import logging
import logging.config
import os

from dotenv import load_dotenv
from fastapi import FastAPI
from fastapi.openapi.utils import get_openapi
from starlette.middleware.cors import CORSMiddleware

from app.config import config as app_config
from app.core.environment import get_environment
from app.core.logger import get_logger
from app.routers.status_router import status_router
from app.routers.ingest_router import ingest_router
from app.routers.retrieve_router import retrieve_router

load_dotenv()

def custom_openapi(app: FastAPI):
    """
    Customize the OpenAPI schema for the FastAPI application.

    This function modifies the default OpenAPI schema generated by FastAPI,
    adding custom metadata for specific routes such as WebSocket endpoints.

    Args:
        app (FastAPI): The FastAPI application instance.

    Returns:
        dict: The modified OpenAPI schema.
    """
    if app.openapi_schema:
        return app.openapi_schema
    openapi_schema = get_openapi(
        title=app_config[get_environment()].APP_NAME,
        version=app_config[get_environment()].APP_VERSION,
        description=app_config[get_environment()].APP_DESCRIPTION,
        routes=app.routes,
    )
    openapi_schema["paths"]["/ingest"] = {
        "post": {
            "summary": "WebSocket Ingest Endpoint",
            "tags": ["Ingestion"],
            "description": "WebSocket endpoint for ingesting documents.",
            "responses": {
                "200": {
                    "description": "Connection established.",
                }
            }
        }
    }
    openapi_schema["paths"]["/retrieve"] = {
        "post": {
            "summary": "WebSocket Retrieve Endpoint",
            "description": "WebSocket endpoint for retrieving documents.",
            "tags": ["Chat completions"],
            "responses": {
                "200": {
                    "description": "Connection established.",
                }
            }
        }
    }
    app.openapi_schema = openapi_schema
    return app.openapi_schema

def create_app() -> FastAPI:
    """
    Create and configure an instance of the FastAPI application.

    This function initializes the FastAPI application with settings and configurations
    based on the environment. It registers routes, middleware, and custom OpenAPI schema.

    Returns:
        FastAPI: The configured FastAPI application instance.

    Raises:
        Exception: If there is an error during the app initialization.
    """
    try:
        APP_ENVIRONMENT = get_environment()

        # Register logging
        logger = get_logger(__name__)
        register_logging(logger, APP_ENVIRONMENT)

        app = FastAPI(
            title=app_config[APP_ENVIRONMENT].APP_NAME,
            version=app_config[APP_ENVIRONMENT].APP_VERSION,
            docs_url=None if APP_ENVIRONMENT == "production" else "/docs",
            redoc_url=None if APP_ENVIRONMENT == "production" else "/redoc",
        )

        # Register the custom OpenAPI function
        app.openapi = lambda: custom_openapi(app)

        # Register routers
        init_routers(app, logger)

        # Make middleware
        make_middleware(app, logger, APP_ENVIRONMENT)

        logger.info("Completed initializing FastAPI app!")
        return app
    except Exception as e:
        logger.error(f"An error occurred during app initialization: {e}")
        raise

def init_routers(app_: FastAPI, logger: logging.Logger) -> None:
    """
    Register routers with the FastAPI application.

    This function includes various routers into the FastAPI application to handle different endpoints.

    Args:
        app_ (FastAPI): The FastAPI application instance.
        logger (logging.Logger): Logger for logging information and errors.
    
    Raises:
        ImportError: If there is an error during router registration.
    """
    try:
        app_.include_router(status_router)
        app_.include_router(ingest_router)
        app_.include_router(retrieve_router)
        logger.info("Registered routes for app!")
    except ImportError as e:
        logger.error(f"Error registering routes: {e}")
        raise

def make_middleware(app_: FastAPI, logger: logging.Logger, environment: str) -> None:
    """
    Configure middleware for the FastAPI application.

    This function sets up middleware, such as CORS, for the FastAPI application based on the environment.

    Args:
        app_ (FastAPI): The FastAPI application instance.
        logger (logging.Logger): Logger for logging information and errors.
        environment (str): The current environment (e.g., "development", "production").
    
    Raises:
        ImportError: If there is an error during middleware configuration.
    """
    try:
        app_.add_middleware(
            CORSMiddleware,
            allow_origins=app_config[environment].CORS_ORIGINS,
            allow_credentials=True,
            allow_methods=app_config[environment].ALLOWED_METHODS,
            allow_headers=app_config[environment].ALLOWED_HEADERS)
        logger.info("Completed making middleware!")
    except ImportError as e:
        logger.error(f"An error occurred during making of middleware: {e}")
        raise

def register_logging(logger: logging.Logger, environment: str) -> None:
    """
    Configure logging for the application.

    This function sets up logging configurations for the application based on the environment settings.

    Args:
        logger (logging.Logger): Logger for logging information and errors.
        environment (str): The current environment (e.g., "development", "production").

    Raises:
        Exception: If there is an error during logging configuration.
    """
    try:
        logging.config.dictConfig(app_config[environment].LOGGING)
        logger.info("Logging configuration was successfully registered.")
    except Exception as e:
        logger.error(f"An error occurred while registering logging configuration: {e}")
        raise
