
import link.kotlin.scripts.dsl.Article
import link.kotlin.scripts.dsl.LinkType.*
import link.kotlin.scripts.dsl.LanguageCodes.*
import java.time.LocalDate

// language=Markdown
val body = """
![](https://cdn-images-1.medium.com/max/880/1*40AToH3O5szPdLfWU3kLVA.jpeg)

> Delegation is an implementation mechanism in which an object forwards or delegates a request to another object. — _(_[_GoF_](https://www.amazon.com/Design-Patterns-Elements-Reusable-Object-Oriented-ebook/dp/B000SEIBB8)_)_

It is said that [inheritance](https://en.wikipedia.org/wiki/Inheritance_%28object-oriented_programming%29) is useful only in case of having [Donald Trump](https://en.wikipedia.org/wiki/Donald_Trump) as a father. Abstract classes have a lot of gotchas in term of designing scalable and reusable architecture. How many times did you create the superclass in your codebase that could not be replaced and blocked you from doing new fancy features? Thanks to [Henry Lieberman](https://g.co/kgs/EzZ2yF) our programming toolbox include an amazing delegation_,_ so it can be used against wrong abstractions. Remember! The wrong abstraction is worst than no abstraction.

Words are cheap, let’s code a little bit.

```kotlin
class CopyPrinter implements Copy, Print {

    private Copy copier;
    private Print printer;

    public CopyPrinter(Copy copier, Print printer) {
        this.copier = copier;
        this.printer = printer;
    }

    @Override
    public Page copy(Page page) {
        return copier.copy(page);
    }

    @Override
    public void print(Page page) {
        printer.print(page);
    }
}

interface Copy {
    Page copy(Page page);
}

interface Print {
    void print(Page page);
}

class Page {}
```

The sample above is pretty straightforward, but still begging for some code generation because of having boilerplate in it. It would be so nice to have delegation as a built-in language feature. In old Java world things like that does not exist, so is there any way to solve that problem differently? It is possible to use annotation processing to create a library for code generation or implement reflection to invoke methods dynamically. That sounds pretty scary, right? As a fan of [Kotlin](https://kotlinlang.org), I will show you how to delegate like a pro.

### [Kotlin](https://kotlinlang.org)

As I have already mentioned, there is a huge need of having delegationpattern implemented in language. Guess what! [Kotlin](https://kotlinlang.org) has it❤.

```kotlin
class CopyPrinter(copier: Copy, printer: Print) : Copy by copier, Print by printer

interface Copy {
    fun copy(page: Page): Page
}

interface Print {
    fun print(page: Page)
}

class Page
```

Just amazing. When I saw it for the first time it literally blew my mind. [Kotlin](https://kotlinlang.org) makes it so clear and concise that I do not have to explain what the code does.

And if that is not enough, [Kotlin](https://kotlinlang.org) have the whole concept of [Delegated Properties](https://kotlinlang.org/docs/reference/delegated-properties.html). If you are not familiar with it, you have to read it and start using it as soon as possible.

> Any piece of code that you’re forced to duplicate or that can be automatically generated by an IDE is a missing language abstraction feature. — ([**Mario Fusco**](https://twitter.com/mariofusco/status/790171082242359296))

#### Notes:

[GoF](https://www.amazon.com/Design-Patterns-Elements-Reusable-Object-Oriented-ebook/dp/B000SEIBB8), [Delegated Properties](https://kotlinlang.org/docs/reference/delegated-properties.html), [Kotlin](https://kotlinlang.org), [Henry Lieberman](https://g.co/kgs/EzZ2yF)

"""

Article(
  title = "Zero boilerplate delegation in Kotlin",
  url = "https://medium.com/@piotr.slesarew/zero-boilerplate-delegation-in-kotlin-e383fdef28eb#.k4sdth6d4",
  categories = listOf(
    "Kotlin"
  ),
  type = article,
  lang = EN,
  author = "Piotr Ślesarew",
  date = LocalDate.of(2016, 10, 28),
  body = body
)
