# Copyright The Lightning AI team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from contextlib import contextmanager
from unittest.mock import ANY, MagicMock, Mock

import pytest
import torch

from lightning.fabric.plugins.precision.utils import _DtypeContextManager
from lightning.pytorch.plugins.precision.fsdp import FSDPPrecision
from tests_pytorch.helpers.runif import RunIf


# Pytest passes args/kwargs to the context manager used with `pytest.warns`.
# `contextlib.nullcontext` doesn't accept them, so this no-op version does.
@contextmanager
def null_ctx(*args, **kwargs):
    yield


@pytest.mark.parametrize(
    ("precision", "expected"),
    [
        ("16-true", (torch.float16, torch.float16, torch.float16)),
        ("bf16-true", (torch.bfloat16, torch.bfloat16, torch.bfloat16)),
        ("16-mixed", (torch.float16, torch.float16, torch.float16)),
        ("bf16-mixed", (torch.bfloat16, torch.bfloat16, torch.bfloat16)),
        ("32-true", (torch.float32, torch.float32, torch.float32)),
    ],
)
def test_fsdp_precision_config(precision, expected):
    plugin = FSDPPrecision(precision=precision)

    warning_ctx = pytest.warns if precision in ("16-true", "bf16-true") else null_ctx

    with warning_ctx(UserWarning, match="enables computation in lower precision"):
        config = plugin.mixed_precision_config

    assert config.param_dtype == expected[0]
    assert config.buffer_dtype == expected[1]
    assert config.reduce_dtype == expected[2]


@pytest.mark.parametrize(
    ("precision", "expected_dtype"),
    [
        ("32-true", torch.float32),
        ("bf16-mixed", torch.float32),
        ("16-mixed", torch.float32),
        ("bf16-true", torch.bfloat16),
        ("16-true", torch.float16),
    ],
)
def test_convert_module(precision, expected_dtype):
    precision = FSDPPrecision(precision=precision)
    module = torch.nn.Linear(2, 2)
    assert module.weight.dtype == module.bias.dtype == torch.float32
    module = precision.convert_module(module)
    assert module.weight.dtype == module.bias.dtype == expected_dtype


def test_fsdp_precision_default_scaler():
    from torch.distributed.fsdp.sharded_grad_scaler import ShardedGradScaler

    precision = FSDPPrecision(precision="16-mixed")
    assert isinstance(precision.scaler, ShardedGradScaler)


def test_fsdp_precision_scaler_with_bf16():
    with pytest.raises(ValueError, match="`precision='bf16-mixed'` does not use a scaler"):
        FSDPPrecision(precision="bf16-mixed", scaler=Mock())

    precision = FSDPPrecision(precision="bf16-mixed")
    assert precision.scaler is None


@RunIf(min_cuda_gpus=1)
def test_fsdp_precision_forward_context_f16():
    """Test to ensure that the context manager correctly is set to float16."""
    from torch.distributed.fsdp.sharded_grad_scaler import ShardedGradScaler

    precision = FSDPPrecision(precision="16-mixed")
    assert isinstance(precision.scaler, ShardedGradScaler)
    assert torch.get_default_dtype() == torch.float32
    with precision.forward_context():
        assert torch.get_autocast_gpu_dtype() == torch.float16
    assert isinstance(precision.forward_context(), torch.autocast)
    assert precision.forward_context().fast_dtype == torch.float16

    precision = FSDPPrecision(precision="16-true")
    assert precision.scaler is None
    assert torch.get_default_dtype() == torch.float32
    with precision.forward_context():
        assert torch.get_default_dtype() == torch.float16
    assert isinstance(precision.forward_context(), _DtypeContextManager)
    assert precision.forward_context()._new_dtype == torch.float16


@RunIf(min_cuda_gpus=1, bf16_cuda=True)
def test_fsdp_precision_forward_context_bf16():
    """Test to ensure that the context manager correctly is set to bfloat16."""
    precision = FSDPPrecision(precision="bf16-mixed")
    assert precision.scaler is None
    with precision.forward_context():
        assert torch.get_autocast_gpu_dtype() == torch.bfloat16
    assert isinstance(precision.forward_context(), torch.autocast)
    assert precision.forward_context().fast_dtype == torch.bfloat16

    precision = FSDPPrecision(precision="bf16-true")
    assert precision.scaler is None
    with precision.forward_context():  # forward context is not using autocast ctx manager
        assert torch.get_default_dtype() == torch.bfloat16
    assert isinstance(precision.forward_context(), _DtypeContextManager)
    assert precision.forward_context()._new_dtype == torch.bfloat16


def test_fsdp_precision_backward():
    precision = FSDPPrecision(precision="16-mixed")
    precision.scaler = Mock()
    precision.scaler.scale = Mock(side_effect=(lambda x: x))
    tensor = Mock()
    model = Mock(trainer=Mock(callbacks=[], profiler=MagicMock()))
    precision.pre_backward(tensor, model)
    precision.backward(tensor, model, None, "positional-arg", keyword="arg")
    precision.scaler.scale.assert_called_once_with(tensor)
    model.backward.assert_called_once_with(tensor, "positional-arg", keyword="arg")


def test_fsdp_precision_optimizer_step_with_scaler():
    precision = FSDPPrecision(precision="16-mixed")
    precision.scaler = Mock()
    model = Mock(trainer=Mock(callbacks=[], profiler=MagicMock()))
    optimizer = Mock()
    closure = Mock()

    precision.optimizer_step(optimizer, model, closure, keyword="arg")
    precision.scaler.step.assert_called_once_with(optimizer, keyword="arg")
    precision.scaler.update.assert_called_once()


def test_fsdp_precision_optimizer_step_without_scaler():
    precision = FSDPPrecision(precision="bf16-mixed")
    assert precision.scaler is None
    model = Mock(trainer=Mock(callbacks=[], profiler=MagicMock()))
    optimizer = Mock()
    closure = Mock()

    precision.optimizer_step(optimizer, model, closure, keyword="arg")
    optimizer.step.assert_called_once_with(closure=ANY, keyword="arg")


def test_invalid_precision_with_fsdp_precision():
    FSDPPrecision("16-mixed")
    FSDPPrecision("bf16-mixed")

    with pytest.raises(ValueError, match="is not supported in FSDP. `precision` must be one of"):
        FSDPPrecision(precision="64-true")
