import React from 'react';
import {
    IconButton,
    Tooltip,
    Typography,
} from '@mui/material';
import { Person, AttachFile, TextSnippet, Engineering, PersonPin, DataObject, QueryBuilder } from '@mui/icons-material';
import { MessageComponentProps, PopulatedMessage } from '../../../../types/MessageTypes';
import CommonCardView from '../../../common/enhanced_component/CardView';
import { hasAnyReferences, References } from '../../../../types/ReferenceTypes';
import AliceMarkdown, { CustomBlockType } from '../../../ui/markdown/alice_markdown/AliceMarkdown';
import EmbeddingChunkViewer from '../../embedding_chunk/embedding_chunk/EmbeddingChunkViewer';
import { getMessageTypeIcon } from '../../../../utils/MessageUtils';
import ContentStats from '../../../ui/markdown/ContentStats';
import GenerationMetadataViewer from '../MetadataViewer';
import DataClusterManager from '../../data_cluster/data_cluster_manager/DataClusterManager';

const MessageCardView: React.FC<MessageComponentProps> = ({
    item,
}) => {
    if (!item) {
        return <Typography>No message data available.</Typography>;
    }
    const populatedItem = item as PopulatedMessage

    const embeddingChunkViewer = populatedItem.embedding && populatedItem.embedding?.length > 0 ?
        populatedItem.embedding.map((chunk, index) => (
            <EmbeddingChunkViewer
                key={chunk._id || `embedding-${index}`}
                item={chunk}
                items={null} onChange={() => null} mode={'view'} handleSave={async () => { }}
            />
        )) : <Typography>No embeddings available</Typography>;


    const subTitle = (
        <ContentStats content={populatedItem.content} >
            <Tooltip title={`Type: ${populatedItem.type}`} arrow>
                <IconButton size="small">{getMessageTypeIcon(populatedItem.type)}</IconButton>
            </Tooltip>
        </ContentStats>
    )
    const listItems = [
        {
            icon: <TextSnippet />,
            primary_text: "Content",
            secondary_text:
                <AliceMarkdown
                    enabledBlocks={[CustomBlockType.ALICE_DOCUMENT, CustomBlockType.ANALYSIS]}
                    role={populatedItem.role}
                    showCopyButton
                >
                    {populatedItem.content}
                </AliceMarkdown>
        },
        {
            icon: <Person />,
            primary_text: "Role",
            secondary_text: populatedItem.role
        },
        {
            icon: <PersonPin />,
            primary_text: "Assistant Name",
            secondary_text: populatedItem.assistant_name ?? "N/A"
        },
        {
            icon: <Engineering />,
            primary_text: "Generated By",
            secondary_text: populatedItem.generated_by
        },
        {
            icon: <AttachFile />,
            primary_text: "References",
            secondary_text: populatedItem.references && hasAnyReferences(populatedItem.references as References) ?
                <DataClusterManager
                    title="References"
                    dataCluster={populatedItem.references}
                    isEditable={false}
                />
                : "N/A"
        },
        {
            icon: <DataObject />,
            primary_text: "Embedding",
            secondary_text: embeddingChunkViewer
        },
        {
            icon: <Person />,
            primary_text: "Metadata",
            secondary_text: populatedItem.creation_metadata && Object.keys(populatedItem.creation_metadata).length > 0 ?
                <GenerationMetadataViewer metadata={populatedItem.creation_metadata} est_tokens={Math.round(item.content.length / 3)} /> : "N/A"
        },
        {
            icon: <QueryBuilder />,
            primary_text: "Created At",
            secondary_text: new Date(populatedItem.createdAt || '').toLocaleString()
        }
    ];

    return (
        <CommonCardView
            elementType='Message'
            title={populatedItem.role}
            subtitle={subTitle}
            id={populatedItem._id}
            listItems={listItems}
            item={populatedItem}
            itemType='messages'
        />
    );
};

export default MessageCardView;