from typing import List
from pydantic import Field
import google.generativeai as genai
from workflow.core.data_structures import (
    ModelConfig, ApiType, FileContentReference, MessageDict, ContentType, FileType, References, FunctionParameters, ParameterDefinition
    )
from workflow.core.api.engines.image_engines.image_gen_engine import ImageGenerationEngine

class GeminiImageGenerationEngine(ImageGenerationEngine):
    input_variables: FunctionParameters = Field(
        default=FunctionParameters(
            type="object",
            properties={
                "prompt": ParameterDefinition(
                    type="string",
                    description="A text description of the desired image(s)."
                ),
                "n": ParameterDefinition(
                    type="integer",
                    description="The number of images to generate.",
                    default=1
                ),
                "size": ParameterDefinition(
                    type="string",
                    description="The size of the generated images.",
                    default="1024x1024"
                )
            },
            required=["prompt"]
        )
    )
    required_api: ApiType = Field(ApiType.IMG_GENERATION, title="The API engine required")

    async def generate_api_response(self, api_data: ModelConfig, prompt: str, n: int = 1, size: str = "1024x1024", **kwargs) -> References:
        """
        Generates images using Google's Gemini model.
        """
        genai.configure(api_key=api_data.api_key)
        imagen = genai.ImageGenerationModel(api_data.model)

        # Map size to aspect ratio (this is an approximation, adjust as needed)
        size_to_aspect_ratio = {
            "1024x1024": "1:1",
            "1024x1792": "9:16",
            "1792x1024": "16:9",
        }
        aspect_ratio = size_to_aspect_ratio.get(size, "1:1")
        try:
            result = imagen.generate_images(
                prompt=prompt,
                number_of_images=n,
                aspect_ratio=aspect_ratio
            )

            file_references: List[FileContentReference] = []
            for index, image in enumerate(result.images):
                filename = self.generate_filename(prompt, api_data.model, index + 1, 'png')
                # Convert the image to base64
                import io
                import base64
                buffered = io.BytesIO()
                image._pil_image.save(buffered, format="PNG")
                img_str = base64.b64encode(buffered.getvalue()).decode()

                file_references.append(FileContentReference(
                    filename=filename,
                    type=FileType.IMAGE,
                    content=img_str,
                    transcript=MessageDict(
                        role='tool', 
                        content=f"Image generated by model {api_data.model}. \nPrompt: '{prompt}' \nAspect ratio: {aspect_ratio}", 
                        type=ContentType.TEXT, 
                        generated_by='tool', 
                        creation_metadata={
                            "prompt": prompt, 
                            "aspect_ratio": aspect_ratio,
                            "model": api_data.model,
                            "cost": api_data.model_costs.cost_per_unit | 0,
                        }
                    )
                ))

            return References(files=file_references)
        except Exception as e:
            raise Exception(f"Error in Gemini image generation API call: {str(e)}")