from typing import Optional, Dict, Any, List
from pydantic import Field, field_validator, BaseModel
from workflow.core.data_structures.base_models import Embeddable
from workflow.core.data_structures.node_response import ExecutionHistoryItem, NodeResponse

class TaskResponse(Embeddable):
    task_id: Optional[str] = Field(None, description="The id of this task node")
    task_name: str = Field(..., description="The name of the task")
    task_description: str = Field(..., description="A detailed description of the task")
    status: str = Field(..., description="The current status of the task: pending, complete, or failed")
    result_code: int = Field(..., description="The result code indicating the success or failure of the task")
    result_diagnostic: Optional[str] = Field(None, description="Diagnostic information for the task, if any")
    task_inputs: Optional[Dict[str, Any]] = Field(None, description="The inputs provided to the task")
    usage_metrics: Optional[Dict[str, Any]] = Field(None, description="Usage metrics for the task, like generated tokens, time taken, and cost.")
    execution_history: Optional[List[ExecutionHistoryItem]] = Field(None, description="Execution history of the task")
    task_outputs: Optional[str] = Field(None, description="The output generated by the task")
    node_references: Optional[List[NodeResponse]] = Field(default_factory=dict, description="A list of NodeReference objects generated by the task")

    def __str__(self) -> str:
        base_str = f"{self.task_name}: {self.task_description}\n\nTask Output:\n\n{self.task_outputs}"
        # if self.node_references:
        #     base_str += "\n\nInner Execution History:\n\n"
        #     base_str += "\n".join([f"{item.node_name}: {item.exit_code}" for item in self.inner_execution_history()])
        return base_str
    
    def model_dump(self, *args, **kwargs):
        data = super().model_dump(*args, **kwargs)
        data['node_references'] = [item.model_dump(*args, **kwargs) if isinstance(item, BaseModel) else item for item in self.node_references]
        return data
        
    def inner_execution_history(self) -> List[NodeResponse]:
        if not self.node_references:
            return []
        return complete_inner_execution_history(self.node_references)
    
    @field_validator('execution_history')
    def validate_history(cls, v):
        if isinstance(v, list):
            if all(isinstance(item, ExecutionHistoryItem) for item in v):
                return v
            else:
                return [ExecutionHistoryItem(**item) for item in v]
        else:
            return []
        
    @field_validator('node_references')
    def validate_references(cls, v):
        if isinstance(v, list):
            if all(isinstance(item, NodeResponse) for item in v):
                return v
            else:
                return [NodeResponse(**item) for item in v]
        else:
            return []
        
def complete_inner_execution_history(nodes: List[NodeResponse], base_order=0) -> List[NodeResponse]:
    """
    Flattens a hierarchy of NodeResponses including task responses found in references.
    
    This function processes a list of NodeResponses and their nested references,
    creating a flattened list where each node's execution order is adjusted based
    on its position in the complete sequence.

    Args:
        nodes (List[NodeResponse]): The list of nodes to process
        base_order (int): The starting execution order for the sequence

    Returns:
        List[NodeResponse]: A flattened list of all nodes including those from nested task responses
    """
    if not nodes:
        return []

    flattened = []
    current_order = base_order

    # Process nodes in execution order
    for node in sorted(nodes, key=lambda x: x.execution_order):
        # Create a new node with updated execution order
        new_node = NodeResponse(
            parent_task_id=node.parent_task_id,
            node_name=node.node_name,
            execution_order=current_order,
            exit_code=node.exit_code,
            references=node.references
        )
        flattened.append(new_node)
        current_order += 1

        # Process task responses in node's references
        if node.references:
            if isinstance(node.references, Dict):
                from workflow.core.data_structures.references import References
                node.references = References(**node.references)
            if node.references.task_responses:
                for task_response in node.references.task_responses:
                    # Recursively process node references from each task response
                    if task_response.node_references:
                        inner_nodes = complete_inner_execution_history(
                            task_response.node_references, 
                            current_order
                        )
                        flattened.extend(inner_nodes)
                        current_order += len(inner_nodes)

    return flattened