"""
Context Composer for PraisonAI Agents.

Assembles context from segments within budget constraints.
"""

from typing import Dict, List, Any, Optional
from .models import (
    ContextLedger, BudgetAllocation, ContextSegment, ContextSnapshot
)
from .tokens import estimate_tokens_heuristic, estimate_messages_tokens
from .ledger import ContextLedgerManager


class ContextComposer:
    """
    Composes context from segments within budget constraints.
    
    Assembles system prompt, rules, skills, memory, tools, and history
    while respecting token budgets and applying trimming as needed.
    
    Example:
        composer = ContextComposer(budget=budgeter.allocate())
        
        result = composer.compose(
            system_prompt="You are a helpful assistant",
            history=messages,
            tools=tool_schemas,
        )
        
        # Use composed messages
        response = llm.chat(result.messages)
    """
    
    def __init__(
        self,
        budget: Optional[BudgetAllocation] = None,
        preserve_tool_pairs: bool = True,
        max_tool_output_tokens: int = 10000,
        keep_recent_turns: int = 5,
    ):
        """
        Initialize composer.
        
        Args:
            budget: Budget allocation (uses defaults if None)
            preserve_tool_pairs: Keep tool_call/tool_result pairs together
            max_tool_output_tokens: Max tokens per tool output
            keep_recent_turns: Minimum recent turns to keep
        """
        self.budget = budget or BudgetAllocation()
        self.preserve_tool_pairs = preserve_tool_pairs
        self.max_tool_output_tokens = max_tool_output_tokens
        self.keep_recent_turns = keep_recent_turns
        
        self._ledger = ContextLedgerManager()
        self._ledger.set_budget(self.budget)
    
    def compose(
        self,
        system_prompt: str = "",
        rules: str = "",
        skills: str = "",
        memory: str = "",
        tools: Optional[List[Dict[str, Any]]] = None,
        history: Optional[List[Dict[str, Any]]] = None,
        agent_id: Optional[str] = None,
    ) -> "ComposedContext":
        """
        Compose context from segments.
        
        Args:
            system_prompt: System prompt content
            rules: Rules content
            skills: Skills content
            memory: Memory content
            tools: Tool schemas
            history: Conversation history
            agent_id: Agent identifier
            
        Returns:
            ComposedContext with messages and ledger
        """
        self._ledger.reset()
        tools = tools or []
        history = history or []
        
        # Track fixed segments
        self._ledger.track_system_prompt(system_prompt)
        self._ledger.track_rules(rules)
        self._ledger.track_skills(skills)
        self._ledger.track_memory(memory)
        self._ledger.track_tools(tools)
        
        # Build combined system message
        system_parts = []
        if system_prompt:
            system_parts.append(system_prompt)
        if rules:
            system_parts.append(f"\n## Rules\n{rules}")
        if skills:
            system_parts.append(f"\n## Skills\n{skills}")
        if memory:
            system_parts.append(f"\n## Memory\n{memory}")
        
        combined_system = "\n".join(system_parts)
        
        # Process history with budget awareness
        processed_history = self._process_history(history)
        
        # Build final messages
        messages = []
        if combined_system:
            messages.append({"role": "system", "content": combined_system})
        messages.extend(processed_history)
        
        # Track final history tokens
        history_tokens = estimate_messages_tokens(processed_history)
        self._ledger.track_segment(
            ContextSegment.HISTORY, 
            history_tokens, 
            len(processed_history)
        )
        
        # Check for overflow
        total = self._ledger.get_total()
        overflow = total > self.budget.usable
        
        return ComposedContext(
            messages=messages,
            tools=tools,
            ledger=self._ledger.get_ledger(),
            total_tokens=total,
            overflow=overflow,
            warnings=self._ledger.get_warnings(),
            agent_id=agent_id,
        )
    
    def _process_history(
        self,
        history: List[Dict[str, Any]]
    ) -> List[Dict[str, Any]]:
        """
        Process history with budget constraints.
        
        Applies:
        - Tool output truncation
        - Sliding window if over budget
        - Tool pair preservation
        """
        if not history:
            return []
        
        # First pass: truncate large tool outputs
        processed = []
        for msg in history:
            processed_msg = self._truncate_tool_output(msg)
            processed.append(processed_msg)
        
        # Check if within budget
        history_budget = self.budget.history_budget
        current_tokens = estimate_messages_tokens(processed)
        
        if current_tokens <= history_budget:
            return processed
        
        # Apply sliding window
        return self._apply_sliding_window(processed, history_budget)
    
    def _truncate_tool_output(self, msg: Dict[str, Any]) -> Dict[str, Any]:
        """Truncate tool output if too large."""
        if msg.get("role") != "tool" and not msg.get("tool_call_id"):
            return msg
        
        content = msg.get("content", "")
        if not isinstance(content, str):
            return msg
        
        tokens = estimate_tokens_heuristic(content)
        if tokens <= self.max_tool_output_tokens:
            return msg
        
        # Truncate to max tokens (rough: 4 chars per token)
        max_chars = self.max_tool_output_tokens * 4
        truncated = content[:max_chars] + "\n...[output truncated]..."
        
        result = msg.copy()
        result["content"] = truncated
        result["_truncated"] = True
        return result
    
    def _apply_sliding_window(
        self,
        history: List[Dict[str, Any]],
        budget: int
    ) -> List[Dict[str, Any]]:
        """Apply sliding window to fit within budget."""
        if not history:
            return []
        
        # Always keep at least recent turns
        min_keep = self.keep_recent_turns * 2  # user + assistant per turn
        
        # Start from end, add messages until budget exceeded
        result = []
        current_tokens = 0
        
        # Track tool call IDs to preserve pairs
        tool_call_ids = set()
        
        for msg in reversed(history):
            msg_tokens = estimate_tokens_heuristic(str(msg.get("content", "")))
            
            if current_tokens + msg_tokens > budget and len(result) >= min_keep:
                break
            
            result.insert(0, msg)
            current_tokens += msg_tokens
            
            # Track tool calls for pair preservation
            if self.preserve_tool_pairs:
                if "tool_calls" in msg:
                    for tc in msg.get("tool_calls", []):
                        if isinstance(tc, dict) and "id" in tc:
                            tool_call_ids.add(tc["id"])
                if msg.get("tool_call_id"):
                    tool_call_ids.add(msg["tool_call_id"])
        
        # Ensure tool pairs are complete
        if self.preserve_tool_pairs:
            result = self._ensure_tool_pairs(result, history, tool_call_ids)
        
        return result
    
    def _ensure_tool_pairs(
        self,
        result: List[Dict[str, Any]],
        full_history: List[Dict[str, Any]],
        tool_call_ids: set
    ) -> List[Dict[str, Any]]:
        """Ensure tool_call/tool_result pairs are complete."""
        # Find tool calls in result
        result_call_ids = set()
        result_response_ids = set()
        
        for msg in result:
            if "tool_calls" in msg:
                for tc in msg.get("tool_calls", []):
                    if isinstance(tc, dict) and "id" in tc:
                        result_call_ids.add(tc["id"])
            if msg.get("tool_call_id"):
                result_response_ids.add(msg["tool_call_id"])
        
        # Find missing pairs
        missing_calls = result_response_ids - result_call_ids
        missing_responses = result_call_ids - result_response_ids
        
        if not missing_calls and not missing_responses:
            return result
        
        # Add missing messages from full history
        for msg in full_history:
            # Add missing tool calls
            if "tool_calls" in msg:
                for tc in msg.get("tool_calls", []):
                    if isinstance(tc, dict) and tc.get("id") in missing_calls:
                        if msg not in result:
                            result.insert(0, msg)
                        break
            
            # Add missing tool responses
            if msg.get("tool_call_id") in missing_responses:
                if msg not in result:
                    result.append(msg)
        
        return result
    
    def get_ledger(self) -> ContextLedger:
        """Get current ledger."""
        return self._ledger.get_ledger()
    
    def get_warnings(self) -> List[str]:
        """Get current warnings."""
        return self._ledger.get_warnings()


class ComposedContext:
    """Result of context composition."""
    
    def __init__(
        self,
        messages: List[Dict[str, Any]],
        tools: List[Dict[str, Any]],
        ledger: ContextLedger,
        total_tokens: int,
        overflow: bool,
        warnings: List[str],
        agent_id: Optional[str] = None,
    ):
        self.messages = messages
        self.tools = tools
        self.ledger = ledger
        self.total_tokens = total_tokens
        self.overflow = overflow
        self.warnings = warnings
        self.agent_id = agent_id
    
    def to_snapshot(
        self,
        session_id: str = "",
        model_name: str = "",
        budget: Optional[BudgetAllocation] = None,
    ) -> ContextSnapshot:
        """Convert to snapshot for monitoring."""
        return ContextSnapshot(
            session_id=session_id,
            agent_name=self.agent_id or "",
            model_name=model_name,
            budget=budget,
            ledger=self.ledger,
            utilization=self.total_tokens / budget.usable if budget else 0.0,
            history_content=self.messages,
            warnings=self.warnings,
        )
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary."""
        return {
            "message_count": len(self.messages),
            "tool_count": len(self.tools),
            "total_tokens": self.total_tokens,
            "overflow": self.overflow,
            "warnings": self.warnings,
            "ledger": self.ledger.to_dict(),
        }
