"""
Context Manager Facade for PraisonAI Agents.

Provides a unified interface for context management:
- Budgeting and allocation
- Token estimation with validation
- Composition within limits
- Optimization with benefit checking
- Monitoring with snapshot hooks
- Multi-agent orchestration support
- Optimization history tracking

This is the main entry point for context management in both SDK and CLI.
"""

import hashlib
import json
import threading
import atexit
from typing import Dict, List, Any, Optional, Literal, Callable, Tuple
from dataclasses import dataclass, field, asdict
from datetime import datetime
from pathlib import Path
from enum import Enum

from .models import (
    ContextLedger, BudgetAllocation, OptimizerStrategy, OptimizationResult
)
from .tokens import estimate_tokens_heuristic, estimate_messages_tokens
from .budgeter import ContextBudgeter
from .ledger import ContextLedgerManager, MultiAgentLedger
from .optimizer import get_optimizer
from .monitor import ContextMonitor, MultiAgentMonitor


class EstimationMode(str, Enum):
    """Token estimation modes."""
    HEURISTIC = "heuristic"
    ACCURATE = "accurate"
    VALIDATED = "validated"


class ContextShareMode(str, Enum):
    """How context is shared between agents."""
    NONE = "none"
    SUMMARY = "summary"
    FULL = "full"


class ToolShareMode(str, Enum):
    """How tools are shared between agents."""
    NONE = "none"
    SAFE = "safe"
    FULL = "full"


class OptimizationEventType(str, Enum):
    """Types of optimization events."""
    NORMALIZE = "normalize"
    CAP_OUTPUTS = "cap_outputs"
    PRUNE_TOOLS = "prune_tools"
    SLIDING_WINDOW = "sliding_window"
    SUMMARIZE = "summarize"
    BENEFIT_CHECK = "benefit_check"
    REVERT = "revert"
    SNAPSHOT = "snapshot"
    OVERFLOW_DETECTED = "overflow_detected"
    AUTO_COMPACT = "auto_compact"


@dataclass
class ContextPolicy:
    """
    Policy for context sharing during agent handoffs.
    
    Controls how context is passed between agents in multi-agent scenarios.
    """
    share: bool = False
    share_mode: ContextShareMode = ContextShareMode.NONE
    max_tokens: int = 0  # 0 = no limit
    tools_share: ToolShareMode = ToolShareMode.NONE
    preserve_system: bool = True
    preserve_recent_turns: int = 3
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            "share": self.share,
            "share_mode": self.share_mode.value,
            "max_tokens": self.max_tokens,
            "tools_share": self.tools_share.value,
            "preserve_system": self.preserve_system,
            "preserve_recent_turns": self.preserve_recent_turns,
        }


@dataclass
class OptimizationEvent:
    """Record of an optimization event."""
    timestamp: str
    event_type: OptimizationEventType
    strategy: Optional[str] = None
    tokens_before: int = 0
    tokens_after: int = 0
    tokens_saved: int = 0
    messages_affected: int = 0
    details: Dict[str, Any] = field(default_factory=dict)
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            "timestamp": self.timestamp,
            "event_type": self.event_type.value,
            "strategy": self.strategy,
            "tokens_before": self.tokens_before,
            "tokens_after": self.tokens_after,
            "tokens_saved": self.tokens_saved,
            "messages_affected": self.messages_affected,
            "details": self.details,
        }


@dataclass
class EstimationMetrics:
    """Metrics for token estimation accuracy."""
    heuristic_estimate: int = 0
    accurate_estimate: int = 0
    error_pct: float = 0.0
    estimator_used: EstimationMode = EstimationMode.HEURISTIC
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            "heuristic_estimate": self.heuristic_estimate,
            "accurate_estimate": self.accurate_estimate,
            "error_pct": self.error_pct,
            "estimator_used": self.estimator_used.value,
        }


@dataclass
class PerToolBudget:
    """Per-tool token budget configuration."""
    tool_name: str
    max_output_tokens: int = 10000
    protected: bool = False
    
    def to_dict(self) -> Dict[str, Any]:
        return asdict(self)


@dataclass
class SnapshotHookData:
    """Data captured at LLM call boundary for exact snapshot."""
    timestamp: str
    messages: List[Dict[str, Any]]
    tools: List[Dict[str, Any]]
    message_hash: str
    tools_hash: str
    ledger: Optional[ContextLedger] = None
    budget: Optional[BudgetAllocation] = None
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            "timestamp": self.timestamp,
            "message_count": len(self.messages),
            "tools_count": len(self.tools),
            "message_hash": self.message_hash,
            "tools_hash": self.tools_hash,
            "ledger": self.ledger.to_dict() if self.ledger else None,
            "budget": self.budget.to_dict() if self.budget else None,
        }


@dataclass
class ManagerConfig:
    """
    Complete configuration for ContextManager.
    
    Consolidates all context management settings with proper precedence.
    """
    # Auto-compaction
    auto_compact: bool = True
    compact_threshold: float = 0.8
    strategy: OptimizerStrategy = OptimizerStrategy.SMART
    
    # Compression benefit check
    compression_min_gain_pct: float = 5.0
    compression_max_attempts: int = 3
    
    # Budget
    output_reserve: int = 8000
    history_ratio: float = 0.6
    default_tool_output_max: int = 10000
    
    # Per-tool budgets
    tool_budgets: Dict[str, PerToolBudget] = field(default_factory=dict)
    protected_tools: List[str] = field(default_factory=list)
    
    # Estimation
    estimation_mode: EstimationMode = EstimationMode.HEURISTIC
    log_estimation_mismatch: bool = False
    mismatch_threshold_pct: float = 15.0
    
    # Monitoring
    monitor_enabled: bool = False
    monitor_path: str = "./context.txt"
    monitor_format: Literal["human", "json"] = "human"
    monitor_frequency: Literal["turn", "tool_call", "manual", "overflow"] = "turn"
    monitor_write_mode: Literal["sync", "async"] = "sync"
    redact_sensitive: bool = True
    snapshot_timing: Literal["pre_optimization", "post_optimization", "both"] = "post_optimization"
    
    # Path validation
    allow_absolute_paths: bool = False
    
    # Pruning
    prune_after_tokens: int = 40000
    keep_recent_turns: int = 5
    
    # Source tracking
    source: str = "defaults"  # defaults, env, config_file, cli
    
    def to_dict(self) -> Dict[str, Any]:
        result = {
            "auto_compact": self.auto_compact,
            "compact_threshold": self.compact_threshold,
            "strategy": self.strategy.value,
            "compression_min_gain_pct": self.compression_min_gain_pct,
            "compression_max_attempts": self.compression_max_attempts,
            "output_reserve": self.output_reserve,
            "history_ratio": self.history_ratio,
            "default_tool_output_max": self.default_tool_output_max,
            "tool_budgets": {k: v.to_dict() for k, v in self.tool_budgets.items()},
            "protected_tools": self.protected_tools,
            "estimation_mode": self.estimation_mode.value,
            "log_estimation_mismatch": self.log_estimation_mismatch,
            "mismatch_threshold_pct": self.mismatch_threshold_pct,
            "monitor_enabled": self.monitor_enabled,
            "monitor_path": self.monitor_path,
            "monitor_format": self.monitor_format,
            "monitor_frequency": self.monitor_frequency,
            "monitor_write_mode": self.monitor_write_mode,
            "redact_sensitive": self.redact_sensitive,
            "snapshot_timing": self.snapshot_timing,
            "allow_absolute_paths": self.allow_absolute_paths,
            "prune_after_tokens": self.prune_after_tokens,
            "keep_recent_turns": self.keep_recent_turns,
            "source": self.source,
        }
        return result
    
    @classmethod
    def from_env(cls) -> "ManagerConfig":
        """Load config from environment variables."""
        import os
        
        def get_bool(key: str, default: bool) -> bool:
            val = os.getenv(key, str(default)).lower()
            return val in ("true", "1", "yes", "on")
        
        def get_float(key: str, default: float) -> float:
            try:
                return float(os.getenv(key, str(default)))
            except ValueError:
                return default
        
        def get_int(key: str, default: int) -> int:
            try:
                return int(os.getenv(key, str(default)))
            except ValueError:
                return default
        
        strategy_str = os.getenv("PRAISONAI_CONTEXT_STRATEGY", "smart")
        try:
            strategy = OptimizerStrategy(strategy_str)
        except ValueError:
            strategy = OptimizerStrategy.SMART
        
        estimation_str = os.getenv("PRAISONAI_CONTEXT_ESTIMATION_MODE", "heuristic")
        try:
            estimation_mode = EstimationMode(estimation_str)
        except ValueError:
            estimation_mode = EstimationMode.HEURISTIC
        
        return cls(
            auto_compact=get_bool("PRAISONAI_CONTEXT_AUTO_COMPACT", True),
            compact_threshold=get_float("PRAISONAI_CONTEXT_THRESHOLD", 0.8),
            strategy=strategy,
            compression_min_gain_pct=get_float("PRAISONAI_CONTEXT_COMPRESSION_MIN_GAIN", 5.0),
            compression_max_attempts=get_int("PRAISONAI_CONTEXT_COMPRESSION_MAX_ATTEMPTS", 3),
            output_reserve=get_int("PRAISONAI_CONTEXT_OUTPUT_RESERVE", 8000),
            default_tool_output_max=get_int("PRAISONAI_CONTEXT_TOOL_OUTPUT_MAX", 10000),
            estimation_mode=estimation_mode,
            log_estimation_mismatch=get_bool("PRAISONAI_CONTEXT_LOG_MISMATCH", False),
            monitor_enabled=get_bool("PRAISONAI_CONTEXT_MONITOR", False),
            monitor_path=os.getenv("PRAISONAI_CONTEXT_MONITOR_PATH", "./context.txt"),
            monitor_format=os.getenv("PRAISONAI_CONTEXT_MONITOR_FORMAT", "human"),
            monitor_frequency=os.getenv("PRAISONAI_CONTEXT_MONITOR_FREQUENCY", "turn"),
            monitor_write_mode=os.getenv("PRAISONAI_CONTEXT_MONITOR_WRITE_MODE", "sync"),
            redact_sensitive=get_bool("PRAISONAI_CONTEXT_REDACT", True),
            source="env",
        )
    
    def merge(self, **overrides) -> "ManagerConfig":
        """Create new config with overrides applied."""
        current = self.to_dict()
        
        for key, value in overrides.items():
            if value is not None and key in current:
                current[key] = value
        
        # Handle enum conversions
        if isinstance(current.get("strategy"), str):
            try:
                current["strategy"] = OptimizerStrategy(current["strategy"])
            except ValueError:
                current["strategy"] = OptimizerStrategy.SMART
        
        if isinstance(current.get("estimation_mode"), str):
            try:
                current["estimation_mode"] = EstimationMode(current["estimation_mode"])
            except ValueError:
                current["estimation_mode"] = EstimationMode.HEURISTIC
        
        # Reconstruct tool budgets
        tool_budgets = {}
        for name, budget_dict in current.get("tool_budgets", {}).items():
            if isinstance(budget_dict, dict):
                tool_budgets[name] = PerToolBudget(**budget_dict)
            elif isinstance(budget_dict, PerToolBudget):
                tool_budgets[name] = budget_dict
        current["tool_budgets"] = tool_budgets
        
        return ManagerConfig(**current)


class ContextManager:
    """
    Unified facade for context management.
    
    Orchestrates budgeting, composition, optimization, and monitoring.
    Provides hooks for exact LLM boundary snapshots.
    Tracks optimization history for debugging.
    
    Example:
        manager = ContextManager(model="gpt-4o")
        
        # Process messages before LLM call
        result = manager.process(
            messages=messages,
            system_prompt=system_prompt,
            tools=tools,
        )
        
        # Get optimized messages
        optimized_messages = result["messages"]
        
        # Check if optimization occurred
        if result["optimized"]:
            print(f"Saved {result['tokens_saved']} tokens")
    """
    
    def __init__(
        self,
        model: str = "gpt-4o-mini",
        config: Optional[ManagerConfig] = None,
        session_id: str = "",
        agent_name: str = "",
    ):
        """
        Initialize context manager.
        
        Args:
            model: Model name for budget calculation
            config: Manager configuration
            session_id: Session identifier
            agent_name: Agent name for monitoring
        """
        self.model = model
        self.config = config or ManagerConfig.from_env()
        self.session_id = session_id
        self.agent_name = agent_name
        
        # Core components
        self._budgeter = ContextBudgeter(
            model=model,
            output_reserve=self.config.output_reserve,
        )
        self._ledger = ContextLedgerManager(agent_id=agent_name)
        self._monitor = ContextMonitor(
            enabled=self.config.monitor_enabled,
            path=self.config.monitor_path,
            format=self.config.monitor_format,
            frequency=self.config.monitor_frequency,
            redact_sensitive=self.config.redact_sensitive,
        )
        
        # Budget allocation
        self._budget = self._budgeter.allocate()
        self._ledger.set_budget(self._budget)
        
        # Optimization history
        self._history: List[OptimizationEvent] = []
        self._max_history = 100
        
        # Estimation metrics
        self._estimation_metrics: Optional[EstimationMetrics] = None
        
        # Snapshot hook data
        self._last_snapshot_hook: Optional[SnapshotHookData] = None
        self._snapshot_callbacks: List[Callable[[SnapshotHookData], None]] = []
        
        # Async write buffer
        self._async_buffer: List[Tuple[Path, str]] = []
        self._async_lock = threading.Lock()
        self._async_thread: Optional[threading.Thread] = None
        
        # Register exit handler for async flush
        if self.config.monitor_write_mode == "async":
            atexit.register(self._flush_async_writes)
        
        # Message hash cache for estimation
        self._estimation_cache: Dict[str, int] = {}
        self._cache_max_size = 1000
    
    def process(
        self,
        messages: List[Dict[str, Any]],
        system_prompt: str = "",
        tools: Optional[List[Dict[str, Any]]] = None,
        trigger: Literal["turn", "tool_call", "manual", "overflow"] = "turn",
    ) -> Dict[str, Any]:
        """
        Process messages through the context pipeline.
        
        Applies budgeting, optimization, and monitoring.
        
        Args:
            messages: Conversation messages
            system_prompt: System prompt content
            tools: Tool schemas
            trigger: What triggered this processing
            
        Returns:
            Dict with processed messages and metadata
        """
        tools = tools or []
        
        # Track segments
        self._ledger.reset()
        if system_prompt:
            self._ledger.track_system_prompt(system_prompt)
        if tools:
            self._ledger.track_tools(tools)
        self._ledger.track_history(messages)
        
        # Check for overflow
        utilization = self._ledger.get_utilization()
        needs_optimization = utilization >= self.config.compact_threshold
        
        result = {
            "messages": messages,
            "optimized": False,
            "tokens_before": self._ledger.get_total(),
            "tokens_after": self._ledger.get_total(),
            "tokens_saved": 0,
            "utilization": utilization,
            "warnings": self._ledger.get_warnings(),
            "optimization_result": None,
        }
        
        # Log overflow detection
        if needs_optimization:
            self._add_history_event(
                OptimizationEventType.OVERFLOW_DETECTED,
                tokens_before=result["tokens_before"],
                details={"utilization": utilization, "threshold": self.config.compact_threshold},
            )
        
        # Auto-compact if needed
        if self.config.auto_compact and needs_optimization:
            optimized_messages, opt_result = self._optimize_with_benefit_check(
                messages,
                self._budget.history_budget,
            )
            
            if opt_result and opt_result.tokens_saved > 0:
                result["messages"] = optimized_messages
                result["optimized"] = True
                result["tokens_after"] = opt_result.optimized_tokens
                result["tokens_saved"] = opt_result.tokens_saved
                result["optimization_result"] = opt_result
                
                # Update ledger
                self._ledger.track_history(optimized_messages)
                result["utilization"] = self._ledger.get_utilization()
                result["warnings"] = self._ledger.get_warnings()
        
        # Pre-optimization snapshot
        if self.config.snapshot_timing in ("pre_optimization", "both"):
            self._take_snapshot(messages, tools, trigger, "pre_optimization")
        
        # Post-optimization snapshot
        if self.config.snapshot_timing in ("post_optimization", "both"):
            self._take_snapshot(result["messages"], tools, trigger, "post_optimization")
        
        return result
    
    def _optimize_with_benefit_check(
        self,
        messages: List[Dict[str, Any]],
        target_tokens: int,
    ) -> Tuple[List[Dict[str, Any]], Optional[OptimizationResult]]:
        """
        Optimize with benefit checking to prevent token inflation.
        
        Args:
            messages: Messages to optimize
            target_tokens: Target token count
            
        Returns:
            Tuple of (optimized_messages, result)
        """
        original_tokens = estimate_messages_tokens(messages)
        
        if original_tokens <= target_tokens:
            return messages, None
        
        # Get optimizer
        optimizer = get_optimizer(
            self.config.strategy,
            preserve_recent=self.config.keep_recent_turns,
            protected_tools=self.config.protected_tools,
        )
        
        # Try optimization
        optimized, result = optimizer.optimize(messages, target_tokens, self._ledger.get_ledger())
        
        # Benefit check
        min_gain = self.config.compression_min_gain_pct / 100.0
        actual_gain = (original_tokens - result.optimized_tokens) / original_tokens if original_tokens > 0 else 0
        
        self._add_history_event(
            OptimizationEventType.BENEFIT_CHECK,
            strategy=self.config.strategy.value,
            tokens_before=original_tokens,
            tokens_after=result.optimized_tokens,
            tokens_saved=result.tokens_saved,
            details={
                "min_gain_required": min_gain,
                "actual_gain": actual_gain,
                "beneficial": actual_gain >= min_gain,
            },
        )
        
        if actual_gain < min_gain:
            # Not beneficial - revert
            self._add_history_event(
                OptimizationEventType.REVERT,
                tokens_before=result.optimized_tokens,
                tokens_after=original_tokens,
                details={"reason": "compression_not_beneficial"},
            )
            return messages, None
        
        # Log successful optimization
        self._add_history_event(
            OptimizationEventType.AUTO_COMPACT,
            strategy=self.config.strategy.value,
            tokens_before=original_tokens,
            tokens_after=result.optimized_tokens,
            tokens_saved=result.tokens_saved,
            messages_affected=result.messages_removed,
        )
        
        return optimized, result
    
    def _take_snapshot(
        self,
        messages: List[Dict[str, Any]],
        tools: List[Dict[str, Any]],
        trigger: str,
        timing: str,
    ) -> None:
        """Take a context snapshot."""
        if not self._monitor.enabled:
            return
        
        self._monitor.snapshot(
            ledger=self._ledger.get_ledger(),
            budget=self._budget,
            messages=messages,
            session_id=self.session_id,
            agent_name=self.agent_name,
            model_name=self.model,
            trigger=trigger,
        )
        
        self._add_history_event(
            OptimizationEventType.SNAPSHOT,
            details={"timing": timing, "trigger": trigger},
        )
    
    def capture_llm_boundary(
        self,
        messages: List[Dict[str, Any]],
        tools: List[Dict[str, Any]],
    ) -> SnapshotHookData:
        """
        Capture exact state at LLM call boundary.
        
        Call this immediately before sending to LLM to get exact snapshot.
        
        Args:
            messages: Exact messages being sent
            tools: Exact tool schemas being sent
            
        Returns:
            SnapshotHookData with hashes for verification
        """
        # Compute hashes
        messages_json = json.dumps(messages, sort_keys=True, default=str)
        tools_json = json.dumps(tools, sort_keys=True, default=str)
        
        message_hash = hashlib.sha256(messages_json.encode()).hexdigest()[:16]
        tools_hash = hashlib.sha256(tools_json.encode()).hexdigest()[:16]
        
        hook_data = SnapshotHookData(
            timestamp=datetime.utcnow().isoformat() + "Z",
            messages=messages,
            tools=tools,
            message_hash=message_hash,
            tools_hash=tools_hash,
            ledger=self._ledger.get_ledger(),
            budget=self._budget,
        )
        
        self._last_snapshot_hook = hook_data
        
        # Call registered callbacks
        for callback in self._snapshot_callbacks:
            try:
                callback(hook_data)
            except Exception:
                pass  # Don't let callbacks break the flow
        
        return hook_data
    
    def register_snapshot_callback(
        self,
        callback: Callable[[SnapshotHookData], None]
    ) -> None:
        """Register a callback for LLM boundary snapshots."""
        self._snapshot_callbacks.append(callback)
    
    def get_last_snapshot_hook(self) -> Optional[SnapshotHookData]:
        """Get the last LLM boundary snapshot."""
        return self._last_snapshot_hook
    
    def estimate_tokens(
        self,
        text: str,
        validate: bool = False,
    ) -> Tuple[int, Optional[EstimationMetrics]]:
        """
        Estimate tokens with optional validation.
        
        Args:
            text: Text to estimate
            validate: Whether to validate against accurate count
            
        Returns:
            Tuple of (token_count, metrics)
        """
        # Check cache
        cache_key = hashlib.md5(text.encode()).hexdigest()
        if cache_key in self._estimation_cache:
            return self._estimation_cache[cache_key], None
        
        heuristic = estimate_tokens_heuristic(text)
        metrics = None
        
        if validate or self.config.estimation_mode == EstimationMode.VALIDATED:
            try:
                from .tokens import estimate_tokens_accurate
                accurate = estimate_tokens_accurate(text, self.model)
                
                error_pct = abs(heuristic - accurate) / accurate * 100 if accurate > 0 else 0
                
                metrics = EstimationMetrics(
                    heuristic_estimate=heuristic,
                    accurate_estimate=accurate,
                    error_pct=error_pct,
                    estimator_used=EstimationMode.VALIDATED,
                )
                
                self._estimation_metrics = metrics
                
                # Log if mismatch exceeds threshold
                if self.config.log_estimation_mismatch and error_pct > self.config.mismatch_threshold_pct:
                    import logging
                    logging.getLogger(__name__).warning(
                        f"Token estimation mismatch: heuristic={heuristic}, accurate={accurate}, error={error_pct:.1f}%"
                    )
                
                # Use accurate if validated mode
                if self.config.estimation_mode == EstimationMode.ACCURATE:
                    result = accurate
                else:
                    result = heuristic
            except ImportError:
                result = heuristic
                metrics = EstimationMetrics(
                    heuristic_estimate=heuristic,
                    estimator_used=EstimationMode.HEURISTIC,
                )
        else:
            result = heuristic
        
        # Cache result
        if len(self._estimation_cache) < self._cache_max_size:
            self._estimation_cache[cache_key] = result
        
        return result, metrics
    
    def get_tool_budget(self, tool_name: str) -> int:
        """Get token budget for a specific tool."""
        if tool_name in self.config.tool_budgets:
            return self.config.tool_budgets[tool_name].max_output_tokens
        return self.config.default_tool_output_max
    
    def set_tool_budget(self, tool_name: str, max_tokens: int, protected: bool = False) -> None:
        """Set token budget for a specific tool."""
        self.config.tool_budgets[tool_name] = PerToolBudget(
            tool_name=tool_name,
            max_output_tokens=max_tokens,
            protected=protected,
        )
        if protected and tool_name not in self.config.protected_tools:
            self.config.protected_tools.append(tool_name)
    
    def truncate_tool_output(self, tool_name: str, output: str) -> str:
        """Truncate tool output according to its budget."""
        max_tokens = self.get_tool_budget(tool_name)
        
        # Estimate current tokens
        current_tokens, _ = self.estimate_tokens(output)
        
        if current_tokens <= max_tokens:
            return output
        
        # Truncate by character ratio (approximate)
        ratio = max_tokens / current_tokens
        max_chars = int(len(output) * ratio * 0.9)  # 10% safety margin
        
        truncated = output[:max_chars] + "\n...[output truncated]..."
        
        self._add_history_event(
            OptimizationEventType.CAP_OUTPUTS,
            tokens_before=current_tokens,
            tokens_after=max_tokens,
            details={"tool_name": tool_name, "original_length": len(output)},
        )
        
        return truncated
    
    def _add_history_event(
        self,
        event_type: OptimizationEventType,
        strategy: Optional[str] = None,
        tokens_before: int = 0,
        tokens_after: int = 0,
        tokens_saved: int = 0,
        messages_affected: int = 0,
        details: Optional[Dict[str, Any]] = None,
    ) -> None:
        """Add an event to optimization history."""
        event = OptimizationEvent(
            timestamp=datetime.utcnow().isoformat() + "Z",
            event_type=event_type,
            strategy=strategy,
            tokens_before=tokens_before,
            tokens_after=tokens_after,
            tokens_saved=tokens_saved,
            messages_affected=messages_affected,
            details=details or {},
        )
        
        self._history.append(event)
        
        # Trim history if too long
        if len(self._history) > self._max_history:
            self._history = self._history[-self._max_history:]
    
    def get_history(self) -> List[Dict[str, Any]]:
        """Get optimization history."""
        return [e.to_dict() for e in self._history]
    
    def get_stats(self) -> Dict[str, Any]:
        """Get current context statistics."""
        return {
            "model": self.model,
            "session_id": self.session_id,
            "agent_name": self.agent_name,
            "budget": self._budget.to_dict(),
            "ledger": self._ledger.to_dict(),
            "utilization": self._ledger.get_utilization(),
            "warnings": self._ledger.get_warnings(),
            "history_events": len(self._history),
            "estimation_metrics": self._estimation_metrics.to_dict() if self._estimation_metrics else None,
            "monitor_stats": self._monitor.get_stats(),
        }
    
    def get_resolved_config(self) -> Dict[str, Any]:
        """Get the fully resolved configuration with source info."""
        return {
            "config": self.config.to_dict(),
            "precedence": "CLI > ENV > config.yaml > defaults",
            "effective_budget": self._budget.to_dict(),
        }
    
    def _flush_async_writes(self) -> None:
        """Flush any pending async writes."""
        with self._async_lock:
            for path, content in self._async_buffer:
                try:
                    path.parent.mkdir(parents=True, exist_ok=True)
                    path.write_text(content, encoding="utf-8")
                except Exception:
                    pass
            self._async_buffer.clear()
    
    def reset(self) -> None:
        """Reset manager state."""
        self._ledger.reset()
        self._history.clear()
        self._estimation_cache.clear()
        self._last_snapshot_hook = None


class MultiAgentContextManager:
    """
    Context manager for multi-agent orchestration.
    
    Provides per-agent isolation with controlled sharing policies.
    """
    
    def __init__(
        self,
        config: Optional[ManagerConfig] = None,
        default_policy: Optional[ContextPolicy] = None,
    ):
        """
        Initialize multi-agent context manager.
        
        Args:
            config: Base configuration
            default_policy: Default context sharing policy
        """
        self.config = config or ManagerConfig.from_env()
        self.default_policy = default_policy or ContextPolicy()
        
        self._agents: Dict[str, ContextManager] = {}
        self._multi_ledger = MultiAgentLedger()
        self._multi_monitor = MultiAgentMonitor(
            enabled=self.config.monitor_enabled,
            format=self.config.monitor_format,
            redact_sensitive=self.config.redact_sensitive,
        )
        
        # Agent policies
        self._policies: Dict[str, ContextPolicy] = {}
        
        # Shared context
        self._shared_context: List[Dict[str, Any]] = []
        self._shared_tokens: int = 0
    
    def get_agent_manager(
        self,
        agent_id: str,
        model: str = "gpt-4o-mini",
    ) -> ContextManager:
        """Get or create context manager for an agent."""
        if agent_id not in self._agents:
            self._agents[agent_id] = ContextManager(
                model=model,
                config=self.config,
                agent_name=agent_id,
            )
        return self._agents[agent_id]
    
    def set_agent_policy(self, agent_id: str, policy: ContextPolicy) -> None:
        """Set context policy for an agent."""
        self._policies[agent_id] = policy
    
    def get_agent_policy(self, agent_id: str) -> ContextPolicy:
        """Get context policy for an agent."""
        return self._policies.get(agent_id, self.default_policy)
    
    def prepare_handoff(
        self,
        from_agent: str,
        to_agent: str,
        messages: List[Dict[str, Any]],
        policy: Optional[ContextPolicy] = None,
    ) -> List[Dict[str, Any]]:
        """
        Prepare context for handoff between agents.
        
        Args:
            from_agent: Source agent ID
            to_agent: Target agent ID
            messages: Current messages
            policy: Override policy for this handoff
            
        Returns:
            Messages to pass to target agent
        """
        policy = policy or self.get_agent_policy(to_agent)
        
        if not policy.share:
            return []
        
        result = []
        
        if policy.share_mode == ContextShareMode.NONE:
            return []
        
        elif policy.share_mode == ContextShareMode.SUMMARY:
            # Create summary of conversation
            summary_parts = []
            for msg in messages[-10:]:  # Last 10 messages
                role = msg.get("role", "")
                content = msg.get("content", "")
                if isinstance(content, str) and content:
                    summary_parts.append(f"[{role}]: {content[:100]}...")
            
            if summary_parts:
                result.append({
                    "role": "system",
                    "content": f"[Context from {from_agent}]\n" + "\n".join(summary_parts),
                    "_handoff_summary": True,
                })
        
        elif policy.share_mode == ContextShareMode.FULL:
            # Share full context with limits
            if policy.preserve_system:
                result.extend([m for m in messages if m.get("role") == "system"])
            
            non_system = [m for m in messages if m.get("role") != "system"]
            
            if policy.preserve_recent_turns > 0:
                result.extend(non_system[-policy.preserve_recent_turns * 2:])
            else:
                result.extend(non_system)
        
        # Apply token limit
        if policy.max_tokens > 0:
            total_tokens = estimate_messages_tokens(result)
            while total_tokens > policy.max_tokens and len(result) > 1:
                # Remove oldest non-system message
                for i, msg in enumerate(result):
                    if msg.get("role") != "system":
                        result.pop(i)
                        break
                total_tokens = estimate_messages_tokens(result)
        
        return result
    
    def get_combined_stats(self) -> Dict[str, Any]:
        """Get combined statistics across all agents."""
        return {
            "agents": {
                agent_id: manager.get_stats()
                for agent_id, manager in self._agents.items()
            },
            "combined_total": self._multi_ledger.get_combined_total(),
            "shared_tokens": self._shared_tokens,
            "policies": {
                agent_id: policy.to_dict()
                for agent_id, policy in self._policies.items()
            },
        }


# Convenience function for creating manager with config precedence
def create_context_manager(
    model: str = "gpt-4o-mini",
    session_id: str = "",
    agent_name: str = "",
    config_file: Optional[str] = None,
    cli_overrides: Optional[Dict[str, Any]] = None,
) -> ContextManager:
    """
    Create a context manager with proper config precedence.
    
    Precedence: CLI > ENV > config_file > defaults
    
    Args:
        model: Model name
        session_id: Session ID
        agent_name: Agent name
        config_file: Path to config.yaml
        cli_overrides: CLI argument overrides
        
    Returns:
        Configured ContextManager
    """
    # Start with defaults
    config = ManagerConfig()
    
    # Load from config file if provided
    if config_file:
        config = _load_config_from_file(config_file, config)
    
    # Apply environment variables
    env_config = ManagerConfig.from_env()
    config = config.merge(**{k: v for k, v in env_config.to_dict().items() if v is not None})
    
    # Apply CLI overrides
    if cli_overrides:
        config = config.merge(**cli_overrides)
        config.source = "cli"
    
    return ContextManager(
        model=model,
        config=config,
        session_id=session_id,
        agent_name=agent_name,
    )


def _load_config_from_file(path: str, base_config: ManagerConfig) -> ManagerConfig:
    """Load config from YAML file."""
    try:
        import yaml
        
        config_path = Path(path)
        if not config_path.exists():
            return base_config
        
        with open(config_path) as f:
            data = yaml.safe_load(f) or {}
        
        context_config = data.get("context", {})
        if not context_config:
            return base_config
        
        return base_config.merge(**context_config, source="config_file")
    except Exception:
        return base_config
