#!/usr/bin/env python3
"""
Demonstration of Issue #896 Enhancement: Callback Needs More Contextual Information

This script demonstrates how the enhanced callback system now provides agent_name, 
agent_role, agent_tools, task_name, task_description, and task_id to callbacks.

Usage:
    python test_issue_896_demo.py

Author: Generated by Claude Code
"""

import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'src', 'praisonai-agents'))

from praisonaiagents import register_display_callback, display_interaction

def enhanced_callback(message=None, response=None, agent_name=None, agent_role=None, 
                     agent_tools=None, task_name=None, task_description=None, 
                     task_id=None, **kwargs):
    """
    Enhanced callback function that demonstrates the new contextual information.
    
    This callback now receives:
    - agent_name: The name of the agent
    - agent_role: The role of the agent  
    - agent_tools: List of tools available to the agent
    - task_name: The name of the task being executed
    - task_description: The description of the task
    - task_id: The unique identifier of the task
    """
    print("="*80)
    print("🔔 ENHANCED CALLBACK TRIGGERED")
    print("="*80)
    print(f"📝 Message: {message}")
    print(f"💬 Response: {response}")
    print(f"🤖 Agent Name: {agent_name}")
    print(f"👤 Agent Role: {agent_role}")
    print(f"🛠️  Agent Tools: {agent_tools}")
    print(f"📋 Task Name: {task_name}")
    print(f"📄 Task Description: {task_description}")
    print(f"🔢 Task ID: {task_id}")
    print(f"⚙️  Other kwargs: {kwargs}")
    print("="*80)
    print()

def legacy_callback(message=None, response=None, **kwargs):
    """
    Legacy callback function that demonstrates backward compatibility.
    
    This callback works exactly as before, proving that existing callbacks
    continue to function without modification.
    """
    print("🔄 Legacy callback still works!")
    print(f"Message: {message}")
    print(f"Response: {response}")
    print(f"Other kwargs: {kwargs}")
    print()

def demo_enhanced_callbacks():
    """Demonstrate the enhanced callback functionality"""
    
    print("🚀 Demonstrating Enhanced Callbacks for Issue #896")
    print("="*60)
    print()
    
    # Register both enhanced and legacy callbacks
    register_display_callback('interaction', enhanced_callback, is_async=False)
    
    print("1. Testing enhanced callback with full context:")
    print("-" * 50)
    
    # Test with full context
    display_interaction(
        message="You need to do the following task: Say the number 1.\nExpected Output: 1.\nPlease provide only the final result of your work.",
        response="1",
        agent_name="MyAgent",
        agent_role="Assistant", 
        agent_tools=["calculator", "text_processor"],
        task_name="simple_task",
        task_description="Say the number 1",
        task_id="task_001",
        markdown=True,
        generation_time=1.3638858795166016
    )
    
    print("2. Testing backward compatibility (missing context):")
    print("-" * 50)
    
    # Test with minimal context (backward compatibility)
    display_interaction(
        message="Test message without context",
        response="Test response", 
        markdown=True,
        generation_time=0.5
    )
    
    print("3. Testing with partial context:")
    print("-" * 50)
    
    # Test with partial context
    display_interaction(
        message="Partial context test",
        response="Response with partial context",
        agent_name="PartialAgent",
        task_name="partial_task",
        markdown=True,
        generation_time=0.8
    )
    
    print("✅ All tests completed successfully!")
    print()
    print("🎉 The enhanced callback system is working correctly!")
    print("   - Callbacks now receive agent and task context")
    print("   - Backward compatibility is maintained")
    print("   - All parameters are optional with sensible defaults")

if __name__ == "__main__":
    demo_enhanced_callbacks()