# Copyright © 2023- Frello Technology Private Limited

import requests
from functools import partial
from pydantic import BaseModel
from typing import Any, List, Tuple, Union, Dict, Optional

from chainfury import (
    Secret,
    model_registry,
    exponential_backoff,
    Model,
    UnAuthException,
)
from chainfury.components.const import Env
from chainfury.types import Thread


class OpenaiGPTModel(Model):
    def __init__(self, id: Optional[str] = None):
        self._openai_model_id = id
        self.openai_api_key = Secret(Env.OPENAI_TOKEN(""))
        super().__init__(
            id="openai-chat",
            description="Use OpenAI chat models",
            usage=["usage", "total_tokens"],
        )

    def set_api_token(self, token: str) -> None:
        self.openai_api_key = Secret(token)

    def chat(
        self,
        chats: Thread,
        model: Optional[str] = None,
        token: Secret = Secret(""),
        temperature: float = 1.0,
        top_p: float = 1.0,
        n: int = 1,
        stop: Union[str, List[str]] = "",
        max_tokens: int = 1024,
        presence_penalty: float = 0.0,
        frequency_penalty: float = 0.0,
        logit_bias: dict = {},
        user: str = "",
        retry_count: int = 3,
        retry_delay: int = 1,
        **kwargs,
    ) -> Any:
        """
        Returns a JSON object containing the OpenAI's API chat response. See [OpenAI documentation](https://platform.openai.com/docs/api-reference/chat/create).

        Args:
            messages: A list of messages describing the conversation so far
            model: ID of the model to use. See [OpenAI documentation](https://platform.openai.com/docs/api-reference/chat/create).
            token (Secret): The OpenAI API key. Defaults to "" or the OPENAI_TOKEN environment variable.
            temperature: Optional. What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic. We generally recommend altering this or top_p but not both. Defaults to 1.
            top_p: Optional. An alternative to sampling with temperature, called nucleus sampling, where the model considers the results of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass are considered. We generally recommend altering this or temperature but not both. Defaults to 1.
            n: Optional. How many chat completion choices to generate for each input message. Defaults to 1.
            stop: Optional. Up to 4 sequences where the API will stop generating further tokens.
            max_tokens: Optional. The maximum number of tokens to generate in the chat completion. The total length of input tokens and generated tokens is limited by the model's context length. Defaults to infinity.
            presence_penalty: Optional. Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model's likelihood to talk about new topics. See more information about frequency and presence penalties. Defaults to 0.
            frequency_penalty: Optional. Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model's likelihood to repeat the same line verbatim. See more information about frequency and presence penalties. Defaults to 0.
            logit_bias: Optional. Modify the likelihood of specified tokens appearing in the completion. Accepts a json object that maps tokens (specified by their token ID in the tokenizer) to an associated bias value from -100 to 100. Mathematically, the bias is added to the logits generated by the model prior to sampling. The exact effect will vary per model, but values between -1 and 1 should decrease or increase likelihood of selection; values like -100 or 100 should result in a ban or exclusive selection of the relevant
            user: Optional. A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse. Defaults to None.
            retry_count: Optional. Number of times to retry the API call. Defaults to 3.
            retry_delay: Optional. Number of seconds to wait before retrying the API call. Defaults to 1.

        Returns:
            Any: The completion(s) generated by the API.
        """
        if not token and not self.openai_api_key.value:
            raise Exception(
                "OpenAI API key not found. Please set OPENAI_TOKEN environment variable or pass through function"
            )

        if isinstance(chats, Thread):
            messages = chats.to_dict()["chats"]
        else:
            messages = chats

        model = model or self._openai_model_id

        def _fn():
            r = requests.post(
                "https://api.openai.com/v1/chat/completions",
                headers={
                    "Content-Type": "application/json",
                    "Authorization": f"Bearer {token}",
                },
                json={
                    "model": model,
                    "messages": messages,
                    "max_tokens": max_tokens,
                    "temperature": temperature,
                    "top_p": top_p,
                    "n": n,
                    "stop": stop,
                    "presence_penalty": presence_penalty,
                    "frequency_penalty": frequency_penalty,
                    "logit_bias": logit_bias,
                    "user": user,
                    **kwargs,
                },
                timeout=(5, 30),
            )
            if r.status_code == 401:
                raise UnAuthException(r.text)
            if r.status_code != 200:
                raise Exception(
                    f"OpenAI API returned status code {r.status_code}: {r.text}"
                )
            return r.json()["choices"][0]["message"]["content"]

        return exponential_backoff(
            _fn, max_retries=retry_count, retry_delay=retry_delay
        )


openai_gpt_model = model_registry.register(OpenaiGPTModel())


def openai_embedding(
    model: str,
    input_strings: Union[str, List[str]],
    openai_api_key: Secret = Secret(""),
    user: str = "",
    *,
    retry_count: int = 3,
    retry_delay: int = 1,
) -> Any:
    """
    Returns a JSON object containing the OpenAI's embeddings API response

    Args:
        model: ID of the model to use. See the model endpoint compatibility table for details on which models work with the Embeddings API.
        input_strings: A list of strings to embed, encoded as a string, array of strings, array of tokens, or array of token arrays.
        openai_api_key (Secret): The OpenAI API key. Defaults to "" or the OPENAI_TOKEN environment variable.
        user: Optional. A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse. Defaults to None.
        retry_count: Optional. Number of times to retry the API call. Defaults to 3.
        retry_delay: Optional. Number of seconds to wait before retrying the API call. Defaults to 1.

    Returns:
        Any: The completion(s) generated by the API.
    """

    if not openai_api_key:
        openai_api_key = Secret(Env.OPENAI_TOKEN("")).value  # type: ignore
    if not openai_api_key:
        raise Exception(
            "OpenAI API key not found. Please set OPENAI_TOKEN environment variable or pass through function"
        )

    def _fn():
        r = requests.post(
            "https://api.openai.com/v1/embeddings",
            headers={
                "Content-Type": "application/json",
                "Authorization": f"Bearer {openai_api_key}",
            },
            json={
                "model": model,
                "input": input_strings,
                "user": user,
            },
        )
        if r.status_code == 401:
            raise UnAuthException(r.text)
        if r.status_code != 200:
            raise Exception(
                f"OpenAI API returned status code {r.status_code}: {r.text}"
            )
        return r.json()

    return exponential_backoff(_fn, max_retries=retry_count, retry_delay=retry_delay)


openai_embedding_model = model_registry.register(
    model=Model(
        id="openai-embedding",
        fn=openai_embedding,
        description="Given a list of messages create embeddings for each message.",
        usage=["usage", "total_tokens"],
    )
)
