import { Button, Tooltip } from "@openbb/ui-pro";
import { type SVGProps, forwardRef, memo } from "react";
import { twMerge } from "tailwind-merge";

const CustomIcon = forwardRef<
    SVGSVGElement,
    { id: string } & SVGProps<SVGSVGElement>
>((props, ref) => {
    const { id, className, ...rest } = props;
    const defaultClass = "w-4 h-4";
    
    // Special case for Jupyter logo which has a different viewBox
    const viewBox = id === "jupyter-logo" ? "0 0 256 300" : "0 0 24 24";
    
    return (
        <svg
            {...rest}
            ref={ref}
            className={twMerge(defaultClass, className)}
            aria-hidden="true"
            role="img"
            viewBox={viewBox}
            width="100%"
            height="100%"
        >
            <use href={`/assets/icons/sprite.svg#${id}`} />
        </svg>
    );
});

CustomIcon.displayName = "CustomIcon";

interface HelpIconProps {
    tooltip: string;
    className?: string;
}

const HelpIcon = ({ tooltip, className = "" }: HelpIconProps) => (
    <Tooltip
        content={tooltip}
        className="tooltip-theme"
    >
        <div
            className={twMerge(
                "info-icon cursor-help opacity-80 hover:opacity-100 transition-opacity duration-200 shadow-sm",
                className
            )}
            style={{
                width: '16px',
                height: '16px',
                borderRadius: '100%',
                backgroundColor: 'none',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                color: 'var(--info-icon)',
                fontSize: '10px',
                fontWeight: 'bold',
                border: '1.2px solid var(--info-icon)',
            }}
        >
            i
        </div>
    </Tooltip>
);

export const ODPLogo = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 249 9"
        width="249px"
        height="100%"
        xmlns="http://www.w3.org/2000/svg"
        fill="currentColor"
        className={twMerge("text-theme-primary", props.className)}
        {...props}
    >
        <path d="M6.58831 0.0126953H0V8.0127H7.53062V0.0126953H6.58943H6.58831ZM6.58831 1.51307V7.01284H0.940066V1.01255H6.58831V1.51307Z" />
        <path d="M20.5331 0.018648H13.5306V8.0127H14.531V5.01418H21.5335V0.0126953H20.5331V0.018648ZM20.5331 1.51791V4.01865H14.531V1.01776H20.5331V1.51791Z" />
        <path d="M34.4889 0.0438232H34.4865H33.9915H33.4954H32.9981H32.502H32.0046H31.5085H31.0112H30.5151H30.0177H29.5216H29.0243H28.5281H28.0308H27.5335V0.540518V1.03603V1.53154V2.02824V2.52493V3.02044V3.51714V4.01265V4.50935V5.00486V5.50155V5.99707V6.49376V6.99045V7.48597V7.98148H28.0308H28.5281H29.0243H29.5216H30.0177H30.5151H31.0112H31.5085H32.0046H32.502H32.9981H33.4954H33.9915H34.4865H34.4889H34.9838H35.4799V7.48597V6.99045H34.9838H34.4889H34.4865H33.9915H33.4954H32.9981H32.502H32.0046H31.5085H31.0112H30.5151H30.0177H29.5216H29.0243H28.5281V6.49376V5.99707V5.50155V5.00486H29.0243H29.5216H30.0177H30.5151H31.0112H31.5085H32.0046H32.502H32.9957H32.9981H33.493H33.9892H34.4865V4.50935V4.01265H33.9892H33.493H32.9981H32.9957H32.502H32.0046H31.5085H31.0112H30.5151H30.0177H29.5216H29.0243H28.5281V3.51714V3.02044V2.52493V2.02824V1.53154V1.03603H29.0243H29.5216H30.0177H30.5151H31.0112H31.5085H32.0046H32.502H32.9981H33.4954H33.9915H34.4865H34.4889H34.9838H35.4799V0.540518V0.0438232H34.9838H34.4889Z" />
        <path d="M48.469 0.0246582V0.52376V1.02167V1.51959V2.01869V2.51779V3.01571V3.51481V4.01272V4.51183V5.00974V5.50884V6.00676L47.9693 5.50884L47.4708 5.00974L46.971 4.51183L46.4725 4.01272L45.9728 3.51481L45.473 3.01571L44.9745 2.51779L44.4748 2.01869L43.9762 1.51959L43.4765 1.02167L42.9779 0.52376L42.4782 0.0246582H41.9797H41.4799V0.52376V1.02167V1.51959V2.01869V2.51779V3.01571V3.51481V4.01272V4.51183V5.00974V5.50884V6.00676V6.50586V7.00496V7.50288V8.00079H41.9797H42.4782V7.50288V7.00496V6.50586V6.00676V5.50884V5.00974V4.51183V4.01272V3.51481V3.01571V2.51779V2.01869V1.51959L42.9779 2.01869L43.4765 2.51779L43.9762 3.01571L44.4748 3.51481L44.9745 4.01272L45.473 4.51183L45.9728 5.00974L46.4725 5.50884L46.971 6.00676L47.4708 6.50586L47.9693 7.00496L48.469 7.50288L48.9676 8.00079H49.4673V7.50288V7.00496V6.50586V6.00676V5.50884V5.00974V4.51183V4.01272V3.51481V3.01571V2.51779V2.01869V1.51959V1.02167V0.52376V0.0246582H48.9676H48.469Z" />
        <path d="M76.2938 0.0126953H70.1772V8.01268H76.2926L76.3183 8.0127L77.3103 7.08677V0.6053L76.2938 0.0126953ZM76.2926 1.51307V7.01283H71.1925V1.01255H76.2926V1.51307Z" />
        <path d="M90.3128 0.018648H83.3103V8.0127H84.3107V5.01418H90.3128V8.0127H91.3132V0.0126953H90.3128V0.018648ZM90.3128 4.01865H84.3107V1.01776H90.3128V4.01865Z" />
        <path d="M105.313 1.0293H101.822V8.0127H100.804V1.0293H97.3132V0.0126953H105.313V1.0293Z" />
        <path d="M118.316 0.018648H111.313V8.0127H112.314V5.01418H118.316V8.0127H119.316V0.0126953H118.316V0.018648ZM118.316 4.01865H112.314V1.01776H118.316V4.01865Z" />
        <path d="M147.029 0.00692926H140.026V8.00098H141.026V5.00247H148.029V0.000976562H147.029V0.00692926ZM147.029 1.50619V4.00693H141.026V1.00604H147.029V1.50619Z" />
        <path d="M154.538 0.0253906H155.555V7.14403H161.657V8.02539H154.538V0.0253906Z" />
        <path d="M175.091 0.00692926H168.089V8.00098H169.089V5.00247H175.091V8.00098H176.091V0.000976562H175.091V0.00692926ZM175.091 4.00693H169.089V1.00604H175.091V4.00693Z" />
        <path d="M190.601 1.04199H187.11V8.02539H186.092V1.04199H182.601V0.0253906H190.601V1.04199Z" />
        <path d="M204.093 0.0253906H204.096H204.594H205.094V0.525986V1.02539H204.594H204.096H204.093H203.594H203.094H202.593H202.093H201.592H201.092H200.591H200.091H199.589H199.089H198.588H198.088V1.52479V2.02539V2.52599V3.02539V3.52599V4.02539H198.588H199.089H199.589H200.091H200.591H201.092H201.592H202.093H202.591H202.593H203.092H203.592H204.594V5.02539H203.592H203.092H202.593H202.591H202.093H201.592H201.092H200.591H200.091H199.589H199.089H198.588H198.088V5.52599V6.02539V6.52599V7.02658V7.64606V8.02539H197.587H197.086V7.52599V7.02658V6.52599V6.02539V5.52599V5.02539V4.52599V4.02539V3.52599V3.02539V2.52599V2.02539V1.52479V1.02539V0.525986V0.0253906H197.587H198.088H198.588H199.089H199.589H200.091H200.591H201.092H201.592H202.093H202.593H203.094H203.594H204.093Z" />
        <path d="M218.587 0.00732422H211.58V4.00732V8.00732H219.59V0.00732422H218.589H218.587ZM218.587 1.5077V7.00747H212.58V1.00718H218.587V1.5077Z" />
        <path d="M233.097 8L231.269 4.85241H232.269L234.097 8H233.097Z" />
        <path d="M233.095 0.00595197H226.093V7.99902H227.094V5.00087H234.095V0H233.095V0.00595197ZM233.095 4.00546H227.094V1.00494H233.095V4.00546Z" />
        <path d="M240.607 8.02258V0.022583H241.624L244.843 3.27575L247.588 0.022583H248.604V8.02258H247.588V1.58139L245.351 4.29236H244.335L241.624 1.58139V8.02258H240.607Z" />
    </svg>
);

export const OpenBBLogo = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 255 157"
        fill="currentColor"
        xmlns="http://www.w3.org/2000/svg"
        {...props}
        className={twMerge("text-theme-primary", props.className)}
        aria-label="OpenBB Logo"
        role="img"
    >
        <path d="M151.404 103.189V95.2524H223.137V111.126H151.404V103.189ZM151.404 39.6893V31.7524H239.064V47.6262H151.404V39.6893ZM247.032 15.8738H135.468V127H239.064V79.3738H151.404V63.5H255V15.8738H247.032ZM31.8726 103.189V95.2524H103.605V111.126H31.8726V103.189ZM15.9363 39.6893V31.7524H103.596V47.6262H15.9363V39.6893ZM119.532 0V15.8738H0V63.5H103.596V79.3738H15.9363V127H119.532V15.8738H135.468V0H119.532Z" />

    </svg>
);


export const FileIcon = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 16 16"
        fill="none"
        stroke="currentColor"
        role="icon"
        strokeWidth={0.8}
        strokeLinecap="round"
        strokeLinejoin="round"
        className={props.className}
        {...props}
    >
        <path d="M9.33341 1.51294V4.26663C9.33341 4.64 9.33341 4.82669 9.40608 4.96929C9.46999 5.09473 9.57198 5.19672 9.69742 5.26064C9.84003 5.3333 10.0267 5.3333 10.4001 5.3333H13.1538M10.6667 8.66659H5.33341M10.6667 11.3333H5.33341M6.66675 5.99992H5.33341M9.33341 1.33325H5.86675C4.74664 1.33325 4.18659 1.33325 3.75877 1.55124C3.38244 1.74299 3.07648 2.04895 2.88473 2.42527C2.66675 2.85309 2.66675 3.41315 2.66675 4.53325V11.4666C2.66675 12.5867 2.66675 13.1467 2.88473 13.5746C3.07648 13.9509 3.38244 14.2569 3.75877 14.4486C4.18659 14.6666 4.74664 14.6666 5.86675 14.6666H10.1334C11.2535 14.6666 11.8136 14.6666 12.2414 14.4486C12.6177 14.2569 12.9237 13.9509 13.1154 13.5746C13.3334 13.1467 13.3334 12.5867 13.3334 11.4666V5.33325L9.33341 1.33325Z" />

    </svg>
);

export const FolderIcon = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 16 16"
        fill="none"
        stroke="currentColor"
        role="icon"
        strokeWidth={1.2}
        strokeLinecap="round"
        strokeLinejoin="round"
        className={props.className}
        {...props}
    >
        <path d="M7.58333 4.08333L6.9326 2.78187C6.74532 2.4073 6.65167 2.22001 6.51196 2.08318C6.38842 1.96218 6.23952 1.87015 6.07604 1.81376C5.89118 1.75 5.68178 1.75 5.263 1.75H3.03333C2.37994 1.75 2.05324 1.75 1.80368 1.87716C1.58415 1.98901 1.40568 2.16749 1.29382 2.38701C1.16666 2.63657 1.16666 2.96327 1.16666 3.61667V4.08333M1.16666 4.08333H10.0333C11.0134 4.08333 11.5035 4.08333 11.8778 4.27407C12.2071 4.44185 12.4748 4.70957 12.6426 5.03885C12.8333 5.4132 12.8333 5.90324 12.8333 6.88333V9.45C12.8333 10.4301 12.8333 10.9201 12.6426 11.2945C12.4748 11.6238 12.2071 11.8915 11.8778 12.0593C11.5035 12.25 11.0134 12.25 10.0333 12.25H3.96666C2.98657 12.25 2.49653 12.25 2.12218 12.0593C1.7929 11.8915 1.52518 11.6238 1.3574 11.2945C1.16666 10.9201 1.16666 10.4301 1.16666 9.45V4.08333Z" />

    </svg>
);

export const DocumentationIcon = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 16 16"
        fill="none"
        stroke="currentColor"
        role="icon"
        strokeWidth={0.8}
        strokeLinecap="round"
        strokeLinejoin="round"
        className={props.className}
        {...props}
    >
        <path d="M7.99992 14L7.93322 13.8999C7.47012 13.2053 7.23857 12.858 6.93265 12.6065C6.66182 12.3839 6.34976 12.2169 6.01432 12.1151C5.63542 12 5.21799 12 4.38313 12H3.46659C2.71985 12 2.34648 12 2.06126 11.8547C1.81038 11.7268 1.60641 11.5229 1.47858 11.272C1.33325 10.9868 1.33325 10.6134 1.33325 9.86667V4.13333C1.33325 3.3866 1.33325 3.01323 1.47858 2.72801C1.60641 2.47713 1.81038 2.27316 2.06126 2.14532C2.34648 2 2.71985 2 3.46659 2H3.73325C5.22673 2 5.97346 2 6.54389 2.29065C7.04566 2.54631 7.45361 2.95426 7.70927 3.45603C7.99992 4.02646 7.99992 4.77319 7.99992 6.26667M7.99992 14V6.26667M7.99992 14L8.06662 13.8999C8.52972 13.2053 8.76126 12.858 9.06718 12.6065C9.33801 12.3839 9.65008 12.2169 9.98552 12.1151C10.3644 12 10.7818 12 11.6167 12H12.5333C13.28 12 13.6534 12 13.9386 11.8547C14.1895 11.7268 14.3934 11.5229 14.5213 11.272C14.6666 10.9868 14.6666 10.6134 14.6666 9.86667V4.13333C14.6666 3.3866 14.6666 3.01323 14.5213 2.72801C14.3934 2.47713 14.1895 2.27316 13.9386 2.14532C13.6534 2 13.28 2 12.5333 2H12.2666C10.7731 2 10.0264 2 9.45594 2.29065C8.95418 2.54631 8.54623 2.95426 8.29057 3.45603C7.99992 4.02646 7.99992 4.77319 7.99992 6.26667" />
    </svg>
);

export const CopyIcon = (props: React.ComponentProps<"svg">) => (
    <svg
        viewBox="0 0 14 14"
        fill="none"
        stroke="currentColor"
        strokeWidth={1.2}
        strokeLinecap="round"
        strokeLinejoin="round"
        className={twMerge("text-theme-muted hover:text-theme-accent focus:text-theme-accent", props.className)}
        {...props}
    >
        {/* Main copy square */}
        <path d="M4.66663 4.66663V3.03329C4.66663 2.3799 4.66663 2.0532 4.79378 1.80364C4.90564 1.58411 5.08411 1.40564 5.30364 1.29379C5.5532 1.16663 5.8799 1.16663 6.53329 1.16663H10.9666C11.62 1.16663 11.9467 1.16663 12.1963 1.29379C12.4158 1.40564 12.5943 1.58411 12.7061 1.80364C12.8333 2.0532 12.8333 2.3799 12.8333 3.03329V7.46663C12.8333 8.12002 12.8333 8.44672 12.7061 8.69628C12.5943 8.9158 12.4158 9.09428 12.1963 9.20613C11.9467 9.33329 11.62 9.33329 10.9666 9.33329H9.33329M3.03329 12.8333H7.46663C8.12002 12.8333 8.44672 12.8333 8.69628 12.7061C8.9158 12.5943 9.09428 12.4158 9.20613 12.1963C9.33329 11.9467 9.33329 11.62 9.33329 10.9666V6.53329C9.33329 5.8799 9.33329 5.5532 9.20613 5.30364C9.09428 5.08411 8.9158 4.90564 8.69628 4.79378C8.44672 4.66663 8.12002 4.66663 7.46663 4.66663H3.03329C2.3799 4.66663 2.0532 4.66663 1.80364 4.79378C1.58411 4.90564 1.40564 5.08411 1.29379 5.30364C1.16663 5.5532 1.16663 5.8799 1.16663 6.53329V10.9666C1.16663 11.62 1.16663 11.9467 1.29379 12.1963C1.40564 12.4158 1.58411 12.5943 1.80364 12.7061C2.0532 12.8333 2.3799 12.8333 3.03329 12.8333Z"/>
    </svg>
);


interface ThemeToggleButtonProps {
    isDarkMode: boolean;
    toggleTheme: () => void;
    className?: string;
    style?: React.CSSProperties;
}

export const ThemeToggleButton = ({
    isDarkMode,
    toggleTheme,
    style,
}: ThemeToggleButtonProps) => (
    <Tooltip
        content={isDarkMode ? "Switch to light mode" : "Switch to dark mode"}
        className="tooltip tooltip-theme"
    >
        <Button
            onClick={toggleTheme}
            className={twMerge(
                "button-ghost"
            )}
            style={style}
            aria-label={isDarkMode ? "Switch to light mode" : "Switch to dark mode"}
            variant="ghost"
            size="icon"
        >
            {isDarkMode ? (
                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 19 19" fill="currentColor">
                    <path fillRule="evenodd" d="M10 2a1 1 0 011 1v1a1 1 0 11-2 0V3a1 1 0 011-1zm4 8a4 4 0 11-8 0 4 4 0 018 0zm-.464 4.95l.707.707a1 1 0 001.414-1.414l-.707-.707a1 1 0 00-1.414 1.414zm2.12-10.607a1 1 0 010 1.414l-.706.707a1 1 0 11-1.414-1.414l.707-.707a1 1 0 011.414 0zM17 11a1 1 0 100-2h-1a1 1 0 100 2h1zm-7 4a1 1 0 011 1v1a1 1 0 11-2 0v-1a1 1 0 011-1zM5.05 6.464A1 1 0 106.465 5.05l-.708-.707a1 1 0 00-1.414 1.414l.707.707zm1.414 8.486l-.707.707a1 1 0 01-1.414-1.414l.707-.707a1 1 0 011.414 1.414zM4 11a1 1 0 100-2H3a1 1 0 000 2h1z" clipRule="evenodd" />
                </svg>
            ) : (
                <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                    <path d="M17.293 13.293A8 8 0 016.707 2.707a8.001 8.001 0 1010.586 10.586z" />
                </svg>
            )}
        </Button>
    </Tooltip>
);

export const SettingsIcon = ({
  className = "",
  ...props
}: React.ComponentProps<"svg">) => (
  <svg
    viewBox="0 0 24 24"
    fill="none"
    xmlns="http://www.w3.org/2000/svg"
    stroke="currentColor"
    strokeWidth={1.25}
    strokeLinecap="round"
    strokeLinejoin="round"
    className={className}
    aria-hidden="true"
    {...props}
  >
    <g clipPath="url(#clip0_3_2291)">
    <path d="M12 13.7501C12.9665 13.7501 13.75 12.9666 13.75 12.0001C13.75 11.0336 12.9665 10.2501 12 10.2501C11.0335 10.2501 10.25 11.0336 10.25 12.0001C10.25 12.9666 11.0335 13.7501 12 13.7501Z" />
    <path d="M15.9242 13.591C15.8536 13.7509 15.8326 13.9284 15.8638 14.1004C15.895 14.2724 15.977 14.4312 16.0992 14.5561L16.1311 14.588C16.2297 14.6865 16.3079 14.8034 16.3613 14.9322C16.4146 15.0609 16.4421 15.199 16.4421 15.3383C16.4421 15.4777 16.4146 15.6157 16.3613 15.7445C16.3079 15.8732 16.2297 15.9902 16.1311 16.0887C16.0325 16.1873 15.9156 16.2656 15.7868 16.3189C15.6581 16.3723 15.5201 16.3998 15.3807 16.3998C15.2413 16.3998 15.1033 16.3723 14.9745 16.3189C14.8458 16.2656 14.7288 16.1873 14.6303 16.0887L14.5985 16.0569C14.4735 15.9346 14.3148 15.8526 14.1427 15.8214C13.9707 15.7903 13.7933 15.8113 13.6333 15.8819C13.4765 15.9491 13.3427 16.0607 13.2485 16.203C13.1543 16.3453 13.1037 16.512 13.103 16.6827V16.7728C13.103 17.0541 12.9913 17.3239 12.7924 17.5228C12.5935 17.7217 12.3237 17.8334 12.0424 17.8334C11.7611 17.8334 11.4914 17.7217 11.2925 17.5228C11.0936 17.3239 10.9818 17.0541 10.9818 16.7728V16.7251C10.9777 16.5496 10.9209 16.3793 10.8187 16.2365C10.7166 16.0937 10.5739 15.9849 10.4091 15.9243C10.2491 15.8537 10.0717 15.8327 9.89968 15.8639C9.72765 15.8951 9.56891 15.9771 9.44393 16.0993L9.41211 16.1311C9.31361 16.2298 9.19664 16.308 9.06788 16.3614C8.93913 16.4147 8.80111 16.4422 8.66173 16.4422C8.52235 16.4422 8.38434 16.4147 8.25558 16.3614C8.12683 16.308 8.00986 16.2298 7.91135 16.1311C7.81274 16.0326 7.73451 15.9157 7.68114 15.7869C7.62776 15.6582 7.60029 15.5201 7.60029 15.3808C7.60029 15.2414 7.62776 15.1034 7.68114 14.9746C7.73451 14.8459 7.81274 14.7289 7.91135 14.6304L7.94317 14.5986C8.06543 14.4736 8.14744 14.3148 8.17863 14.1428C8.20982 13.9708 8.18876 13.7934 8.11817 13.6334C8.05095 13.4766 7.93933 13.3428 7.79705 13.2486C7.65478 13.1544 7.48806 13.1038 7.31741 13.1031H7.22726C6.94597 13.1031 6.6762 12.9914 6.4773 12.7925C6.2784 12.5936 6.16666 12.3238 6.16666 12.0425C6.16666 11.7612 6.2784 11.4914 6.4773 11.2925C6.6762 11.0936 6.94597 10.9819 7.22726 10.9819H7.27499C7.45052 10.9778 7.62075 10.921 7.76356 10.8188C7.90637 10.7167 8.01514 10.574 8.07575 10.4092C8.14634 10.2492 8.1674 10.0718 8.1362 9.89977C8.10501 9.72774 8.023 9.569 7.90075 9.44402L7.86893 9.4122C7.77032 9.3137 7.69209 9.19673 7.63871 9.06797C7.58534 8.93922 7.55787 8.8012 7.55787 8.66182C7.55787 8.52244 7.58534 8.38443 7.63871 8.25567C7.69209 8.12692 7.77032 8.00995 7.86893 7.91144C7.96743 7.81283 8.0844 7.7346 8.21316 7.68123C8.34191 7.62786 8.47993 7.60038 8.61931 7.60038C8.75869 7.60038 8.8967 7.62786 9.02546 7.68123C9.15421 7.7346 9.27118 7.81283 9.36969 7.91144L9.4015 7.94326C9.52649 8.06552 9.68523 8.14753 9.85725 8.17872C10.0293 8.20991 10.2067 8.18885 10.3667 8.11826H10.4091C10.5659 8.05104 10.6997 7.93942 10.7939 7.79715C10.8881 7.65487 10.9387 7.48815 10.9394 7.31751V7.22735C10.9394 6.94606 11.0511 6.67629 11.25 6.47739C11.4489 6.27849 11.7187 6.16675 12 6.16675C12.2813 6.16675 12.551 6.27849 12.75 6.47739C12.9489 6.67629 13.0606 6.94606 13.0606 7.22735V7.27508C13.0613 7.44573 13.1118 7.61245 13.2061 7.75472C13.3003 7.897 13.4341 8.00862 13.5909 8.07584C13.7508 8.14643 13.9283 8.16749 14.1003 8.1363C14.2723 8.1051 14.4311 8.02309 14.5561 7.90084L14.5879 7.86902C14.6864 7.77041 14.8033 7.69218 14.9321 7.63881C15.0609 7.58543 15.1989 7.55796 15.3382 7.55796C15.4776 7.55796 15.6156 7.58543 15.7444 7.63881C15.8732 7.69218 15.9901 7.77041 16.0886 7.86902C16.1872 7.96752 16.2655 8.0845 16.3188 8.21325C16.3722 8.34201 16.3997 8.48002 16.3997 8.6194C16.3997 8.75878 16.3722 8.89679 16.3188 9.02555C16.2655 9.1543 16.1872 9.27128 16.0886 9.36978L16.0568 9.4016C15.9346 9.52658 15.8525 9.68532 15.8214 9.85734C15.7902 10.0294 15.8112 10.2068 15.8818 10.3667V10.4092C15.949 10.566 16.0606 10.6998 16.2029 10.794C16.3452 10.8882 16.5119 10.9388 16.6826 10.9395H16.7727C17.054 10.9395 17.3238 11.0512 17.5227 11.2501C17.7216 11.449 17.8333 11.7188 17.8333 12.0001C17.8333 12.2814 17.7216 12.5511 17.5227 12.75C17.3238 12.9489 17.054 13.0607 16.7727 13.0607H16.725C16.5543 13.0614 16.3876 13.1119 16.2453 13.2062C16.1031 13.3004 15.9915 13.4341 15.9242 13.591Z" />
    </g>
    <defs>
    <clipPath id="clip0_3_2291">
    <rect width="14" height="14" fill="currentColor" transform="translate(5 5)"/>
    </clipPath>
    </defs>
  </svg>
);

export const ChevronIcon = (props: React.ComponentProps<"svg">) => (
	<svg
		xmlns="http://www.w3.org/2000/svg"
		viewBox="0 0 20 20"
		fill="currentColor"
		{...props}
	>
		<path
			fillRule="evenodd"
			d="M7.21 14.77a.75.75 0 01.02-1.06L11.168 10 7.23 6.29a.75.75 0 111.04-1.08l4.5 4.25a.75.75 0 010 1.08l-4.5 4.25a.75.75 0 01-1.06-.02z"
			clipRule="evenodd"
		/>
	</svg>
);

export const RefreshIcon = (props: React.ComponentProps<"svg">) => (
	<svg
		xmlns="http://www.w3.org/2000/svg"
		viewBox="0 0 16 16"
		fill="none"
        stroke="currentColor"
        strokeLinecap="round"
        strokeLinejoin="round"
		{...props}
	>
    <path d="M13.6354 8.59536C13.4502 10.3353 12.4643 11.9658 10.833 12.9076C8.12269 14.4724 4.65701 13.5438 3.0922 10.8335L2.92554 10.5448M2.36417 7.40467C2.54937 5.66474 3.53523 4.03426 5.16655 3.09241C7.87688 1.5276 11.3426 2.45623 12.9074 5.16655L13.074 5.45523M2.32886 12.044L2.81689 10.2227L4.63826 10.7107M11.3617 5.28934L13.183 5.77737L13.6711 3.95601" />
    </svg>
);

export default memo(CustomIcon);

export { HelpIcon, type HelpIconProps };
