"""Report Codes and Metadata for USDA FAS Commodity Production Supply & Distribution Reports."""

# pylint: disable=too-many-lines

from typing import Literal

PsdReportNames = Literal[
    # === COFFEE ===
    "coffee_summary",
    "coffee_summary_2",
    "coffee_summary_3",
    "coffee_summary_4",
    "coffee_production",
    "coffee_arabica_production",
    "coffee_robusta_production",
    "coffee_consumption",
    "coffee_ending_stocks",
    "coffee_exports_total",
    "coffee_exports_green_bean",
    "coffee_exports_soluble",
    "coffee_imports_total",
    "coffee_imports_green_bean",
    "coffee_imports_soluble",
    # === COTTON ===
    "cotton_monthly_changes",
    "cotton_world_supply_use",
    "cotton_world_supply_use_2",
    "cotton_area_yield_production",
    "cotton_supply_distribution",
    "cotton_supply_distribution_2",
    "cotton_by_country",
    "cotton_by_country_2",
    "cotton_world_supply",
    "cotton_us_supply",
    "cotton_foreign_supply",
    # === DAIRY ===
    "butter_production_consumption",
    "butter_trade",
    "cheese_production_consumption",
    "cheese_trade",
    "milk_production_consumption",
    "milk_cow_numbers",
    "nonfat_dry_milk_production_consumption",
    "nonfat_dry_milk_trade",
    "whole_milk_powder_production_consumption",
    "whole_milk_powder_trade",
    # === FIELD CROPS (fcr) ===
    "world_crop_production_summary",
    "wheat_area_yield_production",
    "coarse_grains_area_yield_production",
    "corn_area_yield_production",
    "barley_area_yield_production",
    "oats_area_yield_production",
    "rye_area_yield_production",
    "sorghum_area_yield_production",
    "rice_area_yield_production",
    "oilseeds_area_yield_production",
    "soybeans_area_yield_production",
    "cottonseed_area_yield_production",
    "peanut_area_yield_production",
    "sunflower_area_yield_production",
    "rapeseed_area_yield_production",
    "copra_palm_kernel_palm_oil_production",
    "cotton_area_yield_production_fcr",
    # === FRUITS & VEGETABLES (htp) ===
    "apples_supply_distribution",
    "apples_selected_countries",
    "cherries_supply_distribution",
    "cherries_selected_countries",
    "grapefruit_selected_countries",
    "grapes_supply_distribution",
    "grapes_selected_countries",
    "lemons_limes_selected_countries",
    "oranges_selected_countries",
    "oranges_selected_countries_2",
    "peaches_nectarines_supply_distribution",
    "peaches_nectarines_selected_countries",
    "pears_supply_distribution",
    "pears_selected_countries",
    "raisins_supply_distribution",
    "raisins_selected_countries",
    "tangerines_mandarins_selected_countries",
    # === GRAINS (gra) ===
    "grains_summary_comparison",
    "barley_supply_disappearance",
    "china_grain_supply_demand",
    "corn_supply_disappearance",
    "eu_grain_supply_demand",
    "russia_grain_supply_demand",
    "russia_barley",
    "russia_corn",
    "russia_wheat",
    "other_europe_grain_supply_demand",
    "barley_regional",
    "coarse_grains_regional",
    "corn_regional",
    "oats_regional",
    "rice_regional",
    "rye_regional",
    "sorghum_regional",
    "wheat_regional",
    "sorghum_supply_disappearance",
    "wheat_coarse_grains_supply_demand",
    "us_grains_supply_distribution",
    "wheat_supply_disappearance",
    "barley_world_production_consumption_stocks",
    "barley_world_trade",
    "coarse_grains_world_trade",
    "coarse_grains_world_production_consumption_stocks",
    "corn_barley_supply_demand",
    "corn_world_production_consumption_stocks",
    "corn_world_trade",
    "oats_world_production_consumption_stocks",
    "oats_world_trade",
    "rice_world_production_consumption_stocks",
    "rice_world_trade",
    "rice_supply_demand",
    "rye_world_production_consumption_stocks",
    "rye_world_trade",
    "sorghum_world_production_consumption_stocks",
    "sorghum_world_trade",
    "wheat_coarse_grains_world_supply_demand",
    "wheat_world_production_consumption_stocks",
    "wheat_flour_products_world_trade",
    # === JUICE ===
    "orange_juice_supply_distribution",
    # === LIVESTOCK ===
    "beef_veal_production",
    "beef_veal_trade",
    "cattle_stocks",
    "cattle_trade",
    "chicken_production",
    "chicken_trade",
    "pork_production",
    "pork_trade",
    "swine_stocks",
    "swine_trade",
    # === OILSEEDS (oil) ===
    "oilseeds_world_commodity_view",
    "protein_meals_world_commodity_view",
    "vegetable_oils_world_commodity_view",
    "oilseeds_world_country_view",
    "protein_meals_world_country_view",
    "vegetable_oils_world_country_view",
    "soybeans_world_supply",
    "soybean_meal_world_supply",
    "soybean_oil_world_supply",
    "soybeans_products_world_trade",
    "palm_oil_world_supply",
    "rapeseed_products_world_supply",
    "sunflower_products_world_supply",
    "vegetable_oils_minor_world_supply",
    "oilseeds_products_world_supply_demand",
    "soybeans_products_world_supply_demand",
    "rapeseed_products_world_supply_demand",
    "sunflower_products_world_supply_demand",
    "palm_coconut_fishmeal_world_supply_demand",
    "oilseeds_us_supply_distribution",
    "soybeans_us_supply_distribution",
    "soybeans_brazil_supply_distribution",
    "soybeans_argentina_supply_distribution",
    "oilseeds_southeast_asia",
    "oilseeds_middle_east",
    "oilseeds_eu",
    "oilseeds_china",
    "oilseeds_india",
    # === SUGAR ===
    "sugar_ending_stocks",
    "sugar_imports_exports",
    "sugar_production_consumption",
    # === TREE NUTS ===
    "almonds_supply_distribution",
    "almonds_summary",
    "pistachios_summary",
    "pistachios_supply_distribution",
    "walnuts_supply_distribution",
    "walnuts_summary",
]

PSD_REPORTS_METADATA = {
    "cof": {
        "name": "Coffee",
        "reports": {
            2109: {"title": "Coffee Summary", "templateId": 8},
            2535: {"title": "Coffee Summary, Continued", "templateId": 8},
            2619: {"title": "Coffee Summary, Continued", "templateId": 8},
            2620: {"title": "Coffee Summary, Continued", "templateId": 8},
            1559: {"title": "Table 03A Coffee Production", "templateId": 8},
            1673: {"title": "Table 03B Arabica Coffee Production", "templateId": 8},
            1679: {"title": "Table 03C Robusta Coffee Production", "templateId": 8},
            2245: {"title": "Table 04 Coffee Consumption", "templateId": 8},
            1783: {
                "title": "Table 05 Green Coffee Bean Ending Stocks",
                "templateId": 8,
            },
            2053: {"title": "Table 06A Total Coffee Exports", "templateId": 8},
            1670: {"title": "Table 06B Green Coffee Bean Exports", "templateId": 8},
            2052: {"title": "Table 06C Soluble Coffee Exports", "templateId": 8},
            2055: {"title": "Table 07A Total Coffee Imports", "templateId": 8},
            1774: {"title": "Table 07B Green Coffee Bean Imports", "templateId": 8},
            2056: {"title": "Table 07C Soluble Coffee Imports", "templateId": 8},
        },
    },
    "cot": {
        "name": "Cotton",
        "reports": {
            837: {
                "title": "Summary of Changes in estimates and Forecasts from Last Month",
                "templateId": 17,
            },
            849: {
                "title": "Table 01 Cotton World Supply, Use, and Trade (Season Beginning August 1)",
                "templateId": 8,
            },
            850: {
                "title": "Table 02 Cotton World Supply, Use, and Trade (Season Beginning August 1)",
                "templateId": 8,
            },
            851: {
                "title": "Table 04 Cotton Area, Yield, and Production",
                "templateId": 1,
            },
            852: {"title": "Table 05 Cotton Supply and Distribution", "templateId": 3},
            853: {"title": "Table 05A Cotton Supply and Distribution", "templateId": 3},
            854: {
                "title": "Table 06: Cotton Supply and Distribution by Country",
                "templateId": 3,
            },
            855: {
                "title": "Table 06A: Cotton Supply and Distribution by Country",
                "templateId": 3,
            },
            856: {
                "title": "Table 09:  World Cotton Supply and Distribution",
                "templateId": 3,
            },
            857: {
                "title": "Table 10:  United States Cotton Supply and Distribution",
                "templateId": 3,
            },
            858: {
                "title": "Table 11:  Foreign Cotton Supply and Distribution",
                "templateId": 3,
            },
        },
    },
    "dai": {
        "name": "Dairy",
        "reports": {
            2538: {
                "title": "Butter Production and Consumption:  Summary For Selected Countries",
                "templateId": 7,
            },
            2539: {
                "title": "Butter Trade:  Summary For Selected Countries",
                "templateId": 7,
            },
            2540: {
                "title": "Cheese Production and Consumption: Summary For Selected Countries",
                "templateId": 7,
            },
            2542: {
                "title": "Cheese Trade: Summary For Selected Countries",
                "templateId": 7,
            },
            2544: {
                "title": "Cows Milk Production and Consumption:  Summary For Selected Countries",
                "templateId": 7,
            },
            2543: {
                "title": "Fluid Milk - Cow Numbers: Summary For Selected Countries",
                "templateId": 7,
            },
            2548: {
                "title": "Nonfat Dry Milk  Production and Consumption:  Summary For Selected Countries",
                "templateId": 7,
            },
            2545: {
                "title": "Nonfat Dry Milk Trade: Summary For Selected Countries",
                "templateId": 7,
            },
            2549: {
                "title": "Whole Milk Powder Production And Consumption:  Summary For Selected Countries",
                "templateId": 7,
            },
            2547: {
                "title": "Whole Milk Powder Trade:  Summary For Selected Countries",
                "templateId": 7,
            },
        },
    },
    "fcr": {
        "name": "Field Crops - Production",
        "reports": {
            425: {"title": "Table 01 World Crop Production Summary", "templateId": 2},
            448: {
                "title": "Table 02 Wheat Area, Yield, and Production",
                "templateId": 1,
            },
            453: {
                "title": "Table 03 Total Coarse Grain Area, Yield, and Production",
                "templateId": 1,
            },
            884: {
                "title": "Table 04 Corn Area, Yield, and Production",
                "templateId": 1,
            },
            885: {
                "title": "Table 05 Barley Area, Yield, and Production",
                "templateId": 1,
            },
            890: {
                "title": "Table 06 Oats Area, Yield, and Production",
                "templateId": 1,
            },
            891: {"title": "Table 07 Rye Area, Yield, and Production", "templateId": 1},
            892: {
                "title": "Table 08 Sorghum Area, Yield, and Production",
                "templateId": 1,
            },
            893: {
                "title": "Table 09 Rice Area, Yield, and Production",
                "templateId": 1,
            },
            2403: {
                "title": "Table 10 Total Oilseed Area, Yield, and Production",
                "templateId": 20,
            },
            906: {
                "title": "Table 11 Soybean Area, Yield, and Production",
                "templateId": 1,
            },
            914: {
                "title": "Table 12 Cottonseed Area, Yield, and Production",
                "templateId": 1,
            },
            918: {
                "title": "Table 13 Peanut Area, Yield, and Production",
                "templateId": 1,
            },
            917: {
                "title": "Table 14 Sunflowerseed Area, Yield, and Production",
                "templateId": 1,
            },
            922: {
                "title": "Table 15 Rapeseed Area, Yield, and Production",
                "templateId": 1,
            },
            454: {
                "title": "Table 16 Copra, Palm Kernel, and Palm Oil Production",
                "templateId": 9,
            },
            958: {
                "title": "Table 17 Cotton Area, Yield, and Production",
                "templateId": 1,
            },
        },
    },
    "htp": {
        "name": "Fruits and Vegetables",
        "reports": {
            2418: {
                "title": "Apples, Fresh:  Production, Supply and Distribution",
                "templateId": 3,
            },
            2415: {
                "title": "Apples, Fresh:  Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2560: {
                "title": "Cherries (sweet and sour), Fresh:  Production, Supply and Distribution",
                "templateId": 3,
            },
            2563: {
                "title": "Cherries (sweet and sour), Fresh:  Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2183: {
                "title": "Grapefruit, Fresh: Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2419: {
                "title": "Grapes, Fresh Table:  Production, Supply and Distribution",
                "templateId": 3,
            },
            2416: {
                "title": "Grapes, Fresh Table:  Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2184: {
                "title": "Lemons and Limes, Fresh: Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2366: {
                "title": "Oranges, Fresh: Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2367: {
                "title": "Oranges, Fresh: Production, Supply and Distribution in Selected Countries (Continued)",
                "templateId": 8,
            },
            2561: {
                "title": "Peaches and Nectarines, Fresh:  Production, Supply and Distribution",
                "templateId": 3,
            },
            2562: {
                "title": "Peaches and Nectarines, Fresh:  Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2420: {
                "title": "Pears, Fresh:  Production, Supply and Distribution",
                "templateId": 3,
            },
            2417: {
                "title": "Pears, Fresh:  Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
            2151: {
                "title": "Raisin Production, Supply and Distribution by Country",
                "templateId": 3,
            },
            2304: {
                "title": "Raisin Production, Supply and Distribution for Select Countries",
                "templateId": 8,
            },
            2182: {
                "title": "Tangerines/Mandarins, Fresh: Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
        },
    },
    "gra": {
        "name": "Grains",
        "reports": {
            965: {"title": "All Grain Summary Comparison", "templateId": 11},
            866: {
                "title": "Barley Supply and Disappearance: Selected Exporters",
                "templateId": 3,
            },
            867: {"title": "China: Grain Supply and Demand", "templateId": 13},
            879: {
                "title": "Corn Supply and Disappearance: Selected Exporters",
                "templateId": 3,
            },
            868: {"title": "European Union: Grain Supply and Demand", "templateId": 13},
            869: {
                "title": "Former Soviet Union: Grain Supply and Demand",
                "templateId": 13,
            },
            966: {
                "title": "Former Soviet Union: Grain Supply and Demand (Barley)",
                "templateId": 3,
            },
            969: {
                "title": "Former Soviet Union: Grain Supply and Demand (Corn)",
                "templateId": 3,
            },
            967: {
                "title": "Former Soviet Union: Grain Supply and Demand (Wheat)",
                "templateId": 3,
            },
            870: {"title": "Other Europe: Grain Supply and Demand", "templateId": 13},
            368: {
                "title": "Regional Barley Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            1993: {
                "title": "Regional Coarse Grains Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            1963: {
                "title": "Regional Corn Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            398: {
                "title": "Regional Oats Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            1984: {
                "title": "Regional Rice Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            469: {
                "title": "Regional Rye Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            1992: {
                "title": "Regional Sorghum Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            1983: {
                "title": "Regional Wheat Imports, Production, Consumption, and Stocks",
                "templateId": 8,
            },
            871: {
                "title": "Sorghum Supply and Disappearance: Selected Exporters",
                "templateId": 3,
            },
            872: {
                "title": "Total Wheat and Coarse Grains: Supply and Demand",
                "templateId": 13,
            },
            873: {
                "title": "U.S. Grains Supply and Distribution: Wheat, Corn, Sorghum, Barley, Oats, Rye, and Rice",
                "templateId": 13,
            },
            875: {
                "title": "Wheat Supply and Disappearance: Selected Exporters",
                "templateId": 3,
            },
            2373: {
                "title": "World Barley Production, Consumption, and Stocks",
                "templateId": 7,
            },
            393: {"title": "World Barley Trade", "templateId": 7},
            753: {"title": "World Coarse Grain Trade", "templateId": 7},
            690: {
                "title": "World Coarse Grains Production, Consumption, and Stocks",
                "templateId": 7,
            },
            1811: {
                "title": "World Corn and Barley: Supply and Demand",
                "templateId": 13,
            },
            459: {
                "title": "World Corn Production, Consumption, and Stocks",
                "templateId": 7,
            },
            455: {"title": "World Corn Trade", "templateId": 7},
            401: {
                "title": "World Oats Production, Consumption, and Stocks",
                "templateId": 7,
            },
            466: {"title": "World Oats Trade", "templateId": 7},
            681: {
                "title": "World Rice Production, Consumption, and Stocks",
                "templateId": 7,
            },
            677: {"title": "World Rice Trade", "templateId": 7},
            877: {"title": "World Rice: Supply and Demand", "templateId": 13},
            676: {
                "title": "World Rye Production, Consumption, and Stocks",
                "templateId": 7,
            },
            2503: {"title": "World Rye Trade", "templateId": 7},
            462: {
                "title": "World Sorghum Production, Consumption, and Stocks",
                "templateId": 7,
            },
            460: {"title": "World Sorghum Trade", "templateId": 7},
            878: {
                "title": "World Wheat and Coarse Grains: Supply and Demand",
                "templateId": 13,
            },
            750: {
                "title": "World Wheat Production, Consumption, and Stocks",
                "templateId": 7,
            },
            386: {"title": "World Wheat, Flour, and Products Trade", "templateId": 7},
        },
    },
    "juc": {
        "name": "Juice",
        "reports": {
            2181: {
                "title": "Orange Juice: Production, Supply and Distribution in Selected Countries",
                "templateId": 8,
            },
        },
    },
    "liv": {
        "name": "Livestock and Poultry",
        "reports": {
            2693: {
                "title": "Beef and Veal Production - Top Countries Summary",
                "templateId": 7,
            },
            2694: {
                "title": "Beef and Veal Trade - Top Countries Summary",
                "templateId": 7,
            },
            2695: {"title": "Cattle Stocks - Top Countries Summary", "templateId": 7},
            2696: {"title": "Cattle Trade - Top Countries Summary", "templateId": 7},
            2673: {
                "title": "Chicken Meat Production - Top Countries Summary",
                "templateId": 7,
            },
            2674: {
                "title": "Chicken Meat Trade - Top Countries Summary",
                "templateId": 7,
            },
            2660: {"title": "Pork Production - Top Countries Summary", "templateId": 7},
            2661: {"title": "Pork Trade - Top Countries Summary", "templateId": 7},
            2658: {"title": "Swine Stocks - Top Countries Summary", "templateId": 7},
            2659: {"title": "Swine Trade - Top Countries Summary", "templateId": 7},
        },
    },
    "oil": {
        "name": "Oilseeds",
        "reports": {
            531: {
                "title": "Table 01: Major Oilseeds: World Supply and Distribution (Commodity View)",
                "templateId": 5,
            },
            701: {
                "title": "Table 02: Major Protein Meals: World Supply and Distribution (Commodity View)",
                "templateId": 5,
            },
            533: {
                "title": "Table 03: Major Vegetable Oils: World Supply and Distribution (Commodity View)",
                "templateId": 5,
            },
            703: {
                "title": "Table 04: Major Oilseeds: World Supply and Distribution (Country View)",
                "templateId": 8,
            },
            704: {
                "title": "Table 05: Major Protein Meals: World Supply and Distribution (Country View)",
                "templateId": 8,
            },
            705: {
                "title": "Table 06: Major Vegetable Oils: World Supply and Distribution (Country View)",
                "templateId": 8,
            },
            706: {
                "title": "Table 07: Soybeans: World Supply and Distribution",
                "templateId": 8,
            },
            707: {
                "title": "Table 08: Soybean Meal: World Supply and Distribution",
                "templateId": 8,
            },
            708: {
                "title": "Table 09: Soybean Oil: World Supply and Distribution",
                "templateId": 8,
            },
            709: {
                "title": "Table 10: Soybeans and Products: World Trade",
                "templateId": 11,
            },
            710: {
                "title": "Table 11: Palm Oil: World Supply and Distribution",
                "templateId": 8,
            },
            711: {
                "title": "Table 12: Rapeseed and Products: World Supply and Distribution",
                "templateId": 11,
            },
            712: {
                "title": "Table 13: Sunflowerseed and Products: World Supply and Distribution",
                "templateId": 11,
            },
            713: {
                "title": "Table 14:  Minor Vegetable Oils: World Supply and Distribution",
                "templateId": 11,
            },
            714: {
                "title": "Table 15: World Oilseeds and Products Supply and Distribution",
                "templateId": 13,
            },
            715: {
                "title": "Table 16:  World Soybeans and Products Supply and Distribution",
                "templateId": 13,
            },
            716: {
                "title": "Table 17: World Rapeseed and Products Supply and Distribution",
                "templateId": 13,
            },
            717: {
                "title": "Table 18: World Sunflower and Products Supply and Distribution",
                "templateId": 13,
            },
            718: {
                "title": "Table 19: World Palm Oil, Coconut Oil, and Fish Meal Supply and Distribution",
                "templateId": 13,
            },
            719: {
                "title": "Table 20: United States Oilseeds and Products Supply and Distribution Local Marketing Year",
                "templateId": 13,
            },
            720: {
                "title": "Table 21: United States Soybeans and Products Supply and Distribution (Sep-Aug)",
                "templateId": 13,
            },
            721: {
                "title": "Table 22: Brazil Soybeans and Products Supply and Distribution "
                + "Local Marketing Years (Jan-Dec)",
                "templateId": 13,
            },
            722: {
                "title": "Table 23: Argentina Soybeans and Products Supply and Distribution "
                + "Local Marketing Years (Apr-Mar)",
                "templateId": 13,
            },
            723: {
                "title": "Table 24: South East Asia Oilseeds and Products Supply and Distribution",
                "templateId": 5,
            },
            724: {
                "title": "Table 25: Middle East Oilseeds and Products Supply and Distribution",
                "templateId": 5,
            },
            725: {
                "title": "Table 26: European Union Oilseeds and Products Supply and Distribution",
                "templateId": 5,
            },
            726: {
                "title": "Table 27: China Oilseeds and Products Supply and Distribution",
                "templateId": 5,
            },
            727: {
                "title": "Table 28: India Oilseeds and Products Supply and Distribution",
                "templateId": 5,
            },
        },
    },
    "sug": {
        "name": "Sugar",
        "reports": {
            2423: {"title": "World Centrifugal Sugar: Ending Stocks", "templateId": 8},
            2421: {
                "title": "World Centrifugal Sugar: Imports and Exports",
                "templateId": 8,
            },
            2422: {
                "title": "World Centrifugal Sugar: Production and Consumption",
                "templateId": 8,
            },
        },
    },
    "nut": {
        "name": "Tree Nuts",
        "reports": {
            2077: {
                "title": "Almond Production, Supply and Distribution",
                "templateId": 3,
            },
            2095: {"title": "Almond Summary", "templateId": 8},
            2364: {"title": "Pistachio Summary", "templateId": 8},
            2148: {"title": "Pistachios PSD", "templateId": 3},
            2079: {
                "title": "Walnut Production, Supply and Distribution",
                "templateId": 3,
            },
            2099: {"title": "Walnut Summary", "templateId": 8},
        },
    },
}

# Mapping of human-readable commodity names to group codes
PSD_COMMODITY_GROUPS = {
    "coffee": "cof",
    "cotton": "cot",
    "dairy": "dai",
    "field_crops": "fcr",
    "fruits_vegetables": "htp",
    "grains": "gra",
    "juice": "juc",
    "livestock": "liv",
    "oilseeds": "oil",
    "sugar": "sug",
    "tree_nuts": "nut",
}

# Each friendly name maps to the report_id
PSD_REPORT_NAMES = {
    # === COFFEE ===
    "coffee_summary": 2109,
    "coffee_summary_2": 2535,
    "coffee_summary_3": 2619,
    "coffee_summary_4": 2620,
    "coffee_production": 1559,
    "coffee_arabica_production": 1673,
    "coffee_robusta_production": 1679,
    "coffee_consumption": 2245,
    "coffee_ending_stocks": 1783,
    "coffee_exports_total": 2053,
    "coffee_exports_green_bean": 1670,
    "coffee_exports_soluble": 2052,
    "coffee_imports_total": 2055,
    "coffee_imports_green_bean": 1774,
    "coffee_imports_soluble": 2056,
    # === COTTON ===
    "cotton_monthly_changes": 837,
    "cotton_world_supply_use": 849,
    "cotton_world_supply_use_2": 850,
    "cotton_area_yield_production": 851,
    "cotton_supply_distribution": 852,
    "cotton_supply_distribution_2": 853,
    "cotton_by_country": 854,
    "cotton_by_country_2": 855,
    "cotton_world_supply": 856,
    "cotton_us_supply": 857,
    "cotton_foreign_supply": 858,
    # === DAIRY ===
    "butter_production_consumption": 2538,
    "butter_trade": 2539,
    "cheese_production_consumption": 2540,
    "cheese_trade": 2542,
    "milk_production_consumption": 2544,
    "milk_cow_numbers": 2543,
    "nonfat_dry_milk_production_consumption": 2548,
    "nonfat_dry_milk_trade": 2545,
    "whole_milk_powder_production_consumption": 2549,
    "whole_milk_powder_trade": 2547,
    # === FIELD CROPS (fcr) ===
    "world_crop_production_summary": 425,
    "wheat_area_yield_production": 448,
    "coarse_grains_area_yield_production": 453,
    "corn_area_yield_production": 884,
    "barley_area_yield_production": 885,
    "oats_area_yield_production": 890,
    "rye_area_yield_production": 891,
    "sorghum_area_yield_production": 892,
    "rice_area_yield_production": 893,
    "oilseeds_area_yield_production": 2403,
    "soybeans_area_yield_production": 906,
    "cottonseed_area_yield_production": 914,
    "peanut_area_yield_production": 918,
    "sunflower_area_yield_production": 917,
    "rapeseed_area_yield_production": 922,
    "copra_palm_kernel_palm_oil_production": 454,
    "cotton_area_yield_production_fcr": 958,
    # === FRUITS & VEGETABLES (htp) ===
    "apples_supply_distribution": 2418,
    "apples_selected_countries": 2415,
    "cherries_supply_distribution": 2560,
    "cherries_selected_countries": 2563,
    "grapefruit_selected_countries": 2183,
    "grapes_supply_distribution": 2419,
    "grapes_selected_countries": 2416,
    "lemons_limes_selected_countries": 2184,
    "oranges_selected_countries": 2366,
    "oranges_selected_countries_2": 2367,
    "peaches_nectarines_supply_distribution": 2561,
    "peaches_nectarines_selected_countries": 2562,
    "pears_supply_distribution": 2420,
    "pears_selected_countries": 2417,
    "raisins_supply_distribution": 2151,
    "raisins_selected_countries": 2304,
    "tangerines_mandarins_selected_countries": 2182,
    # === GRAINS (gra) ===
    "grains_summary_comparison": 965,
    "barley_supply_disappearance": 866,
    "china_grain_supply_demand": 867,
    "corn_supply_disappearance": 879,
    "eu_grain_supply_demand": 868,
    "russia_grain_supply_demand": 869,
    "russia_barley": 966,
    "russia_corn": 969,
    "russia_wheat": 967,
    "other_europe_grain_supply_demand": 870,
    "barley_regional": 368,
    "coarse_grains_regional": 1993,
    "corn_regional": 1963,
    "oats_regional": 398,
    "rice_regional": 1984,
    "rye_regional": 469,
    "sorghum_regional": 1992,
    "wheat_regional": 1983,
    "sorghum_supply_disappearance": 871,
    "wheat_coarse_grains_supply_demand": 872,
    "us_grains_supply_distribution": 873,
    "wheat_supply_disappearance": 875,
    "barley_world_production_consumption_stocks": 2373,
    "barley_world_trade": 393,
    "coarse_grains_world_trade": 753,
    "coarse_grains_world_production_consumption_stocks": 690,
    "corn_barley_supply_demand": 1811,
    "corn_world_production_consumption_stocks": 459,
    "corn_world_trade": 455,
    "oats_world_production_consumption_stocks": 401,
    "oats_world_trade": 466,
    "rice_world_production_consumption_stocks": 681,
    "rice_world_trade": 677,
    "rice_supply_demand": 877,
    "rye_world_production_consumption_stocks": 676,
    "rye_world_trade": 2503,
    "sorghum_world_production_consumption_stocks": 462,
    "sorghum_world_trade": 460,
    "wheat_coarse_grains_world_supply_demand": 878,
    "wheat_world_production_consumption_stocks": 750,
    "wheat_flour_products_world_trade": 386,
    # === JUICE ===
    "orange_juice_supply_distribution": 2181,
    # === LIVESTOCK ===
    "beef_veal_production": 2693,
    "beef_veal_trade": 2694,
    "cattle_stocks": 2695,
    "cattle_trade": 2696,
    "chicken_production": 2673,
    "chicken_trade": 2674,
    "pork_production": 2660,
    "pork_trade": 2661,
    "swine_stocks": 2658,
    "swine_trade": 2659,
    # === OILSEEDS (oil) ===
    "oilseeds_world_commodity_view": 531,
    "protein_meals_world_commodity_view": 701,
    "vegetable_oils_world_commodity_view": 533,
    "oilseeds_world_country_view": 703,
    "protein_meals_world_country_view": 704,
    "vegetable_oils_world_country_view": 705,
    "soybeans_world_supply": 706,
    "soybean_meal_world_supply": 707,
    "soybean_oil_world_supply": 708,
    "soybeans_products_world_trade": 709,
    "palm_oil_world_supply": 710,
    "rapeseed_products_world_supply": 711,
    "sunflower_products_world_supply": 712,
    "vegetable_oils_minor_world_supply": 713,
    "oilseeds_products_world_supply_demand": 714,
    "soybeans_products_world_supply_demand": 715,
    "rapeseed_products_world_supply_demand": 716,
    "sunflower_products_world_supply_demand": 717,
    "palm_coconut_fishmeal_world_supply_demand": 718,
    "oilseeds_us_supply_distribution": 719,
    "soybeans_us_supply_distribution": 720,
    "soybeans_brazil_supply_distribution": 721,
    "soybeans_argentina_supply_distribution": 722,
    "oilseeds_southeast_asia": 723,
    "oilseeds_middle_east": 724,
    "oilseeds_eu": 725,
    "oilseeds_china": 726,
    "oilseeds_india": 727,
    # === SUGAR ===
    "sugar_ending_stocks": 2423,
    "sugar_imports_exports": 2421,
    "sugar_production_consumption": 2422,
    # === TREE NUTS ===
    "almonds_supply_distribution": 2077,
    "almonds_summary": 2095,
    "pistachios_summary": 2364,
    "pistachios_supply_distribution": 2148,
    "walnuts_supply_distribution": 2079,
    "walnuts_summary": 2099,
}

AttributeType = Literal[
    "area_planted",
    "area_harvested",
    "catch_for_reduction",
    "cows_in_milk",
    "crush",
    "total_grape_crush",
    "deliv_to_processors",
    "total_trees",
    "bearing_trees",
    "non_bearing_trees",
    "beginning_stocks",
    "sow_beginning_stocks",
    "dairy_cows_beg_stocks",
    "begin_stock_ctrl_app",
    "beef_cows_beg_stocks",
    "begin_stock_other",
    "production",
    "arabica_production",
    "beet_sugar_production",
    "commercial_production",
    "cows_milk_production",
    "farm_sales_weight_prod",
    "filter_production",
    "prod_from_wine_grapes",
    "cane_sugar_production",
    "non_comm_production",
    "non_filter_production",
    "other_milk_production",
    "prod_from_table_grapes",
    "robusta_production",
    "rough_production",
    "other_production",
    "imports",
    "bean_imports",
    "intra_eu_imports",
    "my_imp_from_us",
    "raw_imports",
    "us_leaf_imports",
    "my_imp_from_eu",
    "other_imports",
    "refined_imp_raw_val",
    "roast_ground_imports",
    "cy_imports",
    "ty_imports",
    "soluble_imports",
    "cy_imp_from_us",
    "ty_imp_from_us",
    "total_supply",
    "cy_exp_to_us",
    "exports",
    "raw_exports",
    "bean_exports",
    "intra_eu_exports",
    "intra_eu_exports_alt",
    "my_exp_to_eu",
    "refined_exp_raw_val",
    "other_exports",
    "roast_ground_exports",
    "cy_exports",
    "ty_exports",
    "soluble_exports",
    "slaughter_reference",
    "total_slaughter",
    "cow_slaughter",
    "inventory_reference",
    "sow_slaughter",
    "calf_slaughter",
    "other_slaughter",
    "domestic_consumption",
    "total_disappearance",
    "dom_leaf_consumption",
    "dom_consump_ctrl_app",
    "feed_dom_consumption",
    "fluid_use_dom_consum",
    "for_processing",
    "fresh_dom_consumption",
    "fresh_dom_consumption_alt",
    "human_consumption",
    "human_dom_consumption",
    "industrial_dom_cons",
    "rst_ground_dom_consum",
    "domestic_use",
    "utilization_for_sugar",
    "dom_consump_other",
    "factory_use_consum",
    "food_use_dom_cons",
    "loss",
    "other_disappearance",
    "other_use_losses",
    "soluble_dom_cons",
    "us_leaf_dom_cons",
    "utilization_for_alcohol",
    "feed_use_dom_consum",
    "feed_waste_dom_cons",
    "other_foreign_cons",
    "withdrawal_from_market",
    "loss_and_residual",
    "total_disappearance_alt",
    "total_use",
    "total_utilization",
    "ending_stocks",
    "end_stocks_ctrl_app",
    "total_distribution",
    "end_stocks_other",
    "extr_rate",
    "milling_rate",
    "seed_to_lint_ratio",
    "yield",
    "fsi_consumption",
    "sme",
    "stocks_to_use",
    "exportable_production",
    "balance",
    "inventory_balance",
    "inventory_change",
    "import_change",
    "export_change",
    "consumption_change",
    "production_change",
    "sow_change",
    "cow_change",
    "production_to_cows",
    "production_to_sows",
    "slaughter_to_inventory",
    "weights",
    "population",
    "per_capita_consumption",
    "slaughter_to_total_supply",
    "imports_percent_consumption",
    "exports_percent_production",
    "annual_pct_change_per_cap_cons",
    "stocks_to_use_months",
]

CommodityType = Literal[
    "almonds",
    "cattle",
    "swine",
    "apples",
    "barley",
    "cherries",
    "coffee",
    "corn",
    "cotton",
    "butter",
    "cheese",
    "dry_whole_milk_powder",
    "fluid_milk",
    "nonfat_dry_milk",
    "grapefruit",
    "grapes",
    "lemons_limes",
    "meal_copra",
    "meal_cottonseed",
    "meal_fish",
    "meal_palm_kernel",
    "meal_peanut",
    "meal_rapeseed",
    "meal_soybean",
    "meal_sunflowerseed",
    "beef",
    "chicken",
    "pork",
    "millet",
    "mixed_grain",
    "oats",
    "oil_coconut",
    "oil_cottonseed",
    "oil_olive",
    "oil_palm",
    "oil_palm_kernel",
    "oil_peanut",
    "oil_rapeseed",
    "oil_soybean",
    "oil_sunflowerseed",
    "oilseed_copra",
    "oilseed_cottonseed",
    "oilseed_palm_kernel",
    "oilseed_peanut",
    "oilseed_rapeseed",
    "oilseed_soybean",
    "oilseed_sunflowerseed",
    "orange_juice",
    "oranges",
    "peaches_nectarines",
    "pears",
    "pistachios",
    "broiler",
    "rice",
    "rye",
    "sorghum",
    "sugar",
    "tangerines_mandarins",
    "walnuts",
    "wheat",
]

COMMODITIES = {
    "almonds": "0577400",
    "cattle": "0011000",
    "swine": "0013000",
    "apples": "0574000",
    "barley": "0430000",
    "cherries": "0579305",
    "coffee": "0711100",
    "corn": "0440000",
    "cotton": "2631000",
    "butter": "0230000",
    "cheese": "0240000",
    "dry_whole_milk_powder": "0224400",
    "fluid_milk": "0223000",
    "nonfat_dry_milk": "0224200",
    "grapefruit": "0572220",
    "grapes": "0575100",
    "lemons_limes": "0572120",
    "meal_copra": "0813700",
    "meal_cottonseed": "0813300",
    "meal_fish": "0814200",
    "meal_palm_kernel": "0813800",
    "meal_peanut": "0813200",
    "meal_rapeseed": "0813600",
    "meal_soybean": "0813100",
    "meal_sunflowerseed": "0813500",
    "beef": "0111000",
    "chicken": "0115000",
    "pork": "0113000",
    "millet": "0459100",
    "mixed_grain": "0459900",
    "oats": "0452000",
    "oil_coconut": "4242000",
    "oil_cottonseed": "4233000",
    "oil_olive": "4235000",
    "oil_palm": "4243000",
    "oil_palm_kernel": "4244000",
    "oil_peanut": "4234000",
    "oil_rapeseed": "4239100",
    "oil_soybean": "4232000",
    "oil_sunflowerseed": "4236000",
    "oilseed_copra": "2231000",
    "oilseed_cottonseed": "2223000",
    "oilseed_palm_kernel": "2232000",
    "oilseed_peanut": "2221000",
    "oilseed_rapeseed": "2226000",
    "oilseed_soybean": "2222000",
    "oilseed_sunflowerseed": "2224000",
    "orange_juice": "0585100",
    "oranges": "0571120",
    "peaches_nectarines": "0579309",
    "pears": "0579220",
    "pistachios": "0577907",
    "broiler": "0114200",
    "rice": "0422110",
    "rye": "0451000",
    "sorghum": "0459200",
    "sugar": "0612000",
    "tangerines_mandarins": "0571220",
    "walnuts": "0577901",
    "wheat": "0410000",
}

# Attribute IDs
ATTRIBUTES = {
    "area_planted": 1,
    "area_harvested": 4,
    "catch_for_reduction": 5,
    "cows_in_milk": 6,
    "crush": 7,
    "total_grape_crush": 10,
    "deliv_to_processors": 13,
    "total_trees": 16,
    "bearing_trees": 17,
    "non_bearing_trees": 19,
    "beginning_stocks": 20,
    "sow_beginning_stocks": 22,
    "dairy_cows_beg_stocks": 23,
    "begin_stock_ctrl_app": 24,
    "beef_cows_beg_stocks": 25,
    "begin_stock_other": 26,
    "production": 28,
    "arabica_production": 29,
    "beet_sugar_production": 30,
    "commercial_production": 31,
    "cows_milk_production": 32,
    "farm_sales_weight_prod": 33,
    "filter_production": 34,
    "prod_from_wine_grapes": 40,
    "cane_sugar_production": 43,
    "non_comm_production": 47,
    "non_filter_production": 48,
    "other_milk_production": 49,
    "prod_from_table_grapes": 51,
    "robusta_production": 53,
    "rough_production": 54,
    "other_production": 56,
    "imports": 57,
    "bean_imports": 58,
    "intra_eu_imports": 62,
    "my_imp_from_us": 63,
    "raw_imports": 64,
    "us_leaf_imports": 65,
    "my_imp_from_eu": 70,
    "other_imports": 71,
    "refined_imp_raw_val": 74,
    "roast_ground_imports": 75,
    "cy_imports": 78,
    "ty_imports": 81,
    "soluble_imports": 82,
    "cy_imp_from_us": 83,
    "ty_imp_from_us": 84,
    "total_supply": 86,
    "cy_exp_to_us": 87,
    "exports": 88,
    "raw_exports": 89,
    "bean_exports": 90,
    "intra_eu_exports": 94,
    "intra_eu_exports_alt": 95,
    "my_exp_to_eu": 97,
    "refined_exp_raw_val": 99,
    "other_exports": 104,
    "roast_ground_exports": 107,
    "cy_exports": 110,
    "ty_exports": 113,
    "soluble_exports": 114,
    "slaughter_reference": 116,
    "total_slaughter": 117,
    "cow_slaughter": 118,
    "inventory_reference": 120,
    "sow_slaughter": 121,
    "calf_slaughter": 122,
    "other_slaughter": 124,
    "domestic_consumption": 125,
    "total_disappearance": 126,
    "dom_leaf_consumption": 128,
    "dom_consump_ctrl_app": 129,
    "feed_dom_consumption": 130,
    "fluid_use_dom_consum": 131,
    "for_processing": 132,
    "fresh_dom_consumption": 133,
    "fresh_dom_consumption_alt": 135,
    "human_consumption": 138,
    "human_dom_consumption": 139,
    "industrial_dom_cons": 140,
    "rst_ground_dom_consum": 141,
    "domestic_use": 142,
    "utilization_for_sugar": 143,
    "dom_consump_other": 145,
    "factory_use_consum": 147,
    "food_use_dom_cons": 149,
    "loss": 150,
    "other_disappearance": 151,
    "other_use_losses": 152,
    "soluble_dom_cons": 154,
    "us_leaf_dom_cons": 155,
    "utilization_for_alcohol": 157,
    "feed_use_dom_consum": 158,
    "feed_waste_dom_cons": 161,
    "other_foreign_cons": 167,
    "withdrawal_from_market": 169,
    "loss_and_residual": 172,
    "total_disappearance_alt": 173,
    "total_use": 174,
    "total_utilization": 175,
    "ending_stocks": 176,
    "end_stocks_ctrl_app": 177,
    "total_distribution": 178,
    "end_stocks_other": 179,
    "extr_rate": 181,
    "milling_rate": 182,
    "seed_to_lint_ratio": 183,
    "yield": 184,
    "fsi_consumption": 192,
    "sme": 194,
    "stocks_to_use": 195,
    "exportable_production": 196,
    "balance": 198,
    "inventory_balance": 199,
    "inventory_change": 200,
    "import_change": 201,
    "export_change": 202,
    "consumption_change": 203,
    "production_change": 204,
    "sow_change": 205,
    "cow_change": 206,
    "production_to_cows": 207,
    "production_to_sows": 208,
    "slaughter_to_inventory": 209,
    "weights": 210,
    "population": 211,
    "per_capita_consumption": 212,
    "slaughter_to_total_supply": 213,
    "imports_percent_consumption": 214,
    "exports_percent_production": 215,
    "annual_pct_change_per_cap_cons": 220,
    "stocks_to_use_months": 223,
}

CountryType = Literal[
    # Countries
    "world",
    "afghanistan",
    "albania",
    "algeria",
    "angola",
    "argentina",
    "armenia",
    "australia",
    "austria",
    "azerbaijan",
    "bahamas",
    "bahrain",
    "bangladesh",
    "barbados",
    "belarus",
    "belgium",
    "belize",
    "benin",
    "bhutan",
    "bolivia",
    "bosnia_and_herzegovina",
    "botswana",
    "brazil",
    "brunei",
    "bulgaria",
    "burkina_faso",
    "burma",
    "myanmar",
    "burundi",
    "cabo_verde",
    "cambodia",
    "cameroon",
    "canada",
    "central_african_republic",
    "chad",
    "chile",
    "china",
    "colombia",
    "comoros",
    "congo_brazzaville",
    "congo_kinshasa",
    "costa_rica",
    "cote_divoire",
    "ivory_coast",
    "croatia",
    "cuba",
    "cyprus",
    "czech_republic",
    "czechia",
    "denmark",
    "djibouti",
    "dominica",
    "dominican_republic",
    "ecuador",
    "egypt",
    "el_salvador",
    "equatorial_guinea",
    "eritrea",
    "estonia",
    "eswatini",
    "swaziland",
    "ethiopia",
    "eu_15",
    "eu_25",
    "eu",
    "european_union",
    "fiji",
    "finland",
    "france",
    "gabon",
    "gambia",
    "georgia",
    "germany",
    "ghana",
    "greece",
    "guatemala",
    "guinea",
    "guinea_bissau",
    "guyana",
    "haiti",
    "honduras",
    "hong_kong",
    "hungary",
    "iceland",
    "india",
    "indonesia",
    "iran",
    "iraq",
    "ireland",
    "israel",
    "italy",
    "jamaica",
    "japan",
    "jordan",
    "kazakhstan",
    "kenya",
    "kosovo",
    "kuwait",
    "kyrgyzstan",
    "laos",
    "latvia",
    "lebanon",
    "lesotho",
    "liberia",
    "libya",
    "lithuania",
    "luxembourg",
    "macedonia",
    "macau",
    "madagascar",
    "malawi",
    "malaysia",
    "maldives",
    "mali",
    "malta",
    "mauritania",
    "mauritius",
    "mexico",
    "moldova",
    "mongolia",
    "montenegro",
    "morocco",
    "mozambique",
    "namibia",
    "nepal",
    "netherlands",
    "new_caledonia",
    "new_zealand",
    "nicaragua",
    "niger",
    "nigeria",
    "north_korea",
    "north_macedonia",
    "norway",
    "oman",
    "pakistan",
    "panama",
    "papua_new_guinea",
    "paraguay",
    "peru",
    "philippines",
    "poland",
    "portugal",
    "puerto_rico",
    "qatar",
    "reunion",
    "romania",
    "russia",
    "rwanda",
    "samoa",
    "sao_tome_and_principe",
    "saudi_arabia",
    "senegal",
    "serbia",
    "seychelles",
    "sierra_leone",
    "singapore",
    "slovakia",
    "slovenia",
    "solomon_islands",
    "somalia",
    "south_africa",
    "south_korea",
    "south_sudan",
    "spain",
    "sri_lanka",
    "sudan",
    "suriname",
    "sweden",
    "switzerland",
    "syria",
    "taiwan",
    "tajikistan",
    "tanzania",
    "thailand",
    "togo",
    "tonga",
    "trinidad_and_tobago",
    "tunisia",
    "turkey",
    "turkmenistan",
    "uganda",
    "ukraine",
    "united_arab_emirates",
    "united_kingdom",
    "united_states",
    "uruguay",
    "uzbekistan",
    "vanuatu",
    "venezuela",
    "vietnam",
    "yemen",
    "zambia",
    "zimbabwe",
    # Regions
    "north_america",
    "caribbean",
    "central_america",
    "south_america",
    "former_soviet_union",
    "middle_east",
    "north_africa",
    "sub_saharan_africa",
    "south_asia",
    "oceania",
    "other_europe",
    "southeast_asia",
    "east_asia",
]

COUNTRIES = {
    "world": "00",
    "afghanistan": "AF",
    "albania": "AL",
    "algeria": "AG",
    "angola": "AO",
    "argentina": "AR",
    "armenia": "AM",
    "australia": "AS",
    "austria": "AU",
    "azerbaijan": "AJ",
    "bahamas": "BF",
    "bahrain": "BA",
    "bangladesh": "BG",
    "barbados": "BB",
    "belarus": "BO",
    "belgium": "S8",
    "belize": "BH",
    "benin": "DM",
    "bhutan": "BT",
    "bolivia": "BL",
    "bosnia_and_herzegovina": "BK",
    "botswana": "BC",
    "brazil": "BR",
    "brunei": "BX",
    "bulgaria": "BU",
    "burkina_faso": "UV",
    "burma": "BM",
    "myanmar": "BM",
    "burundi": "BY",
    "cabo_verde": "CV",
    "cambodia": "CB",
    "cameroon": "CM",
    "canada": "CA",
    "central_african_republic": "CT",
    "chad": "CD",
    "chile": "CI",
    "china": "CH",
    "colombia": "CO",
    "comoros": "CN",
    "congo_brazzaville": "CF",
    "congo_kinshasa": "CG",
    "costa_rica": "CS",
    "cote_divoire": "IV",
    "ivory_coast": "IV",
    "croatia": "HR",
    "cuba": "CU",
    "cyprus": "CY",
    "czech_republic": "EZ",
    "czechia": "EZ",
    "denmark": "DA",
    "djibouti": "DJ",
    "dominica": "DO",
    "dominican_republic": "DR",
    "ecuador": "EC",
    "egypt": "EG",
    "el_salvador": "ES",
    "equatorial_guinea": "EK",
    "eritrea": "ER",
    "estonia": "EN",
    "eswatini": "WZ",
    "swaziland": "WZ",
    "ethiopia": "ET",
    "eu_15": "E2",
    "eu_25": "E3",
    "eu": "E4",
    "european_union": "E4",
    "fiji": "FJ",
    "finland": "FI",
    "france": "FR",
    "gabon": "GB",
    "gambia": "GA",
    "georgia": "GG",
    "germany": "GM",
    "ghana": "GH",
    "greece": "GR",
    "guatemala": "GT",
    "guinea": "GU",
    "guinea_bissau": "PU",
    "guyana": "GY",
    "haiti": "HA",
    "honduras": "HO",
    "hong_kong": "HK",
    "hungary": "HU",
    "iceland": "IC",
    "india": "IN",
    "indonesia": "ID",
    "iran": "IR",
    "iraq": "IZ",
    "ireland": "EI",
    "israel": "IS",
    "italy": "IT",
    "jamaica": "JM",
    "japan": "JA",
    "jordan": "JO",
    "kazakhstan": "KZ",
    "kenya": "KE",
    "kosovo": "KV",
    "kuwait": "KU",
    "kyrgyzstan": "KG",
    "laos": "LA",
    "latvia": "LG",
    "lebanon": "LE",
    "lesotho": "LT",
    "liberia": "LI",
    "libya": "LY",
    "lithuania": "LH",
    "luxembourg": "LU",
    "macedonia": "MK",
    "macau": "MC",
    "madagascar": "MA",
    "malawi": "MI",
    "malaysia": "MY",
    "maldives": "MV",
    "mali": "ML",
    "malta": "MT",
    "mauritania": "MR",
    "mauritius": "MP",
    "mexico": "MX",
    "moldova": "MD",
    "mongolia": "MG",
    "montenegro": "MJ",
    "morocco": "MO",
    "mozambique": "MZ",
    "namibia": "WA",
    "nepal": "NP",
    "netherlands": "NL",
    "new_caledonia": "NC",
    "new_zealand": "NZ",
    "nicaragua": "NU",
    "niger": "NG",
    "nigeria": "NI",
    "north_korea": "KN",
    "north_macedonia": "MK",
    "norway": "NO",
    "oman": "MU",
    "pakistan": "PK",
    "panama": "PN",
    "papua_new_guinea": "PP",
    "paraguay": "PA",
    "peru": "PE",
    "philippines": "RP",
    "poland": "PL",
    "portugal": "PO",
    "puerto_rico": "RQ",
    "qatar": "QA",
    "reunion": "RE",
    "romania": "RO",
    "russia": "RS",
    "rwanda": "RW",
    "samoa": "WS",
    "sao_tome_and_principe": "TP",
    "saudi_arabia": "SA",
    "senegal": "SG",
    "serbia": "RB",
    "seychelles": "SE",
    "sierra_leone": "SL",
    "singapore": "SN",
    "slovakia": "LO",
    "slovenia": "SI",
    "solomon_islands": "BP",
    "somalia": "SO",
    "south_africa": "SF",
    "south_korea": "KS",
    "south_sudan": "OD",
    "spain": "SP",
    "sri_lanka": "CE",
    "sudan": "SU",
    "suriname": "NS",
    "sweden": "SW",
    "switzerland": "SZ",
    "syria": "SY",
    "taiwan": "TW",
    "tajikistan": "TI",
    "tanzania": "TZ",
    "thailand": "TH",
    "togo": "TO",
    "tonga": "TN",
    "trinidad_and_tobago": "TD",
    "tunisia": "TS",
    "turkey": "TU",
    "turkmenistan": "TX",
    "uganda": "UG",
    "ukraine": "UP",
    "united_arab_emirates": "TC",
    "united_kingdom": "UK",
    "united_states": "US",
    "uruguay": "UY",
    "uzbekistan": "UZ",
    "vanuatu": "NH",
    "venezuela": "VE",
    "vietnam": "VM",
    "yemen": "YM",
    "zambia": "ZA",
    "zimbabwe": "RH",
}

REGIONS = {
    "world": "R00",
    "north_america": "R01",
    "caribbean": "R02",
    "central_america": "R03",
    "south_america": "R04",
    "european_union": "R05",
    "former_soviet_union": "R07",
    "middle_east": "R09",
    "north_africa": "R10",
    "sub_saharan_africa": "R11",
    "south_asia": "R12",
    "oceania": "R14",
    "other_europe": "R16",
    "southeast_asia": "R17",
    "east_asia": "R18",
}

COUNTRY_TO_REGION = {
    # North America (R01)
    "US": "R01",
    "CA": "R01",
    "MX": "R01",
    # Caribbean (R02)
    "BF": "R02",
    "BB": "R02",
    "CU": "R02",
    "DO": "R02",
    "DR": "R02",
    "HA": "R02",
    "JM": "R02",
    "RQ": "R02",
    "TD": "R02",
    # Central America (R03)
    "BH": "R03",
    "CS": "R03",
    "ES": "R03",
    "GT": "R03",
    "HO": "R03",
    "NU": "R03",
    "PN": "R03",
    # South America (R04)
    "AR": "R04",
    "BL": "R04",
    "BR": "R04",
    "CI": "R04",
    "CO": "R04",
    "EC": "R04",
    "GY": "R04",
    "PA": "R04",
    "PE": "R04",
    "NS": "R04",
    "UY": "R04",
    "VE": "R04",
    # European Union (R05)
    "AU": "R05",
    "S8": "R05",
    "BU": "R05",
    "HR": "R05",
    "CY": "R05",
    "EZ": "R05",
    "DA": "R05",
    "EN": "R05",
    "FI": "R05",
    "FR": "R05",
    "GM": "R05",
    "GR": "R05",
    "HU": "R05",
    "EI": "R05",
    "IT": "R05",
    "LG": "R05",
    "LH": "R05",
    "LU": "R05",
    "MT": "R05",
    "NL": "R05",
    "PL": "R05",
    "PO": "R05",
    "RO": "R05",
    "LO": "R05",
    "SI": "R05",
    "SP": "R05",
    "SW": "R05",
    "E4": "R05",
    # Former Soviet Union (R07)
    "AM": "R07",
    "AJ": "R07",
    "BO": "R07",
    "GG": "R07",
    "KZ": "R07",
    "KG": "R07",
    "MD": "R07",
    "RS": "R07",
    "TI": "R07",
    "TX": "R07",
    "UP": "R07",
    "UZ": "R07",
    # Middle East (R09)
    "BA": "R09",
    "IR": "R09",
    "IZ": "R09",
    "IS": "R09",
    "JO": "R09",
    "KU": "R09",
    "LE": "R09",
    "MU": "R09",
    "QA": "R09",
    "SA": "R09",
    "SY": "R09",
    "TC": "R09",
    "YM": "R09",
    # North Africa (R10)
    "AG": "R10",
    "EG": "R10",
    "LY": "R10",
    "MO": "R10",
    "TS": "R10",
    # Sub-Saharan Africa (R11)
    "AO": "R11",
    "DM": "R11",
    "BC": "R11",
    "UV": "R11",
    "BY": "R11",
    "CV": "R11",
    "CM": "R11",
    "CT": "R11",
    "CD": "R11",
    "CN": "R11",
    "CF": "R11",
    "CG": "R11",
    "IV": "R11",
    "DJ": "R11",
    "EK": "R11",
    "ER": "R11",
    "WZ": "R11",
    "ET": "R11",
    "GB": "R11",
    "GA": "R11",
    "GH": "R11",
    "GU": "R11",
    "PU": "R11",
    "KE": "R11",
    "LT": "R11",
    "LI": "R11",
    "MA": "R11",
    "MI": "R11",
    "ML": "R11",
    "MR": "R11",
    "MP": "R11",
    "MZ": "R11",
    "WA": "R11",
    "NG": "R11",
    "NI": "R11",
    "RE": "R11",
    "RW": "R11",
    "TP": "R11",
    "SG": "R11",
    "SE": "R11",
    "SL": "R11",
    "SO": "R11",
    "SF": "R11",
    "OD": "R11",
    "SU": "R11",
    "TZ": "R11",
    "TO": "R11",
    "UG": "R11",
    "ZA": "R11",
    "RH": "R11",
    # South Asia (R12)
    "AF": "R12",
    "BG": "R12",
    "BT": "R12",
    "IN": "R12",
    "MV": "R12",
    "NP": "R12",
    "PK": "R12",
    "CE": "R12",
    # Oceania (R14)
    "AS": "R14",
    "FJ": "R14",
    "NC": "R14",
    "NZ": "R14",
    "PP": "R14",
    "WS": "R14",
    "BP": "R14",
    "TN": "R14",
    "NH": "R14",
    # Other Europe (R16)
    "AL": "R16",
    "BK": "R16",
    "IC": "R16",
    "KV": "R16",
    "MK": "R16",
    "MJ": "R16",
    "NO": "R16",
    "RB": "R16",
    "SZ": "R16",
    "TU": "R16",
    "UK": "R16",
    # Southeast Asia (R17)
    "BX": "R17",
    "BM": "R17",
    "CB": "R17",
    "ID": "R17",
    "LA": "R17",
    "MY": "R17",
    "RP": "R17",
    "SN": "R17",
    "TH": "R17",
    "VM": "R17",
    # East Asia (R18)
    "CH": "R18",
    "HK": "R18",
    "JA": "R18",
    "KN": "R18",
    "KS": "R18",
    "MC": "R18",
    "MG": "R18",
    "TW": "R18",
}

REGION_TO_COUNTRIES: dict = {}
for country_code, region_code in COUNTRY_TO_REGION.items():
    if region_code not in REGION_TO_COUNTRIES:
        REGION_TO_COUNTRIES[region_code] = []
    REGION_TO_COUNTRIES[region_code].append(country_code)

REGION_DISPLAY = {
    "R00": "World",
    "R01": "North America",
    "R02": "Caribbean",
    "R03": "Central America",
    "R04": "South America",
    "R05": "European Union",
    "R07": "Former Soviet Union - 12",
    "R09": "Middle East",
    "R10": "North Africa",
    "R11": "Sub-Saharan Africa",
    "R12": "South Asia",
    "R14": "Oceania",
    "R16": "Other Europe",
    "R17": "Southeast Asia",
    "R18": "East Asia",
}
