import { categorizeError } from "./error-utils";
import type { components } from "./prefect";

/**
 * UI Settings with camelCase properties for use in React components.
 */
export type UiSettings = {
	apiUrl: string;
	csrfEnabled: boolean;
	auth: string | null;
	flags: string[];
};

// Response type from autogenerated schema
type UiSettingsResponse = components["schemas"]["UISettings"];

/**
 * Singleton class for managing UI settings.
 * Fetches settings from /ui-settings endpoint on first access,
 * then caches for subsequent calls.
 */
class UiSettingsService {
	private settings: UiSettings | null = null;
	private promise: Promise<UiSettings> | null = null;

	/**
	 * Determine the base URL for fetching settings.
	 * - In development: Use VITE_API_URL if set, otherwise localhost:4200
	 * - In production: Use relative URL (same origin as the served UI)
	 */
	private getBaseUrl(): string {
		if (import.meta.env.DEV) {
			// Development mode: use env var or fallback
			const envUrl = import.meta.env.VITE_API_URL as string | undefined;
			if (envUrl) {
				// Strip /api suffix if present to get base URL
				return envUrl.replace(/\/api\/?$/, "");
			}
			return "http://127.0.0.1:4200";
		}
		// Production: UI is served from same origin as API
		return "";
	}

	async load(): Promise<UiSettings> {
		if (this.settings !== null) {
			return this.settings;
		}

		if (this.promise !== null) {
			return this.promise;
		}

		this.promise = (async () => {
			const baseUrl = this.getBaseUrl();

			let response: Response;
			try {
				response = await fetch(`${baseUrl}/ui-settings`);
			} catch (error) {
				// Reset promise so next call can retry
				this.promise = null;
				// Re-throw with categorized error for better messaging
				const categorized = categorizeError(
					error,
					"Failed to connect to Prefect server",
				);
				throw new Error(
					categorized.message +
						(categorized.details ? `: ${categorized.details}` : ""),
				);
			}

			if (!response.ok) {
				// Reset promise so next call can retry
				this.promise = null;
				const categorized = categorizeError(
					new Error(`Failed to fetch UI settings: status ${response.status}`),
					"Server returned an error",
				);
				throw new Error(
					categorized.message +
						(categorized.details ? `: ${categorized.details}` : ""),
				);
			}

			const data = (await response.json()) as UiSettingsResponse;

			return {
				apiUrl: data.api_url,
				csrfEnabled: data.csrf_enabled,
				auth: data.auth,
				flags: data.flags ?? [],
			};
		})();

		try {
			this.settings = await this.promise;
			return this.settings;
		} catch (error) {
			// Ensure promise is reset on any error
			this.promise = null;
			throw error;
		}
	}

	async getApiUrl(): Promise<string> {
		const settings = await this.load();
		return settings.apiUrl;
	}

	/**
	 * Reset settings (useful for testing)
	 */
	reset(): void {
		this.settings = null;
		this.promise = null;
	}
}

export const uiSettings = new UiSettingsService();
