#
# MIT License
#
# Copyright The SCons Foundation
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

"""Tool-specific initialization for yacc.

This tool should support multiple yacc implementations, but is in actuality
biased towards GNU Bison. In particular, it forces the output file name (thus
avoiding the default convention of y.tab.c or foo.tab.c), so the tool *must*
support the -o option, which pure POSIX yacc does not.  byacc should be okay
as an alternative to bison.

There normally shouldn't be any need to import this module directly.
It will usually be imported through the generic SCons.Tool.Tool()
selection method.
"""

from __future__ import annotations

import os.path
import sys

import SCons.Defaults
import SCons.Tool
import SCons.Warnings
from SCons.Platform.mingw import MINGW_DEFAULT_PATHS
from SCons.Platform.cygwin import CYGWIN_DEFAULT_PATHS
from SCons.Platform.win32 import CHOCO_DEFAULT_PATH
from SCons.Util import CLVar, to_String

DEFAULT_PATHS = CHOCO_DEFAULT_PATH + MINGW_DEFAULT_PATHS + CYGWIN_DEFAULT_PATHS

YaccAction = SCons.Action.Action("$YACCCOM", "$YACCCOMSTR")

if sys.platform == 'win32':
    BINS = ['bison', 'yacc', 'win_bison']
else:
    BINS = ["bison", "yacc", "byacc"]  # for byacc, yacc is normally a link


def _yaccEmitter(target, source, env, ysuf, hsuf) -> tuple:
    """Adds extra files generated by yacc program to target list."""

    yaccflags = env.subst_list("$YACCFLAGS", target=target, source=source)
    flags = yaccflags[0]
    targetBase, targetExt = os.path.splitext(to_String(target[0]))

    if '.ym' in ysuf:  # If using Objective-C
        target = [targetBase + ".m"]  # the extension is ".m".

    # If -d is specified on the command line, yacc will emit a .h
    # or .hpp file with the same base name as the .c or .cpp output file.
    # if '-d' in flags:
    # or  bison options -H, --header, --defines (obsolete)
    if "-d" in flags or "-H" in flags or "--header" in flags or "--defines" in flags:
        target.append(targetBase + env.subst(hsuf, target=target, source=source))

    # If -g is specified on the command line, yacc will emit a graph
    # file with the same base name as the .c or .cpp output file.
    # TODO: should this be handled like -v? i.e. a side effect, not target
    # if "-g" in flags:
    # or bison option --graph
    if "-g" in flags or "--graph" in flags:
        target.append(targetBase + env.subst("$YACC_GRAPH_FILE_SUFFIX"))

    # If -v is specified yacc will create the output debug file
    # which is not really source for any process, but should
    # be noted and also be cleaned (issue #2558)
    if "-v" in flags:
        env.SideEffect(targetBase + '.output', target[0])
        env.Clean(target[0], targetBase + '.output')

    # With --defines and --graph, the file to write is defined by the option
    # argument, if present (the no-option-argument cases were caught above).
    # Extract this and include in the list of targets.
    # NOTE: a filename passed to the command this way is not modified by
    # SCons, and so will be interpreted relative to the project top directory
    # at execution time, while the name added to the target list will be
    # interpreted relative to the SConscript directory - a possible mismatch.
    # Better to use YACC_HEADER_FILE and YACC_GRAPH_FILE to pass these.
    #
    # These are GNU bison-only options.
    # Since bison 3.8, --header is the preferred name over --defines
    fileGenOptions = ["--defines=", "--header=", "--graph="]
    for option in flags:
        for fileGenOption in fileGenOptions:
            l = len(fileGenOption)
            if option[:l] == fileGenOption:
                fileName = option[l:].strip()
                target.append(fileName)

    yaccheaderfile = env.subst("$YACC_HEADER_FILE", target=target, source=source)
    if yaccheaderfile:
        target.append(yaccheaderfile)
        # rewrite user-supplied file string with a node, we need later
        env.Replace(YACC_HEADER_FILE=env.File(yaccheaderfile))

    yaccgraphfile = env.subst("$YACC_GRAPH_FILE", target=target, source=source)
    if yaccgraphfile:
        target.append(yaccgraphfile)
        # rewrite user-supplied file string with a node, we need later
        env.Replace(YACC_GRAPH_FILE=env.File(yaccgraphfile))

    return target, source


def yEmitter(target, source, env) -> tuple:
    return _yaccEmitter(target, source, env, ['.y', '.yacc'], '$YACCHFILESUFFIX')


def ymEmitter(target, source, env) -> tuple:
    return _yaccEmitter(target, source, env, ['.ym'], '$YACCHFILESUFFIX')


def yyEmitter(target, source, env) -> tuple:
    return _yaccEmitter(target, source, env, ['.yy'], '$YACCHXXFILESUFFIX')


def get_yacc_path(env, append_paths: bool=False) -> str | None:
    """
    Returns the path to the yacc tool, searching several possible names.

    Only called in the Windows case, so the `default_path` argument to
    :func:`find_program_path` can be Windows-specific.

    Args:
        env: current construction environment
        append_paths: if true, add the path to the tool to PATH
    """
    for prog in BINS:
        bin_path = SCons.Tool.find_program_path(
            env,
            prog,
            default_paths=DEFAULT_PATHS,
            add_path=append_paths,
        )
        if bin_path:
            return bin_path

    SCons.Warnings.warn(
        SCons.Warnings.SConsWarning,
        'yacc tool requested, but yacc or bison binary not found in ENV PATH'
    )


def generate(env) -> None:
    """Add Builders and construction variables for yacc to an Environment."""
    c_file, cxx_file = SCons.Tool.createCFileBuilders(env)

    # C
    c_file.add_action('.y', YaccAction)
    c_file.add_emitter('.y', yEmitter)

    c_file.add_action('.yacc', YaccAction)
    c_file.add_emitter('.yacc', yEmitter)

    # Objective-C
    c_file.add_action('.ym', YaccAction)
    c_file.add_emitter('.ym', ymEmitter)

    # C++
    cxx_file.add_action('.yy', YaccAction)
    cxx_file.add_emitter('.yy', yyEmitter)

    if sys.platform == 'win32':
        # ignore the return, all we need is for the path to be added
        _ = get_yacc_path(env, append_paths=True)

    env.SetDefault(
        YACC=env.Detect(BINS),
        YACCFLAGS=CLVar(""),
        YACC_HEADER_FILE="",
        YACC_GRAPH_FILE="",
    )

    env['YACCCOM'] = '$YACC $YACCFLAGS $_YACC_HEADER $_YACC_GRAPH -o $TARGET $SOURCES'
    env['YACCHFILESUFFIX'] = '.h'
    env['YACCHXXFILESUFFIX'] = '.hpp'
    env['YACCVCGFILESUFFIX'] = '.gv'
    env['YACC_GRAPH_FILE_SUFFIX'] = '$YACCVCGFILESUFFIX'
    env['_YACC_HEADER'] = '${YACC_HEADER_FILE and "--header=" + str(YACC_HEADER_FILE)}'
    env['_YACC_GRAPH'] = '${YACC_GRAPH_FILE and "--graph=" + str(YACC_GRAPH_FILE)}'


def exists(env) -> str | None:
    if 'YACC' in env:
        return env.Detect(env['YACC'])

    if sys.platform == 'win32':
        return get_yacc_path(env)
    else:
        return env.Detect(BINS)
