#!/usr/bin/env python
#
# MIT License
#
# Copyright The SCons Foundation
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

from __future__ import print_function

"""
This configuration comes from the following blog article:

"How scalable is SCons?  The Electric Cloud Blog"
http://blog.electric-cloud.com/2010/03/08/how-scalable-is-scons/

    The test build consists of (a lot of) compiles and links. Starting
    from the bottom, we have N C files each with a unique associated
    header file. The C files and headers were spread across
    N/500 directories in order to eliminate filesystem scalability
    concerns. Both the C files and the header files are trivial: the
    header only includes stdio.h; the C file includes the associated
    header and a second, shared header, then defines a trivial
    function. Objects are collected into groups of 20 and stored into
    a standard archive. Every 20th object is linked into an executable
    along with the archive.

The original ElectricCloud implementation is captured in genscons.pl,
and we just call that script to generate the configuration in a
"sconsbld" subdirectory.
"""

import TestSCons

# The values here were calibrated by hand on the ubuntu-timings slave,
# because the configurations generated by genscons.pl only work if
# the FILES_PER_DIRECTORY value is a multiple of the COMPILES_GROUPED
# value, and it didn't seem worth automating that manipulation.
#
# The key value below is FILES_PER_DIRECTORY; the other values match
# the default from the genscons.pl file.  With the values below,
# it creates a two-deep hierarchy of a single directory with three
# subdirectories.  Each directory (both parent and subdirectories)
# contains sixty source files (each of which includes a .h file)
# that are built into three libraries containing twenty object files
# each, which are then linked into executables.
#
# As of r5143 on 17 August 2010, a value of 60 FILES_PER_DIRECTORY
# performs a full build in 82.5 seconds on the ubuntu-timings slave.
# That's more than our "normal" target of 10 seconds or so for the
# full build, but building anything less than three libraries per
# directory feels like it makes the test too trivial.

import os

test = TestSCons.TimeSCons(variables={
    'NUMBER_OF_LEVELS'      : 2,
    'DIRECTORIES_PER_LEVEL' : 3,
    'FILES_PER_DIRECTORY'   : 60,
    'LOOKUPS_PER_SOURCE'    : 2,
    'COMPILES_GROUPED'      : 20,
},
    calibrate=['FILES_PER_DIRECTORY'])

arguments = [
    '-l %s' % test.variables['NUMBER_OF_LEVELS'],
    '-d %s' % test.variables['DIRECTORIES_PER_LEVEL'],
    '-f %s' % test.variables['FILES_PER_DIRECTORY'],
    '-g %s' % test.variables['COMPILES_GROUPED'],
    '-u %s' % test.variables['LOOKUPS_PER_SOURCE'],
]

test.run(program=test.workpath('genscons.pl'), arguments=' '.join(arguments))
# This print is not for debugging, leave it alone!
# We want to display the output from genscons.pl's generation the build
# configuration, so the buildbot logs contain more info.
print(test.stdout())

test.main(chdir='sconsbld')

test.pass_test()
