## [AKShare](https://github.com/akfamily/akshare) 股票数据

### A股

#### 股票市场总貌

##### 上海证券交易所

接口: stock_sse_summary

目标地址: http://www.sse.com.cn/market/stockdata/statistic/

描述: 上海证券交易所-股票数据总貌

限量: 单次返回最近交易日的股票数据总貌(当前交易日的数据需要交易所收盘后统计)

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数-实时行情数据

| 名称  | 类型     | 描述  |
|-----|--------|-----|
| 项目  | object | -   |
| 股票  | object | -   |
| 科创板 | object | -   |
| 主板  | object | -   |

接口示例

```python
import akshare as ak

stock_sse_summary_df = ak.stock_sse_summary()
print(stock_sse_summary_df)
```

数据示例

```
      项目     股票       科创板         主板
0   流通股本   40403.47    413.63   39989.84
1    总市值  516714.68   55719.6  460995.09
2  平均市盈率      17.92      71.0      16.51
3   上市公司       2036       377       1659
4   上市股票       2078       377       1701
5   流通市值  432772.13   22274.3  410497.83
6   报告时间   20211230  20211230   20211230
8    总股本   46234.03    1211.5   45022.54
```

##### 深圳证券交易所

###### 证券类别统计

接口: stock_szse_summary

目标地址: http://www.szse.cn/market/overview/index.html

描述: 深圳证券交易所-市场总貌-证券类别统计

限量: 单次返回指定 date 的市场总貌数据-证券类别统计(当前交易日的数据需要交易所收盘后统计)

输入参数

| 名称   | 类型  | 描述                                  |
|------|-----|-------------------------------------|
| date | str | date="20200619"; 当前交易日的数据需要交易所收盘后统计 |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 证券类别 | object  | -       |
| 数量   | int64   | 注意单位: 只 |
| 成交金额 | float64 | 注意单位: 元 |
| 总市值  | float64 | -       |
| 流通市值 | float64 | -       |

接口示例

```python
import akshare as ak

stock_szse_summary_df = ak.stock_szse_summary(date="20200619")
print(stock_szse_summary_df)
```

数据示例

```
     证券类别    数量          成交金额           总市值          流通市值
0      股票  2284  4.647749e+11  2.706514e+13  2.104546e+13
1    主板A股   460  9.775950e+10  7.864787e+12  6.943990e+12
2    主板B股    46  8.626816e+07  4.759658e+10  4.706385e+10
3     中小板   960  2.013526e+11  1.130741e+13  8.669555e+12
4   创业板A股   818  1.655765e+11  7.845345e+12  5.384854e+12
5      基金   551  1.362524e+10  2.417277e+11  2.417277e+11
6     ETF   100  1.165436e+10  1.628294e+11  1.628294e+11
7     LOF   250  7.335768e+08  4.043156e+10  4.043156e+10
8   封闭式基金     1  5.527571e+05  7.622440e+08  7.622440e+08
9    分级基金   200  1.236746e+09  3.770451e+10  3.770451e+10
10     债券  7174  1.371389e+11           NaN           NaN
11   债券现券  6599  2.911357e+10  3.683881e+13  1.823072e+12
12   债券回购    13  1.054592e+11           NaN           NaN
13    ABS   562  2.566134e+09  4.849642e+11  4.849642e+11
14     期权   108  2.441560e+08           NaN           NaN
```

###### 地区交易排序

接口: stock_szse_area_summary

目标地址: http://www.szse.cn/market/overview/index.html

描述: 深圳证券交易所-市场总貌-地区交易排序

限量: 单次返回指定 date 的市场总貌数据-地区交易排序数据

输入参数

| 名称   | 类型  | 描述                |
|------|-----|-------------------|
| date | str | date="202203"; 年月 |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 地区    | object  | -       |
| 总交易额  | float64 | 注意单位: 元 |
| 占市场   | float64 | 注意单位: % |
| 股票交易额 | float64 | 注意单位: 元 |
| 基金交易额 | float64 | 注意单位: 元 |
| 债券交易额 | float64 | 注意单位: 元 |

2025年添加优先股交易额与期权交易额

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 地区     | object  | -       |
| 总交易额   | float64 | 注意单位: 元 |
| 占市场    | float64 | 注意单位: % |
| 股票交易额  | float64 | 注意单位: 元 |
| 基金交易额  | float64 | 注意单位: 元 |
| 债券交易额  | float64 | 注意单位: 元 |
| 优先股交易额 | float64 | 注意单位: 元 |
| 期权交易额  | float64 | 注意单位: 元 |
| 接口示例   |         |         |

```python
import akshare as ak

stock_szse_area_summary_df = ak.stock_szse_area_summary(date="202412")
print(stock_szse_area_summary_df)
```

数据示例

```
    序号   地区  总交易额       占市场   股票交易额     基金交易额         债券交易额
0    1    上海  9.487525e+12  16.349  5.951120e+12  4.247199e+11  3.111261e+12
1    2    深圳  7.224594e+12  12.450  4.630469e+12  3.935335e+11  2.200312e+12
2    3    浙江  5.290404e+12   9.117  4.360136e+12  1.182085e+11  8.120595e+11
3    4    北京  5.233418e+12   9.018  3.416063e+12  2.662883e+11  1.550712e+12
4    5    江苏  4.293698e+12   7.399  3.172531e+12  2.320341e+11  8.891332e+11
5    6    广东  2.270127e+12   3.912  1.901562e+12  4.564880e+10  3.229161e+11
6    7    福建  2.253878e+12   3.884  1.623762e+12  6.501030e+10  5.651060e+11
7    8    西藏  2.206716e+12   3.803  1.905686e+12  7.203090e+10  2.289994e+11
8    9    广州  2.197150e+12   3.786  1.463381e+12  1.648445e+11  5.689248e+11
9   10  境外地区  1.984124e+12   3.419  1.970216e+12  1.390804e+10  0.000000e+00
10  11    四川  1.930677e+12   3.327  1.435194e+12  4.522977e+10  4.502540e+11
11  12    湖北  1.696401e+12   2.923  1.158206e+12  4.017641e+10  4.980182e+11
12  13    山东  1.682048e+12   2.899  1.327438e+12  5.365067e+10  3.009584e+11
13  14    湖南  1.219426e+12   2.101  9.349975e+11  1.289489e+11  1.554799e+11
14  15    河南  9.646033e+11   1.662  7.822876e+11  1.960984e+10  1.627058e+11
15  16    安徽  9.465931e+11   1.631  7.593304e+11  1.469990e+10  1.725628e+11
16  17    江西  9.396623e+11   1.619  6.881626e+11  1.311965e+10  2.383801e+11
17  18    陕西  8.237220e+11   1.419  6.012667e+11  1.214389e+10  2.103114e+11
18  19    重庆  7.391354e+11   1.274  5.619510e+11  1.890194e+10  1.582824e+11
19  20    辽宁  6.931954e+11   1.195  5.449654e+11  1.301426e+10  1.352158e+11
20  21    河北  5.669541e+11   0.977  4.501636e+11  1.233615e+10  1.044544e+11
21  22    广西  4.686531e+11   0.808  3.364304e+11  7.822388e+09  1.244003e+11
22  23    山西  4.365001e+11   0.752  3.284633e+11  1.062648e+10  9.741041e+10
23  24    天津  4.343598e+11   0.749  3.184588e+11  2.337921e+10  9.252183e+10
24  25   黑龙江  3.168352e+11   0.546  2.459350e+11  1.226023e+10  5.863994e+10
25  26    吉林  2.993509e+11   0.516  1.989160e+11  1.141372e+10  8.902121e+10
26  27    云南  2.849838e+11   0.491  2.304918e+11  5.161953e+09  4.933003e+10
27  28    贵州  2.687234e+11   0.463  1.523773e+11  7.489895e+09  1.088562e+11
28  29    新疆  2.012105e+11   0.347  1.723385e+11  2.794241e+09  2.607780e+10
29  30    海南  1.803439e+11   0.311  1.573740e+11  4.553905e+09  1.841596e+10
30  31    甘肃  1.739891e+11   0.300  1.526379e+11  2.707475e+09  1.864376e+10
31  32   内蒙古  1.694187e+11   0.292  1.277484e+11  2.512480e+09  3.915781e+10
32  33    宁夏  1.117968e+11   0.193  9.845570e+10  2.297211e+09  1.104391e+10
33  34    青海  3.967756e+10   0.068  2.884644e+10  4.814743e+08  1.034965e+10
```

```python
import akshare as ak

stock_szse_area_summary_df = ak.stock_szse_area_summary(date="202508")
print(stock_szse_area_summary_df)
```

```
    序号    地区  总交易额  ...       债券交易额       优先股交易额    期权交易额
0    1    上海  6.371903e+12  ...  2.220920e+12  876893300.0  1.243253e+10
1    2    深圳  5.121851e+12  ...  1.769490e+12          0.0  4.254038e+09
2    3    北京  3.649473e+12  ...  1.204900e+12  876893300.0  2.817363e+09
3    4    浙江  3.339664e+12  ...  6.396964e+11          0.0  1.360305e+09
4    5    江苏  2.872589e+12  ...  7.602330e+11          0.0  1.447631e+09
5    6  境外地区  1.634432e+12  ...  0.000000e+00          0.0  0.000000e+00
6    7    广州  1.500926e+12  ...  4.780673e+11    8067200.0  1.804676e+09
7    8    福建  1.493353e+12  ...  4.740214e+11          0.0  7.236936e+08
8    9    西藏  1.370016e+12  ...  1.646981e+11          0.0  1.294816e+08
9   10    广东  1.353391e+12  ...  2.476401e+11          0.0  4.377234e+08
10  11    四川  1.200289e+12  ...  3.341463e+11          0.0  2.535704e+08
11  12    湖北  1.185315e+12  ...  4.680616e+11          0.0  1.431620e+08
12  13    山东  1.076383e+12  ...  2.650255e+11          0.0  1.963902e+09
13  14    湖南  7.683324e+11  ...  1.377730e+11          0.0  1.828164e+08
14  15    江西  6.164555e+11  ...  2.175697e+11          0.0  7.712560e+07
15  16    河南  5.856996e+11  ...  1.258403e+11          0.0  1.469152e+08
16  17    安徽  5.759514e+11  ...  1.272217e+11          0.0  1.640928e+08
17  18    陕西  5.025968e+11  ...  1.510291e+11          0.0  1.016968e+08
18  19    重庆  4.503018e+11  ...  1.099315e+11          0.0  1.190079e+08
19  20    辽宁  4.321530e+11  ...  1.090213e+11          0.0  1.779504e+08
20  21    河北  3.498134e+11  ...  8.627298e+10          0.0  1.030772e+08
21  22    广西  3.058458e+11  ...  9.918531e+10          0.0  1.234911e+08
22  23    天津  2.745559e+11  ...  7.141631e+10          0.0  1.061408e+08
23  24    山西  2.727539e+11  ...  6.757100e+10          0.0  3.207566e+07
24  25    贵州  2.045149e+11  ...  9.961597e+10          0.0  3.725579e+07
25  26    吉林  1.972995e+11  ...  7.551653e+10          0.0  6.118616e+07
26  27   黑龙江  1.903182e+11  ...  4.610176e+10          0.0  5.380192e+07
27  28    云南  1.724363e+11  ...  3.210926e+10          0.0  1.397156e+08
28  29    新疆  1.201365e+11  ...  2.366439e+10          0.0  2.407912e+07
29  30   内蒙古  1.122930e+11  ...  3.212444e+10          0.0  4.740066e+07
30  31    海南  1.072153e+11  ...  1.562485e+10          0.0  9.121949e+06
31  32    甘肃  1.034123e+11  ...  1.490509e+10          0.0  3.174230e+07
32  33    宁夏  6.513347e+10  ...  8.624815e+09          0.0  1.210827e+07
33  34    青海  2.453409e+10  ...  6.640867e+09          0.0  2.349068e+06
```

######

###### 股票行业成交

接口: stock_szse_sector_summary

目标地址: http://docs.static.szse.cn/www/market/periodical/month/W020220511355248518608.html

描述: 深圳证券交易所-统计资料-股票行业成交数据

限量: 单次返回指定 symbol 和 date 的统计资料-股票行业成交数据

输入参数

| 名称     | 类型  | 描述                                  |
|--------|-----|-------------------------------------|
| symbol | str | symbol="当月"; choice of {"当月", "当年"} |
| date   | str | date="202501"; 年月                   |

输出参数

| 名称        | 类型      | 描述      |
|-----------|---------|---------|
| 项目名称      | object  | -       |
| 项目名称-英文   | object  | -       |
| 交易天数      | int64   | -       |
| 成交金额-人民币元 | int64   |         |
| 成交金额-占总计  | float64 | 注意单位: % |
| 成交股数-股数   | int64   | -       |
| 成交股数-占总计  | float64 | 注意单位: % |
| 成交笔数-笔    | int64   | -       |
| 成交笔数-占总计  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_szse_sector_summary_df = ak.stock_szse_sector_summary(symbol="当年", date="202501")
print(stock_szse_sector_summary_df)
```

数据示例

```
    项目名称            项目名称-英文 交易天数  ... 成交股数-占总计 成交笔数-笔 成交笔数-占总计
0     合计                     Total    18  ...    100.00  1072706301    100.00
1   农林牧渔               Agriculture    18  ...      0.85     7661044      0.71
2    采矿业                    Mining    18  ...      0.90     9355248      0.87
3    制造业             Manufacturing    18  ...     59.75   715973899     66.74
4   水电煤气                 Utilities    18  ...      1.47    12802444      1.19
5    建筑业              Construction    18  ...      1.39    10222345      0.95
6   批发零售        Wholesale & Retail    18  ...      4.73    40706994      3.79
7   运输仓储            Transportation    18  ...      0.99    11426446      1.07
8   住宿餐饮         Hotels & Catering    18  ...      0.19     1728478      0.16
9   信息技术                        IT    18  ...     11.73   137478307     12.82
10   金融业                   Finance    18  ...      3.76    31633210      2.95
11   房地产               Real Estate    18  ...      2.88    13878880      1.29
12  商务服务          Business Support    18  ...      6.48    38310245      3.57
13  科研服务    Research & Development    18  ...      0.63    10972681      1.02
14  公共环保  Environmental Protection    18  ...      1.61    10919804      1.02
15  居民服务         Resident Services    18  ...      0.01      217920      0.02
16    教育                 Education    18  ...      0.64     3040949      0.28
17    卫生             Public Health    18  ...      0.51     4706600      0.44
18  文化传播                     Media    18  ...      1.33    10017645      0.93
19    综合             Conglomerates    18  ...      0.14     1653162      0.15
[20 rows x 9 columns]
```

##### 上海证券交易所-每日概况

接口: stock_sse_deal_daily

目标地址: http://www.sse.com.cn/market/stockdata/overview/day/

描述: 上海证券交易所-数据-股票数据-成交概况-股票成交概况-每日股票情况

限量: 单次返回指定日期的每日概况数据, 当前交易日数据需要在收盘后获取; 注意仅支持获取在 20211227（包含）之后的数据

输入参数

| 名称   | 类型  | 描述                                                              |
|------|-----|-----------------------------------------------------------------|
| date | str | date="20250221"; 当前交易日的数据需要交易所收盘后统计; 注意仅支持获取在 20211227（包含）之后的数据 |

输出参数

| 名称   | 类型      | 描述        |
|------|---------|-----------|
| 单日情况 | object  | 包含了网页所有字段 |
| 股票   | float64 | -         |
| 主板A  | float64 | -         |
| 主板B  | float64 | -         |
| 科创板  | float64 | -         |
| 股票回购 | float64 | -         |

接口示例

```python
import akshare as ak

stock_sse_deal_daily_df = ak.stock_sse_deal_daily(date="20250221")
print(stock_sse_deal_daily_df)
```

数据示例

```
    单日情况           股票          主板A       主板B         科创板  股票回购
0    挂牌数    2321.0000    1693.0000   43.0000    585.0000   0.0
1   市价总值  529981.4800  456997.7000  942.6300  72041.1500   0.0
2   流通市值  501613.5100  445348.4700  713.7700  55551.2700   0.0
3   成交金额    8561.3100    6413.6300    4.3000   2143.3700   0.3
4    成交量     608.5800     556.5800    0.7200     51.2900   0.1
5  平均市盈率      14.3200      13.2000    7.1600     45.7800   NaN
6    换手率       1.6154       1.4034    0.4565      2.9752   0.0
7  流通换手率       1.7068       1.4401    0.6029      3.8584   0.0
```

#### 个股信息查询-东财

接口: stock_individual_info_em

目标地址: http://quote.eastmoney.com/concept/sh603777.html?from=classic

描述: 东方财富-个股-股票信息

限量: 单次返回指定 symbol 的个股信息

输入参数

| 名称      | 类型    | 描述                      |
|---------|-------|-------------------------|
| symbol  | str   | symbol="603777"; 股票代码   |
| timeout | float | timeout=None; 默认不设置超时参数 |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_individual_info_em_df = ak.stock_individual_info_em(symbol="000001")
print(stock_individual_info_em_df)
```

数据示例

```
   item               value
0    最新               7.05
1  股票代码             000002
2  股票简称            万  科Ａ
3   总股本       11930709471.0
4   流通股        9716935865.0
5   总市值  84111501770.550003
6  流通市值      68504397848.25
7    行业              房地产开发
8  上市时间            19910129
```

#### 个股信息查询-雪球

接口: stock_individual_basic_info_xq

目标地址: https://xueqiu.com/snowman/S/SH601127/detail#/GSJJ

描述: 雪球财经-个股-公司概况-公司简介

限量: 单次返回指定 symbol 的个股信息

输入参数

| 名称      | 类型    | 描述                      |
|---------|-------|-------------------------|
| symbol  | str   | symbol="SH601127"; 股票代码 |
| token   | str   | token=None;             |
| timeout | float | timeout=None; 默认不设置超时参数 |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_individual_basic_info_xq_df = ak.stock_individual_basic_info_xq(symbol="SH601127")
print(stock_individual_basic_info_xq_df)
```

数据示例

```
                            item                                              value
0                         org_id                                         T000071215
1                    org_name_cn                                        赛力斯集团股份有限公司
2              org_short_name_cn                                                赛力斯
3                    org_name_en                               Seres Group Co.,Ltd.
4              org_short_name_en                                              SERES
5        main_operation_business      新能源汽车及核心三电(电池、电驱、电控)、传统汽车及核心部件总成的研发、制造、销售及服务。
6                operating_scope  　　一般项目：制造、销售：汽车零部件、机动车辆零部件、普通机械、电器机械、电器、电子产品（不...
7                district_encode                                             500106
8            org_cn_introduction  赛力斯始创于1986年，是以新能源汽车为核心业务的技术科技型汽车企业。现有员工1.6万人，A...
9           legal_representative                                                张正萍
10               general_manager                                                张正萍
11                     secretary                                                 申薇
12              established_date                                      1178812800000
13                     reg_asset                                       1509782193.0
14                     staff_num                                              16102
15                     telephone                                     86-23-65179666
16                      postcode                                             401335
17                           fax                                     86-23-65179777
18                         email                                    601127@seres.cn
19                   org_website                                   www.seres.com.cn
20                reg_address_cn                                      重庆市沙坪坝区五云湖路7号
21                reg_address_en                                               None
22             office_address_cn                                      重庆市沙坪坝区五云湖路7号
23             office_address_en                                               None
24               currency_encode                                             019001
25                      currency                                                CNY
26                   listed_date                                      1465920000000
27               provincial_name                                                重庆市
28             actual_controller                                       张兴海 (13.79%)
29                   classi_name                                               民营企业
30                   pre_name_cn                                     重庆小康工业集团股份有限公司
31                      chairman                                                张正萍
32               executives_nums                                                 20
33              actual_issue_vol                                        142500000.0
34                   issue_price                                               5.81
35             actual_rc_net_amt                                        738451000.0
36              pe_after_issuing                                              18.19
37  online_success_rate_of_issue                                           0.110176
38            affiliate_industry         {'ind_code': 'BK0025', 'ind_name': '汽车整车'}
```

#### 行情报价

接口: stock_bid_ask_em

目标地址: https://quote.eastmoney.com/sz000001.html

描述: 东方财富-行情报价

限量: 单次返回指定股票的行情报价数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="000001"; 股票代码 |

输出参数

| 名称    | 类型      | 描述 |
|-------|---------|----|
| item  | object  | -  |
| value | float64 | -  |

接口示例

```python
import akshare as ak

stock_bid_ask_em_df = ak.stock_bid_ask_em(symbol="000001")
print(stock_bid_ask_em_df)
```

数据示例

```
          item         value
0       sell_5  1.049000e+01
1   sell_5_vol  1.147100e+06
2       sell_4  1.048000e+01
3   sell_4_vol  1.035700e+06
4       sell_3  1.047000e+01
5   sell_3_vol  1.489100e+06
6       sell_2  1.046000e+01
7   sell_2_vol  1.608400e+06
8       sell_1  1.045000e+01
9   sell_1_vol  2.339000e+05
10       buy_1  1.044000e+01
11   buy_1_vol  3.690000e+05
12       buy_2  1.043000e+01
13   buy_2_vol  8.359000e+05
14       buy_3  1.042000e+01
15   buy_3_vol  6.016000e+05
16       buy_4  1.041000e+01
17   buy_4_vol  7.381000e+05
18       buy_5  1.040000e+01
19   buy_5_vol  1.301900e+06
20          最新  1.045000e+01
21          均价  1.043000e+01
22          涨幅  4.800000e-01
23          涨跌  5.000000e-02
24          总手  8.726630e+05
25          金额  9.102786e+08
26          换手  4.500000e-01
27          量比  4.400000e-01
28          最高  1.047000e+01
29          最低  1.037000e+01
30          今开  1.038000e+01
31          昨收  1.040000e+01
32          涨停  1.144000e+01
33          跌停  9.360000e+00
34          外盘  4.715810e+05
35          内盘  4.010820e+05
```

#### 实时行情数据

##### 实时行情数据-东财

###### 沪深京 A 股

接口: stock_zh_a_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#hs_a_board

描述: 东方财富网-沪深京 A 股-实时行情数据

限量: 单次返回所有沪深京 A 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_zh_a_spot_em_df = ak.stock_zh_a_spot_em()
print(stock_zh_a_spot_em_df)
```

数据示例

```
        序号 代码    名称    最新价  ...   涨速 5分钟涨跌  60日涨跌幅  年初至今涨跌幅
0        1  836149  旭杰科技   7.31  ...  0.00   0.00   28.47    -9.53
1        2  300767  震安科技   9.28  ...  0.00   0.00   -9.55   -46.61
2        3  300125  ST聆达   4.67  ...  0.00   0.00  177.98   -62.97
3        4  301027  华蓝集团   8.93  ...  0.00   0.00   17.81   -28.56
4        5  300584  海辰药业  23.26  ...  0.00   0.00   42.44     0.13
...    ...     ...   ...    ...  ...   ...    ...     ...      ...
5630  5631  300531   优博讯  12.45  ... -0.16  -0.24   25.50   -18.52
5631  5632  301302  华如科技  13.47  ...  0.07  -0.07  -28.88   -46.36
5632  5633  300496  中科创达  32.30  ... -0.09  -0.12  -44.60   -59.53
5633  5634  300050  世纪鼎利   5.04  ... -0.20   0.60   72.01     6.55
5634  5635  832175  东方碳素   6.48  ...  0.00   0.00  -21.64   -47.57
[5635 rows x 23 columns]
```

###### 沪 A 股

接口: stock_sh_a_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#sh_a_board

描述: 东方财富网-沪 A 股-实时行情数据

限量: 单次返回所有沪 A 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_sh_a_spot_em_df = ak.stock_sh_a_spot_em()
print(stock_sh_a_spot_em_df)
```

数据示例

```
      序号    代码    名称   最新价  ...  涨速  5分钟涨跌 60日涨跌幅 年初至今涨跌幅
0        1  688670   金迪克  12.80  ... -0.16   0.00  -19.14   -55.02
1        2  688656   浩欧博  22.40  ... -0.62  -0.40   -4.92   -34.81
2        3  603183   建研院   3.71  ...  0.00   0.00   10.09   -19.87
3        4  603909  建发合诚   7.98  ...  0.13   0.25  -16.09   -29.26
4        5  603700  宁水集团   9.97  ...  0.00   0.00    0.30   -28.63
...    ...     ...   ...    ...  ...   ...    ...     ...      ...
2380  2381  601512  中新集团   6.70  ...  0.00  -0.30  -18.69    -9.95
2381  2382  600187  国中水务   3.02  ...  0.00   0.00   41.12    14.39
2382  2383  603499  翔港科技  12.32  ...  0.00   0.00  -30.24    14.82
2383  2384  605178  时空科技  13.48  ...  0.00  -0.74    9.06   -51.39
2384  2385  688175  高凌信息  11.49  ...  0.00   0.00  -30.15   -42.00
[2385 rows x 23 columns]
```

###### 深 A 股

接口: stock_sz_a_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#sz_a_board

描述: 东方财富网-深 A 股-实时行情数据

限量: 单次返回所有深 A 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_sz_a_spot_em_df = ak.stock_sz_a_spot_em()
print(stock_sz_a_spot_em_df)
```

数据示例

```
      序号    代码   名称    最新价  ...   涨速 5分钟涨跌 60日涨跌幅 年初至今涨跌幅
0        1  300767  震安科技   9.28  ...  0.00   0.00   -9.55   -46.61
1        2  300125  ST聆达   4.67  ...  0.00   0.00  177.98   -62.97
2        3  301027  华蓝集团   8.93  ...  0.00   0.00   17.81   -28.56
3        4  300584  海辰药业  23.26  ...  0.00   0.00   42.44     0.13
4        5  300977  深圳瑞捷  19.85  ...  0.00   0.00   32.95     7.47
...    ...     ...   ...    ...  ...   ...    ...     ...      ...
2991  2992  301326  捷邦科技  33.76  ...  0.12  -0.50   18.96   -15.83
2992  2993  300531   优博讯  12.45  ... -0.16  -0.24   25.50   -18.52
2993  2994  301302  华如科技  13.47  ...  0.07  -0.07  -28.88   -46.36
2994  2995  300496  中科创达  32.30  ... -0.09  -0.12  -44.60   -59.53
2995  2996  300050  世纪鼎利   5.04  ... -0.20   0.60   72.01     6.55
[2996 rows x 23 columns]
```

###### 京 A 股

接口: stock_bj_a_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#bj_a_board

描述: 东方财富网-京 A 股-实时行情数据

限量: 单次返回所有京 A 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_bj_a_spot_em_df = ak.stock_bj_a_spot_em()
print(stock_bj_a_spot_em_df)
```

数据示例

```
      序号  代码   名称      最新价 涨跌幅  ...    流通市值  涨速 5分钟涨跌 60日涨跌幅 年初至今涨跌幅
0      1  836149  旭杰科技   7.31  29.84  ...   349965819  0.00   0.00   28.47    -9.53
1      2  871263  莱赛激光   9.95   8.86  ...   271814568 -0.40   1.84   -8.63   -50.74
2      3  833873  中设咨询   3.32   4.73  ...   362479267  0.00  -0.90   13.70   -23.33
3      4  833230  欧康医药   9.61   4.46  ...   321772274  0.31   0.21    6.31   -31.60
4      5  836892  广咨国际   7.68   4.35  ...  1025658954 -0.13   0.92  -11.52   -30.94
..   ...     ...   ...    ...    ...  ...         ...   ...    ...     ...      ...
249  250  833575  康乐卫士  14.93  -8.69  ...  2789441772 -0.86  -0.99  -27.63   -44.99
250  251  430476  海能技术   6.10  -9.09  ...   392764330 -0.16   0.16  -25.52   -59.03
251  252  832149   利尔达   4.28  -9.51  ...   490226248 -0.47  -0.93   25.15   -41.61
252  253  872374  云里物里  11.30  -9.89  ...   330132913  0.71   0.09   29.74   -27.47
253  254  832175  东方碳素   6.48 -20.20  ...   391996578  0.00   0.00  -21.64   -47.57
[254 rows x 23 columns]
```

###### 新股

接口: stock_new_a_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#newshares

描述: 东方财富网-新股-实时行情数据

限量: 单次返回所有新股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 上市时间    | object  | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_new_a_spot_em_df = ak.stock_new_a_spot_em()
print(stock_new_a_spot_em_df)
```

数据示例

```
    序号   代码    名称   最新价    涨跌幅  ... 流通市值   涨速  5分钟涨跌  60日涨跌幅  年初至今涨跌幅
0    1  688583   C思看   91.96   2.17  ...  1175280710  0.09   0.17  174.84   174.84
1    2  301601   C惠通   39.69   1.69  ...  1189592887  0.05  -0.08  236.36   236.36
2    3  688758  赛分科技   15.87   1.54  ...   626933019  0.06   0.00  267.36   267.36
3    4  301458  钧崴电子   40.15  19.99  ...  2452564476  0.00   0.00  286.06   286.06
4    5  603072  天和磁材   38.01  -0.99  ...  2460981244  0.00   0.16  209.02   209.02
..  ..     ...   ...     ...    ...  ...         ...   ...    ...     ...      ...
68  69  301591  肯特股份   36.84   6.17  ...   774745200 -0.08  -0.19   -4.56    -3.79
69  70  688584  上海合晶   18.62  -0.96  ...  1048019122 -0.11  -0.21    5.32    -4.81
70  71  301589  诺瓦星云  163.23   1.45  ...  3018057408 -0.06   0.19  -13.63    -5.97
71  72  688709  成都华微   29.38   1.31  ...  2265703747  0.20  -0.03   33.24    -4.86
72  73  301502  华阳智能   41.98  -0.52  ...   599096580  0.43   0.55   -7.49    -2.60
[73 rows x 24 columns]
```

###### 创业板

接口: stock_cy_a_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#gem_board

描述: 东方财富网-创业板-实时行情

限量: 单次返回所有创业板的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_cy_a_spot_em_df = ak.stock_cy_a_spot_em()
print(stock_cy_a_spot_em_df)
```

数据示例

```
     序号      代码   名称    最新价  ...    涨速  5分钟涨跌 60日涨跌幅 年初至今涨跌幅
0        1  301602    N超研  32.10  ...  0.25   3.65  379.10   379.10
1        2  300996   普联软件  20.63  ...  0.00   0.00    2.84    13.85
2        3  300117  *ST嘉寓   1.08  ...  0.00   0.00  -31.21    -0.92
3        4  300843   胜蓝股份  37.90  ... -0.21   1.64   21.28     8.25
4        5  300476   胜宏科技  58.50  ...  0.10   0.00   28.09    38.99
...    ...     ...    ...    ...  ...   ...    ...     ...      ...
1396  1397  300043   星辉娱乐   3.66  ...  0.00   0.00   14.73     0.55
1397  1398  300827   上能电气  41.48  ... -0.05   0.12    2.37    -5.51
1398  1399  300377    赢时胜  27.62  ...  0.00  -1.81  100.87     0.22
1399  1400  300515   三德科技  16.00  ... -2.20  -3.50   30.93    34.34
1400  1401  300853   申昊科技  18.67  ...  0.05   0.32    4.59   -25.56
[1401 rows x 23 columns]
```

###### 科创板

接口: stock_kc_a_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#kcb_board

描述: 东方财富网-科创板-实时行情

限量: 单次返回所有科创板的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_kc_a_spot_em_df = ak.stock_kc_a_spot_em()
print(stock_kc_a_spot_em_df)
```

数据示例

```
    序号   代码     名称     最新价  ...  涨速 5分钟涨跌 60日涨跌幅 年初至今涨跌幅
0      1  688545    N兴福   27.39  ...  0.11   0.59  134.50   134.50
1      2  688205    德科立  108.13  ... -1.30   2.01  102.87    18.82
2      3  688577   浙海德曼   47.20  ... -0.13  -0.44   30.03    20.47
3      4  688246   嘉和美康   23.99  ...  0.00   0.00  -17.13    -3.27
4      5  688622   禾信仪器   69.00  ...  0.09   0.19  131.70    13.84
..   ...     ...    ...     ...  ...   ...    ...     ...      ...
582  583  688410    山外山   10.13  ... -0.10  -0.10  -11.37    -9.23
583  584  688311   盟升电子   27.78  ... -0.04  -0.22   -8.26   -16.10
584  585  688085   三友医疗   18.52  ... -1.17  -1.23  -14.62   -12.60
585  586  688165  埃夫特-U   33.70  ... -0.09  -0.82  243.53    27.65
586  587  688408    中信博   61.18  ...  0.30   0.30  -23.46   -15.03
[587 rows x 23 columns]
```

###### AB 股比价

接口: stock_zh_ab_comparison_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#ab_comparison

描述: 东方财富网-行情中心-沪深京个股-AB股比价-全部AB股比价

限量: 单次返回全部 AB 股比价的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型      | 描述 |
|------|---------|----|
| 序号   | int64   | -  |
| B股代码 | object  | -  |
| B股名称 | object  | -  |
| 最新价B | float64 | -  |
| 涨跌幅B | float64 | -  |
| A股代码 | object  | -  |
| A股名称 | object  | -  |
| 最新价A | float64 | -  |
| 涨跌幅A | float64 | -  |
| 比价   | float64 | -  |

接口示例

```python
import akshare as ak

stock_zh_ab_comparison_em_df = ak.stock_zh_ab_comparison_em()
print(stock_zh_ab_comparison_em_df)
```

数据示例

```
    序号 B股代码   B股名称   最新价B 涨跌幅B A股代码  A股名称 最新价A 涨跌幅A   比价
0    1  200026    飞亚达Ｂ  14.38  10.02  000026    飞亚达  6.70  0.60  2.35
1    2  200030     富奥B   6.05   2.54  000030   富奥股份  3.09  0.65  2.14
2    3  900917    海欣Ｂ股   6.70   2.13  600851   海欣股份  2.63  0.38  3.55
3    4  900916    凤凰B股  13.00   1.72  600679   上海凤凰  3.47  0.58  5.22
4    5  900920    动力Ｂ股   6.23   1.63  600841   动力新科  1.97  1.03  4.41
..  ..     ...     ...    ...    ...     ...    ...   ...   ...   ...
68  69  900942    黄山Ｂ股  11.67  -2.99  600054   黄山旅游  7.25 -0.55  2.24
69  70  900902    市北B股   5.12  -4.30  600604   市北高新  1.63  0.00  4.38
70  71  900946    天雁B股   8.27  -5.05  600698   湖南天雁  2.13 -1.84  5.41
71  72  900906    中毅达B  16.88  -5.06  600610    中毅达  2.77 -0.72  8.49
72  73  900952  *ST锦港B    NaN    NaN  600190  *ST锦港   NaN   NaN  2.82
[73 rows x 10 columns]
```

##### 实时行情数据-新浪

接口: stock_zh_a_spot

目标地址: https://vip.stock.finance.sina.com.cn/mkt/#hs_a

描述: 新浪财经-沪深京 A 股数据, 重复运行本函数会被新浪暂时封 IP, 建议增加时间间隔

限量: 单次返回沪深京 A 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 代码  | object  | -       |
| 名称  | object  | -       |
| 最新价 | float64 | -       |
| 涨跌额 | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 买入  | float64 | -       |
| 卖出  | float64 | -       |
| 昨收  | float64 | -       |
| 今开  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 成交量 | float64 | 注意单位: 股 |
| 成交额 | float64 | 注意单位: 元 |
| 时间戳 | object  | -       |

接口示例

```python
import akshare as ak

stock_zh_a_spot_df = ak.stock_zh_a_spot()
print(stock_zh_a_spot_df)
```

数据示例

```
      代码        名称    最新价   涨跌额  ...  最低  成交量          成交额       时间戳
0     bj430017  星昊医药  15.41  1.15  ...  14.02   8411380.0  127141645.0  15:30:01
1     bj430047  诺思兰德  18.17  0.67  ...  17.45   4167518.0   75391105.0  15:30:01
2     bj430090  同辉信息   4.10  0.31  ...   3.75  22609069.0   90900380.0  15:30:01
3     bj430139  华岭股份  13.66  0.86  ...  12.73   4712891.0   63247000.0  15:30:01
4     bj430198  微创光电   0.00  0.00  ...   0.00         0.0          0.0  15:30:01
        ...   ...    ...   ...  ...    ...         ...          ...       ...
5311  sz301550  斯菱股份  48.53 -1.92  ...  48.50   2589623.0  127379535.0  15:35:30
5312  sz301555  惠柏新材  42.12 -1.40  ...  41.90   3411454.0  146471807.0  15:35:45
5313  sz301558  三态股份  14.33 -0.92  ...  14.32  21311135.0  313639930.0  15:35:00
5314  sz301559  中集环科  19.01 -0.49  ...  19.00   2980100.0   57148056.0  15:35:15
5315  sz301568   思泰克  49.08 -3.00  ...  48.83   9336045.0  469276460.0  15:35:30
[5316 rows x 14 columns]
```

##### 实时行情数据-雪球

接口: stock_individual_spot_xq

目标地址: https://xueqiu.com/S/SH513520

描述: 雪球-行情中心-个股

限量: 单次获取指定 symbol 的最新行情数据

输入参数

| 名称      | 类型    | 描述                                                             |
|---------|-------|----------------------------------------------------------------|
| symbol  | str   | symbol="SH600000"; 证券代码，可以是 A 股个股代码，A 股场内基金代码，A 股指数，美股代码, 美股指数 |
| token   | float | token=None; 默认不设置token                                         |
| timeout | float | timeout=None; 默认不设置超时参数                                        |

输出参数

| 名称    | 类型     | 描述 |
|-------|--------|----|
| item  | object | -  |
| value | object | -  |

接口示例

```python
import akshare as ak

stock_individual_spot_xq_df = ak.stock_individual_spot_xq(symbol="SH600000")
print(stock_individual_spot_xq_df)
```

数据示例

```
        item                value
0         代码             SH600000
1      52周最高                11.02
2        流通股          29352178996
3         跌停                 8.69
4         最高                10.29
5        流通值       299392225759.0
6     最小交易单位                  100
7         涨跌                 0.55
8       每股收益                 1.54
9         昨收                 9.65
10       成交量            149422915
11       周转率                 0.51
12     52周最低               6.8673
13        名称                 浦发银行
14       交易所                   SH
15    市盈率(动)                6.615
16  基金份额/总股本          29352178996
17   净资产中的商誉             0.726713
18        均价               10.048
19        涨幅                  5.7
20        振幅                  5.6
21        现价                 10.2
22    今年以来涨幅                -0.87
23      发行日期  1999-11-10 00:00:00
24        最低                 9.75
25  资产净值/总市值       299392225759.0
26   股息(TTM)                0.321
27  股息率(TTM)                3.147
28        货币                  CNY
29     每股净资产                22.36
30    市盈率(静)                6.615
31       成交额         1501459278.0
32       市净率                0.456
33        涨停                10.62
34  市盈率(TTM)                6.615
35        时间  2025-04-08 15:00:00
36        今开                 9.77
```

#### 历史行情数据

##### 历史行情数据-东财

接口: stock_zh_a_hist

目标地址: https://quote.eastmoney.com/concept/sh603777.html?from=classic(示例)

描述: 东方财富-沪深京 A 股日频率数据; 历史数据按日频率更新, 当日收盘价请在收盘后获取

限量: 单次返回指定沪深京 A 股上市公司、指定周期和指定日期间的历史行情日频率数据

输入参数

| 名称         | 类型    | 描述                                                       |
|------------|-------|----------------------------------------------------------|
| symbol     | str   | symbol='603777'; 股票代码可以在 **ak.stock_zh_a_spot_em()** 中获取 |
| period     | str   | period='daily'; choice of {'daily', 'weekly', 'monthly'} |
| start_date | str   | start_date='20210301'; 开始查询的日期                           |
| end_date   | str   | end_date='20210616'; 结束查询的日期                             |
| adjust     | str   | 默认返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据               |
| timeout    | float | timeout=None; 默认不设置超时参数                                  |

**股票数据复权**

1. 为何要复权：由于股票存在配股、分拆、合并和发放股息等事件，会导致股价出现较大的缺口。
若使用不复权的价格处理数据、计算各种指标，将会导致它们失去连续性，且使用不复权价格计算收益也会出现错误。
为了保证数据连贯性，常通过前复权和后复权对价格序列进行调整。

2. 前复权：保持当前价格不变，将历史价格进行增减，从而使股价连续。
前复权用来看盘非常方便，能一眼看出股价的历史走势，叠加各种技术指标也比较顺畅，是各种行情软件默认的复权方式。
这种方法虽然很常见，但也有两个缺陷需要注意。

    2.1 为了保证当前价格不变，每次股票除权除息，均需要重新调整历史价格，因此其历史价格是时变的。
这会导致在不同时点看到的历史前复权价可能出现差异。

    2.2 对于有持续分红的公司来说，前复权价可能出现负值。

3. 后复权：保证历史价格不变，在每次股票权益事件发生后，调整当前的股票价格。
后复权价格和真实股票价格可能差别较大，不适合用来看盘。
其优点在于，可以被看作投资者的长期财富增长曲线，反映投资者的真实收益率情况。

4. 在量化投资研究中普遍采用后复权数据。

输出参数-历史行情数据

| 名称   | 类型      | 描述          |
|------|---------|-------------|
| 日期   | object  | 交易日         |
| 股票代码 | object  | 不带市场标识的股票代码 |
| 开盘   | float64 | 开盘价         |
| 收盘   | float64 | 收盘价         |
| 最高   | float64 | 最高价         |
| 最低   | float64 | 最低价         |
| 成交量  | int64   | 注意单位: 手     |
| 成交额  | float64 | 注意单位: 元     |
| 振幅   | float64 | 注意单位: %     |
| 涨跌幅  | float64 | 注意单位: %     |
| 涨跌额  | float64 | 注意单位: 元     |
| 换手率  | float64 | 注意单位: %     |

接口示例-历史行情数据-不复权

```python
import akshare as ak

stock_zh_a_hist_df = ak.stock_zh_a_hist(symbol="000001", period="daily", start_date="20170301", end_date='20240528', adjust="")
print(stock_zh_a_hist_df)
```

数据示例-历史行情数据-不复权

```
            日期    股票代码   开盘   收盘  ... 振幅  涨跌幅  涨跌额 换手率
0     2017-03-01  000001   9.49   9.49  ...  0.84  0.11  0.01  0.21
1     2017-03-02  000001   9.51   9.43  ...  1.26 -0.63 -0.06  0.24
2     2017-03-03  000001   9.41   9.40  ...  0.74 -0.32 -0.03  0.20
3     2017-03-06  000001   9.40   9.45  ...  0.74  0.53  0.05  0.24
4     2017-03-07  000001   9.44   9.45  ...  0.63  0.00  0.00  0.17
...          ...     ...    ...    ...  ...   ...   ...   ...   ...
1755  2024-05-22  000001  11.56  11.56  ...  2.42  0.09  0.01  1.09
1756  2024-05-23  000001  11.53  11.40  ...  1.90 -1.38 -0.16  0.95
1757  2024-05-24  000001  11.37  11.31  ...  1.67 -0.79 -0.09  0.72
1758  2024-05-27  000001  11.31  11.51  ...  1.95  1.77  0.20  0.75
1759  2024-05-28  000001  11.50  11.40  ...  1.91 -0.96 -0.11  0.62
[1760 rows x 12 columns]
```

接口示例-历史行情数据-前复权

```python
import akshare as ak

stock_zh_a_hist_df = ak.stock_zh_a_hist(symbol="000001", period="daily", start_date="20170301", end_date='20240528', adjust="qfq")
print(stock_zh_a_hist_df)
```

数据示例-历史行情数据-前复权

```
           日期    股票代码   开盘    收盘  ...  振幅  涨跌幅  涨跌额 换手率
0     2017-03-01  000001   8.14   8.14  ...  0.98  0.12  0.01  0.21
1     2017-03-02  000001   8.16   8.08  ...  1.47 -0.74 -0.06  0.24
2     2017-03-03  000001   8.06   8.05  ...  0.87 -0.37 -0.03  0.20
3     2017-03-06  000001   8.05   8.10  ...  0.87  0.62  0.05  0.24
4     2017-03-07  000001   8.09   8.10  ...  0.74  0.00  0.00  0.17
...          ...     ...    ...    ...  ...   ...   ...   ...   ...
1755  2024-05-22  000001  11.56  11.56  ...  2.42  0.09  0.01  1.09
1756  2024-05-23  000001  11.53  11.40  ...  1.90 -1.38 -0.16  0.95
1757  2024-05-24  000001  11.37  11.31  ...  1.67 -0.79 -0.09  0.72
1758  2024-05-27  000001  11.31  11.51  ...  1.95  1.77  0.20  0.75
1759  2024-05-28  000001  11.50  11.40  ...  1.91 -0.96 -0.11  0.62
[1760 rows x 12 columns]
```

接口示例-历史行情数据-后复权

```python
import akshare as ak

stock_zh_a_hist_df = ak.stock_zh_a_hist(symbol="000001", period="daily", start_date="20170301", end_date='20240528', adjust="hfq")
print(stock_zh_a_hist_df)
```

数据示例-历史行情数据-后复权

```
           日期    股票代码   开盘     收盘  ...    振幅   涨跌幅   涨跌额 换手率
0     2017-03-01  000001  1575.20  1575.20  ...  0.83  0.10   1.63  0.21
1     2017-03-02  000001  1578.45  1565.45  ...  1.24 -0.62  -9.75  0.24
2     2017-03-03  000001  1562.20  1560.57  ...  0.73 -0.31  -4.88  0.20
3     2017-03-06  000001  1560.57  1568.70  ...  0.73  0.52   8.13  0.24
4     2017-03-07  000001  1567.07  1568.70  ...  0.62  0.00   0.00  0.17
...          ...     ...      ...      ...  ...   ...   ...    ...   ...
1755  2024-05-22  000001  2131.04  2131.04  ...  2.14  0.08   1.62  1.09
1756  2024-05-23  000001  2126.17  2105.04  ...  1.68 -1.22 -26.00  0.95
1757  2024-05-24  000001  2100.16  2090.41  ...  1.47 -0.69 -14.63  0.72
1758  2024-05-27  000001  2090.41  2122.92  ...  1.71  1.56  32.51  0.75
1759  2024-05-28  000001  2121.29  2105.04  ...  1.68 -0.84 -17.88  0.62
[1760 rows x 12 columns]
```

##### 历史行情数据-新浪

接口: stock_zh_a_daily

P.S. 建议切换为 stock_zh_a_hist 接口使用(该接口数据质量高, 访问无限制)

目标地址: https://finance.sina.com.cn/realstock/company/sh600006/nc.shtml(示例)

描述: 新浪财经-沪深京 A 股的数据, 历史数据按日频率更新; 注意其中的 **sh689009** 为 CDR, 请 通过 **ak.stock_zh_a_cdr_daily** 接口获取

限量: 单次返回指定沪深京 A 股上市公司指定日期间的历史行情日频率数据, 多次获取容易封禁 IP

输入参数

| 名称         | 类型  | 描述                                                                                   |
|------------|-----|--------------------------------------------------------------------------------------|
| symbol     | str | symbol='sh600000'; 股票代码可以在 **ak.stock_zh_a_spot()** 中获取                              |
| start_date | str | start_date='20201103'; 开始查询的日期                                                       |
| end_date   | str | end_date='20201116'; 结束查询的日期                                                         |
| adjust     | str | 默认返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; hfq-factor: 返回后复权因子; qfq-factor: 返回前复权因子 |

**股票数据复权**

1.为何要复权：由于股票存在配股、分拆、合并和发放股息等事件，会导致股价出现较大的缺口。
若使用不复权的价格处理数据、计算各种指标，将会导致它们失去连续性，且使用不复权价格计算收益也会出现错误。
为了保证数据连贯性，常通过前复权和后复权对价格序列进行调整。

2.前复权：保持当前价格不变，将历史价格进行增减，从而使股价连续。
前复权用来看盘非常方便，能一眼看出股价的历史走势，叠加各种技术指标也比较顺畅，是各种行情软件默认的复权方式。
这种方法虽然很常见，但也有两个缺陷需要注意。

2.1 为了保证当前价格不变，每次股票除权除息，均需要重新调整历史价格，因此其历史价格是时变的。
这会导致在不同时点看到的历史前复权价可能出现差异。

2.2 对于有持续分红的公司来说，前复权价可能出现负值。

3.后复权：保证历史价格不变，在每次股票权益事件发生后，调整当前的股票价格。
后复权价格和真实股票价格可能差别较大，不适合用来看盘。
其优点在于，可以被看作投资者的长期财富增长曲线，反映投资者的真实收益率情况。

4.在量化投资研究中普遍采用后复权数据。

输出参数-历史行情数据

| 名称                | 类型      | 描述            |
|-------------------|---------|---------------|
| date              | object  | 交易日           |
| open              | float64 | 开盘价           |
| high              | float64 | 最高价           |
| low               | float64 | 最低价           |
| close             | float64 | 收盘价           |
| volume            | float64 | 成交量; 注意单位: 股  |
| amount            | float64 | 成交额; 注意单位: 元  |
| outstanding_share | float64 | 流动股本; 注意单位: 股 |
| turnover          | float64 | 换手率=成交量/流动股本  |

接口示例-历史行情数据(前复权)

```python
import akshare as ak

stock_zh_a_daily_qfq_df = ak.stock_zh_a_daily(symbol="sz000001", start_date="19910403", end_date="20231027", adjust="qfq")
print(stock_zh_a_daily_qfq_df)
```

数据示例-历史行情数据(前复权)

```
           date   open   high  ...        amount  outstanding_share  turnover
0    1991-04-03   0.39   0.39  ...  5.000000e+03       3.710000e+07  0.000003
1    1991-04-04   0.39   0.39  ...  1.500000e+04       3.710000e+07  0.000008
2    1991-04-05   0.38   0.38  ...  1.000000e+04       3.710000e+07  0.000005
3    1991-04-08   0.38   0.38  ...  1.000000e+04       3.710000e+07  0.000005
4    1991-04-09   0.38   0.38  ...  1.900000e+04       3.710000e+07  0.000011
         ...    ...    ...  ...           ...                ...       ...
7730 2023-10-23  10.59  10.60  ...  5.570322e+08       1.940555e+10  0.002733
7731 2023-10-24  10.54  10.61  ...  8.015284e+08       1.940555e+10  0.003920
7732 2023-10-25  10.51  10.54  ...  1.470972e+09       1.940555e+10  0.007273
7733 2023-10-26  10.31  10.42  ...  6.219153e+08       1.940555e+10  0.003092
7734 2023-10-27  10.38  10.48  ...  9.575875e+08       1.940555e+10  0.004740
[7735 rows x 9 columns]
```

接口示例-历史行情数据(后复权)

```python
import akshare as ak

stock_zh_a_daily_hfq_df = ak.stock_zh_a_daily(symbol="sz000001", start_date="19910403", end_date="20231027", adjust="hfq")
print(stock_zh_a_daily_hfq_df)
```

数据示例-历史行情数据(后复权)

```
           date     open     high  ...        amount  outstanding_share  turnover
0    1991-04-03    49.00    49.00  ...  5.000000e+03       3.710000e+07  0.000003
1    1991-04-04    48.76    48.76  ...  1.500000e+04       3.710000e+07  0.000008
2    1991-04-05    48.52    48.52  ...  1.000000e+04       3.710000e+07  0.000005
3    1991-04-08    48.04    48.04  ...  1.000000e+04       3.710000e+07  0.000005
4    1991-04-09    47.80    47.80  ...  1.900000e+04       3.710000e+07  0.000011
         ...      ...      ...  ...           ...                ...       ...
7730 2023-10-23  1341.02  1342.28  ...  5.570322e+08       1.940555e+10  0.002733
7731 2023-10-24  1334.69  1343.55  ...  8.015284e+08       1.940555e+10  0.003920
7732 2023-10-25  1330.89  1334.69  ...  1.470972e+09       1.940555e+10  0.007273
7733 2023-10-26  1305.56  1319.49  ...  6.219153e+08       1.940555e+10  0.003092
7734 2023-10-27  1314.42  1327.09  ...  9.575875e+08       1.940555e+10  0.004740
[7735 rows x 9 columns]
```

接口示例-前复权因子

```python
import akshare as ak

qfq_factor_df = ak.stock_zh_a_daily(symbol="sz000002", adjust="qfq-factor")
print(qfq_factor_df)
```

数据示例-前复权因子

```
         date            qfq_factor
0  2023-08-25    1.0000000000000000
1  2022-08-25    1.0517754291451000
2  2021-08-25    1.1168798473057000
3  2020-08-14    1.1810093745970000
4  2019-08-15    1.2245366427798000
5  2018-08-23    1.2735981991808000
6  2017-08-29    1.3233913520593000
7  2016-07-29    1.3684162947004000
8  2015-07-21    1.4256988372693000
9  2014-05-08    1.4737021651235000
10 2013-05-16    1.5546965200431000
11 2012-07-05    1.5798398599899000
12 2011-05-27    1.6027105929296000
13 2010-05-18    1.6232581646338000
14 2009-06-08    1.6398462042724000
15 2008-06-16    1.6475522484654000
16 2007-05-16    2.6521572780175000
17 2006-07-21    4.0043855834397000
18 2005-06-29    4.1056769557021000
19 2004-05-26    6.3462750504298000
20 2003-05-23    9.5848830885446000
21 2002-07-17   19.4514673699928000
22 2001-08-21   19.7687833140383020
23 2000-08-17   20.0160648287379000
24 2000-01-10   20.2241320306998030
25 1999-08-06   21.7007260053956000
26 1998-07-10   24.0549868667835000
27 1997-07-14   26.7769985385511000
28 1997-06-27   31.2296603923877000
29 1996-08-06   36.0689784010141000
30 1995-07-04   40.3371408451341000
31 1994-06-21   47.9136235499274000
32 1993-04-05   65.9178076663026900
33 1992-03-30   99.0712868582592000
34 1991-06-10  118.8855442299110900
35 1991-01-29  169.1453113717332000
36 1900-01-01  169.1453113717332000
```

接口示例-后复权因子

```python
import akshare as ak

hfq_factor_df = ak.stock_zh_a_daily(symbol="sz000002", adjust="hfq-factor")
print(hfq_factor_df)
```

数据示例-后复权因子

```
         date            hfq_factor
0  2023-08-25  169.1453113717332000
1  2022-08-25  160.8188465756604000
2  2021-08-25  151.4445011965874000
3  2020-08-14  143.2209726780933200
4  2019-08-15  138.1300530033661900
5  2018-08-23  132.8090063887726200
6  2017-08-29  127.8120120012352000
7  2016-07-29  123.6066188533395000
8  2015-07-21  118.6402814886964000
9  2014-05-08  114.7757772056770000
10 2013-05-16  108.7963529802224000
11 2012-07-05  107.0648460362401000
12 2011-05-27  105.5370271575670000
13 2010-05-18  104.2011154213954000
14 2009-06-08  103.1470578954563000
15 2008-06-16  102.6646114132513100
16 2007-05-16   63.7765010294439100
17 2006-07-21   42.2400160641971000
18 2005-06-29   41.1979104047185000
19 2004-05-26   26.6526915438809980
20 2003-05-23   17.6470917599285000
21 2002-07-17    8.6957610011813000
22 2001-08-21    8.5561821729119010
23 2000-08-17    8.4504777946604000
24 2000-01-10    8.3635387226989000
25 1999-08-06    7.7944540348408990
26 1998-07-10    7.0316110463273000
27 1997-07-14    6.3168137059205000
28 1997-06-27    5.4161751759863000
29 1996-08-06    4.6894954853220000
30 1995-07-04    4.1932895546844000
31 1994-06-21    3.5302133055222000
32 1993-04-05    2.5660032904614000
33 1992-03-30    1.7073091178651000
34 1991-06-10    1.4227575982209000
35 1991-01-29    1.0000000000000000
36 1900-01-01    1.0000000000000000
```

##### 历史行情数据-腾讯

接口: stock_zh_a_hist_tx

目标地址: https://gu.qq.com/sh000919/zs

描述: 腾讯证券-日频-股票历史数据; 历史数据按日频率更新, 当日收盘价请在收盘后获取

限量: 单次返回指定沪深京 A 股上市公司、指定周期和指定日期间的历史行情日频率数据

输入参数

| 名称         | 类型    | 描述                                         |
|------------|-------|--------------------------------------------|
| symbol     | str   | symbol='sz000001'; 带市场标识                   |
| start_date | str   | start_date='19000101'; 开始查询的日期             |
| end_date   | str   | end_date='20500101'; 结束查询的日期               |
| adjust     | str   | 默认返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据 |
| timeout    | float | timeout=None; 默认不设置超时参数                    |

**股票数据复权**

1.为何要复权：由于股票存在配股、分拆、合并和发放股息等事件，会导致股价出现较大的缺口。
若使用不复权的价格处理数据、计算各种指标，将会导致它们失去连续性，且使用不复权价格计算收益也会出现错误。
为了保证数据连贯性，常通过前复权和后复权对价格序列进行调整。

2.前复权：保持当前价格不变，将历史价格进行增减，从而使股价连续。
前复权用来看盘非常方便，能一眼看出股价的历史走势，叠加各种技术指标也比较顺畅，是各种行情软件默认的复权方式。
这种方法虽然很常见，但也有两个缺陷需要注意。

2.1 为了保证当前价格不变，每次股票除权除息，均需要重新调整历史价格，因此其历史价格是时变的。
这会导致在不同时点看到的历史前复权价可能出现差异。

2.2 对于有持续分红的公司来说，前复权价可能出现负值。

3.后复权：保证历史价格不变，在每次股票权益事件发生后，调整当前的股票价格。
后复权价格和真实股票价格可能差别较大，不适合用来看盘。
其优点在于，可以被看作投资者的长期财富增长曲线，反映投资者的真实收益率情况。

4.在量化投资研究中普遍采用后复权数据。

输出参数-历史行情数据

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| date   | object  | 交易日     |
| open   | float64 | 开盘价     |
| close  | float64 | 收盘价     |
| high   | float64 | 最高价     |
| low    | float64 | 最低价     |
| amount | int64   | 注意单位: 手 |

接口示例-不复权

```python
import akshare as ak

stock_zh_a_hist_tx_df = ak.stock_zh_a_hist_tx(symbol="sz000001", start_date="20200101", end_date="20231027", adjust="")
print(stock_zh_a_hist_tx_df)
```

数据示例-不复权

```
     date   open  close   high    low      amount
0    2020-01-02  16.65  16.87  16.95  16.55  1530231.87
1    2020-01-03  16.94  17.18  17.31  16.92  1116194.81
2    2020-01-06  17.01  17.07  17.34  16.91   862083.50
3    2020-01-07  17.13  17.15  17.28  16.95   728607.56
4    2020-01-08  17.00  16.66  17.05  16.63   847824.12
..          ...    ...    ...    ...    ...         ...
920  2023-10-23  10.59  10.50  10.60  10.43   530404.00
921  2023-10-24  10.54  10.55  10.61  10.44   760653.00
922  2023-10-25  10.51  10.38  10.54  10.36  1411450.00
923  2023-10-26  10.31  10.41  10.42  10.30   599991.00
924  2023-10-27  10.38  10.45  10.48  10.33   919771.00
[925 rows x 6 columns]
```

接口示例-前复权

```python
import akshare as ak

stock_zh_a_hist_tx_df = ak.stock_zh_a_hist_tx(symbol="sz000001", start_date="20200101", end_date="20231027", adjust="qfq")
print(stock_zh_a_hist_tx_df)
```

数据示例-前复权

```
     date   open  close   high    low      amount
0    2020-01-02  15.74  15.96  16.04  15.64  1530231.87
1    2020-01-03  16.03  16.27  16.40  16.01  1116194.81
2    2020-01-06  16.10  16.16  16.43  16.00   862083.50
3    2020-01-07  16.22  16.24  16.37  16.04   728607.56
4    2020-01-08  16.09  15.75  16.14  15.72   847824.12
..          ...    ...    ...    ...    ...         ...
920  2023-10-23  10.59  10.50  10.60  10.43   530404.00
921  2023-10-24  10.54  10.55  10.61  10.44   760653.00
922  2023-10-25  10.51  10.38  10.54  10.36  1411450.00
923  2023-10-26  10.31  10.41  10.42  10.30   599991.00
924  2023-10-27  10.38  10.45  10.48  10.33   919771.00
[925 rows x 6 columns]
```

接口示例-后复权

```python
import akshare as ak

stock_zh_a_hist_tx_df = ak.stock_zh_a_hist_tx(symbol="sz000001", start_date="20200101", end_date="20231027", adjust="hfq")
print(stock_zh_a_hist_tx_df)
```

数据示例-后复权

```
     date     open    close     high      low      amount
0    2020-01-02  1880.18  1904.07  1912.76  1869.33  1530231.87
1    2020-01-03  1911.67  1937.73  1951.85  1909.50  1116194.81
2    2020-01-06  1919.27  1925.79  1955.11  1908.42   862083.50
3    2020-01-07  1932.30  1934.48  1948.59  1912.76   728607.56
4    2020-01-08  1918.19  1881.27  1923.62  1878.01   847824.12
..          ...      ...      ...      ...      ...         ...
920  2023-10-23  1321.10  1311.33  1322.19  1303.73   530404.00
921  2023-10-24  1315.67  1316.76  1323.27  1304.81   760653.00
922  2023-10-25  1312.41  1298.30  1315.67  1296.13  1411450.00
923  2023-10-26  1290.70  1301.56  1302.64  1289.61   599991.00
924  2023-10-27  1298.30  1305.90  1309.16  1292.87   919771.00
[925 rows x 6 columns]
```

##### 分时数据-新浪

接口: stock_zh_a_minute

目标地址: http://finance.sina.com.cn/realstock/company/sh600519/nc.shtml

描述: 新浪财经-沪深京 A 股股票或者指数的分时数据，目前可以获取 1, 5, 15, 30, 60 分钟的数据频率, 可以指定是否复权

限量: 单次返回指定股票或指数的指定频率的最近交易日的历史分时行情数据; 注意调用频率

输入参数

| 名称     | 类型  | 描述                                                         |
|--------|-----|------------------------------------------------------------|
| symbol | str | symbol='sh000300'; 同日频率数据接口                                |
| period | str | period='1'; 获取 1, 5, 15, 30, 60 分钟的数据频率                    |
| adjust | str | adjust=""; 默认为空: 返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| day    | object  | -   |
| open   | float64 | -   |
| high   | float64 | -   |
| low    | float64 | -   |
| close  | float64 | -   |
| volume | float64 | -   |

接口示例

```python
import akshare as ak

stock_zh_a_minute_df = ak.stock_zh_a_minute(symbol='sh600751', period='1', adjust="qfq")
print(stock_zh_a_minute_df)
```

数据示例

```
                       day  open  high   low  close  volume
0      2020-06-17 14:49:00  3.05  3.05  3.04   3.04  133200
1      2020-06-17 14:50:00  3.05  3.05  3.04   3.04  131900
2      2020-06-17 14:51:00  3.04  3.05  3.04   3.04  332700
3      2020-06-17 14:52:00  3.05  3.05  3.04   3.04   71100
4      2020-06-17 14:53:00  3.04  3.05  3.04   3.04   49200
                    ...   ...   ...   ...    ...     ...
19995  2020-10-23 14:54:00  3.52  3.52  3.51   3.52  234900
19996  2020-10-23 14:55:00  3.51  3.52  3.51   3.52  337300
19997  2020-10-23 14:56:00  3.52  3.53  3.51   3.52  198067
19998  2020-10-23 14:57:00  3.53  3.53  3.52   3.53  225800
19999  2020-10-23 15:00:00  3.52  3.52  3.52   3.52  259123
```

##### 分时数据-东财

接口: stock_zh_a_hist_min_em

目标地址: https://quote.eastmoney.com/concept/sh603777.html

描述: 东方财富网-行情首页-沪深京 A 股-每日分时行情; 该接口只能获取近期的分时数据，注意时间周期的设置

限量: 单次返回指定股票、频率、复权调整和时间区间的分时数据, 其中 1 分钟数据只返回近 5 个交易日数据且不复权

输入参数

| 名称         | 类型  | 描述                                                                                                  |
|------------|-----|-----------------------------------------------------------------------------------------------------|
| symbol     | str | symbol='000300'; 股票代码                                                                               |
| start_date | str | start_date="1979-09-01 09:32:00"; 日期时间; 默认返回所有数据                                                    |
| end_date   | str | end_date="2222-01-01 09:32:00"; 日期时间; 默认返回所有数据                                                      |
| period     | str | period='5'; choice of {'1', '5', '15', '30', '60'}; 其中 1 分钟数据返回近 5 个交易日数据且不复权                       |
| adjust     | str | adjust=''; choice of {'', 'qfq', 'hfq'}; '': 不复权, 'qfq': 前复权, 'hfq': 后复权, 其中 1 分钟数据返回近 5 个交易日数据且不复权 |

输出参数-1分钟数据

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 时间  | object  | -       |
| 开盘  | float64 | -       |
| 收盘  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 成交量 | float64 | 注意单位: 手 |
| 成交额 | float64 | -       |
| 均价  | float64 | -       |

接口示例-1分钟数据

```python
import akshare as ak

# 注意：该接口返回的数据只有最近一个交易日的有开盘价，其他日期开盘价为 0
stock_zh_a_hist_min_em_df = ak.stock_zh_a_hist_min_em(symbol="000001", start_date="2024-03-20 09:30:00", end_date="2024-03-20 15:00:00", period="1", adjust="")
print(stock_zh_a_hist_min_em_df)
```

数据示例-1分钟数据

```
               时间     开盘     收盘     最高     最低    成交量    成交额     均价
0    2024-03-20 09:30:00  10.38  10.38  10.38  10.38   7174   7446612.0  10.380
1    2024-03-20 09:31:00  10.38  10.40  10.40  10.38  12967  13471486.0  10.386
2    2024-03-20 09:32:00  10.40  10.41  10.41  10.39  10409  10824353.0  10.390
3    2024-03-20 09:33:00  10.41  10.41  10.41  10.40   4896   5095656.0  10.393
4    2024-03-20 09:34:00  10.40  10.39  10.41  10.39   7060   7341787.0  10.394
..                   ...    ...    ...    ...    ...    ...         ...     ...
236  2024-03-20 14:56:00  10.44  10.44  10.45  10.44   2646   2762914.0  10.431
237  2024-03-20 14:57:00  10.44  10.44  10.45  10.44   7613   7951367.0  10.431
238  2024-03-20 14:58:00  10.45  10.45  10.45  10.45    110    114850.0  10.431
239  2024-03-20 14:59:00  10.45  10.45  10.45  10.45      0         0.0  10.431
240  2024-03-20 15:00:00  10.45  10.45  10.45  10.45  13079  13667555.0  10.431
[241 rows x 8 columns]
```

输出参数-其他

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 时间  | object  | -       |
| 开盘  | float64 | -       |
| 收盘  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 涨跌额 | float64 | -       |
| 成交量 | float64 | 注意单位: 手 |
| 成交额 | float64 | -       |
| 振幅  | float64 | 注意单位: % |
| 换手率 | float64 | 注意单位: % |

接口示例-其他

```python
import akshare as ak

stock_zh_a_hist_min_em_df = ak.stock_zh_a_hist_min_em(symbol="000001", start_date="2024-03-20 09:30:00", end_date="2024-03-20 15:00:00", period="5", adjust="hfq")
print(stock_zh_a_hist_min_em_df)
```

数据示例-其他

```
             时间       开盘       收盘  ...         成交额    振幅   换手率
0   2024-03-20 09:35:00  1939.26  1940.89  ...  53652034.0  0.25  0.03
1   2024-03-20 09:40:00  1939.26  1940.89  ...  27557310.0  0.17  0.01
2   2024-03-20 09:45:00  1939.26  1939.26  ...  11556035.0  0.08  0.01
3   2024-03-20 09:50:00  1940.89  1945.76  ...  27377765.0  0.42  0.01
4   2024-03-20 09:55:00  1947.39  1944.14  ...  37867949.0  0.17  0.02
5   2024-03-20 10:00:00  1945.76  1947.39  ...  27334901.0  0.25  0.01
6   2024-03-20 10:05:00  1945.76  1945.76  ...  14407157.0  0.17  0.01
7   2024-03-20 10:10:00  1944.14  1950.64  ...  46100711.0  0.33  0.02
8   2024-03-20 10:15:00  1950.64  1949.01  ...  35531670.0  0.25  0.02
9   2024-03-20 10:20:00  1949.01  1945.76  ...  20072949.0  0.17  0.01
10  2024-03-20 10:25:00  1947.39  1945.76  ...  11039368.0  0.17  0.01
11  2024-03-20 10:30:00  1945.76  1942.51  ...  29766381.0  0.17  0.01
12  2024-03-20 10:35:00  1942.51  1942.51  ...   9093103.0  0.08  0.00
13  2024-03-20 10:40:00  1944.14  1944.14  ...   9585217.0  0.17  0.00
14  2024-03-20 10:45:00  1944.14  1944.14  ...   9546537.0  0.17  0.00
15  2024-03-20 10:50:00  1944.14  1945.76  ...   9917328.0  0.08  0.00
16  2024-03-20 10:55:00  1945.76  1942.51  ...  10891142.0  0.17  0.01
17  2024-03-20 11:00:00  1944.14  1944.14  ...   7600144.0  0.08  0.00
18  2024-03-20 11:05:00  1944.14  1944.14  ...   7900327.0  0.17  0.00
19  2024-03-20 11:10:00  1942.51  1944.14  ...   7612710.0  0.08  0.00
20  2024-03-20 11:15:00  1942.51  1949.01  ...  47061728.0  0.42  0.02
21  2024-03-20 11:20:00  1950.64  1947.39  ...  19120650.0  0.25  0.01
22  2024-03-20 11:25:00  1947.39  1952.26  ...  58101226.0  0.33  0.03
23  2024-03-20 11:30:00  1953.89  1952.26  ...  36299314.0  0.17  0.02
24  2024-03-20 13:05:00  1952.26  1950.64  ...  49781778.0  0.33  0.02
25  2024-03-20 13:10:00  1950.64  1949.01  ...  25451905.0  0.25  0.01
26  2024-03-20 13:15:00  1949.01  1949.01  ...  17763129.0  0.17  0.01
27  2024-03-20 13:20:00  1949.01  1950.64  ...  17839258.0  0.17  0.01
28  2024-03-20 13:25:00  1949.01  1950.64  ...  13335325.0  0.17  0.01
29  2024-03-20 13:30:00  1950.64  1949.01  ...   5756958.0  0.08  0.00
30  2024-03-20 13:35:00  1949.01  1950.64  ...   4852792.0  0.08  0.00
31  2024-03-20 13:40:00  1950.64  1950.64  ...  16137370.0  0.17  0.01
32  2024-03-20 13:45:00  1949.01  1949.01  ...   4450010.0  0.08  0.00
33  2024-03-20 13:50:00  1949.01  1950.64  ...   4778321.0  0.08  0.00
34  2024-03-20 13:55:00  1949.01  1949.01  ...  12025904.0  0.17  0.01
35  2024-03-20 14:00:00  1947.39  1949.01  ...   3459610.0  0.08  0.00
36  2024-03-20 14:05:00  1947.39  1949.01  ...   7610675.0  0.17  0.00
37  2024-03-20 14:10:00  1950.64  1950.64  ...  13966439.0  0.17  0.01
38  2024-03-20 14:15:00  1950.64  1952.26  ...   5955098.0  0.08  0.00
39  2024-03-20 14:20:00  1950.64  1950.64  ...  17385018.0  0.17  0.01
40  2024-03-20 14:25:00  1949.01  1950.64  ...   4410047.0  0.08  0.00
41  2024-03-20 14:30:00  1950.64  1947.39  ...  14639483.0  0.17  0.01
42  2024-03-20 14:35:00  1949.01  1947.39  ...  13156283.0  0.08  0.01
43  2024-03-20 14:40:00  1949.01  1949.01  ...  12261331.0  0.08  0.01
44  2024-03-20 14:45:00  1949.01  1950.64  ...  10861743.0  0.08  0.01
45  2024-03-20 14:50:00  1949.01  1950.64  ...  22807259.0  0.17  0.01
46  2024-03-20 14:55:00  1950.64  1949.01  ...  12102477.0  0.08  0.01
47  2024-03-20 15:00:00  1949.01  1950.64  ...  24496686.0  0.08  0.01
[48 rows x 11 columns]
```

##### 日内分时数据-东财

接口: stock_intraday_em

目标地址: https://quote.eastmoney.com/f1.html?newcode=0.000001

描述: 东方财富-分时数据

限量: 单次返回指定股票最近一个交易日的分时数据, 包含盘前数据

输入参数

| 名称         | 类型  | 描述                                  |
|------------|-----|-------------------------------------|
| symbol     | str | symbol="000001"; 股票代码               |

输出参数

| 名称    | 类型      | 描述 |
|-------|---------|----|
| 时间    | object  | -  |
| 成交价   | float64 | -  |
| 手数    | int64   | -  |
| 买卖盘性质 | object  | -  |

接口示例

```python
import akshare as ak

stock_intraday_em_df = ak.stock_intraday_em(symbol="000001")
print(stock_intraday_em_df)
```

数据示例

```
         时间    成交价    手数 买卖盘性质
0     09:15:00  10.60    11   中性盘
1     09:15:09  10.57  1547   中性盘
2     09:15:18  10.57  1549   中性盘
3     09:15:27  10.57  1549   中性盘
4     09:15:36  10.57  1552   中性盘
        ...    ...   ...   ...
4396  14:56:51  10.50    30    买盘
4397  14:56:54  10.49   109    卖盘
4398  14:56:57  10.49    27    卖盘
4399  14:57:00  10.50    68    买盘
4400  15:00:00  10.50  8338    买盘
[4401 rows x 4 columns]
```

##### 日内分时数据-新浪

接口: stock_intraday_sina

目标地址: https://vip.stock.finance.sina.com.cn/quotes_service/view/cn_bill.php?symbol=sz000001

描述: 新浪财经-日内分时数据

限量: 单次返回指定交易日的分时数据；只能获取近期的数据，此处仅返回大单数据（成交量大于等于: 400手）

输入参数

| 名称     | 类型  | 描述                            |
|--------|-----|-------------------------------|
| symbol | str | symbol="sz000001"; 带市场标识的股票代码 |
| date   | str | date="20240321"; 交易日          |

输出参数

| 名称         | 类型      | 描述            |
|------------|---------|---------------|
| symbol     | object  | -             |
| name       | object  | -             |
| ticktime   | object  | -             |
| price      | float64 | -             |
| volume     | int64   | 注意单位: 股       |
| prev_price | float64 | -             |
| kind       | object  | D 表示卖盘，表示 是买盘 |

接口示例

```python
import akshare as ak

stock_intraday_sina_df = ak.stock_intraday_sina(symbol="sz000001", date="20240321")
print(stock_intraday_sina_df)
```

数据示例

```
        symbol  name  ticktime  price   volume  prev_price kind
0    sz000001  平安银行  09:25:00  10.45   437400        0.00    U
1    sz000001  平安银行  09:30:00  10.44    29100       10.45    D
2    sz000001  平安银行  09:30:03  10.45   356400       10.44    U
3    sz000001  平安银行  09:30:06  10.45    65500       10.45    D
4    sz000001  平安银行  09:30:09  10.46    35800       10.45    U
..        ...   ...       ...    ...      ...         ...  ...
818  sz000001  平安银行  14:56:03  10.46    22100       10.46    D
819  sz000001  平安银行  14:56:18  10.47    20700       10.46    U
820  sz000001  平安银行  14:56:24  10.47   156000       10.47    U
821  sz000001  平安银行  14:56:45  10.46    78900       10.47    D
822  sz000001  平安银行  15:00:00  10.47  1472200       10.46    E
[823 rows x 7 columns]
```

##### 盘前数据

接口: stock_zh_a_hist_pre_min_em

目标地址: https://quote.eastmoney.com/concept/sh603777.html

描述: 东方财富-股票行情-盘前数据

限量: 单次返回指定 symbol 的最近一个交易日的股票分钟数据, 包含盘前分钟数据

输入参数

| 名称         | 类型  | 描述                                  |
|------------|-----|-------------------------------------|
| symbol     | str | symbol="000001"; 股票代码               |
| start_time | str | start_time="09:00:00"; 时间; 默认返回所有数据 |
| end_time   | str | end_time="15:40:00"; 时间; 默认返回所有数据   |

输出参数

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 时间  | object  | -       |
| 开盘  | float64 | -       |
| 收盘  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 成交量 | float64 | 注意单位: 手 |
| 成交额 | float64 | -       |
| 最新价 | float64 | -       |

接口示例

```python
import akshare as ak

stock_zh_a_hist_pre_min_em_df = ak.stock_zh_a_hist_pre_min_em(symbol="000001", start_time="09:00:00", end_time="15:40:00")
print(stock_zh_a_hist_pre_min_em_df)
```

数据示例

```
          时间             开盘  收盘   最高  最低    成交量    成交额  最新价
0    2023-12-11 09:15:00  9.30  9.30  9.30  9.30     0        0.0  9.300
1    2023-12-11 09:16:00  9.25  9.28  9.28  9.25     0        0.0  9.300
2    2023-12-11 09:17:00  9.28  9.28  9.28  9.28     0        0.0  9.300
3    2023-12-11 09:18:00  9.28  9.28  9.28  9.28     0        0.0  9.300
4    2023-12-11 09:19:00  9.28  9.27  9.28  9.27     0        0.0  9.300
..                   ...   ...   ...   ...   ...   ...        ...    ...
251  2023-12-11 14:56:00  9.35  9.35  9.35  9.34  7793  7283721.0  9.214
252  2023-12-11 14:57:00  9.34  9.36  9.36  9.34  5169  4833563.0  9.215
253  2023-12-11 14:58:00  9.36  9.36  9.36  9.36    22    20592.0  9.215
254  2023-12-11 14:59:00  9.36  9.36  9.36  9.36     0        0.0  9.215
255  2023-12-11 15:00:00  9.35  9.35  9.35  9.35  7138  6674030.0  9.216
[256 rows x 8 columns]
```

#### 历史分笔数据

##### 腾讯财经

接口: stock_zh_a_tick_tx

目标地址: http://gu.qq.com/sz300494/gp/detail(示例)

描述: 每个交易日 16:00 提供当日数据; 如遇到数据缺失, 请使用 **ak.stock_zh_a_tick_163()** 接口(注意数据会有一定差异)

限量: 单次返回最近交易日的历史分笔行情数据

输入参数-历史行情数据

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="sh600000"     |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 成交时间 | object  | -       |
| 成交价格 | float64 | 注意单位: 元 |
| 价格变动 | float64 | 注意单位: 元 |
| 成交量  | int32   | 注意单位: 手 |
| 成交额  | int32   | 注意单位: 元 |
| 性质   | object  | 买卖盘标记   |

接口示例

```python
import akshare as ak

stock_zh_a_tick_tx_js_df = ak.stock_zh_a_tick_tx_js(symbol="sz000001")
print(stock_zh_a_tick_tx_js_df)
```

数据示例

```
      成交时间  成交价格  价格变动  成交量  成交额 性质
0     09:25:04  3.45  0.01       4    1380   卖盘
1     09:30:02  3.46  0.01       1     346   买盘
2     09:30:08  3.46  0.00      20    6920   买盘
3     09:30:20  3.46  0.00      28    9688   买盘
4     09:30:23  3.46  0.00      25    8665   买盘
        ...   ...   ...     ...     ...  ..
1405  14:59:29  3.51 -0.02      14    4914   卖盘
1406  14:59:50  3.51  0.00       5    1755   卖盘
1407  14:59:50  3.52  0.01      69   24288   买盘
1408  14:59:59  3.51 -0.01     100   35100   卖盘
1409  15:00:02  3.52  0.01      10    3520   买盘
```


#### 同行比较

##### 成长性比较

接口: stock_zh_growth_comparison_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=000895&color=b#/thbj/czxbj

描述: 东方财富-行情中心-同行比较-成长性比较

限量: 单次返回全部数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="SZ000895"     |

输出参数

| 名称               | 类型      | 描述 |
|------------------|---------|----|
| 代码               | object  | -  |
| 简称               | object  | -  |
| 基本每股收益增长率-3年复合   | float64 | -  |
| 基本每股收益增长率-24A    | float64 | -  |
| 基本每股收益增长率-TTM    | float64 | -  |
| 基本每股收益增长率-25E    | float64 | -  |
| 基本每股收益增长率-26E    | float64 | -  |
| 基本每股收益增长率-27E    | float64 | -  |
| 营业收入增长率-3年复合     | float64 | -  |
| 营业收入增长率-24A      | float64 | -  |
| 营业收入增长率-TTM      | float64 | -  |
| 营业收入增长率-25E      | float64 | -  |
| 营业收入增长率-26E      | float64 | -  |
| 营业收入增长率-27E      | float64 | -  |
| 净利润增长率-3年复合      | float64 | -  |
| 净利润增长率-24A       | float64 | -  |
| 净利润增长率-TTM       | float64 | -  |
| 净利润增长率-25E       | float64 | -  |
| 净利润增长率-26E       | float64 | -  |
| 净利润增长率-27E       | float64 | -  |
| 基本每股收益增长率-3年复合排名 | float64 | -  |

接口示例

```python
import akshare as ak

stock_zh_growth_comparison_em_df = ak.stock_zh_growth_comparison_em(symbol="SZ000895")
print(stock_zh_growth_comparison_em_df)
```

数据示例

```
       代码    简称  基本每股收益增长率-3年复合  ...  净利润增长率-26E  净利润增长率-27E  基本每股收益增长率-3年复合排名
0    行业中值  行业中值       -8.790000  ...   21.290000   16.135000               NaN
1    行业平均  行业平均      -31.127395  ...   57.622875   18.847125               NaN
2  600530  交大昂立       81.710000  ...         NaN         NaN               1.0
3  600186  莲花控股       58.740000  ...   28.700000   22.480000               2.0
4  600962  国投中鲁       51.860000  ...         NaN         NaN               3.0
5  600737  中粮糖业       48.850000  ...   39.030000   22.690000               4.0
6  003000  劲仔食品       46.100000  ...   22.070000   17.520000               5.0
7  000895  双汇发展        0.840000  ...    4.320000    3.980000              38.0
[8 rows x 21 columns]
```

##### 估值比较

接口: stock_zh_valuation_comparison_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=000895&color=b#/thbj/gzbj

描述: 东方财富-行情中心-同行比较-估值比较

限量: 单次返回全部数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="SZ000895"     |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| 排名            | object  | -  |
| 代码            | object  | -  |
| 简称            | object  | -  |
| PEG           | float64 | -  |
| 市盈率-24A       | float64 | -  |
| 市盈率-TTM       | float64 | -  |
| 市盈率-25E       | float64 | -  |
| 市盈率-26E       | float64 | -  |
| 市盈率-27E       | float64 | -  |
| 市销率-24A       | float64 | -  |
| 市销率-TTM       | float64 | -  |
| 市销率-25E       | float64 | -  |
| 市销率-26E       | float64 | -  |
| 市销率-27E       | float64 | -  |
| 市净率-24A       | float64 | -  |
| 市净率-MRQ       | float64 | -  |
| 市现率1-24A      | float64 | -  |
| 市现率1-TTM      | float64 | -  |
| 市现率2-24A      | float64 | -  |
| 市现率2-TTM      | float64 | -  |
| EV/EBITDA-24A | float64 | -  |

接口示例

```python
import akshare as ak

stock_zh_valuation_comparison_em_df = ak.stock_zh_valuation_comparison_em(symbol="SZ000895")
print(stock_zh_valuation_comparison_em_df)
```

数据示例

```
         排名      代码    简称  ...     市现率2-24A     市现率2-TTM  EV/EBITDA-24A
0  42.0/120  000895  双汇发展  ...    29.790457 -1045.264127      12.503574
1       nan    行业平均  行业平均  ...  1036.299305   -81.550319      12.794686
2       nan    行业中值  行业中值  ...   -11.801449   -13.610393      18.565517
3       1.0  920786  骑士乳业  ...   -10.676185   -23.320786      14.613055
4       2.0  002852   道道全  ...    94.382638   -14.822839      10.933433
5       3.0  002840  华统股份  ...   -62.597528    39.150932      19.671557
6       4.0  605077  华康股份  ...    -2.588921   -50.802629      15.723042
7       5.0  002286   保龄宝  ...   257.860564  -114.930447      12.453163
[8 rows x 20 columns]
```

##### 杜邦分析比较

接口: stock_zh_dupont_comparison_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=000895&color=b#/thbj/dbfxbj

描述: 东方财富-行情中心-同行比较-杜邦分析比较

限量: 单次返回全部数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="SZ000895"     |

输出参数

| 名称          | 类型      | 描述 |
|-------------|---------|----|
| 代码          | object  | -  |
| 简称          | object  | -  |
| ROE-3年平均    | float64 | -  |
| ROE-22A     | float64 | -  |
| ROE-23A     | float64 | -  |
| ROE-24A     | float64 | -  |
| 净利率-3年平均    | float64 | -  |
| 净利率-22A     | float64 | -  |
| 净利率-23A     | float64 | -  |
| 净利率-24A     | float64 | -  |
| 总资产周转率-3年平均 | float64 | -  |
| 总资产周转率-22A  | float64 | -  |
| 总资产周转率-23A  | float64 | -  |
| 总资产周转率-24A  | float64 | -  |
| 权益乘数-3年平均   | float64 | -  |
| 权益乘数-22A    | float64 | -  |
| 权益乘数-23A    | float64 | -  |
| 权益乘数-24A    | float64 | -  |
| ROE-3年平均排名  | float64 | -  |


接口示例

```python
import akshare as ak

stock_zh_dupont_comparison_em_df = ak.stock_zh_dupont_comparison_em(symbol="SZ000895")
print(stock_zh_dupont_comparison_em_df)
```

数据示例

```
    代码    简称  ROE-3年平均  ROE-22A  ...  权益乘数-22A  权益乘数-23A  权益乘数-24A  ROE-3年平均排名
0    行业平均  行业平均      5.70     5.51  ...    191.76    189.10   185.080         NaN
1    行业中值  行业中值      7.71     7.89  ...    149.35    142.50   143.105         NaN
2  605499  东鹏饮料     38.09    30.97  ...    234.37    232.62   294.820         1.0
3  002847  盐津铺子     36.48    30.03  ...    213.82    196.34   203.650         2.0
4  000895  双汇发展     24.21    25.17  ...    164.15    173.44   174.840         3.0
5  603262  技源集团     24.02    28.06  ...    152.21    132.11   125.360         4.0
6  603288  海天味业     22.24    24.89  ...    126.69    132.34   130.110         5.0
7  000848  承德露露     21.92    23.53  ...    136.51    133.85   133.510         6.0
[8 rows x 19 columns]
```

##### 公司规模

接口: stock_zh_scale_comparison_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=000895&color=b#/thbj/gsgm

描述: 东方财富-行情中心-同行比较-公司规模

限量: 单次返回全部数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="SZ000895"     |

输出参数

| 名称     | 类型      | 描述 |
|--------|---------|----|
| 代码     | object  | -  |
| 简称     | object  | -  |
| 总市值    | float64 | -  |
| 总市值排名  | int64   | -  |
| 流通市值   | float64 | -  |
| 流通市值排名 | int64   | -  |
| 营业收入   | float64 | -  |
| 营业收入排名 | int64   | -  |
| 净利润    | float64 | -  |
| 净利润排名  | int64   | -  |

接口示例

```python
import akshare as ak

stock_zh_scale_comparison_em_df = ak.stock_zh_scale_comparison_em(symbol="SZ000895")
print(stock_zh_scale_comparison_em_df)
```

数据示例

```
       代码    简称           总市值  总市值排名    流通市值  流通市值排名          营业收入  营业收入排名           净利润  净利润排名
0  000895  双汇发展  8.685906e+10      5  868.48       4  2.850309e+10       3  2.351218e+09      4
```

### A股-CDR

#### 历史行情数据

接口: stock_zh_a_cdr_daily

目标地址: https://finance.sina.com.cn/realstock/company/sh689009/nc.shtml

描述: 上海证券交易所-科创板-CDR

限量: 单次返回指定 CDR 的日频率数据, 分钟历史行情数据可以通过 stock_zh_a_minute 获取

名词解释:

1. [Investopedia-CDR](https://www.investopedia.com/terms/c/cdr.asp)
2. [百度百科-中国存托凭证](https://baike.baidu.com/item/%E4%B8%AD%E5%9B%BD%E5%AD%98%E6%89%98%E5%87%AD%E8%AF%81/2489906?fr=aladdin)

输入参数

| 名称         | 类型  | 描述                          |
|------------|-----|-----------------------------|
| symbol     | str | symbol='sh689009'; CDR 股票代码 |
| start_date | str | start_date='20201103'       |
| end_date   | str | end_date='20201116'         |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| date   | object  | 交易日     |
| open   | float64 | -       |
| high   | float64 | -       |
| low    | float64 | -       |
| close  | float64 | -       |
| volume | float64 | 注意单位: 手 |

接口示例

```python
import akshare as ak

stock_zh_a_cdr_daily_df = ak.stock_zh_a_cdr_daily(symbol='sh689009', start_date='20201103', end_date='20201116')
print(stock_zh_a_cdr_daily_df)
```

数据示例

```
           date   open   high    low  close      volume
0    2020-10-29  33.00  49.80  33.00  38.50  40954922.0
1    2020-10-30  40.02  51.56  40.02  47.60  33600551.0
2    2020-11-02  50.20  56.78  48.81  56.77  27193402.0
3    2020-11-03  56.50  59.55  53.36  57.39  25121445.0
4    2020-11-04  57.45  57.80  51.90  54.40  20846450.0
..          ...    ...    ...    ...    ...         ...
265  2021-11-30  60.50  61.20  60.36  60.84   1535244.0
266  2021-12-01  60.99  61.00  59.78  59.78   2116728.0
267  2021-12-02  59.67  59.77  57.66  57.73   2421344.0
268  2021-12-03  57.55  58.78  57.50  58.60   1709167.0
269  2021-12-06  58.99  58.99  56.30  56.75   1719351.0
```

### B股

#### 实时行情数据

##### 实时行情数据-东财

接口: stock_zh_b_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#hs_b_board

描述: 东方财富网-实时行情数据

限量: 单次返回所有 B 股上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 最新价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 涨跌额     | float64 | -       |
| 成交量     | float64 | 注意单位: 手 |
| 成交额     | float64 | 注意单位: 元 |
| 振幅      | float64 | 注意单位: % |
| 最高      | float64 | -       |
| 最低      | float64 | -       |
| 今开      | float64 | -       |
| 昨收      | float64 | -       |
| 量比      | float64 | -       |
| 换手率     | float64 | 注意单位: % |
| 市盈率-动态  | float64 | -       |
| 市净率     | float64 | -       |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 涨速      | float64 | -       |
| 5分钟涨跌   | float64 | 注意单位: % |
| 60日涨跌幅  | float64 | 注意单位: % |
| 年初至今涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_zh_b_spot_em_df = ak.stock_zh_b_spot_em()
print(stock_zh_b_spot_em_df)
```

数据示例

```
      序号      代码     名称    最新价  ...    涨速  5分钟涨跌  60日涨跌幅  年初至今涨跌幅
0      1  900953    凯马Ｂ  0.330  ...  0.00   0.92    0.00    -2.65
1      2  200625  长  安Ｂ  4.420  ...  0.00   0.23   20.11     1.38
2      3  900914   锦在线B  0.702  ...  0.14   0.29   13.96    16.42
3      4  200761   本钢板Ｂ  2.670  ...  0.00   0.38    8.54     1.14
4      5  900957   凌云Ｂ股  0.568  ...  0.00  -0.35   -6.73    -9.98
..   ...     ...    ...    ...  ...   ...    ...     ...      ...
109  110  200017   深中华B  1.210  ... -1.63  -0.82  -14.18   -15.38
110  111  900936   鄂资Ｂ股  2.630  ...  0.00  -0.53   30.46    27.92
111  112  900919   退市庭B  0.026  ...  0.00   0.00  -78.15   -86.32
112  113  900906   中毅达B  0.571  ...  0.71  -0.70   17.73    13.07
113  114  900948   伊泰Ｂ股  1.465  ... -0.61   0.14   47.38    67.81
```

##### 实时行情数据-新浪

接口: stock_zh_b_spot

目标地址: http://vip.stock.finance.sina.com.cn/mkt/#hs_b

描述: B 股数据是从新浪财经获取的数据, 重复运行本函数会被新浪暂时封 IP, 建议增加时间间隔

限量: 单次返回所有 B 股上市公司的实时行情数据

输入参数-实时行情数据

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数-实时行情数据

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 代码  | object  | -       |
| 名称  | object  | -       |
| 最新价 | float64 | -       |
| 涨跌额 | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 买入  | float64 | -       |
| 卖出  | float64 | -       |
| 昨收  | float64 | -       |
| 今开  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 成交量 | float64 | 注意单位: 股 |
| 成交额 | float64 | 注意单位: 元 |

接口示例-实时行情数据

```python
import akshare as ak

stock_zh_b_spot_df = ak.stock_zh_b_spot()
print(stock_zh_b_spot_df)
```

数据示例-实时行情数据

```
    代码        名称     最新价    涨跌额  ...      最高      最低         成交量         成交额
0   sh900901  云赛Ｂ股   0.451  0.001  ...   0.459   0.447   869400.0    395346.0
1   sh900902  市北B股   0.290 -0.004  ...   0.294   0.290   858200.0    251011.0
2   sh900903  大众Ｂ股   0.290 -0.006  ...   0.296   0.289  1775362.0    518333.0
3   sh900904  神奇B股   0.536 -0.006  ...   0.540   0.531    37060.0     19907.0
4   sh900905  老凤祥Ｂ   3.498 -0.011  ...   3.509   3.495   300400.0   1051414.0
..       ...   ...     ...    ...  ...     ...     ...        ...         ...
85  sz200761  本钢板Ｂ   3.260 -0.120  ...   3.380   3.240  2190800.0   7214306.0
86  sz200771  杭汽轮Ｂ  14.940 -0.760  ...  15.800  14.660   839948.0  12673391.0
87  sz200869  张 裕Ｂ  14.000  0.000  ...  14.000  13.810   176407.0   2461886.0
88  sz200992  中 鲁Ｂ   3.120  0.000  ...   3.130   3.080    17700.0     55129.0
89  sz201872   招港B   8.780 -0.220  ...   9.050   8.750   313566.0   2776833.0
```

#### 历史行情数据

##### 历史行情数据

接口: stock_zh_b_daily

目标地址: https://finance.sina.com.cn/realstock/company/sh900901/nc.shtml

描述: B 股数据是从新浪财经获取的数据, 历史数据按日频率更新

限量: 单次返回指定 B 股上市公司指定日期间的历史行情日频率数据

输入参数

| 名称         | 类型  | 描述                                                                                   |
|------------|-----|--------------------------------------------------------------------------------------|
| symbol     | str | symbol='sh900901'; 股票代码可以在 **ak.stock_zh_b_spot()** 中获取                              |
| start_date | str | start_date='20201103'; 开始查询的日期                                                       |
| end_date   | str | end_date='20201116'; 结束查询的日期                                                         |
| adjust     | str | 默认返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; hfq-factor: 返回后复权因子; qfq-factor: 返回前复权因子 |

**股票数据复权**

1.为何要复权：由于股票存在配股、分拆、合并和发放股息等事件，会导致股价出现较大的缺口。
若使用不复权的价格处理数据、计算各种指标，将会导致它们失去连续性，且使用不复权价格计算收益也会出现错误。
为了保证数据连贯性，常通过前复权和后复权对价格序列进行调整。

2.前复权：保持当前价格不变，将历史价格进行增减，从而使股价连续。
前复权用来看盘非常方便，能一眼看出股价的历史走势，叠加各种技术指标也比较顺畅，是各种行情软件默认的复权方式。
这种方法虽然很常见，但也有两个缺陷需要注意。

2.1 为了保证当前价格不变，每次股票除权除息，均需要重新调整历史价格，因此其历史价格是时变的。
这会导致在不同时点看到的历史前复权价可能出现差异。

2.2 对于有持续分红的公司来说，前复权价可能出现负值。

3.后复权：保证历史价格不变，在每次股票权益事件发生后，调整当前的股票价格。
后复权价格和真实股票价格可能差别较大，不适合用来看盘。
其优点在于，可以被看作投资者的长期财富增长曲线，反映投资者的真实收益率情况。

4.在量化投资研究中普遍采用后复权数据。

输出参数-历史行情数据

| 名称                | 类型      | 描述            |
|-------------------|---------|---------------|
| date              | object  | 交易日           |
| close             | float64 | 收盘价           |
| high              | float64 | 最高价           |
| low               | float64 | 最低价           |
| open              | float64 | 开盘价           |
| volume            | float64 | 成交量; 注意单位: 股  |
| outstanding_share | float64 | 流动股本; 注意单位: 股 |
| turnover          | float64 | 换手率=成交量/流动股本  |

接口示例-历史行情数据(前复权)

```python
import akshare as ak

stock_zh_b_daily_qfq_df = ak.stock_zh_b_daily(symbol="sh900901", start_date="19900103", end_date="20240722", adjust="qfq")
print(stock_zh_b_daily_qfq_df)
```

数据示例-历史行情数据(前复权)

```
           date  open  high  ...     volume  outstanding_share  turnover
0    1992-02-21  0.18  0.23  ...   343000.0          1000000.0  0.343000
1    1992-02-24  0.22  0.23  ...  2896000.0          1000000.0  2.896000
2    1992-02-25  0.20  0.21  ...   181000.0          1000000.0  0.181000
3    1992-02-26  0.20  0.20  ...   305000.0          1000000.0  0.305000
4    1992-02-27  0.20  0.21  ...   330000.0          1000000.0  0.330000
...         ...   ...   ...  ...        ...                ...       ...
7418 2024-07-16  0.49  0.49  ...   540175.0        293370465.0  0.001841
7419 2024-07-17  0.49  0.49  ...   527360.0        293370465.0  0.001798
7420 2024-07-18  0.48  0.48  ...   825069.0        293370465.0  0.002812
7421 2024-07-19  0.48  0.50  ...   902574.0        293370465.0  0.003077
7422 2024-07-22  0.49  0.50  ...   994281.0        293370465.0  0.003389
[7423 rows x 8 columns]
```

接口示例-历史行情数据(后复权)

```python
import akshare as ak

stock_zh_b_daily_hfq_df = ak.stock_zh_b_daily(symbol="sh900901", start_date="19900103", end_date="20240722", adjust="hfq")
print(stock_zh_b_daily_hfq_df)
```

数据示例-历史行情数据(后复权)

```
           date    open    high  ...     volume  outstanding_share  turnover
0    1992-02-21   72.00   92.40  ...   343000.0          1000000.0  0.343000
1    1992-02-24   88.70   92.20  ...  2896000.0          1000000.0  2.896000
2    1992-02-25   80.00   84.00  ...   181000.0          1000000.0  0.181000
3    1992-02-26   81.00   81.00  ...   305000.0          1000000.0  0.305000
4    1992-02-27   80.00   84.00  ...   330000.0          1000000.0  0.330000
...         ...     ...     ...  ...        ...                ...       ...
7418 2024-07-16  196.42  197.63  ...   540175.0        293370465.0  0.001841
7419 2024-07-17  198.03  198.44  ...   527360.0        293370465.0  0.001798
7420 2024-07-18  195.61  196.01  ...   825069.0        293370465.0  0.002812
7421 2024-07-19  195.20  200.05  ...   902574.0        293370465.0  0.003077
7422 2024-07-22  198.44  202.48  ...   994281.0        293370465.0  0.003389
[7423 rows x 8 columns]
```

接口示例-前复权因子

```python
import akshare as ak

qfq_factor_df = ak.stock_zh_b_daily(symbol="sh900901", adjust="qfq-factor")
print(qfq_factor_df)
```

数据示例-前复权因子

```
         date            qfq_factor
0  2024-06-28    1.0000000000000000
1  2023-06-29    1.0128968423207000
2  2022-06-27    1.0235450730903000
3  2021-07-28    1.0414120633892000
4  2020-07-24    1.0621222316638000
5  2019-07-26    1.0776646873343000
6  2018-07-27    1.0933643235806000
7  2017-06-01    1.1126090142905000
8  2016-06-07    1.1260358798118000
9  2007-06-26    1.1339015467992000
10 2006-06-07    1.2472917014791000
11 2005-06-09    1.4454867952888000
12 2004-06-09    1.4781966336173000
13 2003-06-11    1.4917479863778000
14 2002-06-19    1.5062371240633000
15 2001-07-17    1.6596648576557000
16 2000-06-23    1.6787984081928000
17 1998-07-07    1.8889236772532000
18 1996-06-03    2.2667084127038000
19 1995-05-29    2.4933792539742000
20 1994-05-09    2.9920551047690000
21 1993-03-29    3.5280306737986000
22 1992-12-01    4.0414826776136000
23 1992-02-21  404.1482677613642500
24 1900-01-01  404.1482677613642500
```

接口示例-后复权因子

```python
import akshare as ak

hfq_factor_df = ak.stock_zh_b_daily(symbol="sh900901", adjust="hfq-factor")
print(hfq_factor_df)
```

数据示例-后复权因子

```
         date            hfq_factor
0  2024-06-28  404.1482677613642500
1  2023-06-29  399.0023967646893000
2  2022-06-27  394.8514612465220400
3  2021-07-28  388.0771905465590000
4  2020-07-24  380.5101293551464000
5  2019-07-26  375.0222796675719000
6  2018-07-27  369.6373286059189000
7  2017-06-01  363.2437474175006300
8  2016-06-07  358.9124245569568600
9  2007-06-26  356.4227149192948300
10 2006-06-07  324.0206499266317000
11 2005-06-09  279.5931924653872300
12 2004-06-09  273.4062969500634000
13 2003-06-11  270.9226165893487000
14 2002-06-19  268.3164963237175600
15 2001-07-17  243.5119752624272000
16 2000-06-23  240.7366279292711800
17 1998-07-07  213.9569071149911000
18 1996-06-03  178.2974225958258000
19 1995-05-29  162.0885659962053200
20 1994-05-09  135.0738049968378200
21 1993-03-29  114.5535016922676000
22 1992-12-01  100.0000000000000000
23 1992-02-21    1.0000000000000000
24 1900-01-01    1.0000000000000000
```

##### 分时数据

接口: stock_zh_b_minute

目标地址: http://finance.sina.com.cn/realstock/company/sh900901/nc.shtml

描述: 新浪财经 B 股股票或者指数的分时数据，目前可以获取 1, 5, 15, 30, 60 分钟的数据频率, 可以指定是否复权

限量: 单次返回指定股票或指数的指定频率的最近交易日的历史分时行情数据

输入参数

| 名称     | 类型  | 描述                                                         |
|--------|-----|------------------------------------------------------------|
| symbol | str | symbol='sh900901'; 同日频率数据接口                                |
| period | str | period='1'; 获取 1, 5, 15, 30, 60 分钟的数据频率                    |
| adjust | str | adjust=""; 默认为空: 返回不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| day    | object  | -   |
| open   | float64 | -   |
| high   | float64 | -   |
| low    | float64 | -   |
| close  | float64 | -   |
| volume | float64 | -   |

接口示例

```python
import akshare as ak

stock_zh_b_minute_df = ak.stock_zh_b_minute(symbol='sh900901', period='1', adjust="qfq")
print(stock_zh_b_minute_df)
```

数据示例

```
                      day      open      high       low     close volume
0     2024-06-28 14:41:00  0.472987  0.472987  0.472987  0.472987    100
1     2024-06-28 14:42:00  0.470996  0.470996  0.470000  0.470000  12000
2     2024-06-28 14:44:00  0.470996  0.470996  0.470996  0.470996   5000
3     2024-06-28 14:45:00  0.470996  0.470996  0.470996  0.470996  43600
4     2024-06-28 14:48:00  0.470000  0.470000  0.469004  0.469004  16000
...                   ...       ...       ...       ...       ...    ...
1965  2024-07-22 14:43:00  0.498994  0.498994  0.498994  0.498994   5000
1966  2024-07-22 14:48:00  0.497988  0.497988  0.497988  0.497988   1300
1967  2024-07-22 14:56:00  0.496982  0.496982  0.496982  0.496982    200
1968  2024-07-22 14:57:00  0.497988  0.497988  0.497988  0.497988   5100
1969  2024-07-22 15:00:00  0.500000  0.500000  0.500000  0.500000    400
[1970 rows x 6 columns]
```

### 次新股

接口: stock_zh_a_new

目标地址: http://vip.stock.finance.sina.com.cn/mkt/#new_stock

描述: 新浪财经-行情中心-沪深股市-次新股

限量: 单次返回所有次新股行情数据, 由于次新股名单随着交易日变化而变化，只能获取最近交易日的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称            | 类型      | 描述   |
|---------------|---------|------|
| symbol        | object  | 新浪代码 |
| code          | object  | 股票代码 |
| name          | object  | 股票简称 |
| open          | float64 | 开盘价  |
| high          | float64 | 最高价  |
| low           | float64 | 最低价  |
| volume        | int64   | 成交量  |
| amount        | int64   | 成交额  |
| mktcap        | float64 | 市值   |
| turnoverratio | float64 | 换手率  |

接口示例

```python
import akshare as ak

stock_zh_a_new_df = ak.stock_zh_a_new()
print(stock_zh_a_new_df)
```

数据示例

```
     symbol    code  name  ...      amount        mktcap  turnoverratio
0   sh601728  601728  中国电信  ...  2078538311  4.146314e+07        7.91425
1   sh601825  601825  沪农商行  ...   511698399  7.657689e+06        6.67390
2   sh601921  601921  浙版传媒  ...   218150606  2.284444e+06        9.66828
3   sh605028  605028  世茂能源  ...   248080378  3.996800e+05       24.98649
4   sh605056  605056  咸亨国际  ...    60691275  7.736193e+05        7.80149
..       ...     ...   ...  ...         ...           ...            ...
79  sz301052  301052  果麦文化  ...   231553223  2.541569e+05       38.94978
80  sz301053  301053  远信工业  ...   214016261  3.033018e+05       33.99790
81  sz301055  301055  C张小泉  ...   679460624  5.285280e+05       64.66610
82  sz301056  301056   C森赫  ...   423894365  3.694995e+05       48.84555
83  sz301058  301058  中粮工科  ...           0  0.000000e+00        0.00000
```

### 股市日历

#### 公司动态

接口: stock_gsrl_gsdt_em

目标地址: https://data.eastmoney.com/gsrl/gsdt.html

描述: 东方财富网-数据中心-股市日历-公司动态

限量: 单次返回指定交易日的数据

输入参数

| 名称   | 类型  | 描述                   |
|------|-----|----------------------|
| date | str | date="20230808"; 交易日 |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 序号   | int64  | -  |
| 代码   | object | -  |
| 简称   | object | -  |
| 事件类型 | object | -  |
| 具体事项 | object | -  |
| 交易日  | object | -  |

接口示例

```python
import akshare as ak

stock_gsrl_gsdt_em_df = ak.stock_gsrl_gsdt_em(date="20230808")
print(stock_gsrl_gsdt_em_df)
```

数据示例

```
    序号      代码  ...                                               具体事项         交易日
0    1  000504  ...  为盘活闲置资产,优化资产结构,提高资产运营效率,南华生物医药股份有限公司(以下简称“公司”)...  2023-08-08
1    2  000669  ...  公司于近日收到新能国际告知函,获悉北京市第一中级人民法院将于2023年8月3日10时至202...  2023-08-08
2    3  000707  ...  根据2022年10月13日新修订的《企业集团财务公司管理办法》,监管部门要求财务公司需在20...  2023-08-08
3    4  000778  ...  新兴铸管股份有限公司(以下简称“公司”或“本公司”)与长江生态环保集团有限公司(以下简称“长...  2023-08-08
4    5  000790  ...  为加速拓展公司在医药产业的战略布局,提高公司抵御市场变化的抗风险能力,整合优势经营资源,丰富...  2023-08-08
..  ..     ...  ...                                                ...         ...
93  94  688360  ...  德马科技集团股份有限公司(以下简称“公司”或“上市公司”)拟通过发行股份及支付现金方式购买江...  2023-08-08
94  95  688516  ...  为支持各子公司业务发展的融资需求,公司为全资子公司智能装备向商业银行申请总额不超过4.5亿元...  2023-08-08
95  96  831152  ...  为了满足项目建设需要,公司拟以现金方式向全资子公司“昆工恒达(云南)新能源科技有限公司”(以...  2023-08-08
96  97  871694  ...  公司根据战略发展规划及生产经营需要,拟在阿拉伯联合酋长国投资新设全资子公司中裕阿联酋管道制造...  2023-08-08
97  98  872351  ...  华光源海国际物流集团股份有限公司(以下简称“公司”)因业务发展需要,经公司董事长审批,拟通过...  2023-08-08
[98 rows x 6 columns]
```

### 风险警示板

接口: stock_zh_a_st_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#st_board

描述: 东方财富网-行情中心-沪深个股-风险警示板

限量: 单次返回当前交易日风险警示板的所有股票的行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 涨跌额    | float64 | -       |
| 成交量    | float64 | -       |
| 成交额    | float64 | -       |
| 振幅     | float64 | 注意单位: % |
| 最高     | float64 | -       |
| 最低     | float64 | -       |
| 今开     | float64 | -       |
| 昨收     | float64 | -       |
| 量比     | float64 | -       |
| 换手率    | float64 | 注意单位: % |
| 市盈率-动态 | float64 | -       |
| 市净率    | float64 | -       |

接口示例

```python
import akshare as ak

stock_zh_a_st_em_df = ak.stock_zh_a_st_em()
print(stock_zh_a_st_em_df)
```

数据示例

```
      序号      代码     名称    最新价    涨跌幅  ...     昨收    量比    换手率   市盈率-动态    市净率
0      1  300313  *ST天山   7.61  10.45  ...   6.89  7.84  11.96   -90.65  33.49
1      2  300167  ST迪威迅   3.19   7.41  ...   2.97  3.86   9.31    -5.82  54.39
2      3  002569   ST步森   6.90   5.02  ...   6.57  0.88   0.93   -27.27   7.05
3      4  000996  *ST中期   5.24   5.01  ...   4.99  0.81   4.47  6823.87   3.73
4      5  600589  *ST榕泰   5.48   4.98  ...   5.22  1.71   4.07   -24.53  -5.13
..   ...     ...    ...    ...    ...  ...    ...   ...    ...      ...    ...
113  114  000615  *ST美谷   4.10  -4.87  ...   4.31  1.25   5.02   -37.97 -15.44
114  115  000697  *ST炼石   4.82  -4.93  ...   5.07  2.60   1.62    -9.97  -5.81
115  116  000889   ST中嘉   2.12  -4.93  ...   2.23  0.40   0.91   -68.39   8.04
116  117  000752  *ST西发   9.01  -4.96  ...   9.48  0.19   1.26   -64.79 -48.62
117  118  600766  *ST园城  12.34  -5.00  ...  12.99  1.56   1.05  1372.29  47.59
[118 rows x 17 columns]
```

### 新股

接口: stock_zh_a_new_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#newshares

描述: 东方财富网-行情中心-沪深个股-新股

限量: 单次返回当前交易日新股板块的所有股票的行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 涨跌额    | float64 | -       |
| 成交量    | float64 | -       |
| 成交额    | float64 | -       |
| 振幅     | float64 | 注意单位: % |
| 最高     | float64 | -       |
| 最低     | float64 | -       |
| 今开     | float64 | -       |
| 昨收     | float64 | -       |
| 量比     | float64 | -       |
| 换手率    | float64 | 注意单位: % |
| 市盈率-动态 | float64 | -       |
| 市净率    | float64 | -       |

接口示例

```python
import akshare as ak

stock_zh_a_new_em_df = ak.stock_zh_a_new_em()
print(stock_zh_a_new_em_df)
```

数据示例

```
      序号      代码    名称     最新价   涨跌幅  ...      昨收    量比    换手率  市盈率-动态   市净率
0      1  301559  N中集环   23.95 -1.11  ...   24.22   NaN  50.66   20.05  2.92
1      2  688657   C浩辰   89.26  0.42  ...   88.89  0.62  29.22   76.48  2.91
2      3  688719   C爱科   60.33  0.05  ...   60.30  0.44  18.92   45.79  2.86
3      4  301558   C三态   14.72  1.73  ...   14.47  0.66  38.74   71.25  8.58
4      5  603276  恒兴新材   24.17 -0.62  ...   24.32  0.40  12.34   43.70  2.24
..   ...     ...   ...     ...   ...  ...     ...   ...    ...     ...   ...
261  262  688152  麒麟信安   81.80  2.06  ...   80.15  1.26   9.45 -225.22  5.02
262  263  688372  伟测科技  103.70  2.65  ...  101.02  0.98   5.67   83.07  4.95
263  264  688291   金橙子   30.13  4.22  ...   28.91  1.09   6.76   60.43  3.37
264  265  001322  箭牌家居   13.56  1.12  ...   13.41  0.82   3.14   38.40  2.75
265  266  688426  康为世纪   32.25  3.80  ...   31.07  1.76   6.44 -183.28  2.10
[266 rows x 17 columns]
```

### 新股上市首日

接口: stock_xgsr_ths

目标地址: https://data.10jqka.com.cn/ipo/xgsr/

描述: 同花顺-数据中心-新股数据-新股上市首日

限量: 单次返回当前交易日的所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型      | 描述 |
|-------|---------|----|
| 序号    | int64   | -  |
| 股票代码  | object  | -  |
| 股票简称  | object  | -  |
| 上市日期  | object  | -  |
| 发行价   | float64 | -  |
| 最新价   | float64 | -  |
| 首日开盘价 | float64 | -  |
| 首日收盘价 | float64 | -  |
| 首日最高价 | float64 | -  |
| 首日最低价 | float64 | -  |
| 首日涨跌幅 | float64 | -  |
| 是否破发  | object  | -  |

接口示例

```python
import akshare as ak

stock_xgsr_ths_df = ak.stock_xgsr_ths()
print(stock_xgsr_ths_df)
```

数据示例

```
        序号  股票代码 股票简称  上市日期  ...   首日最高价  首日最低价 首日涨跌幅 是否破发
0        1  920019  铜冠矿建  2024-10-11  ...   55.55  33.33   7.3141     否
1        2  001279  强邦新材  2024-10-11  ...  245.00  87.12  17.3895     否
2        3  301618  长联科技  2024-09-30  ...  408.00  99.00  17.0398     否
3        4  688615  合合信息  2024-09-26  ...  116.30  88.93   1.0591     否
4        5  301551  无线传媒  2024-09-26  ...   39.39  29.11   3.1883     否
...    ...     ...   ...         ...  ...     ...    ...      ...   ...
3780  3781  300042  朗科科技  2010-01-08  ...   56.50  51.00   0.3449     是
3781  3782  601117  中国化学  2010-01-07  ...    5.86   5.65   0.0589     否
3782  3783  002329  皇氏集团  2010-01-06  ...   29.76  27.68   0.4308     是
3783  3784  002330   得利斯  2010-01-06  ...   23.98  21.00   0.8194     是
3784  3785  002331  皖通科技  2010-01-06  ...   45.00  40.62   0.6011     是
[3785 rows x 12 columns]
```

### IPO 受益股

接口: stock_ipo_benefit_ths

目标地址: https://data.10jqka.com.cn/ipo/syg/

描述: 同花顺-数据中心-新股数据-IPO受益股

限量: 单次返回当前交易日的所有数据; 该数据每周更新一次, 返回最近一周的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 收盘价    | float64 | 注意单位: 元 |
| 涨跌幅    | float64 | 注意单位: % |
| 市值     | object  | 注意单位: 元 |
| 参股家数   | int64   | -       |
| 投资总额   | object  | 注意单位: 元 |
| 投资占市值比 | float64 | 注意单位: % |
| 参股对象   | object  | -       |

接口示例

```python
import akshare as ak

stock_ipo_benefit_ths_df = ak.stock_ipo_benefit_ths()
print(stock_ipo_benefit_ths_df)
```

数据示例

```
    序号  股票代码  股票简称 收盘价  涨跌幅  市值  参股家数    投资总额  投资占市值比 参股对象
0      1  601601  中国太保  29.12 -0.206  1993.28亿     2  34.17亿    1.71   明细
1      2  600716  凤凰股份   3.82 -1.036    35.76亿     1   4.76亿   13.33   明细
2      3  600649  城投控股   4.08 -0.730   103.21亿     1   4.67亿    4.52   明细
3      4  601963  重庆银行   8.59 -1.151   162.82亿     1   3.79亿    2.33   明细
4      5  600639  浦东金桥  11.10 -0.893    94.38亿     2   3.18亿    3.37   明细
..   ...     ...   ...    ...    ...       ...   ...     ...     ...  ...
107  108  000685  中山公用   8.59 -0.116   126.71亿     1      0万    0.00   明细
108  109  600052  浙江广厦   3.98 -2.690    33.60亿     1      0万    0.00   明细
109  110  600261  阳光照明   3.62 -0.550    52.57亿     1      0万    0.00   明细
110  111  600496  精工钢构   3.90 -2.500    78.50亿     1      0万    0.00   明细
111  112  603987   康德莱  20.00  0.200    88.31亿     1      0万    0.00   明细
```

### 两网及退市

接口: stock_zh_a_stop_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#staq_net_board

描述: 东方财富网-行情中心-沪深个股-两网及退市

限量: 单次返回当前交易日两网及退市的所有股票的行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 涨跌额    | float64 | -       |
| 成交量    | float64 | -       |
| 成交额    | float64 | -       |
| 振幅     | float64 | 注意单位: % |
| 最高     | float64 | -       |
| 最低     | float64 | -       |
| 今开     | float64 | -       |
| 昨收     | float64 | -       |
| 量比     | float64 | -       |
| 换手率    | float64 | 注意单位: % |
| 市盈率-动态 | float64 | -       |
| 市净率    | float64 | -       |

接口示例

```python
import akshare as ak

stock_zh_a_stop_em_df = ak.stock_zh_a_stop_em()
print(stock_zh_a_stop_em_df)
```

数据示例

```
      序号 代码    名称      最新价 涨跌幅 涨跌额  ... 今开 昨收 量比 换手率 市盈率-动态市净率
0      1  400010  鹫  峰 5  1.90  4.97  0.09  ... NaN  1.81 NaN  NaN  -87.16  20.41
1      2  400069     吉恩5  9.96  4.95  0.47  ... NaN  9.49 NaN  NaN   19.61   3.78
2      3  400073    上普A5  7.08  4.89  0.33  ... NaN  6.75 NaN  NaN   97.79  11.69
3      4  400008   水仙Ａ 5  4.96  4.86  0.23  ... NaN  4.73 NaN  NaN -539.13  992.0
4      5  400059     天珑5  3.50  4.79  0.16  ... NaN  3.34 NaN  NaN   50.29   3.72
..   ...     ...     ...   ...   ...   ...  ...  ..   ...  ..  ...     ...    ...
100  101  400027  生  态 5  1.05 -4.55 -0.05  ... NaN   1.1 NaN  NaN  -57.07    4.7
101  102  400036     天创5  1.64 -4.65 -0.08  ... NaN  1.72 NaN  NaN   103.8   1.57
102  103  400005   海国实 5  1.01 -4.72 -0.05  ... NaN  1.06 NaN  NaN  -91.82   15.3
103  104  400039  华  圣 5  0.95 -5.00 -0.05  ... NaN   1.0 NaN  NaN  -65.97   4.64
104  105  400028  金  马 5  0.95 -5.00 -0.05  ... NaN   1.0 NaN  NaN -226.19    9.3
```

### 科创板

#### 实时行情数据

接口: stock_zh_kcb_spot

目标地址: http://vip.stock.finance.sina.com.cn/mkt/#kcb

描述: 新浪财经-科创板股票实时行情数据

限量: 单次返回所有科创板上市公司的实时行情数据; 请控制采集的频率, 大量抓取容易封IP

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数-实时行情数据

| 名称   | 类型      | 描述           |
|------|---------|--------------|
| 代码   | object  | -            |
| 名称   | object  | -            |
| 最新价  | float64 | -            |
| 涨跌额  | float64 | -            |
| 涨跌幅  | float64 | -            |
| 买入   | float64 | -            |
| 卖出   | float64 | -            |
| 昨收   | float64 | -            |
| 今开   | float64 | -            |
| 最高   | float64 | -            |
| 最低   | float64 | -            |
| 成交量  | float64 | 注意单位: 股      |
| 成交额  | float64 | 注意单位: 元      |
| 时点   | object  | 注意: 数据获取的时间点 |
| 市盈率  | float64 | -            |
| 市净率  | float64 | -            |
| 流通市值 | float64 | -            |
| 总市值  | float64 | -            |
| 换手率  | float64 | -            |

接口示例

```python
import akshare as ak

stock_zh_kcb_spot_df = ak.stock_zh_kcb_spot()
print(stock_zh_kcb_spot_df)
```

数据示例

```
           代码    名称     最新价   涨跌额  ...     市净率          流通市值           总市值      换手率
0    sh688001  华兴源创   31.72 -0.43  ...   4.102  1.393734e+06  1.598691e+05  1.77051
1    sh688002  睿创微纳   58.78  0.28  ...   7.950  2.615710e+06  1.505524e+06  0.97091
2    sh688003  天准科技   32.15 -0.46  ...   4.173  6.224240e+05  2.356659e+05  0.88843
3    sh688004  博汇科技   32.88 -0.59  ...   2.612  1.867584e+05  1.299385e+05  0.78243
4    sh688005  容百科技  135.74 -3.76  ...  12.098  6.081663e+06  3.743849e+06  1.87571
..        ...   ...     ...   ...  ...     ...           ...           ...      ...
386  sh688799  华纳药厂   37.84  0.39  ...   2.496  3.549392e+05  7.558540e+04  4.39538
387  sh688800   瑞可达   95.54 -4.45  ...  10.805  1.031832e+06  2.192643e+05  4.70915
388  sh688819  天能股份   35.15 -0.61  ...   2.854  3.416932e+06  4.517024e+05  1.03151
389  sh688981  中芯国际   50.88 -0.18  ...   3.769  4.021520e+07  9.519842e+06  0.64778
390  sh689009  九号公司   54.75 -0.91  ...   0.927  3.875682e+06  2.425397e+06  0.29878
```

#### 历史行情数据

接口: stock_zh_kcb_daily

目标地址: https://finance.sina.com.cn/realstock/company/sh688001/nc.shtml(示例)

描述: 新浪财经-科创板股票历史行情数据

限量: 单次返回指定 symbol 和 adjust 的所有历史行情数据; 请控制采集的频率, 大量抓取容易封IP

输入参数

| 名称     | 类型  | 描述                                                                                 |
|--------|-----|------------------------------------------------------------------------------------|
| symbol | str | symbol="sh688008"; 带市场标识的股票代码                                                      |
| adjust | str | 默认不复权的数据; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; hfq-factor: 返回后复权因子; qfq-factor: 返回前复权因子 |

输出参数

| 名称                | 类型      | 描述                                                                                           |
|-------------------|---------|----------------------------------------------------------------------------------------------|
| date              | object  | -                                                                                            |
| close             | float64 | 收盘价                                                                                          |
| high              | float64 | 最高价                                                                                          |
| low               | float64 | 最低价                                                                                          |
| open              | float64 | 开盘价                                                                                          |
| volume            | float64 | 成交量(股)                                                                                       |
| after_volume      | float64 | 盘后量; 参见[科创板盘后固定价格交易](http://www.sse.com.cn/lawandrules/sserules/tib/trading/c/4729491.shtml) |
| after_amount      | float64 | 盘后额; 参见[科创板盘后固定价格交易](http://www.sse.com.cn/lawandrules/sserules/tib/trading/c/4729491.shtml) |
| outstanding_share | float64 | 流通股本(股)                                                                                      |
| turnover          | float64 | 换手率=成交量(股)/流通股本(股)                                                                           |

接口示例

```python
import akshare as ak

stock_zh_kcb_daily_df = ak.stock_zh_kcb_daily(symbol="sh688399", adjust="hfq")
print(stock_zh_kcb_daily_df)
```

数据示例-历史行情数据(后复权)

```
 date            close    high    low  ...  after_amount  outstanding_share  turnover
2019-12-05  52.33   62.50  50.25  ...      508408.0         13338470.0  0.740740
2019-12-06  52.79   55.89  50.53  ...       30600.0         13338470.0  0.473652
2019-12-09  51.00   51.28  48.00  ...           0.0         13338470.0  0.335540
2019-12-10  50.10   57.51  49.60  ...       75983.0         13338470.0  0.441293
2019-12-11  53.01   56.58  52.85  ...       94051.0         13338470.0  0.303628
           ...     ...    ...  ...           ...                ...       ...
2020-04-14  79.61   81.18  78.33  ...           0.0         13338470.0  0.050237
2020-04-15  80.75   84.34  80.30  ...       32800.0         13338470.0  0.078527
2020-04-16  83.40   96.00  82.44  ...       47450.0         13338470.0  0.234366
2020-04-17  95.95   95.95  88.88  ...           0.0         13338470.0  0.180283
2020-04-20  95.79  109.97  95.20  ...      188212.0         13338470.0  0.259677
```

数据示例-后复权因子

```
  date              hfq_factor
2019-12-05  1.0000000000000000
1900-01-01  1.0000000000000000
```

数据示例-前复权因子

```
 date               qfq_factor
2019-12-05  1.0000000000000000
1900-01-01  1.0000000000000000
```

#### 科创板公告

接口: stock_zh_kcb_report_em

目标地址: https://data.eastmoney.com/notices/kcb.html

描述: 东方财富-科创板报告数据

限量: 单次返回所有科创板上市公司的报告数据

输入参数

| 名称        | 类型  | 描述                   |
|-----------|-----|----------------------|
| from_page | int | from_page=1; 始获取的页码  |
| to_page   | int | to_page=100; 结束获取的页码 |

输出参数

| 名称   | 类型     | 描述                                                                          |
|------|--------|-----------------------------------------------------------------------------|
| 代码   | object | -                                                                           |
| 名称   | object | -                                                                           |
| 公告标题 | object | -                                                                           |
| 公告类型 | object | -                                                                           |
| 公告日期 | object | -                                                                           |
| 公告代码 | object | 本代码可以用来获取公告详情: http://data.eastmoney.com/notices/detail/688595/{替换到此处}.html |

接口示例

```python
import akshare as ak

stock_zh_kcb_report_em_df = ak.stock_zh_kcb_report_em(from_page=1, to_page=100)
print(stock_zh_kcb_report_em_df)
```

数据示例

```
        代码    名称  ...       公告日期                公告代码
0     688092  爱科科技  ...  2024-07-22  AN202407211638153639
1     688303  大全能源  ...  2024-07-22  AN202407211638152326
2     688536   思瑞浦  ...  2024-07-22  AN202407211638152204
3     688599  天合光能  ...  2024-07-22  AN202407211638152159
4     688352  颀中科技  ...  2024-07-22  AN202407211638151001
...      ...   ...  ...         ...                   ...
9995  688370  丛麟科技  ...  2024-05-11  AN202405101632831676
9996  688131  皓元医药  ...  2024-05-11  AN202405101632831212
9997  688199  久日新材  ...  2024-05-11  AN202405101632831202
9998  688199  久日新材  ...  2024-05-11  AN202405101632831197
9999  688519  南亚新材  ...  2024-05-11  AN202405101632831199
[10000 rows x 6 columns]
```

### A+H股

#### 实时行情数据-东财

接口: stock_zh_ah_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#ah_comparison

描述: 东方财富网-行情中心-沪深港通-AH股比价-实时行情, 延迟 15 分钟更新

限量: 单次返回所有 A+H 上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述        |
|---------|---------|-----------|
| 序号      | int64   | -         |
| 名称      | object  | -         |
| H股代码    | object  | -         |
| 最新价-HKD | float64 | 注意单位: HKD |
| H股-涨跌幅  | float64 | 注意单位: %   |
| A股代码    | object  | -         |
| 最新价-RMB | float64 | 注意单位: RMB |
| A股-涨跌幅  | float64 | 注意单位: %   |
| 比价      | float64 | -         |
| 溢价      | float64 | 注意单位: %   |

接口示例

```python
import akshare as ak

stock_zh_ah_spot_em_df = ak.stock_zh_ah_spot_em()
print(stock_zh_ah_spot_em_df)
```

数据示例

```
      序号    名称   H股代码  最新价-HKD  H股-涨跌幅 A股代码  最新价-RMB  A股-涨跌幅 比价 溢价
0      1    复旦微电  01385    25.65    4.27  688385    47.60    8.92  1.98   98.26
1      2    长飞光纤  06869    18.40    3.95  601869    36.90    3.19  2.14  114.26
2      3    山东黄金  01787    15.24    3.67  600547    23.96    2.00  1.68   67.97
3      4   亿华通-U  02402    27.20    3.62  688339    24.23    0.00  0.95   -4.83
4      5    南京熊猫  00553     3.28    3.47  600775    10.29    3.31  3.35  235.17
..   ...     ...    ...      ...     ...     ...      ...     ...   ...     ...
145  146   万  科Ａ  02202     6.26   -4.13  000002     7.64   -1.93  1.30   30.39
146  147  百济神州-U  06160   152.10   -4.70  688235   226.35   -4.49  1.59   58.99
147  148    荣昌生物  09995    16.64   -5.45  688331    32.11   -1.11  2.06  106.16
148  149     比亚迪  01211   339.00   -6.77  002594   344.84   -4.27  1.09    8.68
149  150    龙蟠科技  02465     6.16  -13.60  603906    12.13  -10.01  2.10  110.38
[150 rows x 10 columns]
```

#### 实时行情数据-腾讯

接口: stock_zh_ah_spot

目标地址: https://stockapp.finance.qq.com/mstats/#mod=list&id=hk_ah&module=HK&type=AH

描述: A+H 股数据是从腾讯财经获取的数据, 延迟 15 分钟更新

限量: 单次返回所有 A+H 上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 代码  | object  | -       |
| 名称  | object  | -       |
| 最新价 | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 涨跌额 | float64 | -       |
| 买入  | float64 | -       |
| 卖出  | float64 | -       |
| 成交量 | float64 | -       |
| 成交额 | float64 | -       |
| 今开  | float64 | -       |
| 昨收  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |

接口示例

```python
import akshare as ak

stock_zh_ah_spot_df = ak.stock_zh_ah_spot()
print(stock_zh_ah_spot_df)
```

数据示例

```
        代码       名称   最新价   涨跌幅   涨跌额  ...          成交额    今开    昨收    最高    最低
0    00525   广深铁路股份  1.66 -0.60 -0.01  ...  18427959.54  1.67  1.67  1.69  1.63
1    01618     中国中冶  1.59 -1.85 -0.03  ...  11468956.78  1.62  1.62  1.63  1.59
2    02727     上海电气  1.54  0.00  0.00  ...  12801574.00  1.54  1.54  1.57  1.53
3    03369     秦港股份  1.49  0.00  0.00  ...   6051885.00  1.49  1.49  1.58  1.48
4    03678     弘业期货  1.41 -0.70 -0.01  ...   1871710.00  1.43  1.42  1.44  1.41
..     ...      ...   ...   ...   ...  ...          ...   ...   ...   ...   ...
155  06806     申万宏源  1.39 -1.42 -0.02  ...   4312447.20  1.41  1.41  1.42  1.39
156  00991     大唐发电  1.24 -2.36 -0.03  ...  29927046.00  1.26  1.27  1.27  1.23
157  00323  马鞍山钢铁股份  1.22 -0.81 -0.01  ...   2345447.26  1.23  1.23  1.25  1.22
158  01635     大众公用  1.20  0.84  0.01  ...    371140.00  1.20  1.19  1.20  1.19
159  01375     中州证券  1.11 -1.77 -0.02  ...   4867240.00  1.13  1.13  1.13  1.09
[160 rows x 13 columns]
```

#### 历史行情数据

接口: stock_zh_ah_daily

目标地址: https://gu.qq.com/hk02359/gp

描述: 腾讯财经-A+H 股数据

限量: 单次返回指定参数的 A+H 上市公司的历史行情数据

输入参数

| 名称         | 类型  | 描述                                                          |
|------------|-----|-------------------------------------------------------------|
| symbol     | str | symbol="02318"; 港股股票代码, 可以通过 **ak.stock_zh_ah_name()** 函数获取 |
| start_year | str | start_year="2000"; 开始年份                                     |
| end_year   | str | end_year="2019"; 结束年份                                       |
| adjust     | str | adjust=""; 默认为空不复权; 'qfq': 前复权, 'hfq': 后复权                  |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 开盘  | float64 | -   |
| 收盘  | float64 | -   |
| 最高  | float64 | -   |
| 最低  | float64 | -   |
| 成交量 | float64 | -   |

接口示例

```python
import akshare as ak

stock_zh_ah_daily_df = ak.stock_zh_ah_daily(symbol="02318", start_year="2022", end_year="2024", adjust="")
print(stock_zh_ah_daily_df)
```

数据示例

```
     日期         开盘   收盘     最高     最低    成交量
0    2022-01-03  56.80  55.75  56.95  55.60   7231921.0
1    2022-01-04  55.95  56.50  56.95  55.75  17975862.0
2    2022-01-05  57.50  56.80  58.00  56.60  28768656.0
3    2022-01-06  56.90  58.00  58.00  56.90  15464523.0
4    2022-01-07  58.00  61.10  61.10  57.55  46904651.0
..          ...    ...    ...    ...    ...         ...
753  2024-01-23  30.00  31.30  31.85  29.60  69397766.0
754  2024-01-24  31.75  33.00  33.35  31.10  79928025.0
755  2024-01-25  33.50  34.60  34.75  32.85  86827346.0
756  2024-01-26  34.15  33.75  34.70  33.35  56539336.0
757  2024-01-29  34.05  33.85  34.90  33.75  41424070.0
[758 rows x 6 columns]
```

#### A+H股票字典

接口: stock_zh_ah_name

目标地址: https://stockapp.finance.qq.com/mstats/#mod=list&id=hk_ah&module=HK&type=AH

描述: A+H 股数据是从腾讯财经获取的数据, 历史数据按日频率更新

限量: 单次返回所有 A+H 上市公司的代码和名称

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型     | 描述  |
|-----|--------|-----|
| 代码  | object | -   |
| 名称  | object | -   |

接口示例

```python
import akshare as ak

stock_zh_ah_name_df = ak.stock_zh_ah_name()
print(stock_zh_ah_name_df)
```

数据示例

```
     代码        名称
0    01211     比亚迪股份
1    06160      百济神州
2    01880      中国中免
3    00941      中国移动
4    06821       凯莱英
..     ...       ...
144  01053    重庆钢铁股份
145  00588  北京北辰实业股份
146  02009      金隅集团
147  02880      辽港股份
148  01033     中石化油服
[149 rows x 2 columns]
```

### 美股

#### 实时行情数据-东财

接口: stock_us_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#us_stocks

描述: 东方财富网-美股-实时行情

限量: 单次返回美股所有上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述              |
|-----|---------|-----------------|
| 序号  | int64   | -               |
| 名称  | object  | -               |
| 最新价 | float64 | 注意单位: 美元        |
| 涨跌额 | float64 | 注意单位: 美元        |
| 涨跌幅 | float64 | 注意单位: %         |
| 开盘价 | float64 | 注意单位: 美元        |
| 最高价 | float64 | 注意单位: 美元        |
| 最低价 | float64 | 注意单位: 美元        |
| 昨收价 | float64 | 注意单位: 美元        |
| 总市值 | float64 | 注意单位: 美元        |
| 市盈率 | float64 | -               |
| 成交量 | float64 | -               |
| 成交额 | float64 | 注意单位: 美元        |
| 振幅  | float64 | 注意单位: %         |
| 换手率 | float64 | 注意单位: %         |
| 代码  | object  | 注意: 用来获取历史数据的代码 |

接口示例

```python
import akshare as ak

stock_us_spot_em_df = ak.stock_us_spot_em()
print(stock_us_spot_em_df)
```

数据示例

```
          序号                         名称  ...      换手率         代码
0          1        Nexalin Technology Inc Wt  ...      NaN  105.NXLIW
1          2           Bionexus Gene Lab Corp  ...   427.44   105.BGLC
2          3  PepperLime Health Acquisition C  ...      NaN  105.PEPLW
3          4  Alliance Entertainment Holding   ...      NaN  105.AENTW
4          5         Digital Brands Group Inc  ...  6569.86   105.DBGI
      ...                              ...  ...      ...        ...
11616  11617                      BIOLASE Inc  ...   582.75   105.BIOL
11617  11618           Sunshine Biopharma Inc  ...   144.85   105.SBFM
11618  11619                      Sientra Inc  ...    42.00   105.SIEN
11619  11620        Sunshine Biopharma Inc Wt  ...      NaN  105.SBFMW
11620  11621  Social Leverage Acquisition Cor  ...      NaN  105.SLACW
[11621 rows x 16 columns]
```

#### 实时行情数据-新浪

接口: stock_us_spot

目标地址: https://finance.sina.com.cn/stock/usstock/sector.shtml

描述: 新浪财经-美股; 获取的数据有 15 分钟延迟; 建议使用 ak.stock_us_spot_em() 来获取数据

限量: 单次返回美股所有上市公司的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型  | 描述   |
|-----|-----|------|
| -   | -   | 新浪默认 |

接口示例

```python
import akshare as ak

us_stock_current_df = ak.stock_us_spot()
print(us_stock_current_df)
```

数据示例(美股股票代码可通过 ak.get_us_stock_name() 函数返回值)

```
                                                 name  \
0                                         Apple, Inc.
1                                     Microsoft Corp.
2                                    Amazon.com, Inc.
3                                      Alphabet, Inc.
4                                      Alphabet, Inc.
..                                                ...
75                               salesforce.com, inc.
76                     Thermo Fisher Scientific, Inc.
77                                       AbbVie, Inc.
78  iPath Series B Bloomberg Energy Subindex Total...
79              International Business Machines Corp.
                                                cname category symbol  \
0                                                苹果公司      计算机   AAPL
1                                                微软公司       软件   MSFT
2                                               亚马逊公司      互联网   AMZN
3                                                  谷歌      互联网   GOOG
4                                               谷歌A类股     媒体内容  GOOGL
..                                                ...      ...    ...
75                                              赛富时公司       软件    CRM
76                                          赛默飞世尔科技公司             TMO
77                                              艾伯维公司  生物技术和制药   ABBV
78  iPath Series B Bloomberg Energy Subindex Total...     None   JJEB
79                                    IBM(国际商业机器有限公司)             IBM
      price   diff   chg preclose     open     high      low amplitude  \
0    255.82   7.06  2.84   248.76   249.54   255.93   249.16     2.72%
1    143.72   0.35  0.24   143.37   144.26   144.42   142.97     1.01%
2   1791.44  14.78  0.83  1776.66  1788.01  1797.44  1785.21     0.69%
3   1273.74  13.63  1.08  1260.11  1265.00  1274.62  1260.50     1.12%
4   1272.25  13.45  1.07  1258.80  1265.80  1273.00  1259.71     1.06%
..      ...    ...   ...      ...      ...      ...      ...       ...
75   159.74   3.25  2.08   156.49   157.91   160.09   156.75     2.13%
76   303.60   1.62  0.54   301.98   305.00   305.28   303.12     0.72%
77    81.75   2.20  2.77    79.55    80.03    82.18    79.13     3.83%
78    51.74   0.00  0.00    51.74     0.00     0.00     0.00     3.54%
79   135.53   1.80  1.35   133.73   134.50   135.56   134.09     1.10%
      volume         mktcap            pe  market category_id
0   37781334  1156096660699   21.62468377  NASDAQ           5
1   33128366  1097361048289   26.81343240  NASDAQ          14
2    2790354   881483397489   77.65236086  NASDAQ          41
3    1670072   874498927695   27.11238831  NASDAQ          41
4    1440607   873475960000   27.08067289  NASDAQ         702
..       ...            ...           ...     ...         ...
75   4771053   122239344864  129.86992115    NYSE          14
76    990471   122222192032   33.88392911    NYSE         738
77  13356251   121302156830   29.94505474    NYSE         700
78         0   121175090554          None    AMEX        None
79   3089789   120621698914   15.65011561    NYSE         750
```

#### 历史行情数据-东财

接口: stock_us_hist

目标地址: https://quote.eastmoney.com/us/ENTX.html#fullScreenChart

描述: 东方财富网-行情-美股-每日行情

限量: 单次返回指定上市公司的指定 adjust 后的所有历史行情数据；注意其中复权参数是否生效！

输入参数

| 名称         | 类型  | 描述                                                                          |
|------------|-----|-----------------------------------------------------------------------------|
| symbol     | str | 美股代码, 可以通过 **ak.stock_us_spot_em()** 函数返回所有的 pandas.DataFrame 里面的 `代码` 字段获取 |
| period     | str | period='daily'; choice of {'daily', 'weekly', 'monthly'}                    |
| start_date | str | start_date="20210101"                                                       |
| end_date   | str | end_date="20210601"                                                         |
| adjust     | str | 默认 adjust="", 则返回未复权的数据; adjust="qfq" 则返回前复权的数据, adjust="hfq" 则返回后复权的数据     |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 日期  | object  | -        |
| 开盘  | float64 | 注意单位: 美元 |
| 收盘  | float64 | 注意单位: 美元 |
| 最高  | float64 | 注意单位: 美元 |
| 最低  | float64 | 注意单位: 美元 |
| 成交量 | int32   | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 美元 |
| 振幅  | float64 | 注意单位: %  |
| 涨跌幅 | float64 | 注意单位: %  |
| 涨跌额 | float64 | 注意单位: 美元 |
| 换手率 | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_us_hist_df = ak.stock_us_hist(symbol='106.TTE', period="daily", start_date="20200101", end_date="20240214", adjust="qfq")
print(stock_us_hist_df)
```

数据示例

```
     日期         开盘   收盘   最高  ...    振幅   涨跌幅  涨跌额 换手率
0    2021-06-11  41.75  41.24  41.75  ...  0.00  0.00  0.00  0.03
1    2021-06-14  41.56  41.61  42.17  ...  1.94  0.90  0.37  0.04
2    2021-06-15  41.40  42.19  42.19  ...  2.04  1.39  0.58  0.09
3    2021-06-16  42.04  41.71  42.16  ...  1.59 -1.14 -0.48  0.06
4    2021-06-17  41.61  40.69  41.80  ...  3.72 -2.45 -1.02  0.05
..          ...    ...    ...    ...  ...   ...   ...   ...   ...
668  2024-02-07  62.90  63.05  63.26  ...  0.94 -3.07 -2.00  0.27
669  2024-02-08  63.39  64.19  64.33  ...  1.67  1.81  1.14  0.14
670  2024-02-09  64.41  64.31  64.53  ...  0.69  0.19  0.12  0.15
671  2024-02-12  64.40  64.63  64.82  ...  0.98  0.50  0.32  0.14
672  2024-02-13  65.21  64.43  65.28  ...  1.81 -0.31 -0.20  0.14
[673 rows x 11 columns]
```

#### 个股信息查询-雪球

接口: stock_individual_basic_info_us_xq

目标地址: https://xueqiu.com/snowman/S/NVDA/detail#/GSJJ

描述: 雪球-个股-公司概况-公司简介

限量: 单次返回指定 symbol 的个股信息

输入参数

| 名称      | 类型    | 描述                      |
|---------|-------|-------------------------|
| symbol  | str   | symbol="NVDA"; 股票代码     |
| token   | str   | token=None;             |
| timeout | float | timeout=None; 默认不设置超时参数 |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_individual_basic_info_us_xq_df = ak.stock_individual_basic_info_us_xq(symbol="SH601127")
print(stock_individual_basic_info_us_xq_df)
```

数据示例

```
                             item                                              value
0                          org_id                                         T000040433
1                     org_name_cn                                              英伟达公司
2               org_short_name_cn                                                英伟达
3                     org_name_en                                 Nvidia Corporation
4               org_short_name_en                                             Nvidia
5         main_operation_business                                           图形和通信处理器
6                 operating_scope  公司的图形和通信处理器已被多种多样的计算平台采用，包括个人数字媒体PC、商用PC、专业工作站...
7                 district_encode                                             001008
8             org_cn_introduction  英伟达公司于1993年4月在加利福尼亚州注册成立，并于1998年4月在特拉华州重新注册成立。...
9            legal_representative                                               None
10                general_manager                                               None
11                      secretary                                               None
12               established_date                                               None
13                      reg_asset                                               None
14                      staff_num                                              36000
15                      telephone                                      1-408-4862000
16                       postcode                                              95051
17                            fax                                               None
18                          email                                               None
19                    org_website                                     www.nvidia.com
20                 reg_address_cn                                               特拉华州
21                 reg_address_en                                               特拉华州
22              office_address_cn                                               None
23              office_address_en  2788 San Tomas Expressway\r\nSanta Clara\r\nCa...
24                currency_encode                                               None
25                       currency
26                    listed_date                                       916981200000
27                         td_mkt                                      美国NASDAQ证券交易所
28                       chairman                                               None
29                executives_nums                                                  6
30  actual_issue_total_shares_num                                               None
31             actual_issue_price                                               None
32            total_raise_capital                                               None
33                     mainholder                                       领航集团 (8.30%)
```

#### 分时数据-东财

接口: stock_us_hist_min_em

目标地址: https://quote.eastmoney.com/us/ATER.html

描述: 东方财富网-行情首页-美股-每日分时行情

限量: 单次返回指定上市公司最近 5 个交易日分钟数据, 注意美股数据更新有延时

输入参数

| 名称         | 类型  | 描述                                                                                           |
|------------|-----|----------------------------------------------------------------------------------------------|
| symbol     | str | symbol="105.ATER"; 美股代码可以通过 **ak.stock_us_spot_em()** 函数返回所有的 pandas.DataFrame 里面的 `代码` 字段获取 |
| start_date | str | start_date="1979-09-01 09:32:00"; 日期时间; 默认返回所有数据                                             |
| end_date   | str | end_date="2222-01-01 09:32:00"; 日期时间; 默认返回所有数据                                               |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 时间  | object  | -        |
| 开盘  | float64 | 注意单位: 美元 |
| 收盘  | float64 | 注意单位: 美元 |
| 最高  | float64 | 注意单位: 美元 |
| 最低  | float64 | 注意单位: 美元 |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 美元 |
| 最新价 | float64 | 注意单位: 美元 |

接口示例

```python
import akshare as ak

stock_us_hist_min_em_df = ak.stock_us_hist_min_em(symbol="105.ATER")
print(stock_us_hist_min_em_df)
```

数据示例

```
          时间    开盘    收盘    最高    最低       成交量          成交额    最新价
0     2021-08-30 21:30:00  0.00  8.86  8.92  8.86  12147424  108372137.0  8.898
1     2021-08-30 21:31:00  0.00  8.72  8.90  8.57    998571    8927364.0  8.813
2     2021-08-30 21:32:00  0.00  8.73  8.73  8.44    725257    6193335.0  8.735
3     2021-08-30 21:33:00  0.00  8.90  8.98  8.67    546888    4965712.0  8.747
4     2021-08-30 21:34:00  0.00  8.60  9.00  8.34   1040362    9456790.0  8.739
                   ...   ...   ...   ...   ...       ...          ...    ...
1950  2021-09-04 03:56:00  6.67  6.65  6.70  6.65     26769     180608.0  6.478
1951  2021-09-04 03:57:00  6.65  6.68  6.68  6.65     13643      90551.0  6.478
1952  2021-09-04 03:58:00  6.68  6.70  6.70  6.67     46469     314155.0  6.480
1953  2021-09-04 03:59:00  6.70  6.69  6.70  6.69     31059     209314.0  6.481
1954  2021-09-04 04:00:00  6.70  6.70  6.70  6.67    243168    1549848.0  6.478
```

#### 历史行情数据-新浪

接口: stock_us_daily

目标地址: http://finance.sina.com.cn/stock/usstock/sector.shtml

描述: 美股历史行情数据，设定 adjust="qfq" 则返回前复权后的数据，默认 adjust="", 则返回未复权的数据，历史数据按日频率更新

限量: 单次返回指定上市公司的指定 adjust 后的所有历史行情数据

输入参数

| 名称     | 类型  | 描述                                                                  |
|--------|-----|---------------------------------------------------------------------|
| symbol | str | 美股代码, 可以通过 **ak.get_us_stock_name()** 函数返回所有美股代码, 由于美股数据量大, 建议按需要获取 |
| adjust | str | adjust="qfq" 则返回前复权后的数据，默认 adjust="", 则返回未复权的数据                     |

**ak.get_us_stock_name()**: will return a pandas.DataFrame, which contains name, cname and symbol, you should use
symbol!

输出参数-历史数据

| 名称     | 类型         | 描述  |
|--------|------------|-----|
| date   | datetime64 | -   |
| open   | float64    | 开盘价 |
| high   | float64    | 最高价 |
| low    | float64    | 最低价 |
| close  | float64    | 收盘价 |
| volume | float64    | 成交量 |

输出参数-前复权因子

| 名称         | 类型         | 描述                  |
|------------|------------|---------------------|
| date       | datetime64 | 日期                  |
| qfq_factor | float      | 前复权因子               |
| adjust     | float      | 由于前复权会出现负值, 该值为调整因子 |

P.S. 复权计算公式: 未复权数据 * qfq_factor + adjust

P.S. "CIEN" 股票的新浪美股数据由于复权因子错误，暂不返回前复权数据

接口示例-未复权数据

```python
import akshare as ak

stock_us_daily_df = ak.stock_us_daily(symbol="AAPL", adjust="")
print(stock_us_daily_df)
```

数据示例-未复权数据

```
 date         open      high      low    close       volume
1980-12-12   28.735   28.8750   28.735   28.735    2093900.0
1980-12-15   27.265   27.3700   27.265   27.265     785200.0
1980-12-16   25.235   25.3750   25.235   25.235     472000.0
1980-12-17   25.865   26.0050   25.865   25.865     385900.0
1980-12-18   26.635   26.7400   26.635   26.635     327900.0
             ...       ...      ...      ...          ...
2021-01-04  133.520  133.6116  126.760  129.410  134308607.0
2021-01-05  128.890  131.7400  128.430  131.010   90749416.0
2021-01-06  127.720  131.0499  126.382  126.600  139351145.0
2021-01-07  128.360  131.6300  127.860  130.920  101546989.0
2021-01-08  132.430  132.6300  130.230  132.050   98373014.0
```

接口示例-前复权调整后的数据

```python
import akshare as ak

stock_us_daily_df = ak.stock_us_daily(symbol="AAPL", adjust="qfq")
print(stock_us_daily_df)
```

数据示例-前复权调整后的数据

```
 date          open      high       low     close       volume
1980-12-12   -4.8164   -4.8158   -4.8164   -4.8164    2093900.0
1980-12-15   -4.8230   -4.8225   -4.8230   -4.8230     785200.0
1980-12-16   -4.8321   -4.8314   -4.8321   -4.8321     472000.0
1980-12-17   -4.8293   -4.8286   -4.8293   -4.8293     385900.0
1980-12-18   -4.8258   -4.8253   -4.8258   -4.8258     327900.0
              ...       ...       ...       ...          ...
2021-01-04  133.5200  133.6116  126.7600  129.4100  134308607.0
2021-01-05  128.8900  131.7400  128.4300  131.0100   90749416.0
2021-01-06  127.7200  131.0499  126.3820  126.6000  139351145.0
2021-01-07  128.3600  131.6300  127.8600  130.9200  101546989.0
2021-01-08  132.4300  132.6300  130.2300  132.0500   98373014.0
```

接口示例-前复权因子

```python
import akshare as ak

qfq_df = ak.stock_us_daily(symbol="AAPL", adjust="qfq-factor")
print(qfq_df)
```

数据示例-前复权因子

```
 date               qfq_factor            adjust
2020-08-07                  1                 0
2020-05-08                  1      -0.819999993
2020-02-07                  1      -1.639999986
2019-11-07                  1      -2.409999967
2019-08-09                  1      -3.179999948
                       ...               ...
1987-11-17  0.035714285714285     -18.958676644
1987-08-10  0.035714285714285  -18.958778786857
1987-06-16  0.035714285714285  -18.958855215429
1987-05-11  0.017857142857142  -18.958855215429
1900-01-01  0.017857142857142  -18.958893429714
```

#### 粉单市场

接口: stock_us_pink_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#us_pinksheet

描述: 美股粉单市场的实时行情数据

限量: 单次返回指定所有粉单市场的行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述              |
|-----|---------|-----------------|
| 序号  | int64   | -               |
| 名称  | object  | -               |
| 最新价 | float64 | 注意单位: 美元        |
| 涨跌额 | float64 | 注意单位: 美元        |
| 涨跌幅 | float64 | 注意单位: %         |
| 开盘价 | float64 | 注意单位: 美元        |
| 最高价 | float64 | 注意单位: 美元        |
| 最低价 | float64 | 注意单位: 美元        |
| 昨收价 | float64 | 注意单位: 美元        |
| 总市值 | float64 | 注意单位: 美元        |
| 市盈率 | float64 | -               |
| 代码  | object  | 注意: 用来获取历史数据的代码 |

接口示例

```python
import akshare as ak

stock_us_pink_spot_em_df = ak.stock_us_pink_spot_em()
print(stock_us_pink_spot_em_df)
```

数据示例

```
      序号           名称                    最新价  ...  总市值  市盈率  代码
0      1           LiNiu Technology Group  0.0189  ...  NaN  NaN  153.LINUF
1      2              NOBILIS HEALTH CORP  0.0016  ...  NaN  NaN  153.NRTSF
2      3                  Yuma Energy Inc  0.0999  ...  NaN  NaN  153.YUMAQ
3      4                      HHGregg Inc  0.0120  ...  NaN  NaN  153.HGGGQ
4      5  Helios and Matheson Analytics I  0.0010  ...  NaN  NaN   153.HMNY
..   ...                              ...     ...  ...  ...  ...        ...
215  216              root9B Holdings Inc  0.0100  ...  NaN  NaN   153.RTNB
216  217       Immune Pharmaceuticals Inc  0.0012  ...  NaN  NaN  153.IMNPQ
217  218                       JRjr33 Inc  0.0005  ...  NaN  NaN  153.JRJRQ
218  219         Mad Catz Interactive Inc  0.0001  ...  NaN  NaN  153.MCZAF
219  220         Taronis Technologies Inc  0.0010  ...  NaN  NaN   153.TRNX
```

#### 知名美股

接口: stock_us_famous_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#us_wellknown

描述: 美股-知名美股的实时行情数据

限量: 单次返回指定 symbol 的行情数据

输入参数

| 名称     | 类型  | 描述                                                                       |
|--------|-----|--------------------------------------------------------------------------|
| symbol | str | symbol="科技类"; choice of {'科技类', '金融类', '医药食品类', '媒体类', '汽车能源类', '制造零售类'} |

输出参数

| 名称  | 类型      | 描述              |
|-----|---------|-----------------|
| 序号  | int64   | -               |
| 名称  | object  | -               |
| 最新价 | float64 | 注意单位: 美元        |
| 涨跌额 | float64 | 注意单位: 美元        |
| 涨跌幅 | float64 | 注意单位: %         |
| 开盘价 | float64 | 注意单位: 美元        |
| 最高价 | float64 | 注意单位: 美元        |
| 最低价 | float64 | 注意单位: 美元        |
| 昨收价 | float64 | 注意单位: 美元        |
| 总市值 | float64 | 注意单位: 美元        |
| 市盈率 | float64 | -               |
| 代码  | object  | 注意: 用来获取历史数据的代码 |

接口示例

```python
import akshare as ak

stock_us_famous_spot_em_df = ak.stock_us_famous_spot_em(symbol='科技类')
print(stock_us_famous_spot_em_df)
```

数据示例

```
    序号              名称           最新价  ...         总市值     市盈率        代码
0    1  Silvergate Capital Corp-A   116.34  ...     3085409047   61.93    106.SI
1    2  Opendoor Technologies Inc    18.94  ...    11451903533  -19.65  105.OPEN
2    3                     阿勒格尼技术    17.55  ...     2233160842   -1.82   106.ATI
3    4                Yandex NV-A    78.71  ...    28129406798  131.13  105.YNDX
4    5                        爱立信    11.83  ...    39443015025   16.44  105.ERIC
5    6                        诺基亚     5.91  ...    33269348763  -14.11   106.NOK
6    7              Groupon Inc-A    22.16  ...      654264338   73.65  105.GRPN
7    8                         推特    62.46  ...    49840992399  129.65  106.TWTR
8    9             Facebook Inc-A   378.00  ...  1065750021378   27.36    105.FB
9   10                         惠普    28.21  ...    32512553599    7.98   106.HPQ
10  11                       谷歌-C  2898.27  ...  1932435735616   16.84  105.GOOG
11  12                      阿卡迈技术   113.38  ...    18461647373   31.60  105.AKAM
12  13                      超威半导体   106.15  ...   128756264684   37.47   105.AMD
13  14                         思科    58.60  ...   247159324736   23.34  105.CSCO
14  15                       中华电信    40.05  ...    31068573413   24.84   106.CHT
15  16                       德州仪器   188.47  ...   173997383234   25.87   105.TXN
16  17                        奥多比   661.68  ...   315224352000   56.47  105.ADBE
17  18                        高知特    76.48  ...    40197673856   23.91  105.CTSH
18  19                        英特尔    53.40  ...   216643800000   11.68  105.INTC
19  20                     美国电话电报    27.42  ...   195778800000 -100.55     106.T
20  21                         高通   141.58  ...   159702240000   17.35  105.QCOM
21  22                         苹果   154.07  ...  2546802675620   29.34  105.AAPL
22  23              IBM国际商业机器(US)   137.74  ...   123459126717   23.15   106.IBM
23  24                         陶氏    60.28  ...    44955123381   10.96   106.DOW
24  25                        思爱普   145.68  ...   171841118251   23.58   106.SAP
25  26                        英伟达   221.77  ...   554425000000   78.33  105.NVDA
26  27                      威瑞森通讯    54.44  ...   225387915421   11.28    106.VZ
27  28                         微软   297.25  ...  2233801423468   36.46  105.MSFT
28  29                   摩托罗拉解决方案   244.00  ...    41315202400   35.80   106.MSI
29  30                        亚马逊  3484.16  ...  1764519802163   59.94  105.AMZN
30  31                         易趣    73.00  ...    47454073765    3.68  105.EBAY
31  32                    沃达丰(US)    16.61  ...    46152628937  351.55   105.VOD
32  33                Zynga Inc-A     8.28  ...     9040854574  -53.06  105.ZNGA
33  34          SentinelOne Inc-A    66.04  ...    16930759265  -85.17     106.S
```

### 港股

#### 实时行情数据-东财

接口: stock_hk_spot_em

目标地址: http://quote.eastmoney.com/center/gridlist.html#hk_stocks

描述: 所有港股的实时行情数据; 该数据有 15 分钟延时

限量: 单次返回最近交易日的所有港股的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 序号  | int64   | -        |
| 代码  | object  | -        |
| 名称  | object  | -        |
| 最新价 | float64 | 注意单位: 港元 |
| 涨跌额 | float64 | 注意单位: 港元 |
| 涨跌幅 | float64 | 注意单位: %  |
| 今开  | float64 | -        |
| 最高  | float64 | -        |
| 最低  | float64 | -        |
| 昨收  | float64 | -        |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |

接口示例

```python
import akshare as ak

stock_hk_spot_em_df = ak.stock_hk_spot_em()
print(stock_hk_spot_em_df)
```

数据示例

```
     序号     代码      名称    最新价  ...    最低   昨收  成交量         成交额
0        1  00593     梦东方   2.62  ...    1.6   1.51   2582500   7104955.0
1        2  08367    倩碧控股  0.225  ...  0.153  0.152  82770000  17723337.0
2        3  03886  康健国际医疗  0.395  ...  0.305   0.29  54347051  19867777.0
3        4  00205    财讯传媒  0.475  ...  0.305   0.35   6920400   3218611.0
4        5  08166  中国农业生态  0.047  ...   0.04  0.037    120000      5230.0
    ...    ...     ...    ...  ...    ...    ...       ...         ...
4523  4524  01335    顺泰控股  0.161  ...  0.152  0.195   2310000    376096.0
4524  4525  08088  八零八八投资  0.051  ...   0.05  0.062   1216000     65144.0
4525  4526  00809  大成生化科技  0.205  ...  0.195   0.25   4024000    875026.0
4526  4527  00378    五龙动力  0.012  ...  0.011  0.015  80412000    991172.0
4527  4528  03638    华邦科技  0.099  ...  0.099  0.128    972000    100308.0
```

#### 港股主板实时行情数据-东财

接口: stock_hk_main_board_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#hk_mainboard

描述: 港股主板的实时行情数据; 该数据有 15 分钟延时

限量: 单次返回港股主板的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 序号  | int64   | -        |
| 代码  | object  | -        |
| 名称  | object  | -        |
| 最新价 | float64 | 注意单位: 港元 |
| 涨跌额 | float64 | 注意单位: 港元 |
| 涨跌幅 | float64 | 注意单位: %  |
| 今开  | float64 | -        |
| 最高  | float64 | -        |
| 最低  | float64 | -        |
| 昨收  | float64 | -        |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |

接口示例

```python
import akshare as ak

stock_hk_main_board_spot_em_df = ak.stock_hk_main_board_spot_em()
print(stock_hk_main_board_spot_em_df)
```

数据示例

```
        序号     代码            名称  ...      昨收          成交量          成交额
0        1  02048        易居企业控股  ...   0.385   25513200.0   15353803.0
1        2  02121          创新奇智  ...  14.920   16877100.0  317864528.0
2        3  03896           金山云  ...   4.300  109607960.0  558129600.0
3        4  01552  BHCC HOLDING  ...   0.098     555000.0      63190.0
4        5  00799           IGG  ...   3.080   42094549.0  155345439.0
    ...    ...           ...  ...     ...          ...          ...
2278  2279  02181        迈博药业-B  ...   0.740      16000.0       9060.0
2279  2280  00974         中国顺客隆  ...   0.670       3000.0       1580.0
2280  2281  01146        中国服饰控股  ...   0.194       6000.0        806.0
2281  2282  06928          万马控股  ...   2.410    1816000.0    3545800.0
2282  2283  01160        金石资本集团  ...   0.750    1600000.0     747500.0
```

#### 实时行情数据-新浪

接口: stock_hk_spot

目标地址: https://vip.stock.finance.sina.com.cn/mkt/#qbgg_hk

描述: 获取所有港股的实时行情数据 15 分钟延时

限量: 单次返回当前时间戳的所有港股的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型      | 描述 |
|------|---------|----|
| 日期时间 | object  | -  |
| 代码   | object  | -  |
| 中文名称 | object  | -  |
| 英文名称 | object  | -  |
| 交易类型 | object  | -  |
| 最新价  | float64 | -  |
| 涨跌额  | float64 | -  |
| 涨跌幅  | float64 | -  |
| 昨收   | float64 | -  |
| 今开   | float64 | -  |
| 最高   | float64 | -  |
| 最低   | float64 | -  |
| 成交量  | float64 | -  |
| 成交额  | float64 | -  |
| 买一   | float64 | -  |
| 卖一   | float64 | -  |

接口示例

```python
import akshare as ak

stock_hk_spot_df = ak.stock_hk_spot()
print(stock_hk_spot_df)
```

数据示例

```
       日期时间     代码      中文名称  ...         成交额      买一      卖一
0     2025/07/07 16:08:06  00001        长和  ...   198584360   48.00   48.05
1     2025/07/07 16:08:08  00002      中电控股  ...   162263333   66.00   66.05
2     2025/07/07 16:08:08  00003    香港中华煤气  ...   117155173    6.71    6.72
3     2025/07/07 16:08:06  00004     九龙仓集团  ...     9018213   24.55   24.60
4     2025/07/07 16:08:08  00005      汇丰控股  ...  1045740332   95.20   95.25
...                   ...    ...       ...  ...         ...     ...     ...
2557  2025/07/07 16:08:10  83690     美团－ＷＲ  ...     5132670  107.00  109.00
2558  2025/07/07 16:08:10  86618    京东健康－Ｒ  ...           0   38.15   38.45
2559  2025/07/07 16:08:10  89618  京东集团－ＳＷＲ  ...      811085  112.90  126.00
2560  2025/07/07 16:08:10  89888  百度集团－ＳＷＲ  ...      712615   78.00   78.75
2561  2025/07/07 16:08:10  89988   阿里巴巴－ＷＲ  ...     5516825   96.10   96.50
[2562 rows x 16 columns]
```

#### 个股信息查询-雪球

接口: stock_individual_basic_info_hk_xq

目标地址: https://xueqiu.com/S/00700

描述: 雪球-个股-公司概况-公司简介

限量: 单次返回指定 symbol 的个股信息

输入参数

| 名称      | 类型    | 描述                      |
|---------|-------|-------------------------|
| symbol  | str   | symbol="02097"; 股票代码    |
| token   | str   | token=None;             |
| timeout | float | timeout=None; 默认不设置超时参数 |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_individual_basic_info_hk_xq_df = ak.stock_individual_basic_info_hk_xq(symbol="02097")
print(stock_individual_basic_info_hk_xq_df)
```

数据示例

```
           item                                              value
0       comunic                                        231269720.0
1     comcnname                                         蜜雪冰城股份有限公司
2     comenname                                        MIXUE Group
3       incdate                                    1209484800000.0
4        rgiofc                 中国河南省郑州市金水区北三环南、文化路东瀚海北金商业中心16004室
5    hofclctmbu                 中国河南省郑州市金水区北三环南、文化路东瀚海北金商业中心16004室
6      chairman                                                张红超
7           mbu                                             现制饮品企业
8       comintr  我们是一家领先的现制饮品企业,聚焦为广大消费者提供单价约6元人民币(约1美元)的高质平价的现...
9     refccomty                                                1.0
10     numtissh                                         17059900.0
11         ispr                                              202.5
12         nrfd                                       3291000000.0
13  nation_name                                                 中国
14          tel                                      0371-89834090
15          fax                                      0371-89916887
16        email                                dongshihui@mxbc.com
17     web_site                                http://www.mxbc.com
18    lsdateipo                                    1740931200000.0
19   mainholder                                                张红超
```

#### 分时数据-东财

接口: stock_hk_hist_min_em

目标地址: http://quote.eastmoney.com/hk/00948.html

描述: 东方财富网-行情首页-港股-每日分时行情

限量: 单次返回指定上市公司最近 5 个交易日分钟数据, 注意港股有延时

输入参数

| 名称         | 类型  | 描述                                                                                                  |
|------------|-----|-----------------------------------------------------------------------------------------------------|
| symbol     | str | symbol="01611"; 港股代码可以通过 **ak.stock_hk_spot_em()** 函数返回所有的 pandas.DataFrame 里面的 `代码` 字段获取           |
| period     | str | period='5'; choice of {'1', '5', '15', '30', '60'}; 其中 1 分钟数据返回近 5 个交易日数据且不复权                       |
| adjust     | str | adjust=''; choice of {'', 'qfq', 'hfq'}; '': 不复权, 'qfq': 前复权, 'hfq': 后复权, 其中 1 分钟数据返回近 5 个交易日数据且不复权 |
| start_date | str | start_date="1979-09-01 09:32:00"; 日期时间; 默认返回所有数据                                                    |
| end_date   | str | end_date="2222-01-01 09:32:00"; 日期时间; 默认返回所有数据                                                      |

输出参数-1分钟数据

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 时间  | object  | -        |
| 开盘  | float64 | 注意单位: 港元 |
| 收盘  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |
| 最新价 | float64 | 注意单位: 港元 |

接口示例-1分钟数据

```python
import akshare as ak

stock_hk_hist_min_em_df = ak.stock_hk_hist_min_em(symbol="01611", period='1', adjust='',
                                                  start_date="2021-09-01 09:32:00",
                                                  end_date="2021-09-07 18:32:00")  # 其中的 start_date 和 end_date 需要设定为近期
print(stock_hk_hist_min_em_df)
```

数据示例-1分钟数据

```
                    时间     开盘  收盘    最高    最低   成交量   成交额      最新价
0     2021-09-01 09:32:00   0.00  11.90  11.90  11.90  15500  184450.0  11.9215
1     2021-09-01 09:33:00   0.00  11.86  11.88  11.70  27500  326030.0  11.8837
2     2021-09-01 09:34:00   0.00  11.84  11.86  11.84      0       0.0  11.8837
3     2021-09-01 09:35:00   0.00  11.80  11.80  11.80      0       0.0  11.8837
4     2021-09-01 09:36:00   0.00  11.60  11.70  11.60   5000   58200.0  11.8608
                   ...    ...    ...    ...    ...    ...       ...      ...
1648  2021-09-07 15:56:00  14.04  14.04  14.04  14.04   1000   14040.0  13.9076
1649  2021-09-07 15:57:00  14.08  14.06  14.08  14.04   4500   63330.0  13.9078
1650  2021-09-07 15:58:00  14.04  14.08  14.08  14.04   6500   91390.0  13.9081
1651  2021-09-07 15:59:00  14.08  14.06  14.08  14.06   1000   14070.0  13.9082
1652  2021-09-07 16:00:00  14.06  14.10  14.10  14.06  34500  486320.0  13.9102
```

输出参数-其他

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 时间  | object  | -        |
| 开盘  | float64 | 注意单位: 港元 |
| 收盘  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 涨跌幅 | float64 | 注意单位: %  |
| 涨跌额 | float64 | 注意单位: 港元 |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |
| 振幅  | float64 | 注意单位: %  |
| 换手率 | float64 | 注意单位: %  |

接口示例-其他

```python
import akshare as ak

stock_hk_hist_min_em_df = ak.stock_hk_hist_min_em(symbol="01611", period='5', adjust='hfq',
                                                  start_date="2021-09-01 09:32:00",
                                                  end_date="2021-09-07 18:32:00")  # 其中的 start_date 和 end_date 需要设定为近期
print(stock_hk_hist_min_em_df)
```

数据示例-其他

```
                  时间     开盘   收盘   最高  ...    成交量   成交额    振幅   换手率
0    2021-09-01 09:35:00  12.05  11.85  12.05  ...  48000   570420.0  2.47  0.02
1    2021-09-01 09:40:00  11.75  11.57  11.75  ...  41500   478780.0  1.86  0.01
2    2021-09-01 09:45:00  11.57  11.57  11.67  ...  17000   196350.0  0.86  0.01
3    2021-09-01 09:50:00  11.57  11.67  11.67  ...  21500   248120.0  1.38  0.01
4    2021-09-01 09:55:00  11.65  11.63  11.69  ...  36500   423360.0  0.51  0.01
..                   ...    ...    ...    ...  ...    ...        ...   ...   ...
325  2021-09-07 15:40:00  14.11  14.05  14.11  ...  76500  1073240.0  0.43  0.02
326  2021-09-07 15:45:00  14.09  14.07  14.13  ...  23500   329810.0  0.43  0.01
327  2021-09-07 15:50:00  14.07  14.13  14.13  ...  50000   701710.0  0.43  0.02
328  2021-09-07 15:55:00  14.13  14.13  14.15  ...  54500   767760.0  0.28  0.02
329  2021-09-07 16:00:00  14.09  14.15  14.15  ...  47500   669150.0  0.42  0.02
```

#### 历史行情数据-东财

接口: stock_hk_hist

目标地址: https://quote.eastmoney.com/hk/08367.html

描述: 港股-历史行情数据, 可以选择返回复权后数据, 更新频率为日频

限量: 单次返回指定上市公司的历史行情数据

输入参数

| 名称         | 类型  | 描述                                                             |
|------------|-----|----------------------------------------------------------------|
| symbol     | str | symbol="00593"; 港股代码,可以通过 **ak.stock_hk_spot_em()** 函数返回所有港股代码 |
| period     | str | period='daily'; choice of {'daily', 'weekly', 'monthly'}       |
| start_date | str | start_date="19700101"; 开始日期                                    |
| end_date   | str | end_date="22220101"; 结束日期                                      |
| adjust     | str | adjust="": 返回未复权的数据, 默认; qfq: 返回前复权数据; hfq: 返回后复权数据;           |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 日期  | object  | -        |
| 开盘  | float64 | 注意单位: 港元 |
| 收盘  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 成交量 | int64   | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |
| 振幅  | float64 | 注意单位: %  |
| 涨跌幅 | float64 | 注意单位: %  |
| 涨跌额 | float64 | 注意单位: 港元 |
| 换手率 | float64 | 注意单位: %  |

接口示例-未复权

```python
import akshare as ak

stock_hk_hist_df = ak.stock_hk_hist(symbol="00593", period="daily", start_date="19700101", end_date="22220101", adjust="")
print(stock_hk_hist_df)
```

数据示例-未复权

```
       日期        开盘    收盘  最高    最低  ...  成交额     振幅    涨跌幅   涨跌额   换手率
0     1998-01-02  0.64  0.64  0.67  0.62  ...    805000.0   0.00   0.00  0.00  0.45
1     1998-01-05  0.63  0.59  0.63  0.59  ...   2512000.0   6.25  -7.81 -0.05  1.47
2     1998-01-06  0.60  0.60  0.63  0.60  ...   1199000.0   5.08   1.69  0.01  0.69
3     1998-01-07  0.60  0.58  0.60  0.58  ...   1765000.0   3.33  -3.33 -0.02  1.05
4     1998-01-08  0.59  0.58  0.59  0.56  ...    864000.0   5.17   0.00  0.00  0.53
          ...   ...   ...   ...   ...  ...         ...    ...    ...   ...   ...
4770  2023-11-13  1.80  1.39  1.80  1.30  ...     85720.0  31.65 -12.03 -0.19  0.02
4771  2023-11-14  1.31  1.26  1.58  1.10  ...  18368185.0  34.53  -9.35 -0.13  5.09
4772  2023-11-15  1.40  1.35  1.40  1.23  ...  20318650.0  13.49   7.14  0.09  5.35
4773  2023-11-16  1.35  1.38  1.39  1.34  ...  15420060.0   3.70   2.22  0.03  4.00
4774  2023-11-17  1.38  1.47  1.47  1.38  ...   7020800.0   6.52   6.52  0.09  1.74
[4775 rows x 11 columns]
```

输出参数-前复权

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 日期  | object  | -        |
| 开盘  | float64 | 注意单位: 港元 |
| 收盘  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 成交量 | int64   | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |
| 振幅  | float64 | 注意单位: %  |
| 涨跌幅 | float64 | 注意单位: %  |
| 涨跌额 | float64 | 注意单位: 港元 |
| 换手率 | float64 | 注意单位: %  |

接口示例-前复权

```python
import akshare as ak

stock_hk_hist_qfq_df = ak.stock_hk_hist(symbol="00593", period="daily", start_date="19700101", end_date="22220101", adjust="qfq")
print(stock_hk_hist_qfq_df)
```

数据示例-前复权

```
       日期     开盘     收盘     最高  ...     振幅    涨跌幅   涨跌额   换手率
0     1998-01-02  1.713  1.713  1.863  ...   0.00   0.00  0.00  0.45
1     1998-01-05  1.663  1.463  1.663  ...  11.68 -14.59 -0.25  1.47
2     1998-01-06  1.513  1.513  1.663  ...  10.25   3.42  0.05  0.69
3     1998-01-07  1.513  1.413  1.513  ...   6.61  -6.61 -0.10  1.05
4     1998-01-08  1.463  1.413  1.463  ...  10.62   0.00  0.00  0.53
          ...    ...    ...    ...  ...    ...    ...   ...   ...
4770  2023-11-13  1.800  1.390  1.800  ...  31.65 -12.03 -0.19  0.02
4771  2023-11-14  1.310  1.260  1.580  ...  34.53  -9.35 -0.13  5.09
4772  2023-11-15  1.400  1.350  1.400  ...  13.49   7.14  0.09  5.35
4773  2023-11-16  1.350  1.380  1.390  ...   3.70   2.22  0.03  4.00
4774  2023-11-17  1.380  1.470  1.470  ...   6.52   6.52  0.09  1.74
[4775 rows x 11 columns]
```

输出参数-后复权

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 日期  | object  | -        |
| 开盘  | float64 | 注意单位: 港元 |
| 收盘  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 成交量 | int32   | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |
| 振幅  | float64 | 注意单位: %  |
| 涨跌幅 | float64 | 注意单位: %  |
| 涨跌额 | float64 | 注意单位: 港元 |
| 换手率 | float64 | 注意单位: %  |

接口示例-后复权

```python
import akshare as ak

stock_hk_hist_hfq_df = ak.stock_hk_hist(symbol="00593", period="daily", start_date="19700101", end_date="22220101", adjust="hfq")
print(stock_hk_hist_hfq_df)
```

数据示例-后复权

```
      日期       开盘     收盘     最高  ...   振幅   涨跌幅  涨跌额   换手率
0     1998-01-02  0.715  0.715  0.745  ...   0.00  0.00  0.000  0.45
1     1998-01-05  0.705  0.665  0.705  ...   5.59 -6.99 -0.050  1.47
2     1998-01-06  0.675  0.675  0.705  ...   4.51  1.50  0.010  0.69
3     1998-01-07  0.675  0.655  0.675  ...   2.96 -2.96 -0.020  1.05
4     1998-01-08  0.665  0.655  0.665  ...   4.58  0.00  0.000  0.53
          ...    ...    ...    ...  ...    ...   ...    ...   ...
4770  2023-11-13  0.733  0.651  0.733  ...  14.51 -5.52 -0.038  0.02
4771  2023-11-14  0.635  0.625  0.689  ...  14.75 -3.99 -0.026  5.09
4772  2023-11-15  0.653  0.643  0.653  ...   5.44  2.88  0.018  5.35
4773  2023-11-16  0.643  0.649  0.651  ...   1.56  0.93  0.006  4.00
4774  2023-11-17  0.649  0.667  0.667  ...   2.77  2.77  0.018  1.74
[4775 rows x 11 columns]
```

#### 历史行情数据-新浪

接口: stock_hk_daily

目标地址: http://stock.finance.sina.com.cn/hkstock/quotes/01336.html(个例)

描述:港股-历史行情数据, 可以选择返回复权后数据,更新频率为日频

限量: 单次返回指定上市公司的历史行情数据(包括前后复权因子), 提供新浪财经拥有的该股票的所有数据(
并不等于该股票从上市至今的数据)

输入参数

| 名称     | 类型  | 描述                                                                                             |
|--------|-----|------------------------------------------------------------------------------------------------|
| symbol | str | 港股代码,可以通过 **ak.stock_hk_spot()** 函数返回所有港股代码                                                    |
| adjust | str | "": 返回未复权的数据 ; qfq: 返回前复权后的数据; hfq: 返回后复权后的数据; qfq-factor: 返回前复权因子和调整; hfq-factor: 返回后复权因子和调整; |

输出参数-历史行情数据(后复权)

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| date   | object  | 日期  |
| open   | float64 | 开盘价 |
| high   | float64 | 最高价 |
| low    | float64 | 最低价 |
| close  | float64 | 收盘价 |
| volume | float64 | 成交量 |

接口示例-历史行情数据(后复权)

```python
import akshare as ak

stock_hk_daily_hfq_df = ak.stock_hk_daily(symbol="00700", adjust="hfq")
print(stock_hk_daily_hfq_df)
```

数据示例-历史行情数据(后复权)

```
           date      open      high       low     close       volume
0    2004-06-16     4.375     4.625     4.075     4.150  439775000.0
1    2004-06-17     4.150     4.375     4.125     4.225   83801500.0
2    2004-06-18     4.200     4.250     3.950     4.025   36598000.0
3    2004-06-21     4.125     4.125     3.950     4.000   22817000.0
4    2004-06-23     4.050     4.450     4.025     4.425   55016000.0
         ...       ...       ...       ...       ...          ...
4163 2021-05-24  2968.040  2975.540  2918.040  2963.040   16608372.0
4164 2021-05-25  2983.040  3100.540  2978.040  3085.540   27044587.0
4165 2021-05-26  3158.040  3158.040  3095.540  3125.540   18726374.0
4166 2021-05-27  3085.540  3100.540  3045.540  3063.040   25744926.0
4167 2021-05-28  3075.540  3095.540  3040.540  3043.040   15183086.0
```

输出参数-历史行情数据(未复权)

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| date   | object  | 日期  |
| open   | float64 | 开盘价 |
| high   | float64 | 最高价 |
| low    | float64 | 最低价 |
| close  | float64 | 收盘价 |
| volume | float64 | 成交量 |

接口示例-历史行情数据(未复权)

```python
import akshare as ak

stock_hk_daily_df = ak.stock_hk_daily(symbol="00700", adjust="")
print(stock_hk_daily_df)
```

数据示例-历史行情数据(未复权)

```
            date     open     high      low    close       volume
0     2004-06-16    4.375    4.625    4.075    4.150  439775000.0
1     2004-06-17    4.150    4.375    4.125    4.225   83801500.0
2     2004-06-18    4.200    4.250    3.950    4.025   36598000.0
3     2004-06-21    4.125    4.125    3.950    4.000   22817000.0
4     2004-06-23    4.050    4.450    4.025    4.425   55016000.0
          ...      ...      ...      ...      ...          ...
4165  2021-05-24  586.500  588.000  576.500  585.500   16608372.0
4166  2021-05-25  589.500  613.000  588.500  610.000   27044587.0
4167  2021-05-26  624.500  624.500  612.000  618.000   18726374.0
4168  2021-05-27  610.000  613.000  602.000  605.500   25744926.0
4169  2021-05-28  608.000  612.000  601.000  601.500   15183086.0
```

输出参数-后复权因子

| 名称         | 类型     | 描述    |
|------------|--------|-------|
| date       | object | 日期    |
| hfq_factor | object | 后复权因子 |
| cash       | object | 现金分红  |

接口示例-后复权因子

```python
import akshare as ak

stock_hk_daily_hfq_factor_df = ak.stock_hk_daily(symbol="00700", adjust="hfq-factor")
print(stock_hk_daily_hfq_factor_df)
```

数据示例-后复权因子

```
         date hfq_factor   cash
0  2021-05-24          5  35.54
1  2020-05-15          5  27.54
2  2019-05-17          5  21.54
3  2018-12-28          5  16.54
4  2018-05-18          5  16.28
5  2017-05-19          5  11.88
6  2016-05-20          5   8.83
7  2015-05-15          5   6.48
8  2014-05-16          5   4.68
9  2014-05-15          5   3.48
10 2013-05-20          1   3.48
11 2012-05-18          1   2.48
12 2011-05-03          1   1.73
13 2010-05-05          1   1.18
14 2009-05-06          1   0.78
15 2008-05-06          1   0.43
16 2007-05-09          1   0.27
17 2006-05-15          1   0.15
18 2005-04-19          1   0.07
19 1900-01-01          1      0
```

#### 知名港股

接口: stock_hk_famous_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#hk_wellknown

描述: 东方财富网-行情中心-港股市场-知名港股实时行情数据

限量: 单次返回全部行情数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称  | 类型      | 描述       |
|-----|---------|----------|
| 序号  | int64   | -        |
| 代码  | object  | -        |
| 名称  | object  | -        |
| 最新价 | float64 | 注意单位: 港元 |
| 涨跌额 | float64 | 注意单位: 港元 |
| 涨跌幅 | float64 | 注意单位: %  |
| 今开  | float64 | 注意单位: 港元 |
| 最高  | float64 | 注意单位: 港元 |
| 最低  | float64 | 注意单位: 港元 |
| 昨收  | float64 | 注意单位: 港元 |
| 成交量 | float64 | 注意单位: 股  |
| 成交额 | float64 | 注意单位: 港元 |

接口示例

```python
import akshare as ak

stock_hk_famous_spot_em_df = ak.stock_hk_famous_spot_em()
print(stock_hk_famous_spot_em_df)
```

数据示例

```
      序号 代码       名称    最新价  ...    最低     昨收    成交量           成交额
0      1  01918      融创中国   2.04  ...   1.91   1.91  633638656.0  1.295074e+09
1      2  00763      中兴通讯  34.65  ...  31.50  32.85   90643056.0  3.088137e+09
2      3  00753      中国国航   4.56  ...   4.25   4.33   34639744.0  1.560062e+08
3      4  01928  金沙中国有限公司  18.08  ...  17.20  17.18   37260253.0  6.721171e+08
4      5  03900      绿城中国  10.38  ...   9.89   9.91   33031905.0  3.421672e+08
..   ...    ...       ...    ...  ...    ...    ...          ...           ...
113  114  02400      心动公司  32.15  ...  31.60  34.20    4769000.0  1.544492e+08
114  115  01833     平安好医生   8.68  ...   8.55   9.31   53910271.0  4.845739e+08
115  116  02269      药明生物  23.65  ...  23.10  26.00  170040773.0  4.052070e+09
116  117  02359      药明康德  62.90  ...  62.40  70.00   20403989.0  1.314193e+09
117  118  09698   万国数据-SW  42.70  ...  41.30  48.30   24768786.0  1.068253e+09
[118 rows x 12 columns]
```

#### 证券资料

接口: stock_hk_security_profile_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/CompanyProfile

描述: 东方财富-港股-证券资料

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称             | 类型      | 描述 |
|----------------|---------|----|
| 证券代码           | object  | -  |
| 证券简称           | object  | -  |
| 上市日期           | object  | -  |
| 证券类型           | object  | -  |
| 发行价            | float64 | -  |
| 发行量(股)         | int64   | -  |
| 每手股数           | int64   | -  |
| 每股面值           | object  | -  |
| 交易所            | object  | -  |
| 板块             | object  | -  |
| 年结日            | object  | -  |
| ISIN（国际证券识别编码） | object  | -  |
| 是否沪港通标的        | object  | -  |

接口示例

```python
import akshare as ak

stock_hk_security_profile_em_df = ak.stock_hk_security_profile_em(symbol="03900")
print(stock_hk_security_profile_em_df)
```

数据示例

```
       证券代码  证券简称                 上市日期  ... ISIN（国际证券识别编码）  是否沪港通标的  是否深港通标的
0  03900.HK  绿城中国  2006-07-13 00:00:00  ...   KYG4100M1050        是        是
[1 rows x 14 columns]
```

#### 公司资料

接口: stock_hk_company_profile_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/CompanyProfile

描述: 东方财富-港股-公司资料

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称     | 类型     | 描述 |
|--------|--------|----|
| 公司名称   | object | -  |
| 英文名称   | object | -  |
| 注册地    | object | -  |
| 公司成立日期 | object | -  |
| 所属行业   | object | -  |
| 董事长    | object | -  |
| 公司秘书   | object | -  |
| 员工人数   | int64  | -  |
| 办公地址   | object | -  |
| 公司网址   | object | -  |
| E-MAIL | object | -  |
| 年结日    | object | -  |
| 联系电话   | object | -  |
| 核数师    | object | -  |
| 传真     | object | -  |
| 公司介绍   | object | -  |

接口示例

```python
import akshare as ak

stock_hk_company_profile_em_df = ak.stock_hk_company_profile_em(symbol="03900")
print(stock_hk_company_profile_em_df)
```

数据示例

```
         公司名称  ...                                               公司介绍
0  绿城中国控股有限公司  ...      绿城中国控股有限公司(以下简称“绿城中国”)(股票代码03900.HK),1995年...
[1 rows x 17 columns]
```

#### 财务指标

接口: stock_hk_financial_indicator_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/CoreReading

描述: 东方财富-港股-核心必读-最新指标

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称             | 类型     | 描述 |
|----------------|--------|----|
| 基本每股收益(元)      | object | -  |
| 每股净资产(元)       | object | -  |
| 法定股本(股)        | object | -  |
| 每手股            | object | -  |
| 每股股息TTM(港元)    | object | -  |
| 派息比率(%)        | object | -  |
| 已发行股本(股)       | object | -  |
| 已发行股本-H股(股)    | int64  | -  |
| 每股经营现金流(元)     | object | -  |
| 股息率TTM(%)      | object | -  |
| 总市值(港元)        | object | -  |
| 港股市值(港元)       | object | -  |
| 营业总收入          | object | -  |
| 营业总收入滚动环比增长(%) | object | -  |
| 销售净利率(%)       | object | -  |
| 净利润            | object | -  |
| 净利润滚动环比增长(%)   | object | -  |
| 股东权益回报率(%)     | object | -  |
| 市盈率            | object | -  |
| 市净率            | object | -  |
| 总资产回报率(%)      | object | -  |

接口示例

```python
import akshare as ak

stock_hk_financial_indicator_em_df = ak.stock_hk_financial_indicator_em(symbol="03900")
print(stock_hk_financial_indicator_em_df)
```

数据示例

```
   基本每股收益(元)   每股净资产(元)      法定股本(股)  每手股  每股股息TTM(港元)   派息比率(%)    已发行股本(股)  ...  销售净利率(%)        净利润  净利润滚动环比增长(%)  股东权益回报率(%)        市盈率       市净率  总资产回报率(%)
0       0.08  14.006448  10000000000  500        0.328 -322.1807  2539598690  ...  2.270029  209907000   -114.943944    0.583899 -87.240653  0.640675   0.040922
[1 rows x 21 columns]
```


#### 分红派息

接口: stock_hk_dividend_payout_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/CoreReading

描述: 东方财富-港股-核心必读-分红派息

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称     | 类型     | 描述 |
|--------|--------|----|
| 最新公告日期 | object | -  |
| 财政年度   | object | -  |
| 分红方案   | object | -  |
| 分配类型   | object | -  |
| 除净日    | object | -  |
| 截至过户日  | object | -  |
| 发放日    | object | -  |

接口示例

```python
import akshare as ak

stock_hk_dividend_payout_em_df = ak.stock_hk_dividend_payout_em(symbol="03900")
print(stock_hk_dividend_payout_em_df)
```

数据示例

```
    最新公告日期  财政年度  ...      截至过户日         发放日
0   2025-06-20  2024  ...  2025/06/27-2025/07/02  2025-07-31
1   2024-06-14  2023  ...  2024/06/21-2024/06/25  2024-07-31
2   2023-06-16  2022  ...  2023/06/23-2023/06/27  2023-07-31
3   2022-06-17  2021  ...  2022/06/24-2022/06/28  2022-07-29
4   2021-06-18  2020  ...  2021/06/25-2021/06/29  2021-07-31
5   2020-06-12  2019  ...  2020/06/19-2020/06/23  2020-07-31
6   2019-06-14  2018  ...  2019/06/21-2019/06/25  2019-07-29
7   2018-06-15  2017  ...  2018/06/22-2018/06/26  2018-07-18
8   2017-06-16  2016  ...  2017/06/22-2017/06/26  2017-07-31
9   2014-06-27  2013  ...  2014/07/04-2014/07/09  2014-07-18
10  2013-06-18  2012  ...  2013/06/27-2013/07/02  2013-07-12
11  2012-02-07  2011  ...  2012/02/29-2012/03/02  2012-03-19
12  2011-06-14  2010  ...  2011/06/07-2011/06/13  2011-06-17
13  2010-08-25  2010  ...  2010/09/13-2010/09/15  2010-09-27
14  2010-06-04  2009  ...  2010/06/01-2010/06/03  2010-06-11
15  2009-09-18  2009  ...  2009/10/12-2009/10/14  2009-10-23
16  2012-06-18  2008  ...  2009/07/04-2009/07/08  2009-07-17
17  2008-05-26  2007  ...  2008/05/21-2008/05/23  2008-05-30
18  2007-05-14  2006  ...  2007/05/07-2007/05/11  2007-05-22
[19 rows x 7 columns]
```


#### 行业对比

##### 成长性对比

接口: stock_hk_growth_comparison_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/IndustryComparison

描述: 东方财富-港股-行业对比-成长性对比

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称                  | 类型      | 描述 |
|---------------------|---------|----|
| 代码                  | object  | -  |
| 简称                  | object  | -  |
| 基本每股收益同比增长率         | float64 | -  |
| 基本每股收益同比增长率排名       | int64   | -  |
| 营业收入同比增长率           | float64 | -  |
| 营业收入同比增长率排名         | int64   | -  |
| 营业利润率同比增长率          | float64 | -  |
| 营业利润率同比增长率排名        | int64   | -  |
| 基本每股收总资产同比增长率益同比增长率 | float64 | -  |
| 总资产同比增长率排名          | int64   | -  |

接口示例

```python
import akshare as ak

stock_hk_growth_comparison_em_df = ak.stock_hk_growth_comparison_em(symbol="03900")
print(stock_hk_growth_comparison_em_df)
```

数据示例

```
      代码    简称  基本每股收益同比增长率  基本每股收益同比增长率排名 ...  总资产同比增长率排名
0  03900  绿城中国   -90.123457            171          ...          91
```

##### 估值对比

接口: stock_hk_valuation_comparison_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/IndustryComparison

描述: 东方财富-港股-行业对比-估值对比

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称        | 类型      | 描述 |
|-----------|---------|----|
| 代码        | object  | -  |
| 简称        | object  | -  |
| 市盈率-TTM   | float64 | -  |
| 市盈率-TTM排名 | int64   | -  |
| 市盈率-LYR   | float64 | -  |
| 市盈率-LYR排名 | int64   | -  |
| 市净率-MRQ   | float64 | -  |
| 市净率-MRQ排名 | int64   | -  |
| 市净率-LYR   | float64 | -  |
| 市净率-LYR排名 | int64   | -  |
| 市销率-TTM   | float64 | -  |
| 市销率-TTM排名 | int64   | -  |
| 市销率-LYR   | float64 | -  |
| 市销率-LYR排名 | int64   | -  |
| 市现率-TTM   | float64 | -  |
| 市现率-TTM排名 | int64   | -  |
| 市现率-LYR   | float64 | -  |
| 市现率-LYR排名 | int64   | -  |

接口示例

```python
import akshare as ak

stock_hk_valuation_comparison_em_df = ak.stock_hk_valuation_comparison_em(symbol="03900")
print(stock_hk_valuation_comparison_em_df)
```

数据示例

```
      代码    简称   市盈率-TTM  市盈率-TTM排名    ...   市现率-LYR  市现率-LYR排名
0  03900  绿城中国 -86.44272         97  14.363182      -30.427808      121
[1 rows x 18 columns]
```


##### 规模对比

接口: stock_hk_scale_comparison_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/pages/home/index.html?code=03900&type=web&color=w#/IndustryComparison

描述: 东方财富-港股-行业对比-规模对比

限量: 单次返回全部数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="03900" |

输出参数

| 名称      | 类型      | 描述 |
|---------|---------|----|
| 代码      | object  | -  |
| 简称      | object  | -  |
| 总市值     | float64 | -  |
| 总市值排名   | int64   | -  |
| 流通市值    | float64 | -  |
| 流通市值排名  | int64   | -  |
| 营业总收入   | int64   | -  |
| 营业总收入排名 | int64   | -  |
| 净利润     | int64   | -  |
| 净利润排名   | int64   | -  |

接口示例

```python
import akshare as ak

stock_hk_scale_comparison_em_df = ak.stock_hk_scale_comparison_em(symbol="03900")
print(stock_hk_scale_comparison_em_df)
```

数据示例

```
    代码   简称       总市值  总市值排名  ...    营业总收入  营业总收入排名  净利润  净利润排名
0  03900  绿城中国  2.201719e+10     20  ...  53368264000        6  209907000     37
[1 rows x 10 columns]
```

### 机构调研

#### 机构调研-统计

接口: stock_jgdy_tj_em

目标地址: http://data.eastmoney.com/jgdy/tj.html

描述: 东方财富网-数据中心-特色数据-机构调研-机构调研统计

限量: 单次返回所有历史数据

输入参数

| 名称   | 类型  | 描述                       |
|------|-----|--------------------------|
| date | str | date="20180928"; 开始查询的时间 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 接待机构数量 | int64   | -       |
| 接待方式   | object  | -       |
| 接待人员   | object  | -       |
| 接待地点   | object  | -       |
| 接待日期   | object  | -       |
| 公告日期   | object  | -       |

接口示例

```python
import akshare as ak

stock_jgdy_tj_em_df = ak.stock_jgdy_tj_em(date="20210128")
print(stock_jgdy_tj_em_df)
```

数据示例

```
      序号      代码    名称  ...                 接待地点        接待日期        公告日期
0      1  002315  焦点科技  ...               线上电话会议  2021-02-02  2021-02-03
1      2  300415   伊之密  ...  广东省佛山市顺德顺昌路12号3号会议室  2021-02-02  2021-02-03
2      3  002092  中泰化学  ...                公司会议室  2021-02-02  2021-02-03
3      4  002562  兄弟科技  ...                   线上  2021-02-02  2021-02-03
4      5  002749  国光股份  ...              公司龙泉办公区  2021-02-02  2021-02-03
..   ...     ...   ...  ...                  ...         ...         ...
111  112  000698  沈阳化工  ...             公司董事会办公室  2021-01-29  2021-01-29
112  113  688669  聚石化学  ...                 电话会议  2021-01-29  2021-01-29
113  114  300143  盈康生命  ...                 电话会议  2021-01-29  2021-01-29
114  115  002438  江苏神通  ...           公司董事会秘书办公室  2021-01-29  2021-01-29
115  116  002735  王子新材  ...                公司会议室  2021-01-29  2021-01-29
```

#### 机构调研-详细

接口: stock_jgdy_detail_em

目标地址: http://data.eastmoney.com/jgdy/xx.html

描述: 东方财富网-数据中心-特色数据-机构调研-机构调研详细

限量: 单次所有历史数据, 由于数据量比较大需要等待一定时间

输入参数

| 名称   | 类型  | 描述                       |
|------|-----|--------------------------|
| date | str | date="20241211"; 开始查询的时间 |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int64   | -       |
| 代码   | object  | -       |
| 名称   | object  | -       |
| 最新价  | float64 | -       |
| 涨跌幅  | float64 | 注意单位: % |
| 调研机构 | object  | -       |
| 机构类型 | object  | -       |
| 调研人员 | object  | -       |
| 接待方式 | object  | -       |
| 接待人员 | object  | -       |
| 接待地点 | object  | -       |
| 调研日期 | object  | -       |
| 公告日期 | object  | -       |

接口示例

```python
import akshare as ak

stock_jgdy_detail_em_df = ak.stock_jgdy_detail_em(date="20241211")
print(stock_jgdy_detail_em_df)
```

数据示例

```
   序号   代码    名称  ...     接待地点      调研日期      公告日期
0   1  002024  ST易购  ...     武汉中南店  2024-12-12  2024-12-12
1   2  002024  ST易购  ...     武汉中南店  2024-12-12  2024-12-12
2   3  002521  齐峰新材  ...     公司接待室  2024-12-12  2024-12-12
3   4  002521  齐峰新材  ...     公司接待室  2024-12-12  2024-12-12
4   5  301390  经纬股份  ...        公司  2024-12-12  2024-12-12
5   6  600030  中信证券  ...  北京中信证券大厦  2024-12-12  2024-12-12
6   7  600030  中信证券  ...  北京中信证券大厦  2024-12-12  2024-12-12
7   8  688528  秦川物联  ...    上证路演中心  2024-12-12  2024-12-12
[8 rows x 13 columns]
```

### 主营介绍-同花顺

接口: stock_zyjs_ths

目标地址: https://basic.10jqka.com.cn/new/000066/operate.html

描述: 同花顺-主营介绍

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="000066" |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 股票代码 | object  | -       |
| 主营业务 | object  | -       |
| 产品类型 | object  | -       |
| 产品名称 | object  | -       |
| 经营范围 | object  | -       |

接口示例

```python
import akshare as ak

stock_zyjs_ths_df = ak.stock_zyjs_ths(symbol="000066")
print(stock_zyjs_ths_df)
```

数据示例

```
     股票代码  ...                                               经营范围
0  000066  ...  计算机软件、硬件、终端及其外部设备、网络系统及系统集成、电子产品及零部件、金融机具、税控机具...
[1 rows x 5 columns]
```

### 主营构成-东财

接口: stock_zygc_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/BusinessAnalysis/Index?type=web&code=SH688041#

描述: 东方财富网-个股-主营构成

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SH688041" |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 股票代码 | object  | -       |
| 报告日期 | object  | -       |
| 分类类型 | object  | -       |
| 主营构成 | int64   | -       |
| 主营收入 | float64 | 注意单位: 元 |
| 收入比例 | float64 | -       |
| 主营成本 | float64 | 注意单位: 元 |
| 成本比例 | float64 | -       |
| 主营利润 | float64 | 注意单位: 元 |
| 利润比例 | float64 | -       |
| 毛利率  | float64 | -       |

接口示例

```python
import akshare as ak

stock_zygc_em_df = ak.stock_zygc_em(symbol="SH688041")
print(stock_zygc_em_df)
```

数据示例

```
    股票代码  报告日期   分类类型  ...      主营利润      利润比例       毛利率
0   688041  2024-06-30  按产品分类  ...  2.385020e+09  0.999313  0.634303
1   688041  2024-06-30  按产品分类  ...  1.639645e+06  0.000687  0.575505
2   688041  2024-06-30  按地区分类  ...  2.386659e+09  1.000000  0.634259
3   688041  2023-12-31    NaN  ...  3.587141e+09  0.999963  0.596682
4   688041  2023-12-31    NaN  ...  1.337029e+05  0.000037  0.708626
..     ...         ...    ...  ...           ...       ...       ...
60  688041  2018-12-31  按产品分类  ...  3.538980e+07  0.874818  0.841592
61  688041  2018-12-31  按产品分类  ...  4.352600e+06  0.107594  0.792982
62  688041  2018-12-31  按产品分类  ...           NaN       NaN       NaN
63  688041  2018-12-31  按地区分类  ...           NaN       NaN       NaN
64  688041  2018-12-31  按地区分类  ...           NaN       NaN       NaN
[65 rows x 11 columns]
```

### 股票质押

#### 股权质押市场概况

接口: stock_gpzy_profile_em

目标地址: https://data.eastmoney.com/gpzy/marketProfile.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-股权质押市场概况

限量: 单次所有历史数据, 由于数据量比较大需要等待一定时间

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 交易日期    | object  | -       |
| A股质押总比例 | float64 | 注意单位: % |
| 质押公司数量  | float64 | -       |
| 质押笔数    | float64 | 注意单位: 笔 |
| 质押总股数   | float64 | 注意单位: 股 |
| 质押总市值   | float64 | 注意单位: 元 |
| 沪深300指数 | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gpzy_profile_em_df = ak.stock_gpzy_profile_em()
print(stock_gpzy_profile_em_df)
```

数据示例

```
     交易日期   A股质押总比例  质押公司数量  ... 质押总市值  沪深300指数  涨跌幅
0    2014-03-07  6.731948    1609  ...  1.629956e+08  2168.3580 -0.4871
1    2014-03-14  6.754048    1617  ...  1.593885e+08  2122.8360 -2.0994
2    2014-03-21  6.750488    1614  ...  1.624715e+08  2158.7980  1.6941
3    2014-03-28  6.716687    1623  ...  1.591070e+08  2151.9650 -0.3165
4    2014-04-04  6.801723    1628  ...  1.635032e+08  2185.4720  1.5570
..          ...       ...     ...  ...           ...        ...     ...
544  2024-11-15  3.619449    2335  ...  2.867691e+08  3968.8308 -3.2947
545  2024-11-22  3.622143    2326  ...  2.812394e+08  3865.6989 -2.5985
546  2024-11-29  3.616201    2319  ...  2.863046e+08  3916.5832  1.3163
547  2024-12-06  3.612298    2322  ...  2.929517e+08  3973.1405  1.4440
548  2024-12-13  3.617088    2318  ...  2.933758e+08  3933.1808 -1.0057
[549 rows x 8 columns]
```

#### 上市公司质押比例

接口: stock_gpzy_pledge_ratio_em

目标地址: https://data.eastmoney.com/gpzy/pledgeRatio.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-上市公司质押比例

限量: 单次返回指定交易日的所有历史数据; 其中的交易日需要根据网站提供的为准; 请访问 http://data.eastmoney.com/gpzy/pledgeRatio.aspx 查询具体交易日

输入参数

| 名称   | 类型  | 描述                                                                           |
|------|-----|------------------------------------------------------------------------------|
| date | str | date="20240906"; 请访问 http://data.eastmoney.com/gpzy/pledgeRatio.aspx 查询具体交易日 |

输出参数

| 名称      | 类型      | 描述       |
|---------|---------|----------|
| 序号      | int64   | -        |
| 股票代码    | object  | -        |
| 股票简称    | object  | -        |
| 交易日期    | object  | -        |
| 所属行业    | object  | -        |
| 质押比例    | float64 | 注意单位: %  |
| 质押股数    | float64 | 注意单位: 万股 |
| 质押市值    | float64 | 注意单位: 万元 |
| 质押笔数    | float64 | -        |
| 无限售股质押数 | float64 | 注意单位: 万股 |
| 限售股质押数  | float64 | 注意单位: 万股 |
| 近一年涨跌幅  | float64 | 注意单位: %  |
| 所属行业代码  | object  | -        |

接口示例

```python
import akshare as ak

stock_gpzy_pledge_ratio_em_df = ak.stock_gpzy_pledge_ratio_em(date="20241220")
print(stock_gpzy_pledge_ratio_em_df)
```

数据示例

```
      序号  股票代码  股票简称  交易日期  ...   无限售股质押数   限售股质押数 近一年涨跌幅  所属行业代码
0        1  000567  海德股份  2024-12-20  ...  146779.06      0.0   6.965560  016058
1        2  000688  国城矿业  2024-12-20  ...   78404.69      0.0  40.018282  016032
2        3  002485  ST雪发  2024-12-20  ...   37266.20      0.0 -31.400000  016010
3        4  000020  深华发A  2024-12-20  ...   11610.00      0.0  13.596838  016079
4        5  603132  金徽股份  2024-12-20  ...       0.00  59960.0  -5.566960  016032
...    ...     ...   ...         ...  ...        ...      ...        ...     ...
2305  2306  600640  国脉文化  2024-12-20  ...       0.22      0.0   8.495713  016039
2306  2307  600630  龙头股份  2024-12-20  ...       0.57      0.0  -3.577285  016010
2307  2308  002432  九安医疗  2024-12-20  ...       1.20      0.0  13.687060  016082
2308  2309  601989  中国重工  2024-12-20  ...      76.66      0.0  18.591879  016049
2309  2310  600642  申能股份  2024-12-20  ...       5.00      0.0  51.730585  016007
[2310 rows x 13 columns]
```

#### 重要股东股权质押明细

接口: stock_gpzy_pledge_ratio_detail_em

目标地址: https://data.eastmoney.com/gpzy/pledgeDetail.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-重要股东股权质押明细

限量: 单次所有历史数据, 由于数据量比较大需要等待一定时间

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 股票代码    | object  | -       |
| 股票简称    | object  | -       |
| 股东名称    | object  | -       |
| 质押股份数量  | float64 | 注意单位: 股 |
| 占所持股份比例 | float64 | 注意单位: % |
| 占总股本比例  | float64 | 注意单位: % |
| 质押机构    | object  | -       |
| 最新价     | float64 | 注意单位: 元 |
| 质押日收盘价  | float64 | 注意单位: 元 |
| 预估平仓线   | float64 | 注意单位: 元 |
| 公告日期    | object  | -       |
| 质押开始日期  | object  | -       |

接口示例

```python
import akshare as ak

stock_gpzy_pledge_ratio_detail_em_df = ak.stock_gpzy_pledge_ratio_detail_em()
print(stock_gpzy_pledge_ratio_detail_em_df)
```

数据示例

```
        序号    股票代码  股票简称  ...  预估平仓线  质押开始日期    公告日期
0            1  600172  黄河旋风  ...  2.027200  2024-01-04  2024-02-09
1            2  300121  阳谷华泰  ...  2.902200  2024-02-05  2024-02-09
2            3  603866  桃李面包  ...  7.482211  2023-02-08  2024-02-09
3            4  603866  桃李面包  ...  3.830400  2024-02-07  2024-02-09
4            5  603866  桃李面包  ...  3.494400  2024-02-06  2024-02-09
        ...     ...   ...  ...       ...         ...         ...
117840  117841  000028  国药一致  ...  8.605947  2000-12-20  2002-03-29
117841  117842  200028   一致B  ...  8.605947  2000-12-20  2002-03-29
117842  117843  000028  国药一致  ...  8.605947  2000-12-20  2002-03-29
117843  117844  200028   一致B  ...  8.605947  2000-12-20  2002-03-29
117844  117845  600745  闻泰科技  ...  9.841600  2001-11-20  2001-11-24
[117845 rows x 13 columns]
```

#### 质押机构分布统计-证券公司

接口: stock_gpzy_distribute_statistics_company_em

目标地址: https://data.eastmoney.com/gpzy/distributeStatistics.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-质押机构分布统计-证券公司

限量: 单次返回当前时点所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 质押机构         | object  | -       |
| 质押公司数量       | int64   | -       |
| 质押笔数         | int64   | -       |
| 质押数量         | float64 | 注意单位: 股 |
| 未达预警线比例      | float64 | 注意单位: % |
| 达到预警线未达平仓线比例 | float64 | 注意单位: % |
| 达到平仓线比例      | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gpzy_distribute_statistics_company_em_df = ak.stock_gpzy_distribute_statistics_company_em()
print(stock_gpzy_distribute_statistics_company_em_df)
```

数据示例

```
    序号  质押机构  质押公司数量  质押笔数          质押数量   未达预警线比例  达到预警线未达平仓线比例   达到平仓线比例
0    1  海通证券     304  1261  1.402742e+06  0.753370      0.045202  0.201427
1    2  国泰君安     268   904  1.231400e+06  0.743079      0.034330  0.222591
2    3  中信证券     244   639  1.612960e+06  0.668232      0.067293  0.264476
3    4  华泰证券     181   723  1.040893e+06  0.673130      0.036011  0.290859
4    5  招商证券     166   450  4.972209e+05  0.777778      0.068889  0.153333
5    6  广发证券     153   584  5.437655e+05  0.785959      0.046233  0.167808
6    7  中国银河     127   318  6.123945e+05  0.694969      0.050314  0.254717
7    8  中信建投     105   328  5.657925e+05  0.579268      0.085366  0.335366
8    9  中泰证券     101   346  5.231542e+05  0.754335      0.066474  0.179191
9   10  兴业证券      93   326  3.394603e+05  0.616564      0.076687  0.306748
10  11  国信证券      91   451  6.067771e+05  0.558758      0.075388  0.365854
11  12  长江证券      83   272  3.178707e+05  0.573529      0.055147  0.371324
12  13  浙商证券      80   152  1.261287e+05  0.842105      0.039474  0.118421
13  14  国金证券      79   191  2.726742e+05  0.848168      0.047120  0.104712
14  15  东吴证券      63   207  2.763393e+05  0.574879      0.048309  0.376812
15  16  东方证券      62   246  3.957932e+05  0.475610      0.093496  0.430894
16  17  财通证券      59   105  1.227393e+05  0.752381      0.076190  0.171429
17  18  光大证券      56   145  2.391708e+05  0.572414      0.048276  0.379310
18  19  红塔证券      54   157  1.529568e+05  0.878981      0.012739  0.108280
19  20  东北证券      51   152  1.392464e+05  0.539474      0.032895  0.427632
20  21  国联证券      44   115  1.111010e+05  0.730435      0.043478  0.226087
21  22  华西证券      43   190  1.581871e+05  0.821053      0.063158  0.115789
22  23  国元证券      42   125  1.909249e+05  0.792000      0.064000  0.144000
23  24  中金公司      41   125  1.260170e+05  0.832000      0.024000  0.144000
24  25  国海证券      39   196  1.276768e+05  0.500000      0.015306  0.484694
25  26  华安证券      35    75  7.706306e+04  0.933333      0.026667  0.040000
26  27  长城证券      35   107  1.837452e+05  0.542056      0.037383  0.420561
27  28  第一创业      34   179  4.349701e+05  0.240223      0.039106  0.720670
28  29  东兴证券      32   104  1.423795e+05  0.365385      0.057692  0.576923
29  30  方正证券      29   115  1.857242e+05  0.504348      0.095652  0.400000
30  31  西南证券      28    52  8.773574e+04  0.730769      0.038462  0.230769
31  32  中银证券      28    59  5.708684e+04  0.745763      0.000000  0.254237
32  33  天风证券      26    87  1.803283e+05  0.264368      0.011494  0.724138
33  34  山西证券      23    47  7.069562e+04  0.914894      0.000000  0.085106
34  35  中原证券      22    76  7.726518e+04  0.750000      0.052632  0.197368
35  36   太平洋      20    39  6.065551e+04  0.435897      0.025641  0.538462
36  37  财达证券      17    66  7.506754e+04  0.848485      0.030303  0.121212
37  38  南京证券      16    37  2.476809e+04  0.945946      0.054054  0.000000
38  39  西部证券       9    20  2.626103e+04  0.250000      0.000000  0.750000
39  40  华林证券       5     7  3.709550e+03  0.571429      0.000000  0.428571
40  41  申万宏源       3    11  4.012356e+04  1.000000      0.000000  0.000000
```

#### 质押机构分布统计-银行

接口: stock_gpzy_distribute_statistics_bank_em

目标地址: https://data.eastmoney.com/gpzy/distributeStatistics.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-质押机构分布统计-银行

限量: 单次返回当前时点所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 质押机构         | object  | -       |
| 质押公司数量       | int64   | -       |
| 质押笔数         | int64   | -       |
| 质押数量         | float64 | 注意单位: 股 |
| 未达预警线比例      | float64 | 注意单位: % |
| 达到预警线未达平仓线比例 | float64 | 注意单位: % |
| 达到平仓线比例      | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_em_gpzy_distribute_statistics_bank_df = ak.stock_gpzy_distribute_statistics_bank_em()
print(stock_em_gpzy_distribute_statistics_bank_df)
```

数据示例

```
    序号 质押机构 质押公司数量 质押笔数 质押数量 未达预警线比例 达到预警线未达平仓线比例 达到平仓线比例
0    1  浦发银行     200   384  1.051488e+06  0.774278      0.026247  0.199475
1    2  工商银行     194   348  1.124183e+06  0.804734      0.032544  0.162722
2    3  中信银行     155   330  1.017684e+06  0.766667      0.069697  0.163636
3    4  农业银行     126   215  9.640506e+05  0.823256      0.027907  0.148837
4    5  兴业银行     123   221  6.218067e+05  0.854545      0.018182  0.127273
5    6  建设银行     117   223  7.545851e+05  0.820628      0.022422  0.156951
6    7  民生银行     108   180  8.829439e+05  0.782123      0.033520  0.184358
7    8  华夏银行     107   217  4.349071e+05  0.851852      0.027778  0.120370
8    9  交通银行     104   168  6.636462e+05  0.875000      0.029762  0.095238
9   10  中国银行     104   173  1.666726e+06  0.843023      0.023256  0.133721
10  11  招商银行      97   148  4.823712e+05  0.795918      0.034014  0.170068
11  12  浙商银行      89   209  9.515983e+05  0.779904      0.047847  0.172249
12  13  光大银行      81   128  2.443465e+05  0.822581      0.040323  0.137097
13  14  北京银行      56   100  2.244431e+05  0.828283      0.010101  0.161616
14  15  平安银行      51    91  5.740625e+05  0.822222      0.044444  0.133333
15  16  杭州银行      41    68  9.376276e+04  0.800000      0.092308  0.107692
16  17  江苏银行      36    59  1.707125e+05  0.813559      0.050847  0.135593
17  18  上海银行      29    61  2.313567e+05  0.852459      0.016393  0.131148
18  19  南京银行      26    39  5.202060e+04  0.675676      0.081081  0.243243
19  20  长沙银行      16    40  1.029290e+05  0.725000      0.050000  0.225000
20  21  苏州银行      15    25  3.934501e+04  0.869565      0.043478  0.086957
21  22  宁波银行      15    19  3.920183e+04  0.631579      0.052632  0.315789
22  23  邮储银行      12    22  6.008693e+04  0.954545      0.045455  0.000000
23  24  青岛银行      11    17  4.050235e+04  0.882353      0.058824  0.058824
24  25  沪农商行       9    17  6.490389e+04  1.000000      0.000000  0.000000
25  26  兰州银行       8    12  3.771000e+04  0.916667      0.000000  0.083333
26  27  苏农银行       7    18  2.268900e+04  1.000000      0.000000  0.000000
27  28  无锡银行       7    12  1.764020e+04  0.750000      0.083333  0.166667
28  29  郑州银行       6    24  5.408552e+04  0.916667      0.000000  0.083333
29  30  厦门银行       5    14  3.593239e+04  0.785714      0.071429  0.142857
30  31  西安银行       5     6  4.349401e+04  1.000000      0.000000  0.000000
31  32  成都银行       5     5  2.991586e+04  1.000000      0.000000  0.000000
32  33  齐鲁银行       5    10  1.715242e+04  0.875000      0.000000  0.125000
33  34  张家港行       4     6  5.063094e+03  0.833333      0.000000  0.166667
34  35  瑞丰银行       4     5  7.770000e+03  0.400000      0.000000  0.600000
35  36  紫金银行       3     5  2.280000e+03  1.000000      0.000000  0.000000
36  37  青农商行       3     9  1.980000e+04  1.000000      0.000000  0.000000
37  38  渝农商行       3    24  2.165320e+05  0.666667      0.041667  0.291667
38  39  重庆银行       2     2  5.800000e+03  1.000000      0.000000  0.000000
39  40  江阴银行       2     3  3.160000e+03  1.000000      0.000000  0.000000
40  41  贵阳银行       1     2  8.930000e+02  1.000000      0.000000  0.000000
41  42  常熟银行       1     2  2.750000e+03  1.000000      0.000000  0.000000
```

#### 上市公司质押比例

接口: stock_gpzy_industry_data_em

目标地址: https://data.eastmoney.com/gpzy/industryData.aspx

描述: 东方财富网-数据中心-特色数据-股权质押-上市公司质押比例-行业数据

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 行业     | object  | -       |
| 平均质押比例 | float64 | 注意单位: % |
| 公司家数   | float64 | -       |
| 质押总笔数  | float64 | -       |
| 质押总股本  | float64 | -       |
| 最新质押市值 | float64 | -       |
| 统计时间   | object  | -       |

接口示例

```python
import akshare as ak

stock_gpzy_industry_data_em_df = ak.stock_gpzy_industry_data_em()
print(stock_gpzy_industry_data_em_df)
```

数据示例

```
    序号  行业    平均质押比例  公司家数  质押总笔数 质押总股本 最新质押市值     统计时间
0    1   商业百货  20.886207    29    288  1301231.97  5.579849e+06  2024-12-20
1    2   航空机场  19.988000     5    109  1361284.27  4.552621e+06  2024-12-20
2    3   多元金融  19.492308    13    111   711850.81  3.957603e+06  2024-12-20
3    4   综合行业  18.979091    11    113   424101.84  2.326736e+06  2024-12-20
4    5  房地产开发  18.584063    32    313  1702899.22  6.166608e+06  2024-12-20
..  ..    ...        ...   ...    ...         ...           ...         ...
81  82   专业服务   5.231111     9     19    20018.84  1.993565e+05  2024-12-20
82  83  电子化学品   5.005385    13     27    30328.42  5.075712e+05  2024-12-20
83  84   铁路公路   4.925556     9     16   183142.87  1.489794e+06  2024-12-20
84  85     银行   4.596190    42    684  1827327.07  8.201688e+06  2024-12-20
85  86     保险   0.420000     3     12     6925.00  9.603522e+04  2024-12-20
[86 rows x 8 columns]
```

### 商誉专题

#### A股商誉市场概况

接口: stock_sy_profile_em

目标地址:  https://data.eastmoney.com/sy/scgk.html

描述: 东方财富网-数据中心-特色数据-商誉-A股商誉市场概况

限量: 单次所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 报告期        | object  | -       |
| 商誉         | float64 | 注意单位: 元 |
| 商誉减值       | float64 | 注意单位: 元 |
| 净资产        | float64 | 注意单位: 元 |
| 商誉占净资产比例   | float64 | -       |
| 商誉减值占净资产比例 | float64 | -       |
| 净利润规模      | float64 | 注意单位: 元 |
| 商誉减值占净利润比例 | float64 | -       |

接口示例

```python
import akshare as ak

stock_sy_profile_em_df = ak.stock_sy_profile_em()
print(stock_sy_profile_em_df)
```

数据示例

```
           报告期            商誉  ...         净利润规模  商誉减值占净利润比例
0   2010-12-31  9.305439e+10  ...  8.646720e+11   -0.008547
1   2011-12-31  1.334065e+11  ...  1.030624e+12   -0.001177
2   2012-12-31  1.639409e+11  ...  1.224410e+12   -0.000985
3   2013-12-31  2.051656e+11  ...  1.441550e+12   -0.001916
4   2014-12-31  3.246068e+11  ...  1.604929e+12   -0.001784
5   2015-12-31  5.985505e+11  ...  1.757708e+12   -0.005302
6   2016-12-31  9.667876e+11  ...  1.873202e+12   -0.008425
7   2017-12-31  1.195754e+12  ...  2.244254e+12   -0.011668
8   2018-12-31  1.219315e+12  ...  2.327512e+12   -0.049232
9   2019-12-31  1.190470e+12  ...  2.506131e+12   -0.045300
10  2020-12-31  1.157767e+12  ...  2.606449e+12   -0.028383
11  2021-12-31  1.156240e+12  ...  3.307810e+12   -0.015126
12  2022-12-31  1.198481e+12  ...  3.773819e+12   -0.015187
13  2023-12-31  1.226307e+12  ...  3.745003e+12         NaN
14  2024-06-30  1.246240e+12  ...  2.049446e+12         NaN
15  2024-09-30  1.254842e+12  ...  3.089467e+12         NaN
[16 rows x 8 columns]
```

#### 商誉减值预期明细

接口: stock_sy_yq_em

目标地址: https://data.eastmoney.com/sy/yqlist.html

描述: 东方财富网-数据中心-特色数据-商誉-商誉减值预期明细

限量: 单次所有历史数据

输入参数

| 名称   | 类型  | 描述                      |
|------|-----|-------------------------|
| date | str | date="20221231"; 参见网页选项 |

输出参数

| 名称        | 类型      | 描述      |
|-----------|---------|---------|
| 序号        | int64   | -       |
| 股票代码      | object  | -       |
| 股票简称      | object  | -       |
| 业绩变动原因    | object  | -       |
| 最新商誉报告期   | object  | -       |
| 最新一期商誉    | float64 | 主要单位: 元 |
| 上年商誉      | float64 | 主要单位: 元 |
| 预计净利润-下限  | int64   | 主要单位: 元 |
| 预计净利润-上限  | int64   | 主要单位: 元 |
| 业绩变动幅度-下限 | float64 | 主要单位: % |
| 业绩变动幅度-上限 | float64 | 主要单位: % |
| 上年度同期净利润  | float64 | 主要单位: 元 |
| 公告日期      | object  | -       |
| 交易市场      | object  | -       |

接口示例

```python
import akshare as ak

stock_sy_yq_em_df = ak.stock_sy_yq_em(date="20221231")
print(stock_sy_yq_em_df)
```

数据示例

```
      序号    股票代码  股票简称  ...     上年度同期净利润        公告日期  交易市场
0      1  000010  美丽生态  ...   23458700.0  2023-04-22  深市主板
1      2  300069  金利华电  ...  -40111200.0  2023-04-18   创业板
2      3  000546  ST金圆  ...   86300100.0  2023-04-17  深市主板
3      4  002636  金安国纪  ...  690190000.0  2023-04-15  深市主板
4      5  002251   步步高  ... -184132100.0  2023-04-15  深市主板
..   ...     ...   ...  ...          ...         ...   ...
292  293  300452  山河药辅  ...   89253400.0  2023-01-10   创业板
293  294  300009  安科生物  ...  206629500.0  2023-01-10   创业板
294  295  002137   实益达  ...  -95330400.0  2023-01-10  深市主板
295  296  000576  甘化科工  ...   31860000.0  2023-01-06  深市主板
296  297  002111  威海广泰  ...   67754400.0  2022-10-29  深市主板
[297 rows x 14 columns]
```

#### 个股商誉减值明细

接口: stock_sy_jz_em

目标地址: https://data.eastmoney.com/sy/jzlist.html

描述: 东方财富网-数据中心-特色数据-商誉-个股商誉减值明细

限量: 单次返回所有历史数据

输入参数

| 名称   | 类型  | 描述                      |
|------|-----|-------------------------|
| date | str | date="20230331"; 参见网页选项 |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 股票代码       | object  | -       |
| 股票简称       | object  | -       |
| 商誉         | float64 | 注意单位: 元 |
| 商誉减值       | float64 | 注意单位: 元 |
| 商誉减值占净资产比例 | float64 | -       |
| 净利润        | float64 | 注意单位: 元 |
| 商誉减值占净利润比例 | float64 | -       |
| 公告日期       | object  | -       |
| 交易市场       | object  | -       |

接口示例

```python
import akshare as ak

stock_sy_jz_em_df = ak.stock_sy_jz_em(date="20230331")
print(stock_sy_jz_em_df)
```

数据示例

```
        序号    股票代码  股票简称  ...  商誉减值占净利润比例        公告日期  交易市场
0        1  600587  新华医疗  ...         NaN  2023-04-28  沪市主板
1        2  000829  天音控股  ...         NaN  2023-04-29  深市主板
2        3  600271  航天信息  ...         NaN  2023-04-29  沪市主板
3        4  600383  金地集团  ...         NaN  2023-04-29  沪市主板
4        5  000917  电广传媒  ...         NaN  2023-04-28  深市主板
    ...     ...   ...  ...         ...         ...   ...
2446  2447  002846  英联股份  ...         NaN  2023-04-29  深市主板
2447  2448  688625  呈和科技  ...         NaN  2023-04-25   科创板
2448  2449  601995  中金公司  ...         NaN  2023-04-29  沪市主板
2449  2450  002724   海洋王  ...         NaN  2023-04-28  深市主板
2450  2451  002815  崇达技术  ...         NaN  2023-04-27  深市主板
[2451 rows x 11 columns]
```

#### 个股商誉明细

接口: stock_sy_em

目标地址: https://data.eastmoney.com/sy/list.html

描述: 东方财富网-数据中心-特色数据-商誉-个股商誉明细

限量: 单次返回所有历史数据

输入参数

| 名称   | 类型  | 描述                      |
|------|-----|-------------------------|
| date | str | date="20240630"; 参见网页选项 |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 股票代码     | object  | -       |
| 股票简称     | object  | -       |
| 商誉       | float64 | 注意单位: 元 |
| 商誉占净资产比例 | float64 |         |
| 净利润      | float64 | 注意单位: 元 |
| 净利润同比    | float64 |         |
| 上年商誉     | float64 | 注意单位: 元 |
| 公告日期     | object  | -       |
| 交易市场     | object  | -       |

接口示例

```python
import akshare as ak

stock_sy_em_df = ak.stock_sy_em(date="20240630")
print(stock_sy_em_df)
```

数据示例

```
     序号    股票代码  股票简称  ...      上年商誉    公告日期  交易市场
0        1  688799  华纳药厂  ...  3.238198e+07  2024-08-31   科创板
1        2  688665  四方光电  ...           NaN  2024-08-31   科创板
2        3  688601   力芯微  ...  8.354801e+06  2024-08-31   科创板
3        4  688599  天合光能  ...  1.542976e+08  2024-08-31   科创板
4        5  688559   海目星  ...           NaN  2024-08-31   科创板
...    ...     ...   ...  ...           ...         ...   ...
2630  2631  000990  诚志股份  ...  6.754695e+09  2024-07-26  深市主板
2631  2632  601231  环旭电子  ...  6.090252e+08  2024-07-25  沪市主板
2632  2633  603668  天马科技  ...  1.170249e+08  2024-07-23  沪市主板
2633  2634  002245  蔚蓝锂芯  ...  5.973838e+08  2024-07-23  深市主板
2634  2635  002648  卫星化学  ...  4.439731e+07  2024-07-18  深市主板
[2635 rows x 10 columns]
```

#### 行业商誉

接口: stock_sy_hy_em

目标地址: https://data.eastmoney.com/sy/hylist.html

描述: 东方财富网-数据中心-特色数据-商誉-行业商誉

限量: 单次返回所有历史数据

输入参数

| 名称   | 类型  | 描述                      |
|------|-----|-------------------------|
| date | str | date="20240930"; 参见网页选项 |

输出参数

| 名称           | 类型      | 描述 |
|--------------|---------|----|
| 行业名称         | object  | -  |
| 公司家数         | int64   | -  |
| 商誉规模         | float64 | -  |
| 净资产          | float64 | -  |
| 商誉规模占净资产规模比例 | float64 | -  |
| 净利润规模        | float64 | -  |

接口示例

```python
import akshare as ak

stock_sy_hy_em_df = ak.stock_sy_hy_em(date="20240930")
print(stock_sy_hy_em_df)
```

数据示例

```
     行业名称  公司家数  商誉规模    净资产  商誉规模占净资产规模比例   净利润规模
0      游戏    22  3.021155e+10  1.206944e+11      0.250314  8.439657e+09
1      教育    11  3.216185e+09  1.383355e+10      0.232492  5.010300e+08
2   房地产服务    12  8.534555e+09  4.474023e+10      0.190758  3.724827e+07
3    航空机场     3  1.397192e+10  8.271383e+10      0.168919  3.396607e+09
4    医药商业    25  4.362616e+10  2.754810e+11      0.158364  1.496122e+10
..    ...   ...           ...           ...           ...           ...
81   工程建设    44  1.700281e+10  3.263842e+12      0.005209  1.297234e+11
82   钢铁行业    18  1.975099e+09  4.997408e+11      0.003952  1.588223e+10
83   煤炭行业    12  1.658931e+09  5.478767e+11      0.003028  3.957996e+10
84   化纤行业    11  4.119727e+08  1.538797e+11      0.002677  8.734288e+09
85     银行    12  3.267882e+10  1.433628e+13      0.002279  1.027638e+12
[86 rows x 6 columns]
```

### 股票账户统计

#### 股票账户统计月度

接口: stock_account_statistics_em

目标地址: https://data.eastmoney.com/cjsj/gpkhsj.html

描述: 东方财富网-数据中心-特色数据-股票账户统计

限量: 单次返回从 201504 开始 202308 的所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称         | 类型      | 描述       |
|------------|---------|----------|
| 数据日期       | object  | -        |
| 新增投资者-数量   | float64 | 注意单位: 万户 |
| 新增投资者-环比   | float64 | -        |
| 新增投资者-同比   | float64 | -        |
| 期末投资者-总量   | float64 | 注意单位: 万户 |
| 期末投资者-A股账户 | float64 | 注意单位: 万户 |
| 期末投资者-B股账户 | float64 | 注意单位: 万户 |
| 沪深总市值      | float64 | -        |
| 沪深户均市值     | float64 | 注意单位: 万  |
| 上证指数-收盘    | float64 | -        |
| 上证指数-涨跌幅   | float64 | -        |

接口示例

```python
import akshare as ak

stock_account_statistics_em_df = ak.stock_account_statistics_em()
print(stock_account_statistics_em_df)
```

数据示例

```
     数据日期  新增投资者-数量  新增投资者-环比  ...   沪深户均市值  上证指数-收盘  上证指数-涨跌幅
0    2015-04    497.53       NaN  ...  69.4956  4441.6550   18.5105
1    2015-05    415.87   -0.1641  ...  73.6062  4611.7440    3.8294
2    2015-06    464.22    0.1163  ...  65.0300  4277.2219   -7.2537
3    2015-07    204.87   -0.5587  ...  54.9082  3663.7256  -14.3433
4    2015-08    136.85   -0.3320  ...  46.9417  3205.9855  -12.4938
..       ...       ...       ...  ...      ...        ...       ...
96   2023-04    100.19   -0.4716  ...  39.1565  3323.2746    1.5404
97   2023-05     97.26   -0.0292  ...  37.8637  3204.5644   -3.5721
98   2023-06     98.13    0.0089  ...  38.0336  3202.0623   -0.0781
99   2023-07     91.03   -0.0724  ...  38.6950  3291.0398    2.7788
100  2023-08     99.59    0.0940  ...  36.6206  3119.8764   -5.2009
[101 rows x 11 columns]
```

### 分析师指数

#### 分析师指数排行

接口: stock_analyst_rank_em

目标地址: https://data.eastmoney.com/invest/invest/list.html

描述: 东方财富网-数据中心-研究报告-东方财富分析师指数

限量: 单次获取指定年份的所有数据

输入参数

| 名称   | 类型  | 描述                      |
|------|-----|-------------------------|
| year | str | year='2024'; 从 2013 年至今 |

输出参数

| 名称              | 类型      | 描述                       |
|-----------------|---------|--------------------------|
| 序号              | int64   | -                        |
| 分析师名称           | object  | -                        |
| 分析师单位           | object  | -                        |
| 年度指数            | float64 | -                        |
| xxxx年收益率        | float64 | 其中 xxxx 表示指定的年份; 注意单位: % |
| 3个月收益率          | float64 | 注意单位: %                  |
| 6个月收益率          | float64 | 注意单位: %                  |
| 12个月收益率         | float64 | 注意单位: %                  |
| 成分股个数           | int64   | -                        |
| xxxx最新个股评级-股票名称 | object  | 其中 xxxx 表示指定的年份          |
| xxxx最新个股评级-股票代码 | object  | 其中 xxxx 表示指定的年份          |
| 分析师ID           | object  | -                        |
| 行业代码            | object  | -                        |
| 行业              | object  | -                        |
| 更新日期            | object  | 数据更新日期                   |
| 年度              | object  | 数据更新年度                   |

接口示例

```python
import akshare as ak

stock_analyst_rank_em_df = ak.stock_analyst_rank_em(year='2024')
print(stock_analyst_rank_em_df)
```

数据示例

```
    序号 分析师名称 分析师单位 年度指数  ... 行业代码    行业  更新日期    年度
0     1   任志强  华福证券  6424.01  ...  270000    电子  2024-12-27  2024
1     2    洪烨  中国银河  2303.30  ...  280000    汽车  2024-12-27  2024
2     3   范想想  中国银河  2521.64  ...    None  None  2024-12-27  2024
3     4    张宁  国联证券  2207.86  ...  730000    通信  2024-12-27  2024
4     5    周晴  中邮证券  2118.82  ...  270000    电子  2024-12-27  2024
..  ...   ...   ...      ...  ...     ...   ...         ...   ...
95   96   舒思勤  国金证券  1356.88  ...  490000  非银金融  2024-12-27  2024
96   97   郑庆明  申万宏源  1188.05  ...  480000    银行  2024-12-27  2024
97   98   陈宁玉  中泰证券  1304.21  ...  730000    通信  2024-12-27  2024
98   99    田源  华源证券  1555.00  ...  240000  有色金属  2024-12-27  2024
99  100    常菁  中金公司  2595.04  ...  280000    汽车  2024-12-27  2024
[100 rows x 16 columns]
```

#### 分析师详情

接口: stock_analyst_detail_em

目标地址: https://data.eastmoney.com/invest/invest/11000257131.html

描述: 东方财富网-数据中心-研究报告-东方财富分析师指数-分析师详情

限量: 单次获取指定 indicator 指定的数据

输入参数

| 名称         | 类型  | 描述                                                               |
|------------|-----|------------------------------------------------------------------|
| analyst_id | str | analyst_id="11000257131"; 分析师ID, 从 ak.stock_analyst_rank_em() 获取 |
| indicator  | str | indicator="最新跟踪成分股"; 从 {"最新跟踪成分股", "历史跟踪成分股", "历史指数"} 中选择        |

输出参数-最新跟踪成分股

| 名称        | 类型      | 描述      |
|-----------|---------|---------|
| 序号        | int64   | -       |
| 股票代码      | object  | -       |
| 股票名称      | object  | -       |
| 调入日期      | object  | -       |
| 最新评级日期    | object  | -       |
| 当前评级名称    | object  | -       |
| 成交价格(前复权) | float64 | -       |
| 最新价格      | float64 | -       |
| 阶段涨跌幅     | float64 | 注意单位: % |

接口示例-最新跟踪成分股

```python
import akshare as ak

stock_analyst_detail_em_df = ak.stock_analyst_detail_em(analyst_id="11000200926", indicator="最新跟踪成分股")
print(stock_analyst_detail_em_df)
```

数据示例-最新跟踪成分股

```
    序号    股票代码   股票名称        调入日期  ... 当前评级名称   成交价格(前复权)    最新价格   阶段涨跌幅
0    1  001269   欧晶科技  2023-01-31  ...     买入  118.980000  114.97   -3.37
1    2  301155   海力风电  2022-12-12  ...     买入   90.880000   95.09    4.63
2    3  002865   钧达股份  2022-12-05  ...     买入  198.990000  177.10  -11.00
3    4  600152   维科技术  2022-12-02  ...     买入   17.490000   14.10  -19.38
4    5  300037    新宙邦  2022-11-08  ...     买入   41.200000   47.27   14.73
5    6  603876   鼎胜新材  2022-11-07  ...     买入   53.000000   47.64  -10.11
6    7  603606   东方电缆  2022-11-03  ...     买入   73.600000   58.12  -21.03
7    8  600732   爱旭股份  2022-10-18  ...     买入   36.140000   36.38    0.66
8    9  603799   华友钴业  2022-09-19  ...     买入   72.400000   64.75  -10.57
9   10  300438   鹏辉能源  2022-08-31  ...     买入   82.000000   74.23   -9.48
10  11  002709   天赐材料  2022-08-29  ...     买入   49.380000   47.18   -4.46
11  12  002129  TCL中环  2022-08-26  ...     买入   51.050000   42.30  -17.14
12  13  603218   日月股份  2022-08-25  ...     买入   26.050000   23.99   -7.91
13  14  600869   远东股份  2022-08-12  ...     买入    7.030000    5.44  -22.62
14  15  300274   阳光电源  2022-04-21  ...     买入   64.404849  121.39   88.49
15  16  002623    亚玛顿  2022-04-01  ...     买入   25.824010   32.72   26.72
16  17  605117   德业股份  2022-03-09  ...     买入  189.839120  322.50   69.88
17  18  300772   运达股份  2022-02-14  ...     买入   18.627516   16.24  -12.83
18  19  300014   亿纬锂能  2021-07-15  ...     买入  126.804616   86.79  -31.55
19  20  002074   国轩高科  2021-07-07  ...     买入   47.194889   32.14  -31.89
20  21  688063   派能科技  2021-04-20  ...     买入  166.678770  287.20   72.31
21  22  300568   星源材质  2021-03-17  ...     买入   10.698750   23.06  115.51
22  23  002594    比亚迪  2021-03-02  ...     买入  211.229182  289.98   37.28
23  24  300861   美畅股份  2021-03-01  ...     买入   47.082762   53.04   12.66
24  25  600438   通威股份  2021-01-26  ...     买入   44.615150   40.75   -8.67
25  26  300073   当升科技  2021-01-07  ...     买入   64.161457   65.35    1.85
26  27  300035   中科电气  2020-09-28  ...     买入    8.613592   19.90  131.13
27  28  002812   恩捷股份  2020-06-01  ...     买入   59.938614  145.99  143.56
28  29  601012   隆基绿能  2020-05-26  ...     买入   15.825182   46.40  193.11
29  30  300750   宁德时代  2020-05-18  ...     买入  141.320309  450.60  218.85
30  31  002080   中材科技  2020-02-10  ...     买入   12.347362   23.89   93.44
```

输出参数-历史跟踪成分股

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 股票代码    | object  | -       |
| 股票名称    | object  | -       |
| 调入日期    | object  | -       |
| 调出日期    | object  | -       |
| 调入时评级名称 | object  | -       |
| 调出原因    | object  | -       |
| 累计涨跌幅   | float64 | 注意单位: % |

接口示例-历史跟踪成分股

```python
import akshare as ak

stock_em_analyst_detail_df = ak.stock_analyst_detail_em(analyst_id="11000200926", indicator="历史跟踪成分股")
print(stock_em_analyst_detail_df)
```

数据示例-历史跟踪成分股

```
    序号 股票代码   股票名称 调入日期    调出日期 调入时评级名称  调出原因   累计涨跌幅
0    1  603659    璞泰来  2024-05-23  2024-11-25      买入  到期调出   22.20
1    2  002850    科达利  2024-05-22  2024-11-22      买入  到期调出    4.73
2    3  301155   海力风电  2024-05-17  2024-11-18      买入  到期调出   21.72
3    4  688390    固德威  2024-05-13  2024-11-12      买入  到期调出  -26.18
4    5  603507   振江股份  2024-05-07  2024-11-07      买入  到期调出   12.23
..  ..     ...    ...         ...         ...     ...   ...     ...
70  71  002129  TCL中环  2019-08-09  2020-04-29      买入  到期调出   48.61
71  72  603659    璞泰来  2019-03-19  2019-09-25      买入  到期调出   -6.26
72  73  002202   金风科技  2019-02-25  2020-04-29      买入  到期调出  -27.07
73  74  601012   隆基绿能  2018-12-27  2020-05-06      买入  到期调出  114.24
74  75  600438   通威股份  2018-11-19  2020-04-27      买入  到期调出   68.18
[75 rows x 8 columns]
```

输出参数-历史指数

| 名称    | 类型      | 描述                   |
|-------|---------|----------------------|
| date  | object  | 日期                   |
| value | float64 | 指数数值; 注意: 此指数为东方财富制定 |

接口示例-历史指数

```python
import akshare as ak

stock_em_analyst_detail_df = ak.stock_analyst_detail_em(analyst_id="11000200926", indicator="历史指数")
print(stock_em_analyst_detail_df)
```

数据示例-历史指数

```
            date        value
0     2018-11-19  1000.000000
1     2018-11-20   970.738195
2     2018-11-21  1011.450675
3     2018-11-22  1003.817085
4     2018-11-23   989.822170
...          ...          ...
1479  2024-12-23  5466.064721
1480  2024-12-24  5572.478788
1481  2024-12-25  5521.539014
1482  2024-12-26  5550.570662
1483  2024-12-27  5507.349611
[1484 rows x 2 columns]
```

### 千股千评

接口: stock_comment_em

目标地址: https://data.eastmoney.com/stockcomment/

描述: 东方财富网-数据中心-特色数据-千股千评

限量: 单次获取所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 代码    | object  | -       |
| 名称    | object  | -       |
| 最新价   | float64 | -       |
| 涨跌幅   | float64 | -       |
| 换手率   | float64 | 注意单位: % |
| 市盈率   | float64 | -       |
| 主力成本  | float64 | -       |
| 机构参与度 | float64 | -       |
| 综合得分  | float64 | -       |
| 上升    | int64   | 注意: 正负号 |
| 目前排名  | int64   | -       |
| 关注指数  | float64 | -       |
| 交易日   | float64 | -       |

接口示例

```python
import akshare as ak

stock_comment_em_df = ak.stock_comment_em()
print(stock_comment_em_df)
```

数据示例

```
        序号    代码    名称    最新价  ...    上升  目前排名  关注指数  交易日
0        1  000001     平安银行  10.50  ...  1090  1608  87.6  2024-09-25
1        2  000002    万  科Ａ   7.31  ...  -675   293  90.4  2024-09-25
2        3  000004     国华网安  15.28  ...  -643  4283  91.6  2024-09-25
3        4  000006     深振业Ａ   4.38  ...   704  4043  83.2  2024-09-25
4        5  000007      全新好   5.62  ...   -86  2940  73.2  2024-09-25
...    ...     ...      ...    ...  ...   ...   ...   ...         ...
5076  5077  688799     华纳药厂  40.10  ...  1005  2019  63.2  2024-09-25
5077  5078  688800      瑞可达  21.11  ...  -326  1166  63.6  2024-09-25
5078  5079  688819     天能股份  27.86  ...  -504   568  74.8  2024-09-25
5079  5080  688981     中芯国际  44.90  ...  1274  2669  79.6  2024-09-25
5080  5081  689009  九号公司-WD  40.34  ...  -519   574  70.8  2024-09-25
[5081 rows x 14 columns]
```

### 千股千评详情

#### 主力控盘

##### 机构参与度

接口: stock_comment_detail_zlkp_jgcyd_em

目标地址: https://data.eastmoney.com/stockcomment/stock/600000.html

描述: 东方财富网-数据中心-特色数据-千股千评-主力控盘-机构参与度

限量: 单次获取所有 symbol 的数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600000" |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 交易日   | object  | -       |
| 机构参与度 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_comment_detail_zlkp_jgcyd_em_df = ak.stock_comment_detail_zlkp_jgcyd_em(symbol="600000")
print(stock_comment_detail_zlkp_jgcyd_em_df)
```

数据示例

```
         交易日  机构参与度
0   2024-07-25  28.90860
1   2024-07-26  30.96772
2   2024-07-29  33.18196
3   2024-07-30  29.74948
4   2024-07-31  26.50768
5   2024-08-01  31.43596
6   2024-08-02  30.74792
7   2024-08-05  30.00552
8   2024-08-06  28.46816
9   2024-08-07  27.28048
10  2024-08-08  25.62168
11  2024-08-09  28.76180
12  2024-08-12  22.34396
13  2024-08-13  25.12816
14  2024-08-14  19.10376
15  2024-08-15  26.32104
16  2024-08-16  27.09764
17  2024-08-19  33.85404
18  2024-08-20  26.12312
19  2024-08-21  24.80908
20  2024-08-22  23.36916
21  2024-08-23  27.57504
22  2024-08-26  29.63956
23  2024-08-27  23.29028
24  2024-08-28  20.18936
25  2024-08-29  24.87316
26  2024-08-30  24.68128
27  2024-09-02  25.37456
28  2024-09-03  27.86884
29  2024-09-04  25.54656
30  2024-09-05  22.02124
31  2024-09-06  19.94916
32  2024-09-09  17.86148
33  2024-09-10  34.34128
34  2024-09-11  24.62652
35  2024-09-12  34.83748
36  2024-09-13  24.70888
37  2024-09-18  26.05112
38  2024-09-19  24.63440
39  2024-09-20  18.15912
40  2024-09-23  24.80804
41  2024-09-24  30.84384
```

#### 综合评价

##### 历史评分

接口: stock_comment_detail_zhpj_lspf_em

目标地址: https://data.eastmoney.com/stockcomment/stock/600000.html

描述: 东方财富网-数据中心-特色数据-千股千评-综合评价-历史评分

限量: 单次获取指定 symbol 的数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600000" |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 评分  | float64 | -   |

接口示例

```python
import akshare as ak

stock_comment_detail_zhpj_lspf_em_df = ak.stock_comment_detail_zhpj_lspf_em(symbol="600000")
print(stock_comment_detail_zhpj_lspf_em_df)
```

数据示例

```
      交易日         评分
0   2024-08-13  62.512173
1   2024-08-14  61.670775
2   2024-08-15  63.852884
3   2024-08-16  63.075082
4   2024-08-19  68.835333
5   2024-08-20  64.755213
6   2024-08-21  64.289915
7   2024-08-22  64.394597
8   2024-08-23  61.212378
9   2024-08-26  63.819157
10  2024-08-27  62.614203
11  2024-08-28  63.118215
12  2024-08-29  60.579603
13  2024-08-30  62.166671
14  2024-09-02  66.166225
15  2024-09-03  61.715196
16  2024-09-04  61.016823
17  2024-09-05  60.898236
18  2024-09-06  59.813002
19  2024-09-09  60.014454
20  2024-09-10  64.019089
21  2024-09-11  59.987986
22  2024-09-12  62.374324
23  2024-09-13  63.184084
24  2024-09-18  62.877090
25  2024-09-19  62.136208
26  2024-09-20  59.141378
27  2024-09-23  60.765932
28  2024-09-24  67.036800
29  2024-09-25  73.137028
```

#### 市场热度

##### 用户关注指数

接口: stock_comment_detail_scrd_focus_em

目标地址: https://data.eastmoney.com/stockcomment/stock/600000.html

描述: 东方财富网-数据中心-特色数据-千股千评-市场热度-用户关注指数

限量: 单次获取所有数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600000" |

输出参数

| 名称     | 类型      | 描述 |
|--------|---------|----|
| 交易日    | object  | -  |
| 用户关注指数 | float64 | -  |

接口示例

```python
import akshare as ak

stock_comment_detail_scrd_focus_em_df = ak.stock_comment_detail_scrd_focus_em(symbol="600000")
print(stock_comment_detail_scrd_focus_em_df)
```

数据示例

```
       交易日  用户关注指数
0   2024-08-15    91.2
1   2024-08-16    91.6
2   2024-08-19    92.4
3   2024-08-20    92.4
4   2024-08-21    92.0
5   2024-08-22    92.0
6   2024-08-23    92.4
7   2024-08-26    92.4
8   2024-08-27    91.6
9   2024-08-28    92.0
10  2024-08-29    92.0
11  2024-08-30    91.2
12  2024-09-02    91.2
13  2024-09-03    91.2
14  2024-09-04    91.2
15  2024-09-05    91.2
16  2024-09-06    91.2
17  2024-09-09    92.4
18  2024-09-10    92.0
19  2024-09-11    91.6
20  2024-09-12    91.6
21  2024-09-13    93.2
22  2024-09-18    93.2
23  2024-09-19    92.8
24  2024-09-20    92.4
25  2024-09-23    92.8
26  2024-09-24    93.2
27  2024-09-25    93.2
28  2024-09-26    93.2
29  2024-09-27    92.4
```

##### 市场参与意愿

接口: stock_comment_detail_scrd_desire_em

目标地址: https://data.eastmoney.com/stockcomment/stock/600000.html

描述: 东方财富网-数据中心-特色数据-千股千评-市场热度-市场参与意愿

限量: 单次获取所有数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600000" |

输出参数

| 名称       | 类型      | 描述 |
|----------|---------|----|
| 交易日期     | object  | -  |
| 股票代码     | object  | -  |
| 参与意愿     | float64 | -  |
| 5日平均参与意愿 | float64 | -  |
| 参与意愿变化   | float64 | -  |
| 5日平均变化   | float64 | -  |

接口示例

```python
import akshare as ak

stock_comment_detail_scrd_desire_em_df = ak.stock_comment_detail_scrd_desire_em(symbol="600000")
print(stock_comment_detail_scrd_desire_em_df)
```

数据示例

```
   交易日期    股票代码   参与意愿  5日平均参与意愿  参与意愿变化  5日平均变化
0  2025-12-25  600000  47.31     51.41   -9.50   -4.04
1  2025-12-26  600000  35.05     50.74  -12.26    1.00
2  2025-12-29  600000  65.52     54.55   30.47    3.81
3  2025-12-30  600000  52.52     51.44  -13.00   -3.10
4  2025-12-31  600000  51.85     50.45    1.00    1.00
```

### 沪深港通资金流向

接口: stock_hsgt_fund_flow_summary_em

目标地址: https://data.eastmoney.com/hsgt/index.html#lssj

描述: 东方财富网-数据中心-资金流向-沪深港通资金流向

限量: 单次获取沪深港通资金流向数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述       |
|--------|---------|----------|
| 交易日    | object  | -        |
| 类型     | object  | -        |
| 板块     | object  | -        |
| 资金方向   | object  | -        |
| 交易状态   | int64   | 3 为收盘    |
| 成交净买额  | float64 | 注意单位: 亿元 |
| 资金净流入  | float64 | 注意单位: 亿元 |
| 当日资金余额 | float64 | 注意单位: 亿元 |
| 上涨数    | int64   | -        |
| 持平数    | int64   | -        |
| 下跌数    | int64   | -        |
| 相关指数   | object  | -        |
| 指数涨跌幅  | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_hsgt_fund_flow_summary_em_df = ak.stock_hsgt_fund_flow_summary_em()
print(stock_hsgt_fund_flow_summary_em_df)
```

数据示例

```
      交易日    类型      板块 资金方向  交易状态  ... 上涨数  持平数  下跌数  相关指数  指数涨跌幅
0  2022-11-25  沪港通     沪股通   北向     3  ...  302    8  284  上证指数   0.40
1  2022-11-25  沪港通  港股通(沪)   南向     3  ...  179   16  186  恒生指数  -0.49
2  2022-11-25  深港通     深股通   北向     3  ...  283   29  621  深证成指  -0.48
3  2022-11-25  深港通  港股通(深)   南向     3  ...  247   33  268  恒生指数  -0.49
[4 rows x 13 columns]
```

### 沪深港通持股

#### 结算汇率-深港通

接口: stock_sgt_settlement_exchange_rate_szse

目标地址: https://www.szse.cn/szhk/hkbussiness/exchangerate/index.html

描述: 深港通-港股通业务信息-结算汇率

限量: 单次获取所有深港通结算汇率数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述  |
|----------|---------|-----|
| 适用日期     | object  | -   |
| 买入结算汇兑比率 | float64 | -   |
| 卖出结算汇兑比率 | float64 | -   |
| 货币种类     | object  | -   |

接口示例

```python
import akshare as ak

stock_sgt_settlement_exchange_rate_szse_df = ak.stock_sgt_settlement_exchange_rate_szse()
print(stock_sgt_settlement_exchange_rate_szse_df)
```

数据示例

```
     适用日期  买入结算汇兑比率  卖出结算汇兑比率 货币种类
0     2016-12-05   0.88621   0.88679  HKD
1     2016-12-06   0.88698   0.88682  HKD
2     2016-12-07   0.88525   0.88935  HKD
3     2016-12-08   0.89041   0.88899  HKD
4     2016-12-09   0.88567   0.89133  HKD
          ...       ...       ...  ...
1371  2022-09-23   0.90209   0.90311  HKD
1372  2022-09-26   0.90826   0.90814  HKD
1373  2022-09-27   0.91179   0.91181  HKD
1374  2022-09-28   0.91024   0.91636  HKD
1375  2022-09-29   0.93270   0.91810  HKD
```

#### 结算汇率-沪港通

接口: stock_sgt_settlement_exchange_rate_sse

目标地址: http://www.sse.com.cn/services/hkexsc/disclo/ratios

描述: 沪港通-港股通信息披露-结算汇兑

限量: 单次获取所有沪港通结算汇率数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述  |
|----------|---------|-----|
| 适用日期     | object  | -   |
| 买入结算汇兑比率 | float64 | -   |
| 卖出结算汇兑比率 | float64 | -   |
| 货币种类     | object  | -   |

接口示例

```python
import akshare as ak

stock_sgt_settlement_exchange_rate_sse_df = ak.stock_sgt_settlement_exchange_rate_sse()
print(stock_sgt_settlement_exchange_rate_sse_df)
```

数据示例

```
      适用日期  买入结算汇兑比率  卖出结算汇兑比率 货币种类
0     2014-11-17   0.79143   0.79017  HKD
1     2014-11-18   0.79068   0.78932  HKD
2     2014-11-19   0.78873   0.78927  HKD
3     2014-11-20   0.78980   0.78980  HKD
4     2014-11-21   0.79021   0.78999  HKD
          ...       ...       ...  ...
1860  2022-09-23   0.90170   0.90350  HKD
1861  2022-09-26   0.90812   0.90828  HKD
1862  2022-09-27   0.91177   0.91183  HKD
1863  2022-09-28   0.91038   0.91622  HKD
1864  2022-09-29   0.93260   0.91820  HKD
```

#### 参考汇率-深港通

接口: stock_sgt_reference_exchange_rate_szse

目标地址: https://www.szse.cn/szhk/hkbussiness/exchangerate/index.html

描述: 深港通-港股通业务信息-参考汇率

限量: 单次获取所有深港通参考汇率数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述  |
|---------|---------|-----|
| 适用日期    | object  | -   |
| 参考汇率买入价 | float64 | -   |
| 参考汇率卖出价 | float64 | -   |
| 货币种类    | object  | -   |

接口示例

```python
import akshare as ak

stock_sgt_reference_exchange_rate_szse_df = ak.stock_sgt_reference_exchange_rate_szse()
print(stock_sgt_reference_exchange_rate_szse_df)
```

数据示例

```
     适用日期  参考汇率买入价  参考汇率卖出价 货币种类
0     2016-12-05   0.8599   0.9131  HKD
1     2016-12-06   0.8603   0.9135  HKD
2     2016-12-07   0.8607   0.9139  HKD
3     2016-12-08   0.8630   0.9164  HKD
4     2016-12-09   0.8618   0.9152  HKD
          ...      ...      ...  ...
1375  2022-09-26   0.8810   0.9354  HKD
1376  2022-09-27   0.8844   0.9392  HKD
1377  2022-09-28   0.8859   0.9407  HKD
1378  2022-09-29   0.8976   0.9532  HKD
1379  2022-09-30   0.8875   0.9423  HKD
```

#### 参考汇率-沪港通

接口: stock_sgt_reference_exchange_rate_sse

目标地址: http://www.sse.com.cn/services/hkexsc/disclo/ratios/

描述: 沪港通-港股通信息披露-参考汇率

限量: 单次获取所有沪港通参考汇率数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述  |
|---------|---------|-----|
| 适用日期    | object  | -   |
| 参考汇率买入价 | float64 | -   |
| 参考汇率卖出价 | float64 | -   |
| 货币种类    | object  | -   |

接口示例

```python
import akshare as ak

stock_sgt_reference_exchange_rate_sse_df = ak.stock_sgt_reference_exchange_rate_sse()
print(stock_sgt_reference_exchange_rate_sse_df)
```

数据示例

```
     适用日期 参考汇率买入价 参考汇率卖出价 货币种类
0     2016-11-23   0.8635   0.9169  HKD
1     2016-11-24   0.8666   0.9202  HKD
2     2016-11-25   0.8684   0.9222  HKD
3     2016-11-28   0.8689   0.9227  HKD
4     2016-11-29   0.8669   0.9205  HKD
...          ...      ...      ...  ...
1995  2025-02-13   0.9108   0.9672  HKD
1996  2025-02-14   0.9089   0.9651  HKD
1997  2025-02-17   0.9049   0.9609  HKD
1998  2025-02-18   0.9050   0.9610  HKD
1999  2025-02-19   0.9079   0.9641  HKD
[2000 rows x 4 columns]
```

#### 港股通成份股

接口: stock_hk_ggt_components_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#hk_components

描述: 东方财富网-行情中心-港股市场-港股通成份股

限量: 单次获取所有港股通成份股数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述        |
|-----|---------|-----------|
| 序号  | int64   | -         |
| 代码  | object  | -         |
| 名称  | object  | -         |
| 最新价 | float64 | 注意单位: HKD |
| 涨跌额 | float64 | -         |
| 涨跌幅 | float64 | -         |
| 今开  | float64 | -         |
| 最高  | float64 | -         |
| 最低  | float64 | -         |
| 昨收  | float64 | -         |
| 成交量 | float64 | 注意单位: 股   |
| 成交额 | float64 | 注意单位: 港元  |

接口示例

```python
import akshare as ak

stock_hk_ggt_components_em_df = ak.stock_hk_ggt_components_em()
print(stock_hk_ggt_components_em_df)
```

数据示例

```
     序号  代码     名称    最新价  ...    最低   昨收     成交量           成交额
0      1  06680    金力永磁  14.32  ...   11.2  11.26   37038146   505702480.0
1      2  01347   华虹半导体   38.3  ...   31.0  31.15  156847850  5640157952.0
2      3  02228  晶泰控股-P   7.71  ...   6.35   6.48  367666400  2637056752.0
3      4  09989     海普瑞   4.72  ...   3.94   3.97   52468000   248839660.0
4      5  02556     迈富时  66.35  ...  58.85   58.0   15602500   994784016.0
..   ...    ...     ...    ...  ...    ...    ...        ...           ...
542  543  03738    阜博集团   3.98  ...   3.78   4.18   60483000   240326764.0
543  544  02469      粉笔   2.84  ...   2.76   2.99   67413000   190089816.0
544  545  01060    阿里影业   0.56  ...   0.56   0.59  419548960   239843136.0
545  546  01208    五矿资源   2.53  ...    2.5   2.68   93202524   237766414.0
546  547  01691  JS环球生活   1.85  ...   1.83   1.97  101497000   188240267.0
[547 rows x 12 columns]
```

#### 沪深港通分时数据

接口: stock_hsgt_fund_min_em

目标地址: https://data.eastmoney.com/hsgt/hsgtDetail/scgk.html

描述: 东方财富-数据中心-沪深港通-市场概括-分时数据

限量: 单次返回指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                        |
|--------|-----|-------------------------------------------|
| symbol | str | symbol="北向资金"; choice of {"北向资金", "南向资金"} |

输出参数-北向资金

| 名称   | 类型      | 描述       |
|------|---------|----------|
| 日期   | object  | 日期       |
| 时间   | object  | 时间       |
| 沪股通  | float64 | 注意单位: 万元 |
| 深股通  | float64 | 注意单位: 万元 |
| 北向资金 | float64 | 注意单位: 万元 |

接口示例-北向资金

```python
import akshare as ak

stock_hsgt_fund_min_em_df = ak.stock_hsgt_fund_min_em(symbol="北向资金")
print(stock_hsgt_fund_min_em_df)
```

数据示例-北向资金

```
         日期     时间     沪股通      深股通     北向资金
0    2024-02-05   9:30   -9222.18    768.61   -8453.57
1    2024-02-05   9:31  -19235.49 -14207.98  -33443.47
2    2024-02-05   9:32    7372.63  10008.57   17381.20
3    2024-02-05   9:33      68.98  -6103.43   -6034.45
4    2024-02-05   9:34  -19283.54 -30185.11  -49468.65
..          ...    ...        ...       ...        ...
236  2024-02-05  14:56  147348.10 -85482.41   61865.69
237  2024-02-05  14:57  152668.52 -77390.79   75277.73
238  2024-02-05  14:58  152620.49 -77390.79   75229.70
239  2024-02-05  14:59  152593.72 -77390.79   75202.93
240  2024-02-05  15:00  161430.55 -40329.50  121101.05
[241 rows x 5 columns]
```

输出参数-南向资金

| 名称     | 类型      | 描述       |
|--------|---------|----------|
| 日期     | object  | 日期       |
| 时间     | object  | 时间       |
| 港股通(沪) | float64 | 注意单位: 万元 |
| 港股通(深) | float64 | 注意单位: 万元 |
| 南向资金   | float64 | 注意单位: 万元 |

接口示例-南向资金

```python
import akshare as ak

stock_hsgt_fund_min_em_df = ak.stock_hsgt_fund_min_em(symbol="南向资金")
print(stock_hsgt_fund_min_em_df)
```

数据示例-南向资金

```
       日期       时间     港股通(沪)     港股通(深)     南向资金
0    2024-02-05   9:00       0.00       0.00       0.00
1    2024-02-05   9:01       0.00       0.00       0.00
2    2024-02-05   9:02       0.00       0.00       0.00
3    2024-02-05   9:03       0.00       0.00       0.00
4    2024-02-05   9:04       0.00       0.00       0.00
..          ...    ...        ...        ...        ...
366  2024-02-05  16:06  239979.44  149214.83  389194.27
367  2024-02-05  16:07  239979.44  149214.83  389194.27
368  2024-02-05  16:08  239979.44  149214.83  389194.27
369  2024-02-05  16:09  240475.76  149721.67  390197.43
370  2024-02-05  16:10  240475.76  149721.67  390197.43
[371 rows x 5 columns]
```

#### 板块排行

接口: stock_hsgt_board_rank_em

目标地址: https://data.eastmoney.com/hsgtcg/bk.html

描述: 东方财富网-数据中心-沪深港通持股-板块排行

限量: 单次获取指定 symbol 和 indicator 的所有数据

输入参数

| 名称        | 类型  | 描述                                                                                |
|-----------|-----|-----------------------------------------------------------------------------------|
| symbol    | str | symbol="北向资金增持行业板块排行"; choice of {"北向资金增持行业板块排行", "北向资金增持概念板块排行", "北向资金增持地域板块排行"} |
| indicator | str | indicator="今日"; choice of {"今日", "3日", "5日", "10日", "1月", "1季", "1年"}             |

输出参数

| 名称                | 类型      | 描述      |
|-------------------|---------|---------|
| 序号                | int64   | -       |
| 名称                | object  | -       |
| 最新涨跌幅             | float64 | 注意单位: % |
| 北向资金今日持股-股票只数     | float64 | -       |
| 北向资金今日持股-市值       | float64 | 注意单位: 元 |
| 北向资金今日持股-占板块比     | float64 | -       |
| 北向资金今日持股-占北向资金比   | float64 | -       |
| 北向资金今日增持估计-股票只数   | float64 | -       |
| 北向资金今日增持估计-市值     | float64 | 注意单位: 元 |
| 北向资金今日增持估计-市值增幅   | float64 | -       |
| 北向资金今日增持估计-占板块比   | float64 | -       |
| 北向资金今日增持估计-占北向资金比 | float64 | -       |
| 今日增持最大股-市值        | float64 | -       |
| 今日增持最大股-占股本比      | float64 | -       |
| 今日减持最大股-占股本比      | float64 | -       |
| 今日减持最大股-市值        | float64 | -       |
| 报告时间              | object  | -       |

接口示例

```python
import akshare as ak

stock_hsgt_board_rank_em_df = ak.stock_hsgt_board_rank_em(symbol="北向资金增持行业板块排行", indicator="今日")
print(stock_hsgt_board_rank_em_df)
```

数据示例

```
    序号   名称  最新涨跌幅  ...  今日减持最大股-市值  今日减持最大股-占总市值比 报告时间
0    1   软件开发  -3.17  ...        用友网络            指南针  2024-01-11
1    2  互联网服务  -2.18  ...        东方财富           中科创达  2024-01-11
2    3   煤炭行业  -0.04  ...        潞安环能           山西焦化  2024-01-11
3    4   能源金属  -0.98  ...        华友钴业           腾远钴业  2024-01-11
4    5   汽车整车  -0.98  ...        长安汽车           江淮汽车  2024-01-11
..  ..    ...    ...  ...         ...            ...         ...
81  82   美容护理  -1.17  ...         爱美客            爱美客  2024-01-11
82  83   钢铁行业  -0.54  ...        宝钢股份           常宝股份  2024-01-11
83  84   酿酒行业  -0.47  ...        贵州茅台           舍得酒业  2024-01-11
84  85     银行   0.99  ...        农业银行           常熟银行  2024-01-11
85  86   光伏设备   0.43  ...        隆基绿能           隆基绿能  2024-01-11
[86 rows x 17 columns]
```

#### 个股排行

接口: stock_hsgt_hold_stock_em

目标地址: https://data.eastmoney.com/hsgtcg/list.html

描述: 东方财富网-数据中心-沪深港通持股-个股排行

限量: 单次获取指定 market 和 indicator 的所有数据

输入参数

| 名称        | 类型  | 描述                                                                                |
|-----------|-----|-----------------------------------------------------------------------------------|
| market    | str | market="沪股通"; choice of {"北向", "沪股通", "深股通"}                                      |
| indicator | str | indicator="沪股通"; choice of {"今日排行", "3日排行", "5日排行", "10日排行", "月排行", "季排行", "年排行"} |

输出参数

| 名称         | 类型      | 描述                            |
|------------|---------|-------------------------------|
| 序号         | int32   | -                             |
| 代码         | object  | -                             |
| 名称         | object  | -                             |
| 今日收盘价      | float64 | -                             |
| 今日涨跌幅      | float64 | 注意单位: %                       |
| 今日持股-股数    | float64 | 注意单位: 万                       |
| 今日持股-市值    | float64 | 注意单位: 万                       |
| 今日持股-占流通股比 | float64 | 注意单位: %                       |
| 今日持股-占总股本比 | float64 | 注意单位: %                       |
| 增持估计-股数    | float64 | 注意单位: 万; 主要字段名根据 indicator 变化 |
| 增持估计-市值    | float64 | 注意单位: 万; 主要字段名根据 indicator 变化 |
| 增持估计-市值增幅  | object  | 注意单位: %; 主要字段名根据 indicator 变化 |
| 增持估计-占流通股比 | float64 | 注意单位: ‰; 主要字段名根据 indicator 变化 |
| 增持估计-占总股本比 | float64 | 注意单位: ‰; 主要字段名根据 indicator 变化 |
| 所属板块       | object  | -                             |
| 日期         | object  | -                             |

接口示例

```python
import akshare as ak

stock_em_hsgt_hold_stock_df = ak.stock_hsgt_hold_stock_em(market="北向", indicator="今日排行")
print(stock_em_hsgt_hold_stock_df)
```

数据示例

```
        序号   代码   名称    今日收盘价  ...  5日增持估计-占流通股比  5日增持估计-占总股本比  所属板块   日期
0        1  601127   赛力斯    68.50  ...          5.54          5.54  汽车整车  2024-01-10
1        2  603259  药明康德    67.62  ...          3.06          2.65  医疗服务  2024-01-10
2        3  600660  福耀玻璃    38.88  ...          5.03          3.86  玻璃玻纤  2024-01-10
3        4  600309  万华化学    73.00  ...          1.47          1.47  化学制品  2024-01-10
4        5  600188  兖矿能源    21.67  ...          3.37          2.04  煤炭行业  2024-01-10
    ...     ...   ...      ...  ...           ...           ...   ...         ...
1335  1336  601088  中国神华    33.85  ...         -0.58         -0.48  煤炭行业  2024-01-10
1336  1337  601012  隆基绿能    22.23  ...         -2.44         -2.44  光伏设备  2024-01-10
1337  1338  601899  紫金矿业    12.28  ...         -1.61         -1.26   贵金属  2024-01-10
1338  1339  601318  中国平安    37.82  ...         -1.15         -0.68    保险  2024-01-10
1339  1340  600519  贵州茅台  1641.50  ...         -0.43         -0.43  酿酒行业  2024-01-10
[1340 rows x 16 columns]
```

#### 每日个股统计

接口: stock_hsgt_stock_statistics_em

目标地址: http://data.eastmoney.com/hsgtcg/StockStatistics.aspx

描述: 东方财富网-数据中心-沪深港通-沪深港通持股-每日个股统计

限量: 单次获取指定 market 的 start_date 和 end_date 之间的所有数据, 该接口只能获取近期的数据

输入参数

| 名称         | 类型  | 描述                                                          |
|------------|-----|-------------------------------------------------------------|
| symbol     | str | symbol="北向持股"; choice of {"北向持股", "沪股通持股", "深股通持股", "南向持股"} |
| start_date | str | start_date="20210601"; 此处指定近期交易日                            |
| end_date   | str | end_date="20210608"; 此处指定近期交易日                              |

输出参数

| 名称          | 类型      | 描述                   |
|-------------|---------|----------------------|
| 持股日期        | object  | -                    |
| 股票代码        | object  | -                    |
| 股票简称        | object  | -                    |
| 当日收盘价       | float64 | 注意单位: 元; 南向持股单位为: 港元 |
| 当日涨跌幅       | float64 | 注意单位: %              |
| 持股数量        | float64 | 注意单位: 万股             |
| 持股市值        | float64 | 注意单位: 万元             |
| 持股数量占发行股百分比 | float64 | 注意单位: %              |
| 持股市值变化-1日   | float64 | 注意单位: 元              |
| 持股市值变化-5日   | float64 | 注意单位: 元              |
| 持股市值变化-10日  | float64 | 注意单位: 元              |

接口示例

```python
import akshare as ak

stock_hsgt_stock_statistics_em_df = ak.stock_hsgt_stock_statistics_em(symbol="北向持股", start_date="20211027", end_date="20211027")
print(stock_hsgt_stock_statistics_em_df)
```

数据示例

```
      持股日期    股票代码  股票简称  ...     持股市值变化-1日     持股市值变化-5日   持股市值变化-10日
0     2021-10-27  603327  福蓉科技  ... -1.876431e+05 -7.623525e+04 -1.999411e+06
1     2021-10-27  605199   葫芦娃  ... -1.408119e+06 -8.786002e+05 -8.750511e+05
2     2021-10-27  605136  丽人丽妆  ... -4.739965e+05 -4.834552e+05 -2.335740e+06
3     2021-10-27  300396  迪瑞医疗  ... -2.088927e+05 -7.029425e+05 -8.044508e+05
4     2021-10-27  603598  引力传媒  ...  1.090273e+05 -4.007397e+06 -6.420819e+06
          ...     ...   ...  ...           ...           ...           ...
1436  2021-10-27  601012  隆基股份  ...  1.753065e+09  4.817712e+08  7.829894e+09
1437  2021-10-27  600036  招商银行  ... -8.938440e+08  4.831173e+09  4.401946e+09
1438  2021-10-27  000333  美的集团  ... -5.719656e+09 -3.952053e+09 -7.030619e+09
1439  2021-10-27  300750  宁德时代  ...  3.142970e+09  7.133537e+09  2.157141e+10
1440  2021-10-27  600519  贵州茅台  ... -3.964678e+09 -2.334647e+09 -5.475161e+09
```

#### 机构排行

接口: stock_hsgt_institution_statistics_em

目标地址: http://data.eastmoney.com/hsgtcg/InstitutionStatistics.aspx

描述: 东方财富网-数据中心-沪深港通-沪深港通持股-机构排行

限量: 单次获取指定 market 的所有数据, 该接口只能获取近期的数据

输入参数

| 名称         | 类型  | 描述                                                          |
|------------|-----|-------------------------------------------------------------|
| market     | str | market="北向持股"; choice of {"北向持股", "沪股通持股", "深股通持股", "南向持股"} |
| start_date | str | start_date="20201218"; 此处指定近期交易日                            |
| end_date   | str | end_date="20201218"; 此处指定近期交易日                              |

输出参数

| 名称         | 类型      | 描述                   |
|------------|---------|----------------------|
| 持股日期       | object  | -                    |
| 机构名称       | object  | -                    |
| 持股只数       | float64 | 注意单位: 只              |
| 持股市值       | float64 | 注意单位: 元; 南向持股单位为: 港元 |
| 持股市值变化-1日  | float64 | 注意单位: 元; 南向持股单位为: 港元 |
| 持股市值变化-5日  | float64 | 注意单位: 元; 南向持股单位为: 港元 |
| 持股市值变化-10日 | float64 | 注意单位: 元; 南向持股单位为: 港元 |

接口示例

```python
import akshare as ak

stock_hsgt_institution_statistics_em_df = ak.stock_hsgt_institution_statistics_em(market="北向持股", start_date="20201218", end_date="20201218")
print(stock_hsgt_institution_statistics_em_df)
```

数据示例

```
     持股日期  ...           持股市值变化-10日
0    2020-12-18  ...  -1853745798.3000183
1    2020-12-18  ...         -6.01983e+09
2    2020-12-18  ...   3296301138.0900955
3    2020-12-18  ...   299245525.25022888
4    2020-12-18  ...   18389735021.519775
..          ...  ...                  ...
153  2020-12-18  ...   1455359.9999999981
154  2020-12-18  ...                  -10
155  2020-12-18  ...  -2035718.2199999988
156  2020-12-18  ...                -5425
157  2020-12-18  ...                  -94
```

#### 沪深港通-港股通(沪>港)实时行情

接口: stock_hsgt_sh_hk_spot_em

目标地址: https://quote.eastmoney.com/center/gridlist.html#hk_sh_stocks

描述: 东方财富网-行情中心-沪深港通-港股通(沪>港)-股票；按股票代码排序

限量: 单次获取所有数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称  | 类型      | 描述        |
|-----|---------|-----------|
| 序号  | int64   | -         |
| 代码  | object  | -         |
| 名称  | object  | -         |
| 最新价 | float64 | 注意单位: HKD |
| 涨跌额 | float64 | -         |
| 涨跌幅 | float64 | 注意单位: %   |
| 今开  | float64 | -         |
| 最高  | float64 | -         |
| 最低  | float64 | -         |
| 昨收  | float64 | -         |
| 成交量 | float64 | 注意单位: 亿股  |
| 成交额 | float64 | 注意单位: 亿港元 |

接口示例

```python
import akshare as ak

stock_hsgt_sh_hk_spot_em_df = ak.stock_hsgt_sh_hk_spot_em()
print(stock_hsgt_sh_hk_spot_em_df)
```

数据示例

```
     序号  代码      名称     最新价  ...    最低    昨收     成交量       成交额
0      1  00001      长和   49.50  ...   49.40   51.55  0.453758  22.712456
1      2  00002    中电控股   64.25  ...   64.15   64.99  0.054139   3.488916
2      3  00003  香港中华煤气    6.25  ...    6.17    6.27  0.343850   2.151922
3      4  00004   九龙仓集团   19.38  ...   19.24   19.70  0.016805   0.327002
4      5  00005    汇丰控股   87.95  ...   87.95   90.00  0.374613  33.070786
..   ...    ...     ...     ...  ...     ...     ...       ...        ...
540  541  09989     海普瑞    4.48  ...    4.43    4.50  0.023235   0.103881
541  542  09992    泡泡玛特  116.40  ...  114.10  116.20  0.061158   7.172251
542  543  09993    金辉控股    2.70  ...    2.69    2.80  0.007670   0.020909
543  544  09995    荣昌生物   17.80  ...   17.42   17.90  0.031735   0.566258
544  545  09997    康基医疗    7.53  ...    7.32    7.40  0.040656   0.303262
[545 rows x 12 columns]
```

#### 沪深港通历史数据

接口: stock_hsgt_hist_em

目标地址: https://data.eastmoney.com/hsgt/index.html

描述: 东方财富网-数据中心-资金流向-沪深港通资金流向-沪深港通历史数据

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                                      |
|--------|-----|-------------------------------------------------------------------------|
| symbol | str | symbol="北向资金"; choice of {"北向资金", "沪股通", "深股通", "南向资金", "港股通沪", "港股通深"} |

输出参数-北向资金

| 名称        | 类型      | 描述        |
|-----------|---------|-----------|
| 日期        | object  | -         |
| 当日成交净买额   | float64 | 注意单位: 亿元  |
| 买入成交额     | float64 | 注意单位: 亿元  |
| 卖出成交额     | float64 | 注意单位: 亿元  |
| 历史累计净买额   | float64 | 注意单位: 万亿元 |
| 当日资金流入    | float64 | 注意单位: 亿元  |
| 当日余额      | float64 | 注意单位: 亿元  |
| 持股市值      | float64 | 注意单位: 元   |
| 领涨股       | object  | -         |
| 领涨股-涨跌幅   | float64 | 注意单位: %   |
| 沪深300     | float64 | -         |
| 沪深300-涨跌幅 | float64 | 注意单位: %   |
| 领涨股-代码    | object  | -         |

接口示例-北向资金

```python
import akshare as ak

stock_hsgt_hist_em_df = ak.stock_hsgt_hist_em(symbol="北向资金")
print(stock_hsgt_hist_em_df)
```

数据示例-北向资金

```
           日期   当日成交净买额  买入成交额  ...  沪深300  沪深300-涨跌幅  领涨股-代码
0     2014-11-17  120.8233  120.8233  ...  2474.01      -0.19  601000.SH
1     2014-11-18   47.1967   49.4367  ...  2456.37      -0.71  600755.SH
2     2014-11-19   24.9677   26.6951  ...  2450.99      -0.22  601216.SH
3     2014-11-20   21.4194   23.0973  ...  2452.66       0.07  600755.SH
4     2014-11-21   21.9953   24.3055  ...  2486.79       1.39  600635.SH
          ...       ...       ...  ...      ...        ...        ...
2175  2024-04-08  -30.4468  676.5491  ...  3047.05      -0.72  603110.SH
2176  2024-04-09   10.5503  609.6675  ...  3048.54       0.05  605366.SH
2177  2024-04-10  -41.1411  565.4525  ...  3027.33      -0.70  605376.SH
2178  2024-04-11   20.2153  635.6282  ...  3034.25       0.23  002455.SZ
2179  2024-04-12  -73.8533  584.2191  ...  3019.47      -0.49  600984.SH
[2180 rows x 13 columns]
```

输出参数-港股通沪

| 名称       | 类型      | 描述        |
|----------|---------|-----------|
| 日期       | object  | -         |
| 当日成交净买额  | float64 | 注意单位: 亿港元 |
| 买入成交额    | float64 | 注意单位: 亿港元 |
| 卖出成交额    | float64 | 注意单位: 亿港元 |
| 历史累计净买额  | float64 | 注意单位: 万亿元 |
| 当日资金流入   | float64 | 注意单位: 亿元  |
| 当日余额     | float64 | 注意单位: 亿元  |
| 持股市值     | float64 | 注意单位: 元   |
| 领涨股      | object  | -         |
| 领涨股-涨跌幅  | float64 | 注意单位: %   |
| 恒生指数     | float64 | -         |
| 恒生指数-涨跌幅 | float64 | 注意单位: %   |
| 领涨股-代码   | object  | -         |

接口示例-港股通沪

```python
import akshare as ak

stock_hsgt_hist_em_df = ak.stock_hsgt_hist_em(symbol="港股通沪")
print(stock_hsgt_hist_em_df)
```

数据示例-港股通沪

```
    日期  当日成交净买额     买入成交额  ...    恒生指数  恒生指数-涨跌幅    领涨股-代码
0     2014-11-17  21.3208   22.3811  ...  23797.08     -1.21  02607.HK
1     2014-11-18   9.3716   10.8311  ...  23529.17     -1.13  00220.HK
2     2014-11-19   2.6592    3.8092  ...  23373.31     -0.66  02196.HK
3     2014-11-20   1.8736    2.8150  ...  23349.64     -0.10  01072.HK
4     2014-11-21   1.7453    3.0867  ...  23437.12      0.37  00038.HK
          ...      ...       ...  ...       ...       ...       ...
2136  2024-04-08  31.2429  115.3674  ...  16732.85      0.05  09890.HK
2137  2024-04-09  12.5875   78.8646  ...  16828.07      0.57  09890.HK
2138  2024-04-10   5.8779   92.2470  ...  17139.17      1.85  02038.HK
2139  2024-04-11  43.1305  109.2427  ...  17095.03     -0.26  03650.HK
2140  2024-04-12  32.6043  109.1773  ...  16721.69     -2.18  01563.HK
[2141 rows x 13 columns]
```

#### 沪深港通持股-个股

接口: stock_hsgt_individual_em

目标地址: https://data.eastmoney.com/hsgt/StockHdDetail/002008.html

描述: 东方财富网-数据中心-沪深港通-沪深港通持股-具体股票

限量: 单次获取指定 symbol 的截至 20240816 的数据

输入参数

| 名称     | 类型  | 描述                       |
|--------|-----|--------------------------|
| symbol | str | symbol="002008"; 支持港股和A股 |

输出参数-A股

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 持股日期       | object  | -       |
| 当日收盘价      | float64 | 注意单位: 元 |
| 当日涨跌幅      | float64 | 注意单位: % |
| 持股数量       | int64   | 注意单位: 股 |
| 持股市值       | float64 | 注意单位: 元 |
| 持股数量占A股百分比 | float64 | 注意单位: % |
| 今日增持股数     | float64 | 注意单位: 股 |
| 今日增持资金     | float64 | 注意单位: 元 |
| 今日持股市值变化   | float64 | 注意单位: 元 |

输出参数-港股

| 名称         | 类型      | 描述       |
|------------|---------|----------|
| 持股日期       | object  | -        |
| 当日收盘价      | float64 | 注意单位: 港元 |
| 当日涨跌幅      | float64 | 注意单位: %  |
| 持股数量       | int64   | 注意单位: 股  |
| 持股市值       | float64 | 注意单位: 港元 |
| 持股数量占A股百分比 | float64 | 注意单位: %  |
| 持股市值变化-1日  | float64 | 注意单位: 港元 |
| 持股市值变化-5日  | float64 | 注意单位: 港元 |
| 持股市值变化-10日 | float64 | 注意单位: 港元 |

接口示例-A股

```python
import akshare as ak

stock_hsgt_individual_em_df = ak.stock_hsgt_individual_em(symbol="002008")
print(stock_hsgt_individual_em_df)
```

数据示例-A股

```
      持股日期  当日收盘价  当日涨跌幅  ... 今日增持股数   今日增持资金 今日持股市值变化
0     2017-03-16  26.79 -0.630564  ...        NaN           NaN          NaN
1     2017-03-17  26.29 -1.866368  ...  2581506.0  6.838900e+07  56636924.24
2     2017-03-20  26.68  1.483454  ...      197.0  5.201903e+03   9772120.73
3     2017-03-21  26.52 -0.599700  ...   222538.0  5.916751e+06   1894757.36
4     2017-03-22  26.31 -0.791855  ...   870007.0  2.283707e+07  17584028.79
...          ...    ...       ...  ...        ...           ...          ...
1667  2024-08-12  20.11 -1.179361  ...   198214.0  3.996153e+06   -156132.62
1668  2024-08-13  20.24  0.646445  ...   -67398.0 -1.352516e+06    905332.72
1669  2024-08-14  20.03 -1.037549  ...   911900.0  1.828341e+07  14613446.50
1670  2024-08-15  20.19  0.798802  ...  -920522.0 -1.860605e+07 -15657027.18
1671  2024-08-16  20.12 -0.346706  ...  -835600.0 -1.685664e+07 -18028971.96
[1672 rows x 9 columns]
```

#### 沪深港通持股-个股详情

接口: stock_hsgt_individual_detail_em

目标地址: http://data.eastmoney.com/hsgtcg/StockHdStatistics/002008.html(示例)

描述: 东方财富网-数据中心-沪深港通-沪深港通持股-具体股票-个股详情

限量: 单次获取指定 symbol 的在 start_date 和 end_date 之间的所有数据; 注意只能返回 90 个交易日内的数据

输入参数

| 名称         | 类型  | 描述                                              |
|------------|-----|-------------------------------------------------|
| symbol     | str | symbol="002008"                                 |
| start_date | str | start_date="20210830"; 注意只能返回离最近交易日 90 个交易日内的数据 |
| end_date   | str | end_date="20211026"; 注意只能返回离最近交易日 90 个交易日内的数据   |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 持股日期       | object  | -       |
| 当日收盘价      | float64 | 注意单位: 元 |
| 当日涨跌幅      | float64 | 注意单位: % |
| 机构名称       | object  | -       |
| 持股数量       | int64   | 注意单位: 股 |
| 持股市值       | float64 | 注意单位: 元 |
| 持股数量占A股百分比 | float64 | 注意单位: % |
| 持股市值变化-1日  | float64 | 注意单位: 元 |
| 持股市值变化-5日  | float64 | 注意单位: 元 |
| 持股市值变化-10日 | float64 | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_hsgt_individual_detail_em_df = ak.stock_hsgt_individual_detail_em(
	symbol="002008",
	start_date="20210830",
	end_date="20211026"
)
print(stock_hsgt_individual_detail_em_df)
```

数据示例

```
      持股日期  当日收盘价   当日涨跌幅  ...  持股市值变化-1日  持股市值变化-5日  持股市值变化-10日
0     2021-10-25  40.05  0.5271  ...   112008.0 -9329832.0    211866.0
1     2021-10-25  40.05  0.5271  ...     1155.0     3795.0     12210.0
2     2021-10-25  40.05  0.5271  ...     8442.0    27738.0    437280.0
3     2021-10-25  40.05  0.5271  ...    26922.0    88458.0    224076.0
4     2021-10-25  40.05  0.5271  ...     1071.0     3519.0     11322.0
          ...    ...     ...  ...        ...        ...         ...
1791  2021-08-30  46.53  5.3668  ...    10428.0    29732.0    142362.0
1792  2021-08-30  46.53  5.3668  ...   120633.0  2329002.0   2330955.0
1793  2021-08-30  46.53  5.3668  ...  5289183.0  4723917.0   7891389.0
1794  2021-08-30  46.53  5.3668  ...   -15768.0  -225276.0         0.0
1795  2021-08-30  46.53  5.3668  ...      474.0      556.0     -3168.0
```

### 停复牌信息

接口: stock_tfp_em

目标地址: https://data.eastmoney.com/tfpxx/

描述: 东方财富网-数据中心-特色数据-停复牌信息

限量: 单次获取指定 date 的停复牌数据, 具体更新逻辑跟目标网页统一

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20240426" |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int64  |     |
| 代码     | object |     |
| 名称     | object |     |
| 停牌时间   | object |     |
| 停牌截止时间 | object |     |
| 停牌期限   | object |     |
| 停牌原因   | object |     |
| 所属市场   | object |     |
| 预计复牌时间 | object |     |

接口示例

```python
import akshare as ak

stock_tfp_em_df = ak.stock_tfp_em(date="20240426")
print(stock_tfp_em_df)
```

数据示例

```
    序号      代码     名称        停牌时间  ...  停牌期限       停牌原因      所属市场      预计复牌时间
0    0  300208   青岛中程  2024-04-29  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-30
1    1  300125   聆达股份  2024-04-29  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-30
2    2  002310   东方园林  2024-04-29  ...  停牌一天     刊登重要公告     深交所主板  2024-04-30
3    3  600165   ST宁科  2024-04-29  ...  停牌一天     刊登重要公告  上交所风险警示板  2024-04-30
4    4  300116    保力新  2024-04-29  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-30
5    5  300167  ST迪威迅  2024-04-29  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-30
6    6  000971   ST高升  2024-04-29  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-30
7    7  300536   农尚环境  2024-04-29  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-30
8    8  600777   新潮能源  2024-04-29  ...  停牌一天     刊登重要公告     上交所主板  2024-04-30
9    9  002650   加加食品  2024-04-29  ...  停牌一天     刊登重要公告     深交所主板  2024-04-30
10  10  300013   新宁物流  2024-04-29  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-30
11  11  603388   元成股份  2024-04-29  ...  停牌一天     刊登重要公告     上交所主板  2024-04-30
12  12  600070   ST富润  2024-04-29  ...  停牌一天     刊登重要公告  上交所风险警示板  2024-04-30
13  13  600234   科新发展  2024-04-29  ...  停牌一天     刊登重要公告     上交所主板  2024-04-30
14  14  002665   首航高科  2024-04-29  ...  停牌一天     刊登重要公告     深交所主板  2024-04-30
15  15  300799  *ST左江  2024-04-29  ...  连续停牌     刊登重要公告  深交所风险警示板         NaT
16  16  300108   ST吉药  2024-04-29  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-30
17  17  300209  ST有棵树  2024-04-26  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-29
18  18  600715   文投控股  2024-04-26  ...  停牌一天     刊登重要公告     上交所主板  2024-04-29
19  19  603963   大理药业  2024-04-26  ...  停牌一天     刊登重要公告     上交所主板  2024-04-29
20  20  300506    名家汇  2024-04-26  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-29
21  21  002699  *ST美盛  2024-04-26  ...  连续停牌     刊登重要公告  深交所风险警示板         NaT
22  22  002263    大东南  2024-04-26  ...  连续停牌     刊登重要公告     深交所主板  2024-04-30
23  23  000809   铁岭新城  2024-04-26  ...  停牌一天     刊登重要公告     深交所主板  2024-04-29
24  24  000609   中迪投资  2024-04-26  ...  停牌一天     刊登重要公告     深交所主板  2024-04-29
25  25  688085   三友医疗  2024-04-26  ...  连续停牌     刊登重要公告    上交所科创板  2024-05-08
26  26  300117   嘉寓股份  2024-04-26  ...  停牌一天     刊登重要公告    深交所创业板  2024-04-29
27  27  002141   贤丰控股  2024-04-26  ...  停牌一天     刊登重要公告     深交所主板  2024-04-29
28  28  601456   国联证券  2024-04-26  ...  连续停牌  拟筹划重大资产重组     上交所主板  2024-05-15
29  29  600898   ST美讯  2024-04-25  ...  停牌一天     刊登重要公告  上交所风险警示板  2024-04-26
30  30  300965   恒宇信通  2024-04-25  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-26
31  31  000989    九芝堂  2024-04-25  ...  停牌一天     刊登重要公告  深交所风险警示板  2024-04-26
32  32  000416  *ST民控  2024-04-25  ...  连续停牌     刊登重要公告  深交所风险警示板         NaT
33  33  603088   宁波精达  2024-04-23  ...  连续停牌     刊登重要公告     上交所主板  2024-05-10
34  34  000559   万向钱潮  2024-04-17  ...  连续停牌  拟筹划重大资产重组     深交所主板  2024-05-06
35  35  002715   登云股份  2024-04-15  ...  连续停牌  拟筹划重大资产重组     深交所主板  2024-04-29
36  36  000005   ST星源  2024-03-06  ...  连续停牌     刊登重要公告  深交所风险警示板  2024-04-26
[37 rows x 9 columns]
```

### 停复牌

接口: news_trade_notify_suspend_baidu

目标地址: https://gushitong.baidu.com/calendar

描述: 百度股市通-交易提醒-停复牌

限量: 单次获取指定 date 的停复牌数据, 提供港股的停复牌数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20241107" |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 股票代码   | object |     |
| 股票简称   | object |     |
| 交易所    | object |     |
| 停牌时间   | object |     |
| 复牌时间   | object |     |
| 停牌事项说明 | object |     |

接口示例

```python
import akshare as ak

news_trade_notify_suspend_baidu_df = ak.news_trade_notify_suspend_baidu(date="20241107")
print(news_trade_notify_suspend_baidu_df)
```

数据示例

```
     股票代码   股票简称  ...  复牌时间                                  停牌事项说明
0  002602   世纪华通  ...  2024-11-08                                实行其他风险警示
1  300473   德尔股份  ...         NaT                                    重大事项
2  833534   神玥软件  ...         NaT  全国股转公司认定的其他重大事项，具体内容为：公司股价出现较大波动，停牌核查。
3  836226   卡友信息  ...         NaT                                做市商不足两家。
4  870215  ST未来能  ...         NaT         主办券商单方解除持续督导且公司无其他主办券商承接持续督导工作。
5  872577   浏经水务  ...         NaT                        向全国股转公司主动申请终止挂牌。
6   02288   宏基资本  ...         NaT            将于今天（7/11/2024）上午九时正起短暂停止买卖。
[7 rows x 6 columns]
```

### 分红派息

接口: news_trade_notify_dividend_baidu

目标地址: https://gushitong.baidu.com/calendar

描述: 百度股市通-交易提醒-分红派息

限量: 单次获取指定 date 的分红派息数据, 提供港股的分红派息数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| date   | str | date="20241107" |
| cookie | str | 可以指定 cookie     |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 股票代码 | object |     |
| 除权日  | object |     |
| 分红   | object |     |
| 送股   | object |     |
| 转增   | object |     |
| 实物   | object |     |
| 交易所  | object |     |
| 股票简称 | object |     |
| 报告期  | object |     |

接口示例

```python
import akshare as ak

news_trade_notify_dividend_baidu_df = ak.news_trade_notify_dividend_baidu(date="20251126")
print(news_trade_notify_dividend_baidu_df)
```

数据示例

```
      股票代码   除权日     分红 送股 转增 实物 交易所    股票简称 报告期
0   601788  2025-11-26   1.10元  -  -  -  SH    光大证券  2025-11-26
1   600482  2025-11-26   0.82元  -  -  -  SH    中国动力  2025-11-26
2   603298  2025-11-26   2.00元  -  -  -  SH    杭叉集团  2025-11-26
3   000088  2025-11-26   0.88元  -  -  -  SZ     盐田港  2025-11-26
4    06808  2025-11-26  0.08港元  -  -  -  HK    高鑫零售  2025-11-26
5   603040  2025-11-26   2.50元  -  -  -  SH     新坐标  2025-11-26
6   603008  2025-11-26   2.80元  -  -  -  SH     喜临门  2025-11-26
7   603676  2025-11-26   1.47元  -  -  -  SH     卫信康  2025-11-26
8   920112  2025-11-26   5.00元  -  -  -  BJ     巴兰仕  2025-11-26
9   301023  2025-11-26   4.00元  -  -  -  SZ    江南奕帆  2025-11-26
10  001228  2025-11-26   3.00元  -  -  -  SZ     永泰运  2025-11-26
11   00331  2025-11-26  0.18港元  -  -  -  HK  丰盛生活服务  2025-11-26
12  300246  2025-11-26   0.50元  -  -  -  SZ     宝莱特  2025-11-26
13  688193  2025-11-26   1.50元  -  -  -  SH    仁度生物  2025-11-26
14   06668  2025-11-26  0.05港元  -  -  -  HK    星盛商业  2025-11-26
```

### 个股新闻

接口: stock_news_em

目标地址: https://so.eastmoney.com/news/s?keyword=603777

描述: 东方财富指定个股的新闻资讯数据

限量: 指定 symbol 当日最近 100 条新闻资讯数据

输入参数

| 名称     | 类型  | 描述                          |
|--------|-----|-----------------------------|
| symbol | str | symbol="603777"; 股票代码或其他关键词 |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 关键词  | object | -   |
| 新闻标题 | object | -   |
| 新闻内容 | object | -   |
| 发布时间 | object | -   |
| 文章来源 | object | -   |
| 新闻链接 | object | -   |

接口示例

```python
import akshare as ak

stock_news_em_df = ak.stock_news_em(symbol="603777")
print(stock_news_em_df)
```

数据示例

```
       关键词  ...                                               新闻链接
0   603777  ...  http://finance.eastmoney.com/a/202506163431529...
1   603777  ...  http://finance.eastmoney.com/a/202506113427724...
2   603777  ...  http://finance.eastmoney.com/a/202506093425572...
3   603777  ...  http://finance.eastmoney.com/a/202506093425584...
4   603777  ...  http://finance.eastmoney.com/a/202506123429086...
..     ...  ...                                                ...
95  603777  ...  http://finance.eastmoney.com/a/202505123401879...
96  603777  ...  http://finance.eastmoney.com/a/202503203351336...
97  603777  ...  http://finance.eastmoney.com/a/202505133403529...
98  603777  ...  http://finance.eastmoney.com/a/202504293392475...
99  603777  ...  http://finance.eastmoney.com/a/202505123402073...
[100 rows x 6 columns]
```

### 财经内容精选

接口: stock_news_main_cx

目标地址: https://cxdata.caixin.com/pc/

描述: 财新网-财新数据通-最新

限量: 返回最新 100 条新闻数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称            | 类型     | 描述 |
|---------------|--------|----|
| tag           | object | -  |
| summary       | object | -  |
| url           | object | -  |

接口示例

```python
import akshare as ak

stock_news_main_cx_df = ak.stock_news_main_cx()
print(stock_news_main_cx_df)
```

数据示例

```
      tag  ...                                                url
0    今日热点  ...  https://database.caixin.com/2025-12-25/1023970...
5    市场动态  ...  https://database.caixin.com/2025-12-25/1023969...
6    市场动态  ...  https://database.caixin.com/2025-12-25/1023969...
9    市场动态  ...  https://database.caixin.com/2025-12-25/1023969...
10   市场动态  ...  https://database.caixin.com/2025-12-25/1023969...
..    ...  ...                                                ...
105  市场动态  ...  https://database.caixin.com/2025-12-15/1023934...
106  市场动态  ...  https://database.caixin.com/2025-12-15/1023934...
107  市场动态  ...  https://database.caixin.com/2025-12-15/1023934...
108  市场洞察  ...  https://database.caixin.com/2025-12-15/1023933...
109  市场洞察  ...  https://database.caixin.com/2025-12-15/1023933...
[100 rows x 3 columns]
```

### 财报发行

接口: news_report_time_baidu

目标地址: https://gushitong.baidu.com/calendar

描述: 百度股市通-财报发行

限量: 单次获取指定 date 的财报发行, 提供港股的财报发行数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20241107" |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 股票代码 | object |     |
| 交易所  | object |     |
| 股票简称 | object |     |
| 财报期  | object |     |

接口示例

```python
import akshare as ak

news_report_time_baidu_df = ak.news_report_time_baidu(date="20241107")
print(news_report_time_baidu_df)
```

数据示例

```
     股票代码 交易所                       股票简称               财报期
0   00945  HK                         宏利金融-S          2024年三季报
1   00981  HK                           中芯国际          2024年三季报
2   01347  HK                          华虹半导体          2024年三季报
3   01665  HK                           槟杰科达          2024年三季报
4   08476  HK                         大洋环球控股           2024年中报
..    ...  ..                            ...               ...
95   XRAY  US                         登士柏西诺德    美东时间发布2024年三季报
96   UPST  US         Upstart Holdings, Inc.  美东时间盘后发布2024年三季报
97    MUR  US                           墨菲石油    美东时间发布2024年三季报
98     SG  US       Sweetgreen, Inc. Class A    美东时间发布2024年三季报
99   STEP  US  StepStone Group, Inc. Class A   美东时间发布24/25年二季报
[100 rows x 4 columns]
```

### 新股数据

#### 打新收益率

接口: stock_dxsyl_em

目标地址: https://data.eastmoney.com/xg/xg/dxsyl.html

描述: 东方财富网-数据中心-新股申购-打新收益率

限量: 单次获取所有打新收益率数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 股票代码     | object  | -       |
| 股票简称     | object  | -       |
| 发行价      | float64 | -       |
| 最新价      | float64 | -       |
| 网上发行中签率  | float64 | 注意单位: % |
| 网上有效申购股数 | int64   | -       |
| 网上有效申购户数 | int64   | 注意单位: 户 |
| 网上超额认购倍数 | float64 | -       |
| 网下配售中签率  | float64 | 注意单位: % |
| 网下有效申购股数 | int64   | -       |
| 网下有效申购户数 | int64   | 注意单位: 户 |
| 网下配售认购倍数 | float64 | -       |
| 总发行数量    | int64   | -       |
| 开盘溢价     | float64 | -       |
| 首日涨幅     | float64 | -       |
| 上市日期     | object  | -       |

接口示例

```python
import akshare as ak

stock_dxsyl_em_df = ak.stock_dxsyl_em()
print(stock_dxsyl_em_df)
```

数据示例

```
      序号   股票代码  股票简称  发行价  ...   总发行数量    开盘溢价  首日涨幅    上市日期
0        1  301502  N华阳智   28.01  ...  1427.1000  114.209211  114.1021  2024-02-02
1        2  603082   C北自   21.28  ...  4055.6900  116.165414  118.0451  2024-01-30
2        3  603375   盛景微   38.18  ...  2516.6667   70.246202   54.4002  2024-01-24
3        4  301577  美信科技   36.51  ...  1109.5149  119.118050   88.7976  2024-01-24
4        5  001379  腾达科技   16.98  ...  5000.0000   64.899882   42.6973  2024-01-19
    ...     ...   ...     ...  ...        ...         ...       ...         ...
3494  3495  688709  成都华微   15.69  ...  9560.0000         NaN       NaN         NaT
3495  3496  688584  上海合晶   22.66  ...  6620.6036         NaN       NaN         NaT
3496  3497  603341  龙旗科技     NaN  ...  6000.0000         NaN       NaN         NaT
3497  3498  301591  肯特股份     NaN  ...  2103.0000         NaN       NaN         NaT
3498  3499  301589  诺瓦星云  126.89  ...  1284.0000         NaN       NaN         NaT
[3499 rows x 17 columns]
```

#### 新股申购与中签

接口: stock_xgsglb_em

目标地址: https://data.eastmoney.com/xg/xg/default_2.html

描述: 东方财富网-数据中心-新股数据-新股申购-新股申购与中签查询

限量: 单次获取指定 market 的新股申购与中签查询数据

输入参数

| 名称     | 类型  | 描述                                                                     |
|--------|-----|------------------------------------------------------------------------|
| symbol | str | symbol="全部股票"; choice of {"全部股票", "沪市主板", "科创板", "深市主板", "创业板", "北交所"} |

输出参数-其他（除北交所外）

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 股票代码     | object  | -       |
| 股票简称     | object  | -       |
| 交易所      | object  | -       |
| 板块       | object  | -       |
| 申购代码     | object  | -       |
| 发行总数     | float64 | 注意单位: 股 |
| 网上发行     | int64   | 注意单位: 股 |
| 顶格申购需配市值 | float64 | 注意单位: 股 |
| 申购上限     | int64   | -       |
| 发行价格     | float64 | -       |
| 最新价      | float64 | -       |
| 首日收盘价    | float64 | -       |
| 申购日期     | object  | -       |
| 中签号公布日   | object  | -       |
| 中签缴款日期   | object  | -       |
| 上市日期     | object  | -       |
| 发行市盈率    | float64 | -       |
| 行业市盈率    | float64 | -       |
| 中签率      | float64 | 注意单位: % |
| 询价累计报价倍数 | float64 | -       |
| 配售对象报价家数 | float64 | -       |
| 连续一字板数量  | object  | -       |
| 涨幅       | float64 | 注意单位: % |
| 每中一签获利   | float64 | 注意单位: 元 |

接口示例-其他（除北交所外）

```python
import akshare as ak

stock_xgsglb_em_df = ak.stock_xgsglb_em(symbol="全部股票")
print(stock_xgsglb_em_df)
```

数据示例-其他（除北交所外）

```
     股票代码  股票简称    申购代码    交易所  ...  配售对象报价家数  连续一字板数量        涨幅   每中一签获利
0     301575   艾芬达  301575  深圳证券交易所  ...       NaN      待上市       NaN      NaN
1     920100  三协电机  920100  北京证券交易所  ...       NaN     None       NaN      NaN
2     603370  华新精科  732370  上海证券交易所  ...    9264.0      待上市       NaN      NaN
3     920112   巴兰仕  920112  北京证券交易所  ...       NaN     None       NaN      NaN
4     920056   能之光  920056  北京证券交易所  ...       NaN     None  369.4868  13320.0
...      ...   ...     ...      ...  ...       ...      ...       ...      ...
3861  601801  皖新传媒  780801  上海证券交易所  ...       0.0        1   55.4700   6550.0
3862  002338  奥普光电  002338  深圳证券交易所  ...     285.0        1  136.0400  14965.0
3863  002337  赛象科技  002337  深圳证券交易所  ...     181.0        1   40.6300   6295.0
3864  002335  科华数据  002335  深圳证券交易所  ...     188.0        1   56.7200   7755.0
3865  002334   英威腾  002334  深圳证券交易所  ...     190.0        1   37.7000   9050.0
[3866 rows x 23 columns]
```

输出参数-北交所

| 名称          | 类型      | 描述      |
|-------------|---------|---------|
| 代码          | object  | -       |
| 简称          | object  | -       |
| 申购代码        | object  | -       |
| 发行总数        | int64   | 注意单位: 股 |
| 网上-发行数量     | int64   | 注意单位: 股 |
| 网上-申购上限     | int64   | 注意单位: 股 |
| 网上-顶格所需资金   | int64   | 注意单位: 元 |
| 发行价格        | float64 | -       |
| 申购日         | object  | -       |
| 中签率         | float64 | -       |
| 稳获百股需配资金    | float64 | -       |
| 最新价格-价格     | float64 | -       |
| 最新价格-累计涨幅   | float64 | -       |
| 上市首日-上市日    | object  | -       |
| 上市首日-均价     | float64 | -       |
| 上市首日-涨幅     | float64 | -       |
| 上市首日-每百股获利  | float64 | -       |
| 上市首日-约合年化收益 | float64 | -       |
| 发行市盈率       | float64 | -       |
| 行业市盈率       | float64 | -       |
| 参与申购资金      | float64 | -       |
| 参与申购人数      | float64 | -       |

接口示例-北交所

```python
import akshare as ak

stock_xgsglb_em_df = ak.stock_xgsglb_em(symbol="北交所")
print(stock_xgsglb_em_df)
```

数据示例-北交所

```
     代码    简称    申购代码       发行总数  ...  发行市盈率  行业市盈率        参与申购资金    参与申购人数
0    836547  无锡晶海  889028   15600000  ...  15.31  23.83           NaN       NaN
1    873703  广厦环能  889788   15000000  ...  14.00  27.40  1.348229e+11  189150.0
2    835579  机科股份  889579   31200000  ...  19.54  30.00  1.180983e+11  210819.0
3    873570  坤博精工  889669    7850000  ...  18.51  29.91  8.476524e+10  129579.0
4    873132  泰鹏智能  889878   12000000  ...  11.41  19.66  7.025810e+10  132026.0
..      ...   ...     ...        ...  ...    ...    ...           ...       ...
233  839167  同享科技  889007   12000000  ...  19.82    NaN  4.660134e+10  434040.0
234  430489  佳先股份  889998   21321800  ...  25.00    NaN  3.085403e+10  381892.0
235  834682  球冠电缆  889888   40000000  ...  19.73    NaN  3.238970e+10  381155.0
236  833819  颖泰生物  889001  100000000  ...  24.77    NaN  1.012017e+11  542873.0
237  830799  艾融软件  889999    8800000  ...  49.31    NaN  1.243882e+11  519980.0
[238 rows x 22 columns]
```

### 年报季报

#### 业绩报表

接口: stock_yjbb_em

目标地址: http://data.eastmoney.com/bbsj/202003/yjbb.html

描述: 东方财富-数据中心-年报季报-业绩报表

限量: 单次获取指定 date 的业绩报告数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20200331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20100331 开始 |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 股票代码         | object  | -       |
| 股票简称         | object  | -       |
| 每股收益         | float64 | 注意单位: 元 |
| 营业总收入-营业总收入  | float64 | 注意单位: 元 |
| 营业总收入-同比增长   | float64 | 注意单位: % |
| 营业总收入-季度环比增长 | float64 | 注意单位: % |
| 净利润-净利润      | float64 | 注意单位: 元 |
| 净利润-同比增长     | float64 | 注意单位: % |
| 净利润-季度环比增长   | float64 | 注意单位: % |
| 每股净资产        | float64 | 注意单位: 元 |
| 净资产收益率       | float64 | 注意单位: % |
| 每股经营现金流量     | float64 | 注意单位: 元 |
| 销售毛利率        | float64 | 注意单位: % |
| 所处行业         | object  | -       |
| 最新公告日期       | object  | -       |

接口示例

```python
import akshare as ak

stock_yjbb_em_df = ak.stock_yjbb_em(date="20220331")
print(stock_yjbb_em_df)
```

数据示例

```
    序号   股票代码  股票简称  每股收益  ... 每股经营现金流量 销售毛利率 所处行业   最新公告日期
0        1  002512  达华智能 -0.0349  ... -0.028029  10.287193  计算机设备  2025-01-27
1        2  000833  粤桂股份  0.0533  ... -0.027977  22.838955   综合行业  2025-01-25
2        3  600576  祥源文旅 -0.0050  ... -0.075190  22.145564   旅游酒店  2025-01-03
3        4  688687  凯因科技  0.1500  ...  0.328110  84.879441   生物制品  2024-12-20
4        5  873989  中达新材  0.0700  ... -0.019306  12.369830   None  2024-12-11
...    ...     ...   ...     ...  ...       ...        ...    ...         ...
5850  5851  400055   国瓷5 -0.0080  ... -0.020637  -5.145723   None  2022-04-25
5851  5852  873599  振通检测  0.0700  ... -0.203169  34.648249   None  2022-04-22
5852  5853  420063  武锅B3 -0.1100  ... -0.068036  -0.167132   None  2022-04-22
5853  5854  430717  源通机械  0.0300  ...  0.152183  25.958049   None  2022-04-15
5854  5855  870529  东铭新材  0.0200  ... -0.003474  29.472422   None  2022-04-11
[5855 rows x 16 columns]
```

#### 业绩快报

接口: stock_yjkb_em

目标地址: https://data.eastmoney.com/bbsj/202003/yjkb.html

描述: 东方财富-数据中心-年报季报-业绩快报

限量: 单次获取指定 date 的业绩快报数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20200331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20100331 开始 |

输出参数

| 名称          | 类型     | 描述  |
|-------------|--------|-----|
| 序号          | object | -   |
| 股票代码        | object | -   |
| 股票简称        | object | -   |
| 每股收益        | object | -   |
| 营业收入-营业收入   | object | -   |
| 营业收入-去年同期   | object | -   |
| 营业收入-同比增长   | str    | -   |
| 营业收入-季度环比增长 | object | -   |
| 净利润-净利润     | object | -   |
| 净利润-去年同期    | object | -   |
| 净利润-同比增长    | str    | -   |
| 净利润-季度环比增长  | object | -   |
| 每股净资产       | object | -   |
| 净资产收益率      | object | -   |
| 所处行业        | object | -   |
| 公告日期        | object | -   |
| 市场板块        | object | -   |
| 证券类型        | object | -   |

接口示例

```python
import akshare as ak

stock_yjkb_em_df = ak.stock_yjkb_em(date="20200331")
print(stock_yjkb_em_df)
```

数据示例

```
    序号  股票代码  股票简称   每股收益  ... 每股净资产 净资产收益率 所处行业  公告日期
0      1  301386  未来电器     NaN  ...    NaN     NaN    电源设备  2022-05-31
1      2  301296   新巨丰     NaN  ...    NaN     NaN    包装材料  2021-10-22
2      3  301302  华如科技     NaN  ...    NaN     NaN    软件开发  2021-09-30
3      4  301167  建研设计     NaN  ...    NaN     NaN  工程咨询服务  2021-09-29
4      5  301169  零点有数     NaN  ...    NaN     NaN    专业服务  2021-09-28
..   ...     ...   ...     ...  ...    ...     ...     ...         ...
410  411  000591   太阳能  0.0514  ...   4.49    1.15    电力行业  2020-04-18
411  412  600161  天坛生物  0.1300  ...   3.89    3.29    生物制品  2020-04-14
412  413  300831  派瑞股份     NaN  ...    NaN     NaN     半导体  2020-04-14
413  414  601231  环旭电子     NaN  ...    NaN     NaN    消费电子  2020-04-11
414  415  688528  秦川物联     NaN  ...    NaN     NaN    仪器仪表  2020-04-09
[415 rows x 16 columns]
```

#### 业绩预告

接口: stock_yjyg_em

目标地址: https://data.eastmoney.com/bbsj/202003/yjyg.html

描述: 东方财富-数据中心-年报季报-业绩预告

限量: 单次获取指定 date 的业绩预告数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20200331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20081231 开始 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | object  | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 预测指标   | float64 | -       |
| 业绩变动   | float64 | -       |
| 预测数值   | float64 | 注意单位: 元 |
| 业绩变动幅度 | float64 | 注意单位: % |
| 业绩变动原因 | float64 | -       |
| 预告类型   | float64 | -       |
| 上年同期值  | float64 | 注意单位: 元 |
| 公告日期   | float64 | -       |

接口示例

```python
import akshare as ak

stock_yjyg_em_df = ak.stock_yjyg_em(date="20190331")
print(stock_yjyg_em_df)
```

数据示例

```
     序号   股票代码  股票简称  ... 预告类型    上年同期值     公告日期
0        1  300886  华业香料  ...   略减  2.580923e+08  2020-07-22
1        2  300886  华业香料  ...   略减  5.621616e+07  2020-07-22
2        3  600189   泉阳泉  ...   首亏  4.197830e+07  2020-06-24
3        4  600189   泉阳泉  ...   增亏 -8.628030e+07  2020-06-24
4        5  000980  众泰汽车  ...   首亏  7.998868e+08  2020-06-19
    ...     ...   ...  ...  ...           ...         ...
3806  3807  603687   大胜达  ...   略增  1.413420e+08  2019-06-17
3807  3808  603687   大胜达  ...   略增  1.334714e+08  2019-06-17
3808  3809  603687   大胜达  ...   略增  1.285381e+09  2019-06-17
3809  3810  601975  招商南油  ...   略减  3.642135e+08  2018-12-28
3810  3811  601975  招商南油  ...   略增  3.378241e+09  2018-12-28
[3811 rows x 11 columns]
```

#### 预约披露时间-东方财富

接口: stock_yysj_em

目标地址: https://data.eastmoney.com/bbsj/202003/yysj.html

描述: 东方财富-数据中心-年报季报-预约披露时间

限量: 单次获取指定 symbol 和 date 的预约披露时间数据

输入参数

| 名称     | 类型  | 描述                                                                                         |
|--------|-----|--------------------------------------------------------------------------------------------|
| symbol | str | symbol="沪深A股"; choice of {'沪深A股', '沪市A股', '科创板', '深市A股', '创业板', '京市A股', 'ST板'}             |
| date   | str | date="20200331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20081231 开始 |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int64  | -   |
| 股票代码   | object | -   |
| 股票简称   | object | -   |
| 首次预约时间 | object | -   |
| 一次变更日期 | object | -   |
| 二次变更日期 | object | -   |
| 三次变更日期 | object | -   |
| 实际披露时间 | object | -   |

接口示例

```python
import akshare as ak

stock_yysj_em_df = ak.stock_yysj_em(symbol="沪深A股", date="20211231")
print(stock_yysj_em_df)
```

数据示例

```
     序号    股票代码  股票简称  首次预约时间  一次变更日期 二次变更日期 三次变更日期  实际披露时间
0        1  000509  华塑控股  2022-01-18         NaT         NaT    NaT  2022-01-18
1        2  002107  沃华医药  2022-01-21         NaT         NaT    NaT  2022-01-21
2        3  600272  开开实业  2022-01-28         NaT         NaT    NaT  2022-01-28
3        4  600599  ST熊猫  2022-01-28  2022-03-01  2022-04-23    NaT  2022-04-23
4        5  002984   森麒麟  2022-02-15         NaT         NaT    NaT  2022-02-15
    ...     ...   ...         ...         ...         ...    ...         ...
4578  4579  688597  煜邦电力  2022-04-30  2022-04-27         NaT    NaT  2022-04-27
4579  4580  688613  奥精医疗  2022-04-30  2022-04-28         NaT    NaT  2022-04-28
4580  4581  688701  卓锦股份  2022-04-30  2022-04-16  2022-04-29    NaT  2022-04-29
4581  4582  688711  宏微科技  2022-04-30         NaT         NaT    NaT  2022-04-30
4582  4583  688728   格科微  2022-04-30  2022-04-14  2022-04-28    NaT  2022-04-28
[4583 rows x 8 columns]
```

#### 预约披露时间-巨潮资讯

接口: stock_report_disclosure

目标地址: http://www.cninfo.com.cn/new/commonUrl?url=data/yypl

描述: 巨潮资讯-数据-预约披露的数据

限量: 单次获取指定 market 和 period 的预约披露数据

输入参数

| 名称     | 类型  | 描述                                                                                   |
|--------|-----|--------------------------------------------------------------------------------------|
| market | str | market="沪深京"; choice of {"沪深京", "深市", "深主板", "创业板", "沪市", "沪主板", "科创板", "北交所"}       |
| period | str | period="2021年报"; 近四期的财务报告; e.g., choice of {"2021一季", "2021半年报", "2021三季", "2021年报"} |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 股票代码 | object | -   |
| 股票简称 | object | -   |
| 首次预约 | object | -   |
| 初次变更 | object | -   |
| 二次变更 | object | -   |
| 三次变更 | object | -   |
| 实际披露 | object | -   |

接口示例

```python
import akshare as ak

stock_report_disclosure_df = ak.stock_report_disclosure(market="沪深京", period="2022年报")
print(stock_report_disclosure_df)
```

数据示例

```
      股票代码   股票简称  首次预约        初次变更 二次变更 三次变更 实际披露
0     000001   平安银行  2023-03-09         NaT  NaN  NaN  2023-03-09
1     000002  万  科Ａ  2023-03-31         NaT  NaN  NaN  2023-03-31
2     000004   国华网安  2023-04-21  2023-04-29  NaN  NaN  2023-04-29
3     000005   ST星源  2023-04-29         NaT  NaN  NaN  2023-04-29
4     000006   深振业Ａ  2023-04-28         NaT  NaN  NaN  2023-04-28
      ...    ...         ...         ...  ...  ...         ...
5102  873223   荣亿精密  2023-04-26         NaT  NaN  NaN  2023-04-26
5103  873305   九菱科技  2023-04-26         NaT  NaN  NaN  2023-04-26
5104  873339   恒太照明  2023-04-25  2023-04-12  NaN  NaN  2023-04-12
5105  873527    夜光明  2023-04-17         NaT  NaN  NaN  2023-04-17
5106  873593   鼎智科技  2023-04-25  2023-03-17  NaN  NaN         NaT
[5107 rows x 7 columns]
```

#### 信息披露公告-巨潮资讯

接口: stock_zh_a_disclosure_report_cninfo

目标地址: http://www.cninfo.com.cn/new/commonUrl/pageOfSearch?url=disclosure/list/search

描述: 巨潮资讯-首页-公告查询-信息披露公告-沪深京

限量: 单次获取指定 symbol 的信息披露公告数据

输入参数

| 名称         | 类型  | 描述                                                                                                                                                                                                                         |
|------------|-----|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| symbol     | str | symbol="000001"; 股票代码                                                                                                                                                                                                      |
| market     | str | market="沪深京"; choice of {"沪深京", "港股", "三板", "基金", "债券", "监管", "预披露"}                                                                                                                                                       |
| keyword    | str | keyword=""; 关键词                                                                                                                                                                                                            |
| category   | str | category=""; choice of {'年报', '半年报', '一季报', '三季报', '业绩预告', '权益分派', '董事会', '监事会', '股东大会', '日常经营', '公司治理', '中介报告', '首发', '增发', '股权激励', '配股', '解禁', '公司债', '可转债', '其他融资', '股权变动', '补充更正', '澄清致歉', '风险提示', '特别处理和退市', '退市整理期'} |
| start_date | str | start_date="20230618"                                                                                                                                                                                                      |
| end_date   | str | end_date="20231219"                                                                                                                                                                                                        |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 代码   | object | -  |
| 简称   | object | -  |
| 公告标题 | object | -  |
| 公告时间 | object | -  |
| 公告链接 | object | -  |

接口示例

```python
import akshare as ak

stock_zh_a_disclosure_report_cninfo_df = ak.stock_zh_a_disclosure_report_cninfo(symbol="000001", market="沪深京", category="公司治理", start_date="20230619", end_date="20231220")
print(stock_zh_a_disclosure_report_cninfo_df)
```

数据示例

```
   代码    简称  ...        公告时间                               公告链接
0  000001  平安银行  ...  2023-12-09  http://www.cninfo.com.cn/new/disclosure/detail...
1  000001  平安银行  ...  2023-11-21  http://www.cninfo.com.cn/new/disclosure/detail...
2  000001  平安银行  ...  2023-10-25  http://www.cninfo.com.cn/new/disclosure/detail...
3  000001  平安银行  ...  2023-08-24  http://www.cninfo.com.cn/new/disclosure/detail...
4  000001  平安银行  ...  2023-07-29  http://www.cninfo.com.cn/new/disclosure/detail...
5  000001  平安银行  ...  2023-07-25  http://www.cninfo.com.cn/new/disclosure/detail...
[6 rows x 5 columns]
```

#### 信息披露调研-巨潮资讯

接口: stock_zh_a_disclosure_relation_cninfo

目标地址: http://www.cninfo.com.cn/new/commonUrl/pageOfSearch?url=disclosure/list/search

描述: 巨潮资讯-首页-公告查询-信息披露调研-沪深京

限量: 单次获取指定 symbol 的信息披露调研数据

输入参数

| 名称         | 类型  | 描述                                                                   |
|------------|-----|----------------------------------------------------------------------|
| symbol     | str | symbol="000001"; 股票代码                                                |
| market     | str | market="沪深京"; choice of {"沪深京", "港股", "三板", "基金", "债券", "监管", "预披露"} |
| start_date | str | start_date="20230618"                                                |
| end_date   | str | end_date="20231219"                                                  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 代码   | object | -  |
| 简称   | object | -  |
| 公告标题 | object | -  |
| 公告时间 | object | -  |
| 公告链接 | object | -  |

接口示例

```python
import akshare as ak

stock_zh_a_disclosure_relation_cninfo_df = ak.stock_zh_a_disclosure_relation_cninfo(symbol="000001", market="沪深京", start_date="20230619", end_date="20231220")
print(stock_zh_a_disclosure_relation_cninfo_df)
```

数据示例

```
   代码    简称  ...        公告时间                               公告链接
0  000001  平安银行  ...  2023-11-08  http://www.cninfo.com.cn/new/disclosure/detail...
1  000001  平安银行  ...  2023-11-01  http://www.cninfo.com.cn/new/disclosure/detail...
2  000001  平安银行  ...  2023-10-24  http://www.cninfo.com.cn/new/disclosure/detail...
3  000001  平安银行  ...  2023-10-24  http://www.cninfo.com.cn/new/disclosure/detail...
4  000001  平安银行  ...  2023-10-24  http://www.cninfo.com.cn/new/disclosure/detail...
5  000001  平安银行  ...  2023-10-24  http://www.cninfo.com.cn/new/disclosure/detail...
6  000001  平安银行  ...  2023-08-23  http://www.cninfo.com.cn/new/disclosure/detail...
7  000001  平安银行  ...  2023-08-23  http://www.cninfo.com.cn/new/disclosure/detail...
8  000001  平安银行  ...  2023-08-23  http://www.cninfo.com.cn/new/disclosure/detail...
9  000001  平安银行  ...  2023-08-23  http://www.cninfo.com.cn/new/disclosure/detail...
[10 rows x 5 columns]
```

#### 行业分类数据-巨潮资讯

接口: stock_industry_category_cninfo

目标地址: https://webapi.cninfo.com.cn/#/apiDoc

描述: 巨潮资讯-数据-行业分类数据

限量: 单次获取指定 symbol 的行业分类数据

输入参数

| 名称     | 类型  | 描述                                                                                                                                 |
|--------|-----|------------------------------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="巨潮行业分类标准"; choice of {"证监会行业分类标准", "巨潮行业分类标准", "申银万国行业分类标准", "新财富行业分类标准", "国资委行业分类标准", "巨潮产业细分标准", "天相行业分类标准", "全球行业分类标准"} |

输出参数

| 名称     | 类型         | 描述  |
|--------|------------|-----|
| 类目编码   | object     | -   |
| 类目名称   | object     | -   |
| 终止日期   | datetime64 | -   |
| 行业类型   | object     | -   |
| 行业类型编码 | object     | -   |
| 类目名称英文 | object     | -   |
| 父类编码   | object     | -   |
| 分级     | int32      | -   |

接口示例

```python
import akshare as ak

stock_industry_category_cninfo_df = ak.stock_industry_category_cninfo(symbol="巨潮行业分类标准")
print(stock_industry_category_cninfo_df)
```

数据示例

```
          类目编码      类目名称  ...     父类编码 分级
0            Z  巨潮行业分类标准  ...      008  0
1          Z01        能源  ...        Z  1
2        Z0101        能源  ...      Z01  2
3      Z010101   能源设备与服务  ...    Z0101  3
4    Z01010101      能源设备  ...  Z010101  4
..         ...       ...  ...      ... ..
289  Z11020101     房地产信托  ...  Z110201  4
290    Z110202     房地产租赁  ...    Z1102  3
291  Z11020201     房地产租赁  ...  Z110202  4
292    Z110203   其他房地产服务  ...    Z1102  3
293  Z11020301   其他房地产服务  ...  Z110203  4
[294 rows x 8 columns]
```

#### 上市公司行业归属的变动情况-巨潮资讯

接口: stock_industry_change_cninfo

目标地址: http://webapi.cninfo.com.cn/#/apiDoc

描述: 巨潮资讯-数据-上市公司行业归属的变动情况

限量: 单次获取指定 symbol 在 start_date 和 end_date 之间的上市公司行业归属的变动情况数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="002594"       |
| start_date | str | start_date="20091227" |
| end_date   | str | end_date="20220708"   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 新证券简称  | object | -   |
| 行业中类   | object | -   |
| 行业大类   | object | -   |
| 行业次类   | object | -   |
| 行业门类   | object | -   |
| 机构名称   | object | -   |
| 行业编码   | object | -   |
| 分类标准   | object | -   |
| 分类标准编码 | object | -   |
| 证券代码   | object | -   |
| 变更日期   | object | -   |

接口示例

```python
import akshare as ak

stock_industry_change_cninfo_df = ak.stock_industry_change_cninfo(symbol="002594", start_date="20091227", end_date="20220708")
print(stock_industry_change_cninfo_df)
```

数据示例

```
   新证券简称      行业中类      行业大类  ...  分类标准编码    证券代码        变更日期
0    比亚迪  汽车和汽车零部件  汽车和汽车零部件  ...  008004  002594  2011-06-08
1    比亚迪     其他制造业     其他制造业  ...  008009  002594  2011-06-08
2    比亚迪       乘用车       乘用车  ...  008018  002594  2011-06-21
3    比亚迪       NaN     汽车制造业  ...  008001  002594  2011-06-30
4    比亚迪        轿车        汽车  ...  008013  002594  2011-06-30
5    比亚迪        轿车        汽车  ...  008013  002594  2017-10-16
6    比亚迪        轿车        汽车  ...  008002  002594  2019-05-28
7    比亚迪        汽车    汽车与摩托车  ...  008019  002594  2020-03-16
8    比亚迪       NaN        汽车  ...  008016  002594  2021-06-21
9    比亚迪     电动乘用车     电动乘用车  ...  008003  002594  2021-07-30
10   比亚迪       乘用车       乘用车  ...  008014  002594  2021-12-17
[11 rows x 11 columns]
```

#### 公司股本变动-巨潮资讯

接口: stock_share_change_cninfo

目标地址: https://webapi.cninfo.com.cn/#/apiDoc

描述: 巨潮资讯-数据-公司股本变动

限量: 单次获取指定 symbol 在 start_date 和 end_date 之间的公司股本变动数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="002594"       |
| start_date | str | start_date="20091227" |
| end_date   | str | end_date="20241021"   |

输出参数

| 名称         | 类型      | 描述  |
|------------|---------|-----|
| 证券简称       | object  | -   |
| 机构名称       | object  | -   |
| 境外法人持股     | float64 | -   |
| 证券投资基金持股   | float64 | -   |
| 国家持股-受限    | float64 | -   |
| 国有法人持股     | float64 | -   |
| 配售法人股      | float64 | -   |
| 发起人股份      | float64 | -   |
| 未流通股份      | float64 | -   |
| 其中：境外自然人持股 | float64 | -   |
| 其他流通受限股份   | float64 | -   |
| 其他流通股      | float64 | -   |
| 外资持股-受限    | float64 | -   |
| 内部职工股      | float64 | -   |
| 境外上市外资股-H股 | float64 | -   |
| 其中：境内法人持股  | float64 | -   |
| 自然人持股      | float64 | -   |
| 人民币普通股     | float64 | -   |
| 国有法人持股-受限  | float64 | -   |
| 一般法人持股     | float64 | -   |
| 控股股东、实际控制人 | float64 | -   |
| 其中：限售H股    | float64 | -   |
| 变动原因       | object  | -   |
| 公告日期       | object  | -   |
| 境内法人持股     | float64 | -   |
| 证券代码       | object  | -   |
| 变动日期       | object  | -   |
| 战略投资者持股    | float64 | -   |
| 国家持股       | float64 | -   |
| 其中：限售B股    | float64 | -   |
| 其他未流通股     | float64 | -   |
| 流通受限股份     | float64 | -   |
| 优先股        | float64 | -   |
| 高管股        | float64 | -   |
| 总股本        | float64 | -   |
| 其中：限售高管股   | float64 | -   |
| 转配股        | float64 | -   |
| 境内上市外资股-B股 | float64 | -   |
| 其中：境外法人持股  | float64 | -   |
| 募集法人股      | float64 | -   |
| 已流通股份      | float64 | -   |
| 其中：境内自然人持股 | float64 | -   |
| 其他内资持股-受限  | float64 | -   |
| 变动原因编码     | object  | -   |

接口示例

```python
import akshare as ak

stock_share_change_cninfo_df = ak.stock_share_change_cninfo(symbol="002594", start_date="20091227", end_date="20241021")
print(stock_share_change_cninfo_df)
```

数据示例

```
   证券简称     机构名称 境外法人持股  ...   其中：境内自然人持股  其他内资持股-受限  变动原因编码
0   比亚迪  比亚迪股份有限公司    NaN  ...  129518.2500  146457.3960  015001,015019
1   比亚迪  比亚迪股份有限公司    NaN  ...  129518.2500  146457.3960         015009
2   比亚迪  比亚迪股份有限公司    NaN  ...  129518.2500  146457.3960         015019
3   比亚迪  比亚迪股份有限公司    NaN  ...  103909.7150  120525.4050         015019
4   比亚迪  比亚迪股份有限公司    NaN  ...   82121.6860   98737.3760         015027
5   比亚迪  比亚迪股份有限公司    NaN  ...  105034.8585  121650.5485         015019
6   比亚迪  比亚迪股份有限公司    NaN  ...  101057.0665  117672.7565         015019
7   比亚迪  比亚迪股份有限公司    NaN  ...  101057.0665  117672.7565         015019
8   比亚迪  比亚迪股份有限公司    NaN  ...  101057.0665  117672.7565         015005
9   比亚迪  比亚迪股份有限公司    NaN  ...   74641.3870   86835.0265         015019
10  比亚迪  比亚迪股份有限公司    NaN  ...   77933.1165   90126.7560         015027
11  比亚迪  比亚迪股份有限公司    NaN  ...   73295.2720   85488.9115         015019
12  比亚迪  比亚迪股份有限公司    NaN  ...   74625.3530   84758.9925         015019
13  比亚迪  比亚迪股份有限公司    NaN  ...   73295.2720   81424.3650         015027
14  比亚迪  比亚迪股份有限公司    NaN  ...   72387.8051   80516.8981         015019
15  比亚迪  比亚迪股份有限公司    NaN  ...   67996.6256   72061.1721         015019
16  比亚迪  比亚迪股份有限公司    NaN  ...   72387.8051   76452.3516         015027
17  比亚迪  比亚迪股份有限公司    NaN  ...   67996.6256   95881.7294         015004
18  比亚迪  比亚迪股份有限公司    NaN  ...   67044.1386   96322.9706         015019
19  比亚迪  比亚迪股份有限公司    NaN  ...   67480.9896   92695.2751         015019
20  比亚迪  比亚迪股份有限公司    NaN  ...   67044.1386   92258.4241         015027
21  比亚迪  比亚迪股份有限公司    NaN  ...   67480.9896   67480.9896         015027
22  比亚迪  比亚迪股份有限公司    NaN  ...   67480.9896   67480.9896         015019
23  比亚迪  比亚迪股份有限公司    NaN  ...   67134.3301   67134.3301         015019
24  比亚迪  比亚迪股份有限公司    NaN  ...   67092.8936   67092.8936         015019
25  比亚迪  比亚迪股份有限公司    NaN  ...   67248.1221   67248.1221         015019
26  比亚迪  比亚迪股份有限公司    NaN  ...   67151.6826   67151.6826         015019
27  比亚迪  比亚迪股份有限公司    NaN  ...   66747.5312   66747.5312         015019
28  比亚迪  比亚迪股份有限公司    NaN  ...   66714.9897   66714.9897         015019
29  比亚迪  比亚迪股份有限公司    NaN  ...   66747.5312   66747.5312         015004
30  比亚迪  比亚迪股份有限公司    NaN  ...   65808.4620   65808.4620         015019
31  比亚迪  比亚迪股份有限公司    NaN  ...   65808.4620   65808.4620         015004
32  比亚迪  比亚迪股份有限公司    NaN  ...   65808.4620   65808.4620         015019
33  比亚迪  比亚迪股份有限公司    NaN  ...   64837.8251   64837.8251         015999
34  比亚迪  比亚迪股份有限公司    NaN  ...   64837.8251   64837.8251         015019
35  比亚迪  比亚迪股份有限公司    NaN  ...   64837.8251   64837.8251         015019
36  比亚迪  比亚迪股份有限公司    NaN  ...   64837.8251   64837.8251         015019
37  比亚迪  比亚迪股份有限公司    NaN  ...   64888.0804   64888.0804         015019
38  比亚迪  比亚迪股份有限公司    NaN  ...   64879.5289   64879.5289         015999
39  比亚迪  比亚迪股份有限公司    NaN  ...   64879.5289   64879.5289         015016
40  比亚迪  比亚迪股份有限公司    NaN  ...   64880.8339   64880.8339         015019
[41 rows x 44 columns]
```

#### 配股实施方案-巨潮资讯

接口: stock_allotment_cninfo

目标地址: http://webapi.cninfo.com.cn/#/dataBrowse

描述: 巨潮资讯-个股-配股实施方案

限量: 单次获取指定 symbol 在 start_date 和 end_date 之间的公司股本变动数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="600030"       |
| start_date | str | start_date="19700101" |
| end_date   | str | end_date="22220222"   |

输出参数

| 名称           | 类型         | 描述  |
|--------------|------------|-----|
| 记录标识         | int64      | -   |
| 证券简称         | object     | -   |
| 停牌起始日        | object     | -   |
| 上市公告日期       | object     | -   |
| 配股缴款起始日      | object     | -   |
| 可转配股数量       | float64    | -   |
| 停牌截止日        | object     | -   |
| 实际配股数量       | float64    | -   |
| 配股价格         | float64    | -   |
| 配股比例         | float64    | -   |
| 配股前总股本       | float64    | -   |
| 每股配权转让费(元)   | float64    | -   |
| 法人股实配数量      | float64    | -   |
| 实际募资净额       | float64    | -   |
| 大股东认购方式      | object     | -   |
| 其他配售简称       | object     | -   |
| 发行方式         | object     | -   |
| 配股失败，退还申购款日期 | object     | -   |
| 除权基准日        | object     | -   |
| 预计发行费用       | float64    | -   |
| 配股发行结果公告日    | object     | -   |
| 证券代码         | object     | -   |
| 配股权证交易截止日    | datetime64 | -   |
| 其他股份实配数量     | float64    | -   |
| 国家股实配数量      | float64    | -   |
| 委托单位         | object     | -   |
| 公众获转配数量      | float64    | -   |
| 其他配售代码       | object     | -   |
| 配售对象         | object     | -   |
| 配股权证交易起始日    | datetime64 | -   |
| 资金到账日        | datetime64 | -   |
| 机构名称         | object     | -   |
| 股权登记日        | object     | -   |
| 实际募资总额       | float64    | -   |
| 预计募集资金       | float64    | -   |
| 大股东认购数量      | float64    | -   |
| 公众股实配数量      | float64    | -   |
| 转配股实配数量      | float64    | -   |
| 承销费用         | float64    | -   |
| 法人获转配数量      | float64    | -   |
| 配股后流通股本      | float64    | -   |
| 股票类别         | object     | -   |
| 公众配售简称       | object     | -   |
| 发行方式编码       | object     | -   |
| 承销方式         | object     | -   |
| 公告日期         | object     | -   |
| 配股上市日        | object     | -   |
| 配股缴款截止日      | object     | -   |
| 承销余额(股)      | float64    | -   |
| 预计配股数量       | float64    | -   |
| 配股后总股本       | float64    | -   |
| 职工股实配数量      | float64    | -   |
| 承销方式编码       | object     | -   |
| 发行费用总额       | float64    | -   |
| 配股前流通股本      | float64    | -   |
| 股票类别编码       | object     | -   |
| 公众配售代码       | object     | -   |

接口示例

```python
import akshare as ak

stock_allotment_cninfo_df = ak.stock_allotment_cninfo(symbol="600030", start_date="19900101", end_date="20241022")
print(stock_allotment_cninfo_df)
```

数据示例

```
    记录标识  证券简称     停牌起始日  ...  配股前流通股本  股票类别编码 公众配售代码
0  4475575959  中信证券  2022-01-19  ...  1209298.94  001001  700030
[1 rows x 57 columns]
```

#### 公司概况-巨潮资讯

接口: stock_profile_cninfo

目标地址: http://webapi.cninfo.com.cn/#/company

描述: 巨潮资讯-个股-公司概况

限量: 单次获取指定 symbol 的公司概况

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="600030"       |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 公司名称 | object | -   |
| 英文名称 | object | -   |
| 曾用简称 | object | -   |
| A股代码 | object | -   |
| A股简称 | object | -   |
| B股代码 | object | -   |
| B股简称 | object | -   |
| H股代码 | object | -   |
| H股简称 | object | -   |
| 入选指数 | object | -   |
| 所属市场 | object | -   |
| 所属行业 | object | -   |
| 法人代表 | object | -   |
| 注册资金 | object | -   |
| 成立日期 | object | -   |
| 上市日期 | object | -   |
| 官方网站 | object | -   |
| 电子邮箱 | object | -   |
| 联系电话 | object | -   |
| 传真   | object | -   |
| 注册地址 | object | -   |
| 办公地址 | object | -   |
| 邮政编码 | object | -   |
| 主营业务 | object | -   |
| 经营范围 | object | -   |
| 机构简介 | object | -   |

接口示例

```python
import akshare as ak

stock_profile_cninfo_df = ak.stock_profile_cninfo(symbol="600030")
print(stock_profile_cninfo_df)
```

数据示例

```
         公司名称  ...                                               机构简介
0  中信证券股份有限公司  ...  公司的前身中信证券有限责任公司是经中国人民银行银复[1995]313号文批准，由中信公司，中...
```

#### 上市相关-巨潮资讯

接口: stock_ipo_summary_cninfo

目标地址: https://webapi.cninfo.com.cn/#/company

描述: 巨潮资讯-个股-上市相关

限量: 单次获取指定 symbol 的上市相关数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| symbol     | str | symbol="600030"       |

输出参数

| 名称       | 类型      | 描述       |
|----------|---------|----------|
| 股票代码     | object  | -        |
| 招股公告日期   | object  | -        |
| 中签率公告日   | object  | -        |
| 每股面值     | float64 | 注意单位: 元  |
| 总发行数量    | float64 | 注意单位: 万股 |
| 发行前每股净资产 | float64 | 注意单位: 元  |
| 摊薄发行市盈率  | float64 | -        |
| 募集资金净额   | float64 | 注意单位: 万元 |
| 上网发行日期   | object  | -        |
| 上市日期     | object  | -        |
| 发行价格     | float64 | 注意单位: 元  |
| 发行费用总额   | float64 | 注意单位: 万元 |
| 发行后每股净资产 | float64 | 注意单位: 元  |
| 上网发行中签率  | float64 | 注意单位: %  |
| 主承销商     | float64 | -        |

接口示例

```python
import akshare as ak

stock_ipo_summary_cninfo_df = ak.stock_ipo_summary_cninfo(symbol="600030")
print(stock_ipo_summary_cninfo_df)
```

数据示例

```
   股票代码      招股公告日期 中签率公告日  每股面值  ...     发行费用总额  发行后每股净资产  上网发行中签率        主承销商
0  600030  2002-12-13    NaT   1.0  ...  4032.6625      2.11    0.282  广发证券股份有限公司
[1 rows x 15 columns]
```

#### 资产负债表-沪深

接口: stock_zcfz_em

目标地址: https://data.eastmoney.com/bbsj/202003/zcfz.html

描述: 东方财富-数据中心-年报季报-业绩快报-资产负债表

限量: 单次获取指定 date 的资产负债表数据

输入参数

| 名称   | 类型  |  描述                                                                                        |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20240331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20081231 开始 |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 股票代码     | object  | -       |
| 股票简称     | object  | -       |
| 资产-货币资金  | float64 | 注意单位: 元 |
| 资产-应收账款  | float64 | 注意单位: 元 |
| 资产-存货    | float64 | 注意单位: 元 |
| 资产-总资产   | float64 | 注意单位: 元 |
| 资产-总资产同比 | float64 | 注意单位: % |
| 负债-应付账款  | float64 | 注意单位: 元 |
| 负债-总负债   | float64 | 注意单位: 元 |
| 负债-预收账款  | float64 | 注意单位: 元 |
| 负债-总负债同比 | float64 | 注意单位: % |
| 资产负债率    | float64 | 注意单位: % |
| 股东权益合计   | float64 | 注意单位: 元 |
| 公告日期     | object  | -       |

接口示例

```python
import akshare as ak

stock_zcfz_em_df = ak.stock_zcfz_em(date="20240331")
print(stock_zcfz_em_df)
```

数据示例

```
     序号    股票代码   股票简称  ...   资产负债率  股东权益合计       公告日期
0        1  603156   养元饮品  ...  20.992996  1.160939e+10  2024-09-10
1        2  002569   ST步森  ...  62.799334  8.476112e+07  2024-09-07
2        3  603260   合盛硅业  ...  63.774717  3.253375e+10  2024-08-30
3        4  300417   南华仪器  ...   8.130360  4.507915e+08  2024-08-30
4        5  300081   恒信东方  ...  30.634305  1.319848e+09  2024-08-30
...    ...     ...    ...  ...        ...           ...         ...
5123  5124  300076  GQY视讯  ...  10.010959  9.880683e+08  2024-04-10
5124  5125  002644   佛慈制药  ...  29.294961  1.787072e+09  2024-04-10
5125  5126  603058   永吉股份  ...  35.639627  1.237896e+09  2024-04-09
5126  5127  600873   梅花生物  ...  36.564962  1.486533e+10  2024-04-09
5127  5128  000818   航锦科技  ...  40.879097  4.118588e+09  2024-04-03
[5128 rows x 15 columns]
```

#### 资产负债表-北交所

接口: stock_zcfz_bj_em

目标地址: https://data.eastmoney.com/bbsj/202003/zcfz.html

描述: 东方财富-数据中心-年报季报-业绩快报-资产负债表

限量: 单次获取指定 date 的资产负债表数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20240331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20081231 开始 |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 股票代码     | object  | -       |
| 股票简称     | object  | -       |
| 资产-货币资金  | float64 | 注意单位: 元 |
| 资产-应收账款  | float64 | 注意单位: 元 |
| 资产-存货    | float64 | 注意单位: 元 |
| 资产-总资产   | float64 | 注意单位: 元 |
| 资产-总资产同比 | float64 | 注意单位: % |
| 负债-应付账款  | float64 | 注意单位: 元 |
| 负债-总负债   | float64 | 注意单位: 元 |
| 负债-预收账款  | float64 | 注意单位: 元 |
| 负债-总负债同比 | float64 | 注意单位: % |
| 资产负债率    | float64 | 注意单位: % |
| 股东权益合计   | float64 | 注意单位: 元 |
| 公告日期     | object  | -       |

接口示例

```python
import akshare as ak

stock_zcfz_bj_em_df = ak.stock_zcfz_bj_em(date="20240331")
print(stock_zcfz_bj_em_df)
```

数据示例

```
     序号 股票代码  股票简称  ...  资产负债率    股东权益合计     公告日期
0      1  873223  荣亿精密  ...  38.708142  2.856643e+08  2024-08-29
1      2  838030  德众汽车  ...  72.413909  4.696866e+08  2024-08-28
2      3  830974  凯大催化  ...  26.834128  6.476251e+08  2024-08-28
3      4  834415  恒拓开源  ...  16.546915  5.229393e+08  2024-06-27
4      5  920118  太湖远大  ...  57.054651  4.325410e+08  2024-05-23
..   ...     ...   ...  ...        ...           ...         ...
249  250  835892  中科美菱  ...  19.425219  6.066262e+08  2024-04-18
250  251  834261   一诺威  ...  38.597474  1.475388e+09  2024-04-18
251  252  873703  广厦环能  ...  25.348767  1.018675e+09  2024-04-16
252  253  838262   太湖雪  ...  36.163348  3.227086e+08  2024-04-15
253  254  833394   民士达  ...  19.625162  6.865148e+08  2024-04-11
[254 rows x 15 columns]
```

#### 利润表

接口: stock_lrb_em

目标地址: http://data.eastmoney.com/bbsj/202003/lrb.html

描述: 东方财富-数据中心-年报季报-业绩快报-利润表

限量: 单次获取指定 date 的利润表数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20240331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20120331 开始 |

输出参数

| 名称          | 类型      | 描述      |
|-------------|---------|---------|
| 序号          | int64   | -       |
| 股票代码        | object  | -       |
| 股票简称        | object  | -       |
| 净利润         | float64 | 注意单位: 元 |
| 净利润同比       | float64 | 注意单位: % |
| 营业总收入       | float64 | 注意单位: 元 |
| 营业总收入同比     | float64 | 注意单位: % |
| 营业总支出-营业支出  | float64 | 注意单位: 元 |
| 营业总支出-销售费用  | float64 | 注意单位: 元 |
| 营业总支出-管理费用  | float64 | 注意单位: 元 |
| 营业总支出-财务费用  | float64 | 注意单位: 元 |
| 营业总支出-营业总支出 | float64 | 注意单位: 元 |
| 营业利润        | float64 | 注意单位: 元 |
| 利润总额        | float64 | 注意单位: 元 |
| 公告日期        | object  | -       |

接口示例

```python
import akshare as ak

stock_lrb_em_df = ak.stock_lrb_em(date="20240331")
print(stock_lrb_em_df)
```

数据示例

```
     序号    股票代码   股票简称  ...   营业利润         利润总额       公告日期
0        1  603156   养元饮品  ...  1.078460e+09  1.078821e+09  2024-09-10
1        2  002569   ST步森  ... -8.504272e+06 -8.601409e+06  2024-09-07
2        3  603260   合盛硅业  ...  7.390596e+08  7.561255e+08  2024-08-30
3        4  300417   南华仪器  ... -4.807409e+06 -4.815009e+06  2024-08-30
4        5  300081   恒信东方  ... -3.694795e+07 -3.697999e+07  2024-08-30
...    ...     ...    ...  ...           ...           ...         ...
5123  5124  300076  GQY视讯  ... -2.973570e+05 -2.575340e+05  2024-04-10
5124  5125  002644   佛慈制药  ...  2.299932e+07  2.260271e+07  2024-04-10
5125  5126  603058   永吉股份  ...  5.088006e+07  5.084289e+07  2024-04-09
5126  5127  600873   梅花生物  ...  8.858530e+08  8.755312e+08  2024-04-09
5127  5128  000818   航锦科技  ...  8.761395e+07  8.890215e+07  2024-04-03
[5128 rows x 15 columns]
```

#### 现金流量表

接口: stock_xjll_em

目标地址: http://data.eastmoney.com/bbsj/202003/xjll.html

描述: 东方财富-数据中心-年报季报-业绩快报-现金流量表

限量: 单次获取指定 date 的现金流量表数据

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20200331"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20081231 开始 |

输出参数

| 名称            | 类型      | 描述      |
|---------------|---------|---------|
| 序号            | int64   | -       |
| 股票代码          | object  | -       |
| 股票简称          | object  | -       |
| 净现金流-净现金流     | float64 | 注意单位: 元 |
| 净现金流-同比增长     | float64 | 注意单位: % |
| 经营性现金流-现金流量净额 | float64 | 注意单位: 元 |
| 经营性现金流-净现金流占比 | float64 | 注意单位: % |
| 投资性现金流-现金流量净额 | float64 | 注意单位: 元 |
| 投资性现金流-净现金流占比 | float64 | 注意单位: % |
| 融资性现金流-现金流量净额 | float64 | 注意单位: 元 |
| 融资性现金流-净现金流占比 | float64 | 注意单位: % |
| 公告日期          | object  | -       |

接口示例

```python
import akshare as ak

stock_xjll_em_df = ak.stock_xjll_em(date="20240331")
print(stock_xjll_em_df)
```

数据示例

```
      序号   股票代码   股票简称  ... 融资性现金流-现金流量净额 融资性现金流-净现金流占比  公告日期
0        1  603156   养元饮品  ...   8.906149e+07       8.556244  2024-09-10
1        2  002569   ST步森  ...            NaN            NaN  2024-09-07
2        3  603260   合盛硅业  ...   3.236085e+09    7017.983443  2024-08-30
3        4  300417   南华仪器  ...  -1.980000e+04      -0.047350  2024-08-30
4        5  300081   恒信东方  ...  -6.422119e+06     -30.968169  2024-08-30
...    ...     ...    ...  ...            ...            ...         ...
5123  5124  300076  GQY视讯  ...   3.252000e+05       1.435320  2024-04-10
5124  5125  002644   佛慈制药  ...  -1.388667e+07     -43.325298  2024-04-10
5125  5126  603058   永吉股份  ...   3.204449e+06       5.454042  2024-04-09
5126  5127  600873   梅花生物  ...   1.604316e+08      47.970510  2024-04-09
5127  5128  000818   航锦科技  ...   5.813673e+08     274.656484  2024-04-03
[5128 rows x 12 columns]
```

### 高管持股

#### 股东增减持

接口: stock_ggcg_em

目标地址: http://data.eastmoney.com/executive/gdzjc.html

描述: 东方财富网-数据中心-特色数据-高管持股

限量: 单次获取所有高管持股数据数据

输入参数

| 名称     | 类型  | 描述                                            |
|--------|-----|-----------------------------------------------|
| symbol | str | symbol="全部"; choice of {"全部", "股东增持", "股东减持"} |

输出参数

| 名称             | 类型      | 描述       |
|----------------|---------|----------|
| 代码             | object  | -        |
| 名称             | object  | -        |
| 最新价            | float64 | -        |
| 涨跌幅            | float64 | 注意单位: %  |
| 股东名称           | object  | -        |
| 持股变动信息-增减      | float64 | -        |
| 持股变动信息-变动数量    | float64 | 注意单位: 万股 |
| 持股变动信息-占总股本比例  | float64 | 注意单位: %  |
| 持股变动信息-占流通股比例  | float64 | 注意单位: %  |
| 变动后持股情况-持股总数   | float64 | 注意单位: 万股 |
| 变动后持股情况-占总股本比例 | float64 | 注意单位: %  |
| 变动后持股情况-持流通股数  | float64 | 注意单位: 万股 |
| 变动后持股情况-占流通股比例 | float64 | 注意单位: %  |
| 变动开始日          | object  | -        |
| 变动截止日          | object  | -        |
| 公告日            | object  | -        |

接口示例

```python
import akshare as ak

stock_ggcg_em_df = ak.stock_ggcg_em(symbol="全部")
print(stock_ggcg_em_df)
```

数据示例

```
        代码   名称       最新价  ...    变动开始日   变动截止日       公告日
0       688369  致远互联  44.19  ...  2022-01-05  2022-04-29  2022-04-30
1       688106  金宏气体  16.33  ...         NaT  2022-04-29  2022-04-30
2       601966  玲珑轮胎  17.77  ...  2022-04-29  2022-04-29  2022-04-30
3       601199  江南水务   5.80  ...  2022-01-29  2022-04-29  2022-04-30
4       600346  恒力石化  20.95  ...         NaT  2022-04-29  2022-04-30
        ...   ...    ...  ...         ...         ...         ...
112446  600653  申华控股   1.83  ...  1996-11-29  1996-12-03  1996-12-04
112447  600653  申华控股   1.83  ...  1996-11-26  1996-11-28  1996-11-29
112448  600653  申华控股   1.83  ...  1996-11-21  1996-11-25  1996-11-26
112449  600653  申华控股   1.83  ...         NaT  1996-11-20  1996-11-21
112450  600651  飞乐音响   2.96  ...         NaT  1994-08-05  1994-08-10
```

### 分红配送

#### 分红配送-东财

接口: stock_fhps_em

目标地址: https://data.eastmoney.com/yjfp/

描述: 东方财富-数据中心-年报季报-分红配送

限量: 单次获取指定日期的分红配送数据

输入参数

| 名称   | 类型  | 描述                                                                 |
|------|-----|--------------------------------------------------------------------|
| date | str | date="20231231"; choice of {"XXXX0630", "XXXX1231"}; 从 19901231 开始 |

输出参数

| 名称          | 类型      | 描述  |
|-------------|---------|-----|
| 代码          | object  | -   |
| 名称          | object  | -   |
| 送转股份-送转总比例  | float64 | -   |
| 送转股份-送转比例   | float64 | -   |
| 送转股份-转股比例   | float64 | -   |
| 现金分红-现金分红比例 | float64 | -   |
| 现金分红-股息率    | float64 | -   |
| 每股收益        | float64 | -   |
| 每股净资产       | float64 | -   |
| 每股公积金       | float64 | -   |
| 每股未分配利润     | float64 | -   |
| 净利润同比增长     | float64 | -   |
| 总股本         | int64   | -   |
| 预案公告日       | object  | -   |
| 股权登记日       | object  | -   |
| 除权除息日       | object  | -   |
| 方案进度        | object  | -   |
| 最新公告日期      | object  | -   |

接口示例

```python
import akshare as ak

stock_fhps_em_df = ak.stock_fhps_em(date="20231231")
print(stock_fhps_em_df)
```

数据示例

```
      代码    名称  送转股份-送转总比例  ...    除权除息日      方案进度   最新公告日期
0     300708  聚灿光电         NaN  ...  2024-02-28      实施分配  2024-02-21
1     688082  盛美上海         NaN  ...         NaT   董事会决议通过  2024-02-29
2     002286   保龄宝         NaN  ...  2024-03-08      实施分配  2024-03-01
3     605198   安德利         NaN  ...         NaT  股东大会决议通过  2024-03-07
4     000063  中兴通讯         NaN  ...         NaT   董事会决议通过  2024-03-09
...      ...   ...         ...  ...         ...       ...         ...
3861  600449  宁夏建材         NaN  ...  2024-06-13      实施分配  2024-06-05
3862  002737  葵花药业         NaN  ...  2024-06-14      实施分配  2024-06-05
3863  603915  国茂股份         NaN  ...  2024-06-12      实施分配  2024-06-05
3864  688390   固德威         4.0  ...  2024-06-12      实施分配  2024-06-05
3865  603002  宏昌电子         NaN  ...  2024-06-14      实施分配  2024-06-05
[3866 rows x 18 columns]
```

#### 分红配送详情-东财

接口: stock_fhps_detail_em

目标地址: https://data.eastmoney.com/yjfp/detail/300073.html

描述: 东方财富网-数据中心-分红送配-分红送配详情

限量: 单次获取指定 symbol 的分红配送详情数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="300073" |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| 报告期           | object  | -  |
| 业绩披露日期        | object  | -  |
| 送转股份-送转总比例    | float64 | -  |
| 送转股份-送股比例     | float64 | -  |
| 送转股份-转股比例     | float64 | -  |
| 现金分红-现金分红比例   | float64 | -  |
| 现金分红-现金分红比例描述 | object  | -  |
| 现金分红-股息率      | float64 | -  |
| 每股收益          | float64 | -  |
| 每股净资产         | float64 | -  |
| 每股公积金         | float64 | -  |
| 每股未分配利润       | float64 | -  |
| 净利润同比增长       | float64 | -  |
| 总股本           | int64   | -  |
| 预案公告日         | object  | -  |
| 股权登记日         | object  | -  |
| 除权除息日         | object  | -  |
| 方案进度          | object  | -  |
| 最新公告日期        | object  | -  |

接口示例

```python
import akshare as ak

stock_fhps_detail_em_df = ak.stock_fhps_detail_em(symbol="300073")
print(stock_fhps_detail_em_df)
```

数据示例

```
   报告期      业绩披露日期  送转股份-送转总比例  ... 除权除息日  方案进度  最新公告日期
0  2010-12-31  2011-03-31        10.0  ...  2011-05-20  实施分配  2011-05-16
1  2012-12-31  2013-04-20         NaN  ...  2013-07-09  实施分配  2013-07-02
2  2013-12-31  2014-04-15         NaN  ...  2014-06-26  实施分配  2014-06-18
3  2016-12-31  2017-03-14        10.0  ...  2017-04-26  实施分配  2017-04-18
4  2017-12-31  2018-04-03         NaN  ...  2018-05-29  实施分配  2018-05-23
5  2018-12-31  2019-04-23         NaN  ...  2019-06-21  实施分配  2019-06-15
6  2020-12-31  2021-03-23         NaN  ...  2021-05-28  实施分配  2021-05-21
7  2021-12-31  2022-03-31         NaN  ...  2022-05-26  实施分配  2022-05-19
8  2022-12-31  2023-04-10         NaN  ...  2023-06-29  实施分配  2023-06-19
9  2023-12-31  2024-03-30         NaN  ...  2024-06-07  实施分配  2024-05-31
[10 rows x 19 columns]
```

#### 分红情况-同花顺

接口: stock_fhps_detail_ths

目标地址: https://basic.10jqka.com.cn/new/603444/bonus.html

描述: 同花顺-分红情况

限量: 单次获取指定 symbol 的分红情况数据

输入参数

| 名称     | 类型  | 描述                           |
|--------|-----|------------------------------|
| symbol | str | symbol="603444"; 兼容 A 股和 B 股 |

输出参数

| 名称         | 类型     | 描述                |
|------------|--------|-------------------|
| 报告期        | object | -                 |
| 董事会日期      | object | -                 |
| 股东大会预案公告日期 | object | -                 |
| 实施公告日      | object | -                 |
| 分红方案说明     | object | -                 |
| A股股权登记日    | object | 注意: 根据 A 股和 B 股变化 |
| A股除权除息日    | object | 注意: 根据 A 股和 B 股变化 |
| 分红总额       | object | -                 |
| 方案进度       | object | -                 |
| 股利支付率      | object | -                 |
| 税前分红率      | object | -                 |

接口示例

```python
import akshare as ak

stock_fhps_detail_ths_df = ak.stock_fhps_detail_ths(symbol="603444")
print(stock_fhps_detail_ths_df)
```

数据示例

```
        报告期   董事会日期  股东大会预案公告日期  实施公告日  ... 分红总额 方案进度 股利支付率 税前分红率
0    2016年报  2017-03-28  2017-04-18  2017-04-28  ...   2.94亿    实施方案  37.37%  1.52%
1    2017中报  2017-08-15         NaT         NaT  ...      --   董事会预案      --     --
2    2017年报  2018-04-03  2018-04-25  2018-05-07  ...   1.87亿    实施方案  30.48%  1.66%
3    2018中报  2018-08-14         NaT         NaT  ...      --   董事会预案      --     --
4    2018年报  2019-04-10  2019-05-01  2019-05-16  ...   7.19亿    实施方案  98.81%  4.47%
5    2019中报  2019-08-13         NaT         NaT  ...      --   董事会预案      --     --
6    2019年报  2020-04-09  2020-04-30  2020-05-14  ...   3.59亿    实施方案  44.33%  1.17%
7    2020中报  2020-08-18         NaT         NaT  ...      --   董事会预案      --     --
8    2020年报  2021-03-31  2021-05-13  2021-05-31  ...   8.62亿    实施方案   82.3%  2.08%
9    2021中报  2021-08-13         NaT         NaT  ...      --   董事会预案      --     --
10   2021年报  2022-04-08  2022-04-30  2022-05-13  ...  11.50亿    实施方案  78.32%  4.78%
11   2022中报  2022-08-16         NaT         NaT  ...      --   董事会预案      --     --
12  2022三季报  2022-10-27  2022-11-16  2022-12-22  ...  10.06亿    实施方案  99.43%  4.41%
13   2022年报  2023-03-31  2023-04-22  2023-05-05  ...   2.16亿    实施方案  14.76%  0.61%
14   2023中报  2023-08-16  2023-09-09  2023-09-21  ...   5.04亿    实施方案  74.47%  1.88%
15   2023年报  2024-03-29  2024-04-19         NaT  ...      --  股东大会预案      --     --
[16 rows x 11 columns]
```

#### 分红配送详情-港股-同花顺

接口: stock_hk_fhpx_detail_ths

目标地址: https://stockpage.10jqka.com.cn/HK0700/bonus/

描述: 同花顺-港股-分红派息

限量: 单次获取指定股票的分红派息数据

输入参数

| 名称     | 类型  | 描述                  |
|--------|-----|---------------------|
| symbol | str | symbol="0700"; 港股代码 |

输出参数

| 名称         | 类型     | 描述 |
|------------|--------|----|
| 公告日期       | object | -  |
| 方案         | object | -  |
| 除净日        | object | -  |
| 派息日        | object | -  |
| 过户日期起止日-起始 | object | -  |
| 过户日期起止日-截止 | object | -  |
| 类型         | object | -  |
| 进度         | object | -  |
| 以股代息       | object | -  |

接口示例

```python
import akshare as ak

stock_hk_fhpx_detail_ths_df = ak.stock_hk_fhpx_detail_ths(symbol="0700")
print(stock_hk_fhpx_detail_ths_df)
```

数据示例

```
        公告日期        方案         除净日  ...       类型    进度 以股代息
0   2004-08-19       不分红         NaT  ...       中报    预案    否
1   2004-11-18       不分红         NaT  ...      三季报    预案    否
2   2005-03-17  每股0.07港元  2005-04-19  ...       年报  实施完成    否
3   2005-05-18       不分红         NaT  ...      一季报    预案    否
4   2005-08-24       不分红         NaT  ...       中报    预案    否
..         ...       ...         ...  ...      ...   ...  ...
78  2022-11-16       不分红         NaT  ...      三季报    预案    否
79  2023-03-22   每股2.4港元  2023-05-19  ...       年报  实施完成    否
80  2023-05-17       不分红         NaT  ...      一季报    预案    否
81  2023-08-16       不分红         NaT  ...       中报    预案    否
82  2023-11-15       不分红         NaT  ...  三季报(累计)    预案    否
[83 rows x 9 columns]
```

### 资金流向

#### 同花顺

##### 个股资金流

接口: stock_fund_flow_individual

目标地址: https://data.10jqka.com.cn/funds/ggzjl/#refCountId=data_55f13c2c_254

描述: 同花顺-数据中心-资金流向-个股资金流

限量: 单次获取指定 symbol 的概念资金流数据

输入参数

| 名称     | 类型  | 描述                                                              |
|--------|-----|-----------------------------------------------------------------|
| symbol | str | symbol="即时"; choice of {“即时”, "3日排行", "5日排行", "10日排行", "20日排行"} |

输出参数-即时

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int32   | -       |
| 股票代码 | int64   | -       |
| 股票简称 | object  | -       |
| 最新价  | float64 | -       |
| 涨跌幅  | object  | 注意单位: % |
| 换手率  | object  | -       |
| 流入资金 | object  | 注意单位: 元 |
| 流出资金 | object  | 注意单位: 元 |
| 净额   | object  | 注意单位: 元 |
| 成交额  | object  | 注意单位: 元 |

接口示例-即时

```python
import akshare as ak

stock_fund_flow_individual_df = ak.stock_fund_flow_individual(symbol="即时")
print(stock_fund_flow_individual_df)
```

数据示例-即时

```
      序号  股票代码 股票简称   最新价  ...     流入资金    流出资金     净额       成交额
0        1  300256  星星科技   3.40  ...     6.49亿     5.01亿      1.48亿    11.50亿
1        2  300269  联建光电   4.00  ...     3.53亿     2.77亿   7588.54万     6.30亿
2        3  300299  富春股份   5.26  ...     3.48亿     3.53亿   -571.77万     7.01亿
3        4  300050  世纪鼎利   4.07  ...     2.91亿     3.63亿  -7186.49万     6.53亿
4        5  300323  华灿光电   5.09  ...  5916.58万     1.73亿     -1.13亿     2.32亿
...    ...     ...   ...    ...  ...       ...       ...        ...       ...
5084  5085  603398  沐邦高科  10.85  ...     1.18亿     1.17亿    185.74万     2.35亿
5085  5086  603310  巍华新材  21.56  ...     3.13亿     4.02亿  -8883.27万     7.15亿
5086  5087     509  华塑控股   2.92  ...     2.28亿     1.67亿   6070.40万     3.95亿
5087  5088  300599  雄塑科技   6.19  ...  8622.01万     1.47亿  -6126.79万     2.34亿
5088  5089  688426  康为世纪  17.86  ...  3342.07万  5921.77万  -2579.71万  9263.83万
[5089 rows x 10 columns]
```

输出参数-3日、5日、10日和20日

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int32   | -       |
| 股票代码   | int64   | -       |
| 股票简称   | object  | -       |
| 最新价    | float64 | -       |
| 阶段涨跌幅  | object  | 注意单位: % |
| 连续换手率  | object  | 注意单位: % |
| 资金流入净额 | float64 | 注意单位: 元 |

接口示例-3日、5日、10日和20日

```python
import akshare as ak

stock_fund_flow_individual_df = ak.stock_fund_flow_individual(symbol="3日排行")
print(stock_fund_flow_individual_df)
```

数据示例-3日、5日、10日和20日

```
     序号    股票代码 股票简称   最新价  阶段涨跌幅  连续换手率 资金流入净额
0        1  300622  博士眼镜  31.51   72.85%  103.83%     -1.30亿
1        2  300269  联建光电   4.00   53.26%   75.76%    913.14万
2        3  301288  清研环境  21.39   48.75%  115.83%     -1.43亿
3        4  300157  新锦动力   2.80   46.60%   36.12%   2178.04万
4        5  300211  亿通科技   7.39   46.34%   19.53%  -6246.22万
...    ...     ...   ...    ...      ...      ...        ...
5084  5085    2890  弘宇股份   8.51  -16.07%   19.54%  -6274.33万
5085  5086  603398  沐邦高科  10.85  -17.55%   11.35%  -2952.80万
5086  5087  603139  康惠制药  12.26  -17.72%   17.01%     93.55万
5087  5088    2871  伟隆股份   8.01  -20.69%   25.36%  -1586.36万
5088  5089  688426  康为世纪  17.86  -28.84%   25.86%    465.97万
[5089 rows x 7 columns]
```

##### 概念资金流

接口: stock_fund_flow_concept

目标地址: https://data.10jqka.com.cn/funds/gnzjl/#refCountId=data_55f13c2c_254

描述: 同花顺-数据中心-资金流向-概念资金流

限量: 单次获取指定 symbol 的概念资金流数据

输入参数

| 名称     | 类型  | 描述                                                              |
|--------|-----|-----------------------------------------------------------------|
| symbol | str | symbol="即时"; choice of {“即时”, "3日排行", "5日排行", "10日排行", "20日排行"} |

输出参数-即时

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int32   | -       |
| 行业      | object  | -       |
| 行业指数    | float64 | -       |
| 行业-涨跌幅  | float64 | 注意单位: % |
| 流入资金    | float64 | 注意单位: 亿 |
| 流出资金    | float64 | 注意单位: 亿 |
| 净额      | float64 | 注意单位: 亿 |
| 公司家数    | float64 | -       |
| 领涨股     | object  | -       |
| 领涨股-涨跌幅 | float64 | 注意单位: % |
| 当前价     | float64 | 注意单位: 元 |

接口示例-即时

```python
import akshare as ak

stock_fund_flow_concept_df = ak.stock_fund_flow_concept(symbol="即时")
print(stock_fund_flow_concept_df)
```

数据示例-即时

```
     序号      行业      行业指数  行业-涨跌幅   流入资金  ...  净额  公司家数 领涨股 领涨股-涨跌幅 当前价
0      1  华为海思概念股  1750.270    6.17  45.26  ...  3.54    36  力源信息   20.04   6.11
1      2     猴痘概念   718.597    4.85  33.86  ...  6.82    61  透景生命   20.02  14.15
2      3     AI眼镜  1168.340    3.37  67.39  ... -8.94    51  联合光电   20.00  18.90
3      4     基因测序  1411.750    2.37  22.31  ...  1.41    51  透景生命   20.02  14.15
4      5  幽门螺杆菌概念   792.010    2.37  28.64  ...  3.61    66  海辰药业   19.98  23.54
..   ...      ...       ...     ...    ...  ...   ...   ...   ...     ...    ...
395  396   PVDF概念   649.872   -2.00   2.20  ... -2.07    13  中创环保    0.74   8.19
396  397      可燃冰   989.282   -2.11   4.64  ... -0.96    12  中国石化   -0.15   6.68
397  398      转基因  1139.540   -2.37   1.43  ... -1.75    13  康农种业   -0.33   9.03
398  399       猪肉  2367.260   -2.53   9.65  ... -8.29    32  金字火腿    0.23   4.28
399  400       养鸡  1048.980   -2.55   4.44  ... -4.79    22  春雪食品   -0.39   7.64
[400 rows x 11 columns]
```

输出参数-3日、5日、10日和20日

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int32   | -       |
| 行业    | object  | -       |
| 公司家数  | int64   | -       |
| 行业指数  | float64 | -       |
| 阶段涨跌幅 | object  | 注意单位: % |
| 流入资金  | float64 | 注意单位: 亿 |
| 流出资金  | float64 | 注意单位: 亿 |
| 净额    | float64 | 注意单位: 亿 |

接口示例-3日、5日、10日和20日

```python
import akshare as ak

stock_fund_flow_concept_df = ak.stock_fund_flow_concept(symbol="3日排行")
print(stock_fund_flow_concept_df)
```

数据示例-3日、5日、10日和20日

```
     序号     行业  公司家数  行业指数  阶段涨跌幅  流入资金  流出资金 净额
0      1   华为海思概念股    36  1750.27  11.45%  13.06  13.22 -0.16
1      2  MR(混合现实)    59   797.57   7.68%  23.07  21.81  1.26
2      3      智能音箱    49  1337.42   7.52%  26.52  26.17  0.35
3      4       云游戏    25   780.74   7.45%   6.33   7.15 -0.82
4      5      智能穿戴    94  2109.07   6.94%  45.96  43.13  2.82
..   ...       ...   ...      ...     ...    ...    ...   ...
394  395       禽流感    30  1800.47  -2.64%  19.05  26.21 -7.16
395  396      毛发医疗    18   800.82  -2.78%   2.68   3.93 -1.25
396  397        养鸡    22  1048.98  -3.20%   7.02   5.81  1.21
397  398    PVDF概念    13   649.87  -3.40%   2.83   3.05 -0.23
398  399        猪肉    32  2367.26  -3.63%  13.90  13.43  0.47
[399 rows x 8 columns]
```

##### 行业资金流

接口: stock_fund_flow_industry

目标地址: http://data.10jqka.com.cn/funds/hyzjl/#refCountId=data_55f13c2c_254

描述: 同花顺-数据中心-资金流向-行业资金流

限量: 单次获取指定 symbol 的行业资金流数据

输入参数

| 名称     | 类型  | 描述                                                              |
|--------|-----|-----------------------------------------------------------------|
| symbol | str | symbol="即时"; choice of {“即时”, "3日排行", "5日排行", "10日排行", "20日排行"} |

输出参数-即时

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int32   | -       |
| 行业      | object  | -       |
| 行业指数    | float64 | -       |
| 行业-涨跌幅  | object  | 注意单位: % |
| 流入资金    | float64 | 注意单位: 亿 |
| 流出资金    | float64 | 注意单位: 亿 |
| 净额      | float64 | 注意单位: 亿 |
| 公司家数    | float64 | -       |
| 领涨股     | object  | -       |
| 领涨股-涨跌幅 | object  | 注意单位: % |
| 当前价     | float64 | -       |

接口示例-即时

```python
import akshare as ak

stock_fund_flow_industry_df = ak.stock_fund_flow_industry(symbol="即时")
print(stock_fund_flow_industry_df)
```

数据示例-即时

```
   序号  行业     行业指数  行业-涨跌幅 流入资金 流出资金 净额 公司家数 领涨股 领涨股-涨跌幅  当前价
0    1  黑色家电  2572.370    3.49   7.13   7.41  -0.28     9  辰奕智能    19.99  42.50
1    2  其他电子  7585.180    2.38  27.76  26.92   0.85    31  力源信息    20.04   6.11
2    3    元件  7383.480    2.25  37.39  31.42   5.97    56  则成电子    24.20  28.18
3    4  消费电子  5983.640    2.03  86.85  87.11  -0.26    95  凯旺科技    20.02  28.36
4    5  医疗器械  8014.560    1.64  27.10  23.25   3.84   126  透景生命    20.02  14.15
..  ..   ...       ...     ...    ...    ...    ...   ...   ...      ...    ...
85  86  建筑材料  3105.260   -1.78  14.99  23.36  -8.36    74  扬子新材     8.83   3.08
86  87  建筑装饰  2437.520   -1.85  27.42  37.43 -10.01   151  志特新材     5.51   8.23
87  88   养殖业  1955.710   -2.17   7.22  13.98  -6.76    40  海利生物     0.73   6.93
88  89  影视院线   972.325   -2.29   6.99  10.90  -3.91    21  捷成股份     0.27   3.76
89  90   贵金属  2037.310   -2.42  12.46  19.62  -7.16    12  湖南黄金     1.40  15.89
[90 rows x 11 columns]
```

输出参数-3日、5日、10日和20日

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int32   | -       |
| 行业    | object  | -       |
| 公司家数  | int64   | -       |
| 行业指数  | float64 | -       |
| 阶段涨跌幅 | object  | 注意单位: % |
| 流入资金  | float64 | 注意单位: 亿 |
| 流出资金  | float64 | 注意单位: 亿 |
| 净额    | float64 | 注意单位: 亿 |

接口示例-3日、5日、10日和20日

```python
import akshare as ak

stock_fund_flow_industry_df = ak.stock_fund_flow_industry(symbol="3日排行")
print(stock_fund_flow_industry_df)
```

数据示例-3日、5日、10日和20日

```
   序号     行业  公司家数  行业指数  阶段涨跌幅 流入资金 流出资金  净额
0    1    消费电子    95  5983.64   8.39%  42.71  40.22   2.49
1    2      游戏    25  5613.77   7.46%   8.01   9.16  -1.15
2    3    其他电子    31  7585.18   6.50%  12.07  10.24   1.83
3    4    黑色家电     9  2572.37   6.20%   2.90   2.68   0.22
4    5   光学光电子   107  2677.80   5.71%  26.55  25.45   1.09
..  ..     ...   ...      ...     ...    ...    ...    ...
85  86      中药    72  2977.40  -2.65%  29.00  40.11 -11.11
86  87     养殖业    40  1955.71  -2.71%  10.41  10.26   0.16
87  88      白酒    20  2860.75  -2.74%  24.41  30.48  -6.07
88  89  食品加工制造    65  3414.99  -3.16%   6.05   7.84  -1.79
89  90     贵金属    12  2037.31  -3.32%  12.30  12.24   0.06
[90 rows x 8 columns]
```

##### 大单追踪

接口: stock_fund_flow_big_deal

目标地址: https://data.10jqka.com.cn/funds/ddzz

描述: 同花顺-数据中心-资金流向-大单追踪

限量: 单次获取当前时点的所有大单追踪数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数-即时

| 名称   | 类型      | 描述       |
|------|---------|----------|
| 成交时间 | object  | -        |
| 股票代码 | int64   | -        |
| 股票简称 | object  | -        |
| 成交价格 | float64 | -        |
| 成交量  | int64   | 注意单位: 股  |
| 成交额  | float64 | 注意单位: 万元 |
| 大单性质 | object  | -        |
| 涨跌幅  | object  | -        |
| 涨跌额  | object  | -        |

接口示例-即时

```python
import akshare as ak

stock_fund_flow_big_deal_df = ak.stock_fund_flow_big_deal()
print(stock_fund_flow_big_deal_df)
```

数据示例

```
       成交时间   股票代码  股票简称   成交价格  ...    成交额  大单性质     涨跌幅   涨跌额
0     2024-08-19 15:00:01  601668  中国建筑    5.67  ...  111.98    买盘   0.53%  0.03
1     2024-08-19 15:00:01  688347  华虹公司   31.10  ...   82.61    买盘   0.71%  0.22
2     2024-08-19 15:00:01  688223  晶科能源    6.98  ...   69.80    买盘  -0.57% -0.04
3     2024-08-19 15:00:01  688223  晶科能源    6.98  ...   65.75    卖盘  -0.57% -0.04
4     2024-08-19 15:00:01  688169  石头科技  198.91  ...  114.89    买盘  -4.83% -9.60
...                   ...     ...   ...     ...  ...     ...   ...     ...   ...
4995  2024-08-19 14:30:52  600487  亨通光电   13.46  ...  207.36    买盘  -1.68% -0.23
4996  2024-08-19 14:30:52  688183  生益电子   19.90  ...   97.58    买盘  -1.24% -0.25
4997  2024-08-19 14:30:50    2500  山西证券    4.95  ...  314.41    买盘   1.23%  0.06
4998  2024-08-19 14:30:49  300152   新动力    1.92  ...   38.60    买盘  -0.52% -0.01
4999  2024-08-19 14:30:48  301101  明月镜片   25.26  ...   52.35    买盘  -7.34% -1.85
[5000 rows x 9 columns]
```

#### 东方财富

##### 个股资金流

接口: stock_individual_fund_flow

目标地址: https://data.eastmoney.com/zjlx/detail.html

描述: 东方财富网-数据中心-个股资金流向

限量: 单次获取指定市场和股票的近 100 个交易日的资金流数据

输入参数

| 名称     | 类型  | 描述                                                 |
|--------|-----|----------------------------------------------------|
| stock  | str | stock="000425"; 股票代码                               |
| market | str | market="sh"; 上海证券交易所: sh, 深证证券交易所: sz, 北京证券交易所: bj |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 日期         | object  | -       |
| 收盘价        | float64 | -       |
| 涨跌幅        | float64 | 注意单位: % |
| 主力净流入-净额   | float64 | -       |
| 主力净流入-净占比  | float64 | 注意单位: % |
| 超大单净流入-净额  | float64 | -       |
| 超大单净流入-净占比 | float64 | 注意单位: % |
| 大单净流入-净额   | float64 | -       |
| 大单净流入-净占比  | float64 | 注意单位: % |
| 中单净流入-净额   | float64 | -       |
| 中单净流入-净占比  | float64 | 注意单位: % |
| 小单净流入-净额   | float64 | -       |
| 小单净流入-净占比  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_individual_fund_flow_df = ak.stock_individual_fund_flow(stock="600094", market="sh")
print(stock_individual_fund_flow_df)
```

数据示例

```
    日期   收盘价   涨跌幅  ...  中单净流入-净占比    小单净流入-净额  小单净流入-净占比
0    2023-03-31  3.13 -0.32  ...       4.02    240002.0       0.95
1    2023-04-03  3.18  1.60  ...      -0.93    910076.0       2.50
2    2023-04-04  3.13 -1.57  ...      10.87    929360.0       2.25
3    2023-04-06  3.11 -0.64  ...      -4.87   5618110.0      17.43
4    2023-04-07  3.15  1.29  ...      -7.44   -839635.0      -3.00
..          ...   ...   ...  ...        ...         ...        ...
96   2023-08-22  3.50 -0.57  ...       2.99  10665906.0      10.36
97   2023-08-23  3.46 -1.14  ...       1.45   6542085.0       8.41
98   2023-08-24  3.45 -0.29  ...       5.15  -4355064.0      -5.41
99   2023-08-25  3.43 -0.58  ...      -1.51 -18759517.0     -22.13
100  2023-08-28  3.51  2.33  ...      14.49  10409307.0       5.47
[101 rows x 13 columns]
```

##### 个股资金流排名

接口: stock_individual_fund_flow_rank

目标地址: http://data.eastmoney.com/zjlx/detail.html

描述: 东方财富网-数据中心-资金流向-排名

限量: 单次获取指定类型的个股资金流排名数据

输入参数

| 名称        | 类型  | 描述                                               |
|-----------|-----|--------------------------------------------------|
| indicator | str | indicator="今日"; choice {"今日", "3日", "5日", "10日"} |

输出参数-今日

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | -       |
| 今日涨跌幅        | float64 | 注意单位: % |
| 今日主力净流入-净额   | float64 | -       |
| 今日主力净流入-净占比  | float64 | 注意单位: % |
| 今日超大单净流入-净额  | float64 | -       |
| 今日超大单净流入-净占比 | float64 | 注意单位: % |
| 今日大单净流入-净额   | float64 | -       |
| 今日大单净流入-净占比  | float64 | 注意单位: % |
| 今日中单净流入-净额   | float64 | -       |
| 今日中单净流入-净占比  | float64 | 注意单位: % |
| 今日小单净流入-净额   | float64 | -       |
| 今日小单净流入-净占比  | float64 | 注意单位: % |

接口示例-今日

```python
import akshare as ak

stock_individual_fund_flow_rank_df = ak.stock_individual_fund_flow_rank(indicator="今日")
print(stock_individual_fund_flow_rank_df)
```

数据示例-今日

```
    序号 代码    名称  ...  今日中单净流入-净占比   今日小单净流入-净额  今日小单净流入-净占比
0    1  600196  复星医药  ...       -11.80 -554888208.0        -8.76
1    2  601012  隆基股份  ...        -5.09 -284138560.0        -3.55
2    3  000963  华东医药  ...        -6.39 -291504016.0        -4.76
3    4  300986   N志特  ...         2.28 -416998433.0       -47.90
4    5  300059  东方财富  ...        -4.01 -168403920.0        -3.02
5    6  002460  赣锋锂业  ...        -4.44  -83612896.0        -1.32
6    7  603799  华友钴业  ...        -4.30 -162641040.0        -3.89
7    8  000908  景峰医药  ...        -8.53 -183772494.0       -12.59
8    9  000519  中兵红箭  ...       -10.68 -149982694.0       -10.75
9   10  300274  阳光电源  ...        -2.34 -158902688.0        -3.01
10  11  600276  恒瑞医药  ...        -3.34 -135060768.0        -3.13
11  12  002714  牧原股份  ...        -4.59 -174012720.0        -8.76
12  13  002241  歌尔股份  ...        -4.76 -148783568.0        -6.22
13  14  601127  小康股份  ...        -3.51 -130833968.0        -3.57
14  15  002385   大北农  ...       -10.22 -128105833.0       -13.10
15  16  003040   楚天龙  ...        -7.24 -134830833.0       -10.48
16  17  600223  鲁商发展  ...        -5.69  -97029040.0        -4.48
17  18  000650  仁和药业  ...       -12.98  -79890962.0        -7.68
18  19  300601  康泰生物  ...        -4.51  -82006064.0        -2.83
19  20  000928  中钢国际  ...        -8.78  -75598962.0        -4.99
20  21  002176  江特电机  ...        -9.13 -102992833.0        -9.05
```

输出参数-3日

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | -       |
| 3日涨跌幅        | float64 | 注意单位: % |
| 3日主力净流入-净额   | float64 | -       |
| 3日主力净流入-净占比  | float64 | 注意单位: % |
| 3日超大单净流入-净额  | float64 | -       |
| 3日超大单净流入-净占比 | float64 | 注意单位: % |
| 3日大单净流入-净额   | float64 | -       |
| 3日大单净流入-净占比  | float64 | 注意单位: % |
| 3日中单净流入-净额   | float64 | -       |
| 3日中单净流入-净占比  | float64 | 注意单位: % |
| 3日小单净流入-净额   | float64 | -       |
| 3日小单净流入-净占比  | float64 | 注意单位: % |

接口示例-3日

```python
import akshare as ak

stock_individual_fund_flow_rank_df = ak.stock_individual_fund_flow_rank(indicator="3日")
print(stock_individual_fund_flow_rank_df)
```

数据示例-3日

```
    序号      代码    名称  ...  3日中单净流入-净占比    3日小单净流入-净额  3日小单净流入-净占比
0    1  600196  复星医药  ...        -9.33 -1.157038e+09        -6.84
1    2  300059  东方财富  ...        -3.29 -7.832414e+08        -3.89
2    3  002466  天齐锂业  ...        -3.91 -2.935912e+08        -1.63
3    4  601012  隆基股份  ...        -2.84 -3.713241e+08        -2.14
4    5  002304  洋河股份  ...        -5.98 -3.069602e+08        -4.01
5    6  000963  华东医药  ...        -3.58 -1.615584e+08        -1.11
6    7  300601  康泰生物  ...        -3.55 -3.172210e+08        -4.44
7    8  002460  赣锋锂业  ...        -2.17 -1.881166e+08        -1.24
8    9  603799  华友钴业  ...        -3.09 -2.247308e+08        -2.52
9   10  300760  迈瑞医疗  ...        -7.66 -1.840590e+06        -0.03
10  11  603259  药明康德  ...        -3.41 -1.396783e+08        -1.37
11  12  300750  宁德时代  ...        -0.95 -2.482142e+08        -0.98
12  13  688383   新益昌  ...         1.17 -4.520943e+08       -21.70
13  14  300986   N志特  ...         2.28 -4.169984e+08       -47.90
14  15  000001  平安银行  ...        -4.37 -1.960774e+08        -4.76
15  16  600216  浙江医药  ...        -6.40 -1.923193e+08        -8.57
16  17  002714  牧原股份  ...        -1.31 -2.559579e+08        -4.58
17  18  600030  中信证券  ...        -2.19 -1.662703e+08        -2.24
18  19  000650  仁和药业  ...       -10.18 -1.381347e+08        -7.78
19  20  000718  苏宁环球  ...        -4.32 -1.313346e+08        -3.15
20  21  002821   凯莱英  ...        -5.28 -1.838423e+08        -8.10
```

输出参数-5日

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | -       |
| 5日涨跌幅        | float64 | 注意单位: % |
| 5日主力净流入-净额   | float64 | -       |
| 5日主力净流入-净占比  | float64 | 注意单位: % |
| 5日超大单净流入-净额  | float64 | -       |
| 5日超大单净流入-净占比 | float64 | 注意单位: % |
| 5日大单净流入-净额   | float64 | -       |
| 5日大单净流入-净占比  | float64 | 注意单位: % |
| 5日中单净流入-净额   | float64 | -       |
| 5日中单净流入-净占比  | float64 | 注意单位: % |
| 5日小单净流入-净额   | float64 | -       |
| 5日小单净流入-净占比  | float64 | 注意单位: % |

接口示例-5日

```python
import akshare as ak

stock_individual_fund_flow_rank_df = ak.stock_individual_fund_flow_rank(indicator="5日")
print(stock_individual_fund_flow_rank_df)
```

数据示例-5日

```
    序号      代码    名称  ...  5日中单净流入-净占比    5日小单净流入-净额  5日小单净流入-净占比
0    1  600196  复星医药  ...        -7.51 -1.193192e+09        -5.48
1    2  300979   C华利  ...        -7.16 -1.151424e+09        -9.62
2    3  002304  洋河股份  ...        -5.52 -2.617723e+08        -2.48
3    4  300760  迈瑞医疗  ...        -7.26 -3.271416e+06        -0.03
4    5  300601  康泰生物  ...        -3.09 -4.497565e+08        -4.51
5    6  300981   C中红  ...        -6.22 -1.023511e+08        -1.11
6    7  603259  药明康德  ...        -3.23 -1.252767e+08        -0.76
7    8  002027  分众传媒  ...        -3.80 -3.990743e+08        -5.94
8    9  000928  中钢国际  ...        -5.31 -2.201884e+08        -3.79
9   10  601012  隆基股份  ...        -1.66 -6.462818e+07        -0.24
10  11  002142  宁波银行  ...        -5.03 -2.290891e+08        -4.35
11  12  300059  东方财富  ...        -1.85  2.324636e+08         0.62
12  13  000963  华东医药  ...        -2.00 -8.412000e+06        -0.04
13  14  688383   新益昌  ...         1.17 -4.520943e+08       -21.70
14  15  002241  歌尔股份  ...        -3.00  3.025771e+07         0.20
15  16  000661  长春高新  ...        -4.94 -2.646385e+06        -0.03
16  17  002821   凯莱英  ...        -5.35 -1.858551e+08        -4.39
17  18  300986   N志特  ...         2.28 -4.169984e+08       -47.90
18  19  300982   C苏文  ...         1.20 -4.231374e+08       -14.36
19  20  002466  天齐锂业  ...        -2.37  1.762265e+08         0.75
20  21  601919  中远海控  ...        -1.10 -1.222831e+08        -0.54
```

输出参数-10日

| 名称            | 类型      | 描述      |
|---------------|---------|---------|
| 序号            | int32   | -       |
| 代码            | object  | -       |
| 名称            | object  | -       |
| 最新价           | float64 | -       |
| 10日涨跌幅        | float64 | 注意单位: % |
| 10日主力净流入-净额   | float64 | -       |
| 10日主力净流入-净占比  | float64 | 注意单位: % |
| 10日超大单净流入-净额  | float64 | -       |
| 10日超大单净流入-净占比 | float64 | 注意单位: % |
| 10日大单净流入-净额   | float64 | -       |
| 10日大单净流入-净占比  | float64 | 注意单位: % |
| 10日中单净流入-净额   | float64 | -       |
| 10日中单净流入-净占比  | float64 | 注意单位: % |
| 10日小单净流入-净额   | float64 | -       |
| 10日小单净流入-净占比  | float64 | 注意单位: % |

接口示例-10日

```python
import akshare as ak

stock_individual_fund_flow_rank_df = ak.stock_individual_fund_flow_rank(indicator="10日")
print(stock_individual_fund_flow_rank_df)
```

数据示例-10日

```
    序号      代码     名称  ...  10日中单净流入-净占比   10日小单净流入-净额  10日小单净流入-净占比
0    1  600196   复星医药  ...         -6.51 -1.620072e+09         -4.96
1    2  002241   歌尔股份  ...         -4.37 -1.504247e+09         -3.82
2    3  300059   东方财富  ...         -1.28 -1.498377e+09         -2.20
3    4  300979    C华利  ...         -7.16 -1.151424e+09         -9.62
4    5  002304   洋河股份  ...         -4.77 -5.633588e+08         -2.79
5    6  002027   分众传媒  ...         -4.08 -9.081865e+08         -6.91
6    7  300015   爱尔眼科  ...         -3.03 -5.969998e+08         -2.28
7    8  300760   迈瑞医疗  ...         -6.79 -4.882262e+06         -0.02
8    9  603259   药明康德  ...         -2.87 -3.367873e+08         -1.09
9   10  601012   隆基股份  ...         -1.47 -2.309467e+08         -0.39
10  11  002415   海康威视  ...         -3.18 -4.275869e+08         -2.12
11  12  002142   宁波银行  ...         -5.86 -3.757830e+08         -3.51
12  13  300601   康泰生物  ...         -3.19 -4.968827e+08         -3.39
13  14  603799   华友钴业  ...         -2.14 -3.544022e+08         -1.37
14  15  000661   长春高新  ...         -5.88 -4.703516e+06         -0.03
15  16  000858  五 粮 液  ...         -2.30  3.079020e+08          0.62
16  17  600570   恒生电子  ...         -4.15 -3.146686e+08         -3.05
17  18  300981    C中红  ...         -6.22 -1.023511e+08         -1.11
18  19  002460   赣锋锂业  ...         -2.59  3.715398e+08          0.93
19  20  300142   沃森生物  ...         -1.47 -9.540762e+07         -0.25
20  21  300347   泰格医药  ...         -4.72 -1.195595e+08         -1.08
```

##### 大盘资金流

接口: stock_market_fund_flow

目标地址: https://data.eastmoney.com/zjlx/dpzjlx.html

描述: 东方财富网-数据中心-资金流向-大盘

限量: 单次获取大盘资金流向历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 日期         | object  | -       |
| 上证-收盘价     | float64 | -       |
| 上证-涨跌幅     | float64 | 注意单位: % |
| 深证-收盘价     | float64 | -       |
| 深证-涨跌幅     | float64 | 注意单位: % |
| 主力净流入-净额   | float64 | -       |
| 主力净流入-净占比  | float64 | 注意单位: % |
| 超大单净流入-净额  | float64 | -       |
| 超大单净流入-净占比 | float64 | 注意单位: % |
| 大单净流入-净额   | float64 | -       |
| 大单净流入-净占比  | float64 | 注意单位: % |
| 中单净流入-净额   | float64 | -       |
| 中单净流入-净占比  | float64 | 注意单位: % |
| 小单净流入-净额   | float64 | -       |
| 小单净流入-净占比  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_market_fund_flow_df = ak.stock_market_fund_flow()
print(stock_market_fund_flow_df)
```

数据示例

```
       日期      上证-收盘价 上证-涨跌幅  ...  中单净流入-净占比  小单净流入-净额  小单净流入-净占比
0    2024-02-23  3004.88    0.55  ...      -0.17  5.147947e+09       0.56
1    2024-02-26  2977.02   -0.93  ...       0.08  1.855043e+10       1.88
2    2024-02-27  3015.48    1.29  ...      -1.05 -3.641864e+09      -0.37
3    2024-02-28  2957.85   -1.91  ...       0.12  7.204130e+10       5.31
4    2024-02-29  3015.17    1.94  ...      -1.01 -9.629499e+09      -0.91
..          ...      ...     ...  ...        ...           ...        ...
116  2024-08-13  2867.95    0.34  ...      -0.39  9.309733e+09       1.95
117  2024-08-14  2850.65   -0.60  ...       0.02  8.340414e+09       1.75
118  2024-08-15  2877.36    0.94  ...      -0.97  7.407305e+09       1.25
119  2024-08-16  2879.43    0.07  ...       0.07  1.201517e+10       2.03
120  2024-08-19  2893.67    0.49  ...      -0.25  1.239419e+10       2.17
[121 rows x 15 columns]
```

##### 板块资金流排名

接口: stock_sector_fund_flow_rank

目标地址: https://data.eastmoney.com/bkzj/hy.html

描述: 东方财富网-数据中心-资金流向-板块资金流-排名

限量: 单次获取指定板块的指定期限的资金流排名数据

输入参数

| 名称          | 类型  | 描述                                                         |
|-------------|-----|------------------------------------------------------------|
| indicator   | str | indicator="今日"; choice of {"今日", "5日", "10日"}              |
| sector_type | str | sector_type="行业资金流"; choice of {"行业资金流", "概念资金流", "地域资金流"} |

输出参数-行业资金流-今日

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 名称         | object  | -       |
| 今日涨跌幅      | float64 | 注意单位: % |
| 主力净流入-净额   | float64 | -       |
| 主力净流入-净占比  | float64 | 注意单位: % |
| 超大单净流入-净额  | float64 | -       |
| 超大单净流入-净占比 | float64 | 注意单位: % |
| 大单净流入-净额   | float64 | -       |
| 大单净流入-净占比  | float64 | 注意单位: % |
| 中单净流入-净额   | float64 | -       |
| 中单净流入-净占比  | float64 | 注意单位: % |
| 小单净流入-净额   | float64 | -       |
| 小单净流入-净占比  | float64 | 注意单位: % |
| 主力净流入最大股   | object  | -       |

接口示例-行业资金流-今日

```python
import akshare as ak

stock_sector_fund_flow_rank_df = ak.stock_sector_fund_flow_rank(indicator="今日", sector_type="行业资金流")
print(stock_sector_fund_flow_rank_df)
```

数据示例-行业资金流-今日

```
    序号   名称  今日涨跌幅  ... 今日小单净流入-净额  今日小单净流入-净占比  今日主力净流入最大股
0    1    小金属   2.36  ... -9.101186e+08        -6.32        中国稀土
1    2   有色金属   1.96  ... -1.032419e+09        -5.49        铜陵有色
2    3    贵金属   3.94  ... -4.187458e+08        -4.25        紫金矿业
3    4   交运设备   4.51  ... -2.841601e+08        -3.35        中国通号
4    5   钢铁行业   1.69  ... -3.279502e+08        -3.82        包钢股份
..  ..    ...    ...  ...           ...          ...         ...
81  82   文化传媒  -2.23  ...  2.344617e+09         7.19         人民网
82  83    半导体  -2.20  ...  2.329264e+09         5.38        富创精密
83  84  互联网服务  -2.39  ...  3.482296e+09         7.76        荣联科技
84  85   软件开发  -2.82  ...  3.012122e+09         6.56        云赛智联
85  86   通信设备  -2.24  ...  3.339649e+09         7.95        精伦电子
[86 rows x 14 columns]
```

##### 主力净流入排名

接口: stock_main_fund_flow

目标地址: https://data.eastmoney.com/zjlx/list.html

描述: 东方财富网-数据中心-资金流向-主力净流入排名

限量: 单次获取指定 symbol 的主力净流入排名数据

输入参数

| 名称     | 类型  | 描述                                                                                     |
|--------|-----|----------------------------------------------------------------------------------------|
| symbol | str | symbol="全部股票"；choice of {"全部股票", "沪深A股", "沪市A股", "科创板", "深市A股", "创业板", "沪市B股", "深市B股"} |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | -       |
| 今日排行榜-主力净占比  | float64 | 注意单位: % |
| 今日排行榜-今日排名   | float64 | -       |
| 今日排行榜-今日涨跌   | float64 | 注意单位: % |
| 5日排行榜-主力净占比  | float64 | 注意单位: % |
| 5日排行榜-5日排名   | int64   | -       |
| 5日排行榜-5日涨跌   | float64 | 注意单位: % |
| 10日排行榜-主力净占比 | float64 | 注意单位: % |
| 10日排行榜-10日排名 | int64   | -       |
| 10日排行榜-10日涨跌 | float64 | 注意单位: % |
| 所属板块         | object  | -       |


接口示例

```python
import akshare as ak

stock_main_fund_flow_df = ak.stock_main_fund_flow(symbol="全部股票")
print(stock_main_fund_flow_df)
```

数据示例

```
        序号  代码     名称  ...  10日排行榜-10日排名  10日排行榜-10日涨跌   所属板块
0        0  200017   深中华B  ...            10          7.77   交运设备
1        1  001366   播恩集团  ...           124         22.84   农牧饲渔
2        2  900943   开开Ｂ股  ...            27          2.88   医药商业
3        3  200771   杭汽轮Ｂ  ...           361          0.28   通用设备
4        4  002861   瀛通通讯  ...            98         23.18   消费电子
    ...     ...    ...  ...           ...           ...    ...
5198  5198  301536   星宸科技  ...          1187          0.00   电子元件
5199  5199  002089  *ST新海  ...          1180          0.00   通信设备
5200  5200  001389   广合科技  ...          1185          0.00   电子元件
5201  5201  001359   平安电工  ...          1184          0.00  非金属材料
5202  5202  000005   ST星源  ...          1179          0.00   环保行业
[5203 rows x 14 columns]
```

##### 行业个股资金流

接口: stock_sector_fund_flow_summary

目标地址: https://data.eastmoney.com/bkzj/BK1034.html

描述: 东方财富网-数据中心-资金流向-行业资金流-xx行业个股资金流

限量: 单次获取指定 symbol 的个股资金流

输入参数

| 名称        | 类型  | 描述                                            |
|-----------|-----|-----------------------------------------------|
| symbol    | str | symbol="电源设备"                                 |
| indicator | str | indicator="今日"; choice of {"今日", "5日", "10日"} |

输出参数-今日

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | -       |
| 今日涨跌幅        | float64 | 注意单位: % |
| 今日主力净流入-净额   | float64 | -       |
| 今日主力净流入-净占比  | float64 | 注意单位: % |
| 今日超大单净流入-净额  | float64 | -       |
| 今日超大单净流入-净占比 | float64 | 注意单位: % |
| 今日大单净流入-净额   | float64 | -       |
| 今日大单净流入-净占比  | float64 | 注意单位: % |
| 今日中单净流入-净额   | float64 | -       |
| 今日中单净流入-净占比  | float64 | 注意单位: % |
| 今日小单净流入-净额   | float64 | -       |
| 今日小单净流入-净占比  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_sector_fund_flow_summary_df = ak.stock_sector_fund_flow_summary(symbol="电源设备", indicator="今日")
print(stock_sector_fund_flow_summary_df)
```

数据示例

```
    序号  代码     名称    最新价  ...  今日中单净流入-净额  今日中单净流入-净占比  今日小单净流入-净额  今日小单净流入-净占比
0    1  002665   首航高科   2.38  ... -33548194.0        -6.94 -41505162.0        -8.59
1    2  300870    欧陆通  46.95  ...  -7415353.0        -2.54 -11275570.0        -3.86
2    3  002851   麦格米特  25.70  ...  -1949180.0        -1.29 -14834710.0        -9.81
3    4  601727   上海电气   4.22  ...  -7677061.0        -6.71  -1779576.0        -1.56
4    5  300693   盛弘股份  29.95  ...  -2055055.0        -0.66  -4721271.0        -1.52
5    6  002335   科华数据  27.84  ...   3145038.0         0.75  -7959318.0        -1.89
6    7  300376    易事特   6.13  ...  -5406549.0        -6.38    737086.0         0.87
7    8  301386   未来电器  20.79  ...    589994.0         0.78  -4472460.0        -5.88
8    9  002364   中恒电气   5.98  ...   -713101.0        -1.94  -2889873.0        -7.86
9   10  688551    科威尔  53.96  ...  -1881214.0        -4.68    273893.0         0.68
10  11  300105   龙源技术   5.69  ...   -634763.0        -1.44   -438132.0        -1.00
11  12  301327   华宝新能  56.47  ...   -894989.0        -2.22    431792.0         1.07
12  13  688719   爱科赛博  57.91  ...   1827068.0         7.46   -938083.0        -3.83
13  14  300820   英杰电气  49.10  ...   4970648.0         2.22  -3876803.0        -1.73
14  15  300491   通合科技  18.80  ...   -372862.0        -0.65   1477839.0         2.58
15  16  002227  奥 特 迅   9.03  ...  -1979752.0        -4.66   3161510.0         7.43
16  17  300153   科泰电源   6.12  ...  -1515565.0        -6.23   2736469.0        11.25
17  18  600202    哈空调   4.53  ...    819676.0         3.08    564984.0         2.12
18  19  002630   华西能源   2.37  ...  -2249478.0        -1.68   3763541.0         2.82
19  20  301516    中远通  22.87  ...  -1230094.0        -0.61   3600893.0         1.78
20  21  600405    动力源   4.48  ...  -1001268.0        -1.43   4608713.0         6.57
21  22  600475   华光环能  11.07  ...   2154127.0         1.78   1958723.0         1.62
22  23  300593    新雷能  10.52  ... -14577276.0       -13.04  21502506.0        19.23
23  24  002534   西子洁能  10.36  ...   3695777.0         6.81   3304816.0         6.09
24  25  300713    英可瑞  13.95  ...   1636964.0         1.93   6529631.0         7.70
25  26  002255   海陆重工   5.03  ...   5416980.0         6.22   2929205.0         3.36
26  27  002366   融发核电   4.48  ...   6440715.0         6.28   3416059.0         3.33
27  28  600875   东方电气  16.39  ...   5033565.0         2.47  16389039.0         8.03
28  29  002518    科士达  24.15  ...   1447020.0         0.53  22734013.0         8.32
[29 rows x 15 columns]
```

##### 行业历史资金流

接口: stock_sector_fund_flow_hist

目标地址: https://data.eastmoney.com/bkzj/BK1034.html

描述: 东方财富网-数据中心-资金流向-行业资金流-行业历史资金流

限量: 单次获取指定行业的行业历史资金流数据

输入参数

| 名称     | 类型  | 描述            |
|--------|-----|---------------|
| symbol | str | symbol="汽车服务" |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 日期         | object  | 注意单位: % |
| 主力净流入-净额   | float64 | -       |
| 主力净流入-净占比  | float64 | 注意单位: % |
| 超大单净流入-净额  | float64 | -       |
| 超大单净流入-净占比 | float64 | 注意单位: % |
| 大单净流入-净额   | float64 | -       |
| 大单净流入-净占比  | float64 | 注意单位: % |
| 中单净流入-净额   | float64 | -       |
| 中单净流入-净占比  | float64 | 注意单位: % |
| 小单净流入-净额   | float64 | -       |
| 小单净流入-净占比  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_sector_fund_flow_hist_df = ak.stock_sector_fund_flow_hist(symbol="汽车服务")
print(stock_sector_fund_flow_hist_df)
```

数据示例

```
      日期    主力净流入-净额  主力净流入-净占比  ...  中单净流入-净占比 小单净流入-净额  小单净流入-净占比
0    2024-08-20 -54471010.0     -10.09  ...       2.28  37641239.0       6.97
1    2024-08-21  10178690.0       1.91  ...      -0.79 -10308872.0      -1.94
2    2024-08-22 -44754582.0      -8.39  ...       1.49  32534534.0       6.10
3    2024-08-23 -27649222.0      -6.52  ...      -0.86  29341810.0       6.92
4    2024-08-26 -42767862.0     -10.11  ...      -2.63  53748702.0      12.70
..          ...         ...        ...  ...        ...         ...        ...
116  2025-02-18 -93269892.0      -6.56  ...       1.78  58625760.0       4.12
117  2025-02-19  21681328.0       1.51  ...       1.97 -32209872.0      -2.25
118  2025-02-20  11260284.0       0.89  ...       0.43  -6260576.0      -0.50
119  2025-02-21 -81443762.0      -5.66  ...       3.29  47217792.0       3.28
120  2025-02-24  20329299.0       1.44  ...       1.47 -58079088.0      -4.12
[121 rows x 11 columns]
```

##### 概念历史资金流

接口: stock_concept_fund_flow_hist

目标地址: https://data.eastmoney.com/bkzj/BK0574.html

描述: 东方财富网-数据中心-资金流向-概念资金流-概念历史资金流

限量: 单次获取指定 symbol 的近期概念历史资金流数据

输入参数

| 名称     | 类型  | 描述            |
|--------|-----|---------------|
| symbol | str | symbol="数据要素" |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 日期         | object  | 注意单位: % |
| 主力净流入-净额   | float64 | -       |
| 主力净流入-净占比  | float64 | 注意单位: % |
| 超大单净流入-净额  | float64 | -       |
| 超大单净流入-净占比 | float64 | 注意单位: % |
| 大单净流入-净额   | float64 | -       |
| 大单净流入-净占比  | float64 | 注意单位: % |
| 中单净流入-净额   | float64 | -       |
| 中单净流入-净占比  | float64 | 注意单位: % |
| 小单净流入-净额   | float64 | -       |
| 小单净流入-净占比  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_concept_fund_flow_hist_df = ak.stock_concept_fund_flow_hist(symbol="数据要素")
print(stock_concept_fund_flow_hist_df)
```

数据示例

```
      日期      主力净流入-净额  主力净流入-净占比  ... 中单净流入-净占比 小单净流入-净额  小单净流入-净占比
0    2024-08-20 -6.313736e+08      -4.65  ...      -1.32  8.085588e+08       5.96
1    2024-08-21 -3.658787e+08      -2.77  ...      -0.68  4.585833e+08       3.47
2    2024-08-22 -8.680331e+08      -6.05  ...      -1.09  1.004540e+09       7.00
3    2024-08-23 -1.840573e+08      -1.41  ...      -2.69  5.319882e+08       4.07
4    2024-08-26 -1.560898e+08      -1.42  ...      -2.13  3.791972e+08       3.44
..          ...           ...        ...  ...        ...           ...        ...
116  2025-02-18 -1.405755e+10      -8.92  ...       1.16  1.216662e+10       7.72
117  2025-02-19 -9.526664e+08      -0.73  ...      -0.99  2.248251e+09       1.71
118  2025-02-20 -5.894632e+09      -4.55  ...       0.36  5.343512e+09       4.12
119  2025-02-21  2.832990e+09       1.60  ...      -1.23 -5.548521e+08      -0.31
120  2025-02-24 -1.108665e+10      -7.88  ...       1.80  8.521074e+09       6.06
[121 rows x 11 columns]
```

### 筹码分布

接口: stock_cyq_em

目标地址: https://quote.eastmoney.com/concept/sz000001.html

描述: 东方财富网-概念板-行情中心-日K-筹码分布

限量: 单次返回指定 symbol 和 adjust 的近 90 个交易日数据

输入参数

| 名称     | 类型  | 描述                                                           |
|--------|-----|--------------------------------------------------------------|
| symbol | str | symbol="000001"; 股票代码                                        |
| adjust | str | adjust=""; choice of {"qfq": "前复权", "hfq": "后复权", "": "不复权"} |

输出参数

| 名称     | 类型      | 描述 |
|--------|---------|----|
| 日期     | object  | -  |
| 获利比例   | float64 | -  |
| 平均成本   | float64 | -  |
| 90成本-低 | float64 | -  |
| 90成本-高 | float64 | -  |
| 90集中度  | float64 | -  |
| 70成本-低 | float64 | -  |
| 70成本-高 | float64 | -  |
| 70集中度  | float64 | -  |

接口示例

```python
import akshare as ak

stock_cyq_em_df = ak.stock_cyq_em(symbol="000001", adjust="")
print(stock_cyq_em_df)
```

数据示例

```
      日期      获利比例   平均成本 90成本-低  ...  90集中度  70成本-低  70成本-高  70集中度
0   2023-08-30  0.002484  12.09   11.23  ...  0.081965   11.36   12.87  0.062519
1   2023-08-31  0.002459  12.09   11.21  ...  0.082870   11.36   12.85  0.061726
2   2023-09-01  0.146772  12.07   11.21  ...  0.082102   11.34   12.85  0.062625
3   2023-09-04  0.267725  12.05   11.21  ...  0.082102   11.34   12.85  0.062625
4   2023-09-05  0.186573  12.05   11.21  ...  0.082102   11.34   12.85  0.062625
..         ...       ...    ...     ...  ...       ...     ...     ...       ...
85  2024-01-05  0.103536  11.29    9.16  ...  0.174579    9.44   12.59  0.143242
86  2024-01-08  0.034893  11.29    9.16  ...  0.173302    9.40   12.59  0.145025
87  2024-01-09  0.061026  11.29    9.16  ...  0.173302    9.37   12.59  0.146813
88  2024-01-10  0.006555  11.29    9.16  ...  0.173302    9.37   12.59  0.146813
89  2024-01-11  0.074399  11.25    9.16  ...  0.173302    9.33   12.56  0.147273
[90 rows x 9 columns]
```

### 基本面数据

#### 股东大会

接口: stock_gddh_em

目标地址: https://data.eastmoney.com/gddh/

描述: 东方财富网-数据中心-股东大会

限量: 单次返回所有数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称         | 类型     | 描述 |
|------------|--------|----|
| 代码         | object | -  |
| 简称         | object | -  |
| 股东大会名称     | object | -  |
| 召开开始日      | object | -  |
| 股权登记日      | object | -  |
| 现场登记日      | object | -  |
| 网络投票时间-开始日 | object | -  |
| 网络投票时间-结束日 | object | -  |
| 决议公告日      | object | -  |
| 公告日        | object | -  |
| 序列号        | object | -  |
| 提案         | object | -  |

接口示例

```python
import akshare as ak

stock_gddh_em_df = ak.stock_gddh_em()
print(stock_gddh_em_df)
```

数据示例

```
       代码     简称  ...     序列号                                                 提案
0     603131   上海沪工  ...  209796                                 1、关于提名公司独立董事候选人的议案
1     600252   中恒集团  ...  209759  1、广西梧州中恒集团股份有限公司关于控股子公司莱美药业为其参股公司康德赛借款提供担保展期的议...
2     300426   唐德影视  ...  209760                     1、《关于接受控股股东担保并向其提供反担保暨关联交易的议案》
3     300887   谱尼测试  ...  209756                            1、《关于使用闲置募集资金进行现金管理的议案》
4     002212    天融信  ...  209757                     1、《关于公司为全资孙公司银行综合授信提供担保总额度的议案》
...      ...    ...  ...     ...                                                ...
5021  002113  *ST天润  ...  181437  1、审议关于《2022年度董事会工作报告》的议案、审议关于《2022年年度报告及摘要》的议案...
5022  002118  *ST紫鑫  ...  181461                            1、《关于提名选举第八届董事会独立董事的议案》
5023  600777   ST新潮  ...  173337  1、《关于董事会提前换届选举非独立董事的议案》、《关于董事会提前换届选举独立董事的议案》、《...
5024  600739   辽宁成大  ...   35446  1、董事会工作报告\n2、监事会工作报告\n3、公司2011年年度报告全文及摘要\n4、公司...
5025  600739   辽宁成大  ...   35446  1、董事会工作报告\n2、监事会工作报告\n3、公司2011年年度报告全文及摘要\n4、公司...
[5026 rows x 12 columns]
```

#### 重大合同

接口: stock_zdhtmx_em

目标地址: https://data.eastmoney.com/zdht/mx.html

描述: 东方财富网-数据中心-重大合同-重大合同明细

限量: 单次返回指定 start_date 和 end_date 的所有数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| start_date | str | start_date="20200819" |
| end_date   | str | end_date="20230819"   |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| 序号            | int64   | -  |
| 股票代码          | object  | -  |
| 股票简称          | object  | -  |
| 签署主体          | object  | -  |
| 签署主体-与上市公司关系  | object  | -  |
| 其他签署方         | object  | -  |
| 其他签署方-与上市公司关系 | object  | -  |
| 合同类型          | object  | -  |
| 合同名称          | object  | -  |
| 合同金额          | float64 | -  |
| 上年度营业收入       | float64 | -  |
| 占上年度营业收入比例    | float64 | -  |
| 最新财务报表的营业收入   | float64 | -  |
| 签署日期          | object  | -  |
| 公告日期          | object  | -  |

接口示例

```python
import akshare as ak

stock_zdhtmx_em_df = ak.stock_zdhtmx_em(start_date="20220819", end_date="20230819")
print(stock_zdhtmx_em_df)
```

数据示例

```
      序号  股票代码  股票简称  ... 最新财务报表的营业收入 签署日期 公告日期
0        1  688516   奥特维  ...           NaN  NaT  2023-08-18
1        2  688132  邦彦技术  ...           NaN  NaT  2023-08-18
2        3  831526  凯华材料  ...  5.101723e+07  NaT  2023-08-17
3        4  603085  天成自控  ...           NaN  NaT  2023-08-17
4        5  300345  华民股份  ...           NaN  NaT  2023-08-17
...    ...     ...   ...  ...           ...  ...         ...
1012  1013  300197  节能铁汉  ...  2.434768e+09  NaT  2022-08-22
1013  1014  688516   奥特维  ...           NaN  NaT  2022-08-20
1014  1015  688022  瀚川智能  ...           NaN  NaT  2022-08-19
1015  1016  603815  交建股份  ...           NaN  NaT  2022-08-19
1016  1017  603815  交建股份  ...           NaN  NaT  2022-08-19
[1017 rows x 15 columns]
```

#### 个股研报

接口: stock_research_report_em

目标地址: https://data.eastmoney.com/report/stock.jshtml

描述: 东方财富网-数据中心-研究报告-个股研报

限量: 单次返回指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="000001" |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| 序号            | int64   | -  |
| 股票代码          | object  | -  |
| 股票简称          | object  | -  |
| 报告名称          | object  | -  |
| 东财评级          | object  | -  |
| 机构            | object  | -  |
| 近一月个股研报数      | int64   | -  |
| 2024-盈利预测-收益  | float64 | -  |
| 2024-盈利预测-市盈率 | float64 | -  |
| 2025-盈利预测-收益  | float64 | -  |
| 2025-盈利预测-市盈率 | float64 | -  |
| 2026-盈利预测-收益  | float64 | -  |
| 2026-盈利预测-市盈率 | float64 | -  |
| 行业            | object  | -  |
| 日期            | object  | -  |
| 报告PDF链接       | object  | -  |

接口示例

```python
import akshare as ak

stock_research_report_em_df = ak.stock_research_report_em(symbol="000001")
print(stock_research_report_em_df)
```

数据示例

```
    序号    股票代码  ...  日期                           报告PDF链接
0      1  000001  ...  2025-01-10  https://pdf.dfcfw.com/pdf/H3_AP202501101641890...
1      2  000001  ...  2024-10-22  https://pdf.dfcfw.com/pdf/H3_AP202410221640400...
2      3  000001  ...  2024-10-22  https://pdf.dfcfw.com/pdf/H3_AP202410221640398...
3      4  000001  ...  2024-10-20  https://pdf.dfcfw.com/pdf/H3_AP202410201640374...
4      5  000001  ...  2024-10-19  https://pdf.dfcfw.com/pdf/H3_AP202410191640373...
..   ...     ...  ...         ...                                                ...
272  273  000001  ...  2017-03-22  https://pdf.dfcfw.com/pdf/H3_AP201703220427468...
273  274  000001  ...  2017-03-20  https://pdf.dfcfw.com/pdf/H3_AP201703200421759...
274  275  000001  ...  2017-03-17  https://pdf.dfcfw.com/pdf/H3_AP201703170415156...
275  276  000001  ...  2017-03-07  https://pdf.dfcfw.com/pdf/H3_AP201705190591661...
276  277  000001  ...  2017-02-03  https://pdf.dfcfw.com/pdf/H3_AP201702030311575...
[277 rows x 16 columns]
```

#### 沪深京 A 股公告

接口: stock_notice_report

目标地址: https://data.eastmoney.com/notices/hsa/5.html

描述: 东方财富网-数据中心-公告大全-沪深京 A 股公告

限量: 单次获取指定 symbol 和 date 的数据

输入参数

| 名称     | 类型  | 描述                                                                                      |
|--------|-----|-----------------------------------------------------------------------------------------|
| symbol | str | symbol='财务报告'; choice of {"全部", "重大事项", "财务报告", "融资公告", "风险提示", "资产重组", "信息变更", "持股变动"} |
| date   | str | date="20220511"; 指定日期                                                                   |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 代码   | object | -   |
| 名称   | object | -   |
| 公告标题 | object | -   |
| 公告类型 | object | -   |
| 公告日期 | object | -   |
| 网址   | object | -   |

接口示例

```python
import akshare as ak

stock_notice_report_df = ak.stock_notice_report(symbol='财务报告', date="20240613")
print(stock_notice_report_df)
```

数据示例

```
      代码  ...                                       网址
0    123122  ...  https://data.eastmoney.com/notices/detail/1231...
1    123107  ...  https://data.eastmoney.com/notices/detail/1231...
2    300941  ...  https://data.eastmoney.com/notices/detail/3009...
3    300689  ...  https://data.eastmoney.com/notices/detail/3006...
4    300854  ...  https://data.eastmoney.com/notices/detail/3008...
..      ...  ...                                                ...
134  000159  ...  https://data.eastmoney.com/notices/detail/0001...
135  688478  ...  https://data.eastmoney.com/notices/detail/6884...
136  688513  ...  https://data.eastmoney.com/notices/detail/6885...
137  600583  ...  https://data.eastmoney.com/notices/detail/6005...
138  001301  ...  https://data.eastmoney.com/notices/detail/0013...
[139 rows x 6 columns]
```

#### 财务报表-新浪

接口: stock_financial_report_sina

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vFD_FinanceSummary/stockid/600600/displaytype/4.phtml?source=fzb&qq-pf-to=pcqq.group

描述: 新浪财经-财务报表-三大报表

限量: 单次获取指定报表的所有年份数据的历史数据

注意: 原始数据中有 `国内票证结算` 和 `内部应收款` 字段重, 返回数据中已经剔除

输入参数

| 名称     | 类型  | 描述                                                  |
|--------|-----|-----------------------------------------------------|
| stock  | str | stock="sh600600"; 带市场标识的股票代码                        |
| symbol | str | symbol="现金流量表"; choice of {"资产负债表", "利润表", "现金流量表"} |

输出参数

| 名称   | 类型     | 描述   |
|------|--------|------|
| 报告日  | object | 报告日期 |
| 流动资产 | object | -    |
| ...  | object | -    |
| 类型   | object | -    |
| 更新日期 | object | -    |

接口示例

```python
import akshare as ak

stock_financial_report_sina_df = ak.stock_financial_report_sina(stock="sh600600", symbol="资产负债表")
print(stock_financial_report_sina_df)
```

数据示例

```
      报告日 流动资产           货币资金  ...   币种    类型      更新日期
0   20230331  NaN  15554422004.0  ...  CNY  合并期末  2023-04-24T18:05:05
1   20221231  NaN  17854931855.0  ...  CNY  合并期末  2023-03-22T20:50:09
2   20220930  NaN  17045567340.0  ...  CNY  合并期末  2022-10-26T17:20:07
3   20220630  NaN  17587197778.0  ...  CNY  合并期末  2022-08-25T21:25:04
4   20220331  NaN  12818371932.0  ...  CNY  合并期末  2022-04-28T16:20:04
..       ...  ...            ...  ...  ...   ...                  ...
95  19961231  NaN    671405728.0  ...  CNY  合并期末  2020-03-13T15:29:48
96  19960630  NaN            NaN  ...  CNY  合并期末  2020-03-13T15:29:48
97  19951231  NaN    741738874.0  ...  CNY  合并期末  2020-03-13T15:29:48
98  19950630  NaN            NaN  ...  CNY  合并期末  2020-03-13T15:29:48
99  19941231  NaN    712428000.0  ...  CNY  合并期末  2020-03-13T15:29:48
[100 rows x 147 columns]
```

#### 财务报表-东财

##### 资产负债表-按报告期

接口: stock_balance_sheet_by_report_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-资产负债表-按报告期

限量: 单次获取指定 symbol 的资产负债表-按报告期数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 319 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_balance_sheet_by_report_em_df = ak.stock_balance_sheet_by_report_em(symbol="SH600519")
print(stock_balance_sheet_by_report_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... OSOPINION_TYPE LISTING_STATE
0   600519.SH        600519  ...           None             0
1   600519.SH        600519  ...           None             0
2   600519.SH        600519  ...           None             0
3   600519.SH        600519  ...           None             0
4   600519.SH        600519  ...           None             0
..        ...           ...  ...            ...           ...
87  600519.SH        600519  ...           None             0
88  600519.SH        600519  ...           None             0
89  600519.SH        600519  ...           None             0
90  600519.SH        600519  ...           None             0
91  600519.SH        600519  ...           None             0
[92 rows x 319 columns]
```

##### 资产负债表-按年度

接口: stock_balance_sheet_by_yearly_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-资产负债表-按年度

限量: 单次获取指定 symbol 的资产负债表-按年度数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 319 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_balance_sheet_by_yearly_em_df = ak.stock_balance_sheet_by_yearly_em(symbol="SH600519")
print(stock_balance_sheet_by_yearly_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ... OSOPINION_TYPE LISTING_STATE
0   600519.SH        600519  ...           None             0
1   600519.SH        600519  ...           None             0
2   600519.SH        600519  ...           None             0
3   600519.SH        600519  ...           None             0
4   600519.SH        600519  ...           None             0
5   600519.SH        600519  ...           None             0
6   600519.SH        600519  ...           None             0
7   600519.SH        600519  ...           None             0
8   600519.SH        600519  ...           None             0
9   600519.SH        600519  ...           None             0
10  600519.SH        600519  ...           None             0
11  600519.SH        600519  ...           None             0
12  600519.SH        600519  ...           None             0
13  600519.SH        600519  ...           None             0
14  600519.SH        600519  ...           None             0
15  600519.SH        600519  ...           None             0
16  600519.SH        600519  ...           None             0
17  600519.SH        600519  ...           None             0
18  600519.SH        600519  ...           None             0
19  600519.SH        600519  ...           None             0
20  600519.SH        600519  ...           None             0
21  600519.SH        600519  ...           None             0
22  600519.SH        600519  ...           None             0
23  600519.SH        600519  ...           None             0
24  600519.SH        600519  ...           None             0
[25 rows x 319 columns]
```

##### 利润表-按报告期

接口: stock_profit_sheet_by_report_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-利润表-报告期

限量: 单次获取指定 symbol 的利润表-报告期数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 203 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_profit_sheet_by_report_em_df = ak.stock_profit_sheet_by_report_em(symbol="SH600519")
print(stock_profit_sheet_by_report_em_df)
```

数据示例

```
   SECUCODE SECURITY_CODE  ... ACF_END_INCOME_YOY OPINION_TYPE
0   600519.SH        600519  ...               None         None
1   600519.SH        600519  ...               None         None
2   600519.SH        600519  ...               None         None
3   600519.SH        600519  ...               None      标准无保留意见
4   600519.SH        600519  ...               None         None
..        ...           ...  ...                ...          ...
87  600519.SH        600519  ...               None      标准无保留意见
88  600519.SH        600519  ...               None      标准无保留意见
89  600519.SH        600519  ...               None      标准无保留意见
90  600519.SH        600519  ...               None      标准无保留意见
91  600519.SH        600519  ...               None      标准无保留意见
[92 rows x 203 columns]
```

##### 利润表-按年度

接口: stock_profit_sheet_by_yearly_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-利润表-按年度

限量: 单次获取指定 symbol 的利润表-按年度数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 203 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_profit_sheet_by_yearly_em_df = ak.stock_profit_sheet_by_yearly_em(symbol="SH600519")
print(stock_profit_sheet_by_yearly_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... ACF_END_INCOME_YOY OPINION_TYPE
0   600519.SH        600519  ...               None      标准无保留意见
1   600519.SH        600519  ...               None      标准无保留意见
2   600519.SH        600519  ...               None      标准无保留意见
3   600519.SH        600519  ...               None      标准无保留意见
4   600519.SH        600519  ...               None      标准无保留意见
5   600519.SH        600519  ...               None      标准无保留意见
6   600519.SH        600519  ...               None      标准无保留意见
7   600519.SH        600519  ...               None      标准无保留意见
8   600519.SH        600519  ...               None      标准无保留意见
9   600519.SH        600519  ...               None      标准无保留意见
10  600519.SH        600519  ...               None      标准无保留意见
11  600519.SH        600519  ...               None      标准无保留意见
12  600519.SH        600519  ...               None      标准无保留意见
13  600519.SH        600519  ...               None      标准无保留意见
14  600519.SH        600519  ...               None      标准无保留意见
15  600519.SH        600519  ...               None      标准无保留意见
16  600519.SH        600519  ...               None      标准无保留意见
17  600519.SH        600519  ...               None      标准无保留意见
18  600519.SH        600519  ...               None      标准无保留意见
19  600519.SH        600519  ...               None      标准无保留意见
20  600519.SH        600519  ...               None      标准无保留意见
21  600519.SH        600519  ...               None      标准无保留意见
22  600519.SH        600519  ...               None      标准无保留意见
23  600519.SH        600519  ...               None      标准无保留意见
24  600519.SH        600519  ...               None      标准无保留意见
[25 rows x 203 columns]
```

##### 利润表-按单季度

接口: stock_profit_sheet_by_quarterly_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-利润表-按单季度

限量: 单次获取指定 symbol 的利润表-按单季度数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 204 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_profit_sheet_by_quarterly_em_df = ak.stock_profit_sheet_by_quarterly_em(symbol="SH600519")
print(stock_profit_sheet_by_quarterly_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ... DEDUCT_PARENT_NETPROFIT DEDUCT_PARENT_NETPROFIT_QOQ
0   600519.SH        600519  ...            1.686819e+10                   11.201931
1   600519.SH        600519  ...            1.516897e+10                  -26.996696
2   600519.SH        600519  ...            2.077848e+10                   12.934838
3   600519.SH        600519  ...            1.839864e+10                   25.756158
4   600519.SH        600519  ...            1.463041e+10                   16.858574
..        ...           ...  ...                     ...                         ...
82  600519.SH        600519  ...                     NaN                         NaN
83  600519.SH        600519  ...            6.878058e+07                   -3.336461
84  600519.SH        600519  ...            7.115463e+07                         NaN
85  600519.SH        600519  ...                     NaN                         NaN
86  600519.SH        600519  ...                     NaN                         NaN
[87 rows x 204 columns]
```

##### 现金流量表-按报告期

接口: stock_cash_flow_sheet_by_report_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-现金流量表-按报告期

限量: 单次获取指定 symbol 的现金流量表-按报告期数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 252 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_cash_flow_sheet_by_report_em_df = ak.stock_cash_flow_sheet_by_report_em(symbol="SH600519")
print(stock_cash_flow_sheet_by_report_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ... MINORITY_INTEREST MINORITY_INTEREST_YOY
0   600519.SH        600519  ...               NaN                   NaN
1   600519.SH        600519  ...               NaN                   NaN
2   600519.SH        600519  ...               NaN                   NaN
3   600519.SH        600519  ...               NaN                   NaN
4   600519.SH        600519  ...               NaN                   NaN
..        ...           ...  ...               ...                   ...
83  600519.SH        600519  ...               NaN                   NaN
84  600519.SH        600519  ...               NaN                   NaN
85  600519.SH        600519  ...               NaN                   NaN
86  600519.SH        600519  ...               NaN                   NaN
87  600519.SH        600519  ...               NaN                   NaN
[88 rows x 252 columns]
```

##### 现金流量表-按年度

接口: stock_cash_flow_sheet_by_yearly_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-现金流量表-按年度

限量: 单次获取指定 symbol 的现金流量表-按年度数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 314 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_cash_flow_sheet_by_yearly_em_df = ak.stock_cash_flow_sheet_by_yearly_em(symbol="SH600519")
print(stock_cash_flow_sheet_by_yearly_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ... FBCCE_ADD_YOY CREDIT_IMPAIRMENT_INCOME_YOY
0   601398.SH        601398  ...    238.302033                         None
1   601398.SH        601398  ...   -204.008112                         None
2   601398.SH        601398  ...    676.398241                         None
3   601398.SH        601398  ...   -446.960304                         None
4   601398.SH        601398  ...   -103.265330                         None
5   601398.SH        601398  ...    231.370619                         None
6   601398.SH        601398  ...   -156.355892                         None
7   601398.SH        601398  ...   1112.723130                         None
8   601398.SH        601398  ...    115.092223                         None
9   601398.SH        601398  ...   -169.124835                         None
10  601398.SH        601398  ...     10.647686                         None
11  601398.SH        601398  ...    167.055537                         None
12  601398.SH        601398  ...    160.423857                         None
13  601398.SH        601398  ...           NaN                         None
14  601398.SH        601398  ...           NaN                         None
15  601398.SH        601398  ...           NaN                         None
16  601398.SH        601398  ...           NaN                         None
17  601398.SH        601398  ...           NaN                         None
18  601398.SH        601398  ...           NaN                         None
19  601398.SH        601398  ...           NaN                         None
[20 rows x 314 columns]
```

##### 现金流量表-按单季度

接口: stock_cash_flow_sheet_by_quarterly_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/NewFinanceAnalysis/Index?type=web&code=sh600519#lrb-0

描述: 东方财富-股票-财务分析-现金流量表-按单季度

限量: 单次获取指定 symbol 的现金流量表-按单季度数据

输入参数

| 名称     | 类型  | 描述                      |
|--------|-----|-------------------------|
| symbol | str | symbol="SH600519"; 股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 315 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_cash_flow_sheet_by_quarterly_em_df = ak.stock_cash_flow_sheet_by_quarterly_em(symbol="SH600519")
print(stock_cash_flow_sheet_by_quarterly_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ... OPINION_TYPE OSOPINION_TYPE
0   601398.SH        601398  ...         None           None
1   601398.SH        601398  ...         None           None
2   601398.SH        601398  ...      标准无保留意见           None
3   601398.SH        601398  ...         None           None
4   601398.SH        601398  ...         None           None
..        ...           ...  ...          ...            ...
61  601398.SH        601398  ...         None           None
62  601398.SH        601398  ...      标准无保留意见           None
63  601398.SH        601398  ...         None           None
64  601398.SH        601398  ...         None           None
65  601398.SH        601398  ...         None           None
[66 rows x 315 columns]
```

#### 财务报表-同花顺

##### 资产负债表

接口: stock_financial_debt_new_ths

目标地址: https://basic.10jqka.com.cn/astockpc/astockmain/index.html#/financen?code=000063

描述: 同花顺-财务指标-资产负债表；替换 stock_financial_debt_ths 接口

限量: 单次获取资产负债表所有历史数据

输入参数

| 名称        | 类型  | 描述                                          |
|-----------|-----|---------------------------------------------|
| symbol    | str | symbol="000063"; 股票代码                       |
| indicator | str | indicator="按报告期"; choice of {"按报告期", "按年度"} |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| report_date   | object  | -  |
| report_name   | object  | -  |
| report_period | object  | -  |
| quarter_name  | object  | -  |
| metric_name   | object  | -  |
| value         | float64 | -  |
| single        | object  | -  |
| yoy           | float64 | -  |
| mom           | object  | -  |
| single_yoy    | object  | -  |

接口示例

```python
import akshare as ak

stock_financial_debt_new_ths_df = ak.stock_financial_debt_new_ths(symbol="000063", indicator="按年度")
print(stock_financial_debt_new_ths_df)
```

数据示例

```
     report_date report_name report_period  ...          yoy   mom single_yoy
0     2024-12-31      2024年报        2024-4  ...         <NA>  <NA>       <NA>
1     2024-12-31      2024年报        2024-4  ...         <NA>  <NA>       <NA>
2     2024-12-31      2024年报        2024-4  ...   0.16420728  <NA>       <NA>
3     2024-12-31      2024年报        2024-4  ...  -0.15807123  <NA>       <NA>
4     2024-12-31      2024年报        2024-4  ...   0.00005916  <NA>       <NA>
...          ...         ...           ...  ...          ...   ...        ...
3684  1994-12-31      1994年报        1994-4  ...         <NA>  <NA>       <NA>
3685  1994-12-31      1994年报        1994-4  ...         <NA>  <NA>       <NA>
3686  1994-12-31      1994年报        1994-4  ...         <NA>  <NA>       <NA>
3687  1994-12-31      1994年报        1994-4  ...         <NA>  <NA>       <NA>
3688  1994-12-31      1994年报        1994-4  ...         <NA>  <NA>       <NA>
[3689 rows x 10 columns]
```

##### 利润表

接口: stock_financial_benefit_new_ths

目标地址: https://basic.10jqka.com.cn/astockpc/astockmain/index.html#/financen?code=000063

描述: 同花顺-财务指标-利润表；替换 stock_financial_benefit_ths 接口

限量: 单次获取利润表所有历史数据

输入参数

| 名称        | 类型  | 描述                                                                      |
|-----------|-----|-------------------------------------------------------------------------|
| symbol    | str | symbol="000063"; 股票代码                                                   |
| indicator | str | indicator="按报告期"; choice of {"按报告期", "一季度", "二季度", "三季度", "四季度", "按年度"} |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| report_date   | object  | -  |
| report_name   | object  | -  |
| report_period | object  | -  |
| quarter_name  | object  | -  |
| metric_name   | object  | -  |
| value         | float64 | -  |
| single        | object  | -  |
| yoy           | float64 | -  |
| mom           | object  | -  |
| single_yoy    | object  | -  |

接口示例

```python
import akshare as ak

stock_financial_benefit_new_ths_df = ak.stock_financial_benefit_new_ths(symbol="000063", indicator="按报告期")
print(stock_financial_benefit_new_ths_df)
```

数据示例

```
     report_date report_name report_period  ...        yoy       mom  single_yoy
0     2025-09-30     2025三季报        2025-3  ...  -0.307496 -0.912701   -0.893714
1     2025-09-30     2025三季报        2025-3  ...        NaN       NaN         NaN
2     2025-09-30     2025三季报        2025-3  ...   0.116943 -0.221946   -0.139571
3     2025-09-30     2025三季报        2025-3  ...        NaN       NaN         NaN
4     2025-09-30     2025三季报        2025-3  ...        NaN       NaN         NaN
...          ...         ...           ...  ...        ...       ...         ...
2545  2013-06-30      2013中报        2013-2  ...   0.285714 -0.500000    0.000000
2546  2013-06-30      2013中报        2013-2  ...        NaN       NaN         NaN
2547  2013-06-30      2013中报        2013-2  ...        NaN       NaN         NaN
2548  2013-06-30      2013中报        2013-2  ...        NaN       NaN         NaN
2549  2013-06-30      2013中报        2013-2  ... -11.669821  0.777595    0.078450
[2550 rows x 10 columns]
```

##### 现金流量表

接口: stock_financial_cash_new_ths

目标地址: https://basic.10jqka.com.cn/astockpc/astockmain/index.html#/financen?code=000063

描述: 同花顺-财务指标-现金流量表；替换 stock_financial_cash_ths 接口

限量: 单次获取现金流量表所有历史数据

输入参数

| 名称        | 类型  | 描述                                                                      |
|-----------|-----|-------------------------------------------------------------------------|
| symbol    | str | symbol="000063"; 股票代码                                                   |
| indicator | str | indicator="按报告期"; choice of {"按报告期", "一季度", "二季度", "三季度", "四季度", "按年度"} |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| report_date   | object  | -  |
| report_name   | object  | -  |
| report_period | object  | -  |
| quarter_name  | object  | -  |
| metric_name   | object  | -  |
| value         | float64 | -  |
| single        | object  | -  |
| yoy           | float64 | -  |
| mom           | object  | -  |
| single_yoy    | object  | -  |

接口示例

```python
import akshare as ak

stock_financial_cash_new_ths_df = ak.stock_financial_cash_new_ths(symbol="000063", indicator="按年度")
print(stock_financial_cash_new_ths_df)
```

数据示例

```
     report_date report_name  ...          mom   single_yoy
0     2024-12-31      2024年报  ...         <NA>         <NA>
1     2024-12-31      2024年报  ...  -0.23161757   0.57803131
2     2024-12-31      2024年报  ...   0.36015447  -0.45630237
3     2024-12-31      2024年报  ...   0.17900055  -0.47569575
4     2024-12-31      2024年报  ...   0.05788915  -0.47553164
...          ...         ...  ...          ...          ...
2425  1998-12-31      1998年报  ...         <NA>         <NA>
2426  1998-12-31      1998年报  ...         <NA>         <NA>
2427  1998-12-31      1998年报  ...         <NA>         <NA>
2428  1998-12-31      1998年报  ...         <NA>         <NA>
2429  1998-12-31      1998年报  ...         <NA>         <NA>
[2430 rows x 10 columns]
```

#### 财务报表-东财-已退市股票

##### 资产负债表-按报告期

接口: stock_balance_sheet_by_report_delisted_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=SZ000013#/cwfx/zcfzb

描述: 东方财富-股票-财务分析-资产负债表-已退市股票-按报告期

限量: 单次获取指定 symbol 的资产负债表-按报告期数据

输入参数

| 名称     | 类型  | 描述                               |
|--------|-----|----------------------------------|
| symbol | str | symbol="SZ000013"; 带市场标识的已退市股票代码 |

输出参数

| 名称  | 类型  | 描述         |
|-----|-----|------------|
| -   | -   | 319项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_balance_sheet_by_report_delisted_em_df = ak.stock_balance_sheet_by_report_delisted_em(symbol="SZ000013")
print(stock_balance_sheet_by_report_delisted_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... OSOPINION_TYPE LISTING_STATE
0   000013.SZ        000013  ...           None             2
1   000013.SZ        000013  ...           None             2
2   000013.SZ        000013  ...           None             2
3   000013.SZ        000013  ...           None             2
4   000013.SZ        000013  ...           None             2
5   000013.SZ        000013  ...           None             2
6   000013.SZ        000013  ...           None             2
7   000013.SZ        000013  ...           None             2
8   000013.SZ        000013  ...           None             2
9   000013.SZ        000013  ...           None             2
10  000013.SZ        000013  ...           None             2
11  000013.SZ        000013  ...           None             2
12  000013.SZ        000013  ...           None             2
13  000013.SZ        000013  ...           None             2
14  000013.SZ        000013  ...           None             2
15  000013.SZ        000013  ...           None             2
16  000013.SZ        000013  ...           None             2
17  000013.SZ        000013  ...           None             2
18  000013.SZ        000013  ...           None             2
19  000013.SZ        000013  ...           None             2
20  000013.SZ        000013  ...           None             2
21  000013.SZ        000013  ...           None             2
22  000013.SZ        000013  ...           None             2
23  000013.SZ        000013  ...           None             2
24  000013.SZ        000013  ...           None             2
25  000013.SZ        000013  ...           None             2
26  000013.SZ        000013  ...           None             2
27  000013.SZ        000013  ...           None             2
28  000013.SZ        000013  ...           None             2
29  000013.SZ        000013  ...           None             2
30  000013.SZ        000013  ...           None             2
31  000013.SZ        000013  ...           None             2
32  000013.SZ        000013  ...           None             2
33  000013.SZ        000013  ...           None             2
34  000013.SZ        000013  ...           None             2
35  000013.SZ        000013  ...           None             2
36  000013.SZ        000013  ...           None             2
37  000013.SZ        000013  ...           None             2
[38 rows x 319 columns]
```

##### 利润表-按报告期

接口: stock_profit_sheet_by_report_delisted_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=SZ000013#/cwfx/lrb

描述: 东方财富-股票-财务分析-利润表-已退市股票-按报告期

限量: 单次获取指定 symbol 的利润表-按报告期数据

输入参数

| 名称     | 类型  | 描述                               |
|--------|-----|----------------------------------|
| symbol | str | symbol="SZ000013"; 带市场标识的已退市股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 203 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_profit_sheet_by_report_delisted_em_df = ak.stock_profit_sheet_by_report_delisted_em(symbol="SZ000013")
print(stock_profit_sheet_by_report_delisted_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... ACF_END_INCOME_YOY  OPINION_TYPE
0   000013.SZ        000013  ...               None          None
1   000013.SZ        000013  ...               None          None
2   000013.SZ        000013  ...               None          None
3   000013.SZ        000013  ...               None          None
4   000013.SZ        000013  ...               None          None
5   000013.SZ        000013  ...               None          None
6   000013.SZ        000013  ...               None          None
7   000013.SZ        000013  ...               None          None
8   000013.SZ        000013  ...               None          None
9   000013.SZ        000013  ...               None          None
10  000013.SZ        000013  ...               None          None
11  000013.SZ        000013  ...               None          None
12  000013.SZ        000013  ...               None          None
13  000013.SZ        000013  ...               None          None
14  000013.SZ        000013  ...               None       标准无保留意见
15  000013.SZ        000013  ...               None          None
16  000013.SZ        000013  ...               None  带解释性说明的无保留意见
17  000013.SZ        000013  ...               None          None
18  000013.SZ        000013  ...               None          None
19  000013.SZ        000013  ...               None        无法表示意见
20  000013.SZ        000013  ...               None          None
21  000013.SZ        000013  ...               None          未经审计
22  000013.SZ        000013  ...               None          None
23  000013.SZ        000013  ...               None        无法表示意见
24  000013.SZ        000013  ...               None          None
25  000013.SZ        000013  ...               None          未经审计
26  000013.SZ        000013  ...               None          None
27  000013.SZ        000013  ...               None        无法表示意见
28  000013.SZ        000013  ...               None          None
29  000013.SZ        000013  ...               None        无法表示意见
30  000013.SZ        000013  ...               None          None
31  000013.SZ        000013  ...               None          None
32  000013.SZ        000013  ...               None          None
33  000013.SZ        000013  ...               None          None
34  000013.SZ        000013  ...               None          None
35  000013.SZ        000013  ...               None          None
36  000013.SZ        000013  ...               None          None
37  000013.SZ        000013  ...               None          None
38  000013.SZ        000013  ...               None          None
[39 rows x 203 columns]
```

##### 现金流量表-按报告期

接口: stock_cash_flow_sheet_by_report_delisted_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=SZ000013#/cwfx/xjllb

描述: 东方财富-股票-财务分析-现金流量表-已退市股票-按报告期

限量: 单次获取指定 symbol 的现金流量表-按报告期数据

输入参数

| 名称     | 类型  | 描述                               |
|--------|-----|----------------------------------|
| symbol | str | symbol="SZ000013"; 带市场标识的已退市股票代码 |

输出参数

| 名称  | 类型  | 描述          |
|-----|-----|-------------|
| -   | -   | 252 项，不逐一列出 |

接口示例

```python
import akshare as ak

stock_cash_flow_sheet_by_report_delisted_em_df = ak.stock_cash_flow_sheet_by_report_delisted_em(symbol="SZ000013")
print(stock_cash_flow_sheet_by_report_delisted_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... MINORITY_INTEREST MINORITY_INTEREST_YOY
0   000013.SZ        000013  ...       18337254.13                   NaN
1   000013.SZ        000013  ...       14847516.38            -19.030863
2   000013.SZ        000013  ...        9677540.34                   NaN
3   000013.SZ        000013  ...       23357763.71             57.317649
4   000013.SZ        000013  ...        9262252.60             -4.291253
5   000013.SZ        000013  ...       -3961320.04           -116.959329
6   000013.SZ        000013  ...       -1234838.91           -113.331950
7   000013.SZ        000013  ...      -35258015.13           -790.057223
8   000013.SZ        000013  ...        -179278.95                   NaN
9   000013.SZ        000013  ...       -4739838.45           -283.842654
10  000013.SZ        000013  ...      -28385113.99                   NaN
11  000013.SZ        000013  ...               NaN                   NaN
12  000013.SZ        000013  ...               NaN                   NaN
13  000013.SZ        000013  ...               NaN                   NaN
14  000013.SZ        000013  ...               NaN                   NaN
15  000013.SZ        000013  ...               NaN                   NaN
16  000013.SZ        000013  ...               NaN                   NaN
17  000013.SZ        000013  ...               NaN                   NaN
18  000013.SZ        000013  ...               NaN                   NaN
19  000013.SZ        000013  ...               NaN                   NaN
20  000013.SZ        000013  ...               NaN                   NaN
[21 rows x 252 columns]
```

#### 港股财务报表

接口: stock_financial_hk_report_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/FinancialAnalysis/index?type=web&code=00700

描述: 东方财富-港股-财务报表-三大报表

限量: 单次获取指定股票、指定报告且指定报告期的数据

输入参数

| 名称        | 类型  | 描述                                                  |
|-----------|-----|-----------------------------------------------------|
| stock     | str | stock="00700"; 股票代码                                 |
| symbol    | str | symbol="现金流量表"; choice of {"资产负债表", "利润表", "现金流量表"} |
| indicator | str | indicator="年度"; choice of {"年度", "报告期"}             |

输出参数

| 名称                 | 类型      | 描述 |
|--------------------|---------|----|
| SECUCODE           | object  | -  |
| SECURITY_CODE      | object  | -  |
| SECURITY_NAME_ABBR | object  | -  |
| ORG_CODE           | object  | -  |
| REPORT_DATE        | object  | -  |
| DATE_TYPE_CODE     | object  | -  |
| FISCAL_YEAR        | object  | -  |
| STD_ITEM_CODE      | object  | -  |
| STD_ITEM_NAME      | object  | -  |
| AMOUNT             | float64 | -  |
| STD_REPORT_DATE    | object  | -  |

```python
import akshare as ak

stock_financial_hk_report_em_df = ak.stock_financial_hk_report_em(stock="00700", symbol="资产负债表", indicator="年度")
print(stock_financial_hk_report_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ...        AMOUNT      STD_REPORT_DATE
0    00700.HK         00700  ...  5.397800e+10  2022-12-31 00:00:00
1    00700.HK         00700  ...  5.590000e+08  2022-12-31 00:00:00
2    00700.HK         00700  ...  1.618020e+11  2022-12-31 00:00:00
3    00700.HK         00700  ...  1.804600e+10  2022-12-31 00:00:00
4    00700.HK         00700  ...  9.229000e+09  2022-12-31 00:00:00
..        ...           ...  ...           ...                  ...
965  00700.HK         00700  ...  4.817800e+07  2001-12-31 00:00:00
966  00700.HK         00700  ...  4.832400e+07  2001-12-31 00:00:00
967  00700.HK         00700  ...  4.832400e+07  2001-12-31 00:00:00
968  00700.HK         00700  ...  4.832400e+07  2001-12-31 00:00:00
969  00700.HK         00700  ...  6.554200e+07  2001-12-31 00:00:00
[970 rows x 11 columns]
```

#### 美股财务报表

接口: stock_financial_us_report_em

目标地址: https://emweb.eastmoney.com/PC_USF10/pages/index.html?code=TSLA&type=web&color=w#/cwfx/zyzb

描述: 东方财富-美股-财务分析-三大报表

限量: 单次获取指定股票、指定报告且指定报告期的数据

输入参数

| 名称        | 类型  | 描述                                                    |
|-----------|-----|-------------------------------------------------------|
| stock     | str | stock="TSLA"; 股票代码, 比如 BRK.A 需修改为 BRK_A 再获取           |
| symbol    | str | symbol="资产负债表"; choice of {"资产负债表", "综合损益表", "现金流量表"} |
| indicator | str | indicator="年报"; choice of {"年报", "单季报", "累计季报"}       |

输出参数

| 名称                 | 类型      | 描述 |
|--------------------|---------|----|
| SECUCODE           | object  | -  |
| SECURITY_CODE      | object  | -  |
| SECURITY_NAME_ABBR | object  | -  |
| REPORT_DATE        | object  | -  |
| REPORT_TYPE        | object  | -  |
| REPORT             | object  | -  |
| STD_ITEM_CODE      | object  | -  |
| AMOUNT             | float64 | -  |
| ITEM_NAME          | object  | -  |


```python
import akshare as ak

stock_financial_us_report_em_df = ak.stock_financial_us_report_em(stock="TSLA", symbol="资产负债表", indicator="年报")
print(stock_financial_us_report_em_df)
```

数据示例

```
    SECUCODE SECURITY_CODE  ...        AMOUNT ITEM_NAME
0     TSLA.O          TSLA  ...  1.613900e+10  现金及现金等价物
1     TSLA.O          TSLA  ...  1.639800e+10  现金及现金等价物
2     TSLA.O          TSLA  ...  1.625300e+10  现金及现金等价物
3     TSLA.O          TSLA  ...  1.757600e+10  现金及现金等价物
4     TSLA.O          TSLA  ...  1.938400e+10  现金及现金等价物
..       ...           ...  ...           ...       ...
619   TSLA.O          TSLA  ...  3.670390e+08     非运算项目
620   TSLA.O          TSLA  ...           NaN     非运算项目
621   TSLA.O          TSLA  ...  3.192250e+08     非运算项目
622   TSLA.O          TSLA  ...  1.011780e+08     非运算项目
623   TSLA.O          TSLA  ...  1.011780e+08     非运算项目
[624 rows x 9 columns]
```

#### 关键指标-新浪

接口: stock_financial_abstract

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vFD_FinanceSummary/stockid/600004.phtml

描述: 新浪财经-财务报表-关键指标

限量: 单次获取关键指标所有历史数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="600004"; 股票代码 |

输出参数

| 名称       | 类型     | 描述  |
|----------|--------|-----|
| 选项       | object | -   |
| 指标       | object | -   |
| 【具体的报告期】 | object | -   |

接口示例

```python
import akshare as ak

stock_financial_abstract_df = ak.stock_financial_abstract(symbol="600004")
print(stock_financial_abstract_df)
```

数据示例

```
      选项        指标      20220930  ...      20020630      20011231      20001231
0   常用指标     归母净利润 -6.271849e+08  ...  1.365679e+08  2.967902e+08  1.967577e+08
1   常用指标     营业总收入  3.307660e+09  ...  4.381153e+08  8.515877e+08  7.833500e+08
2   常用指标      营业成本  4.212766e+09  ...  2.664150e+08  5.317826e+08  4.654766e+08
3   常用指标       净利润 -6.321978e+08  ...  1.406112e+08  3.009773e+08  2.019293e+08
4   常用指标     扣非净利润 -6.532030e+08  ...  1.364059e+08  1.985243e+08  1.967577e+08
..   ...       ...           ...  ...           ...           ...           ...
74  营运能力    总资产周转率  1.202050e-01  ...           NaN  6.501820e-01           NaN
75  营运能力   总资产周转天数  2.246150e+03  ...           NaN  5.536909e+02           NaN
76  营运能力   流动资产周转率  8.849150e-01  ...           NaN  9.480880e-01           NaN
77  营运能力  流动资产周转天数  3.051139e+02  ...           NaN  3.797112e+02           NaN
78  营运能力   应付账款周转率  5.311671e+00  ...           NaN  2.140125e+01           NaN
```

#### 关键指标-同花顺

接口: stock_financial_abstract_new_ths

目标地址: https://basic.10jqka.com.cn/new/000063/finance.html

描述: 同花顺-财务指标-重要指标；替换 stock_financial_abstract_ths 接口

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称        | 类型  | 描述                                                                      |
|-----------|-----|-------------------------------------------------------------------------|
| symbol    | str | symbol="000063"; 股票代码                                                   |
| indicator | str | indicator="按报告期"; choice of {"按报告期", "一季度", "二季度", "三季度", "四季度", "按年度"} |

输出参数

| 名称            | 类型      | 描述 |
|---------------|---------|----|
| report_date   | object  | -  |
| report_name   | object  | -  |
| report_period | object  | -  |
| quarter_name  | object  | -  |
| metric_name   | object  | -  |
| value         | float64 | -  |
| single        | object  | -  |
| yoy           | float64 | -  |
| mom           | object  | -  |
| single_yoy    | object  | -  |

接口示例

```python
import akshare as ak

stock_financial_abstract_new_ths_df = ak.stock_financial_abstract_new_ths(symbol="000063", indicator="按报告期")
print(stock_financial_abstract_new_ths_df)
```

数据示例

```
     report_date report_name  ...           mom   single_yoy
0     2025-09-30     2025三季报  ...    1.86837607  -0.54398564
1     2025-09-30     2025三季报  ...  -14.45035019  -8.51550645
2     2025-09-30     2025三季报  ...   -0.89847408  -0.87837202
3     2025-09-30     2025三季报  ...          <NA>         <NA>
4     2025-09-30     2025三季报  ...          <NA>         <NA>
...          ...         ...  ...           ...          ...
1195  2013-06-30      2013中报  ...   -0.49016497   0.25253313
1196  2013-06-30      2013中报  ...   -0.67334254   1.13728066
1197  2013-06-30      2013中报  ...          <NA>         <NA>
1198  2013-06-30      2013中报  ...    0.77759490   0.07845025
1199  2013-06-30      2013中报  ...   -0.49473684   0.26315789
[1200 rows x 10 columns]
```

#### 主要指标-东方财富

接口: stock_financial_analysis_indicator_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html?type=web&code=SZ301389&color=b#/cwfx

描述: 东方财富-A股-财务分析-主要指标

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称        | 类型  | 描述                                           |
|-----------|-----|----------------------------------------------|
| symbol    | str | symbol="301389.SZ"; 股票代码                     |
| indicator | str | indicator="按报告期"; choice of {"按报告期", "按单季度"} |

输出参数

| 名称                 | 类型      | 描述               |
|--------------------|---------|------------------|
| SECUCODE           | object  | 股票代码(带后缀)        |
| SECURITY_CODE      | object  | 股票代码             |
| SECURITY_NAME_ABBR | object  | 股票名称             |
| REPORT_DATE        | object  | 报告日期             |
| REPORT_TYPE        | object  | 报告类型             |
| REPORT_DATE_NAME   | object  | 报告日期名称           |
| EPSJB              | float64 | 基本每股收益(元)        |
| EPSKCJB            | float64 | 扣非每股收益(元)        |
| EPSXS              | float64 | 稀释每股收益(元)        |
| BPS                | float64 | 每股净资产(元)         |
| MGZBGJ             | float64 | 每股公积金(元)         |
| MGWFPLR            | float64 | 每股未分配利润(元)       |
| MGJYXJJE           | float64 | 每股经营现金流(元)       |
| TOTALOPERATEREVE   | float64 | 营业总收入(元)         |
| MLR                | float64 | 毛利润(元)           |
| PARENTNETPROFIT    | float64 | 归属净利润(元)         |
| KCFJCXSYJLR        | float64 | 扣非净利润(元)         |
| TOTALOPERATEREVETZ | float64 | 营业总收入同比增长(%)     |
| PARENTNETPROFITTZ  | float64 | 归属净利润同比增长(%)     |
| KCFJCXSYJLRTZ      | float64 | 扣非净利润同比增长(%)     |
| YYZSRGDHBZC        | float64 | 营业总收入滚动环比增长(%)   |
| NETPROFITRPHBZC    | float64 | 归属净利润滚动环比增长(%)   |
| KFJLRGDHBZC        | float64 | 扣非净利润滚动环比增长(%)   |
| ROEJQ              | float64 | 净资产收益率(加权)(%)    |
| ROEKCJQ            | float64 | 净资产收益率(扣非/加权)(%) |
| ZZCJLL             | float64 | 总资产收益率(加权)(%)    |
| XSJLL              | float64 | 净利率(%)           |
| XSMLL              | float64 | 毛利率(%)           |
| YSZKYYSR           | float64 | 预收账款/营业收入        |
| XSJXLYYSR          | float64 | 销售净现金流/营业收入      |
| JYXJLYYSR          | float64 | 经营净现金流/营业收入      |
| TAXRATE            | float64 | 实际税率(%)          |
| LD                 | float64 | 流动比率             |
| SD                 | float64 | 速动比率             |
| XJLLB              | float64 | 现金流量比率           |
| ZCFZL              | float64 | 资产负债率(%)         |
| QYCS               | float64 | 权益系数             |
| CQBL               | float64 | 产权比率             |
| ZZCZZTS            | float64 | 总资产周转天数(天)       |
| CHZZTS             | float64 | 存货周转天数(天)        |
| YSZKZZTS           | float64 | 应收账款周转天数(天)      |
| TOAZZL             | float64 | 总资产周转率(次)        |
| CHZZL              | float64 | 存货周转率(次)         |
| YSZKZZL            | float64 | 应收账款周转率(次)       |
| ...                | ...     | ...              |

接口示例

```python
import akshare as ak

stock_financial_analysis_indicator_em_df = ak.stock_financial_analysis_indicator_em(symbol="301389.SZ", indicator="按报告期")
print(stock_financial_analysis_indicator_em_df)
```

数据示例

```
     SECUCODE SECURITY_CODE  ... NET_ASSETS_LIABILITIES PROPRIETARY_CAPITAL
0   301389.SZ        301389  ...                   None                None
1   301389.SZ        301389  ...                   None                None
2   301389.SZ        301389  ...                   None                None
3   301389.SZ        301389  ...                   None                None
4   301389.SZ        301389  ...                   None                None
5   301389.SZ        301389  ...                   None                None
6   301389.SZ        301389  ...                   None                None
7   301389.SZ        301389  ...                   None                None
8   301389.SZ        301389  ...                   None                None
9   301389.SZ        301389  ...                   None                None
10  301389.SZ        301389  ...                   None                None
11  301389.SZ        301389  ...                   None                None
12  301389.SZ        301389  ...                   None                None
13  301389.SZ        301389  ...                   None                None
14  301389.SZ        301389  ...                   None                None
15  301389.SZ        301389  ...                   None                None
16  301389.SZ        301389  ...                   None                None
17  301389.SZ        301389  ...                   None                None
18  301389.SZ        301389  ...                   None                None
19  301389.SZ        301389  ...                   None                None
[20 rows x 140 columns]
```

#### 财务指标

接口: stock_financial_analysis_indicator

目标地址: https://money.finance.sina.com.cn/corp/go.php/vFD_FinancialGuideLine/stockid/600004/ctrl/2019/displaytype/4.phtml

描述: 新浪财经-财务分析-财务指标

限量: 单次获取指定 symbol 和 start_year 的所有财务指标历史数据

输入参数

| 名称         | 类型  | 描述                         |
|------------|-----|----------------------------|
| symbol     | str | symbol="600004"; 股票代码      |
| start_year | str | start_year="2020"; 开始查询的时间 |

输出参数

| 名称                | 类型      | 描述 |
|-------------------|---------|----|
| 日期                | object  | -  |
| 摊薄每股收益(元)         | float64 | -  |
| 加权每股收益(元)         | float64 | -  |
| 每股收益_调整后(元)       | float64 | -  |
| 扣除非经常性损益后的每股收益(元) | float64 | -  |
| 每股净资产_调整前(元)      | float64 | -  |
| 每股净资产_调整后(元)      | float64 | -  |
| 每股经营性现金流(元)       | float64 | -  |
| 每股资本公积金(元)        | float64 | -  |
| 每股未分配利润(元)        | float64 | -  |
| 调整后的每股净资产(元)      | float64 | -  |
| 总资产利润率(%)         | float64 | -  |
| 主营业务利润率(%)        | float64 | -  |
| 总资产净利润率(%)        | float64 | -  |
| 成本费用利润率(%)        | float64 | -  |
| 营业利润率(%)          | float64 | -  |
| 主营业务成本率(%)        | float64 | -  |
| 销售净利率(%)          | float64 | -  |
| 股本报酬率(%)          | float64 | -  |
| 净资产报酬率(%)         | float64 | -  |
| 资产报酬率(%)          | float64 | -  |
| 销售毛利率(%)          | float64 | -  |
| 三项费用比重            | float64 | -  |
| 非主营比重             | float64 | -  |
| 主营利润比重            | float64 | -  |
| 股息发放率(%)          | float64 | -  |
| 投资收益率(%)          | float64 | -  |
| 主营业务利润(元)         | float64 | -  |
| 净资产收益率(%)         | float64 | -  |
| 加权净资产收益率(%)       | float64 | -  |
| 扣除非经常性损益后的净利润(元)  | float64 | -  |
| 主营业务收入增长率(%)      | float64 | -  |
| 净利润增长率(%)         | float64 | -  |
| 净资产增长率(%)         | float64 | -  |
| 总资产增长率(%)         | float64 | -  |
| 应收账款周转率(次)        | float64 | -  |
| 应收账款周转天数(天)       | float64 | -  |
| 存货周转天数(天)         | float64 | -  |
| 存货周转率(次)          | float64 | -  |
| 固定资产周转率(次)        | float64 | -  |
| 总资产周转率(次)         | float64 | -  |
| 总资产周转天数(天)        | float64 | -  |
| 流动资产周转率(次)        | float64 | -  |
| 流动资产周转天数(天)       | float64 | -  |
| 股东权益周转率(次)        | float64 | -  |
| 流动比率              | float64 | -  |
| 速动比率              | float64 | -  |
| 现金比率(%)           | float64 | -  |
| 利息支付倍数            | float64 | -  |
| 长期债务与营运资金比率(%)    | float64 | -  |
| 股东权益比率(%)         | float64 | -  |
| 长期负债比率(%)         | float64 | -  |
| 股东权益与固定资产比率(%)    | float64 | -  |
| 负债与所有者权益比率(%)     | float64 | -  |
| 长期资产与长期资金比率(%)    | float64 | -  |
| 资本化比率(%)          | float64 | -  |
| 固定资产净值率(%)        | float64 | -  |
| 资本固定化比率(%)        | float64 | -  |
| 产权比率(%)           | float64 | -  |
| 清算价值比率(%)         | float64 | -  |
| 固定资产比重(%)         | float64 | -  |
| 资产负债率(%)          | float64 | -  |
| 总资产(元)            | float64 | -  |
| 经营现金净流量对销售收入比率(%) | float64 | -  |
| 资产的经营现金流量回报率(%)   | float64 | -  |
| 经营现金净流量与净利润的比率(%) | float64 | -  |
| 经营现金净流量对负债比率(%)   | float64 | -  |
| 现金流量比率(%)         | float64 | -  |
| 短期股票投资(元)         | float64 | -  |
| 短期债券投资(元)         | float64 | -  |
| 短期其它经营性投资(元)      | float64 | -  |
| 长期股票投资(元)         | float64 | -  |
| 长期债券投资(元)         | float64 | -  |
| 长期其它经营性投资(元)      | float64 | -  |
| 1年以内应收帐款(元)       | float64 | -  |
| 1-2年以内应收帐款(元)     | float64 | -  |
| 2-3年以内应收帐款(元)     | float64 | -  |
| 3年以内应收帐款(元)       | float64 | -  |
| 1年以内预付货款(元)       | float64 | -  |
| 1-2年以内预付货款(元)     | float64 | -  |
| 2-3年以内预付货款(元)     | float64 | -  |
| 3年以内预付货款(元)       | float64 | -  |
| 1年以内其它应收款(元)      | float64 | -  |
| 1-2年以内其它应收款(元)    | float64 | -  |
| 2-3年以内其它应收款(元)    | float64 | -  |
| 3年以内其它应收款(元)      | float64 | -  |

接口示例

```python
import akshare as ak

stock_financial_analysis_indicator_df = ak.stock_financial_analysis_indicator(symbol="600004", start_year="2020")
print(stock_financial_analysis_indicator_df)
```

数据示例

```
         日期  摊薄每股收益(元)  ... 2-3年以内其它应收款(元) 3年以内其它应收款(元)
0   2020-03-31    -0.0307  ...             NaN           NaN
1   2020-06-30    -0.0816  ...      1189862.00           NaN
2   2020-09-30    -0.1380  ...             NaN           NaN
3   2020-12-31    -0.0980  ...      1495234.99           NaN
4   2021-03-31    -0.0645  ...             NaN           NaN
5   2021-06-30    -0.1686  ...      3471186.42           NaN
6   2021-09-30    -0.2038  ...             NaN           NaN
7   2021-12-31    -0.1628  ...      1380992.96           NaN
8   2022-03-31    -0.0326  ...             NaN           NaN
9   2022-06-30    -0.2242  ...      1680204.08           NaN
10  2022-09-30    -0.2671  ...             NaN           NaN
11  2022-12-31    -0.4613  ...      2459538.50           NaN
12  2023-03-31     0.0216  ...             NaN           NaN
13  2023-06-30     0.0720  ...      2591827.74           NaN
14  2023-09-30     0.1232  ...             NaN           NaN
15  2023-12-31     0.2032  ...      7162683.42           NaN
16  2024-03-31     0.0841  ...             NaN           NaN
[17 rows x 86 columns]
```

#### 港股财务指标

接口: stock_financial_hk_analysis_indicator_em

目标地址: https://emweb.securities.eastmoney.com/PC_HKF10/NewFinancialAnalysis/index?type=web&code=00700

描述: 东方财富-港股-财务分析-主要指标

限量: 单次获取财务指标所有历史数据

输入参数

| 名称        | 类型  | 描述                                      |
|-----------|-----|-----------------------------------------|
| symbol    | str | symbol="00700"; 股票代码                    |
| indicator | str | indicator="年度"; choice of {"年度", "报告期"} |

输出参数

| 名称                  | 类型      | 描述             |
|---------------------|---------|----------------|
| SECUCODE            | object  | 股票代码(带HK后缀)    |
| SECURITY_CODE       | object  | 股票代码(不带HK后缀)   |
| SECURITY_NAME_ABBR  | object  | 股票名称           |
| ORG_CODE            | object  | ORG_CODE       |
| REPORT_DATE         | object  | 报告日期           |
| DATE_TYPE_CODE      | object  | 报告日期类型         |
| PER_NETCASH_OPERATE | float64 | 每股经营现金流(元)     |
| PER_OI              | float64 | 每股营业收入(元)      |
| BPS                 | float64 | 每股净资产(元)       |
| BASIC_EPS           | float64 | 基本每股收益(元)      |
| DILUTED_EPS         | float64 | 稀释每股收益(元)      |
| OPERATE_INCOME      | int64   | 营业总收入(元)       |
| OPERATE_INCOME_YOY  | float64 | 营业总收入同比增长(%)   |
| GROSS_PROFIT        | int64   | 毛利润(元)         |
| GROSS_PROFIT_YOY    | float64 | 毛利润同比增长(%)     |
| HOLDER_PROFIT       | int64   | 归母净利润(元)       |
| HOLDER_PROFIT_YOY   | float64 | 归母净利润同比增长(%)   |
| GROSS_PROFIT_RATIO  | float64 | 毛利率(%)         |
| EPS_TTM             | float64 | TTM每股收益(元)     |
| OPERATE_INCOME_QOQ  | float64 | 营业总收入滚动环比增长(%) |
| NET_PROFIT_RATIO    | float64 | 净利率(%)         |
| ROE_AVG             | float64 | 平均净资产收益率(%)    |
| GROSS_PROFIT_QOQ    | float64 | 毛利润滚动环比增长(%)   |
| ROA                 | float64 | 总资产净利率(%)      |
| HOLDER_PROFIT_QOQ   | float64 | 归母净利润滚动环比增长(%) |
| ROE_YEARLY          | float64 | 年化净资产收益率(%)    |
| ROIC_YEARLY         | float64 | 年化投资回报率(%)     |
| TAX_EBT             | float64 | 所得税/利润总额(%)    |
| OCF_SALES           | float64 | 经营现金流/营业收入(%)  |
| DEBT_ASSET_RATIO    | float64 | 资产负债率(%)       |
| CURRENT_RATIO       | float64 | 流动比率(倍)        |
| CURRENTDEBT_DEBT    | float64 | 流动负债/总负债(%)    |
| START_DATE          | object  | START_DATE     |
| FISCAL_YEAR         | object  | 年结日            |
| CURRENCY            | object  | CURRENCY       |
| IS_CNY_CODE         | int64   | IS_CNY_CODE    |

接口示例

```python
import akshare as ak

stock_financial_hk_analysis_indicator_em_df = ak.stock_financial_hk_analysis_indicator_em(symbol="00700", indicator="年度")
print(stock_financial_hk_analysis_indicator_em_df)
```

数据示例

```
   SECUCODE SECURITY_CODE SECURITY_NAME_ABBR  ... FISCAL_YEAR CURRENCY IS_CNY_CODE
0  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
1  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
2  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
3  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
4  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
5  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
6  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
7  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
8  00700.HK         00700               腾讯控股  ...       12-31      HKD           0
[9 rows x 36 columns]
```

#### 美股财务指标

接口: stock_financial_us_analysis_indicator_em

目标地址: https://emweb.eastmoney.com/PC_USF10/pages/index.html?code=TSLA&type=web&color=w#/cwfx/zyzb

描述: 东方财富-美股-财务分析-主要指标

限量: 单次获取指定股票的所有历史数据

输入参数

| 名称        | 类型  | 描述                                              |
|-----------|-----|-------------------------------------------------|
| symbol    | str | symbol="TSLA"; 股票代码                             |
| indicator | str | indicator="年报"; choice of {"年报", "单季报", "累计季报"} |

输出参数

| 名称                          | 类型      | 描述 |
|-----------------------------|---------|----|
| SECUCODE                    | object  | -  |
| SECURITY_CODE               | object  | -  |
| SECURITY_NAME_ABBR          | object  | -  |
| ORG_CODE                    | object  | -  |
| SECURITY_INNER_CODE         | object  | -  |
| ACCOUNTING_STANDARDS        | object  | -  |
| NOTICE_DATE                 | object  | -  |
| START_DATE                  | object  | -  |
| REPORT_DATE                 | object  | -  |
| FINANCIAL_DATE              | object  | -  |
| STD_REPORT_DATE             | object  | -  |
| CURRENCY                    | object  | -  |
| DATE_TYPE                   | object  | -  |
| DATE_TYPE_CODE              | object  | -  |
| REPORT_TYPE                 | object  | -  |
| REPORT_DATA_TYPE            | object  | -  |
| ORGTYPE                     | object  | -  |
| OPERATE_INCOME              | float64 | -  |
| OPERATE_INCOME_YOY          | float64 | -  |
| GROSS_PROFIT                | float64 | -  |
| GROSS_PROFIT_YOY            | float64 | -  |
| PARENT_HOLDER_NETPROFIT     | int64   | -  |
| PARENT_HOLDER_NETPROFIT_YOY | float64 | -  |
| BASIC_EPS                   | float64 | -  |
| DILUTED_EPS                 | float64 | -  |
| GROSS_PROFIT_RATIO          | float64 | -  |
| NET_PROFIT_RATIO            | float64 | -  |
| ACCOUNTS_RECE_TR            | float64 | -  |
| INVENTORY_TR                | float64 | -  |
| TOTAL_ASSETS_TR             | float64 | -  |
| ACCOUNTS_RECE_TDAYS         | float64 | -  |
| INVENTORY_TDAYS             | float64 | -  |
| TOTAL_ASSETS_TDAYS          | float64 | -  |
| ROE_AVG                     | float64 | -  |
| ROA                         | float64 | -  |
| CURRENT_RATIO               | float64 | -  |
| SPEED_RATIO                 | float64 | -  |
| OCF_LIQDEBT                 | float64 | -  |
| DEBT_ASSET_RATIO            | float64 | -  |
| EQUITY_RATIO                | float64 | -  |
| BASIC_EPS_YOY               | float64 | -  |
| GROSS_PROFIT_RATIO_YOY      | float64 | -  |
| NET_PROFIT_RATIO_YOY        | float64 | -  |
| ROE_AVG_YOY                 | float64 | -  |
| ROA_YOY                     | float64 | -  |
| DEBT_ASSET_RATIO_YOY        | float64 | -  |
| CURRENT_RATIO_YOY           | float64 | -  |
| SPEED_RATIO_YOY             | float64 | -  |

接口示例

```python
import akshare as ak

stock_financial_us_analysis_indicator_em_df = ak.stock_financial_us_analysis_indicator_em(symbol="TSLA", indicator="年报")
print(stock_financial_us_analysis_indicator_em_df)
```

数据示例

```
   SECUCODE SECURITY_CODE  ... CURRENT_RATIO_YOY SPEED_RATIO_YOY
0    TSLA.O          TSLA  ...         17.325422       28.440175
1    TSLA.O          TSLA  ...         12.659536       19.087360
2    TSLA.O          TSLA  ...         11.391821       -2.942407
3    TSLA.O          TSLA  ...        -26.656933      -31.763438
4    TSLA.O          TSLA  ...         65.265821       98.010070
5    TSLA.O          TSLA  ...         36.490497       54.229892
6    TSLA.O          TSLA  ...         -2.902445       -7.382595
7    TSLA.O          TSLA  ...        -20.306070      -21.998647
8    TSLA.O          TSLA  ...          8.548288       34.456320
9    TSLA.O          TSLA  ...        -34.422709      -49.356229
10   TSLA.O          TSLA  ...        -19.511791      -22.928303
11   TSLA.O          TSLA  ...         92.625823      188.401364
12   TSLA.O          TSLA  ...        -50.045468      -71.819999
13   TSLA.O          TSLA  ...        -29.317702      -24.315460
14   TSLA.O          TSLA  ...         57.604938       65.676565
15   TSLA.O          TSLA  ...        389.435012      700.530829
16   TSLA.O          TSLA  ...        -17.820400      -57.326660
17   TSLA.O          TSLA  ...               NaN             NaN
18   TSLA.O          TSLA  ...               NaN             NaN
[19 rows x 48 columns]
```

#### 历史分红

接口: stock_history_dividend

目标地址: http://vip.stock.finance.sina.com.cn/q/go.php/vInvestConsult/kind/lsfh/index.phtml

描述: 新浪财经-发行与分配-历史分红

限量: 单次获取所有股票的历史分红数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 代码   | object  | -       |
| 名称   | object  | -       |
| 上市日期 | object  | -       |
| 累计股息 | float64 | 注意单位: % |
| 年均股息 | float64 | 注意单位: % |
| 分红次数 | float64 | -       |
| 融资总额 | float64 | 注意单位: 亿 |
| 融资次数 | float64 | -       |

接口示例

```python
import akshare as ak

stock_history_dividend_df = ak.stock_history_dividend()
print(stock_history_dividend_df)
```

数据示例

```
     代码      名称     上市日期   累计股息  年均股息 分红次数 融资总额 融资次数
0     000550  江铃汽车  1993-12-01  213.0  6.87    51   0.0000     0
1     000541  佛山照明  1993-11-23  192.0  6.19    56  10.8842     1
2     000429  粤高速A  1998-02-20  171.4  6.35    50  16.3350     1
3     000726   鲁泰A  2000-12-25  143.9  6.00    47   9.5082     1
4     000581  威孚高科  1998-09-24  143.1  5.50    48  28.5012     1
...      ...   ...         ...    ...   ...   ...      ...   ...
5556  920002  万达轴承  2024-05-30    0.0  0.00     0   1.0235     1
5557  920008  成电光信  2024-08-29    0.0  0.00     0   0.8949     1
5558  920016  中草香料  2024-09-13    0.0  0.00     0   1.1465     1
5559  920099  瑞华技术  2024-09-25    0.0  0.00     0   0.0000     0
5560  920118  太湖远大  2024-08-22    0.0  0.00     0   0.0000     0
[5561 rows x 8 columns]
```

#### 十大流通股东(个股)

接口: stock_gdfx_free_top_10_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/ShareholderResearch/Index?type=web&code=SH688686#sdltgd-0

描述: 东方财富网-个股-十大流通股东

限量: 单次返回指定 symbol 和 date 的所有数据

输入参数

| 名称     | 类型  | 描述                            |
|--------|-----|-------------------------------|
| symbol | str | symbol="sh688686"; 带市场标识的股票代码 |
| date   | str | date="20240930"; 财报发布季度最后日    |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 名次         | int64   | -       |
| 股东名称       | object  | -       |
| 股东性质       | object  | -       |
| 股份类型       | object  | -       |
| 持股数        | int64   | 注意单位: 股 |
| 占总流通股本持股比例 | float64 | 注意单位: % |
| 增减         | object  | 注意单位: 股 |
| 变动比率       | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gdfx_free_top_10_em_df = ak.stock_gdfx_free_top_10_em(symbol="sh688686", date="20240930")
print(stock_gdfx_free_top_10_em_df)
```

数据示例

```
   名次                               股东名称  ...        增减       变动比率
0   1                                 卢治临  ...        不变        NaN
1   2                                 卢盛林  ...        不变        NaN
2   3                                 许学亮  ...        不变        NaN
3   4                  宁波千智创业投资合伙企业(有限合伙)  ...        不变        NaN
4   5                          香港中央结算有限公司  ...  -1132820 -25.435429
5   6  国信证券-招商银行-国信证券鼎信10号科创板战略配售集合资产管理计划  ...        不变        NaN
6   7        摩根资产管理(新加坡)有限公司-摩根中国A股市场机会基金  ...        不变        NaN
7   8                                 李茂波  ...        不变        NaN
8   9     招商银行股份有限公司-惠升惠泽灵活配置混合型发起式证券投资基金  ...        不变        NaN
9  10         MERRILL LYNCH INTERNATIONAL  ...        新进        NaN
[10 rows x 8 columns]
```

#### 十大股东(个股)

接口: stock_gdfx_top_10_em

目标地址: https://emweb.securities.eastmoney.com/PC_HSF10/ShareholderResearch/Index?type=web&code=SH688686#sdltgd-0

描述: 东方财富网-个股-十大股东

限量: 单次返回指定 symbol 和 date 的所有数据

输入参数

| 名称     | 类型  | 描述                            |
|--------|-----|-------------------------------|
| symbol | str | symbol="sh688686"; 带市场标识的股票代码 |
| date   | str | date="20210630"; 财报发布季度最后日    |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 名次       | int64   | -       |
| 股东名称     | object  | -       |
| 股份类型     | object  | -       |
| 持股数      | int64   | 注意单位: 股 |
| 占总股本持股比例 | float64 | 注意单位: % |
| 增减       | object  | 注意单位: 股 |
| 变动比率     | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gdfx_top_10_em_df = ak.stock_gdfx_top_10_em(symbol="sh688686", date="20210630")
print(stock_gdfx_top_10_em_df)
```

数据示例

```
   名次                                        股东名称  ...       增减       变动比率
0   1                                         卢治临  ...       不变        NaN
1   2                                         卢盛林  ...       不变        NaN
2   3                                         许学亮  ...       不变        NaN
3   4                          东莞千智股权投资合伙企业(有限合伙)  ...       不变        NaN
4   5                  高华-汇丰-GOLDMAN,SACHS&CO.LLC  ...   946997  54.311934
5   6            铭基国际投资公司-MATTHEWS ASIA FUNDS(US)  ...  -358837 -15.460528
6   7     MORGAN STANLEY & CO. INTERNATIONAL PLC.  ...    70003   4.082755
7   8  宁波梅山保税港区晨道投资合伙企业-长江晨道(湖北)新能源产业投资合伙企业(有限合伙)  ...       不变        NaN
8   9          国信证券-招商银行-国信证券鼎信10号科创板战略配售集合资产管理计划  ...  -639100 -30.994180
9  10                      FMR 有限公司-富达新兴市场基金(交易所)  ...       新进        NaN
```

#### 股东持股变动统计-十大流通股东

接口: stock_gdfx_free_holding_change_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股变动统计-十大流通股东

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 股东名称         | object  | -       |
| 股东类型         | object  | -       |
| 期末持股只数统计-总持有 | float64 | -       |
| 期末持股只数统计-新进  | float64 | -       |
| 期末持股只数统计-增加  | float64 | -       |
| 期末持股只数统计-不变  | float64 | -       |
| 期末持股只数统计-减少  | float64 | -       |
| 流通市值统计       | float64 | 注意单位: 元 |
| 持有个股         | object  | -       |

接口示例

```python
import akshare as ak

stock_gdfx_free_holding_change_em_df = ak.stock_gdfx_free_holding_change_em(date="20210930")
print(stock_gdfx_free_holding_change_em_df)
```

数据示例

```
          序号  ...                                               持有个股
0          1  ...  603218|日月股份,600988|赤峰黄金,600298|安琪酵母,300232|洲明科...
1          2  ...  600519|贵州茅台,000686|东北证券,002533|金杯电工,002066|瑞泰科...
2          3  ...  600519|贵州茅台,300015|爱尔眼科,600674|川投能源,600009|上海机...
3          4  ...  000026|飞亚达,200026|飞亚达B,300946|恒而达,000751|锌业股份,...
4          5  ...  300005|探路者,300946|恒而达,000985|大庆华科,300927|江天化学,...
      ...  ...                                                ...
30953  30954  ...                                        002982|湘佳股份
30954  30955  ...                                        300917|特发服务
30955  30956  ...                                         688733|壹石通
30956  30957  ...                                        300969|恒帅股份
30957  30958  ...                                        002896|中大力德
```

#### 股东持股变动统计-十大股东

接口: stock_gdfx_holding_change_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股变动统计-十大股东

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 股东名称         | object  | -       |
| 股东类型         | object  | -       |
| 期末持股只数统计-总持有 | float64 | -       |
| 期末持股只数统计-新进  | float64 | -       |
| 期末持股只数统计-增加  | float64 | -       |
| 期末持股只数统计-不变  | float64 | -       |
| 期末持股只数统计-减少  | float64 | -       |
| 流通市值统计       | float64 | 注意单位: 元 |
| 持有个股         | object  | -       |

接口示例

```python
import akshare as ak

stock_gdfx_holding_change_em_df = ak.stock_gdfx_holding_change_em(date="20210930")
print(stock_gdfx_holding_change_em_df)
```

数据示例

```
          序号  ...                                               持有个股
0          1  ...  600763|通策医疗,300229|拓尔思,603039|泛微网络,300006|莱美药业...
1          2  ...  002225|濮耐股份,600435|北方导航,600422|昆药集团,600009|上海机...
2          3  ...  600340|华夏幸福,600009|上海机场,000001|平安银行,601899|紫金矿...
3          4  ...  601865|福莱特,601899|紫金矿业,000756|新华制药,600036|招商银行...
4          5  ...  002479|富春环保,300582|英飞特,300358|楚天科技,002892|科力尔,...
      ...  ...                                                ...
33896  33897  ...                                        300047|天源迪科
33897  33898  ...                                        300969|恒帅股份
33898  33899  ...                                        000425|徐工机械
33899  33900  ...                                        002896|中大力德
33900  33901  ...                            000761|本钢板材,200761|本钢板B
```

#### 高管持股变动统计

接口: stock_management_change_ths

目标地址: https://basic.10jqka.com.cn/new/688981/event.html

描述: 同花顺-公司大事-高管持股变动

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="688981"; 股票代码 |

输出参数

| 名称      | 类型     | 描述      |
|---------|--------|---------|
| 公告日期    | object | -       |
| 变动人     | object | -       |
| 与公司高管关系 | object | -       |
| 变动数量    | object | 注意单位: 股 |
| 交易均价    | object | 注意单位: 元 |
| 剩余股数    | object | 注意单位: 股 |
| 变动途径    | object | -       |

接口示例

```python
import akshare as ak

stock_management_change_ths_df = ak.stock_management_change_ths(symbol="688981")
print(stock_management_change_ths_df)
```

数据示例

```
          变动日期  变动人 与公司高管关系     ...   交易均价    剩余股数  股份变动途径
0   2022-09-01   张昕  核心技术人员    ...  20.00   9.60万    股权激励
1   2022-09-01   金达  核心技术人员    ...  20.00   4.80万    股权激励
2   2022-09-01  阎大勇  核心技术人员    ...  20.00   4.20万    股权激励
3   2022-10-10  阎大勇  核心技术人员  减持4.20万  37.98    0.00  二级市场买卖
  ...             ...             ...
10  2024-09-05  阎大勇  核心技术人员  增持3.50万  20.00   7.00万  股权激励实施
11  2024-09-05   张昕  核心技术人员  增持8.00万  20.00  16.00万  股权激励实施
12  2024-09-25   张昕  核心技术人员  减持1.80万  45.31  14.20万  二级市场买卖
13  2024-09-30   金达  核心技术人员  减持4.00万  54.87   4.00万  二级市场买卖
[13 rows x 7 columns]
```

#### 股东持股变动统计

接口: stock_shareholder_change_ths

目标地址: https://basic.10jqka.com.cn/new/688981/event.html

描述: 同花顺-公司大事-股东持股变动

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="688981"; 股票代码 |

输出参数

| 名称     | 类型     | 描述      |
|--------|--------|---------|
| 公告日期   | object | -       |
| 变动股东   | object | -       |
| 变动数量   | object | 注意单位: 股 |
| 交易均价   | object | 注意单位: 元 |
| 剩余股份总数 | object | 注意单位: 股 |
| 变动期间   | object | -       |
| 变动途径   | object | -       |

接口示例

```python
import akshare as ak

stock_shareholder_change_ths_df = ak.stock_shareholder_change_ths(symbol="688981")
print(stock_shareholder_change_ths_df)
```

数据示例

```
         公告日期                        变动股东  ...                   变动期间         变动途径
0  2021-08-11           GICPrivateLimited  ...                  2021.08.04-2021.08.04  二级市场
1  2022-12-16   华夏上证科创板50成份交易型开放式指数证券投资基金  ...  2022.12.14-2022.12.14  二级市场
2  2023-04-13   华夏上证科创板50成份交易型开放式指数证券投资基金  ...  2023.04.10-2023.04.10  二级市场
3  2023-08-31   华夏上证科创板50成份交易型开放式指数证券投资基金  ...  2023.08.29-2023.08.29  二级市场
4  2024-02-28   华夏上证科创板50成份交易型开放式指数证券投资基金  ...  2024.02.26-2024.02.26  二级市场
5  2024-10-12  易方达上证科创板50成份交易型开放式指数证券投资基金  ...  2024.10.10-2024.10.10  二级市场
6  2024-10-23  易方达上证科创板50成份交易型开放式指数证券投资基金  ...  2024.10.21-2024.10.21  二级市场
[7 rows x 7 columns]
```

#### 股东持股分析-十大流通股东

接口: stock_gdfx_free_holding_analyse_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股分析-十大流通股东

限量: 单次获取返回所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20230930"; 财报发布季度最后日 |

输出参数

| 名称             | 类型      | 描述      |
|----------------|---------|---------|
| 序号             | int64   | -       |
| 股东名称           | object  | -       |
| 股东类型           | object  | -       |
| 股票代码           | object  | -       |
| 股票简称           | object  | -       |
| 报告期            | object  | -       |
| 期末持股-数量        | float64 | 注意单位: 股 |
| 期末持股-数量变化      | float64 | 注意单位: 股 |
| 期末持股-数量变化比例    | float64 | 注意单位: % |
| 期末持股-持股变动      | float64 | -       |
| 期末持股-流通市值      | float64 | 注意单位: 元 |
| 公告日            | object  | -       |
| 公告日后涨跌幅-10个交易日 | float64 | 注意单位: % |
| 公告日后涨跌幅-30个交易日 | float64 | 注意单位: % |
| 公告日后涨跌幅-60个交易日 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gdfx_free_holding_analyse_em_df = ak.stock_gdfx_free_holding_analyse_em(date="20230930")
print(stock_gdfx_free_holding_analyse_em_df)
```

数据示例

```
          序号  ... 公告日后涨跌幅-60个交易日
0          1  ...      -0.959312
1          2  ...      -0.959312
2          3  ...      -0.959312
3          4  ...      -0.959312
4          5  ...      -0.959312
      ...  ...            ...
53269  53270  ...       2.169464
53270  53271  ...       2.169464
53271  53272  ...       2.169464
53272  53273  ...       2.169464
53273  53274  ...       2.169464
[53274 rows x 15 columns]
```

#### 股东持股分析-十大股东

接口: stock_gdfx_holding_analyse_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股分析-十大股东

限量: 单次获取返回所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称             | 类型      | 描述      |
|----------------|---------|---------|
| 序号             | int64   | -       |
| 股东名称           | object  | -       |
| 股东类型           | object  | -       |
| 股票代码           | object  | -       |
| 股票简称           | object  | -       |
| 报告期            | object  | -       |
| 期末持股-数量        | float64 | 注意单位: 股 |
| 期末持股-数量变化      | float64 | 注意单位: 股 |
| 期末持股-数量变化比例    | float64 | 注意单位: % |
| 期末持股-持股变动      | float64 | -       |
| 期末持股-流通市值      | float64 | 注意单位: 元 |
| 公告日            | object  | -       |
| 公告日后涨跌幅-10个交易日 | float64 | 注意单位: % |
| 公告日后涨跌幅-30个交易日 | float64 | 注意单位: % |
| 公告日后涨跌幅-60个交易日 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_gdfx_holding_analyse_em_df = ak.stock_gdfx_holding_analyse_em(date="20210930")
print(stock_gdfx_holding_analyse_em_df)
```

数据示例

```
          序号               股东名称  ... 公告日后涨跌幅-30个交易日 公告日后涨跌幅-60个交易日
0          1     厦门象屿资产管理运营有限公司  ...            NaN            NaN
1          2       厦门港务控股集团有限公司  ...            NaN            NaN
2          3       厦门国际会展控股有限公司  ...            NaN            NaN
3          4       厦门国贸金融控股有限公司  ...            NaN            NaN
4          5         福建奥元集团有限公司  ...            NaN            NaN
      ...                ...  ...            ...            ...
55198  55199  深圳市德益共赢投资企业(有限合伙)  ...            NaN            NaN
55199  55200                王小峰  ...            NaN            NaN
55200  55201                卢达华  ...            NaN            NaN
55201  55202      东莞市通辉模具股份有限公司  ...            NaN            NaN
55202  55203                 赵禹  ...            NaN            NaN
```

#### 股东持股明细-十大流通股东

接口: stock_gdfx_free_holding_detail_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股明细-十大流通股东

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 股东名称         | object  | -       |
| 股东类型         | object  | -       |
| 股票代码         | object  | -       |
| 股票简称         | object  | -       |
| 报告期          | object  | -       |
| 期末持股-数量      | float64 | 注意单位: 股 |
| 期末持股-数量变化    | float64 | 注意单位: 股 |
| 期末持股-数量变化比例  | float64 | 注意单位: % |
| 期末持股-持股变动    | float64 | -       |
| 期末持股-流通市值    | float64 | 注意单位: 元 |
| 公告日          | object  | -       |

接口示例

```python
import akshare as ak

stock_gdfx_free_holding_detail_em_df = ak.stock_gdfx_free_holding_detail_em(date="20210930")
print(stock_gdfx_free_holding_detail_em_df)
```

数据示例

```
     序号                                 股东名称  ...     期末持股-流通市值         公告日
0          1                         漯河银鸽实业集团有限公司  ...  2.152836e+08  2022-04-29
1          2                       永城煤电控股集团上海有限公司  ...  1.363201e+07  2022-04-29
2          3                      漯河市发展投资控股集团有限公司  ...  5.661508e+06  2022-04-29
3          4                                  黄毅明  ...  4.968781e+06  2022-04-29
4          5                                  何卫昌  ...  3.609256e+06  2022-04-29
      ...                                  ...  ...           ...         ...
46546  46547     中国工商银行股份有限公司-东方红启恒三年持有期混合型证券投资基金  ...  2.441112e+08  2021-10-12
46547  46548        招商银行股份有限公司-泓德瑞兴三年持有期混合型证券投资基金  ...  2.201811e+08  2021-10-12
46548  46549  中国工商银行股份有限公司-富国天惠精选成长混合型证券投资基金(LOF)  ...  2.132101e+08  2021-10-12
46549  46550                          全国社保基金四零二组合  ...  2.128475e+08  2021-10-12
46550  46551              中国工商银行股份有限公司-博时信用债券投资基金  ...  2.079605e+08  2021-10-12
```

#### 股东持股明细-十大股东

接口: stock_gdfx_holding_detail_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股明细-十大股东

限量: 单次返回指定参数的所有数据

输入参数

| 名称        | 类型  | 描述                                                                     |
|-----------|-----|------------------------------------------------------------------------|
| date      | str | date="20230331"; 财报发布季度最后日                                             |
| indicator | str | indicator="个人"; 股东类型; choice of {"个人", "基金", "QFII", "社保", "券商", "信托"} |
| symbol    | str | symbol="新进"; 持股变动; choice of {"新进", "增加", "不变", "减少"}                  |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int64   | -       |
| 股东名称         | object  | -       |
| 股东排名         | object  | -       |
| 股票代码         | object  | -       |
| 股票简称         | object  | -       |
| 报告期          | object  | -       |
| 期末持股-数量      | float64 | 注意单位: 股 |
| 期末持股-数量变化    | float64 | 注意单位: 股 |
| 期末持股-数量变化比例  | float64 | 注意单位: % |
| 期末持股-持股变动    | float64 | -       |
| 期末持股-流通市值    | float64 | 注意单位: 元 |
| 公告日          | object  | -       |
| 股东类型         | object  | -       |

接口示例

```python
import akshare as ak

stock_gdfx_holding_detail_em_df = ak.stock_gdfx_holding_detail_em(date="20230331", indicator="个人", symbol="新进")
print(stock_gdfx_holding_detail_em_df)
```

数据示例

```
        序号           股东名称 股东类型    股票代码  ... 期末持股-持股变动 期末持股-流通市值         公告日  股东排名
0        1            金凤霞   个人  688651  ...        新进       NaN  2023-07-21     6
1        2            周华金   个人  688651  ...        新进       NaN  2023-07-21     9
2        3            王润合   个人  688651  ...        新进       NaN  2023-07-21    10
3        4            魏春梅   个人  688651  ...        新进       NaN  2023-07-21    12
4        5            何永华   个人  688651  ...        新进       NaN  2023-07-21    14
    ...            ...  ...     ...  ...       ...       ...         ...   ...
3974  3975             黄敏   个人  873761  ...        新进       NaN  2023-03-28     3
3975  3976            姜海林   个人  873761  ...        新进       NaN  2023-03-28     4
3976  3977            廖道训   个人  873761  ...        新进       NaN  2023-03-28     5
3977  3978            李天兵   个人  873761  ...        新进       NaN  2023-03-28     6
3978  3979  LIANG AMY JUN   个人  873761  ...        新进       NaN  2023-03-28     7
[3979 rows x 13 columns]
```

#### 股东持股统计-十大流通股东

接口: stock_gdfx_free_holding_statistics_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股统计-十大股东

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称                   | 类型      | 描述  |
|----------------------|---------|-----|
| 序号                   | int64   | -   |
| 股东名称                 | object  | -   |
| 股东类型                 | object  | -   |
| 统计次数                 | int64   | -   |
| 公告日后涨幅统计-10个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-10个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-10个交易日-最小涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-最小涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-最小涨幅 | float64 | -   |
| 持有个股                 | object  | -   |

接口示例

```python
import akshare as ak

stock_gdfx_free_holding_statistics_em_df = ak.stock_gdfx_free_holding_statistics_em(date="20210930")
print(stock_gdfx_free_holding_statistics_em_df)
```

数据示例

```
          序号  ...                                               持有个股
0          1  ...  603218|日月股份,600988|赤峰黄金,600298|安琪酵母,300232|洲明科...
1          2  ...  600519|贵州茅台,000686|东北证券,002533|金杯电工,002066|瑞泰科...
2          3  ...  600519|贵州茅台,300015|爱尔眼科,600674|川投能源,600009|上海机...
3          4  ...  000026|飞亚达,300946|恒而达,000751|锌业股份,300927|江天化学,...
4          5  ...  300005|探路者,300946|恒而达,000985|大庆华科,300927|江天化学,...
      ...  ...                                                ...
30963  30964  ...                                        002982|湘佳股份
30964  30965  ...                                        300917|特发服务
30965  30966  ...                                         688733|壹石通
30966  30967  ...                                        300969|恒帅股份
30967  30968  ...                                        002896|中大力德
```

#### 股东持股统计-十大股东

接口: stock_gdfx_holding_statistics_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东持股统计-十大股东

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20210930"; 财报发布季度最后日 |

输出参数

| 名称                   | 类型      | 描述  |
|----------------------|---------|-----|
| 序号                   | int64   | -   |
| 股东名称                 | object  | -   |
| 股东类型                 | object  | -   |
| 统计次数                 | int64   | -   |
| 公告日后涨幅统计-10个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-10个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-10个交易日-最小涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-30个交易日-最小涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-平均涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-最大涨幅 | float64 | -   |
| 公告日后涨幅统计-60个交易日-最小涨幅 | float64 | -   |
| 持有个股                 | object  | -   |

接口示例

```python
import akshare as ak

stock_gdfx_holding_statistics_em_df = ak.stock_gdfx_holding_statistics_em(date="20210930")
print(stock_gdfx_holding_statistics_em_df)
```

数据示例

```
          序号  ...                                               持有个股
0          1  ...  600763|通策医疗,300229|拓尔思,603039|泛微网络,300006|莱美药业...
1          2  ...  002225|濮耐股份,600435|北方导航,600422|昆药集团,600009|上海机...
2          3  ...  600340|华夏幸福,600009|上海机场,000001|平安银行,601899|紫金矿...
3          4  ...  601865|福莱特,601899|紫金矿业,000756|新华制药,600036|招商银行...
4          5  ...  002479|富春环保,300582|英飞特,300358|楚天科技,002892|科力尔,...
      ...  ...                                                ...
33819  33820  ...                                        300047|天源迪科
33820  33821  ...                                        300969|恒帅股份
33821  33822  ...                                        000425|徐工机械
33822  33823  ...                                        002896|中大力德
33823  33824  ...                                        000761|本钢板材
```

#### 股东协同-十大流通股东

接口: stock_gdfx_free_holding_teamwork_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东协同-十大流通股东

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                  |
|--------|-----|---------------------------------------------------------------------|
| symbol | str | symbol="社保"; choice of {"全部", "个人", "基金", "QFII", "社保", "券商", "信托"} |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int64  | -   |
| 股东名称   | object | -   |
| 股东类型   | object | -   |
| 协同股东名称 | object | -   |
| 协同股东类型 | object | -   |
| 协同次数   | int64  | -   |
| 个股详情   | object | -   |

接口示例

```python
import akshare as ak

stock_gdfx_free_holding_teamwork_em_df = ak.stock_gdfx_free_holding_teamwork_em(symbol="社保")
print(stock_gdfx_free_holding_teamwork_em_df)
```

数据示例

```
          序号  ...                                               个股详情
0          1  ...  601668|中国建筑|2023-09-30,600066|宇通客车|2023-06-30,...
1          2  ...  600028|中国石化|2022-09-30,300059|东方财富|2022-12-31,...
2          3  ...  600782|新钢股份|2022-09-30,002463|沪电股份|2021-12-31,...
3          4  ...  601899|紫金矿业|2023-03-31,600031|三一重工|2023-09-30,...
4          5  ...  002340|格林美|2022-06-30,600968|海油发展|2022-09-30,6...
      ...  ...                                                ...
19421  19422  ...                             600019|宝钢股份|2021-03-31
19422  19423  ...                             600019|宝钢股份|2021-03-31
19423  19424  ...                             600019|宝钢股份|2021-03-31
19424  19425  ...                             600019|宝钢股份|2021-03-31
19425  19426  ...                             600019|宝钢股份|2021-03-31
[19426 rows x 7 columns]
```

#### 股东协同-十大股东

接口: stock_gdfx_holding_teamwork_em

目标地址: https://data.eastmoney.com/gdfx/HoldingAnalyse.html

描述: 东方财富网-数据中心-股东分析-股东协同-十大股东

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                  |
|--------|-----|---------------------------------------------------------------------|
| symbol | str | symbol="社保"; choice of {"全部", "个人", "基金", "QFII", "社保", "券商", "信托"} |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int64  | -   |
| 股东名称   | object | -   |
| 股东类型   | object | -   |
| 协同股东名称 | object | -   |
| 协同股东类型 | object | -   |
| 协同次数   | int64  | -   |
| 个股详情   | object | -   |

接口示例

```python
import akshare as ak

stock_gdfx_holding_teamwork_em_df = ak.stock_gdfx_holding_teamwork_em(symbol="社保")
print(stock_gdfx_holding_teamwork_em_df)
```

数据示例

```
       序号  ...                                               个股详情
0          1  ...  601668|中国建筑|2023-09-30,600066|宇通客车|2023-06-30,...
1          2  ...  600028|中国石化|2022-09-30,300059|东方财富|2022-12-31,...
2          3  ...  600782|新钢股份|2022-09-30,002463|沪电股份|2021-12-31,...
3          4  ...  601899|紫金矿业|2023-03-31,600031|三一重工|2023-09-30,...
4          5  ...  601168|西部矿业|2023-09-30,002439|启明星辰|2023-09-30,...
      ...  ...                                                ...
15214  15215  ...                             600019|宝钢股份|2021-03-31
15215  15216  ...                             600019|宝钢股份|2021-03-31
15216  15217  ...                             600019|宝钢股份|2021-03-31
15217  15218  ...                             600019|宝钢股份|2021-03-31
15218  15219  ...                             600019|宝钢股份|2021-03-31
[15219 rows x 7 columns]
```

#### 股东户数

接口: stock_zh_a_gdhs

目标地址: http://data.eastmoney.com/gdhs/

描述: 东方财富网-数据中心-特色数据-股东户数数据

限量: 单次获取返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                     |
|--------|-----|------------------------------------------------------------------------|
| symbol | str | symbol="20230930"; choice of {"最新", 每个季度末}, 其中 每个季度末需要写成 `20230930` 格式 |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 代码           | object  | -       |
| 名称           | object  | -       |
| 最新价          | float64 | 注意单位: 元 |
| 涨跌幅          | float64 | 注意单位: % |
| 股东户数-本次      | int64   | -       |
| 股东户数-上次      | int64   | -       |
| 股东户数-增减      | int64   | -       |
| 股东户数-增减比例    | float64 | 注意单位: % |
| 区间涨跌幅        | float64 | 注意单位: % |
| 股东户数统计截止日-本次 | object  | -       |
| 股东户数统计截止日-上次 | object  | -       |
| 户均持股市值       | float64 | -       |
| 户均持股数量       | float64 | -       |
| 总市值          | float64 | -       |
| 总股本          | float64 | -       |
| 公告日期         | object  | -       |

接口示例

```python
import akshare as ak

stock_zh_a_gdhs_df = ak.stock_zh_a_gdhs(symbol="20230930")
print(stock_zh_a_gdhs_df)
```

数据示例

```
      代码    名称     最新价  ...     总市值           总股本        公告日期
0     688235  百济神州  139.04  ...  1.625156e+10  1.150553e+08  2023-11-14
1     688981  中芯国际   53.02  ...  1.009501e+11  1.973609e+09  2023-11-10
2     688793   倍轻松   33.65  ...  3.142165e+09  8.594542e+07  2023-10-31
3     688789  宏华数科   99.91  ...  1.156317e+10  1.204371e+08  2023-10-31
4     688787  海天瑞声   72.10  ...  5.248291e+09  6.032518e+07  2023-10-31
      ...   ...     ...  ...           ...           ...         ...
5153  600810  神马股份    7.44  ...  7.810436e+09  1.044178e+09  2023-10-12
5154  300809  华辰装备   29.25  ...  7.587976e+09  2.521760e+08  2023-10-12
5155  002817  黄山胶囊    8.76  ...  2.548316e+09  2.990982e+08  2023-10-12
5156  688188  柏楚电子  253.11  ...  3.648391e+10  1.462926e+08  2023-10-11
5157  601666  平煤股份   11.56  ...  2.385597e+10  2.313876e+09  2023-10-11
[5158 rows x 16 columns]
```

#### 股东户数详情

接口: stock_zh_a_gdhs_detail_em

目标地址: https://data.eastmoney.com/gdhs/detail/000002.html

描述: 东方财富网-数据中心-特色数据-股东户数详情

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="000001"; 股票代码 |

输出参数

| 名称        | 类型      | 描述      |
|-----------|---------|---------|
| 股东户数统计截止日 | object  | -       |
| 区间涨跌幅     | float64 | 注意单位: % |
| 股东户数-本次   | int64   | -       |
| 股东户数-上次   | int64   | -       |
| 股东户数-增减   | int64   | -       |
| 股东户数-增减比例 | float64 | 注意单位: % |
| 户均持股市值    | float64 | -       |
| 户均持股数量    | float64 | -       |
| 总市值       | float64 | -       |
| 总股本       | int64   | -       |
| 股本变动      | int64   | -       |
| 股本变动原因    | object  | -       |
| 股东户数公告日期  | object  | -       |
| 代码        | object  | -       |
| 名称        | object  | -       |

接口示例

```python
import akshare as ak

stock_zh_a_gdhs_detail_em_df = ak.stock_zh_a_gdhs_detail_em(symbol="000001")
print(stock_zh_a_gdhs_detail_em_df)
```

数据示例

```
    股东户数统计截止日 区间涨跌幅  股东户数-本次  ...股东户数公告日期  代码    名称
0   2023-09-30  -0.267142   530229  ...  2023-10-25  000001  平安银行
1   2023-06-30  -8.191107   536701  ...  2023-08-24  000001  平安银行
2   2023-03-31  -9.071118   506867  ...  2023-04-25  000001  平安银行
3   2023-02-28   4.711246   477304  ...  2023-03-09  000001  平安银行
4   2022-12-31  11.148649   487200  ...  2023-03-09  000001  平安银行
5   2022-09-30 -19.538845   547735  ...  2022-10-25  000001  平安银行
6   2022-06-30  -2.600780   493029  ...  2022-08-18  000001  平安银行
7   2022-03-31  -2.349206   508679  ...  2022-04-27  000001  平安银行
8   2022-02-28  -4.429612   506734  ...  2022-03-10  000001  平安银行
9   2021-12-31  -8.087005   502716  ...  2022-03-10  000001  平安银行
10  2021-09-30 -20.733864   390490  ...  2021-10-21  000001  平安银行
11  2021-06-30   3.579634   296828  ...  2021-08-20  000001  平安银行
12  2021-03-31  -4.677350   331829  ...  2021-04-21  000001  平安银行
13  2021-01-31  19.389866   290750  ...  2021-02-02  000001  平安银行
14  2020-12-31  27.488464   308724  ...  2021-02-02  000001  平安银行
15  2020-09-30  18.515625   351374  ...  2020-10-22  000001  平安银行
16  2020-06-30   1.721531   431036  ...  2020-08-28  000001  平安银行
17  2020-03-31 -17.631918   397399  ...  2020-04-21  000001  平安银行
18  2020-01-31  -5.531915   340920  ...  2020-02-14  000001  平安银行
19  2019-12-31   5.516357   322864  ...  2020-02-14  000001  平安银行
20  2019-09-30  13.134978   299958  ...  2019-10-22  000001  平安银行
21  2019-06-30   8.620562   321929  ...  2019-08-08  000001  平安银行
22  2019-03-31   3.721683   354508  ...  2019-04-24  000001  平安银行
23  2019-02-28  31.769723   369119  ...  2019-03-07  000001  平安银行
24  2018-12-31 -15.113122   429409  ...  2019-03-07  000001  平安银行
25  2018-09-30  23.531940   406242  ...  2018-10-24  000001  平安银行
26  2018-06-30 -16.605505   435978  ...  2018-08-16  000001  平安银行
27  2018-03-31  -9.543568   421677  ...  2018-04-20  000001  平安银行
28  2018-02-28  -9.398496   392992  ...  2018-03-15  000001  平安银行
29  2017-12-31  19.711971   343994  ...  2018-03-15  000001  平安银行
30  2017-09-30  20.068588   331474  ...  2017-10-21  000001  平安银行
31  2017-06-30   2.399128   379179  ...  2017-08-11  000001  平安银行
32  2017-03-31  -3.270042   371177  ...  2017-04-22  000001  平安银行
33  2017-02-28   4.175824   349253  ...  2017-03-17  000001  平安银行
34  2016-12-31   0.330761   350142  ...  2017-03-17  000001  平安银行
35  2016-09-30   4.252874   340324  ...  2016-10-21  000001  平安银行
36  2016-06-30  -0.391340   340528  ...  2016-08-12  000001  平安银行
37  2016-03-31  11.297071   341525  ...  2016-04-21  000001  平安银行
38  2016-02-29 -20.266889   347147  ...  2016-03-10  000001  平安银行
39  2015-12-31  14.299333   332918  ...  2016-03-10  000001  平安银行
40  2015-09-30 -27.854195   358049  ...  2015-10-23  000001  平安银行
41  2015-06-30  11.728949   418294  ...  2015-08-14  000001  平安银行
42  2015-03-31  17.187500   313029  ...  2015-04-24  000001  平安银行
43  2015-03-06 -15.151515   335403  ...  2015-03-13  000001  平安银行
44  2014-12-31  56.213018   275838  ...  2015-03-13  000001  平安银行
45  2014-09-30   2.320888   311571  ...  2014-10-24  000001  平安银行
46  2014-06-30  11.976813   298305  ...  2014-08-14  000001  平安银行
47  2014-03-31  -3.234501   319109  ...  2014-04-24  000001  平安银行
48  2014-02-28  -9.142857   310335  ...  2014-03-07  000001  平安银行
49  2013-12-31   3.114478   298070  ...  2014-03-07  000001  平安银行
50  2013-09-30  19.157472   291054  ...  2013-10-23  000001  平安银行
51  2013-06-30 -20.017323   281819  ...  2013-08-23  000001  平安银行
52  2013-03-31 -12.749350   256985  ...  2013-04-24  000001  平安银行
53  2013-03-01  43.945069   233517  ...  2013-03-08  000001  平安银行
[54 rows x 15 columns]
```

#### 分红配股

接口: stock_history_dividend_detail

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vISSUE_ShareBonus/stockid/300670.phtml

描述: 新浪财经-发行与分配-分红配股

限量: 单次获取指定股票的新浪财经-发行与分配-分红配股详情

输入参数

| 名称        | 类型  | 描述                                               |
|-----------|-----|--------------------------------------------------|
| symbol    | str | symbol="600012"; 股票代码                            |
| indicator | str | indicator="配股"; choice of {"分红", "配股"}           |
| date      | str | date="1994-12-24"; 分红配股的具体日期, e.g., "1994-12-24" |

输出参数-分红历史

| 名称    | 类型      | 描述          |
|-------|---------|-------------|
| 公告日期  | object  | -           |
| 送股    | int64   | 注意单位: 股     |
| 转增    | int64   | 注意单位: 股     |
| 派息    | float64 | 注意单位: 元; 税前 |
| 进度    | object  | -           |
| 除权除息日 | object  | -           |
| 股权登记日 | object  | -           |
| 红股上市日 | object  | -           |

接口示例-分红历史

```python
import akshare as ak

stock_history_dividend_detail_df = ak.stock_history_dividend_detail(symbol="600012", indicator="分红")
print(stock_history_dividend_detail_df)
```

数据示例-分红历史

```
     公告日期  送股  转增   派息  进度  除权除息日   股权登记日 红股上市日
0   2021-07-14   0   0  2.3  实施  2021-07-21  2021-07-20   NaT
1   2020-07-14   0   0  2.3  实施  2020-07-22  2020-07-21   NaT
2   2019-07-08   0   0  2.5  实施  2019-07-16  2019-07-15   NaT
3   2018-07-09   0   0  2.3  实施  2018-07-17  2018-07-16   NaT
4   2017-07-10   0   0  2.3  实施  2017-07-18  2017-07-17   NaT
5   2016-07-12   0   0  2.3  实施  2016-07-19  2016-07-18   NaT
6   2015-07-13   0   0  2.3  实施  2015-07-20  2015-07-17   NaT
7   2014-07-07   0   0  2.2  实施  2014-07-11  2014-07-10   NaT
8   2013-06-18   0   0  2.0  实施  2013-06-24  2013-06-21   NaT
9   2012-07-10   0   0  2.1  实施  2012-07-16  2012-07-13   NaT
10  2011-06-08   0   0  2.1  实施  2011-06-14  2011-06-13   NaT
11  2010-06-10   0   0  2.0  实施  2010-06-21  2010-06-18   NaT
12  2009-05-25   0   0  2.3  实施  2009-06-02  2009-06-01   NaT
13  2008-06-18   0   0  2.0  实施  2008-06-25  2008-06-24   NaT
14  2007-06-20   0   0  2.5  实施  2007-06-26  2007-06-25   NaT
15  2006-07-12   0   0  2.8  实施  2006-07-18  2006-07-17   NaT
16  2005-07-11   0   0  1.0  实施  2005-07-15  2005-07-14   NaT
17  2004-06-14   0   0  0.6  实施  2004-06-18  2004-06-17   NaT
18  2003-07-07   0   0  0.3  实施  2003-07-11  2003-07-10   NaT
```

输出参数-分红详情

| 名称    | 类型     | 描述   |
|-------|--------|------|
| item  | object | 所列项目 |
| value | object | 项目值  |

接口示例-分红详情

```python
import akshare as ak

stock_history_dividend_detail_df = ak.stock_history_dividend_detail(symbol="600012", indicator="分红",
                                                                    date="2019-07-08")
print(stock_history_dividend_detail_df)
```

数据示例-分红详情

```
                 item             value
0          税前红利（报价币种）        2.50
1          税后红利（报价币种）        2.50
2       B、H股税前红利（人民币）          --
3       B、H股税后红利（人民币）          --
4           送股比例（10送）          --
5          转增比例（10转增）          --
6     盈余公积金转增比例（10转增）          --
7     资本公积金转增比例（10转增）          --
8                发放对象          --
9               股本基准日  2018-12-31
10              最后交易日          --
11                登记日  2019-07-15
12                除息日  2019-07-16
13  红利/配股起始日（送、转股到账日）  2019-07-16
14           红利/配股终止日          --
15                上市日          --
16         股东大会决议公告日期  2019-05-18
17       可转债享受权益转股截止日          --
18          配股比例（10配）          --
19                配股价          --
20               转配比例          --
21                转配价          --
22              配股有效期          --
23              实际配股数          --
24             配股前总股本          --
25             实际配股比例          --
26              每股拆细数          --
27             外币折算汇率          --
28               权息说明          --
```

输出参数-配股历史

| 名称     | 类型      | 描述             |
|--------|---------|----------------|
| 公告日期   | object  | -              |
| 配股方案   | float64 | 注意单位: 每10股配股股数 |
| 配股价格   | float64 | 注意单位: 元        |
| 基准股本   | int64   | 注意单位: 股        |
| 除权日    | object  | -              |
| 股权登记日  | object  | -              |
| 缴款起始日  | object  | -              |
| 缴款终止日  | object  | -              |
| 配股上市日  | object  | -              |
| 募集资金合计 | float64 | 注意单位: 元        |

接口示例-配股历史

```python
import akshare as ak

stock_history_dividend_detail_df = ak.stock_history_dividend_detail(symbol="000002", indicator="配股")
print(stock_history_dividend_detail_df)
```

数据示例-配股历史

```
   公告日期   配股方案  配股价格  ... 缴款终止日    配股上市日 募集资金合计
0  1999-12-22  2.727   7.5  ...  2000-01-24  2000-02-16    NaN
1  1997-06-26  2.370   4.5  ...  1997-07-29  1997-08-22    NaN
2  1900-01-01  5.000   4.4  ...  1991-06-08  1900-01-01    NaN
```

输出参数-配股详情

| 名称    | 类型     | 描述   |
|-------|--------|------|
| item  | object | 所列项目 |
| value | object | 项目值  |

接口示例-配股详情

```python
import akshare as ak

stock_history_dividend_detail_df = ak.stock_history_dividend_detail(indicator="配股", symbol="000002",
                                                                    date="1999-12-22")
print(stock_history_dividend_detail_df)
```

数据示例-配股详情

```
                 item         value
0          税前红利（报价币种）            --
1          税后红利（报价币种）            --
2       B、H股税前红利（人民币）            --
3       B、H股税后红利（人民币）            --
4           送股比例（10送）            --
5          转增比例（10转增）            --
6     盈余公积金转增比例（10转增）            --
7     资本公积金转增比例（10转增）            --
8                发放对象            --
9               股本基准日            --
10              最后交易日            --
11                登记日    2000-01-07
12                除息日    2000-01-10
13  红利/配股起始日（送、转股到账日）    2000-01-11
14           红利/配股终止日    2000-01-24
15                上市日    2000-02-16
16         股东大会决议公告日期            --
17       可转债享受权益转股截止日            --
18          配股比例（10配）          2.73
19                配股价          7.50
20               转配比例            --
21                转配价            --
22              配股有效期            --
23              实际配股数   85434500.00
24             配股前总股本  545537000.00
25             实际配股比例          1.57
26              每股拆细数            --
27             外币折算汇率            --
28               权息说明            --
```

#### 历史分红

接口: stock_dividend_cninfo

目标地址: http://webapi.cninfo.com.cn/#/company?companyid=600009

描述: 巨潮资讯-个股-历史分红

限量: 单次获取指定股票的历史分红数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600009" |

输出参数

| 名称       | 类型      | 描述           |
|----------|---------|--------------|
| 实施方案公告日期 | object  | -            |
| 送股比例     | float64 | 注意单位: 每 10 股 |
| 转增比例     | float64 | 注意单位: 每 10 股 |
| 派息比例     | float64 | 注意单位: 每 10 股 |
| 股权登记日    | object  | -            |
| 除权日      | object  | -            |
| 派息日      | object  | -            |
| 股份到账日    | object  | -            |
| 实施方案分红说明 | object  | -            |
| 分红类型     | object  | -            |
| 报告时间     | object  | -            |

接口示例

```python
import akshare as ak

stock_dividend_cninfo_df = ak.stock_dividend_cninfo(symbol="600009")
print(stock_dividend_cninfo_df)
```

数据示例

```
    实施方案公告日期  分红类型  送股比例  转增比例  ...派息日 股份到账日   实施方案分红说明  报告时间
0   1998-11-28  中期分红   0.0   5.0  ...         NaT  None          10转增5股  1998半年报
1   2001-06-23  年度分红   NaN   NaN  ...  2001-07-06  None    10派2.93元(含税)   2000年报
2   2002-07-18  年度分红   NaN   NaN  ...  2002-08-01  None     10派1.5元(含税)   2001年报
3   2003-08-05  年度分红   NaN   NaN  ...  2003-08-14  None   10股派1.99元（含税）   2002年报
4   2003-12-22  中期分红   3.0   NaN  ...         NaT  None           10送3股  2003半年报
5   2004-07-16  年度分红   NaN   NaN  ...  2004-07-28  None    10派0.96元（含税）   2003年报
6   2005-05-17  年度分红   NaN   NaN  ...  2005-05-27  None       10派1元(含税)   2004年报
7   2006-08-15  年度分红   NaN   NaN  ...  2006-08-25  None     10派1.5元(含税)   2005年报
8   2006-12-01  中期分红   NaN   NaN  ...  2006-12-13  None       10派1元（含税）  2006半年报
9   2007-08-16  年度分红   NaN   NaN  ...  2007-08-28  None     10派1.5元（含税）   2006年报
10  2008-07-24  年度分红   NaN   NaN  ...  2008-08-05  None       10派1元（含税）   2007年报
11  2009-07-30  年度分红   NaN   NaN  ...  2009-08-11  None     10派0.5元（含税）   2008年报
12  2010-07-29  年度分红   NaN   NaN  ...  2010-08-10  None     10派0.5元（含税）   2009年报
13  2011-08-08  年度分红   NaN   NaN  ...  2011-08-18  None       10派1元(含税)   2010年报
14  2012-08-07  年度分红   NaN   NaN  ...  2012-08-17  None       10派6元(含税)   2011年报
15  2013-08-12  年度分红   NaN   NaN  ...  2013-08-22  None     10派3.7元(含税)   2012年报
16  2014-08-11  年度分红   NaN   NaN  ...  2014-08-15  None       10派3元(含税)   2013年报
17  2015-08-14  年度分红   NaN   NaN  ...  2015-08-20  None     10派3.5元(含税)   2014年报
18  2016-08-12  年度分红   NaN   NaN  ...  2016-08-18  None     10派4.3元(含税)   2015年报
19  2017-08-18  年度分红   NaN   NaN  ...  2017-08-24  None     10派4.4元(含税)   2016年报
20  2018-08-17  年度分红   NaN   NaN  ...  2018-08-23  None     10派5.8元(含税)   2017年报
21  2019-08-16  年度分红   NaN   NaN  ...  2019-08-22  None     10派6.6元(含税)   2018年报
22  2020-08-14  年度分红   NaN   NaN  ...  2020-08-20  None     10派7.9元(含税)   2019年报
23  2024-07-15  年度分红   NaN   NaN  ...  2024-07-19  None     10派1.2元(含税)   2023年报
24  2024-09-03  中期分红   NaN   NaN  ...  2024-09-09  None  10派1.0034元(含税)  2024半年报
25  2025-08-04  年度分红   NaN   NaN  ...  2025-08-08  None       10派3元(含税)   2024年报
26  2025-10-10  中期分红   NaN   NaN  ...  2025-10-16  None     10派2.1元(含税)  2025半年报
[27 rows x 11 columns]
```

#### 新股发行

接口: stock_ipo_info

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vISSUE_NewStock/stockid/600004.phtml

描述: 新浪财经-发行与分配-新股发行

限量: 单次获取新股发行的基本信息数据

输入参数

| 名称    | 类型  | 描述                   |
|-------|-----|----------------------|
| stock | str | stock="600004"; 股票代码 |

输出参数

| 名称    | 类型     | 描述   |
|-------|--------|------|
| item  | object | 所列项目 |
| value | object | 项目值  |

接口示例

```python
import akshare as ak

stock_ipo_info_df = ak.stock_ipo_info(stock="600004")
print(stock_ipo_info_df)
```

数据示例

```
              item       value
0              上市地     上海证券交易所
1             主承销商  中国国际金融有限公司
2             承销方式        余额包销
3            上市推荐人  中国国际金融有限公司
4         每股发行价(元)        5.88
5             发行方式    二级市场定价配售
6   发行市盈率（按发行后总股本）          --
7       首发前总股本（万股）    60000.00
8       首发后总股本（万股）   100000.00
9        实际发行量（万股）    40000.00
10      预计募集资金（万元）   235200.00
11    实际募集资金合计（万元）   235200.00
12      发行费用总额（万元）     6936.40
13      募集资金净额（万元）   228263.60
14        承销费用（万元）          --
15           招股公告日  2003-04-09
16            上市日期  2003-04-28
```

#### 新股上会信息

接口: stock_ipo_review_em

目标地址: https://data.eastmoney.com/xg/gh/default.html

描述: 东方财富网-数据中心-新股申购-新股上会信息

限量: 单次获取所有数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称      | 类型     | 描述 |
|---------|--------|----|
| 序号      | int64  | -  |
| 企业名称    | object | -  |
| 股票简称    | object | -  |
| 股票代码    | object | -  |
| 上市板块    | object | -  |
| 上会日期    | object | -  |
| 审核状态    | object | -  |
| 发审委委员   | object | -  |
| 主承销商    | object | -  |
| 发行数量(股) | object | -  |
| 拟融资额(元) | object | -  |
| 公告日期    | object | -  |
| 上市日期    | object | -  |

接口示例

```python
import akshare as ak

stock_ipo_review_em_df = ak.stock_ipo_review_em()
print(stock_ipo_review_em_df)
```

数据示例

```
     序号              企业名称  股票简称  ...       拟融资额(元)        公告日期 上市日期
0        1     北京维通利电气股份有限公司   维通利  ...  1.593741e+09  2025-11-19  NaT
1        2        杭州福恩股份有限公司  福恩股份  ...  1.250000e+09  2025-12-17  NaT
2        3    南昌三瑞智能科技股份有限公司  三瑞智能  ...  7.688829e+08  2025-12-08  NaT
3        4     深圳大普微电子股份有限公司   大普微  ...  1.877852e+09  2025-09-30  NaT
4        5        视涯科技股份有限公司  视涯科技  ...  2.014572e+09  2025-09-30  NaT
...    ...               ...   ...  ...           ...         ...  ...
5056  5057     合肥芯谷微电子股份有限公司   芯谷微  ...  8.500000e+08  2023-05-05  NaT
5057  5058    安翰科技(武汉)股份有限公司  安翰科技  ...  1.200000e+09  2019-03-22  NaT
5058  5059      浙江高信技术股份有限公司  高信技术  ...  7.300000e+08  2023-06-26  NaT
5059  5060    华夏天信智能物联股份有限公司  华夏天信  ...  9.280300e+08  2019-05-31  NaT
5060  5061  广西北部湾陆海新能源股份有限公司  陆海新能  ...  1.600000e+09  2023-09-22  NaT
[5061 rows x 13 columns]
```

#### IPO辅导信息

接口: stock_ipo_tutor_em

目标地址: https://data.eastmoney.com/xg/gh/default.html

描述: 东方财富网-数据中心-新股申购-IPO辅导信息

限量: 单次获取所有数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称      | 类型     | 描述 |
|---------|--------|----|
| 序号      | int64  | -  |
| 企业名称    | object | -  |
| 辅导机构    | object | -  |
| 辅导状态    | object | -  |
| 报告类型    | object | -  |
| 派出机构    | object | -  |
| 备案日期    | object | -  |

接口示例

```python
import akshare as ak

stock_ipo_tutor_em_df = ak.stock_ipo_tutor_em()
print(stock_ipo_tutor_em_df)
```

数据示例

```
     序号  ...        备案日期
0        1  ...  2025-12-23
1        2  ...  2025-12-23
2        3  ...  2025-12-23
3        4  ...  2025-12-23
4        5  ...  2025-12-23
...    ...  ...         ...
5129  5130  ...         NaT
5130  5131  ...         NaT
5131  5132  ...         NaT
5132  5133  ...         NaT
5133  5134  ...         NaT
[5134 rows x 8 columns]
```

#### 股票增发

接口: stock_add_stock

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vISSUE_AddStock/stockid/600004.phtml

描述: 新浪财经-发行与分配-增发

限量: 单次指定 symbol 的股票增发详情数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="600004"; 股票代码 |

输出参数

| 名称         | 类型     | 描述 |
|------------|--------|----|
| 公告日期       | object | -  |
| 发行方式       | object | -  |
| 发行价格       | object | -  |
| 实际公司募集资金总额 | object | -  |
| 发行费用总额     | object | -  |
| 实际发行数量     | object | -  |

接口示例

```python
import akshare as ak

stock_add_stock_df = ak.stock_add_stock(symbol="600004")
print(stock_add_stock_df)
```

数据示例

```
    公告日期    发行方式  发行价格  实际公司募集资金总额  发行费用总额      实际发行数量
0  2020-11-06    定向配售  10.76元  320,000.00万元  1,253.53万元  29739.7769万股
1  2007-12-29  网下定价发行  15.05元   22,575.00万元  1,887.00万元        1500万股
2  2007-12-18    定向配售   7.09元         0.00元       0.00元       13500万股
```

#### 限售解禁

##### 个股限售解禁-新浪

接口: stock_restricted_release_queue_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vInvestConsult/kind/xsjj/index.phtml?symbol=sh600000

描述: 新浪财经-发行分配-限售解禁

限量: 单次获取指定 symbol 的限售解禁数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="600000"; 股票代码 |

输出参数

| 名称      | 类型      | 描述       |
|---------|---------|----------|
| 代码      | object  | -        |
| 名称      | object  | -        |
| 解禁日期    | object  | -        |
| 解禁数量    | float64 | 注意单位: 万股 |
| 解禁股流通市值 | float64 | 注意单位: 亿元 |
| 上市批次    | int64   | -        |
| 公告日期    | object  | -        |

接口示例

```python
import akshare as ak

stock_restricted_release_queue_sina_df = ak.stock_restricted_release_queue_sina(symbol="600000")
print(stock_restricted_release_queue_sina_df)
```

数据示例

```
    代码    名称        解禁日期       解禁数量   解禁股流通市值  上市批次        公告日期
0  600000  浦发银行  2020-09-04  124831.65  127.0786    10  2017-09-06
1  600000  浦发银行  2017-03-20  109946.14  179.1023     9  2016-03-22
2  600000  浦发银行  2015-10-14  373069.43  599.1495     8  2010-10-16
3  600000  浦发银行  2010-09-29  117540.69  151.9801     7  2009-09-30
4  600000  浦发银行  2009-05-12   64917.32  161.1248     6  2006-05-09
5  600000  浦发银行  2008-05-12   41247.84  120.7737     5  2008-05-07
6  600000  浦发银行  2007-11-30   27761.94  144.0567     4  2006-11-21
7  600000  浦发银行  2007-05-14  192834.49  564.2337     3  2007-05-09
```

##### 限售股解禁

接口: stock_restricted_release_summary_em

目标地址: https://data.eastmoney.com/dxf/marketStatistics.html?type=day&startdate=2022-11-08&enddate=2022-12-19

描述: 东方财富网-数据中心-特色数据-限售股解禁

限量: 单次获取指定 symbol 在近期限售股解禁数据

输入参数

| 名称         | 类型  | 描述                                                                      |
|------------|-----|-------------------------------------------------------------------------|
| symbol     | str | symbol="全部股票"; choice of {"全部股票", "沪市A股", "科创板", "深市A股", "创业板", "京市A股"} |
| start_date | str | start_date="20221101"                                                   |
| end_date   | str | end_date="20221209"                                                     |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 解禁时间       | object  | -       |
| 当日解禁股票家数   | int64   | -       |
| 解禁数量       | float64 | 注意单位: 股 |
| 实际解禁数量     | float64 | 注意单位: 股 |
| 实际解禁市值     | int64   | 注意单位: 元 |
| 沪深300指数    | float64 | -       |
| 沪深300指数涨跌幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_restricted_release_summary_em_df = ak.stock_restricted_release_summary_em(symbol="全部股票", start_date="20221108", end_date="20221209")
print(stock_restricted_release_summary_em_df)
```

数据示例

```
    序号        解禁时间  当日解禁股票家数  ...        实际解禁市值    沪深300指数  沪深300指数涨跌幅
0    1  2022-11-08        15  ...  1.172407e+10  3749.3251     -0.6879
1    2  2022-11-09        10  ...  2.572546e+09  3714.2704     -0.9350
2    3  2022-11-10        15  ...  9.329462e+09  3685.6885     -0.7695
3    4  2022-11-11        17  ...  4.912280e+09  3788.4387      2.7878
4    5  2022-11-14        15  ...  2.318019e+10  3794.0171      0.1472
5    6  2022-11-15        14  ...  1.714207e+10  3865.9726      1.8966
6    7  2022-11-16        12  ...  1.209874e+10  3834.3899     -0.8169
7    8  2022-11-17        11  ...  4.802956e+09  3818.6640     -0.4101
8    9  2022-11-18        22  ...  8.274363e+10  3801.5686     -0.4477
9   10  2022-11-21        20  ...  1.283129e+10  3769.1269     -0.8534
10  11  2022-11-22         9  ...  9.232523e+09  3769.5727      0.0118
11  12  2022-11-23         7  ...  3.737618e+08  3773.5287      0.1049
12  13  2022-11-24         7  ...  4.706748e+09  3756.8082     -0.4431
13  14  2022-11-25        11  ...  2.465310e+09  3775.7764      0.5049
14  15  2022-11-28        17  ...  4.328994e+10  3733.2424     -1.1265
15  16  2022-11-29        13  ...  7.017021e+10  3848.4224      3.0853
16  17  2022-11-30        11  ...  1.060871e+10  3853.0365      0.1199
17  18  2022-12-01        15  ...  3.325098e+09  3894.7687      1.0831
18  19  2022-12-02        15  ...  4.087764e+09  3870.9478     -0.6116
19  20  2022-12-05        35  ...  1.952387e+10        NaN         NaN
20  21  2022-12-06        19  ...  1.600255e+10        NaN         NaN
21  22  2022-12-07        14  ...  2.661094e+09        NaN         NaN
22  23  2022-12-08        10  ...  2.100872e+09        NaN         NaN
23  24  2022-12-09         8  ...  1.630020e+10        NaN         NaN
```

##### 限售股解禁详情

接口: stock_restricted_release_detail_em

目标地址: https://data.eastmoney.com/dxf/detail.html

描述: 东方财富网-数据中心-限售股解禁-解禁详情一览

限量: 单次获取指定时间段限售股解禁数据

输入参数

| 名称         | 类型  | 描述                                                                      |
|------------|-----|-------------------------------------------------------------------------|
| start_date | str | start_date="20221202"                                                   |
| end_date   | str | end_date="20241202"                                                     |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 股票代码       | object  | -       |
| 股票简称       | object  | -       |
| 解禁时间       | object  | -       |
| 限售股类型      | object  | 注意单位: 股 |
| 解禁数量       | float64 | 注意单位: 股 |
| 实际解禁数量     | float64 | 注意单位: 股 |
| 实际解禁市值     | float64 | 注意单位: 元 |
| 占解禁前流通市值比例 | float64 | -       |
| 解禁前一交易日收盘价 | float64 | -       |
| 解禁前20日涨跌幅  | float64 | 注意单位: % |
| 解禁后20日涨跌幅  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_restricted_release_detail_em_df = ak.stock_restricted_release_detail_em(start_date="20221202", end_date="20221204")
print(stock_restricted_release_detail_em_df)
```

数据示例

```
    序号    股票代码  股票简称        解禁时间  ... 占解禁前流通市值比例  解禁前一交易日收盘价  解禁前20日涨跌幅  解禁后20日涨跌幅
0    1  002548   金新农  2022-12-02  ...   0.000292        5.98   3.866432       1.18
1    2  873527   夜光明  2022-12-02  ...   0.006829        9.02  -2.162162      -0.33
2    3  603127  昭衍新药  2022-12-02  ...   0.000445       58.30  -6.029040      -2.08
3    4  002511  中顺洁柔  2022-12-02  ...   0.000604       11.97  17.993795       4.91
4    5  688557  兰剑智能  2022-12-02  ...   0.020572       38.04  -4.086845       1.30
5    6  603283  赛腾股份  2022-12-02  ...   0.007451       35.55  -2.860892      -3.94
6    7  688007  光峰科技  2022-12-02  ...   0.003214       26.28   7.978723      -0.42
7    8  688578   艾力斯  2022-12-02  ...   0.013058       20.80  -9.883470      -0.38
8    9  300227   光韵达  2022-12-02  ...   0.021127        6.89  -5.793103       0.88
9   10  300525  博思软件  2022-12-02  ...   0.017589       19.87  -4.481132      -1.88
10  11  831641   格利尔  2022-12-02  ...   0.422407       12.65        NaN      31.77
11  12  688489  三未信安  2022-12-02  ...   1.000000      119.50        NaN        NaN
12  13  301198  喜悦智行  2022-12-02  ...   0.457230       18.83   1.796407       0.70
13  14  600848  上海临港  2022-12-02  ...   0.015465       12.64   6.260575       0.64
14  15  000415  渤海租赁  2022-12-02  ...   0.061934        2.29   4.954955      -1.72
```

##### 解禁批次

接口: stock_restricted_release_queue_em

目标地址: https://data.eastmoney.com/dxf/q/600000.html

描述: 东方财富网-数据中心-个股限售解禁-解禁批次

限量: 单次获取指定 symbol 的解禁批次数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="600000" |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 解禁时间       | object  | -       |
| 解禁股东数      | int64   | -       |
| 解禁数量       | float64 | 注意单位: 股 |
| 实际解禁数量     | float64 | 注意单位: 股 |
| 未解禁数量      | int64   | 注意单位: 股 |
| 实际解禁数量市值   | float64 | 注意单位: 元 |
| 占总市值比例     | float64 | -       |
| 占流通市值比例    | float64 | -       |
| 解禁前一交易日收盘价 | float64 | 注意单位: 元 |
| 限售股类型      | object  | -       |
| 解禁前20日涨跌幅  | float64 | 注意单位: % |
| 解禁后20日涨跌幅  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_restricted_release_queue_em_df = ak.stock_restricted_release_queue_em(symbol="600000")
print(stock_restricted_release_queue_em_df)
```

数据示例

```
   序号   解禁时间  解禁股东数  ...   限售股类型  解禁前20日涨跌幅  解禁后20日涨跌幅
0   1  2020-09-04      2  ...  定向增发机构配售股份  -3.893637  -6.916996
1   2  2017-03-20     11  ...  定向增发机构配售股份  -2.644231  -5.555556
2   3  2015-10-14      1  ...  定向增发机构配售股份   6.042497  12.085160
3   4  2010-09-29     38  ...  定向增发机构配售股份 -10.334996  15.262321
[4 rows x 13 columns]
```

##### 解禁股东

接口: stock_restricted_release_stockholder_em

目标地址: https://data.eastmoney.com/dxf/q/600000.html

描述: 东方财富网-数据中心-个股限售解禁-解禁股东

限量: 单次获取指定 symbol 的解禁批次数据

输入参数

| 名称     | 类型  | 描述                                                                           |
|--------|-----|------------------------------------------------------------------------------|
| symbol | str | symbol="600000"                                                              |
| date   | str | date="20200904"; 通过 ak.stock_restricted_release_queue_em(symbol="600000") 获取 |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 股东名称    | object  | -       |
| 解禁数量    | int64   | 注意单位: 股 |
| 实际解禁数量  | int64   | 注意单位: 股 |
| 解禁市值    | float64 | 注意单位: 元 |
| 锁定期     | int64   | 注意单位: 月 |
| 剩余未解禁数量 | int64   | 注意单位: 股 |
| 限售股类型   | object  | -       |
| 进度      | object  | -       |

接口示例

```python
import akshare as ak

stock_restricted_release_stockholder_em_df = ak.stock_restricted_release_stockholder_em(symbol="600000", date="20200904")
print(stock_restricted_release_stockholder_em_df)
```

数据示例

```
   序号          股东名称       解禁数量     实际解禁数量  ...  锁定期  剩余未解禁数量       限售股类型  进度
0   1    上海国际集团有限公司  842003367  842003367  ...   36        0  定向增发机构配售股份  实施
1   2  上海国鑫投资发展有限公司  406313131  406313131  ...   36        0  定向增发机构配售股份  实施
```

#### 流通股东

接口: stock_circulate_stock_holder

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vCI_CirculateStockHolder/stockid/600000.phtml

描述: 新浪财经-股东股本-流通股东

限量: 单次获取指定 symbol 的流通股东数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="600000"; 股票代码 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 截止日期   | object  | -       |
| 公告日期   | object  | -       |
| 编号     | int64   | -       |
| 股东名称   | object  | -       |
| 持股数量   | int64   | 注意单位: 股 |
| 占流通股比例 | float64 | 注意单位: % |
| 股本性质   | object  | -       |

接口示例

```python
import akshare as ak

stock_circulate_stock_holder_df = ak.stock_circulate_stock_holder(symbol="600000")
print(stock_circulate_stock_holder_df)
```

数据示例

```
      截止日期        公告日期  编号                股东名称        持股数量  占流通股比例   股本性质
0    2022-09-30  2022-10-29   1          上海国际集团有限公司  6331322671  21.570    国有股
1    2022-09-30  2022-10-29   2      中国移动通信集团广东有限公司  5334892824  18.175    国有股
2    2022-09-30  2022-10-29   3   富德生命人寿保险股份有限公司-传统  2779437274   9.469  境内法人股
3    2022-09-30  2022-10-29   4  富德生命人寿保险股份有限公司-资本金  1763232325   6.007  境内法人股
4    2022-09-30  2022-10-29   5       上海上国投资产管理有限公司  1395571025   4.755    国有股
..          ...         ...  ..                 ...         ...     ...    ...
745  2003-12-31         NaN   6          世纪证券有限责任公司     6882195   0.765    国有股
746  2003-12-31         NaN   7            汉兴证券投资基金     5921727   0.658  境内法人股
747  2003-12-31         NaN   8            泰和证券投资基金     4534267   0.504  境内法人股
748  2003-12-31         NaN   9          博时裕富证券投资基金     4197332   0.466  境内法人股
749  2003-12-31         NaN  10            同智证券投资基金     3880427   0.431  境内法人股
```

#### 板块行情

接口: stock_sector_spot

目标地址: http://finance.sina.com.cn/stock/sl/

描述: 新浪行业-板块行情

限量: 单次获取指定的板块行情实时数据

输入参数

| 名称        | 类型  | 描述                                                              |
|-----------|-----|-----------------------------------------------------------------|
| indicator | str | indicator="新浪行业"; choice of {"新浪行业", "启明星行业", "概念", "地域", "行业"} |

输出参数

| 名称     | 类型      | 描述       |
|--------|---------|----------|
| label  | object  | -        |
| 板块     | object  | -        |
| 公司家数   | int64   | -        |
| 平均价格   | float64 | -        |
| 涨跌额    | float64 | -        |
| 涨跌幅    | float64 | 注意单位: %  |
| 总成交量   | int64   | 注意单位: 手  |
| 总成交额   | int64   | 注意单位: 万元 |
| 股票代码   | object  | -        |
| 个股-涨跌幅 | float64 | 注意单位: %  |
| 个股-当前价 | float64 | -        |
| 个股-涨跌额 | float64 | -        |
| 股票名称   | object  | -        |

接口示例

```python
import akshare as ak

stock_industry_sina_df = ak.stock_sector_spot(indicator="新浪行业")
print(stock_industry_sina_df)
```

数据示例

```
        label    板块  公司家数        平均价格  ...  个股-涨跌幅  个股-当前价  个股-涨跌额   股票名称
0    new_blhy  玻璃行业    19   15.913158  ...   4.214   21.27    0.86   金刚玻璃
1    new_cbzz  船舶制造     8   10.561250  ...   5.374   22.55    1.15   中国船舶
2    new_cmyl  传媒娱乐    40    6.491053  ...  13.826    3.54    0.43   华谊兄弟
3    new_dlhy  电力行业    62    7.798667  ...  10.011   20.66    1.88   文山电力
4    new_dqhy  电器行业    58   15.341897  ...   7.950    7.74    0.57   精达股份
5    new_dzqj  电子器件   152   22.652349  ...  20.055    8.74    1.46   珈伟新能
6    new_dzxx  电子信息   247   12.984641  ...  20.022   21.46    3.58    中青宝
7     new_fdc   房地产   123    7.213445  ...  10.189    2.92    0.27   铁岭新城
8    new_fdsb  发电设备    65   11.620000  ...   6.644   18.46    1.15    天融信
9    new_fjzz  飞机制造    14   24.420714  ...   6.079   62.82    3.60   中直股份
10   new_fzhy  纺织行业    42    5.945750  ...   7.362    5.25    0.36   凤竹纺织
11   new_fzjx  纺织机械     8   10.077500  ...   5.704   19.83    1.07   三联虹普
12   new_fzxl  服装鞋类    49    9.784375  ...  10.066    6.67    0.61    乔治白
13   new_glql  公路桥梁    20    4.771667  ...   2.979    2.42    0.07   吉林高速
14   new_gsgq  供水供气    25    6.512000  ...   4.945    3.82    0.18   辽宁能源
15   new_gthy  钢铁行业    60    8.681455  ...  10.036   15.24    1.39   海南矿业
16   new_hbhy  环保行业    27    7.172308  ...  20.023   10.61    1.77   天壕环境
17   new_hghy  化工行业   150   17.143537  ...  11.946   10.87    1.16   海达股份
18   new_hqhy  化纤行业    26    9.303846  ...  10.027    4.06    0.37   吉林化纤
19   new_jdhy  家电行业    37   13.049118  ...   6.127   53.00    3.06  苏 泊 尔
20   new_jdly  酒店旅游    35   16.324571  ...   6.809    2.51    0.16  *ST腾邦
21   new_jjhy  家具行业    16   14.207333  ...   3.830    4.88    0.18   海鸥住工
22   new_jrhy  金融行业    51   12.387200  ...   4.566   25.88    1.13   中信证券
23   new_jtys  交通运输    87    7.466835  ...   6.642    8.51    0.53   中国国航
24   new_jxhy  机械行业   211   12.314412  ...  10.012    9.56    0.87   宁波精达
25   new_jzjc  建筑建材    91    6.915568  ...   7.900   12.02    0.88   四川路桥
26    new_kfq   开发区    10    9.127000  ...   3.902   13.58    0.51   中国国贸
27   new_ljhy  酿酒行业    33  124.060303  ...  10.002  150.45   13.68   重庆啤酒
28    new_mtc   摩托车     6    9.040000  ...   5.878   13.87    0.77   钱江摩托
29   new_mthy  煤炭行业    41    8.143171  ...   5.121    3.90    0.19   ST安泰
30   new_nlmy  农林牧渔    64   11.500161  ...   7.095   14.49    0.96   盈康生命
31   new_nyhf  农药化肥    46   15.010000  ...   8.333   35.36    2.72   广信股份
32   new_qczz  汽车制造   103   17.203465  ...   9.967    9.93    0.90   天润工业
33   new_qtxy  其它行业   202   12.054028  ...  14.844    4.41    0.57    汤姆猫
34   new_slzp  塑料制品    33   12.096061  ...   5.975    6.74    0.38   浙江众成
35   new_snhy  水泥行业    26   10.774800  ...   5.584    8.13    0.43   博闻科技
36   new_sphy  食品行业    58   16.108214  ...  10.031   10.64    0.97   中粮科技
37  new_stock   次新股    76   41.233443  ...  10.015   35.92    3.27   青岛食品
38   new_swzz  生物制药   155   20.037059  ...  10.131    3.37    0.31   交大昂立
39   new_sybh  商业百货    93   10.116111  ...  10.023   23.82    2.17   天音控股
40   new_syhy  石油行业    24    6.421739  ...   7.438    5.20    0.36   泰山石油
41   new_tchy  陶瓷行业     8    9.295714  ...   5.040    3.96    0.19   ST冠福
42   new_wzwm  物资外贸    21    8.492000  ...  10.025   13.17    1.20   江苏国泰
43   new_ylqx  医疗器械    31   27.988333  ...   6.168  170.40    9.90   泰格医药
44   new_yqyb  仪器仪表    48   15.053125  ...   5.106    2.47    0.12   ST安控
45   new_ysbz  印刷包装    20    5.662000  ...   6.150    7.94    0.46   万顺新材
46   new_ysjs  有色金属    72   15.427206  ...   9.871   41.74    3.75   钢研高纳
47   new_zhhy  综合行业    33    6.043929  ...   6.833    6.41    0.41   合金投资
48   new_zzhy  造纸行业    24    6.515652  ...   6.367   17.04    1.02   中顺洁柔
```

#### 板块详情

接口: stock_sector_detail

目标地址: http://finance.sina.com.cn/stock/sl/#area_1

描述: 新浪行业-板块行情-成份详情, 由于新浪网页提供的统计数据有误, 部分行业数量大于统计数

限量: 单次获取指定的新浪行业-板块行情-成份详情

输入参数

| 名称     | 类型  | 描述                                                                        |
|--------|-----|---------------------------------------------------------------------------|
| sector | str | sector="hangye_ZL01"; 通过 **ak.stock_sector_spot** 返回数据的 label 字段选择 sector |

输出参数

| 名称            | 类型      | 描述  |
|---------------|---------|-----|
| symbol        | object  | -   |
| code          | object  | -   |
| name          | object  | -   |
| trade         | float64 | -   |
| pricechange   | float64 | -   |
| changepercent | float64 | -   |
| buy           | float64 | -   |
| sell          | float64 | -   |
| settlement    | float64 | -   |
| open          | float64 | -   |
| high          | float64 | -   |
| low           | float64 | -   |
| volume        | int64   | -   |
| amount        | int64   | -   |
| ticktime      | object  | -   |
| per           | float64 | -   |
| pb            | float64 | -   |
| mktcap        | float64 | -   |
| nmc           | float64 | -   |
| turnoverratio | float64 | -   |

接口示例

```python
import akshare as ak

stock_sector_detail_df = ak.stock_sector_detail(sector="hangye_ZL01")
print(stock_sector_detail_df)
```

数据示例

```
      symbol    code   name  ...        mktcap           nmc turnoverratio
0   sh600373  600373   中文传媒  ...  1.619301e+06  1.619301e+06       0.64600
1   sh600551  600551   时代出版  ...  3.616651e+05  3.616651e+05       0.22616
2   sh600633  600633   浙数文化  ...  1.249847e+06  1.249847e+06       1.06662
3   sh600757  600757   长江传媒  ...  6.310981e+05  6.310185e+05       0.26325
4   sh601098  601098   中南传媒  ...  1.925312e+06  1.925312e+06       0.23095
5   sh601801  601801   皖新传媒  ...  9.727211e+05  9.727211e+05       0.22932
6   sh601858  601858   中国科传  ...  7.280505e+05  7.280505e+05       0.59413
7   sh601900  601900   南方传媒  ...  8.206229e+05  8.147362e+05       0.31126
8   sh601928  601928   凤凰传媒  ...  1.702538e+06  1.702538e+06       0.11763
9   sh601999  601999   出版传媒  ...  3.261415e+05  3.261415e+05       0.60498
10  sh603999  603999   读者传媒  ...  3.317760e+05  3.317760e+05       0.61157
11  sz000504  000504  *ST生物  ...  2.246448e+05  2.239954e+05       0.83369
12  sz000719  000719   中原传媒  ...  6.855465e+05  4.469889e+05       0.32643
13  sz300148  300148   天舟文化  ...  2.999517e+05  2.770519e+05       4.02933
```

#### 股票列表-A股

接口: stock_info_a_code_name

目标地址: 沪深京三个交易所

描述: 沪深京 A 股股票代码和股票简称数据

限量: 单次获取所有 A 股股票代码和简称数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| code | object | -   |
| name | object | -   |

接口示例

```python
import akshare as ak

stock_info_a_code_name_df = ak.stock_info_a_code_name()
print(stock_info_a_code_name_df)
```

数据示例

```
        code   name
0     000001   平安银行
1     000002  万  科Ａ
2     000004   国华网安
3     000005   ST星源
4     000006   深振业Ａ
      ...    ...
4623  871396   常辅股份
4624  871553   凯腾精工
4625  871642   通易航天
4626  871981   晶赛科技
4627  872925   锦好医疗
```

#### 股票列表-上证

接口: stock_info_sh_name_code

目标地址: https://www.sse.com.cn/assortment/stock/list/share/

描述: 上海证券交易所股票代码和简称数据

限量: 单次获取所有上海证券交易所股票代码和简称数据

输入参数

| 名称     | 类型  | 描述                                               |
|--------|-----|--------------------------------------------------|
| symbol | str | symbol="主板A股"; choice of {"主板A股", "主板B股", "科创板"} |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 证券代码 | object | -   |
| 证券简称 | object | -   |
| 公司全称 | object | -   |
| 上市日期 | object | -   |

接口示例

```python
import akshare as ak

stock_info_sh_name_code_df = ak.stock_info_sh_name_code(symbol="主板A股")
print(stock_info_sh_name_code_df)
```

数据示例

```
        证券代码  证券简称                公司全称        上市日期
0     600000  浦发银行      上海浦东发展银行股份有限公司  1999-11-10
1     600004  白云机场      广州白云国际机场股份有限公司  2003-04-28
2     600006  东风汽车          东风汽车股份有限公司  1999-07-27
3     600007  中国国贸      中国国际贸易中心股份有限公司  1999-03-12
4     600008  首创环保    北京首创生态环保集团股份有限公司  2000-04-27
      ...   ...                 ...         ...
1691  605580  恒盛能源          恒盛能源股份有限公司  2021-08-19
1692  605588  冠石科技        南京冠石科技股份有限公司  2021-08-12
1693  605589  圣泉集团        济南圣泉集团股份有限公司  2021-08-10
1694  605598  上海港湾  上海港湾基础建设(集团)股份有限公司  2021-09-17
1695  605599  菜百股份       北京菜市口百货股份有限公司  2021-09-09
[1696 rows x 4 columns]
```

#### 股票列表-深证

接口: stock_info_sz_name_code

目标地址: https://www.szse.cn/market/product/stock/list/index.html

描述: 深证证券交易所股票代码和股票简称数据

限量: 单次获取深证证券交易所股票代码和简称数据

输入参数

| 名称     | 类型  | 描述                                                          |
|--------|-----|-------------------------------------------------------------|
| symbol | str | symbol="A股列表"; choice of {"A股列表", "B股列表", "CDR列表", "AB股列表"} |

输出参数-A股列表

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 板块     | object | -   |
| A股代码   | object | -   |
| A股简称   | object | -   |
| A股上市日期 | object | -   |
| A股总股本  | object | -   |
| A股流通股本 | object | -   |
| 所属行业   | object | -   |

接口示例

```python
import akshare as ak

stock_info_sz_name_code_df = ak.stock_info_sz_name_code(symbol="A股列表")
print(stock_info_sz_name_code_df)
```

数据示例

```
       板块   A股代码  A股简称   A股上市日期     A股总股本        A股流通股本    所属行业
0      主板  000001   平安银行  1991-04-03  19,405,918,198  19,405,513,424   J 金融业
1      主板  000002  万  科Ａ  1991-01-29   9,724,196,533   9,716,935,865   K 房地产
2      主板  000004   国华网安  1990-12-01     132,380,282     126,288,093  I 信息技术
3      主板  000005   ST星源  1990-12-10   1,058,536,842   1,057,875,742  N 公共环保
4      主板  000006   深振业Ａ  1992-04-27   1,349,995,046   1,349,987,396   K 房地产
   ...     ...    ...         ...             ...             ...     ...
2838  创业板  301568    思泰克  2023-11-28     103,258,400      24,487,502   C 制造业
2839  创业板  301577   美信科技  2024-01-24      44,260,000      11,095,149   C 制造业
2840  创业板  301578   辰奕智能  2023-12-28      48,000,000      11,378,776   C 制造业
2841  创业板  301589   诺瓦星云  2024-02-08      51,360,000       7,702,090   C 制造业
2842  创业板  301591  C肯特股份  2024-02-28      84,120,000      19,943,633   C 制造业
[2843 rows x 7 columns]
```

#### 股票列表-北证

接口: stock_info_bj_name_code

目标地址: https://www.bse.cn/nq/listedcompany.html

描述: 北京证券交易所股票代码和简称数据

限量: 单次获取北京证券交易所所有的股票代码和简称数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型     | 描述      |
|------|--------|---------|
| 证券代码 | object | -       |
| 证券简称 | object | -       |
| 总股本  | int64  | 注意单位: 股 |
| 流通股本 | int64  | 注意单位: 股 |
| 上市日期 | object | -       |
| 所属行业 | object | -       |
| 地区   | object | -       |
| 报告日期 | object | -       |

接口示例

```python
import akshare as ak

stock_info_bj_name_code_df = ak.stock_info_bj_name_code()
print(stock_info_bj_name_code_df)
```

数据示例

```
       证券代码  证券简称   总股本  ...              所属行业   地区        报告日期
0    430017  星昊医药  122577200  ...             医药制造业  北京市  2024-02-29
1    430047  诺思兰德  274873974  ...             医药制造业  北京市  2024-02-29
2    430090  同辉信息  199333546  ...        软件和信息技术服务业  北京市  2024-02-29
3    430139  华岭股份  266800000  ...  计算机、通信和其他电子设备制造业  上海市  2024-02-29
4    430198  微创光电  161363872  ...  计算机、通信和其他电子设备制造业  湖北省  2024-02-29
..      ...   ...        ...  ...               ...  ...         ...
239  873693   阿为特   72700000  ...             金属制品业  上海市  2024-02-29
240  873703  广厦环能   76900000  ...           专用设备制造业  北京市  2024-02-29
241  873726  卓兆点胶   82077246  ...           专用设备制造业  江苏省  2024-02-29
242  873806   云星宇  300736667  ...        软件和信息技术服务业  北京市  2024-02-29
243  873833  美心翼申   82360000  ...           通用设备制造业  重庆市  2024-02-29
[244 rows x 8 columns]
```

#### 终止/暂停上市-深证

接口: stock_info_sz_delist

目标地址: https://www.szse.cn/market/stock/suspend/index.html

描述: 深证证券交易所终止/暂停上市股票

限量: 单次获取深证证券交易所终止/暂停上市数据

输入参数

| 名称     | 类型  | 描述                                              |
|--------|-----|-------------------------------------------------|
| symbol | str | symbol="终止上市公司"; choice of {"暂停上市公司", "终止上市公司"} |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 证券代码   | object | -   |
| 证券简称   | object | -   |
| 上市日期   | object | -   |
| 终止上市日期 | object | -   |

接口示例

```python
import akshare as ak

stock_info_sz_delist_df = ak.stock_info_sz_delist(symbol="终止上市公司")
print(stock_info_sz_delist_df)
```

数据示例

```
      证券代码   证券简称  上市日期    终止上市日期
0    000003   PT金田Ａ  1991-01-14  2002-06-14
1    000013  *ST石化A  1992-05-06  2004-09-20
2    000015   PT中浩Ａ  1992-06-25  2001-10-25
3    000018    神城A退  1992-06-16  2020-01-07
4    000024    招商地产  1993-06-07  2015-12-30
..      ...     ...         ...         ...
145  300367     网力退  2014-01-29  2022-06-30
146  300372     欣泰退  2014-01-27  2017-08-28
147  300392     腾信退  2014-09-10  2023-07-10
148  300431     暴风退  2015-03-24  2020-11-10
149  300526     中潜退  2016-08-02  2023-07-12
[150 rows x 4 columns]
```

#### 两网及退市

接口: stock_staq_net_stop

目标地址: https://quote.eastmoney.com/center/gridlist.html#staq_net_board

描述: 东方财富网-行情中心-沪深个股-两网及退市

限量: 单次获取所有两网及退市的股票数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型     | 描述  |
|-----|--------|-----|
| 序号  | int64  | -   |
| 代码  | object | -   |
| 名称  | object | -   |

接口示例

```python
import akshare as ak

stock_staq_net_stop_df = ak.stock_staq_net_stop()
print(stock_staq_net_stop_df)
```

数据示例

```
      序号   代码      名称
0      1  400199    阳光城5
1      2  400170     运盛5
2      3  400039  华  圣 5
3      4  400119     西创5
4      5  400045     猴王5
..   ...     ...     ...
199  200  400149     金泰5
200  201  400155     海医5
201  202  400142     宝德5
202  203  400163     计通5
203  204  400046     博嘉5
[204 rows x 3 columns]
```

#### 暂停/终止上市-上证

接口: stock_info_sh_delist

目标地址: https://www.sse.com.cn/assortment/stock/list/delisting/

描述: 上海证券交易所暂停/终止上市股票

限量: 单次获取上海证券交易所暂停/终止上市股票

输入参数

| 名称     | 类型  | 描述                                         |
|--------|-----|--------------------------------------------|
| symbol | str | symbol="全部"; choice of {"全部", "沪市", "科创板"} |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 公司代码   | object | -   |
| 公司简称   | object | -   |
| 上市日期   | object | -   |
| 暂停上市日期 | object | -   |

接口示例

```python
import akshare as ak

stock_info_sh_delist_df = ak.stock_info_sh_delist(symbol="全部")
print(stock_info_sh_delist_df)
```

数据示例

```
     公司代码    公司简称   上市日期      暂停上市日期
0    600001    邯郸钢铁  1998-01-22  2009-12-29
1    600002    齐鲁退市  1998-04-08  2006-04-24
2    600003   ST东北高  1999-08-10  2010-02-26
3    600005    武钢股份  1999-08-03  2017-02-14
4    600065  *ST联 谊  1997-05-23  2007-12-13
..      ...     ...         ...         ...
109  900935    阳晨Ｂ股  1995-07-27  2016-12-16
110  900949    东电Ｂ股  1997-09-23  2013-11-07
111  900950    新城Ｂ股  1997-10-16  2015-11-23
112  900951    退市大化  1997-10-21  2020-08-28
113  900956    东贝Ｂ股  1999-07-15  2020-11-23
[114 rows x 4 columns]
```

#### 股票更名

接口: stock_info_change_name

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vCI_CorpInfo/stockid/300378.phtml

描述: 新浪财经-股票曾用名

限量: 单次指定 symbol 的所有历史曾用名称

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="000503"; 股票代码 |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| index | int64  | -   |
| name  | object | -   |

接口示例

```python
import akshare as ak

stock_info_change_name_list = ak.stock_info_change_name(symbol="000503")
print(stock_info_change_name_list)
```

数据示例

```
   index  name
0      1   琼海虹
1      2  海虹控股
2      3  ST海虹
3      4  海虹控股
4      5   G海虹
5      6  海虹控股
6      7  国新健康
```

#### 名称变更-深证

接口: stock_info_sz_change_name

目标地址: https://www.szse.cn/www/market/stock/changename/index.html

描述: 深证证券交易所-市场数据-股票数据-名称变更

限量: 单次获取所有历史数据

输入参数

| 名称     | 类型  | 描述                                        |
|--------|-----|-------------------------------------------|
| symbol | str | symbol="全称变更"; choice of {"全称变更", "简称变更"} |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| 变更日期  | object | -   |
| 证券代码  | object | -   |
| 证券简称  | object | -   |
| 变更前全称 | object | -   |
| 变更后全称 | object | -   |

接口示例

```python
import akshare as ak

stock_info_sz_change_name_df = ak.stock_info_sz_change_name(symbol="全称变更")
print(stock_info_sz_change_name_df)
```

数据示例

```
       变更日期    证券代码   证券简称               变更前全称             变更后全称
0     1992-02-28  000004   国华网安        蛇口安达运输股份有限公司   深圳市蛇口安达实业股份有限公司
1     1993-03-08  200012  南  玻Ｂ        中国南方玻璃股份有限公司      中国南玻集团股份有限公司
2     1993-03-08  000012  南  玻Ａ        中国南方玻璃股份有限公司      中国南玻集团股份有限公司
3     1993-07-12  000009   中国宝安   深圳市宝安企业(集团)股份有限公司      中国宝安集团股份有限公司
4     1993-12-20  000005   ST星源        深圳原野实业股份有限公司      深圳世纪星源股份有限公司
          ...     ...    ...                 ...               ...
1614  2024-01-30  002986   宇新股份      湖南宇新能源科技股份有限公司    广东宇新能源科技股份有限公司
1615  2024-01-31  301418   协昌科技      江苏协昌电子科技股份有限公司  江苏协昌电子科技集团股份有限公司
1616  2024-02-05  300617   安靠智电  江苏安靠智能输电工程科技股份有限公司      江苏安靠智电股份有限公司
1617  2024-02-07  002060   广东建工        广东水电二局股份有限公司   广东省建筑工程集团股份有限公司
1618  2024-02-28  300478   杭州高新      杭州高新橡塑材料股份有限公司    杭州高新材料科技股份有限公司
[1619 rows x 5 columns]
```

#### 基金持股

接口: stock_fund_stock_holder

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vCI_FundStockHolder/stockid/600004.phtml

描述: 新浪财经-股本股东-基金持股

限量: 新浪财经-股本股东-基金持股所有历史数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="600004"; 股票代码 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 基金名称   | object  | -       |
| 基金代码   | object  | -       |
| 持仓数量   | int64   | 注意单位: 股 |
| 占流通股比例 | float64 | 注意单位: % |
| 持股市值   | int64   | 注意单位: 元 |
| 占净值比例  | float64 | 注意单位: % |
| 截止日期   | object  | -       |

接口示例

```python
import akshare as ak

stock_fund_stock_holder_df = ak.stock_fund_stock_holder(symbol="601318")
print(stock_fund_stock_holder_df)
```

数据示例

```
    基金名称    基金代码      持仓数量  占流通股比例        持股市值  占净值比例        截止日期
0      银华沪深300价值ETF  562320    265300  0.0015    12981100   3.85  2023-01-05
1    华安沪深300增强策略ETF  561000     91200  0.0005     4425940   0.53  2023-01-04
2         华夏上证50ETF  510050  92794672  0.5076  4361350000   7.48  2022-12-31
3      华泰柏瑞沪深300ETF  510300  44522018  0.2436  2092530000   2.70  2022-12-31
4     易方达沪深300非银ETF  512070  34406219  0.1882  1617090000  29.08  2022-12-31
..              ...     ...       ...     ...         ...    ...         ...
995       安信民稳增长混合A  008809   1125037  0.0062    46779000   2.68  2022-09-30
996       建信上证50ETF  510800   1084221  0.0059    45081900   6.86  2022-09-30
997   国寿安保沪港深300ETF  517300   1016300  0.0056    42257800   2.09  2022-09-30
998      泰达宏利市值优选混合  162209   1013832  0.0055    42155100   6.45  2022-09-30
999      南方沪深300ETF  159925    961652  0.0053    39985500   2.52  2022-09-30
```

#### 主要股东

接口: stock_main_stock_holder

目标地址: https://vip.stock.finance.sina.com.cn/corp/go.php/vCI_StockHolder/stockid/600004.phtml

描述: 新浪财经-股本股东-主要股东

限量: 单次获取所有历史数据

输入参数

| 名称    | 类型  | 描述                   |
|-------|-----|----------------------|
| stock | str | stock="600004"; 股票代码 |

输出参数

| 名称    | 类型      | 描述         |
|-------|---------|------------|
| 编号    | object  | -          |
| 股东名称  | object  | -          |
| 持股数量  | float64 | 注意单位: 股    |
| 持股比例  | float64 | 注意单位: %    |
| 股本性质  | object  | -          |
| 截至日期  | object  | -          |
| 公告日期  | object  | -          |
| 股东说明  | object  | -          |
| 股东总数  | float64 | -          |
| 平均持股数 | float64 | 备注: 按总股本计算 |

接口示例

```python
import akshare as ak

stock_main_stock_holder_df = ak.stock_main_stock_holder(stock="600004")
print(stock_main_stock_holder_df)
```

数据示例

```
      编号                股东名称          持股数量  ...  股东说明      股东总数     平均持股数
0      1          上海国际集团有限公司  6.331323e+09  ...   NaN  164613.0  178312.0
1      2      中国移动通信集团广东有限公司  5.334893e+09  ...   NaN  164613.0  178312.0
2      3   富德生命人寿保险股份有限公司-传统  2.779437e+09  ...   NaN  164613.0  178312.0
3      4  富德生命人寿保险股份有限公司-资本金  1.763232e+09  ...   NaN  164613.0  178312.0
4      5       上海上国投资产管理有限公司  1.395571e+09  ...   NaN  164613.0  178312.0
...   ..                 ...           ...  ...   ...       ...       ...
1383   6        东方国际（集团）有限公司  5.000000e+07  ...   NaN       NaN       NaN
1384   7            申能股份有限公司  5.000000e+07  ...   NaN       NaN       NaN
1385   8    上海外高桥保税区开发股份有限公司  4.000000e+07  ...   NaN       NaN       NaN
1386   9      上海市城市建设投资开发总公司  3.800000e+07  ...   NaN       NaN       NaN
1387  10          中国华北电力集团公司  3.000000e+07  ...   NaN       NaN       NaN
[1388 rows x 10 columns]
```

#### 机构持股

##### 机构持股一览表

接口: stock_institute_hold

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vComStockHold/kind/jgcg/index.phtml

描述: 新浪财经-机构持股-机构持股一览表

限量: 单次获取所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                                                              |
|--------|-----|-----------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="20051"; 从 2005 年开始, {"一季报":1, "中报":2 "三季报":3 "年报":4}, e.g., "20191", 其中的 1 表示一季报; "20193", 其中的 3 表示三季报; |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 证券代码     | object  | -       |
| 证券简称     | object  | -       |
| 机构数      | int64   | -       |
| 机构数变化    | int64   | -       |
| 持股比例     | float64 | 注意单位: % |
| 持股比例增幅   | float64 | 注意单位: % |
| 占流通股比例   | float64 | 注意单位: % |
| 占流通股比例增幅 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_institute_hold_df = ak.stock_institute_hold(symbol="20201")
print(stock_institute_hold_df)
```

数据示例

```
      证券代码  证券简称 机构数  机构数变化   持股比例  持股比例增幅  占流通股比例  占流通股比例增幅
0     002296  辉煌科技    1      1   0.85    0.85    0.85      0.85
1     300354  东华测试    1      0   0.21   -0.59    0.21     -0.59
2     000656  金科股份    1      0   2.12    0.04    2.14      0.03
3     002859  洁美科技    1      0   0.97    0.00    2.28      0.00
4     603866  桃李面包    1      0   1.34    0.00    1.34      0.00
      ...   ...  ...    ...    ...     ...     ...       ...
1790  300782   卓胜微   10      2   6.00    2.02   24.00      8.07
1791  300788  中信出版   10      0  10.26    2.88   41.03     11.50
1792  300785   值得买   10      1  11.62    4.14   46.48     16.56
1793  688188  柏楚电子   10      0   5.17   -0.13   21.43     -1.70
1794  300003  乐普医疗   16      1  11.16    0.99   11.78      1.05
[1795 rows x 8 columns]
```

##### 机构持股详情

接口: stock_institute_hold_detail

目标地址: http://vip.stock.finance.sina.com.cn/q/go.php/vComStockHold/kind/jgcg/index.phtml

描述: 新浪财经-机构持股-机构持股详情

限量: 单次所有历史数据

输入参数

| 名称      | 类型  | 描述                                                                                                               |
|---------|-----|------------------------------------------------------------------------------------------------------------------|
| stock   | str | stock="300003"; 股票代码                                                                                             |
| quarter | str | quarter="20201"; 从 2005 年开始, {"一季报":1, "中报":2 "三季报":3 "年报":4}, e.g., "20191", 其中的 1 表示一季报; "20193", 其中的 3 表示三季报; |

输出参数

| 名称       | 类型      | 描述       |
|----------|---------|----------|
| 持股机构类型   | object  | -        |
| 持股机构代码   | object  | -        |
| 持股机构简称   | object  | -        |
| 持股机构全称   | object  | -        |
| 持股数      | float64 | 注意单位: 万股 |
| 最新持股数    | float64 | 注意单位: 万股 |
| 持股比例     | float64 | 注意单位: %  |
| 最新持股比例   | float64 | 注意单位: %  |
| 占流通股比例   | float64 | 注意单位: %  |
| 最新占流通股比例 | float64 | 注意单位: %  |
| 持股比例增幅   | float64 | 注意单位: %  |
| 占流通股比例增幅 | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_institute_hold_detail_df = ak.stock_institute_hold_detail(stock="300003", quarter="20201")
print(stock_institute_hold_detail_df)
```

数据示例

```
   持股机构类型  持股机构代码      持股机构简称  ... 最新占流通股比例  持股比例增幅  占流通股比例增幅
0      基金    519069          汇添富价值精选混合A  ...     1.18   -0.22     -0.23
1      基金    159915           易方达创业板ETF  ...     0.73    0.02      0.02
2      基金    001417      汇添富医疗服务灵活配置混合A  ...     1.01   -0.35     -0.37
3      基金    470009          汇添富民营活力混合A  ...     0.48    0.03      0.02
4      基金    159949          华安创业板50ETF  ...     0.36    0.13      0.13
5      基金    000831            工银医疗保健股票  ...     0.57   -0.09     -0.10
6    全国社保  70010117         全国社保基金一一七组合  ...     1.30   -0.22     -0.23
7    全国社保  70010104         全国社保基金一零四组合  ...     0.60   -0.06     -0.07
8    全国社保  70010503         全国社保基金五零三组合  ...     0.38    0.07      0.07
9      保险  70304136  华夏人寿保险股份有限公司－自有...  ...     0.00    1.39      1.47
10   QFII  70304002         挪威中央银行－自有资金  ...     0.90    0.25      0.27
11   QFII  70304274        澳门金融管理局－自有资金  ...     0.99    0.00      0.00
12   QFII  80126479           魁北克储蓄投资集团  ...     0.71    0.00      0.00
13   QFII  70304150  鼎晖投资咨询新加坡有限公司－鼎...  ...     0.57    0.01      0.01
14   QFII  80128964             阿布达比投资局  ...     0.56    0.02      0.02
15   QFII  80195934            科威特政府投资局  ...     0.39    0.04      0.04
[16 rows x 12 columns]
```

#### 机构推荐

##### 机构推荐池

接口: stock_institute_recommend

目标地址: http://stock.finance.sina.com.cn/stock/go.php/vIR_RatingNewest/index.phtml

描述: 新浪财经-机构推荐池-具体指标的数据

限量: 单次获取新浪财经-机构推荐池-具体指标的所有数据

输入参数

| 名称     | 类型  | 描述                                                                                                                 |
|--------|-----|--------------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="行业关注度"; choice of {'最新投资评级', '上调评级股票', '下调评级股票', '股票综合评级', '首次评级股票', '目标涨幅排名', '机构关注度', '行业关注度', '投资评级选股'} |

输出参数

| 名称  | 类型  | 描述                |
|-----|-----|-------------------|
| -   | -   | 根据特定 indicator 而定 |

接口示例

```python
import akshare as ak

stock_institute_recommend_df = ak.stock_institute_recommend(symbol="投资评级选股")
print(stock_institute_recommend_df)
```

数据示例

```
        股票代码   股票名称 最新评级     目标价       评级日期↓ 综合评级    平均涨幅    行业
0     000001   平安银行   买入   19.73  2020-05-27   买入  51.75%    银行
1     300403   汉宇集团   中性    5.70  2020-05-27   中性  18.26%  白色家电
2     603288   海天味业   买入  112.85  2020-05-27   买入  -1.32%  食品加工
3     601238   广汽集团   买入   12.72  2020-05-27   买入  34.62%  汽车整车
4     002709   天赐材料   增持   34.24  2020-05-27   买入   18.1%  化学制品
      ...    ...  ...     ...         ...  ...     ...   ...
1053  601398   工商银行   买入    7.14  2020-05-27   买入  39.54%    银行
1054  002078   太阳纸业   买入   11.22  2020-05-27   买入  35.64%    造纸
1055  002080   中材科技   买入   14.59  2020-05-27   买入  22.61%  化学制品
1056  002081  金 螳 螂   增持   12.55  2020-05-27   买入  55.51%  装修装饰
1057  601658   邮储银行   增持    6.52  2020-05-27   买入  28.25%    银行
```

##### 股票评级记录

接口: stock_institute_recommend_detail

目标地址: http://stock.finance.sina.com.cn/stock/go.php/vIR_StockSearch/key/sz000001.phtml

描述: 新浪财经-机构推荐池-股票评级记录

限量: 单次获取新浪财经-机构推荐池-股票评级记录的所有数据

输入参数

| 名称     | 类型  | 描述              |
|--------|-----|-----------------|
| symbol | str | symbol="000001" |

输出参数

| 名称   | 类型  | 描述  |
|------|-----|-----|
| 股票代码 | str | -   |
| 股票名称 | str | -   |
| 目标价  | str | -   |
| 最新评级 | str | -   |
| 评级机构 | str | -   |
| 分析师  | str | -   |
| 行业   | str | -   |
| 评级日期 | str | -   |

接口示例

```python
import akshare as ak

stock_institute_recommend_detail_df = ak.stock_institute_recommend_detail(symbol="002709")
print(stock_institute_recommend_detail_df)
```

数据示例

```
       股票代码  股票名称    目标价 最新评级  评级机构        分析师    行业        评级日期
0    002709  天赐材料    NaN   增持  国海证券      谭倩,尹斌  化学制品  2020-05-27
1    002709  天赐材料    NaN   增持  光大证券     吴裕,裘孝锋  化学制品  2020-05-07
2    002709  天赐材料  35.20   买入  华创证券  杨达伟,于潇,邱迪  化学制品  2020-05-06
3    002709  天赐材料  33.25   买入  东吴证券        曾朵红  化学制品  2020-05-06
4    002709  天赐材料  35.50   买入  招商证券    游家训,刘珺涵  化学制品  2020-05-01
..      ...   ...    ...  ...   ...        ...   ...         ...
241  002709  天赐材料    NaN   买入  招商证券    游家训,陈雁冰  化学制品  2017-09-07
242  002709  天赐材料    NaN   买入  方正证券        韩振国  化学制品  2017-09-07
243  002709  天赐材料  61.82   买入  招商证券    游家训,陈雁冰  化学制品  2017-09-07
244  002709  天赐材料  59.82   买入  中泰证券        邵晶鑫  化学制品  2017-09-06
245  002709  天赐材料    NaN   买入  国海证券        代鹏举  化学制品  2017-09-04
```

##### 投资评级

接口: stock_rank_forecast_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-评级预测-投资评级

限量: 单次获取指定交易日的所有数据

输入参数

| 名称   | 类型  | 描述                   |
|------|-----|----------------------|
| date | str | date="20210910"; 交易日 |

输出参数

| 名称      | 类型      | 描述  |
|---------|---------|-----|
| 证券代码    | object  | -   |
| 证券简称    | object  | -   |
| 发布日期    | object  | -   |
| 研究机构简称  | object  | -   |
| 研究员名称   | object  | -   |
| 投资评级    | object  | -   |
| 是否首次评级  | object  | -   |
| 评级变化    | object  | -   |
| 前一次投资评级 | object  | -   |
| 目标价格-下限 | float64 | -   |
| 目标价格-上限 | float64 | -   |

接口示例

```python
import akshare as ak

stock_rank_forecast_cninfo_df = ak.stock_rank_forecast_cninfo(date="20230817")
print(stock_rank_forecast_cninfo_df)
```

数据示例

```
     证券代码  证券简称   发布日期 研究机构简称  ... 评级变化 前一次投资评级 目标价格-下限 目标价格-上限
0    000552  甘肃能化  2023-08-17   民生证券  ...   维持      增持     NaN     NaN
1    000552  甘肃能化  2023-08-17   山西证券  ...   未知     不评级     NaN     NaN
2    000552  甘肃能化  2023-08-17   开源证券  ...   维持      买入     NaN     NaN
3    000567  海德股份  2023-08-17   中信证券  ...   未知     不评级     NaN     NaN
4    000682  东方电子  2023-08-17   光大证券  ...   维持      买入     NaN     NaN
..      ...   ...         ...    ...  ...  ...     ...     ...     ...
329  688668  鼎通科技  2023-08-17   民生证券  ...   维持      增持     NaN     NaN
330  688981  中芯国际  2023-08-17   中信建投  ...   未知     不评级     NaN     NaN
331  688981  中芯国际  2023-08-17   中泰证券  ...   维持      买入     NaN     NaN
332  836419  万德股份  2023-08-17   开源证券  ...   未知     不评级     NaN     NaN
333  839371  欧福蛋业  2023-08-17   开源证券  ...   维持      增持     NaN     NaN
[334 rows x 11 columns]
```

##### 申万个股行业分类变动历史

接口: stock_industry_clf_hist_sw

目标地址: http://www.swhyresearch.com/institute_sw/allIndex/downloadCenter/industryType

描述: 申万宏源研究-行业分类-全部行业分类

限量: 单次获取所有个股的行业分类变动历史数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称            | 类型     | 描述     |
|---------------|--------|--------|
| symbol        | object | 股票代码   |
| start_date    | object | 计入日期   |
| industry_code | object | 申万行业代码 |
| update_time   | object | 更新日期   |

接口示例

```python
import akshare as ak

stock_industry_clf_hist_sw_df = ak.stock_industry_clf_hist_sw()
print(stock_industry_clf_hist_sw_df)
```

数据示例

```
       symbol  start_date industry_code update_time
0      000001  1991-04-03        440101  2015-10-27
1      000001  2014-01-01        480101  2015-10-27
2      000001  2021-07-30        480301  2021-07-31
3      000002  1991-01-29        430101  2015-10-27
4      000003  1991-04-14        510101  2015-10-27
...       ...         ...           ...         ...
12360  873706  2024-03-12        640601  2024-03-13
12361  873726  2023-10-30        640209  2023-10-30
12362  873806  2024-01-17        710301  2024-01-17
12363  873833  2023-11-20        640106  2023-11-20
12364  874090  2023-08-21        370304  2023-08-21
[12365 rows x 4 columns]
```

##### 行业市盈率

接口: stock_industry_pe_ratio_cninfo

目标地址: http://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-行业分析-行业市盈率

限量: 单次获取指定 symbol 在指定交易日的所有数据; 只能获取近期的数据

输入参数

| 名称     | 类型  | 描述                                                |
|--------|-----|---------------------------------------------------|
| symbol | str | symbol="证监会行业分类"; choice of {"证监会行业分类", "国证行业分类"} |
| date   | str | date="20210910"; 交易日                              |

输出参数

| 名称         | 类型      | 描述       |
|------------|---------|----------|
| 变动日期       | object  | -        |
| 行业分类       | object  | -        |
| 行业层级       | int64   | -        |
| 行业编码       | object  | -        |
| 行业名称       | object  | -        |
| 公司数量       | float64 | -        |
| 纳入计算公司数量   | float64 | -        |
| 总市值-静态     | float64 | 注意单位: 亿元 |
| 净利润-静态     | float64 | 注意单位: 亿元 |
| 静态市盈率-加权平均 | float64 | -        |
| 静态市盈率-中位数  | float64 | -        |
| 静态市盈率-算术平均 | float64 | -        |

接口示例

```python
import akshare as ak

stock_industry_pe_ratio_cninfo_df = ak.stock_industry_pe_ratio_cninfo(symbol="国证行业分类", date="20240617")
print(stock_industry_pe_ratio_cninfo_df)
```

数据示例

```
        变动日期          行业分类  行业层级  ... 静态市盈率-加权平均 静态市盈率-中位数  静态市盈率-算术平均
0    2024-06-17  国证行业分类标准2019     1  ...      11.54     15.75       49.19
1    2024-06-17  国证行业分类标准2019     2  ...      11.54     15.75       49.19
2    2024-06-17  国证行业分类标准2019     3  ...      18.44     22.05       28.24
3    2024-06-17  国证行业分类标准2019     4  ...      19.23     21.45       30.02
4    2024-06-17  国证行业分类标准2019     4  ...      18.10     24.86       25.35
..          ...           ...   ...  ...        ...       ...         ...
288  2024-06-17  国证行业分类标准2019     4  ...        NaN       NaN         NaN
289  2024-06-17  国证行业分类标准2019     3  ...      19.59     22.28       43.17
290  2024-06-17  国证行业分类标准2019     4  ...      19.59     22.28       43.17
291  2024-06-17  国证行业分类标准2019     3  ...      14.12     14.12       14.12
292  2024-06-17  国证行业分类标准2019     4  ...      14.12     14.12       14.12
[293 rows x 12 columns]
```

##### 新股过会

接口: stock_new_gh_cninfo

目标地址: https://webapi.cninfo.com.cn/#/xinguList

描述: 巨潮资讯-数据中心-新股数据-新股过会

限量: 单次获取近一年所有新股过会的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| 公司名称  | object | -   |
| 上会日期  | object | -   |
| 审核类型  | object | -   |
| 审议内容  | object | -   |
| 审核结果  | object | -   |
| 审核公告日 | object | -   |

接口示例

```python
import akshare as ak

stock_new_gh_cninfo_df = ak.stock_new_gh_cninfo()
print(stock_new_gh_cninfo_df)
```

数据示例

```
              公司名称        上会日期  审核类型 审议内容 审核结果       审核公告日
0          兰州银行股份有限公司  2021-09-09  首发新股   首发   通过  2021-09-09
1          永安期货股份有限公司  2021-09-09  首发新股   首发   通过  2021-09-09
2        杭州福莱蒽特股份有限公司  2021-09-09  首发新股   首发   通过  2021-09-09
3    新疆派特罗尔能源服务股份有限公司  2021-09-02  首发新股   首发  未通过  2021-09-02
4      浙江盛泰服装集团股份有限公司  2021-08-26  首发新股   首发   通过  2021-08-26
..                ...         ...   ...  ...  ...         ...
125      河南蓝天燃气股份有限公司  2020-09-17  首发新股   首发   通过  2020-09-17
126      北京声迅电子股份有限公司  2020-09-17  首发新股   首发   通过  2020-09-17
127    天津友发钢管集团股份有限公司  2020-09-17  首发新股   首发   通过  2020-09-17
128     联泓新材料科技股份有限公司  2020-09-17  首发新股   首发   通过  2020-09-17
129      中国国际金融股份有限公司  2020-09-17  首发新股   首发   通过  2020-09-17
```

##### 新股发行

接口: stock_new_ipo_cninfo

目标地址: https://webapi.cninfo.com.cn/#/xinguList

描述: 巨潮资讯-数据中心-新股数据-新股发行

限量: 单次获取近三年所有新股发行的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述       |
|---------|---------|----------|
| 证劵代码    | object  | -        |
| 证券简称    | object  | -        |
| 上市日期    | object  | -        |
| 申购日期    | object  | -        |
| 发行价     | float64 | 注意单位: 元  |
| 总发行数量   | float64 | 注意单位: 万股 |
| 发行市盈率   | float64 | -        |
| 上网发行中签率 | float64 | 注意单位: %  |
| 摇号结果公告日 | object  | -        |
| 中签公告日   | object  | -        |
| 中签缴款日   | object  | -        |
| 网上申购上限  | float64 | -        |
| 上网发行数量  | float64 | -        |

接口示例

```python
import akshare as ak

stock_new_ipo_cninfo_df = ak.stock_new_ipo_cninfo()
print(stock_new_ipo_cninfo_df)
```

数据示例

```
     证劵代码  证券简称   上市日期  ...   中签缴款日  网上申购上限 上网发行数量
0    301592  六九一二         NaT  ...  2024-10-15    1.75  1750.00
1    301613  新铝时代         NaT  ...  2024-10-18    0.65  1162.70
2    301626  苏州天脉         NaT  ...  2024-10-17    0.70  1258.05
3    301628  强达电路         NaT  ...  2024-10-23    1.85  1884.40
4    603205   健尔康         NaT  ...  2024-10-29     NaN  1200.00
..      ...   ...         ...  ...         ...     ...      ...
926  301087  可孚医疗  2021-10-25  ...  2021-10-15    1.10  1918.55
927  605566  福莱蒽特  2021-10-25  ...  2021-10-13    1.30  3000.60
928  688255   凯尔达  2021-10-25  ...  2021-10-14    0.45   674.75
929  872925  锦好医疗  2021-10-25  ...  2021-10-08   51.63  1032.60
930  688737  中自科技  2021-10-22  ...  2021-10-14    0.50   734.70
[931 rows x 13 columns]
```

##### 董监高及相关人员持股变动-上证

接口: stock_share_hold_change_sse

目标地址: http://www.sse.com.cn/disclosure/credibility/supervision/change/

描述: 上海证券交易所-披露-监管信息公开-公司监管-董董监高人员股份变动

限量: 单次获取指定 symbol 的数据

输入参数

| 名称     | 类型  | 描述                                          |
|--------|-----|---------------------------------------------|
| symbol | str | symbol="600000"; choice of {"全部", "具体股票代码"} |

输出参数

| 名称       | 类型      | 描述 |
|----------|---------|----|
| 公司代码     | object  | -  |
| 公司名称     | object  | -  |
| 姓名       | object  | -  |
| 职务       | object  | -  |
| 股票种类     | object  | -  |
| 货币种类     | object  | -  |
| 本次变动前持股数 | int64   | -  |
| 变动数      | int64   | -  |
| 本次变动平均价格 | float64 | -  |
| 变动后持股数   | int64   | -  |
| 变动原因     | object  | -  |
| 变动日期     | object  | -  |
| 填报日期     | object  | -  |

接口示例

```python
import akshare as ak

stock_share_hold_change_sse_df = ak.stock_share_hold_change_sse(symbol="600000")
print(stock_share_hold_change_sse_df)
```

数据示例

```
    公司代码  公司名称   姓名         职务  ...  变动后持股数    变动原因   变动日期        填报日期
0   600000  浦发银行   谢伟     高级管理人员  ...  217000  二级市场买卖  2021-07-15  2021-07-16
1   600000  浦发银行  王新浩     高级管理人员  ...  231000  二级市场买卖  2021-07-15  2021-07-16
2   600000  浦发银行  潘卫东  董事、高级管理人员  ...  400000  二级市场买卖  2021-07-15  2021-07-16
3   600000  浦发银行  刘以研  董事、高级管理人员  ...  235900  二级市场买卖  2021-07-15  2021-07-16
4   600000  浦发银行  崔炳文     高级管理人员  ...  206700  二级市场买卖  2021-07-15  2021-07-16
5   600000  浦发银行   谢伟     高级管理人员  ...  158000  二级市场买卖  2020-07-16  2020-07-17
6   600000  浦发银行  王新浩     高级管理人员  ...  171000  二级市场买卖  2020-07-16  2020-07-17
7   600000  浦发银行  潘卫东  董事、高级管理人员  ...  200000  二级市场买卖  2020-07-16  2020-07-17
8   600000  浦发银行  刘以研  董事、高级管理人员  ...  177400  二级市场买卖  2020-07-15  2020-07-17
9   600000  浦发银行  崔炳文     高级管理人员  ...  148700  二级市场买卖  2020-07-16  2020-07-17
10  600000  浦发银行  刘以研  董事、高级管理人员  ...  163500  二级市场买卖  2020-07-10  2020-07-15
11  600000  浦发银行  刘以研  董事、高级管理人员  ...  168500  二级市场买卖  2020-07-13  2020-07-15
12  600000  浦发银行  刘以研  董事、高级管理人员  ...  173500  二级市场买卖  2020-07-14  2020-07-15
13  600000  浦发银行  徐海燕     高级管理人员  ...  108000  二级市场买卖  2019-06-10  2019-06-11
14  600000  浦发银行   谢伟     高级管理人员  ...  106000  二级市场买卖  2019-06-10  2019-06-11
15  600000  浦发银行  王新浩     高级管理人员  ...  111000  二级市场买卖  2019-06-10  2019-06-11
16  600000  浦发银行  潘卫东     高级管理人员  ...  120000  二级市场买卖  2019-06-10  2019-06-11
17  600000  浦发银行  刘以研     高级管理人员  ...  103500  二级市场买卖  2019-06-10  2019-06-11
18  600000  浦发银行  刘信义     高级管理人员  ...  160000  二级市场买卖  2019-06-10  2019-06-11
19  600000  浦发银行  崔炳文     高级管理人员  ...   99700  二级市场买卖  2019-06-10  2019-06-11
20  600000  浦发银行  崔炳文     高级管理人员  ...   51700  二级市场买卖  2018-07-17  2018-07-17
21  600000  浦发银行  徐海燕     高级管理人员  ...   48000  二级市场买卖  2018-07-12  2018-07-12
22  600000  浦发银行   谢伟     高级管理人员  ...   53000  二级市场买卖  2018-07-11  2018-07-12
23  600000  浦发银行  王新浩     高级管理人员  ...   60000  二级市场买卖  2018-07-11  2018-07-12
24  600000  浦发银行  潘卫东     高级管理人员  ...   55000  二级市场买卖  2018-07-11  2018-07-12
25  600000  浦发银行  刘以研     高级管理人员  ...   52500  二级市场买卖  2018-07-11  2018-07-12
26  600000  浦发银行  刘信义     高级管理人员  ...   80000  二级市场买卖  2018-07-11  2018-07-12
[27 rows x 13 columns]
```

##### 董监高及相关人员持股变动-深证

接口: stock_share_hold_change_szse

目标地址: http://www.szse.cn/disclosure/supervision/change/index.html

描述: 深圳证券交易所-信息披露-监管信息公开-董监高人员股份变动

限量: 单次获取指定 symbol 的数据

输入参数

| 名称     | 类型  | 描述                                          |
|--------|-----|---------------------------------------------|
| symbol | str | symbol="001308"; choice of {"全部", "具体股票代码"} |

输出参数

| 名称         | 类型      | 描述         |
|------------|---------|------------|
| 证券代码       | object  | -          |
| 证券简称       | object  | -          |
| 董监高姓名      | object  | -          |
| 变动日期       | object  | -          |
| 变动股份数量     | float64 | 注意单位: 万股   |
| 成交均价       | float64 | -          |
| 变动原因       | object  | -          |
| 变动比例       | float64 | 注意单位: 千分之一 |
| 当日结存股数     | float64 | 注意单位: 万股   |
| 股份变动人姓名    | object  | -          |
| 职务         | object  | -          |
| 变动人与董监高的关系 | object  | -          |

接口示例

```python
import akshare as ak

stock_share_hold_change_szse_df = ak.stock_share_hold_change_szse(symbol="001308")
print(stock_share_hold_change_szse_df)
```

数据示例

```
   证券代码  证券简称 董监高姓名 变动日期  ...    当日结存股数  股份变动人姓名     职务  变动人与董监高的关系
0  001308  康冠科技    凌斌  2023-08-21  ...  18752.80       凌斌     董事          本人
1  001308  康冠科技    凌斌  2023-08-18  ...  18743.80       凌斌     董事          本人
2  001308  康冠科技    凌斌  2023-08-17  ...  18737.80       凌斌     董事          本人
3  001308  康冠科技    凌斌  2023-08-16  ...  18734.80       凌斌     董事          本人
4  001308  康冠科技    江微  2023-08-09  ...      1.00       江微     监事          本人
5  001308  康冠科技    吴远  2023-08-08  ...      7.48       吴远     高管          本人
6  001308  康冠科技   孙建华  2023-08-08  ...      3.88      孙建华  董秘、高管          本人
7  001308  康冠科技    吴远  2022-08-24  ...      4.95       吴远     高管          本人
8  001308  康冠科技    吴远  2022-06-17  ...      3.55       吴远     高管          本人
[9 rows x 12 columns]
```

##### 董监高及相关人员持股变动-北证

接口: stock_share_hold_change_bse

目标地址: https://www.bse.cn/disclosure/djg_sharehold_change.html

描述: 北京证券交易所-信息披露-监管信息-董监高及相关人员持股变动

限量: 单次获取指定 symbol 的数据

输入参数

| 名称     | 类型  | 描述                                          |
|--------|-----|---------------------------------------------|
| symbol | str | symbol="430489"; choice of {"全部", "具体股票代码"} |

输出参数

| 名称     | 类型      | 描述       |
|--------|---------|----------|
| 代码     | object  | -        |
| 简称     | object  | -        |
| 姓名     | object  | -        |
| 职务     | object  | -        |
| 变动日期   | object  | -        |
| 变动股数   | float64 | 注意单位: 万股 |
| 变动前持股数 | float64 | 注意单位: 万股 |
| 变动后持股数 | float64 | 注意单位: 万股 |
| 变动均价   | float64 | 注意单位: 元  |
| 变动原因   | object  | -        |

接口示例

```python
import akshare as ak

stock_share_hold_change_bse_df = ak.stock_share_hold_change_bse(symbol="430489")
print(stock_share_hold_change_bse_df)
```

数据示例

```
   代码    简称   姓名     职务        变动日期   变动股数   变动前持股数   变动后持股数  变动均价  变动原因
0  430489  佳先股份   李兑      董事  2023-07-28  7.151   0.0000   7.1510  4.66  竞价交易
1  430489  佳先股份   李平  高级管理人员  2023-07-14  2.000  23.0565  25.0565  4.64  竞价交易
2  430489  佳先股份  周星源  高级管理人员  2023-06-21  2.000  28.2896  30.2896  4.59  竞价交易
3  430489  佳先股份   汪静  高级管理人员  2023-06-20  1.000  70.0360  71.0360  4.52  竞价交易
4  430489  佳先股份   汪静  高级管理人员  2023-06-19  1.000  69.0360  70.0360  4.56  竞价交易
5  430489  佳先股份   丁柱  高级管理人员  2023-06-16  0.500  53.2920  53.7920  4.50  竞价交易
6  430489  佳先股份   丁柱  高级管理人员  2023-06-15  0.500  52.7920  53.2920  4.48  竞价交易
7  430489  佳先股份   丁柱  高级管理人员  2023-06-14  1.000  51.7920  52.7920  4.48  竞价交易
```

##### 股东人数及持股集中度

接口: stock_hold_num_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-股东股本-股东人数及持股集中度

限量: 单次指定 date 的股东人数及持股集中度数据, 从 20170331 开始

输入参数

| 名称   | 类型  | 描述                                                                                         |
|------|-----|--------------------------------------------------------------------------------------------|
| date | str | date="20210630"; choice of {"XXXX0331", "XXXX0630", "XXXX0930", "XXXX1231"}; 从 20170331 开始 |

输出参数

| 名称       | 类型      | 描述       |
|----------|---------|----------|
| 证劵代码     | object  | -        |
| 证券简称     | object  | -        |
| 变动日期     | object  | -        |
| 本期股东人数   | int64   | -        |
| 上期股东人数   | float64 | -        |
| 股东人数增幅   | float64 | 注意单位: %  |
| 本期人均持股数量 | int64   | 注意单位: 万股 |
| 上期人均持股数量 | float64 | 注意单位: %  |
| 人均持股数量增幅 | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_hold_num_cninfo_df = ak.stock_hold_num_cninfo(date="20210630")
print(stock_hold_num_cninfo_df)
```

数据示例

```
      证券代码  证券简称   变动日期  本期股东人数  ... 股东人数增幅  本期人均持股数量  上期人均持股数量  人均持股数量增幅
0     002054  德美化工  2021-06-30   17768  ...   -8.71   27134.0   24771.0      9.54
1     002055  得润电子  2021-06-30   57449  ...   15.55    8242.0    9524.0    -13.46
2     002056  横店东磁  2021-06-30   94339  ...    8.63   17243.0   18927.0     -8.90
3     600048  保利发展  2021-06-30  264212  ...    5.63   45303.0   47855.0     -5.33
4     002057  中钢天源  2021-06-30   44197  ...   -3.58   16883.0   12550.0     34.53
...      ...   ...         ...     ...  ...     ...       ...       ...       ...
4203  600918  中泰证券  2021-06-30  152424  ...   11.38   45719.0   50923.0    -10.22
4204  601375  中原证券  2021-06-30  158626  ...    2.38   29269.0   29966.0     -2.33
4205  601108  财通证券  2021-06-30  158061  ...   -5.50   22706.0   21458.0      5.82
4206  002945  华林证券  2021-06-30   54407  ...  -11.32   49626.0   44008.0     12.77
4207  601066  中信建投  2021-06-30  201162  ...    1.18   38559.0   39013.0     -1.16
[4208 rows x 9 columns]
```

##### 股本变动

接口: stock_hold_change_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-股东股本-股本变动

限量: 单次指定 symbol 的股本变动数据

输入参数

| 名称     | 类型  | 描述                                                               |
|--------|-----|------------------------------------------------------------------|
| symbol | str | symbol="全部"; choice of {"深市主板", "沪市", "创业板", "科创板", "北交所", "全部"} |

输出参数

| 名称     | 类型      | 描述     |
|--------|---------|--------|
| 证券代码   | object  | -      |
| 证券简称   | object  | -      |
| 交易市场   | object  | -      |
| 公告日期   | object  | -      |
| 变动日期   | object  | -      |
| 变动原因   | object  | -      |
| 总股本    | float64 | 单位: 万股 |
| 已流通股份  | float64 | 单位: 万股 |
| 已流通比例  | float64 | 单位: %  |
| 流通受限股份 | float64 | 单位: 万股 |

接口示例

```python
import akshare as ak

stock_hold_change_cninfo_df = ak.stock_hold_change_cninfo(symbol="全部")
print(stock_hold_change_cninfo_df)
```

数据示例

```
      证券代码   证券简称   交易市场  ...   已流通股份  已流通比例  流通受限股份
0     000001   平安银行  深交所主板  ...  1940561.75  100.00     30.07
1     000002  万  科Ａ  深交所主板  ...  1192344.88   99.94    726.07
2     000004   国华网安  深交所主板  ...    12628.81   95.40    609.22
3     000006   深振业Ａ  深交所主板  ...   134998.74  100.00      0.77
4     000007    全新好  深交所主板  ...    34644.80  100.00       NaN
...      ...    ...    ...  ...         ...     ...       ...
5463  920106   林泰新材    北交所  ...     1210.57   31.04   2689.43
5464  920111   聚星科技    北交所  ...     3688.85   23.78  11826.60
5465  920116   星图测控    北交所  ...     2612.50   23.75   8387.50
5466  920118   太湖远大    北交所  ...     1379.50   27.10   3710.10
5467  920128   胜业电气    北交所  ...     1710.00   21.06   6409.00
[5468 rows x 10 columns]
```

##### 实际控制人持股变动

接口: stock_hold_control_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-股东股本-实际控制人持股变动

限量: 单次指定 symbol 的实际控制人持股变动数据, 从 2010 开始

输入参数

| 名称     | 类型  | 描述                                                                         |
|--------|-----|----------------------------------------------------------------------------|
| symbol | str | symbol="全部"; choice of {"单独控制", "实际控制人", "一致行动人", "家族控制", "全部"}; 从 2010 开始 |

输出参数

| 名称      | 类型      | 描述       |
|---------|---------|----------|
| 证劵代码    | object  | -        |
| 证券简称    | object  | -        |
| 变动日期    | object  | -        |
| 实际控制人名称 | object  | -        |
| 控股数量    | float64 | 注意单位: 万股 |
| 控股比例    | float64 | 注意单位: %  |
| 直接控制人名称 | object  | -        |
| 控制类型    | object  | -        |

接口示例

```python
import akshare as ak

stock_hold_control_cninfo_df = ak.stock_hold_control_cninfo(symbol="全部")
print(stock_hold_control_cninfo_df)
```

数据示例

```
        证券代码  证券简称  ...                                        直接控制人名称   控制类型
0     601777  力帆股份  ...                 尹明善\r;陈巧凤\r;尹喜地\r;尹索微\r;重庆力帆控股有限公司  一致行动人
1     002460  赣锋锂业  ...                           李良彬;黄闻;熊剑浪;李良学;罗顺香\r;李华彪  一致行动人
2     000966  长源电力  ...                                           中国国电集团公司   单独控制
3     600019  宝钢股份  ...                                           宝钢集团有限公司   单独控制
4     600031  三一重工  ...                                       三一集团有限公司;梁稳根   单独控制
...      ...   ...  ...                                                ...    ...
5391  600354  敦煌种业  ...                                     酒泉钢铁(集团)有限责任公司   单独控制
5392  600208  新湖中宝  ...                          衢州工业投资集团有限公司;衢州工业发展集团有限公司   单独控制
5393  600399  抚顺特钢  ...  江苏沙钢集团有限公司;沙钢集团的股东张家港市保税区润源不锈钢贸易有限公司;宁波梅山保税港区锦...  一致行动人
5394  002075  沙钢股份  ...  江苏沙钢集团有限公司;宁波梅山保税港区锦程沙洲股权投资有限公司;张家港保税区润源不锈钢贸易有限公司   单独控制
5395  301552  科力装备  ...                   张万武;郭艳芝;张子恒;天津科达企业管理咨询合伙企业(有限合伙)   家族控制
[5396 rows x 8 columns]
```

##### 高管持股变动明细

接口: stock_hold_management_detail_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-股东股本-高管持股变动明细

限量: 单次指定 symbol 的高管持股变动明细数据, 返回近一年的数据

输入参数

| 名称     | 类型  | 描述                                  |
|--------|-----|-------------------------------------|
| symbol | str | symbol="增持"; choice of {"增持", "减持"} |

输出参数

| 名称        | 类型      | 描述       |
|-----------|---------|----------|
| 证劵代码      | object  | -        |
| 证券简称      | object  | -        |
| 截止日期      | object  | -        |
| 公告日期      | object  | -        |
| 高管姓名      | object  | -        |
| 董监高姓名     | object  | -        |
| 董监高职务     | object  | -        |
| 变动人与董监高关系 | object  | -        |
| 期初持股数量    | float64 | 注意单位: 万股 |
| 期末持股数量    | float64 | 注意单位: 万股 |
| 变动数量      | float64 | -        |
| 变动比例      | int64   | 注意单位: %  |
| 成交均价      | float64 | 注意单位: 元  |
| 期末市值      | float64 | 注意单位: 万元 |
| 持股变动原因    | object  | -        |
| 数据来源      | object  | -        |

接口示例

```python
import akshare as ak

stock_hold_management_detail_cninfo_df = ak.stock_hold_management_detail_cninfo(symbol="增持")
print(stock_hold_management_detail_cninfo_df)
```

数据示例

```
        证券代码  证券简称  截止日期    公告日期 高管姓名  ... 变动比例 成交均价 期末市值  持股变动原因  数据来源
0      000010  美丽生态  2023-12-31  2024-04-27   张琳  ...  NaN  NaN  NaN    None  定期报告
1      000010  美丽生态  2023-12-31  2024-04-27   张龙  ...  NaN  NaN  NaN    None  定期报告
2      000010  美丽生态  2023-12-31  2024-04-27  周成斌  ...  NaN  NaN  NaN    None  定期报告
3      000010  美丽生态  2023-12-31  2024-04-27  林孔凤  ...  NaN  NaN  NaN    None  定期报告
4      000010  美丽生态  2023-12-31  2024-04-27  江成汉  ...  NaN  NaN  NaN    None  定期报告
...       ...   ...         ...         ...  ...  ...  ...  ...  ...     ...   ...
15229  900946  天雁B股  2023-12-31  2024-04-20  何光清  ...  NaN  NaN  NaN    None  定期报告
15230  900946  天雁B股  2023-12-31  2024-04-20  刘青娥  ...  NaN  NaN  NaN    None  定期报告
15231  900946  天雁B股  2023-12-31  2024-04-20  杨国旗  ...  NaN  NaN  NaN    None  定期报告
15232  900946  天雁B股  2023-12-31  2024-04-20  杨宝全  ...  NaN  NaN  NaN    None  定期报告
15233  900946  天雁B股  2023-12-31  2024-04-20  胡辽平  ...  NaN  NaN  NaN    None  定期报告
[15234 rows x 16 columns]
```

##### 董监高及相关人员持股变动明细

接口: stock_hold_management_detail_em

目标地址: https://data.eastmoney.com/executive/list.html

描述: 东方财富网-数据中心-特色数据-高管持股-董监高及相关人员持股变动明细

限量: 单次返回所有数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称         | 类型      | 描述 |
|------------|---------|----|
| 日期         | object  | -  |
| 代码         | object  | -  |
| 名称         | object  | -  |
| 变动人        | object  | -  |
| 变动股数       | int64   | -  |
| 成交均价       | int64   | -  |
| 变动金额       | float64 | -  |
| 变动原因       | object  | -  |
| 变动比例       | float64 | -  |
| 变动后持股数     | float64 | -  |
| 持股种类       | object  | -  |
| 董监高人员姓名    | object  | -  |
| 职务         | object  | -  |
| 变动人与董监高的关系 | object  | -  |
| 开始时持有      | float64 | -  |
| 结束后持有      | float64 | -  |

接口示例

```python
import akshare as ak

stock_hold_management_detail_em_df = ak.stock_hold_management_detail_em()
print(stock_hold_management_detail_em_df)
```

数据示例

```
        日期      代码    名称  ... 变动人与董监高的关系    开始时持有  结束后持有
0       2023-08-08  001308  康冠科技  ...         本人  64350.0     74350.0
1       2023-08-08  001308  康冠科技  ...         本人      0.0     38800.0
2       2023-08-07  002533  金杯电工  ...         本人      NaN  45718400.0
3       2023-08-07  300389   艾比森  ...         本人      NaN     50600.0
4       2023-08-07  300827  上能电气  ...         本人      NaN  31879000.0
            ...     ...   ...  ...        ...      ...         ...
141380  2006-05-26  600500  中化国际  ...         本人      0.0      9000.0
141381  2006-03-02  000045  深纺织A  ...         本人      0.0      7050.0
141382  2006-02-17  002039  黔源电力  ...         本人      0.0     10000.0
141383  2005-09-22  000999  华润三九  ...         本人      0.0      2000.0
141384  2004-11-29  000030  富奥股份  ...         本人      0.0      6000.0
[141385 rows x 16 columns]
```

##### 人员增减持股变动明细

接口: stock_hold_management_person_em

目标地址: https://data.eastmoney.com/executive/personinfo.html?name=%E5%90%B4%E8%BF%9C&code=001308

描述: 东方财富网-数据中心-特色数据-高管持股-人员增减持股变动明细

限量: 单次返回指定 symbol 和 name 的数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="001308"; 股票代码 |
| name   | str | name="吴远"; 高管名称       |

输出参数

| 名称         | 类型      | 描述 |
|------------|---------|----|
| 日期         | object  | -  |
| 代码         | object  | -  |
| 名称         | object  | -  |
| 变动人        | object  | -  |
| 变动股数       | int64   | -  |
| 成交均价       | int64   | -  |
| 变动金额       | float64 | -  |
| 变动原因       | object  | -  |
| 变动比例       | float64 | -  |
| 变动后持股数     | float64 | -  |
| 持股种类       | object  | -  |
| 董监高人员姓名    | object  | -  |
| 职务         | object  | -  |
| 变动人与董监高的关系 | object  | -  |
| 开始时持有      | float64 | -  |
| 结束后持有      | float64 | -  |

接口示例

```python
import akshare as ak

stock_hold_management_person_em_df = ak.stock_hold_management_person_em(symbol="001308", name="孙建华")
print(stock_hold_management_person_em_df)
```

数据示例

```
   日期      代码    名称 变动人   变动股数  ...  董监高人员姓名    职务 变动人与董监高的关系    开始时持有  结束后持有
0  2023-08-08  001308  康冠科技  吴远  10000  ...       吴远  财务总监         本人  64350.0  74350
1  2022-08-24  001308  康冠科技  吴远  14000  ...       吴远    高管         本人      NaN  49500
2  2022-06-17  001308  康冠科技  吴远  35500  ...       吴远  财务总监         本人      0.0  35500
[3 rows x 16 columns]
```

##### 对外担保

接口: stock_cg_guarantee_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-公司治理-对外担保

限量: 单次指定 symbol 和起始日期的对外担保数据

输入参数

| 名称         | 类型  | 描述                                                        |
|------------|-----|-----------------------------------------------------------|
| symbol     | str | symbol="全部"; choice of {"全部", "深市主板", "沪市", "创业板", "科创板"} |
| start_date | str | start_date="20180630"                                     |
| end_date   | str | end_date="20210927"                                       |

输出参数

| 名称          | 类型      | 描述       |
|-------------|---------|----------|
| 证劵代码        | object  | -        |
| 证券简称        | object  | -        |
| 公告统计区间      | object  | -        |
| 担保笔数        | int64   | -        |
| 担保金额        | float64 | 注意单位: 万元 |
| 归属于母公司所有者权益 | float64 | 注意单位: 万元 |
| 担保金融占净资产比例  | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_corporate_governance_guarantee_df = ak.stock_cg_guarantee_cninfo(symbol="全部", start_date="20180630", end_date="20210927")
print(stock_corporate_governance_guarantee_df)
```

数据示例

```
      证券代码  证券简称  ... 归属于母公司所有者权益  担保金融占净资产比例
0     000683  远兴能源  ...  1416683.55       93.19
1     000698  沈阳化工  ...   152836.81     1413.11
2     000732  ST泰禾  ...   387149.45    11353.60
3     000751  锌业股份  ...   358173.79       64.21
4     000818  航锦科技  ...   368945.73       30.90
...      ...   ...  ...         ...         ...
3075  688025   杰普特  ...   196706.09        5.08
3076  300822  贝仕达克  ...   126328.08        3.96
3077  300907  康平科技  ...    78724.85       10.16
3078  688296  和达科技  ...    70535.91        1.42
3079  300632  光莆股份  ...   189380.21       24.29
[3080 rows x 7 columns]
```

##### 公司诉讼

接口: stock_cg_lawsuit_cninfo

目标地址: http://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-公司治理-公司诉讼

限量: 单次指定 symbol 和起始日期的公司诉讼数据

输入参数

| 名称         | 类型  | 描述                                                        |
|------------|-----|-----------------------------------------------------------|
| symbol     | str | symbol="全部"; choice of {"全部", "深市主板", "沪市", "创业板", "科创板"} |
| start_date | str | start_date="20180630"                                     |
| end_date   | str | end_date="20210927"                                       |

输出参数

| 名称     | 类型      | 描述       |
|--------|---------|----------|
| 证劵代码   | object  | -        |
| 证券简称   | object  | -        |
| 公告统计区间 | object  | -        |
| 诉讼次数   | int64   | -        |
| 诉讼金额   | float64 | 注意单位: 万元 |

接口示例

```python
import akshare as ak

stock_cg_lawsuit_cninfo_df = ak.stock_cg_lawsuit_cninfo(symbol="全部", start_date="20180630", end_date="20210927")
print(stock_cg_lawsuit_cninfo_df)
```

数据示例

```
        证券代码  证券简称                   公告统计区间  诉讼次数      诉讼金额
0     002524  光正眼科  2018-09-28---2021-09-27     1       NaN
1     603507  振江股份  2018-09-28---2021-09-27     1       NaN
2     600410  华胜天成  2018-09-28---2021-09-27     2       NaN
3     601333  广深铁路  2018-09-28---2021-09-27     1       NaN
4     301013   利和兴  2018-09-28---2021-09-27     1       NaN
      ...   ...                      ...   ...       ...
1020  600310  桂东电力  2018-09-28---2021-09-27     2   5360.63
1021  600815  厦工股份  2018-09-28---2021-09-27    75  75569.33
1022  600830  香溢融通  2018-09-28---2021-09-27     6  17109.27
1023  603986  兆易创新  2018-09-28---2021-09-27     6  29200.00
1024  002673  西部证券  2018-09-28---2021-09-27     4  75380.45
```

##### 股权质押

接口: stock_cg_equity_mortgage_cninfo

目标地址: https://webapi.cninfo.com.cn/#/thematicStatistics

描述: 巨潮资讯-数据中心-专题统计-公司治理-股权质押

限量: 单次指定 date 的股权质押数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20210930" |

输出参数

| 名称         | 类型      | 描述       |
|------------|---------|----------|
| 股票代码       | object  | -        |
| 股票简称       | object  | -        |
| 公告日期       | object  | -        |
| 出质人        | object  | -        |
| 质权人        | object  | -        |
| 质押数量       | float64 | 注意单位: 万股 |
| 占总股本比例     | float64 | 注意单位: %  |
| 质押解除数量     | float64 | 注意单位: 万股 |
| 质押事项       | object  | 注意单位: 万元 |
| 累计质押占总股本比例 | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_cg_equity_mortgage_cninfo_df = ak.stock_cg_equity_mortgage_cninfo(date="20210930")
print(stock_cg_equity_mortgage_cninfo_df)
```

数据示例

```
      股票代码   股票简称  ...                                               质押事项  \
0   000892   欢瑞世纪  ...  本公司持股5%以上股东欢瑞世纪(天津)资产管理合伙企业(有限合伙)将2020年04月20日质...
1   000892   欢瑞世纪  ...  本公司持股5%以上股东赵枳程将持有的本公司4743836股流通A股质押给方正证券股份有限公司...
2   000948   南天信息  ...  本公司控股股东云南省工业投资控股集团有限责任公司将2020年09月18日质押给云南国际信托有...
3   002074   国轩高科  ...  本公司控股股东南京国轩控股集团有限公司将持有的本公司20620000股质押给海通证券股份有限...
4   002074   国轩高科  ...  本公司实际控制人李缜将2019年09月25日质押给华安证券股份有限公司的11270000股解...
5   002074   国轩高科  ...  本公司实际控制人李缜将2019年09月27日质押给华安证券股份有限公司的11100000股解...
6   002138   顺络电子  ...  本公司股东新余市恒顺通电子科技开发有限公司将持有的本公司3635000股流通A股质押给广发证...
7   002375   亚厦股份  ...  本公司控股股东亚厦控股有限公司将持有的本公司68000000股流通A股质押给杭州银行股份有限...
8   002375   亚厦股份  ...  本公司控股股东亚厦控股有限公司将持有的本公司97000000股流通A股质押给绍兴银行股份有限...
9   002466   天齐锂业  ...  本公司控股股东成都天齐实业(集团)有限公司将持有的本公司12000000股流通A股质押给中国...
10  002466   天齐锂业  ...  本公司控股股东成都天齐实业(集团)有限公司将持有的本公司4600000股流通A股质押给交通银...
11  002475   立讯精密  ...  本公司控股股东香港立讯有限公司将持有的本公司72620000股流通A股质押给招商银行股份有限...
12  002584   西陇科学  ...  本公司实际控制人黄伟波将持有的本公司8470000股流通受限股份质押给华鑫证券有限责任公司用...
13  002584   西陇科学  ...  本公司实际控制人黄伟波将持有的本公司9480000股流通受限股份质押给华鑫证券有限责任公司用...
14  002584   西陇科学  ...  本公司实际控制人黄伟鹏将2017年08月10日质押给华鑫证券有限责任公司的4480000股解...
15  002584   西陇科学  ...  本公司实际控制人黄伟鹏将持有的本公司5260000股流通受限股份质押给华鑫证券有限责任公司用...
16  002584   西陇科学  ...  本公司实际控制人黄伟鹏将持有的本公司7680000股流通受限股份质押给华鑫证券有限责任公司用...
17  002584   西陇科学  ...  本公司实际控制人黄伟鹏将持有的本公司9970000股流通受限股份质押给华鑫证券有限责任公司用...
18  002584   西陇科学  ...  本公司实际控制人黄伟鹏将持有的本公司5260000股流通受限股份质押给华鑫证券有限责任公司用...
19  002584   西陇科学  ...  本公司实际控制人黄少群将持有的本公司6290000股流通A股质押给华鑫证券有限责任公司用于办...
20  002584   西陇科学  ...  本公司实际控制人黄少群将持有的本公司7360000股流通受限股份质押给华鑫证券有限责任公司用...
21  002584   西陇科学  ...  本公司实际控制人黄少群将持有的本公司8200000股流通受限股份质押给华鑫证券有限责任公司用...
22  002584   西陇科学  ...  本公司实际控制人黄少群将持有的本公司9700000股流通受限股份质押给华鑫证券有限责任公司用...
23  002584   西陇科学  ...  本公司实际控制人黄少群将持有的本公司6290000股流通A股质押给华鑫证券有限责任公司用于办...
24  002602   世纪华通  ...  本公司大股东王佶将2018年02月12日质押给大连银行股份有限公司上海分行的25000000...
25  002602   世纪华通  ...  本公司大股东王佶将2018年02月12日质押给大连银行股份有限公司上海分行的26000000...
26  002602   世纪华通  ...  本公司大股东王佶将持有的本公司26000000股流通A股质押给泰安复星招银壹号股权投资基金合...
27  002602   世纪华通  ...  本公司大股东王佶将持有的本公司5000000股流通A股质押给泰安复星招银壹号股权投资基金合伙...
28  002677   浙江美大  ...  本公司股东夏鼎将2020年12月28日质押给上海海通证券资产管理有限公司用于办理质押式回购业...
29  002677   浙江美大  ...  本公司股东夏鼎将2021年05月18日质押给上海海通证券资产管理有限公司用于办理质押式回购业...
30  002677   浙江美大  ...  本公司股东夏鼎将2021年08月25日质押给上海海通证券资产管理有限公司用于办理质押式回购业...
31  002853    皮阿诺  ...  本公司控股股东、实际控制人马礼斌将持有的本公司1000000股流通A股质押给中信证券股份有限...
32  002922    伊戈尔  ...  本公司控股股东佛山市麦格斯投资有限公司将持有的本公司5000000股流通A股质押给国金证券股...
33  300175   朗源股份  ...  本公司控股股东新疆尚龙股权投资管理有限公司将持有的本公司17000000股流通A股质押给邳州...
34  300275    梅安森  ...  本公司控股股东马焰将持有的本公司2000000股流通受限股份质押给交通银行股份有限公司重庆市...
35  600093  *ST易见  ...  本公司股东云南省滇中产业发展集团有限责任公司将质押给中国工商银行股份有限公司昆明正义支行的8...
36  600093  *ST易见  ...  本公司股东云南省滇中产业发展集团有限责任公司将持有的本公司86380000股流通A股质押给中...
37  600176   中国巨石  ...  本公司股东振石控股集团有限公司将持有的本公司126680000股流通A股质押给中国民生银行股...
38  600176   中国巨石  ...  本公司股东振石控股集团有限公司将质押的211672605股解除质押，解除日期为2021年09...
39  600246   万通发展  ...  本公司第二大股东万通投资控股股份有限公司将2021年07月12日质押给江苏银行股份有限公司北...
40  600516   方大炭素  ...  本公司控股股东辽宁方大集团实业有限公司将持有的本公司56000000股流通A股质押给平安银行...
41  600516   方大炭素  ...  本公司控股股东辽宁方大集团实业有限公司将质押给平安银行股份有限公司沈阳分行的70700000...
42  600572    康恩贝  ...  本公司持股5%以上股东胡季强将持有的本公司27000000股流通A股质押给中信证券股份有限公...
43  600572    康恩贝  ...  本公司持股5%以上股东胡季强将质押给中信证券股份有限公司用于办理质押式回购业务的210000...
44  600739   辽宁成大  ...  本公司控股股东辽宁省国有资产经营有限公司将持有的本公司5500000股流通A股质押给华夏银行...
45  600745   闻泰科技  ...  本公司股东张丹琳将持有的本公司171095股流通A股质押给云南国际信托有限公司，质押开始日期...
46  600745   闻泰科技  ...  本公司股东张秋红将持有的本公司305364股流通A股质押给云南国际信托有限公司，质押开始日期...
47  600866   星湖科技  ...  本公司股东深圳长城汇理资产服务企业(有限合伙)将质押给东方证券股份有限公司的13080000...
48  600903   贵州燃气  ...  本公司控股股东北京东嘉投资有限公司将质押给华能贵诚信托有限公司的91507780股解除质押，...
49  600998    九州通  ...  本公司股东上海弘康实业投资有限公司将持有的本公司15600000股流通A股质押给汉口银行股份...
50  601012   隆基股份  ...  本公司控股股东李振国将质押给国信证券股份有限公司的28000000股流通A股解除质押，解除日...
51  603501   韦尔股份  ...  本公司控股股东虞仁荣将持有的本公司590000股流通A股质押给平安证券股份有限公司，质押开始...
52  603619   中曼石油  ...     本公司实际控制人李玉池将质押的16038460股解除质押，解除日期为2021年09月28日。
53  603638   艾迪精密  ...  本公司控股股东烟台翔宇投资有限公司将持有的本公司6000000股流通A股质押给云南国际信托有...
54  603661   恒林股份  ...  本公司持股5%以上股东安吉恒林商贸有限公司将持有的本公司1500000股流通A股质押给浙商证...
```

#### 美港目标价

接口: stock_price_js

目标地址: https://www.ushknews.com/report.html

描述: 美港电讯-美港目标价数据

限量: 单次获取所有数据, 数据从 2019-至今; 该接口暂时不能使用

输入参数

| 名称     | 类型  | 描述                                  |
|--------|-----|-------------------------------------|
| symbol | str | symbol="us"; choice of {"us", "hk"} |

输出参数

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| 日期    | object  | -   |
| 个股名称  | object  | -   |
| 评级    | object  | -   |
| 先前目标价 | float64 | -   |
| 最新目标价 | float64 | -   |
| 机构名称  | object  | -   |

接口示例

```python
import akshare as ak

stock_price_js_df = ak.stock_price_js(symbol="us")
print(stock_price_js_df)
```

数据示例

```
               日期                    个股名称     评级  先前目标价  最新目标价       机构名称
0      2022-02-12  Teladoc Health(TDOC.N)     买入    NaN  121.0         高盛
1      2022-02-12       Cloudflare(NET.N)   None  132.0  150.0    KeyBanc
2      2022-02-12             Zillow(Z.O)   None   74.0   76.0      摩根士丹利
3      2022-02-11             Zillow(Z.O)   None  105.0  115.0  Benchmark
4      2022-02-11       Cloudflare(NET.N)     中性  210.0  130.0         贝雅
...           ...                     ...    ...    ...    ...        ...
46039  2019-01-02             美国银行(BAC.N)  与大市持平   37.0   34.0        巴克莱
46040  2019-01-02             亚马逊(AMZN.O)     买入    NaN    NaN    加拿大皇家银行
46041  2019-01-02              苹果(AAPL.O)   跑赢大市  266.0  228.0       摩根大通
46042  2019-01-02              苹果(AAPL.O)   跑赢大市    NaN  220.0    加拿大皇家银行
46043  2019-01-01               强生(JNJ.N)     中性  139.0  148.0         花旗
```

#### 券商业绩月报

接口: stock_qsjy_em

目标地址: http://data.eastmoney.com/other/qsjy.html

描述: 东方财富网-数据中心-特色数据-券商业绩月报

限量: 单次获取所有数据, 数据从 201006-202007, 月频率

输入参数

| 名称   | 类型  | 描述                                |
|------|-----|-----------------------------------|
| date | str | date="20200430"; 输入需要查询月份的最后一天的日期 |

输出参数

| 名称              | 类型      | 描述       |
|-----------------|---------|----------|
| 简称              | object  | -        |
| 代码              | object  | -        |
| 当月净利润-净利润       | float64 | 注意单位: 万元 |
| 当月净利润-同比增长      | float64 | -        |
| 当月净利润-环比增长      | float64 | -        |
| 当年累计净利润-累计净利润   | float64 | 注意单位: 万元 |
| 当年累计净利润-同比增长    | float64 | -        |
| 当月营业收入-营业收入     | float64 | 注意单位: 万元 |
| 当月营业收入-环比增长     | float64 | -        |
| 当月营业收入-同比增长     | float64 | -        |
| 当年累计营业收入-累计营业收入 | float64 | 注意单位: 万元 |
| 当年累计营业收入-同比增长   | float64 | -        |
| 净资产-净资产         | float64 | 注意单位: 万元 |
| 净资产-同比增长        | float64 | -        |

接口示例

```python
import akshare as ak

stock_qsjy_em_df = ak.stock_qsjy_em(date="20200430")
print(stock_qsjy_em_df)
```

数据示例

```
      简称      代码      当月净利润-净利润  ...  当年累计营业收入-同比增长       净资产-净资产   净资产-同比增长
0   国海证券  000750   19194.615013  ...      54.032161  1.776790e+06  33.096526
1   国金证券  600109   20891.770000  ...      50.235759  2.097736e+06   6.747984
2   光大证券  601788   60001.753808  ...       1.940449  5.191309e+06   2.343186
3   方正证券  601901   12395.300000  ...      15.571724  4.278250e+06  -8.135272
4   兴业证券  601377   32467.150000  ...     -14.595211  3.430436e+06   5.441332
5   国信证券  002736   56775.290000  ...      32.460693  5.529023e+06   2.995151
6   华泰证券  601688   78019.450000  ...       5.858672  1.234187e+07  14.817958
7   天风证券  601162    7704.010000  ...      13.050550  1.687778e+06  50.755090
8   华西证券  002926   24595.360000  ...       1.709712  2.002422e+06   6.198522
9   西南证券  600369   10153.900000  ...     -32.599011  1.976813e+06   5.182443
10  长城证券  002939   12596.910000  ...      14.546146  1.725532e+06   3.114500
11  东吴证券  601555   15202.270000  ...       8.220710  2.664589e+06  30.763186
12  东兴证券  601198   23724.000000  ...       1.505702  2.069308e+06   4.788353
13  浙商证券  601878   10729.640000  ...      71.999008  1.508543e+06   4.119564
14  申万宏源  000166  156658.000000  ...      29.308936  7.945952e+06  10.579253
15  华林证券  002945    3286.100000  ...      43.888841  5.492524e+05   8.617837
16  国泰君安  601211   79724.170000  ...      -9.160164  1.364813e+07   1.784078
17  东北证券  000686   19464.850000  ...       3.469984  1.618389e+06   4.375444
18  华安证券  600909   11005.490000  ...       7.830548  1.371776e+06   9.652564
19  中原证券  601375    8586.950000  ...      15.216216  1.004165e+06   2.114908
20  长江证券  000783   28988.140000  ...            NaN  3.115692e+06  10.237273
21  东方证券  600958   59493.090000  ...      36.465438  5.598729e+06   4.538278
22  红塔证券  601236   21168.110000  ...            NaN  1.352354e+06        NaN
23   太平洋  601099    1074.460000  ...     -35.504783  1.046198e+06   3.302652
24  中信证券  600030  162426.000000  ...      52.561662  1.683170e+07  23.936907
25  山西证券  002500   11175.370000  ...      35.841437  1.411993e+06   4.424647
26  中银证券  601696   10172.510000  ...            NaN  1.415213e+06        NaN
27  第一创业  002797   10003.990000  ...      32.569921  9.662962e+05   5.396208
28  招商证券  600999   82020.637477  ...      24.449430  8.590377e+06   6.038244
29  中信建投  601066  103886.720000  ...      61.480355  5.799100e+06  20.862475
30  国元证券  000728    4848.630000  ...      27.615097  2.366435e+06  -1.110733
31  华鑫股份  600621    4442.740000  ...      58.946622  6.125237e+05  14.145368
32  中国银河  601881   83429.360000  ...      30.440181  7.297874e+06   7.543491
33  西部证券  002673   19674.240000  ...     -10.968199  1.828071e+06   1.750810
34  广发证券  000776   68079.485932  ...      -4.933729  8.878713e+06   6.098383
35  财通证券  601108   17561.630000  ...      -8.775579  2.116647e+06   6.241030
36  南京证券  601990    9131.740000  ...      19.802440  1.139362e+06   5.430794
37  海通证券  600837   96773.380000  ...       3.011836  1.203286e+07   3.822430
```

#### A 股股息率

接口: stock_a_gxl_lg

目标地址: https://legulegu.com/stockdata/guxilv

描述: 乐咕乐股-股息率-A 股股息率

限量: 单次获取指定 symbol 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                      |
|--------|-----|---------------------------------------------------------|
| symbol | str | symbol="上证A股"; choice of {"上证A股", "深证A股", "创业板", "科创板"} |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 股息率 | float64 | -   |

接口示例

```python
import akshare as ak

stock_a_gxl_lg_df = ak.stock_a_gxl_lg(symbol="上证A股")
print(stock_a_gxl_lg_df)
```

数据示例

```
           日期   股息率
0     2005-01-04  2.03
1     2005-01-05  2.05
2     2005-01-06  2.07
3     2005-01-07  2.06
4     2005-01-10  2.05
          ...   ...
4685  2024-04-18  3.44
4686  2024-04-19  3.44
4687  2024-04-22  3.48
4688  2024-04-23  3.52
4689  2024-04-24  3.50
[4690 rows x 2 columns]
```

#### 恒生指数股息率

接口: stock_hk_gxl_lg

目标地址: https://legulegu.com/stockdata/market/hk/dv/hsi

描述: 乐咕乐股-股息率-恒生指数股息率

限量: 单次获取所有月度历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 股息率 | float64 | -   |

接口示例

```python
import akshare as ak

stock_hk_gxl_lg_df = ak.stock_hk_gxl_lg()
print(stock_hk_gxl_lg_df)
```

数据示例

```
          日期   股息率
0    1973-05-31  0.00
1    1973-06-30  0.00
2    1973-07-31  0.00
3    1973-08-31  0.00
4    1973-09-30  0.00
..          ...   ...
606  2023-11-30  4.10
607  2023-12-31  3.97
608  2024-01-31  4.37
609  2024-02-28  4.45
610  2024-03-31  4.34
[611 rows x 2 columns]
```

#### 大盘拥挤度

接口: stock_a_congestion_lg

目标地址: https://legulegu.com/stockdata/ashares-congestion

描述: 乐咕乐股-大盘拥挤度

限量: 单次获取近 4 年的历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称         | 类型      | 描述  |
|------------|---------|-----|
| date       | object  | 日期  |
| close      | float64 | 收盘价 |
| congestion | float64 | 拥挤度 |

接口示例

```python
import akshare as ak

stock_a_congestion_lg_df = ak.stock_a_congestion_lg()
print(stock_a_congestion_lg_df)
```

数据示例

```
           date    close  congestion
0    2020-04-27  2815.49      0.3783
1    2020-04-28  2810.02      0.3797
2    2020-04-29  2822.44      0.3876
3    2020-04-30  2860.08      0.4009
4    2020-05-06  2878.14      0.4080
..          ...      ...         ...
962  2024-04-18  3074.22      0.3848
963  2024-04-19  3065.26      0.3896
964  2024-04-22  3044.60      0.3834
965  2024-04-23  3021.98      0.3808
966  2024-04-24  3044.82      0.3786
[967 rows x 3 columns]
```

#### 股债利差

接口: stock_ebs_lg

目标地址: https://legulegu.com/stockdata/equity-bond-spread

描述: 乐咕乐股-股债利差

限量: 单次所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述  |
|---------|---------|-----|
| 日期      | object  | -   |
| 沪深300指数 | float64 | -   |
| 股债利差    | float64 | -   |
| 股债利差均线  | float64 | -   |

接口示例

```python
import akshare as ak

stock_ebs_lg_df = ak.stock_ebs_lg()
print(stock_ebs_lg_df)
```

数据示例

```
            日期  沪深300指数   股债利差 股债利差均线
0     2005-04-08  1003.45  0.022656  0.022656
1     2005-04-11   995.42  0.021938  0.022297
2     2005-04-12   978.70  0.024697  0.023097
3     2005-04-13  1000.90  0.022538  0.022957
4     2005-04-14   986.98  0.022811  0.022928
          ...      ...       ...       ...
4619  2024-04-18  3569.80  0.064876  0.061882
4620  2024-04-19  3541.66  0.064949  0.061909
4621  2024-04-22  3530.90  0.065748  0.061983
4622  2024-04-23  3506.22  0.066550  0.062014
4623  2024-04-24  3521.62  0.065847  0.062020
[4624 rows x 4 columns]
```

#### 巴菲特指标

接口: stock_buffett_index_lg

目标地址: https://legulegu.com/stockdata/marketcap-gdp

描述: 乐估乐股-底部研究-巴菲特指标

限量: 单次获取所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述                             |
|--------|---------|--------------------------------|
| 日期     | object  | 交易日                            |
| 收盘价    | float64 | -                              |
| 总市值    | float64 | A股收盘价*已发行股票总股本（A股+B股+H股）       |
| GDP    | float64 | 上年度国内生产总值（例如：2019年，则取2018年GDP） |
| 近十年分位数 | float64 | 当前"总市值/GDP"在历史数据上的分位数          |
| 总历史分位数 | float64 | 当前"总市值/GDP"在历史数据上的分位数          |

接口示例

```python
import akshare as ak

stock_buffett_index_lg_df = ak.stock_buffett_index_lg()
print(stock_buffett_index_lg_df)
```

数据示例

```
            日期   收盘价       总市值      GDP   近十年分位数  总历史分位数
0     2005-04-07  1003.45   38470.47   161840.16  1.00000  1.00000
1     2005-04-10   995.42   39184.81   161840.16  1.00000  1.00000
2     2005-04-11   978.70   38955.09   161840.16  0.66667  0.66667
3     2005-04-12  1000.90   38287.33   161840.16  0.25000  0.25000
4     2005-04-13   986.98   39166.32   161840.16  0.80000  0.80000
...          ...      ...        ...         ...      ...      ...
4646  2024-05-26  3635.71  894146.43  1260582.10  0.34239  0.47837
4647  2024-05-27  3609.17  888429.78  1260582.10  0.32593  0.46601
4648  2024-05-28  3613.52  889703.17  1260582.10  0.32840  0.46870
4649  2024-05-29  3594.31  884208.63  1260582.10  0.31235  0.45699
4650  2024-05-30  3579.92  884322.56  1260582.10  0.31304  0.45732
[4651 rows x 6 columns]
```

#### A 股等权重与中位数市盈率

接口: stock_a_ttm_lyr

目标地址: https://www.legulegu.com/stockdata/a-ttm-lyr

描述: 乐咕乐股-A 股等权重市盈率与中位数市盈率

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称                                  | 类型      | 描述                               |
|-------------------------------------|---------|----------------------------------|
| date                                | object  | 日期                               |
| middlePETTM                         | float64 | 全A股滚动市盈率(TTM)中位数                 |
| averagePETTM                        | float64 | 全A股滚动市盈率(TTM)等权平均                |
| middlePELYR                         | float64 | 全A股静态市盈率(LYR)中位数                 |
| averagePELYR                        | float64 | 全A股静态市盈率(LYR)等权平均                |
| quantileInAllHistoryMiddlePeTtm     | float64 | 当前"TTM(滚动市盈率)中位数"在历史数据上的分位数      |
| quantileInRecent10YearsMiddlePeTtm  | float64 | 当前"TTM(滚动市盈率)中位数"在最近10年数据上的分位数   |
| quantileInAllHistoryAveragePeTtm    | float64 | 当前"TTM(滚动市盈率)等权平均"在历史数据上的分位数     |
| quantileInRecent10YearsAveragePeTtm | float64 | 当前"TTM(滚动市盈率)等权平均"在在最近10年数据上的分位数 |
| quantileInAllHistoryMiddlePeLyr     | float64 | 当前"LYR(静态市盈率)中位数"在历史数据上的分位数      |
| quantileInRecent10YearsMiddlePeLyr  | float64 | 当前"LYR(静态市盈率)中位数"在最近10年数据上的分位数   |
| quantileInAllHistoryAveragePeLyr    | float64 | 当前"LYR(静态市盈率)等权平均"在历史数据上的分位数     |
| quantileInRecent10YearsAveragePeLyr | float64 | 当前"LYR(静态市盈率)等权平均"在最近10年数据上的分位数  |
| close                               | float64 | 沪深300指数                          |

接口示例

```python
import akshare as ak

stock_a_ttm_lyr_df = ak.stock_a_ttm_lyr()
print(stock_a_ttm_lyr_df)
```

数据示例

```
            date  middlePETTM  ...  quantileInRecent10YearsAveragePeLyr    close
0     2005-01-05        28.79  ...                              1.00000     0.00
1     2005-01-06        29.18  ...                              1.00000     0.00
2     2005-01-07        28.73  ...                              0.66667     0.00
3     2005-01-10        28.84  ...                              0.50000     0.00
4     2005-01-11        29.09  ...                              1.00000     0.00
...          ...          ...  ...                                  ...      ...
4795  2024-10-13        29.58  ...                              0.15534  3887.17
4796  2024-10-14        30.34  ...                              0.20725  3961.34
4797  2024-10-15        29.75  ...                              0.18995  3855.99
4798  2024-10-16        29.67  ...                              0.20643  3831.59
4799  2024-10-17        29.56  ...                              0.20107  3788.22
[4800 rows x 14 columns]
```

#### A 股等权重与中位数市净率

接口: stock_a_all_pb

目标地址: https://www.legulegu.com/stockdata/all-pb

描述: 乐咕乐股-A 股等权重与中位数市净率

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述 |
|-----|-----|----|
| -   | -   | -  |

输出参数

| 名称                                          | 类型      | 描述                     |
|---------------------------------------------|---------|------------------------|
| date                                        | object  | 日期                     |
| middlePB                                    | float64 | 全部A股市净率中位数             |
| equalWeightAveragePB                        | float64 | 全部A股市净率等权平均            |
| close                                       | float64 | 上证指数                   |
| quantileInAllHistoryMiddlePB                | float64 | 当前市净率中位数在历史数据上的分位数     |
| quantileInRecent10YearsMiddlePB             | float64 | 当前市净率中位数在最近10年数据上的分位数  |
| quantileInAllHistoryEqualWeightAveragePB    | float64 | 当前市净率等权平均在历史数据上的分位数    |
| quantileInRecent10YearsEqualWeightAveragePB | float64 | 当前市净率等权平均在最近10年数据上的分位数 |

接口示例

```python
import akshare as ak

stock_a_all_pb_df = ak.stock_a_all_pb()
print(stock_a_all_pb_df)
```

数据示例

```
            date  ...  quantileInRecent10YearsEqualWeightAveragePB
0     2005-01-04  ...                                      1.00000
1     2005-01-05  ...                                      1.00000
2     2005-01-06  ...                                      0.66667
3     2005-01-07  ...                                      0.75000
4     2005-01-10  ...                                      1.00000
...          ...  ...                                          ...
4793  2024-10-11  ...                                      0.10763
4794  2024-10-14  ...                                      0.12170
4795  2024-10-15  ...                                      0.11551
4796  2024-10-16  ...                                      0.11840
4797  2024-10-17  ...                                      0.12129
[4798 rows x 8 columns]
```

#### 主板市盈率

接口: stock_market_pe_lg

目标地址: https://legulegu.com/stockdata/shanghaiPE

描述: 乐咕乐股-主板市盈率

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                |
|--------|-----|---------------------------------------------------|
| symbol | str | symbol="上证"; choice of {"上证", "深证", "创业板", "科创版"} |

输出参数-上证, 深证, 创业板

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| 日期    | object  | -   |
| 指数    | float64 | -   |
| 平均市盈率 | float64 | -   |

接口示例-上证, 深证, 创业板

```python
import akshare as ak

stock_market_pe_lg_df = ak.stock_market_pe_lg(symbol="上证")
print(stock_market_pe_lg_df)
```

数据示例-上证, 深证, 创业板

```
        日期       指数  平均市盈率
0    1999-01-29  1134.67  34.03
1    1999-02-09  1090.08  33.50
2    1999-03-31  1158.05  34.30
3    1999-04-30  1120.92  34.39
4    1999-05-31  1279.32  35.30
..          ...      ...    ...
306  2024-06-28  2967.40  12.69
307  2024-07-31  2938.75  12.55
308  2024-08-30  2842.21  12.16
309  2024-09-30  3336.50  14.24
310  2024-10-18  3261.56  13.89
[311 rows x 3 columns]
```

输出参数-科创版

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 总市值 | float64 | -   |
| 市盈率 | float64 | -   |

接口示例-科创版

```python
import akshare as ak

stock_market_pe_lg_df = ak.stock_market_pe_lg(symbol="科创版")
print(stock_market_pe_lg_df)
```

数据示例-科创版

```
        日期       总市值    市盈率
0     2019-07-22   5293.39  81.43
1     2019-07-23   4821.95  74.18
2     2019-07-24   5135.78  79.00
3     2019-07-25   5373.12  82.65
4     2019-07-26   5000.56  76.92
...          ...       ...    ...
1265  2024-10-14  58559.78  39.19
1266  2024-10-15  56996.68  38.18
1267  2024-10-16  55961.92  37.50
1268  2024-10-17  56181.48  37.54
1269  2024-10-18  61039.80  40.72
[1270 rows x 3 columns]
```

#### 指数市盈率

接口: stock_index_pe_lg

目标地址: https://legulegu.com/stockdata/sz50-ttm-lyr

描述: 乐咕乐股-指数市盈率

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                                                                                                  |
|--------|-----|-------------------------------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="上证50"; choice of {"上证50", "沪深300", "上证380", "创业板50", "中证500", "上证180", "深证红利", "深证100", "中证1000", "上证红利", "中证100", "中证800"} |

输出参数

| 名称       | 类型      | 描述  |
|----------|---------|-----|
| 日期       | object  | -   |
| 指数       | float64 | -   |
| 等权静态市盈率  | float64 | -   |
| 静态市盈率    | float64 | -   |
| 静态市盈率中位数 | float64 | -   |
| 等权滚动市盈率  | float64 | -   |
| 滚动市盈率    | float64 | -   |
| 滚动市盈率中位数 | float64 | -   |

接口示例

```python
import akshare as ak

stock_index_pe_lg_df = ak.stock_index_pe_lg(symbol="上证50")
print(stock_index_pe_lg_df)
```

数据示例

```
      日期         指数  等权静态市盈率  静态市盈率  静态市盈率中位数  等权滚动市盈率  滚动市盈率  滚动市盈率中位数
0     2005-01-05   831.43    32.48  20.69     27.02    29.52  14.93     18.97
1     2005-01-06   822.50    32.57  20.37     27.00    29.74  14.69     18.81
2     2005-01-07   823.62    32.59  20.50     27.10    29.81  14.79     18.73
3     2005-01-10   832.99    32.99  20.58     27.29    30.13  14.83     19.07
4     2005-01-11   837.86    33.13  20.69     27.25    30.26  14.91     19.26
...          ...      ...      ...    ...       ...      ...    ...       ...
4800  2024-10-14  2723.18    33.03  11.48     17.17    32.32  11.27     17.79
4801  2024-10-15  2655.12    32.16  11.17     16.68    31.50  10.97     17.43
4802  2024-10-16  2650.18    31.65  11.19     16.42    30.20  11.00     17.45
4803  2024-10-17  2610.47    31.47  11.05     16.13    30.05  10.86     16.94
4804  2024-10-18  2681.91    33.61  11.29     16.60    32.22  11.09     17.35
[4805 rows x 8 columns]
```

#### 主板市净率

接口: stock_market_pb_lg

目标地址: https://legulegu.com/stockdata/shanghaiPB

描述: 乐咕乐股-主板市净率

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                |
|--------|-----|---------------------------------------------------|
| symbol | str | symbol="上证"; choice of {"上证", "深证", "创业板", "科创版"} |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| 日期     | object  | -   |
| 指数     | float64 | -   |
| 市净率    | float64 | -   |
| 等权市净率  | float64 | -   |
| 市净率中位数 | float64 | -   |

接口示例

```python
import akshare as ak

stock_market_pb_lg_df = ak.stock_market_pb_lg(symbol="上证")
print(stock_market_pb_lg_df)
```

数据示例

```
          日期     指数     市净率  等权市净率 市净率中位数
0     2005-01-04  3025.42  2.09   2.57    2.06
1     2005-01-05  3052.86  2.09   2.58    2.06
2     2005-01-06  3028.73  2.06   2.56    2.05
3     2005-01-07  3035.59  2.06   2.57    2.06
4     2005-01-10  3062.73  2.08   2.60    2.08
          ...      ...   ...    ...     ...
4686  2024-04-19  9279.46  1.79   2.65    1.87
4687  2024-04-22  9239.14  1.78   2.62    1.84
4688  2024-04-23  9183.14  1.77   2.62    1.86
4689  2024-04-24  9251.13  1.78   2.67    1.89
4690  2024-04-25  9264.48  1.78   2.69    1.90
[4691 rows x 5 columns]
```

#### 指数市净率

接口: stock_index_pb_lg

目标地址: https://legulegu.com/stockdata/sz50-pb

描述: 乐咕乐股-指数市净率

限量: 单次获取指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                                                                                                  |
|--------|-----|-------------------------------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="上证50"; choice of {"上证50", "沪深300", "上证380", "创业板50", "中证500", "上证180", "深证红利", "深证100", "中证1000", "上证红利", "中证100", "中证800"} |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| 日期     | object  | -   |
| 指数     | float64 | -   |
| 市净率    | float64 | -   |
| 等权市净率  | float64 | -   |
| 市净率中位数 | float64 | -   |

接口示例

```python
import akshare as ak

stock_index_pb_lg_df = ak.stock_index_pb_lg(symbol="上证50")
print(stock_index_pb_lg_df)
```

数据示例

```
          日期     指数   市净率  等权市净率  市净率中位数
0     2005-01-05   831.43  2.10   2.35    2.12
1     2005-01-06   822.50  2.07   2.34    2.12
2     2005-01-07   823.62  2.08   2.34    2.13
3     2005-01-10   832.99  2.08   2.37    2.16
4     2005-01-11   837.86  2.10   2.38    2.16
          ...      ...   ...    ...     ...
4686  2024-04-22  2424.75  1.14   2.97    1.92
4687  2024-04-23  2413.74  1.14   2.96    1.90
4688  2024-04-24  2424.12  1.14   2.98    1.90
4689  2024-04-25  2431.62  1.14   2.96    1.90
4690  2024-04-26  2461.86  1.15   2.96    1.94
[4691 rows x 5 columns]
```

#### A 股估值指标

接口: stock_zh_valuation_baidu

目标地址: https://gushitong.baidu.com/stock/ab-002044

描述: 百度股市通-A 股-财务报表-估值数据

限量: 单次获取指定 symbol 和 indicator 的所有历史数据

输入参数

| 名称        | 类型  | 描述                                                                     |
|-----------|-----|------------------------------------------------------------------------|
| symbol    | str | symbol="002044"; A 股代码                                                 |
| indicator | str | indicator="总市值"; choice of {"总市值", "市盈率(TTM)", "市盈率(静)", "市净率", "市现率"} |
| period    | str | period="近一年"; choice of {"近一年", "近三年", "近五年", "近十年", "全部"}             |

输出参数

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| date  | object  | -   |
| value | float64 | -   |

接口示例

```python
import akshare as ak

stock_zh_valuation_baidu_df = ak.stock_zh_valuation_baidu(symbol="002044", indicator="总市值", period="近一年")
print(stock_zh_valuation_baidu_df)
```

数据示例

```
           date   value
0    2023-05-29  245.42
1    2023-05-30  246.60
2    2023-05-31  249.73
3    2023-06-01  253.64
4    2023-06-02  259.52
..          ...     ...
362  2024-05-25  167.92
363  2024-05-26  167.92
364  2024-05-27  165.96
365  2024-05-28  164.40
366  2024-05-29  167.14
[367 rows x 2 columns]
```

#### 个股估值

接口: stock_value_em

目标地址: https://data.eastmoney.com/gzfx/detail/300766.html

描述: 东方财富网-数据中心-估值分析-每日互动-每日互动-估值分析

限量: 单次获取指定 symbol 的所有历史数据

输入参数

| 名称        | 类型  | 描述                                                                     |
|-----------|-----|------------------------------------------------------------------------|
| symbol    | str | symbol="002044"; A 股代码                                                 |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 数据日期    | object  | -       |
| 当日收盘价   | float64 | 注意单位: 元 |
| 当日涨跌幅   | float64 | 注意单位: % |
| 总市值     | float64 | 注意单位: 元 |
| 流通市值    | float64 | 注意单位: 元 |
| 总股本     | float64 | 注意单位: 股 |
| 流通股本    | float64 | -       |
| PE(TTM) | float64 | -       |
| PE(静)   | float64 | -       |
| 市净率     | float64 | -       |
| PEG值    | float64 | -       |
| 市现率     | float64 | -       |
| 市销率     | float64 | -       |

接口示例

```python
import akshare as ak

stock_value_em_df = ak.stock_value_em(symbol="300766")
print(stock_value_em_df)
```

数据示例

```
        数据日期  当日收盘价   当日涨跌幅  ...  PEG值      市现率       市销率
0     2019-03-25  18.84  44.036697  ... -1.428399   34.410716  13.976938
1     2019-03-26  20.72   9.978769  ... -1.570936   37.844482  15.371664
2     2019-03-27  22.79   9.990347  ... -1.727878   41.625277  16.907347
3     2019-03-28  25.07  10.004388  ... -1.900741   45.789631  18.598823
4     2019-03-29  27.58  10.011966  ... -2.091043   50.374074  20.460931
...          ...    ...        ...  ...       ...         ...        ...
1373  2024-11-20  15.30   6.397775  ... -0.055226 -128.339565  13.236721
1374  2024-11-21  15.33   0.196078  ... -0.055334 -128.591212  13.262675
1375  2024-11-22  14.82  -3.326810  ... -0.053493 -124.313226  12.821451
1376  2024-11-25  14.80  -0.134953  ... -0.053421 -124.145462  12.804148
1377  2024-11-26  14.85   0.337838  ... -0.053601 -124.564872  12.847406
[1378 rows x 13 columns]
```

#### 涨跌投票

接口: stock_zh_vote_baidu

目标地址: https://gushitong.baidu.com/index/ab-000001

描述: 百度股市通- A 股或指数-股评-投票

限量: 单次获取指定 symbol 和 indicator 的所有数据

输入参数

| 名称        | 类型  | 描述                                     |
|-----------|-----|----------------------------------------|
| symbol    | str | symbol="000001"; A 股股票或指数代码            |
| indicator | str | indicator="指数"; choice of {"指数", "股票"} |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 周期   | object | -   |
| 看涨   | object | -   |
| 看跌   | object | -   |
| 看涨比例 | object | -   |
| 看跌比例 | object | -   |

接口示例

```python
import akshare as ak

stock_zh_vote_baidu_df = ak.stock_zh_vote_baidu(symbol="000001", indicator="指数")
print(stock_zh_vote_baidu_df)
```

数据示例

```
   周期    看涨      看跌 看涨比例 看跌比例
0  今日    1531    4765  24%  76%
1  本周    1531    4765  24%  76%
2  本月    2442    7109  26%  74%
3  今年  15.04万  36.01万  29%  71%
```

#### 港股个股指标

P.S. 该数据源暂未更新数据

接口: stock_hk_indicator_eniu

目标地址: https://eniu.com/gu/hk01093/roe

描述: 亿牛网-港股个股指标: 市盈率, 市净率, 股息率, ROE, 市值

限量: 单次获取指定 symbol 和 indicator 的所有历史数据

输入参数

| 名称        | 类型  | 描述                                                                                              |
|-----------|-----|-------------------------------------------------------------------------------------------------|
| symbol    | str | symbol="hk01093"; 可通过调用 **ak.stock_hk_indicator_eniu(symbol="hk01093", indicator="港股")** 获取股票代码 |
| indicator | str | indicator="市盈率"; choice of {"港股", "市盈率", "市净率", "股息率", "ROE", "市值"}                             |

输出参数

| 名称  | 类型  | 描述              |
|-----|-----|-----------------|
| -   | -   | 根据 indicator 而异 |

接口示例-ROE

```python
import akshare as ak

stock_hk_indicator_eniu_df = ak.stock_hk_indicator_eniu(symbol="hk01093", indicator="市净率")
print(stock_hk_indicator_eniu_df)
```

数据示例-ROE

```
            date    pb  price
0     2000-01-03  1.17   0.00
1     2000-01-04  1.24   0.00
2     2000-01-05  1.19   0.00
3     2000-01-06  1.06   0.00
4     2000-01-07  1.08   0.00
          ...   ...    ...
5041  2020-08-07  7.31  15.74
5042  2020-08-10  7.22  15.54
5043  2020-08-11  7.31  15.74
5044  2020-08-12  7.15  15.40
5045  2020-08-13  7.18  15.46
```

#### 港股估值指标

接口: stock_hk_valuation_baidu

目标地址: https://gushitong.baidu.com/stock/hk-06969

描述: 百度股市通-港股-财务报表-估值数据

限量: 单次获取指定 symbol 的指定 indicator 的特定 period 的历史数据

输入参数

| 名称        | 类型  | 描述                                                                     |
|-----------|-----|------------------------------------------------------------------------|
| symbol    | str | symbol="02358"; 港股代码                                                   |
| indicator | str | indicator="总市值"; choice of {"总市值", "市盈率(TTM)", "市盈率(静)", "市净率", "市现率"} |
| period    | str | period="近一年"; choice of {"近一年", "近三年", "全部"}                           |

输出参数

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| date  | object  | -   |
| value | float64 | -   |

接口示例

```python
import akshare as ak

stock_hk_valuation_baidu_df = ak.stock_hk_valuation_baidu(symbol="06969", indicator="总市值", period="近一年")
print(stock_hk_valuation_baidu_df)
```

数据示例

```
           date   value
0    2023-11-21  427.15
1    2023-11-22  428.99
2    2023-11-23  441.88
3    2023-11-24  430.83
4    2023-11-25  430.83
..          ...     ...
362  2024-11-17  602.46
363  2024-11-18  632.73
364  2024-11-19  679.69
365  2024-11-20  713.06
366  2024-11-21  688.35
[367 rows x 2 columns]
```

#### 美股估值指标

接口: stock_us_valuation_baidu

目标地址: https://gushitong.baidu.com/stock/us-NVDA

描述: 百度股市通-美股-财务报表-估值数据

限量: 单次获取指定 symbol 的指定 indicator 的特定 period 的历史数据

输入参数

| 名称        | 类型  | 描述                                                                     |
|-----------|-----|------------------------------------------------------------------------|
| symbol    | str | symbol="NVDA"; 美股代码                                                    |
| indicator | str | indicator="总市值"; choice of {"总市值", "市盈率(TTM)", "市盈率(静)", "市净率", "市现率"} |
| period    | str | period="近一年"; choice of {"近一年", "近三年", "全部"}                           |

输出参数

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| date  | object  | -   |
| value | float64 | -   |

接口示例

```python
import akshare as ak

stock_us_valuation_baidu_df = ak.stock_us_valuation_baidu(symbol="NVDA", indicator="总市值", period="近一年")
print(stock_us_valuation_baidu_df)
```

数据示例

```
           date     value
0    2024-12-24  34339.88
1    2024-12-26  34268.86
2    2024-12-27  33553.75
3    2024-12-30  33671.30
4    2024-12-31  32887.62
..          ...       ...
245  2025-12-17  41538.42
246  2025-12-18  42316.02
247  2025-12-19  43980.57
248  2025-12-22  44636.67
249  2025-12-23  45978.03
[250 rows x 2 columns]
```

#### 创新高和新低的股票数量

接口: stock_a_high_low_statistics

目标地址: https://www.legulegu.com/stockdata/high-low-statistics

描述: 不同市场的创新高和新低的股票数量

限量: 单次获取指定 market 的近两年的历史数据

输入参数

| 名称     | 类型  | 描述                                                                                |
|--------|-----|-----------------------------------------------------------------------------------|
| symbol | str | symbol="all"; {"all": "全部A股", "sz50": "上证50", "hs300": "沪深300", "zz500": "中证500"} |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| date    | object  | 交易日     |
| close   | float64 | 相关指数收盘价 |
| high20  | int64   | 20日新高   |
| low20   | int64   | 20日新低   |
| high60  | int64   | 60日新高   |
| low60   | int64   | 60日新低   |
| high120 | int64   | 120日新高  |
| low120  | int64   | 120日新低  |

接口示例

```python
import akshare as ak

stock_a_high_low_statistics_df = ak.stock_a_high_low_statistics(symbol="all")
print(stock_a_high_low_statistics_df)
```

数据示例

```
           date    close  high20  low20  high60  low60  high120  low120
0    2021-08-18  3485.29     441    293     212    143      143      67
1    2021-08-19  3465.55     303    332     162    189      132     106
2    2021-08-20  3427.33     361    628     214    399      168     227
3    2021-08-23  3477.13     827    102     417     68      332      44
4    2021-08-24  3514.47     843    128     399     83      306      50
..          ...      ...     ...    ...     ...    ...      ...     ...
495  2023-09-04  3177.06     811     65     269     30      139      17
496  2023-09-05  3154.37     512    203     125     57       70      24
497  2023-09-06  3122.35     254    240     103    125       74      65
498  2023-09-06  3158.08     867    200     169     84       91      44
499  2023-09-08  3116.72     393    513     104    281       60     167
[500 rows x 8 columns]
```

#### 破净股统计

接口: stock_a_below_net_asset_statistics

目标地址: https://www.legulegu.com/stockdata/below-net-asset-statistics

描述: 乐咕乐股-A 股破净股统计数据

限量: 单次获取指定 symbol 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                          |
|--------|-----|-------------------------------------------------------------|
| symbol | str | symbol="全部A股"; choice of {"全部A股", "沪深300", "上证50", "中证500"} |

输出参数-全部A股

| 名称                    | 类型      | 描述    |
|-----------------------|---------|-------|
| date                  | object  | 交易日   |
| below_net_asset       | float64 | 破净股家数 |
| total_company         | float64 | 总公司数  |
| below_net_asset_ratio | float64 | 破净股比率 |

接口示例-全部 A 股

```python
import akshare as ak

stock_a_below_net_asset_statistics_df = ak.stock_a_below_net_asset_statistics(symbol="全部A股")
print(stock_a_below_net_asset_statistics_df)
```

数据示例-全部 A 股

```
            date  below_net_asset  total_company  below_net_asset_ratio
0     2005-01-04               36           1352                 0.0266
1     2005-01-05               31           1352                 0.0229
2     2005-01-06               32           1352                 0.0237
3     2005-01-09               33           1352                 0.0244
4     2005-01-10               29           1352                 0.0214
          ...              ...            ...                    ...
4682  2024-04-21              577           5114                 0.1128
4683  2024-04-22              561           5114                 0.1097
4684  2024-04-23              551           5114                 0.1077
4685  2024-04-24              533           5114                 0.1042
4686  2024-04-25              516           5113                 0.1009
[4687 rows x 4 columns]
```

输出参数-沪深 300

| 名称                    | 类型      | 描述    |
|-----------------------|---------|-------|
| date                  | object  | 交易日   |
| below_net_asset       | float64 | 破净股家数 |
| total_company         | float64 | 总公司数  |
| below_net_asset_ratio | float64 | 破净股比率 |

接口示例-沪深 300

```python
import akshare as ak

stock_a_below_net_asset_statistics_df = ak.stock_a_below_net_asset_statistics(symbol="沪深300")
print(stock_a_below_net_asset_statistics_df)
```

数据示例-沪深 300

```
            date  below_net_asset  total_company  below_net_asset_ratio
0     2005-04-07               22            299                 0.0736
1     2005-04-10               21            299                 0.0702
2     2005-04-11               23            299                 0.0769
3     2005-04-12               20            299                 0.0669
4     2005-04-13               22            299                 0.0736
          ...              ...            ...                    ...
4627  2024-04-21               63            300                 0.2100
4628  2024-04-22               63            300                 0.2100
4629  2024-04-23               63            300                 0.2100
4630  2024-04-24               63            300                 0.2100
4631  2024-04-25               62            300                 0.2067
[4632 rows x 4 columns]
```

#### 基金持股

接口: stock_report_fund_hold

目标地址: http://data.eastmoney.com/zlsj/2020-06-30-1-2.html

描述: 东方财富网-数据中心-主力数据-基金持仓

限量: 单次返回指定 symbol 和 date 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                          |
|--------|-----|-----------------------------------------------------------------------------|
| symbol | str | symbol="基金持仓"; choice of {"基金持仓", "QFII持仓", "社保持仓", "券商持仓", "保险持仓", "信托持仓"} |
| date   | str | date="20200630"; 财报发布日期, xxxx-03-31, xxxx-06-30, xxxx-09-30, xxxx-12-31     |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 持有基金家数 | int64   | 注意单位: 家 |
| 持股总数   | int64   | 注意单位: 股 |
| 持股市值   | float64 | 注意单位: 元 |
| 持股变化   | object  | -       |
| 持股变动数值 | int64   | 注意单位: 股 |
| 持股变动比例 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_report_fund_hold_df = ak.stock_report_fund_hold(symbol="基金持仓", date="20200630")
print(stock_report_fund_hold_df)
```

数据示例

```
        序号    股票代码  股票简称  ... 持股变化   持股变动数值             持股变动比例
0        1  300846  首都在线  ...   新进  3024892         0.00219976
1        2  300843  胜蓝股份  ...   新进  1984496         0.00293824
2        3  300842  帝科股份  ...   新进  1190928   1.19092800000001
3        4  600956  新天绿能  ...   新进  6460517  0.167734880000004
4        5  300845  捷安高科  ...   新进  1211353         0.00388496
    ...     ...   ...  ...  ...      ...                ...
3555  3556  300752  隆利科技  ...   增持     3100         0.00258868
3556  3557  300778   新城市  ...   新进      125           0.000125
3557  3558  300786  国林科技  ...   增持     2800         0.00327715
3558  3559  300818  耐普矿机  ...   新进      500         0.00071429
3559  3560  688981  中芯国际  ...   新进   139500         0.00255926
```

#### 基金持股明细

接口: stock_report_fund_hold_detail

目标地址: http://data.eastmoney.com/zlsj/ccjj/2020-12-31-008286.html

描述: 东方财富网-数据中心-主力数据-基金持仓-基金持仓明细表

限量: 单次返回指定 symbol 和 date 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                      |
|--------|-----|-------------------------------------------------------------------------|
| symbol | str | symbol="005827"; 基金代码                                                   |
| date   | str | date="20200630"; 财报发布日期, xxxx-03-31, xxxx-06-30, xxxx-09-30, xxxx-12-31 |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 股票代码    | object  | -       |
| 股票简称    | object  | -       |
| 持股数     | int64   | 注意单位: 股 |
| 持股市值    | float64 | 注意单位: 元 |
| 占总股本比例  | float64 | 注意单位: % |
| 占流通股本比例 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_report_fund_hold_detail_df = ak.stock_report_fund_hold_detail(symbol="005827", date="20201231")
print(stock_report_fund_hold_detail_df)
```

数据示例

```
   序号       股票代码  股票简称       持股数           持股市值 占总股本比例 占流通股本比例
0   1  600519.SH  贵州茅台   3265532     6524532936   0.26    0.26
1   2  300015.SZ  爱尔眼科  32000005  2396480374.45   0.78    0.80
2   3  002415.SZ  海康威视  70000025  3395701212.75   0.75    0.85
3   4  002304.SZ  洋河股份  27200000     6418928000   1.80    2.16
4   5  000858.SZ   五粮液  21858499  6379402933.15   0.56    0.58
5   6  000568.SZ  泸州老窖  27700000     6264632000   1.89    1.89
```

#### 龙虎榜

##### 龙虎榜-东财

###### 龙虎榜详情

接口: stock_lhb_detail_em

目标地址: https://data.eastmoney.com/stock/tradedetail.html

描述: 东方财富网-数据中心-龙虎榜单-龙虎榜详情

限量: 单次返回所有历史数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| start_date | str | start_date="20220314" |
| end_date   | str | end_date="20220315"   |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 代码       | object  | -       |
| 名称       | object  | -       |
| 上榜日      | object  | -       |
| 解读       | object  | -       |
| 收盘价      | float64 | -       |
| 涨跌幅      | float64 | 注意单位: % |
| 龙虎榜净买额   | float64 | 注意单位: 元 |
| 龙虎榜买入额   | float64 | 注意单位: 元 |
| 龙虎榜卖出额   | float64 | 注意单位: 元 |
| 龙虎榜成交额   | float64 | 注意单位: 元 |
| 市场总成交额   | int64   | 注意单位: 元 |
| 净买额占总成交比 | float64 | 注意单位: % |
| 成交额占总成交比 | float64 | 注意单位: % |
| 换手率      | float64 | 注意单位: % |
| 流通市值     | float64 | 注意单位: 元 |
| 上榜原因     | object  | -       |
| 上榜后1日    | float64 | 注意单位: % |
| 上榜后2日    | float64 | 注意单位: % |
| 上榜后5日    | float64 | 注意单位: % |
| 上榜后10日   | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_lhb_detail_em_df = ak.stock_lhb_detail_em(start_date="20230403", end_date="20230417")
print(stock_lhb_detail_em_df)
```

数据示例

```
      序号  代码    名称    上榜日  ...     上榜后1日      上榜后2日      上榜后5日    上榜后10日
0      1  000021   深科技  2023-04-06  ... -1.304348  -0.193237  -8.212560       NaN
1      2  000021   深科技  2023-04-03  ...  2.338227  12.561175  11.201740  7.667210
2      3  000021   深科技  2023-04-03  ...  2.338227  12.561175  11.201740  7.667210
3      4  000032  深桑达A  2023-04-06  ...  2.087576  -2.214868  -2.265784       NaN
4      5  000063  中兴通讯  2023-04-07  ... -3.783784  -5.270270  -7.027027       NaN
..   ...     ...   ...         ...  ...       ...        ...        ...       ...
630  631  688698  伟创电气  2023-04-12  ... -4.407407  -2.592593        NaN       NaN
631  632  688787  海天瑞声  2023-04-10  ...  2.028249   5.644068   6.073446       NaN
632  633  873593  鼎智科技  2023-04-17  ... -7.805655        NaN        NaN       NaN
633  634  900915  中路B股  2023-04-12  ...  2.981030   1.490515        NaN       NaN
634  635  900915  中路B股  2023-04-03  ...  5.807365   3.824363  12.039660  3.399433
```

###### 个股上榜统计

接口: stock_lhb_stock_statistic_em

目标地址: https://data.eastmoney.com/stock/tradedetail.html

描述: 东方财富网-数据中心-龙虎榜单-个股上榜统计

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol="近一月"; choice of {"近一月", "近三月", "近六月", "近一年"} |

输出参数

| 名称      | 类型      | 描述 |
|---------|---------|----|
| 序号      | int64   | -  |
| 代码      | object  | -  |
| 名称      | object  | -  |
| 最近上榜日   | object  | -  |
| 收盘价     | float64 | -  |
| 涨跌幅     | float64 | -  |
| 上榜次数    | int64   | -  |
| 龙虎榜净买额  | float64 | -  |
| 龙虎榜买入额  | float64 | -  |
| 龙虎榜卖出额  | float64 | -  |
| 龙虎榜总成交额 | float64 | -  |
| 买方机构次数  | int64   | -  |
| 卖方机构次数  | int64   | -  |
| 机构买入净额  | float64 | -  |
| 机构买入总额  | float64 | -  |
| 机构卖出总额  | float64 | -  |
| 近1个月涨跌幅 | float64 | -  |
| 近3个月涨跌幅 | float64 | -  |
| 近6个月涨跌幅 | float64 | -  |
| 近1年涨跌幅  | float64 | -  |

接口示例

```python
import akshare as ak

stock_lhb_stock_statistic_em_df = ak.stock_lhb_stock_statistic_em(symbol="近一月")
print(stock_lhb_stock_statistic_em_df)
```

数据示例

```
      序号  代码    名称     最近上榜日  ...   近1个月涨跌幅  近3个月涨跌幅 近6个月涨跌幅  近1年涨跌幅
0      1  605180  华生科技  2024-04-18  ...  39.940828  -8.274079   7.744875   19.350450
1      2  603603  退市博天  2024-04-18  ... -61.000000 -61.386139 -86.315789  -90.250000
2      3  002089   新海退  2024-04-17  ... -69.230769 -76.699029 -80.645161  -75.510204
3      4  000506  中润资源  2024-04-19  ...  55.844156  61.725067  26.849894   11.940299
4      5  301500  飞南资源  2024-04-19  ...  16.005874  21.788284   3.765324   -1.126408
..   ...     ...   ...         ...  ...        ...        ...        ...         ...
653  654  603322  超讯通信  2024-03-21  ...  22.253172   4.969439  39.035551  127.403569
654  655  603528  多伦科技  2024-03-21  ...  57.685353  -4.803493  20.275862   12.761000
655  656  605296  神农集团  2024-03-21  ...  19.154229  10.477759  59.590671   33.666462
656  657  688103  国力股份  2024-03-21  ...  22.362869 -16.219563 -24.842651  -32.964050
657  658  900921  丹科B股  2024-03-21  ...   2.097902   8.955224   5.797101  -18.435754
[658 rows x 20 columns]
```

###### 机构买卖每日统计

接口: stock_lhb_jgmmtj_em

目标地址: https://data.eastmoney.com/stock/jgmmtj.html

描述: 东方财富网-数据中心-龙虎榜单-机构买卖每日统计

限量: 单次返回所有历史数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| start_date | str | start_date="20240417" |
| end_date   | str | end_date="20240430"   |

输出参数

| 名称          | 类型      | 描述       |
|-------------|---------|----------|
| 序号          | int64   | -        |
| 代码          | object  | -        |
| 名称          | object  | -        |
| 收盘价         | float64 | -        |
| 涨跌幅         | float64 | -        |
| 买方机构数       | float64 | -        |
| 卖方机构数       | float64 | -        |
| 机构买入总额      | float64 | 注意单位: 元  |
| 机构卖出总额      | float64 | 注意单位: 元  |
| 机构买入净额      | float64 | 注意单位: 元  |
| 市场总成交额      | float64 | 注意单位: 元  |
| 机构净买额占总成交额比 | float64 | -        |
| 换手率         | float64 | -        |
| 流通市值        | float64 | 注意单位: 亿元 |
| 上榜原因        | object  | -        |
| 上榜日期        | object  | -        |

接口示例

```python
import akshare as ak

stock_lhb_jgmmtj_em_df = ak.stock_lhb_jgmmtj_em(start_date="20240417", end_date="20240430")
print(stock_lhb_jgmmtj_em_df)
```

数据示例

```
      序号  代码  ...                                    上榜原因                   上榜日期
0      1  600919  ...           有价格涨跌幅限制的日收盘价格跌幅偏离值达到7%的前五只证券  2024-04-26
1      2  603486  ...  非ST、*ST和S证券连续三个交易日内收盘价格涨幅偏离值累计达到20%的证券  2024-04-22
2      3  688019  ...             有价格涨跌幅限制的日收盘价格涨幅达到15%的前五只证券  2024-04-29
3      4  002920  ...                        日跌幅偏离值达到7%的前5只证券  2024-04-19
4      5  000628  ...                         日换手率达到20%的前5只证券  2024-04-29
..   ...     ...  ...                                     ...         ...
334  335  000628  ...                连续三个交易日内，涨幅偏离值累计达到20%的证券  2024-04-26
335  336  000099  ...                        日涨幅偏离值达到7%的前5只证券  2024-04-18
336  337  688027  ...             有价格涨跌幅限制的日收盘价格涨幅达到15%的前五只证券  2024-04-26
337  338  002085  ...                        日涨幅偏离值达到7%的前5只证券  2024-04-24
338  339  603882  ...           有价格涨跌幅限制的日收盘价格跌幅偏离值达到7%的前五只证券  2024-04-26
[339 rows x 16 columns]
```

###### 机构席位追踪

接口: stock_lhb_jgstatistic_em

目标地址: https://data.eastmoney.com/stock/jgstatistic.html

描述: 东方财富网-数据中心-龙虎榜单-机构席位追踪

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol="近一月"; choice of {"近一月", "近三月", "近六月", "近一年"} |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 代码      | object  | -       |
| 名称      | object  | -       |
| 收盘价     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |
| 龙虎榜成交金额 | float64 | 注意单位: 元 |
| 上榜次数    | int64   | -       |
| 机构买入额   | float64 | 注意单位: 元 |
| 机构买入次数  | int64   | -       |
| 机构卖出额   | float64 | 注意单位: 元 |
| 机构卖出次数  | int64   | -       |
| 机构净买额   | float64 | 注意单位: 元 |
| 近1个月涨跌幅 | float64 | 注意单位: % |
| 近3个月涨跌幅 | float64 | 注意单位: % |
| 近6个月涨跌幅 | float64 | 注意单位: % |
| 近1年涨跌幅  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_lhb_jgstatistic_em_df = ak.stock_lhb_jgstatistic_em(symbol="近一月")
print(stock_lhb_jgstatistic_em_df)
```

数据示例

```
    序号   代码    名称    收盘价  ...   近1个月涨跌幅  近3个月涨跌幅  近6个月涨跌幅  近1年涨跌幅
0      1  000099  中信海直  22.28  ...   41.730280  142.966194  179.197995  196.852398
1      2  000506  中润资源   5.40  ...   27.962085   45.945946   13.924051   -1.459854
2      3  301500  飞南资源  23.90  ...   18.199802   25.459318    5.752212   -0.292032
3      4  001376  百通能源  20.05  ...   15.537337   10.231084  344.168612  344.168612
4      5  300139  晓程科技  13.70  ...   56.930126   50.715072   27.205200   22.759857
..   ...     ...   ...    ...  ...         ...         ...         ...         ...
366  367  688691  灿芯股份  50.37  ...  153.625378  153.625378  153.625378  153.625378
367  368  688701  卓锦股份   4.80  ...  -20.265781  -47.426068  -45.080092  -43.195266
368  369  830839  万通液压  10.99  ...   -8.263773   -2.397869   45.370370   77.031095
369  370  835438   戈碧迦  26.93  ...  169.300000  169.300000  169.300000  169.300000
370  371  837006  晟楠科技  14.20  ...   -0.350877   -5.396402   40.454995   74.433640
[371 rows x 16 columns]
```

###### 每日活跃营业部

接口: stock_lhb_hyyyb_em

目标地址: https://data.eastmoney.com/stock/hyyyb.html

描述: 东方财富网-数据中心-龙虎榜单-每日活跃营业部

限量: 单次返回所有历史数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| start_date | str | start_date="20220311" |
| end_date   | str | end_date="20220315"   |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 营业部名称 | object  | -       |
| 上榜日   | object  | -       |
| 买入个股数 | float64 | -       |
| 卖出个股数 | float64 | -       |
| 买入总金额 | float64 | 注意单位: 元 |
| 卖出总金额 | float64 | 注意单位: 元 |
| 总买卖净额 | float64 | 注意单位: 元 |
| 买入股票  | object  | -       |

接口示例

```python
import akshare as ak

stock_lhb_hyyyb_em_df = ak.stock_lhb_hyyyb_em(start_date="20220324", end_date="20220324")
print(stock_lhb_hyyyb_em_df)
```

数据示例

```
       序号  ...                                               买入股票
0        1  ...  中信海直 高新发展 张裕A 新乡化纤 宗申动力 湖南黄金 桂林三金 科华数据 柘中股份 *S...
1        2  ...  中信海直 焦作万方 北方铜业 钱江摩托 新乡化纤 ST中南 百通能源 宗申动力 万丰奥威 百...
2        3  ...                                               宗申动力
3        4  ...                                               罗博特科
4        5  ...                                          中科通达 神州信息
...    ...  ...                                                ...
7532  7533  ...  万科A 中信海直 神州信息 高新发展 康冠科技 宗申动力 万丰奥威 沃尔核材 江特电机 雪迪...
7533  7534  ...  中信海直 我爱我家 长安汽车 高新发展 天保基建 万丰奥威 荣盛发展 百川股份 万安科技 奥...
7534  7535  ...                                               长安汽车
7535  7536  ...  双良节能 鲁信创投 江苏银行 太平洋 东方材料 日出东方 金域医学 国盾量子 信安世纪 财富...
7536  7537  ...                                          长安汽车 荣盛发展
[7537 rows x 9 columns]
```

#### 营业部详情数据-东财

接口: stock_lhb_yyb_detail_em

目标地址: https://data.eastmoney.com/stock/lhb/yyb/10188715.html

描述: 东方财富网-数据中心-龙虎榜单-营业部历史交易明细-营业部交易明细

限量: 单次返回指定营业部的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                        |
|--------|-----|-----------------------------------------------------------|
| symbol | str | symbol="10026729"; 营业部代码, 通过 ak.stock_lhb_hyyyb_em() 接口获取 |

输出参数

| 名称       | 类型      | 描述                     |
|----------|---------|------------------------|
| 序号       | int64   | -                      |
| 营业部代码    | object  | -                      |
| 营业部名称    | object  | -                      |
| 营业部简称    | object  | -                      |
| 交易日期     | object  | -                      |
| 股票代码     | object  | -                      |
| 股票名称     | object  | -                      |
| 涨跌幅      | float64 | 注意单位: %                |
| 买入金额     | float64 | 注意单位: 元                |
| 卖出金额     | float64 | 注意单位: 元                |
| 净额       | float64 | 注意单位: 元                |
| 上榜原因     | object  | -                      |
| 1日后涨跌幅   | float64 | 注意单位: %                |
| 2日后涨跌幅   | float64 | 注意单位: %                |
| 3日后涨跌幅   | float64 | 注意单位: %                |
| 5日后涨跌幅   | float64 | 注意单位: %                |
| 10日后涨跌幅  | float64 | 注意单位: %                |
| 20日后涨跌幅  | float64 | 注意单位: %                |
| 30日后涨跌幅  | float64 | 注意单位: %                |

接口示例

```python
import akshare as ak

stock_lhb_yyb_detail_em_df = ak.stock_lhb_yyb_detail_em(symbol="10188715")
print(stock_lhb_yyb_detail_em_df)
```

数据示例

```
     序号  营业部代码                      营业部名称  ...    10日后涨跌幅    20日后涨跌幅    30日后涨跌幅
0      1  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ... -18.448276        NaN        NaN
1      2  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ... -16.309524 -32.023810        NaN
2      3  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...  -0.361011   2.045728 -12.033694
3      4  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...   8.633094  14.388489  -0.719424
4      5  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...  13.247863  38.632479  22.564103
..   ...       ...                       ...  ...        ...        ...        ...
97    98  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...  -2.818991  -9.792285  -1.483680
98    99  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...  12.643678  24.856322  30.747126
99   100  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...  84.564546  58.426793  58.604601
100  101  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ...   6.680370  -7.502569  -6.269270
101  102  10188715  东方证券股份有限公司上海普陀区云岭东路证券营业部  ... -19.966159 -20.135364  -9.983080
[102 rows x 19 columns]
```

###### 营业部排行

接口: stock_lhb_yybph_em

目标地址: https://data.eastmoney.com/stock/yybph.html

描述: 东方财富网-数据中心-龙虎榜单-营业部排行

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol="近一月"; choice of {"近一月", "近三月", "近六月", "近一年"} |

输出参数

| 名称          | 类型      | 描述      |
|-------------|---------|---------|
| 序号          | int64   | -       |
| 营业部名称       | object  | -       |
| 上榜后1天-买入次数  | int64   | -       |
| 上榜后1天-平均涨幅  | float64 | 注意单位: % |
| 上榜后1天-上涨概率  | float64 | 注意单位: % |
| 上榜后2天-买入次数  | int64   | -       |
| 上榜后2天-平均涨幅  | float64 | 注意单位: % |
| 上榜后2天-上涨概率  | float64 | 注意单位: % |
| 上榜后3天-买入次数  | int64   | -       |
| 上榜后3天-平均涨幅  | float64 | 注意单位: % |
| 上榜后3天-上涨概率  | float64 | 注意单位: % |
| 上榜后4天-买入次数  | int64   | -       |
| 上榜后4天-平均涨幅  | float64 | 注意单位: % |
| 上榜后4天-上涨概率  | float64 | 注意单位: % |
| 上榜后10天-买入次数 | int64   | -       |
| 上榜后10天-平均涨幅 | float64 | 注意单位: % |
| 上榜后10天-上涨概率 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_lhb_yybph_em_df = ak.stock_lhb_yybph_em(symbol="近一月")
print(stock_lhb_yybph_em_df)
```

数据示例

```
      序号               营业部名称  ...  上榜后10天-平均涨幅  上榜后10天-上涨概率
0        1  东方财富证券拉萨东环路第二证券营业部  ...   -10.314787    26.900585
1        2    东方财富证券拉萨八一路证券营业部  ...   -11.566581    19.161677
2        3  东方财富证券拉萨团结路第二证券营业部  ...    -9.394130    30.147059
3        4  东方财富证券拉萨团结路第一证券营业部  ...   -10.351127    20.833333
4        5               深股通专用  ...    -4.880778    30.769231
...    ...                 ...  ...          ...          ...
1619  1620   中银国际证券焦作解放东路证券营业部  ...     8.383234   100.000000
1620  1621    华西证券北京马家堡东路证券营业部  ...   -14.800262     0.000000
1621  1622      平安证券东莞石竹路证券营业部  ...     0.000000     0.000000
1622  1623  中国银河证券北京东四环南路证券营业部  ...     0.000000     0.000000
1623  1624      招商证券郑州北龙湖证券营业部  ...     9.370988   100.000000
[1624 rows x 17 columns]
```

###### 营业部统计

接口: stock_lhb_traderstatistic_em

目标地址: https://data.eastmoney.com/stock/traderstatistic.html

描述: 东方财富网-数据中心-龙虎榜单-营业部统计

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol="近一月"; choice of {"近一月", "近三月", "近六月", "近一年"} |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 序号      | int64   | -       |
| 营业部名称   | object  | -       |
| 龙虎榜成交金额 | float64 | -       |
| 上榜次数    | int64   | -       |
| 买入额     | float64 | 注意单位: 元 |
| 买入次数    | int64   | -       |
| 卖出额     | float64 | 注意单位: 元 |
| 卖出次数    | int64   | -       |

接口示例

```python
import akshare as ak

stock_lhb_traderstatistic_em_df = ak.stock_lhb_traderstatistic_em(symbol="近一月")
print(stock_lhb_traderstatistic_em_df)
```

数据示例

```
    序号                     营业部名称       龙虎榜成交金额  ...  买入次数           卖出额  卖出次数
0        1                     深股通专用  2.143017e+10  ...   269  1.116966e+10   269
1        2     华宝证券股份有限公司上海东大名路证券营业部  1.043461e+10  ...    46  5.103830e+09    42
2        3           华鑫证券有限责任公司上海分公司  7.489912e+09  ...   180  3.422379e+09   155
3        4    国泰君安证券股份有限公司成都双庆路证券营业部  7.023039e+09  ...    14  3.509692e+09    13
4        5                     沪股通专用  6.663151e+09  ...    96  3.758107e+09   107
...    ...                       ...           ...  ...   ...           ...   ...
1620  1621    中信建投证券股份有限公司锦州解放路证券营业部  1.794000e+04  ...     1  0.000000e+00     0
1621  1622      中信证券股份有限公司如皋福寿路证券营业部  1.346730e+04  ...     1  0.000000e+00     0
1622  1623  申万宏源证券有限公司上海浦东新区沪南路证券营业部  1.178100e+04  ...     1  0.000000e+00     0
1623  1624      华泰证券股份有限公司沈阳光荣街证券营业部  1.155000e+04  ...     1  0.000000e+00     0
1624  1625      国信证券股份有限公司天津友谊路证券营业部  6.560000e+02  ...     1  0.000000e+00     0
[1625 rows x 8 columns]
```

###### 个股龙虎榜详情

接口: stock_lhb_stock_detail_em

目标地址: https://data.eastmoney.com/stock/lhb/600077.html

描述: 东方财富网-数据中心-龙虎榜单-个股龙虎榜详情

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                                            |
|--------|-----|-----------------------------------------------------------------------------------------------|
| symbol | str | symbol="600077";                                                                              |
| date   | str | date="20220310"; 需要通过 ak.stock_lhb_stock_detail_date_em(symbol="600077") 接口获取相应股票的有龙虎榜详情数据的日期 |
| flag   | str | flag="卖出";  choice of {"买入", "卖出"}                                                            |

输出参数

| 名称          | 类型      | 描述               |
|-------------|---------|------------------|
| 序号          | int64   | -                |
| 交易营业部名称     | object  | -                |
| 买入金额        | float64 | -                |
| 买入金额-占总成交比例 | float64 | -                |
| 卖出金额-占总成交比例 | float64 | -                |
| 净额          | float64 | -                |
| 类型          | object  | 该字段主要处理多种龙虎榜标准问题 |

接口示例

```python
import akshare as ak

stock_lhb_stock_detail_em_df = ak.stock_lhb_stock_detail_em(symbol="600077", date="20070416", flag="买入")
print(stock_lhb_stock_detail_em_df)
```

数据示例

```
   序号                  交易营业部名称  ...          净额                        类型
0   1           国泰君安证券股份有限公司总部  ...  5025114.99  有价格涨跌幅限制的日换手率达到20％的前三只证券
1   2  中银国际证券有限责任公司武汉黄孝河路证券营业部  ...  3435790.02  有价格涨跌幅限制的日换手率达到20％的前三只证券
2   3    华泰证券有限责任公司梧州蝶山二路证券营业部  ...  2723460.00  有价格涨跌幅限制的日换手率达到20％的前三只证券
3   4    东吴证券有限责任公司昆山前进中路证券营业部  ...  1844064.62  有价格涨跌幅限制的日换手率达到20％的前三只证券
4   5    国信证券有限责任公司深圳红岭中路证券营业部  ...  1770274.92  有价格涨跌幅限制的日换手率达到20％的前三只证券
[5 rows x 8 columns]
```

##### 龙虎榜-营业部排行

###### 龙虎榜-营业部排行-上榜次数最多

接口: stock_lh_yyb_most

目标地址: https://data.10jqka.com.cn/market/longhu/

描述: 龙虎榜-营业部排行-上榜次数最多

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称        | 类型     | 描述  |
|-----------|--------|-----|
| 序号        | int64  | -   |
| 营业部名称     | object | -   |
| 上榜次数      | int64  | -   |
| 合计动用资金    | object | -   |
| 年内上榜次数    | int64  | -   |
| 年内买入股票只数  | int64  | -   |
| 年内3日跟买成功率 | object | -   |

接口示例

```python
import akshare as ak

stock_lh_yyb_most_df = ak.stock_lh_yyb_most()
print(stock_lh_yyb_most_df)
```

数据示例

```
      序号                         营业部名称  上榜次数  ... 年内上榜次数  年内买入股票只数  年内3日跟买成功率
0      1  东方财富证券股份有限公司拉萨金融城南环...  一线游资    15  ...   3160      2369     41.40%
1      2  东方财富证券股份有限公司拉萨东环路第二...  一线游资    14  ...   4145      2978     43.94%
2      3  东方财富证券股份有限公司拉萨团结路第二...  一线游资    13  ...   3985      2856     43.21%
3      4        东方财富证券股份有限公司拉萨团结路第一...    13  ...   2685      1858     44.00%
4      5                   深股通专用  一线游资    11  ...   1780      1567     44.99%
..   ...                           ...   ...  ...    ...       ...        ...
288  289  华泰证券股份有限公司上海奉贤区望园南路...  跟风高手     1  ...     37        11    100.00%
289  290          华创证券有限责任公司贵阳北京路证券营业部     1  ...      4         1      无符合数据
290  291        中信建投证券股份有限公司东三环中路证券...     1  ...      4         1      无符合数据
291  292        国泰君安证券股份有限公司深圳福田中心区...     1  ...     43        24     48.00%
292  293         大同证券有限责任公司运城人民北路证券营业部     1  ...      1         1      无符合数据
[293 rows x 7 columns]
```

###### 龙虎榜-营业部排行-资金实力最强

接口: stock_lh_yyb_capital

目标地址: https://data.10jqka.com.cn/market/longhu/

描述: 龙虎榜-营业部排行-资金实力最强

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型     | 描述  |
|----------|--------|-----|
| 序号       | int64  | -   |
| 营业部名称    | object | -   |
| 今日最高操作   | int64  | -   |
| 今日最高金额   | object | -   |
| 今日最高买入金额 | object | -   |
| 累计参与金额   | object | -   |
| 累计买入金额   | object | -   |

接口示例

```python
import akshare as ak

stock_lh_yyb_capital_df = ak.stock_lh_yyb_capital()
print(stock_lh_yyb_capital_df)
```

数据示例

```
      序号                         营业部名称  今日最高操作  今日最高金额 今日最高买入金额  累计参与金额  累计买入金额
0      1   华宝证券股份有限公司上海东大名路证券营业部  一线游资       5  14.72亿    7.35亿  20.32亿  10.18亿
1      2  东方证券股份有限公司上海松江区沪亭北路...  知名游资       2   8.98亿    4.49亿  10.80亿   5.40亿
2      3         华鑫证券有限责任公司上海分公司  一线游资       8   8.63亿    4.32亿  12.23亿   6.52亿
3      4        国泰君安证券股份有限公司成都双庆路证券...       2   8.00亿    4.00亿   9.77亿   4.88亿
4      5        东方证券股份有限公司上海静安区延平路证...       2   4.60亿    2.30亿   6.36亿   3.18亿
..   ...                           ...     ...     ...      ...     ...     ...
288  289        长江证券股份有限公司北京百万庄大街证券...       1   2.00万    2.00万   2.00万   2.00万
289  290          国联证券股份有限公司无锡中山路证券营业部       1   1.87万     0.00   1.87万    0.00
290  291        中信建投证券股份有限公司武汉建设八路证...       1   1.79万    1.79万   1.79万   1.79万
291  292          长江证券股份有限公司成都东大街证券营业部       1   1.74万     0.00   1.74万    0.00
292  293           中国银河证券股份有限公司德清证券营业部       1   1.21万     0.00   1.21万    0.00
[293 rows x 7 columns]
```

###### 龙虎榜-营业部排行-抱团操作实力

接口: stock_lh_yyb_control

目标地址: https://data.10jqka.com.cn/market/longhu/

描述: 龙虎榜-营业部排行-抱团操作实力

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称        | 类型     | 描述  |
|-----------|--------|-----|
| 序号        | int64  | -   |
| 营业部名称     | object | -   |
| 携手营业部家数   | int64  | -   |
| 年内最佳携手对象  | object | -   |
| 年内最佳携手股票数 | int64  | -   |
| 年内最佳携手成功率 | object | -   |

接口示例

```python
import akshare as ak

stock_lh_yyb_control_df = ak.stock_lh_yyb_control()
print(stock_lh_yyb_control_df)
```

数据示例

```
      序号                         营业部名称  ...  年内最佳携手股票数 年内最佳携手成功率
0      1                   深股通专用  一线游资  ...        207    50.73%
1      2  东方财富证券股份有限公司拉萨金融城南环...  一线游资  ...        240    42.44%
2      3  东方财富证券股份有限公司拉萨团结路第二...  一线游资  ...        241    46.86%
3      4        东方财富证券股份有限公司拉萨团结路第一...  ...        241    46.86%
4      5  东方财富证券股份有限公司拉萨东环路第二...  一线游资  ...        241    46.03%
..   ...                           ...  ...        ...       ...
263  264         海通证券股份有限公司常熟海虞北路证券营业部  ...          1     无符合数据
264  265          兴业证券股份有限公司福州湖东路证券营业部  ...          4     无符合数据
265  266         光大证券股份有限公司贵阳长岭北路证券营业部  ...          2     无符合数据
266  267        华泰证券股份有限公司深圳平安金融中心证...  ...          3     无符合数据
267  268               中信证券股份有限公司深圳分公司  ...          4     无符合数据
[268 rows x 6 columns]
```

##### 龙虎榜-每日详情

接口: stock_lhb_detail_daily_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vInvestConsult/kind/lhb/index.phtml

描述: 新浪财经-龙虎榜-每日详情

限量: 单次返回指定 date 的所有数据

输入参数

| 名称   | 类型  | 描述                   |
|------|-----|----------------------|
| date | str | date="20240222"; 交易日 |

输出参数

| 名称   | 类型      | 描述       |
|------|---------|----------|
| 序号   | int64   | -        |
| 股票代码 | object  | -        |
| 股票名称 | object  | -        |
| 收盘价  | float64 | 注意单位: 元  |
| 对应值  | float64 | 注意单位: %  |
| 成交量  | float64 | 注意单位: 万股 |
| 成交额  | float64 | 注意单位: 万元 |
| 指标   | object  | 注意单位: 万元 |

接口示例

```python
import akshare as ak

stock_lhb_detail_daily_sina_df = ak.stock_lhb_detail_daily_sina(date="20240222")
print(stock_lhb_detail_daily_sina_df)
```

数据示例

```
    序号    股票代码  ...          成交额                                                 指标
0    1  000005  ...   12350.1150           连续三个交易日内，涨幅偏离值累计达到12%的ST证券、*ST证券和未完成股改证券
1    2  000023  ...    5569.3887           连续三个交易日内，涨幅偏离值累计达到12%的ST证券、*ST证券和未完成股改证券
2    3  002087  ...    7799.9948           连续三个交易日内，涨幅偏离值累计达到12%的ST证券、*ST证券和未完成股改证券
3    4  002251  ...   17824.5730           连续三个交易日内，涨幅偏离值累计达到12%的ST证券、*ST证券和未完成股改证券
4    5  002309  ...     625.8694           连续三个交易日内，涨幅偏离值累计达到12%的ST证券、*ST证券和未完成股改证券
5    1  000017  ...  182261.0964                                         振幅值达15%的证券
6    2  002767  ...   19027.4927                                         振幅值达15%的证券
7    3  002779  ...   26323.3751                                         振幅值达15%的证券
8    4  002976  ...   21965.7357                                         振幅值达15%的证券
9    5  603958  ...   88089.9000                                         振幅值达15%的证券
10   6  600715  ...   30483.4000                                         振幅值达15%的证券
11   7  601599  ...   83517.0300                                         振幅值达15%的证券
12   1  000017  ...  182261.0964                                         换手率达20%的证券
13   2  000628  ...  338104.2782                                         换手率达20%的证券
14   3  001300  ...   26419.9729                                         换手率达20%的证券
15   4  001314  ...   55316.8666                                         换手率达20%的证券
16   5  001376  ...   22526.3696                                         换手率达20%的证券
17   6  603082  ...   57471.2900                                         换手率达20%的证券
18   7  600520  ...   96413.6700                                         换手率达20%的证券
19   8  601136  ...  254881.6100                                         换手率达20%的证券
20   9  600355  ...   43150.3900                                         换手率达20%的证券
21  10  603375  ...   21939.2800                                         换手率达20%的证券
22  11  837748  ...   17402.7500                                         换手率达20%的证券
23  12  871753  ...   32244.5500                                         换手率达20%的证券
24  13  873122  ...   29711.8900                                         换手率达20%的证券
25  14  870976  ...    9089.6900                                         换手率达20%的证券
26  15  837592  ...   11167.0400                                         换手率达20%的证券
27   1  000070  ...   87688.7663                                        涨幅偏离值达7%的证券
28   2  000536  ...   40405.8869                                        涨幅偏离值达7%的证券
29   3  000628  ...  338104.2782                                        涨幅偏离值达7%的证券
30   4  002657  ...   58771.5639                                        涨幅偏离值达7%的证券
31   5  002916  ...   48520.0120                                        涨幅偏离值达7%的证券
32   6  601996  ...   22519.3300                                        涨幅偏离值达7%的证券
33   7  600792  ...   30895.5200                                        涨幅偏离值达7%的证券
34   8  603528  ...   29088.9000                                        涨幅偏离值达7%的证券
35   9  603963  ...   10527.9800                                        涨幅偏离值达7%的证券
36  10  600602  ...   87019.3500                                        涨幅偏离值达7%的证券
37   1  000416  ...   15786.0704  连续三个交易日内，日均换手率与前五个交易日的日均换手率的比值达到30倍，且换手率累计达20%的股票
38   1  002217  ...   76899.5978                            连续三个交易日内，涨幅偏离值累计达20%的证券
39   2  002789  ...   10830.8319                            连续三个交易日内，涨幅偏离值累计达20%的证券
40   3  603363  ...   59315.0400                            连续三个交易日内，涨幅偏离值累计达20%的证券
41   1  300157  ...   30524.4363                                                NaN
42   2  300209  ...    1587.7636                                                NaN
43   3  300321  ...   13573.3626                                                NaN
44   1  300188  ...   48544.7464                                                NaN
45   2  300210  ...   19151.3472                                                NaN
46   3  300249  ...   34312.9720                                                NaN
47   4  301252  ...   13744.3625                                                NaN
48   5  301318  ...   50108.3350                                                NaN
49   1  300781  ...  153086.7566                                                NaN
50   2  300949  ...   26507.2743                                                NaN
51   3  301502  ...   25688.4905                                                NaN
52   4  301516  ...   60720.4662                                                NaN
53   5  301577  ...   24623.2346                                                NaN
54   6  688709  ...   58862.6200                                                NaN
[55 rows x 8 columns]
```

##### 龙虎榜-个股上榜统计

接口: stock_lhb_ggtj_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vLHBData/kind/ggtj/index.phtml

描述: 新浪财经-龙虎榜-个股上榜统计

限量: 单次返回指定 symbol 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                                |
|--------|-----|-----------------------------------------------------------------------------------|
| symbol | str | symbol="5"; choice of {"5": 最近 5 天; "10": 最近 10 天; "30": 最近 30 天; "60": 最近 60 天;} |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 股票代码  | object  | -       |
| 股票名称  | object  | -       |
| 上榜次数  | int64   | -       |
| 累积购买额 | float64 | 注意单位: 万 |
| 累积卖出额 | float64 | 注意单位: 万 |
| 净额    | float64 | 注意单位: 万 |
| 买入席位数 | int64   | -       |
| 卖出席位数 | int64   | -       |

接口示例

```python
import akshare as ak

stock_lhb_ggtj_sina_df = ak.stock_lhb_ggtj_sina(symbol="5")
print(stock_lhb_ggtj_sina_df)
```

数据示例

```
     股票代码 股票名称  上榜次数  累积购买额   累积卖出额   净额  买入席位数  卖出席位数
0    300364  中文在线     2  171289.25   97538.13   73751.12      9      9
1    000001  平安银行     1  126080.24   64719.50   61360.74      5      4
2    002673  西部证券     1   22404.44    8677.21   13727.22      5      4
3    601595  上海电影     1   26953.47   13537.77   13415.70      3      4
4    601136  首创证券     2   50349.55   37874.81   12474.74      9      7
..      ...   ...   ...        ...        ...        ...    ...    ...
310  001314  亿道信息     4   27524.67   46756.32  -19231.65     13     10
311  300502   新易盛     1  122834.08  142750.78  -19916.70      5      5
312  002229  鸿博股份     2  116669.40  137884.32  -21214.92     10      9
313  000628  高新发展     4  372969.77  421664.91  -48695.14     14     15
314  603259  药明康德     1   66436.38  248845.50 -182409.12      5      3
[315 rows x 8 columns]
```

##### 龙虎榜-营业上榜统计

接口: stock_lhb_yytj_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vLHBData/kind/yytj/index.phtml

描述: 新浪财经-龙虎榜-营业上榜统计

限量: 单次返回指定 symbol 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                                |
|--------|-----|-----------------------------------------------------------------------------------|
| symbol | str | symbol="5"; choice of {"5": 最近 5 天; "10": 最近 10 天; "30": 最近 30 天; "60": 最近 60 天;} |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 营业部名称  | object  | -       |
| 上榜次数   | int64   | -       |
| 累积购买额  | float64 | 注意单位: 万 |
| 买入席位数  | int64   | -       |
| 累积卖出额  | float64 | 注意单位: 万 |
| 卖出席位数  | int64   | -       |
| 买入前三股票 | object  | -       |

接口示例

```python
import akshare as ak

stock_lhb_yytj_sina_df = ak.stock_lhb_yytj_sina(symbol="5")
print(stock_lhb_yytj_sina_df)
```

数据示例

```
                营业部名称  上榜次数    累积购买额  ...    累积卖出额  卖出席位数           买入前三股票
0      渤海证券股份有限公司天津滨海新区第一分公司     1     0.00  ...   116.29      1             ST星源
1       平安证券股份有限公司深圳香蜜湖证券营业部     3   879.44  ...   522.60      1   科新发展,ST星源,中重科技
2       华泰证券股份有限公司姜堰东大街证券营业部     2     0.00  ...   165.14      2       ST星源,*ST西发
3       海通证券股份有限公司庆阳西大街证券营业部     2     0.00  ...   504.69      2             ST星源
4    东方财富证券股份有限公司山南香曲东路证券营业部    26  5593.35  ...  5962.82     17  亚世光电,中视传媒,光大嘉宝,
..                       ...   ...      ...  ...      ...    ...              ...
315          华泰证券股份有限公司湖南分公司     7  6542.40  ...  3553.71      2        哈森股份,开元教育
316  上海证券有限责任公司绍兴新昌人民西路证券营业部     1  1458.00  ...     0.00      0             昂立教育
317     华安证券股份有限公司合肥高新区证券营业部     5  1868.41  ...  1788.15      2   中科金财,昀冢科技,普路通,
318     兴业证券股份有限公司福州湖东路证券营业部     1  1047.27  ...     0.00      0              香飘飘
319  申万宏源证券有限公司上海徐汇区龙漕路证券营业部     1   678.26  ...     0.00      0            *ST碳元
[320 rows x 7 columns]
```

##### 龙虎榜-机构席位追踪

接口: stock_lhb_jgzz_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vLHBData/kind/jgzz/index.phtml

描述: 新浪财经-龙虎榜-机构席位追踪

限量: 单次返回指定 symbol 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                                |
|--------|-----|-----------------------------------------------------------------------------------|
| symbol | str | symbol="5"; choice of {"5": 最近 5 天; "10": 最近 10 天; "30": 最近 30 天; "60": 最近 60 天;} |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 股票代码  | object  | -       |
| 股票名称  | object  | -       |
| 累积买入额 | float64 | 注意单位: 万 |
| 买入次数  | float64 | -       |
| 累积卖出额 | float64 | 注意单位: 万 |
| 卖出次数  | float64 | -       |
| 净额    | float64 | 注意单位: 万 |

接口示例

```python
import akshare as ak

stock_lhb_jgzz_sina_df = ak.stock_lhb_jgzz_sina(symbol="5")
print(stock_lhb_jgzz_sina_df)
```

数据示例

```
     股票代码  股票名称  累积买入额  买入次数  累积卖出额  卖出次数  净额
0    688498  源杰科技  19589.28     6   3035.09     2  16554.19
1    300308  中际旭创  19269.73     1   9088.94     1  10180.79
2    603135  中重科技   5824.15     1      0.00     0   5824.15
3    003032  传智教育   5388.51     4    118.46     0   5270.05
4    688601   力芯微  10427.99     6   5463.37     4   4964.62
..      ...   ...       ...   ...       ...   ...       ...
192  000628  高新发展   8635.84     2  21545.41     3 -12909.57
193  300063  天龙集团   7426.88     1  20736.26     5 -13309.38
194  002463  沪电股份   4700.87     1  18997.89     3 -14297.02
195  300781  因赛集团  13758.05     2  39308.41     8 -25550.36
196  603259  药明康德      0.00     0  54874.30     3 -54874.30
[197 rows x 7 columns]
```

##### 龙虎榜-机构席位成交明细

接口: stock_lhb_jgmx_sina

目标地址: https://vip.stock.finance.sina.com.cn/q/go.php/vLHBData/kind/jgzz/index.phtml

描述: 新浪财经-龙虎榜-机构席位成交明细

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 股票代码    | object  | -       |
| 股票名称    | object  | -       |
| 交易日期    | object  | -       |
| 机构席位买入额 | float64 | 注意单位: 万 |
| 机构席位卖出额 | float64 | 注意单位: 万 |
| 类型      | object  | -       |

接口示例

```python
import akshare as ak

stock_lhb_jgmx_sina_df = ak.stock_lhb_jgmx_sina()
print(stock_lhb_jgmx_sina_df)
```

数据示例

```
    股票代码  股票名称    交易日期  机构席位买入额(万)  机构席位卖出额(万)   类型
0    001300   三柏硕  2024-02-23     2231.89     5214.88   换手率达20%的证券
1    301378   通达海  2024-02-23      249.88     2318.76          NaN
2    300781  因赛集团  2024-02-23     7237.09    20345.79          NaN
3    300486  东杰智能  2024-02-23     3921.15     7344.49          NaN
4    300078  思创医惠  2024-02-23     2423.62     4956.84          NaN
..      ...   ...         ...         ...         ...          ...
235  301516   中远通  2024-02-19     1606.70       78.07          NaN
236  300624  万兴科技  2024-02-19        0.00    10913.05          NaN
237  002281  光迅科技  2024-02-19      224.76     2618.77  涨幅偏离值达7%的证券
238  300067   安诺其  2024-02-19     1329.52     2784.87          NaN
239  002621   美吉姆  2024-02-19      833.72     1391.98   振幅值达15%的证券
[240 rows x 6 columns]
```

#### 首发申报信息

接口: stock_ipo_declare_em

目标地址: https://data.eastmoney.com/xg/xg/sbqy.html

描述: 东方财富网-数据中心-新股申购-首发申报信息-首发申报企业信息

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述 |
|--------|--------|----|
| 序号     | int64  | -  |
| 企业名称   | object | -  |
| 最新状态   | object | -  |
| 注册地    | object | -  |
| 保荐机构   | object | -  |
| 律师事务所  | object | -  |
| 会计师事务所 | object | -  |
| 拟上市地点  | object | -  |
| 更新日期   | object | -  |
| 招股说明书  | object | -  |

接口示例

```python
import akshare as ak

stock_ipo_declare_em_df = ak.stock_ipo_declare_em()
print(stock_ipo_declare_em_df)
```

数据示例

```
      序号     申报企业 拟上市地  ...               会计师事务所        律师事务所      备注
0      1     奥精医疗科技股份有限公司  科创板  ...     立信会计师事务所(特殊普通合伙)  北京观韬中茂律师事务所  进一步问询中
1      2       爱威科技股份有限公司  科创板  ...     天健会计师事务所(特殊普通合伙)    湖南启元律师事务所  进一步问询中
2      3   北京弘成立业科技股份有限公司  创业板  ...     大华会计师事务所(特殊普通合伙)   北京市君合律师事务所  进一步问询中
3      4   北京海天瑞声科技股份有限公司  科创板  ...  毕马威华振会计师事务所(特殊普通合伙)   北京市天元律师事务所  进一步问询中
4      5  北京挖金客信息科技股份有限公司  创业板  ...     容诚会计师事务所(特殊普通合伙)  国浩律师(北京)事务所  进一步问询中
..   ...              ...  ...  ...                  ...          ...     ...
106  107   浙江嘉益保温科技股份有限公司  创业板  ...     天健会计师事务所(特殊普通合伙)   北京市金杜律师事务所  进一步问询中
107  108   浙江日发纺织机械股份有限公司  创业板  ...     天健会计师事务所(特殊普通合伙)    浙江天册律师事务所  进一步问询中
108  109     浙江泰福泵业股份有限公司  创业板  ...     天健会计师事务所(特殊普通合伙)  国浩律师(杭州)事务所  进一步问询中
109  110     中兰环保科技股份有限公司  创业板  ...     立信会计师事务所(特殊普通合伙)    北京国枫律师事务所  进一步问询中
110  111   中联云港数据科技股份有限公司  创业板  ...   天职国际会计师事务所(特殊普通合伙)  国浩律师(北京)事务所  进一步问询中
```

#### IPO审核信息

##### 全部

接口: stock_register_all_em

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-全部

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述 |
|--------|--------|----|
| 序号     | int64  | -  |
| 企业名称   | object | -  |
| 最新状态   | object | -  |
| 注册地    | object | -  |
| 行业     | object | -  |
| 保荐机构   | object | -  |
| 律师事务所  | object | -  |
| 会计师事务所 | object | -  |
| 更新日期   | object | -  |
| 受理日期   | object | -  |
| 拟上市地点  | object | -  |
| 招股说明书  | object | -  |

接口示例

```python
import akshare as ak

stock_register_all_em_df = ak.stock_register_all_em()
print(stock_register_all_em_df)
```

数据示例

```
       序号  ...                                    招股说明书
0        1  ...  https://pdf.dfcfw.com/pdf/H2_AN202512251807880...
1        2  ...  https://pdf.dfcfw.com/pdf/H2_AN202506301700470...
2        3  ...  https://pdf.dfcfw.com/pdf/H2_AN202512241807247...
3        4  ...  https://pdf.dfcfw.com/pdf/H2_AN202512241807270...
4        5  ...  https://pdf.dfcfw.com/pdf/H2_AN202512241807300...
...    ...  ...                                                ...
4145  4146  ...  https://pdf.dfcfw.com/pdf/H2_AN201609240017728...
4146  4147  ...  https://pdf.dfcfw.com/pdf/H2_AN201607140016578...
4147  4148  ...  https://pdf.dfcfw.com/pdf/H2_AN201611160102988...
4148  4149  ...  https://pdf.dfcfw.com/pdf/H2_AN201410300007493...
4149  4150  ...  https://pdf.dfcfw.com/pdf/H2_AN201611080074226...
[4150 rows x 12 columns]
```

##### 科创板

接口: stock_register_kcb

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-科创板

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int64  | -   |
| 发行人全称  | object | -   |
| 审核状态   | object | -   |
| 注册地    | object | -   |
| 证监会行业  | object | -   |
| 保荐机构   | object | -   |
| 律师事务所  | object | -   |
| 会计师事务所 | object | -   |
| 更新日期   | object | -   |
| 受理日期   | object | -   |
| 拟上市地点  | object | -   |
| 招股说明书  | object | -   |

接口示例

```python
import akshare as ak

stock_register_kcb_df = ak.stock_register_kcb()
print(stock_register_kcb_df)
```

数据示例

```
       序号              企业名称     最新状态 注册地                行业          保荐机构  \
0      1    联芸科技(杭州)股份有限公司     注册生效  浙江  计算机、通信和其他电子设备制造业  中信建投证券股份有限公司
1      2  赛诺威盛科技(北京)股份有限公司       终止  北京           专用设备制造业    国金证券股份有限公司
2      3   青岛信芯微电子科技股份有限公司       终止  山东        软件和信息技术服务业  中国国际金融股份有限公司
3      4   北京凯普林光电科技股份有限公司  上市委会议通过  北京  计算机、通信和其他电子设备制造业  国泰君安证券股份有限公司
4      5    深圳垦拓流体技术股份有限公司       终止  广东           通用设备制造业    中信证券股份有限公司
..   ...               ...      ...  ..               ...           ...
866  867        澜起科技股份有限公司     注册生效  上海  计算机、通信和其他电子设备制造业    中信证券股份有限公司
867  868      浙江杭可科技股份有限公司     注册生效  浙江           专用设备制造业    国信证券股份有限公司
868  869      苏州天准科技股份有限公司     注册生效  江苏           专用设备制造业    海通证券股份有限公司
869  870    苏州华兴源创科技股份有限公司     注册生效  江苏           专用设备制造业  华泰联合证券有限责任公司
870  871    烟台睿创微纳技术股份有限公司     注册生效  山东  计算机、通信和其他电子设备制造业    中信证券股份有限公司
           律师事务所              会计师事务所        更新日期        受理日期 拟上市地点  \
0     北京市君合律师事务所  德勤华永会计师事务所(特殊普通合伙)  2024-06-14  2022-12-28   科创板
1     北京市中伦律师事务所    大信会计师事务所(特殊普通合伙)  2024-06-14  2023-03-27   科创板
2     北京市君合律师事务所    天健会计师事务所(特殊普通合伙)  2024-06-14  2023-06-12   科创板
3     北京市天元律师事务所    天健会计师事务所(特殊普通合伙)  2024-06-14  2023-05-09   科创板
4     北京市中伦律师事务所    致同会计师事务所(特殊普通合伙)  2024-06-13  2023-12-27   科创板
..           ...                 ...         ...         ...   ...
866  国浩律师(上海)事务所    瑞华会计师事务所(特殊普通合伙)  2019-06-26  2019-04-01   科创板
867   北京市金杜律师事务所    天健会计师事务所(特殊普通合伙)  2019-06-24  2019-04-15   科创板
868    浙江六和律师事务所    瑞华会计师事务所(特殊普通合伙)  2019-06-20  2019-04-02   科创板
869   上海市通力律师事务所  华普天健会计师事务所(特殊普通合伙)  2019-06-18  2019-03-27   科创板
870   北京市中银律师事务所  信永中和会计师事务所(特殊普通合伙)  2019-06-18  2019-03-22   科创板
                                                 招股说明书
0    https://pdf.dfcfw.com/pdf/H2_AN202302171583296...
1    https://pdf.dfcfw.com/pdf/H2_AN202303271584589...
2    https://pdf.dfcfw.com/pdf/H2_AN202306121590862...
3    https://pdf.dfcfw.com/pdf/H2_AN202305091586404...
4    https://pdf.dfcfw.com/pdf/H2_AN202312271615032...
..                                                 ...
866  https://pdf.dfcfw.com/pdf/H2_AN201904011312907...
867  https://pdf.dfcfw.com/pdf/H2_AN201904151319351...
868  https://pdf.dfcfw.com/pdf/H2_AN201904021313966...
869  https://pdf.dfcfw.com/pdf/H2_AN201903271310281...
870  https://pdf.dfcfw.com/pdf/H2_AN201903221308711...
[871 rows x 12 columns]
```

###### 创业板

接口: stock_register_cyb

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-创业板

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int32  | -   |
| 发行人全称  | object | -   |
| 审核状态   | object | -   |
| 注册地    | object | -   |
| 证监会行业  | object | -   |
| 保荐机构   | object | -   |
| 律师事务所  | object | -   |
| 会计师事务所 | object | -   |
| 更新日期   | object | -   |
| 受理日期   | object | -   |
| 拟上市地点  | object | -   |
| 招股说明书  | object | -   |

接口示例

```python
import akshare as ak

stock_register_cyb_df = ak.stock_register_cyb()
print(stock_register_cyb_df)
```

数据示例

```
       序号               企业名称  最新状态 注册地          行业              保荐机构  \
0        1   北京易诚互动网络技术股份有限公司    撤回  北京  软件和信息技术服务业        东北证券股份有限公司
1        2     基因科技(上海)股份有限公司    撤回  上海  科技推广和应用服务业        广发证券股份有限公司
2        3         黄山谷捷股份有限公司   已问询  安徽       汽车制造业        国元证券股份有限公司
3        4       迪嘉药业集团股份有限公司    撤回  山东       医药制造业        民生证券股份有限公司
4        5   中航上大高温合金材料股份有限公司  提交注册  河北       金属制品业      中国国际金融股份有限公司
...    ...                ...   ...  ..         ...               ...
1122  1123    宁波卡倍亿电气技术股份有限公司  注册生效  浙江       汽车制造业        东莞证券股份有限公司
1123  1124      杨凌美畅新材料股份有限公司  注册生效  陕西    非金属矿物制品业      中信建投证券股份有限公司
1124  1125   北京锋尚世纪文化传媒股份有限公司  注册生效  北京       文化艺术业      中信建投证券股份有限公司
1125  1126  康泰医学系统(秦皇岛)股份有限公司  注册生效  河北     专用设备制造业  申万宏源证券承销保荐有限责任公司
1126  1127      安徽蓝盾光电子股份有限公司  注册生效  安徽     仪器仪表制造业        华龙证券股份有限公司
            律师事务所              会计师事务所        更新日期        受理日期 拟上市地点  \
0       北京德恒律师事务所  信永中和会计师事务所(特殊普通合伙)  2024-06-14  2022-06-27   创业板
1      上海市广发律师事务所    容诚会计师事务所(特殊普通合伙)  2024-06-13  2023-06-27   创业板
2       安徽天禾律师事务所  中审众环会计师事务所(特殊普通合伙)  2024-06-12  2023-05-08   创业板
3       上海泽昌律师事务所    天健会计师事务所(特殊普通合伙)  2024-06-12  2023-06-16   创业板
4      北京市嘉源律师事务所    立信会计师事务所(特殊普通合伙)  2024-06-11  2022-06-29   创业板
...           ...                 ...         ...         ...   ...
1122  上海市锦天城律师事务所    立信会计师事务所(特殊普通合伙)  2020-07-29  2020-06-22   创业板
1123    北京国枫律师事务所  立信中联会计师事务所(特殊普通合伙)  2020-07-24  2020-06-24   创业板
1124   北京市中伦律师事务所  信永中和会计师事务所(特殊普通合伙)  2020-07-24  2020-06-22   创业板
1125   北京市中伦律师事务所  德勤华永会计师事务所(特殊普通合伙)  2020-07-24  2020-07-02   创业板
1126    安徽承义律师事务所    容诚会计师事务所(特殊普通合伙)  2020-07-24  2020-06-30   创业板
                                                  招股说明书
0     https://pdf.dfcfw.com/pdf/H2_AN202401251618103...
1     https://pdf.dfcfw.com/pdf/H2_AN202312291615486...
2     https://pdf.dfcfw.com/pdf/H2_AN202406121636034...
3     https://pdf.dfcfw.com/pdf/H2_AN202311161611034...
4     https://pdf.dfcfw.com/pdf/H2_AN202406111635973...
...                                                 ...
1122  https://pdf.dfcfw.com/pdf/H2_AN202008121398114...
1123  https://pdf.dfcfw.com/pdf/H2_AN202008091397452...
1124  https://pdf.dfcfw.com/pdf/H2_AN202008091397452...
1125  https://pdf.dfcfw.com/pdf/H2_AN202008131398337...
1126  https://pdf.dfcfw.com/pdf/H2_AN202008161398984...
[1127 rows x 12 columns]
```

###### 上海主板

接口: stock_register_sh

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-上海主板

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int32  | -   |
| 发行人全称  | object | -   |
| 审核状态   | object | -   |
| 注册地    | object | -   |
| 证监会行业  | object | -   |
| 保荐机构   | object | -   |
| 律师事务所  | object | -   |
| 会计师事务所 | object | -   |
| 更新日期   | object | -   |
| 受理日期   | object | -   |
| 拟上市地点  | object | -   |
| 招股说明书  | object | -   |

接口示例

```python
import akshare as ak

stock_register_sh_df = ak.stock_register_sh()
print(stock_register_sh_df)
```

数据示例

```
      序号              企业名称       最新状态 注册地          行业              保荐机构  \
0      1      浙江亘古电缆股份有限公司         终止  浙江  电气机械和器材制造业        招商证券股份有限公司
1      2  上海生生医药冷链科技股份有限公司         终止  上海  多式联运和运输代理业      中国国际金融股份有限公司
2      3    新疆凯龙清洁能源股份有限公司         终止  新疆  开采专业及辅助性活动        东兴证券股份有限公司
3      4      广州三晶电气股份有限公司  已收到注册申请材料  广东        None        民生证券股份有限公司
4      5      合肥汇通控股股份有限公司  已收到注册申请材料  安徽        None      中银国际证券股份有限公司
..   ...               ...        ...  ..         ...               ...
874  875             上海数据港     已通过发审会  上海        None      中信建投证券股份有限公司
875  876            镇海石化工程     已通过发审会  浙江        None        浙商证券股份有限公司
876  877          浙江德创环保科技     已通过发审会  浙江        None        民生证券股份有限公司
877  878           重庆建工集团*     已通过发审会  重庆        None        华融证券股份有限公司
878  879            大连百傲化学     已通过发审会  辽宁        None  申万宏源证券承销保荐有限责任公司
           律师事务所              会计师事务所        更新日期        受理日期 拟上市地点  \
0      北京国枫律师事务所    天健会计师事务所(特殊普通合伙)  2024-06-17  2023-03-03   沪主板
1    上海市锦天城律师事务所    立信会计师事务所(特殊普通合伙)  2024-06-17  2023-06-30   沪主板
2     北京市中伦律师事务所  信永中和会计师事务所(特殊普通合伙)  2024-06-14  2023-06-15   沪主板
3      国信信扬律师事务所    立信会计师事务所(特殊普通合伙)  2024-06-13         NaT   沪主板
4      安徽天禾律师事务所    容诚会计师事务所(特殊普通合伙)  2024-06-13         NaT   沪主板
..           ...                 ...         ...         ...   ...
874  上海市锦天城律师事务所    立信会计师事务所(特殊普通合伙)  2017-01-05         NaT   沪主板
875   北京市天元律师事务所    天健会计师事务所(特殊普通合伙)  2017-01-05         NaT   沪主板
876   北京市天元律师事务所    天健会计师事务所(特殊普通合伙)  2017-01-05         NaT   沪主板
877   北京市金杜律师事务所    大信会计师事务所(特殊普通合伙)  2017-01-05         NaT   沪主板
878    北京德恒律师事务所    瑞华会计师事务所(特殊普通合伙)  2017-01-05         NaT   沪主板
                                                 招股说明书
0    https://pdf.dfcfw.com/pdf/H2_AN202303031584027...
1    https://pdf.dfcfw.com/pdf/H2_AN202306301591956...
2    https://pdf.dfcfw.com/pdf/H2_AN202306151590972...
3    https://pdf.dfcfw.com/pdf/H2_AN202303031584027...
4    https://pdf.dfcfw.com/pdf/H2_AN202303031584027...
..                                                 ...
874  https://pdf.dfcfw.com/pdf/H2_AN201701150271855...
875  https://pdf.dfcfw.com/pdf/H2_AN201701180277302...
876  https://pdf.dfcfw.com/pdf/H2_AN201701170274362...
877  https://pdf.dfcfw.com/pdf/H2_AN201701170274362...
878  https://pdf.dfcfw.com/pdf/H2_AN201701160272431...
[879 rows x 12 columns]
```

###### 深圳主板

接口: stock_register_sz

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-深圳主板

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int32  | -   |
| 发行人全称  | object | -   |
| 审核状态   | object | -   |
| 注册地    | object | -   |
| 证监会行业  | object | -   |
| 保荐机构   | object | -   |
| 律师事务所  | object | -   |
| 会计师事务所 | object | -   |
| 更新日期   | object | -   |
| 受理日期   | object | -   |
| 拟上市地点  | object | -   |
| 招股说明书  | object | -   |

接口示例

```python
import akshare as ak

stock_register_sz_df = ak.stock_register_sz()
print(stock_register_sz_df)
```

数据示例

```
     序号             企业名称    最新状态 注册地            行业          保荐机构  \
0      1    山东金鸿新材料股份有限公司      撤回  山东      非金属矿物制品业    中泰证券股份有限公司
1      2    金永和精工制造股份有限公司      撤回  山东         汽车制造业    中德证券有限责任公司
2      3     江苏大艺科技股份有限公司      撤回  江苏       通用设备制造业  中信建投证券股份有限公司
3      4   浙江新纳材料科技股份有限公司      撤回  浙江  化学原料及化学制品制造业    中信证券股份有限公司
4      5  深圳宏业基岩土科技股份有限公司      中止  广东          None    招商证券股份有限公司
..   ...              ...     ...  ..           ...           ...
512  513         深圳市科达利实业  已通过发审会  广东          None  中国国际金融股份有限公司
513  514         浙江威星智能仪表  已通过发审会  浙江          None    东吴证券股份有限公司
514  515         高斯贝尔数码科技  已通过发审会  湖南          None    招商证券股份有限公司
515  516           盐津铺子食品  已通过发审会  湖南          None    西部证券股份有限公司
516  517           广东英联包装  已通过发审会  广东          None    国海证券股份有限公司
           律师事务所              会计师事务所        更新日期        受理日期 拟上市地点  \
0     北京市汉坤律师事务所    永拓会计师事务所(特殊普通合伙)  2024-06-17  2023-12-28   深主板
1      北京大成律师事务所  信永中和会计师事务所(特殊普通合伙)  2024-06-16  2023-06-30   深主板
2    国浩律师(上海)事务所    天健会计师事务所(特殊普通合伙)  2024-06-15  2023-03-02   深主板
3     北京市康达律师事务所    天健会计师事务所(特殊普通合伙)  2024-06-14  2023-03-01   深主板
4      广东华商律师事务所    大华会计师事务所(特殊普通合伙)  2024-06-13         NaT   深主板
..           ...                 ...         ...         ...   ...
512    广东信达律师事务所    瑞华会计师事务所(特殊普通合伙)  2017-02-09         NaT   深主板
513  国浩律师(杭州)事务所    天健会计师事务所(特殊普通合伙)  2017-01-19         NaT   深主板
514    湖南启元律师事务所    天健会计师事务所(特殊普通合伙)  2017-01-12         NaT   深主板
515    湖南启元律师事务所    天健会计师事务所(特殊普通合伙)  2017-01-05         NaT   深主板
516    北京国枫律师事务所    立信会计师事务所(特殊普通合伙)  2017-01-05         NaT   深主板
                                                 招股说明书
0    https://pdf.dfcfw.com/pdf/H2_AN202312281615219...
1    https://pdf.dfcfw.com/pdf/H2_AN202312261614904...
2    https://pdf.dfcfw.com/pdf/H2_AN202312271615081...
3    https://pdf.dfcfw.com/pdf/H2_AN202312291615534...
4    https://pdf.dfcfw.com/pdf/H2_AN202307061592065...
..                                                 ...
512  https://pdf.dfcfw.com/pdf/H2_AN201702190353685...
513  https://pdf.dfcfw.com/pdf/H2_AN201702060321215...
514  https://pdf.dfcfw.com/pdf/H2_AN201701250290802...
515  https://pdf.dfcfw.com/pdf/H2_AN201701180274398...
516  https://pdf.dfcfw.com/pdf/H2_AN201701180274398...
[517 rows x 12 columns]
```

###### 北交所

接口: stock_register_bj

目标地址: https://data.eastmoney.com/xg/ipo/

描述: 东方财富网-数据中心-新股数据-IPO审核信息-北交所

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述  |
|--------|--------|-----|
| 序号     | int32  | -   |
| 发行人全称  | object | -   |
| 审核状态   | object | -   |
| 注册地    | object | -   |
| 证监会行业  | object | -   |
| 保荐机构   | object | -   |
| 律师事务所  | object | -   |
| 会计师事务所 | object | -   |
| 更新日期   | object | -   |
| 受理日期   | object | -   |
| 拟上市地点  | object | -   |
| 招股说明书  | object | -   |

接口示例

```python
import akshare as ak

stock_register_bj_df = ak.stock_register_bj()
print(stock_register_bj_df)
```

数据示例

```
      序号              企业名称 最新状态      注册地                行业        保荐机构  \
0      1   浙江太湖远大新材料股份有限公司  已问询  浙江省 湖州市          橡胶和塑料制品业  招商证券股份有限公司
1      2      大连瑞克科技股份有限公司  已问询  辽宁省 大连市      化学原料和化学制品制造业  国融证券股份有限公司
2      3    江苏永创医药科技股份有限公司   终止  江苏省 淮安市      化学原料和化学制品制造业  开源证券股份有限公司
3      4    成都成电光信科技股份有限公司  已问询  四川省 成都市        软件和信息技术服务业  广发证券股份有限公司
4      5    浙江圣兆药物科技股份有限公司   终止  浙江省 杭州市             医药制造业  东吴证券股份有限公司
..   ...               ...  ...      ...               ...         ...
519  520      上海艾融软件股份有限公司   核准  上海市 崇明县        软件和信息技术服务业  光大证券股份有限公司
520  521  北京颖泰嘉和生物科技股份有限公司   核准  北京市 昌平区      化学原料和化学制品制造业  西南证券股份有限公司
521  522  晨越建设项目管理集团股份有限公司   终止  四川省 成都市           专业技术服务业  华福证券有限责任公司
522  523        东亚装饰股份有限公司   终止  山东省 青岛市        建筑装饰和其他建筑业  红塔证券股份有限公司
523  524      北京蓝山科技股份有限公司   终止  北京市 海淀区  计算机、通信和其他电子设备制造业  华龙证券股份有限公司
           律师事务所                  会计师事务所        更新日期        受理日期 拟上市地点  \
0      北京大成律师事务所      公证天业会计师事务所(特殊普通合伙)  2024-06-13  2023-11-30   北交所
1    北京海润天睿律师事务所        和信会计师事务所(特殊普通合伙)  2024-06-11  2023-12-28   北交所
2      北京大成律师事务所      苏亚金诚会计师事务所(特殊普通合伙)  2024-06-07  2023-09-28   北交所
3      北京国枫律师事务所      信永中和会计师事务所(特殊普通合伙)  2024-06-07  2023-09-27   北交所
4     上海市联合律师事务所      立信中联会计师事务所(特殊普通合伙)  2024-06-06  2023-06-30   北交所
..           ...                     ...         ...         ...   ...
519   上海天衍禾律师事务所        天健会计师事务所(特殊普通合伙)  2020-06-12  2020-04-29   北交所
520   北京市康达律师事务所  四川华信(集团)会计师事务所(特殊普通合伙)  2020-06-12  2020-04-28   北交所
521   北京市中伦律师事务所      信永中和会计师事务所(特殊普通合伙)  2020-06-10  2020-04-28   北交所
522   北京市中银律师事务所        立信会计师事务所(特殊普通合伙)  2020-05-26  2020-05-19   北交所
523   北京市天元律师事务所     中兴财光华会计师事务所(特殊普通合伙)  2020-05-13  2020-04-29   北交所
                                                 招股说明书
0    https://pdf.dfcfw.com/pdf/H2_AN202406131636105...
1    https://pdf.dfcfw.com/pdf/H2_AN202312281615150...
2    https://pdf.dfcfw.com/pdf/H2_AN202309281600193...
3    https://pdf.dfcfw.com/pdf/H2_AN202406071635887...
4    https://pdf.dfcfw.com/pdf/H2_AN202306301591947...
..                                                 ...
519  https://pdf.dfcfw.com/pdf/H2_AN202007061389895...
520  https://pdf.dfcfw.com/pdf/H2_AN202007061389821...
521  https://pdf.dfcfw.com/pdf/H2_AN201507130010192...
522  https://pdf.dfcfw.com/pdf/H2_AN201401220005024...
523  https://pdf.dfcfw.com/pdf/H2_AN201406160006074...
[524 rows x 12 columns]
```

##### 达标企业

接口: stock_register_db

目标地址: https://data.eastmoney.com/xg/cyb/

描述: 东方财富网-数据中心-新股数据-注册制审核-达标企业

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称           | 类型      | 描述      |
|--------------|---------|---------|
| 序号           | int32   | -       |
| 企业名称         | object  | -       |
| 经营范围         | object  | -       |
| 近三年营业收入-2019 | float64 | 注意单位: 元 |
| 近三年净利润-2019  | float64 | 注意单位: 元 |
| 近三年研发费用-2019 | object  | 注意单位: 元 |
| 近三年营业收入-2018 | float64 | 注意单位: 元 |
| 近三年净利润-2018  | float64 | 注意单位: 元 |
| 近三年研发费用-2018 | object  | 注意单位: 元 |
| 近三年营业收入-2017 | object  | 注意单位: 元 |
| 近三年净利润-2017  | object  | 注意单位: 元 |
| 近三年研发费用-2017 | object  | 注意单位: 元 |
| 近两年累计净利润     | float64 | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_register_db_df = ak.stock_register_db()
print(stock_register_db_df)
```

数据示例

```
        序号              企业名称  ... 近三年研发费用-2017      近两年累计净利润
0        1   上海宏力达信息技术股份有限公司  ...  19932141.20  3.360614e+08
1        2   上海阿拉丁生化科技股份有限公司  ...   8962209.85  1.171064e+08
2        3  河北华友文化遗产保护股份有限公司  ...   2203545.49  5.992881e+07
3        4    大连豪森设备制造股份有限公司  ...  38364081.96  6.224462e+07
4        5            九号有限公司  ...  91330278.85 -2.258809e+09
    ...               ...  ...          ...           ...
1713  1714  武汉同济现代医药科技股份有限公司  ...   2796051.27  2.762509e+07
1714  1715      上海雷腾软件股份有限公司  ...   7793124.89  1.014579e+08
1715  1716    上海银音信息科技股份有限公司  ...  12518776.19  1.107391e+07
1716  1717    上海致远绿色能源股份有限公司  ...  18035201.48  3.533665e+07
1717  1718    上海四维文化传媒股份有限公司  ...  16961496.21  2.896218e+07
```

#### 增发

接口: stock_qbzf_em

目标地址: https://data.eastmoney.com/other/gkzf.html

描述: 东方财富网-数据中心-新股数据-增发-全部增发

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 增发代码   | object  | -       |
| 发行方式   | object  | -       |
| 发行总数   | int64   | 注意单位: 股 |
| 网上发行   | object  | 注意单位: 股 |
| 发行价格   | float64 | -       |
| 最新价    | float64 | -       |
| 发行日期   | object  | -       |
| 增发上市日期 | object  | -       |
| 锁定期    | object  | -       |

接口示例

```python
import akshare as ak

stock_qbzf_em_df = ak.stock_qbzf_em()
print(stock_qbzf_em_df)
```

数据示例

```
      股票代码  股票简称 增发代码  发行方式  ...  最新价    发行日期  增发上市日期  锁定期
0     872931  无锡鼎邦  None  公开增发  ...    NaN  2024-03-27  2024-04-08  0.5年
1     837023  芭薇股份  None  公开增发  ...  13.10  2024-03-20  2024-03-29  0.5年
2     600773  西藏城投  None  定向增发  ...  12.21  2024-03-15  2024-04-02  0.5年
3     600038  中直股份  None  定向增发  ...  39.27  2024-03-14  2024-03-19    3年
4     600028  中国石化  None  定向增发  ...   6.50  2024-03-13  2024-03-18    3年
      ...   ...   ...   ...  ...    ...         ...         ...   ...
5656  601880  辽港股份  None  定向增发  ...   1.42         NaT  2021-02-09  None
5657  600056  中国医药  None  定向增发  ...  11.03         NaT  2013-08-01  None
5658  600372  中航机载  None  定向增发  ...  11.57         NaT  2023-04-19  None
5659  600827  百联股份  None  定向增发  ...   9.09         NaT  2011-08-31  None
5660  600332   白云山  None  定向增发  ...  29.13         NaT  2013-05-23  None
[5661 rows x 11 columns]
```

#### 配股

接口: stock_pg_em

目标地址: https://data.eastmoney.com/xg/pg/

描述: 东方财富网-数据中心-新股数据-配股

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 配售代码   | object  | -       |
| 配股数量   | int64   | 注意单位: 股 |
| 配股比例   | object  | -       |
| 配股价    | float64 | -       |
| 最新价    | float64 | -       |
| 配股前总股本 | int64   | 注意单位: 股 |
| 配股后总股本 | int64   | 注意单位: 股 |
| 股权登记日  | object  | -       |
| 缴款起始日期 | object  | -       |
| 缴款截止日期 | object  | -       |
| 上市日    | object  | -       |

接口示例

```python
import akshare as ak

stock_pg_em_df = ak.stock_pg_em()
print(stock_pg_em_df)
```

数据示例

```
     股票代码  股票简称   配售代码  ... 缴款起始日期   缴款截止日期    上市日
0     000049  德赛电池  080049  ...  2023-11-30  2023-12-06  2023-12-25
1     600081  东风科技  700081  ...  2023-08-02  2023-08-08  2023-08-24
2     601916  浙商银行  760916  ...  2023-06-15  2023-06-21  2023-07-06
3     300475  香农芯创  380475  ...  2023-02-08  2023-02-14  2023-03-03
4     600459  贵研铂业  700459  ...  2022-12-14  2022-12-20  2023-01-04
      ...   ...     ...  ...         ...         ...         ...
1017  600651  飞乐音响    None  ...         NaT         NaT  1994-04-25
1018  600653  申华控股    None  ...         NaT         NaT  1993-06-10
1019  600609  金杯汽车    None  ...         NaT         NaT  1993-01-29
1020  600601  方正科技    None  ...         NaT         NaT  1992-06-10
1021  000002   万科A    None  ...  1991-06-01  1991-06-08         NaT
[1022 rows x 13 columns]
```

#### 股票回购数据

接口: stock_repurchase_em

目标地址: https://data.eastmoney.com/gphg/hglist.html

描述: 东方财富网-数据中心-股票回购-股票回购数据

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称             | 类型      | 描述      |
|----------------|---------|---------|
| 序号             | int64   | -       |
| 股票代码           | object  | -       |
| 股票简称           | object  | -       |
| 最新价            | float64 | -       |
| 计划回购价格区间       | float64 | 注意单位: 元 |
| 计划回购数量区间-下限    | float64 | 注意单位: 股 |
| 计划回购数量区间-上限    | float64 | 注意单位: 股 |
| 占公告前一日总股本比例-下限 | float64 | 注意单位: % |
| 占公告前一日总股本比例-上限 | float64 | 注意单位: % |
| 计划回购金额区间-下限    | float64 | 注意单位: 元 |
| 计划回购金额区间-上限    | float64 | 注意单位: 元 |
| 回购起始时间         | object  | -       |
| 实施进度           | object  | -       |
| 已回购股份价格区间-下限   | float64 | 注意单位: % |
| 已回购股份价格区间-上限   | float64 | 注意单位: % |
| 已回购股份数量        | float64 | 注意单位: 股 |
| 已回购金额          | float64 | 注意单位: 元 |
| 最新公告日期         | object  | -       |

接口示例

```python
import akshare as ak

stock_repurchase_em_df = ak.stock_repurchase_em()
print(stock_repurchase_em_df)
```

数据示例

```
        序号 股票代码  股票简称  ...  已回购股份数量    已回购金额      最新公告日期
0        1  600028  中国石化  ...         NaN           NaN  2022-08-29
1        2  301021  英诺激光  ...      6500.0  1.625000e+05  2022-08-29
2        3  688185   康希诺  ...    500000.0  1.137614e+08  2022-08-29
3        4  603700  宁水集团  ...   2291336.0  4.005009e+07  2021-09-15
4        5  000338  潍柴动力  ...  55950000.0  6.822452e+08  2022-08-27
    ...     ...   ...  ...         ...           ...         ...
1977  1978  002048  宁波华翔  ...  23152838.0  1.598245e+08  2012-08-16
1978  1979  002032   苏泊尔  ...   2750000.0  3.467148e+07  2012-07-18
1979  1980  002054  德美化工  ...   5484773.0  5.473750e+07  2012-01-05
1980  1981  002048  宁波华翔  ...  13940012.0  1.442085e+08  2011-12-02
1981  1982  600380   健康元  ...  29252223.0  2.999997e+08  2011-11-30
```

#### 股本结构

接口: stock_zh_a_gbjg_em

目标地址: https://emweb.securities.eastmoney.com/pc_hsf10/pages/index.html#/gbjg

描述: 东方财富-A股数据-股本结构

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                 |
|--------|-----|--------------------|
| symbol | str | symbol="603392.SH" |

输出参数

| 名称          | 类型      | 描述 |
|-------------|---------|----|
| 变更日期        | object  | -  |
| 总股本         | int64   | -  |
| 流通受限股份      | float64 | -  |
| 其他内资持股(受限)  | float64 | -  |
| 境内法人持股(受限)  | float64 | -  |
| 境内自然人持股(受限) | float64 | -  |
| 已流通股份       | float64 | -  |
| 已上市流通A股     | int64   | -  |
| 变动原因        | object  | -  |

接口示例

```python
import akshare as ak

stock_zh_a_gbjg_em_df = ak.stock_zh_a_gbjg_em(symbol="603392.SH")
print(stock_zh_a_gbjg_em_df)
```

数据示例

```
    变更日期       总股本    流通受限股份  ...    已流通股份    已上市流通A股        变动原因
0   2024-09-12  1265122774          NaN  ...  1265122774  1.265123e+09          回购
1   2024-01-15  1268206999          NaN  ...  1268206999  1.268207e+09    网下配售股份上市
2   2023-06-05  1268206999    3620778.0  ...  1268206999  1.264586e+09       转增股上市
3   2023-05-04   906070705    2586270.0  ...   906070705  9.034844e+08    首发限售股份上市
4   2023-01-13   906070705  664113571.0  ...   906070705  2.419571e+08    网下配售股份上市
5   2022-07-13   906070705  687390006.0  ...   906070705  2.186807e+08      增发A股上市
6   2022-04-21   880208000  661527301.0  ...   880208000  2.186807e+08  送股上市,转增股上市
7   2021-05-18   607040000  456225725.0  ...   607040000  1.508143e+08        送股上市
8   2021-04-29   433600000  325875518.0  ...   433600000  1.077245e+08    首发限售股份上市
9   2020-04-15   433600000  390000000.0  ...   433600000  4.360000e+07      首发A股上市
10  2018-11-16   390000000  390000000.0  ...   390000000           NaN       上市前股本
11  2016-06-24   361200000  361200000.0  ...   361200000           NaN       上市前股本
[12 rows x 9 columns]
```

### 大宗交易

#### 市场统计

接口: stock_dzjy_sctj

目标地址: https://data.eastmoney.com/dzjy/dzjy_sctj.html

描述: 东方财富网-数据中心-大宗交易-市场统计

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 交易日期     | object  | -       |
| 上证指数     | float64 | -       |
| 上证指数涨跌幅  | float64 | 注意单位: % |
| 大宗交易成交总额 | float64 | 注意单位: 元 |
| 溢价成交总额   | float64 | 注意单位: 元 |
| 溢价成交总额占比 | float64 | 注意单位: % |
| 折价成交总额   | float64 | 注意单位: 元 |
| 折价成交总额占比 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_dzjy_sctj_df = ak.stock_dzjy_sctj()
print(stock_dzjy_sctj_df)
```

数据示例

```
        序号 交易日期      上证指数  ...    溢价成交总额占比 折价成交总额 折价成交总额占比
0        1  2025-01-13  3160.7550  ...    3.952040  414053400.0   65.605380
1        2  2025-01-10  3168.5238  ...    5.291598  736478342.1   93.934773
2        3  2025-01-09  3211.3933  ...   29.431536  426177100.0   66.837397
3        4  2025-01-08  3230.1679  ...    3.562115  493815200.0   89.406862
4        5  2025-01-07  3229.6439  ...   26.093856  683763300.0   61.112711
...    ...         ...        ...  ...         ...          ...         ...
4230  4231  2003-03-14  1466.0440  ...    0.000000   62720000.0  100.000000
4231  4232  2003-01-20  1482.6810  ...  100.000000          0.0    0.000000
4232  4233  2003-01-08  1372.0670  ...    0.000000  151742709.0  100.000000
4233  4234  2002-03-19  1665.3570  ...    0.000000  121770000.0  100.000000
4234  4235  2000-08-29  2105.4760  ...    0.000000   14910000.0  100.000000
[4235 rows x 9 columns]
```

#### 每日明细

接口: stock_dzjy_mrmx

目标地址: https://data.eastmoney.com/dzjy/dzjy_mrmx.html

描述: 东方财富网-数据中心-大宗交易-每日明细

限量: 单次返回所有历史数据

输入参数

| 名称         | 类型  | 描述                                              |
|------------|-----|-------------------------------------------------|
| symbol     | str | symbol='债券'; choice of {'A股', 'B股', '基金', '债券'} |
| start_date | str | start_date='20201123'; 开始日期                     |
| end_date   | sr  | end_date='20201204'; 结束日期                       |

输出参数-A股

| 名称       | 类型      | 描述      |
|----------|---------|---------|
| 序号       | int64   | -       |
| 交易日期     | object  | -       |
| 证券代码     | object  | -       |
| 证券简称     | object  | -       |
| 涨跌幅      | float64 | 注意单位: % |
| 收盘价      | float64 | -       |
| 成交价      | float64 | -       |
| 折溢率      | float64 | -       |
| 成交量      | float64 | 注意单位: 股 |
| 成交额      | float64 | 注意单位: 元 |
| 成交额/流通市值 | float64 | 注意单位: % |
| 买方营业部    | object  | -       |
| 卖方营业部    | object  | -       |

接口示例-A股

```python
import akshare as ak

stock_dzjy_mrmx_df = ak.stock_dzjy_mrmx(symbol='A股', start_date='20220104', end_date='20220104')
print(stock_dzjy_mrmx_df)
```

数据示例-A股

```
      序号   交易日期  ...                       买方营业部                     卖方营业部
0      1  2022-01-04  ...  中信建投证券股份有限公司上海浦东新区福山路证券营业部    国泰君安证券股份有限公司上海江苏路证券营业部
1      2  2022-01-04  ...       国信证券股份有限公司株洲神农大道证券营业部      长城证券股份有限公司株洲天台路证券营业部
2      3  2022-01-04  ...             开源证券股份有限公司上海分公司       中信建投证券股份有限公司北京东城分公司
3      4  2022-01-04  ...                        机构专用                      机构专用
4      5  2022-01-04  ...      华泰证券股份有限公司南京清凉门大街证券营业部      华泰证券股份有限公司广州天河城证券营业部
..   ...         ...  ...                         ...                       ...
130  131  2022-01-04  ...        信达证券股份有限公司深圳白石路证券营业部    兴业证券股份有限公司上海自由贸易试验区分公司
131  132  2022-01-04  ...        民生证券股份有限公司上海自贸试验区分公司         国信证券股份有限公司客户资产管理部
132  133  2022-01-04  ...               中信证券(山东)青岛分公司             中信证券(山东)青岛分公司
133  134  2022-01-04  ...           广发证券股份有限公司深圳前海营业部      财通证券股份有限公司江山中山路证券营业部
134  135  2022-01-04  ...     申万宏源证券有限公司成都火车南站东路证券营业部  申万宏源西部证券有限公司霍尔果斯亚欧路证券营业部
[135 rows x 13 columns]
```

输出参数-B股

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 交易日期  | object  | -       |
| 证券代码  | object  | -       |
| 证券简称  | object  | -       |
| 成交价   | float64 | -       |
| 成交量   | float64 | 注意单位: 股 |
| 成交额   | float64 | 注意单位: 元 |
| 买方营业部 | object  | -       |
| 卖方营业部 | object  | -       |

接口示例-B股

```python
import akshare as ak

stock_dzjy_mrmx_df = ak.stock_dzjy_mrmx(symbol='B股', start_date='20211104', end_date='20220104')
print(stock_dzjy_mrmx_df)
```

数据示例-B股

```
   序号        交易日期    证券代码  ...          成交额                 买方营业部             卖方营业部
0   1  2021-11-19  200596  ...  29031500.00  华鑫证券有限责任公司上海茅台路证券营业部  申万宏源交易单元(001698)
1   2  2021-12-02  900926  ...    762204.34         申万宏源证券有限公司国际部     申万宏源证券有限公司国际部
[2 rows x 9 columns]
```

输出参数-基金

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 交易日期  | object  | -       |
| 证券代码  | object  | -       |
| 证券简称  | object  | -       |
| 成交价   | float64 | -       |
| 成交量   | float64 | 注意单位: 股 |
| 成交额   | float64 | 注意单位: 元 |
| 买方营业部 | object  | -       |
| 卖方营业部 | object  | -       |

接口示例-基金

```python
import akshare as ak

stock_dzjy_mrmx_df = ak.stock_dzjy_mrmx(symbol='基金', start_date='20220104', end_date='20220104')
print(stock_dzjy_mrmx_df)
```

数据示例-基金

```
   序号        交易日期    证券代码  ...        成交额            买方营业部                  卖方营业部
0   1  2022-01-04  159919  ...   13503700  东北证券股份有限公司四川分公司  长城证券股份有限公司深圳深南大道证券营业部
1   2  2022-01-04  510050  ...  162350000   国泰君安证券股份有限公司总部  广发证券股份有限公司广州环市东路证券营业部
2   3  2022-01-04  510050  ...  584460000   国泰君安证券股份有限公司总部  广发证券股份有限公司广州环市东路证券营业部
[3 rows x 9 columns]
```

输出参数-债券

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 交易日期  | object  | -       |
| 证券代码  | object  | -       |
| 证券简称  | object  | -       |
| 成交价   | float64 | -       |
| 成交量   | float64 | 注意单位: 股 |
| 成交额   | float64 | 注意单位: 元 |
| 买方营业部 | object  | -       |
| 卖方营业部 | object  | -       |

接口示例-债券

```python
import akshare as ak

stock_dzjy_mrmx_df = ak.stock_dzjy_mrmx(symbol='债券', start_date='20220104', end_date='20220104')
print(stock_dzjy_mrmx_df)
```

数据示例-债券

```
      序号     交易日期  ...                 买方营业部                 卖方营业部
0      1  2022-01-04  ...  华安证券股份有限公司合肥高新区证券营业部  华福证券有限责任公司上海银城路证券营业部
1      2  2022-01-04  ...                  机构专用                  机构专用
2      3  2022-01-04  ...                  机构专用                  机构专用
3      4  2022-01-04  ...                  机构专用                  机构专用
4      5  2022-01-04  ...                  机构专用                  机构专用
..   ...         ...  ...                   ...                   ...
162  163  2022-01-04  ...                  机构专用                  机构专用
163  164  2022-01-04  ...                  机构专用                  机构专用
164  165  2022-01-04  ...                  机构专用                  机构专用
165  166  2022-01-04  ...                  机构专用                  机构专用
166  167  2022-01-04  ...                  机构专用                  机构专用
[167 rows x 9 columns]
```

#### 每日统计

接口: stock_dzjy_mrtj

目标地址: https://data.eastmoney.com/dzjy/dzjy_mrtj.html

描述: 东方财富网-数据中心-大宗交易-每日统计

限量: 单次返回所有历史数据

输入参数

| 名称         | 类型  | 描述                          |
|------------|-----|-----------------------------|
| start_date | str | start_date='20220105'; 开始日期 |
| end_date   | sr  | end_date='20220105'; 结束日期   |

输出参数

| 名称        | 类型      | 描述       |
|-----------|---------|----------|
| 序号        | int64   | -        |
| 交易日期      | object  | -        |
| 证券代码      | object  | -        |
| 证券简称      | object  | -        |
| 涨跌幅       | float64 | 注意单位: %  |
| 收盘价       | float64 | -        |
| 成交均价      | float64 | -        |
| 折溢率       | float64 | -        |
| 成交笔数      | int64   |          |
| 成交总量      | float64 | 注意单位: 万股 |
| 成交总额      | float64 | 注意单位: 万元 |
| 成交总额/流通市值 | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_dzjy_mrtj_df = ak.stock_dzjy_mrtj(start_date='20220105', end_date='20220105')
print(stock_dzjy_mrtj_df)
```

数据示例

```
    序号  交易日期    证券代码 证券简称  ...  成交笔数 成交总量 成交总额  成交总额/流通市值
0    1  2022-01-05  833509  同惠电子  ...     1   130.00  1783.6000   2.574944
1    2  2022-01-05  600617  国新能源  ...     1  1825.55  7046.6230   1.712782
2    3  2022-01-05  002919  名臣健康  ...     3   191.79  7427.8500   1.604464
3    4  2022-01-05  300858  科拓生物  ...     2    75.00  2779.7500   1.389459
4    5  2022-01-05  836263  中航泰达  ...     1   101.39   654.9794   1.380876
..  ..         ...     ...   ...  ...   ...      ...        ...        ...
76  77  2022-01-05  002475  立讯精密  ...     1    10.00   482.1000   0.001415
77  78  2022-01-05  600036  招商银行  ...     1    20.43   987.7905   0.000976
78  79  2022-01-05  601138  工业富联  ...     1    16.48   200.0672   0.000837
79  80  2022-01-05  603288  海天味业  ...     1     2.30   219.5120   0.000480
80  81  2022-01-05  600519  贵州茅台  ...     1     0.45   923.0535   0.000363
[81 rows x 12 columns]
```

#### 活跃 A 股统计

接口: stock_dzjy_hygtj

目标地址: https://data.eastmoney.com/dzjy/dzjy_hygtj.html

描述: 东方财富网-数据中心-大宗交易-活跃 A 股统计

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol='近三月'; choice of {'近一月', '近三月', '近六月', '近一年'} |

输出参数

| 名称            | 类型      | 描述       |
|---------------|---------|----------|
| 序号            | int64   | -        |
| 证券代码          | object  | -        |
| 证券简称          | object  | -        |
| 最新价           | float64 | -        |
| 涨跌幅           | float64 | 注意单位: %  |
| 最近上榜日         | object  | -        |
| 上榜次数-总计       | int64   | -        |
| 上榜次数-溢价       | int64   | -        |
| 上榜次数-折价       | int64   |          |
| 总成交额          | float64 | 注意单位: 万元 |
| 折溢率           | float64 | 注意单位: 万元 |
| 成交总额/流通市值     | float64 | -        |
| 上榜日后平均涨跌幅-1日  | float64 | 注意符号: %  |
| 上榜日后平均涨跌幅-5日  | float64 | 注意符号: %  |
| 上榜日后平均涨跌幅-10日 | float64 | 注意符号: %  |
| 上榜日后平均涨跌幅-20日 | float64 | 注意符号: %  |

接口示例

```python
import akshare as ak

stock_dzjy_hygtj_df = ak.stock_dzjy_hygtj(symbol='近三月')
print(stock_dzjy_hygtj_df)
```

数据示例

```
        序号 证券代码  证券简称  ... 上榜日后平均涨跌幅-5日  上榜日后平均涨跌幅-10日 上榜日后平均涨跌幅-20日
0        1  300750  宁德时代  ...     -1.088728      -1.637595      0.231556
1        2  601156  东航物流  ...      2.363677       3.253657      2.078001
2        3  601127   赛力斯  ...      3.137805       7.196673      3.697366
3        4  301498  乖宝宠物  ...      1.984983       4.387132     10.152126
4        5  689009  九号公司  ...      2.749486       4.166386     -0.388344
...    ...     ...   ...  ...           ...            ...           ...
1595  1596  000034  神州数码  ...     -8.193717     -15.026178     -8.664921
1596  1597  000021   深科技  ...     -1.542700       0.000000      0.000000
1597  1598  000016  深康佳A  ...     -7.954545      -1.136364      0.000000
1598  1599  000010  美丽生态  ...     20.187793      60.093897     31.455399
1599  1600  000009  中国宝安  ...     -5.544148       2.874743     -2.464066
[1600 rows x 16 columns]
```

#### 活跃营业部统计

接口: stock_dzjy_hyyybtj

目标地址: https://data.eastmoney.com/dzjy/dzjy_hyyybtj.html

描述: 东方财富网-数据中心-大宗交易-活跃营业部统计

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                              |
|--------|-----|-----------------------------------------------------------------|
| symbol | str | symbol='近3日'; choice of {'当前交易日', '近3日', '近5日', '近10日', '近30日'} |

输出参数

| 名称          | 类型      | 描述       |
|-------------|---------|----------|
| 序号          | int64   | -        |
| 营业部名称       | str     | -        |
| 最近上榜日       | object  | -        |
| 次数总计-买入     | float64 | -        |
| 次数总计-卖出     | float64 | 注意单位: %  |
| 成交金额统计-买入   | float64 | 注意单位: 万元 |
| 成交金额统计-卖出   | float64 | 注意单位: 万元 |
| 成交金额统计-净买入额 | float64 | 注意单位: 万元 |
| 买入的股票       | object  |          |

接口示例

```python
import akshare as ak

stock_dzjy_hyyybtj_df = ak.stock_dzjy_hyyybtj(symbol='近3日')
print(stock_dzjy_hyyybtj_df)
```

数据示例

```
      序号  ...                                              买入的股票
0      1  ...  000426|兴业银锡,000703|恒逸石化,000736|中交地产,001236|弘业期...
1      2  ...  605376|博迁新材,688165|埃夫特,688234|天岳先进,688469|芯联集成...
2      3  ...  002156|通富微电,300398|飞凯材料,300783|三只松鼠,301091|深城交...
3      4  ...  002156|通富微电,003009|中天火箭,300398|飞凯材料,688234|天岳先...
4      5  ...    600021|上海电力,600033|福建高速,601326|秦港股份,603378|亚士创能
..   ...  ...                                                ...
242  243  ...                                               None
243  244  ...                                               None
244  245  ...                                               None
245  246  ...                                               None
246  247  ...                                               None
[247 rows x 9 columns]
```

#### 营业部排行

接口: stock_dzjy_yybph

目标地址: https://data.eastmoney.com/dzjy/dzjy_yybph.html

描述: 东方财富网-数据中心-大宗交易-营业部排行

限量: 单次返回所有历史数据

输入参数

| 名称     | 类型  | 描述                                                   |
|--------|-----|------------------------------------------------------|
| symbol | str | symbol='近三月'; choice of {'近一月', '近三月', '近六月', '近一年'} |

输出参数

| 名称          | 类型      | 描述      |
|-------------|---------|---------|
| 序号          | int64   | -       |
| 营业部名称       | object  | -       |
| 上榜后1天-买入次数  | float64 | -       |
| 上榜后1天-平均涨幅  | float64 | 注意单位: % |
| 上榜后1天-上涨概率  | float64 | -       |
| 上榜后5天-买入次数  | float64 | -       |
| 上榜后5天-平均涨幅  | float64 | 注意单位: % |
| 上榜后5天-上涨概率  | float64 | -       |
| 上榜后10天-买入次数 | float64 | -       |
| 上榜后10天-平均涨幅 | float64 | 注意单位: % |
| 上榜后10天-上涨概率 | float64 |         |
| 上榜后20天-买入次数 | float64 | -       |
| 上榜后20天-平均涨幅 | float64 | 注意单位: % |
| 上榜后20天-上涨概率 | float64 |         |

接口示例

```python
import akshare as ak

stock_dzjy_yybph_df = ak.stock_dzjy_yybph(symbol='近三月')
print(stock_dzjy_yybph_df)
```

数据示例

```
        序号               营业部名称  ...  上榜后20天-平均涨幅  上榜后20天-上涨概率
0        1                机构专用  ...     4.105711    52.522101
1        2     华泰证券营业总部(非营业场所)  ...     2.933068    51.394422
2        3       中信证券总部(非营业场所)  ...     7.533781    60.317460
3        4            国泰君安证券总部  ...     6.994607    60.606061
4        5      招商证券西安北大街证券营业部  ...     7.372532    67.346939
...    ...                 ...  ...          ...          ...
1226  1227           华福证券浙江分公司  ...          NaN          NaN
1227  1228    财通证券杭州富阳公望街证券营业部  ...          NaN          NaN
1228  1229   申万宏源证券重庆中山一路证券营业部  ...          NaN          NaN
1229  1230  国泰君安证券苏州苏州大道东证券营业部  ...          NaN          NaN
1230  1231           兴业证券莆田分公司  ...          NaN          NaN
[1231 rows x 14 columns]
```

### 一致行动人

接口: stock_yzxdr_em

目标地址: http://data.eastmoney.com/yzxdr/

描述: 东方财富网-数据中心-特色数据-一致行动人

限量: 单次返回所有历史数据

输入参数

| 名称   | 类型  | 描述                         |
|------|-----|----------------------------|
| date | str | date="20200930"; 每年的季度末时间点 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 一致行动人  | object  | -       |
| 股东排名   | object  | -       |
| 持股数量   | int64   | -       |
| 持股比例   | float64 | -       |
| 持股数量变动 | object  | 注意单位: % |
| 行业     | object  | -       |
| 公告日期   | object  | -       |

接口示例

```python
import akshare as ak

stock_yzxdr_em_df = ak.stock_yzxdr_em(date="20210331")
print(stock_yzxdr_em_df)
```

数据示例

```
      序号    股票代码  股票简称  ... 持股数量变动     行业        公告日期
0        1  688981  中芯国际  ...    NaN    半导体  2024-11-08
1        2  688981  中芯国际  ...    NaN    半导体  2024-11-08
2        3  688981  中芯国际  ...    NaN    半导体  2024-11-08
3        4  688347  华虹公司  ...    NaN    半导体  2024-11-08
4        5  688347  华虹公司  ...    NaN    半导体  2024-11-08
...    ...     ...   ...  ...    ...    ...         ...
4637  4638  002521  齐峰新材  ...    NaN   造纸印刷  2024-10-12
4638  4639  300102  乾照光电  ...    NaN  光学光电子  2024-10-12
4639  4640  301227  森鹰窗业  ...    NaN   装修建材  2024-10-12
4640  4641  301227  森鹰窗业  ...    NaN   装修建材  2024-10-12
4641  4642  300708  聚灿光电  ...    NaN  光学光电子  2024-10-11
[4642 rows x 10 columns]
```

### 融资融券

#### 标的证券名单及保证金比例查询

接口: stock_margin_ratio_pa

目标地址: https://stock.pingan.com/static/webinfo/margin/business.html?businessType=0

描述: 融资融券-标的证券名单及保证金比例查询

限量: 单次返回指定交易所和交易日的所有历史数据

输入参数

| 名称     | 类型  | 描述                                         |
|--------|-----|--------------------------------------------|
| symbol | str | symbol="深市"; choice of {"深市", "沪市", "北交所"} |
| date   | str | date="20260113"                            |

输出参数

| 名称   | 类型      | 描述 |
|------|---------|----|
| 证券代码 | object  | -  |
| 证券简称 | object  | -  |
| 融资比例 | float64 | -  |
| 融券比例 | float64 | -  |

接口示例

```python
import akshare as ak

stock_margin_ratio_pa_df = ak.stock_margin_ratio_pa(symbol="沪市", date="20260113")
print(stock_margin_ratio_pa_df)
```

数据示例

```
      证券代码   证券简称  融资比例 融券比例
0     510050    50ETF   0.8   1.0
1     510100  SZ50ETF   0.8   1.0
2     510150    消费ETF   0.8   1.0
3     510180   180ETF   0.8   1.0
4     510210    综指ETF   0.8   1.0
...      ...      ...   ...   ...
1950  688807     优迅股份   1.0   1.8
1951  688809     强一股份   1.0   1.8
1952  688819     天能股份   0.8   1.5
1953  688981     中芯国际   0.8   1.5
1954  689009     九号公司   1.3   1.5
[1955 rows x 4 columns]
```

#### 两融账户信息

接口: stock_margin_account_info

目标地址: https://data.eastmoney.com/rzrq/zhtjday.html

描述: 东方财富网-数据中心-融资融券-融资融券账户统计-两融账户信息

限量: 单次返回所有历史数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称            | 类型      | 描述       |
|---------------|---------|----------|
| 日期            | object  | -        |
| 融资余额          | float64 | 注意单位: 亿  |
| 融券余额          | float64 | 注意单位: 亿  |
| 融资买入额         | float64 | 注意单位: 亿  |
| 融券卖出额         | float64 | 注意单位: 亿  |
| 证券公司数量        | float64 | 注意单位: 家  |
| 营业部数量         | float64 | 注意单位: 家  |
| 个人投资者数量       | float64 | 注意单位: 万名 |
| 机构投资者数量       | float64 | 注意单位: 家  |
| 参与交易的投资者数量    | float64 | 注意单位: 名  |
| 有融资融券负债的投资者数量 | float64 | 注意单位: 名  |
| 担保物总价值        | float64 | 注意单位: 亿  |
| 平均维持担保比例      | float64 | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_margin_account_info_df = ak.stock_margin_account_info()
print(stock_margin_account_info_df)
```

数据示例

```
      日期           融资余额 融券余额  ...  有融资融券负债的投资者数量   担保物总价值 平均维持担保比例
0     2012-09-27    704.53   15.31  ...            NaN       NaN       NaN
1     2012-09-28    685.47   15.20  ...            NaN       NaN       NaN
2     2012-10-08    696.69   15.64  ...            NaN       NaN       NaN
3     2012-10-09    699.63   17.59  ...            NaN       NaN       NaN
4     2012-10-10    708.57   17.63  ...            NaN       NaN       NaN
...          ...       ...     ...  ...            ...       ...       ...
2836  2024-06-06  14761.62  367.59  ...      1545084.0  44142.04     247.5
2837  2024-06-07  14697.46  341.56  ...      1538584.0  44096.42     248.3
2838  2024-06-11  14744.14  336.41  ...      1545026.0  44228.19     249.0
2839  2024-06-12  14761.55  338.57  ...      1547653.0  44506.55     250.1
2840  2024-06-13  14769.50  338.84  ...      1548858.0  44328.60     249.3
[2841 rows x 13 columns]
```

#### 上海证券交易所

##### 融资融券汇总

接口: stock_margin_sse

目标地址: http://www.sse.com.cn/market/othersdata/margin/sum/

描述: 上海证券交易所-融资融券数据-融资融券汇总数据

限量: 单次返回指定时间段内的所有历史数据

输入参数

| 名称         | 类型  | 描述                    |
|------------|-----|-----------------------|
| start_date | str | start_date="20010106" |
| end_date   | str | end_date="20010106"   |

输出参数

| 名称     | 类型     | 描述      |
|--------|--------|---------|
| 信用交易日期 | object | -       |
| 融资余额   | int64  | 注意单位: 元 |
| 融资买入额  | int64  | 注意单位: 元 |
| 融券余量   | int64  | -       |
| 融券余量金额 | int64  | 注意单位: 元 |
| 融券卖出量  | int64  | -       |
| 融资融券余额 | int64  | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_margin_sse_df = ak.stock_margin_sse(start_date="20010106", end_date="20210208")
print(stock_margin_sse_df)
```

数据示例

```
      信用交易日期   融资余额     融资买入额  ...    融券余量金额   融券卖出量   融资融券余额
0     20210208  796355371775  33386752258  ...  87718590706  148374362  884073962481
1     20210205  798537947597  39256505179  ...  87230537840  169481352  885768485437
2     20210204  802902827539  41486860920  ...  87780973049  164920794  890683800588
3     20210203  807189978760  42621756166  ...  88548469683  251518719  895738448443
4     20210202  807418158794  37533872673  ...  88303226275  230032183  895721385069
        ...           ...          ...  ...          ...        ...           ...
1995  20121127   52059352556   1561381214  ...   1384086520  255616186   53443439076
1996  20121126   52224867623   1322180521  ...   1366826765  234474541   53591694388
1997  20121123   52163842479   1543727769  ...   1347840020  257859771   53511682499
1998  20121122   52326315202   1332145449  ...   1302639373  214932697   53628954575
1999  20121121   52338359165   1943146103  ...   1306569346  308581665   53644928511
[2000 rows x 7 columns]
```

##### 融资融券明细

接口: stock_margin_detail_sse

目标地址: http://www.sse.com.cn/market/othersdata/margin/detail/

描述: 上海证券交易所-融资融券数据-融资融券明细数据

限量: 单次返回交易日的所有历史数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20210205" |

输出参数

| 名称     | 类型     | 描述      |
|--------|--------|---------|
| 信用交易日期 | object | -       |
| 标的证券代码 | object | -       |
| 标的证券简称 | object | -       |
| 融资余额   | int64  | 注意单位: 元 |
| 融资买入额  | int64  | 注意单位: 元 |
| 融资偿还额  | int64  | 注意单位: 元 |
| 融券余量   | int64  | -       |
| 融券卖出量  | int64  | -       |
| 融券偿还量  | int64  | -       |

接口示例

```python
import akshare as ak

stock_margin_detail_sse_df = ak.stock_margin_detail_sse(date="20230922")
print(stock_margin_detail_sse_df)
```

数据示例

```
      信用交易日期  标的证券代码   标的证券简称  ...  融券余量 融券卖出量  融券偿还量
0     20230922  510050    50ETF  ...  31081200  21114800  11402100
1     20230922  510100  SZ50ETF  ...         0         0         0
2     20230922  510150    消费ETF  ...         0         0         0
3     20230922  510180   180ETF  ...    409300    173200    130900
4     20230922  510210    综指ETF  ...         0         0         0
        ...     ...      ...  ...       ...       ...       ...
1735  20230922  688799     华纳药厂  ...         0         0         0
1736  20230922  688800      瑞可达  ...    124592      3617     21777
1737  20230922  688819     天能股份  ...   1231449     75962      3000
1738  20230922  688981     中芯国际  ...   4747379    389501    606169
1739  20230922  689009     九号公司  ...  11236510    112012     27234
[1740 rows x 9 columns]
```

#### 深圳证券交易所

##### 融资融券汇总

接口: stock_margin_szse

目标地址: https://www.szse.cn/disclosure/margin/margin/index.html

描述: 深圳证券交易所-融资融券数据-融资融券汇总数据

限量: 单次返回指定时间内的所有历史数据

输入参数

| 名称   | 类型  | 描述                    |
|------|-----|-----------------------|
| date | str | date="20240411"; 交易日期 |

输出参数

| 名称     | 类型      | 描述          |
|--------|---------|-------------|
| 融资买入额  | float64 | 注意单位: 亿元    |
| 融资余额   | float64 | 注意单位: 亿元    |
| 融券卖出量  | float64 | 注意单位: 亿股/亿份 |
| 融券余量   | float64 | 注意单位: 亿股/亿份 |
| 融券余额   | float64 | 注意单位: 亿元    |
| 融资融券余额 | float64 | 注意单位: 亿元    |

接口示例

```python
import akshare as ak

stock_margin_sse_df = ak.stock_margin_szse(date="20240411")
print(stock_margin_sse_df)
```

数据示例

```
    融资买入额     融资余额  融券卖出量   融券余量   融券余额   融资融券余额
0  321.08  7077.67   0.28  24.34  157.3  7234.97
```

##### 融资融券明细

接口: stock_margin_detail_szse

目标地址: https://www.szse.cn/disclosure/margin/margin/index.html

描述: 深证证券交易所-融资融券数据-融资融券交易明细数据

限量: 单次返回指定 date 的所有历史数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20220118" |

输出参数

| 名称     | 类型     | 描述        |
|--------|--------|-----------|
| 证券代码   | object | -         |
| 证券简称   | object | -         |
| 融资买入额  | int64  | 注意单位: 元   |
| 融资余额   | int64  | 注意单位: 元   |
| 融券卖出量  | int64  | 注意单位: 股/份 |
| 融券余量   | int64  | 注意单位: 股/份 |
| 融券余额   | int64  | 注意单位: 元   |
| 融资融券余额 | int64  | 注意单位: 元   |

接口示例

```python
import akshare as ak

stock_margin_detail_sse_df = ak.stock_margin_detail_szse(date="20230925")
print(stock_margin_detail_sse_df)
```

数据示例

```
        证券代码   证券简称     融资买入额  ...     融券余量      融券余额      融资融券余额
0     000001   平安银行  67452975  ...  1791277  20098128  4365763792
1     000002  万  科Ａ  78869157  ...  6637572  87217696  4819262864
2     000006   深振业Ａ  11862407  ...   893154   4037056   471912480
3     000008   神州高铁   6467970  ...    37100     89411   264908600
4     000009   中国宝安   5545586  ...  2540168  26036723  1501197300
      ...    ...       ...  ...      ...       ...         ...
1806  301528    多浦乐   9457046  ...      600     52212    60245085
1807  301529   福赛科技  10576005  ...    67138   3056121    41545975
1808  301533   威马农机   1615519  ...     1000     43110    23167466
1809  301548  C崇德科技  23852209  ...        0         0    41105017
1810  301550   斯菱股份  17612051  ...     2300     98486    36562401
[1811 rows x 8 columns]
```

##### 标的证券信息

接口: stock_margin_underlying_info_szse

目标地址: https://www.szse.cn/disclosure/margin/object/index.html

描述: 深圳证券交易所-融资融券数据-标的证券信息

限量: 单次返回交易日的所有历史数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date="20210205" |

输出参数

| 名称       | 类型     | 描述  |
|----------|--------|-----|
| 证券代码     | object | -   |
| 证券简称     | object | -   |
| 融资标的     | object | -   |
| 融券标的     | object | -   |
| 当日可融资    | object | -   |
| 当日可融券    | object | -   |
| 融券卖出价格限制 | object | -   |
| 涨跌幅限制    | object | -   |

接口示例

```python
import akshare as ak

stock_margin_underlying_info_szse_df = ak.stock_margin_underlying_info_szse(date="20210727")
print(stock_margin_underlying_info_szse_df)
```

数据示例

```
    证券代码  证券简称 融资标的 融券标的 当日可融资 当日可融券 融券卖出价格限制 涨跌幅限制
0     000001  平安银行    Y    Y     Y     Y        Y   10%
1     000002   万科Ａ    Y    Y     Y     Y        Y   10%
2     000006  深振业Ａ    Y    Y     Y     Y        Y   10%
3     000008  神州高铁    Y    Y     Y     Y        Y   10%
4     000009  中国宝安    Y    Y     Y     Y        Y   10%
      ...   ...  ...  ...   ...   ...      ...   ...
1000  301030   C仕净    Y    Y     Y     Y        Y   无限制
1001  301031  中熔电气    Y    Y     Y     Y        Y   20%
1002  301032   C新柴    Y    Y     Y     Y        Y   无限制
1003  301033   C迈普    Y    Y     Y     Y        Y   无限制
1004  301039  中集车辆    Y    Y     Y     Y        Y   20%
```

### 盈利预测-东方财富

接口: stock_profit_forecast_em

目标地址: http://data.eastmoney.com/report/profitforecast.jshtml

描述: 东方财富网-数据中心-研究报告-盈利预测; 该数据源网页端返回数据有异常, 本接口已修复该异常

限量: 单次返回指定 symbol 的数据

输入参数

| 名称     | 类型  | 描述                                                                                                 |
|--------|-----|----------------------------------------------------------------------------------------------------|
| symbol | str | symbol="", 默认为获取全部数据; symbol="船舶制造", 则获取具体行业板块的数据; 行业板块可以通过 ak.stock_board_industry_name_em() 接口获取 |

输出参数

| 名称              | 类型      | 描述  |
|-----------------|---------|-----|
| 序号              | int64   | -   |
| 代码              | object  | -   |
| 名称              | object  | -   |
| 研报数             | int64   | -   |
| 机构投资评级(近六个月)-买入 | float64 | -   |
| 机构投资评级(近六个月)-增持 | float64 | -   |
| 机构投资评级(近六个月)-中性 | float64 | -   |
| 机构投资评级(近六个月)-减持 | int64   | -   |
| 机构投资评级(近六个月)-卖出 | int64   | -   |
| xxxx预测每股收益      | float64 | -   |
| xxxx预测每股收益      | float64 | -   |
| xxxx预测每股收益      | float64 | -   |
| xxxx预测每股收益      | float64 | -   |

接口示例

```python
import akshare as ak

stock_profit_forecast_em_df = ak.stock_profit_forecast_em()
print(stock_profit_forecast_em_df)
```

数据示例

```
      序号  代码    名称  研报数  ...  2022预测每股收益  2023预测每股收益  2024预测每股收益  2025预测每股收益
0        1  600519  贵州茅台   57  ...   49.925612   58.933509   69.944737   81.378268
1        2  603605   珀莱雅   53  ...    2.060203    2.832283    3.571340    4.412750
2        3  000858   五粮液   52  ...    6.876187    7.795654    8.837846   10.038078
3        4  300957   贝泰妮   50  ...    2.481654    3.066469    3.869959    4.834542
4        5  300896   爱美客   47  ...    5.840074    8.793660   12.255915   16.379000
    ...     ...   ...  ...  ...         ...         ...         ...         ...
3026  3027  002428  云南锗业    1  ...   -0.095546    0.020000    0.050000    0.060000
3027  3028  688058   宝兰德    1  ...   -0.618857    1.060000    1.710000    2.660000
3028  3029  688309  恒誉环保    1  ...    0.182819    0.720000    0.810000    0.910000
3029  3030  300552  万集科技    1  ...   -0.136009   -0.050000    0.380000    0.670000
3030  3031  600637  东方明珠    1  ...    0.051296    0.280000    0.290000         NaN
[3031 rows x 13 columns]
```

### 港股盈利预测-经济通

接口: stock_hk_profit_forecast_et

目标地址: https://www.etnet.com.hk/www/sc/stocks/realtime/quote_profit.php?code=9999

描述: 经济通-公司资料-盈利预测

限量: 单次返回指定 symbol 和 indicator 的数据

输入参数

| 名称        | 类型  | 描述                                                                    |
|-----------|-----|-----------------------------------------------------------------------|
| symbol    | str | symbol="09999"                                                        |
| indicator | str | indicator="盈利预测概览"; choice of {"评级总览", "去年度业绩表现", "综合盈利预测", "盈利预测概览"} |

输出参数-盈利预测概览

| 名称    | 类型      | 描述               |
|-------|---------|------------------|
| 财政年度  | object  | -                |
| 纯利/亏损 | float64 | 注意单位：百万元人民币/百万港元 |
| 每股盈利  | float64 | 注意单位：分/港仙        |
| 每股派息  | float64 | 注意单位：分/港仙        |
| 证券商   | object  | -                |
| 评级    | object  | -                |
| 目标价   | float64 | 注意单位：港元          |
| 更新日期  | object  | -                |

接口示例-盈利预测概览

```python
import akshare as ak

stock_hk_profit_forecast_et_df = ak.stock_hk_profit_forecast_et(symbol="09999", indicator="盈利预测概览")
print(stock_hk_profit_forecast_et_df)
```

数据示例-盈利预测概览

```
    财政年度    纯利/亏损    每股盈利     每股派息     证券商    评级    目标价       更新日期
0   2023  26568.0   814.0   256.62   星展唯高达    买入  237.0 2023-08-25
1   2023  29960.0   909.9   272.90      大和    买入  215.0 2023-11-16
2   2023  29679.0   913.0  1364.00      野村    买入  215.0 2023-11-17
3   2023  29968.0   922.0   260.30    中银国际    买入  212.0 2023-12-18
4   2023  32835.0  1010.0   255.00  中国国际金融  优於大市  211.0 2023-11-17
5   2023  29719.2   924.4  1453.00      高盛    买入  206.0 2023-11-17
6   2023  36308.0  1013.3   258.19    大华继显    买入  204.0 2023-11-17
7   2023  27710.0   860.0      NaN    中信证券    买入  200.0 2023-11-02
8   2023  33891.0   953.0  1435.00    摩根大通  优於大市  195.0 2023-12-13
10  2024  28143.0   862.0   271.72   星展唯高达    --    NaN 2023-08-25
11  2024  28151.0   875.0  1294.00      野村    --    NaN 2023-11-17
12  2024  28799.0   874.6   262.30      大和    --    NaN 2023-11-16
13  2024  30892.0   954.0   269.30    中银国际    --    NaN 2023-12-18
14  2024  33983.0  1046.0   264.00  中国国际金融    --    NaN 2023-11-17
15  2024  30592.2   957.0  1508.00      高盛    --    NaN 2023-11-17
16  2024  36514.0  1021.8  1113.46    大华继显    --    NaN 2023-11-17
17  2024  28630.0   890.0      NaN    中信证券    --    NaN 2023-11-02
18  2024  35252.0   996.0  1476.00    摩根大通    --    NaN 2023-12-13
20  2025  31448.0   964.0   301.91   星展唯高达    --    NaN 2023-08-25
21  2025  31188.0   947.2   284.10      大和    --    NaN 2023-11-16
22  2025  30540.0   959.0  1404.00      野村    --    NaN 2023-11-17
23  2025  33620.0  1037.2   292.80    中银国际    --    NaN 2023-12-18
24  2025  34070.8  1064.4  1678.00      高盛    --    NaN 2023-11-17
25  2025  31709.0  1106.7  1210.28    大华继显    --    NaN 2023-11-17
26  2025  31400.0   970.0      NaN    中信证券    --    NaN 2023-11-02
27  2025  39789.0  1130.0  1674.00    摩根大通    --    NaN 2023-12-13
```

### 盈利预测-同花顺

接口: stock_profit_forecast_ths

目标地址: http://basic.10jqka.com.cn/new/600519/worth.html

描述: 同花顺-盈利预测

限量: 单次返回指定 symbol 和 indicator 的数据

输入参数

| 名称        | 类型  | 描述                                                                                    |
|-----------|-----|---------------------------------------------------------------------------------------|
| symbol    | str | symbol="600519"; 股票代码                                                                 |
| indicator | str | indicator="预测年报每股收益"; choice of {"预测年报每股收益", "预测年报净利润", "业绩预测详表-机构", "业绩预测详表-详细指标预测"} |

输出参数-预测年报每股收益

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| 年度    | object  | -   |
| 预测机构数 | int64   | -   |
| 最小值   | float64 | -   |
| 均值    | float64 | -   |
| 最大值   | float64 | -   |
| 行业平均数 | float64 | -   |

接口示例-预测年报每股收益

```python
import akshare as ak

stock_profit_forecast_ths_df = ak.stock_profit_forecast_ths(symbol="600519", indicator="预测年报每股收益")
print(stock_profit_forecast_ths_df)
```

数据示例-预测年报每股收益

```
     年度  预测机构数    最小值     均值    最大值  行业平均数
0  2022     48  48.46  49.73  50.58   7.00
1  2023     48  56.01  58.31  61.53   8.45
2  2024     48  63.79  67.59  73.67  10.12
```

输出参数-预测年报净利润

| 名称    | 类型      | 描述  |
|-------|---------|-----|
| 年度    | object  | -   |
| 预测机构数 | int64   | -   |
| 最小值   | float64 | -   |
| 均值    | float64 | -   |
| 最大值   | float64 | -   |
| 行业平均数 | float64 | -   |

接口示例-预测年报净利润

```python
import akshare as ak

stock_profit_forecast_ths_df = ak.stock_profit_forecast_ths(symbol="600519", indicator="预测年报净利润")
print(stock_profit_forecast_ths_df)
```

数据示例-预测年报净利润

```
     年度  预测机构数     最小值      均值     最大值   行业平均数
0  2022     48  608.70  624.70  635.37   92.07
1  2023     48  703.55  732.50  772.89  109.92
2  2024     48  801.30  849.09  925.44  130.04
```

输出参数-业绩预测详表-机构

| 名称             | 类型      | 描述  |
|----------------|---------|-----|
| 机构名称           | object  | -   |
| 研究员            | object  | -   |
| 预测年报每股收益2022预测 | float64 | -   |
| 预测年报每股收益2023预测 | float64 | -   |
| 预测年报每股收益2024预测 | float64 | -   |
| 预测年报净利润2022预测  | object  | -   |
| 预测年报净利润2023预测  | object  | -   |
| 预测年报净利润2024预测  | object  | -   |
| 报告日期           | object  | -   |

接口示例-业绩预测详表-机构

```python
import akshare as ak

stock_profit_forecast_ths_df = ak.stock_profit_forecast_ths(symbol="600519", indicator="业绩预测详表-机构")
print(stock_profit_forecast_ths_df)
```

数据示例-业绩预测详表-机构

```
       机构名称          研究员  ...  预测年报净利润2024预测        报告日期
0      东方证券          叶书怀  ...        829.14亿  2023-03-07
1      中泰证券          范劲松  ...        848.80亿  2023-02-13
2      中信证券           李鑫  ...        832.79亿  2023-02-13
3      招银证券  Joseph Wong  ...        866.26亿  2023-02-06
4      华鑫证券          孙山山  ...        860.24亿  2023-01-03
5      开源证券          张宇光  ...        860.49亿  2023-01-03
6  群益证券(香港)          顾向君  ...        845.28亿  2023-01-03
7      财通证券           苏铖  ...        850.53亿  2023-01-02
8      东吴证券           汤军  ...        845.27亿  2022-12-30
9      东莞证券          魏红梅  ...        846.24亿  2022-12-30
```

输出参数-业绩预测详表-详细指标预测

| 名称        | 类型     | 描述  |
|-----------|--------|-----|
| 预测指标      | object | -   |
| 2019-实际值  | object | -   |
| 2020-实际值  | object | -   |
| 2021-实际值  | object | -   |
| 预测2022-平均 | object | -   |
| 预测2023-平均 | object | -   |
| 预测2024-平均 | object | -   |

接口示例-业绩预测详表-详细指标预测

```python
import akshare as ak

stock_profit_forecast_ths_df = ak.stock_profit_forecast_ths(symbol="600519", indicator="业绩预测详表-详细指标预测")
print(stock_profit_forecast_ths_df)
```

数据示例-业绩预测详表-详细指标预测

```
       预测指标 2019-实际值 2020-实际值  2021-实际值 预测2022-平均 预测2023-平均 预测2024-平均
0   营业收入(元)  854.30亿  949.15亿  1061.90亿  1268.88亿  1471.77亿  1691.60亿
1   营业收入增长率   15.10%   10.29%    11.71%    19.49%    15.99%    14.93%
2   利润总额(元)  587.83亿  661.97亿   745.28亿   883.27亿  1034.55亿  1198.90亿
3    净利润(元)  412.06亿  466.97亿   524.60亿   624.70亿   732.50亿   849.09亿
4    净利润增长率   17.05%   13.33%    12.34%    18.92%    17.23%    15.96%
5  每股现金流(元)    35.99    41.13     50.97     55.46     59.37     72.57
6  每股净资产(元)   108.27   128.42    150.88    177.18    210.97    251.11
7    净资产收益率   33.09%   31.41%    29.90%    28.62%    28.26%    27.64%
8   市盈率(动态)    53.98    47.63     42.40     35.65     30.41     26.24
```

### 概念板块

#### 同花顺-概念板块指数

接口: stock_board_concept_index_ths

目标地址: https://q.10jqka.com.cn/gn/detail/code/301558

描述: 同花顺-板块-概念板块-指数日频率数据

限量: 单次返回所有日频指数数据

输入参数

| 名称         | 类型  | 描述                                                                         |
|------------|-----|----------------------------------------------------------------------------|
| symbol     | str | symbol="阿里巴巴概念"; 可以通过调用 **ak.stock_board_concept_name_ths()** 查看同花顺的所有概念名称 |
| start_date | str | start_date="20200101"; 开始时间                                                |
| end_date   | str | end_date="20250228"; 结束时间                                                  |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 开盘价 | float64 | -   |
| 最高价 | float64 | -   |
| 最低价 | float64 | -   |
| 收盘价 | float64 | -   |
| 成交量 | int64   | -   |
| 成交额 | float64 | -   |

接口示例

```python
import akshare as ak

stock_board_concept_index_ths_df = ak.stock_board_concept_index_ths(symbol="阿里巴巴概念", start_date="20200101", end_date="20250321")
print(stock_board_concept_index_ths_df)
```

数据示例

```
        日期       开盘价     最高价  ...     收盘价      成交量           成交额
0     2020-01-02  1105.430  1133.391  ...  1130.280   1867106700  2.270406e+10
1     2020-01-03  1133.673  1143.881  ...  1140.087   1734555400  2.049213e+10
2     2020-01-06  1132.865  1160.416  ...  1152.178   1926283700  2.349216e+10
3     2020-01-07  1154.766  1173.157  ...  1173.068   1750895500  2.130472e+10
4     2020-01-08  1163.282  1165.660  ...  1145.718   1803020900  2.088292e+10
...          ...       ...       ...  ...       ...          ...           ...
1258  2025-03-17  1876.853  1885.138  ...  1873.981  10514337100  1.814942e+11
1259  2025-03-18  1878.808  1895.155  ...  1880.904  10643759300  1.781277e+11
1260  2025-03-19  1869.726  1870.423  ...  1853.801  10066894000  1.679532e+11
1261  2025-03-20  1846.304  1869.670  ...  1839.816   9905099800  1.734386e+11
1262  2025-03-21  1826.952  1830.013  ...  1790.115  10158278900  1.719123e+11
[1263 rows x 7 columns]
```

#### 同花顺-概念板块简介

接口: stock_board_concept_info_ths

目标地址: http://q.10jqka.com.cn/gn/detail/code/301558/

描述: 同花顺-板块-概念板块-板块简介

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                                |
|--------|-----|-----------------------------------------------------------------------------------|
| symbol | str | symbol: str = "阿里巴巴概念"; 可以通过调用 **ak.stock_board_concept_name_ths()** 查看同花顺的所有概念名称 |

输出参数

| 名称 | 类型      | 描述 |
|----|---------|----|
| 项目 | object  | -  |
| 值  | float64 | -  |

接口示例

```python
import akshare as ak

stock_board_concept_info_ths_df = ak.stock_board_concept_info_ths(symbol="阿里巴巴概念")
print(stock_board_concept_info_ths_df)
```

数据示例

```
         项目         值
0        今开   1825.71
1        昨收   1844.18
2        最低   1752.74
3        最高   1827.99
4   成交量(万手)  12324.70
5      板块涨幅    -4.96%
6      涨幅排名   317/396
7      涨跌家数    25/222
8  资金净流入(亿)   -251.00
9    成交额(亿)   2167.43
```

#### 东方财富-概念板块

接口: stock_board_concept_name_em

目标地址: https://quote.eastmoney.com/center/boardlist.html#concept_board

描述: 东方财富网-行情中心-沪深京板块-概念板块

限量: 单次返回当前时刻所有概念板块的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述     |
|----------|---------|--------|
| 排名       | int64   | -      |
| 板块名称     | object  | -      |
| 板块代码     | object  | -      |
| 最新价      | float64 | -      |
| 涨跌额      | float64 | -      |
| 涨跌幅      | float64 | 注意单位：% |
| 总市值      | int64   | -      |
| 换手率      | float64 | 注意单位：% |
| 上涨家数     | int64   | -      |
| 下跌家数     | int64   | -      |
| 领涨股票     | object  | -      |
| 领涨股票-涨跌幅 | float64 | 注意单位：% |

接口示例

```python
import akshare as ak

stock_board_concept_name_em_df = ak.stock_board_concept_name_em()
print(stock_board_concept_name_em_df)
```

数据示例

```
      排名  板块名称    板块代码  最新价    涨跌额  ...  换手率  上涨家数  下跌家数  领涨股票  领涨股票-涨跌幅
0      1        乳业  BK0892  1045.53   42.05  ...   2.85    26     3  庄园牧场     10.04
1      2      免税概念  BK0927  1308.35   48.13  ...   2.58    30     1  海南发展     10.01
2      3      退税商店  BK0933  1214.86   40.97  ...   2.33    10     0  广百股份      9.97
3      4  昨日连板_含一字  BK1051  3886.96  110.94  ...  16.32    13     7  广聚能源     10.01
4      5      社区团购  BK0965   890.88   25.13  ...   5.56    36     3  宏辉果蔬      9.95
..   ...       ...     ...      ...     ...  ...    ...   ...   ...   ...       ...
457  458      商汤概念  BK0942  1309.35  -37.91  ...   3.60     1    13  上海临港      0.20
458  459      数字哨兵  BK1084  1581.94  -45.94  ...   6.93     1    11  雄帝科技      1.21
459  460     CPO概念  BK1128  2681.19  -79.49  ...   4.25     2    41  奥飞数据     11.02
460  461   MLOps概念  BK1132  1171.08  -36.92  ...   5.50     1     9  安恒信息      2.16
461  462     国资云概念  BK1008  1769.93  -60.23  ...  11.46     3    16  杭钢股份     10.01
[462 rows x 12 columns]
```

#### 东方财富-概念板块-实时行情

接口: stock_board_concept_spot_em

目标地址: https://quote.eastmoney.com/bk/90.BK0818.html

描述: 东方财富网-行情中心-沪深京板块-概念板块-实时行情

限量: 单次返回指定概念板块的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型      | 描述 |
|-------|---------|----|
| item  | object  | -  |
| value | float64 | -  |

接口示例

```python
import akshare as ak

stock_board_concept_spot_em_df = ak.stock_board_concept_spot_em(symbol="可燃冰")
print(stock_board_concept_spot_em_df)
```

数据示例

```
  item         value
0   最新  1.138760e+03
1   最高  1.149370e+03
2   最低  1.119960e+03
3   开盘  1.119960e+03
4  成交量  4.543638e+06
5  成交额  2.916916e+09
6  换手率  1.190000e+00
7  涨跌额  2.487000e+01
8  涨跌幅  2.230000e+00
9   振幅  2.640000e+00
```

#### 东方财富-成份股

接口: stock_board_concept_cons_em

目标地址: http://quote.eastmoney.com/center/boardlist.html#boards-BK06551

描述: 东方财富-沪深板块-概念板块-板块成份

限量: 单次返回当前时刻所有成份股

输入参数

| 名称     | 类型  | 描述                                                                                              |
|--------|-----|-------------------------------------------------------------------------------------------------|
| symbol | str | symbol="融资融券"; 支持传入板块代码比如：BK0655，可以通过调用 **ak.stock_board_concept_name_em()** 查看东方财富-概念板块的所有行业名称 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 涨跌额    | float64 | -       |
| 成交量    | float64 | 注意单位: 手 |
| 成交额    | float64 | -       |
| 振幅     | float64 | 注意单位: % |
| 最高     | float64 | -       |
| 最低     | float64 | -       |
| 今开     | float64 | -       |
| 昨收     | float64 | -       |
| 换手率    | float64 | 注意单位: % |
| 市盈率-动态 | float64 | -       |
| 市净率    | float64 | -       |

接口示例

```python
import akshare as ak

stock_board_concept_cons_em_df = ak.stock_board_concept_cons_em(symbol="融资融券")
print(stock_board_concept_cons_em_df)
```

数据示例

```
       序号  代码    名称    最新价     涨跌幅  ...     今开      昨收    换手率  市盈率-动态    市净率
0        1  301557   N常友  89.99  211.60  ...  80.23   28.88  83.66   37.89   4.00
1        2  603409   N汇通  50.81  110.13  ...  48.50   24.18  75.43   38.13   4.44
2        3  832149   利尔达  17.29   30.00  ...  13.01   13.30  37.53  -49.74  10.49
3        4  837006  晟楠科技  31.39   29.98  ...  24.24   24.15  47.44 -195.48   8.85
4        5  831167   鑫汇科  33.87   29.97  ...  29.90   26.06  15.82   59.27   6.26
...    ...     ...   ...    ...     ...  ...    ...     ...    ...     ...    ...
3669  3670  603663  三祥新材  24.75  -10.00  ...  24.75   27.50  14.56  102.19   8.02
3670  3671  603906  龙蟠科技  12.13  -10.01  ...  13.01   13.48   9.35  -19.98   2.73
3671  3672  000042  中洲控股   6.71  -10.05  ...   7.45    7.46   3.17   -6.45   0.86
3672  3673  870436  大地电气  25.53  -10.20  ...  26.72   28.43  11.62  -32.46   6.16
3673  3674  920106  林泰新材  93.35  -11.93  ...  98.00  106.00  25.92   34.45   8.44
[3674 rows x 16 columns]
```

#### 东方财富-指数

接口: stock_board_concept_hist_em

目标地址: http://quote.eastmoney.com/bk/90.BK0715.html

描述: 东方财富-沪深板块-概念板块-历史行情数据

限量: 单次返回指定 symbol 和 adjust 的历史数据

输入参数

| 名称         | 类型  | 描述                                                                            |
|------------|-----|-------------------------------------------------------------------------------|
| symbol     | str | symbol="绿色电力"; 可以通过调用 **ak.stock_board_concept_name_em()** 查看东方财富-概念板块的所有概念代码 |
| period     | str | period="daily"; choice of {"daily", "weekly", "monthly"}                      |
| start_date | str | start_date="20220101"                                                         |
| end_date   | str | end_date="20221128"                                                           |
| adjust     | str | adjust=""; choice of {'': 不复权, 默认; "qfq": 前复权, "hfq": 后复权}                    |

输出参数

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 日期  | object  | -       |
| 开盘  | float64 | -       |
| 收盘  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 涨跌额 | float64 | -       |
| 成交量 | int64   | -       |
| 成交额 | float64 | -       |
| 振幅  | float64 | 注意单位: % |
| 换手率 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_board_concept_hist_em_df = ak.stock_board_concept_hist_em(symbol="绿色电力", period="daily", start_date="20220101", end_date="20250227", adjust="")
print(stock_board_concept_hist_em_df)
```

数据示例

```
      日期        开盘      收盘      最高  ...     成交量       成交额    振幅   换手率
0    2022-01-04  1151.48  1143.13  1155.59  ...  57503811  4.502800e+10  1.22  1.42
1    2022-01-05  1141.91  1106.74  1141.91  ...  54859005  4.212589e+10  3.74  1.35
2    2022-01-06  1100.52  1115.16  1122.27  ...  42174619  3.282551e+10  2.25  1.04
3    2022-01-07  1115.49  1089.20  1118.08  ...  44604672  3.478526e+10  2.61  1.10
4    2022-01-10  1083.43  1082.51  1088.21  ...  36095282  2.706919e+10  1.23  0.89
..          ...      ...      ...      ...  ...       ...           ...   ...   ...
756  2025-02-21   985.80   986.71   987.88  ...  42772223  2.825437e+10  0.91  1.06
757  2025-02-24   986.75   996.56   997.62  ...  43658453  3.027056e+10  1.13  1.08
758  2025-02-25   990.79   990.51   997.28  ...  38009252  2.627609e+10  1.00  0.94
759  2025-02-26   991.87  1002.54  1002.57  ...  39788170  2.634823e+10  1.08  0.98
760  2025-02-27  1002.54   995.28  1003.97  ...  38629884  2.581886e+10  1.89  0.95
[761 rows x 11 columns]
```

#### 东方财富-指数-分时

接口: stock_board_concept_hist_min_em

目标地址: http://quote.eastmoney.com/bk/90.BK0715.html

描述: 东方财富-沪深板块-概念板块-分时历史行情数据

限量: 单次返回指定 symbol 和 period 的历史数据

输入参数

| 名称     | 类型  | 描述                                                                           |
|--------|-----|------------------------------------------------------------------------------|
| symbol | str | symbol="长寿药"; 可以通过调用 **ak.stock_board_concept_name_em()** 查看东方财富-概念板块的所有概念代码 |
| period | str | period="5"; choice of {"1", "5", "15", "30", "60"}                           |

输出参数-1分钟

| 名称   | 类型      | 描述 |
|------|---------|----|
| 日期时间 | object  | -  |
| 开盘   | float64 | -  |
| 收盘   | float64 | -  |
| 最高   | float64 | -  |
| 最低   | float64 | -  |
| 成交量  | int64   | -  |
| 成交额  | float64 | -  |
| 最新价  | float64 | -  |

接口示例-1分钟

```python
import akshare as ak

stock_board_concept_hist_min_em_df = ak.stock_board_concept_hist_min_em(symbol="长寿药", period="1")
print(stock_board_concept_hist_min_em_df)
```

数据示例-1分钟

```
     日期时间      开盘      收盘  ...    成交量         成交额      最新价
0    2023-06-29 09:30  982.68  982.68  ...   4108   3937927.0  982.676
1    2023-06-29 09:31  982.68  983.77  ...  17078  17015428.0  982.802
2    2023-06-29 09:32  983.74  982.80  ...   7764   8582773.0  982.864
3    2023-06-29 09:33  982.70  984.44  ...  14440  12411281.0  983.023
4    2023-06-29 09:34  984.15  986.00  ...  11301  10606470.0  983.438
..                ...     ...     ...  ...    ...         ...      ...
236  2023-06-29 14:56  989.46  988.81  ...  10894  10521292.0  989.834
237  2023-06-29 14:57  988.91  988.82  ...  11955  13259401.0  989.828
238  2023-06-29 14:58  988.93  989.12  ...    615    927170.0  989.828
239  2023-06-29 14:59  989.12  989.12  ...      0         0.0  989.828
240  2023-06-29 15:00  989.12  989.03  ...  13803  14810685.0  989.819
[241 rows x 8 columns]
```

输出参数-其他

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 日期时间 | object  | -       |
| 开盘   | float64 | -       |
| 收盘   | float64 | -       |
| 最高   | float64 | -       |
| 最低   | float64 | -       |
| 涨跌幅  | float64 | 注意单位: % |
| 涨跌额  | float64 | -       |
| 成交量  | int64   | -       |
| 成交额  | float64 | -       |
| 振幅   | float64 | 注意单位: % |
| 换手率  | float64 | 注意单位: % |

接口示例-其他

```python
import akshare as ak

stock_board_concept_hist_min_em_df = ak.stock_board_concept_hist_min_em(symbol="长寿药", period="5")
print(stock_board_concept_hist_min_em_df)
```

数据示例-其他

```
        日期时间      开盘      收盘      最高  ...     成交量          成交额    振幅   换手率
0     2023-05-15 09:35  991.11  989.41  991.29  ...  143820  179947201.0  0.67  0.12
1     2023-05-15 09:40  989.10  985.13  991.33  ...  106496  125452569.0  0.67  0.09
2     2023-05-15 09:45  985.10  983.65  985.35  ...   79590  103164009.0  0.22  0.07
3     2023-05-15 09:50  983.35  979.80  983.38  ...   95082  128923628.0  0.41  0.08
4     2023-05-15 09:55  979.61  979.81  981.33  ...   62847   84142446.0  0.25  0.05
                ...     ...     ...     ...  ...     ...          ...   ...   ...
1531  2023-06-29 14:40  989.05  989.00  989.70  ...   26887   29158991.0  0.10  0.02
1532  2023-06-29 14:45  989.00  988.57  989.31  ...   23680   26278319.0  0.12  0.02
1533  2023-06-29 14:50  988.41  988.41  989.17  ...   32369   38999798.0  0.11  0.03
1534  2023-06-29 14:55  988.54  989.29  989.34  ...   43791   49956030.0  0.12  0.04
1535  2023-06-29 15:00  989.46  989.03  989.71  ...   37267   39518548.0  0.10  0.03
[1536 rows x 11 columns]
```

#### 富途牛牛-美股概念-成分股

接口: stock_concept_cons_futu

目标地址: https://www.futunn.com/quote/sparks-us

描述: 富途牛牛-主题投资-概念板块-成分股

限量: 单次返回指定概念板块成分股数据

输入参数

| 名称     | 类型  | 描述                                                      |
|--------|-----|---------------------------------------------------------|
| symbol | str | symbol="特朗普概念股"; choice of {"巴菲特持仓", "佩洛西持仓", "特朗普概念股"} |

输出参数

| 名称   | 类型      | 描述 |
|------|---------|----|
| 代码   | object  | -  |
| 股票名称 | object  | -  |
| 最新价  | float64 | -  |
| 涨跌额  | float64 | -  |
| 涨跌幅  | object  | -  |
| 成交量  | object  | -  |
| 成交额  | object  | -  |

接口示例

```python
import akshare as ak

stock_concept_cons_futu_df = ak.stock_concept_cons_futu(symbol="特朗普概念股")
print(stock_concept_cons_futu_df)
```

数据示例

```
      代码     股票名称     最新价    涨跌额   涨跌幅   成交量       成交额
0   RIOT   Riot Platforms    9.71   0.73  +8.13%  2032.27万     1.94亿
1   CLSK       CleanSpark   10.56   0.76  +7.76%  2610.58万     2.71亿
2   MARA    MARA Holdings   16.24   0.74  +4.77%  3255.48万     5.23亿
3   COIN         Coinbase  193.96   7.69  +4.13%   995.08万    19.27亿
4     GM             通用汽车   53.70   1.90  +3.67%  1153.40万     6.15亿
5   INTC              英特尔   23.32   0.80  +3.55%  5915.33万    13.73亿
6   TSLA              特斯拉  251.44   8.60  +3.54%  6928.25万   174.84亿
7    FLR               福陆   54.59   1.78  +3.37%   257.97万     1.40亿
8    BTU   Peabody Energy   27.20   0.87  +3.30%   290.58万  7876.67万
9    RUM           Rumble    5.95   0.18  +3.12%   295.22万  1744.41万
10   EVR         Evercore  271.84   8.08  +3.06%    42.95万     1.17亿
11    GS               高盛  526.96  15.49  +3.03%   216.99万    11.35亿
12  AMAT             应用材料  186.48   4.66  +2.56%   405.30万     7.54亿
13     F             福特汽车   10.60   0.24  +2.32%  4930.90万     5.20亿
14  TSCO          拖拉机供应公司  276.21   5.85  +2.16%   105.19万     2.89亿
15  MSTR    MicroStrategy  227.80   4.81  +2.16%  1799.39万    42.09亿
16   LNG  Cheniere Energy  192.58   3.81  +2.02%   166.95万     3.21亿
17   HUM              哈门那  260.60   5.05  +1.98%   146.09万     3.80亿
18   UNH             联合健康  567.03   9.26  +1.66%   260.81万    14.75亿
19  STLD   Steel Dynamics  135.65   2.07  +1.55%    92.67万     1.25亿
20    MS            摩根士丹利  117.81   1.67  +1.44%   479.84万     5.64亿
21   NUE             纽柯钢铁  144.61   2.01  +1.41%   122.78万     1.77亿
22   BAC             美国银行   41.88   0.55  +1.33%  3085.14万    12.90亿
23  KLAC               科磊  670.33   6.96  +1.05%   101.66万     6.84亿
24   JPM             摩根大通  221.49   1.71  +0.78%   560.07万    12.38亿
25   WFC             富国银行   64.07   0.37  +0.58%  1344.72万     8.60亿
26   COP             康菲石油  109.21   0.35  +0.32%   600.31万     6.56亿
27   XOM            埃克森美孚  118.96   0.35  +0.30%  1043.67万    12.40亿
28   LLY               礼来  806.06  -0.08  -0.01%   373.33万    30.00亿
29   MRK              默沙东  101.65  -0.22  -0.22%   674.14万     6.84亿
30   OXY             西方石油   50.40  -0.16  -0.32%   799.79万     4.03亿
31    DE             迪尔股份  400.37  -1.31  -0.33%    90.26万     3.61亿
32   CVX              雪佛龙  153.41  -0.56  -0.36%   597.89万     9.18亿
33   TXN             德州仪器  202.46  -1.20  -0.59%   525.32万    10.65亿
34   DJT        特朗普媒体科技集团   33.94  -0.40  -1.16%     1.69亿    63.70亿
35  PHUN         Phunware    6.37  -0.54  -7.81%  1239.14万  8446.10万
```

### 行业板块

#### 同花顺-同花顺行业一览表

接口: stock_board_industry_summary_ths

目标地址: https://q.10jqka.com.cn/thshy/

描述: 同花顺-同花顺行业一览表

限量: 单次返回当前时刻同花顺行业一览表

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述       |
|---------|---------|----------|
| 序号      | int64   | -        |
| 板块      | object  | -        |
| 涨跌幅     | object  | 注意单位: %  |
| 总成交量    | float64 | 注意单位: 万手 |
| 总成交额    | float64 | 注意单位: 亿元 |
| 净流入     | float64 | 注意单位: 亿元 |
| 上涨家数    | float64 | -        |
| 下跌家数    | float64 | -        |
| 均价      | float64 | -        |
| 领涨股     | float64 | -        |
| 领涨股-最新价 | object  | -        |
| 领涨股-涨跌幅 | object  | 注意单位: %  |

接口示例

```python
import akshare as ak

stock_board_industry_summary_ths_df = ak.stock_board_industry_summary_ths()
print(stock_board_industry_summary_ths_df)
```

数据示例

```
   序号     板块  涨跌幅 总成交量  总成交额  ...  下跌家数 均价 领涨股  领涨股-最新价 领涨股-涨跌幅
0    1     传媒  6.17  3737.04  278.19  ...     6   7.44  旗天科技     4.54   20.11
1    2   通信服务  5.28   764.85   63.64  ...     3   8.32  线上线下    27.64   20.02
2    3  计算机应用  4.97  3581.73  423.25  ...     9  11.82  当虹科技    24.72   20.00
3    4  互联网电商  4.46   254.22   15.38  ...     1   6.05   跨境通     1.38    7.81
4    5   其他电子  4.23   444.29   66.62  ...     2  14.99  百邦科技    20.83   17.29
..  ..    ...   ...      ...     ...  ...   ...    ...   ...      ...     ...
71  72  酒店及餐饮  0.70    68.58    7.73  ...     2  11.27  金陵饭店     6.41    2.40
72  73   港口航运  0.68   779.30   49.41  ...     9   6.34  国航远洋     3.88    2.65
73  74     银行  0.50  2349.83  147.23  ...     8   6.27  渝农商行     5.15    1.78
74  75  房地产服务  0.06   267.35   11.01  ...     4   4.12   世联行     1.63    2.52
75  76    贵金属 -0.73   270.28   35.48  ...     8  13.13  西部黄金    11.69    1.39
[76 rows x 12 columns]
```

#### 同花顺-指数

接口: stock_board_industry_index_ths

目标地址: https://q.10jqka.com.cn/thshy/detail/code/881270/

描述: 同花顺-板块-行业板块-指数日频率数据

限量: 单次返回所有日频指数数据

输入参数

| 名称         | 类型  | 描述                                                                      |
|------------|-----|-------------------------------------------------------------------------|
| symbol     | str | symbol="元件"; 可以通过调用 **ak.stock_board_industry_name_ths()** 查看同花顺的所有行业名称 |
| start_date | str | start_date="20200101"; 开始时间                                             |
| end_date   | str | end_date="20211027"; 结束时间                                               |

输出参数

| 名称  | 类型      | 描述  |
|-----|---------|-----|
| 日期  | object  | -   |
| 开盘价 | float64 | -   |
| 最高价 | float64 | -   |
| 最低价 | float64 | -   |
| 收盘价 | float64 | -   |
| 成交量 | int64   | -   |
| 成交额 | float64 | -   |

接口示例

```python
import akshare as ak

stock_board_industry_index_ths_df = ak.stock_board_industry_index_ths(symbol="元件", start_date="20240101", end_date="20240718")
print(stock_board_industry_index_ths_df)
```

数据示例

```
      日期       开盘价      最高价  ...      收盘价        成交量          成交额
0    2024-01-02  8383.164  8434.157  ...  8369.686   378801210  6.537645e+09
1    2024-01-03  8336.885  8348.720  ...  8200.462   397687380  6.681469e+09
2    2024-01-04  8181.495  8195.721  ...  8122.310   320899470  5.532075e+09
3    2024-01-05  8129.346  8193.125  ...  7927.934   400481990  6.126120e+09
4    2024-01-08  7920.488  7936.585  ...  7687.373   425991150  5.932317e+09
..          ...       ...       ...  ...       ...         ...           ...
126  2024-07-12  7963.094  8054.884  ...  7975.548  1195854040  2.105850e+10
127  2024-07-15  7951.025  7975.132  ...  7909.781   992999090  1.719067e+10
128  2024-07-16  7883.687  8248.755  ...  8239.074  1328925150  2.387857e+10
129  2024-07-17  8258.454  8260.010  ...  7946.988  1309263550  2.460237e+10
130  2024-07-18  7810.944  7838.241  ...  7755.153  1248902920  2.164933e+10
[131 rows x 7 columns]
```

#### 东方财富-行业板块

接口: stock_board_industry_name_em

目标地址: https://quote.eastmoney.com/center/boardlist.html#industry_board

描述: 东方财富-沪深京板块-行业板块

限量: 单次返回当前时刻所有行业板的实时行情数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述     |
|----------|---------|--------|
| 排名       | int64   | -      |
| 板块名称     | object  | -      |
| 板块代码     | object  | -      |
| 最新价      | float64 | -      |
| 涨跌额      | float64 | -      |
| 涨跌幅      | float64 | 注意单位：% |
| 总市值      | int64   | -      |
| 换手率      | float64 | 注意单位：% |
| 上涨家数     | int64   | -      |
| 下跌家数     | int64   | -      |
| 领涨股票     | object  | -      |
| 领涨股票-涨跌幅 | float64 | 注意单位：% |

接口示例

```python
import akshare as ak

stock_board_industry_name_em_df = ak.stock_board_industry_name_em()
print(stock_board_industry_name_em_df)
```

数据示例

```
    排名   板块名称 板块代码 最新价     涨跌额  ...   换手率  上涨家数  下跌家数  领涨股票  领涨股票-涨跌幅
0    1   煤炭行业  BK0437  11332.18  518.57  ...  1.34    35     0  云煤能源     10.12
1    2  互联网服务  BK0447  16332.38  673.12  ...  3.38   144     3   信雅达      9.97
2    3  计算机设备  BK0735    867.59   34.86  ...  3.00    50     1  中威电子     20.08
3    4   通信服务  BK0736    793.26   28.90  ...  1.56    40     1   二六三      9.95
4    5   软件开发  BK0737    691.48   24.85  ...  3.02   179     8  国投智能     20.00
..  ..    ...     ...       ...     ...  ...   ...   ...   ...   ...       ...
81  82   航空机场  BK0420   4109.35   11.15  ...  0.43     9     5  南方航空      1.67
82  83   航运港口  BK0450   8431.11   22.32  ...  0.45    19     8   唐山港      2.33
83  84   能源金属  BK1015    399.07    0.36  ...  1.19     6     5  寒锐钴业      1.03
84  85     银行  BK0475   3050.18   -0.42  ...  0.23    16    16  成都银行      1.80
85  86   珠宝首饰  BK0734    351.08   -0.05  ...  1.56     8     9   曼卡龙      3.36
[86 rows x 12 columns]
```

#### 东方财富-行业板块-实时行情

接口: stock_board_industry_spot_em

目标地址: https://quote.eastmoney.com/bk/90.BK1027.html

描述: 东方财富网-沪深板块-行业板块-实时行情

限量: 单次返回指定板块的实时行情数据

输入参数

| 名称     | 类型  | 描述           |
|--------|-----|--------------|
| symbol | str | symbol="小金属" |

输出参数

| 名称    | 类型      | 描述 |
|-------|---------|----|
| item  | object  | -  |
| value | float64 | -  |

接口示例

```python
import akshare as ak

stock_board_industry_spot_em_df = ak.stock_board_industry_spot_em(symbol="小金属")
print(stock_board_industry_spot_em_df)
```

数据示例

```
  item         value
0   最新  1.957370e+03
1   最高  1.990130e+03
2   最低  1.953700e+03
3   开盘  1.964160e+03
4  成交量  1.386981e+07
5  成交额  2.165428e+10
6  换手率  2.970000e+00
7  涨跌额 -1.791000e+01
8  涨跌幅 -9.100000e-01
9   振幅  1.840000e+00
```

#### 东方财富-成份股

接口: stock_board_industry_cons_em

目标地址: https://data.eastmoney.com/bkzj/BK1027.html

描述: 东方财富-沪深板块-行业板块-板块成份

限量: 单次返回指定 symbol 的所有成份股

输入参数

| 名称     | 类型  | 描述                                                                                              |
|--------|-----|-------------------------------------------------------------------------------------------------|
| symbol | str | symbol="小金属"; 支持传入板块代码比如：BK1027，可以通过调用 **ak.stock_board_industry_name_em()** 查看东方财富-行业板块的所有行业代码 |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 代码     | object  | -       |
| 名称     | object  | -       |
| 最新价    | float64 | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 涨跌额    | float64 | -       |
| 成交量    | float64 | 注意单位: 手 |
| 成交额    | float64 | -       |
| 振幅     | float64 | 注意单位: % |
| 最高     | float64 | -       |
| 最低     | float64 | -       |
| 今开     | float64 | -       |
| 昨收     | float64 | -       |
| 换手率    | float64 | 注意单位: % |
| 市盈率-动态 | float64 | -       |
| 市净率    | float64 | -       |

接口示例

```python
import akshare as ak

stock_board_industry_cons_em_df = ak.stock_board_industry_cons_em(symbol="小金属")
print(stock_board_industry_cons_em_df)
```

数据示例

```
    序号  代码   名称    最新价   涨跌幅  ...  今开 昨收    换手率   市盈率-动态 市净率
0    1  300811  铂科新材  53.34  5.17  ...  52.18  50.72   2.93     42.12  5.82
1    2  002842  翔鹭钨业   5.36  3.68  ...   5.16   5.17   5.22    -27.42  1.77
2    3  000962  东方钽业   9.89  3.13  ...   9.58   9.59   1.94     25.59  2.15
3    4  600980  北矿科技  13.21  2.72  ...  12.76  12.86   2.96     26.17  1.95
4    5  300328  宜安科技   4.92  2.71  ...   4.77   4.79   1.37    495.38  3.10
5    6  002428  云南锗业   9.84  2.29  ...   9.63   9.62   2.53   -991.04  4.58
6    7  603993  洛阳钼业   6.13  2.17  ...   5.98   6.00   0.59     40.65  2.48
7    8  600281  华阳新材   2.95  2.08  ...   2.85   2.89   4.80     -9.12  4.74
8    9  688077   大地熊  15.40  2.05  ...  15.17  15.09   1.26    -38.03  1.58
9   10  300127  银河磁体  13.90  1.98  ...  13.56  13.63   2.18     23.07  3.17
10  11  000831  中国稀土  28.14  1.74  ...  26.90  27.66   3.43    178.41  5.76
11  12  000960  锡业股份  13.26  1.69  ...  13.00  13.04   0.82     14.94  1.25
12  13  605376  博迁新材  19.39  1.52  ...  18.99  19.10   0.73    296.20  3.12
13  14  301141  中科磁业  45.27  1.39  ...  43.90  44.65  19.94     78.91  3.19
14  15  600259  广晟有色  29.04  1.29  ...  28.50  28.67   1.07     44.03  2.71
15  16  002149  西部材料  12.68  1.20  ...  12.47  12.53   0.87     30.81  2.14
16  17  600206  有研新材  10.22  1.19  ...  10.11  10.10   0.94     46.45  2.34
17  18  603399  吉翔股份   6.96  1.16  ...   6.83   6.88   2.23    -13.83  1.80
18  19  300835  龙磁科技  23.70  1.11  ...  23.23  23.44   2.30     32.74  2.73
19  20  600111  北方稀土  18.59  1.03  ...  18.28  18.40   0.88     36.51  3.25
20  21  301026  浩通科技  21.66  1.03  ...  21.44  21.44   1.76     17.86  1.70
21  22  002167  东方锆业   4.94  1.02  ...   4.87   4.89   1.46   -380.23  2.53
22  23  600456  宝钛股份  26.40  0.92  ...  26.30  26.16   0.83     17.22  1.93
23  24  300224  正海磁材  10.32  0.58  ...  10.20  10.26   0.66     17.27  2.43
24  25  002378  章源钨业   4.73  0.42  ...   4.71   4.71   0.52     38.67  2.83
25  26  601958  金钼股份   9.86  0.41  ...   9.82   9.82   0.57     10.18  2.07
26  27  600366  宁波韵升   5.86  0.34  ...   5.85   5.84   0.78    -19.79  1.12
27  28  000970  中科三环   8.95  0.34  ...   8.89   8.92   1.04     36.51  1.67
28  29  600549  厦门钨业  16.19  0.31  ...  16.05  16.14   0.68     14.36  2.05
29  30  600392  盛和资源   8.93  0.22  ...   8.87   8.91   1.01     74.21  1.77
30  31  600459  贵研铂业  13.54  0.22  ...  13.50  13.51   0.44     19.26  1.63
31  32  600615  丰华股份   9.44  0.21  ...   9.35   9.42   2.03 -20739.46  2.80
32  33  000657  中钨高新   8.08  0.12  ...   8.02   8.07   0.89     26.84  1.97
33  34  300748  金力永磁  16.25  0.00  ...  16.10  16.25   1.02     33.16  3.09
34  35  000795   英洛华   5.04  0.00  ...   5.04   5.04   1.00     52.08  2.16
35  36  688190  云路股份  61.66 -0.48  ...  61.99  61.96   0.67     23.29  3.38
36  37  002738  中矿资源  32.29 -1.10  ...  32.10  32.65   2.25      8.54  1.96
[37 rows x 16 columns]
```

#### 东方财富-指数-日频

接口: stock_board_industry_hist_em

目标地址: https://quote.eastmoney.com/bk/90.BK1027.html

描述: 东方财富-沪深板块-行业板块-历史行情数据

限量: 单次返回指定 symbol 和 adjust 的所有历史数据

输入参数

| 名称         | 类型  | 描述                                                                            |
|------------|-----|-------------------------------------------------------------------------------|
| symbol     | str | symbol="小金属"; 可以通过调用 **ak.stock_board_industry_name_em()** 查看东方财富-行业板块的所有行业代码 |
| start_date | str | start_date="20211201";                                                        |
| end_date   | str | end_date="20220401";                                                          |
| period     | str | period="日k"; 周期; choice of {"日k", "周k", "月k"}                                 |
| adjust     | str | adjust=""; choice of {'': 不复权, 默认; "qfq": 前复权, "hfq": 后复权}                    |

输出参数

| 名称  | 类型      | 描述      |
|-----|---------|---------|
| 日期  | object  | -       |
| 开盘  | float64 | -       |
| 收盘  | float64 | -       |
| 最高  | float64 | -       |
| 最低  | float64 | -       |
| 涨跌幅 | float64 | 注意单位: % |
| 涨跌额 | float64 | -       |
| 成交量 | int64   | -       |
| 成交额 | float64 | -       |
| 振幅  | float64 | 注意单位: % |
| 换手率 | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_board_industry_hist_em_df = ak.stock_board_industry_hist_em(symbol="小金属", start_date="20211201", end_date="20240222", period="日k", adjust="")
print(stock_board_industry_hist_em_df)
```

数据示例

```
        日期      开盘      收盘     最高  ...       成交量        成交额    振幅   换手率
0    2021-12-01  1915.46  1915.23  1919.43  ...  17207231  3.134633e+10  1.77  3.68
1    2021-12-02  1909.70  1925.26  1947.38  ...  17723129  3.914958e+10  3.32  3.79
2    2021-12-03  1912.86  1923.48  1930.06  ...  14027893  3.125242e+10  2.52  3.00
3    2021-12-06  1924.43  1901.92  1940.57  ...  15612344  3.377371e+10  2.07  3.34
4    2021-12-07  1914.66  1851.77  1917.07  ...  14501660  3.131906e+10  4.61  3.10
..          ...      ...      ...      ...  ...       ...           ...   ...   ...
534  2024-02-08  1234.37  1271.41  1284.17  ...   9309314  1.033029e+10  5.01  1.99
535  2024-02-19  1271.91  1285.67  1294.59  ...   7797152  9.248781e+09  2.22  1.67
536  2024-02-20  1279.11  1285.77  1288.37  ...   5182130  6.270108e+09  1.86  1.11
537  2024-02-21  1274.53  1303.25  1336.49  ...   6059774  7.920041e+09  5.11  1.30
538  2024-02-22  1298.16  1320.27  1320.27  ...   4594781  5.684279e+09  1.94  0.98
[539 rows x 11 columns]
```

#### 东方财富-指数-分时

接口: stock_board_industry_hist_min_em

目标地址: http://quote.eastmoney.com/bk/90.BK1027.html

描述: 东方财富-沪深板块-行业板块-分时历史行情数据

限量: 单次返回指定 symbol 和 period 的所有历史数据

输入参数

| 名称     | 类型  | 描述                                                                            |
|--------|-----|-------------------------------------------------------------------------------|
| symbol | str | symbol="小金属"; 可以通过调用 **ak.stock_board_industry_name_em()** 查看东方财富-行业板块的所有行业代码 |
| period | str | period=""; choice of {"1", "5", "15", "30", "60"}                             |

输出参数-1分钟

| 名称   | 类型      | 描述 |
|------|---------|----|
| 日期时间 | object  | -  |
| 开盘   | float64 | -  |
| 收盘   | float64 | -  |
| 最高   | float64 | -  |
| 最低   | float64 | -  |
| 成交量  | int64   | -  |
| 成交额  | float64 | -  |
| 最新价  | float64 | -  |

接口示例-1分钟

```python
import akshare as ak

stock_board_industry_hist_min_em_df = ak.stock_board_industry_hist_min_em(symbol="小金属", period="1")
print(stock_board_industry_hist_min_em_df)
```

数据示例-1分钟

```
         日期时间       开盘       收盘  ...    成交量          成交额       最新价
0    2023-06-27 09:30  1560.07  1560.07  ...  19551   27815455.0  1560.071
1    2023-06-27 09:31  1560.07  1560.12  ...  82736  120214342.0  1559.491
2    2023-06-27 09:32  1560.22  1563.32  ...  43374   70802338.0  1560.403
3    2023-06-27 09:33  1563.78  1564.85  ...  46300   71769286.0  1561.553
4    2023-06-27 09:34  1564.86  1566.13  ...  55577   77527242.0  1562.524
..                ...      ...      ...  ...    ...          ...       ...
236  2023-06-27 14:56  1581.26  1581.44  ...  21701   30935379.0  1572.689
237  2023-06-27 14:57  1581.79  1581.75  ...  30772   40940268.0  1572.758
238  2023-06-27 14:58  1581.62  1581.64  ...   2585    2789725.0  1572.763
239  2023-06-27 14:59  1581.64  1581.64  ...      0          0.0  1572.763
240  2023-06-27 15:00  1581.64  1581.39  ...  52691   74537630.0  1572.874
[241 rows x 8 columns]
```

输出参数-其他

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 日期时间 | object  | -       |
| 开盘   | float64 | -       |
| 收盘   | float64 | -       |
| 最高   | float64 | -       |
| 最低   | float64 | -       |
| 涨跌幅  | float64 | 注意单位: % |
| 涨跌额  | float64 | -       |
| 成交量  | int64   | -       |
| 成交额  | float64 | -       |
| 振幅   | float64 | 注意单位: % |
| 换手率  | float64 | 注意单位: % |

接口示例-其他

```python
import akshare as ak

stock_board_industry_hist_min_em_df = ak.stock_board_industry_hist_min_em(symbol="小金属", period="5")
print(stock_board_industry_hist_min_em_df)
```

数据示例-其他

```
                  日期时间       开盘       收盘  ...          成交额    振幅   换手率
0     2023-05-11 09:35  1601.02  1599.16  ...  472122686.0  0.19  0.04
1     2023-05-11 09:40  1598.86  1595.23  ...  307441109.0  0.39  0.04
2     2023-05-11 09:45  1595.47  1592.76  ...  231066676.0  0.27  0.03
3     2023-05-11 09:50  1592.91  1596.50  ...  165816236.0  0.24  0.02
4     2023-05-11 09:55  1596.65  1598.37  ...  128737348.0  0.16  0.02
                ...      ...      ...  ...          ...   ...   ...
1531  2023-06-27 14:40  1580.65  1581.91  ...  117492687.0  0.09  0.03
1532  2023-06-27 14:45  1581.77  1581.88  ...  106035779.0  0.05  0.02
1533  2023-06-27 14:50  1581.86  1581.49  ...  129199767.0  0.07  0.02
1534  2023-06-27 14:55  1581.20  1580.99  ...  147442240.0  0.07  0.02
1535  2023-06-27 15:00  1581.26  1581.39  ...  149203002.0  0.06  0.02
[1536 rows x 11 columns]
```

### 股票热度

#### 股票热度-雪球

##### 关注排行榜

接口: stock_hot_follow_xq

目标地址: https://xueqiu.com/hq

描述: 雪球-沪深股市-热度排行榜-关注排行榜

限量: 单次返回指定 symbol 的排行数据

输入参数

| 名称     | 类型  | 描述                                      |
|--------|-----|-----------------------------------------|
| symbol | str | symbol="最热门"; choice of {"本周新增", "最热门"} |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 股票代码 | object  | -       |
| 股票简称 | object  | -       |
| 关注   | float64 | -       |
| 最新价  | float64 | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_hot_follow_xq_df = ak.stock_hot_follow_xq(symbol="最热门")
print(stock_hot_follow_xq_df)
```

数据示例

```
     股票代码  股票简称         关注      最新价
0     SH600519  贵州茅台  2763065.0  1663.36
1     SH601318  中国平安  2321952.0    38.97
2     SH600036  招商银行  2039407.0    28.29
3     SZ000651  格力电器  1768422.0    32.98
4     SZ002594   比亚迪  1494585.0   192.68
        ...   ...        ...      ...
5420  BJ836957  汉维科技      180.0     9.42
5421  BJ836942  恒立钻具      178.0    13.07
5422  BJ836419  万德股份      176.0    11.15
5423  BJ873690   N捷众       30.0    21.01
5424  SZ300307  慈星股份        NaN     6.20
[5425 rows x 4 columns]
```

##### 讨论排行榜

接口: stock_hot_tweet_xq

目标地址: https://xueqiu.com/hq

描述: 雪球-沪深股市-热度排行榜-讨论排行榜

限量: 单次返回指定 symbol 的排行数据

输入参数

| 名称     | 类型  | 描述                                      |
|--------|-----|-----------------------------------------|
| symbol | str | symbol="最热门"; choice of {"本周新增", "最热门"} |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 股票代码 | object  | -       |
| 股票简称 | object  | -       |
| 关注   | float64 | -       |
| 最新价  | float64 | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_hot_tweet_xq_df = ak.stock_hot_tweet_xq(symbol="最热门")
print(stock_hot_tweet_xq_df)
```

数据示例

```
      股票代码  股票简称     关注       最新价
0     SZ002594   比亚迪  89745   192.680
1     SH600519  贵州茅台  85990  1663.360
2     SZ300750  宁德时代  52705   150.900
3     SZ000977  浪潮信息  50664    30.160
4     SZ002229  鸿博股份  46794    24.690
        ...   ...    ...       ...
5420  SH900913  国新B股      0     0.246
5421  SH900901  云赛B股      0     0.496
5422  BJ870508  丰安股份      0    12.530
5423  BJ836149  旭杰科技      0     7.970
5424  BJ831175  派诺科技      0    22.490
[5425 rows x 4 columns]
```

##### 交易排行榜

接口: stock_hot_deal_xq

目标地址: https://xueqiu.com/hq

描述: 雪球-沪深股市-热度排行榜-交易排行榜

限量: 单次返回指定 symbol 的排行数据

输入参数

| 名称     | 类型  | 描述                                      |
|--------|-----|-----------------------------------------|
| symbol | str | symbol="最热门"; choice of {"本周新增", "最热门"} |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 股票代码 | object  | -       |
| 股票简称 | object  | -       |
| 关注   | float64 | -       |
| 最新价  | float64 | 注意单位: 元 |

接口示例

```python
import akshare as ak

stock_hot_deal_xq_df = ak.stock_hot_deal_xq(symbol="最热门")
print(stock_hot_deal_xq_df)
```

数据示例

```
     股票代码  股票简称   关注      最新价
0     SH601318  中国平安  304    38.97
1     SZ002229  鸿博股份  258    24.69
2     SH600519  贵州茅台  257  1663.36
3     SZ002594   比亚迪  257   192.68
4     SH600880  博瑞传播  231     4.90
        ...   ...  ...      ...
5420  SZ300932  三友联众    0    15.95
5421  SZ300956  英力股份    0    18.21
5422  SH688618  三旺通信    0    52.82
5423  BJ836149  旭杰科技    0     7.97
5424  SH688655   迅捷兴    0    14.64
[5425 rows x 4 columns]
```

#### 股票热度-东财

##### 人气榜-A股

接口: stock_hot_rank_em

目标地址: http://guba.eastmoney.com/rank/

描述: 东方财富网站-股票热度

限量: 单次返回当前交易日前 100 个股票的人气排名数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 当前排名 | int64   | -       |
| 代码   | object  | -       |
| 股票名称 | object  | -       |
| 最新价  | float64 | -       |
| 涨跌额  | float64 | -       |
| 涨跌幅  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_hot_rank_em_df = ak.stock_hot_rank_em()
print(stock_hot_rank_em_df)
```

数据示例

```
    当前排名   代码  股票名称    最新价       涨跌额    涨跌幅
0      1  SZ002261  拓维信息  18.83  0.090384   0.48
1      2  SH603000   人民网  34.35  2.881965   8.39
2      3  SH601127   赛力斯  33.72  3.378744  10.02
3      4  SZ002229  鸿博股份  35.09  0.021054   0.06
4      5  SZ000936  华西股份  10.41  1.045164  10.04
..   ...       ...   ...    ...       ...    ...
95    96  SZ002585  双星新材   9.69  0.968031   9.99
96    97  SZ002670  国盛金控   8.46  0.520290   6.15
97    98  SZ300116   保力新   1.53  0.191250  12.50
98    99  SH603533  掌阅科技  33.14 -3.231150  -9.75
99   100  SZ301368  丰立智能  58.32  3.563352   6.11
[100 rows x 6 columns]
```

##### 飙升榜-A股

接口: stock_hot_up_em

目标地址: http://guba.eastmoney.com/rank/

描述: 东方财富-个股人气榜-飙升榜

限量: 单次返回当前交易日前 100 个股票的飙升榜排名数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称      | 类型      | 描述      |
|---------|---------|---------|
| 排名较昨日变动 | int64   | -       |
| 当前排名    | int64   | -       |
| 代码      | object  | -       |
| 股票名称    | object  | -       |
| 最新价     | float64 | -       |
| 涨跌额     | float64 | -       |
| 涨跌幅     | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_hot_up_em_df = ak.stock_hot_up_em()
print(stock_hot_up_em_df)
```

数据示例

```
    排名较昨日变动  当前排名    代码  股票名称   最新价  涨跌额    涨跌幅
0      4491   484  SH688226  威腾电气  18.89  2.349916  12.44
1      4240   454  SZ301079  邵阳液压  19.42  1.648758   8.49
2      4029   645  SH688698  伟创电气  31.45  2.858805   9.09
3      4023   366  SZ002523  天桥起重   2.87  0.180810   6.30
4      4019   568  SZ300402  宝色股份  20.50  1.377600   6.72
..      ...   ...       ...   ...    ...       ...    ...
95     2261   504  SZ000795   英洛华   6.31  0.217064   3.44
96     2253  2816  SZ430425  乐创技术  13.78  0.826800   6.00
97     2234  1515  SZ300861  美畅股份  41.60  2.025920   4.87
98     2230   367  SH688003  天准科技  46.06  4.900784  10.64
99     2230  2825  SZ835985  海泰新能   6.56  0.217136   3.31
[100 rows x 7 columns]
```

##### 人气榜-港股

接口: stock_hk_hot_rank_em

目标地址: https://guba.eastmoney.com/rank/

描述: 东方财富-个股人气榜-人气榜-港股市场

限量: 单次返回当前交易日前 100 个股票的人气排名数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 当前排名 | int64   | -       |
| 代码   | object  | -       |
| 股票名称 | object  | -       |
| 最新价  | float64 | -       |
| 涨跌幅  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_hk_hot_rank_em_df = ak.stock_hk_hot_rank_em()
print(stock_hk_hot_rank_em_df)
```

数据示例

```
    当前排名     代码    股票名称      最新价   涨跌幅
0      1  00700    腾讯控股  376.800  0.32
1      2  01810  小米集团-W   12.440  1.80
2      3  02158    医渡科技    8.350  3.73
3      4  03690    美团-W  140.200 -0.64
4      5  09955    智云健康    7.880  4.51
..   ...    ...     ...      ...   ...
95    96  01951    锦欣生殖    5.380 -1.28
96    97  01898    中煤能源    5.910 -8.51
97    98  02768  佳源国际控股    0.179 -2.19
98    99  00388   香港交易所  342.000  0.00
99   100  01816   中广核电力    1.900 -1.55
```

#### 历史趋势及粉丝特征

##### A股

接口: stock_hot_rank_detail_em

目标地址: http://guba.eastmoney.com/rank/stock?code=000665

描述: 东方财富网-股票热度-历史趋势及粉丝特征

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SZ000665" |

输出参数

| 名称   | 类型      | 描述  |
|------|---------|-----|
| 时间   | object  | -   |
| 排名   | int64   | -   |
| 证券代码 | object  | -   |
| 新晋粉丝 | float64 | -   |
| 铁杆粉丝 | float64 | -   |

接口示例

```python
import akshare as ak

stock_hot_rank_detail_em_df = ak.stock_hot_rank_detail_em(symbol="SZ000665")
print(stock_hot_rank_detail_em_df)
```

数据示例

```
         时间    排名   证券代码   新晋粉丝 铁杆粉丝
0    2024-07-24  4973  SZ871245  0.4259  0.5741
1    2024-07-25  2231  SZ871245  0.5476  0.4524
2    2024-07-26  2870  SZ871245  0.6578  0.3422
3    2024-07-27  4862  SZ871245  0.6669  0.3331
4    2024-07-28  4452  SZ871245  0.6725  0.3275
..          ...   ...       ...     ...     ...
361  2025-07-20  5052  SZ871245  0.5067  0.4933
362  2025-07-21   459  SZ871245  0.8479  0.1521
363  2025-07-22   548  SZ871245  0.9044  0.0956
364  2025-07-23  1610  SZ871245  0.9134  0.0866
365  2025-07-24  3646  SZ871245  0.9175  0.0825
[366 rows x 5 columns]
```

##### 港股

接口: stock_hk_hot_rank_detail_em

目标地址: https://guba.eastmoney.com/rank/stock?code=HK_00700

描述: 东方财富网-股票热度-历史趋势

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="00700" |

输出参数

| 名称   | 类型      | 描述  |
|------|---------|-----|
| 时间   | object  | -   |
| 排名   | int64   | -   |
| 证券代码 | object  | -   |

接口示例

```python
import akshare as ak

stock_hk_hot_rank_detail_em_df = ak.stock_hk_hot_rank_detail_em(symbol="00700")
print(stock_hk_hot_rank_detail_em_df)
```

数据示例

```
      时间    排名   证券代码
0    2022-11-26   1  00700
1    2022-11-27   2  00700
2    2022-11-28   1  00700
3    2022-11-29   1  00700
4    2022-11-30   1  00700
..          ...  ..    ...
115  2023-03-21   1  00700
116  2023-03-22   1  00700
117  2023-03-23   1  00700
118  2023-03-24   1  00700
119  2023-03-25   1  00700
```

#### 互动平台

##### 互动易-提问

接口: stock_irm_cninfo

目标地址: https://irm.cninfo.com.cn/

描述: 互动易-提问

限量: 单次返回近期 10000 条提问数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="002594";  |

输出参数

| 名称    | 类型             | 描述 |
|-------|----------------|----|
| 股票代码  | object         | -  |
| 公司简称  | object         | -  |
| 行业    | object         | -  |
| 行业代码  | object         | -  |
| 问题    | object         | -  |
| 提问者   | object         | -  |
| 来源    | object         | -  |
| 提问时间  | datetime64[ns] | -  |
| 更新时间  | datetime64[ns] | -  |
| 提问者编号 | object         | -  |
| 问题编号  | object         | -  |
| 回答ID  | object         | -  |
| 回答内容  | object         | -  |
| 回答者   | object         | -  |

接口示例

```python
import akshare as ak

stock_irm_cninfo_df = ak.stock_irm_cninfo(symbol="002594")
print(stock_irm_cninfo_df)
```

数据示例

```
      股票代码 公司简称  ...                                               回答内容   回答者
0     002594  比亚迪  ...                                               None  None
1     002594  比亚迪  ...                                               None  None
2     002594  比亚迪  ...                                               None  None
3     002594  比亚迪  ...                                               None  None
4     002594  比亚迪  ...                                               None  None
...      ...  ...  ...                                                ...   ...
9995  002594  比亚迪  ...                                            感谢您的建议！   比亚迪
9996  002594  比亚迪  ...             感谢对公司的关注，未来公司将综合各个基地的产能协调生产，保证产能充足，谢谢！   比亚迪
9997  002594  比亚迪  ...  目前电动叉车项目还在产品推广的初期，市场尚不够熟悉，未来公司会通过创新销售方式、拓宽销售渠道...   比亚迪
9998  002594  比亚迪  ...                                         请见之前回复，谢谢！   比亚迪
9999  002594  比亚迪  ...                                                谢谢！   比亚迪
[10000 rows x 14 columns]
```

##### 互动易-回答

接口: stock_irm_ans_cninfo

目标地址: https://irm.cninfo.com.cn/

描述: 互动易-回答

限量: 单次返回指定 symbol 的回答数据

输入参数

| 名称     | 类型  | 描述                                                               |
|--------|-----|------------------------------------------------------------------|
| symbol | str | symbol="1495108801386602496"; 通过 ak.stock_irm_cninfo 来获取具体的提问者编号 |

输出参数

| 名称   | 类型             | 描述 |
|------|----------------|----|
| 股票代码 | object         | -  |
| 公司简称 | object         | -  |
| 问题   | object         | -  |
| 回答内容 | object         | -  |
| 提问者  | object         | -  |
| 提问时间 | datetime64[ns] | -  |
| 回答时间 | datetime64[ns] | -  |

接口示例

```python
import akshare as ak

stock_irm_ans_cninfo_df = ak.stock_irm_ans_cninfo(symbol="1495108801386602496")
print(stock_irm_ans_cninfo_df)
```

数据示例

```
     股票代码 公司简称  ...                提问时间                回答时间
0  002594  比亚迪  ... 2023-07-08 04:12:53 2023-07-12 00:34:31
[1 rows x 7 columns]
```

##### 上证e互动

接口: stock_sns_sseinfo

目标地址: https://sns.sseinfo.com/company.do?uid=65

描述: 上证e互动-提问与回答

限量: 单次返回指定 symbol 的提问与回答数据

输入参数

| 名称     | 类型  | 描述                    |
|--------|-----|-----------------------|
| symbol | str | symbol="603119"; 股票代码 |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 股票代码 | object | -  |
| 公司简称 | object | -  |
| 问题   | object | -  |
| 回答   | object | -  |
| 问题时间 | object | -  |
| 回答时间 | object | -  |
| 问题来源 | object | -  |
| 回答来源 | object | -  |
| 用户名  | object | -  |

接口示例

```python
import akshare as ak

stock_sns_sseinfo_df = ak.stock_sns_sseinfo(symbol="603119")
print(stock_sns_sseinfo_df)
```

数据示例

```
       股票代码  公司简称  ... 回答来源              用户名
0    603119  浙江荣泰  ...   网站             YiQi
1    603119  浙江荣泰  ...   网站             YiQi
2    603119  浙江荣泰  ...   网站            春暖花未开
3    603119  浙江荣泰  ...   网站  guest_zT0NfLL0L
4    603119  浙江荣泰  ...   网站              Kgk
..      ...   ...  ...  ...              ...
150  603119  浙江荣泰  ...   网站              龙投宝
151  603119  浙江荣泰  ...   网站       SummerIcey
152  603119  浙江荣泰  ...   网站  guest_Usy8rr8Ik
153  603119  浙江荣泰  ...   网站            赵子龙常山
154  603119  浙江荣泰  ...   网站            赵子龙常山
[155 rows x 9 columns]
```

#### 个股人气榜-实时变动

##### A股

接口: stock_hot_rank_detail_realtime_em

目标地址: http://guba.eastmoney.com/rank/stock?code=000665

描述: 东方财富网-个股人气榜-实时变动

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SZ000665" |

输出参数

| 名称   | 类型      | 描述  |
|------|---------|-----|
| 时间   | object  | -   |
| 排名   | int64   | -   |

接口示例

```python
import akshare as ak

stock_hot_rank_detail_realtime_em_df = ak.stock_hot_rank_detail_realtime_em(symbol="SZ000665")
print(stock_hot_rank_detail_realtime_em_df)
```

数据示例

```
                  时间   排名
0   2022-02-26 00:00:00  170
1   2022-02-26 00:10:00  170
2   2022-02-26 00:20:00  170
3   2022-02-26 00:30:00  170
4   2022-02-26 00:40:00  171
..                  ...  ...
94  2022-02-26 15:40:00  225
95  2022-02-26 15:50:00  224
96  2022-02-26 16:00:00  222
97  2022-02-26 16:10:00  215
98  2022-02-26 16:20:00  211
```

##### 港股

接口: stock_hk_hot_rank_detail_realtime_em

目标地址: https://guba.eastmoney.com/rank/stock?code=HK_00700

描述: 东方财富网-个股人气榜-实时变动

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="00700" |

输出参数

| 名称   | 类型      | 描述  |
|------|---------|-----|
| 时间   | object  | -   |
| 排名   | int64   | -   |

接口示例

```python
import akshare as ak

stock_hk_hot_rank_detail_realtime_em_df = ak.stock_hk_hot_rank_detail_realtime_em(symbol="00700")
print(stock_hk_hot_rank_detail_realtime_em_df)
```

数据示例

```
                      时间  排名
0    2023-03-25 00:00:00   1
1    2023-03-25 00:10:00   1
2    2023-03-25 00:20:00   1
3    2023-03-25 00:30:00   1
4    2023-03-25 00:40:00   1
..                   ...  ..
137  2023-03-25 22:50:00   1
138  2023-03-25 23:00:00   1
139  2023-03-25 23:10:00   1
140  2023-03-25 23:20:00   1
141  2023-03-25 23:30:00   1
```

#### 热门关键词

接口: stock_hot_keyword_em

目标地址: http://guba.eastmoney.com/rank/stock?code=000665

描述: 东方财富-个股人气榜-热门关键词

限量: 单次返回指定 symbol 的最近交易日时点数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SZ000665" |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 时间   | object | -   |
| 股票代码 | object | -   |
| 概念名称 | object | -   |
| 概念代码 | object | -   |
| 热度   | int64  | -   |

接口示例

```python
import akshare as ak

stock_hot_keyword_em_df = ak.stock_hot_keyword_em(symbol="SZ000665")
print(stock_hot_keyword_em_df)
```

数据示例

```
               时间      股票代码   概念名称    概念代码  热度
0  2022-02-28 12:00:00  SZ000665  元宇宙概念  BK1009  2138
1  2022-02-28 12:00:00  SZ000665     广电  BK0904  1082
2  2022-02-28 12:00:00  SZ000665    云计算  BK0579   411
3  2022-02-28 12:00:00  SZ000665   虚拟现实  BK0722   152
4  2022-02-28 12:00:00  SZ000665   彩票概念  BK0671   131
5  2022-02-28 12:00:00  SZ000665   转债标的  BK0528    51
6  2022-02-28 12:00:00  SZ000665   华为概念  BK0854    36
7  2022-02-28 12:00:00  SZ000665   智慧城市  BK0628    31
8  2022-02-28 12:00:00  SZ000665   预盈预增  BK0571    13
9  2022-02-28 12:00:00  SZ000665   超清视频  BK0859     1
```

#### 内部交易

接口: stock_inner_trade_xq

目标地址: https://xueqiu.com/hq/insider

描述: 雪球-行情中心-沪深股市-内部交易

限量: 单次返回所有历史数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| 股票代码   | object  | -   |
| 股票名称   | object  | -   |
| 变动日期   | object  | -   |
| 变动人    | object  | -   |
| 变动股数   | int64   | -   |
| 成交均价   | float64 | -   |
| 变动后持股数 | float64 | -   |
| 与董监高关系 | object  | -   |
| 董监高职务  | object  | -   |

接口示例

```python
import akshare as ak

stock_inner_trade_xq_df = ak.stock_inner_trade_xq()
print(stock_inner_trade_xq_df)
```

数据示例

```
        股票代码  股票名称   变动日期  变动人  ...   成交均价    变动后持股数 与董监高关系  董监高职务
0      SH688787  海天瑞声  2024-11-04  吕思遥  ...  63.00     42850.0      本人      董事
1      SH688550  瑞联新材  2024-11-04   胡湛  ...  31.11     39014.0      本人  高级管理人员
2      SH600868  梅雁吉祥  2024-11-04   李明  ...   2.39  26650702.0      本人      董事
3      SZ301206  三元生物  2024-11-01  聂在建  ...  27.71  93698700.0      本人      董事
4      SZ300963  中洲特材  2024-11-01   韩明  ...   9.69  32119500.0      本人      董事
...         ...   ...         ...  ...  ...    ...         ...     ...     ...
23300  SH601028  玉龙股份  2023-01-03   张鹏  ...  11.69      7000.0      本人      董事
23301  SH600875  东方电气  2023-01-03  胡修奎  ...  22.60     65000.0      本人  高级管理人员
23302  SH600089  特变电工  2023-01-03  胡有成  ...   6.51    700000.0      本人  高级管理人员
23303  SH600079  人福医药  2023-01-03  王学海  ...  24.00   4615346.0      本人      董事
23304  SH600079  人福医药  2023-01-03  徐华斌  ...  25.04  14300716.0
[23305 rows x 9 columns]
```

#### 个股人气榜-最新排名

##### A股

接口: stock_hot_rank_latest_em

目标地址: http://guba.eastmoney.com/rank/stock?code=000665

描述: 东方财富-个股人气榜-最新排名

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SZ000665" |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_hot_rank_latest_em_df = ak.stock_hot_rank_latest_em(symbol="SZ000665")
print(stock_hot_rank_latest_em_df)
```

数据示例

```
                 item                value
0          marketType               000001
1      marketAllCount                 4735
2            calcTime  2022-02-28 13:30:00
3           innerCode               000665
4     srcSecurityCode             SZ000665
5                rank                  155
6          rankChange                   -2
7       hisRankChange                  141
8  hisRankChange_rank                 1654
9                flag                    0
```

##### 港股

接口: stock_hk_hot_rank_latest_em

目标地址: https://guba.eastmoney.com/rank/stock?code=HK_00700

描述: 东方财富-个股人气榜-最新排名

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述             |
|--------|-----|----------------|
| symbol | str | symbol="00700" |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_hk_hot_rank_latest_em_df = ak.stock_hk_hot_rank_latest_em(symbol="00700")
print(stock_hk_hot_rank_latest_em_df)
```

数据示例

```
                 item                value
0          marketType               000003
1      marketAllCount                 2613
2            calcTime  2023-03-25 23:30:00
3           innerCode              00700_2
4     srcSecurityCode             HK|00700
5                rank                    1
6          rankChange                    0
7       hisRankChange                    0
8  hisRankChange_rank                 1253
9                flag                    0
```

#### 热搜股票

接口: stock_hot_search_baidu

目标地址: https://gushitong.baidu.com/expressnews

描述: 百度股市通-热搜股票

限量: 单次返回指定 symbol, date 和 time 的热搜股票数据

输入参数

| 名称     | 类型  | 描述                                              |
|--------|-----|-------------------------------------------------|
| symbol | str | symbol="A股"; choice of {"全部", "A股", "港股", "美股"} |
| date   | str | date="20250616"                                 |
| time   | str | time="今日"; choice of {"今日", "1小时"}              |

输出参数

| 名称    | 类型     | 描述 |
|-------|--------|----|
| 名称/代码 | object | -  |
| 涨跌幅   | object | -  |
| 综合热度  | int64  | -  |

接口示例

```python
import akshare as ak

stock_hot_search_baidu_df = ak.stock_hot_search_baidu(symbol="A股", date="20250616", time="今日")
print(stock_hot_search_baidu_df)
```

数据示例

```
    名称/代码      涨跌幅    综合热度
0    融发核电   +1.72%  866000
1    永安药业   +7.13%  865000
2   中毅达    +10.00%  857000
3    恒宝股份   +9.98%  828000
4    利民股份   +9.98%  788000
5    中超控股   -1.34%  786000
6    百利电气   -4.24%  734000
7     比亚迪   -0.45%  716000
8    贵州茅台   -0.33%  569000
9    元隆雅图   +9.99%  561000
10   宁波韵升   +4.04%  555000
11   长安汽车   -1.47%  553000
```

#### 相关股票

接口: stock_hot_rank_relate_em

目标地址: http://guba.eastmoney.com/rank/stock?code=000665

描述: 东方财富-个股人气榜-相关股票

限量: 单次返回指定 symbol 的股票近期历史数据

输入参数

| 名称     | 类型  | 描述                |
|--------|-----|-------------------|
| symbol | str | symbol="SZ000665" |

输出参数

| 名称     | 类型      | 描述  |
|--------|---------|-----|
| 时间     | object  | -   |
| 股票代码   | object  | -   |
| 相关股票代码 | object  | -   |
| 涨跌幅    | float64 | -   |

接口示例

```python
import akshare as ak

stock_hot_rank_relate_em_df = ak.stock_hot_rank_relate_em(symbol="SZ000665")
print(stock_hot_rank_relate_em_df)
```

数据示例

```
              时间       股票代码   相关股票代码 涨跌幅
0  2024-09-30 20:00:00  SZ000665  SZ300059  6.18
1  2024-09-30 20:00:00  SZ000665  SZ000158  4.46
2  2024-09-30 20:00:00  SZ000665  SH600519  2.71
3  2024-09-30 20:00:00  SZ000665  SH601162  2.29
4  2024-09-30 20:00:00  SZ000665  SZ000595  2.01
5  2024-09-30 20:00:00  SZ000665  SZ002717  1.72
6  2024-09-30 20:00:00  SZ000665  SZ300085  1.72
7  2024-09-30 20:00:00  SZ000665  SZ301551  1.59
8  2024-09-30 20:00:00  SZ000665  SZ000002  1.54
9  2024-09-30 20:00:00  SZ000665  SZ000980  1.50
```

### 盘口异动

接口: stock_changes_em

目标地址: http://quote.eastmoney.com/changes/

描述: 东方财富-行情中心-盘口异动数据

限量: 单次指定 symbol 的最近交易日的盘口异动数据

输入参数

| 名称     | 类型  | 描述                                                                                                                                                                                                                    |
|--------|-----|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="大笔买入"; choice of {'火箭发射', '快速反弹', '大笔买入', '封涨停板', '打开跌停板', '有大买盘', '竞价上涨', '高开5日线', '向上缺口', '60日新高', '60日大幅上涨', '加速下跌', '高台跳水', '大笔卖出', '封跌停板', '打开涨停板', '有大卖盘', '竞价下跌', '低开5日线', '向下缺口', '60日新低', '60日大幅下跌'} |

输出参数

| 名称   | 类型     | 描述                   |
|------|--------|----------------------|
| 时间   | object | -                    |
| 代码   | object | -                    |
| 名称   | object | -                    |
| 板块   | object | -                    |
| 相关信息 | object | 注意: 不同的 symbol 的单位不同 |

接口示例

```python
import akshare as ak

stock_changes_em_df = ak.stock_changes_em(symbol="大笔买入")
print(stock_changes_em_df)
```

数据示例

```
            时间   代码    名称    板块                       相关信息
0     14:55:51  872953  国子软件  大笔买入   124230,19.24000,0.300000
1     14:55:43  834021  流金科技  大笔买入    296001,4.52000,0.280453
2     14:55:28  301292  海科新源  大笔买入   142800,14.90000,0.119459
3     14:55:27  001277  速达股份  大笔买入    54500,38.29000,0.099971
4     14:55:22  301052  果麦文化  大笔买入    98571,30.31000,0.181676
...        ...     ...   ...   ...                        ...
3169  09:25:19  603918  金桥信息  大笔买入   496600,11.83000,0.045053
3170  09:25:19  601136  首创证券  大笔买入  2743700,23.50000,0.088467
3171  09:25:17  688631  莱斯信息  大笔买入   118450,67.00000,0.078905
3172  09:25:16  600864  哈投股份  大笔买入   3876600,6.37000,0.083333
3173  09:25:16  688709  成都华微  大笔买入   130346,17.40000,0.055825
[3174 rows x 5 columns]
```

### 板块异动详情

接口: stock_board_change_em

目标地址: https://quote.eastmoney.com/changes/

描述: 东方财富-行情中心-当日板块异动详情

限量: 返回最近交易日的数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称                  | 类型      | 描述        |
|---------------------|---------|-----------|
| 板块名称                | object  | -         |
| 涨跌幅                 | float64 | 注意单位: %   |
| 主力净流入               | float64 | 注意单位: 万元  |
| 板块异动总次数             | float64 | -         |
| 板块异动最频繁个股及所属类型-股票代码 | object  | -         |
| 板块异动最频繁个股及所属类型-股票名称 | object  | -         |
| 板块异动最频繁个股及所属类型-买卖方向 | object  | -         |
| 板块具体异动类型列表及出现次数     | object  | 返回具体异动的字典 |

接口示例

```python
import akshare as ak

stock_board_change_em_df = ak.stock_board_change_em()
print(stock_board_change_em_df)
```

数据示例

```
       板块名称  ...                                    板块具体异动类型列表及出现次数
0      融资融券  ...  [{'t': 8201, 'ct': 10821}, {'t': 8219, 'ct': 1...
1       深股通  ...  [{'t': 8219, 'ct': 4564}, {'t': 8201, 'ct': 38...
2      创业板综  ...  [{'t': 8201, 'ct': 5584}, {'t': 8203, 'ct': 40...
3       沪股通  ...  [{'t': 8219, 'ct': 4973}, {'t': 8201, 'ct': 27...
4      富时罗素  ...  [{'t': 8219, 'ct': 4654}, {'t': 8201, 'ct': 26...
..      ...  ...                                                ...
563    千金藤素  ...  [{'t': 8201, 'ct': 18}, {'t': 8219, 'ct': 13},...
564  同步磁阻电机  ...  [{'t': 8201, 'ct': 18}, {'t': 8219, 'ct': 18},...
565  第四代半导体  ...  [{'t': 8201, 'ct': 10}, {'t': 8203, 'ct': 10},...
566  光伏高速公路  ...  [{'t': 8219, 'ct': 17}, {'t': 8201, 'ct': 11},...
567    赛马概念  ...  [{'t': 8219, 'ct': 17}, {'t': 8203, 'ct': 10},...
[568 rows x 8 columns]
```

### 涨停板行情

#### 涨停股池

接口: stock_zt_pool_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=ztgc

描述: 东方财富网-行情中心-涨停板行情-涨停股池

限量: 单次返回指定 date 的涨停股池数据; 该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20241008' |

输出参数

| 名称     | 类型      | 描述             |
|--------|---------|----------------|
| 序号     | int64   | -              |
| 代码     | object  | -              |
| 名称     | object  | -              |
| 涨跌幅    | float64 | 注意单位: %        |
| 最新价    | float64 | -              |
| 成交额    | int64   | -              |
| 流通市值   | float64 | -              |
| 总市值    | float64 | -              |
| 换手率    | float64 | 注意单位: %        |
| 封板资金   | int64   | -              |
| 首次封板时间 | object  | 注意格式: 09:25:00 |
| 最后封板时间 | object  | 注意格式: 09:25:00 |
| 炸板次数   | int64   | -              |
| 涨停统计   | object  | -              |
| 连板数    | int64   | 注意格式: 1 为首板    |
| 所属行业   | object  | -              |

接口示例

```python
import akshare as ak

stock_zt_pool_em_df = ak.stock_zt_pool_em(date='20241008')
print(stock_zt_pool_em_df)
```

数据示例

```
      序号 代码    名称      涨跌幅    最新价  ...  最后封板时间 炸板次数 涨停统计 连板数 所属行业
0      1  000004  国华网安  10.000000  17.93  ...  141354     1  10/6    2  软件开发
1      2  000021   深科技   9.981852  18.18  ...  142309     1   1/1    1  消费电子
2      3  000027  深圳能源  10.014306   7.69  ...  142912     3   1/1    1  电力行业
3      4  000032  深桑达Ａ   9.994223  19.04  ...  144248     1   3/3    3  通信设备
4      5  000034  神州数码  10.006497  33.86  ...  144533     2   2/2    2  计算机设
..   ...     ...   ...        ...    ...  ...     ...   ...   ...  ...   ...
771  772  300222  科大智能  20.053122   9.04  ...  150000     0   1/1    1  电网设备
772  773  300248   新开普  20.022123  10.85  ...  150000     0   1/1    1  互联网服
773  774  688095  福昕软件  20.002993  80.21  ...  150000     0   1/1    1  软件开发
774  775  688205   德科立  20.009701  49.48  ...  150000     0   1/1    1  光学光电
775  776  688653  康希通信  20.030350  15.82  ...  150000     0   1/1    1   半导体
[776 rows x 16 columns]
```

#### 昨日涨停股池

接口: stock_zt_pool_previous_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=zrzt

描述: 东方财富网-行情中心-涨停板行情-昨日涨停股池

限量: 单次返回指定 date 的昨日涨停股池数据; 该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20240415' |

输出参数

| 名称     | 类型      | 描述             |
|--------|---------|----------------|
| 序号     | int32   | -              |
| 代码     | object  | -              |
| 名称     | object  | -              |
| 涨跌幅    | float64 | 注意单位: %        |
| 最新价    | int64   | -              |
| 涨停价    | int64   | -              |
| 成交额    | int64   | -              |
| 流通市值   | float64 | -              |
| 总市值    | float64 | -              |
| 换手率    | float64 | 注意单位: %        |
| 涨速     | float64 | 注意单位: %        |
| 振幅     | float64 | 注意单位: %        |
| 昨日封板时间 | int64   | 注意格式: 09:25:00 |
| 昨日连板数  | int64   | 注意格式: 1 为首板    |
| 涨停统计   | object  | -              |
| 所属行业   | object  | -              |

接口示例

```python
import akshare as ak

stock_zt_pool_previous_em_df = ak.stock_zt_pool_previous_em(date='20240415')
print(stock_zt_pool_previous_em_df)
```

数据示例

```
    序号   代码    名称      涨跌幅    最新价  ...     振幅  昨日封板时间  昨日连板数  涨停统计  所属行业
0    1  001202   炬申股份  -5.521886  14.03  ...  18.585859  093957      1   2/1  物流行业
1    2  603822   嘉澳环保  -5.987780  23.08  ...   8.472506  131416      1   2/1  化学制品
2    3  001333   光华股份  -8.636837  17.56  ...   9.781478  130703      1   2/1  化学制品
3    4  603276   恒兴新材  -8.216784  21.00  ...  10.926574  142402      1   2/1  化学原料
4    5  603968   醋化股份  -8.055329  11.30  ...   7.648495  112937      1   2/1  化学制品
5    6  600854   春兰股份  -6.808510   4.38  ...  14.042553  092501      1   2/1  家电行业
6    7  603067   振华股份  -4.146101   9.71  ...   6.910168  144529      1   2/1  化学原料
7    8  601179   中国西电   6.184012   7.04  ...  14.027150  142250      2   3/2  电网设备
8    9  603335    迪生力  -8.802817   5.18  ...   8.802817  093119      1   2/1  汽车零部
9   10  605118   力鼎光电  -5.382586  17.93  ...   7.229552  095804      1   2/1  光学光电
10  11  001228    永泰运  -1.821494  26.95  ...   5.173042  132433      1   2/1  物流行业
11  12  002552   宝鼎科技  -6.556334  14.68  ...   7.702100  095245      1   2/1  通用设备
12  13  601956   东贝集团  -6.972477   5.07  ...  13.761468  094931      1   2/1  家电行业
13  14  603116    红蜻蜓  -9.844560   5.22  ...   8.635578  144142      1   2/1  纺织服装
14  15  002264  新 华 都   5.389221   5.28  ...  10.778442  093200      1   2/1  互联网服
15  16  000888   峨眉山Ａ   0.271924  14.75  ...  12.984364  140624      2   5/3  旅游酒店
16  17  605336   帅丰电器   1.520637  14.02  ...  15.133961  093000      1   2/1  家电行业
17  18  605378   野马电池  -4.830054  21.28  ...   9.749553  093211      1   2/1    电池
18  19  000059   华锦股份 -10.016420   5.48  ...   8.045978  130045      1   2/1  石油行业
19  20  000506   中润资源  -9.956710   6.24  ...   0.000000  140100      2   8/5   贵金属
20  21  000603   盛达资源  -7.873528  14.86  ...   5.455673  130006      1   2/1  有色金属
21  22  000622   恒立实业  -9.883721   3.10  ...   6.686047  093215      1   2/1  汽车零部
22  23  000856   冀东装备  -9.967846   8.40  ...  10.610932  094142      1   2/1  专用设备
23  24  002059   云南旅游   5.111821   6.58  ...  15.814696  145415      1   2/1  旅游酒店
24  25  002476   宝莫股份  -9.960938   4.61  ...   6.250000  100633      1   2/1  化学制品
25  26  002562   兄弟科技  -5.191257   3.47  ...   6.557377  112212      1   2/1  化学制品
26  27  002574   明牌珠宝  -9.941521   6.16  ...   5.555556  142530      1   2/1  珠宝首饰
27  28  002716   湖南白银  -7.579462   3.78  ...   5.134474  145154      1   2/1   贵金属
28  29  002828   贝肯能源  -3.445065  10.37  ...  13.966480  093821      1   2/1  采掘行业
29  30  002835   同为股份   9.973227  16.43  ...   5.488621  092500      1   2/2  计算机设
30  31  300641   正丹股份  11.195929   8.74  ...   9.796437  093248      2   3/2  化学制品
31  32  300721   怡达股份 -14.406779  11.11  ...  16.409863  145109      1   2/1  化学制品
32  33  300824   北鼎股份   3.828829   9.22  ...  25.788290  093606      1   2/1  家电行业
33  34  300931   通用电梯  -8.356164   6.69  ...  16.164383  144054      1   2/1  专用设备
34  35  300945    曼卡龙  -9.520401  12.64  ...   8.017179  102900      1   2/1  珠宝首饰
35  36  301008   宏昌科技  -1.414743  26.48  ...  20.476545  093254      1   2/1  家电行业
36  37  600363   联创光电  -0.864074  29.83  ...   5.882353  132434      1   2/1  光学光电
37  38  600857   宁波中百  -9.979633   8.84  ...  16.395111  092500      3   4/3  商业百货
38  39  600984   建设机械  10.129870   4.24  ...   0.000000  092502      3   4/4  工程机械
39  40  603213   镇洋发展  -9.933143   9.43  ...  10.028653  093011      1   2/1  化学原料
40  41  603320   迪贝电气 -10.029283  12.29  ...  15.080526  093804      1   2/1    电机
41  42  603657   春光科技  10.029717  14.81  ...   8.989598  100149      1   2/2  家电行业
42  43  603813   原尚股份 -10.012754  14.11  ...  20.025509  092503      2   5/3  物流行业
43  44  603900   莱绅通灵  -9.981685   9.83  ...   0.000000  093620      9  10/9  珠宝首饰
44  45  688663    新风光  -2.955271  24.30  ...   6.669329  144856      1   2/1  电网设备
45  46  603082   北自科技   4.895862  38.78  ...   7.222072  144152      1   2/1  通用设备
46  47  001389   广合科技   2.957523  51.87  ...   9.170305  112151      1   2/1  电子元件
47  48  603893    瑞芯微   2.849442  53.42  ...   5.602618  131557      1   2/1   半导体
48  49  603789   星光农机  -4.013015   8.85  ...   6.832972  100438      1   4/2  专用设备
49  50  001239   永达股份   4.634496  21.90  ...  14.524606  093021      2   3/2  专用设备
50  51  002313   日海智能  -9.167544   8.62  ...  10.010537  100815      1   2/1  通信设备
51  52  870726   鸿智科技 -15.000001  17.00  ...  19.500000  094042      1   2/1  家电行业
52  53  600490   鹏欣资源  -6.077348   3.40  ...   7.182320  095846      1   2/1  有色金属
53  54  603661   恒林股份   1.083032  47.60  ...   8.027182  131046      1   2/1  装修建材
54  55  002295   精艺股份  -3.393665   8.54  ...  19.683258  093233      1   2/1  有色金属
55  56  600815   厦工股份 -10.033445   2.69  ...  14.046823  145336      2   3/2  工程机械
56  57  002455   百川股份  -5.270656   6.65  ...  15.242166  100733      2   5/3  化学制品
57  58  001387   雪祺电气  -0.170285  23.45  ...  10.174542  092500      1   2/1  家电行业
58  59  002102   冠福股份  -7.801419   2.60  ...   6.737588  102645      1   2/1  化学制药
[59 rows x 16 columns]
```

#### 强势股池

接口: stock_zt_pool_strong_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=qsgc

描述: 东方财富网-行情中心-涨停板行情-强势股池

限量: 单次返回指定 date 的强势股池数据；该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20241009' |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int64   | -       |
| 代码   | object  | -       |
| 名称   | object  | -       |
| 涨跌幅  | float64 | 注意单位: % |
| 最新价  | float64 | -       |
| 涨停价  | float64 | -       |
| 成交额  | int64   | -       |
| 流通市值 | float64 | -       |
| 总市值  | float64 | -       |
| 换手率  | float64 | 注意单位: % |
| 涨速   | float64 | 注意单位: % |
| 是否新高 | object  | -       |
| 量比   | float64 | -       |
| 涨停统计 | object  | -       |
| 入选理由 | object  | -       |
| 所属行业 | object  | -       |

接口示例

```python
import akshare as ak

stock_zt_pool_strong_em_df = ak.stock_zt_pool_strong_em(date='20241231')
print(stock_zt_pool_strong_em_df)
```

数据示例

```
      序号    代码   名称      涨跌幅  ...       量比  涨停统计        入选理由  所属行业
0      1  301202  朗威股份  19.994860  ...  2.298022   1/1         60日新高  通用设备
1      2  002629  仁智股份  10.103627  ...  1.863184   2/2        近期多次涨停  采掘行业
2      3  002361  神剑股份  10.101010  ...  8.426764   3/3  60日新高且近期多次涨停  塑料制品
3      4  002442  龙星化工  10.073260  ...  1.979328   1/1         60日新高  橡胶制品
4      5  600693  东百集团  10.062893  ...  1.475128  16/8  60日新高且近期多次涨停  商业百货
..   ...     ...   ...        ...  ...       ...   ...           ...   ...
122  123  002577  雷柏科技  -9.985735  ...  1.397206   5/3        近期多次涨停  消费电子
123  124  002067  景兴纸业 -10.000000  ...  1.227895   6/3        近期多次涨停  造纸印刷
124  125  002137   实益达 -10.010010  ...  1.372619   9/6        近期多次涨停  电子元件
125  126  603958  哈森股份 -10.015175  ...  2.310775   5/4  60日新高且近期多次涨停  纺织服装
126  127  603980  吉华集团 -10.082304  ...  0.823222   6/3        近期多次涨停  化学制品
[127 rows x 16 columns]
```

#### 次新股池

接口: stock_zt_pool_sub_new_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=cxgc

描述: 东方财富网-行情中心-涨停板行情-次新股池

限量: 单次返回指定 date 的次新股池数据；该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20241231' |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int32   | -       |
| 代码   | object  | -       |
| 名称   | object  | -       |
| 涨跌幅  | float64 | 注意单位: % |
| 最新价  | float64 | -       |
| 涨停价  | float64 | -       |
| 成交额  | int64   | -       |
| 流通市值 | float64 | -       |
| 总市值  | float64 | -       |
| 转手率  | float64 | 注意单位: % |
| 开板几日 | int64   | -       |
| 开板日期 | int64   | -       |
| 上市日期 | int64   | -       |
| 是否新高 | int64   | -       |
| 涨停统计 | object  | -       |
| 所属行业 | object  | -       |

接口示例

```python
import akshare as ak

stock_zt_pool_sub_new_em_df = ak.stock_zt_pool_sub_new_em(date='20241231')
print(stock_zt_pool_sub_new_em_df)
```

数据示例

```
    序号  代码   名称      涨跌幅    最新价  ...   开板日期    上市日期  是否新高  涨停统计  所属行业
0    1  001391  C国货航 -0.967742   9.21  ...  2024-12-30  2024-12-30     否   0/0  物流行业
1    2  920082  方正阀门  1.915185  14.90  ...  2024-12-26  2024-12-26     否   0/0  通用设备
2    3  603194  中力股份 -6.406904  35.79  ...  2024-12-24  2024-12-24     否   0/0  工程机械
3    4  301585  蓝宇股份 -1.644876  46.64  ...  2024-12-20  2024-12-20     否   0/0  化学制品
4    5  301598  博科测试 -7.260816  67.95  ...  2024-12-20  2024-12-20     否   0/0  专用设备
..  ..     ...   ...       ...    ...  ...         ...         ...   ...   ...   ...
86  87  301567  贝隆精密 -4.102564  44.88  ...  2024-01-16  2024-01-16     否   0/0  消费电子
87  88  001387  雪祺电气 -3.814147  13.87  ...  2024-01-11  2024-01-11     否   0/0  家电行业
88  89  603312  西典新能 -2.946482  32.28  ...  2024-01-11  2024-01-11     否   0/0  电网设备
89  90  603325  博隆技术  0.012642  79.11  ...  2024-01-10  2024-01-10     否   0/0  通用设备
90  91  688717  艾罗能源 -0.383877  46.71  ...  2024-01-03  2024-01-03     否   0/0  光伏设备
[91 rows x 16 columns]
```

#### 炸板股池

接口: stock_zt_pool_zbgc_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=zbgc

描述: 东方财富网-行情中心-涨停板行情-炸板股池

限量: 单次返回指定 date 的炸板股池数据；该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20241011' |

输出参数

| 名称     | 类型      | 描述             |
|--------|---------|----------------|
| 序号     | int32   | -              |
| 代码     | object  | -              |
| 名称     | object  | -              |
| 涨跌幅    | float64 | 注意单位: %        |
| 最新价    | float64 | -              |
| 涨停价    | float64 | -              |
| 成交额    | int64   | -              |
| 流通市值   | float64 | -              |
| 总市值    | float64 | -              |
| 换手率    | float64 | 注意单位: %        |
| 涨速     | int64   | -              |
| 首次封板时间 | object  | 注意格式: 09:25:00 |
| 炸板次数   | int64   | -              |
| 涨停统计   | int64   | -              |
| 振幅     | object  | -              |
| 所属行业   | object  | -              |

接口示例

```python
import akshare as ak

stock_zt_pool_zbgc_em_df = ak.stock_zt_pool_zbgc_em(date='20241011')
print(stock_zt_pool_zbgc_em_df)
```

数据示例

```
    序号  代码   名称       涨跌幅    最新价  ... 首次封板时间  炸板次数  涨停统计  振幅  所属行业
0    1  002347  泰尔股份  -0.920810   5.38  ...  092500     3   3/2  17.311234  通用设备
1    2  002835  同为股份   5.509817  16.66  ...  092500     2   0/0   5.509817  计算机设
2    3  002570   贝因美   7.619048   3.39  ...  093336     1   0/0   7.301588  食品饮料
3    4  301178   天亿马   0.065466  30.57  ...  094215     1   4/2  20.000000  软件开发
4    5  600239  云南城投   8.363637   2.98  ...  094658     4   7/4   9.454545  房地产服
5    6  603016   新宏泰   9.251101  24.80  ...  095251     2   0/0  12.202642  电网设备
6    7  301058  中粮科工  11.724724  13.15  ...  100542     1   2/1  19.456245  工程建设
7    8  002423  中粮资本   2.907712  16.28  ...  102224     8   9/8  16.434893  多元金融
8    9  002537  海联金汇   8.580858   6.58  ...  103254     4   0/0  12.211222  汽车零部
9   10  002703  浙江世宝   7.892528  12.85  ...  110724     2   0/0  11.335012  汽车零部
10  11  603767  中马传动   3.935599  11.62  ...  111110     1   0/0  14.221825  汽车零部
11  12  001278  一彬科技   4.915993  16.86  ...  111357     2   0/0  14.747977  汽车零部
12  13  301398  星源卓镁   8.176848  45.51  ...  111430     1   0/0  25.623959  汽车零部
13  14  603499  翔港科技   7.900001  21.58  ...  112102     1   0/0  12.850000  造纸印刷
14  15  002075  沙钢股份   7.460035   6.05  ...  131539     2   0/0  10.124334  钢铁行业
15  16  600770  综艺股份  10.000000   3.74  ...  132618     2   0/0  11.470589  综合行业
[16 rows x 16 columns]
```

#### 跌停股池

接口: stock_zt_pool_dtgc_em

目标地址: https://quote.eastmoney.com/ztb/detail#type=zbgc

描述: 东方财富网-行情中心-涨停板行情-跌停股池

限量: 单次返回指定 date 的跌停股池数据；该接口只能获取近期的数据

输入参数

| 名称   | 类型  | 描述              |
|------|-----|-----------------|
| date | str | date='20241011' |

输出参数

| 名称     | 类型      | 描述             |
|--------|---------|----------------|
| 序号     | int64   | -              |
| 代码     | object  | -              |
| 名称     | object  | -              |
| 涨跌幅    | float64 | 注意单位: %        |
| 最新价    | float64 | -              |
| 成交额    | int64   | -              |
| 流通市值   | float64 | -              |
| 总市值    | float64 | -              |
| 动态市盈率  | float64 | -              |
| 换手率    | float64 | 注意单位: %        |
| 封单资金   | int64   | -              |
| 最后封板时间 | object  | 注意格式: 09:25:00 |
| 板上成交额  | int64   | -              |
| 连续跌停   | int64   | -              |
| 开板次数   | int64   | -              |
| 所属行业   | object  | -              |

接口示例

```python
import akshare as ak

stock_zt_pool_dtgc_em_df = ak.stock_zt_pool_dtgc_em(date='20241011')
print(stock_zt_pool_dtgc_em_df)
```

数据示例

```
    序号  代码     名称      涨跌幅    最新价  ... 最后封板时间  板上成交额  连续跌停  开板次数  所属行业
0    1  002795   永和智控  -9.905661   3.82  ...  143233     24222238     1     1  通用设备
1    2  603978   深圳新星  -9.981167  14.34  ...  144754    143723833     1     9  有色金属
2    3  000668   荣丰控股 -10.000000   7.38  ...  145651     14545980     1     2  房地产开
3    4  000697   炼石航空 -10.012836   7.01  ...  143442     32945926     1     1  航天航空
4    5  603496   恒为科技 -10.010427  25.89  ...  143203    123642303     1     4  互联网服
5    6  000850   华茂股份  -9.902912   4.64  ...  145618     16800512     1     2  纺织服装
6    7  002355   兴民智通  -9.985315   6.13  ...  145403     55676338     1     4  汽车零部
7    8  002883   中设股份  -9.991236  10.27  ...  140433     20144605     1     1  工程咨询
8    9  600099   林海股份  -9.970090   9.03  ...  140728     31065906     1     3  交运设备
9   10  300853   申昊科技 -19.991270  18.33  ...  142454    557961581     1    12  通用设备
10  11  601858   中国科传  -9.986801  20.46  ...  140544     68199318     1     3  文化传媒
11  12  600650   锦江在线  -9.993109  13.06  ...  145053     27841308     1     2  铁路公路
12  13  002908   德生科技  -9.979633   8.84  ...  140139     25946284     1     1  软件开发
13  14  600696   岩石股份 -10.018726   9.61  ...  134317     20594230     1     1  酿酒行业
14  15  600560   金自天正 -10.000000  14.49  ...  135223     39492234     1     2  专用设备
15  16  603728   鸣志电器 -10.008137  44.24  ...  140306     49712488     1     0    电机
16  17  600990   四创电子 -10.000000  21.15  ...  112710     65182833     1     7  通信设备
17  18  601777   力帆科技  -9.925094   4.81  ...  134409    123372309     1     7  汽车整车
18  19  600666    奥瑞德  -9.767442   1.94  ...  135702     28302466     1     0  光学光电
19  20  000536   华映科技  -9.944751   3.26  ...  134951    103476310     2     1  光学光电
20  21  000717   中南股份  -9.824561   2.57  ...  132045    104712542     1     0  钢铁行业
21  22  600390  XD五矿资  -9.972299   6.50  ...  144845  15307499959     2     2  多元金融
22  23  603188   亚邦股份  -9.975062   3.61  ...  093132     30793661     1     0  化学制品
23  24  603383   顶点软件 -10.007468  36.15  ...  134435   2683979703     2     9  软件开发
24  25  000062   深圳华强  -9.985528  24.88  ...  143854    137266692     2     0  电子元件
25  26  000628   高新发展  -9.994529  49.35  ...  140412    141856575     1     0  工程建设
26  27  600702   舍得酒业 -10.001374  65.51  ...  135511    133242819     1     1  酿酒行业
27  28  601162   天风证券 -10.041842   4.30  ...  145050  45859221240     2     9    证券
28  29  601933   永辉超市  -9.907121   2.91  ...  135047   1763205191     3     0  商业百货
29  30  603106   恒银科技  -9.967497   8.31  ...  092501    818423646     2     0  计算机设
30  31  000032   深桑达Ａ  -9.984795  17.76  ...  093206   1348257504     1     0  通信设备
31  32  002693   双成药业  -9.988776  16.04  ...  092500   1244147412     2     0  生物制品
[32 rows x 16 columns]
```

### 赚钱效应分析

接口: stock_market_activity_legu

目标地址: https://www.legulegu.com/stockdata/market-activity

描述: 乐咕乐股网-赚钱效应分析数据

限量: 单次返回当前赚钱效应分析数据

说明：

1. 涨跌比：即沪深两市上涨个股所占比例，体现的是市场整体涨跌，占比越大则代表大部分个股表现活跃。
2. 涨停板数与跌停板数的意义：涨停家数在一定程度上反映了市场的投机氛围。当涨停家数越多，则市场的多头氛围越强。真实涨停是非一字无量涨停。真实跌停是非一字无量跌停。

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型     | 描述  |
|-------|--------|-----|
| item  | object | -   |
| value | object | -   |

接口示例

```python
import akshare as ak

stock_market_activity_legu_df = ak.stock_market_activity_legu()
print(stock_market_activity_legu_df)
```

数据示例

```
        item                value
0         上涨               4770.0
1         涨停                119.0
2       真实涨停                101.0
3   st st*涨停                 10.0
4         下跌                281.0
5         跌停                  6.0
6       真实跌停                  4.0
7   st st*跌停                  4.0
8         平盘                 39.0
9         停牌                 10.0
10       活跃度               93.53%
11      统计日期  2024-10-14 15:00:00
```

### 资讯数据

#### 财经早餐-东财财富

接口：stock_info_cjzc_em

目标地址：https://stock.eastmoney.com/a/czpnc.html

描述：东方财富-财经早餐

限量：单次返回全部历史数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 标题   | object | -  |
| 摘要   | object | -  |
| 发布时间 | object | -  |
| 链接   | object | -  |

接口示例：

```python
import akshare as ak

stock_info_cjzc_em_df = ak.stock_info_cjzc_em()
print(stock_info_cjzc_em_df)
```

数据示例

```
                                       标题  ...                                                 链接
0                        东方财富财经早餐 3月13日周三  ...  http://finance.eastmoney.com/a/202403133009961...
1                        东方财富财经早餐 3月12日周二  ...  http://finance.eastmoney.com/a/202403113008262...
2                        东方财富财经早餐 3月11日周一  ...  http://finance.eastmoney.com/a/202403103007214...
3                         东方财富财经早餐 3月8日周五  ...  http://finance.eastmoney.com/a/202403073005282...
4                         东方财富财经早餐 3月7日周四  ...  http://finance.eastmoney.com/a/202403073004092...
..                                    ...  ...                                                ...
338                     东方财富财经早餐 10月14日周五  ...  http://finance.eastmoney.com/a/202210142528794...
339                     东方财富早盘内参 10月13日周四  ...  http://finance.eastmoney.com/a/202210132527637...
340                     东方财富早盘内参 10月12日周三  ...  http://finance.eastmoney.com/a/202210122526415...
341  【早盘内参】宁德时代：前三季度净利同比预增112.87%-132.22%  ...  http://finance.eastmoney.com/a/202210112525318...
342           【早盘内参】央行：进一步简化境外投资者进入中国投资流程  ...  http://finance.eastmoney.com/a/202210102524284...
[343 rows x 4 columns]
```

#### 全球财经快讯-东财财富

接口：stock_info_global_em

目标地址：https://kuaixun.eastmoney.com/7_24.html

描述：东方财富-全球财经快讯

限量：单次返回最近 200 条新闻数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 标题   | object | -  |
| 摘要   | object | -  |
| 发布时间 | object | -  |
| 链接   | object | -  |

接口示例：

```python
import akshare as ak

stock_info_global_em_df = ak.stock_info_global_em()
print(stock_info_global_em_df)
```

数据示例

```
                                     标题  ...                                                 链接
0                     法本信息与鸿蒙生态服务公司签约合作  ...  https://finance.eastmoney.com/a/20240313301075...
1    欧洲央行管委维勒鲁瓦：6月降息的可能性高于4月 但仍有可能在春季降息  ...  https://finance.eastmoney.com/a/20240313301075...
2        海马汽车：目前公司氢燃料电池汽车7X-H已开始小批量上线生产  ...  https://finance.eastmoney.com/a/20240313301075...
3     茉酸奶回应被消保委点名：目前配方只有酸奶和水果 后续有官方渠道回应  ...  https://finance.eastmoney.com/a/20240313301075...
4         大众重押合肥：ID.品牌将推U系列车型 计划建独立销售渠道  ...  https://finance.eastmoney.com/a/20240313301075...
..                                  ...  ...                                                ...
195              加速电动化 江淮汽车与大众中国拟增资合资公司  ...  https://finance.eastmoney.com/a/20240313301015...
196                    农业银行普惠贷款余额突破4万亿元  ...  https://finance.eastmoney.com/a/20240313301015...
197            中金公司：AI Agent或开启AI原生应用时代  ...  https://finance.eastmoney.com/a/20240313301015...
198                          日韩股市周三双双高开  ...  https://finance.eastmoney.com/a/20240313301015...
199             委内瑞拉：不允许阿根廷任何形式的飞机使用委领空  ...  https://finance.eastmoney.com/a/20240313301015...
[200 rows x 4 columns]
```

#### 全球财经快讯-新浪财经

接口：stock_info_global_sina

目标地址：https://finance.sina.com.cn/7x24

描述：新浪财经-全球财经快讯

限量：单次返回最近 20 条新闻数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 时间   | object | -  |
| 内容   | object | -  |

接口示例

```python
import akshare as ak

stock_info_global_sina_df = ak.stock_info_global_sina()
print(stock_info_global_sina_df)
```

数据示例

```
                     时间                                                 内容
0   2024-03-13 16:18:40  【Canalys：2023年Q4英特尔CPU出货5000万颗占比78% 是AMD的6倍】根据...
1   2024-03-13 16:18:19  俄罗斯别尔哥罗德市联邦安全局发布消息称，该部门大楼遭到乌军无人机袭击，没有造成人员伤亡，大楼...
2   2024-03-13 16:18:08  日本首相顾问Yata：（当被问及工资提议是否可能会推动在三月结束负利率时）政府不会干涉日本央...
3   2024-03-13 16:16:35  【基金经理：日企大赚而劳动力短缺，加薪5%不足为奇】“与往年相比，日本企业在加薪方面要开放得...
4   2024-03-13 16:15:41  【梅西代言白酒重新上架】多款梅西代言白酒已重新上架。3月13日搜索发现，赤水河白酒电商平台店...
5   2024-03-13 16:15:35         土耳其银行家表示，资产规模超过1000亿里拉的土耳其银行将执行15%的里拉准备金率。
6   2024-03-13 16:15:10                          地区官员表示，俄罗斯梁赞石油精炼厂的火灾已被扑灭。
7   2024-03-13 16:14:46  【九部门：鼓励在有条件的村布放智能快件箱】商务部等9部门发布关于推动农村电商高质量发展的实施...
8   2024-03-13 16:14:23        两名安全消息人士表示，以色列袭击了黎巴嫩南部城市泰尔附近的一辆汽车，导致两名乘客死亡。
9   2024-03-13 16:13:38  【九部门：造县域直播电商基地】商务部等9部门发布关于推动农村电商高质量发展的实施意见。其中提...
10  2024-03-13 16:12:30       土耳其银行家表示，资产规模超过5000亿里拉的土耳其银行将执行25%的里拉存款准备金率。
11  2024-03-13 16:12:25           名创优品美股盘前涨超4%，去年第四季度营收及毛利率双双创新高，拟派特别现金股息。
12  2024-03-13 16:11:08  【商务部等9部门发布关于推动农村电商高质量发展的实施】意见商务部等9部门发布关于推动农村电商...
13  2024-03-13 16:09:50  【乘联会：3月1-10日乘用车市场零售35.5万辆 同比去年同期增长4%】乘联会数据显示，3...
14  2024-03-13 16:08:56  【港股收评：恒指收跌0.07% 科指收涨0.34%】港股收盘，恒指收跌0.07%，科指收涨0...
15  2024-03-13 16:08:01  【韩国总统尹锡悦承诺到2027年将航天产业发展相关预算增至1.5万亿韩元】韩国总统尹锡悦3月...
16  2024-03-13 16:05:34  【恢复两融业务正常平仓措施？业内人士否认】今天盘中，市场突然传出：风险解除，市场秩序恢复正常...
17  2024-03-13 16:05:26  【中欧民航合作项目预先飞行计划研讨会在京召开】日前，中欧民航合作项目（EU-CHINA AP...
18  2024-03-13 16:05:21          英伟达在美股盘前上涨1.8%，报936美元/股。上一交易日，英伟达收盘大涨超7%。
19  2024-03-13 16:04:07  【中国或将调查法国酒业是否与欧盟调查中国电车有关？外交部回应】外交部发言人汪文斌主持例行记者...
```

#### 快讯-富途牛牛

接口：stock_info_global_futu

目标地址：https://news.futunn.com/main/live

描述：富途牛牛-快讯

限量：单次返回最近 50 条新闻数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 标题   | object | -  |
| 内容   | object | -  |
| 发布时间 | object | -  |
| 链接   | object | -  |

接口示例

```python
import akshare as ak

stock_info_global_futu_df = ak.stock_info_global_futu()
print(stock_info_global_futu_df)
```

数据示例

```
                                               标题  ...                                             链接
0                                                  ...  https://news.futunn.com/flash/16697438?src=48
1      Canalys：2023年Q4英特尔CPU出货5000万颗占比78% 是AMD的6倍  ...  https://news.futunn.com/flash/16697433?src=48
2                                                  ...  https://news.futunn.com/flash/16697427?src=48
3                        基金经理：日企大赚而劳动力短缺，加薪5%不足为奇  ...  https://news.futunn.com/flash/16697407?src=48
4                             九部门：鼓励在有条件的村布放智能快件箱  ...  https://news.futunn.com/flash/16697383?src=48
5                                   九部门：造县域直播电商基地  ...  https://news.futunn.com/flash/16697371?src=48
6                                                  ...  https://news.futunn.com/flash/16697361?src=48
7                       商务部等9部门发布关于推动农村电商高质量发展的实施  ...  https://news.futunn.com/flash/16697341?src=48
8                                                  ...  https://news.futunn.com/flash/16697328?src=48
9             乘联会：3月1-10日乘用车市场零售35.5万辆 同比去年同期增长4%  ...  https://news.futunn.com/flash/16697322?src=48
10                            恢复两融业务正常平仓措施？业内人士否认  ...  https://news.futunn.com/flash/16697275?src=48
11                          中欧民航合作项目预先飞行计划研讨会在京召开  ...  https://news.futunn.com/flash/16697270?src=48
12                                                 ...  https://news.futunn.com/flash/16697271?src=48
13                          蔚来推出买车送手机活动，面向2024款车型  ...  https://news.futunn.com/flash/16697255?src=48
14                                                 ...  https://news.futunn.com/flash/16697185?src=48
15                           洛阳钼业：KFM 2月产铜量再创历史新高  ...  https://news.futunn.com/flash/16697158?src=48
16                     策略师：人为压低通胀，日本央行无权继续维持超宽松政策  ...  https://news.futunn.com/flash/16697130?src=48
17                                                 ...  https://news.futunn.com/flash/16697108?src=48
18                                                 ...  https://news.futunn.com/flash/16697090?src=48
19                              法本信息与鸿蒙生态服务公司签约合作  ...  https://news.futunn.com/flash/16697083?src=48
20                            港股医药股尾盘拉升 和黄医药涨超10%  ...  https://news.futunn.com/flash/16697084?src=48
21                   美众议院将就“封禁TikTok”法案进行表决 外交部回应  ...  https://news.futunn.com/flash/16697069?src=48
22                       互金协会：建议将所有涉金融APP纳入备案管理范围  ...  https://news.futunn.com/flash/16697058?src=48
23                           凯投宏观：英国经济在隧道的尽头看到了光明  ...  https://news.futunn.com/flash/16697054?src=48
24                  报告：去年全球艺术品销售额下降4%，中国销售额逆势增长9%  ...  https://news.futunn.com/flash/16697051?src=48
25                          高盛：沟通不足，日本央行会推迟一个月再加息  ...  https://news.futunn.com/flash/16697046?src=48
26                                                 ...  https://news.futunn.com/flash/16697048?src=48
27                                                 ...  https://news.futunn.com/flash/16696995?src=48
28                             天津调整公积金贷款首付比例 首套两成  ...  https://news.futunn.com/flash/16696984?src=48
29                                                 ...  https://news.futunn.com/flash/16696915?src=48
30                                                 ...  https://news.futunn.com/flash/16696860?src=48
31                        小米生态链模式十年来最大调整，提出分级管理策略  ...  https://news.futunn.com/flash/16696189?src=48
32                         《我国支持科技创新主要税费优惠政策指引》发布  ...  https://news.futunn.com/flash/16696803?src=48
33               美国家运输安全委员会将于8月就波音客机“掉门”事故举行调查听证会  ...  https://news.futunn.com/flash/16696764?src=48
34                                                 ...  https://news.futunn.com/flash/16696744?src=48
35                                                 ...  https://news.futunn.com/flash/16696702?src=48
36                                                 ...  https://news.futunn.com/flash/16696682?src=48
37                 香港文旅局：首两个月日均访港人次达13万 旅游业复苏势头强劲  ...  https://news.futunn.com/flash/16696679?src=48
38                                                 ...  https://news.futunn.com/flash/16696632?src=48
39                                                 ...  https://news.futunn.com/flash/16696592?src=48
40                         北向资金净买入15.67亿元，连续4日净买入  ...  https://news.futunn.com/flash/16696597?src=48
41                          财经网站Forexlive评英国GDP数据  ...  https://news.futunn.com/flash/16696577?src=48
42                                                 ...  https://news.futunn.com/flash/16696578?src=48
43  AI研发投资超100亿：荣耀AI PC技术将在荣耀MagicBook Pro 16全面落地  ...  https://news.futunn.com/flash/16696540?src=48
44                    A股收评：沪指缩量调整跌0.4% 传媒、游戏股逆势大涨  ...  https://news.futunn.com/flash/16696512?src=48
45                   二手平台VisionPro租赁每小时199元 需3万押金  ...  https://news.futunn.com/flash/16696336?src=48
46            小米汽车 SU7 新配置申报：宁德时代磷酸铁锂电池，220kW 单电机  ...  https://news.futunn.com/flash/16696331?src=48
47                              大众汽车计划今年推出30多款新产品  ...  https://news.futunn.com/flash/16696328?src=48
48               极氪发布安徽马鞍山极氪 001 过火事件说明：电池无异常、无燃烧  ...  https://news.futunn.com/flash/16696320?src=48
49                 中钢协：3月上旬重点统计钢铁企业共生产钢材1927.62万吨  ...  https://news.futunn.com/flash/16696278?src=48
[50 rows x 4 columns]
```

#### 全球财经直播-同花顺财经

接口：stock_info_global_ths

目标地址：https://news.10jqka.com.cn/realtimenews.html

描述：同花顺财经-全球财经直播

限量：单次返回最近 20 条新闻数据

输入参数

| 名称 | 类型 | 描述 |
|----|----|----|
| -  | -  | -  |

输出参数

| 名称   | 类型     | 描述 |
|------|--------|----|
| 标题   | object | -  |
| 内容   | object | -  |
| 发布时间 | object | -  |
| 链接   | object | -  |

接口示例

```python
import akshare as ak

stock_info_global_ths_df = ak.stock_info_global_ths()
print(stock_info_global_ths_df)
```

数据示例

```
                                            标题  ...                                                 链接
0                       机构论市：指数一波三折 传媒、游戏逆势走强！  ...  https://news.10jqka.com.cn/20240313/c655877020...
1                            日本决定启动与孟加拉国的EPA谈判  ...  https://news.10jqka.com.cn/20240313/c655876966...
2   Canalys：2023年Q4英特尔CPU出货5000万颗占比78% 是AMD的6倍  ...  https://news.10jqka.com.cn/20240313/c655876921...
3                  英伟达美股盘前涨近2% 市值再度升至2.3万亿美元上方  ...  https://news.10jqka.com.cn/20240313/c655876900...
4                          九部门：鼓励在有条件的村布放智能快件箱  ...  https://news.10jqka.com.cn/20240313/c655876856...
5                               九部门：打造县域直播电商基地  ...  https://news.10jqka.com.cn/20240313/c655876844...
6          乘联会：3月1-10日乘用车市场零售35.5万辆 同比去年同期增长4%  ...  https://news.10jqka.com.cn/20240313/c655876794...
7            商务部等9部门：用5年时间在全国培育100个左右农村电商“领跑县”  ...  https://news.10jqka.com.cn/20240313/c655876780...
8                     港股收评：恒指收跌0.07% 科指收涨0.34%  ...  https://news.10jqka.com.cn/20240313/c655876701...
9                        蔚来推出买车送手机活动，面向2024款车型  ...  https://news.10jqka.com.cn/20240313/c655876687...
10         韩国总统尹锡悦承诺到2027年将航天产业发展相关预算增至1.5万亿韩元  ...  https://news.10jqka.com.cn/20240313/c655876677...
11                         恢复两融业务正常平仓措施？业内人士否认  ...  https://news.10jqka.com.cn/20240313/c655876644...
12                       中欧民航合作项目预先飞行计划研讨会在京召开  ...  https://news.10jqka.com.cn/20240313/c655876634...
13                                      欧股多数高开  ...  https://news.10jqka.com.cn/20240313/c655876611...
14                        洛阳钼业：KFM 2月产铜量再创历史新高  ...  https://news.10jqka.com.cn/20240313/c655876547...
15                    第十三届中国数控机床展览会将于4月8日在上海启幕  ...  https://news.10jqka.com.cn/20240313/c655876458...
16                        浙大网新：子公司中标1.28亿元施工项目  ...  https://news.10jqka.com.cn/20240313/c655876449...
17                外交部：不管谁当选下一届美国总统 我们都希望更好造福两国  ...  https://news.10jqka.com.cn/20240313/c655876428...
18                   小米SU7展车已进驻王府井门店，店内仍在装修未开业  ...  https://news.10jqka.com.cn/20240313/c655876412...
19       中证指数有限公司将发布中证港股通人工智能主题指数和中证香港人工智能主题指数  ...  https://news.10jqka.com.cn/20240313/c655876408...
[20 rows x 4 columns]
```

#### 电报-财联社

接口：stock_info_global_cls

目标地址：https://www.cls.cn/telegraph

描述：财联社-电报

限量：单次返回指定 symbol 的最近 20 条财联社-电报的数据

输入参数

| 名称     | 类型  | 描述                                 |
|--------|-----|------------------------------------|
| symbol | str | symbol="全部"；choice of {"全部", "重点"} |

输出参数

| 名称   | 类型     | 描述  |
|------|--------|-----|
| 标题   | object | -   |
| 内容   | object | -   |
| 发布日期 | object | -   |
| 发布时间 | object | -   |

接口示例：

```python
import akshare as ak

stock_info_global_cls_df = ak.stock_info_global_cls(symbol="全部")
print(stock_info_global_cls_df)
```

数据示例

```
                               标题  ...      发布时间
0    华为轮值董事长徐直军谈鸿蒙生态未来目标：拥有10万个应用  ...  14:05:03
1       中国牵头首个冷链物流无接触配送领域国际标准正式发布  ...  14:12:02
2           以军袭击黎巴嫩首都住宅楼 死亡人数升至5人  ...  14:37:34
3            上交所与三大石油石化集团将进一步深化合作  ...  14:50:34
4                                  ...  14:56:22
5    至少19人食用后患病 美国企业紧急召回近76吨牛肉泥产品  ...  15:13:18
6       《加强长江流域生物多样性司法保护倡议书》在武汉发布  ...  15:27:53
7               阿联酋哈伊马角酋长一行到访亿航智能  ...  15:41:35
8           以军称空袭贝鲁特南郊多个真主党武装军事目标  ...  15:43:30
9          以军空袭加沙地带多地 致17名巴勒斯坦人死亡  ...  15:46:34
10             北约秘书长在美国佛州与特朗普举行会谈  ...  15:49:04
11     经济观察报：央国企市值管理更多相关政策在酝酿和推进中  ...  15:56:06
12      华为徐直军：鸿蒙生态就是基于开源鸿蒙共建共享的生态  ...  16:00:29
13  我国牵头的首个工业化建造自动标识与数据采集应用国际标准发布  ...  16:02:03
14     AI辅助诊断首次被列入 国家医保局解读17批价格立项  ...  16:12:22
15            以军袭击贝鲁特中部住宅楼 已致9人死亡  ...  16:32:07
16        俄宣布12月1日起临时禁止废旧贵金属出口6个月  ...  16:36:40
17                                 ...  16:41:55
18          波兰农民在波乌边境抗议 将封锁梅迪卡过境点  ...  16:56:41
19            吉林省将迎大范围明显雨雪及寒潮大风天气  ...  17:17:38
[20 rows x 4 columns]
```

### 手续费

#### 国内券商佣金

##### 以万 2.5 佣金结构为例

深证证券交易所

| 项目   | 费用      | 收取部门    |
|------|---------|---------|
| 过户费  | 万 0.2   | 由中国结算收取 |
| 证管费  | 万 0.2   | 由证监会收取  |
| 经手费  | 万 0.487 | 由交易所收取  |
| 券商收入 | 万 1.613 | 由券商收取   |

上海证券交易所

| 项目   | 费用      | 收取部门   |
|------|---------|--------|
| 证管费  | 万 0.2   | 由证监会收取 |
| 经手费  | 万 0.487 | 由交易所收取 |
| 券商收入 | 万 1.813 | 由券商收取  |

##### 以万 1.2 佣金结构为例

深证证券交易所

| 项目   | 费用      | 收取部门    |
|------|---------|---------|
| 过户费  | 万 0.2   | 由中国结算收取 |
| 证管费  | 万 0.2   | 由证监会收取  |
| 经手费  | 万 0.487 | 由交易所收取  |
| 券商收入 | 万 0.313 | 由券商收取   |

上海证券交易所

| 项目   | 费用      | 收取部门   |
|------|---------|--------|
| 证管费  | 万 0.2   | 由证监会收取 |
| 经手费  | 万 0.487 | 由交易所收取 |
| 券商收入 | 万 0.513 | 由券商收取  |

##### 结构图

![交易费用](https://jfds-1252952517.cos.ap-chengdu.myqcloud.com/akshare/readme/mindmap/cost_of_trade.svg)

### 技术指标

#### 创新高

接口：stock_rank_cxg_ths

目标地址：https://data.10jqka.com.cn/rank/cxg/

描述：同花顺-数据中心-技术选股-创新高

限量：单次指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                        |
|--------|-----|-----------------------------------------------------------|
| symbol | str | symbol="创月新高"; choice of {"创月新高", "半年新高", "一年新高", "历史新高"} |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 换手率    | float64 | 注意单位: % |
| 最新价    | float64 | 注意单位: 元 |
| 前期高点   | float64 | -       |
| 前期高点日期 | object  | -       |

接口示例：

```python
import akshare as ak

stock_rank_cxg_ths_df = ak.stock_rank_cxg_ths(symbol="创月新高")
print(stock_rank_cxg_ths_df)
```

数据示例

```
    序号 股票代码  股票简称  涨跌幅 换手率  最新价  前期高点  前期高点日期
0    1  000157  中联重科   1.23  1.20   7.44   7.37  2024-01-30
1    2  000333  美的集团   2.06  0.67  59.45  58.81  2024-02-01
2    3  000423  东阿阿胶   3.25  1.86  55.25  55.09  2024-01-29
3    4  000425  徐工机械   2.71  1.57   5.69   5.68  2024-01-29
4    5  000429  粤高速A   4.28  1.54   9.75   9.35  2024-02-02
..  ..     ...   ...    ...   ...    ...    ...         ...
63  64  603156  养元饮品   5.03  0.70  24.62  23.44  2024-02-02
64  65  603195  公牛集团   2.95  0.25  96.96  95.03  2024-01-29
65  66  603369   今世缘   1.00  0.80  47.55  47.08  2024-02-02
66  67  603871  嘉友国际   4.32  1.48  19.81  19.64  2024-01-31
67  68  603960  克来机电  10.01  2.62  15.82  14.38  2024-02-02
[68 rows x 8 columns]
```

#### 创新低

接口：stock_rank_cxd_ths

目标地址：https://data.10jqka.com.cn/rank/cxd/

描述：同花顺-数据中心-技术选股-创新低

限量：单次指定 symbol 的所有数据

输入参数

| 名称     | 类型  | 描述                                                        |
|--------|-----|-----------------------------------------------------------|
| symbol | str | symbol="创月新低"; choice of {"创月新低", "半年新低", "一年新低", "历史新低"} |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 换手率    | float64 | 注意单位: % |
| 最新价    | float64 | 注意单位: 元 |
| 前期低点   | float64 | -       |
| 前期低点日期 | object  | -       |

接口示例：

```python
import akshare as ak

stock_rank_cxd_ths_df = ak.stock_rank_cxd_ths(symbol="创月新低")
print(stock_rank_cxd_ths_df)
```

数据示例

```
     序号   股票代码  股票简称  涨跌幅 换手率  最新价  前期低点  前期低点日期
0        1  000004  国华网安 -10.04  3.97  10.22  11.36  2024-02-02
1        2  000005  ST星源  -4.71  0.22   0.81   0.85  2024-02-02
2        3  000006  深振业A  -9.62  3.56   3.57   3.93  2024-02-01
3        4  000008  神州高铁  -5.37  1.17   1.94   2.05  2024-02-02
4        5  000009  中国宝安  -4.21  1.20  10.00  10.44  2024-02-02
    ...     ...   ...    ...   ...    ...    ...         ...
3535  3536  605580  恒盛能源 -10.01  4.73   8.09   8.99  2024-02-02
3536  3537  605588  冠石科技 -10.00  7.30  32.77  36.41  2024-02-02
3537  3538  605589  圣泉集团  -4.05  2.18  16.10  16.78  2024-02-02
3538  3539  605598  上海港湾  -9.97  4.55  15.35  17.05  2024-02-02
3539  3540  605599  菜百股份  -2.93  0.74  11.27  11.61  2024-02-02
[3540 rows x 8 columns]
```

#### 连续上涨

接口：stock_rank_lxsz_ths

目标地址：https://data.10jqka.com.cn/rank/lxsz/

描述：同花顺-数据中心-技术选股-连续上涨

限量：单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 股票代码  | object  | -       |
| 股票简称  | object  | -       |
| 收盘价   | float64 | 注意单位: 元 |
| 最高价   | float64 | 注意单位: 元 |
| 最低价   | float64 | 注意单位: 元 |
| 连涨天数  | int64   | -       |
| 连续涨跌幅 | float64 | 注意单位: % |
| 累计换手率 | float64 | 注意单位: % |
| 所属行业  | object  | -       |

接口示例

```python
import akshare as ak

stock_rank_lxsz_ths_df = ak.stock_rank_lxsz_ths()
print(stock_rank_lxsz_ths_df)
```

数据示例

```
    序号股票代码  股票简称  收盘价  最高价  最低价  连涨天数  连续涨跌幅 累计换手率 所属行业
0    1  603192  汇得科技   27.33   27.87   24.31     7   2.90  10.05   化学制品
1    2  688330   宏力达   98.50   99.30   83.24     7   6.41  21.48   电力设备
2    3  300870   欧陆通   67.60   67.99   58.08     5   6.46   7.75   电力设备
3    4  603507  振江股份   47.70   48.21   35.28     5  21.59  36.38   电力设备
4    5  603855  华荣股份   23.50   24.12   20.91     5   8.60   6.72   专用设备
..  ..     ...   ...     ...     ...     ...   ...    ...    ...    ...
58  59  688122  西部超导   81.26   82.50   75.37     3   5.12   5.13   国防军工
59  60  688166  博瑞医药   33.39   34.16   30.31     3   0.57   4.75   化学制药
60  61  688369  致远互联   59.30   60.97   55.22     3   6.23   1.74  计算机应用
61  62  688395  正弦电气   25.45   25.93   23.85     3   6.71  10.89  自动化设备
62  63  688788  科思科技  139.13  140.20  126.53     3   7.84   7.82   国防军工
```

#### 连续下跌

接口：stock_rank_lxxd_ths

目标地址：https://data.10jqka.com.cn/rank/lxxd/

描述：同花顺-数据中心-技术选股-连续下跌

限量：单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型      | 描述      |
|-------|---------|---------|
| 序号    | int64   | -       |
| 股票代码  | object  | -       |
| 股票简称  | object  | -       |
| 收盘价   | float64 | 注意单位: 元 |
| 最高价   | float64 | 注意单位: 元 |
| 最低价   | float64 | 注意单位: 元 |
| 连涨天数  | int64   | -       |
| 连续涨跌幅 | float64 | 注意单位: % |
| 累计换手率 | float64 | 注意单位: % |
| 所属行业  | object  | -       |

接口示例：

```python
import akshare as ak

stock_rank_lxxd_ths_df = ak.stock_rank_lxxd_ths()
print(stock_rank_lxxd_ths_df)
```

数据示例

```
     序号   股票代码  股票简称  收盘价 最高价  最低价  连涨天数 连续涨跌幅 累计换手率 所属行业
0        1  603038  华立股份   8.02  19.60   7.75    14 -55.30  178.79    建筑材料
1        2  605180  华生科技   7.38  16.87   7.38    14 -56.64  265.53    纺织制造
2        3  605255  天普股份   9.82  22.00   9.82    12 -52.24   73.00   汽车零部件
3        4  000889  ST中嘉   1.46   2.28   1.46    11 -35.40   27.92    通信服务
4        5  001217   华尔泰   8.34  17.03   8.06    11 -50.00  234.23    化学原料
    ...     ...   ...    ...    ...    ...   ...    ...     ...     ...
1315  1316  603922   金鸿顺  21.70  23.00  20.70     3  -9.43    2.94   汽车零部件
1316  1317  605218  伟时电子  13.55  19.74  13.55     3 -27.03   21.34   光学光电子
1317  1318  605296  神农集团  27.26  28.54  26.60     3  -4.35   14.78     养殖业
1318  1319  688087  英科再生  19.05  20.50  18.21     3  -7.07    3.97  化工合成材料
1319  1320  688398  赛特新材  25.79  30.80  24.37     3 -17.97    4.15    化学制品
[1320 rows x 10 columns]
```

#### 持续放量

接口: stock_rank_cxfl_ths

目标地址: https://data.10jqka.com.cn/rank/cxfl/

描述: 同花顺-数据中心-技术选股-持续放量

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 最新价    | float64 | 注意单位: 元 |
| 成交量    | object  | 注意单位: 股 |
| 基准日成交量 | object  | 注意单位: 股 |
| 放量天数   | int64   | -       |
| 阶段涨跌幅  | float64 | 注意单位: % |
| 所属行业   | object  | -       |

接口示例

```python
import akshare as ak

stock_rank_cxfl_ths_df = ak.stock_rank_cxfl_ths()
print(stock_rank_cxfl_ths_df)
```

数据示例

```
     序号    股票代码  股票简称    涨跌幅  ...      基准日成交量 放量天数  阶段涨跌幅   所属行业
0        1  000565  渝三峡A   9.97  ...   486.72万(01月26日)    8 -31.55   化学制品
1        2  000880  潍柴重机  -5.37  ...   354.53万(01月26日)    8 -29.51  汽车零部件
2        3  000959  首钢股份   2.82  ...  2486.70万(01月26日)    8  -1.80     钢铁
3        4  002159  三特索道  -6.66  ...   418.88万(01月26日)    8 -30.14  景点及旅游
4        5  002286   保龄宝  -9.55  ...   298.40万(01月26日)    8 -36.36  农产品加工
    ...     ...   ...    ...  ...               ...  ...    ...    ...
1139  1140  688656   浩欧博  -7.63  ...    67.71万(02月02日)    3 -16.46   医疗器械
1140  1141  688658  悦康药业   2.36  ...   364.38万(02月02日)    3  -1.42   化学制药
1141  1142  688677  海泰新光   4.57  ...   116.19万(02月02日)    3  13.83   医疗器械
1142  1143  688683  莱尔科技 -16.04  ...   133.01万(02月02日)    3 -31.20   消费电子
1143  1144  688701  卓锦股份 -14.72  ...   219.35万(02月02日)    3 -34.92     环保
[1144 rows x 10 columns]
```

#### 持续缩量

接口: stock_rank_cxsl_ths

目标地址: https://data.10jqka.com.cn/rank/cxsl/

描述: 同花顺-数据中心-技术选股-持续缩量

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 涨跌幅    | float64 | 注意单位: % |
| 最新价    | float64 | 注意单位: 元 |
| 成交量    | object  | 注意单位: 股 |
| 基准日成交量 | object  | 注意单位: 股 |
| 缩量天数   | int64   | -       |
| 阶段涨跌幅  | float64 | 注意单位: % |
| 所属行业   | object  | -       |

接口示例

```python
import akshare as ak

stock_rank_cxsl_ths_df = ak.stock_rank_cxsl_ths()
print(stock_rank_cxsl_ths_df)
```

数据示例

```
    序号  股票代码  股票简称 涨跌幅  ...     基准日成交量 缩量天数  阶段涨跌幅    所属行业
0      1  300350   华鹏飞 -4.59  ...     1.14亿(01月24日)   10 -37.46   计算机应用
1      2  688648  中邮科技 -5.08  ...  1622.39万(01月29日)    7 -28.90    通用设备
2      3  000058   深赛格  1.89  ...  5392.48万(01月30日)    6 -22.18      零售
3      4  600121  郑州煤电 -2.80  ...  5509.10万(01月30日)    6 -22.19  煤炭开采加工
4      5  600816  建元信托 -1.93  ...     1.94亿(01月30日)    6  -9.29   保险及其他
..   ...     ...   ...   ...  ...               ...  ...    ...     ...
112  113  688093  世华科技 -0.98  ...   483.17万(02月02日)    3  -5.13    其他电子
113  114  688293   奥浦迈  0.80  ...   170.12万(02月02日)    3   5.61    医疗服务
114  115  688334   西高院 -1.69  ...   278.80万(02月02日)    3  -1.20  其他社会服务
115  116  688376  美埃科技 -1.18  ...   158.60万(02月02日)    3   5.94      环保
116  117  688787  海天瑞声 -4.58  ...   302.30万(02月02日)    3 -17.38   计算机应用
[117 rows x 10 columns]
```

#### 向上突破

接口: stock_rank_xstp_ths

目标地址: https://data.10jqka.com.cn/rank/xstp/

描述: 同花顺-数据中心-技术选股-向上突破

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                                                   |
|--------|-----|------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="500日均线"; choice of {"5日均线", "10日均线", "20日均线", "30日均线", "60日均线", "90日均线", "250日均线", "500日均线"} |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int64   | -       |
| 股票代码 | object  | -       |
| 股票简称 | object  | -       |
| 最新价  | float64 | 注意单位: 元 |
| 成交额  | object  | 注意单位: 元 |
| 成交量  | object  | 注意单位: 股 |
| 涨跌幅  | float64 | 注意单位: % |
| 换手率  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_rank_xstp_ths_df = ak.stock_rank_xstp_ths(symbol="500日均线")
print(stock_rank_xstp_ths_df)
```

数据示例

```
   序号    股票代码  股票简称  最新价  成交额  成交量    涨跌幅    换手率
0      1  300466  赛摩智能   6.13  6.42亿  9499.20万 -11.03  22.62
1      2  300374  中铁装配  14.28  6.98亿  4488.57万 -10.69  23.72
2      3  000972  中基健康   4.58  4.62亿  8499.21万 -10.02  11.02
3      4  600630  龙头股份   9.70  8.09亿  7629.75万 -10.02  17.96
4      5  603721  中广天择  17.24  3.40亿  1934.10万 -10.02  14.88
..   ...     ...   ...    ...    ...       ...    ...    ...
385  386  603178  圣龙股份  27.89  2.87亿  1066.88万  10.02   4.51
386  387  603225   新凤鸣  13.37  2.67亿  2070.78万  10.04   1.35
387  388  688728   格科微  19.96  3.61亿  1882.35万  11.63   1.69
388  389  688127  蓝特光学  19.74  1.54亿   818.44万  12.74   3.40
389  390  300128  锦富技术   4.20  3.43亿  8830.80万  16.02   8.07
[390 rows x 8 columns]
```

#### 向下突破

接口: stock_rank_xxtp_ths

目标地址: https://data.10jqka.com.cn/rank/xxtp/

描述: 同花顺-数据中心-技术选股-向下突破

限量: 单次返回所有数据

输入参数

| 名称     | 类型  | 描述                                                                                                   |
|--------|-----|------------------------------------------------------------------------------------------------------|
| symbol | str | symbol="500日均线"; choice of {"5日均线", "10日均线", "20日均线", "30日均线", "60日均线", "90日均线", "250日均线", "500日均线"} |

输出参数

| 名称   | 类型      | 描述      |
|------|---------|---------|
| 序号   | int64   | -       |
| 股票代码 | object  | -       |
| 股票简称 | object  | -       |
| 最新价  | float64 | 注意单位: 元 |
| 成交额  | object  | 注意单位: 元 |
| 成交量  | object  | 注意单位: 股 |
| 涨跌幅  | float64 | 注意单位: % |
| 换手率  | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_rank_xxtp_ths_df = ak.stock_rank_xxtp_ths(symbol="500日均线")
print(stock_rank_xxtp_ths_df)
```

数据示例

```
     序号    股票代码   股票简称  最新价   成交额     成交量    涨跌幅   换手率
0        1  301106   骏成科技  28.00  7312.44万   234.10万 -20.00  11.23
1        2  300405   科隆股份   3.61     1.25亿  3107.45万 -19.96  14.20
2        3  300391   长药控股   3.16  8421.59万  2479.90万 -19.80   7.08
3        4  300716   泉为科技   5.01  6863.18万  1282.18万 -18.40   8.01
4        5  300710   万隆光电  11.06  6636.05万   559.99万 -17.95   6.30
    ...     ...    ...    ...       ...       ...    ...    ...
3267  3268  000565   渝三峡A   3.97     1.00亿  2680.46万   9.97   6.18
3268  3269  605155    西大门  10.91     1.04亿  1054.60万   9.98   7.85
3269  3270  002453   华软科技   6.05  9283.25万  1578.67万  10.00   2.60
3270  3271  002819   东方中科  22.40     1.41亿   649.21万  10.02   2.77
3271  3272  300799  *ST左江  20.86     2.54亿  1554.33万  20.02  23.00
[3272 rows x 8 columns]
```

#### 量价齐升

接口: stock_rank_ljqs_ths

目标地址: https://data.10jqka.com.cn/rank/ljqs/

描述: 同花顺-数据中心-技术选股-量价齐升

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 最新价    | float64 | 注意单位: 元 |
| 量价齐升天数 | int64   | -       |
| 阶段涨幅   | float64 | 注意单位: % |
| 累计换手率  | float64 | 注意单位: % |
| 所属行业   | object  | -       |

接口示例

```python
import akshare as ak

stock_rank_ljqs_ths_df = ak.stock_rank_ljqs_ths()
print(stock_rank_ljqs_ths_df)
```

数据示例

```
    序号    股票代码  股票简称  最新价  量价齐升天数   阶段涨幅  累计换手率    所属行业
0        1  000423  东阿阿胶  60.12       4  14.51   6.77      中药
1        2  000932  华菱钢铁   5.71       4   5.94   5.57      钢铁
2        3  002028  思源电气  57.99       4  14.02   7.04    电力设备
3        4  002444  巨星科技  22.04       4  18.88   6.16    通用设备
4        5  002595  豪迈科技  32.80       4  21.48   4.45    专用设备
    ...     ...   ...    ...     ...    ...    ...     ...
1588  1589  688711  宏微科技  26.54       1   6.29   5.47  半导体及元件
1589  1590  688739  成大生物  28.90       1   2.88   1.99    生物制品
1590  1591  688798  艾为电子  54.03       1   3.31   2.17  半导体及元件
1591  1592  688799  华纳药厂  36.95       1   4.23   4.11    化学制药
1592  1593  689009  九号公司  28.04       1   7.68   1.76   非汽车交运
[1593 rows x 8 columns]
```

#### 量价齐跌

接口: stock_rank_ljqd_ths

目标地址: https://data.10jqka.com.cn/rank/ljqd/

描述: 同花顺-数据中心-技术选股-量价齐跌

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述      |
|--------|---------|---------|
| 序号     | int64   | -       |
| 股票代码   | object  | -       |
| 股票简称   | object  | -       |
| 最新价    | float64 | 注意单位: 元 |
| 量价齐跌天数 | int64   | -       |
| 阶段涨幅   | float64 | 注意单位: % |
| 累计换手率  | float64 | 注意单位: % |
| 所属行业   | object  | -       |

接口示例

```python
import akshare as ak

stock_rank_ljqd_ths_df = ak.stock_rank_ljqd_ths()
print(stock_rank_ljqd_ths_df)
```

数据示例

```
    序号    股票代码  股票简称  最新价  量价齐跌天数  阶段涨幅 累计换手率  所属行业
0        1  600322  津投城开   1.53       5 -22.73   21.21   房地产开发
1        2  600622  光大嘉宝   2.21       5 -28.94   45.42   房地产开发
2        3  603082  北自科技  25.84       5 -45.83  248.34    通用设备
3        4  002269  美邦服饰   1.53       4 -17.30   20.10    服装家纺
4        5  002523  天桥起重   2.52       4 -22.94   24.58    专用设备
    ...     ...   ...    ...     ...    ...     ...     ...
1464  1465  688719  爱科赛博  44.18       1  -3.85    4.42    电力设备
1465  1466  688722   同益中  10.76       1  -0.46    1.45  化工合成材料
1466  1467  688737  中自科技  19.15       1  -5.53    1.59   汽车零部件
1467  1468  688767  博拓生物  23.20       1  -2.93    1.80    医疗器械
1468  1469  688787  海天瑞声  40.17       1  -4.58    5.64   计算机应用
[1469 rows x 8 columns]
```

#### 险资举牌

接口: stock_rank_xzjp_ths

目标地址: https://data.10jqka.com.cn/financial/xzjp/

描述: 同花顺-数据中心-技术选股-险资举牌

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称         | 类型      | 描述      |
|------------|---------|---------|
| 序号         | int64   | -       |
| 举牌公告日      | object  | -       |
| 股票代码       | object  | -       |
| 股票简称       | object  | -       |
| 现价         | float64 | 注意单位: 元 |
| 涨跌幅        | float64 | 注意单位: % |
| 举牌方        | object  | -       |
| 增持数量       | object  | 注意单位: 股 |
| 交易均价       | float64 | 注意单位: 元 |
| 增持数量占总股本比例 | float64 | 注意单位: % |
| 变动后持股总数    | object  | 注意单位: 股 |
| 变动后持股比例    | float64 | 注意单位: % |

接口示例

```python
import akshare as ak

stock_rank_xzjp_ths_df = ak.stock_rank_xzjp_ths()
print(stock_rank_xzjp_ths_df)
```

数据示例

```
    序号       举牌公告日    股票代码   股票简称  ...   交易均价  增持数量占总股本比例   变动后持股总数 变动后持股比例
0    1  2025-01-04  600582   天地科技  ...   6.23        0.10     2.07亿    5.00
1    2  2024-12-24  600236   桂冠电力  ...   6.60        1.00    10.25亿   13.00
2    3  2024-12-24  002005   ST德豪  ...   1.58        0.67     1.23亿    7.04
3    4  2024-12-17  600016   民生银行  ...   2.92        0.04    21.89亿    5.00
4    5  2024-12-10  300036   超图软件  ...  19.39        4.80  2538.69万    5.15
5    6  2024-12-07  601330   绿色动力  ...   6.38        0.03     1.14亿    8.16
6    7  2024-11-27  600527   江南高纤  ...   1.97        0.02     1.10亿    6.35
7    8  2024-11-21  600113   浙江东日  ...  10.16        2.53  2129.42万    5.05
8    9  2024-11-19  001288   运机集团  ...  28.14        0.99   987.89万    5.93
9   10  2024-11-15  300436    广生堂  ...  24.50        5.02   800.00万    5.02
10  11  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
11  12  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
12  13  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
13  14  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
14  15  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
15  16  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
16  17  2024-11-15  000088    盐田港  ...   0.00        6.66     2.85亿    5.49
17  18  2024-11-14  601607   上海医药  ...  21.23        0.31     1.87亿    5.04
18  19  2024-11-14  600511   国药股份  ...  35.22        0.20  3824.44万    5.07
19  20  2024-11-13  300201    海伦哲  ...   5.07        0.85  4974.50万    5.00
20  21  2024-11-12  300499   高澜股份  ...   0.00        0.00  1729.40万    5.67
21  22  2024-11-11  300499   高澜股份  ...  13.77        5.67  1729.40万    5.67
22  23  2024-11-09  688118   普元信息  ...  22.11        5.01   471.13万    5.01
23  24  2024-11-08  002488   金固股份  ...   0.00        0.15  4978.08万    5.00
24  25  2024-10-31  601963   重庆银行  ...   7.51        0.10     1.75亿    5.03
25  26  2024-10-31  601963   重庆银行  ...   7.51        0.00     1.75亿    5.03
26  27  2024-10-17  000007    全新好  ...   5.38        5.00  1732.25万    5.00
27  28  2024-09-26  002870   香山股份  ...  28.73        0.85  2493.19万   18.88
28  29  2024-09-14  002870   香山股份  ...  28.98        0.38  2380.74万   18.03
29  30  2024-09-11  002052  *ST同洲  ...   2.38        0.45  5735.96万    7.69
30  31  2024-09-07  002870   香山股份  ...  28.87        0.63  2330.34万   17.64
31  32  2024-09-06  002005   ST德豪  ...   1.27        1.16     1.12亿    6.36
32  33  2024-09-03  002052  *ST同洲  ...   2.26        0.32  5396.98万    7.24
33  34  2024-08-30  601199   江南水务  ...   5.31        1.00  5611.41万    6.00
34  35  2024-08-28  002870   香山股份  ...  27.93        0.52  2246.98万   17.01
35  36  2024-08-24  002870   香山股份  ...  28.95        1.05  2178.52万   16.49
36  37  2024-08-23  002052  *ST同洲  ...   1.54        0.13  5155.56万    6.91
37  38  2024-08-10  603018   华设集团  ...   8.42        0.37  3612.86万    5.28
38  39  2024-08-10  603018   华设集团  ...   8.42        0.37  3612.86万    5.28
39  40  2024-08-08  002870   香山股份  ...  28.58        1.01  2039.34万   15.44
40  41  2024-08-03  002005   ST德豪  ...   1.00        5.21  9127.13万    5.21
41  42  2024-08-02  601330   绿色动力  ...   2.64        5.03  7015.00万    5.03
42  43  2024-07-27  001288   运机集团  ...  26.59        5.14   822.59万    5.14
43  44  2024-07-26  601963   重庆银行  ...   5.03        0.00     1.77亿    5.10
44  45  2024-07-26  601963   重庆银行  ...   7.97        0.00     2.95亿    8.50
[45 rows x 12 columns]
```

### ESG 评级

#### ESG 评级数据

接口: stock_esg_rate_sina

目标地址: https://finance.sina.com.cn/esg/grade.shtml

描述: 新浪财经-ESG评级中心-ESG评级-ESG评级数据

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称    | 类型     | 描述 |
|-------|--------|----|
| 成分股代码 | object | -  |
| 评级机构  | object | -  |
| 评级    | object | -  |
| 评级季度  | object | -  |
| 标识    | object | -  |
| 交易市场  | object | -  |

接口示例

```python
import akshare as ak

stock_esg_rate_sina_df = ak.stock_esg_rate_sina()
print(stock_esg_rate_sina_df)
```

数据示例

```
    成分股代码              评级机构     评级    评级季度             标识 交易市场
0      SZ000001             中财绿金院     A-  2022Q4            NaN   cn
1      SZ000001              商道融绿     B+  2022Q4            NaN   cn
2      SZ000001                盟浪      A  2022Q2            NaN   cn
3      SZ000001               中诚信    AA-  2023Q3            NaN   cn
4      SZ000001  晨星Sustainalytics  24.96  2022Q4  Comprehensive   cn
         ...               ...    ...     ...            ...  ...
46888   HK02361                盟浪      -  2022Q2            NaN   hk
46889   HK02361               中诚信     BB  2023Q3            NaN   hk
46890   HK02361  晨星Sustainalytics      -  2022Q4                  hk
46891   HK02361                妙盈      -  2022Q2            NaN   hk
46892   HK02361             华测CTI      -  2022Q1            NaN   hk
[46893 rows x 6 columns]
```

#### MSCI

接口: stock_esg_msci_sina

目标地址: https://finance.sina.com.cn/esg/grade.shtml

描述: 新浪财经-ESG评级中心-ESG评级-MSCI

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述 |
|--------|---------|----|
| 股票代码   | object  | -  |
| ESG评分  | object  | -  |
| 环境总评   | float64 | -  |
| 社会责任总评 | float64 | -  |
| 治理总评   | float64 | -  |
| 评级日期   | object  | -  |
| 交易市场   | object  | -  |

接口示例

```python
import akshare as ak

stock_esg_msci_sina_df = ak.stock_esg_msci_sina()
print(stock_esg_msci_sina_df)
```

数据示例

```
        股票代码 ESG评分 环境总评 社会责任总评 治理总评 评级日期 交易市场
0      00019.HK   AAA   7.0     7.3   6.3  2024-04-24   HK
1     000513.SZ   AAA   6.8     6.4   6.1  2024-06-25   CN
2      00066.HK   AAA   7.2     5.6   6.5  2024-07-11   HK
3      00087.HK   AAA   7.0     7.3   6.3  2024-04-24   HK
4      00992.HK   AAA   5.1     6.2   5.5  2024-07-08   HK
...         ...   ...   ...     ...   ...         ...  ...
4619   UONEK.US   CCC   7.0     3.2   1.0  2024-06-24   US
4620     UVE.US   CCC   1.9     2.0   5.4  2024-07-08   US
4621     VTS.US   CCC   1.1     2.4   6.8  2024-04-24   US
4622    WULF.US   CCC   0.3     1.3   3.9  2024-05-20   US
4623    ZETA.US   CCC   6.7     3.1   2.0  2024-06-24   US
[4624 rows x 7 columns]
```

#### 路孚特

接口: stock_esg_rft_sina

目标地址: https://finance.sina.com.cn/esg/grade.shtml

描述: 新浪财经-ESG评级中心-ESG评级-路孚特

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称       | 类型      | 描述 |
|----------|---------|----|
| 股票代码     | object  | -  |
| ESG评分    | object  | -  |
| ESG评分日期  | object  | -  |
| 环境总评     | float64 | -  |
| 环境总评日期   | float64 | -  |
| 社会责任总评   | float64 | -  |
| 社会责任总评日期 | object  | -  |
| 治理总评     | object  | -  |
| 治理总评日期   | object  | -  |
| 争议总评     | object  | -  |
| 争议总评日期   | object  | -  |
| 行业       | object  | -  |
| 交易所      | object  | -  |

接口示例

```python
import akshare as ak

stock_esg_rft_sina_df = ak.stock_esg_rft_sina()
print(stock_esg_rft_sina_df)
```

数据示例

```
           股票代码     ESG评分     ESG评分日期  ...      争议总评日期          行业   交易所
0          MNTX   10.9(D)  2024-03-23  ...  2023-08-26  卡车-建筑-农用机械  纳斯达克
1     600958.SH  52.3(B-)  2024-03-23  ...  2023-08-26         证券Ⅱ   上交所
2     000799.SZ  27.4(C-)  2024-03-23  ...  2023-08-26         白酒Ⅱ   深交所
3     600745.SH  52.5(B-)  2024-03-23  ...  2023-09-09        消费电子   上交所
4         DISCA   59.9(B)  2024-03-23  ...  2024-03-16     有线-卫星电视  纳斯达克
         ...       ...         ...  ...         ...         ...   ...
5418    BILI.US   35.3(C)  2024-03-23  ...  2024-03-23        ????  纳斯达克
5419    TCOM.US  29.0(C-)  2024-03-23  ...  2023-12-16        ????  纳斯达克
5420     HCM.US  52.4(B-)  2024-03-23  ...  2023-08-26     ???????  纳斯达克
5421    XPEV.US   60.4(B)  2024-03-23  ...  2023-08-26          ??   纽交所
5422      LI.US   35.7(C)  2024-03-23  ...  2023-09-23          ??  纳斯达克
[5423 rows x 13 columns]
```

#### 秩鼎

接口: stock_esg_zd_sina

目标地址: https://finance.sina.com.cn/esg/grade.shtml

描述: 新浪财经-ESG评级中心-ESG评级-秩鼎

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型     | 描述 |
|--------|--------|----|
| 股票代码   | object | -  |
| ESG评分  | object | -  |
| 环境总评   | object | -  |
| 社会责任总评 | object | -  |
| 治理总评   | object | -  |
| 评分日期   | object | -  |


接口示例

```python
import akshare as ak

stock_esg_zd_sina_df = ak.stock_esg_zd_sina()
print(stock_esg_zd_sina_df)
```

数据示例

```
       股票代码      ESG评分     环境总评   社会责任总评   治理总评    评分日期
0     600775.SH   70.96(AA)    68.36(A)    65.06(A)   78.76(AA)  2024-03-31
1     601717.SH   78.64(AA)  87.28(AAA)  57.09(BBB)  89.15(AAA)  2024-03-31
2     600801.SH   79.78(AA)  85.55(AAA)    68.95(A)  82.06(AAA)  2024-03-31
3     601618.SH   78.31(AA)  86.90(AAA)    64.58(A)  81.37(AAA)  2024-03-31
4     600808.SH   78.17(AA)   78.22(AA)    67.38(A)  87.46(AAA)  2024-03-31
         ...         ...         ...         ...         ...         ...
7670   00247.HK   70.46(AA)   78.32(AA)    63.93(A)    67.50(A)  2024-03-31
7671  301016.SZ   51.54(BB)   36.89(CC)    46.12(B)   73.94(AA)  2024-03-31
7672   03309.HK  56.38(BBB)  56.22(BBB)   35.18(CC)   72.42(AA)  2024-03-31
7673  301166.SZ   51.82(BB)    27.78(C)   39.81(CC)  81.81(AAA)  2024-03-31
7674  837212.BJ    46.57(B)    34.34(C)   51.46(BB)  55.51(BBB)  2024-03-31
[7675 rows x 6 columns]
```

#### 华证指数

接口: stock_esg_hz_sina

目标地址: https://finance.sina.com.cn/esg/grade.shtml

描述: 新浪财经-ESG评级中心-ESG评级-华证指数

限量: 单次返回所有数据

输入参数

| 名称  | 类型  | 描述  |
|-----|-----|-----|
| -   | -   | -   |

输出参数

| 名称     | 类型      | 描述 |
|--------|---------|----|
| 日期     | object  | -  |
| 股票代码   | object  | -  |
| 交易市场   | object  | -  |
| 股票名称   | object  | -  |
| ESG评分  | float64 | -  |
| ESG等级  | object  | -  |
| 环境     | float64 | -  |
| 环境等级   | object  | -  |
| 社会     | float64 | -  |
| 社会等级   | object  | -  |
| 公司治理   | float64 | -  |
| 公司治理等级 | object  | -  |

接口示例

```python
import akshare as ak

stock_esg_hz_sina_df = ak.stock_esg_hz_sina()
print(stock_esg_hz_sina_df)
```

数据示例

```
        日期       股票代码 交易市场   股票名称  ESG评分  ... 环境等级     社会 社会等级   公司治理 公司治理等级
0     2023-04-30  300760.SZ   cn   迈瑞医疗  88.85  ...  BBB  92.77   AA  89.90      A
1     2023-04-30  601825.SH   cn   沪农商行  88.09  ...  BBB  88.64    A  89.79      A
2     2023-04-30  601166.SH   cn   兴业银行  88.08  ...   AA  86.00    A  88.78      A
3     2023-04-30  300021.SZ   cn   大禹节水  87.91  ...    A  91.78   AA  85.77      A
4     2023-04-30  600803.SH   cn   新奥股份  87.90  ...    A  90.49   AA  85.99      A
          ...        ...  ...    ...    ...  ...  ...    ...  ...    ...    ...
5159  2023-04-30  000150.SZ   cn  *ST宜康  60.24  ...    C  69.49  CCC  53.14      C
5160  2023-04-30  600242.SH   cn  *ST中昌  59.79  ...    C  68.36  CCC  54.08      C
5161  2023-04-30  000752.SZ   cn  *ST西发  59.55  ...    C  67.36  CCC  56.29      C
5162  2023-04-30  688086.SH   cn  *ST紫晶  57.11  ...   CC  63.37   CC  47.49      C
5163  2023-04-30  000038.SZ   cn  *ST大通  56.95  ...   CC  65.05  CCC  49.79      C
[5164 rows x 12 columns]
```
