######################################################################
# 2024-12: Backwards compatibility for Toga < 0.5.0
######################################################################
import warnings

from travertino.constants import (
    BOTTOM,
    CENTER,
    COLUMN,
    END,
    LEFT,
    LTR,
    RIGHT,
    ROW,
    RTL,
    START,
    TOP,
)
from travertino.properties.aliased import Condition
from travertino.properties.validated import validated_property

# Make sure deprecation warnings are shown by default
warnings.filterwarnings("default", category=DeprecationWarning)


class _AlignmentCondition(Condition):
    def __init__(self, main_value, /, **properties):
        super().__init__(**properties)
        self.main_value = main_value

    def match(self, style, main_name=None):
        # main_name can't be accessed the "normal" way without causing a loop; we need
        # to access the private stored value.
        return (
            super().match(style) and getattr(style, f"_{main_name}") == self.main_value
        )


class _alignment_property(validated_property):
    # Alignment is deprecated in favor of align_items, but the two share a complex
    # relationship because they don't use the same set of values; translating from one
    # to the other may require knowing the value of direction and text_direction as
    # well.

    # Both names exist as actual properties on the style object. If one of them has been
    # set, that one is the source of truth; if the other name is requested, its value
    # is computed / translated. They can never both be set at the same time; setting
    # one deletes any value stored in the other.

    def __set_name__(self, owner, name):
        # Hard-coded because it's only called on alignment, not align_items.

        self.name = "alignment"
        owner._BASE_ALL_PROPERTIES[owner].add("alignment")
        self.other = "align_items"
        self.derive = {
            _AlignmentCondition(CENTER): CENTER,
            _AlignmentCondition(START, direction=COLUMN, text_direction=LTR): LEFT,
            _AlignmentCondition(START, direction=COLUMN, text_direction=RTL): RIGHT,
            _AlignmentCondition(START, direction=ROW): TOP,
            _AlignmentCondition(END, direction=COLUMN, text_direction=LTR): RIGHT,
            _AlignmentCondition(END, direction=COLUMN, text_direction=RTL): LEFT,
            _AlignmentCondition(END, direction=ROW): BOTTOM,
        }

        # Replace the align_items validated_property with another instance of this
        # class. This is needed so accessing or setting either one will properly
        # reference the other.
        owner.align_items = _alignment_property(START, CENTER, END)
        owner.align_items.name = "align_items"
        owner.align_items.other = "alignment"
        owner.align_items.derive = {
            # Invert each condition so that it maps in the opposite direction.
            _AlignmentCondition(result, **condition.properties): condition.main_value
            for condition, result in self.derive.items()
        }

    def __get__(self, obj, objtype=None):
        if obj is None:
            return self

        self.warn_if_deprecated()

        if hasattr(obj, f"_{self.other}"):
            # If the other property is set, attempt to translate.
            for condition, value in self.derive.items():
                if condition.match(obj, main_name=self.other):
                    return value

        # If the other property isn't set (or no condition is valid), access this
        # property as usual.
        return super().__get__(obj)

    def __set__(self, obj, value):
        if value is self:
            # This happens during autogenerated dataclass __init__ when no value is
            # supplied.
            return

        self.warn_if_deprecated()

        # Delete the other property when setting this one.
        try:
            delattr(obj, f"_{self.other}")
        except AttributeError:
            pass
        super().__set__(obj, value)

    def __delete__(self, obj):
        self.warn_if_deprecated()

        # Delete the other property too.
        try:
            delattr(obj, f"_{self.other}")
        except AttributeError:
            pass
        super().__delete__(obj)

    def is_set_on(self, obj):
        self.warn_if_deprecated()

        # Counts as set if *either* of the two properties is set.
        return super().is_set_on(obj) or hasattr(obj, f"_{self.other}")

    def warn_if_deprecated(self):
        if self.name == "alignment":
            warnings.warn(
                "Pack.alignment is deprecated. Use Pack.align_items instead.",
                DeprecationWarning,
                stacklevel=3,
            )


######################################################################
# End backwards compatibility
######################################################################
