from http.cookiejar import Cookie, CookieJar

from PySide6.QtCore import QUrl, Signal
from PySide6.QtWebEngineCore import QWebEnginePage
from PySide6.QtWebEngineWidgets import QWebEngineView
from travertino.size import at_least

from toga.widgets.webview import CookiesResult, JavaScriptResult

from .base import Widget


class TogaWebViewPage(QWebEnginePage):
    """A QWebEnginePage that exposes a signal for JavaScript console messages."""

    #: Signal that emits console messages generated by JavaScript.
    consoleMessage = Signal(
        QWebEnginePage.JavaScriptConsoleMessageLevel,
        str,
        int,
        str,
    )

    def javaScriptConsoleMessage(
        self,
        level: QWebEnginePage.JavaScriptConsoleMessageLevel,
        message: str,
        line: int,
        sourceID: str,
    ):
        """Forward console messages to the consoleMessage signal."""
        self.consoleMessage.emit(level, message, line, sourceID)


class JavaScriptHandler:
    """A callback handler for running JavaScript.

    To catch JavaScript errors, connect handle_console_message to the consoleMessage
    signal in a TogaWebViewPage.
    """

    def __init__(self, result: JavaScriptResult):
        self.result = result
        self.error = None

    def __call__(self, value):
        if self.error:
            exc = RuntimeError(self.error)
            self.result.set_exception(exc)
        else:
            self.result.set_result(value)

    def handle_console_message(
        self,
        level: QWebEnginePage.JavaScriptConsoleMessageLevel,
        message: str,
        line: int,
        sourceID: str,
    ):
        """Signal handler for JavaScript console messages."""
        # only record errors, don't want tests which cover other cases
        if (
            level == QWebEnginePage.JavaScriptConsoleMessageLevel.ErrorMessageLevel
        ):  # pragma: no branch
            self.error = (
                f"JavaScriptError: message={message!r}, "
                f"line={line}, source={sourceID!r}"
            )


class WebView(Widget):
    SUPPORTS_ON_NAVIGATION_STARTING = False

    def create(self):
        self._cookies = CookieJar()
        self.load_future = None

        self.native = QWebEngineView()
        page = TogaWebViewPage(self.native)
        self.native.setPage(page)

        self.native.page().profile().cookieStore().cookieAdded.connect(
            self.qt_on_cookie_added
        )
        self.native.loadFinished.connect(self.qt_on_webview_load)

    def qt_on_webview_load(self, ok: bool):
        self.interface.on_webview_load()
        if self.load_future:
            self.load_future.set_result(None)
            self.load_future = None

    def qt_on_cookie_added(self, qt_cookie):
        if qt_cookie.isSessionCookie():
            expiration = None
        else:
            expiration = qt_cookie.expirationDate().toPython().timestamp()
        cookie = Cookie(
            version=None,
            name=bytearray(qt_cookie.name()).decode(),
            value=bytearray(qt_cookie.value()).decode(),
            port=None,
            port_specified=False,
            domain=qt_cookie.domain(),
            domain_specified=bool(qt_cookie.domain()),
            domain_initial_dot=(qt_cookie.domain().startswith(".")),
            path=qt_cookie.path(),
            path_specified=True,
            expires=expiration,
            secure=qt_cookie.isSecure(),
            discard=qt_cookie.isSessionCookie(),
            comment=None,
            comment_url=None,
            rest={},
        )
        self._cookies.set_cookie(cookie)

    def get_url(self):
        url = self.native.url().toString()
        return None if url == "about:blank" else url

    def set_url(self, value, future=None):
        self.load_future = future
        if value:
            self.native.setUrl(value)
        else:
            self.native.setUrl(QUrl())

    def get_user_agent(self):
        return self.native.page().profile().httpUserAgent()

    def set_user_agent(self, value):
        # replace user agent of webview (webview has own one)
        return self.native.page().profile().setHttpUserAgent(value)

    def set_content(self, root_url, content):
        self.native.setHtml(content, baseUrl=root_url)

    def get_cookies(self):
        result = CookiesResult()
        # copy current state of cookies
        cookie_jar = CookieJar()
        for cookie in self._cookies:
            cookie_jar.set_cookie(cookie)
        result.set_result(cookie_jar)
        return result

    def evaluate_javascript(self, javascript, on_result=None):
        result = JavaScriptResult(on_result)

        handler = JavaScriptHandler(result)
        page = self.native.page()
        page.consoleMessage.connect(handler.handle_console_message)
        page.runJavaScript(javascript, handler)

        # wait for the future, and return the result
        return result

    def rehint(self):
        size = self.native.sizeHint()
        self.interface.intrinsic.width = at_least(
            max(size.width(), self.interface._MIN_WIDTH)
        )
        self.interface.intrinsic.height = at_least(
            max(size.height(), self.interface._MIN_HEIGHT)
        )
