import {Model} from "../model"
import type {HasProps} from "./has_props"
import type {Attrs} from "./types"
import {isPlainObject} from "./util/types"
import {assert} from "./util/assert"
import type {GeometryData} from "./geometry"
import type {Class} from "./class"
import type {KeyModifiers} from "./ui_gestures"
import type {Serializable, Serializer} from "./serialization"
import {serialize} from "./serialization"
import {Deserializer} from "./serialization/deserializer"
import type {Equatable, Comparator} from "./util/eq"
import {equals} from "./util/eq"
import type {Legend} from "../models/annotations/legend"
import type {Axis} from "../models/axes/axis"
import type {LegendItem} from "../models/annotations/legend_item"
import type {Factor} from "../models/ranges/factor_range"
import type {ClearInput} from "../models/widgets/input_widget"
import type {ClientConnection} from "../client/connection"

Deserializer.register("event", (rep: BokehEventRep, deserializer: Deserializer): BokehEvent => {
  const cls = deserializable_events.get(rep.name)
  if (cls !== undefined && cls.from_values != null) {
    const values = deserializer.decode(rep.values)
    assert(isPlainObject(values))
    return cls.from_values(values)
  } else {
    deserializer.error(`deserialization of '${rep.name}' event is not supported`)
  }
})

export type BokehEventType =
  DocumentEventType |
  ModelEventType

export type DocumentEventType =
  "document_ready" |
  ConnectionEventType

export type ConnectionEventType =
  "connection_lost" |
  "client_reconnected"

export type ModelEventType =
  "axis_click" |
  "button_click" |
  "legend_item_click" |
  "menu_item_click" |
  "value_submit" |
  UIEventType

export type UIEventType =
  "lodstart" |
  "lodend" |
  "rangesupdate" |
  "selectiongeometry" |
  "reset" |
  PointEventType

export type PointEventType =
  "pan" |
  "pinch" |
  "rotate" |
  "wheel" |
  "mousemove" |
  "mouseenter" |
  "mouseleave" |
  "tap" |
  "doubletap" |
  "press" |
  "pressup" |
  "panstart" |
  "panend" |
  "pinchstart" |
  "pinchend" |
  "rotatestart" |
  "rotateend"

/**
 * Events known to bokeh by name, for type-safety of Model.on_event(event_name, (EventType) => void).
 * Other events, including user defined events, can be referred to by event's class object.
 */
export type BokehEventMap = {
  axis_click: AxisClick
  button_click: ButtonClick
  clear_input: ClearInput
  connection_lost: ConnectionLost
  client_reconnected: ClientReconnected
  document_ready: DocumentReady
  doubletap: DoubleTap
  legend_item_click: LegendItemClick
  lodend: LODEnd
  lodstart: LODStart
  menu_item_click: MenuItemClick
  mouseenter: MouseEnter
  mouseleave: MouseLeave
  mousemove: MouseMove
  pan: Pan
  panend: PanEnd
  panstart: PanStart
  pinch: Pinch
  pinchend: PinchEnd
  pinchstart: PinchStart
  press: Press
  pressup: PressUp
  rangesupdate: RangesUpdate
  reset: Reset
  rotate: Rotate
  rotateend: RotateEnd
  rotatestart: RotateStart
  selectiongeometry: SelectionGeometry
  tap: Tap
  value_submit: ValueSubmit
  wheel: MouseWheel
}

export type BokehEventRep = {
  type: "event"
  name: string
  values: unknown
}

function event(event_name: string) {
  return (cls: Class<BokehEvent>) => {
    cls.prototype.event_name = event_name
  }
}

const deserializable_events: Map<string, typeof BokehEvent> = new Map()

/**
 * Marks and registers a class as a one way (server -> client) event.
 */
export function server_event(event_name: string) {
  return (cls: Class<BokehEvent>) => {
    if (deserializable_events.has(event_name)) {
      throw new Error(`'${event_name}' event is already registered`)
    }
    deserializable_events.set(event_name, cls)
    cls.prototype.event_name = event_name
    cls.prototype.publish = false
  }
}

export abstract class BokehEvent implements Serializable, Equatable {
  declare event_name: string
  declare publish: boolean

  [serialize](serializer: Serializer): BokehEventRep {
    const {event_name: name, event_values} = this
    const values = serializer.encode(event_values)
    return {type: "event", name, values}
  }

  [equals](that: this, cmp: Comparator): boolean {
    return this.event_name == that.event_name && cmp.eq(this.event_values, that.event_values)
  }

  protected abstract get event_values(): Attrs

  static from_values?(values: Attrs): BokehEvent

  static {
    this.prototype.publish = true
  }
}

export abstract class ModelEvent extends BokehEvent {
  origin: HasProps | null = null

  protected get event_values(): Attrs {
    return {model: this.origin}
  }
}

export abstract class UserEvent extends ModelEvent {
  constructor(readonly values: Attrs) {
    super()
  }

  protected override get event_values(): Attrs {
    return {...super.event_values, ...this.values}
  }

  static override from_values(values: Attrs): UserEvent {
    const origin = (() => {
      if ("model" in values) {
        const {model} = values
        assert(model === null || model instanceof Model)
        delete values.model
        return model
      } else {
        return null
      }
    })()
    const event = new (this as any)(values)
    event.origin = origin
    return event
  }
}

export abstract class DocumentEvent extends BokehEvent {}

@event("document_ready")
export class DocumentReady extends DocumentEvent {
  protected get event_values(): Attrs {
    return {}
  }
}

export abstract class ConnectionEvent extends DocumentEvent {}

/**
 * Announce when a WebSocket connection was disconnected.
 *
 * @member timestamp when the last connection attempt was made
 * @member attempts  the number of times reconnection was attempted
 * @member timeout   milliseconds till next reconnection attempt or `null`
 *                   indicating that no further attempts will be made
 */
export class ConnectionLost extends ConnectionEvent {
  readonly timestamp = Date.now()

  constructor(private readonly connection: WeakRef<ClientConnection>, readonly attempts: number, readonly timeout: number | null) {
    super()
  }

  protected get event_values(): Attrs {
    const {timestamp, attempts, timeout} = this
    return {timestamp, attempts, timeout}
  }

  static {
    this.prototype.event_name = "connection_lost"
    this.prototype.publish = false
  }

  reconnect(): void {
    void this.connection.deref()?.reconnect()
  }
}

/**
 * Announce when a connection to the client has been reconnected.
 */
export class ClientReconnected extends ConnectionEvent {

  protected get event_values(): Attrs {
    return {}
  }

  static {
    this.prototype.event_name = "client_reconnected"
  }
}

@event("axis_click")
export class AxisClick extends ModelEvent {

  constructor(readonly model: Axis, readonly value: number | Factor) {
    super()
  }

  protected override get event_values(): Attrs {
    const {value} = this
    return {...super.event_values, value}
  }
}

@event("button_click")
export class ButtonClick extends ModelEvent {}

@event("legend_item_click")
export class LegendItemClick extends ModelEvent {

  constructor(readonly model: Legend, readonly item: LegendItem) {
    super()
  }

  protected override get event_values(): Attrs {
    const {item} = this
    return {...super.event_values, item}
  }
}

@event("menu_item_click")
export class MenuItemClick extends ModelEvent {

  constructor(readonly item: string) {
    super()
  }

  protected override get event_values(): Attrs {
    const {item} = this
    return {...super.event_values, item}
  }
}

@event("value_submit")
export class ValueSubmit extends ModelEvent {

  constructor(readonly value: string) {
    super()
  }

  protected override get event_values(): Attrs {
    const {value} = this
    return {...super.event_values, value}
  }
}

// A UIEvent is an event originating on a canvas this includes.
// DOM events such as keystrokes as well as hammer, LOD, and range events.
export abstract class UIEvent extends ModelEvent {}

@event("lodstart")
export class LODStart extends UIEvent {}

@event("lodend")
export class LODEnd extends UIEvent {}

@event("rangesupdate")
export class RangesUpdate extends UIEvent {

  constructor(readonly x0: number, readonly x1: number, readonly y0: number, readonly y1: number) {
    super()
  }

  protected override get event_values(): Attrs {
    const {x0, x1, y0, y1} = this
    return {...super.event_values, x0, x1, y0, y1}
  }
}

@event("selectiongeometry")
export class SelectionGeometry extends UIEvent {

  constructor(readonly geometry: GeometryData, readonly final: boolean) {
    super()
  }

  protected override get event_values(): Attrs {
    const {geometry, final} = this
    return {...super.event_values, geometry, final}
  }
}

@event("reset")
export class Reset extends UIEvent {}

export abstract class PointEvent extends UIEvent {

  constructor(
    readonly sx: number, readonly sy: number,
    readonly x: number, readonly y: number,
    readonly modifiers: KeyModifiers,
  ) {
    super()
  }

  protected override get event_values(): Attrs {
    const {sx, sy, x, y, modifiers} = this
    return {...super.event_values, sx, sy, x, y, modifiers}
  }
}

@event("pan")
export class Pan extends PointEvent {

  /* TODO: direction: -1 | 1 */
  constructor(
    sx: number, sy: number,
    x: number, y: number,
    readonly delta_x: number, readonly delta_y: number,
    modifiers: KeyModifiers,
  ) {
    super(sx, sy, x, y, modifiers)
  }

  protected override get event_values(): Attrs {
    const {delta_x, delta_y/*, direction*/} = this
    return {...super.event_values, delta_x, delta_y/*, direction*/}
  }
}

@event("pinch")
export class Pinch extends PointEvent {

  constructor(
    sx: number, sy: number,
    x: number, y: number,
    readonly scale: number,
    modifiers: KeyModifiers,
  ) {
    super(sx, sy, x, y, modifiers)
  }

  protected override get event_values(): Attrs {
    const {scale} = this
    return {...super.event_values, scale}
  }
}

@event("rotate")
export class Rotate extends PointEvent {

  constructor(
    sx: number, sy: number,
    x: number, y: number,
    readonly rotation: number,
    modifiers: KeyModifiers,
  ) {
    super(sx, sy, x, y, modifiers)
  }

  protected override get event_values(): Attrs {
    const {rotation} = this
    return {...super.event_values, rotation}
  }
}

@event("wheel")
export class MouseWheel extends PointEvent {

  constructor(
    sx: number, sy: number,
    x: number, y: number,
    readonly delta: number,
    modifiers: KeyModifiers,
  ) {
    super(sx, sy, x, y, modifiers)
  }

  protected override get event_values(): Attrs {
    const {delta} = this
    return {...super.event_values, delta}
  }
}

@event("mousemove")
export class MouseMove extends PointEvent {}

@event("mouseenter")
export class MouseEnter extends PointEvent {}

@event("mouseleave")
export class MouseLeave extends PointEvent {}

@event("tap")
export class Tap extends PointEvent {}

@event("doubletap")
export class DoubleTap extends PointEvent {}

@event("press")
export class Press extends PointEvent {}

@event("pressup")
export class PressUp extends PointEvent {}

@event("panstart")
export class PanStart extends PointEvent {}

@event("panend")
export class PanEnd extends PointEvent {}

@event("pinchstart")
export class PinchStart extends PointEvent {}

@event("pinchend")
export class PinchEnd extends PointEvent {}

@event("rotatestart")
export class RotateStart extends PointEvent {}

@event("rotateend")
export class RotateEnd extends PointEvent {}
