# -----------------------------------------------------------------------------
# Copyright (c) Anaconda, Inc., and Bokeh Contributors.
# All rights reserved.
#
# The full license is in the file LICENSE.txt, distributed with this software.
# -----------------------------------------------------------------------------
"""

"""
from __future__ import annotations

# Standard library imports
import os
from functools import wraps
from typing import Callable

# External imports
import boto3

# Bokeh imports
from . import BOKEHJS_BUCKETS
from .action import FAILED, PASSED, ActionReturn
from .config import Config
from .pipeline import StepType
from .system import System

__all__ = (
    "verify_anaconda_credentials",
    "verify_aws_credentials",
    "verify_google_credentials",
    "verify_npm_credentials",
    "verify_pypi_credentials",
)


VerifyFunctionType = Callable[..., None]  # Unfortunately best current solution see https://github.com/python/typing/issues/696


def collect_credential(**kw: str) -> Callable[[VerifyFunctionType], StepType]:
    def decorator(func: VerifyFunctionType) -> StepType:
        service = func.__name__.split("_")[1].upper()

        @wraps(func)
        def wrapper(config: Config, system: System) -> ActionReturn:
            secrets = dict()
            for argname, envname in kw.items():
                if envname not in os.environ:
                    return FAILED(f"Credential {envname} is not set")
                secrets[argname] = os.environ[envname]

                # this must be added immediately before anything else so a scrubber is registered
                config.add_secret(envname, secrets[argname])

            try:
                func(config, system, **secrets)
                return PASSED(f"Verified {service} credentials")
            except Exception as e:
                return FAILED(f"Could NOT verify {service} credentials", details=e.args)

        return wrapper

    return decorator


@collect_credential(token="ANACONDA_TOKEN")
def verify_anaconda_credentials(config: Config, system: System, *, token: str) -> None:
    out = system.run(f"anaconda -t {token} whoami")
    if "Username: bokeh" not in out:
        raise RuntimeError(*out.strip().split("\n"))


@collect_credential(token="PYPI_TOKEN")
def verify_pypi_credentials(config: Config, system: System, *, token: str) -> None:
    # TODO is there a way to actually test that the creds work?
    pass


@collect_credential(token="GOOGLE_API_KEY")
def verify_google_credentials(config: Config, system: System, *, token: str) -> None:
    # TODO is there a way to actually test that the creds work?
    pass


def verify_npm_credentials(config: Config, system: System) -> ActionReturn:
    service = "npm"
    return PASSED(f"Verification of {service} credentials is not currently supported")

    # NOTE no token, because we use trusted providers to deploy npm packages
    # TODO restore this when OICD support is added to npm whoami
    # out = system.run("npm whoami")
    # if out.strip() == "bokeh-service":
    #     return PASSED(f"Verified {service} credentials")
    # else:
    #     return FAILED(f"Could NOT verify {service} credentials")


@collect_credential(access_key_id="AWS_ACCESS_KEY_ID", secret_access_key="AWS_SECRET_ACCESS_KEY")
def verify_aws_credentials(config: Config, system: System, *, access_key_id: str, secret_access_key: str) -> None:
    for bucket, region_name in BOKEHJS_BUCKETS:
        s3 = boto3.client(
            "s3",
            region_name=region_name,
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key,
        )
        s3.head_bucket(Bucket=bucket)
