// Copyright 2025 The Casibase Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package controllers

import (
	"encoding/json"

	"github.com/beego/beego/utils/pagination"
	"github.com/casibase/casibase/object"
	"github.com/casibase/casibase/util"
)

// GetDoctors
// @Title GetDoctors
// @Tag Doctor API
// @Description get all doctors
// @Param   pageSize     query    string  true        "The size of each page"
// @Param   p     query    string  true        "The number of the page"
// @Success 200 {object} object.Doctor The Response object
// @router /get-doctors [get]
func (c *ApiController) GetDoctors() {
	owner := c.Input().Get("owner")
	limit := c.Input().Get("pageSize")
	page := c.Input().Get("p")
	field := c.Input().Get("field")
	value := c.Input().Get("value")
	sortField := c.Input().Get("sortField")
	sortOrder := c.Input().Get("sortOrder")

	if limit == "" || page == "" {
		doctors, err := object.GetMaskedDoctors(object.GetDoctors(owner))
		if err != nil {
			c.ResponseError(err.Error())
			return
		}

		c.ResponseOk(doctors)
	} else {
		limit := util.ParseInt(limit)
		count, err := object.GetDoctorCount(owner, field, value)
		if err != nil {
			c.ResponseError(err.Error())
			return
		}

		paginator := pagination.SetPaginator(c.Ctx, limit, count)
		doctors, err := object.GetMaskedDoctors(object.GetPaginationDoctors(owner, paginator.Offset(), limit, field, value, sortField, sortOrder))
		if err != nil {
			c.ResponseError(err.Error())
			return
		}

		c.ResponseOk(doctors, paginator.Nums())
	}
}

// GetDoctor
// @Title GetDoctor
// @Tag Doctor API
// @Description get doctor
// @Param   id     query    string  true        "The id ( owner/name ) of the doctor"
// @Success 200 {object} object.Doctor The Response object
// @router /get-doctor [get]
func (c *ApiController) GetDoctor() {
	id := c.Input().Get("id")

	doctor, err := object.GetMaskedDoctor(object.GetDoctor(id))
	if err != nil {
		c.ResponseError(err.Error())
		return
	}

	c.ResponseOk(doctor)
}

// UpdateDoctor
// @Title UpdateDoctor
// @Tag Doctor API
// @Description update doctor
// @Param   id     query    string  true        "The id ( owner/name ) of the doctor"
// @Param   body    body   object.Doctor  true        "The details of the doctor"
// @Success 200 {object} controllers.Response The Response object
// @router /update-doctor [post]
func (c *ApiController) UpdateDoctor() {
	id := c.Input().Get("id")

	var doctor object.Doctor
	err := json.Unmarshal(c.Ctx.Input.RequestBody, &doctor)
	if err != nil {
		c.ResponseError(err.Error())
		return
	}

	c.Data["json"] = wrapActionResponse(object.UpdateDoctor(id, &doctor))
	c.ServeJSON()
}

// AddDoctor
// @Title AddDoctor
// @Tag Doctor API
// @Description add a doctor
// @Param   body    body   object.Doctor  true        "The details of the doctor"
// @Success 200 {object} controllers.Response The Response object
// @router /add-doctor [post]
func (c *ApiController) AddDoctor() {
	var doctor object.Doctor
	err := json.Unmarshal(c.Ctx.Input.RequestBody, &doctor)
	if err != nil {
		c.ResponseError(err.Error())
		return
	}

	c.Data["json"] = wrapActionResponse(object.AddDoctor(&doctor))
	c.ServeJSON()
}

// DeleteDoctor
// @Title DeleteDoctor
// @Tag Doctor API
// @Description delete a doctor
// @Param   body    body   object.Doctor  true        "The details of the doctor"
// @Success 200 {object} controllers.Response The Response object
// @router /delete-doctor [post]
func (c *ApiController) DeleteDoctor() {
	var doctor object.Doctor
	err := json.Unmarshal(c.Ctx.Input.RequestBody, &doctor)
	if err != nil {
		c.ResponseError(err.Error())
		return
	}

	c.Data["json"] = wrapActionResponse(object.DeleteDoctor(&doctor))
	c.ServeJSON()
}
