// Copyright 2025 The Casibase Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package object

import (
	"fmt"

	"github.com/casibase/casibase/util"
	"xorm.io/core"
)

type Asset struct {
	Owner       string `xorm:"varchar(100) notnull pk" json:"owner"`
	Name        string `xorm:"varchar(100) notnull pk" json:"name"`
	CreatedTime string `xorm:"varchar(100)" json:"createdTime"`
	UpdatedTime string `xorm:"varchar(100)" json:"updatedTime"`
	DisplayName string `xorm:"varchar(200)" json:"displayName"`

	Provider   string `xorm:"varchar(100)" json:"provider"`
	Id         string `xorm:"varchar(200)" json:"id"`
	Type       string `xorm:"varchar(100)" json:"type"`
	Region     string `xorm:"varchar(100)" json:"region"`
	Zone       string `xorm:"varchar(100)" json:"zone"`
	State      string `xorm:"varchar(100)" json:"state"`
	Tag        string `xorm:"varchar(500)" json:"tag"`
	Username   string `xorm:"varchar(100)" json:"username"`
	Password   string `xorm:"varchar(200)" json:"password"`
	Properties string `xorm:"mediumtext" json:"properties"`
}

func GetMaskedAsset(asset *Asset, isMaskEnabled bool) *Asset {
	if !isMaskEnabled {
		return asset
	}

	if asset == nil {
		return nil
	}

	// Create a copy to avoid modifying the original
	maskedAsset := *asset
	if maskedAsset.Password != "" {
		maskedAsset.Password = "***"
	}

	return &maskedAsset
}

func GetMaskedAssets(assets []*Asset, isMaskEnabled bool) []*Asset {
	if !isMaskEnabled {
		return assets
	}

	for i := range assets {
		assets[i] = GetMaskedAsset(assets[i], isMaskEnabled)
	}
	return assets
}

func GetAssetCount(owner, field, value string) (int64, error) {
	session := GetDbSession(owner, -1, -1, field, value, "", "")
	return session.Count(&Asset{})
}

func GetAssets(owner string) ([]*Asset, error) {
	assets := []*Asset{}
	err := adapter.engine.Desc("created_time").Find(&assets, &Asset{Owner: owner})
	if err != nil {
		return assets, err
	}
	return assets, nil
}

func GetPaginationAssets(owner string, offset, limit int, field, value, sortField, sortOrder string) ([]*Asset, error) {
	assets := []*Asset{}
	session := GetDbSession(owner, offset, limit, field, value, sortField, sortOrder)
	err := session.Find(&assets)
	if err != nil {
		return assets, err
	}

	return assets, nil
}

func getAsset(owner string, name string) (*Asset, error) {
	if owner == "" || name == "" {
		return nil, nil
	}

	asset := Asset{Owner: owner, Name: name}
	existed, err := adapter.engine.Get(&asset)
	if err != nil {
		return &asset, err
	}

	if existed {
		return &asset, nil
	} else {
		return nil, nil
	}
}

func GetAsset(id string) (*Asset, error) {
	owner, name, err := util.GetOwnerAndNameFromIdWithError(id)
	if err != nil {
		return nil, err
	}
	return getAsset(owner, name)
}

func UpdateAsset(id string, asset *Asset) (bool, error) {
	owner, name, err := util.GetOwnerAndNameFromIdWithError(id)
	if err != nil {
		return false, err
	}
	assetDb, err := getAsset(owner, name)
	if err != nil {
		return false, err
	}
	if assetDb == nil {
		return false, nil
	}

	asset.processAssetParams(assetDb)

	affected, err := adapter.engine.ID(core.PK{owner, name}).AllCols().Update(asset)
	if err != nil {
		return false, err
	}

	return affected != 0, nil
}

func AddAsset(asset *Asset) (bool, error) {
	affected, err := adapter.engine.Insert(asset)
	if err != nil {
		return false, err
	}

	return affected != 0, nil
}

func addAssets(assets []*Asset) (bool, error) {
	affected, err := adapter.engine.Insert(assets)
	if err != nil {
		return false, err
	}

	return affected != 0, nil
}

func DeleteAsset(asset *Asset) (bool, error) {
	affected, err := adapter.engine.ID(core.PK{asset.Owner, asset.Name}).Delete(&Asset{})
	if err != nil {
		return false, err
	}

	return affected != 0, nil
}

func deleteAssets(owner string) (bool, error) {
	affected, err := adapter.engine.Delete(&Asset{Owner: owner})
	if err != nil {
		return false, err
	}

	return affected != 0, nil
}

func (a *Asset) processAssetParams(assetDb *Asset) {
	if a.Password == "***" {
		a.Password = assetDb.Password
	}
}

func (asset *Asset) GetId() string {
	return fmt.Sprintf("%s/%s", asset.Owner, asset.Name)
}

func (asset *Asset) GetScanTarget() (string, error) {
	if asset.Type == "Virtual Machine" {
		publicIp, err := util.GetFieldFromJsonString(asset.Properties, "publicIp")
		if err != nil {
			return "", fmt.Errorf("failed to parse publicIp from properties: %v", err)
		}
		if publicIp != "" {
			return publicIp, nil
		}
		// Fallback to asset.Id if publicIp is not available
		return asset.Id, nil
	}
	// For non-Virtual Machine types, use asset.Id
	return asset.Id, nil
}
