---
title: Connectors
id: connectors-api
description: Various connectors to integrate with external services.
slug: "/connectors-api"
---

<a id="openapi_service"></a>

# Module openapi\_service

<a id="openapi_service.OpenAPIServiceConnector"></a>

## OpenAPIServiceConnector

A component which connects the Haystack framework to OpenAPI services.

The `OpenAPIServiceConnector` component connects the Haystack framework to OpenAPI services, enabling it to call
operations as defined in the OpenAPI specification of the service.

It integrates with `ChatMessage` dataclass, where the payload in messages is used to determine the method to be
called and the parameters to be passed. The message payload should be an OpenAI JSON formatted function calling
string consisting of the method name and the parameters to be passed to the method. The method name and parameters
are then used to invoke the method on the OpenAPI service. The response from the service is returned as a
`ChatMessage`.

Before using this component, users usually resolve service endpoint parameters with a help of
`OpenAPIServiceToFunctions` component.

The example below demonstrates how to use the `OpenAPIServiceConnector` to invoke a method on a https://serper.dev/
service specified via OpenAPI specification.

Note, however, that `OpenAPIServiceConnector` is usually not meant to be used directly, but rather as part of a
pipeline that includes the `OpenAPIServiceToFunctions` component and an `OpenAIChatGenerator` component using LLM
with the function calling capabilities. In the example below we use the function calling payload directly, but in a
real-world scenario, the function calling payload would usually be generated by the `OpenAIChatGenerator` component.

Usage example:

```python
import json
import requests

from haystack.components.connectors import OpenAPIServiceConnector
from haystack.dataclasses import ChatMessage


fc_payload = [{'function': {'arguments': '{"q": "Why was Sam Altman ousted from OpenAI?"}', 'name': 'search'},
               'id': 'call_PmEBYvZ7mGrQP5PUASA5m9wO', 'type': 'function'}]

serper_token = <your_serper_dev_token>
serperdev_openapi_spec = json.loads(requests.get("https://bit.ly/serper_dev_spec").text)
service_connector = OpenAPIServiceConnector()
result = service_connector.run(messages=[ChatMessage.from_assistant(json.dumps(fc_payload))],
                               service_openapi_spec=serperdev_openapi_spec, service_credentials=serper_token)
print(result)

>> {'service_response': [ChatMessage(_role=<ChatRole.ASSISTANT: 'assistant'>, _content=[TextContent(text=
>> '{"searchParameters": {"q": "Why was Sam Altman ousted from OpenAI?",
>> "type": "search", "engine": "google"}, "answerBox": {"snippet": "Concerns over AI safety and OpenAI's role
>> in protecting were at the center of Altman's brief ouster from the company."...
```

<a id="openapi_service.OpenAPIServiceConnector.__init__"></a>

#### OpenAPIServiceConnector.\_\_init\_\_

```python
def __init__(ssl_verify: Optional[Union[bool, str]] = None)
```

Initializes the OpenAPIServiceConnector instance

**Arguments**:

- `ssl_verify`: Decide if to use SSL verification to the requests or not,
in case a string is passed, will be used as the CA.

<a id="openapi_service.OpenAPIServiceConnector.run"></a>

#### OpenAPIServiceConnector.run

```python
@component.output_types(service_response=dict[str, Any])
def run(
    messages: list[ChatMessage],
    service_openapi_spec: dict[str, Any],
    service_credentials: Optional[Union[dict, str]] = None
) -> dict[str, list[ChatMessage]]
```

Processes a list of chat messages to invoke a method on an OpenAPI service.

It parses the last message in the list, expecting it to contain tool calls.

**Arguments**:

- `messages`: A list of `ChatMessage` objects containing the messages to be processed. The last message
should contain the tool calls.
- `service_openapi_spec`: The OpenAPI JSON specification object of the service to be invoked. All the refs
should already be resolved.
- `service_credentials`: The credentials to be used for authentication with the service.
Currently, only the http and apiKey OpenAPI security schemes are supported.

**Raises**:

- `ValueError`: If the last message is not from the assistant or if it does not contain tool calls.

**Returns**:

A dictionary with the following keys:
- `service_response`:  a list of `ChatMessage` objects, each containing the response from the service. The
response is in JSON format, and the `content` attribute of the `ChatMessage` contains
the JSON string.

<a id="openapi_service.OpenAPIServiceConnector.to_dict"></a>

#### OpenAPIServiceConnector.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="openapi_service.OpenAPIServiceConnector.from_dict"></a>

#### OpenAPIServiceConnector.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "OpenAPIServiceConnector"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: The dictionary to deserialize from.

**Returns**:

The deserialized component.

<a id="openapi"></a>

# Module openapi

<a id="openapi.OpenAPIConnector"></a>

## OpenAPIConnector

OpenAPIConnector enables direct invocation of REST endpoints defined in an OpenAPI specification.

The OpenAPIConnector serves as a bridge between Haystack pipelines and any REST API that follows
the OpenAPI(formerly Swagger) specification. It dynamically interprets the API specification and
provides an interface for executing API operations. It is usually invoked by passing input
arguments to it from a Haystack pipeline run method or by other components in a pipeline that
pass input arguments to this component.

**Example**:

```python
from haystack.utils import Secret
from haystack.components.connectors.openapi import OpenAPIConnector

connector = OpenAPIConnector(
    openapi_spec="https://bit.ly/serperdev_openapi",
    credentials=Secret.from_env_var("SERPERDEV_API_KEY"),
    service_kwargs={"config_factory": my_custom_config_factory}
)
response = connector.run(
    operation_id="search",
    arguments={"q": "Who was Nikola Tesla?"}
)
```

**Notes**:

  - The `parameters` argument is required for this component.
  - The `service_kwargs` argument is optional, it can be used to pass additional options to the OpenAPIClient.

<a id="openapi.OpenAPIConnector.__init__"></a>

#### OpenAPIConnector.\_\_init\_\_

```python
def __init__(openapi_spec: str,
             credentials: Optional[Secret] = None,
             service_kwargs: Optional[dict[str, Any]] = None)
```

Initialize the OpenAPIConnector with a specification and optional credentials.

**Arguments**:

- `openapi_spec`: URL, file path, or raw string of the OpenAPI specification
- `credentials`: Optional API key or credentials for the service wrapped in a Secret
- `service_kwargs`: Additional keyword arguments passed to OpenAPIClient.from_spec()
For example, you can pass a custom config_factory or other configuration options.

<a id="openapi.OpenAPIConnector.to_dict"></a>

#### OpenAPIConnector.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serialize this component to a dictionary.

<a id="openapi.OpenAPIConnector.from_dict"></a>

#### OpenAPIConnector.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "OpenAPIConnector"
```

Deserialize this component from a dictionary.

<a id="openapi.OpenAPIConnector.run"></a>

#### OpenAPIConnector.run

```python
@component.output_types(response=dict[str, Any])
def run(operation_id: str,
        arguments: Optional[dict[str, Any]] = None) -> dict[str, Any]
```

Invokes a REST endpoint specified in the OpenAPI specification.

**Arguments**:

- `operation_id`: The operationId from the OpenAPI spec to invoke
- `arguments`: Optional parameters for the endpoint (query, path, or body parameters)

**Returns**:

Dictionary containing the service response
