---
title: "Amazon Bedrock"
id: integrations-amazon-bedrock
description: "Amazon Bedrock integration for Haystack"
slug: "/integrations-amazon-bedrock"
---

<a id="haystack_integrations.common.amazon_bedrock.errors"></a>

## Module haystack\_integrations.common.amazon\_bedrock.errors

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockError"></a>

### AmazonBedrockError

Any error generated by the Amazon Bedrock integration.

This error wraps its source transparently in such a way that its attributes
can be accessed directly: for example, if the original error has a `message` attribute,
`AmazonBedrockError.message` will exist and have the expected content.

<a id="haystack_integrations.common.amazon_bedrock.errors.AWSConfigurationError"></a>

### AWSConfigurationError

Exception raised when AWS is not configured correctly

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockConfigurationError"></a>

### AmazonBedrockConfigurationError

Exception raised when AmazonBedrock node is not configured correctly

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockInferenceError"></a>

### AmazonBedrockInferenceError

Exception for issues that occur in the Bedrock inference node

<a id="haystack_integrations.common.amazon_bedrock.errors"></a>

## Module haystack\_integrations.common.amazon\_bedrock.errors

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockError"></a>

### AmazonBedrockError

Any error generated by the Amazon Bedrock integration.

This error wraps its source transparently in such a way that its attributes
can be accessed directly: for example, if the original error has a `message` attribute,
`AmazonBedrockError.message` will exist and have the expected content.

<a id="haystack_integrations.common.amazon_bedrock.errors.AWSConfigurationError"></a>

### AWSConfigurationError

Exception raised when AWS is not configured correctly

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockConfigurationError"></a>

### AmazonBedrockConfigurationError

Exception raised when AmazonBedrock node is not configured correctly

<a id="haystack_integrations.common.amazon_bedrock.errors.AmazonBedrockInferenceError"></a>

### AmazonBedrockInferenceError

Exception for issues that occur in the Bedrock inference node

<a id="haystack_integrations.common.s3.errors"></a>

## Module haystack\_integrations.common.s3.errors

<a id="haystack_integrations.common.s3.errors.S3Error"></a>

### S3Error

Exception for issues that occur in the S3 based components

<a id="haystack_integrations.common.s3.errors.S3ConfigurationError"></a>

### S3ConfigurationError

Exception raised when AmazonS3 node is not configured correctly

<a id="haystack_integrations.common.s3.errors.S3StorageError"></a>

### S3StorageError

This exception is raised when an error occurs while interacting with a S3Storage object.

<a id="haystack_integrations.common.s3.utils"></a>

## Module haystack\_integrations.common.s3.utils

<a id="haystack_integrations.common.s3.utils.S3Storage"></a>

### S3Storage

This class provides a storage class for downloading files from an AWS S3 bucket.

<a id="haystack_integrations.common.s3.utils.S3Storage.__init__"></a>

#### S3Storage.\_\_init\_\_

```python
def __init__(s3_bucket: str,
             session: Session,
             s3_prefix: str | None = None,
             endpoint_url: str | None = None,
             config: Config | None = None) -> None
```

Initializes the S3Storage object with the provided parameters.

**Arguments**:

- `s3_bucket`: The name of the S3 bucket to download files from.
- `session`: The session to use for the S3 client.
- `s3_prefix`: The optional prefix of the files in the S3 bucket.
Can be used to specify folder or naming structure.
For example, if the file is in the folder "folder/subfolder/file.txt",
the s3_prefix should be "folder/subfolder/". If the file is in the root of the S3 bucket,
the s3_prefix should be None.
- `endpoint_url`: The endpoint URL of the S3 bucket to download files from.
- `config`: The configuration to use for the S3 client.

<a id="haystack_integrations.common.s3.utils.S3Storage.download"></a>

#### S3Storage.download

```python
def download(key: str, local_file_path: Path) -> None
```

Download a file from S3.

**Arguments**:

- `key`: The key of the file to download.
- `local_file_path`: The folder path to download the file to.
It will be created if it does not exist. The file will be downloaded to
the folder with the same name as the key.

**Raises**:

- `S3ConfigurationError`: If the S3 session client cannot be created.
- `S3StorageError`: If the file does not exist in the S3 bucket
or the file cannot be downloaded.

<a id="haystack_integrations.common.s3.utils.S3Storage.from_env"></a>

#### S3Storage.from\_env

```python
@classmethod
def from_env(cls, *, session: Session, config: Config) -> "S3Storage"
```

Create a S3Storage object from environment variables.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader"></a>

## Module haystack\_integrations.components.downloaders.s3.s3\_downloader

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader"></a>

### S3Downloader

A component for downloading files from AWS S3 Buckets to local filesystem.
Supports filtering by file extensions.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader.__init__"></a>

#### S3Downloader.\_\_init\_\_

```python
def __init__(
    *,
    aws_access_key_id: Secret | None = Secret.from_env_var("AWS_ACCESS_KEY_ID",
                                                           strict=False),
    aws_secret_access_key: Secret | None = Secret.from_env_var(  # noqa: B008
        "AWS_SECRET_ACCESS_KEY", strict=False),
    aws_session_token: Secret | None = Secret.from_env_var("AWS_SESSION_TOKEN",
                                                           strict=False),
    aws_region_name: Secret | None = Secret.from_env_var("AWS_DEFAULT_REGION",
                                                         strict=False),
    aws_profile_name: Secret | None = Secret.from_env_var("AWS_PROFILE",
                                                          strict=False),
    boto3_config: dict[str, Any] | None = None,
    file_root_path: str | None = None,
    file_extensions: list[str] | None = None,
    file_name_meta_key: str = "file_name",
    max_workers: int = 32,
    max_cache_size: int = 100,
    s3_key_generation_function: Callable[[Document], str] | None = None
) -> None
```

Initializes the `S3Downloader` with the provided parameters.

Note that the AWS credentials are not required if the AWS environment is configured correctly. These are loaded
automatically from the environment or the AWS configuration file and do not need to be provided explicitly via
the constructor. If the AWS environment is not configured users need to provide the AWS credentials via the
constructor. Three required parameters are `aws_access_key_id`, `aws_secret_access_key`,
and `aws_region_name`.

**Arguments**:

- `aws_access_key_id`: AWS access key ID.
- `aws_secret_access_key`: AWS secret access key.
- `aws_session_token`: AWS session token.
- `aws_region_name`: AWS region name.
- `aws_profile_name`: AWS profile name.
- `boto3_config`: The configuration for the boto3 client.
- `file_root_path`: The path where the file will be downloaded.
Can be set through this parameter or the `FILE_ROOT_PATH` environment variable.
If none of them is set, a `ValueError` is raised.
- `file_extensions`: The file extensions that are permitted to be downloaded.
By default, all file extensions are allowed.
- `max_workers`: The maximum number of workers to use for concurrent downloads.
- `max_cache_size`: The maximum number of files to cache.
- `file_name_meta_key`: The name of the meta key that contains the file name to download. The file name
will also be used to create local file path for download.
By default, the `Document.meta["file_name"]` is used. If you want to use a
different key in `Document.meta`, you can set it here.
- `s3_key_generation_function`: An optional function that generates the S3 key for the file to download.
If not provided, the default behavior is to use `Document.meta[file_name_meta_key]`.
The function must accept a `Document` object and return a string.
If the environment variable `S3_DOWNLOADER_PREFIX` is set, its value will be automatically
prefixed to the generated S3 key.

**Raises**:

- `ValueError`: If the `file_root_path` is not set through
the constructor or the `FILE_ROOT_PATH` environment variable.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader.warm_up"></a>

#### S3Downloader.warm\_up

```python
def warm_up() -> None
```

Warm up the component by initializing the settings and storage.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader.run"></a>

#### S3Downloader.run

```python
@component.output_types(documents=list[Document])
def run(documents: list[Document]) -> dict[str, list[Document]]
```

Download files from AWS S3 Buckets to local filesystem.

Return enriched `Document`s with the path of the downloaded file.

**Arguments**:

- `documents`: Document containing the name of the file to download in the meta field.

**Raises**:

- `S3Error`: If a download attempt fails or the file does not exist in the S3 bucket.
- `ValueError`: If the path where files will be downloaded is not set.

**Returns**:

A dictionary with:
- `documents`: The downloaded `Document`s; each has `meta['file_path']`.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader.to_dict"></a>

#### S3Downloader.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serialize the component to a dictionary.

<a id="haystack_integrations.components.downloaders.s3.s3_downloader.S3Downloader.from_dict"></a>

#### S3Downloader.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "S3Downloader"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary to deserialize from.

**Returns**:

Deserialized component.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder"></a>

## Module haystack\_integrations.components.embedders.amazon\_bedrock.document\_embedder

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder.AmazonBedrockDocumentEmbedder"></a>

### AmazonBedrockDocumentEmbedder

A component for computing Document embeddings using Amazon Bedrock.
The embedding of each Document is stored in the `embedding` field of the Document.

Usage example:
```python
import os
from haystack.dataclasses import Document
from haystack_integrations.components.embedders.amazon_bedrock import AmazonBedrockDocumentEmbedder

os.environ["AWS_ACCESS_KEY_ID"] = "..."
os.environ["AWS_SECRET_ACCESS_KEY_ID"] = "..."
os.environ["AWS_DEFAULT_REGION"] = "..."

embedder = AmazonBedrockDocumentEmbedder(
    model="cohere.embed-english-v3",
    input_type="search_document",
)

doc = Document(content="I love Paris in the winter.", meta={"name": "doc1"})

result = embedder.run([doc])
print(result['documents'][0].embedding)

# [0.002, 0.032, 0.504, ...]
```

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder.AmazonBedrockDocumentEmbedder.__init__"></a>

#### AmazonBedrockDocumentEmbedder.\_\_init\_\_

```python
def __init__(
        model: str,
        aws_access_key_id: Secret | None = Secret.from_env_var(
            "AWS_ACCESS_KEY_ID", strict=False),
        aws_secret_access_key: Secret
    | None = Secret.from_env_var(  # noqa: B008
        "AWS_SECRET_ACCESS_KEY", strict=False),
        aws_session_token: Secret | None = Secret.from_env_var(
            "AWS_SESSION_TOKEN", strict=False),
        aws_region_name: Secret | None = Secret.from_env_var(
            "AWS_DEFAULT_REGION", strict=False),
        aws_profile_name: Secret | None = Secret.from_env_var("AWS_PROFILE",
                                                              strict=False),
        batch_size: int = 32,
        progress_bar: bool = True,
        meta_fields_to_embed: list[str] | None = None,
        embedding_separator: str = "\n",
        boto3_config: dict[str, Any] | None = None,
        **kwargs: Any) -> None
```

Initializes the AmazonBedrockDocumentEmbedder with the provided parameters. The parameters are passed to the

Amazon Bedrock client.

Note that the AWS credentials are not required if the AWS environment is configured correctly. These are loaded
automatically from the environment or the AWS configuration file and do not need to be provided explicitly via
the constructor. If the AWS environment is not configured users need to provide the AWS credentials via the
constructor. Aside from model, three required parameters are `aws_access_key_id`, `aws_secret_access_key`,
 and `aws_region_name`.

**Arguments**:

- `model`: The embedding model to use.
Amazon Titan and Cohere embedding models are supported, for example:
"amazon.titan-embed-text-v1", "amazon.titan-embed-text-v2:0", "amazon.titan-embed-image-v1",
"cohere.embed-english-v3", "cohere.embed-multilingual-v3", "cohere.embed-v4:0".
To find all supported models, refer to the Amazon Bedrock
[documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/models-supported.html) and
filter for "embedding", then select models from the Amazon Titan and Cohere series.
- `aws_access_key_id`: AWS access key ID.
- `aws_secret_access_key`: AWS secret access key.
- `aws_session_token`: AWS session token.
- `aws_region_name`: AWS region name.
- `aws_profile_name`: AWS profile name.
- `batch_size`: Number of Documents to encode at once.
Only Cohere models support batch inference. This parameter is ignored for Amazon Titan models.
- `progress_bar`: Whether to show a progress bar or not. Can be helpful to disable in production deployments
to keep the logs clean.
- `meta_fields_to_embed`: List of meta fields that should be embedded along with the Document text.
- `embedding_separator`: Separator used to concatenate the meta fields to the Document text.
- `boto3_config`: The configuration for the boto3 client.
- `kwargs`: Additional parameters to pass for model inference. For example, `input_type` and `truncate` for
Cohere models.

**Raises**:

- `ValueError`: If the model is not supported.
- `AmazonBedrockConfigurationError`: If the AWS environment is not configured correctly.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder.AmazonBedrockDocumentEmbedder.run"></a>

#### AmazonBedrockDocumentEmbedder.run

```python
@component.output_types(documents=list[Document])
def run(documents: list[Document]) -> dict[str, list[Document]]
```

Embed the provided `Document`s using the specified model.

**Arguments**:

- `documents`: The `Document`s to embed.

**Raises**:

- `AmazonBedrockInferenceError`: If the inference fails.

**Returns**:

A dictionary with the following keys:
- `documents`: The `Document`s with the `embedding` field populated.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder.AmazonBedrockDocumentEmbedder.to_dict"></a>

#### AmazonBedrockDocumentEmbedder.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_embedder.AmazonBedrockDocumentEmbedder.from_dict"></a>

#### AmazonBedrockDocumentEmbedder.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "AmazonBedrockDocumentEmbedder"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary to deserialize from.

**Returns**:

Deserialized component.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder"></a>

## Module haystack\_integrations.components.embedders.amazon\_bedrock.document\_image\_embedder

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder.AmazonBedrockDocumentImageEmbedder"></a>

### AmazonBedrockDocumentImageEmbedder

A component for computing Document embeddings based on images using Amazon Bedrock models.

The embedding of each Document is stored in the `embedding` field of the Document.

### Usage example
```python
from haystack import Document
rom haystack_integrations.components.embedders.amazon_bedrock import AmazonBedrockDocumentImageEmbedder

os.environ["AWS_ACCESS_KEY_ID"] = "..."
os.environ["AWS_SECRET_ACCESS_KEY_ID"] = "..."
os.environ["AWS_DEFAULT_REGION"] = "..."

embedder = AmazonBedrockDocumentImageEmbedder(model="amazon.titan-embed-image-v1")

documents = [
    Document(content="A photo of a cat", meta={"file_path": "cat.jpg"}),
    Document(content="A photo of a dog", meta={"file_path": "dog.jpg"}),
]

result = embedder.run(documents=documents)
documents_with_embeddings = result["documents"]
print(documents_with_embeddings)

# [Document(id=...,
#           content='A photo of a cat',
#           meta={'file_path': 'cat.jpg',
#                 'embedding_source': {'type': 'image', 'file_path_meta_field': 'file_path'}},
#           embedding=vector of size 512),
#  ...]
```

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder.AmazonBedrockDocumentImageEmbedder.__init__"></a>

#### AmazonBedrockDocumentImageEmbedder.\_\_init\_\_

```python
def __init__(
        *,
        model: str,
        aws_access_key_id: Secret | None = Secret.from_env_var(
            "AWS_ACCESS_KEY_ID", strict=False),
        aws_secret_access_key: Secret
    | None = Secret.from_env_var(  # noqa: B008
        "AWS_SECRET_ACCESS_KEY", strict=False),
        aws_session_token: Secret | None = Secret.from_env_var(
            "AWS_SESSION_TOKEN", strict=False),
        aws_region_name: Secret | None = Secret.from_env_var(
            "AWS_DEFAULT_REGION", strict=False),
        aws_profile_name: Secret | None = Secret.from_env_var("AWS_PROFILE",
                                                              strict=False),
        file_path_meta_field: str = "file_path",
        root_path: str | None = None,
        image_size: tuple[int, int] | None = None,
        progress_bar: bool = True,
        boto3_config: dict[str, Any] | None = None,
        **kwargs: Any) -> None
```

Creates a AmazonBedrockDocumentImageEmbedder component.

**Arguments**:

- `model`: The embedding model to use.
Amazon Titan and Cohere multimodal embedding models are supported, for example:
"amazon.titan-embed-image-v1", "cohere.embed-english-v3", "cohere.embed-multilingual-v3",
"cohere.embed-v4:0".
To find all supported models, refer to the Amazon Bedrock
[documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/models-supported.html) and
filter for "embedding", then select multimodal models from the Amazon Titan and Cohere series.
- `aws_access_key_id`: AWS access key ID.
- `aws_secret_access_key`: AWS secret access key.
- `aws_session_token`: AWS session token.
- `aws_region_name`: AWS region name.
- `aws_profile_name`: AWS profile name.
- `file_path_meta_field`: The metadata field in the Document that contains the file path to the image or PDF.
- `root_path`: The root directory path where document files are located. If provided, file paths in
document metadata will be resolved relative to this path. If None, file paths are treated as absolute paths.
- `image_size`: If provided, resizes the image to fit within the specified dimensions (width, height) while
maintaining aspect ratio. This reduces file size, memory usage, and processing time, which is beneficial
when working with models that have resolution constraints or when transmitting images to remote services.
- `progress_bar`: If `True`, shows a progress bar when embedding documents.
- `boto3_config`: The configuration for the boto3 client.
- `kwargs`: Additional parameters to pass for model inference.
For example, `embeddingConfig` for Amazon Titan models and
`embedding_types` for Cohere models.

**Raises**:

- `ValueError`: If the model is not supported.
- `AmazonBedrockConfigurationError`: If the AWS environment is not configured correctly.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder.AmazonBedrockDocumentImageEmbedder.to_dict"></a>

#### AmazonBedrockDocumentImageEmbedder.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder.AmazonBedrockDocumentImageEmbedder.from_dict"></a>

#### AmazonBedrockDocumentImageEmbedder.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str,
                              Any]) -> "AmazonBedrockDocumentImageEmbedder"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary to deserialize from.

**Returns**:

Deserialized component.

<a id="haystack_integrations.components.embedders.amazon_bedrock.document_image_embedder.AmazonBedrockDocumentImageEmbedder.run"></a>

#### AmazonBedrockDocumentImageEmbedder.run

```python
@component.output_types(documents=list[Document])
def run(documents: list[Document]) -> dict[str, list[Document]]
```

Embed a list of images.

**Arguments**:

- `documents`: Documents to embed.

**Returns**:

A dictionary with the following keys:
- `documents`: Documents with embeddings.

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder"></a>

## Module haystack\_integrations.components.embedders.amazon\_bedrock.text\_embedder

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder.AmazonBedrockTextEmbedder"></a>

### AmazonBedrockTextEmbedder

A component for embedding strings using Amazon Bedrock.

Usage example:
```python
import os
from haystack_integrations.components.embedders.amazon_bedrock import AmazonBedrockTextEmbedder

os.environ["AWS_ACCESS_KEY_ID"] = "..."
os.environ["AWS_SECRET_ACCESS_KEY_ID"] = "..."
os.environ["AWS_DEFAULT_REGION"] = "..."

embedder = AmazonBedrockTextEmbedder(
    model="cohere.embed-english-v3",
    input_type="search_query",
)

print(text_embedder.run("I love Paris in the summer."))

# {'embedding': [0.002, 0.032, 0.504, ...]}
```

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder.AmazonBedrockTextEmbedder.__init__"></a>

#### AmazonBedrockTextEmbedder.\_\_init\_\_

```python
def __init__(
        model: str,
        aws_access_key_id: Secret | None = Secret.from_env_var(
            "AWS_ACCESS_KEY_ID", strict=False),
        aws_secret_access_key: Secret
    | None = Secret.from_env_var(  # noqa: B008
        "AWS_SECRET_ACCESS_KEY", strict=False),
        aws_session_token: Secret | None = Secret.from_env_var(
            "AWS_SESSION_TOKEN", strict=False),
        aws_region_name: Secret | None = Secret.from_env_var(
            "AWS_DEFAULT_REGION", strict=False),
        aws_profile_name: Secret | None = Secret.from_env_var("AWS_PROFILE",
                                                              strict=False),
        boto3_config: dict[str, Any] | None = None,
        **kwargs: Any) -> None
```

Initializes the AmazonBedrockTextEmbedder with the provided parameters. The parameters are passed to the

Amazon Bedrock client.

Note that the AWS credentials are not required if the AWS environment is configured correctly. These are loaded
automatically from the environment or the AWS configuration file and do not need to be provided explicitly via
the constructor. If the AWS environment is not configured users need to provide the AWS credentials via the
constructor. Aside from model, three required parameters are `aws_access_key_id`, `aws_secret_access_key`,
 and `aws_region_name`.

**Arguments**:

- `model`: The embedding model to use.
Amazon Titan and Cohere embedding models are supported, for example:
"amazon.titan-embed-text-v1", "amazon.titan-embed-text-v2:0", "amazon.titan-embed-image-v1",
"cohere.embed-english-v3", "cohere.embed-multilingual-v3", "cohere.embed-v4:0".
To find all supported models, refer to the Amazon Bedrock
[documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/models-supported.html) and
filter for "embedding", then select models from the Amazon Titan and Cohere series.
- `aws_access_key_id`: AWS access key ID.
- `aws_secret_access_key`: AWS secret access key.
- `aws_session_token`: AWS session token.
- `aws_region_name`: AWS region name.
- `aws_profile_name`: AWS profile name.
- `boto3_config`: The configuration for the boto3 client.
- `kwargs`: Additional parameters to pass for model inference. For example, `input_type` and `truncate` for
Cohere models.

**Raises**:

- `ValueError`: If the model is not supported.
- `AmazonBedrockConfigurationError`: If the AWS environment is not configured correctly.

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder.AmazonBedrockTextEmbedder.run"></a>

#### AmazonBedrockTextEmbedder.run

```python
@component.output_types(embedding=list[float])
def run(text: str) -> dict[str, list[float]]
```

Embeds the input text using the Amazon Bedrock model.

**Arguments**:

- `text`: The input text to embed.

**Raises**:

- `TypeError`: If the input text is not a string.
- `AmazonBedrockInferenceError`: If the model inference fails.

**Returns**:

A dictionary with the following keys:
- `embedding`: The embedding of the input text.

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder.AmazonBedrockTextEmbedder.to_dict"></a>

#### AmazonBedrockTextEmbedder.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.embedders.amazon_bedrock.text_embedder.AmazonBedrockTextEmbedder.from_dict"></a>

#### AmazonBedrockTextEmbedder.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "AmazonBedrockTextEmbedder"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary to deserialize from.

**Returns**:

Deserialized component.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters"></a>

## Module haystack\_integrations.components.generators.amazon\_bedrock.adapters

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.BedrockModelAdapter"></a>

### BedrockModelAdapter

Base class for Amazon Bedrock model adapters.

Each subclass of this class is designed to address the unique specificities of a particular LLM it adapts,
focusing on preparing the requests and extracting the responses from the Amazon Bedrock hosted LLMs.

**Arguments**:

- `model_kwargs`: Keyword arguments for the model. You can find the full list of parameters in the
Amazon Bedrock API [documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html).
- `max_length`: Maximum length of generated text. This is mapped to the correct parameter for each model.
It will be overridden by the corresponding parameter in the `model_kwargs` if it is present.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.BedrockModelAdapter.prepare_body"></a>

#### BedrockModelAdapter.prepare\_body

```python
@abstractmethod
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Amazon Bedrock request.

Each subclass should implement this method to prepare the request body for the specific model.

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary containing the body for the request.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.BedrockModelAdapter.get_responses"></a>

#### BedrockModelAdapter.get\_responses

```python
def get_responses(response_body: dict[str, Any]) -> list[str]
```

Extracts the responses from the Amazon Bedrock response.

**Arguments**:

- `response_body`: The response body from the Amazon Bedrock request.

**Returns**:

A list of responses.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.BedrockModelAdapter.get_stream_responses"></a>

#### BedrockModelAdapter.get\_stream\_responses

```python
def get_stream_responses(
        stream: EventStream,
        streaming_callback: SyncStreamingCallbackT) -> list[str]
```

Extracts the responses from the Amazon Bedrock streaming response.

**Arguments**:

- `stream`: The streaming response from the Amazon Bedrock request.
- `streaming_callback`: The handler for the streaming response.

**Returns**:

A list of string responses.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AnthropicClaudeAdapter"></a>

### AnthropicClaudeAdapter

Adapter for the Anthropic Claude models.

**Arguments**:

- `model_kwargs`: Keyword arguments for the model. You can find the full list of parameters in the
Amazon Bedrock API documentation for the Claude model
[here](https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters-claude.html).
Some example parameters are:
- use_messages_api: Whether to use the messages API, default: True
- include_thinking: Whether to include thinking output, default: True
- thinking_tag: XML tag for thinking content, default: "thinking"
- `max_length`: Maximum length of generated text

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AnthropicClaudeAdapter.prepare_body"></a>

#### AnthropicClaudeAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Claude model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
- `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.MistralAdapter"></a>

### MistralAdapter

Adapter for the Mistral models.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.MistralAdapter.prepare_body"></a>

#### MistralAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Mistral model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
- `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.CohereCommandAdapter"></a>

### CohereCommandAdapter

Adapter for the Cohere Command model.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.CohereCommandAdapter.prepare_body"></a>

#### CohereCommandAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Command model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
- `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.CohereCommandRAdapter"></a>

### CohereCommandRAdapter

Adapter for the Cohere Command R models.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.CohereCommandRAdapter.prepare_body"></a>

#### CohereCommandRAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Command model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
- `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AI21LabsJurassic2Adapter"></a>

### AI21LabsJurassic2Adapter

Model adapter for AI21 Labs' Jurassic 2 models.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AI21LabsJurassic2Adapter.prepare_body"></a>

#### AI21LabsJurassic2Adapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Jurassic 2 model.

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
-  `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AmazonTitanAdapter"></a>

### AmazonTitanAdapter

Adapter for Amazon's Titan models.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.AmazonTitanAdapter.prepare_body"></a>

#### AmazonTitanAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Titan model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys
- `inputText`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.MetaLlamaAdapter"></a>

### MetaLlamaAdapter

Adapter for Meta's Llama2 models.

<a id="haystack_integrations.components.generators.amazon_bedrock.adapters.MetaLlamaAdapter.prepare_body"></a>

#### MetaLlamaAdapter.prepare\_body

```python
def prepare_body(prompt: str, **inference_kwargs: Any) -> dict[str, Any]
```

Prepares the body for the Llama2 model

**Arguments**:

- `prompt`: The prompt to be sent to the model.
- `inference_kwargs`: Additional keyword arguments passed to the handler.

**Returns**:

A dictionary with the following keys:
- `prompt`: The prompt to be sent to the model.
- specified inference parameters.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator"></a>

## Module haystack\_integrations.components.generators.amazon\_bedrock.chat.chat\_generator

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator"></a>

### AmazonBedrockChatGenerator

Completes chats using LLMs hosted on Amazon Bedrock available via the Bedrock Converse API.

For example, to use the Anthropic Claude 3 Sonnet model, initialize this component with the
'anthropic.claude-3-5-sonnet-20240620-v1:0' model name.

### Usage example

```python
from haystack_integrations.components.generators.amazon_bedrock import AmazonBedrockChatGenerator
from haystack.dataclasses import ChatMessage
from haystack.components.generators.utils import print_streaming_chunk

messages = [ChatMessage.from_system("\nYou are a helpful, respectful and honest assistant, answer in German only"),
            ChatMessage.from_user("What's Natural Language Processing?")]


client = AmazonBedrockChatGenerator(model="anthropic.claude-3-5-sonnet-20240620-v1:0",
                                    streaming_callback=print_streaming_chunk)
client.run(messages, generation_kwargs={"max_tokens": 512})
```

### Multimodal example
```python
from haystack.dataclasses import ChatMessage, ImageContent
from haystack_integrations.components.generators.amazon_bedrock import AmazonBedrockChatGenerator

generator = AmazonBedrockChatGenerator(model="anthropic.claude-3-5-sonnet-20240620-v1:0")

image_content = ImageContent.from_file_path(file_path="apple.jpg")

message = ChatMessage.from_user(content_parts=["Describe the image using 10 words at most.", image_content])

response = generator.run(messages=[message])["replies"][0].text

print(response)
> The image shows a red apple.

### Tool usage example
# AmazonBedrockChatGenerator supports Haystack's unified tool architecture, allowing tools to be used
# across different chat generators. The same tool definitions and usage patterns work consistently
# whether using Amazon Bedrock, OpenAI, Ollama, or any other supported LLM providers.

```python
from haystack.dataclasses import ChatMessage
from haystack.tools import Tool
from haystack_integrations.components.generators.amazon_bedrock import AmazonBedrockChatGenerator

def weather(city: str):
    return f'The weather in {city} is sunny and 32°C'

__Define tool parameters__

tool_parameters = {
    "type": "object",
    "properties": {"city": {"type": "string"}},
    "required": ["city"]
}

__Create weather tool__

weather_tool = Tool(
    name="weather",
    description="useful to determine the weather in a given location",
    parameters=tool_parameters,
    function=weather
)

__Initialize generator with tool__

client = AmazonBedrockChatGenerator(
    model="anthropic.claude-3-5-sonnet-20240620-v1:0",
    tools=[weather_tool]
)

__Run initial query__

messages = [ChatMessage.from_user("What's the weather like in Paris?")]
results = client.run(messages=messages)

__Get tool call from response__

tool_message = next(msg for msg in results["replies"] if msg.tool_call)
tool_call = tool_message.tool_call

__Execute tool and send result back__

weather_result = weather(**tool_call.arguments)
new_messages = [
    messages[0],
    tool_message,
    ChatMessage.from_tool(tool_result=weather_result, origin=tool_call)
]

__Get final response__

final_result = client.run(new_messages)
print(final_result["replies"][0].text)

> Based on the information I've received, I can tell you that the weather in Paris is
> currently sunny with a temperature of 32°C (which is about 90°F).

```

AmazonBedrockChatGenerator uses AWS for authentication. You can use the AWS CLI to authenticate through your IAM.
For more information on setting up an IAM identity-based policy, see [Amazon Bedrock documentation]
(https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html).

If the AWS environment is configured correctly, the AWS credentials are not required as they're loaded
automatically from the environment or the AWS configuration file.
If the AWS environment is not configured, set `aws_access_key_id`, `aws_secret_access_key`,
  and `aws_region_name` as environment variables or pass them as
 [Secret](https://docs.haystack.deepset.ai/docs/secret-management) arguments. Make sure the region you set
supports Amazon Bedrock.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator.__init__"></a>

#### AmazonBedrockChatGenerator.\_\_init\_\_

```python
def __init__(
        model: str,
        aws_access_key_id: Secret | None = Secret.from_env_var(
            ["AWS_ACCESS_KEY_ID"], strict=False),
        aws_secret_access_key: Secret
    | None = Secret.from_env_var(  # noqa: B008
        ["AWS_SECRET_ACCESS_KEY"], strict=False),
        aws_session_token: Secret | None = Secret.from_env_var(
            ["AWS_SESSION_TOKEN"], strict=False),
        aws_region_name: Secret | None = Secret.from_env_var(
            ["AWS_DEFAULT_REGION"], strict=False),
        aws_profile_name: Secret | None = Secret.from_env_var(["AWS_PROFILE"],
                                                              strict=False),
        generation_kwargs: dict[str, Any] | None = None,
        streaming_callback: StreamingCallbackT | None = None,
        boto3_config: dict[str, Any] | None = None,
        tools: ToolsType | None = None,
        *,
        guardrail_config: dict[str, str] | None = None) -> None
```

Initializes the `AmazonBedrockChatGenerator` with the provided parameters. The parameters are passed to the

Amazon Bedrock client.

Note that the AWS credentials are not required if the AWS environment is configured correctly. These are loaded
automatically from the environment or the AWS configuration file and do not need to be provided explicitly via
the constructor. If the AWS environment is not configured users need to provide the AWS credentials via the
constructor. Aside from model, three required parameters are `aws_access_key_id`, `aws_secret_access_key`,
and `aws_region_name`.

**Arguments**:

- `model`: The model to use for text generation. The model must be available in Amazon Bedrock and must
be specified in the format outlined in the [Amazon Bedrock documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/model-ids-arns.html).
- `aws_access_key_id`: AWS access key ID.
- `aws_secret_access_key`: AWS secret access key.
- `aws_session_token`: AWS session token.
- `aws_region_name`: AWS region name. Make sure the region you set supports Amazon Bedrock.
- `aws_profile_name`: AWS profile name.
- `generation_kwargs`: Keyword arguments sent to the model. These parameters are specific to a model.
You can find the model specific arguments in the AWS Bedrock API
[documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html).
- `streaming_callback`: A callback function called when a new token is received from the stream.
By default, the model is not set up for streaming. To enable streaming, set this parameter to a callback
function that handles the streaming chunks. The callback function receives a
[StreamingChunk](https://docs.haystack.deepset.ai/docs/data-classes#streamingchunk) object and switches
the streaming mode on.
- `boto3_config`: The configuration for the boto3 client.
- `tools`: A list of Tool and/or Toolset objects, or a single Toolset for which the model can prepare calls.
Each tool should have a unique name.
- `guardrail_config`: Optional configuration for a guardrail that has been created in Amazon Bedrock.
This must be provided as a dictionary matching either
[GuardrailConfiguration](https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GuardrailConfiguration.html).
or, in streaming mode (when `streaming_callback` is set),
[GuardrailStreamConfiguration](https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_GuardrailStreamConfiguration.html).
If `trace` is set to `enabled`, the guardrail trace will be included under the `trace` key in the `meta`
attribute of the resulting `ChatMessage`.
Note: Enabling guardrails in streaming mode may introduce additional latency.
To manage this, you can adjust the `streamProcessingMode` parameter.
See the
[Guardrails Streaming documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/guardrails-streaming.html)
for more information.

**Raises**:

- `ValueError`: If the model name is empty or None.
- `AmazonBedrockConfigurationError`: If the AWS environment is not configured correctly or the model is
not supported.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator.to_dict"></a>

#### AmazonBedrockChatGenerator.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator.from_dict"></a>

#### AmazonBedrockChatGenerator.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "AmazonBedrockChatGenerator"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary with serialized data.

**Returns**:

Instance of `AmazonBedrockChatGenerator`.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator.run"></a>

#### AmazonBedrockChatGenerator.run

```python
@component.output_types(replies=list[ChatMessage])
def run(messages: list[ChatMessage],
        streaming_callback: StreamingCallbackT | None = None,
        generation_kwargs: dict[str, Any] | None = None,
        tools: ToolsType | None = None) -> dict[str, list[ChatMessage]]
```

Executes a synchronous inference call to the Amazon Bedrock model using the Converse API.

Supports both standard and streaming responses depending on whether a streaming callback is provided.

**Arguments**:

- `messages`: A list of `ChatMessage` objects forming the chat history.
- `streaming_callback`: Optional callback for handling streaming outputs.
- `generation_kwargs`: Optional dictionary of generation parameters. Some common parameters are:
- `maxTokens`: Maximum number of tokens to generate.
- `stopSequences`: List of stop sequences to stop generation.
- `temperature`: Sampling temperature.
- `topP`: Nucleus sampling parameter.
- `tools`: A list of Tool and/or Toolset objects, or a single Toolset for which the model can prepare calls.
Each tool should have a unique name.

**Raises**:

- `AmazonBedrockInferenceError`: If the Bedrock inference API call fails.

**Returns**:

A dictionary containing the model-generated replies under the `"replies"` key.

<a id="haystack_integrations.components.generators.amazon_bedrock.chat.chat_generator.AmazonBedrockChatGenerator.run_async"></a>

#### AmazonBedrockChatGenerator.run\_async

```python
@component.output_types(replies=list[ChatMessage])
async def run_async(
        messages: list[ChatMessage],
        streaming_callback: StreamingCallbackT | None = None,
        generation_kwargs: dict[str, Any] | None = None,
        tools: ToolsType | None = None) -> dict[str, list[ChatMessage]]
```

Executes an asynchronous inference call to the Amazon Bedrock model using the Converse API.

Designed for use cases where non-blocking or concurrent execution is desired.

**Arguments**:

- `messages`: A list of `ChatMessage` objects forming the chat history.
- `streaming_callback`: Optional async-compatible callback for handling streaming outputs.
- `generation_kwargs`: Optional dictionary of generation parameters. Some common parameters are:
- `maxTokens`: Maximum number of tokens to generate.
- `stopSequences`: List of stop sequences to stop generation.
- `temperature`: Sampling temperature.
- `topP`: Nucleus sampling parameter.
- `tools`: A list of Tool and/or Toolset objects, or a single Toolset for which the model can prepare calls.
Each tool should have a unique name.

**Raises**:

- `AmazonBedrockInferenceError`: If the Bedrock inference API call fails.

**Returns**:

A dictionary containing the model-generated replies under the `"replies"` key.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator"></a>

## Module haystack\_integrations.components.generators.amazon\_bedrock.generator

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator"></a>

### AmazonBedrockGenerator

Generates text using models hosted on Amazon Bedrock.

For example, to use the Anthropic Claude model, pass 'anthropic.claude-v2' in the `model` parameter.
Provide AWS credentials either through the local AWS profile or directly through
`aws_access_key_id`, `aws_secret_access_key`, `aws_session_token`, and `aws_region_name` parameters.

### Usage example

```python
from haystack_integrations.components.generators.amazon_bedrock import AmazonBedrockGenerator

generator = AmazonBedrockGenerator(
        model="anthropic.claude-v2",
        max_length=99
)

print(generator.run("Who is the best American actor?"))
```

AmazonBedrockGenerator uses AWS for authentication. You can use the AWS CLI to authenticate through your IAM.
For more information on setting up an IAM identity-based policy, see [Amazon Bedrock documentation]
(https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html).
If the AWS environment is configured correctly, the AWS credentials are not required as they're loaded
automatically from the environment or the AWS configuration file.
If the AWS environment is not configured, set `aws_access_key_id`, `aws_secret_access_key`,
`aws_session_token`, and `aws_region_name` as environment variables or pass them as
 [Secret](https://docs.haystack.deepset.ai/docs/secret-management) arguments. Make sure the region you set
supports Amazon Bedrock.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator.__init__"></a>

#### AmazonBedrockGenerator.\_\_init\_\_

```python
def __init__(
        model: str,
        aws_access_key_id: Secret | None = Secret.from_env_var(
            "AWS_ACCESS_KEY_ID", strict=False),
        aws_secret_access_key: Secret
    | None = Secret.from_env_var(  # noqa: B008
        "AWS_SECRET_ACCESS_KEY", strict=False),
        aws_session_token: Secret | None = Secret.from_env_var(
            "AWS_SESSION_TOKEN", strict=False),
        aws_region_name: Secret | None = Secret.from_env_var(
            "AWS_DEFAULT_REGION", strict=False),
        aws_profile_name: Secret | None = Secret.from_env_var("AWS_PROFILE",
                                                              strict=False),
        max_length: int | None = None,
        truncate: bool | None = None,
        streaming_callback: Callable[[StreamingChunk], None] | None = None,
        boto3_config: dict[str, Any] | None = None,
        model_family: MODEL_FAMILIES | None = None,
        **kwargs: Any) -> None
```

Create a new `AmazonBedrockGenerator` instance.

**Arguments**:

- `model`: The name of the model to use.
- `aws_access_key_id`: The AWS access key ID.
- `aws_secret_access_key`: The AWS secret access key.
- `aws_session_token`: The AWS session token.
- `aws_region_name`: The AWS region name. Make sure the region you set supports Amazon Bedrock.
- `aws_profile_name`: The AWS profile name.
- `max_length`: The maximum length of the generated text. This can also be set in the `kwargs` parameter
by using the model specific parameter name.
- `truncate`: Deprecated. This parameter no longer has any effect.
- `streaming_callback`: A callback function that is called when a new token is received from the stream.
The callback function accepts StreamingChunk as an argument.
- `boto3_config`: The configuration for the boto3 client.
- `model_family`: The model family to use. If not provided, the model adapter is selected based on the model
name.
- `kwargs`: Additional keyword arguments to be passed to the model.
You can find the model specific arguments in AWS Bedrock's
    [documentation](https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html).
These arguments are specific to the model. You can find them in the model's documentation.

**Raises**:

- `ValueError`: If the model name is empty or None.
- `AmazonBedrockConfigurationError`: If the AWS environment is not configured correctly or the model is
not supported.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator.run"></a>

#### AmazonBedrockGenerator.run

```python
@component.output_types(replies=list[str], meta=dict[str, Any])
def run(
    prompt: str,
    streaming_callback: Callable[[StreamingChunk], None] | None = None,
    generation_kwargs: dict[str, Any] | None = None
) -> dict[str, list[str] | dict[str, Any]]
```

Generates a list of string response to the given prompt.

**Arguments**:

- `prompt`: The prompt to generate a response for.
- `streaming_callback`: A callback function that is called when a new token is received from the stream.
- `generation_kwargs`: Additional keyword arguments passed to the generator.

**Raises**:

- `ValueError`: If the prompt is empty or None.
- `AmazonBedrockInferenceError`: If the model cannot be invoked.

**Returns**:

A dictionary with the following keys:
- `replies`: A list of generated responses.
- `meta`: A dictionary containing response metadata.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator.get_model_adapter"></a>

#### AmazonBedrockGenerator.get\_model\_adapter

```python
@classmethod
def get_model_adapter(
        cls,
        model: str,
        model_family: str | None = None) -> type[BedrockModelAdapter]
```

Gets the model adapter for the given model.

If `model_family` is provided, the adapter for the model family is returned.
If `model_family` is not provided, the adapter is auto-detected based on the model name.

**Arguments**:

- `model`: The model name.
- `model_family`: The model family.

**Raises**:

- `AmazonBedrockConfigurationError`: If the model family is not supported or the model cannot be
auto-detected.

**Returns**:

The model adapter class, or None if no adapter is found.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator.to_dict"></a>

#### AmazonBedrockGenerator.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.generators.amazon_bedrock.generator.AmazonBedrockGenerator.from_dict"></a>

#### AmazonBedrockGenerator.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "AmazonBedrockGenerator"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: Dictionary to deserialize from.

**Returns**:

Deserialized component.

<a id="haystack_integrations.components.rankers.amazon_bedrock.ranker"></a>

## Module haystack\_integrations.components.rankers.amazon\_bedrock.ranker

<a id="haystack_integrations.components.rankers.amazon_bedrock.ranker.AmazonBedrockRanker"></a>

### AmazonBedrockRanker

Ranks Documents based on their similarity to the query using Amazon Bedrock's Cohere Rerank model.

Documents are indexed from most to least semantically relevant to the query.

Supported Amazon Bedrock models:
- cohere.rerank-v3-5:0
- amazon.rerank-v1:0

Usage example:
```python
from haystack import Document
from haystack.utils import Secret
from haystack_integrations.components.rankers.amazon_bedrock import AmazonBedrockRanker

ranker = AmazonBedrockRanker(
    model="cohere.rerank-v3-5:0",
    top_k=2,
    aws_region_name=Secret.from_token("eu-central-1")
)

docs = [Document(content="Paris"), Document(content="Berlin")]
query = "What is the capital of germany?"
output = ranker.run(query=query, documents=docs)
docs = output["documents"]
```

AmazonBedrockRanker uses AWS for authentication. You can use the AWS CLI to authenticate through your IAM.
For more information on setting up an IAM identity-based policy, see [Amazon Bedrock documentation]
(https://docs.aws.amazon.com/bedrock/latest/userguide/security_iam_id-based-policy-examples.html).

If the AWS environment is configured correctly, the AWS credentials are not required as they're loaded
automatically from the environment or the AWS configuration file.
If the AWS environment is not configured, set `aws_access_key_id`, `aws_secret_access_key`,
and `aws_region_name` as environment variables or pass them as
[Secret](https://docs.haystack.deepset.ai/docs/secret-management) arguments. Make sure the region you set
supports Amazon Bedrock.

<a id="haystack_integrations.components.rankers.amazon_bedrock.ranker.AmazonBedrockRanker.to_dict"></a>

#### AmazonBedrockRanker.to\_dict

```python
def to_dict() -> dict[str, Any]
```

Serializes the component to a dictionary.

**Returns**:

Dictionary with serialized data.

<a id="haystack_integrations.components.rankers.amazon_bedrock.ranker.AmazonBedrockRanker.from_dict"></a>

#### AmazonBedrockRanker.from\_dict

```python
@classmethod
def from_dict(cls, data: dict[str, Any]) -> "AmazonBedrockRanker"
```

Deserializes the component from a dictionary.

**Arguments**:

- `data`: The dictionary to deserialize from.

**Returns**:

The deserialized component.

<a id="haystack_integrations.components.rankers.amazon_bedrock.ranker.AmazonBedrockRanker.run"></a>

#### AmazonBedrockRanker.run

```python
@component.output_types(documents=list[Document])
def run(query: str,
        documents: list[Document],
        top_k: int | None = None) -> dict[str, list[Document]]
```

Use the Amazon Bedrock Reranker to re-rank the list of documents based on the query.

**Arguments**:

- `query`: Query string.
- `documents`: List of Documents.
- `top_k`: The maximum number of Documents you want the Ranker to return.

**Raises**:

- `ValueError`: If `top_k` is not > 0.

**Returns**:

A dictionary with the following keys:
- `documents`: List of Documents most similar to the given query in descending order of similarity.

