from chameleon.config import AUTO_RELOAD
from chameleon.config import DEBUG_MODE
from collections.abc import Mapping
from devpi_common.metadata import get_latest_version
from devpi_common.validation import normalize_name
from devpi_server.log import threadlog
from devpi_server.main import Fatal
from devpi_web.config import add_indexer_backend_option
from devpi_web.config import get_pluginmanager
from devpi_web.doczip import remove_docs
from devpi_web.indexing import ProjectIndexingInfo
from devpi_web.indexing import is_project_cached
from pluggy import HookimplMarker
from pyramid_chameleon.renderer import ChameleonRendererLookup
import os
import sys


hookimpl = HookimplMarker("devpiweb")
devpiserver_hookimpl = HookimplMarker("devpiserver")


def theme_static_url(request, path):
    return request.static_url(
        os.path.join(request.registry['theme_path'], 'static', path))


def add_href_css(request, href, rel="stylesheet", type="text/css"):  # noqa: A002
    css = request.environ.setdefault("devpiweb.head_css", [])
    css.append(dict(href=href, rel=rel, type=type))


def add_src_script(request, src):
    scripts = request.environ.setdefault("devpiweb.head_scripts", [])
    scripts.append(dict(src=src))


def add_static_css(request, href):
    return request.add_href_css(request.static_url(href))


def add_static_script(request, src):
    return request.add_src_script(request.static_url(src))


def navigation_version(context):
    version = context.version
    if version == 'latest':
        stage = context.model.getstage(context.username, context.index)
        version = stage.get_latest_version_perstage(context.project)
    elif version == 'stable':
        stage = context.model.getstage(context.username, context.index)
        version = stage.get_latest_version_perstage(context.project, stable=True)
    return version


def navigation_info(request):
    context = request.context
    matchdict = request.matchdict
    path = [dict(
        url=request.route_url("root"),
        title="devpi")]
    result = dict(path=path)
    if matchdict and "user" in matchdict:
        user = context.username
        path.append(dict(
            url=request.route_url(
                "/{user}", user=user),
            title="%s" % user))
    else:
        return result
    if "index" in matchdict:
        index = context.index
        path.append(dict(
            url=request.stage_url(user, index),
            title="%s" % index))
    else:
        return result
    if "project" in matchdict:
        project = context.matchdict['project']
        name = normalize_name(project)
        path.append(dict(
            url=request.route_url(
                "/{user}/{index}/{project}", user=user, index=index, project=name),
            title=name))
    else:
        return result
    if "version" in matchdict:
        version = navigation_version(context)
        path.append(dict(
            url=request.route_url(
                "/{user}/{index}/{project}/{version}",
                user=user, index=index, project=name, version=version),
            title=version))
    else:
        return result
    return result


def status_info(request):
    msgs = []
    pm = request.registry['devpiweb-pluginmanager']
    for result in pm.hook.devpiweb_get_status_info(request=request):
        msgs.extend(result)
    states = {x["status"] for x in msgs}
    if 'fatal' in states:
        status = 'fatal'
        short_msg = 'fatal'
    elif 'warn' in states:
        status = 'warn'
        short_msg = 'degraded'
    else:
        status = 'ok'
        short_msg = 'ok'
    url = request.route_url('/+status')
    return dict(status=status, short_msg=short_msg, msgs=msgs, url=url)


def query_docs_html(request):
    search_index = request.registry['search_index']
    return search_index.get_query_parser_html_help()


class ThemeChameleonRendererLookup(ChameleonRendererLookup):
    auto_reload = AUTO_RELOAD
    debug = DEBUG_MODE
    theme_path = None

    def __call__(self, info):
        # if the template exists in the theme, we will use it instead of the
        # original template
        if "devpi_web:" not in info.name and (theme_path := self.theme_path):
            theme_file = os.path.join(theme_path, info.name)
            if os.path.exists(theme_file):
                info.name = theme_file
        return ChameleonRendererLookup.__call__(self, info)


def includeme(config):
    from devpi_web import __version__
    from pyramid_chameleon.interfaces import IChameleonLookup
    from pyramid_chameleon.zpt import ZPTTemplateRenderer
    config.include('pyramid_chameleon')
    # we overwrite the template lookup to allow theming
    lookup = ThemeChameleonRendererLookup(ZPTTemplateRenderer, config.registry)
    config.registry.registerUtility(lookup, IChameleonLookup, name='.pt')
    config.add_static_view('+static-%s' % __version__, 'static')
    theme_path = config.registry['theme_path']
    if theme_path:
        # if a theme is used, we set the path on the lookup instance
        lookup.theme_path = theme_path
        # if a 'static' directory exists in the theme, we add it and a helper
        # method 'theme_static_url' on the request
        static_path = os.path.join(theme_path, 'static')
        if os.path.exists(static_path):
            config.add_static_view('+theme-static-%s' % __version__, static_path)
            config.add_request_method(theme_static_url)
    config.include("devpi_web.macroregistry")
    config.add_route('root', '/', accept='text/html')
    config.add_route('search', '/+search', accept='text/html')
    config.add_route('search_help', '/+searchhelp', accept='text/html')
    config.add_route("project_refresh", "/{user}/{index}/{project}/refresh")
    config.add_route(
        "docroot",
        "/{user}/{index}/{project}/{version}/+doc/{relpath:.*}")
    config.add_route(
        "docviewroot",
        "/{user}/{index}/{project}/{version}/+d/{relpath:.*}")
    config.add_route(
        "toxresults",
        "/{user}/{index}/{project}/{version}/+toxresults/{basename}")
    config.add_route(
        "toxresult",
        "/{user}/{index}/{project}/{version}/+toxresults/{basename}/{toxresult}")
    # BBB can be removed once we require devpi-server >= 6.1.0
    config.add_route(
        "/{user}/",
        "/{user:[^+/]+}/")
    config.add_tween("devpi_web.views.tween_trailing_slash_redirect")
    config.add_request_method(add_href_css)
    config.add_request_method(add_src_script)
    config.add_request_method(add_static_css)
    config.add_request_method(add_static_script)
    config.add_request_method(navigation_info, reify=True)
    config.add_request_method(status_info, reify=True)
    config.add_request_method(query_docs_html, reify=True)
    config.scan()


def get_indexer_from_config(config):
    pm = get_pluginmanager(config)
    indexers = {
        x['name']: x
        for x in pm.hook.devpiweb_indexer_backend()}
    # return null indexer if argument doesn't exist, which can happen
    # with devpi-import for example
    indexer_backend = getattr(config.args, "indexer_backend", "null")
    if isinstance(indexer_backend, dict):
        # a yaml config may return a dict
        settings = dict(indexer_backend)
        name = settings.pop('name')
    else:
        (name, sep, setting_str) = indexer_backend.partition(':')
        settings = {}
        if setting_str:
            for item in setting_str.split(','):
                (key, value) = item.split('=', 1)
                settings[key] = value
    indexer = indexers[name]['indexer'](config=config, settings=settings)
    return indexer


def get_indexer(xom):
    # lookup cached value
    indexer = getattr(xom, 'devpiweb_indexer', None)
    if indexer is not None:
        return indexer
    indexer = get_indexer_from_config(xom.config)
    if hasattr(indexer, 'runtime_setup'):
        indexer.runtime_setup(xom)
    # cache the expensive setup
    xom.devpiweb_indexer = indexer
    return indexer


@devpiserver_hookimpl
def devpiserver_pyramid_configure(config, pyramid_config):
    # make the theme path absolute if it exists and make it available via the
    # pyramid registry
    theme_path = config.args.theme
    if theme_path:
        theme_path = os.path.abspath(theme_path)
        if not os.path.exists(theme_path):
            threadlog.error(
                "The theme path '%s' does not exist." % theme_path)
            sys.exit(1)
        if not os.path.isdir(theme_path):
            threadlog.error(
                "The theme path '%s' is not a directory." % theme_path)
            sys.exit(1)
    pyramid_config.registry['theme_path'] = theme_path
    pyramid_config.registry["debug_macros"] = config.args.debug_macros
    # by using include, the package name doesn't need to be set explicitly
    # for registrations of static views etc
    pyramid_config.include('devpi_web.main')
    pyramid_config.registry['devpiweb-pluginmanager'] = get_pluginmanager(config)
    xom = pyramid_config.registry['xom']
    pyramid_config.registry['search_index'] = get_indexer(xom)


@devpiserver_hookimpl
def devpiserver_add_parser_options(parser):
    theme = parser.addgroup("devpi-web theme options")
    theme.addoption(
        "--theme", action="store",
        help="folder with template and resource overwrites for the web interface")
    theme.addoption(
        "--debug-macros",
        action="store_true",
        help="add html comments at start and end of macros, and log deprecation warnings",
    )
    doczip = parser.addgroup("devpi-web doczip options")
    doczip.addoption(
        "--documentation-path", action="store",
        help="path for unzipped documentation. "
             "By default the --serverdir is used.")
    doczip.addoption(
        "--keep-docs-packed", default=False,
        action="store_true",
        help="fetch data from doczips instead of unpacking them")
    indexing = parser.addgroup("devpi-web search indexing")
    add_indexer_backend_option(indexing)


@devpiserver_hookimpl
def devpiserver_mirror_initialnames(stage, projectnames):
    ix = get_indexer(stage.xom)
    threadlog.info(
        "indexing '%s' mirror with %s projects",
        stage.name,
        len(projectnames))
    if isinstance(projectnames, Mapping):
        # since devpi-server 6.6.0 mirrors return a mapping where
        # the un-normalized names are in the values
        projectnames = projectnames.values()
    ix.update_projects(
        ProjectIndexingInfo(stage=stage, name=name)
        for name in projectnames)
    threadlog.info("finished mirror indexing operation")


@devpiserver_hookimpl
def devpiserver_stage_created(stage):
    # make sure we are at the current state
    stage.keyfs.restart_read_transaction()
    stage = stage.model.getstage(stage.name)
    if stage is None:
        # the stage was deleted
        return
    if stage.ixconfig["type"] == "mirror":
        threadlog.info("triggering load of initial projectnames for %s", stage.name)
        stage.list_projects_perstage()


@devpiserver_hookimpl
def devpiserver_cmdline_run(xom):
    docs_path = xom.config.args.documentation_path
    if docs_path is not None and not os.path.isabs(docs_path):
        raise Fatal("The path for unzipped documentation must be absolute.")
    # return None to allow devpi-server to run


def delete_project(stage, name):
    if stage is None:
        return
    ix = get_indexer(stage.xom)
    ix.delete_projects([ProjectIndexingInfo(stage=stage, name=name)])


def index_project(stage, name):
    if stage is None:
        return
    ix = get_indexer(stage.xom)
    ix.update_projects([ProjectIndexingInfo(stage=stage, name=name)])


@devpiserver_hookimpl
def devpiserver_on_upload(stage, project, version, link):
    if not link.entry.file_exists():
        # on replication or import we might be at a lower than
        # current revision and the file might have been deleted already
        threadlog.debug("ignoring lost upload: %s", link)
    elif link.rel == "doczip":
        index_project(stage, project)


@devpiserver_hookimpl
def devpiserver_on_changed_versiondata(stage, project, version, metadata):
    if stage is None:
        return
    if not metadata:
        if is_project_cached(stage, project) and not stage.has_project_perstage(project):
            delete_project(stage, project)
            return
        versions = stage.list_versions(project)
        if versions:
            version = get_latest_version(versions)
            if version:
                threadlog.debug("A version of %s was deleted, using latest version %s for indexing" % (
                    project, version))
                metadata = stage.get_versiondata(project, version)
    if metadata:
        index_project(stage, metadata['name'])


@devpiserver_hookimpl
def devpiserver_on_remove_file(stage, relpath):
    if relpath.endswith(".doc.zip"):
        project, version = (
            os.path.basename(relpath).rsplit('.doc.zip')[0].rsplit('-', 1)
        )
        remove_docs(stage, project, version)


@devpiserver_hookimpl(optionalhook=True)
def devpiserver_on_replicated_file(stage, project, version, link, serial, back_serial, is_from_mirror):
    if is_from_mirror:
        return
    elif link.rel == "doczip":
        index_project(stage, project)


@devpiserver_hookimpl
def devpiserver_authcheck_always_ok(request):
    route = request.matched_route
    if not route:
        return None
    if "+static" in route.name and "/+static" in request.url:
        return True
    if "+theme-static" in route.name and "/+theme-static" in request.url:
        return True
    return None
