import io
import struct

import pytest

from kafka.protocol.types import UnsignedVarInt32, VarInt32, VarInt64


@pytest.mark.parametrize(('value','expected_encoded'), [
    (0, [0x00]),
    (-1, [0xFF, 0xFF, 0xFF, 0xFF, 0x0F]),
    (1, [0x01]),
    (63, [0x3F]),
    (-64, [0xC0, 0xFF, 0xFF, 0xFF, 0x0F]),
    (64, [0x40]),
    (8191, [0xFF, 0x3F]),
    (-8192, [0x80, 0xC0, 0xFF, 0xFF, 0x0F]),
    (8192, [0x80, 0x40]),
    (-8193, [0xFF, 0xBF, 0xFF, 0xFF, 0x0F]),
    (1048575, [0xFF, 0xFF, 0x3F]),
    (1048576, [0x80, 0x80, 0x40]),
    (2147483647, [0xFF, 0xFF, 0xFF, 0xFF, 0x07]),
    (-2147483648, [0x80, 0x80, 0x80, 0x80, 0x08]),
])
def test_unsigned_varint_serde(value, expected_encoded):
    value &= 0xffffffff
    encoded = UnsignedVarInt32.encode(value)
    assert encoded == b''.join(struct.pack('>B', x) for x in expected_encoded)
    assert value == UnsignedVarInt32.decode(io.BytesIO(encoded))


@pytest.mark.parametrize(('value','expected_encoded'), [
    (0, [0x00]),
    (-1, [0x01]),
    (1, [0x02]),
    (63, [0x7E]),
    (-64, [0x7F]),
    (64, [0x80, 0x01]),
    (-65, [0x81, 0x01]),
    (8191, [0xFE, 0x7F]),
    (-8192, [0xFF, 0x7F]),
    (8192, [0x80, 0x80, 0x01]),
    (-8193, [0x81, 0x80, 0x01]),
    (1048575, [0xFE, 0xFF, 0x7F]),
    (-1048576, [0xFF, 0xFF, 0x7F]),
    (1048576, [0x80, 0x80, 0x80, 0x01]),
    (-1048577, [0x81, 0x80, 0x80, 0x01]),
    (134217727, [0xFE, 0xFF, 0xFF, 0x7F]),
    (-134217728, [0xFF, 0xFF, 0xFF, 0x7F]),
    (134217728, [0x80, 0x80, 0x80, 0x80, 0x01]),
    (-134217729, [0x81, 0x80, 0x80, 0x80, 0x01]),
    (2147483647, [0xFE, 0xFF, 0xFF, 0xFF, 0x0F]),
    (-2147483648, [0xFF, 0xFF, 0xFF, 0xFF, 0x0F]),
])
def test_signed_varint_serde(value, expected_encoded):
    encoded = VarInt32.encode(value)
    assert encoded == b''.join(struct.pack('>B', x) for x in expected_encoded)
    assert value == VarInt32.decode(io.BytesIO(encoded))


@pytest.mark.parametrize(('value','expected_encoded'), [
    (0, [0x00]),
    (-1, [0x01]),
    (1, [0x02]),
    (63, [0x7E]),
    (-64, [0x7F]),
    (64, [0x80, 0x01]),
    (-65, [0x81, 0x01]),
    (8191, [0xFE, 0x7F]),
    (-8192, [0xFF, 0x7F]),
    (8192, [0x80, 0x80, 0x01]),
    (-8193, [0x81, 0x80, 0x01]),
    (1048575, [0xFE, 0xFF, 0x7F]),
    (-1048576, [0xFF, 0xFF, 0x7F]),
    (1048576, [0x80, 0x80, 0x80, 0x01]),
    (-1048577, [0x81, 0x80, 0x80, 0x01]),
    (134217727, [0xFE, 0xFF, 0xFF, 0x7F]),
    (-134217728, [0xFF, 0xFF, 0xFF, 0x7F]),
    (134217728, [0x80, 0x80, 0x80, 0x80, 0x01]),
    (-134217729, [0x81, 0x80, 0x80, 0x80, 0x01]),
    (2147483647, [0xFE, 0xFF, 0xFF, 0xFF, 0x0F]),
    (-2147483648, [0xFF, 0xFF, 0xFF, 0xFF, 0x0F]),
    (17179869183, [0xFE, 0xFF, 0xFF, 0xFF, 0x7F]),
    (-17179869184, [0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (17179869184, [0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (-17179869185, [0x81, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (2199023255551, [0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (-2199023255552, [0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (2199023255552, [0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (-2199023255553, [0x81, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (281474976710655, [0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (-281474976710656, [0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (281474976710656, [0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (-281474976710657, [0x81, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 1]),
    (36028797018963967, [0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (-36028797018963968, [0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (36028797018963968, [0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (-36028797018963969, [0x81, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (4611686018427387903, [0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (-4611686018427387904, [0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x7F]),
    (4611686018427387904, [0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (-4611686018427387905, [0x81, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x01]),
    (9223372036854775807, [0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01]),
    (-9223372036854775808, [0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01]),
])
def test_signed_varlong_serde(value, expected_encoded):
    encoded = VarInt64.encode(value)
    assert encoded == b''.join(struct.pack('>B', x) for x in expected_encoded)
    assert value == VarInt64.decode(io.BytesIO(encoded))
