---
id: code-5001-public
title: 5001 - Code Injection
sidebar_label:  5001 - Code Injection
---
import {OssOnly, FbInternalOnly} from 'docusaurus-plugin-internaldocs-fb/internal';
import SubprocessFbSolution from '../fb/warning_codes/_subprocess_fb_solution.md';
import HydraFbSolution from '../fb/warning_codes/_hydra_fb_solution.md';

## TL;DR

This category indicates that user-controlled input flows into a sink that allows code or shell command execution. This directly leads to Remote Code Execution which can be assumed to mean complete compromise of the server.


## RCE via Code Injection (eval/exec)

### ISSUE

The simplest kind of RCE involves user input flowing into a function such as eval or exec which are intended to interpret or run python code.

### EXAMPLE

```python
def update_search_account_filtering(request: HttpRequest) -> HttpResponse:
    ...
    if action in ["delete", "add", "update"]:
        ...
        filter_by_username = eval(request.POST.get("filter_by_username", "True"))
```
### RECOMMENDED SOLUTION
There are few reasons to use these functions, and even fewer reasons to allow a user to control the content of these functions. Generally, we recommend not making calls to these functions with user input. If you only need to eval python datatypes you can use `ast.literal_eval`. Using it on arbitrary user input can still lead to DOS attack but can't be exploited for code execution ([details](https://docs.python.org/3/library/ast.html#ast.literal_eval)).

## RCE via Command Injection (os.system)

### ISSUE

This kind of RCE involves user input flowing into a command executed in a system shell. If a user can control a portion of the command being executed in a shell, they can potentially add additional arbitrary commands to be executed.

### EXAMPLE

The following code is intended to run the spellcheck binary on a user provided text:
```python
def spellcheck(request: HttpRequest):
    command = f"/usr/bin/spellcheck -l {request.GET['text']}"
    return subprocess.getoutput(command)
```
An attacker, however, can supply a path such as `'test' && rm -rf /`, which would result in the following command being executed: `/usr/bin/spellcheck -l 'test' && rm -rf /`. Since this command is executed in a system shell the `rm -rf /` command will be executed after the spellcheck command.

### RECOMMENDED SOLUTION

<OssOnly>

In general, we recommend avoiding creation of a subprocess and prefer using the API provided by the language.
However, if you need to create a subprocess, we recommend using an API such as `subprocess.run`, which allows you to separate arguments from the executable being invoked. **DO NOT add the `shell=True` argument otherwise the code would still be vulnerable like the previous example**

```python
def spellcheck(request: HttpRequest):
    command = ["/usr/bin/spellcheck", "-l", request.GET['text']]
    subprocess.run(command)
```
*NOTE: be conscious of the fact that arguments to an executable can still lead to code execution (e.g., the `-exec` argument of `find`).*

</OssOnly>

<FbInternalOnly>

<SubprocessFbSolution/>
<HydraFbSolution/>


## SEVERITY SCORING

A vulnerability's severity is context-dependent, but here's rough guidelines with common scenarios.

### CRITICAL
Vulnerability affects an externally exposed endpoint (identified by SAPP feature via:reachable_from_prod or always-via:exposed_externally) OR the service handles sensitive business-critical data/user data. Direct RCE via eval(), os.system(), or subprocess.getoutput() with external user-controlled input. Service has access to sensitive credentials (hipster groups, secrets)

#### Examples:
- T188945618: User input flows into DataFrame.query() which internally calls DataFrame.eval(), allowing code execution by closing quotes and invoking os.system. The vulnerability is exploitable from GraphQL with a logged-out VC, and affects a payments service handling sensitive financial data. Exploitation was confirmed via both Thrift fiddle and GraphQL endpoints.

### SIGNIFICANT
Vulnerability is in internal-only services, but still allows code execution with user-controlled input. The service has access to internal systems/data but not highly sensitive production data. May require internal employee access to exploit. LLM output flowing to eval/exec without isolation/sandboxing.

#### Examples:
- T245530940: User-controlled FBlearner Flow parameters flow into command execution functions. While this leads to RCE, FBlearner workflows have ACLs by default and are internal-only, reducing the exposure and exploitation difficulty for external attackers.
- T71305397: The augmented msgpack deserializer dynamically loads modules and calls arbitrary functions based on user-provided data, allowing code execution via specially crafted ExtType data. While technically RCE is possible via Async job scheduling, the affected component is internal and exploitation requires knowledge of the specific msgpack format and internal service architecture.

### LIMITED/BAD PRACTICE
Vulnerability is in experimental/test code, development tools, or code paths with limited reachability. The input is partially constrained that Pysa cannot detect (e.g. regex) or requires specific conditions to exploit.

#### Examples:
- T226400980: Import injection in Cardinal API. Requires malicious .json on filesystem AND classes that perform code execution, so complex attack chain reduces practical exploitability.

### FALSE POSITIVE
The data flow tracked by Pysa is incorrect (e.g., tainting the wrong fields in an object). The code uses safe deserialization.

### DO NOT CARE
The flow tracked by Pysa is correct, but exploitation is impossible (e.g. input validation that Pysa cannot detect) OR the code path is unreachable.

</FbInternalOnly>
