---
id: code-6065-public
title: 6065 - Commandline arguments injection
sidebar_label: 6065 - Commandline arguments injection
---
import {OssOnly, FbInternalOnly} from 'docusaurus-plugin-internaldocs-fb/internal';
import SubprocessFbSolution from '../fb/warning_codes/_subprocess_fb_solution.md';

## TL;DR

This category indicates that user-controlled input flows into a command-line argument used to execute an external process. Unlike category 5001, this leads to a Remote Code Execution issue only in specific cases (e.g., `shell=True` parameter or when executing particular binaries).

### ISSUE

`subprocess.Popen`, `subprocess.run`, `subprocess.call`, and other functions do a good job in preventing by default the command injection issues we described in category [5001](warning_codes/5001.md). The values supplied in the `args` parameter (excluding the first which represents the executable) are considered only as arguments and not as commands to be interpreted in a system shell (more details in the python [documentation](https://docs.python.org/3/library/subprocess.html#subprocess.Popen)). However, this safe behaviour can be manually bypassed by specifying the `shell=True` parameter, which reintroduces the command injection issue.

### EXAMPLE

The following code is intended to run the spellcheck binary on a user provided text:
```python
def spellcheck(request: HttpRequest):
    command = "/usr/bin/spellcheck -l {}".format(request.GET['text'])
    return subprocess.run(command, shell=True)
```
An attacker, however, can supply a path such as `'test' && rm -rf /`, which would result in the following command being executed: `/usr/bin/spellcheck -l 'test' && rm -rf /`. Since this command is executed in a system shell the `rm -rf /` command will be executed after the spellcheck command.

### RECOMMENDED SOLUTION

<OssOnly>

In general, we recommend avoiding creation of a subprocess and prefer using the API provided by the language.
However, if you need to create a subprocess, we recommend using a safe API such as `subprocess.run` and avoiding use of the `shell=True` argument. If this is not possible, we recommend ensuring that the user-controlled values are shell-escaped with `shlex.quote`.

```python
def spellcheck(request: HttpRequest):
    command = ["/usr/bin/spellcheck", "-l", request.GET['text']]
    subprocess.run(command)
```
*NOTE: be conscious of the fact that arguments to an executable can still lead to code execution (e.g., the `-exec` argument of `find`).*

</OssOnly>


<FbInternalOnly>

<SubprocessFbSolution/>

## SEVERITY SCORING

A vulnerability's severity is context-dependent, but here's rough guidelines with common scenarios.

### CRITICAL
The subprocess call uses `shell=True` AND the endpoint is externally reachable OR handles sensitive data. The attacker can inject arbitrary shell commands. Can be chained to full RCE via shell metacharacters

#### Examples:
- T239908731: User-controllable shard_prefix flows directly into command construction and execution. Direct path to RCE via parameter manipulation. The vulnerability is in Akkio placement handle handlers, which manage ZippyDB database sharding infrastructure. The vulnerability is in a FBJE handler, meaning it processes jobs that could potentially have externally-controlled inputs, and executes on infrastructure managing database configurations.

### SIGNIFICANT
The subprocess call uses `shell=True` in internal services, OR the subprocess call doesn't use `shell=True` but the binary being executed is known to have dangerous argument handling (e.g., allows reading/writing files via arguments).

#### Examples:
- T244241355: Thrift params flow into subprocess.run with shell=True allowing arbitrary command execution. Complete server compromise is possible with low exploitation difficulty. Internal-only service limits the attack surface to Meta employees
- T246423813: User-controlled input from the patches parameter flows directly into subprocess.run() with `shell=True`, allowing arbitrary command execution. Requires internal access to the Confucius platform and attack would need to be conducted through the agent interface, which provides some indirect protection

### LIMITED/BAD PRACTICE
Argument injection without `shell=True` to a binary with no known dangerous argument patterns. Impact is limited to manipulating the specific command's behavior. Code is in experimental/internal tooling.

#### Examples:
- T245745219: Command injection in internal hackathon analect. Attack surface restricted to users with hackathon tool access.

### FALSE POSITIVE
The data flow is incorrect or the subprocess is using proper argument handling (e.g. TrustedSubprocessWithList).

### DO NOT CARE
The flow is correct, but the command arguments are constrained to a safe set of values, OR the binary being called is safe even with arbitrary arguments.

</FbInternalOnly>
