# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved
import datetime
import re
from textwrap import dedent
from typing import Any

try:
    from _pytest.python_api import RaisesContext
except ImportError:
    from _pytest.raises import RaisesExc as RaisesContext  # type: ignore
from pytest import mark, param, raises

from hydra._internal.utils import _locate
from hydra.utils import get_class, get_method, get_object
from tests.instantiate import (
    AClass,
    Adam,
    AnotherClass,
    ASubclass,
    NestingClass,
    Parameters,
    an_object,
)

from .module_shadowed_by_function import a_function


@mark.parametrize(
    "name,expected",
    [
        param(
            "int",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'int':
                    ModuleNotFoundError\("No module named 'int'",?\)
                    Are you sure that module 'int' is installed\?
                    """
                ).strip(),
            ),
            id="int",
        ),
        param("builtins.int", int, id="builtins_explicit"),
        param("builtins.int.from_bytes", int.from_bytes, id="method_of_builtin"),
        param(
            "builtins.int.not_found",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'builtins\.int\.not_found':
                    AttributeError\("type object 'int' has no attribute 'not_found'",?\)
                    Are you sure that 'not_found' is an attribute of 'builtins\.int'\?
                    """
                ).strip(),
            ),
            id="builtin_attribute_error",
        ),
        param(
            "datetime",
            datetime,
            id="top_level_module",
        ),
        ("tests.instantiate.Adam", Adam),
        ("tests.instantiate.Parameters", Parameters),
        ("tests.instantiate.AClass", AClass),
        param(
            "tests.instantiate.AClass.static_method",
            AClass.static_method,
            id="staticmethod",
        ),
        param(
            "tests.instantiate.AClass.not_found",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'tests\.instantiate\.AClass\.not_found':
                    AttributeError\("type object 'AClass' has no attribute 'not_found'",?\)
                    Are you sure that 'not_found' is an attribute of 'tests\.instantiate\.AClass'\?
                    """
                ).strip(),
            ),
            id="class_attribute_error",
        ),
        ("tests.instantiate.ASubclass", ASubclass),
        ("tests.instantiate.NestingClass", NestingClass),
        ("tests.instantiate.AnotherClass", AnotherClass),
        ("tests.instantiate.module_shadowed_by_function", a_function),
        param(
            "",
            raises(ImportError, match=("Empty path")),
            id="invalid-path-empty",
        ),
        param(
            "toplevel_not_found",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'toplevel_not_found':
                    ModuleNotFoundError\("No module named 'toplevel_not_found'",?\)
                    Are you sure that module 'toplevel_not_found' is installed\?
                    """
                ).strip(),
            ),
            id="toplevel_not_found",
        ),
        param(
            "tests.instantiate.b.c.Door",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'tests\.instantiate\.b\.c\.Door':
                    ModuleNotFoundError\("No module named 'tests\.instantiate\.b'",?\)
                    Are you sure that 'b' is importable from module 'tests\.instantiate'\?"""
                ).strip(),
            ),
            id="nested_not_found",
        ),
        param(
            "tests.instantiate.import_error",
            raises(
                ImportError,
                match=re.escape(
                    dedent(
                        """\
                        Error loading 'tests.instantiate.import_error':
                        AssertionError()"""
                    )
                ),
            ),
            id="import_assertion_error",
        ),
        param("tests.instantiate.an_object", an_object, id="object"),
    ],
)
def test_locate(name: str, expected: Any) -> None:
    if isinstance(expected, RaisesContext):
        with expected:
            _locate(name)
    else:
        assert _locate(name) == expected


@mark.parametrize(
    "name",
    [
        param(".", id="invalid-path-period"),
        param("..", id="invalid-path-period2"),
        param(".mod", id="invalid-path-relative"),
        param("..mod", id="invalid-path-relative2"),
        param("mod.", id="invalid-path-trailing-dot"),
        param("mod..another", id="invalid-path-two-dots"),
    ],
)
def test_locate_relative_import_fails(name: str) -> None:
    with raises(
        ValueError,
        match=r"Error loading '.*': invalid dotstring\."
        + re.escape("\nRelative imports are not supported."),
    ):
        _locate(name)


@mark.parametrize(
    "path,expected",
    [
        param("tests.instantiate.AClass", AClass, id="class"),
        param("builtins.print", print, id="callable"),
        param(
            "datetime",
            raises(
                ValueError,
                match="Located non-callable of type 'module' while loading 'datetime'",
            ),
            id="module-error",
        ),
        param(
            "tests.instantiate.an_object",
            raises(
                ValueError,
                match="Located non-callable of type 'object' while loading 'tests.instantiate.an_object'",
            ),
            id="object-error",
        ),
    ],
)
def test_get_method(path: str, expected: Any) -> None:
    if isinstance(expected, RaisesContext):
        with expected:
            get_method(path)
    else:
        assert get_method(path) == expected


@mark.parametrize(
    "path,expected",
    [
        param("tests.instantiate.AClass", AClass, id="class"),
        param(
            "builtins.print",
            raises(
                ValueError,
                match="Located non-class of type 'builtin_function_or_method'"
                + " while loading 'builtins.print'",
            ),
            id="callable-error",
        ),
        param(
            "datetime",
            raises(
                ValueError,
                match="Located non-class of type 'module' while loading 'datetime'",
            ),
            id="module-error",
        ),
        param(
            "tests.instantiate.an_object",
            raises(
                ValueError,
                match="Located non-class of type 'object' while loading 'tests.instantiate.an_object'",
            ),
            id="object-error",
        ),
    ],
)
def test_get_class(path: str, expected: Any) -> None:
    if isinstance(expected, RaisesContext):
        with expected:
            get_class(path)
    else:
        assert get_class(path) == expected


@mark.parametrize(
    "path,expected",
    [
        param("tests.instantiate.AClass", AClass, id="class"),
        param("builtins.print", print, id="callable"),
        param(
            "datetime",
            datetime,
            id="module-error",
        ),
        param("tests.instantiate.an_object", an_object, id="object"),
        param(
            "builtins.int.not_found",
            raises(
                ImportError,
                match=dedent(
                    r"""
                    Error loading 'builtins\.int\.not_found':
                    AttributeError\("type object 'int' has no attribute 'not_found'",?\)
                    Are you sure that 'not_found' is an attribute of 'builtins\.int'\?
                    """
                ).strip(),
            ),
            id="builtin_attribute_error",
        ),
    ],
)
def test_get_object(path: str, expected: Any) -> None:
    if isinstance(expected, RaisesContext):
        with expected:
            get_object(path)
    else:
        assert get_object(path) == expected
