import { test, describe, assert, afterEach, vi } from "vitest";
import { cleanup, getByTestId, render } from "@self/tootils";
import Audio from "./";
import type { LoadingStatus } from "@gradio/statustracker";
import { setupi18n } from "../core/src/i18n";
import ResizeObserver from "resize-observer-polyfill";

global.ResizeObserver = ResizeObserver;

vi.mock("wavesurfer.js", () => ({
	default: {
		create: vi.fn(() => ({
			load: vi.fn(),
			on: vi.fn(),
			play: vi.fn(),
			pause: vi.fn(),
			destroy: vi.fn(),
			getCurrentTime: vi.fn(() => 0),
			getDuration: vi.fn(() => 0),
			setVolume: vi.fn(),
			seekTo: vi.fn(),
			registerPlugin: vi.fn(() => ({
				on: vi.fn(),
				addRegion: vi.fn(() => ({
					start: 0,
					end: 0,
					play: vi.fn(),
					remove: vi.fn(),
					setOptions: vi.fn()
				})),
				getRegions: vi.fn(() => []),
				clearRegions: vi.fn()
			}))
		}))
	}
}));

const loading_status: LoadingStatus = {
	eta: 0,
	queue_position: 1,
	queue_size: 1,
	status: "complete",
	scroll_to_output: false,
	visible: true,
	fn_index: 0,
	show_progress: "full"
};

const default_values = {
	loading_status,
	label: "music",
	value: {
		url: "https://raw.githubusercontent.com/gradio-app/gradio/main/gradio/test/test_files/audio_sample.wav",
		path: "https://raw.githubusercontent.com/gradio-app/gradio/main/gradio/test/test_files/audio_sample.wav",
		orig_name: "audio_sample.wav"
	},
	show_label: true,
	waveform_options: {
		trim_region_color: "#f97316",
		show_recording_waveform: true,
		show_controls: true
	}
};
describe("Audio", () => {
	setupi18n();

	afterEach(() => cleanup());

	test("renders audio component", async () => {
		const { getAllByTestId } = await render(Audio, {
			...default_values,
			interactive: true,
			sources: ["microphone", "upload"],
			pending: false,
			streaming: false
		});

		assert.exists(getAllByTestId("waveform-music"));
	});

	test("renders audio component with audio controls", async () => {
		const { getAllByTestId, getAllByLabelText, getAllByText } = await render(
			Audio,
			{
				...default_values,
				streaming: false,
				pending: false,
				sources: ["microphone"],
				interactive: true
			}
		);

		assert.exists(getAllByTestId("waveform-controls"));

		assert.exists(getAllByLabelText("Trim audio to selection"));
		assert.exists(getAllByLabelText("Reset audio"));
		assert.exists(getAllByText("0:00"));
		assert.exists(getAllByLabelText("audio.play"));
		assert.exists(getAllByLabelText("Adjust volume"));
		assert.exists(getAllByLabelText("Adjust playback speed to 1.5x"));
		assert.exists(getAllByLabelText("Skip forward by 5 seconds"));
	});

	test("does not render with audio editing controls when not interactive", async () => {
		const { getAllByTestId, queryByLabelText } = await render(Audio, {
			...default_values,
			streaming: false,
			pending: false,
			sources: ["microphone"],
			interactive: false
		});

		assert.exists(getAllByTestId("waveform-controls"));
		assert.notExists(queryByLabelText("Trim audio to selection"));
		assert.notExists(queryByLabelText("Reset audio"));
	});

	test("renders source selection with correct selected source", async () => {
		const { getByTestId, getByLabelText } = await render(Audio, {
			...default_values,
			streaming: false,
			pending: false,
			sources: ["microphone", "upload"],
			interactive: true
		});

		assert.exists(getByTestId("source-select"));
		assert.lengthOf(getByTestId("source-select").children, 2);
		assert.exists(getByLabelText("Record audio"));

		assert.equal(
			getByLabelText("Record audio").classList.contains("selected"),
			true
		);

		assert.equal(
			getByLabelText("Upload file").classList.contains("selected"),
			false
		);
	});

	test("does not render source selection when upload is only source", async () => {
		const { queryByTestId } = await render(Audio, {
			...default_values,
			streaming: false,
			pending: false,
			sources: ["upload"],
			interactive: true
		});

		assert.notExists(queryByTestId("source-select"));
	});

	test("renders audio component with subtitles", async () => {
		const { getByTestId } = await render(Audio, {
			...default_values,
			subtitles: {
				url: "https://raw.githubusercontent.com/gradio-app/gradio/refs/heads/main/demo/video_subtitle/files/s2.vtt",
				path: "https://raw.githubusercontent.com/gradio-app/gradio/refs/heads/main/demo/video_subtitle/files/s2.vtt",
				orig_name: "s2.vtt"
			},
			interactive: false,
			waveform_options: {
				trim_region_color: "#f97316",
				show_recording_waveform: true,
				show_controls: true
			}
		});
		assert.equal(getByTestId("subtitle-display").textContent, "");
	});
});
