package hex.deeplearning;

import hex.DataInfo;
import hex.Model;
import hex.deeplearning.DeepLearningModel.DeepLearningParameters;
import hex.genmodel.utils.DistributionFamily;
import org.apache.commons.lang.math.LongRange;
import org.junit.BeforeClass;
import org.junit.Test;
import water.H2O;
import water.Key;
import water.TestUtil;
import water.fvec.Chunk;
import water.fvec.Frame;
import water.rapids.Rapids;
import water.rapids.Val;
import water.rapids.vals.ValFrame;
import water.util.FrameUtils;

import java.util.Arrays;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class DeepLearningSHAPTest extends TestUtil {
  
  /*
   NOTE: These test do not test all required properties for SHAP. 
   To be more sure after doing some changes to the SHAP, please run the python test:
   h2o-py/tests/testdir_misc/pyunit_SHAP_NOPASS.py 
  */

  @BeforeClass
  public static void setup() {
    stall_till_cloudsize(1);
  }


  @Test
  public void testClassificationCompactSHAP() {
    Frame fr = parseTestFile("smalldata/titanic/titanic_expanded.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "survived";
      params._hidden = ari(5, 5);

      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions().setOutputFormat(Model.Contributions.ContributionsOutputFormat.Compact),
              bgFr);

      assert fr.numCols() >= contribs.numCols();

      Val val = Rapids.exec("(sumaxis " + contribs._key + " 0 1)");
      assertTrue(val instanceof ValFrame);
      res = val.getFrame();
      assertColsEquals(scored, res, 2, 0, 1e-3);
    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }


  @Test
  public void testClassificationOriginalSHAP() {
    Frame fr = parseTestFile("smalldata/titanic/titanic_expanded.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "survived";
      params._hidden = ari(5, 5);

      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions().setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original),
              bgFr);

      assert fr.numCols() < contribs.numCols();  // Titanic has categorical vars

      Val val = Rapids.exec("(sumaxis " + contribs._key + " 0 1)");
      assertTrue(val instanceof ValFrame);
      res = val.getFrame();
      assertColsEquals(scored, res, 2, 0, 1e-3);
    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }


  @Test
  public void testRegressionCompactSHAP() {
    Frame fr = parseTestFile("smalldata/titanic/titanic_expanded.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "fare";
      params._hidden = ari(5, 5);

      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions().setOutputFormat(Model.Contributions.ContributionsOutputFormat.Compact),
              bgFr);

      assert fr.numCols() >= contribs.numCols();

      Val val = Rapids.exec("(sumaxis " + contribs._key + " 0 1)");
      assertTrue(val instanceof ValFrame);
      res = val.getFrame();
      assertColsEquals(scored, res, 0, 0, 1e-3);
    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }

  @Test
  public void testRegressionOriginalSHAP() {
    Frame fr = parseTestFile("smalldata/titanic/titanic_expanded.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "fare";
      params._hidden = ari(5, 5);

      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions().setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original),
              bgFr);

      assert fr.numCols() < contribs.numCols(); // Titanic has categorical vars

      Val val = Rapids.exec("(sumaxis " + contribs._key + " 0 1)");
      assertTrue(val instanceof ValFrame);
      res = val.getFrame();
      assertColsEquals(scored, res, 0, 0, 1e-3);
    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }

  //--------------------------------------------------------------------------------------------------------------------
  // Generated tests to compare with DeepSHAP


  String toPyTorch(DeepLearningModel model, Frame fr, Frame bg) {
    double fraction = 0.01; // how much of the rows should be compared (limited by java file size)

    String[] columns = model._output._names;
    int[] perm = new int[columns.length];
    for (int i = 0; i < columns.length; i++) {
      perm[i] = Arrays.asList(fr._names).indexOf(columns[i]);
    }
    fr.reOrder(perm);
    bg.reOrder(perm);

    //used just for manually generating tests
    StringBuilder sb = new StringBuilder();
    sb.append("import torch\n" + "import shap\n");
    sb.append("nn = torch.nn.Sequential(\n");
    int offset = 0;

    for (int i = 0; i < model._parms._hidden.length; i++) {
      sb.append("    torch.nn.Linear(" + model.model_info().get_weights(i).cols() + ", " + model.model_info().get_weights(i).rows() + "),\n");
      switch (model._parms._activation) {
        case Tanh:
          sb.append("    torch.nn.Tanh(),\n");
          break;
        case Rectifier:
          sb.append("    torch.nn.ReLU(),\n");
          break;
        default:
          H2O.unimpl();
      }
    }
    sb.append("    torch.nn.Linear(" + model.model_info().get_weights(model._parms._hidden.length).cols() +
            ", " + model.model_info().get_weights(model._parms._hidden.length).rows() + "),\n");
    if (model.model_info().data_info._normRespMul != null)
      sb.append("    torch.nn.Linear(1, 1),\n");
    if (model._parms._distribution.equals(DistributionFamily.bernoulli))
      sb.append("    torch.nn.Softmax(),\n");
    sb.append(")\n\n");

    for (int i = 0; i <= model._parms._hidden.length; i++) {
      String weights = "";
      String biases = "[";
      weights += "[";
      for (int k = 0; k < model.model_info().get_weights(i).rows(); k++) {
        weights += "[";
        for (int l = 0; l < model.model_info().get_weights(i).cols(); l++) {
          weights += model.model_info().get_weights(i).get(k, l) + ",";
        }
        weights += "],";
        biases += model.model_info().get_biases(i).get(k) + ", ";
      }
      weights += "]";
      biases += "]";


      sb.append("nn[" + ((2 * i) + offset) + "].weight.data = torch.tensor(" + weights + ", dtype=torch.float32)\n");
      sb.append("nn[" + ((2 * i) + offset) + "].bias.data = torch.tensor(" + biases + ", dtype=torch.float32)\n");
    }

    if (model.model_info().data_info._normRespMul != null) {
      sb.append("nn[" + ((2 * model._parms._hidden.length + 1) + offset) + "].weight.data = torch.tensor([[" + (1. / model.model_info().data_info._normRespMul[0]) + "]], dtype=torch.float32)\n");
      sb.append("nn[" + ((2 * model._parms._hidden.length + 1) + offset) + "].bias.data = torch.tensor([" + (model.model_info().data_info._normRespSub[0]) + "], dtype=torch.float32)\n");
    }
    sb.append("\n");

    sb.append("bg = [\n");
    DataInfo.Row row = model.model_info().data_info().newDenseRow();
    for (int i = 0; i < bg.anyVec().nChunks(); i++) {
      Chunk[] cs = FrameUtils.extractChunks(bg, i, false);
      for (int j = 0; j < cs[0]._len; j++) {
        model.model_info().data_info().extractDenseRow(cs, j, row);
        sb.append("    torch.tensor([[");
        for (int k = 0; k < model.model_info().data_info().fullN(); k++) {
          sb.append(row.get(k) + ", ");
        }
        sb.append("]], dtype=torch.float32),\n");
      }
    }
    sb.append("]\n\n");
    sb.append("x = [\n");
    row = model.model_info().data_info().newDenseRow();
    for (int i = 0; i < fr.anyVec().nChunks(); i++) {
      Chunk[] cs = FrameUtils.extractChunks(fr, i, false);
      for (int j = 0; j < cs[0]._len; j++) {
        model.model_info().data_info().extractDenseRow(cs, j, row);
        sb.append("    torch.tensor([[");
        for (int k = 0; k < model.model_info().data_info().fullN(); k++) {
          sb.append(row.get(k) + ", ");
        }
        sb.append("]], dtype=torch.float32),\n");
      }
    }
    sb.append("]\n\n");

    sb.append("cnt=0\n" +
            "for i in range(len(x)):\n" +
            "    import time\n" +
            "    time.sleep(0.5)  # Sleep so that the output doesn't overwhelm jupyter notebook\n" +
            "    import random\n" +
            "    \n" +
            "    for j in range(len(bg)):\n" +
            "        de = shap.DeepExplainer(nn, bg[j])\n" +
            "        svs = (de.shap_values(x[i]))\n" +
            "        if len(svs) > 1:\n" +
            "            svs = svs[1]  # Binomial\n" +
            "        svs = svs[0]\n" +
            "        for col, sv in enumerate(svs):\n" +
            "            if random.randint(0, 100) <= 1.0:\n" +
            "                print(f\"assertTrue(\\\"Error for column '\\\"+contribs.names()[{col}]+\\\"' for rowIdx=\\\"+contribs.vecs()[contribs.numCols()-2].at({cnt})+\\\"; bgRowIdx=\\\"+contribs.vecs()[contribs.numCols()-1].at({cnt})+\\\"; diff=\\\"+(contribs.vecs()[{col}].at({cnt}) - {sv}),  Math.abs(contribs.vecs()[{col}].at({cnt}) - {sv}) < eps);\")\n" +
            "        cnt+=1");
    return sb.toString();
  }


  @Test
  public void testRegressionReLUDeepSHAPComparison() {
    Frame fr = parseTestFile("smalldata/prostate/prostate.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    double eps = 1e-4;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "AGE";
      params._hidden = ari(3, 3);
      params._seed = 123456;
      params._reproducible = true;
      params._activation = DeepLearningParameters.Activation.Rectifier;


      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions()
                      .setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original)
                      .setOutputPerReference(true),
              bgFr);

      //toPyTorch(model, test, bgFr);

      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(3) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(3) + "; diff=" + (contribs.vecs()[2].at(3) - 1.5493541955947876), Math.abs(contribs.vecs()[2].at(3) - 1.5493541955947876) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(8) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(8) + "; diff=" + (contribs.vecs()[4].at(8) - 0.1030421182513237), Math.abs(contribs.vecs()[4].at(8) - 0.1030421182513237) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(9) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(9) + "; diff=" + (contribs.vecs()[1].at(9) - 0.2955318093299866), Math.abs(contribs.vecs()[1].at(9) - 0.2955318093299866) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(25) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(25) + "; diff=" + (contribs.vecs()[2].at(25) - 0.0), Math.abs(contribs.vecs()[2].at(25) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(32) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(32) + "; diff=" + (contribs.vecs()[5].at(32) - -3.8791611194610596), Math.abs(contribs.vecs()[5].at(32) - -3.8791611194610596) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(41) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(41) + "; diff=" + (contribs.vecs()[1].at(41) - 0.3471778929233551), Math.abs(contribs.vecs()[1].at(41) - 0.3471778929233551) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(51) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(51) + "; diff=" + (contribs.vecs()[7].at(51) - 0.0), Math.abs(contribs.vecs()[7].at(51) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(64) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(64) + "; diff=" + (contribs.vecs()[1].at(64) - -0.18146377801895142), Math.abs(contribs.vecs()[1].at(64) - -0.18146377801895142) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(65) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(65) + "; diff=" + (contribs.vecs()[1].at(65) - 0.0), Math.abs(contribs.vecs()[1].at(65) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(82) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(82) + "; diff=" + (contribs.vecs()[4].at(82) - 0.0), Math.abs(contribs.vecs()[4].at(82) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(87) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(87) + "; diff=" + (contribs.vecs()[6].at(87) - -0.026492716744542122), Math.abs(contribs.vecs()[6].at(87) - -0.026492716744542122) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(89) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(89) + "; diff=" + (contribs.vecs()[0].at(89) - -0.06538061052560806), Math.abs(contribs.vecs()[0].at(89) - -0.06538061052560806) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(99) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(99) + "; diff=" + (contribs.vecs()[0].at(99) - -0.01725296676158905), Math.abs(contribs.vecs()[0].at(99) - -0.01725296676158905) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(106) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(106) + "; diff=" + (contribs.vecs()[3].at(106) - -0.7113931775093079), Math.abs(contribs.vecs()[3].at(106) - -0.7113931775093079) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(113) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(113) + "; diff=" + (contribs.vecs()[5].at(113) - -2.1560144424438477), Math.abs(contribs.vecs()[5].at(113) - -2.1560144424438477) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(116) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(116) + "; diff=" + (contribs.vecs()[3].at(116) - -1.2092221975326538), Math.abs(contribs.vecs()[3].at(116) - -1.2092221975326538) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(122) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(122) + "; diff=" + (contribs.vecs()[6].at(122) - 0.0), Math.abs(contribs.vecs()[6].at(122) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(131) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(131) + "; diff=" + (contribs.vecs()[1].at(131) - 0.25997185707092285), Math.abs(contribs.vecs()[1].at(131) - 0.25997185707092285) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(146) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(146) + "; diff=" + (contribs.vecs()[7].at(146) - -0.6751816272735596), Math.abs(contribs.vecs()[7].at(146) - -0.6751816272735596) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(149) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(149) + "; diff=" + (contribs.vecs()[5].at(149) - -1.7813825607299805), Math.abs(contribs.vecs()[5].at(149) - -1.7813825607299805) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(152) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(152) + "; diff=" + (contribs.vecs()[7].at(152) - 0.49694427847862244), Math.abs(contribs.vecs()[7].at(152) - 0.49694427847862244) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(164) + "; diff=" + (contribs.vecs()[3].at(164) - -0.3191453516483307), Math.abs(contribs.vecs()[3].at(164) - -0.3191453516483307) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(173) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(173) + "; diff=" + (contribs.vecs()[0].at(173) - -0.19784973561763763), Math.abs(contribs.vecs()[0].at(173) - -0.19784973561763763) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(175) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(175) + "; diff=" + (contribs.vecs()[2].at(175) - 0.0), Math.abs(contribs.vecs()[2].at(175) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(181) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(181) + "; diff=" + (contribs.vecs()[2].at(181) - 0.0), Math.abs(contribs.vecs()[2].at(181) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(189) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(189) + "; diff=" + (contribs.vecs()[6].at(189) - 0.48990318179130554), Math.abs(contribs.vecs()[6].at(189) - 0.48990318179130554) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(203) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(203) + "; diff=" + (contribs.vecs()[7].at(203) - 0.31132790446281433), Math.abs(contribs.vecs()[7].at(203) - 0.31132790446281433) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(215) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(215) + "; diff=" + (contribs.vecs()[0].at(215) - -0.16607385873794556), Math.abs(contribs.vecs()[0].at(215) - -0.16607385873794556) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(223) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(223) + "; diff=" + (contribs.vecs()[4].at(223) - -0.16986817121505737), Math.abs(contribs.vecs()[4].at(223) - -0.16986817121505737) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(235) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(235) + "; diff=" + (contribs.vecs()[1].at(235) - -0.08299426734447479), Math.abs(contribs.vecs()[1].at(235) - -0.08299426734447479) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(244) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(244) + "; diff=" + (contribs.vecs()[5].at(244) - -0.00259071565233171), Math.abs(contribs.vecs()[5].at(244) - -0.00259071565233171) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(256) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(256) + "; diff=" + (contribs.vecs()[4].at(256) - -0.44498857855796814), Math.abs(contribs.vecs()[4].at(256) - -0.44498857855796814) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(295) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(295) + "; diff=" + (contribs.vecs()[7].at(295) - 0.0), Math.abs(contribs.vecs()[7].at(295) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(299) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(299) + "; diff=" + (contribs.vecs()[6].at(299) - -0.5201129913330078), Math.abs(contribs.vecs()[6].at(299) - -0.5201129913330078) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(301) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(301) + "; diff=" + (contribs.vecs()[4].at(301) - 0.0), Math.abs(contribs.vecs()[4].at(301) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(305) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(305) + "; diff=" + (contribs.vecs()[2].at(305) - 0.0), Math.abs(contribs.vecs()[2].at(305) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(306) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(306) + "; diff=" + (contribs.vecs()[4].at(306) - 0.0), Math.abs(contribs.vecs()[4].at(306) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(314) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(314) + "; diff=" + (contribs.vecs()[6].at(314) - 0.18156881630420685), Math.abs(contribs.vecs()[6].at(314) - 0.18156881630420685) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(324) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(324) + "; diff=" + (contribs.vecs()[5].at(324) - -0.8602394461631775), Math.abs(contribs.vecs()[5].at(324) - -0.8602394461631775) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(333) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(333) + "; diff=" + (contribs.vecs()[1].at(333) - 0.03129155561327934), Math.abs(contribs.vecs()[1].at(333) - 0.03129155561327934) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(343) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(343) + "; diff=" + (contribs.vecs()[6].at(343) - 0.0), Math.abs(contribs.vecs()[6].at(343) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(368) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(368) + "; diff=" + (contribs.vecs()[6].at(368) - 0.31521573662757874), Math.abs(contribs.vecs()[6].at(368) - 0.31521573662757874) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(369) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(369) + "; diff=" + (contribs.vecs()[3].at(369) - 0.7021065354347229), Math.abs(contribs.vecs()[3].at(369) - 0.7021065354347229) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(370) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(370) + "; diff=" + (contribs.vecs()[0].at(370) - -0.43444523215293884), Math.abs(contribs.vecs()[0].at(370) - -0.43444523215293884) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(371) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(371) + "; diff=" + (contribs.vecs()[6].at(371) - 0.7376115918159485), Math.abs(contribs.vecs()[6].at(371) - 0.7376115918159485) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(372) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(372) + "; diff=" + (contribs.vecs()[6].at(372) - 0.7834213972091675), Math.abs(contribs.vecs()[6].at(372) - 0.7834213972091675) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(380) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(380) + "; diff=" + (contribs.vecs()[3].at(380) - -0.3175189793109894), Math.abs(contribs.vecs()[3].at(380) - -0.3175189793109894) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(385) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(385) + "; diff=" + (contribs.vecs()[2].at(385) - 0.0), Math.abs(contribs.vecs()[2].at(385) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(403) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(403) + "; diff=" + (contribs.vecs()[6].at(403) - 0.4235844314098358), Math.abs(contribs.vecs()[6].at(403) - 0.4235844314098358) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(404) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(404) + "; diff=" + (contribs.vecs()[3].at(404) - -0.23187653720378876), Math.abs(contribs.vecs()[3].at(404) - -0.23187653720378876) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(405) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(405) + "; diff=" + (contribs.vecs()[7].at(405) - 0.4493134319782257), Math.abs(contribs.vecs()[7].at(405) - 0.4493134319782257) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(410) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(410) + "; diff=" + (contribs.vecs()[3].at(410) - 0.263580322265625), Math.abs(contribs.vecs()[3].at(410) - 0.263580322265625) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(427) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(427) + "; diff=" + (contribs.vecs()[5].at(427) - 0.1052236258983612), Math.abs(contribs.vecs()[5].at(427) - 0.1052236258983612) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(431) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(431) + "; diff=" + (contribs.vecs()[4].at(431) - 0.0), Math.abs(contribs.vecs()[4].at(431) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(438) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(438) + "; diff=" + (contribs.vecs()[0].at(438) - -0.14045442640781403), Math.abs(contribs.vecs()[0].at(438) - -0.14045442640781403) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(440) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(440) + "; diff=" + (contribs.vecs()[3].at(440) - 0.5502122640609741), Math.abs(contribs.vecs()[3].at(440) - 0.5502122640609741) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(448) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(448) + "; diff=" + (contribs.vecs()[1].at(448) - 0.08299426734447479), Math.abs(contribs.vecs()[1].at(448) - 0.08299426734447479) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(451) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(451) + "; diff=" + (contribs.vecs()[0].at(451) - -0.07680611312389374), Math.abs(contribs.vecs()[0].at(451) - -0.07680611312389374) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(459) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(459) + "; diff=" + (contribs.vecs()[3].at(459) - 0.0), Math.abs(contribs.vecs()[3].at(459) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(460) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(460) + "; diff=" + (contribs.vecs()[1].at(460) - 0.0), Math.abs(contribs.vecs()[1].at(460) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(464) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(464) + "; diff=" + (contribs.vecs()[6].at(464) - -2.696352481842041), Math.abs(contribs.vecs()[6].at(464) - -2.696352481842041) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(471) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(471) + "; diff=" + (contribs.vecs()[0].at(471) - -0.23465709388256073), Math.abs(contribs.vecs()[0].at(471) - -0.23465709388256073) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(479) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(479) + "; diff=" + (contribs.vecs()[7].at(479) - 0.6177017688751221), Math.abs(contribs.vecs()[7].at(479) - 0.6177017688751221) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(493) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(493) + "; diff=" + (contribs.vecs()[1].at(493) - 0.0), Math.abs(contribs.vecs()[1].at(493) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(493) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(493) + "; diff=" + (contribs.vecs()[2].at(493) - 0.0), Math.abs(contribs.vecs()[2].at(493) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(493) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(493) + "; diff=" + (contribs.vecs()[4].at(493) - 0.0), Math.abs(contribs.vecs()[4].at(493) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(504) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(504) + "; diff=" + (contribs.vecs()[7].at(504) - 1.2676347494125366), Math.abs(contribs.vecs()[7].at(504) - 1.2676347494125366) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(516) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(516) + "; diff=" + (contribs.vecs()[0].at(516) - -0.4666285812854767), Math.abs(contribs.vecs()[0].at(516) - -0.4666285812854767) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(517) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(517) + "; diff=" + (contribs.vecs()[1].at(517) - 0.1791084110736847), Math.abs(contribs.vecs()[1].at(517) - 0.1791084110736847) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(522) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(522) + "; diff=" + (contribs.vecs()[1].at(522) - 0.0), Math.abs(contribs.vecs()[1].at(522) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(524) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(524) + "; diff=" + (contribs.vecs()[6].at(524) - 0.23237177729606628), Math.abs(contribs.vecs()[6].at(524) - 0.23237177729606628) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(527) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(527) + "; diff=" + (contribs.vecs()[6].at(527) - 0.17616993188858032), Math.abs(contribs.vecs()[6].at(527) - 0.17616993188858032) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(528) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(528) + "; diff=" + (contribs.vecs()[4].at(528) - -0.2728838324546814), Math.abs(contribs.vecs()[4].at(528) - -0.2728838324546814) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(534) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(534) + "; diff=" + (contribs.vecs()[7].at(534) - 0.48364531993865967), Math.abs(contribs.vecs()[7].at(534) - 0.48364531993865967) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(537) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(537) + "; diff=" + (contribs.vecs()[5].at(537) - -0.06302861869335175), Math.abs(contribs.vecs()[5].at(537) - -0.06302861869335175) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(537) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(537) + "; diff=" + (contribs.vecs()[7].at(537) - 0.03775444254279137), Math.abs(contribs.vecs()[7].at(537) - 0.03775444254279137) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(560) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(560) + "; diff=" + (contribs.vecs()[1].at(560) - -0.11066688597202301), Math.abs(contribs.vecs()[1].at(560) - -0.11066688597202301) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(567) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(567) + "; diff=" + (contribs.vecs()[7].at(567) - 0.0), Math.abs(contribs.vecs()[7].at(567) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(569) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(569) + "; diff=" + (contribs.vecs()[7].at(569) - 0.0), Math.abs(contribs.vecs()[7].at(569) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(581) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(581) + "; diff=" + (contribs.vecs()[4].at(581) - 0.0), Math.abs(contribs.vecs()[4].at(581) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(586) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(586) + "; diff=" + (contribs.vecs()[7].at(586) - 0.5474206209182739), Math.abs(contribs.vecs()[7].at(586) - 0.5474206209182739) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(590) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(590) + "; diff=" + (contribs.vecs()[4].at(590) - 0.0), Math.abs(contribs.vecs()[4].at(590) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(592) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(592) + "; diff=" + (contribs.vecs()[3].at(592) - 0.0), Math.abs(contribs.vecs()[3].at(592) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(602) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(602) + "; diff=" + (contribs.vecs()[1].at(602) - 0.0), Math.abs(contribs.vecs()[1].at(602) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(633) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(633) + "; diff=" + (contribs.vecs()[7].at(633) - 0.0), Math.abs(contribs.vecs()[7].at(633) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(636) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(636) + "; diff=" + (contribs.vecs()[1].at(636) - -0.07549892365932465), Math.abs(contribs.vecs()[1].at(636) - -0.07549892365932465) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(642) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(642) + "; diff=" + (contribs.vecs()[3].at(642) - 0.0), Math.abs(contribs.vecs()[3].at(642) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(645) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(645) + "; diff=" + (contribs.vecs()[2].at(645) - 0.0), Math.abs(contribs.vecs()[2].at(645) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(654) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(654) + "; diff=" + (contribs.vecs()[0].at(654) - -0.14884933829307556), Math.abs(contribs.vecs()[0].at(654) - -0.14884933829307556) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(661) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(661) + "; diff=" + (contribs.vecs()[7].at(661) - 0.049600061029195786), Math.abs(contribs.vecs()[7].at(661) - 0.049600061029195786) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(672) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(672) + "; diff=" + (contribs.vecs()[0].at(672) - -0.528071939945221), Math.abs(contribs.vecs()[0].at(672) - -0.528071939945221) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(679) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(679) + "; diff=" + (contribs.vecs()[0].at(679) - -0.32248973846435547), Math.abs(contribs.vecs()[0].at(679) - -0.32248973846435547) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(679) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(679) + "; diff=" + (contribs.vecs()[4].at(679) - 0.0), Math.abs(contribs.vecs()[4].at(679) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(680) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(680) + "; diff=" + (contribs.vecs()[0].at(680) - -0.2886812090873718), Math.abs(contribs.vecs()[0].at(680) - -0.2886812090873718) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(685) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(685) + "; diff=" + (contribs.vecs()[4].at(685) - 0.0), Math.abs(contribs.vecs()[4].at(685) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(693) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(693) + "; diff=" + (contribs.vecs()[5].at(693) - -0.8544059991836548), Math.abs(contribs.vecs()[5].at(693) - -0.8544059991836548) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(716) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(716) + "; diff=" + (contribs.vecs()[5].at(716) - -0.16469554603099823), Math.abs(contribs.vecs()[5].at(716) - -0.16469554603099823) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(720) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(720) + "; diff=" + (contribs.vecs()[2].at(720) - 0.9636378884315491), Math.abs(contribs.vecs()[2].at(720) - 0.9636378884315491) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(722) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(722) + "; diff=" + (contribs.vecs()[4].at(722) - 0.0), Math.abs(contribs.vecs()[4].at(722) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(730) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(730) + "; diff=" + (contribs.vecs()[7].at(730) - 1.842221975326538), Math.abs(contribs.vecs()[7].at(730) - 1.842221975326538) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(747) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(747) + "; diff=" + (contribs.vecs()[2].at(747) - 0.0), Math.abs(contribs.vecs()[2].at(747) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(764) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(764) + "; diff=" + (contribs.vecs()[3].at(764) - 0.038559362292289734), Math.abs(contribs.vecs()[3].at(764) - 0.038559362292289734) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(764) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(764) + "; diff=" + (contribs.vecs()[7].at(764) - 0.03531688451766968), Math.abs(contribs.vecs()[7].at(764) - 0.03531688451766968) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(774) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(774) + "; diff=" + (contribs.vecs()[7].at(774) - 0.0), Math.abs(contribs.vecs()[7].at(774) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(776) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(776) + "; diff=" + (contribs.vecs()[6].at(776) - 0.0), Math.abs(contribs.vecs()[6].at(776) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(783) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(783) + "; diff=" + (contribs.vecs()[3].at(783) - 0.013799285516142845), Math.abs(contribs.vecs()[3].at(783) - 0.013799285516142845) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(787) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(787) + "; diff=" + (contribs.vecs()[3].at(787) - 0.2147972285747528), Math.abs(contribs.vecs()[3].at(787) - 0.2147972285747528) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(792) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(792) + "; diff=" + (contribs.vecs()[4].at(792) - 0.0), Math.abs(contribs.vecs()[4].at(792) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(797) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(797) + "; diff=" + (contribs.vecs()[2].at(797) - 0.0), Math.abs(contribs.vecs()[2].at(797) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(826) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(826) + "; diff=" + (contribs.vecs()[1].at(826) - -0.1592215746641159), Math.abs(contribs.vecs()[1].at(826) - -0.1592215746641159) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(828) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(828) + "; diff=" + (contribs.vecs()[5].at(828) - 0.06171402707695961), Math.abs(contribs.vecs()[5].at(828) - 0.06171402707695961) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(829) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(829) + "; diff=" + (contribs.vecs()[6].at(829) - 0.0), Math.abs(contribs.vecs()[6].at(829) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(833) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(833) + "; diff=" + (contribs.vecs()[1].at(833) - 0.0), Math.abs(contribs.vecs()[1].at(833) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(836) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(836) + "; diff=" + (contribs.vecs()[0].at(836) - -0.044895239174366), Math.abs(contribs.vecs()[0].at(836) - -0.044895239174366) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(838) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(838) + "; diff=" + (contribs.vecs()[4].at(838) - 0.0), Math.abs(contribs.vecs()[4].at(838) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(842) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(842) + "; diff=" + (contribs.vecs()[5].at(842) - -0.0137413889169693), Math.abs(contribs.vecs()[5].at(842) - -0.0137413889169693) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(864) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(864) + "; diff=" + (contribs.vecs()[3].at(864) - -0.42542293667793274), Math.abs(contribs.vecs()[3].at(864) - -0.42542293667793274) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(865) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(865) + "; diff=" + (contribs.vecs()[7].at(865) - 0.4001626968383789), Math.abs(contribs.vecs()[7].at(865) - 0.4001626968383789) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(874) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(874) + "; diff=" + (contribs.vecs()[0].at(874) - -0.5244303345680237), Math.abs(contribs.vecs()[0].at(874) - -0.5244303345680237) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(878) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(878) + "; diff=" + (contribs.vecs()[0].at(878) - -0.25493234395980835), Math.abs(contribs.vecs()[0].at(878) - -0.25493234395980835) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(880) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(880) + "; diff=" + (contribs.vecs()[5].at(880) - 2.600280284881592), Math.abs(contribs.vecs()[5].at(880) - 2.600280284881592) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(894) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(894) + "; diff=" + (contribs.vecs()[6].at(894) - -0.49846571683883667), Math.abs(contribs.vecs()[6].at(894) - -0.49846571683883667) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(896) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(896) + "; diff=" + (contribs.vecs()[0].at(896) - -0.31608787178993225), Math.abs(contribs.vecs()[0].at(896) - -0.31608787178993225) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(908) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(908) + "; diff=" + (contribs.vecs()[4].at(908) - 0.0), Math.abs(contribs.vecs()[4].at(908) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(921) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(921) + "; diff=" + (contribs.vecs()[7].at(921) - -0.15748555958271027), Math.abs(contribs.vecs()[7].at(921) - -0.15748555958271027) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(928) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(928) + "; diff=" + (contribs.vecs()[2].at(928) - 0.0), Math.abs(contribs.vecs()[2].at(928) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(937) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(937) + "; diff=" + (contribs.vecs()[2].at(937) - 0.0), Math.abs(contribs.vecs()[2].at(937) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(942) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(942) + "; diff=" + (contribs.vecs()[6].at(942) - 0.18452845513820648), Math.abs(contribs.vecs()[6].at(942) - 0.18452845513820648) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(963) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(963) + "; diff=" + (contribs.vecs()[1].at(963) - 0.0), Math.abs(contribs.vecs()[1].at(963) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(968) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(968) + "; diff=" + (contribs.vecs()[0].at(968) - -0.14786988496780396), Math.abs(contribs.vecs()[0].at(968) - -0.14786988496780396) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(970) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(970) + "; diff=" + (contribs.vecs()[2].at(970) - 0.0), Math.abs(contribs.vecs()[2].at(970) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(972) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(972) + "; diff=" + (contribs.vecs()[4].at(972) - 0.0), Math.abs(contribs.vecs()[4].at(972) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(974) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(974) + "; diff=" + (contribs.vecs()[2].at(974) - 0.0), Math.abs(contribs.vecs()[2].at(974) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(984) + "; diff=" + (contribs.vecs()[1].at(984) - 0.0), Math.abs(contribs.vecs()[1].at(984) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(986) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(986) + "; diff=" + (contribs.vecs()[2].at(986) - 0.0), Math.abs(contribs.vecs()[2].at(986) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(990) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(990) + "; diff=" + (contribs.vecs()[0].at(990) - -0.2389090359210968), Math.abs(contribs.vecs()[0].at(990) - -0.2389090359210968) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(995) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(995) + "; diff=" + (contribs.vecs()[0].at(995) - -0.20861928164958954), Math.abs(contribs.vecs()[0].at(995) - -0.20861928164958954) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(996) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(996) + "; diff=" + (contribs.vecs()[4].at(996) - 0.0), Math.abs(contribs.vecs()[4].at(996) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1004) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1004) + "; diff=" + (contribs.vecs()[1].at(1004) - 0.0), Math.abs(contribs.vecs()[1].at(1004) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1013) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1013) + "; diff=" + (contribs.vecs()[4].at(1013) - 0.0), Math.abs(contribs.vecs()[4].at(1013) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1025) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1025) + "; diff=" + (contribs.vecs()[3].at(1025) - 0.0), Math.abs(contribs.vecs()[3].at(1025) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1029) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1029) + "; diff=" + (contribs.vecs()[3].at(1029) - -1.3129854202270508), Math.abs(contribs.vecs()[3].at(1029) - -1.3129854202270508) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1034) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1034) + "; diff=" + (contribs.vecs()[1].at(1034) - 0.0), Math.abs(contribs.vecs()[1].at(1034) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1041) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1041) + "; diff=" + (contribs.vecs()[2].at(1041) - -1.4191868305206299), Math.abs(contribs.vecs()[2].at(1041) - -1.4191868305206299) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1042) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1042) + "; diff=" + (contribs.vecs()[6].at(1042) - 0.01486214529722929), Math.abs(contribs.vecs()[6].at(1042) - 0.01486214529722929) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1044) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1044) + "; diff=" + (contribs.vecs()[1].at(1044) - -0.16042625904083252), Math.abs(contribs.vecs()[1].at(1044) - -0.16042625904083252) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1049) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1049) + "; diff=" + (contribs.vecs()[1].at(1049) - 0.0), Math.abs(contribs.vecs()[1].at(1049) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1050) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1050) + "; diff=" + (contribs.vecs()[1].at(1050) - -0.22404959797859192), Math.abs(contribs.vecs()[1].at(1050) - -0.22404959797859192) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1052) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1052) + "; diff=" + (contribs.vecs()[6].at(1052) - 0.30162322521209717), Math.abs(contribs.vecs()[6].at(1052) - 0.30162322521209717) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1056) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1056) + "; diff=" + (contribs.vecs()[0].at(1056) - -0.20182056725025177), Math.abs(contribs.vecs()[0].at(1056) - -0.20182056725025177) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1062) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1062) + "; diff=" + (contribs.vecs()[6].at(1062) - 1.0653584003448486), Math.abs(contribs.vecs()[6].at(1062) - 1.0653584003448486) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1072) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1072) + "; diff=" + (contribs.vecs()[2].at(1072) - 0.0), Math.abs(contribs.vecs()[2].at(1072) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1078) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1078) + "; diff=" + (contribs.vecs()[3].at(1078) - 0.7021065354347229), Math.abs(contribs.vecs()[3].at(1078) - 0.7021065354347229) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1079) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1079) + "; diff=" + (contribs.vecs()[3].at(1079) - -0.4610143303871155), Math.abs(contribs.vecs()[3].at(1079) - -0.4610143303871155) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1084) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1084) + "; diff=" + (contribs.vecs()[7].at(1084) - 0.0), Math.abs(contribs.vecs()[7].at(1084) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1092) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1092) + "; diff=" + (contribs.vecs()[3].at(1092) - 0.593103289604187), Math.abs(contribs.vecs()[3].at(1092) - 0.593103289604187) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1101) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1101) + "; diff=" + (contribs.vecs()[4].at(1101) - 0.0), Math.abs(contribs.vecs()[4].at(1101) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1102) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1102) + "; diff=" + (contribs.vecs()[5].at(1102) - 0.03232740983366966), Math.abs(contribs.vecs()[5].at(1102) - 0.03232740983366966) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1103) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1103) + "; diff=" + (contribs.vecs()[1].at(1103) - 0.0), Math.abs(contribs.vecs()[1].at(1103) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1111) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1111) + "; diff=" + (contribs.vecs()[6].at(1111) - 0.15899430215358734), Math.abs(contribs.vecs()[6].at(1111) - 0.15899430215358734) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1112) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1112) + "; diff=" + (contribs.vecs()[7].at(1112) - 0.381452351808548), Math.abs(contribs.vecs()[7].at(1112) - 0.381452351808548) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1114) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1114) + "; diff=" + (contribs.vecs()[1].at(1114) - 0.0), Math.abs(contribs.vecs()[1].at(1114) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1119) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1119) + "; diff=" + (contribs.vecs()[5].at(1119) - -0.14440135657787323), Math.abs(contribs.vecs()[5].at(1119) - -0.14440135657787323) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1128) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1128) + "; diff=" + (contribs.vecs()[1].at(1128) - 0.0), Math.abs(contribs.vecs()[1].at(1128) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1130) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1130) + "; diff=" + (contribs.vecs()[4].at(1130) - 0.0), Math.abs(contribs.vecs()[4].at(1130) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1131) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1131) + "; diff=" + (contribs.vecs()[5].at(1131) - 0.2331714779138565), Math.abs(contribs.vecs()[5].at(1131) - 0.2331714779138565) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1135) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1135) + "; diff=" + (contribs.vecs()[6].at(1135) - 0.36708810925483704), Math.abs(contribs.vecs()[6].at(1135) - 0.36708810925483704) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1138) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1138) + "; diff=" + (contribs.vecs()[6].at(1138) - 0.9830343723297119), Math.abs(contribs.vecs()[6].at(1138) - 0.9830343723297119) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1151) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1151) + "; diff=" + (contribs.vecs()[5].at(1151) - -0.05054476857185364), Math.abs(contribs.vecs()[5].at(1151) - -0.05054476857185364) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1156) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1156) + "; diff=" + (contribs.vecs()[0].at(1156) - -0.3296390771865845), Math.abs(contribs.vecs()[0].at(1156) - -0.3296390771865845) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1164) + "; diff=" + (contribs.vecs()[5].at(1164) - 0.03386038914322853), Math.abs(contribs.vecs()[5].at(1164) - 0.03386038914322853) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1177) + "; diff=" + (contribs.vecs()[0].at(1177) - -0.24312438070774078), Math.abs(contribs.vecs()[0].at(1177) - -0.24312438070774078) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1177) + "; diff=" + (contribs.vecs()[4].at(1177) - 0.0), Math.abs(contribs.vecs()[4].at(1177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1180) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1180) + "; diff=" + (contribs.vecs()[3].at(1180) - 1.707529067993164), Math.abs(contribs.vecs()[3].at(1180) - 1.707529067993164) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1193) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1193) + "; diff=" + (contribs.vecs()[1].at(1193) - 0.2128257304430008), Math.abs(contribs.vecs()[1].at(1193) - 0.2128257304430008) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1197) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1197) + "; diff=" + (contribs.vecs()[6].at(1197) - 1.3132168054580688), Math.abs(contribs.vecs()[6].at(1197) - 1.3132168054580688) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1200) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1200) + "; diff=" + (contribs.vecs()[4].at(1200) - 0.0), Math.abs(contribs.vecs()[4].at(1200) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1201) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1201) + "; diff=" + (contribs.vecs()[6].at(1201) - 0.14338259398937225), Math.abs(contribs.vecs()[6].at(1201) - 0.14338259398937225) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1213) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1213) + "; diff=" + (contribs.vecs()[2].at(1213) - 0.0), Math.abs(contribs.vecs()[2].at(1213) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1215) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1215) + "; diff=" + (contribs.vecs()[1].at(1215) - -0.1319882720708847), Math.abs(contribs.vecs()[1].at(1215) - -0.1319882720708847) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1222) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1222) + "; diff=" + (contribs.vecs()[4].at(1222) - 0.0), Math.abs(contribs.vecs()[4].at(1222) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1224) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1224) + "; diff=" + (contribs.vecs()[0].at(1224) - -0.43247270584106445), Math.abs(contribs.vecs()[0].at(1224) - -0.43247270584106445) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1235) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1235) + "; diff=" + (contribs.vecs()[5].at(1235) - -3.0125255584716797), Math.abs(contribs.vecs()[5].at(1235) - -3.0125255584716797) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1257) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1257) + "; diff=" + (contribs.vecs()[7].at(1257) - 0.0), Math.abs(contribs.vecs()[7].at(1257) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1267) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1267) + "; diff=" + (contribs.vecs()[0].at(1267) - -0.18084637820720673), Math.abs(contribs.vecs()[0].at(1267) - -0.18084637820720673) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1270) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1270) + "; diff=" + (contribs.vecs()[2].at(1270) - 0.0), Math.abs(contribs.vecs()[2].at(1270) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1273) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1273) + "; diff=" + (contribs.vecs()[1].at(1273) - 0.0), Math.abs(contribs.vecs()[1].at(1273) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1295) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1295) + "; diff=" + (contribs.vecs()[3].at(1295) - -0.16692787408828735), Math.abs(contribs.vecs()[3].at(1295) - -0.16692787408828735) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1297) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1297) + "; diff=" + (contribs.vecs()[0].at(1297) - -0.3296566307544708), Math.abs(contribs.vecs()[0].at(1297) - -0.3296566307544708) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1299) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1299) + "; diff=" + (contribs.vecs()[2].at(1299) - 0.0), Math.abs(contribs.vecs()[2].at(1299) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1302) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1302) + "; diff=" + (contribs.vecs()[0].at(1302) - -0.4366879165172577), Math.abs(contribs.vecs()[0].at(1302) - -0.4366879165172577) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1309) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1309) + "; diff=" + (contribs.vecs()[0].at(1309) - -0.3352934718132019), Math.abs(contribs.vecs()[0].at(1309) - -0.3352934718132019) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1314) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1314) + "; diff=" + (contribs.vecs()[3].at(1314) - 0.0), Math.abs(contribs.vecs()[3].at(1314) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1331) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1331) + "; diff=" + (contribs.vecs()[1].at(1331) - 0.0), Math.abs(contribs.vecs()[1].at(1331) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1344) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1344) + "; diff=" + (contribs.vecs()[0].at(1344) - -0.07055582106113434), Math.abs(contribs.vecs()[0].at(1344) - -0.07055582106113434) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1353) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1353) + "; diff=" + (contribs.vecs()[3].at(1353) - 0.24225008487701416), Math.abs(contribs.vecs()[3].at(1353) - 0.24225008487701416) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1355) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1355) + "; diff=" + (contribs.vecs()[5].at(1355) - 0.17685917019844055), Math.abs(contribs.vecs()[5].at(1355) - 0.17685917019844055) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1358) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1358) + "; diff=" + (contribs.vecs()[0].at(1358) - -0.3747975826263428), Math.abs(contribs.vecs()[0].at(1358) - -0.3747975826263428) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1383) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1383) + "; diff=" + (contribs.vecs()[4].at(1383) - -0.459643691778183), Math.abs(contribs.vecs()[4].at(1383) - -0.459643691778183) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1387) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1387) + "; diff=" + (contribs.vecs()[5].at(1387) - -0.17861102521419525), Math.abs(contribs.vecs()[5].at(1387) - -0.17861102521419525) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1391) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1391) + "; diff=" + (contribs.vecs()[4].at(1391) - 0.0), Math.abs(contribs.vecs()[4].at(1391) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1392) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1392) + "; diff=" + (contribs.vecs()[1].at(1392) - 0.0), Math.abs(contribs.vecs()[1].at(1392) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1392) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1392) + "; diff=" + (contribs.vecs()[5].at(1392) - -0.23352868854999542), Math.abs(contribs.vecs()[5].at(1392) - -0.23352868854999542) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1393) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1393) + "; diff=" + (contribs.vecs()[3].at(1393) - -0.24374011158943176), Math.abs(contribs.vecs()[3].at(1393) - -0.24374011158943176) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1396) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1396) + "; diff=" + (contribs.vecs()[5].at(1396) - 0.03068203292787075), Math.abs(contribs.vecs()[5].at(1396) - 0.03068203292787075) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1398) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1398) + "; diff=" + (contribs.vecs()[7].at(1398) - -1.0593324899673462), Math.abs(contribs.vecs()[7].at(1398) - -1.0593324899673462) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1401) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1401) + "; diff=" + (contribs.vecs()[7].at(1401) - -0.18291574716567993), Math.abs(contribs.vecs()[7].at(1401) - -0.18291574716567993) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1407) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1407) + "; diff=" + (contribs.vecs()[0].at(1407) - -0.2917780876159668), Math.abs(contribs.vecs()[0].at(1407) - -0.2917780876159668) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1409) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1409) + "; diff=" + (contribs.vecs()[1].at(1409) - 0.0), Math.abs(contribs.vecs()[1].at(1409) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1415) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1415) + "; diff=" + (contribs.vecs()[4].at(1415) - 0.0), Math.abs(contribs.vecs()[4].at(1415) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1415) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1415) + "; diff=" + (contribs.vecs()[6].at(1415) - -0.8198428750038147), Math.abs(contribs.vecs()[6].at(1415) - -0.8198428750038147) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1422) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1422) + "; diff=" + (contribs.vecs()[6].at(1422) - -0.26865124702453613), Math.abs(contribs.vecs()[6].at(1422) - -0.26865124702453613) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1423) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1423) + "; diff=" + (contribs.vecs()[7].at(1423) - -0.19894379377365112), Math.abs(contribs.vecs()[7].at(1423) - -0.19894379377365112) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1429) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1429) + "; diff=" + (contribs.vecs()[7].at(1429) - -0.6879506707191467), Math.abs(contribs.vecs()[7].at(1429) - -0.6879506707191467) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1430) + "; diff=" + (contribs.vecs()[7].at(1430) - -0.34222960472106934), Math.abs(contribs.vecs()[7].at(1430) - -0.34222960472106934) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1431) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1431) + "; diff=" + (contribs.vecs()[3].at(1431) - 0.3927164077758789), Math.abs(contribs.vecs()[3].at(1431) - 0.3927164077758789) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1436) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1436) + "; diff=" + (contribs.vecs()[1].at(1436) - -0.08299426734447479), Math.abs(contribs.vecs()[1].at(1436) - -0.08299426734447479) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1451) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1451) + "; diff=" + (contribs.vecs()[7].at(1451) - -0.7135322690010071), Math.abs(contribs.vecs()[7].at(1451) - -0.7135322690010071) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1455) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1455) + "; diff=" + (contribs.vecs()[3].at(1455) - 0.3649737536907196), Math.abs(contribs.vecs()[3].at(1455) - 0.3649737536907196) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1464) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1464) + "; diff=" + (contribs.vecs()[3].at(1464) - 0.01874369941651821), Math.abs(contribs.vecs()[3].at(1464) - 0.01874369941651821) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1472) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1472) + "; diff=" + (contribs.vecs()[3].at(1472) - 0.8192729949951172), Math.abs(contribs.vecs()[3].at(1472) - 0.8192729949951172) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1476) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1476) + "; diff=" + (contribs.vecs()[4].at(1476) - 0.0), Math.abs(contribs.vecs()[4].at(1476) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1479) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1479) + "; diff=" + (contribs.vecs()[3].at(1479) - -0.15229633450508118), Math.abs(contribs.vecs()[3].at(1479) - -0.15229633450508118) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1479) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1479) + "; diff=" + (contribs.vecs()[6].at(1479) - 0.0), Math.abs(contribs.vecs()[6].at(1479) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1487) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1487) + "; diff=" + (contribs.vecs()[4].at(1487) - 0.0), Math.abs(contribs.vecs()[4].at(1487) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1490) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1490) + "; diff=" + (contribs.vecs()[4].at(1490) - -0.7108476758003235), Math.abs(contribs.vecs()[4].at(1490) - -0.7108476758003235) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1493) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1493) + "; diff=" + (contribs.vecs()[3].at(1493) - -0.850708544254303), Math.abs(contribs.vecs()[3].at(1493) - -0.850708544254303) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1496) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1496) + "; diff=" + (contribs.vecs()[5].at(1496) - -0.1655459851026535), Math.abs(contribs.vecs()[5].at(1496) - -0.1655459851026535) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1536) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1536) + "; diff=" + (contribs.vecs()[4].at(1536) - -0.4172687232494354), Math.abs(contribs.vecs()[4].at(1536) - -0.4172687232494354) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1538) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1538) + "; diff=" + (contribs.vecs()[4].at(1538) - 0.0), Math.abs(contribs.vecs()[4].at(1538) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1540) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1540) + "; diff=" + (contribs.vecs()[5].at(1540) - -0.7688691020011902), Math.abs(contribs.vecs()[5].at(1540) - -0.7688691020011902) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1548) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1548) + "; diff=" + (contribs.vecs()[7].at(1548) - 0.25193867087364197), Math.abs(contribs.vecs()[7].at(1548) - 0.25193867087364197) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1565) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1565) + "; diff=" + (contribs.vecs()[2].at(1565) - 0.0), Math.abs(contribs.vecs()[2].at(1565) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1573) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1573) + "; diff=" + (contribs.vecs()[6].at(1573) - 0.0), Math.abs(contribs.vecs()[6].at(1573) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1574) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1574) + "; diff=" + (contribs.vecs()[1].at(1574) - -0.24510827660560608), Math.abs(contribs.vecs()[1].at(1574) - -0.24510827660560608) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1583) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1583) + "; diff=" + (contribs.vecs()[6].at(1583) - 0.0), Math.abs(contribs.vecs()[6].at(1583) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1585) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1585) + "; diff=" + (contribs.vecs()[4].at(1585) - 0.0), Math.abs(contribs.vecs()[4].at(1585) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1591) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1591) + "; diff=" + (contribs.vecs()[3].at(1591) - 1.244676113128662), Math.abs(contribs.vecs()[3].at(1591) - 1.244676113128662) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1601) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1601) + "; diff=" + (contribs.vecs()[1].at(1601) - -0.06115712597966194), Math.abs(contribs.vecs()[1].at(1601) - -0.06115712597966194) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1605) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1605) + "; diff=" + (contribs.vecs()[7].at(1605) - 0.0), Math.abs(contribs.vecs()[7].at(1605) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1609) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1609) + "; diff=" + (contribs.vecs()[6].at(1609) - -1.2588329315185547), Math.abs(contribs.vecs()[6].at(1609) - -1.2588329315185547) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1612) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1612) + "; diff=" + (contribs.vecs()[0].at(1612) - -0.2908022999763489), Math.abs(contribs.vecs()[0].at(1612) - -0.2908022999763489) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1639) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1639) + "; diff=" + (contribs.vecs()[2].at(1639) - 0.8305550217628479), Math.abs(contribs.vecs()[2].at(1639) - 0.8305550217628479) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1649) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1649) + "; diff=" + (contribs.vecs()[4].at(1649) - 0.0), Math.abs(contribs.vecs()[4].at(1649) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1649) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1649) + "; diff=" + (contribs.vecs()[6].at(1649) - -0.08057282865047455), Math.abs(contribs.vecs()[6].at(1649) - -0.08057282865047455) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1650) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1650) + "; diff=" + (contribs.vecs()[1].at(1650) - 0.0), Math.abs(contribs.vecs()[1].at(1650) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1660) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1660) + "; diff=" + (contribs.vecs()[5].at(1660) - 0.024642355740070343), Math.abs(contribs.vecs()[5].at(1660) - 0.024642355740070343) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1661) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1661) + "; diff=" + (contribs.vecs()[2].at(1661) - 0.0), Math.abs(contribs.vecs()[2].at(1661) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1671) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1671) + "; diff=" + (contribs.vecs()[0].at(1671) - -0.21310321986675262), Math.abs(contribs.vecs()[0].at(1671) - -0.21310321986675262) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1671) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1671) + "; diff=" + (contribs.vecs()[4].at(1671) - 0.0), Math.abs(contribs.vecs()[4].at(1671) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1673) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1673) + "; diff=" + (contribs.vecs()[3].at(1673) - 0.04377976059913635), Math.abs(contribs.vecs()[3].at(1673) - 0.04377976059913635) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1697) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1697) + "; diff=" + (contribs.vecs()[7].at(1697) - 0.3191958963871002), Math.abs(contribs.vecs()[7].at(1697) - 0.3191958963871002) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1715) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1715) + "; diff=" + (contribs.vecs()[1].at(1715) - 0.0), Math.abs(contribs.vecs()[1].at(1715) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1719) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1719) + "; diff=" + (contribs.vecs()[5].at(1719) - -0.23234859108924866), Math.abs(contribs.vecs()[5].at(1719) - -0.23234859108924866) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1723) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1723) + "; diff=" + (contribs.vecs()[5].at(1723) - 0.010362856090068817), Math.abs(contribs.vecs()[5].at(1723) - 0.010362856090068817) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1725) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1725) + "; diff=" + (contribs.vecs()[3].at(1725) - -0.026947757229208946), Math.abs(contribs.vecs()[3].at(1725) - -0.026947757229208946) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1726) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1726) + "; diff=" + (contribs.vecs()[3].at(1726) - 0.06638649851083755), Math.abs(contribs.vecs()[3].at(1726) - 0.06638649851083755) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1730) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1730) + "; diff=" + (contribs.vecs()[2].at(1730) - 0.0), Math.abs(contribs.vecs()[2].at(1730) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1731) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1731) + "; diff=" + (contribs.vecs()[7].at(1731) - 0.2587127387523651), Math.abs(contribs.vecs()[7].at(1731) - 0.2587127387523651) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1769) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1769) + "; diff=" + (contribs.vecs()[2].at(1769) - 0.0), Math.abs(contribs.vecs()[2].at(1769) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1770) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1770) + "; diff=" + (contribs.vecs()[4].at(1770) - 0.0), Math.abs(contribs.vecs()[4].at(1770) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1773) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1773) + "; diff=" + (contribs.vecs()[6].at(1773) - 0.04940139874815941), Math.abs(contribs.vecs()[6].at(1773) - 0.04940139874815941) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1784) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1784) + "; diff=" + (contribs.vecs()[2].at(1784) - 0.0), Math.abs(contribs.vecs()[2].at(1784) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1787) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1787) + "; diff=" + (contribs.vecs()[5].at(1787) - -0.08481662720441818), Math.abs(contribs.vecs()[5].at(1787) - -0.08481662720441818) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1794) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1794) + "; diff=" + (contribs.vecs()[4].at(1794) - 0.0), Math.abs(contribs.vecs()[4].at(1794) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1797) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1797) + "; diff=" + (contribs.vecs()[0].at(1797) - -0.3840912878513336), Math.abs(contribs.vecs()[0].at(1797) - -0.3840912878513336) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1808) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1808) + "; diff=" + (contribs.vecs()[1].at(1808) - -0.08299426734447479), Math.abs(contribs.vecs()[1].at(1808) - -0.08299426734447479) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1811) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1811) + "; diff=" + (contribs.vecs()[3].at(1811) - -0.19499067962169647), Math.abs(contribs.vecs()[3].at(1811) - -0.19499067962169647) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1825) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1825) + "; diff=" + (contribs.vecs()[4].at(1825) - 0.04322551563382149), Math.abs(contribs.vecs()[4].at(1825) - 0.04322551563382149) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1832) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1832) + "; diff=" + (contribs.vecs()[5].at(1832) - 0.20207571983337402), Math.abs(contribs.vecs()[5].at(1832) - 0.20207571983337402) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1835) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1835) + "; diff=" + (contribs.vecs()[1].at(1835) - -0.09596008062362671), Math.abs(contribs.vecs()[1].at(1835) - -0.09596008062362671) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1844) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1844) + "; diff=" + (contribs.vecs()[3].at(1844) - -0.38401108980178833), Math.abs(contribs.vecs()[3].at(1844) - -0.38401108980178833) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1852) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1852) + "; diff=" + (contribs.vecs()[4].at(1852) - 0.6224786639213562), Math.abs(contribs.vecs()[4].at(1852) - 0.6224786639213562) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1854) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1854) + "; diff=" + (contribs.vecs()[1].at(1854) - 0.16771896183490753), Math.abs(contribs.vecs()[1].at(1854) - 0.16771896183490753) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1860) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1860) + "; diff=" + (contribs.vecs()[3].at(1860) - -0.6498047709465027), Math.abs(contribs.vecs()[3].at(1860) - -0.6498047709465027) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1863) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1863) + "; diff=" + (contribs.vecs()[6].at(1863) - 0.5561907291412354), Math.abs(contribs.vecs()[6].at(1863) - 0.5561907291412354) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1869) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1869) + "; diff=" + (contribs.vecs()[5].at(1869) - -3.2439544200897217), Math.abs(contribs.vecs()[5].at(1869) - -3.2439544200897217) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1870) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1870) + "; diff=" + (contribs.vecs()[3].at(1870) - 0.0), Math.abs(contribs.vecs()[3].at(1870) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1877) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1877) + "; diff=" + (contribs.vecs()[3].at(1877) - -0.5020903944969177), Math.abs(contribs.vecs()[3].at(1877) - -0.5020903944969177) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1883) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1883) + "; diff=" + (contribs.vecs()[3].at(1883) - 0.0), Math.abs(contribs.vecs()[3].at(1883) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1887) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1887) + "; diff=" + (contribs.vecs()[6].at(1887) - -0.5655964016914368), Math.abs(contribs.vecs()[6].at(1887) - -0.5655964016914368) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1890) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1890) + "; diff=" + (contribs.vecs()[7].at(1890) - 0.11167144030332565), Math.abs(contribs.vecs()[7].at(1890) - 0.11167144030332565) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1898) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1898) + "; diff=" + (contribs.vecs()[0].at(1898) - -0.34260988235473633), Math.abs(contribs.vecs()[0].at(1898) - -0.34260988235473633) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1902) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1902) + "; diff=" + (contribs.vecs()[3].at(1902) - -0.5574719905853271), Math.abs(contribs.vecs()[3].at(1902) - -0.5574719905853271) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1908) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1908) + "; diff=" + (contribs.vecs()[0].at(1908) - -0.40525177121162415), Math.abs(contribs.vecs()[0].at(1908) - -0.40525177121162415) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1908) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1908) + "; diff=" + (contribs.vecs()[7].at(1908) - 0.23590712249279022), Math.abs(contribs.vecs()[7].at(1908) - 0.23590712249279022) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1912) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1912) + "; diff=" + (contribs.vecs()[1].at(1912) - 0.0), Math.abs(contribs.vecs()[1].at(1912) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1929) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1929) + "; diff=" + (contribs.vecs()[0].at(1929) - -0.2280067354440689), Math.abs(contribs.vecs()[0].at(1929) - -0.2280067354440689) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1935) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1935) + "; diff=" + (contribs.vecs()[3].at(1935) - -0.13118162751197815), Math.abs(contribs.vecs()[3].at(1935) - -0.13118162751197815) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1937) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1937) + "; diff=" + (contribs.vecs()[2].at(1937) - 0.0), Math.abs(contribs.vecs()[2].at(1937) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1938) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1938) + "; diff=" + (contribs.vecs()[3].at(1938) - -0.45896971225738525), Math.abs(contribs.vecs()[3].at(1938) - -0.45896971225738525) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1938) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1938) + "; diff=" + (contribs.vecs()[4].at(1938) - 0.0), Math.abs(contribs.vecs()[4].at(1938) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1939) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1939) + "; diff=" + (contribs.vecs()[4].at(1939) - -0.34655430912971497), Math.abs(contribs.vecs()[4].at(1939) - -0.34655430912971497) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1940) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1940) + "; diff=" + (contribs.vecs()[4].at(1940) - -0.3896315395832062), Math.abs(contribs.vecs()[4].at(1940) - -0.3896315395832062) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1942) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1942) + "; diff=" + (contribs.vecs()[6].at(1942) - 0.8989930152893066), Math.abs(contribs.vecs()[6].at(1942) - 0.8989930152893066) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1945) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1945) + "; diff=" + (contribs.vecs()[5].at(1945) - 2.1151039600372314), Math.abs(contribs.vecs()[5].at(1945) - 2.1151039600372314) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1950) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1950) + "; diff=" + (contribs.vecs()[6].at(1950) - 0.16227500140666962), Math.abs(contribs.vecs()[6].at(1950) - 0.16227500140666962) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1965) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1965) + "; diff=" + (contribs.vecs()[1].at(1965) - 0.0), Math.abs(contribs.vecs()[1].at(1965) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1967) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1967) + "; diff=" + (contribs.vecs()[7].at(1967) - 0.17290009558200836), Math.abs(contribs.vecs()[7].at(1967) - 0.17290009558200836) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1973) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1973) + "; diff=" + (contribs.vecs()[1].at(1973) - 0.0), Math.abs(contribs.vecs()[1].at(1973) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1974) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1974) + "; diff=" + (contribs.vecs()[2].at(1974) - 0.0), Math.abs(contribs.vecs()[2].at(1974) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1984) + "; diff=" + (contribs.vecs()[6].at(1984) - 0.1068720817565918), Math.abs(contribs.vecs()[6].at(1984) - 0.1068720817565918) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1991) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1991) + "; diff=" + (contribs.vecs()[0].at(1991) - -0.24880318343639374), Math.abs(contribs.vecs()[0].at(1991) - -0.24880318343639374) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1995) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1995) + "; diff=" + (contribs.vecs()[1].at(1995) - 0.0), Math.abs(contribs.vecs()[1].at(1995) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1995) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1995) + "; diff=" + (contribs.vecs()[7].at(1995) - -0.002281967317685485), Math.abs(contribs.vecs()[7].at(1995) - -0.002281967317685485) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2008) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2008) + "; diff=" + (contribs.vecs()[2].at(2008) - 0.022622674703598022), Math.abs(contribs.vecs()[2].at(2008) - 0.022622674703598022) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2009) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2009) + "; diff=" + (contribs.vecs()[3].at(2009) - -0.015145013108849525), Math.abs(contribs.vecs()[3].at(2009) - -0.015145013108849525) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2012) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2012) + "; diff=" + (contribs.vecs()[5].at(2012) - -0.040933284908533096), Math.abs(contribs.vecs()[5].at(2012) - -0.040933284908533096) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2013) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2013) + "; diff=" + (contribs.vecs()[3].at(2013) - 0.0), Math.abs(contribs.vecs()[3].at(2013) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2014) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2014) + "; diff=" + (contribs.vecs()[2].at(2014) - 0.0), Math.abs(contribs.vecs()[2].at(2014) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2020) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2020) + "; diff=" + (contribs.vecs()[5].at(2020) - -0.010880997404456139), Math.abs(contribs.vecs()[5].at(2020) - -0.010880997404456139) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2021) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2021) + "; diff=" + (contribs.vecs()[4].at(2021) - 0.0), Math.abs(contribs.vecs()[4].at(2021) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2029) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2029) + "; diff=" + (contribs.vecs()[3].at(2029) - 0.19499067962169647), Math.abs(contribs.vecs()[3].at(2029) - 0.19499067962169647) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2042) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2042) + "; diff=" + (contribs.vecs()[0].at(2042) - -0.22986941039562225), Math.abs(contribs.vecs()[0].at(2042) - -0.22986941039562225) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2045) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2045) + "; diff=" + (contribs.vecs()[1].at(2045) - -0.04082760214805603), Math.abs(contribs.vecs()[1].at(2045) - -0.04082760214805603) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2057) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2057) + "; diff=" + (contribs.vecs()[2].at(2057) - 0.0), Math.abs(contribs.vecs()[2].at(2057) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2057) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2057) + "; diff=" + (contribs.vecs()[7].at(2057) - 0.0), Math.abs(contribs.vecs()[7].at(2057) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2072) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2072) + "; diff=" + (contribs.vecs()[0].at(2072) - -0.42314350605010986), Math.abs(contribs.vecs()[0].at(2072) - -0.42314350605010986) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2074) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2074) + "; diff=" + (contribs.vecs()[3].at(2074) - 0.36947113275527954), Math.abs(contribs.vecs()[3].at(2074) - 0.36947113275527954) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2075) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2075) + "; diff=" + (contribs.vecs()[0].at(2075) - -0.2168416976928711), Math.abs(contribs.vecs()[0].at(2075) - -0.2168416976928711) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2079) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2079) + "; diff=" + (contribs.vecs()[5].at(2079) - -0.06806577742099762), Math.abs(contribs.vecs()[5].at(2079) - -0.06806577742099762) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2081) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2081) + "; diff=" + (contribs.vecs()[0].at(2081) - -0.16542606055736542), Math.abs(contribs.vecs()[0].at(2081) - -0.16542606055736542) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2086) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2086) + "; diff=" + (contribs.vecs()[7].at(2086) - -0.15046222507953644), Math.abs(contribs.vecs()[7].at(2086) - -0.15046222507953644) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2094) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2094) + "; diff=" + (contribs.vecs()[4].at(2094) - 0.0), Math.abs(contribs.vecs()[4].at(2094) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2097) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2097) + "; diff=" + (contribs.vecs()[4].at(2097) - -0.34655430912971497), Math.abs(contribs.vecs()[4].at(2097) - -0.34655430912971497) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2098) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2098) + "; diff=" + (contribs.vecs()[2].at(2098) - 1.2383278608322144), Math.abs(contribs.vecs()[2].at(2098) - 1.2383278608322144) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2107) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2107) + "; diff=" + (contribs.vecs()[3].at(2107) - -0.7089053392410278), Math.abs(contribs.vecs()[3].at(2107) - -0.7089053392410278) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2118) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2118) + "; diff=" + (contribs.vecs()[0].at(2118) - -0.4869399070739746), Math.abs(contribs.vecs()[0].at(2118) - -0.4869399070739746) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2119) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2119) + "; diff=" + (contribs.vecs()[3].at(2119) - 0.0), Math.abs(contribs.vecs()[3].at(2119) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2122) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2122) + "; diff=" + (contribs.vecs()[1].at(2122) - 0.06450061500072479), Math.abs(contribs.vecs()[1].at(2122) - 0.06450061500072479) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2124) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2124) + "; diff=" + (contribs.vecs()[2].at(2124) - 0.0), Math.abs(contribs.vecs()[2].at(2124) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2142) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2142) + "; diff=" + (contribs.vecs()[7].at(2142) - -0.4755629301071167), Math.abs(contribs.vecs()[7].at(2142) - -0.4755629301071167) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2154) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2154) + "; diff=" + (contribs.vecs()[1].at(2154) - 0.0), Math.abs(contribs.vecs()[1].at(2154) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2160) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2160) + "; diff=" + (contribs.vecs()[2].at(2160) - 0.0), Math.abs(contribs.vecs()[2].at(2160) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2195) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2195) + "; diff=" + (contribs.vecs()[2].at(2195) - 0.0), Math.abs(contribs.vecs()[2].at(2195) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2200) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2200) + "; diff=" + (contribs.vecs()[2].at(2200) - 0.6382100582122803), Math.abs(contribs.vecs()[2].at(2200) - 0.6382100582122803) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2205) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2205) + "; diff=" + (contribs.vecs()[4].at(2205) - -0.04793485626578331), Math.abs(contribs.vecs()[4].at(2205) - -0.04793485626578331) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2217) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2217) + "; diff=" + (contribs.vecs()[0].at(2217) - -0.3250027298927307), Math.abs(contribs.vecs()[0].at(2217) - -0.3250027298927307) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2223) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2223) + "; diff=" + (contribs.vecs()[6].at(2223) - -0.7274534106254578), Math.abs(contribs.vecs()[6].at(2223) - -0.7274534106254578) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2225) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2225) + "; diff=" + (contribs.vecs()[1].at(2225) - 0.0), Math.abs(contribs.vecs()[1].at(2225) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2229) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2229) + "; diff=" + (contribs.vecs()[3].at(2229) - 0.0037235745694488287), Math.abs(contribs.vecs()[3].at(2229) - 0.0037235745694488287) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2230) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2230) + "; diff=" + (contribs.vecs()[5].at(2230) - 0.006058772560209036), Math.abs(contribs.vecs()[5].at(2230) - 0.006058772560209036) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2234) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2234) + "; diff=" + (contribs.vecs()[3].at(2234) - 0.0), Math.abs(contribs.vecs()[3].at(2234) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2236) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2236) + "; diff=" + (contribs.vecs()[6].at(2236) - -1.218820333480835), Math.abs(contribs.vecs()[6].at(2236) - -1.218820333480835) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2241) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2241) + "; diff=" + (contribs.vecs()[7].at(2241) - -0.39161959290504456), Math.abs(contribs.vecs()[7].at(2241) - -0.39161959290504456) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2245) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2245) + "; diff=" + (contribs.vecs()[5].at(2245) - 0.04243364557623863), Math.abs(contribs.vecs()[5].at(2245) - 0.04243364557623863) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2247) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2247) + "; diff=" + (contribs.vecs()[0].at(2247) - -0.7501740455627441), Math.abs(contribs.vecs()[0].at(2247) - -0.7501740455627441) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2248) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2248) + "; diff=" + (contribs.vecs()[0].at(2248) - -0.43249085545539856), Math.abs(contribs.vecs()[0].at(2248) - -0.43249085545539856) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2258) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2258) + "; diff=" + (contribs.vecs()[0].at(2258) - -0.09566572308540344), Math.abs(contribs.vecs()[0].at(2258) - -0.09566572308540344) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2259) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2259) + "; diff=" + (contribs.vecs()[5].at(2259) - -0.20722690224647522), Math.abs(contribs.vecs()[5].at(2259) - -0.20722690224647522) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2263) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2263) + "; diff=" + (contribs.vecs()[7].at(2263) - 0.0), Math.abs(contribs.vecs()[7].at(2263) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2271) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2271) + "; diff=" + (contribs.vecs()[5].at(2271) - -0.43483856320381165), Math.abs(contribs.vecs()[5].at(2271) - -0.43483856320381165) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2278) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2278) + "; diff=" + (contribs.vecs()[5].at(2278) - -0.011539543978869915), Math.abs(contribs.vecs()[5].at(2278) - -0.011539543978869915) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2281) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2281) + "; diff=" + (contribs.vecs()[1].at(2281) - 0.0), Math.abs(contribs.vecs()[1].at(2281) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2285) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2285) + "; diff=" + (contribs.vecs()[5].at(2285) - 0.026997782289981842), Math.abs(contribs.vecs()[5].at(2285) - 0.026997782289981842) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2294) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2294) + "; diff=" + (contribs.vecs()[4].at(2294) - 0.0), Math.abs(contribs.vecs()[4].at(2294) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2294) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2294) + "; diff=" + (contribs.vecs()[7].at(2294) - -0.046757809817790985), Math.abs(contribs.vecs()[7].at(2294) - -0.046757809817790985) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2296) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2296) + "; diff=" + (contribs.vecs()[1].at(2296) - 0.0), Math.abs(contribs.vecs()[1].at(2296) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2306) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2306) + "; diff=" + (contribs.vecs()[2].at(2306) - 0.0), Math.abs(contribs.vecs()[2].at(2306) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2315) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2315) + "; diff=" + (contribs.vecs()[0].at(2315) - -0.49826619029045105), Math.abs(contribs.vecs()[0].at(2315) - -0.49826619029045105) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2322) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2322) + "; diff=" + (contribs.vecs()[1].at(2322) - 0.0), Math.abs(contribs.vecs()[1].at(2322) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2329) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2329) + "; diff=" + (contribs.vecs()[2].at(2329) - 0.0), Math.abs(contribs.vecs()[2].at(2329) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2341) + "; diff=" + (contribs.vecs()[5].at(2341) - -0.11458416283130646), Math.abs(contribs.vecs()[5].at(2341) - -0.11458416283130646) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2345) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2345) + "; diff=" + (contribs.vecs()[7].at(2345) - 0.49087974429130554), Math.abs(contribs.vecs()[7].at(2345) - 0.49087974429130554) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2349) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2349) + "; diff=" + (contribs.vecs()[6].at(2349) - -0.760672390460968), Math.abs(contribs.vecs()[6].at(2349) - -0.760672390460968) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2358) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2358) + "; diff=" + (contribs.vecs()[5].at(2358) - 0.7139408588409424), Math.abs(contribs.vecs()[5].at(2358) - 0.7139408588409424) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2374) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2374) + "; diff=" + (contribs.vecs()[2].at(2374) - 0.0), Math.abs(contribs.vecs()[2].at(2374) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2374) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2374) + "; diff=" + (contribs.vecs()[4].at(2374) - 0.0), Math.abs(contribs.vecs()[4].at(2374) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2375) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2375) + "; diff=" + (contribs.vecs()[1].at(2375) - 0.0), Math.abs(contribs.vecs()[1].at(2375) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2379) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2379) + "; diff=" + (contribs.vecs()[0].at(2379) - -0.42750540375709534), Math.abs(contribs.vecs()[0].at(2379) - -0.42750540375709534) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2389) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2389) + "; diff=" + (contribs.vecs()[4].at(2389) - 0.0), Math.abs(contribs.vecs()[4].at(2389) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2395) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2395) + "; diff=" + (contribs.vecs()[7].at(2395) - 0.0), Math.abs(contribs.vecs()[7].at(2395) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2410) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2410) + "; diff=" + (contribs.vecs()[4].at(2410) - -0.46558743715286255), Math.abs(contribs.vecs()[4].at(2410) - -0.46558743715286255) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2435) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2435) + "; diff=" + (contribs.vecs()[2].at(2435) - 0.0), Math.abs(contribs.vecs()[2].at(2435) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2442) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2442) + "; diff=" + (contribs.vecs()[7].at(2442) - -0.4051136374473572), Math.abs(contribs.vecs()[7].at(2442) - -0.4051136374473572) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2451) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2451) + "; diff=" + (contribs.vecs()[0].at(2451) - -0.8399887681007385), Math.abs(contribs.vecs()[0].at(2451) - -0.8399887681007385) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2459) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2459) + "; diff=" + (contribs.vecs()[2].at(2459) - 0.0), Math.abs(contribs.vecs()[2].at(2459) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2462) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2462) + "; diff=" + (contribs.vecs()[4].at(2462) - 0.0), Math.abs(contribs.vecs()[4].at(2462) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2468) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2468) + "; diff=" + (contribs.vecs()[1].at(2468) - 0.0), Math.abs(contribs.vecs()[1].at(2468) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2474) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2474) + "; diff=" + (contribs.vecs()[7].at(2474) - 0.0), Math.abs(contribs.vecs()[7].at(2474) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2478) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2478) + "; diff=" + (contribs.vecs()[7].at(2478) - -0.3567661643028259), Math.abs(contribs.vecs()[7].at(2478) - -0.3567661643028259) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2485) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2485) + "; diff=" + (contribs.vecs()[5].at(2485) - 0.0014219990698620677), Math.abs(contribs.vecs()[5].at(2485) - 0.0014219990698620677) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2490) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2490) + "; diff=" + (contribs.vecs()[6].at(2490) - -0.07401317358016968), Math.abs(contribs.vecs()[6].at(2490) - -0.07401317358016968) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2498) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2498) + "; diff=" + (contribs.vecs()[4].at(2498) - 0.0), Math.abs(contribs.vecs()[4].at(2498) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2506) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2506) + "; diff=" + (contribs.vecs()[1].at(2506) - 0.0), Math.abs(contribs.vecs()[1].at(2506) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2507) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2507) + "; diff=" + (contribs.vecs()[6].at(2507) - -0.23687303066253662), Math.abs(contribs.vecs()[6].at(2507) - -0.23687303066253662) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2509) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2509) + "; diff=" + (contribs.vecs()[0].at(2509) - -0.7939766049385071), Math.abs(contribs.vecs()[0].at(2509) - -0.7939766049385071) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2510) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2510) + "; diff=" + (contribs.vecs()[2].at(2510) - 0.0), Math.abs(contribs.vecs()[2].at(2510) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2551) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2551) + "; diff=" + (contribs.vecs()[1].at(2551) - -0.06138652190566063), Math.abs(contribs.vecs()[1].at(2551) - -0.06138652190566063) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2551) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2551) + "; diff=" + (contribs.vecs()[4].at(2551) - -0.0625661090016365), Math.abs(contribs.vecs()[4].at(2551) - -0.0625661090016365) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2568) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2568) + "; diff=" + (contribs.vecs()[1].at(2568) - -0.14739710092544556), Math.abs(contribs.vecs()[1].at(2568) - -0.14739710092544556) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2571) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2571) + "; diff=" + (contribs.vecs()[6].at(2571) - 0.42717307806015015), Math.abs(contribs.vecs()[6].at(2571) - 0.42717307806015015) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2572) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2572) + "; diff=" + (contribs.vecs()[0].at(2572) - -0.38261720538139343), Math.abs(contribs.vecs()[0].at(2572) - -0.38261720538139343) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2574) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2574) + "; diff=" + (contribs.vecs()[0].at(2574) - -0.36214008927345276), Math.abs(contribs.vecs()[0].at(2574) - -0.36214008927345276) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2585) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2585) + "; diff=" + (contribs.vecs()[5].at(2585) - -0.005181428045034409), Math.abs(contribs.vecs()[5].at(2585) - -0.005181428045034409) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2591) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2591) + "; diff=" + (contribs.vecs()[7].at(2591) - -0.15346978604793549), Math.abs(contribs.vecs()[7].at(2591) - -0.15346978604793549) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2596) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2596) + "; diff=" + (contribs.vecs()[1].at(2596) - -0.06275253742933273), Math.abs(contribs.vecs()[1].at(2596) - -0.06275253742933273) < eps);

    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }


  @Test
  public void testClassificationReLUDeepSHAPComparison() {
    Frame fr = parseTestFile("smalldata/prostate/prostate.csv").toCategoricalCol("CAPSULE");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    double eps = 1e-4;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "CAPSULE";
      params._hidden = ari(3, 3);
      params._seed = 123456;
      params._reproducible = true;
      params._activation = DeepLearningParameters.Activation.Rectifier;


      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions()
                      .setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original)
                      .setOutputPerReference(true),
              bgFr);

      //toPyTorch(model, test, bgFr);

      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(4) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(4) + "; diff=" + (contribs.vecs()[0].at(4) - 0.0022530322894454002), Math.abs(contribs.vecs()[0].at(4) - 0.0022530322894454002) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(7) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(7) + "; diff=" + (contribs.vecs()[4].at(7) - 0.0), Math.abs(contribs.vecs()[4].at(7) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(10) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(10) + "; diff=" + (contribs.vecs()[3].at(10) - 0.0), Math.abs(contribs.vecs()[3].at(10) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(11) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(11) + "; diff=" + (contribs.vecs()[4].at(11) - 0.0), Math.abs(contribs.vecs()[4].at(11) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(14) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(14) + "; diff=" + (contribs.vecs()[2].at(14) - 0.0), Math.abs(contribs.vecs()[2].at(14) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(14) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(14) + "; diff=" + (contribs.vecs()[4].at(14) - 0.030686914920806885), Math.abs(contribs.vecs()[4].at(14) - 0.030686914920806885) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(18) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(18) + "; diff=" + (contribs.vecs()[7].at(18) - 0.08645044267177582), Math.abs(contribs.vecs()[7].at(18) - 0.08645044267177582) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(21) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(21) + "; diff=" + (contribs.vecs()[5].at(21) - 0.008112519048154354), Math.abs(contribs.vecs()[5].at(21) - 0.008112519048154354) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(22) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(22) + "; diff=" + (contribs.vecs()[2].at(22) - 0.0), Math.abs(contribs.vecs()[2].at(22) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(34) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(34) + "; diff=" + (contribs.vecs()[3].at(34) - 0.1263210028409958), Math.abs(contribs.vecs()[3].at(34) - 0.1263210028409958) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(35) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(35) + "; diff=" + (contribs.vecs()[0].at(35) - 0.0007028516847640276), Math.abs(contribs.vecs()[0].at(35) - 0.0007028516847640276) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(36) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(36) + "; diff=" + (contribs.vecs()[6].at(36) - -0.025823963806033134), Math.abs(contribs.vecs()[6].at(36) - -0.025823963806033134) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(42) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(42) + "; diff=" + (contribs.vecs()[6].at(42) - -0.027111750096082687), Math.abs(contribs.vecs()[6].at(42) - -0.027111750096082687) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(50) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(50) + "; diff=" + (contribs.vecs()[4].at(50) - 0.03641768917441368), Math.abs(contribs.vecs()[4].at(50) - 0.03641768917441368) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(58) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(58) + "; diff=" + (contribs.vecs()[5].at(58) - -0.09176137298345566), Math.abs(contribs.vecs()[5].at(58) - -0.09176137298345566) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(68) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(68) + "; diff=" + (contribs.vecs()[0].at(68) - 0.003414984093979001), Math.abs(contribs.vecs()[0].at(68) - 0.003414984093979001) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(92) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(92) + "; diff=" + (contribs.vecs()[1].at(92) - 0.0025313280057162046), Math.abs(contribs.vecs()[1].at(92) - 0.0025313280057162046) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(95) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(95) + "; diff=" + (contribs.vecs()[6].at(95) - -0.003381944028660655), Math.abs(contribs.vecs()[6].at(95) - -0.003381944028660655) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(96) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(96) + "; diff=" + (contribs.vecs()[4].at(96) - -0.12276124954223633), Math.abs(contribs.vecs()[4].at(96) - -0.12276124954223633) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(99) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(99) + "; diff=" + (contribs.vecs()[4].at(99) - 0.0), Math.abs(contribs.vecs()[4].at(99) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(103) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(103) + "; diff=" + (contribs.vecs()[3].at(103) - 0.0), Math.abs(contribs.vecs()[3].at(103) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(104) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(104) + "; diff=" + (contribs.vecs()[7].at(104) - 0.027242304757237434), Math.abs(contribs.vecs()[7].at(104) - 0.027242304757237434) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(109) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(109) + "; diff=" + (contribs.vecs()[4].at(109) - -0.04774857684969902), Math.abs(contribs.vecs()[4].at(109) - -0.04774857684969902) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(121) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(121) + "; diff=" + (contribs.vecs()[4].at(121) - -0.15911060571670532), Math.abs(contribs.vecs()[4].at(121) - -0.15911060571670532) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(138) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(138) + "; diff=" + (contribs.vecs()[6].at(138) - 0.0), Math.abs(contribs.vecs()[6].at(138) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(139) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(139) + "; diff=" + (contribs.vecs()[7].at(139) - -0.006115716882050037), Math.abs(contribs.vecs()[7].at(139) - -0.006115716882050037) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(146) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(146) + "; diff=" + (contribs.vecs()[0].at(146) - 0.0004224328731652349), Math.abs(contribs.vecs()[0].at(146) - 0.0004224328731652349) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(148) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(148) + "; diff=" + (contribs.vecs()[6].at(148) - -0.04519467055797577), Math.abs(contribs.vecs()[6].at(148) - -0.04519467055797577) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(156) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(156) + "; diff=" + (contribs.vecs()[2].at(156) - -0.0960310623049736), Math.abs(contribs.vecs()[2].at(156) - -0.0960310623049736) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(160) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(160) + "; diff=" + (contribs.vecs()[6].at(160) - 0.004778292495757341), Math.abs(contribs.vecs()[6].at(160) - 0.004778292495757341) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(168) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(168) + "; diff=" + (contribs.vecs()[4].at(168) - 0.0), Math.abs(contribs.vecs()[4].at(168) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(172) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(172) + "; diff=" + (contribs.vecs()[1].at(172) - -0.0016800060402601957), Math.abs(contribs.vecs()[1].at(172) - -0.0016800060402601957) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(174) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(174) + "; diff=" + (contribs.vecs()[5].at(174) - 0.041116464883089066), Math.abs(contribs.vecs()[5].at(174) - 0.041116464883089066) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(177) + "; diff=" + (contribs.vecs()[4].at(177) - 0.0), Math.abs(contribs.vecs()[4].at(177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(179) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(179) + "; diff=" + (contribs.vecs()[1].at(179) - -0.002126300474628806), Math.abs(contribs.vecs()[1].at(179) - -0.002126300474628806) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(189) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(189) + "; diff=" + (contribs.vecs()[2].at(189) - 0.0), Math.abs(contribs.vecs()[2].at(189) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(189) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(189) + "; diff=" + (contribs.vecs()[3].at(189) - 0.22724193334579468), Math.abs(contribs.vecs()[3].at(189) - 0.22724193334579468) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(190) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(190) + "; diff=" + (contribs.vecs()[6].at(190) - -0.024751175194978714), Math.abs(contribs.vecs()[6].at(190) - -0.024751175194978714) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(192) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(192) + "; diff=" + (contribs.vecs()[6].at(192) - 0.005886700935661793), Math.abs(contribs.vecs()[6].at(192) - 0.005886700935661793) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(199) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(199) + "; diff=" + (contribs.vecs()[0].at(199) - 0.006494735833257437), Math.abs(contribs.vecs()[0].at(199) - 0.006494735833257437) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(207) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(207) + "; diff=" + (contribs.vecs()[0].at(207) - 0.0025441981852054596), Math.abs(contribs.vecs()[0].at(207) - 0.0025441981852054596) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(210) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(210) + "; diff=" + (contribs.vecs()[0].at(210) - 0.0024206824600696564), Math.abs(contribs.vecs()[0].at(210) - 0.0024206824600696564) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(211) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(211) + "; diff=" + (contribs.vecs()[7].at(211) - 0.0), Math.abs(contribs.vecs()[7].at(211) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(229) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(229) + "; diff=" + (contribs.vecs()[1].at(229) - 0.009207786060869694), Math.abs(contribs.vecs()[1].at(229) - 0.009207786060869694) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(229) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(229) + "; diff=" + (contribs.vecs()[2].at(229) - 0.0), Math.abs(contribs.vecs()[2].at(229) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(231) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(231) + "; diff=" + (contribs.vecs()[5].at(231) - 0.033011410385370255), Math.abs(contribs.vecs()[5].at(231) - 0.033011410385370255) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(236) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(236) + "; diff=" + (contribs.vecs()[3].at(236) - -0.052326466888189316), Math.abs(contribs.vecs()[3].at(236) - -0.052326466888189316) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(242) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(242) + "; diff=" + (contribs.vecs()[1].at(242) - 0.008694488555192947), Math.abs(contribs.vecs()[1].at(242) - 0.008694488555192947) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(246) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(246) + "; diff=" + (contribs.vecs()[1].at(246) - 0.012225069105625153), Math.abs(contribs.vecs()[1].at(246) - 0.012225069105625153) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(247) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(247) + "; diff=" + (contribs.vecs()[0].at(247) - 0.0044067176058888435), Math.abs(contribs.vecs()[0].at(247) - 0.0044067176058888435) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(248) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(248) + "; diff=" + (contribs.vecs()[3].at(248) - 0.0), Math.abs(contribs.vecs()[3].at(248) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(250) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(250) + "; diff=" + (contribs.vecs()[4].at(250) - 0.0), Math.abs(contribs.vecs()[4].at(250) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(260) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(260) + "; diff=" + (contribs.vecs()[3].at(260) - -0.1797863394021988), Math.abs(contribs.vecs()[3].at(260) - -0.1797863394021988) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(265) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(265) + "; diff=" + (contribs.vecs()[1].at(265) - -0.003779401769861579), Math.abs(contribs.vecs()[1].at(265) - -0.003779401769861579) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(269) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(269) + "; diff=" + (contribs.vecs()[3].at(269) - 0.09415392577648163), Math.abs(contribs.vecs()[3].at(269) - 0.09415392577648163) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(271) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(271) + "; diff=" + (contribs.vecs()[1].at(271) - 0.000940272759180516), Math.abs(contribs.vecs()[1].at(271) - 0.000940272759180516) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(281) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(281) + "; diff=" + (contribs.vecs()[6].at(281) - 0.0009703431278467178), Math.abs(contribs.vecs()[6].at(281) - 0.0009703431278467178) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(283) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(283) + "; diff=" + (contribs.vecs()[4].at(283) - 0.0), Math.abs(contribs.vecs()[4].at(283) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(292) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(292) + "; diff=" + (contribs.vecs()[4].at(292) - -0.24485529959201813), Math.abs(contribs.vecs()[4].at(292) - -0.24485529959201813) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(294) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(294) + "; diff=" + (contribs.vecs()[1].at(294) - -0.001794807962141931), Math.abs(contribs.vecs()[1].at(294) - -0.001794807962141931) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(299) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(299) + "; diff=" + (contribs.vecs()[0].at(299) - 0.0007757777348160744), Math.abs(contribs.vecs()[0].at(299) - 0.0007757777348160744) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(301) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(301) + "; diff=" + (contribs.vecs()[3].at(301) - 0.0), Math.abs(contribs.vecs()[3].at(301) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(306) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(306) + "; diff=" + (contribs.vecs()[3].at(306) - 0.1866435706615448), Math.abs(contribs.vecs()[3].at(306) - 0.1866435706615448) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(308) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(308) + "; diff=" + (contribs.vecs()[4].at(308) - -0.1241367757320404), Math.abs(contribs.vecs()[4].at(308) - -0.1241367757320404) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(320) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(320) + "; diff=" + (contribs.vecs()[6].at(320) - 0.0), Math.abs(contribs.vecs()[6].at(320) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(327) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(327) + "; diff=" + (contribs.vecs()[7].at(327) - 0.0), Math.abs(contribs.vecs()[7].at(327) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(332) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(332) + "; diff=" + (contribs.vecs()[4].at(332) - 0.0), Math.abs(contribs.vecs()[4].at(332) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(339) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(339) + "; diff=" + (contribs.vecs()[0].at(339) - 0.004423505626618862), Math.abs(contribs.vecs()[0].at(339) - 0.004423505626618862) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(341) + "; diff=" + (contribs.vecs()[3].at(341) - 0.1960795670747757), Math.abs(contribs.vecs()[3].at(341) - 0.1960795670747757) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(342) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(342) + "; diff=" + (contribs.vecs()[3].at(342) - 0.1916118562221527), Math.abs(contribs.vecs()[3].at(342) - 0.1916118562221527) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(345) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(345) + "; diff=" + (contribs.vecs()[7].at(345) - 0.0), Math.abs(contribs.vecs()[7].at(345) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(354) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(354) + "; diff=" + (contribs.vecs()[6].at(354) - 0.043873969465494156), Math.abs(contribs.vecs()[6].at(354) - 0.043873969465494156) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(372) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(372) + "; diff=" + (contribs.vecs()[6].at(372) - -0.07612083852291107), Math.abs(contribs.vecs()[6].at(372) - -0.07612083852291107) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(377) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(377) + "; diff=" + (contribs.vecs()[2].at(377) - 0.0), Math.abs(contribs.vecs()[2].at(377) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(380) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(380) + "; diff=" + (contribs.vecs()[5].at(380) - -0.015985051169991493), Math.abs(contribs.vecs()[5].at(380) - -0.015985051169991493) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(389) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(389) + "; diff=" + (contribs.vecs()[7].at(389) - -0.19340386986732483), Math.abs(contribs.vecs()[7].at(389) - -0.19340386986732483) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(391) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(391) + "; diff=" + (contribs.vecs()[6].at(391) - 0.0018793083727359772), Math.abs(contribs.vecs()[6].at(391) - 0.0018793083727359772) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(409) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(409) + "; diff=" + (contribs.vecs()[4].at(409) - -0.15650998055934906), Math.abs(contribs.vecs()[4].at(409) - -0.15650998055934906) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(425) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(425) + "; diff=" + (contribs.vecs()[4].at(425) - 0.0), Math.abs(contribs.vecs()[4].at(425) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(443) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(443) + "; diff=" + (contribs.vecs()[4].at(443) - 0.0), Math.abs(contribs.vecs()[4].at(443) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(453) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(453) + "; diff=" + (contribs.vecs()[2].at(453) - 0.43529510498046875), Math.abs(contribs.vecs()[2].at(453) - 0.43529510498046875) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(466) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(466) + "; diff=" + (contribs.vecs()[0].at(466) - 0.0019438443705439568), Math.abs(contribs.vecs()[0].at(466) - 0.0019438443705439568) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(478) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(478) + "; diff=" + (contribs.vecs()[0].at(478) - 0.002516582841053605), Math.abs(contribs.vecs()[0].at(478) - 0.002516582841053605) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(485) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(485) + "; diff=" + (contribs.vecs()[3].at(485) - 0.0), Math.abs(contribs.vecs()[3].at(485) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(495) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(495) + "; diff=" + (contribs.vecs()[2].at(495) - 0.0), Math.abs(contribs.vecs()[2].at(495) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(501) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(501) + "; diff=" + (contribs.vecs()[5].at(501) - 0.00886547937989235), Math.abs(contribs.vecs()[5].at(501) - 0.00886547937989235) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(508) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(508) + "; diff=" + (contribs.vecs()[3].at(508) - 0.0), Math.abs(contribs.vecs()[3].at(508) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(512) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(512) + "; diff=" + (contribs.vecs()[1].at(512) - -0.0006752629997208714), Math.abs(contribs.vecs()[1].at(512) - -0.0006752629997208714) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(516) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(516) + "; diff=" + (contribs.vecs()[6].at(516) - -0.012883334420621395), Math.abs(contribs.vecs()[6].at(516) - -0.012883334420621395) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(531) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(531) + "; diff=" + (contribs.vecs()[5].at(531) - -0.2365799844264984), Math.abs(contribs.vecs()[5].at(531) - -0.2365799844264984) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(540) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(540) + "; diff=" + (contribs.vecs()[4].at(540) - 0.0), Math.abs(contribs.vecs()[4].at(540) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(541) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(541) + "; diff=" + (contribs.vecs()[0].at(541) - 0.005892131477594376), Math.abs(contribs.vecs()[0].at(541) - 0.005892131477594376) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(557) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(557) + "; diff=" + (contribs.vecs()[1].at(557) - -0.0025445776991546154), Math.abs(contribs.vecs()[1].at(557) - -0.0025445776991546154) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(563) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(563) + "; diff=" + (contribs.vecs()[2].at(563) - 0.0), Math.abs(contribs.vecs()[2].at(563) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(565) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(565) + "; diff=" + (contribs.vecs()[7].at(565) - 0.14319023489952087), Math.abs(contribs.vecs()[7].at(565) - 0.14319023489952087) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(566) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(566) + "; diff=" + (contribs.vecs()[0].at(566) - 0.010471895337104797), Math.abs(contribs.vecs()[0].at(566) - 0.010471895337104797) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(576) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(576) + "; diff=" + (contribs.vecs()[1].at(576) - -0.0011600902071222663), Math.abs(contribs.vecs()[1].at(576) - -0.0011600902071222663) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(583) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(583) + "; diff=" + (contribs.vecs()[3].at(583) - -0.29205551743507385), Math.abs(contribs.vecs()[3].at(583) - -0.29205551743507385) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(593) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(593) + "; diff=" + (contribs.vecs()[6].at(593) - -0.013272972777485847), Math.abs(contribs.vecs()[6].at(593) - -0.013272972777485847) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(612) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(612) + "; diff=" + (contribs.vecs()[2].at(612) - 0.0), Math.abs(contribs.vecs()[2].at(612) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(616) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(616) + "; diff=" + (contribs.vecs()[0].at(616) - 0.011769300326704979), Math.abs(contribs.vecs()[0].at(616) - 0.011769300326704979) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(624) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(624) + "; diff=" + (contribs.vecs()[2].at(624) - 0.0), Math.abs(contribs.vecs()[2].at(624) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(628) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(628) + "; diff=" + (contribs.vecs()[2].at(628) - -0.5141113996505737), Math.abs(contribs.vecs()[2].at(628) - -0.5141113996505737) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(628) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(628) + "; diff=" + (contribs.vecs()[5].at(628) - 0.0008381839725188911), Math.abs(contribs.vecs()[5].at(628) - 0.0008381839725188911) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(633) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(633) + "; diff=" + (contribs.vecs()[5].at(633) - -0.07892993092536926), Math.abs(contribs.vecs()[5].at(633) - -0.07892993092536926) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(636) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(636) + "; diff=" + (contribs.vecs()[0].at(636) - 0.00998559221625328), Math.abs(contribs.vecs()[0].at(636) - 0.00998559221625328) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(637) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(637) + "; diff=" + (contribs.vecs()[3].at(637) - 0.2799602746963501), Math.abs(contribs.vecs()[3].at(637) - 0.2799602746963501) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(638) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(638) + "; diff=" + (contribs.vecs()[1].at(638) - 0.0006048274226486683), Math.abs(contribs.vecs()[1].at(638) - 0.0006048274226486683) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(644) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(644) + "; diff=" + (contribs.vecs()[2].at(644) - 0.0), Math.abs(contribs.vecs()[2].at(644) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(666) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(666) + "; diff=" + (contribs.vecs()[6].at(666) - 0.012176294811069965), Math.abs(contribs.vecs()[6].at(666) - 0.012176294811069965) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(667) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(667) + "; diff=" + (contribs.vecs()[1].at(667) - 0.00609327107667923), Math.abs(contribs.vecs()[1].at(667) - 0.00609327107667923) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(669) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(669) + "; diff=" + (contribs.vecs()[4].at(669) - -0.04289707541465759), Math.abs(contribs.vecs()[4].at(669) - -0.04289707541465759) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(673) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(673) + "; diff=" + (contribs.vecs()[3].at(673) - 0.0), Math.abs(contribs.vecs()[3].at(673) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(678) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(678) + "; diff=" + (contribs.vecs()[2].at(678) - 0.0), Math.abs(contribs.vecs()[2].at(678) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(679) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(679) + "; diff=" + (contribs.vecs()[4].at(679) - 0.0), Math.abs(contribs.vecs()[4].at(679) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(687) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(687) + "; diff=" + (contribs.vecs()[0].at(687) - 0.005126929376274347), Math.abs(contribs.vecs()[0].at(687) - 0.005126929376274347) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(688) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(688) + "; diff=" + (contribs.vecs()[4].at(688) - 0.0), Math.abs(contribs.vecs()[4].at(688) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(703) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(703) + "; diff=" + (contribs.vecs()[5].at(703) - 0.023841001093387604), Math.abs(contribs.vecs()[5].at(703) - 0.023841001093387604) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(704) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(704) + "; diff=" + (contribs.vecs()[3].at(704) - 0.32518744468688965), Math.abs(contribs.vecs()[3].at(704) - 0.32518744468688965) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(707) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(707) + "; diff=" + (contribs.vecs()[2].at(707) - -0.06422601640224457), Math.abs(contribs.vecs()[2].at(707) - -0.06422601640224457) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(708) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(708) + "; diff=" + (contribs.vecs()[2].at(708) - 0.14889703691005707), Math.abs(contribs.vecs()[2].at(708) - 0.14889703691005707) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(718) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(718) + "; diff=" + (contribs.vecs()[6].at(718) - -0.1036929041147232), Math.abs(contribs.vecs()[6].at(718) - -0.1036929041147232) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(724) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(724) + "; diff=" + (contribs.vecs()[5].at(724) - 0.019141947850584984), Math.abs(contribs.vecs()[5].at(724) - 0.019141947850584984) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(742) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(742) + "; diff=" + (contribs.vecs()[1].at(742) - -0.0005446047289296985), Math.abs(contribs.vecs()[1].at(742) - -0.0005446047289296985) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(746) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(746) + "; diff=" + (contribs.vecs()[2].at(746) - 0.0), Math.abs(contribs.vecs()[2].at(746) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(750) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(750) + "; diff=" + (contribs.vecs()[7].at(750) - 0.07367128878831863), Math.abs(contribs.vecs()[7].at(750) - 0.07367128878831863) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(755) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(755) + "; diff=" + (contribs.vecs()[0].at(755) - 0.0019438184099271894), Math.abs(contribs.vecs()[0].at(755) - 0.0019438184099271894) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(756) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(756) + "; diff=" + (contribs.vecs()[0].at(756) - 0.0032419762574136257), Math.abs(contribs.vecs()[0].at(756) - 0.0032419762574136257) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(756) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(756) + "; diff=" + (contribs.vecs()[4].at(756) - 0.0), Math.abs(contribs.vecs()[4].at(756) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(756) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(756) + "; diff=" + (contribs.vecs()[5].at(756) - 0.02154417335987091), Math.abs(contribs.vecs()[5].at(756) - 0.02154417335987091) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(760) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(760) + "; diff=" + (contribs.vecs()[1].at(760) - 0.004725531674921513), Math.abs(contribs.vecs()[1].at(760) - 0.004725531674921513) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(763) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(763) + "; diff=" + (contribs.vecs()[7].at(763) - 0.11020446568727493), Math.abs(contribs.vecs()[7].at(763) - 0.11020446568727493) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(766) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(766) + "; diff=" + (contribs.vecs()[7].at(766) - -0.1425885707139969), Math.abs(contribs.vecs()[7].at(766) - -0.1425885707139969) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(767) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(767) + "; diff=" + (contribs.vecs()[6].at(767) - 0.0), Math.abs(contribs.vecs()[6].at(767) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(798) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(798) + "; diff=" + (contribs.vecs()[4].at(798) - -0.1482221931219101), Math.abs(contribs.vecs()[4].at(798) - -0.1482221931219101) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(799) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(799) + "; diff=" + (contribs.vecs()[6].at(799) - 0.06988131254911423), Math.abs(contribs.vecs()[6].at(799) - 0.06988131254911423) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(804) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(804) + "; diff=" + (contribs.vecs()[3].at(804) - -0.32446619868278503), Math.abs(contribs.vecs()[3].at(804) - -0.32446619868278503) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(808) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(808) + "; diff=" + (contribs.vecs()[2].at(808) - 0.0), Math.abs(contribs.vecs()[2].at(808) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(812) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(812) + "; diff=" + (contribs.vecs()[1].at(812) - 0.016178550198674202), Math.abs(contribs.vecs()[1].at(812) - 0.016178550198674202) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(818) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(818) + "; diff=" + (contribs.vecs()[1].at(818) - -0.00013484350347425789), Math.abs(contribs.vecs()[1].at(818) - -0.00013484350347425789) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(827) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(827) + "; diff=" + (contribs.vecs()[1].at(827) - -0.0033020772971212864), Math.abs(contribs.vecs()[1].at(827) - -0.0033020772971212864) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(829) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(829) + "; diff=" + (contribs.vecs()[6].at(829) - 0.0), Math.abs(contribs.vecs()[6].at(829) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(838) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(838) + "; diff=" + (contribs.vecs()[1].at(838) - -0.0028208878356963396), Math.abs(contribs.vecs()[1].at(838) - -0.0028208878356963396) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(842) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(842) + "; diff=" + (contribs.vecs()[1].at(842) - -0.006249197758734226), Math.abs(contribs.vecs()[1].at(842) - -0.006249197758734226) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(861) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(861) + "; diff=" + (contribs.vecs()[0].at(861) - 0.0058035352267324924), Math.abs(contribs.vecs()[0].at(861) - 0.0058035352267324924) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(862) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(862) + "; diff=" + (contribs.vecs()[3].at(862) - -0.11691392958164215), Math.abs(contribs.vecs()[3].at(862) - -0.11691392958164215) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(871) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(871) + "; diff=" + (contribs.vecs()[6].at(871) - -0.09016678482294083), Math.abs(contribs.vecs()[6].at(871) - -0.09016678482294083) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(891) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(891) + "; diff=" + (contribs.vecs()[6].at(891) - 0.2125844955444336), Math.abs(contribs.vecs()[6].at(891) - 0.2125844955444336) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(909) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(909) + "; diff=" + (contribs.vecs()[0].at(909) - 0.001632937346585095), Math.abs(contribs.vecs()[0].at(909) - 0.001632937346585095) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(909) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(909) + "; diff=" + (contribs.vecs()[4].at(909) - 0.0), Math.abs(contribs.vecs()[4].at(909) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(925) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(925) + "; diff=" + (contribs.vecs()[4].at(925) - -0.009214065037667751), Math.abs(contribs.vecs()[4].at(925) - -0.009214065037667751) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(934) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(934) + "; diff=" + (contribs.vecs()[1].at(934) - 0.00641679810360074), Math.abs(contribs.vecs()[1].at(934) - 0.00641679810360074) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(937) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(937) + "; diff=" + (contribs.vecs()[6].at(937) - -0.002604898763820529), Math.abs(contribs.vecs()[6].at(937) - -0.002604898763820529) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(938) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(938) + "; diff=" + (contribs.vecs()[0].at(938) - 0.01168486662209034), Math.abs(contribs.vecs()[0].at(938) - 0.01168486662209034) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(957) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(957) + "; diff=" + (contribs.vecs()[4].at(957) - 0.0), Math.abs(contribs.vecs()[4].at(957) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(963) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(963) + "; diff=" + (contribs.vecs()[7].at(963) - -0.20877961814403534), Math.abs(contribs.vecs()[7].at(963) - -0.20877961814403534) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(968) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(968) + "; diff=" + (contribs.vecs()[1].at(968) - 0.005060562398284674), Math.abs(contribs.vecs()[1].at(968) - 0.005060562398284674) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(971) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(971) + "; diff=" + (contribs.vecs()[6].at(971) - 0.0), Math.abs(contribs.vecs()[6].at(971) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(973) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(973) + "; diff=" + (contribs.vecs()[3].at(973) - 0.12422512471675873), Math.abs(contribs.vecs()[3].at(973) - 0.12422512471675873) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(975) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(975) + "; diff=" + (contribs.vecs()[1].at(975) - 0.0), Math.abs(contribs.vecs()[1].at(975) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(976) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(976) + "; diff=" + (contribs.vecs()[7].at(976) - -0.06154855713248253), Math.abs(contribs.vecs()[7].at(976) - -0.06154855713248253) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(985) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(985) + "; diff=" + (contribs.vecs()[2].at(985) - -0.16767671704292297), Math.abs(contribs.vecs()[2].at(985) - -0.16767671704292297) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(989) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(989) + "; diff=" + (contribs.vecs()[3].at(989) - 0.0), Math.abs(contribs.vecs()[3].at(989) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1000) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1000) + "; diff=" + (contribs.vecs()[7].at(1000) - -0.16114342212677002), Math.abs(contribs.vecs()[7].at(1000) - -0.16114342212677002) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1009) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1009) + "; diff=" + (contribs.vecs()[0].at(1009) - 0.008422031998634338), Math.abs(contribs.vecs()[0].at(1009) - 0.008422031998634338) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1010) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1010) + "; diff=" + (contribs.vecs()[7].at(1010) - 0.0), Math.abs(contribs.vecs()[7].at(1010) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1027) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1027) + "; diff=" + (contribs.vecs()[1].at(1027) - -0.005197695456445217), Math.abs(contribs.vecs()[1].at(1027) - -0.005197695456445217) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1039) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1039) + "; diff=" + (contribs.vecs()[2].at(1039) - -0.31756240129470825), Math.abs(contribs.vecs()[2].at(1039) - -0.31756240129470825) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1047) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1047) + "; diff=" + (contribs.vecs()[7].at(1047) - 0.0), Math.abs(contribs.vecs()[7].at(1047) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1050) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1050) + "; diff=" + (contribs.vecs()[6].at(1050) - 0.002386835403740406), Math.abs(contribs.vecs()[6].at(1050) - 0.002386835403740406) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1056) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1056) + "; diff=" + (contribs.vecs()[4].at(1056) - 0.0), Math.abs(contribs.vecs()[4].at(1056) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1061) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1061) + "; diff=" + (contribs.vecs()[7].at(1061) - -0.008352608419954777), Math.abs(contribs.vecs()[7].at(1061) - -0.008352608419954777) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1065) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1065) + "; diff=" + (contribs.vecs()[2].at(1065) - 0.4403659999370575), Math.abs(contribs.vecs()[2].at(1065) - 0.4403659999370575) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1079) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1079) + "; diff=" + (contribs.vecs()[1].at(1079) - 0.006223337259143591), Math.abs(contribs.vecs()[1].at(1079) - 0.006223337259143591) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1079) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1079) + "; diff=" + (contribs.vecs()[7].at(1079) - 0.0), Math.abs(contribs.vecs()[7].at(1079) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1095) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1095) + "; diff=" + (contribs.vecs()[3].at(1095) - 0.39538267254829407), Math.abs(contribs.vecs()[3].at(1095) - 0.39538267254829407) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1099) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1099) + "; diff=" + (contribs.vecs()[1].at(1099) - 0.0029332577250897884), Math.abs(contribs.vecs()[1].at(1099) - 0.0029332577250897884) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1120) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1120) + "; diff=" + (contribs.vecs()[2].at(1120) - 0.0), Math.abs(contribs.vecs()[2].at(1120) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1130) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1130) + "; diff=" + (contribs.vecs()[4].at(1130) - 0.0), Math.abs(contribs.vecs()[4].at(1130) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1132) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1132) + "; diff=" + (contribs.vecs()[4].at(1132) - -0.09378690272569656), Math.abs(contribs.vecs()[4].at(1132) - -0.09378690272569656) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1133) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1133) + "; diff=" + (contribs.vecs()[7].at(1133) - -0.32589462399482727), Math.abs(contribs.vecs()[7].at(1133) - -0.32589462399482727) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1145) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1145) + "; diff=" + (contribs.vecs()[6].at(1145) - -0.14026089012622833), Math.abs(contribs.vecs()[6].at(1145) - -0.14026089012622833) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1148) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1148) + "; diff=" + (contribs.vecs()[7].at(1148) - -0.4033825397491455), Math.abs(contribs.vecs()[7].at(1148) - -0.4033825397491455) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1159) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1159) + "; diff=" + (contribs.vecs()[5].at(1159) - -0.12920701503753662), Math.abs(contribs.vecs()[5].at(1159) - -0.12920701503753662) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1164) + "; diff=" + (contribs.vecs()[2].at(1164) - 0.0), Math.abs(contribs.vecs()[2].at(1164) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1166) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1166) + "; diff=" + (contribs.vecs()[3].at(1166) - 0.0), Math.abs(contribs.vecs()[3].at(1166) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1180) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1180) + "; diff=" + (contribs.vecs()[1].at(1180) - -0.003569747554138303), Math.abs(contribs.vecs()[1].at(1180) - -0.003569747554138303) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1201) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1201) + "; diff=" + (contribs.vecs()[7].at(1201) - 0.24331693351268768), Math.abs(contribs.vecs()[7].at(1201) - 0.24331693351268768) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1209) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1209) + "; diff=" + (contribs.vecs()[4].at(1209) - 0.0), Math.abs(contribs.vecs()[4].at(1209) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1216) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1216) + "; diff=" + (contribs.vecs()[7].at(1216) - 0.0), Math.abs(contribs.vecs()[7].at(1216) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1219) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1219) + "; diff=" + (contribs.vecs()[7].at(1219) - 0.18400807678699493), Math.abs(contribs.vecs()[7].at(1219) - 0.18400807678699493) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1223) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1223) + "; diff=" + (contribs.vecs()[3].at(1223) - 0.0), Math.abs(contribs.vecs()[3].at(1223) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1223) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1223) + "; diff=" + (contribs.vecs()[7].at(1223) - 0.006256530061364174), Math.abs(contribs.vecs()[7].at(1223) - 0.006256530061364174) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1228) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1228) + "; diff=" + (contribs.vecs()[6].at(1228) - -0.0241287499666214), Math.abs(contribs.vecs()[6].at(1228) - -0.0241287499666214) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1240) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1240) + "; diff=" + (contribs.vecs()[4].at(1240) - 0.0), Math.abs(contribs.vecs()[4].at(1240) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1243) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1243) + "; diff=" + (contribs.vecs()[2].at(1243) - 0.0), Math.abs(contribs.vecs()[2].at(1243) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1248) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1248) + "; diff=" + (contribs.vecs()[0].at(1248) - 0.0068368567153811455), Math.abs(contribs.vecs()[0].at(1248) - 0.0068368567153811455) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1249) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1249) + "; diff=" + (contribs.vecs()[6].at(1249) - 0.0), Math.abs(contribs.vecs()[6].at(1249) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1263) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1263) + "; diff=" + (contribs.vecs()[5].at(1263) - 0.06783655285835266), Math.abs(contribs.vecs()[5].at(1263) - 0.06783655285835266) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1272) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1272) + "; diff=" + (contribs.vecs()[3].at(1272) - -0.09517879039049149), Math.abs(contribs.vecs()[3].at(1272) - -0.09517879039049149) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1275) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1275) + "; diff=" + (contribs.vecs()[1].at(1275) - -0.0066292379051446915), Math.abs(contribs.vecs()[1].at(1275) - -0.0066292379051446915) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1281) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1281) + "; diff=" + (contribs.vecs()[3].at(1281) - -0.08858741074800491), Math.abs(contribs.vecs()[3].at(1281) - -0.08858741074800491) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1284) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1284) + "; diff=" + (contribs.vecs()[5].at(1284) - -0.05671408772468567), Math.abs(contribs.vecs()[5].at(1284) - -0.05671408772468567) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1291) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1291) + "; diff=" + (contribs.vecs()[1].at(1291) - 0.005387417506426573), Math.abs(contribs.vecs()[1].at(1291) - 0.005387417506426573) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1291) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1291) + "; diff=" + (contribs.vecs()[3].at(1291) - 1.266815423965454), Math.abs(contribs.vecs()[3].at(1291) - 1.266815423965454) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1308) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1308) + "; diff=" + (contribs.vecs()[5].at(1308) - -0.04208841919898987), Math.abs(contribs.vecs()[5].at(1308) - -0.04208841919898987) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1316) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1316) + "; diff=" + (contribs.vecs()[6].at(1316) - -0.024270309135317802), Math.abs(contribs.vecs()[6].at(1316) - -0.024270309135317802) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1323) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1323) + "; diff=" + (contribs.vecs()[2].at(1323) - 0.0), Math.abs(contribs.vecs()[2].at(1323) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1326) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1326) + "; diff=" + (contribs.vecs()[3].at(1326) - 0.0), Math.abs(contribs.vecs()[3].at(1326) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1333) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1333) + "; diff=" + (contribs.vecs()[7].at(1333) - -0.06130077689886093), Math.abs(contribs.vecs()[7].at(1333) - -0.06130077689886093) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1346) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1346) + "; diff=" + (contribs.vecs()[1].at(1346) - -0.002618578728288412), Math.abs(contribs.vecs()[1].at(1346) - -0.002618578728288412) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1347) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1347) + "; diff=" + (contribs.vecs()[3].at(1347) - -0.17377300560474396), Math.abs(contribs.vecs()[3].at(1347) - -0.17377300560474396) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1349) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1349) + "; diff=" + (contribs.vecs()[6].at(1349) - -0.19107750058174133), Math.abs(contribs.vecs()[6].at(1349) - -0.19107750058174133) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1357) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1357) + "; diff=" + (contribs.vecs()[2].at(1357) - 0.0), Math.abs(contribs.vecs()[2].at(1357) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1357) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1357) + "; diff=" + (contribs.vecs()[7].at(1357) - -0.17829348146915436), Math.abs(contribs.vecs()[7].at(1357) - -0.17829348146915436) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1364) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1364) + "; diff=" + (contribs.vecs()[1].at(1364) - 0.0007497526239603758), Math.abs(contribs.vecs()[1].at(1364) - 0.0007497526239603758) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1369) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1369) + "; diff=" + (contribs.vecs()[4].at(1369) - 0.0), Math.abs(contribs.vecs()[4].at(1369) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1376) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1376) + "; diff=" + (contribs.vecs()[5].at(1376) - -0.015912482514977455), Math.abs(contribs.vecs()[5].at(1376) - -0.015912482514977455) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1377) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1377) + "; diff=" + (contribs.vecs()[5].at(1377) - 0.035819754004478455), Math.abs(contribs.vecs()[5].at(1377) - 0.035819754004478455) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1378) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1378) + "; diff=" + (contribs.vecs()[2].at(1378) - 0.0), Math.abs(contribs.vecs()[2].at(1378) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1378) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1378) + "; diff=" + (contribs.vecs()[3].at(1378) - -0.1857113540172577), Math.abs(contribs.vecs()[3].at(1378) - -0.1857113540172577) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1382) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1382) + "; diff=" + (contribs.vecs()[1].at(1382) - 0.007968848571181297), Math.abs(contribs.vecs()[1].at(1382) - 0.007968848571181297) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1393) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1393) + "; diff=" + (contribs.vecs()[1].at(1393) - 0.00436181528493762), Math.abs(contribs.vecs()[1].at(1393) - 0.00436181528493762) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1395) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1395) + "; diff=" + (contribs.vecs()[4].at(1395) - -0.03278009220957756), Math.abs(contribs.vecs()[4].at(1395) - -0.03278009220957756) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1398) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1398) + "; diff=" + (contribs.vecs()[0].at(1398) - 0.004399542696774006), Math.abs(contribs.vecs()[0].at(1398) - 0.004399542696774006) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1406) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1406) + "; diff=" + (contribs.vecs()[1].at(1406) - -0.0014128116890788078), Math.abs(contribs.vecs()[1].at(1406) - -0.0014128116890788078) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1406) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1406) + "; diff=" + (contribs.vecs()[7].at(1406) - -0.11391080915927887), Math.abs(contribs.vecs()[7].at(1406) - -0.11391080915927887) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1412) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1412) + "; diff=" + (contribs.vecs()[2].at(1412) - 0.0), Math.abs(contribs.vecs()[2].at(1412) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1416) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1416) + "; diff=" + (contribs.vecs()[1].at(1416) - 0.010678294114768505), Math.abs(contribs.vecs()[1].at(1416) - 0.010678294114768505) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1419) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1419) + "; diff=" + (contribs.vecs()[7].at(1419) - -0.014783057384192944), Math.abs(contribs.vecs()[7].at(1419) - -0.014783057384192944) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1430) + "; diff=" + (contribs.vecs()[7].at(1430) - 0.4513237774372101), Math.abs(contribs.vecs()[7].at(1430) - 0.4513237774372101) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1438) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1438) + "; diff=" + (contribs.vecs()[6].at(1438) - -0.012404222041368484), Math.abs(contribs.vecs()[6].at(1438) - -0.012404222041368484) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1447) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1447) + "; diff=" + (contribs.vecs()[6].at(1447) - -0.004168993793427944), Math.abs(contribs.vecs()[6].at(1447) - -0.004168993793427944) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1455) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1455) + "; diff=" + (contribs.vecs()[4].at(1455) - 0.0), Math.abs(contribs.vecs()[4].at(1455) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1464) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1464) + "; diff=" + (contribs.vecs()[4].at(1464) - 0.0), Math.abs(contribs.vecs()[4].at(1464) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1469) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1469) + "; diff=" + (contribs.vecs()[4].at(1469) - 0.0), Math.abs(contribs.vecs()[4].at(1469) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1470) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1470) + "; diff=" + (contribs.vecs()[6].at(1470) - -0.007941191084682941), Math.abs(contribs.vecs()[6].at(1470) - -0.007941191084682941) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1481) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1481) + "; diff=" + (contribs.vecs()[5].at(1481) - 0.0), Math.abs(contribs.vecs()[5].at(1481) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1481) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1481) + "; diff=" + (contribs.vecs()[7].at(1481) - 0.0), Math.abs(contribs.vecs()[7].at(1481) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1484) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1484) + "; diff=" + (contribs.vecs()[7].at(1484) - -0.28404635190963745), Math.abs(contribs.vecs()[7].at(1484) - -0.28404635190963745) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1499) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1499) + "; diff=" + (contribs.vecs()[0].at(1499) - 0.011148720048367977), Math.abs(contribs.vecs()[0].at(1499) - 0.011148720048367977) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1504) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1504) + "; diff=" + (contribs.vecs()[6].at(1504) - 0.0), Math.abs(contribs.vecs()[6].at(1504) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1505) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1505) + "; diff=" + (contribs.vecs()[4].at(1505) - 0.0), Math.abs(contribs.vecs()[4].at(1505) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1508) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1508) + "; diff=" + (contribs.vecs()[1].at(1508) - -0.019276263192296028), Math.abs(contribs.vecs()[1].at(1508) - -0.019276263192296028) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1515) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1515) + "; diff=" + (contribs.vecs()[7].at(1515) - 0.0), Math.abs(contribs.vecs()[7].at(1515) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1530) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1530) + "; diff=" + (contribs.vecs()[3].at(1530) - -0.08827926963567734), Math.abs(contribs.vecs()[3].at(1530) - -0.08827926963567734) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1532) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1532) + "; diff=" + (contribs.vecs()[4].at(1532) - -0.032891646027565), Math.abs(contribs.vecs()[4].at(1532) - -0.032891646027565) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1533) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1533) + "; diff=" + (contribs.vecs()[1].at(1533) - 0.008597113192081451), Math.abs(contribs.vecs()[1].at(1533) - 0.008597113192081451) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1544) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1544) + "; diff=" + (contribs.vecs()[4].at(1544) - 0.0), Math.abs(contribs.vecs()[4].at(1544) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1553) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1553) + "; diff=" + (contribs.vecs()[4].at(1553) - 0.0), Math.abs(contribs.vecs()[4].at(1553) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1559) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1559) + "; diff=" + (contribs.vecs()[5].at(1559) - 0.12635868787765503), Math.abs(contribs.vecs()[5].at(1559) - 0.12635868787765503) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1564) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1564) + "; diff=" + (contribs.vecs()[2].at(1564) - 0.0), Math.abs(contribs.vecs()[2].at(1564) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1572) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1572) + "; diff=" + (contribs.vecs()[3].at(1572) - -0.08556395024061203), Math.abs(contribs.vecs()[3].at(1572) - -0.08556395024061203) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1575) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1575) + "; diff=" + (contribs.vecs()[0].at(1575) - 0.01057206466794014), Math.abs(contribs.vecs()[0].at(1575) - 0.01057206466794014) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1580) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1580) + "; diff=" + (contribs.vecs()[1].at(1580) - 0.0005981634021736681), Math.abs(contribs.vecs()[1].at(1580) - 0.0005981634021736681) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1586) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1586) + "; diff=" + (contribs.vecs()[7].at(1586) - -0.13915595412254333), Math.abs(contribs.vecs()[7].at(1586) - -0.13915595412254333) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1599) + "; diff=" + (contribs.vecs()[3].at(1599) - 0.10529808700084686), Math.abs(contribs.vecs()[3].at(1599) - 0.10529808700084686) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1599) + "; diff=" + (contribs.vecs()[5].at(1599) - 0.16799549758434296), Math.abs(contribs.vecs()[5].at(1599) - 0.16799549758434296) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1599) + "; diff=" + (contribs.vecs()[6].at(1599) - 0.0), Math.abs(contribs.vecs()[6].at(1599) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1603) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1603) + "; diff=" + (contribs.vecs()[0].at(1603) - 0.05992807075381279), Math.abs(contribs.vecs()[0].at(1603) - 0.05992807075381279) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1615) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1615) + "; diff=" + (contribs.vecs()[3].at(1615) - 0.0), Math.abs(contribs.vecs()[3].at(1615) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1620) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1620) + "; diff=" + (contribs.vecs()[1].at(1620) - -0.006235333625227213), Math.abs(contribs.vecs()[1].at(1620) - -0.006235333625227213) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1643) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1643) + "; diff=" + (contribs.vecs()[7].at(1643) - 0.0), Math.abs(contribs.vecs()[7].at(1643) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1656) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1656) + "; diff=" + (contribs.vecs()[3].at(1656) - 0.2849220931529999), Math.abs(contribs.vecs()[3].at(1656) - 0.2849220931529999) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1657) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1657) + "; diff=" + (contribs.vecs()[5].at(1657) - -0.0028211737517267466), Math.abs(contribs.vecs()[5].at(1657) - -0.0028211737517267466) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1659) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1659) + "; diff=" + (contribs.vecs()[1].at(1659) - -0.0021799656096845865), Math.abs(contribs.vecs()[1].at(1659) - -0.0021799656096845865) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1664) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1664) + "; diff=" + (contribs.vecs()[2].at(1664) - 0.0), Math.abs(contribs.vecs()[2].at(1664) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1664) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1664) + "; diff=" + (contribs.vecs()[4].at(1664) - 0.0), Math.abs(contribs.vecs()[4].at(1664) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1670) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1670) + "; diff=" + (contribs.vecs()[6].at(1670) - -0.0001098226202884689), Math.abs(contribs.vecs()[6].at(1670) - -0.0001098226202884689) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1681) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1681) + "; diff=" + (contribs.vecs()[6].at(1681) - 0.1306278556585312), Math.abs(contribs.vecs()[6].at(1681) - 0.1306278556585312) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1683) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1683) + "; diff=" + (contribs.vecs()[5].at(1683) - -0.0046625640243291855), Math.abs(contribs.vecs()[5].at(1683) - -0.0046625640243291855) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1689) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1689) + "; diff=" + (contribs.vecs()[5].at(1689) - -0.04930208623409271), Math.abs(contribs.vecs()[5].at(1689) - -0.04930208623409271) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1703) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1703) + "; diff=" + (contribs.vecs()[3].at(1703) - -0.10901273787021637), Math.abs(contribs.vecs()[3].at(1703) - -0.10901273787021637) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1710) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1710) + "; diff=" + (contribs.vecs()[4].at(1710) - 0.0), Math.abs(contribs.vecs()[4].at(1710) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1718) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1718) + "; diff=" + (contribs.vecs()[7].at(1718) - 0.0), Math.abs(contribs.vecs()[7].at(1718) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1724) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1724) + "; diff=" + (contribs.vecs()[6].at(1724) - 0.013742632232606411), Math.abs(contribs.vecs()[6].at(1724) - 0.013742632232606411) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1726) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1726) + "; diff=" + (contribs.vecs()[2].at(1726) - 0.0), Math.abs(contribs.vecs()[2].at(1726) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1754) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1754) + "; diff=" + (contribs.vecs()[7].at(1754) - -0.11498484760522842), Math.abs(contribs.vecs()[7].at(1754) - -0.11498484760522842) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1757) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1757) + "; diff=" + (contribs.vecs()[3].at(1757) - -0.48023703694343567), Math.abs(contribs.vecs()[3].at(1757) - -0.48023703694343567) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1762) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1762) + "; diff=" + (contribs.vecs()[5].at(1762) - -0.005961960181593895), Math.abs(contribs.vecs()[5].at(1762) - -0.005961960181593895) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1763) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1763) + "; diff=" + (contribs.vecs()[2].at(1763) - 0.0), Math.abs(contribs.vecs()[2].at(1763) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1763) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1763) + "; diff=" + (contribs.vecs()[3].at(1763) - -0.3179105520248413), Math.abs(contribs.vecs()[3].at(1763) - -0.3179105520248413) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1772) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1772) + "; diff=" + (contribs.vecs()[2].at(1772) - 0.0), Math.abs(contribs.vecs()[2].at(1772) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1777) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1777) + "; diff=" + (contribs.vecs()[1].at(1777) - -0.001197494682855904), Math.abs(contribs.vecs()[1].at(1777) - -0.001197494682855904) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1781) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1781) + "; diff=" + (contribs.vecs()[2].at(1781) - 0.0), Math.abs(contribs.vecs()[2].at(1781) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1795) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1795) + "; diff=" + (contribs.vecs()[6].at(1795) - 1.161454439163208), Math.abs(contribs.vecs()[6].at(1795) - 1.161454439163208) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1797) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1797) + "; diff=" + (contribs.vecs()[0].at(1797) - -0.0018520676530897617), Math.abs(contribs.vecs()[0].at(1797) - -0.0018520676530897617) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1805) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1805) + "; diff=" + (contribs.vecs()[3].at(1805) - 0.18885751068592072), Math.abs(contribs.vecs()[3].at(1805) - 0.18885751068592072) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1825) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1825) + "; diff=" + (contribs.vecs()[1].at(1825) - -0.0005366713739931583), Math.abs(contribs.vecs()[1].at(1825) - -0.0005366713739931583) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1835) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1835) + "; diff=" + (contribs.vecs()[6].at(1835) - -0.04199840500950813), Math.abs(contribs.vecs()[6].at(1835) - -0.04199840500950813) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1840) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1840) + "; diff=" + (contribs.vecs()[7].at(1840) - 0.1172303557395935), Math.abs(contribs.vecs()[7].at(1840) - 0.1172303557395935) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1853) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1853) + "; diff=" + (contribs.vecs()[1].at(1853) - -0.0002167797356378287), Math.abs(contribs.vecs()[1].at(1853) - -0.0002167797356378287) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1860) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1860) + "; diff=" + (contribs.vecs()[1].at(1860) - 0.0008336377213709056), Math.abs(contribs.vecs()[1].at(1860) - 0.0008336377213709056) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1875) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1875) + "; diff=" + (contribs.vecs()[7].at(1875) - -0.002802874194458127), Math.abs(contribs.vecs()[7].at(1875) - -0.002802874194458127) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1876) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1876) + "; diff=" + (contribs.vecs()[7].at(1876) - -0.005018997937440872), Math.abs(contribs.vecs()[7].at(1876) - -0.005018997937440872) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1879) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1879) + "; diff=" + (contribs.vecs()[2].at(1879) - 0.0), Math.abs(contribs.vecs()[2].at(1879) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1883) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1883) + "; diff=" + (contribs.vecs()[7].at(1883) - 0.0), Math.abs(contribs.vecs()[7].at(1883) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1885) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1885) + "; diff=" + (contribs.vecs()[6].at(1885) - -0.007478285580873489), Math.abs(contribs.vecs()[6].at(1885) - -0.007478285580873489) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1888) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1888) + "; diff=" + (contribs.vecs()[6].at(1888) - -0.008211535401642323), Math.abs(contribs.vecs()[6].at(1888) - -0.008211535401642323) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1892) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1892) + "; diff=" + (contribs.vecs()[2].at(1892) - 0.0), Math.abs(contribs.vecs()[2].at(1892) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1897) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1897) + "; diff=" + (contribs.vecs()[6].at(1897) - -0.02180142141878605), Math.abs(contribs.vecs()[6].at(1897) - -0.02180142141878605) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1898) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1898) + "; diff=" + (contribs.vecs()[7].at(1898) - 0.11166847497224808), Math.abs(contribs.vecs()[7].at(1898) - 0.11166847497224808) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1899) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1899) + "; diff=" + (contribs.vecs()[3].at(1899) - 0.0), Math.abs(contribs.vecs()[3].at(1899) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1910) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1910) + "; diff=" + (contribs.vecs()[7].at(1910) - 0.14012974500656128), Math.abs(contribs.vecs()[7].at(1910) - 0.14012974500656128) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1913) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1913) + "; diff=" + (contribs.vecs()[5].at(1913) - 0.02506313845515251), Math.abs(contribs.vecs()[5].at(1913) - 0.02506313845515251) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1923) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1923) + "; diff=" + (contribs.vecs()[1].at(1923) - -0.010175086557865143), Math.abs(contribs.vecs()[1].at(1923) - -0.010175086557865143) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1928) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1928) + "; diff=" + (contribs.vecs()[2].at(1928) - 0.0), Math.abs(contribs.vecs()[2].at(1928) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1937) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1937) + "; diff=" + (contribs.vecs()[5].at(1937) - 0.029316062107682228), Math.abs(contribs.vecs()[5].at(1937) - 0.029316062107682228) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1940) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1940) + "; diff=" + (contribs.vecs()[6].at(1940) - -0.028595473617315292), Math.abs(contribs.vecs()[6].at(1940) - -0.028595473617315292) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1944) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1944) + "; diff=" + (contribs.vecs()[5].at(1944) - -0.02331010065972805), Math.abs(contribs.vecs()[5].at(1944) - -0.02331010065972805) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1960) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1960) + "; diff=" + (contribs.vecs()[0].at(1960) - -0.0011435748310759664), Math.abs(contribs.vecs()[0].at(1960) - -0.0011435748310759664) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2001) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2001) + "; diff=" + (contribs.vecs()[6].at(2001) - 0.011042353697121143), Math.abs(contribs.vecs()[6].at(2001) - 0.011042353697121143) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2014) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2014) + "; diff=" + (contribs.vecs()[3].at(2014) - 0.0), Math.abs(contribs.vecs()[3].at(2014) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2022) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2022) + "; diff=" + (contribs.vecs()[4].at(2022) - -0.14186570048332214), Math.abs(contribs.vecs()[4].at(2022) - -0.14186570048332214) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2023) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2023) + "; diff=" + (contribs.vecs()[6].at(2023) - 0.15967819094657898), Math.abs(contribs.vecs()[6].at(2023) - 0.15967819094657898) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2034) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2034) + "; diff=" + (contribs.vecs()[6].at(2034) - 0.05573802441358566), Math.abs(contribs.vecs()[6].at(2034) - 0.05573802441358566) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2043) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2043) + "; diff=" + (contribs.vecs()[6].at(2043) - 0.049775026738643646), Math.abs(contribs.vecs()[6].at(2043) - 0.049775026738643646) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2049) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2049) + "; diff=" + (contribs.vecs()[5].at(2049) - -0.029783472418785095), Math.abs(contribs.vecs()[5].at(2049) - -0.029783472418785095) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2049) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2049) + "; diff=" + (contribs.vecs()[7].at(2049) - -0.05947677791118622), Math.abs(contribs.vecs()[7].at(2049) - -0.05947677791118622) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2050) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2050) + "; diff=" + (contribs.vecs()[3].at(2050) - -0.3040808141231537), Math.abs(contribs.vecs()[3].at(2050) - -0.3040808141231537) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2052) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2052) + "; diff=" + (contribs.vecs()[6].at(2052) - 0.0), Math.abs(contribs.vecs()[6].at(2052) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2054) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2054) + "; diff=" + (contribs.vecs()[3].at(2054) - 0.0), Math.abs(contribs.vecs()[3].at(2054) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2056) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2056) + "; diff=" + (contribs.vecs()[4].at(2056) - 0.0), Math.abs(contribs.vecs()[4].at(2056) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2064) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2064) + "; diff=" + (contribs.vecs()[5].at(2064) - -0.28235453367233276), Math.abs(contribs.vecs()[5].at(2064) - -0.28235453367233276) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2073) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2073) + "; diff=" + (contribs.vecs()[7].at(2073) - -0.16449123620986938), Math.abs(contribs.vecs()[7].at(2073) - -0.16449123620986938) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2089) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2089) + "; diff=" + (contribs.vecs()[1].at(2089) - 1.3773797945759725e-05), Math.abs(contribs.vecs()[1].at(2089) - 1.3773797945759725e-05) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2116) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2116) + "; diff=" + (contribs.vecs()[5].at(2116) - 0.04062234237790108), Math.abs(contribs.vecs()[5].at(2116) - 0.04062234237790108) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2118) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2118) + "; diff=" + (contribs.vecs()[0].at(2118) - 0.014691373333334923), Math.abs(contribs.vecs()[0].at(2118) - 0.014691373333334923) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2124) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2124) + "; diff=" + (contribs.vecs()[1].at(2124) - -0.010728556662797928), Math.abs(contribs.vecs()[1].at(2124) - -0.010728556662797928) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2150) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2150) + "; diff=" + (contribs.vecs()[7].at(2150) - 0.16496345400810242), Math.abs(contribs.vecs()[7].at(2150) - 0.16496345400810242) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2154) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2154) + "; diff=" + (contribs.vecs()[0].at(2154) - 0.0022897713351994753), Math.abs(contribs.vecs()[0].at(2154) - 0.0022897713351994753) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2159) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2159) + "; diff=" + (contribs.vecs()[2].at(2159) - 0.0), Math.abs(contribs.vecs()[2].at(2159) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2160) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2160) + "; diff=" + (contribs.vecs()[1].at(2160) - -0.007288633845746517), Math.abs(contribs.vecs()[1].at(2160) - -0.007288633845746517) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2168) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2168) + "; diff=" + (contribs.vecs()[7].at(2168) - 0.1368570774793625), Math.abs(contribs.vecs()[7].at(2168) - 0.1368570774793625) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2170) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2170) + "; diff=" + (contribs.vecs()[1].at(2170) - -0.0006547279190272093), Math.abs(contribs.vecs()[1].at(2170) - -0.0006547279190272093) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2176) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2176) + "; diff=" + (contribs.vecs()[1].at(2176) - 0.0), Math.abs(contribs.vecs()[1].at(2176) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2177) + "; diff=" + (contribs.vecs()[2].at(2177) - 0.0), Math.abs(contribs.vecs()[2].at(2177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2180) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2180) + "; diff=" + (contribs.vecs()[3].at(2180) - 0.09390239417552948), Math.abs(contribs.vecs()[3].at(2180) - 0.09390239417552948) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2185) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2185) + "; diff=" + (contribs.vecs()[2].at(2185) - 0.0), Math.abs(contribs.vecs()[2].at(2185) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2190) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2190) + "; diff=" + (contribs.vecs()[0].at(2190) - 0.004904119297862053), Math.abs(contribs.vecs()[0].at(2190) - 0.004904119297862053) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2191) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2191) + "; diff=" + (contribs.vecs()[4].at(2191) - 0.09638641029596329), Math.abs(contribs.vecs()[4].at(2191) - 0.09638641029596329) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2192) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2192) + "; diff=" + (contribs.vecs()[4].at(2192) - 0.07950587570667267), Math.abs(contribs.vecs()[4].at(2192) - 0.07950587570667267) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2194) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2194) + "; diff=" + (contribs.vecs()[4].at(2194) - -0.1291104257106781), Math.abs(contribs.vecs()[4].at(2194) - -0.1291104257106781) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2215) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2215) + "; diff=" + (contribs.vecs()[2].at(2215) - 0.0), Math.abs(contribs.vecs()[2].at(2215) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2223) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2223) + "; diff=" + (contribs.vecs()[6].at(2223) - -0.03995045647025108), Math.abs(contribs.vecs()[6].at(2223) - -0.03995045647025108) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2230) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2230) + "; diff=" + (contribs.vecs()[5].at(2230) - -0.0185830220580101), Math.abs(contribs.vecs()[5].at(2230) - -0.0185830220580101) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2242) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2242) + "; diff=" + (contribs.vecs()[1].at(2242) - -0.0031049144454300404), Math.abs(contribs.vecs()[1].at(2242) - -0.0031049144454300404) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2245) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2245) + "; diff=" + (contribs.vecs()[1].at(2245) - -0.002147829160094261), Math.abs(contribs.vecs()[1].at(2245) - -0.002147829160094261) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2245) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2245) + "; diff=" + (contribs.vecs()[3].at(2245) - -0.39321041107177734), Math.abs(contribs.vecs()[3].at(2245) - -0.39321041107177734) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2248) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2248) + "; diff=" + (contribs.vecs()[7].at(2248) - 0.1401585042476654), Math.abs(contribs.vecs()[7].at(2248) - 0.1401585042476654) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2252) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2252) + "; diff=" + (contribs.vecs()[3].at(2252) - 0.0), Math.abs(contribs.vecs()[3].at(2252) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2253) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2253) + "; diff=" + (contribs.vecs()[5].at(2253) - 0.012461811304092407), Math.abs(contribs.vecs()[5].at(2253) - 0.012461811304092407) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2263) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2263) + "; diff=" + (contribs.vecs()[3].at(2263) - -0.2688072621822357), Math.abs(contribs.vecs()[3].at(2263) - -0.2688072621822357) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2263) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2263) + "; diff=" + (contribs.vecs()[5].at(2263) - 0.022626684978604317), Math.abs(contribs.vecs()[5].at(2263) - 0.022626684978604317) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2267) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2267) + "; diff=" + (contribs.vecs()[0].at(2267) - 0.014528692699968815), Math.abs(contribs.vecs()[0].at(2267) - 0.014528692699968815) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2272) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2272) + "; diff=" + (contribs.vecs()[4].at(2272) - 0.0), Math.abs(contribs.vecs()[4].at(2272) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2302) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2302) + "; diff=" + (contribs.vecs()[2].at(2302) - -0.0977664515376091), Math.abs(contribs.vecs()[2].at(2302) - -0.0977664515376091) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2310) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2310) + "; diff=" + (contribs.vecs()[2].at(2310) - 0.0), Math.abs(contribs.vecs()[2].at(2310) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2314) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2314) + "; diff=" + (contribs.vecs()[3].at(2314) - 0.0), Math.abs(contribs.vecs()[3].at(2314) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2326) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2326) + "; diff=" + (contribs.vecs()[0].at(2326) - 0.015440425835549831), Math.abs(contribs.vecs()[0].at(2326) - 0.015440425835549831) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2326) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2326) + "; diff=" + (contribs.vecs()[6].at(2326) - 0.015849005430936813), Math.abs(contribs.vecs()[6].at(2326) - 0.015849005430936813) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2330) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2330) + "; diff=" + (contribs.vecs()[0].at(2330) - 0.01534988358616829), Math.abs(contribs.vecs()[0].at(2330) - 0.01534988358616829) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2330) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2330) + "; diff=" + (contribs.vecs()[4].at(2330) - 0.0), Math.abs(contribs.vecs()[4].at(2330) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2332) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2332) + "; diff=" + (contribs.vecs()[2].at(2332) - 0.0), Math.abs(contribs.vecs()[2].at(2332) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2333) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2333) + "; diff=" + (contribs.vecs()[3].at(2333) - 0.269255667924881), Math.abs(contribs.vecs()[3].at(2333) - 0.269255667924881) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2337) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2337) + "; diff=" + (contribs.vecs()[0].at(2337) - 0.012155788950622082), Math.abs(contribs.vecs()[0].at(2337) - 0.012155788950622082) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2339) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2339) + "; diff=" + (contribs.vecs()[5].at(2339) - -0.06183215230703354), Math.abs(contribs.vecs()[5].at(2339) - -0.06183215230703354) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2341) + "; diff=" + (contribs.vecs()[1].at(2341) - -0.003940714988857508), Math.abs(contribs.vecs()[1].at(2341) - -0.003940714988857508) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2363) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2363) + "; diff=" + (contribs.vecs()[6].at(2363) - 0.0), Math.abs(contribs.vecs()[6].at(2363) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2383) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2383) + "; diff=" + (contribs.vecs()[6].at(2383) - 0.0), Math.abs(contribs.vecs()[6].at(2383) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2403) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2403) + "; diff=" + (contribs.vecs()[5].at(2403) - -0.00791538879275322), Math.abs(contribs.vecs()[5].at(2403) - -0.00791538879275322) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2406) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2406) + "; diff=" + (contribs.vecs()[3].at(2406) - 0.0), Math.abs(contribs.vecs()[3].at(2406) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2424) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2424) + "; diff=" + (contribs.vecs()[5].at(2424) - 0.14997421205043793), Math.abs(contribs.vecs()[5].at(2424) - 0.14997421205043793) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2426) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2426) + "; diff=" + (contribs.vecs()[0].at(2426) - 0.01698547787964344), Math.abs(contribs.vecs()[0].at(2426) - 0.01698547787964344) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2430) + "; diff=" + (contribs.vecs()[4].at(2430) - -0.11966871470212936), Math.abs(contribs.vecs()[4].at(2430) - -0.11966871470212936) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2437) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2437) + "; diff=" + (contribs.vecs()[6].at(2437) - 0.11984168738126755), Math.abs(contribs.vecs()[6].at(2437) - 0.11984168738126755) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2440) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2440) + "; diff=" + (contribs.vecs()[4].at(2440) - 0.0), Math.abs(contribs.vecs()[4].at(2440) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2442) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2442) + "; diff=" + (contribs.vecs()[1].at(2442) - 0.0009033792885020375), Math.abs(contribs.vecs()[1].at(2442) - 0.0009033792885020375) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2444) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2444) + "; diff=" + (contribs.vecs()[5].at(2444) - 0.18670323491096497), Math.abs(contribs.vecs()[5].at(2444) - 0.18670323491096497) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2446) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2446) + "; diff=" + (contribs.vecs()[5].at(2446) - 0.12360533326864243), Math.abs(contribs.vecs()[5].at(2446) - 0.12360533326864243) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2447) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2447) + "; diff=" + (contribs.vecs()[5].at(2447) - 0.09203118085861206), Math.abs(contribs.vecs()[5].at(2447) - 0.09203118085861206) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2452) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2452) + "; diff=" + (contribs.vecs()[7].at(2452) - 0.1855335831642151), Math.abs(contribs.vecs()[7].at(2452) - 0.1855335831642151) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2455) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2455) + "; diff=" + (contribs.vecs()[0].at(2455) - 0.0029488878790289164), Math.abs(contribs.vecs()[0].at(2455) - 0.0029488878790289164) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2469) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2469) + "; diff=" + (contribs.vecs()[2].at(2469) - -0.07727140188217163), Math.abs(contribs.vecs()[2].at(2469) - -0.07727140188217163) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2474) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2474) + "; diff=" + (contribs.vecs()[6].at(2474) - 0.005472847260534763), Math.abs(contribs.vecs()[6].at(2474) - 0.005472847260534763) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2483) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2483) + "; diff=" + (contribs.vecs()[5].at(2483) - 0.014722280204296112), Math.abs(contribs.vecs()[5].at(2483) - 0.014722280204296112) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2484) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2484) + "; diff=" + (contribs.vecs()[3].at(2484) - 0.2706046402454376), Math.abs(contribs.vecs()[3].at(2484) - 0.2706046402454376) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2492) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2492) + "; diff=" + (contribs.vecs()[0].at(2492) - 0.0013094069436192513), Math.abs(contribs.vecs()[0].at(2492) - 0.0013094069436192513) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2502) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2502) + "; diff=" + (contribs.vecs()[5].at(2502) - -0.168378084897995), Math.abs(contribs.vecs()[5].at(2502) - -0.168378084897995) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2510) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2510) + "; diff=" + (contribs.vecs()[1].at(2510) - 0.006395938340574503), Math.abs(contribs.vecs()[1].at(2510) - 0.006395938340574503) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2524) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2524) + "; diff=" + (contribs.vecs()[1].at(2524) - 0.0005423594266176224), Math.abs(contribs.vecs()[1].at(2524) - 0.0005423594266176224) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2526) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2526) + "; diff=" + (contribs.vecs()[0].at(2526) - 0.004384635481983423), Math.abs(contribs.vecs()[0].at(2526) - 0.004384635481983423) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2536) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2536) + "; diff=" + (contribs.vecs()[0].at(2536) - 0.01582752726972103), Math.abs(contribs.vecs()[0].at(2536) - 0.01582752726972103) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2543) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2543) + "; diff=" + (contribs.vecs()[6].at(2543) - 0.026056285947561264), Math.abs(contribs.vecs()[6].at(2543) - 0.026056285947561264) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2548) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2548) + "; diff=" + (contribs.vecs()[7].at(2548) - 0.0), Math.abs(contribs.vecs()[7].at(2548) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2551) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2551) + "; diff=" + (contribs.vecs()[4].at(2551) - -0.056843940168619156), Math.abs(contribs.vecs()[4].at(2551) - -0.056843940168619156) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2555) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2555) + "; diff=" + (contribs.vecs()[7].at(2555) - -0.04799283668398857), Math.abs(contribs.vecs()[7].at(2555) - -0.04799283668398857) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2557) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2557) + "; diff=" + (contribs.vecs()[1].at(2557) - -0.002549258526414633), Math.abs(contribs.vecs()[1].at(2557) - -0.002549258526414633) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2560) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2560) + "; diff=" + (contribs.vecs()[0].at(2560) - 0.004619837272912264), Math.abs(contribs.vecs()[0].at(2560) - 0.004619837272912264) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2561) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2561) + "; diff=" + (contribs.vecs()[7].at(2561) - 0.0), Math.abs(contribs.vecs()[7].at(2561) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2566) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2566) + "; diff=" + (contribs.vecs()[0].at(2566) - 0.035680223256349564), Math.abs(contribs.vecs()[0].at(2566) - 0.035680223256349564) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2570) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2570) + "; diff=" + (contribs.vecs()[1].at(2570) - -0.017693351954221725), Math.abs(contribs.vecs()[1].at(2570) - -0.017693351954221725) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2572) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2572) + "; diff=" + (contribs.vecs()[4].at(2572) - 0.0), Math.abs(contribs.vecs()[4].at(2572) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2573) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2573) + "; diff=" + (contribs.vecs()[0].at(2573) - 0.018375953659415245), Math.abs(contribs.vecs()[0].at(2573) - 0.018375953659415245) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2575) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2575) + "; diff=" + (contribs.vecs()[1].at(2575) - 0.0023021171800792217), Math.abs(contribs.vecs()[1].at(2575) - 0.0023021171800792217) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2577) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2577) + "; diff=" + (contribs.vecs()[7].at(2577) - 0.43035608530044556), Math.abs(contribs.vecs()[7].at(2577) - 0.43035608530044556) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2581) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2581) + "; diff=" + (contribs.vecs()[6].at(2581) - 0.034452684223651886), Math.abs(contribs.vecs()[6].at(2581) - 0.034452684223651886) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2584) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2584) + "; diff=" + (contribs.vecs()[1].at(2584) - -0.006879712454974651), Math.abs(contribs.vecs()[1].at(2584) - -0.006879712454974651) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2588) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2588) + "; diff=" + (contribs.vecs()[2].at(2588) - 0.0), Math.abs(contribs.vecs()[2].at(2588) - 0.0) < eps);


    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }


  @Test
  public void testRegressionTanhDeepSHAPComparison() {
    Frame fr = parseTestFile("smalldata/prostate/prostate.csv");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    double eps = 1e-4;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "AGE";
      params._hidden = ari(3, 3);
      params._seed = 123456;
      params._reproducible = true;
      params._activation = DeepLearningParameters.Activation.Tanh;


      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions()
                      .setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original)
                      .setOutputPerReference(true),
              bgFr);

      //toPyTorch(model, test, bgFr);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(5) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(5) + "; diff=" + (contribs.vecs()[0].at(5) - -0.3580889105796814), Math.abs(contribs.vecs()[0].at(5) - -0.3580889105796814) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(14) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(14) + "; diff=" + (contribs.vecs()[5].at(14) - -2.5539872646331787), Math.abs(contribs.vecs()[5].at(14) - -2.5539872646331787) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(25) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(25) + "; diff=" + (contribs.vecs()[6].at(25) - 0.3793828785419464), Math.abs(contribs.vecs()[6].at(25) - 0.3793828785419464) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(29) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(29) + "; diff=" + (contribs.vecs()[0].at(29) - -0.16547757387161255), Math.abs(contribs.vecs()[0].at(29) - -0.16547757387161255) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(30) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(30) + "; diff=" + (contribs.vecs()[7].at(30) - 0.4034125506877899), Math.abs(contribs.vecs()[7].at(30) - 0.4034125506877899) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(33) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(33) + "; diff=" + (contribs.vecs()[6].at(33) - 0.4620344042778015), Math.abs(contribs.vecs()[6].at(33) - 0.4620344042778015) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(34) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(34) + "; diff=" + (contribs.vecs()[3].at(34) - -0.7360246777534485), Math.abs(contribs.vecs()[3].at(34) - -0.7360246777534485) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(39) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(39) + "; diff=" + (contribs.vecs()[0].at(39) - -0.09453216940164566), Math.abs(contribs.vecs()[0].at(39) - -0.09453216940164566) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(42) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(42) + "; diff=" + (contribs.vecs()[5].at(42) - -2.6938116550445557), Math.abs(contribs.vecs()[5].at(42) - -2.6938116550445557) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(42) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(42) + "; diff=" + (contribs.vecs()[6].at(42) - 0.32873302698135376), Math.abs(contribs.vecs()[6].at(42) - 0.32873302698135376) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(46) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(46) + "; diff=" + (contribs.vecs()[1].at(46) - 0.16295106709003448), Math.abs(contribs.vecs()[1].at(46) - 0.16295106709003448) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(48) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(48) + "; diff=" + (contribs.vecs()[1].at(48) - 0.07559704780578613), Math.abs(contribs.vecs()[1].at(48) - 0.07559704780578613) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(65) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(65) + "; diff=" + (contribs.vecs()[3].at(65) - -0.8830930590629578), Math.abs(contribs.vecs()[3].at(65) - -0.8830930590629578) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(68) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(68) + "; diff=" + (contribs.vecs()[6].at(68) - 0.4824211597442627), Math.abs(contribs.vecs()[6].at(68) - 0.4824211597442627) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(70) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(70) + "; diff=" + (contribs.vecs()[4].at(70) - -0.011373836547136307), Math.abs(contribs.vecs()[4].at(70) - -0.011373836547136307) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(77) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(77) + "; diff=" + (contribs.vecs()[2].at(77) - 0.0), Math.abs(contribs.vecs()[2].at(77) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(78) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(78) + "; diff=" + (contribs.vecs()[4].at(78) - 0.0), Math.abs(contribs.vecs()[4].at(78) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(87) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(87) + "; diff=" + (contribs.vecs()[1].at(87) - 0.0), Math.abs(contribs.vecs()[1].at(87) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(94) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(94) + "; diff=" + (contribs.vecs()[1].at(94) - 0.0), Math.abs(contribs.vecs()[1].at(94) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(97) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(97) + "; diff=" + (contribs.vecs()[1].at(97) - 0.0), Math.abs(contribs.vecs()[1].at(97) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(100) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(100) + "; diff=" + (contribs.vecs()[5].at(100) - -0.07253534346818924), Math.abs(contribs.vecs()[5].at(100) - -0.07253534346818924) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(121) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(121) + "; diff=" + (contribs.vecs()[0].at(121) - -0.20580261945724487), Math.abs(contribs.vecs()[0].at(121) - -0.20580261945724487) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(121) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(121) + "; diff=" + (contribs.vecs()[6].at(121) - -0.16214485466480255), Math.abs(contribs.vecs()[6].at(121) - -0.16214485466480255) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(122) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(122) + "; diff=" + (contribs.vecs()[2].at(122) - 0.0), Math.abs(contribs.vecs()[2].at(122) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(132) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(132) + "; diff=" + (contribs.vecs()[0].at(132) - -0.14933991432189941), Math.abs(contribs.vecs()[0].at(132) - -0.14933991432189941) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(135) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(135) + "; diff=" + (contribs.vecs()[4].at(135) - -0.2943253815174103), Math.abs(contribs.vecs()[4].at(135) - -0.2943253815174103) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(150) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(150) + "; diff=" + (contribs.vecs()[1].at(150) - 0.10797140002250671), Math.abs(contribs.vecs()[1].at(150) - 0.10797140002250671) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(158) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(158) + "; diff=" + (contribs.vecs()[2].at(158) - 0.0), Math.abs(contribs.vecs()[2].at(158) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(168) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(168) + "; diff=" + (contribs.vecs()[6].at(168) - 0.6330541372299194), Math.abs(contribs.vecs()[6].at(168) - 0.6330541372299194) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(170) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(170) + "; diff=" + (contribs.vecs()[3].at(170) - -0.48850032687187195), Math.abs(contribs.vecs()[3].at(170) - -0.48850032687187195) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(171) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(171) + "; diff=" + (contribs.vecs()[6].at(171) - 0.4266732335090637), Math.abs(contribs.vecs()[6].at(171) - 0.4266732335090637) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(173) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(173) + "; diff=" + (contribs.vecs()[2].at(173) - 0.0), Math.abs(contribs.vecs()[2].at(173) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(177) + "; diff=" + (contribs.vecs()[4].at(177) - 0.0), Math.abs(contribs.vecs()[4].at(177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(177) + "; diff=" + (contribs.vecs()[7].at(177) - 0.0), Math.abs(contribs.vecs()[7].at(177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(191) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(191) + "; diff=" + (contribs.vecs()[1].at(191) - -0.09747468680143356), Math.abs(contribs.vecs()[1].at(191) - -0.09747468680143356) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(193) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(193) + "; diff=" + (contribs.vecs()[7].at(193) - 0.0), Math.abs(contribs.vecs()[7].at(193) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(201) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(201) + "; diff=" + (contribs.vecs()[6].at(201) - -0.1346847116947174), Math.abs(contribs.vecs()[6].at(201) - -0.1346847116947174) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(203) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(203) + "; diff=" + (contribs.vecs()[7].at(203) - 0.4949609041213989), Math.abs(contribs.vecs()[7].at(203) - 0.4949609041213989) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(208) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(208) + "; diff=" + (contribs.vecs()[1].at(208) - 0.14081238210201263), Math.abs(contribs.vecs()[1].at(208) - 0.14081238210201263) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(209) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(209) + "; diff=" + (contribs.vecs()[6].at(209) - -0.29288148880004883), Math.abs(contribs.vecs()[6].at(209) - -0.29288148880004883) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(211) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(211) + "; diff=" + (contribs.vecs()[7].at(211) - 0.0), Math.abs(contribs.vecs()[7].at(211) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(213) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(213) + "; diff=" + (contribs.vecs()[1].at(213) - 0.3521924912929535), Math.abs(contribs.vecs()[1].at(213) - 0.3521924912929535) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(233) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(233) + "; diff=" + (contribs.vecs()[1].at(233) - 0.3087965250015259), Math.abs(contribs.vecs()[1].at(233) - 0.3087965250015259) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(235) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(235) + "; diff=" + (contribs.vecs()[0].at(235) - -0.15379741787910461), Math.abs(contribs.vecs()[0].at(235) - -0.15379741787910461) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(236) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(236) + "; diff=" + (contribs.vecs()[6].at(236) - 0.028442339971661568), Math.abs(contribs.vecs()[6].at(236) - 0.028442339971661568) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(238) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(238) + "; diff=" + (contribs.vecs()[4].at(238) - 0.0), Math.abs(contribs.vecs()[4].at(238) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(240) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(240) + "; diff=" + (contribs.vecs()[4].at(240) - 0.0), Math.abs(contribs.vecs()[4].at(240) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(252) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(252) + "; diff=" + (contribs.vecs()[3].at(252) - -0.4089759290218353), Math.abs(contribs.vecs()[3].at(252) - -0.4089759290218353) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(263) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(263) + "; diff=" + (contribs.vecs()[6].at(263) - -0.27042704820632935), Math.abs(contribs.vecs()[6].at(263) - -0.27042704820632935) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(276) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(276) + "; diff=" + (contribs.vecs()[7].at(276) - 0.0), Math.abs(contribs.vecs()[7].at(276) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(280) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(280) + "; diff=" + (contribs.vecs()[4].at(280) - 0.0), Math.abs(contribs.vecs()[4].at(280) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(289) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(289) + "; diff=" + (contribs.vecs()[5].at(289) - -0.730988621711731), Math.abs(contribs.vecs()[5].at(289) - -0.730988621711731) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(296) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(296) + "; diff=" + (contribs.vecs()[2].at(296) - 0.0), Math.abs(contribs.vecs()[2].at(296) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(297) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(297) + "; diff=" + (contribs.vecs()[6].at(297) - 0.270845890045166), Math.abs(contribs.vecs()[6].at(297) - 0.270845890045166) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(299) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(299) + "; diff=" + (contribs.vecs()[6].at(299) - -0.3666711747646332), Math.abs(contribs.vecs()[6].at(299) - -0.3666711747646332) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(328) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(328) + "; diff=" + (contribs.vecs()[4].at(328) - 0.0), Math.abs(contribs.vecs()[4].at(328) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(337) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(337) + "; diff=" + (contribs.vecs()[4].at(337) - 0.0), Math.abs(contribs.vecs()[4].at(337) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(341) + "; diff=" + (contribs.vecs()[7].at(341) - 0.0), Math.abs(contribs.vecs()[7].at(341) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(346) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(346) + "; diff=" + (contribs.vecs()[1].at(346) - 0.0), Math.abs(contribs.vecs()[1].at(346) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(348) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(348) + "; diff=" + (contribs.vecs()[0].at(348) - -0.12492985278367996), Math.abs(contribs.vecs()[0].at(348) - -0.12492985278367996) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(349) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(349) + "; diff=" + (contribs.vecs()[3].at(349) - -0.4288831055164337), Math.abs(contribs.vecs()[3].at(349) - -0.4288831055164337) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(367) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(367) + "; diff=" + (contribs.vecs()[1].at(367) - -0.3382520377635956), Math.abs(contribs.vecs()[1].at(367) - -0.3382520377635956) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(368) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(368) + "; diff=" + (contribs.vecs()[4].at(368) - -0.48246869444847107), Math.abs(contribs.vecs()[4].at(368) - -0.48246869444847107) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(375) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(375) + "; diff=" + (contribs.vecs()[5].at(375) - -0.00949004478752613), Math.abs(contribs.vecs()[5].at(375) - -0.00949004478752613) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(377) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(377) + "; diff=" + (contribs.vecs()[6].at(377) - 0.43652570247650146), Math.abs(contribs.vecs()[6].at(377) - 0.43652570247650146) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(380) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(380) + "; diff=" + (contribs.vecs()[6].at(380) - 0.4965631067752838), Math.abs(contribs.vecs()[6].at(380) - 0.4965631067752838) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(391) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(391) + "; diff=" + (contribs.vecs()[7].at(391) - 0.0), Math.abs(contribs.vecs()[7].at(391) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(397) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(397) + "; diff=" + (contribs.vecs()[1].at(397) - -0.23866517841815948), Math.abs(contribs.vecs()[1].at(397) - -0.23866517841815948) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(397) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(397) + "; diff=" + (contribs.vecs()[2].at(397) - 0.0), Math.abs(contribs.vecs()[2].at(397) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(400) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(400) + "; diff=" + (contribs.vecs()[5].at(400) - 0.10691917687654495), Math.abs(contribs.vecs()[5].at(400) - 0.10691917687654495) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(400) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(400) + "; diff=" + (contribs.vecs()[7].at(400) - -0.3860916495323181), Math.abs(contribs.vecs()[7].at(400) - -0.3860916495323181) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(402) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(402) + "; diff=" + (contribs.vecs()[5].at(402) - 0.8488811254501343), Math.abs(contribs.vecs()[5].at(402) - 0.8488811254501343) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(409) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(409) + "; diff=" + (contribs.vecs()[2].at(409) - 0.0), Math.abs(contribs.vecs()[2].at(409) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(417) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(417) + "; diff=" + (contribs.vecs()[5].at(417) - 0.17525191605091095), Math.abs(contribs.vecs()[5].at(417) - 0.17525191605091095) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(423) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(423) + "; diff=" + (contribs.vecs()[6].at(423) - 0.9975223541259766), Math.abs(contribs.vecs()[6].at(423) - 0.9975223541259766) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(439) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(439) + "; diff=" + (contribs.vecs()[2].at(439) - 0.0), Math.abs(contribs.vecs()[2].at(439) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(439) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(439) + "; diff=" + (contribs.vecs()[6].at(439) - -0.12148309499025345), Math.abs(contribs.vecs()[6].at(439) - -0.12148309499025345) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(440) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(440) + "; diff=" + (contribs.vecs()[6].at(440) - 0.012547098100185394), Math.abs(contribs.vecs()[6].at(440) - 0.012547098100185394) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(446) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(446) + "; diff=" + (contribs.vecs()[6].at(446) - 0.21073707938194275), Math.abs(contribs.vecs()[6].at(446) - 0.21073707938194275) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(447) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(447) + "; diff=" + (contribs.vecs()[7].at(447) - 0.13029076159000397), Math.abs(contribs.vecs()[7].at(447) - 0.13029076159000397) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(452) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(452) + "; diff=" + (contribs.vecs()[2].at(452) - 1.7142770290374756), Math.abs(contribs.vecs()[2].at(452) - 1.7142770290374756) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(457) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(457) + "; diff=" + (contribs.vecs()[6].at(457) - 0.10284071415662766), Math.abs(contribs.vecs()[6].at(457) - 0.10284071415662766) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(466) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(466) + "; diff=" + (contribs.vecs()[6].at(466) - 0.3274780511856079), Math.abs(contribs.vecs()[6].at(466) - 0.3274780511856079) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(466) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(466) + "; diff=" + (contribs.vecs()[7].at(466) - -0.1420784294605255), Math.abs(contribs.vecs()[7].at(466) - -0.1420784294605255) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(482) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(482) + "; diff=" + (contribs.vecs()[0].at(482) - -0.059232842177152634), Math.abs(contribs.vecs()[0].at(482) - -0.059232842177152634) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(485) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(485) + "; diff=" + (contribs.vecs()[1].at(485) - 0.0), Math.abs(contribs.vecs()[1].at(485) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(492) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(492) + "; diff=" + (contribs.vecs()[3].at(492) - 0.33906880021095276), Math.abs(contribs.vecs()[3].at(492) - 0.33906880021095276) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(493) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(493) + "; diff=" + (contribs.vecs()[0].at(493) - -0.09802514314651489), Math.abs(contribs.vecs()[0].at(493) - -0.09802514314651489) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(494) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(494) + "; diff=" + (contribs.vecs()[2].at(494) - 0.0), Math.abs(contribs.vecs()[2].at(494) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(512) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(512) + "; diff=" + (contribs.vecs()[5].at(512) - -0.03956804797053337), Math.abs(contribs.vecs()[5].at(512) - -0.03956804797053337) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(522) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(522) + "; diff=" + (contribs.vecs()[5].at(522) - 0.294601708650589), Math.abs(contribs.vecs()[5].at(522) - 0.294601708650589) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(534) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(534) + "; diff=" + (contribs.vecs()[7].at(534) - -0.09052468091249466), Math.abs(contribs.vecs()[7].at(534) - -0.09052468091249466) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(573) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(573) + "; diff=" + (contribs.vecs()[6].at(573) - -0.02630389854311943), Math.abs(contribs.vecs()[6].at(573) - -0.02630389854311943) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(578) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(578) + "; diff=" + (contribs.vecs()[4].at(578) - 0.0), Math.abs(contribs.vecs()[4].at(578) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(586) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(586) + "; diff=" + (contribs.vecs()[4].at(586) - 0.0), Math.abs(contribs.vecs()[4].at(586) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(590) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(590) + "; diff=" + (contribs.vecs()[3].at(590) - 1.467305302619934), Math.abs(contribs.vecs()[3].at(590) - 1.467305302619934) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(593) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(593) + "; diff=" + (contribs.vecs()[3].at(593) - 1.4871798753738403), Math.abs(contribs.vecs()[3].at(593) - 1.4871798753738403) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(595) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(595) + "; diff=" + (contribs.vecs()[2].at(595) - 0.0), Math.abs(contribs.vecs()[2].at(595) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(601) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(601) + "; diff=" + (contribs.vecs()[6].at(601) - 0.03095083311200142), Math.abs(contribs.vecs()[6].at(601) - 0.03095083311200142) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(620) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(620) + "; diff=" + (contribs.vecs()[1].at(620) - 0.0), Math.abs(contribs.vecs()[1].at(620) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(630) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(630) + "; diff=" + (contribs.vecs()[7].at(630) - 0.5364130735397339), Math.abs(contribs.vecs()[7].at(630) - 0.5364130735397339) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(631) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(631) + "; diff=" + (contribs.vecs()[1].at(631) - -0.35077521204948425), Math.abs(contribs.vecs()[1].at(631) - -0.35077521204948425) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(633) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(633) + "; diff=" + (contribs.vecs()[4].at(633) - 0.0), Math.abs(contribs.vecs()[4].at(633) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(646) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(646) + "; diff=" + (contribs.vecs()[6].at(646) - -0.5542057156562805), Math.abs(contribs.vecs()[6].at(646) - -0.5542057156562805) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(649) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(649) + "; diff=" + (contribs.vecs()[3].at(649) - -0.599361002445221), Math.abs(contribs.vecs()[3].at(649) - -0.599361002445221) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(654) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(654) + "; diff=" + (contribs.vecs()[5].at(654) - 0.3060382008552551), Math.abs(contribs.vecs()[5].at(654) - 0.3060382008552551) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(655) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(655) + "; diff=" + (contribs.vecs()[1].at(655) - 0.0), Math.abs(contribs.vecs()[1].at(655) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(659) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(659) + "; diff=" + (contribs.vecs()[7].at(659) - -0.03527200594544411), Math.abs(contribs.vecs()[7].at(659) - -0.03527200594544411) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(660) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(660) + "; diff=" + (contribs.vecs()[5].at(660) - -0.12618795037269592), Math.abs(contribs.vecs()[5].at(660) - -0.12618795037269592) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(662) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(662) + "; diff=" + (contribs.vecs()[3].at(662) - -0.7737846374511719), Math.abs(contribs.vecs()[3].at(662) - -0.7737846374511719) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(666) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(666) + "; diff=" + (contribs.vecs()[4].at(666) - 0.0), Math.abs(contribs.vecs()[4].at(666) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(681) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(681) + "; diff=" + (contribs.vecs()[7].at(681) - 0.6601803302764893), Math.abs(contribs.vecs()[7].at(681) - 0.6601803302764893) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(683) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(683) + "; diff=" + (contribs.vecs()[2].at(683) - 0.0), Math.abs(contribs.vecs()[2].at(683) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(684) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(684) + "; diff=" + (contribs.vecs()[0].at(684) - -0.1798228621482849), Math.abs(contribs.vecs()[0].at(684) - -0.1798228621482849) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(692) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(692) + "; diff=" + (contribs.vecs()[3].at(692) - 0.0), Math.abs(contribs.vecs()[3].at(692) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(698) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(698) + "; diff=" + (contribs.vecs()[7].at(698) - 0.0), Math.abs(contribs.vecs()[7].at(698) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(699) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(699) + "; diff=" + (contribs.vecs()[6].at(699) - 0.0), Math.abs(contribs.vecs()[6].at(699) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(700) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(700) + "; diff=" + (contribs.vecs()[5].at(700) - -0.19213362038135529), Math.abs(contribs.vecs()[5].at(700) - -0.19213362038135529) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(704) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(704) + "; diff=" + (contribs.vecs()[0].at(704) - -0.17435787618160248), Math.abs(contribs.vecs()[0].at(704) - -0.17435787618160248) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(705) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(705) + "; diff=" + (contribs.vecs()[7].at(705) - 0.5470529198646545), Math.abs(contribs.vecs()[7].at(705) - 0.5470529198646545) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(706) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(706) + "; diff=" + (contribs.vecs()[7].at(706) - 0.0), Math.abs(contribs.vecs()[7].at(706) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(711) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(711) + "; diff=" + (contribs.vecs()[1].at(711) - 0.009711022488772869), Math.abs(contribs.vecs()[1].at(711) - 0.009711022488772869) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(712) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(712) + "; diff=" + (contribs.vecs()[2].at(712) - 0.0), Math.abs(contribs.vecs()[2].at(712) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(718) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(718) + "; diff=" + (contribs.vecs()[0].at(718) - -0.37298160791397095), Math.abs(contribs.vecs()[0].at(718) - -0.37298160791397095) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(734) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(734) + "; diff=" + (contribs.vecs()[7].at(734) - 0.0), Math.abs(contribs.vecs()[7].at(734) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(737) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(737) + "; diff=" + (contribs.vecs()[5].at(737) - 0.02823474071919918), Math.abs(contribs.vecs()[5].at(737) - 0.02823474071919918) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(740) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(740) + "; diff=" + (contribs.vecs()[5].at(740) - -0.20227950811386108), Math.abs(contribs.vecs()[5].at(740) - -0.20227950811386108) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(750) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(750) + "; diff=" + (contribs.vecs()[1].at(750) - -0.016940223053097725), Math.abs(contribs.vecs()[1].at(750) - -0.016940223053097725) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(753) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(753) + "; diff=" + (contribs.vecs()[5].at(753) - -0.18892410397529602), Math.abs(contribs.vecs()[5].at(753) - -0.18892410397529602) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(753) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(753) + "; diff=" + (contribs.vecs()[6].at(753) - -0.2586790919303894), Math.abs(contribs.vecs()[6].at(753) - -0.2586790919303894) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(759) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(759) + "; diff=" + (contribs.vecs()[2].at(759) - -0.39209118485450745), Math.abs(contribs.vecs()[2].at(759) - -0.39209118485450745) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(759) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(759) + "; diff=" + (contribs.vecs()[4].at(759) - -0.17012961208820343), Math.abs(contribs.vecs()[4].at(759) - -0.17012961208820343) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(762) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(762) + "; diff=" + (contribs.vecs()[4].at(762) - 0.0), Math.abs(contribs.vecs()[4].at(762) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(769) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(769) + "; diff=" + (contribs.vecs()[1].at(769) - 0.0), Math.abs(contribs.vecs()[1].at(769) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(770) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(770) + "; diff=" + (contribs.vecs()[4].at(770) - -0.33179789781570435), Math.abs(contribs.vecs()[4].at(770) - -0.33179789781570435) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(775) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(775) + "; diff=" + (contribs.vecs()[2].at(775) - 1.5633254051208496), Math.abs(contribs.vecs()[2].at(775) - 1.5633254051208496) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(793) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(793) + "; diff=" + (contribs.vecs()[7].at(793) - 0.5479622483253479), Math.abs(contribs.vecs()[7].at(793) - 0.5479622483253479) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(804) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(804) + "; diff=" + (contribs.vecs()[3].at(804) - 0.4731927216053009), Math.abs(contribs.vecs()[3].at(804) - 0.4731927216053009) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(811) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(811) + "; diff=" + (contribs.vecs()[1].at(811) - 0.0), Math.abs(contribs.vecs()[1].at(811) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(817) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(817) + "; diff=" + (contribs.vecs()[3].at(817) - 0.6870976090431213), Math.abs(contribs.vecs()[3].at(817) - 0.6870976090431213) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(818) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(818) + "; diff=" + (contribs.vecs()[1].at(818) - 0.0), Math.abs(contribs.vecs()[1].at(818) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(821) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(821) + "; diff=" + (contribs.vecs()[3].at(821) - 0.543901801109314), Math.abs(contribs.vecs()[3].at(821) - 0.543901801109314) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(821) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(821) + "; diff=" + (contribs.vecs()[5].at(821) - -0.07580218464136124), Math.abs(contribs.vecs()[5].at(821) - -0.07580218464136124) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(833) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(833) + "; diff=" + (contribs.vecs()[0].at(833) - -0.3255879580974579), Math.abs(contribs.vecs()[0].at(833) - -0.3255879580974579) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(844) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(844) + "; diff=" + (contribs.vecs()[5].at(844) - -0.035776060074567795), Math.abs(contribs.vecs()[5].at(844) - -0.035776060074567795) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(856) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(856) + "; diff=" + (contribs.vecs()[4].at(856) - 0.0), Math.abs(contribs.vecs()[4].at(856) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(879) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(879) + "; diff=" + (contribs.vecs()[1].at(879) - -0.19245196878910065), Math.abs(contribs.vecs()[1].at(879) - -0.19245196878910065) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(883) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(883) + "; diff=" + (contribs.vecs()[5].at(883) - 0.07384628057479858), Math.abs(contribs.vecs()[5].at(883) - 0.07384628057479858) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(885) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(885) + "; diff=" + (contribs.vecs()[7].at(885) - -0.6469556093215942), Math.abs(contribs.vecs()[7].at(885) - -0.6469556093215942) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(888) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(888) + "; diff=" + (contribs.vecs()[2].at(888) - -1.5706157684326172), Math.abs(contribs.vecs()[2].at(888) - -1.5706157684326172) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(890) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(890) + "; diff=" + (contribs.vecs()[2].at(890) - 0.0), Math.abs(contribs.vecs()[2].at(890) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(896) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(896) + "; diff=" + (contribs.vecs()[5].at(896) - 0.24125796556472778), Math.abs(contribs.vecs()[5].at(896) - 0.24125796556472778) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(898) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(898) + "; diff=" + (contribs.vecs()[3].at(898) - 0.0), Math.abs(contribs.vecs()[3].at(898) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(900) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(900) + "; diff=" + (contribs.vecs()[4].at(900) - -0.5291658043861389), Math.abs(contribs.vecs()[4].at(900) - -0.5291658043861389) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(904) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(904) + "; diff=" + (contribs.vecs()[1].at(904) - 0.008820213377475739), Math.abs(contribs.vecs()[1].at(904) - 0.008820213377475739) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(912) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(912) + "; diff=" + (contribs.vecs()[4].at(912) - -0.17594759166240692), Math.abs(contribs.vecs()[4].at(912) - -0.17594759166240692) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(913) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(913) + "; diff=" + (contribs.vecs()[6].at(913) - -1.0662415027618408), Math.abs(contribs.vecs()[6].at(913) - -1.0662415027618408) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(915) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(915) + "; diff=" + (contribs.vecs()[6].at(915) - -0.5274701118469238), Math.abs(contribs.vecs()[6].at(915) - -0.5274701118469238) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(920) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(920) + "; diff=" + (contribs.vecs()[1].at(920) - 0.22263449430465698), Math.abs(contribs.vecs()[1].at(920) - 0.22263449430465698) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(929) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(929) + "; diff=" + (contribs.vecs()[0].at(929) - -0.504033625125885), Math.abs(contribs.vecs()[0].at(929) - -0.504033625125885) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(933) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(933) + "; diff=" + (contribs.vecs()[1].at(933) - 0.2341347634792328), Math.abs(contribs.vecs()[1].at(933) - 0.2341347634792328) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(936) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(936) + "; diff=" + (contribs.vecs()[1].at(936) - 0.228414848446846), Math.abs(contribs.vecs()[1].at(936) - 0.228414848446846) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(957) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(957) + "; diff=" + (contribs.vecs()[3].at(957) - 0.0), Math.abs(contribs.vecs()[3].at(957) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(969) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(969) + "; diff=" + (contribs.vecs()[2].at(969) - 0.0), Math.abs(contribs.vecs()[2].at(969) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(973) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(973) + "; diff=" + (contribs.vecs()[7].at(973) - 0.0), Math.abs(contribs.vecs()[7].at(973) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(975) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(975) + "; diff=" + (contribs.vecs()[6].at(975) - 0.0), Math.abs(contribs.vecs()[6].at(975) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(977) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(977) + "; diff=" + (contribs.vecs()[6].at(977) - -0.3834817111492157), Math.abs(contribs.vecs()[6].at(977) - -0.3834817111492157) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(979) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(979) + "; diff=" + (contribs.vecs()[4].at(979) - -0.13317827880382538), Math.abs(contribs.vecs()[4].at(979) - -0.13317827880382538) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(984) + "; diff=" + (contribs.vecs()[7].at(984) - 0.7178128361701965), Math.abs(contribs.vecs()[7].at(984) - 0.7178128361701965) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(994) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(994) + "; diff=" + (contribs.vecs()[7].at(994) - 0.5923293232917786), Math.abs(contribs.vecs()[7].at(994) - 0.5923293232917786) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(998) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(998) + "; diff=" + (contribs.vecs()[1].at(998) - 0.0), Math.abs(contribs.vecs()[1].at(998) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1010) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1010) + "; diff=" + (contribs.vecs()[7].at(1010) - 0.0), Math.abs(contribs.vecs()[7].at(1010) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1016) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1016) + "; diff=" + (contribs.vecs()[2].at(1016) - 0.0), Math.abs(contribs.vecs()[2].at(1016) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1023) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1023) + "; diff=" + (contribs.vecs()[2].at(1023) - 0.9833056330680847), Math.abs(contribs.vecs()[2].at(1023) - 0.9833056330680847) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1028) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1028) + "; diff=" + (contribs.vecs()[0].at(1028) - -0.5017721652984619), Math.abs(contribs.vecs()[0].at(1028) - -0.5017721652984619) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1038) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1038) + "; diff=" + (contribs.vecs()[5].at(1038) - -0.3614657521247864), Math.abs(contribs.vecs()[5].at(1038) - -0.3614657521247864) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1039) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1039) + "; diff=" + (contribs.vecs()[4].at(1039) - -0.1067371591925621), Math.abs(contribs.vecs()[4].at(1039) - -0.1067371591925621) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1045) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1045) + "; diff=" + (contribs.vecs()[3].at(1045) - -1.1568502187728882), Math.abs(contribs.vecs()[3].at(1045) - -1.1568502187728882) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1057) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1057) + "; diff=" + (contribs.vecs()[4].at(1057) - -0.5544533729553223), Math.abs(contribs.vecs()[4].at(1057) - -0.5544533729553223) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1062) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1062) + "; diff=" + (contribs.vecs()[5].at(1062) - -0.05309559404850006), Math.abs(contribs.vecs()[5].at(1062) - -0.05309559404850006) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1064) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1064) + "; diff=" + (contribs.vecs()[4].at(1064) - 0.0), Math.abs(contribs.vecs()[4].at(1064) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1076) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1076) + "; diff=" + (contribs.vecs()[7].at(1076) - -0.025188103318214417), Math.abs(contribs.vecs()[7].at(1076) - -0.025188103318214417) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1078) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1078) + "; diff=" + (contribs.vecs()[7].at(1078) - 0.0), Math.abs(contribs.vecs()[7].at(1078) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1091) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1091) + "; diff=" + (contribs.vecs()[1].at(1091) - -0.016343478113412857), Math.abs(contribs.vecs()[1].at(1091) - -0.016343478113412857) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1107) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1107) + "; diff=" + (contribs.vecs()[0].at(1107) - -0.34347963333129883), Math.abs(contribs.vecs()[0].at(1107) - -0.34347963333129883) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1116) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1116) + "; diff=" + (contribs.vecs()[4].at(1116) - -0.30302712321281433), Math.abs(contribs.vecs()[4].at(1116) - -0.30302712321281433) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1120) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1120) + "; diff=" + (contribs.vecs()[7].at(1120) - 0.4587375819683075), Math.abs(contribs.vecs()[7].at(1120) - 0.4587375819683075) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1134) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1134) + "; diff=" + (contribs.vecs()[1].at(1134) - -0.3090515434741974), Math.abs(contribs.vecs()[1].at(1134) - -0.3090515434741974) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1141) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1141) + "; diff=" + (contribs.vecs()[1].at(1141) - -0.3424721658229828), Math.abs(contribs.vecs()[1].at(1141) - -0.3424721658229828) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1144) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1144) + "; diff=" + (contribs.vecs()[4].at(1144) - 0.0), Math.abs(contribs.vecs()[4].at(1144) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1151) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1151) + "; diff=" + (contribs.vecs()[3].at(1151) - 0.8764688968658447), Math.abs(contribs.vecs()[3].at(1151) - 0.8764688968658447) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1171) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1171) + "; diff=" + (contribs.vecs()[1].at(1171) - 0.0), Math.abs(contribs.vecs()[1].at(1171) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1172) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1172) + "; diff=" + (contribs.vecs()[3].at(1172) - -0.851893424987793), Math.abs(contribs.vecs()[3].at(1172) - -0.851893424987793) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1175) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1175) + "; diff=" + (contribs.vecs()[4].at(1175) - -0.44742950797080994), Math.abs(contribs.vecs()[4].at(1175) - -0.44742950797080994) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1177) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1177) + "; diff=" + (contribs.vecs()[1].at(1177) - 0.0), Math.abs(contribs.vecs()[1].at(1177) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1183) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1183) + "; diff=" + (contribs.vecs()[6].at(1183) - 0.0), Math.abs(contribs.vecs()[6].at(1183) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1185) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1185) + "; diff=" + (contribs.vecs()[0].at(1185) - -0.5246394872665405), Math.abs(contribs.vecs()[0].at(1185) - -0.5246394872665405) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1187) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1187) + "; diff=" + (contribs.vecs()[6].at(1187) - 0.0), Math.abs(contribs.vecs()[6].at(1187) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1194) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1194) + "; diff=" + (contribs.vecs()[4].at(1194) - 0.0), Math.abs(contribs.vecs()[4].at(1194) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1196) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1196) + "; diff=" + (contribs.vecs()[5].at(1196) - 0.020161302760243416), Math.abs(contribs.vecs()[5].at(1196) - 0.020161302760243416) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1200) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1200) + "; diff=" + (contribs.vecs()[3].at(1200) - 0.7104930281639099), Math.abs(contribs.vecs()[3].at(1200) - 0.7104930281639099) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1204) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1204) + "; diff=" + (contribs.vecs()[0].at(1204) - -0.2738960385322571), Math.abs(contribs.vecs()[0].at(1204) - -0.2738960385322571) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1215) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1215) + "; diff=" + (contribs.vecs()[5].at(1215) - -0.15537123382091522), Math.abs(contribs.vecs()[5].at(1215) - -0.15537123382091522) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1222) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1222) + "; diff=" + (contribs.vecs()[5].at(1222) - -0.24451979994773865), Math.abs(contribs.vecs()[5].at(1222) - -0.24451979994773865) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1228) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1228) + "; diff=" + (contribs.vecs()[5].at(1228) - -1.5381841659545898), Math.abs(contribs.vecs()[5].at(1228) - -1.5381841659545898) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1242) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1242) + "; diff=" + (contribs.vecs()[5].at(1242) - -3.5212204456329346), Math.abs(contribs.vecs()[5].at(1242) - -3.5212204456329346) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1247) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1247) + "; diff=" + (contribs.vecs()[3].at(1247) - 0.0), Math.abs(contribs.vecs()[3].at(1247) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1255) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1255) + "; diff=" + (contribs.vecs()[4].at(1255) - 0.0), Math.abs(contribs.vecs()[4].at(1255) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1258) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1258) + "; diff=" + (contribs.vecs()[3].at(1258) - 0.14675742387771606), Math.abs(contribs.vecs()[3].at(1258) - 0.14675742387771606) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1260) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1260) + "; diff=" + (contribs.vecs()[2].at(1260) - 1.1089333295822144), Math.abs(contribs.vecs()[2].at(1260) - 1.1089333295822144) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1303) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1303) + "; diff=" + (contribs.vecs()[7].at(1303) - 0.13960134983062744), Math.abs(contribs.vecs()[7].at(1303) - 0.13960134983062744) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1308) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1308) + "; diff=" + (contribs.vecs()[2].at(1308) - 0.0), Math.abs(contribs.vecs()[2].at(1308) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1309) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1309) + "; diff=" + (contribs.vecs()[4].at(1309) - 0.0), Math.abs(contribs.vecs()[4].at(1309) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1317) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1317) + "; diff=" + (contribs.vecs()[3].at(1317) - -0.514670193195343), Math.abs(contribs.vecs()[3].at(1317) - -0.514670193195343) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1323) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1323) + "; diff=" + (contribs.vecs()[4].at(1323) - 0.0), Math.abs(contribs.vecs()[4].at(1323) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1324) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1324) + "; diff=" + (contribs.vecs()[0].at(1324) - -0.24705907702445984), Math.abs(contribs.vecs()[0].at(1324) - -0.24705907702445984) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1325) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1325) + "; diff=" + (contribs.vecs()[0].at(1325) - -0.21464914083480835), Math.abs(contribs.vecs()[0].at(1325) - -0.21464914083480835) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1325) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1325) + "; diff=" + (contribs.vecs()[6].at(1325) - 0.4443226158618927), Math.abs(contribs.vecs()[6].at(1325) - 0.4443226158618927) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1340) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1340) + "; diff=" + (contribs.vecs()[0].at(1340) - -0.3651164472103119), Math.abs(contribs.vecs()[0].at(1340) - -0.3651164472103119) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1341) + "; diff=" + (contribs.vecs()[1].at(1341) - 0.08190041035413742), Math.abs(contribs.vecs()[1].at(1341) - 0.08190041035413742) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1358) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1358) + "; diff=" + (contribs.vecs()[5].at(1358) - 0.5015586614608765), Math.abs(contribs.vecs()[5].at(1358) - 0.5015586614608765) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1371) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1371) + "; diff=" + (contribs.vecs()[2].at(1371) - -0.5673139691352844), Math.abs(contribs.vecs()[2].at(1371) - -0.5673139691352844) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1386) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1386) + "; diff=" + (contribs.vecs()[3].at(1386) - -0.5862051248550415), Math.abs(contribs.vecs()[3].at(1386) - -0.5862051248550415) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1394) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1394) + "; diff=" + (contribs.vecs()[5].at(1394) - -0.30793464183807373), Math.abs(contribs.vecs()[5].at(1394) - -0.30793464183807373) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1400) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1400) + "; diff=" + (contribs.vecs()[5].at(1400) - 0.13208724558353424), Math.abs(contribs.vecs()[5].at(1400) - 0.13208724558353424) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1402) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1402) + "; diff=" + (contribs.vecs()[1].at(1402) - 0.0), Math.abs(contribs.vecs()[1].at(1402) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1408) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1408) + "; diff=" + (contribs.vecs()[4].at(1408) - 0.0), Math.abs(contribs.vecs()[4].at(1408) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1412) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1412) + "; diff=" + (contribs.vecs()[5].at(1412) - -0.028300341218709946), Math.abs(contribs.vecs()[5].at(1412) - -0.028300341218709946) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1421) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1421) + "; diff=" + (contribs.vecs()[7].at(1421) - -0.7503513097763062), Math.abs(contribs.vecs()[7].at(1421) - -0.7503513097763062) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1423) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1423) + "; diff=" + (contribs.vecs()[6].at(1423) - -1.512215256690979), Math.abs(contribs.vecs()[6].at(1423) - -1.512215256690979) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1429) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1429) + "; diff=" + (contribs.vecs()[7].at(1429) - -0.3178964853286743), Math.abs(contribs.vecs()[7].at(1429) - -0.3178964853286743) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1435) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1435) + "; diff=" + (contribs.vecs()[1].at(1435) - 0.17372745275497437), Math.abs(contribs.vecs()[1].at(1435) - 0.17372745275497437) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1438) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1438) + "; diff=" + (contribs.vecs()[5].at(1438) - -0.2939351797103882), Math.abs(contribs.vecs()[5].at(1438) - -0.2939351797103882) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1451) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1451) + "; diff=" + (contribs.vecs()[7].at(1451) - -0.4010968506336212), Math.abs(contribs.vecs()[7].at(1451) - -0.4010968506336212) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1455) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1455) + "; diff=" + (contribs.vecs()[3].at(1455) - 0.5229577422142029), Math.abs(contribs.vecs()[3].at(1455) - 0.5229577422142029) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1457) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1457) + "; diff=" + (contribs.vecs()[7].at(1457) - -0.20937426388263702), Math.abs(contribs.vecs()[7].at(1457) - -0.20937426388263702) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1469) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1469) + "; diff=" + (contribs.vecs()[6].at(1469) - 0.11448994278907776), Math.abs(contribs.vecs()[6].at(1469) - 0.11448994278907776) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1474) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1474) + "; diff=" + (contribs.vecs()[0].at(1474) - -0.07277217507362366), Math.abs(contribs.vecs()[0].at(1474) - -0.07277217507362366) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1479) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1479) + "; diff=" + (contribs.vecs()[0].at(1479) - -0.7350947260856628), Math.abs(contribs.vecs()[0].at(1479) - -0.7350947260856628) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1482) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1482) + "; diff=" + (contribs.vecs()[3].at(1482) - -0.5479225516319275), Math.abs(contribs.vecs()[3].at(1482) - -0.5479225516319275) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1490) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1490) + "; diff=" + (contribs.vecs()[1].at(1490) - 0.014099462889134884), Math.abs(contribs.vecs()[1].at(1490) - 0.014099462889134884) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1495) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1495) + "; diff=" + (contribs.vecs()[2].at(1495) - 0.4291142523288727), Math.abs(contribs.vecs()[2].at(1495) - 0.4291142523288727) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1511) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1511) + "; diff=" + (contribs.vecs()[2].at(1511) - 0.0), Math.abs(contribs.vecs()[2].at(1511) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1535) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1535) + "; diff=" + (contribs.vecs()[3].at(1535) - 0.7950345277786255), Math.abs(contribs.vecs()[3].at(1535) - 0.7950345277786255) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1550) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1550) + "; diff=" + (contribs.vecs()[7].at(1550) - 0.0), Math.abs(contribs.vecs()[7].at(1550) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1561) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1561) + "; diff=" + (contribs.vecs()[4].at(1561) - 0.0), Math.abs(contribs.vecs()[4].at(1561) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1573) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1573) + "; diff=" + (contribs.vecs()[1].at(1573) - 0.0), Math.abs(contribs.vecs()[1].at(1573) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1574) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1574) + "; diff=" + (contribs.vecs()[1].at(1574) - -0.13196739554405212), Math.abs(contribs.vecs()[1].at(1574) - -0.13196739554405212) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1582) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1582) + "; diff=" + (contribs.vecs()[6].at(1582) - 0.0), Math.abs(contribs.vecs()[6].at(1582) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1587) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1587) + "; diff=" + (contribs.vecs()[0].at(1587) - -0.4875998795032501), Math.abs(contribs.vecs()[0].at(1587) - -0.4875998795032501) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1590) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1590) + "; diff=" + (contribs.vecs()[0].at(1590) - -0.6396923065185547), Math.abs(contribs.vecs()[0].at(1590) - -0.6396923065185547) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1598) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1598) + "; diff=" + (contribs.vecs()[7].at(1598) - 1.2356491088867188), Math.abs(contribs.vecs()[7].at(1598) - 1.2356491088867188) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1601) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1601) + "; diff=" + (contribs.vecs()[3].at(1601) - 0.0), Math.abs(contribs.vecs()[3].at(1601) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1601) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1601) + "; diff=" + (contribs.vecs()[6].at(1601) - 0.0), Math.abs(contribs.vecs()[6].at(1601) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1604) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1604) + "; diff=" + (contribs.vecs()[1].at(1604) - 0.0), Math.abs(contribs.vecs()[1].at(1604) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1609) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1609) + "; diff=" + (contribs.vecs()[1].at(1609) - 0.0), Math.abs(contribs.vecs()[1].at(1609) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1610) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1610) + "; diff=" + (contribs.vecs()[4].at(1610) - 0.0), Math.abs(contribs.vecs()[4].at(1610) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1613) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1613) + "; diff=" + (contribs.vecs()[3].at(1613) - 1.452709674835205), Math.abs(contribs.vecs()[3].at(1613) - 1.452709674835205) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1626) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1626) + "; diff=" + (contribs.vecs()[0].at(1626) - -0.20764008164405823), Math.abs(contribs.vecs()[0].at(1626) - -0.20764008164405823) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1628) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1628) + "; diff=" + (contribs.vecs()[5].at(1628) - -0.821851372718811), Math.abs(contribs.vecs()[5].at(1628) - -0.821851372718811) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1629) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1629) + "; diff=" + (contribs.vecs()[3].at(1629) - 0.0), Math.abs(contribs.vecs()[3].at(1629) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1630) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1630) + "; diff=" + (contribs.vecs()[1].at(1630) - 0.0), Math.abs(contribs.vecs()[1].at(1630) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1635) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1635) + "; diff=" + (contribs.vecs()[0].at(1635) - -0.5778653621673584), Math.abs(contribs.vecs()[0].at(1635) - -0.5778653621673584) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1637) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1637) + "; diff=" + (contribs.vecs()[3].at(1637) - 0.4788239300251007), Math.abs(contribs.vecs()[3].at(1637) - 0.4788239300251007) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1650) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1650) + "; diff=" + (contribs.vecs()[0].at(1650) - -0.37024739384651184), Math.abs(contribs.vecs()[0].at(1650) - -0.37024739384651184) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1655) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1655) + "; diff=" + (contribs.vecs()[1].at(1655) - 0.0), Math.abs(contribs.vecs()[1].at(1655) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1665) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1665) + "; diff=" + (contribs.vecs()[7].at(1665) - 0.0), Math.abs(contribs.vecs()[7].at(1665) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1683) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1683) + "; diff=" + (contribs.vecs()[5].at(1683) - -0.38537460565567017), Math.abs(contribs.vecs()[5].at(1683) - -0.38537460565567017) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1707) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1707) + "; diff=" + (contribs.vecs()[7].at(1707) - 0.0), Math.abs(contribs.vecs()[7].at(1707) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1712) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1712) + "; diff=" + (contribs.vecs()[1].at(1712) - 0.0), Math.abs(contribs.vecs()[1].at(1712) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1715) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1715) + "; diff=" + (contribs.vecs()[1].at(1715) - 0.0), Math.abs(contribs.vecs()[1].at(1715) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1718) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1718) + "; diff=" + (contribs.vecs()[5].at(1718) - -0.04745614156126976), Math.abs(contribs.vecs()[5].at(1718) - -0.04745614156126976) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1724) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1724) + "; diff=" + (contribs.vecs()[0].at(1724) - -0.24005818367004395), Math.abs(contribs.vecs()[0].at(1724) - -0.24005818367004395) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1729) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1729) + "; diff=" + (contribs.vecs()[0].at(1729) - -0.17885462939739227), Math.abs(contribs.vecs()[0].at(1729) - -0.17885462939739227) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1739) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1739) + "; diff=" + (contribs.vecs()[4].at(1739) - -0.31237637996673584), Math.abs(contribs.vecs()[4].at(1739) - -0.31237637996673584) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1755) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1755) + "; diff=" + (contribs.vecs()[1].at(1755) - 0.0), Math.abs(contribs.vecs()[1].at(1755) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1760) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1760) + "; diff=" + (contribs.vecs()[3].at(1760) - -0.4271882176399231), Math.abs(contribs.vecs()[3].at(1760) - -0.4271882176399231) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1770) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1770) + "; diff=" + (contribs.vecs()[7].at(1770) - 0.0), Math.abs(contribs.vecs()[7].at(1770) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1775) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1775) + "; diff=" + (contribs.vecs()[2].at(1775) - 0.0), Math.abs(contribs.vecs()[2].at(1775) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1799) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1799) + "; diff=" + (contribs.vecs()[4].at(1799) - 0.27511513233184814), Math.abs(contribs.vecs()[4].at(1799) - 0.27511513233184814) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1816) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1816) + "; diff=" + (contribs.vecs()[6].at(1816) - -0.001250381814315915), Math.abs(contribs.vecs()[6].at(1816) - -0.001250381814315915) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1821) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1821) + "; diff=" + (contribs.vecs()[1].at(1821) - 0.17878897488117218), Math.abs(contribs.vecs()[1].at(1821) - 0.17878897488117218) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1826) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1826) + "; diff=" + (contribs.vecs()[5].at(1826) - -1.7367584705352783), Math.abs(contribs.vecs()[5].at(1826) - -1.7367584705352783) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1828) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1828) + "; diff=" + (contribs.vecs()[6].at(1828) - 0.7035931944847107), Math.abs(contribs.vecs()[6].at(1828) - 0.7035931944847107) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1831) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1831) + "; diff=" + (contribs.vecs()[2].at(1831) - 1.3278272151947021), Math.abs(contribs.vecs()[2].at(1831) - 1.3278272151947021) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1847) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1847) + "; diff=" + (contribs.vecs()[7].at(1847) - 0.4615137577056885), Math.abs(contribs.vecs()[7].at(1847) - 0.4615137577056885) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1856) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1856) + "; diff=" + (contribs.vecs()[1].at(1856) - 0.0), Math.abs(contribs.vecs()[1].at(1856) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1868) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1868) + "; diff=" + (contribs.vecs()[7].at(1868) - 0.0), Math.abs(contribs.vecs()[7].at(1868) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1879) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1879) + "; diff=" + (contribs.vecs()[1].at(1879) - 0.15033870935440063), Math.abs(contribs.vecs()[1].at(1879) - 0.15033870935440063) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1880) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1880) + "; diff=" + (contribs.vecs()[3].at(1880) - 0.304600328207016), Math.abs(contribs.vecs()[3].at(1880) - 0.304600328207016) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1891) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1891) + "; diff=" + (contribs.vecs()[0].at(1891) - -0.22425580024719238), Math.abs(contribs.vecs()[0].at(1891) - -0.22425580024719238) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1900) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1900) + "; diff=" + (contribs.vecs()[4].at(1900) - -0.15452930331230164), Math.abs(contribs.vecs()[4].at(1900) - -0.15452930331230164) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1907) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1907) + "; diff=" + (contribs.vecs()[4].at(1907) - 0.0), Math.abs(contribs.vecs()[4].at(1907) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1908) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1908) + "; diff=" + (contribs.vecs()[6].at(1908) - 0.8482376337051392), Math.abs(contribs.vecs()[6].at(1908) - 0.8482376337051392) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1914) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1914) + "; diff=" + (contribs.vecs()[2].at(1914) - 0.0), Math.abs(contribs.vecs()[2].at(1914) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1915) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1915) + "; diff=" + (contribs.vecs()[5].at(1915) - -0.2122526913881302), Math.abs(contribs.vecs()[5].at(1915) - -0.2122526913881302) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1917) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1917) + "; diff=" + (contribs.vecs()[5].at(1917) - -0.19786998629570007), Math.abs(contribs.vecs()[5].at(1917) - -0.19786998629570007) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1923) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1923) + "; diff=" + (contribs.vecs()[4].at(1923) - 0.0), Math.abs(contribs.vecs()[4].at(1923) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1929) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1929) + "; diff=" + (contribs.vecs()[1].at(1929) - -0.1432940661907196), Math.abs(contribs.vecs()[1].at(1929) - -0.1432940661907196) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1943) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1943) + "; diff=" + (contribs.vecs()[3].at(1943) - 0.0), Math.abs(contribs.vecs()[3].at(1943) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1948) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1948) + "; diff=" + (contribs.vecs()[7].at(1948) - 0.0), Math.abs(contribs.vecs()[7].at(1948) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1955) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1955) + "; diff=" + (contribs.vecs()[2].at(1955) - 0.0), Math.abs(contribs.vecs()[2].at(1955) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1959) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1959) + "; diff=" + (contribs.vecs()[7].at(1959) - 0.0), Math.abs(contribs.vecs()[7].at(1959) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1960) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1960) + "; diff=" + (contribs.vecs()[7].at(1960) - 0.0), Math.abs(contribs.vecs()[7].at(1960) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1969) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1969) + "; diff=" + (contribs.vecs()[6].at(1969) - 0.012140314094722271), Math.abs(contribs.vecs()[6].at(1969) - 0.012140314094722271) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1970) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1970) + "; diff=" + (contribs.vecs()[5].at(1970) - -0.04970177263021469), Math.abs(contribs.vecs()[5].at(1970) - -0.04970177263021469) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1984) + "; diff=" + (contribs.vecs()[4].at(1984) - 0.0), Math.abs(contribs.vecs()[4].at(1984) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1984) + "; diff=" + (contribs.vecs()[5].at(1984) - -0.16093464195728302), Math.abs(contribs.vecs()[5].at(1984) - -0.16093464195728302) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1993) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1993) + "; diff=" + (contribs.vecs()[1].at(1993) - 0.0), Math.abs(contribs.vecs()[1].at(1993) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1995) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1995) + "; diff=" + (contribs.vecs()[6].at(1995) - -0.4950285851955414), Math.abs(contribs.vecs()[6].at(1995) - -0.4950285851955414) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2002) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2002) + "; diff=" + (contribs.vecs()[6].at(2002) - 0.9328063130378723), Math.abs(contribs.vecs()[6].at(2002) - 0.9328063130378723) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2006) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2006) + "; diff=" + (contribs.vecs()[6].at(2006) - 0.46997368335723877), Math.abs(contribs.vecs()[6].at(2006) - 0.46997368335723877) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2013) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2013) + "; diff=" + (contribs.vecs()[3].at(2013) - 0.0), Math.abs(contribs.vecs()[3].at(2013) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2017) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2017) + "; diff=" + (contribs.vecs()[6].at(2017) - -0.04804735258221626), Math.abs(contribs.vecs()[6].at(2017) - -0.04804735258221626) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2020) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2020) + "; diff=" + (contribs.vecs()[1].at(2020) - 0.0), Math.abs(contribs.vecs()[1].at(2020) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2021) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2021) + "; diff=" + (contribs.vecs()[5].at(2021) - 0.11558608710765839), Math.abs(contribs.vecs()[5].at(2021) - 0.11558608710765839) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2033) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2033) + "; diff=" + (contribs.vecs()[1].at(2033) - -0.18281495571136475), Math.abs(contribs.vecs()[1].at(2033) - -0.18281495571136475) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2040) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2040) + "; diff=" + (contribs.vecs()[2].at(2040) - 0.0), Math.abs(contribs.vecs()[2].at(2040) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2045) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2045) + "; diff=" + (contribs.vecs()[1].at(2045) - -0.10132802277803421), Math.abs(contribs.vecs()[1].at(2045) - -0.10132802277803421) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2059) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2059) + "; diff=" + (contribs.vecs()[0].at(2059) - -0.6166958808898926), Math.abs(contribs.vecs()[0].at(2059) - -0.6166958808898926) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2061) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2061) + "; diff=" + (contribs.vecs()[6].at(2061) - 0.0), Math.abs(contribs.vecs()[6].at(2061) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2063) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2063) + "; diff=" + (contribs.vecs()[5].at(2063) - 0.10107361525297165), Math.abs(contribs.vecs()[5].at(2063) - 0.10107361525297165) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2072) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2072) + "; diff=" + (contribs.vecs()[5].at(2072) - 0.1687183827161789), Math.abs(contribs.vecs()[5].at(2072) - 0.1687183827161789) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2078) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2078) + "; diff=" + (contribs.vecs()[7].at(2078) - -0.9449918866157532), Math.abs(contribs.vecs()[7].at(2078) - -0.9449918866157532) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2094) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2094) + "; diff=" + (contribs.vecs()[6].at(2094) - -0.19366121292114258), Math.abs(contribs.vecs()[6].at(2094) - -0.19366121292114258) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2113) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2113) + "; diff=" + (contribs.vecs()[4].at(2113) - 0.0), Math.abs(contribs.vecs()[4].at(2113) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2120) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2120) + "; diff=" + (contribs.vecs()[2].at(2120) - 0.0), Math.abs(contribs.vecs()[2].at(2120) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2125) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2125) + "; diff=" + (contribs.vecs()[2].at(2125) - 0.0), Math.abs(contribs.vecs()[2].at(2125) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2129) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2129) + "; diff=" + (contribs.vecs()[1].at(2129) - 0.0), Math.abs(contribs.vecs()[1].at(2129) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2131) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2131) + "; diff=" + (contribs.vecs()[3].at(2131) - 0.0), Math.abs(contribs.vecs()[3].at(2131) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2152) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2152) + "; diff=" + (contribs.vecs()[4].at(2152) - 0.0), Math.abs(contribs.vecs()[4].at(2152) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2167) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2167) + "; diff=" + (contribs.vecs()[3].at(2167) - -0.9592239260673523), Math.abs(contribs.vecs()[3].at(2167) - -0.9592239260673523) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2180) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2180) + "; diff=" + (contribs.vecs()[4].at(2180) - 0.28466251492500305), Math.abs(contribs.vecs()[4].at(2180) - 0.28466251492500305) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2183) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2183) + "; diff=" + (contribs.vecs()[4].at(2183) - 0.27689680457115173), Math.abs(contribs.vecs()[4].at(2183) - 0.27689680457115173) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2196) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2196) + "; diff=" + (contribs.vecs()[2].at(2196) - 1.5264776945114136), Math.abs(contribs.vecs()[2].at(2196) - 1.5264776945114136) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2199) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2199) + "; diff=" + (contribs.vecs()[6].at(2199) - -0.4486493766307831), Math.abs(contribs.vecs()[6].at(2199) - -0.4486493766307831) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2200) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2200) + "; diff=" + (contribs.vecs()[1].at(2200) - 0.0), Math.abs(contribs.vecs()[1].at(2200) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2201) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2201) + "; diff=" + (contribs.vecs()[2].at(2201) - 0.0), Math.abs(contribs.vecs()[2].at(2201) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2206) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2206) + "; diff=" + (contribs.vecs()[7].at(2206) - 0.0), Math.abs(contribs.vecs()[7].at(2206) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2215) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2215) + "; diff=" + (contribs.vecs()[2].at(2215) - 0.0), Math.abs(contribs.vecs()[2].at(2215) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2221) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2221) + "; diff=" + (contribs.vecs()[0].at(2221) - -0.39188310503959656), Math.abs(contribs.vecs()[0].at(2221) - -0.39188310503959656) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2230) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2230) + "; diff=" + (contribs.vecs()[5].at(2230) - 0.25613903999328613), Math.abs(contribs.vecs()[5].at(2230) - 0.25613903999328613) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2231) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2231) + "; diff=" + (contribs.vecs()[7].at(2231) - 0.0), Math.abs(contribs.vecs()[7].at(2231) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2240) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2240) + "; diff=" + (contribs.vecs()[7].at(2240) - 0.20388856530189514), Math.abs(contribs.vecs()[7].at(2240) - 0.20388856530189514) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2248) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2248) + "; diff=" + (contribs.vecs()[5].at(2248) - -0.03969041630625725), Math.abs(contribs.vecs()[5].at(2248) - -0.03969041630625725) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2262) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2262) + "; diff=" + (contribs.vecs()[7].at(2262) - 0.3475807011127472), Math.abs(contribs.vecs()[7].at(2262) - 0.3475807011127472) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2268) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2268) + "; diff=" + (contribs.vecs()[3].at(2268) - 0.0), Math.abs(contribs.vecs()[3].at(2268) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2273) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2273) + "; diff=" + (contribs.vecs()[2].at(2273) - 0.0), Math.abs(contribs.vecs()[2].at(2273) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2279) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2279) + "; diff=" + (contribs.vecs()[5].at(2279) - -0.18406447768211365), Math.abs(contribs.vecs()[5].at(2279) - -0.18406447768211365) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2285) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2285) + "; diff=" + (contribs.vecs()[6].at(2285) - 0.3257436156272888), Math.abs(contribs.vecs()[6].at(2285) - 0.3257436156272888) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2286) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2286) + "; diff=" + (contribs.vecs()[6].at(2286) - 0.439421147108078), Math.abs(contribs.vecs()[6].at(2286) - 0.439421147108078) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2292) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2292) + "; diff=" + (contribs.vecs()[4].at(2292) - 0.0), Math.abs(contribs.vecs()[4].at(2292) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2300) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2300) + "; diff=" + (contribs.vecs()[7].at(2300) - -0.1656927764415741), Math.abs(contribs.vecs()[7].at(2300) - -0.1656927764415741) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2312) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2312) + "; diff=" + (contribs.vecs()[5].at(2312) - -0.6124882698059082), Math.abs(contribs.vecs()[5].at(2312) - -0.6124882698059082) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2315) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2315) + "; diff=" + (contribs.vecs()[0].at(2315) - -0.7104811072349548), Math.abs(contribs.vecs()[0].at(2315) - -0.7104811072349548) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2315) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2315) + "; diff=" + (contribs.vecs()[2].at(2315) - 0.0), Math.abs(contribs.vecs()[2].at(2315) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2328) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2328) + "; diff=" + (contribs.vecs()[0].at(2328) - -0.6534627079963684), Math.abs(contribs.vecs()[0].at(2328) - -0.6534627079963684) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2330) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2330) + "; diff=" + (contribs.vecs()[3].at(2330) - -0.5364760160446167), Math.abs(contribs.vecs()[3].at(2330) - -0.5364760160446167) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2361) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2361) + "; diff=" + (contribs.vecs()[2].at(2361) - 0.0), Math.abs(contribs.vecs()[2].at(2361) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2363) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2363) + "; diff=" + (contribs.vecs()[1].at(2363) - 0.0), Math.abs(contribs.vecs()[1].at(2363) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2369) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2369) + "; diff=" + (contribs.vecs()[3].at(2369) - 0.0), Math.abs(contribs.vecs()[3].at(2369) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2371) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2371) + "; diff=" + (contribs.vecs()[5].at(2371) - 0.07379321753978729), Math.abs(contribs.vecs()[5].at(2371) - 0.07379321753978729) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2375) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2375) + "; diff=" + (contribs.vecs()[0].at(2375) - -0.6297346353530884), Math.abs(contribs.vecs()[0].at(2375) - -0.6297346353530884) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2377) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2377) + "; diff=" + (contribs.vecs()[5].at(2377) - -0.016071418300271034), Math.abs(contribs.vecs()[5].at(2377) - -0.016071418300271034) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2394) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2394) + "; diff=" + (contribs.vecs()[5].at(2394) - -0.33734166622161865), Math.abs(contribs.vecs()[5].at(2394) - -0.33734166622161865) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2410) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2410) + "; diff=" + (contribs.vecs()[6].at(2410) - 0.0), Math.abs(contribs.vecs()[6].at(2410) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2411) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2411) + "; diff=" + (contribs.vecs()[1].at(2411) - 0.0), Math.abs(contribs.vecs()[1].at(2411) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2416) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2416) + "; diff=" + (contribs.vecs()[5].at(2416) - -0.9206300973892212), Math.abs(contribs.vecs()[5].at(2416) - -0.9206300973892212) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2421) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2421) + "; diff=" + (contribs.vecs()[6].at(2421) - -0.8671939969062805), Math.abs(contribs.vecs()[6].at(2421) - -0.8671939969062805) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2424) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2424) + "; diff=" + (contribs.vecs()[0].at(2424) - -0.5661767721176147), Math.abs(contribs.vecs()[0].at(2424) - -0.5661767721176147) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2425) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2425) + "; diff=" + (contribs.vecs()[1].at(2425) - 0.0), Math.abs(contribs.vecs()[1].at(2425) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2430) + "; diff=" + (contribs.vecs()[4].at(2430) - -0.3673529624938965), Math.abs(contribs.vecs()[4].at(2430) - -0.3673529624938965) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2442) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2442) + "; diff=" + (contribs.vecs()[2].at(2442) - -1.2806113958358765), Math.abs(contribs.vecs()[2].at(2442) - -1.2806113958358765) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2444) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2444) + "; diff=" + (contribs.vecs()[6].at(2444) - -0.36702486872673035), Math.abs(contribs.vecs()[6].at(2444) - -0.36702486872673035) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2446) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2446) + "; diff=" + (contribs.vecs()[7].at(2446) - 0.0), Math.abs(contribs.vecs()[7].at(2446) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2450) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2450) + "; diff=" + (contribs.vecs()[3].at(2450) - -0.4846079647541046), Math.abs(contribs.vecs()[3].at(2450) - -0.4846079647541046) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2453) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2453) + "; diff=" + (contribs.vecs()[3].at(2453) - 0.0), Math.abs(contribs.vecs()[3].at(2453) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2457) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2457) + "; diff=" + (contribs.vecs()[0].at(2457) - -0.7434551119804382), Math.abs(contribs.vecs()[0].at(2457) - -0.7434551119804382) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2461) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2461) + "; diff=" + (contribs.vecs()[6].at(2461) - 0.6379348039627075), Math.abs(contribs.vecs()[6].at(2461) - 0.6379348039627075) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2483) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2483) + "; diff=" + (contribs.vecs()[4].at(2483) - 0.0), Math.abs(contribs.vecs()[4].at(2483) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2496) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2496) + "; diff=" + (contribs.vecs()[4].at(2496) - 0.0), Math.abs(contribs.vecs()[4].at(2496) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2504) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2504) + "; diff=" + (contribs.vecs()[2].at(2504) - 0.0), Math.abs(contribs.vecs()[2].at(2504) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2504) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2504) + "; diff=" + (contribs.vecs()[4].at(2504) - -0.3213357925415039), Math.abs(contribs.vecs()[4].at(2504) - -0.3213357925415039) < eps);
      assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2507) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2507) + "; diff=" + (contribs.vecs()[0].at(2507) - -0.6453195214271545), Math.abs(contribs.vecs()[0].at(2507) - -0.6453195214271545) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2526) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2526) + "; diff=" + (contribs.vecs()[3].at(2526) - 0.5175634026527405), Math.abs(contribs.vecs()[3].at(2526) - 0.5175634026527405) < eps);
      assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2526) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2526) + "; diff=" + (contribs.vecs()[6].at(2526) - -0.795461893081665), Math.abs(contribs.vecs()[6].at(2526) - -0.795461893081665) < eps);
      assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2534) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2534) + "; diff=" + (contribs.vecs()[2].at(2534) - 0.0), Math.abs(contribs.vecs()[2].at(2534) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2568) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2568) + "; diff=" + (contribs.vecs()[4].at(2568) - -0.2644883692264557), Math.abs(contribs.vecs()[4].at(2568) - -0.2644883692264557) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2576) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2576) + "; diff=" + (contribs.vecs()[7].at(2576) - 0.0), Math.abs(contribs.vecs()[7].at(2576) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2583) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2583) + "; diff=" + (contribs.vecs()[3].at(2583) - 0.0), Math.abs(contribs.vecs()[3].at(2583) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2585) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2585) + "; diff=" + (contribs.vecs()[3].at(2585) - -0.41133856773376465), Math.abs(contribs.vecs()[3].at(2585) - -0.41133856773376465) < eps);
      assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2590) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2590) + "; diff=" + (contribs.vecs()[1].at(2590) - 0.0), Math.abs(contribs.vecs()[1].at(2590) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2594) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2594) + "; diff=" + (contribs.vecs()[4].at(2594) - 0.0), Math.abs(contribs.vecs()[4].at(2594) - 0.0) < eps);
      assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2599) + "; diff=" + (contribs.vecs()[7].at(2599) - 0.06657768040895462), Math.abs(contribs.vecs()[7].at(2599) - 0.06657768040895462) < eps);

    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }


  @Test
  public void testClassificationTanhDeepSHAPComparison() {
    Frame fr = parseTestFile("smalldata/prostate/prostate.csv").toCategoricalCol("CAPSULE");
    Frame bgFr = fr.deepSlice(new LongRange(0, 50).toArray(), null);
    Frame test = fr.deepSlice(new LongRange(51, 101).toArray(), null);
    DeepLearningModel model = null;
    Frame scored = null;
    Frame contribs = null;
    Frame res = null;
    double eps = 1e-4;
    try {
      // Launch Deep Learning
      DeepLearningParameters params = new DeepLearningParameters();
      params._train = fr._key;
      params._epochs = 5;
      params._response_column = "CAPSULE";
      params._hidden = ari(3, 3);
      params._seed = 123456;
      params._reproducible = true;
      params._activation = DeepLearningParameters.Activation.Tanh;
      params._distribution = DistributionFamily.bernoulli;


      model = new DeepLearning(params).trainModel().get();

      assert model != null;
      scored = model.score(test);
      contribs = model.scoreContributions(test, Key.make(), null,
              new Model.Contributions.ContributionsOptions()
                      .setOutputFormat(Model.Contributions.ContributionsOutputFormat.Original)
                      .setOutputPerReference(true),
              bgFr);

      //toPyTorch(model, test, bgFr);
      assertContributionsForTestClassificationTanhDeepSHAPComparison(contribs, eps);
    } finally {
      fr.delete();
      bgFr.delete();
      test.delete();
      if (null != res) res.delete();
      if (null != scored) scored.delete();
      if (null != contribs) contribs.delete();
      if (null != model) model.delete();
    }
  }
  
  private void assertContributionsForTestClassificationTanhDeepSHAPComparison(Frame contribs, double eps) {
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(7) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(7) + "; diff=" + (contribs.vecs()[3].at(7) - 0.0), Math.abs(contribs.vecs()[3].at(7) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(8) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(8) + "; diff=" + (contribs.vecs()[6].at(8) - 0.035676464438438416), Math.abs(contribs.vecs()[6].at(8) - 0.035676464438438416) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(12) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(12) + "; diff=" + (contribs.vecs()[1].at(12) - 0.0), Math.abs(contribs.vecs()[1].at(12) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(23) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(23) + "; diff=" + (contribs.vecs()[5].at(23) - 0.06693841516971588), Math.abs(contribs.vecs()[5].at(23) - 0.06693841516971588) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(26) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(26) + "; diff=" + (contribs.vecs()[4].at(26) - 0.1775183081626892), Math.abs(contribs.vecs()[4].at(26) - 0.1775183081626892) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(33) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(33) + "; diff=" + (contribs.vecs()[6].at(33) - -0.14369148015975952), Math.abs(contribs.vecs()[6].at(33) - -0.14369148015975952) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(37) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(37) + "; diff=" + (contribs.vecs()[0].at(37) - -0.0006938997539691627), Math.abs(contribs.vecs()[0].at(37) - -0.0006938997539691627) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(37) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(37) + "; diff=" + (contribs.vecs()[5].at(37) - -0.167270690202713), Math.abs(contribs.vecs()[5].at(37) - -0.167270690202713) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(38) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(38) + "; diff=" + (contribs.vecs()[0].at(38) - -0.0008466641302220523), Math.abs(contribs.vecs()[0].at(38) - -0.0008466641302220523) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(42) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(42) + "; diff=" + (contribs.vecs()[5].at(42) - 0.0034527077805250883), Math.abs(contribs.vecs()[5].at(42) - 0.0034527077805250883) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(44) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(44) + "; diff=" + (contribs.vecs()[3].at(44) - -0.010672773234546185), Math.abs(contribs.vecs()[3].at(44) - -0.010672773234546185) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(45) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(45) + "; diff=" + (contribs.vecs()[6].at(45) - 0.0017190121579915285), Math.abs(contribs.vecs()[6].at(45) - 0.0017190121579915285) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(50) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(50) + "; diff=" + (contribs.vecs()[0].at(50) - -1.117471583711449e-05), Math.abs(contribs.vecs()[0].at(50) - -1.117471583711449e-05) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(53) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(53) + "; diff=" + (contribs.vecs()[4].at(53) - -0.012516340240836143), Math.abs(contribs.vecs()[4].at(53) - -0.012516340240836143) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(56) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(56) + "; diff=" + (contribs.vecs()[2].at(56) - 0.0), Math.abs(contribs.vecs()[2].at(56) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(63) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(63) + "; diff=" + (contribs.vecs()[0].at(63) - -0.001936570042744279), Math.abs(contribs.vecs()[0].at(63) - -0.001936570042744279) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(63) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(63) + "; diff=" + (contribs.vecs()[4].at(63) - -0.11302074044942856), Math.abs(contribs.vecs()[4].at(63) - -0.11302074044942856) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(68) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(68) + "; diff=" + (contribs.vecs()[0].at(68) - -0.0008717066957615316), Math.abs(contribs.vecs()[0].at(68) - -0.0008717066957615316) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(71) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(71) + "; diff=" + (contribs.vecs()[4].at(71) - 0.0), Math.abs(contribs.vecs()[4].at(71) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(72) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(72) + "; diff=" + (contribs.vecs()[2].at(72) - 0.6356714367866516), Math.abs(contribs.vecs()[2].at(72) - 0.6356714367866516) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(74) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(74) + "; diff=" + (contribs.vecs()[1].at(74) - 0.005337131209671497), Math.abs(contribs.vecs()[1].at(74) - 0.005337131209671497) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(97) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(97) + "; diff=" + (contribs.vecs()[1].at(97) - 0.0035176132805645466), Math.abs(contribs.vecs()[1].at(97) - 0.0035176132805645466) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(98) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(98) + "; diff=" + (contribs.vecs()[6].at(98) - -0.1023840382695198), Math.abs(contribs.vecs()[6].at(98) - -0.1023840382695198) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(103) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(103) + "; diff=" + (contribs.vecs()[4].at(103) - 0.009511320851743221), Math.abs(contribs.vecs()[4].at(103) - 0.009511320851743221) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(106) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(106) + "; diff=" + (contribs.vecs()[1].at(106) - 0.03711974620819092), Math.abs(contribs.vecs()[1].at(106) - 0.03711974620819092) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(106) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(106) + "; diff=" + (contribs.vecs()[7].at(106) - 0.2395717203617096), Math.abs(contribs.vecs()[7].at(106) - 0.2395717203617096) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(109) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(109) + "; diff=" + (contribs.vecs()[4].at(109) - -0.005505495239049196), Math.abs(contribs.vecs()[4].at(109) - -0.005505495239049196) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(122) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(122) + "; diff=" + (contribs.vecs()[7].at(122) - 0.0), Math.abs(contribs.vecs()[7].at(122) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(135) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(135) + "; diff=" + (contribs.vecs()[4].at(135) - 0.003214907133951783), Math.abs(contribs.vecs()[4].at(135) - 0.003214907133951783) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(138) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(138) + "; diff=" + (contribs.vecs()[4].at(138) - 0.0), Math.abs(contribs.vecs()[4].at(138) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(139) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(139) + "; diff=" + (contribs.vecs()[4].at(139) - -0.07137253880500793), Math.abs(contribs.vecs()[4].at(139) - -0.07137253880500793) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(144) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(144) + "; diff=" + (contribs.vecs()[0].at(144) - 0.0006224248791113496), Math.abs(contribs.vecs()[0].at(144) - 0.0006224248791113496) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(159) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(159) + "; diff=" + (contribs.vecs()[2].at(159) - -0.04702525958418846), Math.abs(contribs.vecs()[2].at(159) - -0.04702525958418846) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(164) + "; diff=" + (contribs.vecs()[1].at(164) - 0.0205331202596426), Math.abs(contribs.vecs()[1].at(164) - 0.0205331202596426) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(167) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(167) + "; diff=" + (contribs.vecs()[3].at(167) - 0.2761104106903076), Math.abs(contribs.vecs()[3].at(167) - 0.2761104106903076) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(175) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(175) + "; diff=" + (contribs.vecs()[4].at(175) - 0.0), Math.abs(contribs.vecs()[4].at(175) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(175) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(175) + "; diff=" + (contribs.vecs()[5].at(175) - 0.018196264281868935), Math.abs(contribs.vecs()[5].at(175) - 0.018196264281868935) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(175) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(175) + "; diff=" + (contribs.vecs()[6].at(175) - 0.04160588979721069), Math.abs(contribs.vecs()[6].at(175) - 0.04160588979721069) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(178) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(178) + "; diff=" + (contribs.vecs()[1].at(178) - 0.015565533190965652), Math.abs(contribs.vecs()[1].at(178) - 0.015565533190965652) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(178) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(178) + "; diff=" + (contribs.vecs()[2].at(178) - 0.0), Math.abs(contribs.vecs()[2].at(178) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(192) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(192) + "; diff=" + (contribs.vecs()[6].at(192) - 0.051963962614536285), Math.abs(contribs.vecs()[6].at(192) - 0.051963962614536285) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(193) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(193) + "; diff=" + (contribs.vecs()[5].at(193) - 0.11259891092777252), Math.abs(contribs.vecs()[5].at(193) - 0.11259891092777252) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(195) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(195) + "; diff=" + (contribs.vecs()[0].at(195) - -0.0012207904364913702), Math.abs(contribs.vecs()[0].at(195) - -0.0012207904364913702) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(196) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(196) + "; diff=" + (contribs.vecs()[2].at(196) - 0.0), Math.abs(contribs.vecs()[2].at(196) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(198) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(198) + "; diff=" + (contribs.vecs()[6].at(198) - 0.08799633383750916), Math.abs(contribs.vecs()[6].at(198) - 0.08799633383750916) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(213) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(213) + "; diff=" + (contribs.vecs()[2].at(213) - -1.517763376235962), Math.abs(contribs.vecs()[2].at(213) - -1.517763376235962) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(227) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(227) + "; diff=" + (contribs.vecs()[2].at(227) - 0.0), Math.abs(contribs.vecs()[2].at(227) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(227) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(227) + "; diff=" + (contribs.vecs()[4].at(227) - 0.0), Math.abs(contribs.vecs()[4].at(227) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(228) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(228) + "; diff=" + (contribs.vecs()[5].at(228) - 0.8134714961051941), Math.abs(contribs.vecs()[5].at(228) - 0.8134714961051941) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(229) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(229) + "; diff=" + (contribs.vecs()[0].at(229) - -0.0027693328447639942), Math.abs(contribs.vecs()[0].at(229) - -0.0027693328447639942) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(230) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(230) + "; diff=" + (contribs.vecs()[1].at(230) - 0.03809114545583725), Math.abs(contribs.vecs()[1].at(230) - 0.03809114545583725) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(240) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(240) + "; diff=" + (contribs.vecs()[3].at(240) - 0.21017999947071075), Math.abs(contribs.vecs()[3].at(240) - 0.21017999947071075) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(243) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(243) + "; diff=" + (contribs.vecs()[6].at(243) - 0.034745994955301285), Math.abs(contribs.vecs()[6].at(243) - 0.034745994955301285) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(250) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(250) + "; diff=" + (contribs.vecs()[3].at(250) - 0.18998755514621735), Math.abs(contribs.vecs()[3].at(250) - 0.18998755514621735) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(257) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(257) + "; diff=" + (contribs.vecs()[6].at(257) - -0.01867113821208477), Math.abs(contribs.vecs()[6].at(257) - -0.01867113821208477) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(262) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(262) + "; diff=" + (contribs.vecs()[1].at(262) - -0.008823730982840061), Math.abs(contribs.vecs()[1].at(262) - -0.008823730982840061) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(280) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(280) + "; diff=" + (contribs.vecs()[5].at(280) - 0.1318257749080658), Math.abs(contribs.vecs()[5].at(280) - 0.1318257749080658) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(281) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(281) + "; diff=" + (contribs.vecs()[0].at(281) - -0.004609288647770882), Math.abs(contribs.vecs()[0].at(281) - -0.004609288647770882) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(289) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(289) + "; diff=" + (contribs.vecs()[4].at(289) - 0.0), Math.abs(contribs.vecs()[4].at(289) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(302) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(302) + "; diff=" + (contribs.vecs()[0].at(302) - -0.0003267067950218916), Math.abs(contribs.vecs()[0].at(302) - -0.0003267067950218916) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(306) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(306) + "; diff=" + (contribs.vecs()[0].at(306) - -0.0047950525768101215), Math.abs(contribs.vecs()[0].at(306) - -0.0047950525768101215) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(316) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(316) + "; diff=" + (contribs.vecs()[4].at(316) - -0.034691378474235535), Math.abs(contribs.vecs()[4].at(316) - -0.034691378474235535) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(321) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(321) + "; diff=" + (contribs.vecs()[6].at(321) - 0.0), Math.abs(contribs.vecs()[6].at(321) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(330) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(330) + "; diff=" + (contribs.vecs()[5].at(330) - 0.05748696252703667), Math.abs(contribs.vecs()[5].at(330) - 0.05748696252703667) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(336) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(336) + "; diff=" + (contribs.vecs()[4].at(336) - 0.0), Math.abs(contribs.vecs()[4].at(336) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(338) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(338) + "; diff=" + (contribs.vecs()[2].at(338) - 0.0), Math.abs(contribs.vecs()[2].at(338) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(340) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(340) + "; diff=" + (contribs.vecs()[5].at(340) - 0.07024893164634705), Math.abs(contribs.vecs()[5].at(340) - 0.07024893164634705) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(347) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(347) + "; diff=" + (contribs.vecs()[2].at(347) - 0.0), Math.abs(contribs.vecs()[2].at(347) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(349) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(349) + "; diff=" + (contribs.vecs()[3].at(349) - 0.22013822197914124), Math.abs(contribs.vecs()[3].at(349) - 0.22013822197914124) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(350) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(350) + "; diff=" + (contribs.vecs()[4].at(350) - 0.0), Math.abs(contribs.vecs()[4].at(350) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(355) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(355) + "; diff=" + (contribs.vecs()[3].at(355) - 0.21807730197906494), Math.abs(contribs.vecs()[3].at(355) - 0.21807730197906494) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(358) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(358) + "; diff=" + (contribs.vecs()[4].at(358) - -0.29787445068359375), Math.abs(contribs.vecs()[4].at(358) - -0.29787445068359375) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(363) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(363) + "; diff=" + (contribs.vecs()[4].at(363) - -0.14200395345687866), Math.abs(contribs.vecs()[4].at(363) - -0.14200395345687866) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(380) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(380) + "; diff=" + (contribs.vecs()[5].at(380) - -0.008395775221288204), Math.abs(contribs.vecs()[5].at(380) - -0.008395775221288204) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(389) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(389) + "; diff=" + (contribs.vecs()[1].at(389) - 0.060086384415626526), Math.abs(contribs.vecs()[1].at(389) - 0.060086384415626526) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(389) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(389) + "; diff=" + (contribs.vecs()[2].at(389) - 0.0), Math.abs(contribs.vecs()[2].at(389) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(393) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(393) + "; diff=" + (contribs.vecs()[1].at(393) - -0.010251793079078197), Math.abs(contribs.vecs()[1].at(393) - -0.010251793079078197) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(398) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(398) + "; diff=" + (contribs.vecs()[7].at(398) - 0.0), Math.abs(contribs.vecs()[7].at(398) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(402) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(402) + "; diff=" + (contribs.vecs()[5].at(402) - -0.3221670687198639), Math.abs(contribs.vecs()[5].at(402) - -0.3221670687198639) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(425) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(425) + "; diff=" + (contribs.vecs()[7].at(425) - 0.04865093529224396), Math.abs(contribs.vecs()[7].at(425) - 0.04865093529224396) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(430) + "; diff=" + (contribs.vecs()[2].at(430) - 0.0), Math.abs(contribs.vecs()[2].at(430) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(433) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(433) + "; diff=" + (contribs.vecs()[4].at(433) - 0.0), Math.abs(contribs.vecs()[4].at(433) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(444) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(444) + "; diff=" + (contribs.vecs()[7].at(444) - 0.0), Math.abs(contribs.vecs()[7].at(444) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(481) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(481) + "; diff=" + (contribs.vecs()[0].at(481) - 0.002312108175829053), Math.abs(contribs.vecs()[0].at(481) - 0.002312108175829053) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(482) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(482) + "; diff=" + (contribs.vecs()[0].at(482) - -0.0012754183262586594), Math.abs(contribs.vecs()[0].at(482) - -0.0012754183262586594) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(486) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(486) + "; diff=" + (contribs.vecs()[5].at(486) - -0.0004020605992991477), Math.abs(contribs.vecs()[5].at(486) - -0.0004020605992991477) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(496) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(496) + "; diff=" + (contribs.vecs()[6].at(496) - 0.22847186028957367), Math.abs(contribs.vecs()[6].at(496) - 0.22847186028957367) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(504) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(504) + "; diff=" + (contribs.vecs()[4].at(504) - -0.28271928429603577), Math.abs(contribs.vecs()[4].at(504) - -0.28271928429603577) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(508) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(508) + "; diff=" + (contribs.vecs()[6].at(508) - 0.05244000628590584), Math.abs(contribs.vecs()[6].at(508) - 0.05244000628590584) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(512) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(512) + "; diff=" + (contribs.vecs()[6].at(512) - -0.029761070385575294), Math.abs(contribs.vecs()[6].at(512) - -0.029761070385575294) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(515) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(515) + "; diff=" + (contribs.vecs()[0].at(515) - -0.00326751172542572), Math.abs(contribs.vecs()[0].at(515) - -0.00326751172542572) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(525) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(525) + "; diff=" + (contribs.vecs()[0].at(525) - -0.00017144682351499796), Math.abs(contribs.vecs()[0].at(525) - -0.00017144682351499796) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(528) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(528) + "; diff=" + (contribs.vecs()[4].at(528) - -0.1649795025587082), Math.abs(contribs.vecs()[4].at(528) - -0.1649795025587082) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(538) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(538) + "; diff=" + (contribs.vecs()[7].at(538) - -0.14994977414608002), Math.abs(contribs.vecs()[7].at(538) - -0.14994977414608002) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(550) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(550) + "; diff=" + (contribs.vecs()[4].at(550) - 0.0), Math.abs(contribs.vecs()[4].at(550) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(561) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(561) + "; diff=" + (contribs.vecs()[5].at(561) - 0.10144419968128204), Math.abs(contribs.vecs()[5].at(561) - 0.10144419968128204) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(565) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(565) + "; diff=" + (contribs.vecs()[7].at(565) - 0.21019183099269867), Math.abs(contribs.vecs()[7].at(565) - 0.21019183099269867) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(566) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(566) + "; diff=" + (contribs.vecs()[4].at(566) - -0.16735494136810303), Math.abs(contribs.vecs()[4].at(566) - -0.16735494136810303) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(567) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(567) + "; diff=" + (contribs.vecs()[6].at(567) - 0.006263695657253265), Math.abs(contribs.vecs()[6].at(567) - 0.006263695657253265) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(570) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(570) + "; diff=" + (contribs.vecs()[3].at(570) - 0.015948766842484474), Math.abs(contribs.vecs()[3].at(570) - 0.015948766842484474) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(579) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(579) + "; diff=" + (contribs.vecs()[1].at(579) - -0.1549777239561081), Math.abs(contribs.vecs()[1].at(579) - -0.1549777239561081) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(581) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(581) + "; diff=" + (contribs.vecs()[4].at(581) - 0.0), Math.abs(contribs.vecs()[4].at(581) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(594) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(594) + "; diff=" + (contribs.vecs()[2].at(594) - 0.0), Math.abs(contribs.vecs()[2].at(594) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(598) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(598) + "; diff=" + (contribs.vecs()[4].at(598) - -0.3411220610141754), Math.abs(contribs.vecs()[4].at(598) - -0.3411220610141754) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(602) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(602) + "; diff=" + (contribs.vecs()[5].at(602) - 0.05384654924273491), Math.abs(contribs.vecs()[5].at(602) - 0.05384654924273491) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(605) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(605) + "; diff=" + (contribs.vecs()[4].at(605) - 0.0), Math.abs(contribs.vecs()[4].at(605) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(616) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(616) + "; diff=" + (contribs.vecs()[7].at(616) - 0.19281955063343048), Math.abs(contribs.vecs()[7].at(616) - 0.19281955063343048) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(620) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(620) + "; diff=" + (contribs.vecs()[4].at(620) - 0.0), Math.abs(contribs.vecs()[4].at(620) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(651) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(651) + "; diff=" + (contribs.vecs()[0].at(651) - -0.0031498868484050035), Math.abs(contribs.vecs()[0].at(651) - -0.0031498868484050035) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(660) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(660) + "; diff=" + (contribs.vecs()[0].at(660) - -0.0012070555239915848), Math.abs(contribs.vecs()[0].at(660) - -0.0012070555239915848) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(671) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(671) + "; diff=" + (contribs.vecs()[1].at(671) - 0.03709922730922699), Math.abs(contribs.vecs()[1].at(671) - 0.03709922730922699) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(671) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(671) + "; diff=" + (contribs.vecs()[4].at(671) - 0.0), Math.abs(contribs.vecs()[4].at(671) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(671) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(671) + "; diff=" + (contribs.vecs()[5].at(671) - 0.11867019534111023), Math.abs(contribs.vecs()[5].at(671) - 0.11867019534111023) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(672) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(672) + "; diff=" + (contribs.vecs()[5].at(672) - 0.06331565231084824), Math.abs(contribs.vecs()[5].at(672) - 0.06331565231084824) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(673) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(673) + "; diff=" + (contribs.vecs()[5].at(673) - -4.895442179986276e-05), Math.abs(contribs.vecs()[5].at(673) - -4.895442179986276e-05) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(675) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(675) + "; diff=" + (contribs.vecs()[3].at(675) - 0.0), Math.abs(contribs.vecs()[3].at(675) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(679) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(679) + "; diff=" + (contribs.vecs()[2].at(679) - -0.36490175127983093), Math.abs(contribs.vecs()[2].at(679) - -0.36490175127983093) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(685) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(685) + "; diff=" + (contribs.vecs()[2].at(685) - 0.0), Math.abs(contribs.vecs()[2].at(685) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(692) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(692) + "; diff=" + (contribs.vecs()[6].at(692) - -0.08957762271165848), Math.abs(contribs.vecs()[6].at(692) - -0.08957762271165848) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(693) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(693) + "; diff=" + (contribs.vecs()[4].at(693) - 0.0), Math.abs(contribs.vecs()[4].at(693) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(710) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(710) + "; diff=" + (contribs.vecs()[0].at(710) - -0.00011390092549845576), Math.abs(contribs.vecs()[0].at(710) - -0.00011390092549845576) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(715) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(715) + "; diff=" + (contribs.vecs()[0].at(715) - -0.00037478431477211416), Math.abs(contribs.vecs()[0].at(715) - -0.00037478431477211416) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(718) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(718) + "; diff=" + (contribs.vecs()[7].at(718) - 0.21685810387134552), Math.abs(contribs.vecs()[7].at(718) - 0.21685810387134552) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(719) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(719) + "; diff=" + (contribs.vecs()[4].at(719) - -0.14024557173252106), Math.abs(contribs.vecs()[4].at(719) - -0.14024557173252106) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(725) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(725) + "; diff=" + (contribs.vecs()[3].at(725) - 0.0), Math.abs(contribs.vecs()[3].at(725) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(725) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(725) + "; diff=" + (contribs.vecs()[4].at(725) - -0.28405728936195374), Math.abs(contribs.vecs()[4].at(725) - -0.28405728936195374) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(728) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(728) + "; diff=" + (contribs.vecs()[6].at(728) - 0.0), Math.abs(contribs.vecs()[6].at(728) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(729) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(729) + "; diff=" + (contribs.vecs()[4].at(729) - 0.0), Math.abs(contribs.vecs()[4].at(729) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(730) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(730) + "; diff=" + (contribs.vecs()[1].at(730) - 0.06257335841655731), Math.abs(contribs.vecs()[1].at(730) - 0.06257335841655731) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(734) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(734) + "; diff=" + (contribs.vecs()[0].at(734) - -0.013252747245132923), Math.abs(contribs.vecs()[0].at(734) - -0.013252747245132923) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(735) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(735) + "; diff=" + (contribs.vecs()[3].at(735) - -0.551484227180481), Math.abs(contribs.vecs()[3].at(735) - -0.551484227180481) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(746) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(746) + "; diff=" + (contribs.vecs()[2].at(746) - 0.0), Math.abs(contribs.vecs()[2].at(746) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(748) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(748) + "; diff=" + (contribs.vecs()[6].at(748) - 0.06345286965370178), Math.abs(contribs.vecs()[6].at(748) - 0.06345286965370178) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(765) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(765) + "; diff=" + (contribs.vecs()[2].at(765) - 0.0), Math.abs(contribs.vecs()[2].at(765) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(767) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(767) + "; diff=" + (contribs.vecs()[1].at(767) - 0.027414578944444656), Math.abs(contribs.vecs()[1].at(767) - 0.027414578944444656) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(770) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(770) + "; diff=" + (contribs.vecs()[4].at(770) - -0.18696460127830505), Math.abs(contribs.vecs()[4].at(770) - -0.18696460127830505) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(771) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(771) + "; diff=" + (contribs.vecs()[4].at(771) - -0.12462224811315536), Math.abs(contribs.vecs()[4].at(771) - -0.12462224811315536) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(789) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(789) + "; diff=" + (contribs.vecs()[4].at(789) - 0.0), Math.abs(contribs.vecs()[4].at(789) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(804) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(804) + "; diff=" + (contribs.vecs()[0].at(804) - -0.002962949685752392), Math.abs(contribs.vecs()[0].at(804) - -0.002962949685752392) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(810) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(810) + "; diff=" + (contribs.vecs()[3].at(810) - -0.20188173651695251), Math.abs(contribs.vecs()[3].at(810) - -0.20188173651695251) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(810) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(810) + "; diff=" + (contribs.vecs()[4].at(810) - -0.28023964166641235), Math.abs(contribs.vecs()[4].at(810) - -0.28023964166641235) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(819) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(819) + "; diff=" + (contribs.vecs()[3].at(819) - -0.11223693937063217), Math.abs(contribs.vecs()[3].at(819) - -0.11223693937063217) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(820) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(820) + "; diff=" + (contribs.vecs()[6].at(820) - -0.21019192039966583), Math.abs(contribs.vecs()[6].at(820) - -0.21019192039966583) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(832) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(832) + "; diff=" + (contribs.vecs()[6].at(832) - 0.0), Math.abs(contribs.vecs()[6].at(832) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(837) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(837) + "; diff=" + (contribs.vecs()[4].at(837) - 0.0), Math.abs(contribs.vecs()[4].at(837) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(845) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(845) + "; diff=" + (contribs.vecs()[1].at(845) - -0.04494836926460266), Math.abs(contribs.vecs()[1].at(845) - -0.04494836926460266) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(852) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(852) + "; diff=" + (contribs.vecs()[7].at(852) - -0.24774812161922455), Math.abs(contribs.vecs()[7].at(852) - -0.24774812161922455) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(853) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(853) + "; diff=" + (contribs.vecs()[3].at(853) - 0.0), Math.abs(contribs.vecs()[3].at(853) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(874) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(874) + "; diff=" + (contribs.vecs()[2].at(874) - -0.11167852580547333), Math.abs(contribs.vecs()[2].at(874) - -0.11167852580547333) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(874) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(874) + "; diff=" + (contribs.vecs()[4].at(874) - -0.10100284218788147), Math.abs(contribs.vecs()[4].at(874) - -0.10100284218788147) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(885) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(885) + "; diff=" + (contribs.vecs()[1].at(885) - -0.013131152838468552), Math.abs(contribs.vecs()[1].at(885) - -0.013131152838468552) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(887) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(887) + "; diff=" + (contribs.vecs()[6].at(887) - -0.015229648910462856), Math.abs(contribs.vecs()[6].at(887) - -0.015229648910462856) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(888) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(888) + "; diff=" + (contribs.vecs()[1].at(888) - 0.009290995076298714), Math.abs(contribs.vecs()[1].at(888) - 0.009290995076298714) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(888) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(888) + "; diff=" + (contribs.vecs()[6].at(888) - -0.0035724646877497435), Math.abs(contribs.vecs()[6].at(888) - -0.0035724646877497435) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(891) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(891) + "; diff=" + (contribs.vecs()[6].at(891) - 0.17885321378707886), Math.abs(contribs.vecs()[6].at(891) - 0.17885321378707886) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(894) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(894) + "; diff=" + (contribs.vecs()[4].at(894) - 0.0), Math.abs(contribs.vecs()[4].at(894) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(901) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(901) + "; diff=" + (contribs.vecs()[5].at(901) - -0.030082590878009796), Math.abs(contribs.vecs()[5].at(901) - -0.030082590878009796) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(903) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(903) + "; diff=" + (contribs.vecs()[3].at(903) - 0.0), Math.abs(contribs.vecs()[3].at(903) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(904) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(904) + "; diff=" + (contribs.vecs()[3].at(904) - 0.1725825071334839), Math.abs(contribs.vecs()[3].at(904) - 0.1725825071334839) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(904) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(904) + "; diff=" + (contribs.vecs()[4].at(904) - -0.22601692378520966), Math.abs(contribs.vecs()[4].at(904) - -0.22601692378520966) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(913) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(913) + "; diff=" + (contribs.vecs()[4].at(913) - 0.0), Math.abs(contribs.vecs()[4].at(913) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(917) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(917) + "; diff=" + (contribs.vecs()[3].at(917) - 0.09440826624631882), Math.abs(contribs.vecs()[3].at(917) - 0.09440826624631882) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(926) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(926) + "; diff=" + (contribs.vecs()[1].at(926) - 0.0022241082042455673), Math.abs(contribs.vecs()[1].at(926) - 0.0022241082042455673) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(927) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(927) + "; diff=" + (contribs.vecs()[3].at(927) - 2.547128200531006), Math.abs(contribs.vecs()[3].at(927) - 2.547128200531006) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(930) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(930) + "; diff=" + (contribs.vecs()[3].at(930) - -0.056284964084625244), Math.abs(contribs.vecs()[3].at(930) - -0.056284964084625244) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(931) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(931) + "; diff=" + (contribs.vecs()[0].at(931) - 0.0017306468216702342), Math.abs(contribs.vecs()[0].at(931) - 0.0017306468216702342) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(938) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(938) + "; diff=" + (contribs.vecs()[7].at(938) - -0.15465492010116577), Math.abs(contribs.vecs()[7].at(938) - -0.15465492010116577) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(942) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(942) + "; diff=" + (contribs.vecs()[7].at(942) - 0.5451818108558655), Math.abs(contribs.vecs()[7].at(942) - 0.5451818108558655) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(958) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(958) + "; diff=" + (contribs.vecs()[4].at(958) - 0.0), Math.abs(contribs.vecs()[4].at(958) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(960) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(960) + "; diff=" + (contribs.vecs()[5].at(960) - 0.01734853908419609), Math.abs(contribs.vecs()[5].at(960) - 0.01734853908419609) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(967) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(967) + "; diff=" + (contribs.vecs()[4].at(967) - 0.0), Math.abs(contribs.vecs()[4].at(967) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(981) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(981) + "; diff=" + (contribs.vecs()[7].at(981) - -0.28199946880340576), Math.abs(contribs.vecs()[7].at(981) - -0.28199946880340576) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(983) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(983) + "; diff=" + (contribs.vecs()[5].at(983) - -0.018257103860378265), Math.abs(contribs.vecs()[5].at(983) - -0.018257103860378265) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(984) + "; diff=" + (contribs.vecs()[1].at(984) - 0.0033155090641230345), Math.abs(contribs.vecs()[1].at(984) - 0.0033155090641230345) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(984) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(984) + "; diff=" + (contribs.vecs()[6].at(984) - 0.0), Math.abs(contribs.vecs()[6].at(984) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(988) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(988) + "; diff=" + (contribs.vecs()[7].at(988) - -0.12848593294620514), Math.abs(contribs.vecs()[7].at(988) - -0.12848593294620514) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(989) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(989) + "; diff=" + (contribs.vecs()[0].at(989) - -0.005300444550812244), Math.abs(contribs.vecs()[0].at(989) - -0.005300444550812244) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(995) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(995) + "; diff=" + (contribs.vecs()[1].at(995) - -0.001691723708063364), Math.abs(contribs.vecs()[1].at(995) - -0.001691723708063364) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(999) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(999) + "; diff=" + (contribs.vecs()[2].at(999) - 0.0), Math.abs(contribs.vecs()[2].at(999) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1000) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1000) + "; diff=" + (contribs.vecs()[0].at(1000) - -0.0026468855794519186), Math.abs(contribs.vecs()[0].at(1000) - -0.0026468855794519186) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1005) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1005) + "; diff=" + (contribs.vecs()[1].at(1005) - -0.0022628754377365112), Math.abs(contribs.vecs()[1].at(1005) - -0.0022628754377365112) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1013) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1013) + "; diff=" + (contribs.vecs()[3].at(1013) - -0.07087628543376923), Math.abs(contribs.vecs()[3].at(1013) - -0.07087628543376923) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1014) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1014) + "; diff=" + (contribs.vecs()[0].at(1014) - -0.0017958410317078233), Math.abs(contribs.vecs()[0].at(1014) - -0.0017958410317078233) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1015) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1015) + "; diff=" + (contribs.vecs()[5].at(1015) - -0.014173241332173347), Math.abs(contribs.vecs()[5].at(1015) - -0.014173241332173347) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1017) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1017) + "; diff=" + (contribs.vecs()[6].at(1017) - 0.017004145309329033), Math.abs(contribs.vecs()[6].at(1017) - 0.017004145309329033) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1022) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1022) + "; diff=" + (contribs.vecs()[3].at(1022) - 0.463029146194458), Math.abs(contribs.vecs()[3].at(1022) - 0.463029146194458) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1029) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1029) + "; diff=" + (contribs.vecs()[1].at(1029) - -0.0053933411836624146), Math.abs(contribs.vecs()[1].at(1029) - -0.0053933411836624146) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1032) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1032) + "; diff=" + (contribs.vecs()[0].at(1032) - -0.0011205776827409863), Math.abs(contribs.vecs()[0].at(1032) - -0.0011205776827409863) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1032) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1032) + "; diff=" + (contribs.vecs()[7].at(1032) - -0.39356496930122375), Math.abs(contribs.vecs()[7].at(1032) - -0.39356496930122375) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1033) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1033) + "; diff=" + (contribs.vecs()[2].at(1033) - 0.0), Math.abs(contribs.vecs()[2].at(1033) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1037) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1037) + "; diff=" + (contribs.vecs()[0].at(1037) - -0.0020911283791065216), Math.abs(contribs.vecs()[0].at(1037) - -0.0020911283791065216) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1039) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1039) + "; diff=" + (contribs.vecs()[1].at(1039) - -0.008584234863519669), Math.abs(contribs.vecs()[1].at(1039) - -0.008584234863519669) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1041) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1041) + "; diff=" + (contribs.vecs()[3].at(1041) - -0.16396939754486084), Math.abs(contribs.vecs()[3].at(1041) - -0.16396939754486084) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1064) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1064) + "; diff=" + (contribs.vecs()[3].at(1064) - 0.0), Math.abs(contribs.vecs()[3].at(1064) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1080) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1080) + "; diff=" + (contribs.vecs()[1].at(1080) - 0.501024603843689), Math.abs(contribs.vecs()[1].at(1080) - 0.501024603843689) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1082) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1082) + "; diff=" + (contribs.vecs()[3].at(1082) - 0.381587415933609), Math.abs(contribs.vecs()[3].at(1082) - 0.381587415933609) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1085) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1085) + "; diff=" + (contribs.vecs()[6].at(1085) - 0.0), Math.abs(contribs.vecs()[6].at(1085) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1091) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1091) + "; diff=" + (contribs.vecs()[3].at(1091) - 0.23668000102043152), Math.abs(contribs.vecs()[3].at(1091) - 0.23668000102043152) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1092) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1092) + "; diff=" + (contribs.vecs()[7].at(1092) - 0.0), Math.abs(contribs.vecs()[7].at(1092) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1093) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1093) + "; diff=" + (contribs.vecs()[4].at(1093) - 0.0), Math.abs(contribs.vecs()[4].at(1093) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1097) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1097) + "; diff=" + (contribs.vecs()[7].at(1097) - 0.0), Math.abs(contribs.vecs()[7].at(1097) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1101) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1101) + "; diff=" + (contribs.vecs()[5].at(1101) - -0.07860700786113739), Math.abs(contribs.vecs()[5].at(1101) - -0.07860700786113739) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1107) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1107) + "; diff=" + (contribs.vecs()[0].at(1107) - -0.004041729029268026), Math.abs(contribs.vecs()[0].at(1107) - -0.004041729029268026) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1115) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1115) + "; diff=" + (contribs.vecs()[1].at(1115) - 0.0033762939274311066), Math.abs(contribs.vecs()[1].at(1115) - 0.0033762939274311066) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1124) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1124) + "; diff=" + (contribs.vecs()[6].at(1124) - -0.0993354544043541), Math.abs(contribs.vecs()[6].at(1124) - -0.0993354544043541) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1134) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1134) + "; diff=" + (contribs.vecs()[0].at(1134) - -0.0015937892021611333), Math.abs(contribs.vecs()[0].at(1134) - -0.0015937892021611333) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1135) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1135) + "; diff=" + (contribs.vecs()[2].at(1135) - 0.0), Math.abs(contribs.vecs()[2].at(1135) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1150) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1150) + "; diff=" + (contribs.vecs()[3].at(1150) - 0.0), Math.abs(contribs.vecs()[3].at(1150) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1150) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1150) + "; diff=" + (contribs.vecs()[4].at(1150) - 0.0), Math.abs(contribs.vecs()[4].at(1150) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1151) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1151) + "; diff=" + (contribs.vecs()[4].at(1151) - 0.0), Math.abs(contribs.vecs()[4].at(1151) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1153) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1153) + "; diff=" + (contribs.vecs()[3].at(1153) - 0.19632399082183838), Math.abs(contribs.vecs()[3].at(1153) - 0.19632399082183838) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1162) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1162) + "; diff=" + (contribs.vecs()[6].at(1162) - -0.005759126506745815), Math.abs(contribs.vecs()[6].at(1162) - -0.005759126506745815) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1166) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1166) + "; diff=" + (contribs.vecs()[1].at(1166) - 0.0), Math.abs(contribs.vecs()[1].at(1166) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1185) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1185) + "; diff=" + (contribs.vecs()[6].at(1185) - 0.0), Math.abs(contribs.vecs()[6].at(1185) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1189) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1189) + "; diff=" + (contribs.vecs()[1].at(1189) - 0.16015739738941193), Math.abs(contribs.vecs()[1].at(1189) - 0.16015739738941193) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1193) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1193) + "; diff=" + (contribs.vecs()[5].at(1193) - 0.041188839823007584), Math.abs(contribs.vecs()[5].at(1193) - 0.041188839823007584) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1196) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1196) + "; diff=" + (contribs.vecs()[2].at(1196) - 0.0), Math.abs(contribs.vecs()[2].at(1196) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1200) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1200) + "; diff=" + (contribs.vecs()[3].at(1200) - -0.4612942337989807), Math.abs(contribs.vecs()[3].at(1200) - -0.4612942337989807) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1208) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1208) + "; diff=" + (contribs.vecs()[5].at(1208) - 0.014487700536847115), Math.abs(contribs.vecs()[5].at(1208) - 0.014487700536847115) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1216) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1216) + "; diff=" + (contribs.vecs()[3].at(1216) - -0.16777344048023224), Math.abs(contribs.vecs()[3].at(1216) - -0.16777344048023224) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1233) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1233) + "; diff=" + (contribs.vecs()[7].at(1233) - 0.40806272625923157), Math.abs(contribs.vecs()[7].at(1233) - 0.40806272625923157) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1254) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1254) + "; diff=" + (contribs.vecs()[2].at(1254) - 0.2661345303058624), Math.abs(contribs.vecs()[2].at(1254) - 0.2661345303058624) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1261) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1261) + "; diff=" + (contribs.vecs()[5].at(1261) - -3.249119281768799), Math.abs(contribs.vecs()[5].at(1261) - -3.249119281768799) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1269) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1269) + "; diff=" + (contribs.vecs()[0].at(1269) - -0.015390398912131786), Math.abs(contribs.vecs()[0].at(1269) - -0.015390398912131786) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1273) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1273) + "; diff=" + (contribs.vecs()[6].at(1273) - -0.01992523856461048), Math.abs(contribs.vecs()[6].at(1273) - -0.01992523856461048) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1275) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1275) + "; diff=" + (contribs.vecs()[5].at(1275) - 0.0006850137724541128), Math.abs(contribs.vecs()[5].at(1275) - 0.0006850137724541128) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1277) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1277) + "; diff=" + (contribs.vecs()[1].at(1277) - 0.005575789604336023), Math.abs(contribs.vecs()[1].at(1277) - 0.005575789604336023) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1279) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1279) + "; diff=" + (contribs.vecs()[5].at(1279) - -0.014218656346201897), Math.abs(contribs.vecs()[5].at(1279) - -0.014218656346201897) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1279) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1279) + "; diff=" + (contribs.vecs()[7].at(1279) - 0.0), Math.abs(contribs.vecs()[7].at(1279) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1287) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1287) + "; diff=" + (contribs.vecs()[3].at(1287) - -0.09425235539674759), Math.abs(contribs.vecs()[3].at(1287) - -0.09425235539674759) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1289) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1289) + "; diff=" + (contribs.vecs()[6].at(1289) - -0.01751715876162052), Math.abs(contribs.vecs()[6].at(1289) - -0.01751715876162052) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1295) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1295) + "; diff=" + (contribs.vecs()[3].at(1295) - 0.2736222743988037), Math.abs(contribs.vecs()[3].at(1295) - 0.2736222743988037) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1296) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1296) + "; diff=" + (contribs.vecs()[0].at(1296) - 0.0076947868801653385), Math.abs(contribs.vecs()[0].at(1296) - 0.0076947868801653385) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1320) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1320) + "; diff=" + (contribs.vecs()[1].at(1320) - 0.011189385317265987), Math.abs(contribs.vecs()[1].at(1320) - 0.011189385317265987) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1323) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1323) + "; diff=" + (contribs.vecs()[7].at(1323) - 0.05488421395421028), Math.abs(contribs.vecs()[7].at(1323) - 0.05488421395421028) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1328) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1328) + "; diff=" + (contribs.vecs()[1].at(1328) - 0.02476864494383335), Math.abs(contribs.vecs()[1].at(1328) - 0.02476864494383335) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1328) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1328) + "; diff=" + (contribs.vecs()[7].at(1328) - 0.0), Math.abs(contribs.vecs()[7].at(1328) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1331) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1331) + "; diff=" + (contribs.vecs()[7].at(1331) - -0.29173359274864197), Math.abs(contribs.vecs()[7].at(1331) - -0.29173359274864197) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1333) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1333) + "; diff=" + (contribs.vecs()[5].at(1333) - -0.15405507385730743), Math.abs(contribs.vecs()[5].at(1333) - -0.15405507385730743) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1334) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1334) + "; diff=" + (contribs.vecs()[7].at(1334) - -0.10335371643304825), Math.abs(contribs.vecs()[7].at(1334) - -0.10335371643304825) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1344) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1344) + "; diff=" + (contribs.vecs()[4].at(1344) - -0.1100127249956131), Math.abs(contribs.vecs()[4].at(1344) - -0.1100127249956131) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1346) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1346) + "; diff=" + (contribs.vecs()[1].at(1346) - -0.011288557201623917), Math.abs(contribs.vecs()[1].at(1346) - -0.011288557201623917) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1347) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1347) + "; diff=" + (contribs.vecs()[1].at(1347) - 0.022208405658602715), Math.abs(contribs.vecs()[1].at(1347) - 0.022208405658602715) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1354) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1354) + "; diff=" + (contribs.vecs()[4].at(1354) - 0.0), Math.abs(contribs.vecs()[4].at(1354) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1354) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1354) + "; diff=" + (contribs.vecs()[6].at(1354) - 2.684647798538208), Math.abs(contribs.vecs()[6].at(1354) - 2.684647798538208) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1364) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1364) + "; diff=" + (contribs.vecs()[5].at(1364) - -0.3347826600074768), Math.abs(contribs.vecs()[5].at(1364) - -0.3347826600074768) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1365) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1365) + "; diff=" + (contribs.vecs()[3].at(1365) - -0.18140916526317596), Math.abs(contribs.vecs()[3].at(1365) - -0.18140916526317596) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1368) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1368) + "; diff=" + (contribs.vecs()[2].at(1368) - 0.0), Math.abs(contribs.vecs()[2].at(1368) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1368) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1368) + "; diff=" + (contribs.vecs()[5].at(1368) - -0.07368536293506622), Math.abs(contribs.vecs()[5].at(1368) - -0.07368536293506622) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1374) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1374) + "; diff=" + (contribs.vecs()[3].at(1374) - 0.0), Math.abs(contribs.vecs()[3].at(1374) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1379) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1379) + "; diff=" + (contribs.vecs()[0].at(1379) - -0.008831405080854893), Math.abs(contribs.vecs()[0].at(1379) - -0.008831405080854893) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1392) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1392) + "; diff=" + (contribs.vecs()[0].at(1392) - -0.004599761683493853), Math.abs(contribs.vecs()[0].at(1392) - -0.004599761683493853) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1400) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1400) + "; diff=" + (contribs.vecs()[1].at(1400) - 0.015421858988702297), Math.abs(contribs.vecs()[1].at(1400) - 0.015421858988702297) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1413) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1413) + "; diff=" + (contribs.vecs()[4].at(1413) - 0.0), Math.abs(contribs.vecs()[4].at(1413) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1420) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1420) + "; diff=" + (contribs.vecs()[3].at(1420) - 0.0), Math.abs(contribs.vecs()[3].at(1420) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1425) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1425) + "; diff=" + (contribs.vecs()[2].at(1425) - 0.0), Math.abs(contribs.vecs()[2].at(1425) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1429) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1429) + "; diff=" + (contribs.vecs()[2].at(1429) - 0.0), Math.abs(contribs.vecs()[2].at(1429) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1430) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1430) + "; diff=" + (contribs.vecs()[1].at(1430) - 0.01700463704764843), Math.abs(contribs.vecs()[1].at(1430) - 0.01700463704764843) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1435) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1435) + "; diff=" + (contribs.vecs()[1].at(1435) - -0.0012532093096524477), Math.abs(contribs.vecs()[1].at(1435) - -0.0012532093096524477) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1439) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1439) + "; diff=" + (contribs.vecs()[0].at(1439) - -0.01600862480700016), Math.abs(contribs.vecs()[0].at(1439) - -0.01600862480700016) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1442) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1442) + "; diff=" + (contribs.vecs()[1].at(1442) - 0.021479740738868713), Math.abs(contribs.vecs()[1].at(1442) - 0.021479740738868713) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1447) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1447) + "; diff=" + (contribs.vecs()[3].at(1447) - -0.07704043388366699), Math.abs(contribs.vecs()[3].at(1447) - -0.07704043388366699) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1447) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1447) + "; diff=" + (contribs.vecs()[7].at(1447) - 0.057848796248435974), Math.abs(contribs.vecs()[7].at(1447) - 0.057848796248435974) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1448) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1448) + "; diff=" + (contribs.vecs()[0].at(1448) - -0.01381778996437788), Math.abs(contribs.vecs()[0].at(1448) - -0.01381778996437788) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1448) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1448) + "; diff=" + (contribs.vecs()[1].at(1448) - -0.04383111000061035), Math.abs(contribs.vecs()[1].at(1448) - -0.04383111000061035) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1449) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1449) + "; diff=" + (contribs.vecs()[5].at(1449) - 0.02370190992951393), Math.abs(contribs.vecs()[5].at(1449) - 0.02370190992951393) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1450) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1450) + "; diff=" + (contribs.vecs()[5].at(1450) - 0.3815477192401886), Math.abs(contribs.vecs()[5].at(1450) - 0.3815477192401886) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1455) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1455) + "; diff=" + (contribs.vecs()[0].at(1455) - -0.00895262137055397), Math.abs(contribs.vecs()[0].at(1455) - -0.00895262137055397) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1456) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1456) + "; diff=" + (contribs.vecs()[7].at(1456) - 0.8663000464439392), Math.abs(contribs.vecs()[7].at(1456) - 0.8663000464439392) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1476) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1476) + "; diff=" + (contribs.vecs()[6].at(1476) - -0.03470806032419205), Math.abs(contribs.vecs()[6].at(1476) - -0.03470806032419205) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1477) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1477) + "; diff=" + (contribs.vecs()[1].at(1477) - 0.004093554336577654), Math.abs(contribs.vecs()[1].at(1477) - 0.004093554336577654) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1494) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1494) + "; diff=" + (contribs.vecs()[3].at(1494) - 0.10939344018697739), Math.abs(contribs.vecs()[3].at(1494) - 0.10939344018697739) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1495) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1495) + "; diff=" + (contribs.vecs()[2].at(1495) - -0.6267116069793701), Math.abs(contribs.vecs()[2].at(1495) - -0.6267116069793701) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1497) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1497) + "; diff=" + (contribs.vecs()[0].at(1497) - -0.01180319394916296), Math.abs(contribs.vecs()[0].at(1497) - -0.01180319394916296) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1497) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1497) + "; diff=" + (contribs.vecs()[2].at(1497) - 0.0), Math.abs(contribs.vecs()[2].at(1497) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1498) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1498) + "; diff=" + (contribs.vecs()[2].at(1498) - -0.16438470780849457), Math.abs(contribs.vecs()[2].at(1498) - -0.16438470780849457) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1527) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1527) + "; diff=" + (contribs.vecs()[0].at(1527) - -0.0027826272416859865), Math.abs(contribs.vecs()[0].at(1527) - -0.0027826272416859865) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1539) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1539) + "; diff=" + (contribs.vecs()[1].at(1539) - -0.003227922599762678), Math.abs(contribs.vecs()[1].at(1539) - -0.003227922599762678) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1540) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1540) + "; diff=" + (contribs.vecs()[3].at(1540) - -0.19028088450431824), Math.abs(contribs.vecs()[3].at(1540) - -0.19028088450431824) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1548) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1548) + "; diff=" + (contribs.vecs()[3].at(1548) - 0.0), Math.abs(contribs.vecs()[3].at(1548) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1555) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1555) + "; diff=" + (contribs.vecs()[3].at(1555) - 0.0), Math.abs(contribs.vecs()[3].at(1555) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1565) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1565) + "; diff=" + (contribs.vecs()[5].at(1565) - 0.7737926840782166), Math.abs(contribs.vecs()[5].at(1565) - 0.7737926840782166) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1570) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1570) + "; diff=" + (contribs.vecs()[5].at(1570) - 0.06915128976106644), Math.abs(contribs.vecs()[5].at(1570) - 0.06915128976106644) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1577) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1577) + "; diff=" + (contribs.vecs()[7].at(1577) - -0.1765524297952652), Math.abs(contribs.vecs()[7].at(1577) - -0.1765524297952652) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1578) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1578) + "; diff=" + (contribs.vecs()[0].at(1578) - -0.0036354686599224806), Math.abs(contribs.vecs()[0].at(1578) - -0.0036354686599224806) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1582) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1582) + "; diff=" + (contribs.vecs()[5].at(1582) - 0.007496808189898729), Math.abs(contribs.vecs()[5].at(1582) - 0.007496808189898729) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1589) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1589) + "; diff=" + (contribs.vecs()[2].at(1589) - 0.0), Math.abs(contribs.vecs()[2].at(1589) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1598) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1598) + "; diff=" + (contribs.vecs()[4].at(1598) - 0.0), Math.abs(contribs.vecs()[4].at(1598) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1599) + "; diff=" + (contribs.vecs()[0].at(1599) - -0.005717381369322538), Math.abs(contribs.vecs()[0].at(1599) - -0.005717381369322538) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1599) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1599) + "; diff=" + (contribs.vecs()[7].at(1599) - 0.3121662437915802), Math.abs(contribs.vecs()[7].at(1599) - 0.3121662437915802) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1619) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1619) + "; diff=" + (contribs.vecs()[4].at(1619) - 0.0), Math.abs(contribs.vecs()[4].at(1619) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1620) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1620) + "; diff=" + (contribs.vecs()[7].at(1620) - 0.0), Math.abs(contribs.vecs()[7].at(1620) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1624) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1624) + "; diff=" + (contribs.vecs()[1].at(1624) - -0.011914636008441448), Math.abs(contribs.vecs()[1].at(1624) - -0.011914636008441448) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1630) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1630) + "; diff=" + (contribs.vecs()[7].at(1630) - 0.23249287903308868), Math.abs(contribs.vecs()[7].at(1630) - 0.23249287903308868) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1632) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1632) + "; diff=" + (contribs.vecs()[0].at(1632) - -0.006621595472097397), Math.abs(contribs.vecs()[0].at(1632) - -0.006621595472097397) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1632) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1632) + "; diff=" + (contribs.vecs()[7].at(1632) - 0.1732502579689026), Math.abs(contribs.vecs()[7].at(1632) - 0.1732502579689026) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1639) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1639) + "; diff=" + (contribs.vecs()[2].at(1639) - -0.08769787847995758), Math.abs(contribs.vecs()[2].at(1639) - -0.08769787847995758) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1640) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1640) + "; diff=" + (contribs.vecs()[2].at(1640) - 0.0), Math.abs(contribs.vecs()[2].at(1640) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1643) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1643) + "; diff=" + (contribs.vecs()[1].at(1643) - -0.015649089589715004), Math.abs(contribs.vecs()[1].at(1643) - -0.015649089589715004) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1643) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1643) + "; diff=" + (contribs.vecs()[7].at(1643) - 0.0), Math.abs(contribs.vecs()[7].at(1643) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1646) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1646) + "; diff=" + (contribs.vecs()[5].at(1646) - 0.013410779647529125), Math.abs(contribs.vecs()[5].at(1646) - 0.013410779647529125) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1649) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1649) + "; diff=" + (contribs.vecs()[1].at(1649) - -0.009502182714641094), Math.abs(contribs.vecs()[1].at(1649) - -0.009502182714641094) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1656) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1656) + "; diff=" + (contribs.vecs()[3].at(1656) - 0.449634850025177), Math.abs(contribs.vecs()[3].at(1656) - 0.449634850025177) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1658) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1658) + "; diff=" + (contribs.vecs()[0].at(1658) - -0.006743815261870623), Math.abs(contribs.vecs()[0].at(1658) - -0.006743815261870623) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1664) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1664) + "; diff=" + (contribs.vecs()[1].at(1664) - 0.005839791148900986), Math.abs(contribs.vecs()[1].at(1664) - 0.005839791148900986) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1670) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1670) + "; diff=" + (contribs.vecs()[7].at(1670) - 0.0), Math.abs(contribs.vecs()[7].at(1670) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1676) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1676) + "; diff=" + (contribs.vecs()[4].at(1676) - 0.0), Math.abs(contribs.vecs()[4].at(1676) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1683) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1683) + "; diff=" + (contribs.vecs()[6].at(1683) - 0.1322035789489746), Math.abs(contribs.vecs()[6].at(1683) - 0.1322035789489746) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1684) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1684) + "; diff=" + (contribs.vecs()[5].at(1684) - 0.0218995101749897), Math.abs(contribs.vecs()[5].at(1684) - 0.0218995101749897) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1687) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1687) + "; diff=" + (contribs.vecs()[5].at(1687) - -0.018805718049407005), Math.abs(contribs.vecs()[5].at(1687) - -0.018805718049407005) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1697) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1697) + "; diff=" + (contribs.vecs()[0].at(1697) - -0.0028596660122275352), Math.abs(contribs.vecs()[0].at(1697) - -0.0028596660122275352) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1710) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1710) + "; diff=" + (contribs.vecs()[1].at(1710) - -0.01461546216160059), Math.abs(contribs.vecs()[1].at(1710) - -0.01461546216160059) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1710) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1710) + "; diff=" + (contribs.vecs()[6].at(1710) - -0.007993999868631363), Math.abs(contribs.vecs()[6].at(1710) - -0.007993999868631363) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1716) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1716) + "; diff=" + (contribs.vecs()[3].at(1716) - -0.3035963475704193), Math.abs(contribs.vecs()[3].at(1716) - -0.3035963475704193) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1720) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1720) + "; diff=" + (contribs.vecs()[6].at(1720) - -0.0047856783494353294), Math.abs(contribs.vecs()[6].at(1720) - -0.0047856783494353294) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1722) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1722) + "; diff=" + (contribs.vecs()[4].at(1722) - 0.0), Math.abs(contribs.vecs()[4].at(1722) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1730) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1730) + "; diff=" + (contribs.vecs()[6].at(1730) - 0.002015445614233613), Math.abs(contribs.vecs()[6].at(1730) - 0.002015445614233613) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1736) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1736) + "; diff=" + (contribs.vecs()[0].at(1736) - -0.008973526768386364), Math.abs(contribs.vecs()[0].at(1736) - -0.008973526768386364) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1746) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1746) + "; diff=" + (contribs.vecs()[4].at(1746) - -0.11527284234762192), Math.abs(contribs.vecs()[4].at(1746) - -0.11527284234762192) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1753) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1753) + "; diff=" + (contribs.vecs()[7].at(1753) - -0.12284810096025467), Math.abs(contribs.vecs()[7].at(1753) - -0.12284810096025467) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1754) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1754) + "; diff=" + (contribs.vecs()[3].at(1754) - 0.12224403023719788), Math.abs(contribs.vecs()[3].at(1754) - 0.12224403023719788) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1755) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1755) + "; diff=" + (contribs.vecs()[2].at(1755) - 0.34735265374183655), Math.abs(contribs.vecs()[2].at(1755) - 0.34735265374183655) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1755) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1755) + "; diff=" + (contribs.vecs()[4].at(1755) - 0.0), Math.abs(contribs.vecs()[4].at(1755) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1760) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1760) + "; diff=" + (contribs.vecs()[7].at(1760) - -0.6065792441368103), Math.abs(contribs.vecs()[7].at(1760) - -0.6065792441368103) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1761) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1761) + "; diff=" + (contribs.vecs()[7].at(1761) - 0.14812932908535004), Math.abs(contribs.vecs()[7].at(1761) - 0.14812932908535004) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1762) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1762) + "; diff=" + (contribs.vecs()[3].at(1762) - 0.0), Math.abs(contribs.vecs()[3].at(1762) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1773) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1773) + "; diff=" + (contribs.vecs()[7].at(1773) - -0.18297342956066132), Math.abs(contribs.vecs()[7].at(1773) - -0.18297342956066132) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1789) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1789) + "; diff=" + (contribs.vecs()[3].at(1789) - 0.2250409871339798), Math.abs(contribs.vecs()[3].at(1789) - 0.2250409871339798) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1821) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1821) + "; diff=" + (contribs.vecs()[5].at(1821) - 0.17498348653316498), Math.abs(contribs.vecs()[5].at(1821) - 0.17498348653316498) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1824) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1824) + "; diff=" + (contribs.vecs()[0].at(1824) - -0.0023917905054986477), Math.abs(contribs.vecs()[0].at(1824) - -0.0023917905054986477) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1831) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1831) + "; diff=" + (contribs.vecs()[2].at(1831) - -0.07947985827922821), Math.abs(contribs.vecs()[2].at(1831) - -0.07947985827922821) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1838) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1838) + "; diff=" + (contribs.vecs()[4].at(1838) - 0.0), Math.abs(contribs.vecs()[4].at(1838) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1847) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1847) + "; diff=" + (contribs.vecs()[5].at(1847) - -4.5412702560424805), Math.abs(contribs.vecs()[5].at(1847) - -4.5412702560424805) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1859) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1859) + "; diff=" + (contribs.vecs()[1].at(1859) - 0.008402948267757893), Math.abs(contribs.vecs()[1].at(1859) - 0.008402948267757893) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1860) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1860) + "; diff=" + (contribs.vecs()[1].at(1860) - 0.011777420528233051), Math.abs(contribs.vecs()[1].at(1860) - 0.011777420528233051) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1883) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1883) + "; diff=" + (contribs.vecs()[6].at(1883) - 0.07991338521242142), Math.abs(contribs.vecs()[6].at(1883) - 0.07991338521242142) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1885) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1885) + "; diff=" + (contribs.vecs()[1].at(1885) - -0.01456593070179224), Math.abs(contribs.vecs()[1].at(1885) - -0.01456593070179224) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1886) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1886) + "; diff=" + (contribs.vecs()[1].at(1886) - 0.010519035160541534), Math.abs(contribs.vecs()[1].at(1886) - 0.010519035160541534) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1888) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1888) + "; diff=" + (contribs.vecs()[4].at(1888) - -0.02113964594900608), Math.abs(contribs.vecs()[4].at(1888) - -0.02113964594900608) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1892) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1892) + "; diff=" + (contribs.vecs()[0].at(1892) - 0.00043502598418854177), Math.abs(contribs.vecs()[0].at(1892) - 0.00043502598418854177) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1894) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1894) + "; diff=" + (contribs.vecs()[3].at(1894) - 0.0), Math.abs(contribs.vecs()[3].at(1894) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1897) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1897) + "; diff=" + (contribs.vecs()[0].at(1897) - 7.873147467307717e-08), Math.abs(contribs.vecs()[0].at(1897) - 7.873147467307717e-08) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1920) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1920) + "; diff=" + (contribs.vecs()[4].at(1920) - -0.018893253058195114), Math.abs(contribs.vecs()[4].at(1920) - -0.018893253058195114) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1921) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1921) + "; diff=" + (contribs.vecs()[7].at(1921) - 0.3637378513813019), Math.abs(contribs.vecs()[7].at(1921) - 0.3637378513813019) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1923) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1923) + "; diff=" + (contribs.vecs()[2].at(1923) - 0.0), Math.abs(contribs.vecs()[2].at(1923) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1924) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1924) + "; diff=" + (contribs.vecs()[2].at(1924) - 0.0), Math.abs(contribs.vecs()[2].at(1924) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1925) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1925) + "; diff=" + (contribs.vecs()[1].at(1925) - -0.002803466049954295), Math.abs(contribs.vecs()[1].at(1925) - -0.002803466049954295) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1931) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1931) + "; diff=" + (contribs.vecs()[0].at(1931) - -3.171037315041758e-05), Math.abs(contribs.vecs()[0].at(1931) - -3.171037315041758e-05) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1931) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1931) + "; diff=" + (contribs.vecs()[4].at(1931) - 0.0), Math.abs(contribs.vecs()[4].at(1931) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1934) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1934) + "; diff=" + (contribs.vecs()[2].at(1934) - 0.0), Math.abs(contribs.vecs()[2].at(1934) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1935) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1935) + "; diff=" + (contribs.vecs()[0].at(1935) - -0.0029068992007523775), Math.abs(contribs.vecs()[0].at(1935) - -0.0029068992007523775) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1935) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1935) + "; diff=" + (contribs.vecs()[4].at(1935) - 0.0), Math.abs(contribs.vecs()[4].at(1935) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1939) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1939) + "; diff=" + (contribs.vecs()[1].at(1939) - 0.0036946344189345837), Math.abs(contribs.vecs()[1].at(1939) - 0.0036946344189345837) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1940) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1940) + "; diff=" + (contribs.vecs()[1].at(1940) - 0.0), Math.abs(contribs.vecs()[1].at(1940) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1940) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1940) + "; diff=" + (contribs.vecs()[5].at(1940) - 0.07560829818248749), Math.abs(contribs.vecs()[5].at(1940) - 0.07560829818248749) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1945) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1945) + "; diff=" + (contribs.vecs()[7].at(1945) - 0.0), Math.abs(contribs.vecs()[7].at(1945) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1946) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1946) + "; diff=" + (contribs.vecs()[5].at(1946) - 0.0717889741063118), Math.abs(contribs.vecs()[5].at(1946) - 0.0717889741063118) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1949) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1949) + "; diff=" + (contribs.vecs()[1].at(1949) - 0.016291676089167595), Math.abs(contribs.vecs()[1].at(1949) - 0.016291676089167595) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1960) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1960) + "; diff=" + (contribs.vecs()[2].at(1960) - 0.0), Math.abs(contribs.vecs()[2].at(1960) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1977) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1977) + "; diff=" + (contribs.vecs()[0].at(1977) - -0.008111163042485714), Math.abs(contribs.vecs()[0].at(1977) - -0.008111163042485714) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1983) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1983) + "; diff=" + (contribs.vecs()[0].at(1983) - -0.11220423877239227), Math.abs(contribs.vecs()[0].at(1983) - -0.11220423877239227) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1985) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1985) + "; diff=" + (contribs.vecs()[1].at(1985) - -0.03639569506049156), Math.abs(contribs.vecs()[1].at(1985) - -0.03639569506049156) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1993) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1993) + "; diff=" + (contribs.vecs()[0].at(1993) - 0.0006902207969687879), Math.abs(contribs.vecs()[0].at(1993) - 0.0006902207969687879) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1994) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1994) + "; diff=" + (contribs.vecs()[2].at(1994) - 0.0), Math.abs(contribs.vecs()[2].at(1994) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(1999) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(1999) + "; diff=" + (contribs.vecs()[4].at(1999) - -0.1336108297109604), Math.abs(contribs.vecs()[4].at(1999) - -0.1336108297109604) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2002) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2002) + "; diff=" + (contribs.vecs()[3].at(2002) - -0.3683775067329407), Math.abs(contribs.vecs()[3].at(2002) - -0.3683775067329407) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2027) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2027) + "; diff=" + (contribs.vecs()[0].at(2027) - -0.0012872419320046902), Math.abs(contribs.vecs()[0].at(2027) - -0.0012872419320046902) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2027) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2027) + "; diff=" + (contribs.vecs()[1].at(2027) - -0.020565271377563477), Math.abs(contribs.vecs()[1].at(2027) - -0.020565271377563477) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2029) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2029) + "; diff=" + (contribs.vecs()[0].at(2029) - -0.0003967961820308119), Math.abs(contribs.vecs()[0].at(2029) - -0.0003967961820308119) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2038) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2038) + "; diff=" + (contribs.vecs()[2].at(2038) - 0.0), Math.abs(contribs.vecs()[2].at(2038) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2047) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2047) + "; diff=" + (contribs.vecs()[4].at(2047) - -0.0911482647061348), Math.abs(contribs.vecs()[4].at(2047) - -0.0911482647061348) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2056) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2056) + "; diff=" + (contribs.vecs()[2].at(2056) - 0.04270605742931366), Math.abs(contribs.vecs()[2].at(2056) - 0.04270605742931366) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2059) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2059) + "; diff=" + (contribs.vecs()[0].at(2059) - -0.005321903619915247), Math.abs(contribs.vecs()[0].at(2059) - -0.005321903619915247) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2069) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2069) + "; diff=" + (contribs.vecs()[6].at(2069) - 0.052560243755578995), Math.abs(contribs.vecs()[6].at(2069) - 0.052560243755578995) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2070) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2070) + "; diff=" + (contribs.vecs()[0].at(2070) - -0.0036156296264380217), Math.abs(contribs.vecs()[0].at(2070) - -0.0036156296264380217) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2088) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2088) + "; diff=" + (contribs.vecs()[3].at(2088) - -0.19958917796611786), Math.abs(contribs.vecs()[3].at(2088) - -0.19958917796611786) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2090) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2090) + "; diff=" + (contribs.vecs()[5].at(2090) - 0.0010882415808737278), Math.abs(contribs.vecs()[5].at(2090) - 0.0010882415808737278) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2093) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2093) + "; diff=" + (contribs.vecs()[6].at(2093) - 0.0), Math.abs(contribs.vecs()[6].at(2093) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2119) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2119) + "; diff=" + (contribs.vecs()[2].at(2119) - 0.0), Math.abs(contribs.vecs()[2].at(2119) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2127) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2127) + "; diff=" + (contribs.vecs()[0].at(2127) - -0.0059058484621346), Math.abs(contribs.vecs()[0].at(2127) - -0.0059058484621346) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2146) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2146) + "; diff=" + (contribs.vecs()[2].at(2146) - 0.0), Math.abs(contribs.vecs()[2].at(2146) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2147) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2147) + "; diff=" + (contribs.vecs()[4].at(2147) - 0.0), Math.abs(contribs.vecs()[4].at(2147) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2152) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2152) + "; diff=" + (contribs.vecs()[1].at(2152) - 0.0004429039836395532), Math.abs(contribs.vecs()[1].at(2152) - 0.0004429039836395532) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2155) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2155) + "; diff=" + (contribs.vecs()[5].at(2155) - -0.0005092703504487872), Math.abs(contribs.vecs()[5].at(2155) - -0.0005092703504487872) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2161) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2161) + "; diff=" + (contribs.vecs()[5].at(2161) - 0.0296290535479784), Math.abs(contribs.vecs()[5].at(2161) - 0.0296290535479784) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2161) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2161) + "; diff=" + (contribs.vecs()[7].at(2161) - 0.023025602102279663), Math.abs(contribs.vecs()[7].at(2161) - 0.023025602102279663) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2163) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2163) + "; diff=" + (contribs.vecs()[4].at(2163) - 0.015483974479138851), Math.abs(contribs.vecs()[4].at(2163) - 0.015483974479138851) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2164) + "; diff=" + (contribs.vecs()[6].at(2164) - -0.0016590235754847527), Math.abs(contribs.vecs()[6].at(2164) - -0.0016590235754847527) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2164) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2164) + "; diff=" + (contribs.vecs()[7].at(2164) - 0.09305916726589203), Math.abs(contribs.vecs()[7].at(2164) - 0.09305916726589203) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2166) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2166) + "; diff=" + (contribs.vecs()[1].at(2166) - 0.004852556623518467), Math.abs(contribs.vecs()[1].at(2166) - 0.004852556623518467) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2166) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2166) + "; diff=" + (contribs.vecs()[2].at(2166) - 0.0), Math.abs(contribs.vecs()[2].at(2166) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2166) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2166) + "; diff=" + (contribs.vecs()[3].at(2166) - 0.028813684359192848), Math.abs(contribs.vecs()[3].at(2166) - 0.028813684359192848) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2172) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2172) + "; diff=" + (contribs.vecs()[4].at(2172) - 0.12585368752479553), Math.abs(contribs.vecs()[4].at(2172) - 0.12585368752479553) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2180) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2180) + "; diff=" + (contribs.vecs()[2].at(2180) - 0.0), Math.abs(contribs.vecs()[2].at(2180) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2181) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2181) + "; diff=" + (contribs.vecs()[3].at(2181) - 0.0), Math.abs(contribs.vecs()[3].at(2181) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2193) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2193) + "; diff=" + (contribs.vecs()[3].at(2193) - -0.16385558247566223), Math.abs(contribs.vecs()[3].at(2193) - -0.16385558247566223) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2198) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2198) + "; diff=" + (contribs.vecs()[6].at(2198) - -0.07219620794057846), Math.abs(contribs.vecs()[6].at(2198) - -0.07219620794057846) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2206) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2206) + "; diff=" + (contribs.vecs()[7].at(2206) - 0.0), Math.abs(contribs.vecs()[7].at(2206) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2207) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2207) + "; diff=" + (contribs.vecs()[5].at(2207) - 0.010745230130851269), Math.abs(contribs.vecs()[5].at(2207) - 0.010745230130851269) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2221) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2221) + "; diff=" + (contribs.vecs()[0].at(2221) - 0.0016042409697547555), Math.abs(contribs.vecs()[0].at(2221) - 0.0016042409697547555) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2228) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2228) + "; diff=" + (contribs.vecs()[4].at(2228) - 0.0), Math.abs(contribs.vecs()[4].at(2228) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2228) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2228) + "; diff=" + (contribs.vecs()[7].at(2228) - 0.0), Math.abs(contribs.vecs()[7].at(2228) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2230) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2230) + "; diff=" + (contribs.vecs()[2].at(2230) - 0.0), Math.abs(contribs.vecs()[2].at(2230) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2238) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2238) + "; diff=" + (contribs.vecs()[4].at(2238) - -0.26824507117271423), Math.abs(contribs.vecs()[4].at(2238) - -0.26824507117271423) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2240) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2240) + "; diff=" + (contribs.vecs()[7].at(2240) - -0.21003328263759613), Math.abs(contribs.vecs()[7].at(2240) - -0.21003328263759613) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2242) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2242) + "; diff=" + (contribs.vecs()[3].at(2242) - -0.1447216123342514), Math.abs(contribs.vecs()[3].at(2242) - -0.1447216123342514) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2254) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2254) + "; diff=" + (contribs.vecs()[3].at(2254) - -0.2883744239807129), Math.abs(contribs.vecs()[3].at(2254) - -0.2883744239807129) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2257) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2257) + "; diff=" + (contribs.vecs()[6].at(2257) - 0.029696766287088394), Math.abs(contribs.vecs()[6].at(2257) - 0.029696766287088394) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2260) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2260) + "; diff=" + (contribs.vecs()[2].at(2260) - 6.380661487579346), Math.abs(contribs.vecs()[2].at(2260) - 6.380661487579346) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2269) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2269) + "; diff=" + (contribs.vecs()[2].at(2269) - 0.0), Math.abs(contribs.vecs()[2].at(2269) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2276) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2276) + "; diff=" + (contribs.vecs()[2].at(2276) - 0.0), Math.abs(contribs.vecs()[2].at(2276) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2278) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2278) + "; diff=" + (contribs.vecs()[2].at(2278) - 0.0), Math.abs(contribs.vecs()[2].at(2278) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2279) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2279) + "; diff=" + (contribs.vecs()[1].at(2279) - -0.005762324668467045), Math.abs(contribs.vecs()[1].at(2279) - -0.005762324668467045) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2281) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2281) + "; diff=" + (contribs.vecs()[5].at(2281) - -0.0831969603896141), Math.abs(contribs.vecs()[5].at(2281) - -0.0831969603896141) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2283) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2283) + "; diff=" + (contribs.vecs()[1].at(2283) - -0.004815306980162859), Math.abs(contribs.vecs()[1].at(2283) - -0.004815306980162859) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2286) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2286) + "; diff=" + (contribs.vecs()[7].at(2286) - -0.3489488959312439), Math.abs(contribs.vecs()[7].at(2286) - -0.3489488959312439) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2294) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2294) + "; diff=" + (contribs.vecs()[3].at(2294) - 0.0), Math.abs(contribs.vecs()[3].at(2294) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2296) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2296) + "; diff=" + (contribs.vecs()[7].at(2296) - 0.0), Math.abs(contribs.vecs()[7].at(2296) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2297) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2297) + "; diff=" + (contribs.vecs()[7].at(2297) - 0.2300163358449936), Math.abs(contribs.vecs()[7].at(2297) - 0.2300163358449936) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2303) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2303) + "; diff=" + (contribs.vecs()[6].at(2303) - 0.008638379164040089), Math.abs(contribs.vecs()[6].at(2303) - 0.008638379164040089) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2307) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2307) + "; diff=" + (contribs.vecs()[0].at(2307) - 0.0004552254104055464), Math.abs(contribs.vecs()[0].at(2307) - 0.0004552254104055464) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2309) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2309) + "; diff=" + (contribs.vecs()[6].at(2309) - 0.0), Math.abs(contribs.vecs()[6].at(2309) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2312) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2312) + "; diff=" + (contribs.vecs()[0].at(2312) - -0.008009620010852814), Math.abs(contribs.vecs()[0].at(2312) - -0.008009620010852814) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2320) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2320) + "; diff=" + (contribs.vecs()[5].at(2320) - 0.019398584961891174), Math.abs(contribs.vecs()[5].at(2320) - 0.019398584961891174) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2321) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2321) + "; diff=" + (contribs.vecs()[3].at(2321) - 0.3074694573879242), Math.abs(contribs.vecs()[3].at(2321) - 0.3074694573879242) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2337) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2337) + "; diff=" + (contribs.vecs()[5].at(2337) - 0.033772729337215424), Math.abs(contribs.vecs()[5].at(2337) - 0.033772729337215424) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2338) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2338) + "; diff=" + (contribs.vecs()[5].at(2338) - 0.04788355156779289), Math.abs(contribs.vecs()[5].at(2338) - 0.04788355156779289) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2341) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2341) + "; diff=" + (contribs.vecs()[6].at(2341) - 0.18723702430725098), Math.abs(contribs.vecs()[6].at(2341) - 0.18723702430725098) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2342) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2342) + "; diff=" + (contribs.vecs()[7].at(2342) - -0.10795357078313828), Math.abs(contribs.vecs()[7].at(2342) - -0.10795357078313828) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2350) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2350) + "; diff=" + (contribs.vecs()[5].at(2350) - 0.005689268931746483), Math.abs(contribs.vecs()[5].at(2350) - 0.005689268931746483) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2359) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2359) + "; diff=" + (contribs.vecs()[0].at(2359) - -0.000482442497741431), Math.abs(contribs.vecs()[0].at(2359) - -0.000482442497741431) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2360) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2360) + "; diff=" + (contribs.vecs()[6].at(2360) - 0.0), Math.abs(contribs.vecs()[6].at(2360) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[7] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2371) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2371) + "; diff=" + (contribs.vecs()[7].at(2371) - 0.023364514112472534), Math.abs(contribs.vecs()[7].at(2371) - 0.023364514112472534) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2375) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2375) + "; diff=" + (contribs.vecs()[3].at(2375) - -0.3933889865875244), Math.abs(contribs.vecs()[3].at(2375) - -0.3933889865875244) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2437) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2437) + "; diff=" + (contribs.vecs()[6].at(2437) - 0.04607648402452469), Math.abs(contribs.vecs()[6].at(2437) - 0.04607648402452469) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2438) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2438) + "; diff=" + (contribs.vecs()[5].at(2438) - 0.2191285640001297), Math.abs(contribs.vecs()[5].at(2438) - 0.2191285640001297) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2440) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2440) + "; diff=" + (contribs.vecs()[3].at(2440) - -0.09508752822875977), Math.abs(contribs.vecs()[3].at(2440) - -0.09508752822875977) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2445) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2445) + "; diff=" + (contribs.vecs()[3].at(2445) - -0.10111632198095322), Math.abs(contribs.vecs()[3].at(2445) - -0.10111632198095322) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2450) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2450) + "; diff=" + (contribs.vecs()[1].at(2450) - 0.014004581607878208), Math.abs(contribs.vecs()[1].at(2450) - 0.014004581607878208) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2456) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2456) + "; diff=" + (contribs.vecs()[5].at(2456) - 0.03993678838014603), Math.abs(contribs.vecs()[5].at(2456) - 0.03993678838014603) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2462) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2462) + "; diff=" + (contribs.vecs()[4].at(2462) - 0.0), Math.abs(contribs.vecs()[4].at(2462) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2468) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2468) + "; diff=" + (contribs.vecs()[2].at(2468) - 0.0), Math.abs(contribs.vecs()[2].at(2468) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2473) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2473) + "; diff=" + (contribs.vecs()[2].at(2473) - 0.0), Math.abs(contribs.vecs()[2].at(2473) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2483) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2483) + "; diff=" + (contribs.vecs()[1].at(2483) - 0.01891067437827587), Math.abs(contribs.vecs()[1].at(2483) - 0.01891067437827587) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2484) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2484) + "; diff=" + (contribs.vecs()[6].at(2484) - -0.0926094725728035), Math.abs(contribs.vecs()[6].at(2484) - -0.0926094725728035) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2492) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2492) + "; diff=" + (contribs.vecs()[1].at(2492) - -0.000562994449865073), Math.abs(contribs.vecs()[1].at(2492) - -0.000562994449865073) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2498) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2498) + "; diff=" + (contribs.vecs()[5].at(2498) - 0.014675598591566086), Math.abs(contribs.vecs()[5].at(2498) - 0.014675598591566086) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2514) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2514) + "; diff=" + (contribs.vecs()[6].at(2514) - 0.0), Math.abs(contribs.vecs()[6].at(2514) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2515) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2515) + "; diff=" + (contribs.vecs()[5].at(2515) - -1.952954649925232), Math.abs(contribs.vecs()[5].at(2515) - -1.952954649925232) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2522) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2522) + "; diff=" + (contribs.vecs()[5].at(2522) - 0.006932680960744619), Math.abs(contribs.vecs()[5].at(2522) - 0.006932680960744619) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2525) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2525) + "; diff=" + (contribs.vecs()[5].at(2525) - 0.009310884401202202), Math.abs(contribs.vecs()[5].at(2525) - 0.009310884401202202) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2526) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2526) + "; diff=" + (contribs.vecs()[5].at(2526) - 0.03337103873491287), Math.abs(contribs.vecs()[5].at(2526) - 0.03337103873491287) < eps);
    assertTrue("Error for column '" + contribs.names()[2] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2531) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2531) + "; diff=" + (contribs.vecs()[2].at(2531) - 0.0), Math.abs(contribs.vecs()[2].at(2531) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[3] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2533) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2533) + "; diff=" + (contribs.vecs()[3].at(2533) - 0.0), Math.abs(contribs.vecs()[3].at(2533) - 0.0) < eps);
    assertTrue("Error for column '" + contribs.names()[5] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2535) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2535) + "; diff=" + (contribs.vecs()[5].at(2535) - 0.1955590546131134), Math.abs(contribs.vecs()[5].at(2535) - 0.1955590546131134) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2537) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2537) + "; diff=" + (contribs.vecs()[6].at(2537) - 0.030922507867217064), Math.abs(contribs.vecs()[6].at(2537) - 0.030922507867217064) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2538) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2538) + "; diff=" + (contribs.vecs()[6].at(2538) - 0.07160303741693497), Math.abs(contribs.vecs()[6].at(2538) - 0.07160303741693497) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2546) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2546) + "; diff=" + (contribs.vecs()[6].at(2546) - 0.05333210155367851), Math.abs(contribs.vecs()[6].at(2546) - 0.05333210155367851) < eps);
    assertTrue("Error for column '" + contribs.names()[1] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2563) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2563) + "; diff=" + (contribs.vecs()[1].at(2563) - -0.015552395954728127), Math.abs(contribs.vecs()[1].at(2563) - -0.015552395954728127) < eps);
    assertTrue("Error for column '" + contribs.names()[6] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2566) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2566) + "; diff=" + (contribs.vecs()[6].at(2566) - -0.11050549894571304), Math.abs(contribs.vecs()[6].at(2566) - -0.11050549894571304) < eps);
    assertTrue("Error for column '" + contribs.names()[0] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2593) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2593) + "; diff=" + (contribs.vecs()[0].at(2593) - -0.0031774542294442654), Math.abs(contribs.vecs()[0].at(2593) - -0.0031774542294442654) < eps);
    assertTrue("Error for column '" + contribs.names()[4] + "' for rowIdx=" + contribs.vecs()[contribs.numCols() - 2].at(2593) + "; bgRowIdx=" + contribs.vecs()[contribs.numCols() - 1].at(2593) + "; diff=" + (contribs.vecs()[4].at(2593) - 0.0), Math.abs(contribs.vecs()[4].at(2593) - 0.0) < eps);
  }

  private static void assertColsEquals(Frame expected, Frame actual, int colExpected, int colActual, double eps) {
    assertEquals(expected.numRows(), actual.numRows());
    for (int i = 0; i < expected.numRows(); i++) {
      assertEquals("Wrong sum in row " + i, expected.vec(colExpected).at(i), actual.vec(colActual).at(i), eps);
    }
  }
}
