# This file is part of Indico.
# Copyright (C) 2002 - 2026 CERN
#
# Indico is free software; you can redistribute it and/or
# modify it under the terms of the MIT License; see the
# LICENSE file for more details.

from indico.core.db import db
from indico.core.db.sqlalchemy.util.session import no_autoflush
from indico.modules.events.cloning import EventCloner, get_attrs_to_clone
from indico.modules.events.editing.models.file_types import EditingFileType
from indico.modules.events.editing.models.review_conditions import EditingReviewCondition
from indico.modules.events.editing.models.tags import EditingTag
from indico.modules.events.editing.settings import editable_type_settings, editing_settings
from indico.modules.events.models.events import EventType
from indico.util.i18n import _


NOT_CLONED_TYPE_SETTINGS = ('submission_enabled', 'editing_enabled')


class EditingSettingsCloner(EventCloner):
    name = 'editing_settings'
    friendly_name = _('Editing (configured tags, file types, review conditions)')
    new_event_only = True

    @property
    def is_visible(self):
        return self.old_event.type_ == EventType.conference

    @no_autoflush
    def run(self, new_event, cloners, shared_data, event_exists=False):
        self._filetype_map = {}
        self._clone_settings(new_event)
        self._clone_tags(new_event)
        self._clone_filetypes(new_event)
        self._clone_review_conditions(new_event)
        db.session.flush()

    def _clone_settings(self, new_event):
        editing_settings.set(new_event, 'editable_types', editing_settings.get(self.old_event, 'editable_types'))
        for type_settings in editable_type_settings.values():
            data = type_settings.get_all(self.old_event, no_defaults=True)
            for name in NOT_CLONED_TYPE_SETTINGS:
                data.pop(name, None)
            type_settings.set_multi(new_event, data)

    def _clone_tags(self, new_event):
        attrs = get_attrs_to_clone(EditingTag)
        for old_tag in self.old_event.editing_tags:
            tag = EditingTag()
            tag.populate_from_attrs(old_tag, attrs)
            new_event.editing_tags.append(tag)

    def _clone_filetypes(self, new_event):
        attrs = get_attrs_to_clone(EditingFileType)
        del new_event.editing_file_types[:]
        db.session.flush()
        for old_filetype in self.old_event.editing_file_types:
            filetype = EditingFileType()
            filetype.populate_from_attrs(old_filetype, attrs)
            new_event.editing_file_types.append(filetype)
            db.session.flush()
            self._filetype_map[old_filetype] = filetype

    def _clone_review_conditions(self, new_event):
        old_conditions = EditingReviewCondition.query.with_parent(self.old_event).all()
        for condition in old_conditions:
            new_filetypes = {self._filetype_map[ft] for ft in condition.file_types}
            new_condition = EditingReviewCondition(type=condition.type, file_types=new_filetypes)
            new_event.editing_review_conditions.append(new_condition)
