# This file is part of Indico.
# Copyright (C) 2002 - 2026 CERN
#
# Indico is free software; you can redistribute it and/or
# modify it under the terms of the MIT License; see the
# LICENSE file for more details.

from indico.core.db import db
from indico.core.db.sqlalchemy.principals import PrincipalType
from indico.util.locators import locator_property
from indico.util.string import format_repr


class EventRole(db.Model):
    __tablename__ = 'roles'
    __table_args__ = (db.CheckConstraint('code = upper(code)', 'uppercase_code'),
                      db.Index(None, 'event_id', 'code', unique=True),
                      {'schema': 'events'})

    principal_order = 2
    principal_type = PrincipalType.event_role

    id = db.Column(
        db.Integer,
        primary_key=True
    )
    event_id = db.Column(
        db.Integer,
        db.ForeignKey('events.events.id'),
        nullable=False,
        index=True
    )
    name = db.Column(
        db.String,
        nullable=False
    )
    code = db.Column(
        db.String,
        nullable=False
    )
    color = db.Column(
        db.String,
        nullable=False
    )

    event = db.relationship(
        'Event',
        lazy=True,
        backref=db.backref(
            'roles',
            cascade='all, delete-orphan',
            lazy=True
        )
    )
    members = db.relationship(
        'User',
        secondary='events.role_members',
        lazy=True,
        collection_class=set,
        backref=db.backref('event_roles', lazy=True, collection_class=set),
    )

    # relationship backrefs:
    # - in_attachment_acls (AttachmentPrincipal.event_role)
    # - in_attachment_folder_acls (AttachmentFolderPrincipal.event_role)
    # - in_contribution_acls (ContributionPrincipal.event_role)
    # - in_event_acls (EventPrincipal.event_role)
    # - in_event_settings_acls (EventSettingPrincipal.event_role)
    # - in_menu_entry_acls (MenuEntryPrincipal.event_role)
    # - in_session_acls (SessionPrincipal.event_role)
    # - in_track_acls (TrackPrincipal.event_role)

    def __contains__(self, user):
        return user is not None and self in user.event_roles

    def __repr__(self):
        return format_repr(self, 'id', 'code', _text=self.name)

    @locator_property
    def locator(self):
        return dict(self.event.locator, role_id=self.id)

    @property
    def identifier(self):
        return f'EventRole:{self.id}'

    @property
    def persistent_identifier(self):
        """A persistent version of this object's identifier.

        This is currently identical to the regular identifier, since it does not
        contain any signatures or similar data that could change.
        """
        return self.identifier

    @property
    def obj(self):
        return self.event

    @property
    def css(self):
        return f'color: #{self.color} !important; border-color: #{self.color} !important'

    @property
    def style(self):
        return {'color': '#' + self.color, 'borderColor': '#' + self.color}


role_members_table = db.Table(
    'role_members',
    db.metadata,
    db.Column(
        'role_id',
        db.Integer,
        db.ForeignKey('events.roles.id'),
        primary_key=True,
        nullable=False,
        index=True
    ),
    db.Column(
        'user_id',
        db.Integer,
        db.ForeignKey('users.users.id'),
        primary_key=True,
        nullable=False,
        index=True
    ),
    schema='events'
)
