/**
 * Basic SageMaker Provider Usage Example
 *
 * This example demonstrates the fundamental usage patterns of the SageMaker provider
 * including initialization, text generation, and error handling.
 */

import { NeuroLink } from "@juspay/neurolink";

async function basicUsageExample() {
  console.log("SageMaker Provider - Basic Usage Example\n");

  try {
    // 1. Initialize NeuroLink
    console.log("1. Initializing NeuroLink...");
    const neurolink = new NeuroLink();

    // Get the endpoint name from environment or use default
    const endpointName =
      process.env.SAGEMAKER_ENDPOINT_NAME || "my-custom-endpoint";

    console.log(`   Using SageMaker endpoint: ${endpointName}`);

    // 2. Basic text generation
    console.log("\n2. Generating text...");
    const result = await neurolink.generate({
      input: {
        text: "You are a helpful assistant. What are the benefits of using AWS SageMaker for machine learning?",
      },
      provider: "sagemaker",
      model: endpointName,
      maxTokens: 200,
      temperature: 0.7,
    });

    console.log("Generated Response:");
    console.log(`   Input tokens: ${result.usage?.inputTokens || "N/A"}`);
    console.log(`   Output tokens: ${result.usage?.outputTokens || "N/A"}`);
    console.log(`   Total tokens: ${result.usage?.totalTokens || "N/A"}`);
    console.log(`   Response: ${result.text}\n`);

    // 3. Follow-up question
    console.log("3. Follow-up question...");
    const conversationResult = await neurolink.generate({
      input: {
        text: "Based on your previous answer about SageMaker, can you give me a simple example of how it's used?",
      },
      provider: "sagemaker",
      model: endpointName,
      maxTokens: 150,
      temperature: 0.6,
    });

    console.log("Follow-up Response:");
    console.log(`   Response: ${conversationResult.text}\n`);

    // 4. Provider and model information
    console.log("4. Configuration used:");
    console.log(`   Provider: sagemaker`);
    console.log(`   Endpoint: ${endpointName}`);
    console.log(`   Region: ${process.env.AWS_REGION || "Default"}\n`);

    console.log("Basic usage example completed successfully!");
  } catch (error) {
    console.error("Error in basic usage example:");
    console.error(error instanceof Error ? error.message : String(error));

    // Common error scenarios and solutions
    if (error instanceof Error) {
      const message = error.message.toLowerCase();

      if (message.includes("credentials") || message.includes("access")) {
        console.log("\nSolution: Check your AWS credentials");
        console.log('   export AWS_ACCESS_KEY_ID="your_key"');
        console.log('   export AWS_SECRET_ACCESS_KEY="your_secret"');
        console.log('   export AWS_REGION="us-east-1"');
      } else if (
        message.includes("endpoint") ||
        message.includes("not found")
      ) {
        console.log("\nSolution: Check your SageMaker endpoint");
        console.log('   export SAGEMAKER_ENDPOINT_NAME="your_endpoint"');
        console.log("   aws sagemaker list-endpoints");
      } else if (message.includes("timeout") || message.includes("network")) {
        console.log(
          "\nSolution: Check network connectivity and timeout settings",
        );
        console.log('   export SAGEMAKER_TIMEOUT="60000"');
      }
    }

    process.exit(1);
  }
}

// Run the example
basicUsageExample();
