/**
 * Custom Semantic Release Plugin - Format Changelog
 * 
 * This plugin runs during the prepare step of semantic-release to format
 * the CHANGELOG.md file after it's been generated by @semantic-release/changelog
 */

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

/**
 * Plugin function that formats the changelog
 * @param {Object} pluginConfig - Plugin configuration
 * @param {Object} context - Semantic release context
 */
async function formatChangelog(pluginConfig, context) {
  const { logger } = context;
  
  logger.log('📄 Running changelog formatter...');
  
  const changelogPath = path.join(process.cwd(), 'CHANGELOG.md');
  
  // Check if CHANGELOG.md exists
  if (!fs.existsSync(changelogPath)) {
    logger.log('⚠️  CHANGELOG.md not found, skipping formatting');
    return;
  }
  
  try {
    // Run prettier on CHANGELOG.md specifically
    execSync('npx prettier --write CHANGELOG.md', { 
      stdio: 'inherit',
      cwd: process.cwd()
    });
    
    logger.log('✅ CHANGELOG.md formatted successfully');
  } catch (error) {
    logger.error('❌ Error formatting CHANGELOG.md:', error.message);
    throw error;
  }
}

module.exports = {
  // Run during the prepare step, after @semantic-release/changelog
  // but before @semantic-release/git
  prepare: formatChangelog
};