/**
 * Provider-specific type definitions for NeuroLink
 */

import type { UnknownRecord, JsonValue } from "./common.js";
import {
  AIProviderName,
  AnthropicModels,
  BedrockModels,
  GoogleAIModels,
  OpenAIModels,
  VertexModels,
} from "../constants/enums.js";
import type { Tool } from "ai";
import type { ValidationSchema } from "./typeAliases.js";
import type {
  EnhancedGenerateResult,
  GenerateResult,
  TextGenerationOptions,
} from "./generateTypes.js";
import type { StreamOptions, StreamResult } from "./streamTypes.js";
import type { ExternalMCPToolInfo } from "./externalMcp.js";

// ============================================================================
// TYPE ALIASES
// ============================================================================

/**
 * Generic AI SDK model interface
 */
export type AISDKModel = {
  // This will be refined based on actual AI SDK types
  [key: string]: unknown;
};

/**
 * Union type of all supported model names
 */
export type SupportedModelName =
  | BedrockModels
  | OpenAIModels
  | VertexModels
  | GoogleAIModels
  | AnthropicModels;

/**
 * Extract provider names from enum
 */
export type ProviderName = (typeof AIProviderName)[keyof typeof AIProviderName];

/**
 * Provider status information
 */
export type ProviderStatus = {
  provider: string;
  status: "working" | "failed" | "not-configured";
  configured: boolean;
  authenticated: boolean;
  error?: string;
  responseTime?: number;
  model?: string;
};

/**
 * Provider error information
 */
export type ProviderError = Error & {
  code?: string | number;
  statusCode?: number;
  provider?: string;
  originalError?: unknown;
};

/**
 * AWS Credential Configuration for Bedrock provider
 */
export type AWSCredentialConfig = {
  region?: string;
  profile?: string;
  roleArn?: string;
  roleSessionName?: string;
  timeout?: number;
  /** @deprecated Prefer maxAttempts to match AWS SDK v3 config */
  maxRetries?: number;
  /** Number of attempts as per AWS SDK v3 ("retry-mode") */
  maxAttempts?: number;
  enableDebugLogging?: boolean;
  /** Optional service endpoint override (e.g., VPC/Gov endpoints) */
  endpoint?: string;
};

/**
 * AWS Credential Validation Result
 */
export type CredentialValidationResult = {
  isValid: boolean;
  credentialSource: string;
  region: string;
  hasExpiration: boolean;
  expirationTime?: Date;
  error?: string;
  debugInfo: {
    accessKeyId: string;
    hasSessionToken: boolean;
    providerConfig: Readonly<Required<AWSCredentialConfig>>;
  };
};

/**
 * Service Connectivity Test Result
 */
export type ServiceConnectivityResult = {
  bedrockAccessible: boolean;
  availableModels: number;
  responseTimeMs: number;
  error?: string;
  sampleModels: string[];
};

/**
 * Model Capabilities - Maximally Reusable
 */
export type ModelCapability =
  | "text"
  | "vision"
  | "function-calling"
  | "embedding"
  | "audio"
  | "video"
  | "code"
  | "reasoning"
  | "multimodal";

/**
 * Model Use Cases - High Reusability
 */
export type ModelUseCase =
  | "chat"
  | "completion"
  | "analysis"
  | "coding"
  | "creative"
  | "reasoning"
  | "translation"
  | "summarization"
  | "classification";

/**
 * Provider health status
 */
export type ProviderHealthStatus =
  | "healthy"
  | "degraded"
  | "unhealthy"
  | "unknown";

/**
 * Stream processing phases
 */
export type StreamPhase =
  | "initializing"
  | "streaming"
  | "processing"
  | "complete"
  | "error";

/**
 * Model Filter Configuration - High Reusability
 */
export type ModelFilter = {
  provider?: string;
  capability?: ModelCapability;
  useCase?: ModelUseCase;
  requireVision?: boolean;
  requireFunctionCalling?: boolean;
  maxTokens?: number;
  costLimit?: number;
};

/**
 * Model Resolution Context - High Reusability
 */
export type ModelResolutionContext = {
  requireCapabilities?: ModelCapability[];
  preferredProviders?: string[];
  useCase?: ModelUseCase;
  budgetConstraints?: {
    maxCostPerRequest?: number;
    maxTokens?: number;
  };
  performance?: {
    maxLatency?: number;
    minQuality?: number;
  };
};

/**
 * Model Statistics Object - High Reusability
 */
export type ModelStats = {
  name: string;
  provider: string;
  capabilities: ModelCapability[];
  useCases: ModelUseCase[];
  performance: {
    avgLatency?: number;
    avgTokensPerSecond?: number;
    reliability?: number;
  };
  pricing?: ModelPricing;
  metadata: {
    [key: string]: JsonValue;
  } & {
    version?: string;
    lastUpdated?: Date;
  };
};

/**
 * Model Pricing Information - High Reusability
 */
export type ModelPricing = {
  inputTokens?: {
    price: number;
    currency: string;
    per: number;
  };
  outputTokens?: {
    price: number;
    currency: string;
    per: number;
  };
  requestPrice?: {
    price: number;
    currency: string;
  };
  tier?: "free" | "basic" | "premium" | "enterprise";
  // Additional properties for models command compatibility
  average?: number;
  min?: number;
  max?: number;
  free?: boolean;
};

/**
 * Provider capabilities
 */
export type ProviderCapabilities = {
  supportsStreaming: boolean;
  supportsTools: boolean;
  supportsImages: boolean;
  supportsAudio: boolean;
  maxTokens?: number;
  supportedModels: string[];
};

/**
 * Provider configuration specifying provider and its available models (from core types)
 */
export type AIModelProviderConfig = {
  provider: AIProviderName;
  models: SupportedModelName[];
};

/**
 * Provider configuration for individual providers
 */
export type IndividualProviderConfig = {
  apiKey?: string;
  baseURL?: string;
  timeout?: number;
  retries?: number;
  model?: string;
  [key: string]: unknown;
};

/**
 * Configuration options for provider validation
 */
export type ProviderConfigOptions = {
  providerName: string;
  envVarName: string;
  setupUrl: string;
  description: string;
  instructions: string[];
  fallbackEnvVars?: string[]; // For providers with multiple possible env vars
};

// ============================================================================
// CORE PROVIDER INTERFACES
// ============================================================================

/**
 * AI Provider type with flexible parameter support
 */
export type AIProvider = {
  // Primary streaming method
  stream(
    optionsOrPrompt: StreamOptions | string,
    analysisSchema?: ValidationSchema,
  ): Promise<StreamResult>;

  generate(
    optionsOrPrompt: TextGenerationOptions | string,
    analysisSchema?: ValidationSchema,
  ): Promise<EnhancedGenerateResult | null>;

  gen(
    optionsOrPrompt: TextGenerationOptions | string,
    analysisSchema?: ValidationSchema,
  ): Promise<EnhancedGenerateResult | null>;

  // Tool execution setup - consolidated from NeuroLink SDK
  setupToolExecutor(
    sdk: {
      customTools: Map<string, unknown>;
      executeTool: (toolName: string, params: unknown) => Promise<unknown>;
    },
    functionTag: string,
  ): void;
};

/**
 * Provider attempt result for iteration tracking (converted from interface)
 */
export type ProviderAttempt = {
  provider: AIProviderName;
  model: SupportedModelName;
  success: boolean;
  error?: string;
  stack?: string;
};

/**
 * Error types for provider creation
 */
export type ProviderCreationError = {
  code: "INVALID_PROVIDER" | "CONFIGURATION_ERROR" | "INSTANTIATION_ERROR";
  message: string;
  provider: string;
  details?: Record<string, unknown>;
};

/**
 * Provider factory function type
 */
export type ProviderFactory = (
  modelName?: string,
  providerName?: string,
  sdk?: unknown,
) => Promise<unknown>;

/**
 * Provider constructor type
 */
export type ProviderConstructor = {
  new (modelName?: string, providerName?: string, sdk?: unknown): unknown;
};

/**
 * Provider registration entry
 */
export type ProviderRegistration = {
  name: string;
  constructor: ProviderConstructor | ProviderFactory;
  capabilities?: ProviderCapabilities;
  defaultConfig?: IndividualProviderConfig;
};

/**
 * Configuration options for the provider registry
 */
export type ProviderRegistryOptions = {
  /**
   * Enable loading of manual MCP configurations from .mcp-config.json
   * Should only be true for CLI mode, false for SDK mode
   */
  enableManualMCP?: boolean;
};

/**
 * Provider metadata type
 */
export type ProviderMetadata = {
  name: string;
  version: string;
  capabilities: ProviderCapability[];
  models: string[];
  healthStatus: ProviderHealthStatus;
};

/**
 * Provider capability type
 */
export type ProviderCapability =
  | "text-generation"
  | "streaming"
  | "tool-calling"
  | "image-generation"
  | "embeddings";

/**
 * Extended tool type that combines AI SDK tools with external MCP tool info
 */
export type ExtendedTool = Tool & Partial<ExternalMCPToolInfo>;

/**
 * AI SDK generate result with steps support (extends GenerateResult)
 */
export type AISDKGenerateResult = GenerateResult & {
  steps?: Array<{
    toolCalls?: Array<{
      toolName?: string;
      name?: string;
      [key: string]: unknown;
    }>;
    [key: string]: unknown;
  }>;
  [key: string]: unknown;
};

// ============================================================================
// Provider-Specific Type Definitions
// ============================================================================

// ============================================================================
// Amazon Bedrock Provider Types
// ============================================================================

/**
 * Bedrock tool usage structure
 */
export type BedrockToolUse = {
  toolUseId: string;
  name: string;
  input: Record<string, unknown>;
};

/**
 * Bedrock tool result structure
 */
export type BedrockToolResult = {
  toolUseId: string;
  content: Array<{ text: string }>;
  status: string;
};

/**
 * Bedrock content block structure
 */
export type BedrockContentBlock = {
  text?: string;
  image?: {
    format: "png" | "jpeg" | "gif" | "webp";
    source: {
      bytes?: Uint8Array | Buffer;
    };
  };
  document?: {
    format:
      | "pdf"
      | "csv"
      | "doc"
      | "docx"
      | "xls"
      | "xlsx"
      | "html"
      | "txt"
      | "md";
    name: string;
    source: {
      bytes?: Uint8Array | Buffer;
    };
  };
  toolUse?: BedrockToolUse;
  toolResult?: BedrockToolResult;
};

/**
 * Bedrock message structure
 */
export type BedrockMessage = {
  role: "user" | "assistant";
  content: BedrockContentBlock[];
};

// ============================================================================
// Google AI Studio Provider Types (Live API)
// ============================================================================

/**
 * Google AI Live media configuration
 */
export type GenAILiveMedia = {
  data: string;
  mimeType: string;
};

/**
 * Live server message inline data
 */
export type LiveServerMessagePartInlineData = {
  data?: string;
};

/**
 * Live server message model turn
 */
export type LiveServerMessageModelTurn = {
  parts?: Array<{ inlineData?: LiveServerMessagePartInlineData }>;
};

/**
 * Live server content structure
 */
export type LiveServerContent = {
  modelTurn?: LiveServerMessageModelTurn;
  interrupted?: boolean;
};

/**
 * Live server message structure
 */
export type LiveServerMessage = {
  serverContent?: LiveServerContent;
};

/**
 * Live connection callbacks
 */
export type LiveConnectCallbacks = {
  onopen?: () => void;
  onmessage?: (message: LiveServerMessage) => void;
  onerror?: (e: { message?: string }) => void;
  onclose?: (e: { code?: number; reason?: string }) => void;
};

/**
 * Live connection configuration
 */
export type LiveConnectConfig = {
  model: string;
  callbacks: LiveConnectCallbacks;
  config: {
    responseModalities: ("TEXT" | "IMAGE" | "AUDIO")[];
    speechConfig: {
      voiceConfig: { prebuiltVoiceConfig: { voiceName: string } };
    };
  };
};

/**
 * Google AI Live session interface
 */
export type GenAILiveSession = {
  sendRealtimeInput?: (payload: {
    media?: GenAILiveMedia;
    event?: string;
  }) => Promise<void> | void;
  sendInput?: (payload: {
    event?: string;
    media?: GenAILiveMedia;
  }) => Promise<void> | void;
  close?: (code?: number, reason?: string) => Promise<void> | void;
};

/**
 * Google AI generateContentStream response chunk
 */
export type GenAIStreamChunk = {
  text?: string;
  functionCalls?: Array<{ name: string; args: Record<string, unknown> }>;
  candidates?: Array<{
    content?: {
      parts?: Array<{
        text?: string;
        inlineData?: {
          data?: string;
          mimeType?: string;
        };
      }>;
    };
  }>;
};

/**
 * Google AI generate content response
 */
export type GenAIGenerateContentResponse = {
  candidates?: Array<{
    content?: {
      parts?: Array<{
        text?: string;
        inlineData?: {
          data?: string;
          mimeType?: string;
        };
      }>;
    };
  }>;
};

/**
 * Google AI models API interface
 */
export type GenAIModelsAPI = {
  generateContentStream: (params: {
    model: string;
    contents: Array<{ role: string; parts: unknown[] }>;
    config?: Record<string, unknown>;
  }) => Promise<AsyncIterable<GenAIStreamChunk>>;
  generateContent: (params: {
    model: string;
    contents: Array<{ role: string; parts: unknown[] }>;
    config?: Record<string, unknown>;
  }) => Promise<GenAIGenerateContentResponse>;
};

/**
 * Google AI client interface
 */
export type GenAIClient = {
  live: { connect: (config: LiveConnectConfig) => Promise<GenAILiveSession> };
  models: GenAIModelsAPI;
};

/**
 * Google GenAI constructor type
 * Supports both API key (Google AI Studio) and Vertex AI configurations
 */
export type GoogleGenAIClass = new (
  cfg:
    | { apiKey: string }
    | { vertexai: boolean; project: string; location: string },
) => GenAIClient;

// ============================================================================
// OpenAI Compatible Provider Types
// ============================================================================

/**
 * OpenAI-compatible models endpoint response structure
 */
export type ModelsResponse = {
  data: Array<{
    id: string;
    object: string;
    created?: number;
    owned_by?: string;
  }>;
};

// ============================================================================
// Ollama Provider Types
// ============================================================================

/**
 * Ollama tool call structure
 */
export type OllamaToolCall = {
  id: string;
  type: "function";
  function: {
    name: string;
    arguments: string;
  };
};

/**
 * Ollama tool result structure
 */
export type OllamaToolResult = {
  tool_call_id: string;
  content: string;
};

/**
 * Ollama message structure for conversation and tool execution
 */
export type OllamaMessage = {
  role: "system" | "user" | "assistant" | "tool";
  content:
    | string
    | Array<{ type: string; text?: string; [key: string]: unknown }>;
  tool_calls?: OllamaToolCall[];
  images?: string[];
};

/**
 * Default model aliases for easy reference
 */
export const DEFAULT_MODEL_ALIASES = {
  // Latest recommended models per provider
  LATEST_OPENAI: OpenAIModels.GPT_4O,
  FASTEST_OPENAI: OpenAIModels.GPT_4O_MINI,
  LATEST_ANTHROPIC: AnthropicModels.CLAUDE_3_5_SONNET,
  FASTEST_ANTHROPIC: AnthropicModels.CLAUDE_3_5_HAIKU,
  LATEST_GOOGLE: GoogleAIModels.GEMINI_2_5_PRO,
  FASTEST_GOOGLE: GoogleAIModels.GEMINI_2_5_FLASH,

  // Best models by use case
  BEST_CODING: AnthropicModels.CLAUDE_3_5_SONNET,
  BEST_ANALYSIS: GoogleAIModels.GEMINI_2_5_PRO,
  BEST_CREATIVE: AnthropicModels.CLAUDE_3_5_SONNET,
  BEST_VALUE: GoogleAIModels.GEMINI_2_5_FLASH,
} as const;

/**
 * @deprecated Use DEFAULT_MODEL_ALIASES instead. Will be removed in future version.
 */
export const ModelAliases = DEFAULT_MODEL_ALIASES;

/**
 * Default provider configurations
 */
export const DEFAULT_PROVIDER_CONFIGS: AIModelProviderConfig[] = [
  {
    provider: AIProviderName.BEDROCK,
    models: [BedrockModels.CLAUDE_3_7_SONNET, BedrockModels.CLAUDE_3_5_SONNET],
  },
  {
    provider: AIProviderName.VERTEX,
    models: [VertexModels.CLAUDE_4_0_SONNET, VertexModels.GEMINI_2_5_FLASH],
  },
  {
    provider: AIProviderName.OPENAI,
    models: [OpenAIModels.GPT_4O, OpenAIModels.GPT_4O_MINI],
  },
];

// ============================================================================
// Amazon SageMaker Provider Types
// ============================================================================

/**
 * Adaptive semaphore configuration for concurrency management
 */
export type AdaptiveSemaphoreConfig = {
  initialConcurrency: number;
  maxConcurrency: number;
  minConcurrency: number;
};

/**
 * Metrics for adaptive semaphore performance tracking
 */
export type AdaptiveSemaphoreMetrics = {
  activeRequests: number;
  currentConcurrency: number;
  completedCount: number;
  errorCount: number;
  averageResponseTime: number;
  waitingCount: number;
};

/**
 * AWS configuration options for SageMaker client
 */
export type SageMakerConfig = {
  /** AWS region for SageMaker service */
  region: string;
  /** AWS access key ID */
  accessKeyId: string;
  /** AWS secret access key */
  secretAccessKey: string;
  /** AWS session token (optional, for temporary credentials) */
  sessionToken?: string;
  /** Request timeout in milliseconds */
  timeout?: number;
  /** Maximum number of retry attempts */
  maxRetries?: number;
  /** Custom SageMaker endpoint URL (optional) */
  endpoint?: string;
};

/**
 * Model-specific configuration for SageMaker endpoints
 */
export type SageMakerModelConfig = {
  /** SageMaker endpoint name */
  endpointName: string;
  /** Model type for request/response formatting */
  modelType?:
    | "llama"
    | "mistral"
    | "claude"
    | "huggingface"
    | "jumpstart"
    | "custom";
  /** Content type for requests */
  contentType?: string;
  /** Accept header for responses */
  accept?: string;
  /** Custom attributes for the endpoint */
  customAttributes?: string;
  /** Input format specification */
  inputFormat?: "huggingface" | "jumpstart" | "custom";
  /** Output format specification */
  outputFormat?: "huggingface" | "jumpstart" | "custom";
  /** Maximum tokens for generation */
  maxTokens?: number;
  /** Temperature parameter */
  temperature?: number;
  /** Top-p parameter */
  topP?: number;
  /** Stop sequences */
  stopSequences?: string[];
  /** Initial concurrency for batch processing */
  initialConcurrency?: number;
  /** Maximum concurrency for batch processing */
  maxConcurrency?: number;
  /** Minimum concurrency for batch processing */
  minConcurrency?: number;
  /** Maximum concurrent detection tests */
  maxConcurrentDetectionTests?: number;
};

/**
 * SageMaker endpoint information and metadata
 */
export type SageMakerEndpointInfo = {
  /** Endpoint name */
  endpointName: string;
  /** Endpoint ARN */
  endpointArn: string;
  /** Associated model name */
  modelName: string;
  /** EC2 instance type */
  instanceType: string;
  /** Endpoint creation timestamp */
  creationTime: string; // ISO 8601 date string
  /** Last modification timestamp */
  lastModifiedTime: string; // ISO 8601 date string
  /** Current endpoint status */
  endpointStatus:
    | "InService"
    | "Creating"
    | "Updating"
    | "SystemUpdating"
    | "RollingBack"
    | "Deleting"
    | "Failed";
  /** Current instance count */
  currentInstanceCount?: number;
  /** Variant weights for A/B testing */
  productionVariants?: Array<{
    variantName: string;
    modelName: string;
    initialInstanceCount: number;
    instanceType: string;
    currentWeight?: number;
  }>;
};

/**
 * Token usage and billing information
 */
export type SageMakerUsage = {
  /** Number of prompt tokens */
  promptTokens: number;
  /** Number of completion tokens */
  completionTokens: number;
  /** Total tokens used */
  total: number;
  /** Request processing time in milliseconds */
  requestTime?: number;
  /** Model inference time in milliseconds */
  inferenceTime?: number;
  /** Estimated cost in USD */
  estimatedCost?: number;
};

/**
 * Parameters for SageMaker endpoint invocation
 */
export type InvokeEndpointParams = {
  /** Endpoint name to invoke */
  EndpointName: string;
  /** Request body as string or Uint8Array */
  Body: string | Uint8Array;
  /** Content type of the request */
  ContentType?: string;
  /** Accept header for response format */
  Accept?: string;
  /** Custom attributes for the request */
  CustomAttributes?: string;
  /** Target model for multi-model endpoints */
  TargetModel?: string;
  /** Target variant for A/B testing */
  TargetVariant?: string;
  /** Inference ID for request tracking */
  InferenceId?: string;
};

/**
 * Response from SageMaker endpoint invocation
 */
export type InvokeEndpointResponse = {
  /** Response body */
  Body?: Uint8Array;
  /** Content type of the response */
  ContentType?: string;
  /** Invoked production variant */
  InvokedProductionVariant?: string;
  /** Custom attributes in the response */
  CustomAttributes?: string;
};

/**
 * Streaming response chunk from SageMaker
 */
export type SageMakerStreamChunk = {
  /** Text content in the chunk */
  content?: string;
  /** Indicates if this is the final chunk */
  done?: boolean;
  /** Usage information (only in final chunk) */
  usage?: SageMakerUsage;
  /** Error information if chunk contains error */
  error?: string;
  /** Finish reason for generation */
  finishReason?:
    | "stop"
    | "length"
    | "tool-calls"
    | "content-filter"
    | "unknown";
  /** Tool call in progress (Phase 2.3) */
  toolCall?: SageMakerStreamingToolCall;
  /** Tool result chunk (Phase 2.3) */
  toolResult?: SageMakerStreamingToolResult;
  /** Structured output streaming (Phase 2.3) */
  structuredOutput?: SageMakerStructuredOutput;
};

/**
 * Tool call information for function calling
 */
export type SageMakerToolCall = {
  /** Tool call identifier */
  id: string;
  /** Tool/function name */
  name: string;
  /** Tool arguments as JSON object */
  arguments: Record<string, unknown>;
  /** Tool call type */
  type: "function";
};

/**
 * Tool result information
 */
export type SageMakerToolResult = {
  /** Tool call identifier */
  toolCallId: string;
  /** Tool name */
  toolName: string;
  /** Tool result data */
  result: unknown;
  /** Execution status */
  status: "success" | "error";
  /** Error message if status is error */
  error?: string;
};

/**
 * Streaming tool call information (Phase 2.3)
 */
export type SageMakerStreamingToolCall = {
  /** Tool call identifier */
  id: string;
  /** Tool/function name */
  name?: string;
  /** Partial or complete arguments as JSON string */
  arguments?: string;
  /** Tool call type */
  type: "function";
  /** Indicates if this tool call is complete */
  complete?: boolean;
  /** Delta text for incremental argument building */
  argumentsDelta?: string;
};

/**
 * Streaming tool result information (Phase 2.3)
 */
export type SageMakerStreamingToolResult = {
  /** Tool call identifier */
  toolCallId: string;
  /** Tool name */
  toolName: string;
  /** Partial or complete result data */
  result?: unknown;
  /** Result delta for incremental responses */
  resultDelta?: string;
  /** Execution status */
  status: "pending" | "running" | "success" | "error";
  /** Error message if status is error */
  error?: string;
  /** Indicates if this result is complete */
  complete?: boolean;
};

/**
 * Structured output streaming information (Phase 2.3)
 */
export type SageMakerStructuredOutput = {
  /** Partial JSON object being built */
  partialObject?: Record<string, unknown>;
  /** JSON delta text */
  jsonDelta?: string;
  /** Current parsing path (e.g., "user.name") */
  currentPath?: string;
  /** Schema validation errors */
  validationErrors?: string[];
  /** Indicates if JSON is complete and valid */
  complete?: boolean;
  /** JSON schema being validated against */
  schema?: Record<string, unknown>;
};

/**
 * Enhanced generation request options
 */
export type SageMakerGenerationOptions = {
  /** Input prompt text */
  prompt: string;
  /** System prompt for context */
  systemPrompt?: string;
  /** Maximum tokens to generate */
  maxTokens?: number;
  /** Temperature for randomness (0-1) */
  temperature?: number;
  /** Top-p nucleus sampling (0-1) */
  topP?: number;
  /** Top-k sampling */
  topK?: number;
  /** Stop sequences to end generation */
  stopSequences?: string[];
  /** Enable streaming response */
  stream?: boolean;
  /** Tools available for function calling */
  tools?: Array<{
    name: string;
    description: string;
    parameters: Record<string, unknown>;
  }>;
  /** Tool choice mode */
  toolChoice?: "auto" | "none" | { type: "tool"; name: string };
};

/**
 * Generation response from SageMaker
 */
export type SageMakerGenerationResponse = {
  /** Generated text content */
  text: string;
  /** Token usage information */
  usage: SageMakerUsage;
  /** Finish reason for generation */
  finishReason: "stop" | "length" | "tool-calls" | "content-filter" | "unknown";
  /** Tool calls made during generation */
  toolCalls?: SageMakerToolCall[];
  /** Tool results if tools were executed */
  toolResults?: SageMakerToolResult[];
  /** Additional metadata */
  metadata?: Record<string, unknown>;
  /** Model version or identifier */
  modelVersion?: string;
};

/**
 * Error codes specific to SageMaker operations
 */
export type SageMakerErrorCode =
  | "VALIDATION_ERROR"
  | "MODEL_ERROR"
  | "INTERNAL_ERROR"
  | "SERVICE_UNAVAILABLE"
  | "CREDENTIALS_ERROR"
  | "NETWORK_ERROR"
  | "ENDPOINT_NOT_FOUND"
  | "THROTTLING_ERROR"
  | "UNKNOWN_ERROR";

/**
 * SageMaker-specific error information
 */
export type SageMakerErrorInfo = {
  /** Error code */
  code: SageMakerErrorCode;
  /** Human-readable error message */
  message: string;
  /** HTTP status code if applicable */
  statusCode?: number;
  /** Original error from AWS SDK */
  cause?: Error;
  /** Endpoint name where error occurred */
  endpoint?: string;
  /** Request ID for debugging */
  requestId?: string;
  /** Retry suggestion */
  retryable?: boolean;
};

/**
 * Batch inference job configuration
 */
export type BatchInferenceConfig = {
  /** Input S3 location */
  inputS3Uri: string;
  /** Output S3 location */
  outputS3Uri: string;
  /** SageMaker model name */
  modelName: string;
  /** Instance type for batch job */
  instanceType: string;
  /** Instance count for batch job */
  instanceCount: number;
  /** Maximum payload size in MB */
  maxPayloadInMB?: number;
  /** Batch strategy */
  batchStrategy?: "MultiRecord" | "SingleRecord";
};

/**
 * Model deployment configuration
 */
export type ModelDeploymentConfig = {
  /** Model name */
  modelName: string;
  /** Endpoint name */
  endpointName: string;
  /** EC2 instance type */
  instanceType: string;
  /** Initial instance count */
  initialInstanceCount: number;
  /** Model data S3 location */
  modelDataUrl: string;
  /** Container image URI */
  image: string;
  /** IAM execution role ARN */
  executionRoleArn: string;
  /** Resource tags */
  tags?: Record<string, string>;
  /** Auto scaling configuration */
  autoScaling?: {
    minCapacity: number;
    maxCapacity: number;
    targetValue: number;
    scaleUpCooldown: number;
    scaleDownCooldown: number;
  };
};

/**
 * Endpoint metrics and monitoring data
 */
export type EndpointMetrics = {
  /** Endpoint name */
  endpointName: string;
  /** Total invocations */
  invocations: number;
  /** Average latency in milliseconds */
  averageLatency: number;
  /** Error rate percentage */
  errorRate: number;
  /** CPU utilization percentage */
  cpuUtilization?: number;
  /** Memory utilization percentage */
  memoryUtilization?: number;
  /** Instance count */
  instanceCount: number;
  /** Timestamp of metrics as ISO 8601 date string */
  timestamp: string;
};

/**
 * Cost estimation data
 */
export type CostEstimate = {
  /** Estimated cost in USD */
  estimatedCost: number;
  /** Currency code */
  currency: string;
  /** Cost breakdown */
  breakdown: {
    /** Instance hours cost */
    instanceCost: number;
    /** Request-based cost */
    requestCost: number;
    /** Total processing hours */
    totalHours: number;
  };
  /** Time period for estimate */
  period?: {
    start: string; // ISO 8601 date string
    end: string; // ISO 8601 date string
  };
};

/**
 * SageMaker generation result type for better type safety
 */
export type SageMakerGenerateResult = {
  text?: string;
  reasoning?:
    | string
    | Array<
        | { type: "text"; text: string; signature?: string }
        | { type: "redacted"; data: string }
      >;
  files?: Array<{ data: string | Uint8Array; mimeType: string }>;
  logprobs?: Array<{
    token: string;
    logprob: number;
    topLogprobs: Array<{ token: string; logprob: number }>;
  }>;
  usage: {
    promptTokens: number;
    completionTokens: number;
    totalTokens?: number;
  };
  finishReason:
    | "stop"
    | "length"
    | "content-filter"
    | "tool-calls"
    | "error"
    | "unknown";
  warnings?: Array<{ type: "other"; message: string }>;
  rawCall: { rawPrompt: unknown; rawSettings: Record<string, unknown> };
  rawResponse?: { headers?: Record<string, string> };
  request?: { body?: string };
  toolCalls?: SageMakerToolCall[];
  object?: unknown;
};

export type ProviderHealthStatusOptions = {
  provider: AIProviderName;
  isHealthy: boolean;
  isConfigured: boolean;
  hasApiKey: boolean;
  lastChecked: Date;
  error?: string;
  warning?: string;
  responseTime?: number;
  configurationIssues: string[];
  recommendations: string[];
};

export type ProviderHealthCheckOptions = {
  timeout?: number;
  includeConnectivityTest?: boolean;
  includeModelValidation?: boolean;
  cacheResults?: boolean;
  maxCacheAge?: number;
};

// ============================================================================
// Provider-Specific Namespace Types
// ============================================================================

/**
 * Amazon Bedrock specific types
 */
export namespace BedrockTypes {
  // Based on AWS SDK Bedrock types
  export type Client = {
    send(command: unknown): Promise<unknown>;
    config: {
      region?: string;
      credentials?: unknown;
    };
  };

  // Based on AWS SDK types
  export type InvokeModelCommand = {
    input: {
      modelId: string;
      body: string;
      contentType?: string;
    };
  };
}

/**
 * Mistral specific types
 */
export namespace MistralTypes {
  // Based on Mistral SDK types
  export type Client = {
    chat?: {
      complete?: (options: unknown) => Promise<unknown>;
      stream?: (options: unknown) => AsyncIterable<unknown>;
    };
  };
}

/**
 * OpenTelemetry specific types (for telemetry service)
 */
export namespace TelemetryTypes {
  export type Meter = {
    createCounter(name: string, options?: unknown): Counter;
    createHistogram(name: string, options?: unknown): Histogram;
  };

  export type Tracer = {
    startSpan(name: string, options?: unknown): Span;
  };

  export type Counter = {
    add(value: number, attributes?: UnknownRecord): void;
  };

  export type Histogram = {
    record(value: number, attributes?: UnknownRecord): void;
  };

  export type Span = {
    end(): void;
    setStatus(status: unknown): void;
    recordException(exception: unknown): void;
  };
}

// ============================================================================
// OpenRouter Provider Types
// ============================================================================

/**
 * OpenRouter provider configuration
 */
export type OpenRouterConfig = {
  /** OpenRouter API key */
  apiKey: string;
  /** HTTP Referer header for attribution on openrouter.ai/activity */
  referer?: string;
  /** App name for X-Title header attribution */
  appName?: string;
};

/**
 * OpenRouter model information from /api/v1/models endpoint
 */
export type OpenRouterModelInfo = {
  /** Model ID in format 'provider/model-name' */
  id: string;
  /** Supported parameters (e.g., 'tools', 'temperature') */
  supported_parameters?: string[];
  /** Model name */
  name?: string;
  /** Model description */
  description?: string;
  /** Pricing information */
  pricing?: {
    prompt?: string;
    completion?: string;
  };
  /** Context length */
  context_length?: number;
};

/**
 * OpenRouter models API response
 */
export type OpenRouterModelsResponse = {
  data: OpenRouterModelInfo[];
};

/**
 * OpenRouter provider static cache properties (for testing/internal use)
 */
export type OpenRouterProviderCache = {
  modelsCache: string[];
  modelsCacheTime: number;
  toolCapableModels: Set<string>;
  capabilitiesCached: boolean;
};
