import { z } from "zod";

/**
 * META Ads Analysis Output Schema - Vertex AI Compatible Version
 *
 * This schema enforces the structured output format for META Ads performance analysis
 * (including ROAS - Return on Ad Spend) generated by agentic loops. It ensures consistent,
 * comprehensive audit reports with:
 * - Account performance metrics and trend diagnosis
 * - Campaign and ad set level analysis
 * - Critical issues and risk register
 * - Prioritized action roadmap
 *
 * NOTE: All length constraints are specified in descriptions rather than schema validation
 * to ensure compatibility with Google Vertex AI's constrained decoding system.
 */

/**
 * Metric comparison schema - used for all metrics that need current/previous/change
 */
const MetricComparisonSchema = z.object({
  current: z.number().describe("Current period value"),
  previous: z.number().describe("Comparison period value"),
  changePercent: z
    .number()
    .describe(
      "Percentage change from previous to current period, can be a positive or negative value",
    ),
  trend: z
    .enum(["improving", "declining", "stable"])
    .nullish()
    .describe("Directional trend indicator"),
  businessImplication: z
    .string()
    .describe(
      "REQUIRED: 1 line insight, maximum 20 words. What might be the cause for the change",
    ),
});

/**
 * Root cause categories for issues
 */
const RootCauseCategorySchema = z.enum([
  "Data/Tracking",
  "Targeting/Audience",
  "Creative/Offer",
  "Technical/Budgeting",
]);

/**
 * Responsible teams/areas for action items
 */
const ResponsibleAreaSchema = z.enum([
  "Media Buying",
  "Creative Strategy",
  "Tracking & Analytics",
]);

/**
 * Account-level performance metrics
 * All metrics follow the current/previous/change pattern with business implications
 */
const AccountMetricsSchema = z.object({
  totalSpend: MetricComparisonSchema.describe(
    "Total advertising spend in account currency. Critical for ROAS calculation.",
  ),
  totalReach: MetricComparisonSchema.describe(
    "Total unique users reached by ads. Key indicator of campaign scale.",
  ),
  ctr: MetricComparisonSchema.describe(
    "Click-through rate (clicks/impressions * 100). Measures ad engagement quality.",
  ),
  averageFrequency: MetricComparisonSchema.describe(
    "Average number of times a user saw an ad. High frequency may indicate audience fatigue.",
  ),
  cpc: MetricComparisonSchema.describe(
    "Cost per click. Indicates bid competitiveness and ad relevance.",
  ),
  totalClicks: MetricComparisonSchema.describe(
    "Total number of clicks across all campaigns. Measures overall user interest.",
  ),
  roas: MetricComparisonSchema.nullish().describe(
    "OPTIONAL: Return on Ad Spend. Primary performance indicator if conversion data available.",
  ),
  conversions: MetricComparisonSchema.nullish().describe(
    "OPTIONAL: Total conversions tracked. Include only if conversion tracking is enabled.",
  ),
});

/**
 * Entity-level metrics for campaigns/ad sets
 */
const EntityMetricsSchema = z.object({
  roas: z.number().nullish().describe("Return on Ad Spend for this entity"),
  ctr: z.number().nullish().describe("Click-through rate percentage"),
  cpa: z.number().nullish().describe("OPTIONAL: Cost per acquisition/lead"),
  cpl: z
    .number()
    .nullish()
    .describe("OPTIONAL: Cost per lead (if different from CPA)"),
  spend: z.number().nullish().describe("Total spend on this entity"),
  impressions: z.number().nullish().describe("Total impressions"),
  clicks: z.number().nullish().describe("Total clicks"),
  conversions: z
    .number()
    .nullish()
    .describe("OPTIONAL: Total conversions if tracked"),
});

/**
 * Individual entity (campaign or ad set) performance
 */
const EntityPerformanceSchema = z.object({
  entityId: z.string().describe("Campaign ID or Ad Set ID from META Ads API"),
  entityName: z.string().describe("Human-readable name of the campaign/ad set"),
  entityType: z
    .enum(["campaign", "adset"])
    .describe("Type of entity being analyzed"),
  metrics: EntityMetricsSchema,
  diagnosticReasoning: z
    .string()
    .describe(
      "REQUIRED: 2 lines, maximum 40 words (approximately 80-160 characters). " +
        "Focus on WHY this entity succeeded/failed with ROOT CAUSE analysis: " +
        "targeting accuracy, creative resonance, audience engagement patterns, bid efficiency. " +
        "Avoid just describing metric changes. Be specific with data-driven reasoning.",
    ),
});

/**
 * Campaign and Ad Set level analysis section
 */
const CampaignAnalysisSchema = z.object({
  topPerformers: z
    .array(EntityPerformanceSchema)
    .describe(
      "REQUIRED: Provide 1-3 items. " +
        "Top 3-5 best performing campaigns/ad sets based on ROAS and efficiency",
    ),

  underperformers: z
    .array(EntityPerformanceSchema)
    .describe(
      "REQUIRED: Provide 1-3 items. " +
        "Top 3-5 worst performing campaigns/ad sets requiring optimization",
    ),
});

/**
 * Individual critical issue
 */
const CriticalIssueSchema = z.object({
  issueId: z.string().describe("Unique identifier for this issue"),
  affectedEntity: z
    .string()
    .describe('Campaign ID, Ad Set ID, or "account" for account-level issues'),
  affectedEntityName: z
    .string()
    .describe("Human-readable name of affected entity"),
  rootCauseCategory: RootCauseCategorySchema,
  severity: z
    .enum(["critical", "high", "medium"])
    .describe("Issue severity level"),

  quantifiedImpact: z.object({
    metric: z
      .string()
      .describe("Metric being impacted (e.g., ROAS, CTR, Conversions)"),
    currentValue: z.number().describe("Current value of the metric"),
    expectedValue: z.number().describe("Expected/benchmark value"),
    impactAmount: z
      .number()
      .describe("Quantified impact (difference or percentage loss)"),
  }),

  detailedExplanation: z
    .string()
    .describe(
      "REQUIRED: 2 lines, maximum 40 words (approximately 80-160 characters). " +
        "Focus on ROOT CAUSE and IMPACT: " +
        "What is causing this issue (specific data patterns) and the business consequence if unresolved. " +
        "Use specific numbers and clear cause-effect reasoning. Avoid symptom descriptions.",
    ),
});

/**
 * Critical issues section
 */
const CriticalIssuesSchema = z
  .array(CriticalIssueSchema)
  .describe(
    "OPTIONAL: Can be empty array (0-10 items). " +
      "List of critical issues detected. Provide empty array if no issues found.",
  );

/**
 * Individual action item
 */
const ActionItemSchema = z.object({
  actionId: z.string().describe("Unique identifier for this action"),
  objective: z
    .string()
    .describe(
      "REQUIRED: 30-80 characters. " +
        "Clear, specific objective of this action. What will be done.",
    ),

  expectedOutcome: z
    .string()
    .describe(
      "REQUIRED: 5-10 words. " +
        "Measurable business outcome expected from this action. " +
        'Include specific metrics and quantified improvements (e.g., "Increase ROAS by 15%", "Reduce CPA by $5").',
    ),

  responsibleArea: ResponsibleAreaSchema,

  estimatedImpact: z.object({
    metric: z.string().describe("Primary metric this action will improve"),
    currentValue: z.number().describe("Current value of the metric"),
    projectedValue: z.number().describe("Projected value after action"),
  }),

  rationale: z
    .string()
    .describe(
      "REQUIRED: 2 lines, maximum 40 words (approximately 80-160 characters). " +
        "Focus on WHY this action will work: root cause it addresses, data evidence supporting it, " +
        "and expected business impact. Use professional, CMO-facing language. " +
        "Avoid implementation details.",
    ),

  timeline: z
    .string()
    .nullish()
    .describe(
      'OPTIONAL: Suggested timeline for implementation (e.g., "Immediate", "1-2 weeks")',
    ),
});

/**
 * Action roadmap section with priority tiers
 */
const ActionRoadmapSchema = z.object({
  high: z
    .array(ActionItemSchema)
    .length(1)
    .describe(
      "REQUIRED: Provide exactly 1 item. " +
        "Single highest priority action with immediate, significant business impact",
    ),

  medium: z
    .array(ActionItemSchema)
    .max(1)
    .describe(
      "OPTIONAL: Provide at most 1 item (0-1 items). " +
        "Medium priority action for optimization or improvement",
    ),

  budgetReallocation: z
    .array(ActionItemSchema)
    .max(1)
    .optional()
    .describe(
      "OPTIONAL: Provide at most 1 budget reallocation action item (0-1 items). " +
        "Comprehensive budget efficiency analysis covering: " +
        "1) Optimal spend areas (high ROAS entities), " +
        "2) Inefficient spend areas (low ROAS, high waste), " +
        "3) Budget allocation recommendations, " +
        "4) Expected ROAS improvement from reallocation. " +
        "Include specific campaign/ad set names and quantified impact in the action fields.",
    ),
});

/**
 * Complete META Ads analysis report schema
 * This is the root schema that will be used with neurolinkInstance.generate()
 */
export const MetaAdsAnalysisOutputSchema = z.object({
  analysisMetadata: z
    .object({
      accountId: z.string().describe("META Ads account ID analyzed"),
      accountName: z.string().describe("Account name"),
      analysisDateRange: z.object({
        since: z.string().describe("Analysis period start date (YYYY-MM-DD)"),
        until: z.string().describe("Analysis period end date (YYYY-MM-DD)"),
      }),
      comparisonDateRange: z.object({
        since: z.string().describe("Comparison period start date (YYYY-MM-DD)"),
        until: z.string().describe("Comparison period end date (YYYY-MM-DD)"),
      }),
      generatedAt: z
        .string()
        .describe("ISO timestamp when analysis was generated"),
      currency: z.string().describe("Account currency code (e.g., USD, INR)"),
    })
    .describe("Metadata about the analysis itself"),

  accountPerformance: AccountMetricsSchema.describe(
    "Section 1: Account-level performance metrics with per-metric business implications",
  ),

  campaignAnalysis: CampaignAnalysisSchema.describe(
    "Section 2: Campaign and ad set level performance analysis",
  ),

  criticalIssues: CriticalIssuesSchema.describe(
    "Section 3: Critical issues and risk register",
  ),

  actionRoadmap: ActionRoadmapSchema.describe(
    "Section 4: Prioritized action items organized by priority level",
  ),
});
