#!/usr/bin/env node
/**
 * NeuroLink Shell-to-JavaScript Converter - Phase 2C Implementation
 * Converts shell scripts to cross-platform Node.js JavaScript
 *
 * Features:
 * - Pattern-based conversion of common shell commands
 * - Cross-platform Node.js equivalents
 * - Modern ES modules with async/await
 * - Error handling and validation
 * - Template generation for clean output
 * - Integration with VS Code tasks
 */

import fs from "fs/promises";
import path from "path";

class ShellConverter {
  constructor() {
    this.scriptsDir = "./scripts";
    this.outputDir = "./tools/converted-scripts";

    // Common shell command to Node.js conversions
    this.conversionMap = {
      echo: "console.log",
      "mkdir -p": "await fs.mkdir(path, { recursive: true })",
      "cp -r": "await fs.cp(source, dest, { recursive: true })",
      mv: "await fs.rename(oldPath, newPath)",
      "rm -rf": "await fs.rm(path, { recursive: true, force: true })",
      "rm -f": "await fs.unlink(path)",
      ls: "await fs.readdir(path)",
      pwd: "process.cwd()",
      cd: "process.chdir(path)",
      cat: 'await fs.readFile(path, "utf-8")',
      touch: 'await fs.writeFile(path, "")',
      chmod: "// Use fs.chmod(path, mode) if needed",
    };

    this.results = {
      converted: [],
      failed: [],
      total: 0,
    };
  }

  /**
   * Convert all shell scripts in the scripts directory
   */
  async convertAllShellScripts() {
    console.log("🔄 Starting shell-to-JavaScript conversion...");
    console.log("🎯 Target: Cross-platform Node.js compatibility");

    try {
      // Ensure output directory exists
      await fs.mkdir(this.outputDir, { recursive: true });
      console.log(`📁 Output directory: ${this.outputDir}`);

      // Find all shell scripts
      const shellScripts = await this.findShellScripts();
      this.results.total = shellScripts.length;

      console.log(`🐚 Found ${shellScripts.length} shell scripts to convert:`);
      shellScripts.forEach((script) => console.log(`  • ${script}`));

      // Convert each script
      for (const script of shellScripts) {
        try {
          await this.convertScript(script);
          this.results.converted.push(script);
          console.log(`✅ Converted: ${script}`);
        } catch (error) {
          this.results.failed.push({ script, error: error.message });
          console.log(`❌ Failed: ${script} - ${error.message}`);
        }
      }

      this.reportResults();
      return this.results;
    } catch (error) {
      console.error("❌ Conversion process failed:", error.message);
      throw error;
    }
  }

  /**
   * Find all shell scripts in the scripts directory
   */
  async findShellScripts() {
    try {
      const files = await fs.readdir(this.scriptsDir);
      return files.filter((file) => file.endsWith(".sh"));
    } catch (error) {
      throw new Error(`Failed to read scripts directory: ${error.message}`);
    }
  }

  /**
   * Convert a single shell script to JavaScript
   */
  async convertScript(shellFile) {
    console.log(`🔄 Converting ${shellFile} to JavaScript...`);

    const shellPath = path.join(this.scriptsDir, shellFile);
    const jsFileName = shellFile.replace(".sh", ".js");
    const jsPath = path.join(this.outputDir, jsFileName);

    try {
      // Read shell script content
      const content = await fs.readFile(shellPath, "utf-8");
      const lines = content.split("\n");

      // Generate JavaScript equivalent
      const jsContent =
        this.generateJSTemplate(shellFile) +
        this.convertShellCommands(lines) +
        this.generateJSFooter();

      // Write JavaScript file
      await fs.writeFile(jsPath, jsContent);

      console.log(`✅ Created: ${jsPath}`);
      return jsPath;
    } catch (error) {
      throw new Error(`Failed to convert ${shellFile}: ${error.message}`);
    }
  }

  /**
   * Generate JavaScript template with imports and setup
   */
  generateJSTemplate(originalFile) {
    return `#!/usr/bin/env node
/**
 * JavaScript conversion of ${path.basename(originalFile)}
 * Auto-generated by NeuroLink Shell Converter
 *
 * Original shell script converted to cross-platform Node.js
 * Uses modern ES modules and async/await patterns
 */

import fs from 'fs/promises';
import path from 'path';
import { execSync } from 'child_process';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

/**
 * Main execution function
 */
async function main() {
  try {
    console.log('🚀 Starting ${originalFile} (JavaScript version)...');

`;
  }

  /**
   * Convert shell commands to JavaScript equivalents
   */
  convertShellCommands(lines) {
    let jsCode = "";

    for (const line of lines) {
      const trimmed = line.trim();

      // Skip shebang, empty lines, and comments
      if (trimmed.startsWith("#") || !trimmed) {
        if (trimmed.startsWith("#") && !trimmed.startsWith("#!/")) {
          jsCode += `    // ${trimmed.substring(1)}\n`;
        }
        continue;
      }

      // Convert the line
      const converted = this.convertLine(trimmed);
      jsCode += `    ${converted}\n`;
    }

    return jsCode;
  }

  /**
   * Convert a single shell command line to JavaScript
   */
  convertLine(shellLine) {
    // Handle variable assignments
    if (
      shellLine.includes("=") &&
      !shellLine.includes(" ") &&
      !shellLine.includes("==")
    ) {
      const [key, value] = shellLine.split("=");
      return `const ${key} = ${this.convertValue(value)};`;
    }

    // Handle echo commands
    if (shellLine.startsWith("echo ")) {
      const message = shellLine.substring(5).replace(/['"]/g, "");
      return `console.log('${message}');`;
    }

    // Handle cd commands
    if (shellLine.startsWith("cd ")) {
      const targetDir = shellLine.substring(3).trim().replace(/['"]/g, "");
      return `process.chdir('${targetDir}');`;
    }

    // Handle mkdir commands
    if (shellLine.startsWith("mkdir -p ") || shellLine.startsWith("mkdir ")) {
      const dirPath = shellLine
        .replace(/mkdir\s+(-p\s+)?/, "")
        .trim()
        .replace(/['"]/g, "");
      return `await fs.mkdir('${dirPath}', { recursive: true });`;
    }

    // Handle rm commands
    if (shellLine.startsWith("rm -rf ")) {
      const targetPath = shellLine.substring(7).trim().replace(/['"]/g, "");
      return `await fs.rm('${targetPath}', { recursive: true, force: true });`;
    }

    if (shellLine.startsWith("rm -f ")) {
      const targetPath = shellLine.substring(6).trim().replace(/['"]/g, "");
      return `await fs.unlink('${targetPath}');`;
    }

    // Handle cp commands
    if (shellLine.startsWith("cp -r ")) {
      const parts = shellLine.substring(6).trim().split(/\s+/);
      if (parts.length >= 2) {
        const source = parts[0].replace(/['"]/g, "");
        const dest = parts[1].replace(/['"]/g, "");
        return `await fs.cp('${source}', '${dest}', { recursive: true });`;
      }
    }

    // Handle mv commands
    if (shellLine.startsWith("mv ")) {
      const parts = shellLine.substring(3).trim().split(/\s+/);
      if (parts.length >= 2) {
        const oldPath = parts[0].replace(/['"]/g, "");
        const newPath = parts[1].replace(/['"]/g, "");
        return `await fs.rename('${oldPath}', '${newPath}');`;
      }
    }

    // Handle ls commands
    if (shellLine.startsWith("ls ")) {
      const dirPath = shellLine.substring(3).trim().replace(/['"]/g, "") || ".";
      return `const files = await fs.readdir('${dirPath}'); console.log(files);`;
    }

    // Handle cat commands
    if (shellLine.startsWith("cat ")) {
      const filePath = shellLine.substring(4).trim().replace(/['"]/g, "");
      return `const content = await fs.readFile('${filePath}', 'utf-8'); console.log(content);`;
    }

    // Handle find commands (basic conversion)
    if (shellLine.startsWith("find ")) {
      // Parse basic find patterns
      const findMatch = shellLine.match(
        /find\s+([^\s]+)\s*(-name\s+[^\s]+)?(.+)?/,
      );
      if (findMatch) {
        const searchPath = findMatch[1].replace(/['"]/g, "");
        const namePattern = findMatch[2]
          ? findMatch[2].replace(/-name\s+/, "").replace(/['"]/g, "")
          : "*";
        const additionalArgs = findMatch[3] || "";

        return `// Find files matching pattern
const glob = await import('glob');
const files = await glob.glob('${namePattern}', { 
  cwd: '${searchPath}', 
  absolute: true 
});${additionalArgs.includes("-delete") ? "\n// Delete found files\nfor (const file of files) {\n  await fs.unlink(file);\n}" : ""}`;
      }
      return `// Complex find command - manual conversion needed: ${shellLine}`;
    }

    // Handle grep commands
    if (shellLine.includes(" | grep ")) {
      // Parse grep pipeline patterns
      const grepMatch = shellLine.match(/(.+)\s*\|\s*grep\s+([^\s]+)(.+)?/);
      if (grepMatch) {
        const sourceCommand = grepMatch[1].trim();
        const grepPattern = grepMatch[2].replace(/['"]/g, "");
        const grepOptions = grepMatch[3] || "";

        // Convert common source commands
        let sourceConversion = "";
        if (sourceCommand.startsWith("cat ")) {
          const filePath = sourceCommand
            .substring(4)
            .trim()
            .replace(/['"]/g, "");
          sourceConversion = `const content = await fs.readFile('${filePath}', 'utf-8');`;
        } else if (sourceCommand.startsWith("ls")) {
          sourceConversion = `const content = (await fs.readdir('.')).join('\\n');`;
        } else {
          sourceConversion = `// Source command: ${sourceCommand}\nconst content = /* result from above command */;`;
        }

        return `${sourceConversion}
// Filter content with grep pattern
const lines = content.split('\\n');
const filteredLines = lines.filter(line => {
  const regex = new RegExp('${grepPattern}'${grepOptions.includes("-i") ? ", 'i'" : ""});
  return regex.test(line);
});
console.log(filteredLines.join('\\n'));`;
      }
      return `// Complex grep pipeline - manual conversion needed: ${shellLine}`;
    }

    // Fallback: execute as shell command with warning
    return `// Shell command: ${shellLine}
    try {
      execSync('${shellLine.replace(/'/g, "\\'")}', { stdio: 'inherit' });
    } catch (error) {
      console.warn('Shell command failed:', error.message);
    }`;
  }

  /**
   * Convert shell variable values to JavaScript
   */
  convertValue(value) {
    // Remove quotes and return as string literal
    const cleanValue = value.replace(/^['"]|['"]$/g, "");

    // Check if it's a number
    if (/^\d+$/.test(cleanValue)) {
      return cleanValue;
    }

    // Check if it's a boolean
    if (cleanValue === "true" || cleanValue === "false") {
      return cleanValue;
    }

    // Default to string
    return `'${cleanValue}'`;
  }

  /**
   * Generate JavaScript footer with error handling
   */
  generateJSFooter() {
    return `
    console.log('✅ Script execution completed successfully');

  } catch (error) {
    console.error('❌ Script execution failed:', error.message);
    process.exit(1);
  }
}

// Export for module use
export { main };

// Run if called directly
if (import.meta.url === \`file://\${process.argv[1]}\`) {
  main().catch(error => {
    console.error('Unhandled error:', error);
    process.exit(1);
  });
}
`;
  }

  /**
   * Create specific conversions for known NeuroLink scripts
   */
  async createSpecificConversions() {
    console.log("🎯 Creating specific conversions for NeuroLink scripts...");

    const specificConversions = {
      "cleanupHashNamedVideos.sh": this.generateVideoCleanupScript(),
      "generateAllVideos.sh": this.generateVideoGenerationScript(),
    };

    for (const [shellFile, jsContent] of Object.entries(specificConversions)) {
      const jsFileName = shellFile.replace(".sh", ".js");
      const jsPath = path.join(this.outputDir, jsFileName);

      try {
        await fs.writeFile(jsPath, jsContent);
        console.log(`✅ Created specific conversion: ${jsFileName}`);
      } catch (error) {
        console.log(`❌ Failed to create ${jsFileName}: ${error.message}`);
      }
    }
  }

  /**
   * Generate optimized video cleanup script
   */
  generateVideoCleanupScript() {
    return `#!/usr/bin/env node
/**
 * Video Cleanup Script - JavaScript Version
 * Removes hash-named video files from the visual content directory
 */

import fs from 'fs/promises';
import path from 'path';

async function cleanupHashNamedVideos() {
  console.log('🧹 Starting hash-named video cleanup...');

  try {
    const videoDir = './docs/visual-content/videos';

    // Check if directory exists
    try {
      await fs.access(videoDir);
    } catch {
      console.log('📁 Video directory not found, creating...');
      await fs.mkdir(videoDir, { recursive: true });
      return;
    }

    // Read directory contents
    const files = await fs.readdir(videoDir);

    // Pattern for hash-named files (8+ hex characters)
    const hashPattern = /^[a-f0-9]{8,}\\.(mp4|webm|gif|mov)$/i;
    const cleanupTargets = files.filter(file => hashPattern.test(file));

    console.log(\`🔍 Found \${files.length} total files\`);
    console.log(\`🎯 Found \${cleanupTargets.length} hash-named videos to clean\`);

    if (cleanupTargets.length === 0) {
      console.log('✅ No hash-named videos found - cleanup not needed');
      return;
    }

    // Remove hash-named videos
    for (const file of cleanupTargets) {
      const filePath = path.join(videoDir, file);
      const stats = await fs.stat(filePath);

      try {
        await fs.unlink(filePath);
        console.log(\`🗑️ Removed: \${file} (\${(stats.size / 1024 / 1024).toFixed(1)} MB)\`);
      } catch (error) {
        console.warn(\`⚠️ Failed to remove \${file}: \${error.message}\`);
      }
    }

    console.log(\`🎉 Cleanup complete! Removed \${cleanupTargets.length} files\`);

  } catch (error) {
    console.error('❌ Video cleanup failed:', error.message);
    throw error;
  }
}

// Export for module use
export { cleanupHashNamedVideos };

// Run if called directly
if (import.meta.url === \`file://\${process.argv[1]}\`) {
  cleanupHashNamedVideos().catch(error => {
    console.error('Unhandled error:', error);
    process.exit(1);
  });
}
`;
  }

  /**
   * Generate optimized video generation orchestrator script
   */
  generateVideoGenerationScript() {
    return `#!/usr/bin/env node
/**
 * Video Generation Orchestrator - JavaScript Version
 * Coordinates execution of all video generation scripts
 */

import { execSync } from 'child_process';
import fs from 'fs/promises';

async function generateAllVideos() {
  console.log('🎬 Starting comprehensive video generation pipeline...');

  try {
    const videoScripts = [
      'createCliOverviewVideo.js',
      'createMcpVideos.js',
      'createAiWorkflowCliDemo.js',
      'createCliScreenshots.js',
      'createMcpScreenshots.js'
    ];

    const results = {
      success: [],
      failed: [],
      total: videoScripts.length
    };

    console.log(\`📋 Executing \${videoScripts.length} video generation scripts...\`);

    for (const script of videoScripts) {
      const scriptPath = \`./scripts/\${script}\`;

      try {
        // Check if script exists
        await fs.access(scriptPath);

        console.log(\`🎥 Running: \${script}\`);
        execSync(\`node \${scriptPath}\`, {
          stdio: 'inherit',
          timeout: 300000 // 5 minute timeout per script
        });

        results.success.push(script);
        console.log(\`✅ Completed: \${script}\`);

      } catch (error) {
        results.failed.push({ script, error: error.message });
        console.error(\`❌ Failed: \${script} - \${error.message}\`);
      }
    }

    // Report results
    console.log('\\n📊 VIDEO GENERATION RESULTS');
    console.log('═'.repeat(40));
    console.log(\`✅ Successful: \${results.success.length}/\${results.total}\`);
    console.log(\`❌ Failed: \${results.failed.length}/\${results.total}\`);

    if (results.success.length > 0) {
      console.log('\\n🎉 Successfully generated:');
      results.success.forEach(script => console.log(\`  ✅ \${script}\`));
    }

    if (results.failed.length > 0) {
      console.log('\\n⚠️ Failed scripts:');
      results.failed.forEach(({script, error}) => console.log(\`  ❌ \${script}: \${error}\`));
    }

    console.log(\`\\n🎬 Video generation pipeline complete!\`);
    return results;

  } catch (error) {
    console.error('❌ Video generation pipeline failed:', error.message);
    throw error;
  }
}

// Export for module use
export { generateAllVideos };

// Run if called directly
if (import.meta.url === \`file://\${process.argv[1]}\`) {
  generateAllVideos().catch(error => {
    console.error('Unhandled error:', error);
    process.exit(1);
  });
}
`;
  }

  /**
   * Report conversion results
   */
  reportResults() {
    console.log("\n📊 SHELL-TO-JS CONVERSION RESULTS");
    console.log("═".repeat(50));
    console.log(`🎯 Total scripts: ${this.results.total}`);
    console.log(`✅ Successfully converted: ${this.results.converted.length}`);
    console.log(`❌ Failed conversions: ${this.results.failed.length}`);

    if (this.results.converted.length > 0) {
      console.log("\n🎉 Successfully converted:");
      this.results.converted.forEach((script) => {
        const jsName = script.replace(".sh", ".js");
        console.log(`  ✅ ${script} → ${jsName}`);
      });
    }

    if (this.results.failed.length > 0) {
      console.log("\n⚠️ Failed conversions:");
      this.results.failed.forEach(({ script, error }) => {
        console.log(`  ❌ ${script}: ${error}`);
      });
    }

    const successRate = (
      (this.results.converted.length / this.results.total) *
      100
    ).toFixed(1);
    console.log(`\n📈 Success rate: ${successRate}%`);
    console.log(`📁 Output directory: ${this.outputDir}`);
    console.log("\n💡 Next steps:");
    console.log("1. Review converted scripts for accuracy");
    console.log("2. Test JavaScript versions");
    console.log("3. Update package.json scripts to use JS versions");
    console.log("4. Remove original shell scripts after validation");
  }
}

// Export for module use
export { ShellConverter };

// CLI execution when run directly
if (import.meta.url === `file://${process.argv[1]}`) {
  const converter = new ShellConverter();

  try {
    if (process.argv.includes("--specific")) {
      await converter.createSpecificConversions();
    } else {
      await converter.convertAllShellScripts();
    }

    console.log("\n✅ Shell-to-JavaScript conversion complete!");
    process.exit(0);
  } catch (error) {
    console.error("❌ Conversion failed:", error.message);
    process.exit(1);
  }
}
