from django.core.management.base import BaseCommand

from ix.agents.models import Agent
from ix.chains.models import ChainNode, Chain


ARTIFACT_FORMAT = """
{
    "name": "filename",
    "key": "unique_artifact_key",
    "description": "1-3 sentence description of artifact",
    "identifier": "file_path",
}
"""


CREATE_CODE_ARTIFACTS_V1 = """
You are a python coder. You are planning code edits and will describe files for the user request.

ARTIFACTS: Artifacts represent the results or consequences of executing a command or action.
They can be new objects created (e.g., code components, data, files) or changes to the system
or environment state (e.g., modified settings, enabled features, activated services).

ARTIFACT TYPES:
- file: a file that will be created
- file_edit: an edit to an existing file

ARTIFACT TOOL SELECTION:
- write_file for new files
- append_file for adding new code block to an existing file
- replace_lines for replacing existing code block in an existing file

ARTIFACT_FORMAT:
{artifact_format}

OUTPUT_FORMAT:
###START###
[
    ARTIFACT,
    ARTIFACT
]
###END###

{related_artifacts}

INSTRUCTIONS:
    - generate an ARTIFACT for each file or modification required for the request.
    - only include files that you will generate content for.
    - do not split files into multiple ARTIFACTS.
    - do not include files that are generated by another process or command.
    - structure your response to match OUTPUT_FORMAT without any other text or explanation.
    - DO NOT add fields to the OUTPUT_FORMAT.
    - if editing an artifact, use the same key and storage_id as the original artifact.
    - output must include markers ###START### and ###END###
"""


FILE_CONTENT_FORMAT = """
{
    "file_content": "file content here"
}
"""


GENERATE_CODE_V1 = """
You are a python coder. You will writes files for the user request.

{related_artifacts}

FILE ARTIFACT:
{file_artifact}

OUTPUT_FORMAT:
###START###
{file_content_format}
###END###

INSTRUCTIONS:
    - write the file described by the FILE ARTIFACT description
    - structure your response to match OUTPUT_FORMAT without any other text or explanation.
    - output must include markers ###START### and ###END###
"""


CREATE_CODE_ARTIFACTS = {
    "class_path": "ix.chains.llm_chain.LLMChain",
    "config": {
        "llm": {
            "class_path": "langchain_community.chat_models.ChatOpenAI",
            "config": {
                "model": "gpt-4",
                "request_timeout": 240,
                "temperature": 0.2,
                "verbose": True,
            },
        },
        "prompt": {
            "class_path": "langchain.prompts.chat.ChatPromptTemplate",
            "config": {
                "messages": [
                    {
                        "role": "system",
                        "template": CREATE_CODE_ARTIFACTS_V1,
                        "input_variables": ["related_artifacts"],
                        "partial_variables": {
                            "artifact_format": ARTIFACT_FORMAT,
                        },
                    },
                    {
                        "role": "user",
                        "template": "{user_input}",
                        "input_variables": ["user_input"],
                    },
                ],
            },
        },
    },
}


PARSE_ARTIFACT_LIST_JSON = {
    "class_path": "ix.chains.json.ParseJSON",
    "config": {
        "input_key": "text",
        "output_key": "file_artifacts_json",
    },
}


SAVE_ARTIFACT_LIST = {
    "class_path": "ix.chains.artifacts.SaveArtifact",
    "config": {
        "artifact_key": "file_list",
        "artifact_name": "file_list",
        "artifact_description": "list of files that will be generated",
        "artifact_type": "artifact_list",
        "artifact_storage": "write_file",
        "content_key": "file_artifacts_json",
        "output_key": "file_artifacts_list",
    },
}


GENERATE_FILE = {
    "class_path": "ix.chains.llm_chain.LLMChain",
    "config": {
        "llm": {
            "class_path": "langchain_community.chat_models.ChatOpenAI",
            "config": {
                "model": "gpt-4",
                "request_timeout": 240,
                "temperature": 0.2,
                "verbose": True,
            },
        },
        "prompt": {
            "class_path": "langchain.prompts.chat.ChatPromptTemplate",
            "config": {
                "messages": [
                    {
                        "role": "system",
                        "template": GENERATE_CODE_V1,
                        "input_variables": ["file_artifact", "related_artifacts"],
                        "partial_variables": {
                            "file_content_format": FILE_CONTENT_FORMAT,
                        },
                    },
                    {
                        "role": "user",
                        "template": "file artifact list was generated from this input:\n {user_input}",
                        "input_variables": ["user_input"],
                    },
                ],
            },
        },
    },
}


PARSE_FILE_JSON = {
    "class_path": "ix.chains.json.ParseJSON",
    "config": {
        "input_key": "text",
        "output_key": "generated_file_json",
    },
}


SAVE_FILE_ARTIFACT = {
    "class_path": "ix.chains.artifacts.SaveArtifact",
    "config": {
        "artifact_from_key": "file_artifact",
        "artifact_type": "file",
        "artifact_storage": "write_to_file",
        "content_key": "generated_file_json",
        "content_path": "generated_file_json.file_content",
        "output_key": "generated_file_artifact",
    },
}


GENERATE_FILES_MAP = {
    "name": "generate files",
    "description": "Generates files for each artifact in the input list",
    "class_path": "ix.chains.routing.MapSubchain",
    "config": {
        "input_variables": ["file_artifacts_json", "related_artifacts", "user_input"],
        "map_input": "file_artifacts_json",
        "map_input_to": "file_artifact",
        "output_key": "write_file_output",
        "chains": [
            GENERATE_FILE,
            PARSE_FILE_JSON,
            SAVE_FILE_ARTIFACT,
        ],
    },
}


CODER_SEQUENCE = {
    "name": "create files",
    "description": "Write new files containing code",
    "class_path": "langchain.chains.SequentialChain",
    "hidden": False,
    "config": {
        "input_variables": ["user_input"],
        "memory": {
            "class_path": "ix.memory.artifacts.ArtifactMemory",
        },
        "chains": [
            CREATE_CODE_ARTIFACTS,
            PARSE_ARTIFACT_LIST_JSON,
            SAVE_ARTIFACT_LIST,
            GENERATE_FILES_MAP,
        ],
    },
}


CODER_V1_CHAIN = "b7d8f662-12f6-4525-b07b-c9ea7c10000c"
CODER_V1_AGENT = "b7d8f662-12f6-4525-b07b-c9ea7c10000a"


class Command(BaseCommand):
    help = "Generates planner v4 chain"

    def handle(self, *args, **options):
        chain, is_new = Chain.objects.get_or_create(
            pk=CODER_V1_CHAIN,
            defaults=dict(
                name="Coder chain v1",
                description="Chain used to generate code files",
            ),
        )

        # clear old nodes
        chain.clear_chain()
        ChainNode.objects.create_from_config(
            chain=chain, root=True, config=CODER_SEQUENCE
        )

        Agent.objects.get_or_create(
            id=CODER_V1_AGENT,
            defaults=dict(
                name="Coder v1",
                alias="code",
                purpose="To generate code for a user request. May generate multiple files",
                chain=chain,
                config={},
            ),
        )
