import logging
from uuid import UUID

import graphene
import channels_graphql_ws
from graphene.types.generic import GenericScalar
from graphene_django import DjangoObjectType

from ix.agents.models import Agent
from ix.chat.models import Chat
from ix.runnable_log.subscription import RunEventSubscription

# from ix.runnable_log.subscription import RunEventSubscription
from ix.task_log.models import TaskLogMessage, Artifact, Task

logger = logging.getLogger(__name__)


class AgentType(DjangoObjectType):
    class Meta:
        model = Agent
        fields = "__all__"

    config = GenericScalar()


class TaskLogMessageType(DjangoObjectType):
    class Meta:
        model = TaskLogMessage
        fields = "__all__"

    content = GenericScalar()


class ArtifactType(DjangoObjectType):
    storage = GenericScalar()

    class Meta:
        model = Artifact
        fields = "__all__"


class ChatMessageSubscription(channels_graphql_ws.Subscription):
    """GraphQL subscription to TaskLogMessage instances."""

    # Subscription payload - FKs aren't working over websockets
    # so add related objects manually
    task_log_message = graphene.Field(TaskLogMessageType)
    agent = graphene.Field(AgentType)
    parent_id = graphene.UUID()

    class Arguments:
        chatId = graphene.String()

    @staticmethod
    async def subscribe(root, info, chatId):
        """Called when client subscribes."""
        logger.debug(f"client subscribing to chatId: {chatId}")
        chat = await Chat.objects.aget(id=chatId)
        task_id = chat.task_id
        return [f"task_id_{task_id}"]

    @staticmethod
    async def publish(payload, info, chatId):
        """Called to notify subscribed clients."""
        msg = payload.get("instance")
        agent = payload.get("agent")
        return ChatMessageSubscription(
            task_log_message=msg, parent_id=msg.parent_id, agent=agent
        )

    @classmethod
    def new_task_log_message(cls, sender, **kwargs):
        """
        Called when new task log message instance is saved. Messages are
        Broadcast to all clients subscribed to the chat's task_id.
        Subtasks are broadcast to the parent task's task_id.
        """

        # messages shouldn't be updated but check just in case
        if not kwargs.get("created", False):
            return

        instance = kwargs["instance"]
        root_id = instance.task.root_id if instance.task.root_id else instance.task.id
        cls.broadcast(
            group=f"task_id_{root_id}",
            payload={"instance": instance, "agent": instance.task.agent},
        )


class ChatMessageTokenSubscription(channels_graphql_ws.Subscription):
    """GraphQL subscription to streamed messages.

    This subscription streams messages tokens as they are generated by the
    agent. The stream only includes the message_id and text.
    """

    msg_id = graphene.UUID()
    index = graphene.Int()
    text = graphene.String()

    class Arguments:
        chatId = graphene.String()

    @staticmethod
    async def subscribe(root, info, chatId):
        """Called when client subscribes."""
        chat = await Chat.objects.aget(id=chatId)
        task_id = chat.task_id
        logger.debug(
            f"Client subscribing to STREAM! chatId: {chatId} task_id: {task_id}"
        )

        return [f"stream_task_id_{task_id}"]

    @staticmethod
    async def publish(payload, info, chatId):
        """Called to notify subscribed clients."""
        return ChatMessageTokenSubscription(
            msg_id=payload.get("msg_id"),
            index=payload.get("index"),
            text=payload.get("text"),
        )

    @classmethod
    async def on_new_token(cls, task: Task, message_id: UUID, index: int, text: str):
        """
        Generic handler for new message tokens.
        """
        task_id = task.root_id if task.root_id else task.id
        cls.broadcast(
            group=f"stream_task_id_{task_id}",
            payload={
                "msg_id": str(message_id),
                "index": index,
                "text": text,
            },
        )


class ChatArtifactSubscription(channels_graphql_ws.Subscription):
    """GraphQL subscription to Artifact instances."""

    artifact = graphene.Field(ArtifactType)

    class Arguments:
        chatId = graphene.String()

    @staticmethod
    async def subscribe(root, info, chatId):
        """Called when client subscribes."""
        logger.debug(f"client subscribing for chat artifacts for chat_id={chatId}")
        chat = await Chat.objects.aget(id=chatId)
        task_id = chat.task_id
        return [f"artifacts_task_id_{task_id}"]

    @staticmethod
    async def publish(payload, info, chatId):
        """Called to notify subscribed clients."""
        instance = payload.get("instance")
        return ChatArtifactSubscription(artifact=instance)

    @classmethod
    def new_artifact(cls, sender, **kwargs):
        """
        Called when new task artifact instance is saved. Messages are
        Broadcast to all clients subscribed to the chat's task_id.
        Subtasks are broadcast to the parent task's task_id.
        """
        # messages shouldn't be updated but check just in case
        if not kwargs.get("created", False):
            return

        instance = kwargs["instance"]
        task_id = instance.task.root_id if instance.task.root_id else instance.task.id
        cls.broadcast(
            group=f"artifacts_task_id_{task_id}",
            payload={"instance": instance},
        )


class Subscription(graphene.ObjectType):
    """Root GraphQL subscription."""

    chatMessageSubscription = ChatMessageSubscription.Field()
    chatMessageTokenSubscription = ChatMessageTokenSubscription.Field()
    chatArtifactSubscription = ChatArtifactSubscription.Field()
    runEventSubscription = RunEventSubscription.Field()
