import Page from "@/src/components/layouts/page";
import { BreadcrumbSeparator } from "@/src/components/ui/breadcrumb";
import { BreadcrumbPage } from "@/src/components/ui/breadcrumb";
import { BreadcrumbItem } from "@/src/components/ui/breadcrumb";
import { Check } from "lucide-react";
import { cn } from "@/src/utils/tailwind";
import { BreadcrumbList } from "@/src/components/ui/breadcrumb";
import { Breadcrumb } from "@/src/components/ui/breadcrumb";
import { useRouter } from "next/router";
import { SelectEvaluatorList } from "@/src/features/evals/components/select-evaluator-list";
import { RunEvaluatorForm } from "@/src/features/evals/components/run-evaluator-form";
import { api } from "@/src/utils/api";
import { useHasProjectAccess } from "@/src/features/rbac/utils/checkProjectAccess";
import { getMaintainer } from "@/src/features/evals/utils/typeHelpers";
import { MaintainerTooltip } from "@/src/features/evals/components/maintainer-tooltip";
import { ManageDefaultEvalModel } from "@/src/features/evals/components/manage-default-eval-model";
import { DefaultEvalModelSetup } from "@/src/features/evals/components/default-eval-model-setup";

// Multi-step setup process
// 0. Set up default model (optional, only if no default model exists): /project/:projectId/evals/new
// 1. Select Evaluator: /project/:projectId/evals/new
// 2. Configure Evaluator: /project/:projectId/evals/new?evaluator=:evaluatorId
export default function NewEvaluatorPage() {
  const router = useRouter();
  const projectId = router.query.projectId as string;
  const evaluatorId = router.query.evaluator as string | undefined;

  const hasDefaultModelReadAccess = useHasProjectAccess({
    projectId,
    scope: "evalDefaultModel:read",
  });

  const { data: defaultModel } = api.defaultLlmModel.fetchDefaultModel.useQuery(
    { projectId },
    { enabled: hasDefaultModelReadAccess && !!projectId },
  );

  const hasDefaultModel = !!defaultModel;

  // Calculate step number: if no default model, start at 0, otherwise start at 1
  const stepInt = !hasDefaultModel ? 0 : !evaluatorId ? 1 : 2;

  const hasAccess = useHasProjectAccess({
    projectId,
    scope: "evalTemplate:CUD",
  });

  const evalTemplates = api.evals.allTemplates.useQuery(
    {
      projectId,
      limit: 500,
      page: 0,
    },
    {
      enabled: hasAccess,
    },
  );

  const currentTemplate = evalTemplates.data?.templates.find(
    (t) => t.id === evaluatorId,
  );

  if (!hasAccess) {
    return <div>You do not have access to this page.</div>;
  }

  return (
    <Page
      withPadding
      headerProps={{
        title: "Set up evaluator",
        breadcrumb: [
          {
            name: "Running Evaluators",
            href: `/project/${projectId}/evals`,
          },
        ],
        actionButtonsRight: <ManageDefaultEvalModel projectId={projectId} />,
      }}
    >
      <Breadcrumb className="mb-3">
        <BreadcrumbList>
          {!hasDefaultModel && (
            <>
              <BreadcrumbItem>
                <BreadcrumbPage
                  className={cn(
                    stepInt !== 0
                      ? "text-muted-foreground"
                      : "font-semibold text-foreground",
                  )}
                >
                  0. Set up default model
                  {stepInt > 0 && (
                    <Check className="ml-1 inline-block h-3 w-3" />
                  )}
                </BreadcrumbPage>
              </BreadcrumbItem>
              <BreadcrumbSeparator />
            </>
          )}
          <BreadcrumbItem
            className="hover:cursor-pointer"
            onClick={() => router.push(`/project/${projectId}/evals/new`)}
          >
            <BreadcrumbPage
              className={cn(
                stepInt !== 1
                  ? "text-muted-foreground"
                  : "font-semibold text-foreground",
              )}
            >
              1. Select Evaluator
              {stepInt > 1 && <Check className="ml-1 inline-block h-3 w-3" />}
            </BreadcrumbPage>
          </BreadcrumbItem>
          <BreadcrumbSeparator />
          <BreadcrumbItem>
            <BreadcrumbPage
              className={cn(
                stepInt !== 2
                  ? "text-muted-foreground"
                  : "font-semibold text-foreground",
              )}
            >
              <div className="flex flex-row">
                2. Run Evaluator
                {currentTemplate && (
                  <div className="flex flex-row gap-2">
                    <span>
                      {currentTemplate.name ? `: ${currentTemplate.name}` : ""}
                    </span>
                    <MaintainerTooltip
                      maintainer={getMaintainer(currentTemplate)}
                    />
                  </div>
                )}
              </div>
            </BreadcrumbPage>
          </BreadcrumbItem>
        </BreadcrumbList>
      </Breadcrumb>
      {
        // 0. Set up default model
        stepInt === 0 && projectId && (
          <DefaultEvalModelSetup projectId={projectId} />
        )
      }
      {
        // 1. Select Evaluator
        stepInt === 1 && projectId && (
          <SelectEvaluatorList projectId={projectId} />
        )
      }
      {
        // 2. Run Evaluator
        stepInt === 2 && evaluatorId && projectId && (
          <RunEvaluatorForm
            projectId={projectId}
            evaluatorId={evaluatorId}
            evalTemplates={evalTemplates.data?.templates ?? []}
          />
        )
      }
    </Page>
  );
}
