'use client';

import { ActionIcon, Flexbox, Popover, Tooltip } from '@lobehub/ui';
import { ArrowLeft, ArrowRight, Clock } from 'lucide-react';
import { memo, useCallback, useEffect, useState } from 'react';
import { useTranslation } from 'react-i18next';

import { useGlobalStore } from '@/store/global';
import { systemStatusSelectors } from '@/store/global/selectors';
import { electronStylish } from '@/styles/electron';
import { isMacOS } from '@/utils/platform';

import { useNavigationHistory } from '../navigation/useNavigationHistory';
import RecentlyViewed from './RecentlyViewed';

const isMac = isMacOS();

const useNavPanelWidth = () => {
  return useGlobalStore(systemStatusSelectors.leftPanelWidth);
};

const NavigationBar = memo(() => {
  const { t } = useTranslation('electron');
  const { canGoBack, canGoForward, goBack, goForward } = useNavigationHistory();
  const [historyOpen, setHistoryOpen] = useState(false);
  // Use ResizeObserver for real-time width updates during resize
  const leftPanelWidth = useNavPanelWidth();

  // Toggle history popover
  const toggleHistoryOpen = useCallback(() => {
    setHistoryOpen((prev) => !prev);
  }, []);

  // Listen for keyboard shortcut ⌘Y / Ctrl+Y
  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      const isCmdOrCtrl = isMac ? event.metaKey : event.ctrlKey;
      if (isCmdOrCtrl && event.key.toLowerCase() === 'y') {
        event.preventDefault();
        toggleHistoryOpen();
      }
    };

    window.addEventListener('keydown', handleKeyDown);
    return () => {
      window.removeEventListener('keydown', handleKeyDown);
    };
  }, [toggleHistoryOpen]);

  // Tooltip content for the clock button
  const tooltipContent = t('navigation.recentView');

  return (
    <Flexbox
      align="center"
      className={electronStylish.nodrag}
      data-width={leftPanelWidth}
      horizontal
      justify="end"
      style={{ width: `${leftPanelWidth - 12}px` }}
    >
      <Flexbox align="center" gap={2} horizontal>
        <ActionIcon disabled={!canGoBack} icon={ArrowLeft} onClick={goBack} size="small" />
        <ActionIcon disabled={!canGoForward} icon={ArrowRight} onClick={goForward} size="small" />
        <Popover
          content={<RecentlyViewed onClose={() => setHistoryOpen(false)} />}
          onOpenChange={setHistoryOpen}
          open={historyOpen}
          placement="bottomLeft"
          styles={{ content: { padding: 0 } }}
          trigger="click"
        >
          <div>
            <Tooltip open={historyOpen ? false : undefined} title={tooltipContent}>
              <ActionIcon icon={Clock} size="small" />
            </Tooltip>
          </div>
        </Popover>
      </Flexbox>
    </Flexbox>
  );
});

NavigationBar.displayName = 'NavigationBar';

export default NavigationBar;
