import type {
  GroupOrchestrationEvent,
  GroupOrchestrationExecutor,
  GroupOrchestrationExecutorOutput,
  SupervisorInstruction,
  SupervisorInstructionCallAgent,
  SupervisorInstructionCallSupervisor,
  SupervisorInstructionDelegate,
  SupervisorInstructionExecAsyncTask,
  SupervisorInstructionParallelCallAgents,
} from '@lobechat/agent-runtime';
import type { ConversationContext, UIChatMessage } from '@lobechat/types';
import debug from 'debug';

import { aiAgentService } from '@/services/aiAgent';
import { dbMessageSelectors } from '@/store/chat/slices/message/selectors';
import type { ChatStore } from '@/store/chat/store';
import { messageMapKey } from '@/store/chat/utils/messageMapKey';

const log = debug('lobe-store:group-orchestration-executors');

const sleep = (ms: number) =>
  new Promise<void>((resolve) => {
    setTimeout(resolve, ms);
  });

export interface GroupOrchestrationExecutorsContext {
  get: () => ChatStore;
  /**
   * Message context for fetching messages
   * Contains agentId (groupId for group chat), topicId, scope, etc.
   */
  messageContext: ConversationContext;
  orchestrationOperationId: string;
  supervisorAgentId: string;
}

/**
 * Creates executors for Group Orchestration
 *
 * Architecture:
 * - Supervisor (State Machine): Receives ExecutorResult → Returns SupervisorInstruction
 * - Executor (Execution Layer): Receives SupervisorInstruction → Returns ExecutorResult
 *
 * Flow:
 * ```
 * Supervisor.decide(init)
 *        │
 *        └─► call_supervisor instruction
 *                │
 *                └─► call_supervisor Executor
 *                        │
 *                        ├─► internal_execAgentRuntime(Supervisor)
 *                        │        │
 *                        │        ├─► Supervisor calls speak tool
 *                        │        │        │
 *                        │        │        └─► tool handler triggers orchestration
 *                        │        │
 *                        │        └─► Supervisor finishes normally
 *                        │
 *                        └─► Returns supervisor_decided result
 *                                │
 *                                └─► Supervisor.decide(supervisor_decided)
 *                                        │
 *                                        └─► call_agent instruction
 *                                                │
 *                                                └─► call_agent Executor
 *                                                        │
 *                                                        └─► Returns agent_spoke result
 *                                                                │
 *                                                                └─► Supervisor.decide(agent_spoke)
 *                                                                        │
 *                                                                        └─► call_supervisor OR finish
 * ```
 */
export const createGroupOrchestrationExecutors = (
  context: GroupOrchestrationExecutorsContext,
): Partial<Record<SupervisorInstruction['type'], GroupOrchestrationExecutor>> => {
  const { get, messageContext, orchestrationOperationId, supervisorAgentId } = context;

  // Pre-compute the chat key for message fetching
  const chatKey = messageMapKey(messageContext);

  /**
   * Helper to get current messages for the group conversation
   */
  const getMessages = () => dbMessageSelectors.getDbMessagesByKey(chatKey)(get());

  /* eslint-disable sort-keys-fix/sort-keys-fix */

  return {
    /**
     * call_supervisor Executor
     * Executes the Supervisor Agent completely
     *
     * Returns: supervisor_decided result with the decision made by supervisor
     *
     * Note: When Supervisor calls a group-management tool (speak/broadcast/delegate),
     * the tool returns `stop: true` which terminates the AgentRuntime.
     * The tool also registers an afterCompletion callback to trigger the orchestration.
     */
    call_supervisor: async (instruction, state): Promise<GroupOrchestrationExecutorOutput> => {
      const { supervisorAgentId: agentId } = (instruction as SupervisorInstructionCallSupervisor)
        .payload;

      const sessionLogId = `${state.operationId}:call_supervisor`;
      log(`[${sessionLogId}] Starting supervisor agent: ${agentId}`);

      const messages = getMessages();
      const lastMessage = messages.at(-1);

      if (!lastMessage) {
        log(`[${sessionLogId}] No messages found, cannot execute supervisor`);
        return {
          events: [{ type: 'supervisor_finished' }] as GroupOrchestrationEvent[],
          newState: { ...state, status: 'done' },
          // Supervisor finished without action - end orchestration
          result: undefined,
        };
      }

      // Variable to capture the decision from tool handler
      // let decision: ExecutorResult['type'] | undefined;
      // let decisionParams: Record<string, unknown> = {};
      // let skipCallSupervisor = false;

      // Execute Supervisor agent with the supervisor's agentId in context
      // Mark isSupervisor=true so assistant messages get metadata.isSupervisor for UI rendering
      await get().internal_execAgentRuntime({
        context: { ...messageContext, agentId: supervisorAgentId, isSupervisor: true },
        messages,
        operationId: state.operationId,
        parentMessageId: lastMessage.id,
        parentMessageType: lastMessage.role as 'user' | 'assistant' | 'tool',
        parentOperationId: orchestrationOperationId,
      });

      log(`[${sessionLogId}] Supervisor agent finished`);

      // Check what decision was made by the supervisor
      // This is captured from the groupOrchestration callbacks registered by tools
      // const orchestrationCallbacks = get().getGroupOrchestrationCallbacks();

      // If no tool was called (supervisor finished normally), end orchestration
      // The actual decision is captured via the afterCompletion callbacks
      // For now, return a finish decision if we reach here
      return {
        events: [{ type: 'supervisor_finished' }] as GroupOrchestrationEvent[],
        newState: state,
        result: {
          payload: {
            decision: 'finish',
            params: { reason: 'supervisor_completed_without_action' },
            skipCallSupervisor: false,
          },
          type: 'supervisor_decided',
        },
      };
    },

    /**
     * call_agent Executor
     * Executes a single target Agent completely
     *
     * Returns: agent_spoke result
     *
     * If the Supervisor provides an instruction, it will be injected as a virtual
     * User Message at the end of the messages array. This improves instruction-following
     * as User Messages have stronger influence on model behavior.
     */
    call_agent: async (instruction, state): Promise<GroupOrchestrationExecutorOutput> => {
      const { agentId, instruction: agentInstruction } = (
        instruction as SupervisorInstructionCallAgent
      ).payload;

      const sessionLogId = `${state.operationId}:call_agent`;
      log(`[${sessionLogId}] Calling agent: ${agentId}, instruction: ${agentInstruction}`);

      const messages = getMessages();
      const lastMessage = messages.at(-1);

      if (!lastMessage) {
        log(`[${sessionLogId}] No messages found, cannot execute agent`);
        return {
          events: [{ agentId, type: 'agent_spoke' }] as GroupOrchestrationEvent[],
          newState: state,
          result: { payload: { agentId, completed: true }, type: 'agent_spoke' },
        };
      }

      // If instruction is provided, inject it as a virtual User Message
      // This virtual message is not persisted to database, only used for model context
      // Mark with <speaker> tag so the agent knows this instruction is from the Supervisor
      const now = Date.now();
      const messagesWithInstruction: UIChatMessage[] = agentInstruction
        ? [
            ...messages,
            {
              content: `<speaker name="Supervisor" />\n${agentInstruction}`,
              createdAt: now,
              id: `virtual_speak_instruction_${now}`,
              role: 'user',
              updatedAt: now,
            },
          ]
        : messages;

      // Execute target Agent with subAgentId for agent config retrieval
      // - messageContext keeps the group's main conversation context (for message storage)
      // - subAgentId specifies which agent's config to use
      await get().internal_execAgentRuntime({
        context: { ...messageContext, subAgentId: agentId },
        messages: messagesWithInstruction,
        parentMessageId: lastMessage.id,
        parentMessageType: lastMessage.role as 'user' | 'assistant' | 'tool',
        parentOperationId: orchestrationOperationId,
      });

      log(`[${sessionLogId}] Agent ${agentId} finished speaking`);

      // Return agent_spoke result
      return {
        events: [{ agentId, type: 'agent_spoke' }] as GroupOrchestrationEvent[],
        newState: state,
        result: {
          payload: { agentId, completed: true },
          type: 'agent_spoke',
        },
      };
    },

    /**
     * parallel_call_agents Executor
     * Executes multiple Agents in parallel
     *
     * Returns: agents_broadcasted result
     *
     * If the Supervisor provides an instruction, it will be injected as a virtual
     * User Message at the end of the messages array. This improves instruction-following
     * as User Messages have stronger influence on model behavior.
     */
    parallel_call_agents: async (instruction, state): Promise<GroupOrchestrationExecutorOutput> => {
      const {
        agentIds,
        instruction: agentInstruction,
        toolMessageId,
      } = (instruction as SupervisorInstructionParallelCallAgents).payload;

      const sessionLogId = `${state.operationId}:parallel_call_agents`;
      log(
        `[${sessionLogId}] Broadcasting to agents: ${agentIds.join(', ')}, instruction: ${agentInstruction}, toolMessageId: ${toolMessageId}`,
      );

      const messages = getMessages();

      if (messages.length === 0) {
        log(`[${sessionLogId}] No messages found, cannot execute agents`);
        return {
          events: [{ agentIds, type: 'agents_broadcasted' }] as GroupOrchestrationEvent[],
          newState: state,
          result: {
            payload: { agentIds, completed: true },
            type: 'agents_broadcasted',
          },
        };
      }

      // If instruction is provided, inject it as a virtual User Message
      // This virtual message is not persisted to database, only used for model context
      // Mark with <speaker> tag so the agent knows this instruction is from the Supervisor
      const now = Date.now();
      const messagesWithInstruction: UIChatMessage[] = agentInstruction
        ? [
            ...messages,
            {
              content: `<speaker name="Supervisor" />\n${agentInstruction}`,
              createdAt: now,
              id: `virtual_broadcast_instruction_${now}`,
              role: 'user',
              updatedAt: now,
            },
          ]
        : messages;

      // Execute all Agents in parallel, each with their own subAgentId for config retrieval
      // - messageContext keeps the group's main conversation context (for message storage)
      // - subAgentId specifies which agent's config to use for each agent
      // - toolMessageId is used as parentMessageId so agent responses are children of the tool message
      await Promise.all(
        agentIds.map(async (agentId) => {
          await get().internal_execAgentRuntime({
            context: { ...messageContext, subAgentId: agentId },
            messages: messagesWithInstruction,
            parentMessageId: toolMessageId,
            parentMessageType: 'tool',
            parentOperationId: orchestrationOperationId,
          });
        }),
      );

      log(`[${sessionLogId}] All agents finished broadcasting`);

      // Return agents_broadcasted result
      return {
        events: [{ agentIds, type: 'agents_broadcasted' }] as GroupOrchestrationEvent[],
        newState: state,
        result: {
          payload: { agentIds, completed: true },
          type: 'agents_broadcasted',
        },
      };
    },

    /**
     * delegate Executor
     * Delegates control to another agent
     *
     * Returns: delegated result
     */
    delegate: async (instruction, state): Promise<GroupOrchestrationExecutorOutput> => {
      const { agentId, reason } = (instruction as SupervisorInstructionDelegate).payload;

      const sessionLogId = `${state.operationId}:delegate`;
      log(`[${sessionLogId}] Delegating to agent: ${agentId}, reason: ${reason}`);

      const messages = getMessages();
      const lastMessage = messages.at(-1);

      if (!lastMessage) {
        log(`[${sessionLogId}] No messages found, cannot delegate`);
        return {
          events: [] as GroupOrchestrationEvent[],
          newState: state,
          result: {
            payload: { agentId, completed: true },
            type: 'delegated',
          },
        };
      }

      // Execute delegated Agent
      await get().internal_execAgentRuntime({
        context: { ...messageContext, subAgentId: agentId },
        messages,
        parentMessageId: lastMessage.id,
        parentMessageType: lastMessage.role as 'user' | 'assistant' | 'tool',
        parentOperationId: orchestrationOperationId,
      });

      log(`[${sessionLogId}] Delegated agent ${agentId} finished`);

      // Return delegated result
      return {
        events: [] as GroupOrchestrationEvent[],
        newState: state,
        result: {
          payload: { agentId, completed: true },
          type: 'delegated',
        },
      };
    },

    /**
     * exec_async_task Executor
     * Executes an async task for an agent using aiAgentService with polling
     *
     * Flow:
     * 1. Create a task message (role: 'task') as placeholder
     * 2. Call execGroupSubAgentTask API (backend creates thread with sourceMessageId)
     * 3. Poll for task completion
     * 4. Update task message content with summary on completion
     *
     * Returns: task_completed result
     */
    exec_async_task: async (instruction, state): Promise<GroupOrchestrationExecutorOutput> => {
      const { agentId, task, timeout, title, toolMessageId } = (
        instruction as SupervisorInstructionExecAsyncTask
      ).payload;

      const sessionLogId = `${state.operationId}:exec_async_task`;
      log(
        `[${sessionLogId}] Executing async task for agent: ${agentId}, task: ${task}, timeout: ${timeout}`,
      );

      const { groupId, topicId } = messageContext;

      if (!groupId || !topicId) {
        log(`[${sessionLogId}] No valid context, cannot execute async task`, groupId, topicId);
        return {
          events: [] as GroupOrchestrationEvent[],
          newState: state,
          result: {
            payload: { agentId, error: 'No valid context available', success: false },
            type: 'task_completed',
          },
        };
      }

      try {
        // 1. Create task message as placeholder
        const taskMessageResult = await get().optimisticCreateMessage(
          {
            agentId,
            content: '',
            groupId,
            metadata: { instruction: task, taskTitle: title },
            parentId: toolMessageId,
            role: 'task',
            topicId,
          },
          { operationId: state.operationId },
        );

        if (!taskMessageResult) {
          console.error(`[${sessionLogId}] Failed to create task message`);
          return {
            events: [] as GroupOrchestrationEvent[],
            newState: state,
            result: {
              payload: { agentId, error: 'Failed to create task message', success: false },
              type: 'task_completed',
            },
          };
        }

        const taskMessageId = taskMessageResult.id;
        log(`[${sessionLogId}] Created task message: ${taskMessageId}`);

        // 2. Create task via backend API (backend creates thread with sourceMessageId)
        const createResult = await aiAgentService.execSubAgentTask({
          agentId,
          groupId,
          instruction: task,
          parentMessageId: taskMessageId,
          title,
          topicId,
        });

        if (!createResult.success) {
          log(`[${sessionLogId}] Failed to create task: ${createResult.error}`);
          // Update task message with error
          await get().optimisticUpdateMessageContent(
            taskMessageId,
            `Task creation failed: ${createResult.error}`,
            undefined,
            { operationId: state.operationId },
          );
          return {
            events: [] as GroupOrchestrationEvent[],
            newState: state,
            result: {
              payload: { agentId, error: createResult.error, success: false },
              type: 'task_completed',
            },
          };
        }

        log(`[${sessionLogId}] Task created with threadId: ${createResult.threadId}`);

        // 3. Poll for task completion
        const pollInterval = 3000; // 3 seconds
        const maxWait = timeout || 1_800_000; // Default 30 minutes
        const startTime = Date.now();

        while (Date.now() - startTime < maxWait) {
          // Check if operation has been cancelled
          const currentOperation = get().operations[state.operationId];
          if (currentOperation?.status === 'cancelled') {
            console.warn(`[${sessionLogId}] Operation cancelled, stopping polling`);
            return {
              events: [] as GroupOrchestrationEvent[],
              newState: { ...state, status: 'done' },
              result: {
                payload: { agentId, error: 'Operation cancelled', success: false },
                type: 'task_completed',
              },
            };
          }

          const status = await aiAgentService.getSubAgentTaskStatus({
            threadId: createResult.threadId,
          });

          // Update taskDetail in message if available
          if (status.taskDetail) {
            get().internal_dispatchMessage(
              {
                id: taskMessageId,
                type: 'updateMessage',
                value: { taskDetail: status.taskDetail },
              },
              { operationId: state.operationId },
            );
            log(`[${sessionLogId}] Updated task message with taskDetail`);
          }

          if (status.status === 'completed') {
            log(`[${sessionLogId}] Task completed successfully`);
            // 4. Update task message with summary
            if (status.result) {
              await get().optimisticUpdateMessageContent(taskMessageId, status.result, undefined, {
                operationId: state.operationId,
              });
            }
            return {
              events: [] as GroupOrchestrationEvent[],
              newState: state,
              result: {
                payload: { agentId, result: status.result, success: true },
                type: 'task_completed',
              },
            };
          }

          if (status.status === 'failed') {
            console.error(`[${sessionLogId}] Task failed: ${status.error}`);
            // Update task message with error
            await get().optimisticUpdateMessageContent(
              taskMessageId,
              `Task failed: ${status.error}`,
              undefined,
              { operationId: state.operationId },
            );
            return {
              events: [] as GroupOrchestrationEvent[],
              newState: state,
              result: {
                payload: { agentId, error: status.error, success: false },
                type: 'task_completed',
              },
            };
          }

          if (status.status === 'cancel') {
            log(`[${sessionLogId}] Task was cancelled`);
            // Update task message with cancelled status
            await get().optimisticUpdateMessageContent(
              taskMessageId,
              'Task was cancelled',
              undefined,
              { operationId: state.operationId },
            );
            return {
              events: [] as GroupOrchestrationEvent[],
              newState: state,
              result: {
                payload: { agentId, error: 'Task was cancelled', success: false },
                type: 'task_completed',
              },
            };
          }

          // Still processing, wait and poll again
          await sleep(pollInterval);
        }

        // Timeout reached
        log(`[${sessionLogId}] Task timeout after ${maxWait}ms`);
        // Update task message with timeout error
        await get().optimisticUpdateMessageContent(
          taskMessageId,
          `Task timeout after ${maxWait}ms`,
          undefined,
          { operationId: state.operationId },
        );

        return {
          events: [] as GroupOrchestrationEvent[],
          newState: state,
          result: {
            payload: { agentId, error: `Task timeout after ${maxWait}ms`, success: false },
            type: 'task_completed',
          },
        };
      } catch (error) {
        log(`[${sessionLogId}] Error executing async task: ${error}`);
        return {
          events: [] as GroupOrchestrationEvent[],
          newState: state,
          result: {
            payload: {
              agentId,
              error: error instanceof Error ? error.message : 'Unknown error',
              success: false,
            },
            type: 'task_completed',
          },
        };
      }
    },
  };
};
