/* Copyright 2026 Marimo. All rights reserved. */

import { EditorView } from "@codemirror/view";
import { Logger } from "@/utils/Logger";

export function insertDebuggerAtLine(view: EditorView, line: number): boolean {
  // Get the document
  const { state } = view;
  const doc = state.doc;

  // Check if the line number is valid
  if (line <= 0 || line > doc.lines) {
    Logger.warn(
      `Invalid line number: ${line}. Document has ${doc.lines} lines.`,
    );
    return false;
  }

  // Get the target line
  const targetLine = doc.line(line);

  // Skip if line already contains breakpoint()
  if (targetLine.text.includes("breakpoint()")) {
    return true;
  }

  // Extract the indentation from the target line
  const lineContent = targetLine.text;
  const indentMatch = lineContent.match(/^(\s*)/);
  const indentation = indentMatch ? indentMatch[1] : "";

  // Create the breakpoint statement with the same indentation
  const breakpointStatement = `${indentation}breakpoint()\n`;

  // Get the position where we need to insert the breakpoint statement
  const insertPos = targetLine.from;

  // Create and dispatch the transaction
  view.dispatch({
    changes: {
      from: insertPos,
      to: insertPos,
      insert: breakpointStatement,
    },
    // Scroll to the breakpoint
    selection: {
      anchor: insertPos + breakpointStatement.length - 1,
      head: insertPos + breakpointStatement.length - 1,
    },
    scrollIntoView: true,
    effects: [EditorView.scrollIntoView(insertPos, { y: "center" })],
  });

  // Focus the editor after the transaction
  view.focus();

  return true;
}
