# Copyright 2026 Marimo. All rights reserved.
from __future__ import annotations

import os
import re
import shutil
import time
from pathlib import Path
from typing import TYPE_CHECKING

import pytest

from marimo import __version__
from marimo._dependencies.dependencies import DependencyManager
from marimo._messaging.cell_output import CellChannel, CellOutput
from marimo._messaging.notification import CellNotification
from marimo._output.utils import uri_encode_component
from marimo._types.ids import CellId_t, SessionId
from marimo._utils.platform import is_windows
from tests._server.conftest import get_session_manager
from tests._server.mocks import (
    token_header,
    with_read_session,
    with_session,
)
from tests.mocks import EDGE_CASE_FILENAMES, snapshotter

if TYPE_CHECKING:
    from starlette.testclient import TestClient

snapshot = snapshotter(__file__)

SESSION_ID = SessionId("session-123")
HEADERS = {
    "Marimo-Session-Id": SESSION_ID,
    **token_header("fake-token"),
}

CODE = uri_encode_component("import marimo as mo")


@with_session(SESSION_ID)
def test_export_html(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    body = response.text
    assert '<marimo-code hidden=""></marimo-code>' not in body
    assert CODE in body


@with_session(SESSION_ID)
def test_export_html_skew_protection(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers={
            **HEADERS,
            "Marimo-Server-Token": "old-skew-id",
        },
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    assert response.status_code == 401
    assert response.json() == {"error": "Invalid server token"}


@with_session(SESSION_ID)
def test_export_html_no_code(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": False,
        },
    )
    body = response.text
    assert '<marimo-code hidden=""></marimo-code>' in body
    assert CODE not in body


@with_session(SESSION_ID)
def test_export_html_file_not_found(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": ["/test-10.csv"],
            "includeCode": True,
        },
    )
    assert response.status_code == 200
    assert "<marimo-code hidden=" in response.text


# Read session with include_code=True allows code to be included
@with_read_session(SESSION_ID, include_code=True)
def test_export_html_with_code_in_read_with_include_code(
    client: TestClient,
) -> None:
    """Test that HTML export includes code in run mode when include_code=True."""
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    body = response.text
    # Code should be included when include_code=True
    assert '<marimo-code hidden=""></marimo-code>' not in body
    assert CODE in body


# Read session without include_code forces empty code
@with_read_session(SESSION_ID, include_code=False)
def test_export_html_no_code_in_read(client: TestClient) -> None:
    """Test that HTML export excludes code in run mode when include_code=False."""
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    # Even if request asks for code, it should be denied
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    body = response.text
    assert '<marimo-code hidden=""></marimo-code>' in body
    assert CODE not in body

    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = "test.py"
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": False,
        },
    )
    body = response.text
    assert '<marimo-code hidden=""></marimo-code>' in body
    assert CODE not in body


@with_session(SESSION_ID)
def test_export_script(client: TestClient) -> None:
    response = client.post(
        "/api/export/script",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 200
    assert "__generated_with = " in response.text


@pytest.mark.xfail(reason="flakey", strict=False)
@with_session(SESSION_ID)
def test_export_markdown(client: TestClient) -> None:
    response = client.post(
        "/api/export/markdown",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 200
    assert f"marimo-version: {__version__}" in response.text
    assert "```python {.marimo}" in response.text
    # Check that the Content-Disposition header has the correct .md extension
    assert "Content-Disposition" in response.headers
    # The temp file has .py extension, should be converted to .md
    assert re.match(
        r"filename=.*\.md", response.headers["Content-Disposition"]
    )


@with_read_session(SESSION_ID)
def test_other_exports_dont_work_in_read(client: TestClient) -> None:
    response = client.post(
        "/api/export/markdown",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 401
    response = client.post(
        "/api/export/script",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 401


@with_session(SESSION_ID)
def test_auto_export_html(client: TestClient, temp_marimo_file: str) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    assert temp_marimo_file is not None
    session.app_file_manager.filename = temp_marimo_file
    session.session_view.add_notification(
        CellNotification(
            cell_id=CellId_t("new_cell"),
            output=CellOutput(
                data="hello",
                mimetype="text/plain",
                channel=CellChannel.OUTPUT,
            ),
        )
    )

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    assert response.status_code == 200
    assert response.json() == {"success": True}

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    # Not modified response
    assert response.status_code == 304

    # Assert __marimo__ directory is created
    assert os.path.exists(
        os.path.join(os.path.dirname(temp_marimo_file), "__marimo__")
    )


@with_session(SESSION_ID)
def test_auto_export_html_no_code(
    client: TestClient, temp_marimo_file: str
) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = temp_marimo_file
    session.session_view.add_notification(
        CellNotification(
            cell_id=CellId_t("new_cell"),
            output=None,
            console=[CellOutput.stdout("hello")],
        )
    )

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": False,
        },
    )
    assert response.status_code == 200
    assert response.json() == {"success": True}

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": False,
        },
    )
    # Not modified response
    assert response.status_code == 304

    # Assert __marimo__ file is created
    assert os.path.exists(
        os.path.join(os.path.dirname(temp_marimo_file), "__marimo__")
    )


@with_session(SESSION_ID)
def test_auto_export_html_no_operations(
    client: TestClient, temp_marimo_file: str
) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = temp_marimo_file
    assert session.session_view.is_empty()

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )
    # Not modified response
    assert response.status_code == 304


@with_session(SESSION_ID)
def test_auto_export_markdown(
    client: TestClient, *, temp_marimo_file: str
) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = temp_marimo_file

    response = client.post(
        "/api/export/auto_export/markdown",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 200
    assert response.json() == {"success": True}

    response = client.post(
        "/api/export/auto_export/markdown",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    # Not modified response
    assert response.status_code == 304

    # Assert __marimo__ file is created
    assert os.path.exists(
        os.path.join(os.path.dirname(temp_marimo_file), "__marimo__")
    )


@pytest.mark.skipif(
    not DependencyManager.nbformat.has(), reason="nbformat not installed"
)
@with_session(SESSION_ID)
def test_auto_export_ipynb(
    client: TestClient, *, temp_marimo_file: str
) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = temp_marimo_file

    response = client.post(
        "/api/export/auto_export/ipynb",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert response.status_code == 200
    assert response.json() == {"success": True}

    response = client.post(
        "/api/export/auto_export/ipynb",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    # Not modified response
    assert response.status_code == 304

    # Assert __marimo__ file is created
    assert os.path.exists(
        os.path.join(os.path.dirname(temp_marimo_file), "__marimo__")
    )


@pytest.mark.skipif(
    not DependencyManager.nbformat.has() or is_windows(),
    reason="nbformat not installed or on Windows",
)
@pytest.mark.flaky(reruns=3)
@with_session(SESSION_ID)
def test_auto_export_ipynb_with_new_cell(
    client: TestClient, *, temp_marimo_file: str
) -> None:
    """Test that auto-exporting to ipynb works after creating and running a new cell.

    This test addresses the bug in https://github.com/marimo-team/marimo/issues/3992
    where cell ID inconsistency causes KeyError when auto-exporting as ipynb.
    """
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.app_file_manager.filename = temp_marimo_file

    # First, create and run a cell with constant value 1
    create_cell_response = client.post(
        "/api/kernel/run",
        headers=HEADERS,
        json={
            "cellIds": ["new_cell"],
            "codes": ["3.14"],
        },
    )
    assert create_cell_response.status_code == 200

    time.sleep(0.5)

    # Save the session
    save_response = client.post(
        "/api/kernel/save",
        headers=HEADERS,
        json={
            "cellIds": ["new_cell"],
            "filename": temp_marimo_file,
            "codes": ["3.14"],
            "names": ["_"],
            "configs": [
                {
                    "hideCode": True,
                    "disabled": False,
                }
            ],
        },
    )
    assert save_response.status_code == 200, save_response.text

    # Clean up the marimo directory
    marimo_dir = Path(temp_marimo_file).parent / "__marimo__"
    shutil.rmtree(marimo_dir, ignore_errors=True)

    # Verify the cell output is correct
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session

    # Wait for the cell operation to be created
    timeout = 2
    start = time.time()
    cell_notification = None
    while time.time() - start < timeout:
        if "new_cell" not in session.session_view.cell_notifications:
            time.sleep(0.1)
            continue
        cell_notification = session.session_view.cell_notifications["new_cell"]
        if (
            cell_notification.output is not None
            and cell_notification.output.data
        ):
            break
    assert cell_notification
    assert cell_notification.output is not None
    assert "3.14" in cell_notification.output.data

    # Now attempt to auto-export as ipynb
    export_response = client.post(
        "/api/export/auto_export/ipynb",
        headers=HEADERS,
        json={
            "download": False,
        },
    )
    assert export_response.status_code == 200
    assert export_response.json() == {"success": True}

    # Verify the exported file exists
    assert marimo_dir.exists()

    # Verify the ipynb file exists
    filename = Path(temp_marimo_file).name.replace(".py", ".ipynb")
    ipynb_path = marimo_dir / filename

    # Wait for the ipynb file to be created
    time.sleep(0.2)
    notebook = ipynb_path.read_text()
    assert "<pre class='text-xs'>3.14</pre>" in notebook


@with_session(SESSION_ID)
def test_export_html_with_script_config(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    session.config_manager = session.config_manager.with_overrides(
        {"display": {"code_editor_font_size": 999}}
    )
    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": False,
        },
    )
    body = response.text
    assert '"code_editor_font_size": 999' in body


@with_session(SESSION_ID)
def test_auto_export_html_unnamed_file(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    # Ensure the file is unnamed
    session.app_file_manager.filename = None

    response = client.post(
        "/api/export/auto_export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )

    # Should return 400 Bad Request when file is unnamed
    assert response.status_code == 400
    assert "File must have a name before exporting" in response.text


@with_session(SESSION_ID)
def test_export_html_unnamed_file(client: TestClient) -> None:
    session = get_session_manager(client).get_session(SESSION_ID)
    assert session
    # Ensure the file is unnamed
    session.app_file_manager.filename = None

    response = client.post(
        "/api/export/html",
        headers=HEADERS,
        json={
            "download": False,
            "files": [],
            "includeCode": True,
        },
    )

    # Should return 400 Bad Request when file is unnamed
    assert response.status_code == 400
    assert "File must have a name before exporting" in response.text


@with_session(SESSION_ID)
def test_export_html_download_edge_case_filenames(client: TestClient) -> None:
    """Test that HTML export with download=True works for non-ASCII filenames."""
    for filename in EDGE_CASE_FILENAMES:
        session = get_session_manager(client).get_session(SESSION_ID)
        assert session
        session.app_file_manager.filename = filename
        response = client.post(
            "/api/export/html",
            headers=HEADERS,
            json={
                "download": True,
                "files": [],
                "includeCode": True,
            },
        )
        assert response.status_code == 200, f"Failed for filename: {filename}"
        assert "Content-Disposition" in response.headers
        assert "attachment" in response.headers["Content-Disposition"]


@with_session(SESSION_ID)
def test_export_script_download_edge_case_filenames(
    client: TestClient,
) -> None:
    """Test that script export with download=True works for non-ASCII filenames."""
    for filename in EDGE_CASE_FILENAMES:
        session = get_session_manager(client).get_session(SESSION_ID)
        assert session
        session.app_file_manager.filename = filename
        response = client.post(
            "/api/export/script",
            headers=HEADERS,
            json={
                "download": True,
            },
        )
        assert response.status_code == 200, f"Failed for filename: {filename}"
        assert "Content-Disposition" in response.headers
        assert "attachment" in response.headers["Content-Disposition"]
