/* Copyright 2026 Marimo. All rights reserved. */

import type { Column } from "@tanstack/react-table";
import { render } from "@testing-library/react";
import { I18nProvider } from "react-aria";
import { describe, expect, it, test } from "vitest";
import { TooltipProvider } from "@/components/ui/tooltip";
import { parseContent } from "@/utils/url-parser";
import {
  generateColumns,
  inferFieldTypes,
  LocaleNumber,
  renderCellValue,
} from "../columns";
import { getMimeValues, isMimeValue, MimeCell } from "../mime-cell";
import type { FieldTypesWithExternalType } from "../types";
import { uniformSample } from "../uniformSample";
import { UrlDetector } from "../url-detector";

test("uniformSample", () => {
  const items = ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J"];

  expect(uniformSample(items, 2)).toMatchInlineSnapshot(`
    [
      "A",
      "J",
    ]
  `);
  expect(uniformSample(items, 4)).toMatchInlineSnapshot(`
    [
      "A",
      "C",
      "F",
      "J",
    ]
  `);
  expect(uniformSample(items, 100)).toBe(items);
});

test("UrlDetector renders URLs as hyperlinks", () => {
  const text = "Check this link: https://example.com";
  const { container } = render(<UrlDetector parts={parseContent(text)} />);
  const link = container.querySelector("a");
  expect(link).toBeTruthy();
  expect(link?.href).toBe("https://example.com/");
});

test("inferFieldTypes", () => {
  const data = [
    {
      a: 1,
      b: "foo",
      c: null,
      d: { mime: "text/csv" },
      e: [1, 2, 3],
      f: true,
      g: false,
      h: new Date(),
    },
  ];
  const fieldTypes = inferFieldTypes(data);
  expect(fieldTypes).toMatchInlineSnapshot(`
    [
      [
        "a",
        [
          "number",
          "number",
        ],
      ],
      [
        "b",
        [
          "string",
          "string",
        ],
      ],
      [
        "c",
        [
          "unknown",
          "object",
        ],
      ],
      [
        "d",
        [
          "unknown",
          "object",
        ],
      ],
      [
        "e",
        [
          "unknown",
          "object",
        ],
      ],
      [
        "f",
        [
          "boolean",
          "boolean",
        ],
      ],
      [
        "g",
        [
          "boolean",
          "boolean",
        ],
      ],
      [
        "h",
        [
          "datetime",
          "datetime",
        ],
      ],
    ]
  `);
});

test("inferFieldTypes with nulls", () => {
  const data = [{ a: 1, b: null }];
  const fieldTypes = inferFieldTypes(data);
  expect(fieldTypes).toMatchInlineSnapshot(`
    [
      [
        "a",
        [
          "number",
          "number",
        ],
      ],
      [
        "b",
        [
          "unknown",
          "object",
        ],
      ],
    ]
  `);
});

test("inferFieldTypes with mimetypes", () => {
  const data = [{ a: { mime: "text/csv" }, b: { mime: "image/png" } }];
  const fieldTypes = inferFieldTypes(data);
  expect(fieldTypes).toMatchInlineSnapshot(`
    [
      [
        "a",
        [
          "unknown",
          "object",
        ],
      ],
      [
        "b",
        [
          "unknown",
          "object",
        ],
      ],
    ]
  `);
});

describe("generateColumns", () => {
  const fieldTypes: FieldTypesWithExternalType = [
    ["name", ["string", "text"]],
    ["age", ["number", "integer"]],
  ];

  it("should generate columns with row headers", () => {
    const columns = generateColumns({
      rowHeaders: [["name", ["string", "text"]]],
      selection: null,
      fieldTypes,
    });

    expect(columns).toHaveLength(3);
    expect(columns[0].id).toBe("name");
    expect(columns[0].meta?.rowHeader).toBe(true);
    expect(columns[0].enableSorting).toBe(true);
  });

  it("should generate columns with nameless row headers", () => {
    const columns = generateColumns({
      rowHeaders: [["", ["string", "text"]]],
      selection: null,
      fieldTypes,
    });

    expect(columns).toHaveLength(3);
    expect(columns[0].id).toMatchInlineSnapshot(`"__m_column__0"`);
    expect(columns[0].meta?.rowHeader).toBe(true);
    expect(columns[0].enableSorting).toBe(false);
  });

  it("should include selection column for multi selection", () => {
    const columns = generateColumns({
      rowHeaders: [],
      selection: "multi",
      fieldTypes,
    });

    expect(columns[0].id).toBe("__select__");
    expect(columns[0].enableSorting).toBe(false);
  });

  it("should generate columns with correct meta data", () => {
    const columns = generateColumns({
      rowHeaders: [],
      selection: null,
      fieldTypes,
    });

    expect(columns.length).toBe(2);
    expect(columns[0].meta?.dataType).toBe("string");
    expect(columns[1].meta?.dataType).toBe("number");
  });

  it("should handle text justification and wrapping", () => {
    const columns = generateColumns({
      rowHeaders: [],
      selection: null,
      fieldTypes,
      textJustifyColumns: { name: "center" },
      wrappedColumns: ["age"],
    });

    // Assuming getCellStyleClass is a function that returns a class name
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const cell = (columns[0].cell as any)({
      column: {
        columnDef: columns[0],
      },
      renderValue: () => "John",
      getValue: () => "John",
    });
    expect(cell?.props.className).toContain("center");
  });

  it("should not include index column if it exists", () => {
    const columns = generateColumns({
      rowHeaders: [],
      selection: null,
      fieldTypes: [...fieldTypes, ["_marimo_row_id", ["string", "text"]]],
    });

    expect(columns).toHaveLength(2);
    expect(columns[0].id).toBe("name");
    expect(columns[1].id).toBe("age");
  });

  it("should render header with tooltip when headerTooltip is provided", () => {
    const columns = generateColumns({
      rowHeaders: [],
      selection: null,
      fieldTypes,
      headerTooltip: { name: "Custom Name Tooltip" },
    });

    // Get the header function for the first column
    const headerFunction = columns[0].header;
    expect(headerFunction).toBeTypeOf("function");

    const mockColumn = {
      id: "name",
      getCanSort: () => false,
      getCanFilter: () => false,
      columnDef: {
        meta: {
          dtype: "string",
          dataType: "string",
        },
      },
    };

    const { container } = render(
      <TooltipProvider>
        {/* @ts-expect-error: mock column and header function */}
        {headerFunction({ column: mockColumn })}
      </TooltipProvider>,
    );

    expect(container.textContent).toContain("name");
    // The tooltip functionality is tested by verifying that the header renders correctly
    // when headerTooltip is provided.
    expect(container.firstChild).toBeTruthy();
  });
});

describe("MimeCell", () => {
  it("renders with correct mime data", () => {
    const value = { mimetype: "text/plain", data: "Hello World" };
    const { container } = render(<MimeCell value={value} />);
    expect(container.textContent).toContain("Hello World");
  });
});

describe("isMimeValue", () => {
  it("should return true for valid MimeValue objects", () => {
    const value = { mimetype: "text/plain", data: "test data" };
    expect(isMimeValue(value)).toBe(true);
  });

  it("should return false for null", () => {
    expect(isMimeValue(null)).toBe(false);
  });

  it("should return false for primitive values", () => {
    expect(isMimeValue("string")).toBe(false);
    expect(isMimeValue(123)).toBe(false);
    expect(isMimeValue(true)).toBe(false);
  });

  it("should return false for objects missing required properties", () => {
    expect(isMimeValue({})).toBe(false);
    expect(isMimeValue({ mimetype: "text/plain" })).toBe(false);
    expect(isMimeValue({ data: "test data" })).toBe(false);
  });
});

describe("getMimeValues", () => {
  it("should return array with single MimeValue when input is a MimeValue", () => {
    const value = { mimetype: "text/plain", data: "test data" };
    expect(getMimeValues(value)).toEqual([value]);
  });

  it("should return array with MimeValue when input has serialized_mime_bundle", () => {
    const mimeValue = { mimetype: "text/plain", data: "test data" };
    const value = { serialized_mime_bundle: mimeValue };
    expect(getMimeValues(value)).toEqual([mimeValue]);
  });

  it("should return array with MimeValue when input has _serialized_mime_bundle", () => {
    const mimeValue = { mimetype: "text/plain", data: "test data" };
    const value = { _serialized_mime_bundle: mimeValue };
    expect(getMimeValues(value)).toEqual([mimeValue]);
  });

  it("should return array of MimeValues when input is an array of MimeValues", () => {
    const values = [
      { mimetype: "text/plain", data: "test data 1" },
      { mimetype: "text/html", data: "<p>test data 2</p>" },
    ];
    expect(getMimeValues(values)).toEqual(values);
  });

  it("should return undefined for null input", () => {
    expect(getMimeValues(null)).toBeUndefined();
  });

  it("should return undefined for primitive values", () => {
    expect(getMimeValues("string")).toBeUndefined();
    expect(getMimeValues(123)).toBeUndefined();
    expect(getMimeValues(true)).toBeUndefined();
  });

  it("should return undefined for objects that don't match any pattern", () => {
    expect(getMimeValues({})).toBeUndefined();
    expect(getMimeValues({ random: "property" })).toBeUndefined();
  });

  it("should return undefined for invalid serialized_mime_bundle", () => {
    expect(
      getMimeValues({ serialized_mime_bundle: "not a mime value" }),
    ).toBeUndefined();
  });

  it("should return undefined for invalid _serialized_mime_bundle", () => {
    expect(
      getMimeValues({ _serialized_mime_bundle: "not a mime value" }),
    ).toBeUndefined();
  });

  it("should return undefined for array with non-MimeValue items", () => {
    const values = [
      { mimetype: "text/plain", data: "test data" },
      "not a mime value",
    ];
    expect(getMimeValues(values)).toBeUndefined();
  });
});

describe("LocaleNumber", () => {
  it("should format numbers correctly for en-US locale", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={1_234_567.89} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"1,234,567.89"`);
  });

  it("should format numbers correctly for de-DE locale", () => {
    const { container } = render(
      <I18nProvider locale="de-DE">
        <LocaleNumber value={1_234_567.89} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"1.234.567,89"`);
  });

  it("should format integers correctly", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={1000} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"1,000"`);
  });

  it("should format zero correctly", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={0} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"0"`);
  });

  it("should format negative numbers correctly", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={-1234.56} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"-1,234.56"`);
  });

  it("should format small decimal numbers correctly", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={0.123_456_789} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"0.123456789"`);
  });

  it("should format large numbers correctly", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={999_999_999.99} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"999,999,999.99"`);
  });

  it("should format numbers correctly for fr-FR locale", () => {
    const { container } = render(
      <I18nProvider locale="fr-FR">
        <LocaleNumber value={1_234_567.89} />
      </I18nProvider>,
    );
    // eslint-disable-next-line no-irregular-whitespace
    expect(container.textContent).toMatchInlineSnapshot(`"1 234 567,89"`);
  });

  it("should format numbers correctly for ja-JP locale", () => {
    const { container } = render(
      <I18nProvider locale="ja-JP">
        <LocaleNumber value={1_234_567.89} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"1,234,567.89"`);
  });

  it("should respect maximumFractionDigits based on locale", () => {
    // Test with a number that has many decimal places
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={1.123_456_789_012_345_7} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"1.1234567890123457"`);
  });

  it("should handle very large numbers", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={123_456_789_012_345.67} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(
      `"123,456,789,012,345.67"`,
    );
  });

  it("should handle Infinity", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={Infinity} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"∞"`);
  });

  it("should handle negative Infinity", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={-Infinity} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"-∞"`);
  });

  it("should handle NaN", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={Number.NaN} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"NaN"`);
  });

  it("should handle numbers with scientific notation", () => {
    const { container } = render(
      <I18nProvider locale="en-US">
        <LocaleNumber value={1e10} />
      </I18nProvider>,
    );
    expect(container.textContent).toMatchInlineSnapshot(`"10,000,000,000"`);
  });
});

describe("renderCellValue with datetime values", () => {
  const createMockColumn = () =>
    ({
      id: "created",
      columnDef: {
        meta: {
          dataType: "datetime" as const,
          dtype: "datetime64[ns]",
        },
      },
      getColumnFormatting: () => undefined,
      getColumnWrapping: () => undefined,
      applyColumnFormatting: (value: unknown) => value,
    }) as unknown as Column<unknown>;

  it("should handle null, empty string, and 'null' string datetime values without throwing RangeError", () => {
    const mockColumn = createMockColumn();

    // Test null, empty string, and "null" string (as they might come from SQL)
    const nullishValues = [null, "", "null"];

    nullishValues.forEach((value) => {
      const result = renderCellValue({
        column: mockColumn,
        renderValue: () => value,
        getValue: () => value,
        selectCell: undefined,
        cellStyles: "",
      });

      expect(result).toBeDefined();
      // Should not throw RangeError when rendering
      expect(() => {
        render(
          <I18nProvider locale="en-US">
            <TooltipProvider>{result}</TooltipProvider>
          </I18nProvider>,
        );
      }).not.toThrow();
    });
  });

  it("should handle invalid date strings without throwing RangeError", () => {
    const mockColumn = createMockColumn();

    const invalidDates = [
      "invalid-date",
      "2024-13-45", // Invalid month/day
      "not-a-date",
      "2024-06-14 12:34:20.665332",
    ];

    invalidDates.forEach((invalidDate) => {
      const result = renderCellValue({
        column: mockColumn,
        renderValue: () => invalidDate,
        getValue: () => invalidDate,
        selectCell: undefined,
        cellStyles: "",
      });
      expect(result).toBeDefined();
      // Should not throw RangeError when rendering
      expect(() => {
        render(
          <I18nProvider locale="en-US">
            <TooltipProvider>{result}</TooltipProvider>
          </I18nProvider>,
        );
      }).not.toThrow();
    });
  });

  it("should still render valid datetime strings correctly", () => {
    const mockColumn = createMockColumn();

    const validDates = [
      "2024-06-14T12:34:20Z",
      "2024-06-14 12:34:20",
      "2024-06-14",
    ];

    validDates.forEach((validDate) => {
      const result = renderCellValue({
        column: mockColumn,
        renderValue: () => validDate,
        getValue: () => validDate,
        selectCell: undefined,
        cellStyles: "",
      });
      expect(result).toBeDefined();
      // Should render as a date component, not as plain string
      expect(result).not.toBeNull();
      // Should not throw when rendering
      expect(() => {
        render(
          <I18nProvider locale="en-US">
            <TooltipProvider>{result}</TooltipProvider>
          </I18nProvider>,
        );
      }).not.toThrow();
    });
  });

  it("should handle invalid Date instances without throwing RangeError", () => {
    const mockColumn = createMockColumn();

    const invalidDate = new Date("invalid");

    const result = renderCellValue({
      column: mockColumn,
      renderValue: () => invalidDate,
      getValue: () => invalidDate,
      selectCell: undefined,
      cellStyles: "",
    });
    expect(result).toBeDefined();
    // Should not throw RangeError when rendering
    expect(() => {
      render(
        <I18nProvider locale="en-US">
          <TooltipProvider>{result}</TooltipProvider>
        </I18nProvider>,
      );
    }).not.toThrow();
  });

  it("should handle mixed valid and null datetime values in a column", () => {
    const mockColumn = createMockColumn();

    const values = [
      "2024-06-14T12:34:20Z", // Valid
      null,
      "2024-06-15T12:34:20Z", // Valid
      "",
      "2024-06-16T12:34:20Z", // Valid
    ];

    values.forEach((value) => {
      const result = renderCellValue({
        column: mockColumn,
        renderValue: () => value,
        getValue: () => value,
        selectCell: undefined,
        cellStyles: "",
      });
      expect(result).toBeDefined();
      // Should not throw RangeError when rendering
      expect(() => {
        render(
          <I18nProvider locale="en-US">
            <TooltipProvider>{result}</TooltipProvider>
          </I18nProvider>,
        );
      }).not.toThrow();
    });
  });
});
