/* Copyright 2026 Marimo. All rights reserved. */

import { PlayIcon } from "lucide-react";
import React from "react";
import { MinimalHotkeys } from "@/components/shortcuts/renderShortcut";
import { Button } from "@/components/ui/button";
import { Tooltip } from "@/components/ui/tooltip";
import { isPlatformMac } from "@/core/hotkeys/shortcuts";

/**
 * Common keyboard shortcut handlers for AI completions
 */
export const createAiCompletionOnKeydown = (opts: {
  handleAcceptCompletion: () => void;
  handleDeclineCompletion: () => void;
  isLoading: boolean;
  hasCompletion: boolean;
}) => {
  const {
    handleAcceptCompletion,
    handleDeclineCompletion,
    isLoading,
    hasCompletion,
  } = opts;

  return (e: React.KeyboardEvent<HTMLDivElement>) => {
    const metaKey = isPlatformMac() ? e.metaKey : e.ctrlKey;

    // Mod+Enter should accept the completion, if there is one
    if (metaKey && e.key === "Enter" && !isLoading && hasCompletion) {
      handleAcceptCompletion();
    }

    // Mod+Shift+Delete should decline the completion
    const deleteKey = e.key === "Delete" || e.key === "Backspace";
    if (deleteKey && metaKey && e.shiftKey) {
      handleDeclineCompletion();
      e.preventDefault();
      e.stopPropagation();
    }
  };
};

export const CompletionActionsCellFooter: React.FC<{
  isLoading: boolean;
  onAccept: () => void;
  onDecline: () => void;
  size?: "xs" | "sm";
  multipleCompletions?: boolean;
  runCell?: () => void;
}> = ({ isLoading, onAccept, onDecline, runCell }) => {
  return (
    <>
      <AcceptCompletionButton
        isLoading={isLoading}
        onAccept={onAccept}
        size="xs"
        runCell={runCell}
      />
      <RejectCompletionButton onDecline={onDecline} size="xs" />
    </>
  );
};

export const AcceptCompletionButton: React.FC<{
  isLoading: boolean;
  onAccept: () => void;
  multipleCompletions?: boolean;
  size?: "xs" | "sm";
  buttonStyles?: string;
  playButtonStyles?: string;
  borderless?: boolean;
  acceptShortcut?: string;
  runCell?: () => void;
}> = ({
  isLoading,
  onAccept,
  multipleCompletions = false,
  size = "sm",
  buttonStyles,
  acceptShortcut,
  runCell,
  playButtonStyles,
  borderless = false,
}) => {
  const handleAcceptAndRun = () => {
    onAccept();
    if (runCell) {
      runCell();
    }
  };

  const text = multipleCompletions ? "Accept all" : "Accept";

  const baseClasses = `h-6 text-(--grass-11) bg-(--grass-3)/60
    hover:bg-(--grass-3) dark:bg-(--grass-4)/80 dark:hover:bg-(--grass-3) font-semibold
    active:bg-(--grass-5) dark:active:bg-(--grass-4)
    border-(--green-6) border hover:shadow-xs`;

  if (runCell) {
    return (
      <div className="flex">
        <Button
          variant="text"
          size={size}
          disabled={isLoading}
          onClick={onAccept}
          className={`${baseClasses} rounded-r-none ${borderless && "border-none rounded-md rounded-r-none"} ${buttonStyles}`}
        >
          {text}
          {acceptShortcut && (
            <MinimalHotkeys className="ml-1 inline" shortcut={acceptShortcut} />
          )}
        </Button>
        <Tooltip
          content={
            multipleCompletions
              ? "Accept and run all cells"
              : "Accept and run cell"
          }
        >
          <Button
            variant="text"
            size={size}
            disabled={isLoading}
            onClick={handleAcceptAndRun}
            className={`${baseClasses} rounded-l-none px-1.5 ${borderless && "border-0 border-l-1"} ${playButtonStyles}`}
          >
            <PlayIcon className="h-2.5 w-2.5" />
          </Button>
        </Tooltip>
      </div>
    );
  }

  return (
    <Button
      variant="text"
      size={size}
      disabled={isLoading}
      onClick={onAccept}
      className={`${baseClasses} rounded px-3 ${buttonStyles}`}
    >
      {text}
      {acceptShortcut && (
        <MinimalHotkeys className="ml-1 inline" shortcut={acceptShortcut} />
      )}
    </Button>
  );
};

export const RejectCompletionButton: React.FC<{
  onDecline: () => void;
  multipleCompletions?: boolean;
  size?: "xs" | "sm";
  className?: string;
  borderless?: boolean;
  declineShortcut?: string;
}> = ({
  onDecline,
  multipleCompletions = false,
  size = "sm",
  className,
  declineShortcut,
  borderless = false,
}) => {
  return (
    <Button
      variant="text"
      size={size}
      onClick={onDecline}
      className={`h-6 text-(--red-10) bg-(--red-3)/60 hover:bg-(--red-3)
    dark:bg-(--red-4)/80 dark:hover:bg-(--red-3) rounded px-3 font-semibold
    active:bg-(--red-5) dark:active:bg-(--red-4)
    border-(--red-6) border hover:shadow-xs ${borderless && "border-none rounded-md"} ${className}`}
    >
      Reject{multipleCompletions && " all"}
      {declineShortcut && (
        <MinimalHotkeys className="ml-1 inline" shortcut={declineShortcut} />
      )}
    </Button>
  );
};
