/* Copyright 2026 Marimo. All rights reserved. */

import { cva, type VariantProps } from "class-variance-authority";
import {
  FieldError as AriaFieldError,
  type FieldErrorProps as AriaFieldErrorProps,
  Group as AriaGroup,
  type GroupProps as AriaGroupProps,
  Label as AriaLabel,
  type LabelProps as AriaLabelProps,
  Text as AriaText,
  type TextProps as AriaTextProps,
  composeRenderProps,
} from "react-aria-components";
import { cn } from "@/utils/cn";
import { selectStyles } from "./native-select";

const labelVariants = cva([
  "text-sm font-medium leading-none",
  /* Disabled */
  "data-disabled:cursor-not-allowed data-disabled:opacity-70",
  /* Invalid */
  "group-data-invalid:text-destructive",
]);

const Label = ({ className, ...props }: AriaLabelProps) => (
  <AriaLabel className={cn(labelVariants(), className)} {...props} />
);

const FormDescription = ({ className, ...props }: AriaTextProps) => {
  return (
    <AriaText
      className={cn("text-sm text-muted-foreground", className)}
      {...props}
      slot="description"
    />
  );
};

const FieldError = ({ className, ...props }: AriaFieldErrorProps) => {
  return (
    <AriaFieldError
      className={cn("text-sm font-medium text-destructive", className)}
      {...props}
    />
  );
};

const fieldGroupVariants = cva("", {
  variants: {
    variant: {
      default: [
        "relative flex w-full items-center overflow-hidden rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background",
        selectStyles(),
        /* Focus Within */
        "data-focus-within:outline-hidden data-focus-within:ring-2 data-focus-within:ring-ring data-focus-within:ring-offset-2",
        /* Disabled */
        "data-disabled:opacity-50",
      ],
      ghost: "",
    },
  },
  defaultVariants: {
    variant: "default",
  },
});

interface GroupProps
  extends AriaGroupProps,
    VariantProps<typeof fieldGroupVariants> {}

const FieldGroup = ({ className, variant, ...props }: GroupProps) => {
  return (
    <AriaGroup
      className={composeRenderProps(className, (className) =>
        cn(fieldGroupVariants({ variant }), className),
      )}
      {...props}
    />
  );
};

export {
  Label,
  labelVariants,
  FieldGroup,
  fieldGroupVariants,
  FieldError,
  FormDescription,
};
