# Copyright 2026 Marimo. All rights reserved.
from __future__ import annotations

from datetime import datetime, timedelta
from typing import Any

import pytest

pytest.importorskip("plotly.express")
pytest.importorskip("plotly.graph_objects")

import plotly.express as px
import plotly.graph_objects as go

from marimo._plugins.ui._impl.plotly import (
    _extract_bars_fallback,
    _extract_bars_numpy,
    _extract_heatmap_cells_fallback,
    _extract_heatmap_cells_numpy,
    _extract_scatter_points_fallback,
    _extract_scatter_points_numpy,
    plotly,
)

# ============================================================================
# General / Basic Tests
# ============================================================================


def test_basic_scatter_plot() -> None:
    """Test creating a basic scatter plot."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6], mode="markers"))
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []
    assert plot.ranges == {}
    assert plot.points == []
    assert plot.indices == []


def test_plotly_express_scatter() -> None:
    """Test creating a plot with plotly express."""
    import pandas as pd

    df = pd.DataFrame(
        {"x": [1, 2, 3], "y": [4, 5, 6], "color": ["A", "B", "A"]}
    )
    fig = px.scatter(df, x="x", y="y", color="color")
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []


def test_plotly_with_config() -> None:
    """Test creating a plot with custom configuration."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    config = {"staticPlot": True, "displayModeBar": False}
    plot = plotly(fig, config=config)

    assert plot is not None
    assert plot._component_args["config"] == config


def test_plotly_with_label() -> None:
    """Test creating a plot with a label."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig, label="My Plot")

    assert plot is not None


def test_plotly_with_on_change() -> None:
    """Test creating a plot with on_change callback."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    callback_called = []

    def on_change(value: Any) -> None:
        callback_called.append(value)

    plot = plotly(fig, on_change=on_change)
    assert plot is not None


def test_initial_selection() -> None:
    """Test that initial selection is properly set."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3, 4], y=[1, 2, 3, 4]))

    # Add a selection to the figure
    fig.add_selection(x0=1, x1=3, y0=1, y1=3, xref="x", yref="y")

    # Update layout to include axis titles
    fig.update_xaxes(title_text="X Axis")
    fig.update_yaxes(title_text="Y Axis")

    plot = plotly(fig)

    # Check that initial value contains the selection
    initial_value = plot._args.initial_value
    assert "range" in initial_value
    assert "x" in initial_value["range"]
    assert "y" in initial_value["range"]
    assert initial_value["range"]["x"] == [1, 3]
    assert initial_value["range"]["y"] == [1, 3]

    # Check that points within the selection are included
    assert "points" in initial_value
    assert "indices" in initial_value
    # Points at (1,1), (2,2), and (3,3) should be selected (using <= comparisons)
    assert len(initial_value["indices"]) == 3
    assert initial_value["indices"] == [0, 1, 2]


def test_selection_with_axis_titles() -> None:
    """Test that selection properly extracts axis titles."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    fig.update_xaxes(title_text="Time")
    fig.update_yaxes(title_text="Value")
    fig.add_selection(x0=1, x1=2, y0=4, y1=5, xref="x", yref="y")

    plot = plotly(fig)

    # Check that points have the correct axis labels
    initial_value = plot._args.initial_value
    if initial_value["points"]:
        point = initial_value["points"][0]
        assert "Time" in point or "Value" in point


def test_selection_without_axis_titles() -> None:
    """Test selection when axes don't have titles."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    fig.add_selection(x0=1, x1=2, y0=4, y1=5, xref="x", yref="y")

    plot = plotly(fig)

    # Should still work, but points might be empty or have generic labels
    initial_value = plot._args.initial_value
    assert "points" in initial_value


def test_convert_value_with_selection() -> None:
    """Test _convert_value method with selection data."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    selection = {
        "points": [{"x": 1, "y": 4}, {"x": 2, "y": 5}],
        "range": {"x": [1, 2], "y": [4, 5]},
        "indices": [0, 1],
    }

    result = plot._convert_value(selection)

    # _convert_value should return the points
    assert result == selection["points"]
    assert plot.ranges == {"x": [1, 2], "y": [4, 5]}
    assert plot.indices == [0, 1]


def test_convert_value_empty_selection() -> None:
    """Test _convert_value with empty selection."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    result = plot._convert_value({})

    assert result == []
    assert plot.ranges == {}
    assert plot.points == []
    assert plot.indices == []


def test_ranges_property() -> None:
    """Test the ranges property."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    # Initially empty
    assert plot.ranges == {}

    # Set selection data
    plot._convert_value({"range": {"x": [1, 2], "y": [4, 5]}})
    assert plot.ranges == {"x": [1, 2], "y": [4, 5]}


def test_points_property() -> None:
    """Test the points property."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    # Initially empty
    assert plot.points == []

    # Set selection data
    plot._convert_value({"points": [{"x": 1, "y": 4}]})
    assert plot.points == [{"x": 1, "y": 4}]


def test_indices_property() -> None:
    """Test the indices property."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    # Initially empty
    assert plot.indices == []

    # Set selection data
    plot._convert_value({"indices": [0, 2]})
    assert plot.indices == [0, 2]


def test_treemap() -> None:
    """Test that treemaps can be created (supported chart type)."""
    fig = go.Figure(
        go.Treemap(
            labels=["A", "B", "C"],
            parents=["", "A", "A"],
            values=[10, 5, 5],
        )
    )
    plot = plotly(fig)

    assert plot is not None


def test_sunburst() -> None:
    """Test that sunburst charts can be created (supported chart type)."""
    fig = go.Figure(
        go.Sunburst(
            labels=["A", "B", "C"],
            parents=["", "A", "A"],
            values=[10, 5, 5],
        )
    )
    plot = plotly(fig)

    assert plot is not None


def test_multiple_traces() -> None:
    """Test plot with multiple traces."""
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=[1, 2, 3], y=[4, 5, 6], name="Trace 1"))
    fig.add_trace(go.Scatter(x=[1, 2, 3], y=[6, 5, 4], name="Trace 2"))

    plot = plotly(fig)
    assert plot is not None


def test_selection_across_multiple_traces() -> None:
    """Test that selection works across multiple traces."""
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=[1, 2], y=[1, 2], name="Trace 1"))
    fig.add_trace(go.Scatter(x=[2, 3], y=[2, 3], name="Trace 2"))
    fig.update_xaxes(title_text="X")
    fig.update_yaxes(title_text="Y")
    fig.add_selection(x0=1.5, x1=2.5, y0=1.5, y1=2.5, xref="x", yref="y")

    plot = plotly(fig)

    # Should select points from both traces
    initial_value = plot._args.initial_value
    assert len(initial_value["indices"]) >= 1


def test_selection_with_no_data() -> None:
    """Test selection on a plot with no data."""
    fig = go.Figure()
    fig.add_selection(x0=1, x1=2, y0=1, y1=2, xref="x", yref="y")

    plot = plotly(fig)

    # Should not error, but should have empty selection
    initial_value = plot._args.initial_value
    assert initial_value["points"] == []
    assert initial_value["indices"] == []


def test_selection_partial_attributes() -> None:
    """Test that selection without all required attributes is ignored."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))

    plot = plotly(fig)
    assert plot is not None


def test_figure_serialization() -> None:
    """Test that the figure is properly serialized to JSON."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    # Check that figure is in args as a dictionary
    assert "figure" in plot._component_args
    assert isinstance(plot._component_args["figure"], dict)
    assert "data" in plot._component_args["figure"]


def test_default_config_from_renderer() -> None:
    """Test that default config is pulled from renderer when not provided."""
    import plotly.io as pio

    # Save original renderer
    original_renderer = pio.renderers.default

    try:
        # Set a renderer with custom config
        pio.renderers.default = "browser"

        fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
        plot = plotly(fig)

        # Should have some config (exact config depends on renderer)
        assert "config" in plot._component_args

    finally:
        # Restore original renderer
        pio.renderers.default = original_renderer


def test_explicit_config_overrides_renderer() -> None:
    """Test that explicit config takes precedence over renderer config."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    custom_config = {"displaylogo": False}
    plot = plotly(fig, config=custom_config)

    assert plot._component_args["config"] == custom_config


def test_value_returns_points() -> None:
    """Test that .value returns the points list."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[4, 5, 6]))
    plot = plotly(fig)

    selection = {
        "points": [{"x": 1, "y": 4}],
        "range": {"x": [1, 2], "y": [4, 5]},
        "indices": [0],
    }

    # _convert_value returns the points
    # With the new line chart support, it now extracts ALL points in the x-range
    result = plot._convert_value(selection)
    assert len(result) == 2  # Points at x=1 and x=2
    assert result[0]["x"] == 1
    assert result[0]["y"] == 4
    assert result[1]["x"] == 2
    assert result[1]["y"] == 5


def test_plotly_name() -> None:
    """Test that the component name is correct."""
    assert plotly.name == "marimo-plotly"


def test_selection_boundary_conditions() -> None:
    """Test selection at exact boundaries."""
    fig = go.Figure(data=go.Scatter(x=[1, 2, 3], y=[1, 2, 3]))
    fig.update_xaxes(title_text="X")
    fig.update_yaxes(title_text="Y")

    # Selection that exactly matches point (2, 2)
    fig.add_selection(x0=2, x1=2, y0=2, y1=2, xref="x", yref="y")

    plot = plotly(fig)

    # Point at exactly (2, 2) should be selected (using <= comparisons)
    initial_value = plot._args.initial_value
    assert len(initial_value["indices"]) == 1
    assert 1 in initial_value["indices"]


# ============================================================================
# Heatmap Tests
# ============================================================================


def test_heatmap_basic() -> None:
    """Test that heatmaps can be created (supported chart type)."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=["A", "B", "C"],
            y=["X", "Y", "Z"],
        )
    )
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []


def test_heatmap_selection_numeric() -> None:
    """Test heatmap selection with numeric axes."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=[10, 20, 30],
            y=[100, 200, 300],
        )
    )
    plot = plotly(fig)

    # Simulate a selection from frontend
    selection = {
        "range": {"x": [15, 25], "y": [150, 250]},
        "points": [],  # Frontend might send empty for heatmaps
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract cells at (20, 200) within the range
    assert len(result) > 0
    # Check that extracted cells have x, y, z values
    for cell in result:
        assert "x" in cell
        assert "y" in cell
        assert "z" in cell


def test_heatmap_selection_categorical() -> None:
    """Test heatmap selection with categorical axes."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=["A", "B", "C"],
            y=["X", "Y", "Z"],
        )
    )
    plot = plotly(fig)

    # For categorical axes, selection uses indices
    # Select cells around index 1 (0.5 to 1.5 covers index 1)
    selection = {
        "range": {"x": [0.5, 1.5], "y": [0.5, 1.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract the cell at index (1, 1) which is ("B", "Y")
    assert len(result) > 0
    assert any(cell["x"] == "B" and cell["y"] == "Y" for cell in result)


def test_heatmap_selection_mixed_axes() -> None:
    """Test heatmap with one numeric and one categorical axis."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6]],
            x=[10, 20, 30],  # Numeric
            y=["Row1", "Row2"],  # Categorical
        )
    )
    plot = plotly(fig)

    selection = {
        "range": {"x": [15, 25], "y": [-0.5, 0.5]},  # Select first row
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract cell at (20, "Row1")
    assert len(result) > 0
    assert any(cell["x"] == 20 and cell["y"] == "Row1" for cell in result)


def test_heatmap_selection_all_cells() -> None:
    """Test selecting all cells in a small heatmap."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2], [3, 4]],
            x=["A", "B"],
            y=["X", "Y"],
        )
    )
    plot = plotly(fig)

    # Select entire heatmap (categorical uses indices -0.5 to n-0.5)
    selection = {
        "range": {"x": [-0.5, 1.5], "y": [-0.5, 1.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should have all 4 cells
    assert len(result) == 4
    # Check we have all combinations
    expected_combinations = {("A", "X"), ("A", "Y"), ("B", "X"), ("B", "Y")}
    actual_combinations = {(cell["x"], cell["y"]) for cell in result}
    assert actual_combinations == expected_combinations


def test_heatmap_selection_no_cells() -> None:
    """Test heatmap selection with range that includes no cells."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3]],
            x=[10, 20, 30],
            y=[100],
        )
    )
    plot = plotly(fig)

    # Select a range with no cells
    selection = {
        "range": {"x": [50, 60], "y": [200, 300]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should be empty
    assert result == []


def test_heatmap_selection_invalid_range_type() -> None:
    """Test heatmap with selection where range is not a dict."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2], [3, 4]],
            x=["A", "B"],
            y=["X", "Y"],
        )
    )
    plot = plotly(fig)

    # Selection with invalid range type (should be handled by isinstance check)
    selection = {"range": "invalid", "points": [], "indices": []}

    result = plot._convert_value(selection)

    # Should handle gracefully without crashing
    assert result == []


def test_heatmap_selection_missing_x_or_y() -> None:
    """Test heatmap selection with incomplete range data."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2], [3, 4]],
            x=["A", "B"],
            y=["X", "Y"],
        )
    )
    plot = plotly(fig)

    # Selection with only x range
    selection = {"range": {"x": [0, 1]}, "points": [], "indices": []}

    result = plot._convert_value(selection)

    # Should return empty list (checked in _extract_heatmap_cells_from_range)
    assert result == []


def test_heatmap_curve_number() -> None:
    """Test that heatmap cells include curveNumber for multi-trace plots."""
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=[1, 2], y=[1, 2]))  # trace 0
    fig.add_trace(go.Heatmap(z=[[1, 2]], x=["A", "B"], y=["X"]))  # trace 1

    plot = plotly(fig)

    selection = {
        "range": {"x": [-0.5, 1.5], "y": [-0.5, 0.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should include both scatter points (curveNumber=0) and heatmap cells (curveNumber=1)
    assert len(result) > 0

    # Separate scatter and heatmap points
    scatter_points = [p for p in result if p.get("curveNumber") == 0]
    heatmap_cells = [p for p in result if p.get("curveNumber") == 1]

    # Should have points from both traces
    assert len(scatter_points) > 0, "Should have scatter points from trace 0"
    assert len(heatmap_cells) > 0, "Should have heatmap cells from trace 1"

    # Heatmap cells should have z values
    assert all("z" in cell for cell in heatmap_cells)


def test_heatmap_initial_selection() -> None:
    """Test that initial selection works with heatmaps."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=["A", "B", "C"],
            y=["X", "Y", "Z"],
        )
    )

    # Add an initial selection
    fig.add_selection(x0=0.5, x1=1.5, y0=0.5, y1=1.5, xref="x", yref="y")

    plot = plotly(fig)

    # Check that initial value contains the selection
    initial_value = plot._args.initial_value
    assert "range" in initial_value
    assert initial_value["range"]["x"] == [0.5, 1.5]
    assert initial_value["range"]["y"] == [0.5, 1.5]

    # For heatmap, should extract cells, not scatter points
    assert "points" in initial_value
    assert len(initial_value["points"]) > 0

    # Should have x, y, z values (heatmap cells)
    for point in initial_value["points"]:
        assert "x" in point
        assert "y" in point
        assert "z" in point

    # Should extract cell at index (1, 1) which is ("B", "Y", 5)
    assert any(
        p["x"] == "B" and p["y"] == "Y" and p["z"] == 5
        for p in initial_value["points"]
    )


def test_heatmap_empty() -> None:
    """Test heatmap with empty data."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[],
            x=[],
            y=[],
        )
    )
    plot = plotly(fig)

    selection = {
        "range": {"x": [-0.5, 0.5], "y": [-0.5, 0.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return empty list for empty heatmap
    assert result == []


def test_heatmap_single_cell() -> None:
    """Test heatmap with a single cell."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[42]],
            x=["A"],
            y=["X"],
        )
    )
    plot = plotly(fig)

    # Selection that covers the single cell (categorical at index 0)
    selection = {
        "range": {"x": [-0.5, 0.5], "y": [-0.5, 0.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract the single cell
    assert len(result) == 1
    assert result[0]["x"] == "A"
    assert result[0]["y"] == "X"
    assert result[0]["z"] == 42


def test_heatmap_single_cell_numeric() -> None:
    """Test heatmap with a single cell and numeric axes."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[99]],
            x=[5],
            y=[10],
        )
    )
    plot = plotly(fig)

    # Selection that covers the single numeric cell
    selection = {
        "range": {"x": [0, 10], "y": [5, 15]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract the single cell
    assert len(result) == 1
    assert result[0]["x"] == 5
    assert result[0]["y"] == 10
    assert result[0]["z"] == 99


def test_heatmap_single_cell_outside_selection() -> None:
    """Test single-cell heatmap where selection doesn't cover the cell."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[42]],
            x=["A"],
            y=["X"],
        )
    )
    plot = plotly(fig)

    # Selection that doesn't cover the cell (categorical at index 0)
    selection = {
        "range": {"x": [1.5, 2.5], "y": [1.5, 2.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return empty list since selection doesn't cover the cell
    assert result == []


def test_heatmap_numpy_and_fallback_produce_same_results() -> None:
    """Test that numpy and fallback implementations produce identical results."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=["A", "B", "C"],
            y=["X", "Y", "Z"],
        )
    )

    x_min, x_max = 0.5, 2.5
    y_min, y_max = 0.5, 2.5

    # Get results from both implementations
    numpy_result = _extract_heatmap_cells_numpy(
        fig, x_min, x_max, y_min, y_max
    )
    fallback_result = _extract_heatmap_cells_fallback(
        fig, x_min, x_max, y_min, y_max
    )

    # Both should return the same number of cells
    assert len(numpy_result) == len(fallback_result)

    # Sort results for comparison (order might differ)
    def sort_key(cell: dict[str, Any]) -> tuple[Any, ...]:
        return (cell["x"], cell["y"], cell["z"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    # Compare each cell
    for np_cell, fb_cell in zip(numpy_sorted, fallback_sorted):
        assert np_cell["x"] == fb_cell["x"]
        assert np_cell["y"] == fb_cell["y"]
        assert np_cell["z"] == fb_cell["z"]
        assert np_cell["curveNumber"] == fb_cell["curveNumber"]


# ============================================================================
# Datetime Axis Tests
# ============================================================================


def test_heatmap_selection_datetime_x_axis() -> None:
    """Test heatmap with datetime x-axis and categorical y-axis.

    This is the exact scenario reported in the bug: datetime x-axis,
    string y-axis, which was causing a numpy UFuncNoLoopError.
    """

    # Create datetime x-axis
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]

    # String categories for y-axis
    categories = ["Category A", "Category B", "Category C"]

    # Generate z values (3 rows x 5 columns)
    z_values = [[i * 5 + j for j in range(5)] for i in range(3)]

    fig = go.Figure(
        data=go.Heatmap(
            z=z_values,
            x=dates,
            y=categories,
        )
    )
    plot = plotly(fig)

    # Simulate a range selection covering dates[1] to dates[3]
    # and categories at indices 0 and 1
    # IMPORTANT: Frontend sends ISO strings via JSON, not datetime objects
    selection = {
        "range": {
            "x": [dates[1].isoformat(), dates[3].isoformat()],
            "y": [-0.5, 1.5],
        },
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract cells for dates[1], dates[2], dates[3]
    # and categories at indices 0, 1
    # That's 3 dates x 2 categories = 6 cells
    assert len(result) == 6

    # Verify we got the expected cells
    for cell in result:
        assert cell["x"] in dates[1:4]  # dates[1], dates[2], dates[3]
        assert cell["y"] in ["Category A", "Category B"]
        assert "z" in cell


def test_heatmap_numpy_and_fallback_datetime_x_axis() -> None:
    """Test that numpy and fallback produce same results for datetime x-axis."""
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]
    categories = ["A", "B", "C"]
    z_values = [[i * 5 + j for j in range(5)] for i in range(3)]

    fig = go.Figure(data=go.Heatmap(z=z_values, x=dates, y=categories))

    # Use ISO strings for x range (as frontend sends), indices for y range
    x_min, x_max = dates[1].isoformat(), dates[3].isoformat()
    y_min, y_max = -0.5, 1.5

    numpy_result = _extract_heatmap_cells_numpy(
        fig, x_min, x_max, y_min, y_max
    )
    fallback_result = _extract_heatmap_cells_fallback(
        fig, x_min, x_max, y_min, y_max
    )

    assert len(numpy_result) == len(fallback_result)

    # Sort for comparison
    def sort_key(cell: dict[str, Any]) -> tuple[Any, ...]:
        return (str(cell["x"]), cell["y"], cell["z"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_cell, fb_cell in zip(numpy_sorted, fallback_sorted):
        assert np_cell["x"] == fb_cell["x"]
        assert np_cell["y"] == fb_cell["y"]
        assert np_cell["z"] == fb_cell["z"]


def test_scatter_selection_datetime_x_axis() -> None:
    """Test scatter/line chart with datetime x-axis."""
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]
    values = [10, 20, 15, 25, 30]

    fig = go.Figure(data=go.Scatter(x=dates, y=values, mode="lines"))
    plot = plotly(fig)

    # Select dates[1] to dates[3] - frontend sends ISO strings
    selection = {
        "range": {
            "x": [dates[1].isoformat(), dates[3].isoformat()],
            "y": [0, 50],
        },
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return 3 points: at dates[1], dates[2], dates[3]
    assert len(result) == 3
    assert result[0]["x"] == dates[1]
    assert result[0]["y"] == 20
    assert result[1]["x"] == dates[2]
    assert result[1]["y"] == 15
    assert result[2]["x"] == dates[3]
    assert result[2]["y"] == 25


def test_scatter_numpy_and_fallback_datetime_x_axis() -> None:
    """Test that numpy and fallback produce same results for datetime scatter."""
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]
    values = [10, 20, 15, 25, 30]

    fig = go.Figure(data=go.Scatter(x=dates, y=values, mode="lines"))

    # Frontend sends ISO strings
    x_min, x_max = dates[1].isoformat(), dates[3].isoformat()

    numpy_result = _extract_scatter_points_numpy(fig, x_min, x_max)
    fallback_result = _extract_scatter_points_fallback(fig, x_min, x_max)

    assert len(numpy_result) == len(fallback_result)

    def sort_key(p: dict[str, Any]) -> tuple[Any, ...]:
        return (str(p["x"]), p["y"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_p, fb_p in zip(numpy_sorted, fallback_sorted):
        assert np_p["x"] == fb_p["x"]
        assert np_p["y"] == fb_p["y"]


def test_bar_selection_datetime_x_axis() -> None:
    """Test bar chart with datetime x-axis."""
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]
    values = [10, 20, 15, 25, 30]

    fig = go.Figure(data=go.Bar(x=dates, y=values))
    plot = plotly(fig)

    # Select dates[1] to dates[3] - frontend sends ISO strings
    selection = {
        "range": {
            "x": [dates[1].isoformat(), dates[3].isoformat()],
            "y": [0, 50],
        },
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return 3 bars: at dates[1], dates[2], dates[3]
    assert len(result) == 3
    assert any(bar["x"] == dates[1] and bar["y"] == 20 for bar in result)
    assert any(bar["x"] == dates[2] and bar["y"] == 15 for bar in result)
    assert any(bar["x"] == dates[3] and bar["y"] == 25 for bar in result)


def test_bar_numpy_and_fallback_datetime_x_axis() -> None:
    """Test that numpy and fallback produce same results for datetime bars."""
    base_date = datetime(2024, 1, 1)
    dates = [base_date + timedelta(days=i) for i in range(5)]
    values = [10, 20, 15, 25, 30]

    fig = go.Figure(data=go.Bar(x=dates, y=values))

    # Frontend sends ISO strings
    x_min, x_max = dates[1].isoformat(), dates[3].isoformat()
    y_min, y_max = 0, 50

    numpy_result = _extract_bars_numpy(fig, x_min, x_max, y_min, y_max)
    fallback_result = _extract_bars_fallback(fig, x_min, x_max, y_min, y_max)

    assert len(numpy_result) == len(fallback_result)

    def sort_key(bar: dict[str, Any]) -> tuple[Any, ...]:
        return (str(bar["x"]), bar["y"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_bar, fb_bar in zip(numpy_sorted, fallback_sorted):
        assert np_bar["x"] == fb_bar["x"]
        assert np_bar["y"] == fb_bar["y"]


# ============================================================================
# Scatter / Line Chart Tests
# ============================================================================


def test_line_chart_basic() -> None:
    """Test that line charts can be created (supported chart type)."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []


def test_line_chart_selection() -> None:
    """Test box selection on pure line chart."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )
    fig.update_xaxes(title_text="X")
    fig.update_yaxes(title_text="Y")

    plot = plotly(fig)

    # Simulate box selection from x=2 to x=4
    selection = {
        "range": {"x": [2, 4], "y": [10, 30]},
        "points": [],  # Empty for pure lines
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return points at x=2, 3, 4
    assert len(result) == 3
    assert result[0]["X"] == 2
    assert result[0]["Y"] == 20
    assert result[1]["X"] == 3
    assert result[1]["Y"] == 15
    assert result[2]["X"] == 4
    assert result[2]["Y"] == 25


def test_line_markers_selection() -> None:
    """Test box selection on line chart with markers."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines+markers",
        )
    )
    fig.update_xaxes(title_text="Time")
    fig.update_yaxes(title_text="Value")

    plot = plotly(fig)

    # Simulate box selection from x=1.5 to x=3.5
    selection = {
        "range": {"x": [1.5, 3.5], "y": [10, 25]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return points at x=2, 3
    assert len(result) == 2
    assert result[0]["Time"] == 2
    assert result[0]["Value"] == 20
    assert result[1]["Time"] == 3
    assert result[1]["Value"] == 15


def test_line_chart_with_axis_titles() -> None:
    """Test line chart selection with custom axis titles."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )
    fig.update_xaxes(title_text="Time")
    fig.update_yaxes(title_text="Value")

    plot = plotly(fig)

    selection = {
        "range": {"x": [2, 3], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return points at x=2, 3
    assert len(result) == 2
    assert result[0]["Time"] == 2
    assert result[0]["Value"] == 20
    assert result[1]["Time"] == 3
    assert result[1]["Value"] == 15


def test_multiple_line_traces_selection() -> None:
    """Test box selection on multiple line traces."""
    fig = go.Figure()
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
            name="Series A",
        )
    )
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[15, 10, 20, 18, 25],
            mode="lines",
            name="Series B",
        )
    )
    fig.update_xaxes(title_text="X")
    fig.update_yaxes(title_text="Y")

    plot = plotly(fig)

    # Simulate box selection from x=2 to x=4
    selection = {
        "range": {"x": [2, 4], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return 6 points total: 3 from each trace
    assert len(result) == 6

    # Check Series A points
    series_a_points = [p for p in result if p.get("name") == "Series A"]
    assert len(series_a_points) == 3
    assert series_a_points[0]["X"] == 2
    assert series_a_points[1]["X"] == 3
    assert series_a_points[2]["X"] == 4

    # Check Series B points
    series_b_points = [p for p in result if p.get("name") == "Series B"]
    assert len(series_b_points) == 3
    assert series_b_points[0]["X"] == 2
    assert series_b_points[1]["X"] == 3
    assert series_b_points[2]["X"] == 4


def test_line_chart_no_axis_titles() -> None:
    """Test line chart selection without axis titles."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [2, 4], "y": [10, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should use default field names 'x' and 'y'
    assert len(result) == 3
    assert result[0]["x"] == 2
    assert result[0]["y"] == 20


def test_line_chart_empty_selection() -> None:
    """Test line chart with range that includes no points."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3],
            y=[10, 20, 30],
            mode="lines",
        )
    )

    plot = plotly(fig)

    # Select range outside data
    selection = {
        "range": {"x": [10, 20], "y": [0, 100]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should be empty
    assert result == []


def test_line_chart_single_point_selection() -> None:
    """Test line chart with range that covers only one point."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )

    plot = plotly(fig)

    # Very narrow range around x=3
    selection = {
        "range": {"x": [2.9, 3.1], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return only point at x=3
    assert len(result) == 1
    assert result[0]["x"] == 3
    assert result[0]["y"] == 15


def test_line_chart_with_curve_number() -> None:
    """Test that line chart points include curveNumber."""
    fig = go.Figure()
    fig.add_trace(
        go.Scatter(x=[1, 2, 3], y=[10, 20, 15], mode="lines", name="A")
    )
    fig.add_trace(
        go.Scatter(x=[1, 2, 3], y=[15, 10, 20], mode="lines", name="B")
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 3], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Check that all points have curveNumber
    assert all("curveNumber" in point for point in result)

    # Check we have points from both traces
    curve_numbers = {point["curveNumber"] for point in result}
    assert curve_numbers == {0, 1}


def test_line_chart_filters_by_x_range_only() -> None:
    """Test that line chart selection filters by x-range, matching Altair behavior."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 50, 15, 60, 30],  # Varying y values
            mode="lines",
        )
    )

    plot = plotly(fig)

    # Select x range 2-4, with narrow y range that wouldn't include all points
    selection = {
        "range": {
            "x": [2, 4],
            "y": [10, 20],
        },  # y range only covers some points
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return ALL points in x-range [2,4], regardless of y
    # This matches Altair behavior
    assert len(result) == 3
    assert result[0]["x"] == 2
    assert result[0]["y"] == 50  # y=50 is outside [10,20] but still included
    assert result[1]["x"] == 3
    assert result[1]["y"] == 15
    assert result[2]["x"] == 4
    assert result[2]["y"] == 60  # y=60 is outside [10,20] but still included


def test_scatter_points_numpy_and_fallback() -> None:
    """Test that numpy and fallback implementations produce identical results."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            mode="lines",
        )
    )

    x_min, x_max = 2, 4

    # Get results from both implementations
    numpy_result = _extract_scatter_points_numpy(fig, x_min, x_max)
    fallback_result = _extract_scatter_points_fallback(fig, x_min, x_max)

    # Both should return the same number of points
    assert len(numpy_result) == len(fallback_result)

    # Sort results for comparison (order might differ)
    def sort_key(point: dict[str, Any]) -> tuple[Any, ...]:
        return (point["x"], point["y"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    # Compare each point
    for np_point, fb_point in zip(numpy_sorted, fallback_sorted):
        assert np_point["x"] == fb_point["x"]
        assert np_point["y"] == fb_point["y"]
        assert np_point["curveNumber"] == fb_point["curveNumber"]


def test_mixed_heatmap_and_line_selection() -> None:
    """Test that heatmap and line selections don't interfere with each other."""
    # Create figure with both heatmap and line
    fig = go.Figure()
    fig.add_trace(
        go.Scatter(x=[1, 2, 3], y=[10, 20, 15], mode="lines", name="Line")
    )
    # Note: Not adding heatmap since they typically don't mix,
    # but test that scatter detection works correctly

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 3], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract line points
    assert len(result) == 3


# ============================================================================
# Area Chart Tests
# ============================================================================


def test_area_chart_basic() -> None:
    """Test that area charts can be created (scatter with fill)."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            fill="tozeroy",
            mode="lines",
        )
    )
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []


def test_area_chart_selection() -> None:
    """Test box selection on area chart returns points in x-range."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            fill="tozeroy",
            mode="lines",
        )
    )
    fig.update_xaxes(title_text="X")
    fig.update_yaxes(title_text="Y")

    plot = plotly(fig)

    selection = {
        "range": {"x": [2, 4], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return points at x=2, 3, 4
    assert len(result) == 3
    assert result[0]["X"] == 2
    assert result[0]["Y"] == 20
    assert result[1]["X"] == 3
    assert result[1]["Y"] == 15
    assert result[2]["X"] == 4
    assert result[2]["Y"] == 25


def test_area_chart_tozeroy() -> None:
    """Test area chart with fill='tozeroy' (fill down to x-axis)."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3],
            y=[10, 20, 15],
            fill="tozeroy",
            mode="lines",
            name="Area",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 3], "y": [0, 25]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return all 3 points
    assert len(result) == 3
    assert all("name" in p and p["name"] == "Area" for p in result)


def test_area_chart_tonexty() -> None:
    """Test stacked area chart with fill='tonexty'."""
    fig = go.Figure()
    # First trace fills to zero
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3],
            y=[10, 15, 12],
            fill="tozeroy",
            mode="lines",
            name="Bottom",
        )
    )
    # Second trace fills to previous trace
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3],
            y=[20, 25, 22],
            fill="tonexty",
            mode="lines",
            name="Top",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 3], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return points from both traces (3 + 3 = 6)
    assert len(result) == 6

    # Check we have points from both traces
    bottom_points = [p for p in result if p.get("name") == "Bottom"]
    top_points = [p for p in result if p.get("name") == "Top"]
    assert len(bottom_points) == 3
    assert len(top_points) == 3


def test_area_chart_stackgroup() -> None:
    """Test stacked area chart using stackgroup property."""
    fig = go.Figure()
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3, 4],
            y=[10, 20, 15, 25],
            stackgroup="one",
            mode="lines",
            name="Series A",
        )
    )
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3, 4],
            y=[5, 10, 8, 12],
            stackgroup="one",
            mode="lines",
            name="Series B",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [2, 3], "y": [0, 50]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return 2 points from each trace (4 total)
    assert len(result) == 4

    series_a = [p for p in result if p.get("name") == "Series A"]
    series_b = [p for p in result if p.get("name") == "Series B"]
    assert len(series_a) == 2
    assert len(series_b) == 2


def test_area_chart_mode_none() -> None:
    """Test area chart with mode='none' (fill only, no line)."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4],
            y=[10, 25, 15, 30],
            fill="tozeroy",
            mode="none",  # No line, just fill
            name="Fill Only",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 4], "y": [0, 35]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should still return all 4 points even with mode='none'
    assert len(result) == 4


def test_area_chart_with_axis_titles() -> None:
    """Test area chart selection with custom axis titles."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[100, 200, 150, 250, 300],
            fill="tozeroy",
            mode="lines",
        )
    )
    fig.update_xaxes(title_text="Time")
    fig.update_yaxes(title_text="Value")

    plot = plotly(fig)

    selection = {
        "range": {"x": [2, 4], "y": [0, 300]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should use custom axis titles as field names
    assert len(result) == 3
    assert result[0]["Time"] == 2
    assert result[0]["Value"] == 200
    assert result[1]["Time"] == 3
    assert result[1]["Value"] == 150
    assert result[2]["Time"] == 4
    assert result[2]["Value"] == 250


def test_area_chart_empty_selection() -> None:
    """Test area chart with selection range containing no points."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3],
            y=[10, 20, 30],
            fill="tozeroy",
            mode="lines",
        )
    )

    plot = plotly(fig)

    # Selection range outside data
    selection = {
        "range": {"x": [10, 20], "y": [0, 100]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return empty list
    assert result == []


def test_area_chart_single_point_selection() -> None:
    """Test area chart with selection covering only one point."""
    fig = go.Figure(
        data=go.Scatter(
            x=[1, 2, 3, 4, 5],
            y=[10, 20, 15, 25, 30],
            fill="tozeroy",
            mode="lines",
        )
    )

    plot = plotly(fig)

    # Very narrow range around x=3
    selection = {
        "range": {"x": [2.9, 3.1], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return only the point at x=3
    assert len(result) == 1
    assert result[0]["x"] == 3
    assert result[0]["y"] == 15


def test_area_chart_curve_number() -> None:
    """Test that area chart points include curveNumber for multi-trace plots."""
    fig = go.Figure()
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3],
            y=[10, 20, 15],
            fill="tozeroy",
            mode="lines",
            name="Area A",
        )
    )
    fig.add_trace(
        go.Scatter(
            x=[1, 2, 3],
            y=[15, 10, 20],
            fill="tozeroy",
            mode="lines",
            name="Area B",
        )
    )

    plot = plotly(fig)

    selection = {
        "range": {"x": [1, 3], "y": [0, 25]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should have 6 points total (3 from each trace)
    assert len(result) == 6

    # Check curveNumbers are present and distinct
    assert all("curveNumber" in p for p in result)
    curve_numbers = {p["curveNumber"] for p in result}
    assert curve_numbers == {0, 1}


# ============================================================================
# Bar Chart Tests
# ============================================================================


def test_bar_chart_basic() -> None:
    """Test that bar charts can be created (supported chart type)."""
    fig = go.Figure(
        data=go.Bar(
            x=["A", "B", "C"],
            y=[10, 20, 15],
        )
    )
    plot = plotly(fig)

    assert plot is not None
    assert plot.value == []


def test_bar_chart_selection_vertical() -> None:
    """Test bar chart selection with vertical bars (categorical x-axis)."""
    fig = go.Figure(
        data=go.Bar(
            x=["A", "B", "C", "D"],
            y=[10, 20, 15, 25],
        )
    )
    plot = plotly(fig)

    # Simulate a selection from frontend (selecting bars B and C)
    selection = {
        "range": {"x": [0.5, 2.5], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract bars at indices 1 and 2 (B and C)
    assert len(result) == 2
    assert any(bar["x"] == "B" and bar["y"] == 20 for bar in result)
    assert any(bar["x"] == "C" and bar["y"] == 15 for bar in result)


def test_bar_chart_selection_horizontal() -> None:
    """Test bar chart selection with horizontal bars."""
    fig = go.Figure(
        data=go.Bar(
            x=[10, 20, 15],
            y=["A", "B", "C"],
            orientation="h",
        )
    )
    plot = plotly(fig)

    # Simulate a selection (selecting bars A and B)
    selection = {
        "range": {"x": [0, 30], "y": [-0.5, 1.5]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract bars at y-indices 0 and 1 (A and B)
    assert len(result) == 2
    assert any(bar["y"] == "A" and bar["x"] == 10 for bar in result)
    assert any(bar["y"] == "B" and bar["x"] == 20 for bar in result)


def test_bar_chart_categorical_axis() -> None:
    """Test bar chart with categorical x-axis (most common case)."""
    fig = go.Figure(
        data=go.Bar(
            x=["Product A", "Product B", "Product C"],
            y=[100, 200, 150],
        )
    )
    plot = plotly(fig)

    # Select middle bar (Product B at index 1)
    selection = {
        "range": {"x": [0.7, 1.3], "y": [0, 250]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    assert len(result) == 1
    assert result[0]["x"] == "Product B"
    assert result[0]["y"] == 200


def test_bar_chart_numeric_axis() -> None:
    """Test bar chart with numeric x-axis (histogram-like)."""
    fig = go.Figure(
        data=go.Bar(
            x=[10, 20, 30, 40],
            y=[5, 12, 8, 15],
        )
    )
    plot = plotly(fig)

    # Select bars at x=20 and x=30
    selection = {
        "range": {"x": [15, 35], "y": [0, 20]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    assert len(result) == 2
    assert any(bar["x"] == 20 and bar["y"] == 12 for bar in result)
    assert any(bar["x"] == 30 and bar["y"] == 8 for bar in result)


def test_bar_chart_stacked() -> None:
    """Test stacked bar chart - should return all segments at selected positions."""
    fig = go.Figure()
    # Two traces for stacked bars
    fig.add_trace(go.Bar(x=["A", "B", "C"], y=[10, 15, 12], name="Series1"))
    fig.add_trace(go.Bar(x=["A", "B", "C"], y=[5, 8, 6], name="Series2"))

    plot = plotly(fig)

    # Select bar at position A (index 0)
    selection = {
        "range": {"x": [-0.5, 0.5], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return both segments at position A (one from each trace)
    assert len(result) == 2
    assert any(bar["x"] == "A" and bar["y"] == 10 for bar in result)
    assert any(bar["x"] == "A" and bar["y"] == 5 for bar in result)
    # Check curveNumber to distinguish traces
    assert result[0]["curveNumber"] != result[1]["curveNumber"]


def test_bar_chart_grouped() -> None:
    """Test grouped bar chart - should return all bars at selected position."""
    fig = go.Figure()
    # Two traces for grouped bars
    fig.add_trace(go.Bar(x=["A", "B", "C"], y=[20, 25, 18], name="Series1"))
    fig.add_trace(go.Bar(x=["A", "B", "C"], y=[15, 22, 16], name="Series2"))

    plot = plotly(fig)

    # Select category B (index 1)
    selection = {
        "range": {"x": [0.5, 1.5], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should return both bars at category B
    assert len(result) == 2
    assert any(bar["x"] == "B" and bar["y"] == 25 for bar in result)
    assert any(bar["x"] == "B" and bar["y"] == 22 for bar in result)


def test_bar_chart_empty_selection() -> None:
    """Test bar chart selection with range that includes no bars."""
    fig = go.Figure(
        data=go.Bar(
            x=["A", "B", "C"],
            y=[10, 20, 15],
        )
    )
    plot = plotly(fig)

    # Select a range with no bars (far to the right)
    selection = {
        "range": {"x": [5, 10], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should be empty
    assert result == []


def test_bar_chart_single_bar() -> None:
    """Test bar chart with a single bar."""
    fig = go.Figure(
        data=go.Bar(
            x=["OnlyBar"],
            y=[42],
        )
    )
    plot = plotly(fig)

    # Select the single bar (categorical at index 0)
    selection = {
        "range": {"x": [-0.5, 0.5], "y": [0, 50]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should extract the single bar
    assert len(result) == 1
    assert result[0]["x"] == "OnlyBar"
    assert result[0]["y"] == 42


def test_bar_chart_initial_selection() -> None:
    """Test that initial selection works with bar charts."""
    fig = go.Figure(
        data=go.Bar(
            x=["A", "B", "C", "D"],
            y=[10, 20, 15, 25],
        )
    )

    # Add an initial selection
    fig.add_selection(x0=0.5, x1=2.5, y0=0, y1=30, xref="x", yref="y")

    plot = plotly(fig)

    # Check that initial value contains the selection
    initial_value = plot._args.initial_value
    assert "range" in initial_value
    assert initial_value["range"]["x"] == [0.5, 2.5]
    assert initial_value["range"]["y"] == [0, 30]

    # For bar charts, should extract bars at indices 1 and 2 (B and C)
    assert "points" in initial_value
    assert len(initial_value["points"]) == 2
    assert any(
        bar["x"] == "B" and bar["y"] == 20 for bar in initial_value["points"]
    )
    assert any(
        bar["x"] == "C" and bar["y"] == 15 for bar in initial_value["points"]
    )


def test_bar_chart_curve_number() -> None:
    """Test that bar chart cells include curveNumber for multi-trace plots."""
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=[1, 2], y=[1, 2]))  # trace 0
    fig.add_trace(go.Bar(x=["A", "B"], y=[10, 20]))  # trace 1

    plot = plotly(fig)

    selection = {
        "range": {"x": [-0.5, 0.5], "y": [0, 30]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Bar should have curveNumber = 1
    bar_items = [r for r in result if "x" in r and r["x"] == "A"]
    assert len(bar_items) > 0
    assert bar_items[0]["curveNumber"] == 1


def test_bar_chart_mixed_with_scatter() -> None:
    """Test figure with both scatter and bar traces."""
    fig = go.Figure()
    fig.add_trace(go.Scatter(x=[0, 1, 2], y=[5, 10, 15], mode="markers"))
    fig.add_trace(go.Bar(x=["A", "B", "C"], y=[10, 20, 15]))

    plot = plotly(fig)

    # Select range that includes scatter points and bars
    selection = {
        "range": {"x": [-0.5, 2.5], "y": [5, 25]},
        "points": [],
        "indices": [],
    }

    result = plot._convert_value(selection)

    # Should include both scatter points and bar(s)
    # The exact count depends on the x-range mapping to categorical bar positions
    assert len(result) >= 2
    # Should have at least one bar
    assert any(r.get("x") in ["A", "B", "C"] for r in result)
    # Should have scatter points
    assert any(isinstance(r.get("x"), int) for r in result)


# Test numpy vs fallback implementations


def test_bar_numpy_and_fallback_match_categorical() -> None:
    """Test that numpy and fallback produce identical results for categorical bars."""
    fig = go.Figure(
        data=go.Bar(
            x=["A", "B", "C", "D"],
            y=[10, 20, 15, 25],
        )
    )

    x_min, x_max = 0.5, 2.5
    y_min, y_max = 0, 30

    # Get results from both implementations
    numpy_result = _extract_bars_numpy(fig, x_min, x_max, y_min, y_max)
    fallback_result = _extract_bars_fallback(fig, x_min, x_max, y_min, y_max)

    # Both should return the same number of bars
    assert len(numpy_result) == len(fallback_result)

    # Sort results for comparison
    def sort_key(bar: dict[str, Any]) -> tuple[Any, ...]:
        return (bar["x"], bar["y"], bar["curveNumber"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    # Compare each bar
    for np_bar, fb_bar in zip(numpy_sorted, fallback_sorted):
        assert np_bar["x"] == fb_bar["x"]
        assert np_bar["y"] == fb_bar["y"]
        assert np_bar["curveNumber"] == fb_bar["curveNumber"]


def test_bar_numpy_and_fallback_match_numeric() -> None:
    """Test that numpy and fallback produce identical results for numeric bars."""
    fig = go.Figure(
        data=go.Bar(
            x=[10, 20, 30, 40],
            y=[5, 12, 8, 15],
        )
    )

    x_min, x_max = 15, 35
    y_min, y_max = 0, 20

    numpy_result = _extract_bars_numpy(fig, x_min, x_max, y_min, y_max)
    fallback_result = _extract_bars_fallback(fig, x_min, x_max, y_min, y_max)

    assert len(numpy_result) == len(fallback_result)

    def sort_key(bar: dict[str, Any]) -> tuple[Any, ...]:
        return (bar["x"], bar["y"], bar["curveNumber"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_bar, fb_bar in zip(numpy_sorted, fallback_sorted):
        assert np_bar["x"] == fb_bar["x"]
        assert np_bar["y"] == fb_bar["y"]
        assert np_bar["curveNumber"] == fb_bar["curveNumber"]


def test_bar_numpy_and_fallback_match_horizontal() -> None:
    """Test that numpy and fallback produce identical results for horizontal bars."""
    fig = go.Figure(
        data=go.Bar(
            x=[10, 20, 15],
            y=["A", "B", "C"],
            orientation="h",
        )
    )

    x_min, x_max = 0, 30
    y_min, y_max = -0.5, 1.5

    numpy_result = _extract_bars_numpy(fig, x_min, x_max, y_min, y_max)
    fallback_result = _extract_bars_fallback(fig, x_min, x_max, y_min, y_max)

    assert len(numpy_result) == len(fallback_result)

    def sort_key(bar: dict[str, Any]) -> tuple[Any, ...]:
        return (str(bar["y"]), bar["x"], bar["curveNumber"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_bar, fb_bar in zip(numpy_sorted, fallback_sorted):
        assert np_bar["x"] == fb_bar["x"]
        assert np_bar["y"] == fb_bar["y"]
        assert np_bar["curveNumber"] == fb_bar["curveNumber"]


def test_heatmap_numpy_and_fallback_numeric_axes() -> None:
    """Test numpy and fallback with numeric axes produce identical results."""
    fig = go.Figure(
        data=go.Heatmap(
            z=[[1, 2, 3], [4, 5, 6], [7, 8, 9]],
            x=[10, 20, 30],
            y=[100, 200, 300],
        )
    )

    x_min, x_max = 15, 25
    y_min, y_max = 150, 250

    numpy_result = _extract_heatmap_cells_numpy(
        fig, x_min, x_max, y_min, y_max
    )
    fallback_result = _extract_heatmap_cells_fallback(
        fig, x_min, x_max, y_min, y_max
    )

    assert len(numpy_result) == len(fallback_result)

    def sort_key(cell: dict[str, Any]) -> tuple[Any, ...]:
        return (cell["x"], cell["y"], cell["z"])

    numpy_sorted = sorted(numpy_result, key=sort_key)
    fallback_sorted = sorted(fallback_result, key=sort_key)

    for np_cell, fb_cell in zip(numpy_sorted, fallback_sorted):
        assert np_cell["x"] == fb_cell["x"]
        assert np_cell["y"] == fb_cell["y"]
        assert np_cell["z"] == fb_cell["z"]
        assert np_cell["curveNumber"] == fb_cell["curveNumber"]
