import fs from 'node:fs/promises';
import path from 'node:path';
import { fileURLToPath } from 'node:url';
import { z } from 'zod';
import type { ProviderConfig } from '../src/llm';
import { EXCLUDED_PROVIDERS, PROVIDERS_WITH_INSTALLED_PACKAGES } from '../src/llm/model/gateways/constants';
import { generateProviderOptionsSection } from './generate-provider-options-docs';

/**
 * Generate a comment indicating the file was auto-generated
 */
function getGeneratedComment(): string {
  return `{/* This file is auto-generated by generate-model-docs.ts - DO NOT EDIT MANUALLY */}`;
}

/**
 * Format provider name for display (handles underscores and capitalization)
 */
function formatProviderName(name: string): string {
  // Special cases
  const specialCases: Record<string, string> = {
    'fireworks-ai': 'Fireworks AI',
    openrouter: 'OpenRouter',
    togetherai: 'Together AI',
    huggingface: 'Hugging Face',
    deepseek: 'DeepSeek',
    openai: 'OpenAI',
    xai: 'xAI',
    'github-copilot': 'GitHub Copilot',
    'github-models': 'GitHub Models',
    deepinfra: 'Deep Infra',
    fastrouter: 'FastRouter',
    baseten: 'Baseten',
    lmstudio: 'LMStudio',
    modelscope: 'ModelScope',
    moonshotai: 'Moonshot AI',
    'moonshotai-cn': 'Moonshot AI (China)',
    zhipuai: 'Zhipu AI',
    opencode: 'OpenCode',
  };

  const lower = name.toLowerCase();
  if (specialCases[lower]) {
    return specialCases[lower];
  }

  // Default: capitalize first letter
  return name.charAt(0).toUpperCase() + name.slice(1);
}

/**
 * Clean API URLs to documentation URLs
 * Removes API paths like /v1 and subdomains like api.
 */
function cleanDocumentationUrl(url: string | undefined): string | undefined {
  if (!url) return undefined;

  try {
    const parsed = new URL(url);

    // If it contains API paths, convert to homepage
    if (parsed.pathname.includes('/v1') || parsed.pathname.includes('/api') || parsed.pathname.includes('/inference')) {
      // Remove subdomain if it's an API subdomain
      if (parsed.hostname.startsWith('api.') || parsed.hostname.startsWith('router.')) {
        const domain = parsed.hostname.replace(/^(api|router)\./, '');
        return `https://${domain}`;
      }

      // Just remove the path
      return `${parsed.protocol}//${parsed.host}`;
    }

    return url;
  } catch {
    return url; // Return as-is if parsing fails
  }
}

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Popular providers to show at the top of the sidebar
const POPULAR_PROVIDERS = ['openai', 'anthropic', 'google', 'deepseek', 'groq', 'mistral', 'xai'];

// Providers that are actually gateways (aggregate multiple model providers)
const GATEWAY_PROVIDERS = ['netlify', 'openrouter', 'vercel', 'azure-openai'];

const MANUALLY_DOCUMENTED_PROVIDERS = ['azure-openai'];
const MANUALLY_DOCUMENTED_GATEWAYS = ['azure-openai'];

interface ProviderInfo {
  id: string;
  name: string;
  url?: string;
  apiKeyEnvVar: string | string[];
  apiKeyHeader: string;
  models: readonly string[];
  isGateway: boolean;
  isPopular: boolean;
  baseProvider?: string; // For gateway providers like netlify/openai -> openai
  packageName?: string; // Vercel AI SDK package name from models.dev
}

// Zod schema for models.dev API response
const ModelsDevModelSchema = z
  .object({
    id: z.string(),
    context_window: z.number().optional(),
  })
  .passthrough();

const ModelsDevProviderSchema = z
  .object({
    id: z.string(),
    name: z.string(),
    url: z.string().optional(),
    npm: z.string().optional(),
    models: z.record(ModelsDevModelSchema),
  })
  .passthrough();

const ModelsDevResponseSchema = z.record(ModelsDevProviderSchema);

type ModelsDevProvider = z.infer<typeof ModelsDevProviderSchema>;

interface GroupedProviders {
  gateways: Map<string, ProviderInfo[]>; // gateway -> providers
  popular: ProviderInfo[];
  other: ProviderInfo[];
}

async function parseProviders(): Promise<GroupedProviders> {
  // Load provider registry from JSON
  const registryPath = path.join(__dirname, '../src/llm/model/provider-registry.json');
  const registryContent = await fs.readFile(registryPath, 'utf-8');
  const registryData = JSON.parse(registryContent);
  const PROVIDER_REGISTRY = registryData.providers;

  const gateways = new Map<string, ProviderInfo[]>();
  const popular: ProviderInfo[] = [];
  const other: ProviderInfo[] = [];

  for (const [id, config] of Object.entries<ProviderConfig>(PROVIDER_REGISTRY)) {
    if (MANUALLY_DOCUMENTED_PROVIDERS.includes(id)) {
      continue;
    }

    // Check if it's a standalone gateway (like vercel, netlify, etc.)
    const isGateway = GATEWAY_PROVIDERS.includes(id);

    let gatewayName: string | undefined;
    let baseProvider: string | undefined;

    if (isGateway) {
      gatewayName = id;
      baseProvider = undefined; // Standalone gateways don't have a base provider
    }

    const providerInfo: ProviderInfo = {
      id,
      name: config.name,
      url: config.url,
      apiKeyEnvVar: config.apiKeyEnvVar,
      apiKeyHeader: config.apiKeyHeader || 'Authorization',
      models: config.models,
      isGateway,
      isPopular: !isGateway && POPULAR_PROVIDERS.includes(id),
      baseProvider,
    };

    if (isGateway && gatewayName) {
      if (!gateways.has(gatewayName)) {
        gateways.set(gatewayName, []);
      }
      gateways.get(gatewayName)!.push(providerInfo);
    } else if (!isGateway && providerInfo.isPopular) {
      popular.push(providerInfo);
    } else if (!isGateway) {
      other.push(providerInfo);
    }
  }

  // Sort popular providers by the POPULAR_PROVIDERS order
  popular.sort((a, b) => {
    const aIndex = POPULAR_PROVIDERS.indexOf(a.id);
    const bIndex = POPULAR_PROVIDERS.indexOf(b.id);
    return aIndex - bIndex;
  });

  // Sort other providers alphabetically
  other.sort((a, b) => a.name.localeCompare(b.name));

  // Ensure manually documented gateways are present in the grouped map so that
  // navigation pages can still reference them without generating files.
  for (const gatewayId of MANUALLY_DOCUMENTED_GATEWAYS) {
    if (!gateways.has(gatewayId)) {
      gateways.set(gatewayId, []);
    }
  }

  return { gateways, popular, other };
}

async function fetchProviderInfo(providerId: string): Promise<{ models: any[]; packageName?: string }> {
  try {
    const response = await fetch('https://models.dev/api.json');
    const data = await response.json();
    const provider = data[providerId];

    if (!provider?.models) return { models: [] };

    const models = Object.entries(provider.models)
      .map(([modelId, model]: [string, any]) => ({
        model: `${providerId}/${modelId}`,
        imageInput: model.modalities?.input?.includes('image') || false,
        audioInput: model.modalities?.input?.includes('audio') || false,
        videoInput: model.modalities?.input?.includes('video') || false,
        toolUsage: model.tool_call !== false,
        reasoning: model.reasoning === true,
        contextWindow: model.limit?.context || null,
        maxOutput: model.limit?.output || null,
        inputCost: model.cost?.input || null,
        outputCost: model.cost?.output || null,
      }))
      .sort((a, b) => a.model.localeCompare(b.model));

    return {
      models,
      packageName: provider.npm || undefined,
    };
  } catch (error) {
    console.error(`Failed to fetch models for ${providerId}:`, error);
    return { models: [] };
  }
}

async function generateProviderPage(
  provider: ProviderInfo,
  providerRegistry: Record<string, ProviderConfig>,
): Promise<string> {
  const modelCount = provider.models.length;

  // Get documentation URL if available
  const rawDocUrl = (providerRegistry[provider.id] as any).docUrl;
  const docUrl = cleanDocumentationUrl(rawDocUrl);

  // Create intro with optional documentation link
  const introText = docUrl
    ? `Access ${modelCount} ${provider.name} model${modelCount !== 1 ? 's' : ''} through Mastra's model router. Authentication is handled automatically using the \`${provider.apiKeyEnvVar}\` environment variable.

Learn more in the [${provider.name} documentation](${docUrl}).`
    : `Access ${modelCount} ${provider.name} model${modelCount !== 1 ? 's' : ''} through Mastra's model router. Authentication is handled automatically using the \`${provider.apiKeyEnvVar}\` environment variable.`;

  // Fetch model capabilities from models.dev
  const { models: modelsWithCapabilities, packageName } = await fetchProviderInfo(provider.id);
  provider.packageName = packageName;

  // Check for AI SDK docs link if package is available
  const aiSdkDocsLink = packageName ? await checkAiSdkDocsLink(provider.id) : null;

  // Generate static model data as JSON for the component (show all models)
  const modelDataJson = JSON.stringify(modelsWithCapabilities, null, 2);

  return `---
title: "${provider.name} | Models | Mastra"
description: "Use ${provider.name} models with Mastra. ${modelCount} model${modelCount !== 1 ? 's' : ''} available."
---

${getGeneratedComment()}

${provider.packageName && provider.packageName !== '@ai-sdk/openai-compatible' ? 'import Tabs from "@theme/Tabs";\nimport TabItem from "@theme/TabItem";' : ''}

# <img src="${getLogoUrl(provider.id)}" alt="${provider.name} logo" className="${getLogoClass(provider.id)}" />${provider.name}

${introText}

\`\`\`bash
${provider.apiKeyEnvVar}=your-api-key
\`\`\`

\`\`\`typescript
import { Agent } from "@mastra/core/agent";

const agent = new Agent({
  id: "my-agent",
  name: "My Agent",
  instructions: "You are a helpful assistant",
  model: "${provider.id}/${provider.models[0]}"
});

// Generate a response
const response = await agent.generate("Hello!");

// Stream a response
const stream = await agent.stream("Tell me a story");
for await (const chunk of stream) {
  console.log(chunk);
}
\`\`\`
${
  !PROVIDERS_WITH_INSTALLED_PACKAGES.includes(provider.id)
    ? // if it's not a directly supported provider then it's openai compatible, so warn about it
      `
:::info

Mastra uses the OpenAI-compatible \`/chat/completions\` endpoint. Some provider-specific features may not be available. Check the [${provider.name} documentation](${docUrl || '#'}) for details.

:::
`
    : ``
}
## Models

<ProviderModelsTable
  models={${modelDataJson}}
/>

## Advanced Configuration

### Custom Headers

\`\`\`typescript
const agent = new Agent({
  id: "custom-agent",
  name: "custom-agent",
  model: {${
    provider.url
      ? `
    url: "${provider.url}",`
      : ''
  }
    id: "${provider.id}/${provider.models[0]}",
    apiKey: process.env.${provider.apiKeyEnvVar},
    headers: {
      "X-Custom-Header": "value"
    }
  }
});
\`\`\`

### Dynamic Model Selection

\`\`\`typescript
const agent = new Agent({
  id: "dynamic-agent",
  name: "Dynamic Agent",
  model: ({ requestContext }) => {
    const useAdvanced = requestContext.task === "complex";
    return useAdvanced
      ? "${provider.id}/${provider.models[provider.models.length - 1]}"
      : "${provider.id}/${provider.models[0]}";
  }
});
\`\`\`

${generateProviderOptionsSection(provider.id)}
${
  provider.packageName && provider.packageName !== '@ai-sdk/openai-compatible'
    ? `
## Direct Provider Installation

This provider can also be installed directly as a standalone package, which can be used instead of the Mastra model router string. View the [package documentation](https://www.npmjs.com/package/${provider.packageName}) for more details.

\`\`\`bash npm2yarn copy
npm install ${provider.packageName}
\`\`\`
${
  aiSdkDocsLink
    ? `
For detailed provider-specific documentation, see the [AI SDK ${provider.name} provider docs](${aiSdkDocsLink}).`
    : ''
}
`
    : ''
}`;
}

async function checkAiSdkDocsLink(providerId: string): Promise<string | null> {
  const paths = [
    `https://ai-sdk.dev/providers/ai-sdk-providers/${providerId}`,
    `https://ai-sdk.dev/providers/community-providers/${providerId}`,
  ];

  for (const url of paths) {
    try {
      const response = await fetch(url, { method: 'HEAD' });
      if (response.ok) {
        return url;
      }
    } catch {
      // Continue to next URL
    }
  }

  return null;
}

function getLogoUrl(providerId: string): string {
  return `https://models.dev/logos/${providerId}.svg`;
}

function getLogoClass(providerId: string): string {
  // Providers with colored logos that shouldn't be inverted
  const coloredLogos = ['netlify'];

  const baseClass = 'inline w-8 h-8 mr-2 align-middle';

  if (coloredLogos.includes(providerId)) {
    return baseClass;
  }

  return `${baseClass} dark:invert dark:brightness-0 dark:contrast-200`;
}

/**
 * Check if a provider has a React logo component
 */
function hasLogoComponent(providerId: string): boolean {
  const providersWithComponents = ['netlify'];
  return providersWithComponents.includes(providerId);
}

/**
 * Get the logo component import statement for a provider
 */
function getLogoComponentImport(providerId: string): string {
  const componentName = providerId.charAt(0).toUpperCase() + providerId.slice(1) + 'Logo';
  return `import { ${componentName} } from '@site/src/components/logos/${componentName}';`;
}

/**
 * Get the logo component JSX for a provider
 */
function getLogoComponentJSX(providerId: string): string {
  const componentName = providerId.charAt(0).toUpperCase() + providerId.slice(1) + 'Logo';
  return `<${componentName} className="inline w-8 h-8 mr-2 align-middle" />`;
}

function generateGatewayPage(
  gatewayName: string,
  providers: ProviderInfo[],
  providerRegistry: Record<string, any>,
): string {
  const displayName = formatProviderName(gatewayName);
  const totalModels = providers.reduce((sum, p) => sum + p.models.length, 0);
  // Get documentation URL if available
  // Special override for Vercel to use the AI SDK documentation
  let rawDocUrl: string | undefined;

  if (gatewayName === 'vercel') {
    rawDocUrl = 'https://ai-sdk.dev/providers/ai-sdk-providers';
  } else if (providers[0] && !providers[0].baseProvider) {
    // For standalone gateways like groq, openrouter, etc.
    rawDocUrl = (providerRegistry[providers[0].id] as any).docUrl;
  } else if (providers[0]) {
    // For prefixed gateways like netlify/openai
    rawDocUrl = (providerRegistry[providers[0].id] as any).docUrl;
  }

  const docUrl = cleanDocumentationUrl(rawDocUrl);

  // Create intro with optional documentation link
  const gatewayDescription =
    gatewayName === 'netlify'
      ? 'Netlify AI Gateway provides unified access to multiple providers with built-in caching and observability.'
      : `${displayName} aggregates models from multiple providers with enhanced features like rate limiting and failover.`;

  const introText = docUrl
    ? `${gatewayDescription} Access ${totalModels} models through Mastra's model router.

Learn more in the [${displayName} documentation](${docUrl}).`
    : `${gatewayDescription} Access ${totalModels} models through Mastra's model router.`;

  // Create model table for all models (sorted alphabetically)
  const allModels = providers.flatMap(p => p.models).sort((a, b) => a.localeCompare(b));
  const modelTable =
    allModels.length > 0
      ? `
## Available Models

| Model |
|-------|
${allModels.map(m => `| \`${m}\` |`).join('\n')}
`
      : '';

  // Generate logo markup - use component if available, otherwise use img tag
  const logoImport = hasLogoComponent(gatewayName)
    ? `${getLogoComponentImport(gatewayName)}
`
    : '';
  const logoMarkup = hasLogoComponent(gatewayName)
    ? getLogoComponentJSX(gatewayName)
    : `<img src="${getLogoUrl(gatewayName)}" alt="${displayName} logo" className="${getLogoClass(gatewayName)}" />`;

  return `---
title: "${displayName} | Models | Mastra"
description: "Use AI models through ${displayName}."
---

${getGeneratedComment()}

${logoImport}

# ${logoMarkup}${displayName}

${introText}

## Usage

\`\`\`typescript
import { Agent } from "@mastra/core/agent";

const agent = new Agent({
  id: "my-agent",
  name: "My Agent",
  instructions: "You are a helpful assistant",
  model: "${gatewayName}/${providers[0]?.models[0] || 'model-name'}"
});
\`\`\`

:::info

Mastra uses the OpenAI-compatible \`/chat/completions\` endpoint. Some provider-specific features may not be available. ${docUrl ? `Check the [${displayName} documentation](${docUrl}) for details.` : `Check the ${displayName} documentation for details.`}

:::

## Configuration

\`\`\`bash
# Use gateway API key
${gatewayName.toUpperCase()}_API_KEY=your-gateway-key

# Or use provider API keys directly
OPENAI_API_KEY=sk-...
ANTHROPIC_API_KEY=ant-...
\`\`\`

${modelTable}
`;
}

function generateIndexPage(grouped: GroupedProviders): string {
  const totalProviders = grouped.popular.length + grouped.other.length + grouped.gateways.size;
  const totalModels =
    [...grouped.popular, ...grouped.other].reduce((sum, p) => sum + p.models.length, 0) +
    Array.from(grouped.gateways.values())
      .flat()
      .reduce((sum, p) => sum + p.models.length, 0);

  return `---
title: "Models"
description: "Access ${totalProviders}+ AI providers and ${totalModels}+ models through Mastra's model router."
---

${getGeneratedComment()}

import { CardGrid, CardGridItem } from "@site/src/components/cards/card-grid";
import Tabs from '@theme/Tabs';
import TabItem from '@theme/TabItem';
import { NetlifyLogo } from "@site/src/components/logos/NetlifyLogo";

# Model Providers

Mastra provides a unified interface for working with LLMs across multiple providers, giving you access to ${totalModels} models from ${totalProviders} providers through a single API.

## Features

- **One API for any model** - Access any model without having to install and manage additional provider dependencies.

- **Access the newest AI** - Use new models the moment they're released, no matter which provider they come from. Avoid vendor lock-in with Mastra's provider-agnostic interface.

- [**Mix and match models**](#mix-and-match-models) - Use different models for different tasks. For example, run GPT-4o-mini for large-context processing, then switch to Claude Opus 4.1 for reasoning tasks.

- [**Model fallbacks**](#model-fallbacks) - If a provider experiences an outage, Mastra can automatically switch to another provider at the application level, minimizing latency compared to API gateways.

## Basic usage

Whether you're using OpenAI, Anthropic, Google, or a gateway like OpenRouter, specify the model as \`"provider/model-name"\` and Mastra handles the rest.

Mastra reads the relevant environment variable (e.g. \`ANTHROPIC_API_KEY\`) and routes requests to the provider. If an API key is missing, you'll get a clear runtime error showing exactly which variable to set.

<Tabs>
  <TabItem value="OpenAI" label="OpenAI">
    \`\`\`typescript
    import { Agent } from "@mastra/core/agent";

    const agent = new Agent({
      id: "my-agent",
      name: "My Agent",
      instructions: "You are a helpful assistant",
      model: "openai/gpt-5"
    })
    \`\`\`
  </TabItem>
  <TabItem value="Anthropic" label="Anthropic">
    \`\`\`typescript
    import { Agent } from "@mastra/core/agent";

    const agent = new Agent({
      id: "my-agent",
      name: "My Agent",
      instructions: "You are a helpful assistant",
      model: "anthropic/claude-4-5-sonnet"
    })
    \`\`\`
  </TabItem>
  <TabItem value="Google Gemini" label="Google Gemini">
    \`\`\`typescript
    import { Agent } from "@mastra/core/agent";

    const agent = new Agent({
      id: "my-agent",
      name: "My Agent",
      instructions: "You are a helpful assistant",
      model: "google/gemini-2.5-flash"
    })
    \`\`\`
  </TabItem>
  <TabItem value="xAI" label="xAI">
    \`\`\`typescript
    import { Agent } from "@mastra/core/agent";

    const agent = new Agent({
      id: "my-agent",
      name: "My Agent",
      instructions: "You are a helpful assistant",
      model: "xai/grok-4"
    })
    \`\`\`
  </TabItem>
  <TabItem value="OpenRouter" label="OpenRouter">
    \`\`\`typescript
    import { Agent } from "@mastra/core/agent";

    const agent = new Agent({
      id: "my-agent",
      name: "My Agent",
      instructions: "You are a helpful assistant",
      model: "openrouter/anthropic/claude-haiku-4-5"
    })
    \`\`\`
  </TabItem>
</Tabs>


## Model directory

Browse the directory of available models using the navigation on the left, or explore below.

<CardGrid>
    <CardGridItem
      title="Gateways"
      href="/models/v1/gateways"
    >
      <div className="space-y-3">
        <div className="flex flex-col gap-2">
${(() => {
  const gatewayOrder = ['openrouter', 'netlify', 'vercel'];
  const allGateways = Array.from(grouped.gateways.keys());
  const orderedGateways = gatewayOrder.filter(g => allGateways.includes(g));
  const remainingGateways = allGateways.filter(g => !gatewayOrder.includes(g)).sort((a, b) => a.localeCompare(b));
  const finalOrder = [...orderedGateways, ...remainingGateways];

  return finalOrder
    .slice(0, 3)
    .map(gatewayId => {
      const providers = grouped.gateways.get(gatewayId);
      let displayName = providers?.[0]?.name || gatewayId;
      // Simplify "Vercel AI Gateway" to just "Vercel"
      if (gatewayId === 'vercel') {
        displayName = 'Vercel';
      }

      // Use NetlifyLogo component for Netlify, img tag for others
      const logoMarkup =
        gatewayId === 'netlify'
          ? `<NetlifyLogo className="w-4 h-4" />`
          : `<img src="${getLogoUrl(gatewayId)}" alt="${displayName}" className="w-4 h-4 object-contain dark:invert dark:brightness-0 dark:contrast-200" />`;

      return `          <div className="flex items-center gap-2 text-sm">
            ${logoMarkup}
            <span>${displayName}</span>
          </div>`;
    })
    .join('\n');
})()}
        </div>
${grouped.gateways.size > 3 ? `        <div className="text-sm text-gray-600 dark:text-gray-400 mt-3">+ ${grouped.gateways.size - 3} more</div>` : ''}
      </div>
    </CardGridItem>
    <CardGridItem
      title="Providers"
      href="/models/v1/providers"
    >
      <div className="space-y-3">
        <div className="flex flex-col gap-2">
          <div className="flex items-center gap-2 text-sm">
            <img src="${getLogoUrl('openai')}" alt="OpenAI" className="w-4 h-4 object-contain dark:invert dark:brightness-0 dark:contrast-200" />
            <span>OpenAI</span>
          </div>
          <div className="flex items-center gap-2 text-sm">
            <img src="${getLogoUrl('anthropic')}" alt="Anthropic" className="w-4 h-4 object-contain dark:invert dark:brightness-0 dark:contrast-200" />
            <span>Anthropic</span>
          </div>
          <div className="flex items-center gap-2 text-sm">
            <img src="${getLogoUrl('google')}" alt="Google" className="w-4 h-4 object-contain dark:invert dark:brightness-0 dark:contrast-200" />
            <span>Google</span>
          </div>
        </div>
        <div className="text-sm text-gray-600 dark:text-gray-400 mt-3">+ ${grouped.popular.length + grouped.other.length - 3} more</div>
      </div>
    </CardGridItem>
</CardGrid>

You can also discover models directly in your editor. Mastra provides full autocomplete for the \`model\` field - just start typing, and your IDE will show available options.

Alternatively, browse and test models in [Studio](/docs/v1/getting-started/studio) UI.

:::info

In development, we auto-refresh your local model list every hour, ensuring your TypeScript autocomplete and Studio stay up-to-date with the latest models. To disable, set \`MASTRA_AUTO_REFRESH_PROVIDERS=false\`. Auto-refresh is disabled by default in production.

:::


## Mix and match models

Some models are faster but less capable, while others offer larger context windows or stronger reasoning skills. Use different models from the same provider, or mix and match across providers to fit each task.

\`\`\`typescript
import { Agent } from "@mastra/core/agent";

// Use a cost-effective model for document processing
const documentProcessor = new Agent({
  id: "document-processor",
  name: "Document Processor",
  instructions: "Extract and summarize key information from documents",
  model: "openai/gpt-4o-mini"
})

// Use a powerful reasoning model for complex analysis
const reasoningAgent = new Agent({
  id: "reasoning-agent",
  name: "Reasoning Agent",
  instructions: "Analyze data and provide strategic recommendations",
  model: "anthropic/claude-opus-4-1"
})
\`\`\`
## Dynamic model selection

Since models are just strings, you can select them dynamically based on [request context](/docs/v1/server/request-context), variables, or any other logic.

\`\`\`typescript
const agent = new Agent({
  id: "dynamic-assistant",
  name: "Dynamic Assistant",
  model: ({ requestContext }) => {
    const provider = requestContext.get("provider-id");
    const model = requestContext.get("model-id");
    return \`\${provider}/\${model}\`;
  },
});
\`\`\`

This enables powerful patterns:

- A/B testing - Compare model performance in production.
- User-selectable models - Let users choose their preferred model in your app.
- Multi-tenant applications - Each customer can bring their own API keys and model preferences.

## Provider-specific options

Different model providers expose their own configuration options. With OpenAI, you might adjust the \`reasoningEffort\`. With Anthropic, you might tune \`cacheControl\`. Mastra lets you set these specific \`providerOptions\` either at the agent level or per message.

\`\`\`typescript
// Agent level (apply to all future messages)
const planner = new Agent({
  id: "planner",
  name: "Planner",
  instructions: {
    role: "system",
    content: "You are a helpful assistant.",
    providerOptions: {
      openai: { reasoningEffort: "low" }
    }
  },
  model: "openai/o3-pro",
});

const lowEffort =
  await planner.generate("Plan a simple 3 item dinner menu");

// Message level (apply only to this message)
const highEffort = await planner.generate([
  {
    role: "user",
    content: "Plan a simple 3 item dinner menu for a celiac",
    providerOptions: {
      openai: { reasoningEffort: "high" }
    }
  }
]);
\`\`\`

## Custom headers

If you need to specify custom headers, such as an organization ID or other provider-specific fields, use this syntax.


\`\`\`typescript
const agent = new Agent({
  id: "custom-agent",
  name: "Custom Agent",
  model: {
    id: "openai/gpt-4-turbo",
    apiKey: process.env.OPENAI_API_KEY,
    headers: {
      "OpenAI-Organization": "org-abc123"
    }
  }
});
\`\`\`

:::info

Configuration differs by provider. See the provider pages in the left navigation for details on custom headers.

:::

## Model fallbacks

Relying on a single model creates a single point of failure for your application. Model fallbacks provide automatic failover between models and providers. If the primary model becomes unavailable, requests are retried against the next configured fallback until one succeeds.


\`\`\`typescript
import { Agent } from '@mastra/core/agent';

const agent = new Agent({
  id: 'resilient-assistant',
  name: 'Resilient Assistant',
  instructions: 'You are a helpful assistant.',
  model: [
    {
      model: "openai/gpt-5",
      maxRetries: 3,
    },
    {
      model: "anthropic/claude-4-5-sonnet",
      maxRetries: 2,
    },
    {
      model: "google/gemini-2.5-pro",
      maxRetries: 2,
    },
  ],
});
\`\`\`
Mastra tries your primary model first. If it encounters a 500 error, rate limit, or timeout, it automatically switches to your first fallback. If that fails too, it moves to the next. Each model gets its own retry count before moving on.

Your users never experience the disruption - the response comes back with the same format, just from a different model. The error context is preserved as the system moves through your fallback chain, ensuring clean error propagation while maintaining streaming compatibility.

## Use AI SDK with Mastra

Mastra supports AI SDK provider modules, should you need to use them directly.


\`\`\`typescript
import { groq } from '@ai-sdk/groq';
import { Agent } from "@mastra/core/agent";

const agent = new Agent({
  id: "my-agent",
  name: "My Agent",
  model: groq('gemma2-9b-it')
})
\`\`\`
You can use an AI SDK model (e.g. \`groq('gemma2-9b-it')\`) anywhere that accepts a \`"provider/model"\` string, including within model router fallbacks and [scorers](/docs/v1/evals/overview).`;
}

function generateGatewaysIndexPage(grouped: GroupedProviders): string {
  // Sort gateways alphabetically
  const gatewaysList = Array.from(grouped.gateways.keys()).sort((a, b) => a.localeCompare(b));

  const hasNetlify = gatewaysList.includes('netlify');
  const logoImport = hasNetlify ? 'import { NetlifyLogo } from "@site/src/components/logos/NetlifyLogo";' : '';

  return `---
title: "Gateways"
description: "Access AI models through gateway providers with caching, rate limiting, and analytics."
---

${getGeneratedComment()}

import { CardGrid, CardGridItem } from "@site/src/components/cards/card-grid";${logoImport}

# Gateway Providers

Gateway providers aggregate multiple model providers and add features like caching, rate limiting, analytics, and automatic failover. Use gateways when you need observability, cost management, or simplified multi-provider access.

## Custom Gateways

Create custom gateways for private LLM deployments or specialized provider integrations. See [Custom Gateways](/models/v1/gateways/custom-gateways) for implementation details.

## Built-in Gateways

<CardGrid>
${gatewaysList
  .map(g => {
    // Custom descriptions for manually documented gateways
    if (MANUALLY_DOCUMENTED_GATEWAYS.includes(g)) {
      if (g === 'azure-openai') {
        return `    <CardGridItem
      title="Azure OpenAI"
      description="Use your private Azure OpenAI deployments with associated deployment names"
      href="/models/v1/gateways/${g}"
      logo="${getLogoUrl(g)}"
    />`;
      }
    }

    if (g === 'netlify') {
      return `    <CardGridItem
      title="${formatProviderName(g).replace(/&/g, '&amp;')}"
      description="${grouped.gateways.get(g)?.reduce((sum, p) => sum + p.models.length, 0) || 0} models"
      href="/models/v1/gateways/${g}"
      logo={<NetlifyLogo />}
    />`;
    }
    return `    <CardGridItem
      title="${formatProviderName(g).replace(/&/g, '&amp;')}"
      description="${grouped.gateways.get(g)?.reduce((sum, p) => sum + p.models.length, 0) || 0} models"
      href="/models/v1/gateways/${g}"
      logo="${getLogoUrl(g)}"

    />`;
  })
  .join(
    '\
',
  )}
</CardGrid>`;
}

function generateProvidersIndexPage(grouped: GroupedProviders): string {
  const allProviders = [...grouped.popular, ...grouped.other];

  return `---
title: "Providers"
description: "Direct access to AI model providers."
---

${getGeneratedComment()}

import { CardGrid, CardGridItem } from "@site/src/components/cards/card-grid";

# Model Providers

Direct access to individual AI model providers. Each provider offers unique models with specific capabilities and pricing.

<CardGrid>
${allProviders
  .map(
    p => `    <CardGridItem
      title="${p.name.replace(/&/g, '&amp;')}"
      description="${p.models.length} models"
      href="/models/v1/providers/${p.id}"
      logo="${getLogoUrl(p.id)}"
    />`,
  )
  .join(
    '\
',
  )}
</CardGrid>`;
}

function generateProvidersSidebarItems(grouped: GroupedProviders, aiSdkProviders: ModelsDevProvider[] = []): any[] {
  // Keep popular providers in their original order
  const popularProviders = grouped.popular.map(p => ({
    type: 'doc',
    id: `providers/${p.id}`,
    label: p.name,
  }));

  // Combine "other" model router providers with AI SDK providers and sort alphabetically
  const otherProviders = [
    ...grouped.other.map(p => ({
      type: 'doc',
      id: `providers/${p.id}`,
      label: p.name,
    })),
    ...aiSdkProviders.map(p => {
      let displayName = p.name;
      if (p.id === 'google-vertex') {
        displayName = 'Google Vertex AI';
      } else if (p.id === 'google-vertex-anthropic') {
        displayName = 'Vertex AI (Anthropic)';
      }
      return {
        type: 'doc',
        id: `providers/${p.id}`,
        label: displayName,
      };
    }),
  ].sort((a, b) => a.label.localeCompare(b.label));

  return [{ type: 'doc', id: 'providers/index', label: 'Providers' }, ...popularProviders, ...otherProviders];
}

async function generateAiSdkProviderPage(provider: any, aiSdkDocsUrl: string | null): Promise<string> {
  const packageName = provider.npm;
  const logoUrl = getLogoUrl(provider.id);
  const logoClass = getLogoClass(provider.id);

  const aiSdkDocsText = aiSdkDocsUrl
    ? `

For detailed provider-specific documentation, see the [AI SDK ${provider.name} provider docs](${aiSdkDocsUrl}).`
    : '';

  return `---
title: "${provider.name}"
description: "Use ${provider.name} models via the AI SDK."
---

${getGeneratedComment()}

# <img src="${logoUrl}" alt="${provider.name} logo" className="${logoClass}" />${provider.name}

${provider.name} is available through the AI SDK. Install the provider package to use their models with Mastra.${aiSdkDocsText}

To use this provider with Mastra agents, see the [Agent Overview documentation](/docs/v1/agents/overview).

## Installation

\`\`\`bash npm2yarn copy
npm install ${packageName}
\`\`\`
`;
}

function generateGatewaysSidebarItems(grouped: GroupedProviders): any[] {
  // Sort gateways alphabetically
  const gatewaysList = Array.from(grouped.gateways.keys()).sort((a, b) => a.localeCompare(b));

  const items = [
    { type: 'doc', id: 'gateways/index', label: 'Gateways' },
    { type: 'doc', id: 'gateways/custom-gateways', label: 'Custom Gateways' },
  ];

  for (const gatewayId of gatewaysList) {
    const providers = grouped.gateways.get(gatewayId);
    // Include manually documented gateways even if they have no providers
    const isManuallyDocumented = MANUALLY_DOCUMENTED_GATEWAYS.includes(gatewayId);
    if ((providers && providers.length > 0) || isManuallyDocumented) {
      const name = formatProviderName(gatewayId);
      items.push({
        type: 'doc',
        id: `gateways/${gatewayId}`,
        label: name,
      });
    }
  }

  return items;
}

function generateSidebarsFile(grouped: GroupedProviders, aiSdkProviders: ModelsDevProvider[] = []): string {
  const gatewaysItems = generateGatewaysSidebarItems(grouped);
  const providersItems = generateProvidersSidebarItems(grouped, aiSdkProviders);

  return `/**
 * Sidebar for Models
 */

// @ts-check

/** @type {import('@docusaurus/plugin-content-docs').SidebarsConfig} */
const sidebars = {
  modelsSidebar: [
    "index",
    "embeddings",
    {
      type: "category",
      label: "Gateways",
      collapsed: false,
      items: ${JSON.stringify(gatewaysItems, null, 6).replace(/^/gm, '      ').trim()},
    },
    {
      type: "category",
      label: "Providers",
      collapsed: false,
      items: ${JSON.stringify(providersItems, null, 6).replace(/^/gm, '      ').trim()},
    },
  ],
};

export default sidebars;
`;
}

async function generateDocs() {
  const docsDir = path.join(__dirname, '..', '..', '..', 'docs', 'src', 'content', 'en', 'models');
  const providersDir = path.join(docsDir, 'providers');
  const gatewaysDir = path.join(docsDir, 'gateways');

  // Create directories
  await fs.mkdir(docsDir, { recursive: true });
  await fs.mkdir(providersDir, { recursive: true });
  await fs.mkdir(gatewaysDir, { recursive: true });

  // Load provider registry from JSON
  const registryPath = path.join(__dirname, '../src/llm/model/provider-registry.json');
  const registryContent = await fs.readFile(registryPath, 'utf-8');
  const registryData = JSON.parse(registryContent);
  const providerRegistry = registryData.providers;

  const grouped = await parseProviders();

  // Fetch all providers from models.dev for AI SDK provider filtering
  console.info('🔍 Fetching provider data from models.dev...');
  const modelsDevResponse = await fetch('https://models.dev/api.json');
  const modelsDevData = ModelsDevResponseSchema.parse(await modelsDevResponse.json());
  // Convert object to array of providers
  const allModelsDevProviders: ModelsDevProvider[] = Object.values(modelsDevData);

  // Generate index page
  const indexContent = generateIndexPage(grouped);
  await fs.writeFile(path.join(docsDir, 'index.mdx'), indexContent);
  console.info('✅ Generated models/index.mdx');

  // Generate gateways overview page
  const gatewaysIndexContent = generateGatewaysIndexPage(grouped);
  await fs.writeFile(path.join(gatewaysDir, 'index.mdx'), gatewaysIndexContent);
  console.info('✅ Generated gateways/index.mdx');

  // Generate AI SDK provider documentation
  console.info(
    '\
🔍 Filtering AI SDK providers...',
  );
  const supportedProviderIds = new Set(Object.keys(providerRegistry));
  const aiSdkProviders = allModelsDevProviders
    .filter(p => {
      return (
        p.npm && // Has an npm package
        !supportedProviderIds.has(p.id) && // Not in our model router
        !EXCLUDED_PROVIDERS.includes(p.id) // Not excluded
      );
    })
    .sort((a, b) => a.name.localeCompare(b.name));

  // Add Ollama as a custom AI SDK provider
  aiSdkProviders.push({
    id: 'ollama',
    name: 'Ollama',
    npm: 'ollama-ai-provider-v2',
    models: {},
  });

  console.info(`Found ${aiSdkProviders.length} AI SDK providers`);

  // Generate providers overview page
  const providersIndexContent = generateProvidersIndexPage(grouped);
  await fs.writeFile(path.join(providersDir, 'index.mdx'), providersIndexContent);
  console.info('✅ Generated providers/index.mdx');

  // Generate individual provider pages (parallelized)
  await Promise.all(
    [...grouped.popular, ...grouped.other]
      .filter(provider => !MANUALLY_DOCUMENTED_PROVIDERS.includes(provider.id))
      .map(async provider => {
        const content = await generateProviderPage(provider, providerRegistry);
        await fs.writeFile(path.join(providersDir, `${provider.id}.mdx`), content);
        console.info(`✅ Generated providers/${provider.id}.mdx`);
      }),
  );

  // Generate individual AI SDK provider pages (parallelized, only if they have AI SDK docs)
  const aiSdkProviderResults = await Promise.all(
    aiSdkProviders.map(async provider => {
      const aiSdkDocsUrl = await checkAiSdkDocsLink(provider.id);
      if (!aiSdkDocsUrl) {
        console.info(`⏭️  Skipping providers/${provider.id}.mdx (no AI SDK docs found)`);
        return null;
      }
      const content = await generateAiSdkProviderPage(provider, aiSdkDocsUrl);
      await fs.writeFile(path.join(providersDir, `${provider.id}.mdx`), content);
      console.info(`✅ Generated providers/${provider.id}.mdx (AI SDK)`);
      return provider;
    }),
  );

  const aiSdkProvidersWithDocs = aiSdkProviderResults.filter((p): p is ModelsDevProvider => p !== null);

  // Generate individual gateway pages (parallelized)
  await Promise.all(
    Array.from(grouped.gateways.entries())
      .filter(([gatewayName]) => !MANUALLY_DOCUMENTED_GATEWAYS.includes(gatewayName))
      .map(async ([gatewayName, providers]) => {
        const content = generateGatewayPage(gatewayName, providers, providerRegistry);
        await fs.writeFile(path.join(gatewaysDir, `${gatewayName}.mdx`), content);
        console.info(`✅ Generated gateways/${gatewayName}.mdx`);
      }),
  );

  // Generate sidebars.js (including AI SDK providers with docs)
  const sidebarsContent = generateSidebarsFile(grouped, aiSdkProvidersWithDocs);
  await fs.writeFile(path.join(docsDir, 'sidebars.js'), sidebarsContent);
  console.info('✅ Generated models/sidebars.js');

  console.info(`
📚 Documentation generated successfully!
   - ${grouped.popular.length + grouped.other.length + aiSdkProviders.length} provider pages + 1 overview
   - ${grouped.gateways.size} gateway pages + 1 overview
   - 1 main index page

   Total: ${grouped.popular.length + grouped.other.length + aiSdkProviders.length + grouped.gateways.size + 3} pages generated
  `);
}

// Run if executed directly
if (import.meta.url === `file://${process.argv[1]}`) {
  generateDocs().catch(error => {
    console.error('Failed to generate documentation:', error);
    process.exit(1);
  });
}

export {
  generateDocs,
  parseProviders,
  generateProviderPage,
  generateGatewayPage,
  generateIndexPage,
  generateSidebarsFile,
};
